/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.commons.compress.utils;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNull;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;

import org.apache.commons.lang3.SystemUtils;
import org.junit.jupiter.api.Test;

class FileNameUtilsTest {

    @Test
    void testGetBaseNamePathBaseCases() {
        assertEquals("bar", FileNameUtils.getBaseName(Paths.get("a/b/c/bar.foo")));
        assertEquals("foo", FileNameUtils.getBaseName(Paths.get("foo")));
        assertEquals("", FileNameUtils.getBaseName(Paths.get("")));
        assertEquals("", FileNameUtils.getBaseName(Paths.get(".")));
        for (final File f : File.listRoots()) {
            assertNull(FileNameUtils.getBaseName(f.toPath()));
        }
        if (SystemUtils.IS_OS_WINDOWS) {
            assertNull(FileNameUtils.getBaseName(Paths.get("C:\\")));
        }
    }

    @Test
    void testGetBaseNamePathCornerCases() {
        assertNull(FileNameUtils.getBaseName((Path) null));
        assertEquals("foo", FileNameUtils.getBaseName(Paths.get("foo.")));
        assertEquals("", FileNameUtils.getBaseName(Paths.get("bar/.foo")));
    }

    @Test
    void testGetBaseNameStringBaseCases() {
        assertEquals("bar", FileNameUtils.getBaseName("a/b/c/bar.foo"));
        assertEquals("foo", FileNameUtils.getBaseName("foo"));
    }

    @Test
    void testGetBaseNameStringCornerCases() {
        assertNull(FileNameUtils.getBaseName((String) null));
        assertEquals("foo", FileNameUtils.getBaseName("foo."));
        assertEquals("", FileNameUtils.getBaseName("bar/.foo"));
    }

    @Test
    void testGetExtensionPathBaseCases() {
        assertEquals("foo", FileNameUtils.getExtension(Paths.get("a/b/c/bar.foo")));
        assertEquals("", FileNameUtils.getExtension(Paths.get("foo")));
        assertEquals("", FileNameUtils.getExtension(Paths.get("")));
        assertEquals("", FileNameUtils.getExtension(Paths.get(".")));
        for (final File f : File.listRoots()) {
            assertNull(FileNameUtils.getExtension(f.toPath()));
        }
        if (SystemUtils.IS_OS_WINDOWS) {
            assertNull(FileNameUtils.getExtension(Paths.get("C:\\")));
        }
    }

    @Test
    void testGetExtensionPathCornerCases() {
        assertNull(FileNameUtils.getExtension((String) null));
        assertEquals("", FileNameUtils.getExtension(Paths.get("foo.")));
        assertEquals("foo", FileNameUtils.getExtension(Paths.get("bar/.foo")));
    }

    @Test
    void testGetExtensionStringBaseCases() {
        assertEquals("foo", FileNameUtils.getExtension("a/b/c/bar.foo"));
        assertEquals("", FileNameUtils.getExtension("foo"));
    }

    @Test
    void testGetExtensionStringCornerCases() {
        assertNull(FileNameUtils.getExtension((String) null));
        assertEquals("", FileNameUtils.getExtension("foo."));
        assertEquals("foo", FileNameUtils.getExtension("bar/.foo"));
    }
}
