/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.websocket.core;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.ReadPendingException;
import java.security.SecureRandom;
import java.util.Objects;
import java.util.Random;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.LongAdder;
import org.eclipse.jetty.io.AbstractConnection;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.Connection;
import org.eclipse.jetty.io.EndPoint;
import org.eclipse.jetty.io.RetainableByteBuffer;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.thread.AutoLock;
import org.eclipse.jetty.util.thread.Scheduler;
import org.eclipse.jetty.websocket.core.Behavior;
import org.eclipse.jetty.websocket.core.Frame;
import org.eclipse.jetty.websocket.core.WebSocketCoreSession;
import org.eclipse.jetty.websocket.core.exception.WebSocketTimeoutException;
import org.eclipse.jetty.websocket.core.internal.FrameFlusher;
import org.eclipse.jetty.websocket.core.internal.Generator;
import org.eclipse.jetty.websocket.core.internal.Parser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WebSocketConnection
extends AbstractConnection
implements Connection.UpgradeTo,
Dumpable,
Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(WebSocketConnection.class);
    private static final int MIN_BUFFER_SIZE = 28;
    private final AutoLock lock = new AutoLock();
    private final ByteBufferPool byteBufferPool;
    private final Generator generator;
    private final Parser parser;
    private final WebSocketCoreSession coreSession;
    private final Flusher flusher;
    private final Random random;
    private DemandState demand = DemandState.NOT_DEMANDING;
    private boolean fillingAndParsing = true;
    private final LongAdder messagesIn = new LongAdder();
    private final LongAdder bytesIn = new LongAdder();
    private RetainableByteBuffer networkBuffer;
    private boolean useInputDirectByteBuffers;
    private boolean useOutputDirectByteBuffers;

    public WebSocketConnection(EndPoint endp, Executor executor, Scheduler scheduler, ByteBufferPool byteBufferPool, WebSocketCoreSession coreSession) {
        this(endp, executor, scheduler, byteBufferPool, coreSession, null);
    }

    public WebSocketConnection(EndPoint endp, Executor executor, Scheduler scheduler, ByteBufferPool byteBufferPool, WebSocketCoreSession coreSession, Random randomMask) {
        super(endp, executor);
        Objects.requireNonNull(endp, "EndPoint");
        Objects.requireNonNull(coreSession, "Session");
        Objects.requireNonNull(executor, "Executor");
        Objects.requireNonNull(byteBufferPool, "ByteBufferPool");
        this.byteBufferPool = byteBufferPool;
        this.coreSession = coreSession;
        this.generator = new Generator();
        this.parser = new Parser(byteBufferPool, coreSession);
        this.flusher = new Flusher(scheduler, coreSession.getOutputBufferSize(), this.generator, endp);
        this.setInputBufferSize(coreSession.getInputBufferSize());
        if (this.coreSession.getBehavior() == Behavior.CLIENT && randomMask == null) {
            randomMask = new SecureRandom();
        }
        this.random = randomMask;
    }

    @Override
    public Executor getExecutor() {
        return super.getExecutor();
    }

    @Deprecated
    public InetSocketAddress getLocalAddress() {
        SocketAddress local = this.getLocalSocketAddress();
        if (local instanceof InetSocketAddress) {
            return (InetSocketAddress)local;
        }
        return null;
    }

    public SocketAddress getLocalSocketAddress() {
        return this.getEndPoint().getLocalSocketAddress();
    }

    @Deprecated
    public InetSocketAddress getRemoteAddress() {
        SocketAddress remote = this.getRemoteSocketAddress();
        if (remote instanceof InetSocketAddress) {
            return (InetSocketAddress)remote;
        }
        return null;
    }

    public SocketAddress getRemoteSocketAddress() {
        return this.getEndPoint().getRemoteSocketAddress();
    }

    public boolean isUseInputDirectByteBuffers() {
        return this.useInputDirectByteBuffers;
    }

    public void setWriteTimeout(long writeTimeout) {
        this.flusher.setIdleTimeout(writeTimeout);
    }

    public void setUseInputDirectByteBuffers(boolean useInputDirectByteBuffers) {
        this.useInputDirectByteBuffers = useInputDirectByteBuffers;
    }

    public boolean isUseOutputDirectByteBuffers() {
        return this.useOutputDirectByteBuffers;
    }

    public void setUseOutputDirectByteBuffers(boolean useOutputDirectByteBuffers) {
        this.useOutputDirectByteBuffers = useOutputDirectByteBuffers;
    }

    @Override
    public void onClose(Throwable cause) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("onClose() of physical connection");
        }
        if (!this.coreSession.isClosed()) {
            this.coreSession.onEof();
        }
        this.flusher.onClose(cause);
        try (AutoLock ignored = this.lock.lock();){
            if (this.networkBuffer != null) {
                this.networkBuffer.clear();
                this.releaseNetworkBuffer();
            }
        }
        super.onClose(cause);
    }

    @Override
    public boolean onIdleExpired(TimeoutException timeoutException) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("onIdleExpired()");
        }
        this.coreSession.processHandlerError(new WebSocketTimeoutException("Connection Idle Timeout", timeoutException), Callback.NOOP);
        return true;
    }

    @Override
    protected boolean onReadTimeout(TimeoutException timeout) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("onReadTimeout()");
        }
        this.coreSession.processHandlerError(new WebSocketTimeoutException("Timeout on Read", timeout), Callback.NOOP);
        return false;
    }

    protected void onFrame(final Frame.Parsed frame) {
        RetainableByteBuffer referenced;
        if (LOG.isDebugEnabled()) {
            LOG.debug("onFrame({})", (Object)frame);
        }
        RetainableByteBuffer retainableByteBuffer = referenced = frame.hasPayload() && !frame.isReleaseable() ? this.networkBuffer : null;
        if (referenced != null) {
            referenced.retain();
        }
        this.coreSession.onFrame(frame, new Callback(){
            final /* synthetic */ WebSocketConnection this$0;
            {
                this.this$0 = this$0;
            }

            @Override
            public void succeeded() {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("succeeded onFrame({})", (Object)frame);
                }
                frame.close();
                if (referenced != null) {
                    referenced.release();
                }
            }

            @Override
            public void failed(Throwable cause) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("failed onFrame({}) {}", (Object)frame, (Object)cause.toString());
                }
                frame.close();
                if (referenced != null) {
                    referenced.release();
                }
                this.this$0.coreSession.processHandlerError(cause, NOOP);
            }
        });
    }

    private void acquireNetworkBuffer() {
        try (AutoLock l = this.lock.lock();){
            if (this.networkBuffer == null) {
                this.networkBuffer = this.newNetworkBuffer(this.getInputBufferSize());
            }
        }
    }

    private void reacquireNetworkBuffer() {
        try (AutoLock l = this.lock.lock();){
            if (this.networkBuffer == null) {
                throw new IllegalStateException();
            }
            if (this.networkBuffer.getByteBuffer().hasRemaining()) {
                throw new IllegalStateException();
            }
            this.networkBuffer.release();
            this.networkBuffer = this.newNetworkBuffer(this.getInputBufferSize());
        }
    }

    private RetainableByteBuffer newNetworkBuffer(int capacity) {
        return this.byteBufferPool.acquire(capacity, this.isUseInputDirectByteBuffers());
    }

    private void releaseNetworkBuffer() {
        try (AutoLock l = this.lock.lock();){
            if (this.networkBuffer == null) {
                throw new IllegalStateException();
            }
            if (this.networkBuffer.hasRemaining()) {
                throw new IllegalStateException();
            }
            this.networkBuffer.release();
            this.networkBuffer = null;
        }
    }

    @Override
    public void onFillable() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("onFillable()");
        }
        this.fillAndParse();
    }

    @Override
    public void run() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("run()");
        }
        this.fillAndParse();
    }

    public void demand() {
        boolean fillAndParse = false;
        try (AutoLock l = this.lock.lock();){
            if (LOG.isDebugEnabled()) {
                LOG.debug("demand {} d={} fp={} {}", new Object[]{this.demand, this.fillingAndParsing, this.networkBuffer, this});
            }
            if (this.demand != DemandState.CANCELLED) {
                if (this.demand == DemandState.DEMANDING) {
                    throw new ReadPendingException();
                }
                this.demand = DemandState.DEMANDING;
            }
            if (!this.fillingAndParsing) {
                this.fillingAndParsing = true;
                fillAndParse = true;
            }
        }
        if (fillAndParse) {
            this.getExecutor().execute(this);
        }
    }

    public boolean moreDemand() {
        try (AutoLock l = this.lock.lock();){
            if (LOG.isDebugEnabled()) {
                LOG.debug("moreDemand? d={} fp={} {} {}", new Object[]{this.demand, this.fillingAndParsing, this.networkBuffer, this});
            }
            if (!this.fillingAndParsing) {
                throw new IllegalStateException();
            }
            switch (this.demand.ordinal()) {
                case 1: {
                    this.fillingAndParsing = false;
                    if (this.networkBuffer != null && !this.networkBuffer.hasRemaining()) {
                        this.releaseNetworkBuffer();
                    }
                    boolean bl = false;
                    return bl;
                }
                case 0: 
                case 2: {
                    boolean bl = true;
                    return bl;
                }
            }
            throw new IllegalStateException(this.demand.name());
        }
    }

    public boolean meetDemand() {
        try (AutoLock l = this.lock.lock();){
            if (LOG.isDebugEnabled()) {
                LOG.debug("meetDemand d={} fp={} {} {}", new Object[]{this.demand, this.fillingAndParsing, this.networkBuffer, this});
            }
            if (this.demand == DemandState.NOT_DEMANDING) {
                throw new IllegalStateException();
            }
            if (!this.fillingAndParsing) {
                throw new IllegalStateException();
            }
            if (this.demand != DemandState.CANCELLED) {
                this.demand = DemandState.NOT_DEMANDING;
            }
            boolean bl = true;
            return bl;
        }
    }

    public void cancelDemand() {
        try (AutoLock l = this.lock.lock();){
            if (LOG.isDebugEnabled()) {
                LOG.debug("cancelDemand d={} fp={} {} {}", new Object[]{this.demand, this.fillingAndParsing, this.networkBuffer, this});
            }
            this.demand = DemandState.CANCELLED;
        }
    }

    private void fillAndParse() {
        this.acquireNetworkBuffer();
        try {
            while (true) {
                Frame.Parsed frame;
                if (this.networkBuffer.hasRemaining() && (frame = this.parser.parse(this.networkBuffer.getByteBuffer())) != null) {
                    this.messagesIn.increment();
                    if (this.meetDemand()) {
                        this.onFrame(frame);
                    }
                    if (this.moreDemand()) continue;
                    return;
                }
                assert (!this.networkBuffer.hasRemaining());
                if (!this.getEndPoint().isOpen()) {
                    this.releaseNetworkBuffer();
                    return;
                }
                if (this.networkBuffer.isRetained()) {
                    this.reacquireNetworkBuffer();
                }
                int filled = this.getEndPoint().fill(this.networkBuffer.getByteBuffer());
                if (LOG.isDebugEnabled()) {
                    LOG.debug("endpointFill() filled={}: {}", (Object)filled, (Object)this.networkBuffer);
                }
                if (filled < 0) {
                    this.releaseNetworkBuffer();
                    this.coreSession.onEof();
                    return;
                }
                if (filled == 0) {
                    this.releaseNetworkBuffer();
                    this.fillInterested();
                    return;
                }
                this.bytesIn.add(filled);
            }
        }
        catch (Throwable t) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Error during fillAndParse() {}", (Object)t.toString());
            }
            if (this.networkBuffer != null) {
                BufferUtil.clear(this.networkBuffer.getByteBuffer());
                this.releaseNetworkBuffer();
            }
            this.coreSession.processConnectionError(t, Callback.NOOP);
            return;
        }
    }

    protected void setInitialBuffer(ByteBuffer initialBuffer) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Set initial buffer - {}", (Object)BufferUtil.toDetailString(initialBuffer));
        }
        try (AutoLock l = this.lock.lock();){
            this.networkBuffer = this.newNetworkBuffer(initialBuffer.remaining());
        }
        ByteBuffer buffer = this.networkBuffer.getByteBuffer();
        BufferUtil.clearToFill(buffer);
        BufferUtil.put(initialBuffer, buffer);
        BufferUtil.flipToFlush(buffer, 0);
    }

    @Override
    public void onOpen() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("onOpen() {}", (Object)this);
        }
        super.onOpen();
        this.coreSession.onOpen();
        if (this.moreDemand()) {
            this.fillAndParse();
        }
    }

    @Override
    public void setInputBufferSize(int inputBufferSize) {
        if (inputBufferSize < 28) {
            throw new IllegalArgumentException("Cannot have buffer size less than 28");
        }
        super.setInputBufferSize(inputBufferSize);
    }

    @Override
    public String dump() {
        return Dumpable.dump(this);
    }

    @Override
    public void dump(Appendable out, String indent) throws IOException {
        Dumpable.dumpObjects(out, indent, this, new Object[0]);
    }

    @Override
    public String toConnectionString() {
        return String.format("%s@%x[%s,p=%s,f=%s,g=%s]", new Object[]{this.getClass().getSimpleName(), this.hashCode(), this.coreSession.getBehavior(), this.parser, this.flusher, this.generator});
    }

    @Override
    public void onUpgradeTo(ByteBuffer buffer) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("onUpgradeTo({})", (Object)BufferUtil.toDetailString(buffer));
        }
        this.setInitialBuffer(buffer);
    }

    @Override
    public long getMessagesIn() {
        return this.messagesIn.longValue();
    }

    @Override
    public long getBytesIn() {
        return this.bytesIn.longValue();
    }

    @Override
    public long getMessagesOut() {
        return this.flusher.getMessagesOut();
    }

    @Override
    public long getBytesOut() {
        return this.flusher.getBytesOut();
    }

    public void enqueueFrame(Frame frame, Callback callback, boolean batch) {
        if (this.coreSession.getBehavior() == Behavior.CLIENT) {
            byte[] mask = new byte[4];
            this.random.nextBytes(mask);
            frame.setMask(mask);
        }
        if (this.flusher.enqueue(frame, callback, batch)) {
            this.flusher.iterate();
        }
    }

    private static enum DemandState {
        DEMANDING,
        NOT_DEMANDING,
        CANCELLED;

    }

    private class Flusher
    extends FrameFlusher {
        private Flusher(Scheduler scheduler, int bufferSize, Generator generator, EndPoint endpoint) {
            super(WebSocketConnection.this.byteBufferPool, scheduler, generator, endpoint, bufferSize, 8);
            this.setUseDirectByteBuffers(WebSocketConnection.this.isUseOutputDirectByteBuffers());
        }

        @Override
        public void onCompleteFailure(Throwable x) {
            WebSocketConnection.this.coreSession.processConnectionError(x, NOOP);
            super.onCompleteFailure(x);
        }
    }
}

