/*
 * Copyright 2012 Splunk, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"): you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package com.splunk;

/**
 * The {@code JobExportArgs} class contains arguments for exporting events using
 * the {@link Service#export} method.
 */
public class JobExportArgs extends Args {
    
    /**
     * Specifies the format for the returned output.
     */
    public static enum OutputMode {
        /** Returns output in Atom format. */
        ATOM("atom"),
        /** Returns output in CSV format. */
        CSV("csv"),
        /** Returns output in JSON format. */
        JSON("json"),
        /** Returns output in JSON_COLS format. */
        JSON_COLS("json_cols"),
        /** Returns output in JSON_ROWS format. */
        JSON_ROWS("json_rows"),
        /** Returns output in raw format. */
        RAW("raw"),
        /** Returns output in XML format. */
        XML("xml");
        
        private String value;
        
        private OutputMode(String value) {
            this.value = value;
        }
        
        /**
         * @return The REST API value for this enumerated constant.
         */
        public String toString() {
            return this.value;
        }
    }
    
    /**
     * Specifies how to create a job using the {@link JobCollection#create} 
     * method.
     */
    public static enum SearchMode {
        /**
         * Searches historical data.
         */
        NORMAL("normal"),
        /**
         * Searches live data. A real-time search may also be specified by 
         * setting the "earliest_time" and "latest_time" parameters to begin 
         * with "rt", even if the search_mode is set to "normal" or is not set. 
         * <p>
         * If both the "earliest_time" and "latest_time" parameters are set to 
         * "rt", the search represents all appropriate live data that was 
         * received since the start of the search.
         * <p>
         * If both the "earliest_time" and "latest_time" parameters are set to 
         * "rt" followed by a relative time specifier, a sliding window is used 
         * where the time bounds of the window are determined by the relative 
         * time specifiers and are continuously updated based on current time.
         */
        REALTIME("realtime");
        
        private String value;
        
        private SearchMode(String value) {
            this.value = value;
        }
        
        /**
         * @return The REST API value for this enumerated constant.
         */
        public String toString() {
            return this.value;
        }
    }
    
    /**
     * Specifies how to truncate lines to achieve the value in 
     * {@link #setMaximumLines}.
     */
    public static enum TruncationMode {
        /** Use the "abstract" truncation mode.*/
        ABSTRACT("abstract"),
        /** Use the "truncate" truncation mode.*/
        TRUNCATE("truncate");
        
        private String value;
        
        private TruncationMode(String value) {
            this.value = value;
        }
        
        /**
         * @return The REST API value for this enumerated constant.
         */
        public String toString() {
            return this.value;
        }
    }
    
    /**
     * Class constructor.
     */
    public JobExportArgs() { super(); }
    
    /* BEGIN AUTOGENERATED CODE */
    
    /**
     * Specifies the earliest time in the time range to search, based on the index time. The time string can be a UTC time (with fractional seconds), a relative time specifier (to now), or a formatted time string.
     * 
     * @param indexEarliest
     *      The earliest time. 
     */
    public void setIndexEarliest(String indexEarliest) {
        this.put("index_earliest", indexEarliest);
    }
    
    /**
     * Specifies the latest time in the time range to search, based on the index time. The time string can be a UTC time (with fractional seconds), a relative time specifier (to now), or a formatted time string.
     * 
     * @param indexLatest
     *      The latest time. 
     */
    public void setIndexLatest(String indexLatest) {
        this.put("index_latest", indexLatest);
    }
    
    /**
     * Sets the format of the output.
     * 
     * @param outputMode
     *      The output format.
     */
    public void setOutputMode(OutputMode outputMode) {
        this.put("output_mode", outputMode);
    }
    
    /**
     * Sets the number of seconds of inactivity after which to automatically cancel a job. A value of 0 means never auto-cancel.
     * 
     * @param autoCancel
     *      The number of seconds after which to cancel a job.
     */
    public void setAutoCancel(int autoCancel) {
        this.put("auto_cancel", autoCancel);
    }
    
    /**
     * Sets the number of events to process after which to auto-finalize the search. A value of 0 means no limit.
     * 
     * @param autoFinalizeEventCount
     *      The number of events.
     */
    public void setAutoFinalizeEventCount(int autoFinalizeEventCount) {
        this.put("auto_finalize_ec", autoFinalizeEventCount);
    }
    
    /**
     * Sets the number of seconds of inactivity after which to automatically pause a job. A value of 0 means never auto-pause.
     * 
     * @param autoPause
     *      The number of seconds after which to pause.
     */
    public void setAutoPause(int autoPause) {
        this.put("auto_pause", autoPause);
    }
    
    /**
     * Specifies the earliest time in the time range to search. The time string can be a UTC time (with fractional seconds), a relative time specifier (to now), or a formatted time string.
     * 
     * @param earliestTime
     *      The earliest time. 
     */
    public void setEarliestTime(String earliestTime) {
        this.put("earliest_time", earliestTime);
    }
    
    /**
     * Indicates whether to enable lookups for this search. Enabling lookups might slow searches significantly depending on the nature of the lookups.
     * 
     * @param enableLookups
     *      {@code true} to enable lookups, {@code false} if not.
     */
    public void setEnableLookups(boolean enableLookups) {
        this.put("enable_lookups", enableLookups);
    }
    
    /**
     * Specifies whether this search should cause (and wait depending on the value of {@code setSynchronizeBundleReplication}) bundle synchronization with all search peers.
     * 
     * @param forceBundleReplication
     *      {@code true} if this search should cause bundle synchronization, {@code false} if not.
     */
    public void setForceBundleReplication(boolean forceBundleReplication) {
        this.put("force_bundle_replication", forceBundleReplication);
    }
    
    /**
     * Specifies the latest time in the time range to search. The time string can be a UTC time (with fractional seconds), a relative time specifier (to now), or a formatted time string.
     * 
     * @param latestTime
     *      The latest time. 
     */
    public void setLatestTime(String latestTime) {
        this.put("latest_time", latestTime);
    }
    
    /**
     * Sets the number of seconds to run this search before finalizing. Specify 0 to never finalize.
     * 
     * @param maximumTime
     *      The maximum time, in seconds.
     */
    public void setMaximumTime(int maximumTime) {
        this.put("max_time", maximumTime);
    }
    
    /**
     * Specifies the application namespace in which to restrict searches. 
     * 
     * @param namespace
     *      The namespace.
     */
    public void setNamespace(String namespace) {
        this.put("namespace", namespace);
    }
    
    /**
     * Specifies a time string that sets the absolute time used for any relative time specifier in the search. This value defaults to the current system time.<p> You can specify a relative time modifier for this parameter. For example, specify +2d to specify the current time plus two days. If you specify a relative time modifier both in this parameter and in the search string, the search string modifier takes precedence.<p>For information about relative time modifiers, see <a href="http://docs.splunk.com/Documentation/Splunk/latest/SearchReference/SearchTimeModifiers" target="_blank">Time modifiers for search</a> in the Search Reference. 
     * 
     * @param now
     *      A time string.
     */
    public void setNow(String now) {
        this.put("now", now);
    }
    
    /**
     * Sets the time to wait between running the MapReduce phase on accumulated map values.
     * 
     * @param reduceFrequency
     *      The time to wait, in seconds.
     */
    public void setReduceFrequency(int reduceFrequency) {
        this.put("reduce_freq", reduceFrequency);
    }
    
    /**
     * Indicates whether to reload macro definitions from the macros.conf configuration file.
     * 
     * @param reloadMacros
     *      {@code true} to reload macro definitions, {@code false} if not.
     */
    public void setReloadMacros(boolean reloadMacros) {
        this.put("reload_macros", reloadMacros);
    }
    
    /**
     * Sets a list of (possibly wildcarded) servers from which to pull raw events. This same server list is used in subsearches.
     * 
     * @param remoteServerList
     *      A list of servers.
     */
    public void setRemoteServerList(String[] remoteServerList) {
        StringBuilder csv = new StringBuilder();
        for (int i = 0, n = remoteServerList.length; i < n; i++) {
            if (i != 0) {
                csv.append(",");
            }
            csv.append(remoteServerList[i]);
        }
        
        this.put("remote_server_list", String.valueOf(csv));
    }

    /**
     * Sets one or more fields to the search. These fields, even if not referenced or used directly by the search, are still included by the events and summary endpoints. Splunk Web uses these fields to prepopulate panels in the Search view.
     * 
     * @param fieldList
     *      The list of fields.
     */
    public void setFieldList(String[] fieldList) {
        this.put("f", fieldList);
    }
    
    /**
     * Sets one or more required fields to the search. These fields, even if not referenced or used directly by the search, are still included by the events and summary endpoints. Splunk Web uses these fields to prepopulate panels in the Search view.
     * 
     * @param requiredFieldList
     *      The list of required fields.
     */
    public void setRequiredFieldList(String[] requiredFieldList) {
        this.put("rf", requiredFieldList);
    }
    
    /**
     * Indicates whether the indexer blocks if the queue for this search is full. Only applies to real-time searches.
     * 
     * @param realtimeBlocking
     *      {@code true} to block the indexer for a full queue, {@code false} if not.
     */
    public void setRealtimeBlocking(boolean realtimeBlocking) {
        this.put("rt_blocking", realtimeBlocking);
    }
    
    /**
     * Indicates whether the indexer pre-filters events. Only applies to real-time searches.
     * 
     * @param realtimeIndexFilter
     *      {@code true} to pre-filter events, {@code false} if not.
     */
    public void setRealtimeIndexFilter(boolean realtimeIndexFilter) {
        this.put("rt_indexfilter", realtimeIndexFilter);
    }
    
    /**
     * Sets the number of seconds indicating the maximum time to block. A value of 0 means no limit. For real-time searches with "rt_blocking" set to {@code true}.
     * 
     * @param realtimeMaximumBlockSeconds
     *      The maximum time to block, in seconds.
     */
    public void setRealtimeMaximumBlockSeconds(int realtimeMaximumBlockSeconds) {
        this.put("rt_maxblocksecs", realtimeMaximumBlockSeconds);
    }
    
    /**
     * Sets the number indicating the queue size (in events) that the indexer should use for this search. Only applies to real-time searches.
     * 
     * @param realtimeQueueSize
     *      The queue size, in events.
     */
    public void setRealtimeQueueSize(int realtimeQueueSize) {
        this.put("rt_queue_size", realtimeQueueSize);
    }
    
    /**
     * Sets a string that registers a search state listener with the search. Use the format: {@code search_state;results_condition;http_method;uri;}<p>For example:<p>{@code search_listener=onResults;true;POST;/servicesNS/admin/search/saved/search/foobar/notify;}
     * 
     * @param searchListener
     *      The search listener string.
     */
    public void setSearchListener(String searchListener) {
        this.put("search_listener", searchListener);
    }
    
    /**
     * Sets the search mode ("normal" or "realtime").
     * 
     * @param searchMode
     *      The search mode.
     */
    public void setSearchMode(SearchMode searchMode) {
        this.put("search_mode", searchMode);
    }
    
    /**
     * Indicates whether this search should wait for bundle replication to complete.
     * 
     * @param synchronizeBundleReplication
     *      {@code true} to wait for bundle replication, {@code false} if not.
     */
    public void setSynchronizeBundleReplication(boolean synchronizeBundleReplication) {
        this.put("sync_bundle_replication", synchronizeBundleReplication);
    }
    
    /**
     * Sets the format for converting a formatted time string from {start,end}_time into UTC seconds. The default value is ISO-8601.
     * 
     * @param timeFormat
     *      The time format string.
     */
    public void setTimeFormat(String timeFormat) {
        this.put("time_format", timeFormat);
    }
    
    /**
     * Sets the number of seconds to keep this search after processing has stopped.
     * 
     * @param timeout
     *      The timeout, in seconds.
     */
    public void setTimeout(int timeout) {
        this.put("timeout", timeout);
    }
    
    /**
     * Sets the maximum number of lines that any single event's <b>_raw</b> field should contain.
     * 
     * @param maximumLines
     *      The maximum number of lines. For no limit, specify 0.
     */
    public void setMaximumLines(int maximumLines) {
        this.put("max_lines", maximumLines);
    }
    
    /**
     * Sets a UTC time format.
     * 
     * @param outputTimeFormat
     *      A UTC time format.
     */
    public void setOutputTimeFormat(String outputTimeFormat) {
        this.put("output_time_format", outputTimeFormat);
    }
    
    /**
     * Sets the type of segmentation to perform on the data, including an option to perform k/v segmentation.
     * 
     * @param segmentation
     *      The segmentation type.
     */
    public void setSegmentation(String segmentation) {
        this.put("segmentation", segmentation);
    }
    
    /**
     * Specifies how to truncate lines to achieve the value in {@link #setMaximumLines}.
     * 
     * @param truncationMode
     *      The truncation mode.
     */
    public void setTruncationMode(TruncationMode truncationMode) {
        this.put("truncation_mode", truncationMode);
    }
    
    /* END AUTOGENERATED CODE */
}
