/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.commons.resourcehandler;


/**
 * Utility methods needed in PackagedResource and PackagedResourceHandler. 
 *
 * @author Jakob Korherr
 */
public class ResourceUtils
{
    /**
     * Checks if the user agent supports GZIP compression on basis of the "Accept-Encoding" header field. 
     * Created according to RFC2616, section 14.3 Accept-Encoding.
     *
     * Some examples of Accept-Encoding:
     *
     *     Accept-Encoding: gzip, deflate
     *     Accept-Encoding:
     *     Accept-Encoding: *
     *     Accept-Encoding: compress;q=0.5, gzip;q=1.0
     *     Accept-Encoding: gzip;q=1.0, identity; q=0.5, *;q=0 
     *
     * @param acceptEncodingHeader
     * @return
     */
    public static boolean isGZIPEncodingAccepted(String acceptEncodingHeader)
    {
        if (acceptEncodingHeader == null)
        {
            return false;
        }
        int gzipIndex = acceptEncodingHeader.indexOf("gzip");
        if (gzipIndex != -1)
        {
            // "gzip" appears in the header
            // --> check if banned via q=0
            return !isEncodingQValueZero(acceptEncodingHeader, gzipIndex);
        }

        // no "gzip" in header --> check for "*"
        int starIndex = acceptEncodingHeader.indexOf('*');
        if (starIndex != -1)
        {
            // "*" appears in the header
            // --> check if banned via q=0
            return !isEncodingQValueZero(acceptEncodingHeader, starIndex);
        }

        // neither "gzip" nor "*" in header
        return false;
    }

    private static boolean isEncodingQValueZero(String acceptEncodingHeader, int startIndex)
    {
        // remove any precending definitions
        String encodingSubstring = acceptEncodingHeader.substring(startIndex);

        // remove any subsequent definitions (separated via ,)
        int commaIndex = encodingSubstring.indexOf(',');
        if (commaIndex != -1)
        {
            encodingSubstring = encodingSubstring.substring(0, commaIndex);
        }

        int qZeroIndex = encodingSubstring.indexOf("q=0");
        if (qZeroIndex != -1)
        {
            String qZeroSubstring = encodingSubstring.substring(qZeroIndex).trim();
            if (qZeroSubstring.matches("q=0(\\.(0){0,3})?"))
            {
                // "q=0" or "q=0." or "q=0.0" or "q=0.00" or "q=0.000" found
                // NOTE that there MUST NOT be more than 3 digits after the decimal point (per RFC section 3.9)
                return true;
            }
            else
            {
                // q=0.xyz" found with any of xyz being != 0
                return false;
            }
        }
        else
        {
            // "q=0" not found
            return false;
        }
    }
    
}
