#!BPY

"""
Name: 'Center to verts'
Blender: 248
Group: 'Mesh'
Tooltip: 'Move object center to vertex selection'
"""

__author__ = "Marcus Jenkins (BlenderArtists - FunkyWyrm)"
__url__ = ("blenderartists.org")
__version__ = "1.1 20/03/2009"

__bpydoc__ = """\
This script enables the movement of an object center to the average vertex position
of selected vertices.

Usage:

Select vertices in Edit mode.

Run the script, it will move the object center to the average position of the selected
vertices.

eg.

To set the center of a cube object to the center of the bottom face, select the bottom
face and run the script.

To set the center of a cone object to the tip of the cone, select the vertex at the tip
of the cone and run the script.

To set the center of an object to the center of the mesh data, select all the vertices
and run the script. (This functions the same as the "Center new" button in the "Mesh"
panel in the "Editing" buttons context)

Version notes:

v1.1:
Removed surplus imported modules
Removed print statements used for debugging
Made the script use <scene>.objects.active rather than <scene>.objects.selected
"""

# ***** BEGIN GPL LICENSE BLOCK *****
#
# Script copyright (C) Marcus Jenkins
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
# ***** END GPL LICENCE BLOCK *****
# --------------------------------------------------------------------------

import Blender
from Blender import Mathutils, Scene, Draw, Window
from Blender.Mathutils import Vector

def FindSelectedVertAverage(obj):
	vertAvg = Vector([0,0,0])
	selectedVerts = 0
	for vert in obj.getData(mesh=1).verts:
		if vert.sel:
			vertAvg = (vertAvg + vert.co)
			selectedVerts += 1
	if selectedVerts:
		vertAvg = vertAvg/selectedVerts
		return vertAvg
	else:
		Draw.PupMenu("No vertices are selected")
		return 0

def MoveCenter(obj):
	objLoc = Vector(obj.getLocation())
	vertLoc = FindSelectedVertAverage(obj)
	if vertLoc:
		obj.setLocation(objLoc + vertLoc)
		for vert in obj.getData(mesh=1).verts:
			vert.co = vert.co-vertLoc
			
#-----------------------------------------------------
def main():	
	#Vertex selection status can only be read in Object mode
	
	isEditMode = Window.EditMode()	#Find window status
	
	if isEditMode:					#Exit edit mode if necessary
		Window.EditMode(0)

	obj = Scene.GetCurrent().objects.active
	if obj.type == "Mesh":
		MoveCenter(obj)
	else:
                retval = Draw.PupMenu("Object is not a mesh")
	
	Window.EditMode(isEditMode)		#Restore initial window state
#-----------------------------------------------------

if __name__ == "__main__":
	main()
