/*
   Copyright The containerd Authors.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package image

import (
	"encoding/json"
	"errors"
	"fmt"
	"runtime"
	"strings"
	"testing"

	"gotest.tools/v3/assert"

	"github.com/containerd/nerdctl/mod/tigron/require"
	"github.com/containerd/nerdctl/mod/tigron/test"
	"github.com/containerd/nerdctl/mod/tigron/tig"

	"github.com/containerd/nerdctl/v2/pkg/inspecttypes/dockercompat"
	"github.com/containerd/nerdctl/v2/pkg/testutil"
	"github.com/containerd/nerdctl/v2/pkg/testutil/nerdtest"
)

func TestImageInspectSimpleCases(t *testing.T) {
	nerdtest.Setup()

	testCase := &test.Case{
		Setup: func(data test.Data, helpers test.Helpers) {
			helpers.Ensure("pull", "--quiet", testutil.CommonImage)
		},
		SubTests: []*test.Case{
			{
				Description: "Contains some stuff",
				Command:     test.Command("image", "inspect", testutil.CommonImage),
				Expected: test.Expects(0, nil, func(stdout string, t tig.T) {
					var dc []dockercompat.Image
					err := json.Unmarshal([]byte(stdout), &dc)
					assert.NilError(t, err, "Unable to unmarshal output\n")
					assert.Equal(t, 1, len(dc), "Unexpectedly got multiple results\n")
					assert.Assert(t, len(dc[0].RootFS.Layers) > 0, "there should be at least one rootfs layer\n")
					assert.Assert(t, dc[0].Architecture != "", "architecture should be set\n")
					assert.Assert(t, dc[0].Size > 0, "size should be > 0 \n")
				}),
			},
			{
				Description: "RawFormat support (.Id)",
				Command:     test.Command("image", "inspect", testutil.CommonImage, "--format", "{{.Id}}"),
				Expected:    test.Expects(0, nil, nil),
			},
			{
				Description: "typedFormat support (.ID)",
				Command:     test.Command("image", "inspect", testutil.CommonImage, "--format", "{{.ID}}"),
				Expected:    test.Expects(0, nil, nil),
			},
			{
				Description: "Error for image not found",
				Command:     test.Command("image", "inspect", "dne:latest", "dne2:latest"),
				Expected: test.Expects(1, []error{
					errors.New("no such image: dne:latest"),
					errors.New("no such image: dne2:latest"),
				}, nil),
			},
		},
	}

	if runtime.GOOS == "windows" {
		testCase.Require = nerdtest.IsFlaky("https://github.com/containerd/nerdctl/issues/3524")
	}

	testCase.Run(t)
}

func TestImageInspectDifferentValidReferencesForTheSameImage(t *testing.T) {
	nerdtest.Setup()

	tags := []string{
		"",
		":latest",
	}
	names := []string{
		"busybox",
		"docker.io/library/busybox",
		"registry-1.docker.io/library/busybox",
	}

	testCase := &test.Case{
		Require: require.All(
			require.Not(nerdtest.Docker),
			// FIXME: this test depends on hub images that do not have windows versions
			require.Not(require.Windows),
			// We need a clean slate
			nerdtest.Private,
		),
		Setup: func(data test.Data, helpers test.Helpers) {
			helpers.Ensure("pull", "--quiet", "alpine")
			helpers.Ensure("pull", "--quiet", "busybox")
			helpers.Ensure("pull", "--quiet", "registry-1.docker.io/library/busybox")
		},
		SubTests: []*test.Case{
			{
				Description: "name and tags +/- sha combinations",
				Command:     test.Command("image", "inspect", "busybox"),
				Expected: func(data test.Data, helpers test.Helpers) *test.Expected {
					return &test.Expected{
						Output: func(stdout string, t tig.T) {
							var dc []dockercompat.Image
							err := json.Unmarshal([]byte(stdout), &dc)
							assert.NilError(t, err, "Unable to unmarshal output\n")
							assert.Equal(t, 1, len(dc), "Unexpectedly got multiple results\n")
							reference := dc[0].ID
							sha := strings.TrimPrefix(dc[0].RepoDigests[0], "busybox@sha256:")

							for _, name := range names {
								for _, tag := range tags {
									it := nerdtest.InspectImage(helpers, name+tag)
									assert.Equal(t, it.ID, reference)
									it = nerdtest.InspectImage(helpers, name+tag+"@sha256:"+sha)
									assert.Equal(t, it.ID, reference)
								}
							}
						},
					}
				},
			},
			{
				Description: "by digest, short or long, with or without prefix",
				Command:     test.Command("image", "inspect", "busybox"),
				Expected: func(data test.Data, helpers test.Helpers) *test.Expected {
					return &test.Expected{
						Output: func(stdout string, t tig.T) {
							var dc []dockercompat.Image
							err := json.Unmarshal([]byte(stdout), &dc)
							assert.NilError(t, err, "Unable to unmarshal output\n")
							assert.Equal(t, 1, len(dc), "Unexpectedly got multiple results\n")
							reference := dc[0].ID
							sha := strings.TrimPrefix(dc[0].RepoDigests[0], "busybox@sha256:")

							for _, id := range []string{"sha256:" + sha, sha, sha[0:8], "sha256:" + sha[0:8]} {
								it := nerdtest.InspectImage(helpers, id)
								assert.Equal(t, it.ID, reference)
							}

							// Now, tag alpine with a short id
							// Build reference values for comparison
							alpine := nerdtest.InspectImage(helpers, "alpine")

							// Demonstrate image name precedence over digest lookup
							// Using the shortened sha should no longer get busybox, but rather the newly tagged Alpine
							// FIXME: this is triggering https://github.com/containerd/nerdctl/issues/3016
							// We cannot get rid of that image now, which does break local testing
							helpers.Ensure("tag", "alpine", sha[0:8])
							it := nerdtest.InspectImage(helpers, sha[0:8])
							assert.Equal(t, it.ID, alpine.ID)
						},
					}
				},
			},
			{
				Description: "prove that wrong references with correct digest do not get resolved",
				Command:     test.Command("image", "inspect", "busybox"),
				Expected: func(data test.Data, helpers test.Helpers) *test.Expected {
					return &test.Expected{
						Output: func(stdout string, t tig.T) {
							var dc []dockercompat.Image
							err := json.Unmarshal([]byte(stdout), &dc)
							assert.NilError(t, err, "Unable to unmarshal output\n")
							assert.Equal(t, 1, len(dc), "Unexpectedly got multiple results\n")
							sha := strings.TrimPrefix(dc[0].RepoDigests[0], "busybox@sha256:")

							for _, id := range []string{"doesnotexist", "doesnotexist:either", "busybox:bogustag"} {
								cmd := helpers.Command("image", "inspect", id+"@sha256:"+sha)
								cmd.Run(&test.Expected{
									ExitCode: 1,
									Errors:   []error{fmt.Errorf("no such image: %s@sha256:%s", id, sha)},
								})
							}
						},
					}
				},
			},
			{
				Description: "prove that invalid reference return no result without crashing",
				Command:     test.Command("image", "inspect", "busybox"),
				Expected: func(data test.Data, helpers test.Helpers) *test.Expected {
					return &test.Expected{
						Output: func(stdout string, t tig.T) {
							var dc []dockercompat.Image
							err := json.Unmarshal([]byte(stdout), &dc)
							assert.NilError(t, err, "Unable to unmarshal output\n")
							assert.Equal(t, 1, len(dc), "Unexpectedly got multiple results\n")

							for _, id := range []string{"∞∞∞∞∞∞∞∞∞∞", "busybox:∞∞∞∞∞∞∞∞∞∞"} {
								cmd := helpers.Command("image", "inspect", id)
								cmd.Run(&test.Expected{
									ExitCode: 1,
									Errors:   []error{fmt.Errorf("invalid reference format: %s", id)},
								})
							}
						},
					}
				},
			},
			{
				Description: "retrieving multiple entries at once",
				Command:     test.Command("image", "inspect", "busybox", "busybox"),
				Expected: func(data test.Data, helpers test.Helpers) *test.Expected {
					return &test.Expected{
						Output: func(stdout string, t tig.T) {
							var dc []dockercompat.Image
							err := json.Unmarshal([]byte(stdout), &dc)
							assert.NilError(t, err, "Unable to unmarshal output\n")
							assert.Equal(t, 2, len(dc), "Unexpectedly did not get 2 results\n")
							reference := nerdtest.InspectImage(helpers, "busybox")
							assert.Equal(t, dc[0].ID, reference.ID)
							assert.Equal(t, dc[1].ID, reference.ID)
						},
					}
				},
			},
		},
	}

	testCase.Run(t)
}
