/****************************************************************************
**
** Copyright (C) 2016 Andre Hartmann.
** Contact: aha_1980@gmx.de
**
** This file is part of Qt Creator.
**
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3 as published by the Free Software
** Foundation with exceptions as appearing in the file LICENSE.GPL3-EXCEPT
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-3.0.html.
**
****************************************************************************/

#include "customparser.h"
#include "task.h"
#include "projectexplorerconstants.h"
#include "buildmanager.h"

#include <utils/qtcassert.h>

#include <QString>

using namespace Utils;
using namespace ProjectExplorer;

bool CustomParserExpression::operator ==(const CustomParserExpression &other) const
{
    return pattern() == other.pattern() && fileNameCap() == other.fileNameCap()
            && lineNumberCap() == other.lineNumberCap() && messageCap() == other.messageCap()
            && channel() == other.channel() && example() == other.example();
}

QString CustomParserExpression::pattern() const
{
    return m_regExp.pattern();
}

void ProjectExplorer::CustomParserExpression::setPattern(const QString &pattern)
{
    m_regExp.setPattern(pattern);
    QTC_CHECK(m_regExp.isValid());
}

CustomParserExpression::CustomParserChannel CustomParserExpression::channel() const
{
    return m_channel;
}

void CustomParserExpression::setChannel(CustomParserExpression::CustomParserChannel channel)
{
    QTC_ASSERT(channel > ParseNoChannel && channel <= ParseBothChannels,
               channel = ParseBothChannels);

    m_channel = channel;
}

QString CustomParserExpression::example() const
{
    return m_example;
}

void CustomParserExpression::setExample(const QString &example)
{
    m_example = example;
}

int CustomParserExpression::messageCap() const
{
    return m_messageCap;
}

void CustomParserExpression::setMessageCap(int messageCap)
{
    m_messageCap = messageCap;
}

int CustomParserExpression::lineNumberCap() const
{
    return m_lineNumberCap;
}

void CustomParserExpression::setLineNumberCap(int lineNumberCap)
{
    m_lineNumberCap = lineNumberCap;
}

int CustomParserExpression::fileNameCap() const
{
    return m_fileNameCap;
}

void CustomParserExpression::setFileNameCap(int fileNameCap)
{
    m_fileNameCap = fileNameCap;
}

bool CustomParserSettings::operator ==(const CustomParserSettings &other) const
{
    return error == other.error && warning == other.warning;
}

CustomParser::CustomParser(const CustomParserSettings &settings)
{
    setObjectName("CustomParser");

    setSettings(settings);
}

void CustomParser::stdError(const QString &line)
{
    if (parseLine(line, CustomParserExpression::ParseStdErrChannel))
        return;

    IOutputParser::stdError(line);
}

void CustomParser::stdOutput(const QString &line)
{
    if (parseLine(line, CustomParserExpression::ParseStdOutChannel))
        return;

    IOutputParser::stdOutput(line);
}

void CustomParser::setWorkingDirectory(const QString &workingDirectory)
{
    m_workingDirectory = workingDirectory;
}

void CustomParser::setSettings(const CustomParserSettings &settings)
{
    m_error = settings.error;
    m_warning = settings.warning;
}

Core::Id CustomParser::id()
{
    return Core::Id("ProjectExplorer.OutputParser.Custom");
}

FilePath CustomParser::absoluteFilePath(const QString &filePath) const
{
    if (m_workingDirectory.isEmpty())
        return FilePath::fromUserInput(filePath);

    return FilePath::fromString(m_workingDirectory).resolvePath(filePath);
}

bool CustomParser::hasMatch(const QString &line, CustomParserExpression::CustomParserChannel channel,
                            const CustomParserExpression &expression, Task::TaskType taskType)
{
    if (!(channel & expression.channel()))
        return false;

    if (expression.pattern().isEmpty())
        return false;

    const QRegularExpressionMatch match = expression.match(line);
    if (!match.hasMatch())
        return false;

    const FilePath fileName = absoluteFilePath(match.captured(expression.fileNameCap()));
    const int lineNumber = match.captured(expression.lineNumberCap()).toInt();
    const QString message = match.captured(expression.messageCap());

    emit addTask(CompileTask(taskType, message, fileName, lineNumber), 1);
    return true;
}

bool CustomParser::parseLine(const QString &rawLine, CustomParserExpression::CustomParserChannel channel)
{
    const QString line = rawLine.trimmed();

    if (hasMatch(line, channel, m_error, Task::Error))
        return true;

    return hasMatch(line, channel, m_warning, Task::Warning);
}

// Unit tests:

#ifdef WITH_TESTS
#   include <QTest>

#   include "projectexplorer.h"
#   include "outputparser_test.h"

void ProjectExplorerPlugin::testCustomOutputParsers_data()
{
    QTest::addColumn<QString>("input");
    QTest::addColumn<QString>("workDir");
    QTest::addColumn<OutputParserTester::Channel>("inputChannel");
    QTest::addColumn<CustomParserExpression::CustomParserChannel>("filterErrorChannel");
    QTest::addColumn<CustomParserExpression::CustomParserChannel>("filterWarningChannel");
    QTest::addColumn<QString>("errorPattern");
    QTest::addColumn<int>("errorFileNameCap");
    QTest::addColumn<int>("errorLineNumberCap");
    QTest::addColumn<int>("errorMessageCap");
    QTest::addColumn<QString>("warningPattern");
    QTest::addColumn<int>("warningFileNameCap");
    QTest::addColumn<int>("warningLineNumberCap");
    QTest::addColumn<int>("warningMessageCap");
    QTest::addColumn<QString>("childStdOutLines");
    QTest::addColumn<QString>("childStdErrLines");
    QTest::addColumn<Tasks >("tasks");
    QTest::addColumn<QString>("outputLines");

    const QString simplePattern = "^([a-z]+\\.[a-z]+):(\\d+): error: ([^\\s].+)$";
    const FilePath fileName = FilePath::fromUserInput("main.c");

    QTest::newRow("empty patterns")
            << QString::fromLatin1("Sometext")
            << QString()
            << OutputParserTester::STDOUT
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseBothChannels
            << QString() << 1 << 2 << 3
            << QString() << 1 << 2 << 3
            << QString::fromLatin1("Sometext\n") << QString()
            << Tasks()
            << QString();

    QTest::newRow("pass-through stdout")
            << QString::fromLatin1("Sometext")
            << QString()
            << OutputParserTester::STDOUT
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseBothChannels
            << simplePattern << 1 << 2 << 3
            << QString() << 1 << 2 << 3
            << QString::fromLatin1("Sometext\n") << QString()
            << Tasks()
            << QString();

    QTest::newRow("pass-through stderr")
            << QString::fromLatin1("Sometext")
            << QString()
            << OutputParserTester::STDERR
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseBothChannels
            << simplePattern << 1 << 2 << 3
            << QString() << 1 << 2 << 3
            << QString() << QString::fromLatin1("Sometext\n")
            << Tasks()
            << QString();

    const QString simpleError = "main.c:9: error: `sfasdf' undeclared (first use this function)";
    const QString simpleErrorPassThrough = simpleError + '\n';
    const QString message = "`sfasdf' undeclared (first use this function)";

    QTest::newRow("simple error")
            << simpleError
            << QString()
            << OutputParserTester::STDERR
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseBothChannels
            << simplePattern << 1 << 2 << 3
            << QString() << 0 << 0 << 0
            << QString() << QString()
            << Tasks({CompileTask(Task::Error, message, fileName, 9)})
            << QString();

    const QString pathPattern = "^([a-z\\./]+):(\\d+): error: ([^\\s].+)$";
    QString workingDir = "/home/src/project";
    FilePath expandedFileName = FilePath::fromString("/home/src/project/main.c");

    QTest::newRow("simple error with expanded path")
            << "main.c:9: error: `sfasdf' undeclared (first use this function)"
            << workingDir
            << OutputParserTester::STDERR
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseBothChannels
            << pathPattern << 1 << 2 << 3
            << QString() << 0 << 0 << 0
            << QString() << QString()
            << Tasks({CompileTask(Task::Error, message, expandedFileName, 9)})
            << QString();

    expandedFileName = FilePath::fromString("/home/src/project/subdir/main.c");
    QTest::newRow("simple error with subdir path")
            << "subdir/main.c:9: error: `sfasdf' undeclared (first use this function)"
            << workingDir
            << OutputParserTester::STDERR
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseBothChannels
            << pathPattern << 1 << 2 << 3
            << QString() << 0 << 0 << 0
            << QString() << QString()
            << Tasks({CompileTask(Task::Error, message, expandedFileName, 9)})
            << QString();

    workingDir = "/home/src/build-project";
    QTest::newRow("simple error with buildir path")
            << "../project/subdir/main.c:9: error: `sfasdf' undeclared (first use this function)"
            << workingDir
            << OutputParserTester::STDERR
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseBothChannels
            << pathPattern << 1 << 2 << 3
            << QString() << 0 << 0 << 0
            << QString() << QString()
            << Tasks({CompileTask(Task::Error, message, expandedFileName, 9)})
            << QString();

    QTest::newRow("simple error on wrong channel")
            << simpleError
            << QString()
            << OutputParserTester::STDOUT
            << CustomParserExpression::ParseStdErrChannel << CustomParserExpression::ParseBothChannels
            << simplePattern << 1 << 2 << 3
            << QString() << 0 << 0 << 0
            << simpleErrorPassThrough << QString()
            << Tasks()
            << QString();

    QTest::newRow("simple error on other wrong channel")
            << simpleError
            << QString()
            << OutputParserTester::STDERR
            << CustomParserExpression::ParseStdOutChannel << CustomParserExpression::ParseBothChannels
            << simplePattern << 1 << 2 << 3
            << QString() << 0 << 0 << 0
            << QString() << simpleErrorPassThrough
            << Tasks()
            << QString();

    const QString simpleError2 = "Error: Line 19 in main.c: `sfasdf' undeclared (first use this function)";
    const QString simplePattern2 = "^Error: Line (\\d+) in ([a-z]+\\.[a-z]+): ([^\\s].+)$";
    const int lineNumber2 = 19;

    QTest::newRow("another simple error on stderr")
            << simpleError2
            << QString()
            << OutputParserTester::STDERR
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseBothChannels
            << simplePattern2 << 2 << 1 << 3
            << QString() << 1 << 2 << 3
            << QString() << QString()
            << Tasks({CompileTask(Task::Error, message, fileName, lineNumber2)})
            << QString();

    QTest::newRow("another simple error on stdout")
            << simpleError2
            << QString()
            << OutputParserTester::STDOUT
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseBothChannels
            << simplePattern2 << 2 << 1 << 3
            << QString() << 1 << 2 << 3
            << QString() << QString()
            << Tasks({CompileTask(Task::Error, message, fileName, lineNumber2)})
            << QString();

    const QString simpleWarningPattern = "^([a-z]+\\.[a-z]+):(\\d+): warning: ([^\\s].+)$";
    const QString simpleWarning = "main.c:1234: warning: `helloWorld' declared but not used";
    const QString warningMessage = "`helloWorld' declared but not used";

    QTest::newRow("simple warning")
            << simpleWarning
            << QString()
            << OutputParserTester::STDERR
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseBothChannels
            << QString() << 1 << 2 << 3
            << simpleWarningPattern << 1 << 2 << 3
            << QString() << QString()
            << Tasks({CompileTask(Task::Warning, warningMessage, fileName, 1234)})
            << QString();

    const QString simpleWarning2 = "Warning: `helloWorld' declared but not used (main.c:19)";
    const QString simpleWarningPassThrough2 = simpleWarning2 + '\n';
    const QString simpleWarningPattern2 = "^Warning: (.*) \\(([a-z]+\\.[a-z]+):(\\d+)\\)$";

    QTest::newRow("another simple warning on stdout")
            << simpleWarning2
            << QString()
            << OutputParserTester::STDOUT
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseStdOutChannel
            << simplePattern2 << 1 << 2 << 3
            << simpleWarningPattern2 << 2 << 3 << 1
            << QString() << QString()
            << Tasks({CompileTask(Task::Warning, warningMessage, fileName, lineNumber2)})
            << QString();

    QTest::newRow("warning on wrong channel")
            << simpleWarning2
            << QString()
            << OutputParserTester::STDOUT
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseStdErrChannel
            << QString() << 1 << 2 << 3
            << simpleWarningPattern2 << 2 << 3 << 1
            << simpleWarningPassThrough2 << QString()
            << Tasks()
            << QString();

    QTest::newRow("warning on other wrong channel")
            << simpleWarning2
            << QString()
            << OutputParserTester::STDERR
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseStdOutChannel
            << QString() << 1 << 2 << 3
            << simpleWarningPattern2 << 2 << 3 << 1
            << QString() << simpleWarningPassThrough2
            << Tasks()
            << QString();

    QTest::newRow("error and *warning*")
            << simpleWarning
            << QString()
            << OutputParserTester::STDERR
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseBothChannels
            << simplePattern << 1 << 2 << 3
            << simpleWarningPattern << 1 << 2 << 3
            << QString() << QString()
            << Tasks({CompileTask(Task::Warning, warningMessage, fileName, 1234)})
            << QString();

    QTest::newRow("*error* when equal pattern")
            << simpleError
            << QString()
            << OutputParserTester::STDERR
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseBothChannels
            << simplePattern << 1 << 2 << 3
            << simplePattern << 1 << 2 << 3
            << QString() << QString()
            << Tasks({CompileTask(Task::Error, message, fileName, 9)})
            << QString();

    const QString unitTestError = "../LedDriver/LedDriverTest.c:63: FAIL: Expected 0x0080 Was 0xffff";
    const FilePath unitTestFileName = FilePath::fromUserInput("../LedDriver/LedDriverTest.c");
    const QString unitTestMessage = "Expected 0x0080 Was 0xffff";
    const QString unitTestPattern = "^([^:]+):(\\d+): FAIL: ([^\\s].+)$";
    const int unitTestLineNumber = 63;

    QTest::newRow("unit test error")
            << unitTestError
            << QString()
            << OutputParserTester::STDOUT
            << CustomParserExpression::ParseBothChannels << CustomParserExpression::ParseBothChannels
            << unitTestPattern << 1 << 2 << 3
            << QString() << 1 << 2 << 3
            << QString() << QString()
            << Tasks({CompileTask(Task::Error, unitTestMessage, unitTestFileName, unitTestLineNumber)})
            << QString();
}

void ProjectExplorerPlugin::testCustomOutputParsers()
{
    QFETCH(QString, input);
    QFETCH(QString, workDir);
    QFETCH(OutputParserTester::Channel, inputChannel);
    QFETCH(CustomParserExpression::CustomParserChannel, filterErrorChannel);
    QFETCH(CustomParserExpression::CustomParserChannel, filterWarningChannel);
    QFETCH(QString, errorPattern);
    QFETCH(int,     errorFileNameCap);
    QFETCH(int,     errorLineNumberCap);
    QFETCH(int,     errorMessageCap);
    QFETCH(QString, warningPattern);
    QFETCH(int,     warningFileNameCap);
    QFETCH(int,     warningLineNumberCap);
    QFETCH(int,     warningMessageCap);
    QFETCH(QString, childStdOutLines);
    QFETCH(QString, childStdErrLines);
    QFETCH(Tasks, tasks);
    QFETCH(QString, outputLines);

    CustomParserSettings settings;
    settings.error.setPattern(errorPattern);
    settings.error.setFileNameCap(errorFileNameCap);
    settings.error.setLineNumberCap(errorLineNumberCap);
    settings.error.setMessageCap(errorMessageCap);
    settings.error.setChannel(filterErrorChannel);
    settings.warning.setPattern(warningPattern);
    settings.warning.setFileNameCap(warningFileNameCap);
    settings.warning.setLineNumberCap(warningLineNumberCap);
    settings.warning.setMessageCap(warningMessageCap);
    settings.warning.setChannel(filterWarningChannel);

    CustomParser *parser = new CustomParser;
    parser->setSettings(settings);
    parser->setWorkingDirectory(workDir);

    OutputParserTester testbench;
    testbench.appendOutputParser(parser);
    testbench.testParsing(input, inputChannel,
                          tasks, childStdOutLines, childStdErrLines,
                          outputLines);
}
#endif
