package city

import "testing"

const (
	seed0    uint64 = 1234567
	seed1    uint64 = k0
	dataSize int    = 1 << 20
	testSize int    = 300
)

var (
	seed128 = U128{seed0, seed1}
	data    [dataSize]byte
)

// Initialize data to pseudorandom value.
func setup() {
	a := uint64(9)
	b := uint64(777)
	for i := 0; i < dataSize; i++ {
		a += b
		b += a
		a = (a ^ (a >> 41)) * k0
		b = (b^(b>>41))*k0 + uint64(i)
		u := byte(b >> 37)
		data[i] = u
	}
}

var testdata = [testSize][]uint64{
	{
		0x9ae16a3b2f90404f, 0x75106db890237a4a, 0x3feac5f636039766, 0x3df09dfc64c09a2b,
		0x3cb540c392e51e29, 0x6b56343feac0663, 0x5b7bc50fd8e8ad92, 0x3df09dfc64c09a2b,
		0x3cb540c392e51e29, 0x6b56343feac0663, 0x5b7bc50fd8e8ad92, 0x95162f24e6a5f930,
		0x6808bdf4f1eb06e0, 0xb3b1f3a67b624d82, 0xc9a62f12bd4cd80b, 0xdc56d17a,
	},
	{
		0x541150e87f415e96, 0x1aef0d24b3148a1a, 0xbacc300e1e82345a, 0xc3cdc41e1df33513,
		0x2c138ff2596d42f6, 0xf58e9082aed3055f, 0x162e192b2957163d, 0xc3cdc41e1df33513,
		0x2c138ff2596d42f6, 0xf58e9082aed3055f, 0x162e192b2957163d, 0xfb99e85e0d16f90c,
		0x608462c15bdf27e8, 0xe7d2c5c943572b62, 0x1baaa9327642798c, 0x99929334,
	},
	{
		0xf3786a4b25827c1, 0x34ee1a2bf767bd1c, 0x2f15ca2ebfb631f2, 0x3149ba1dac77270d,
		0x70e2e076e30703c, 0x59bcc9659bc5296, 0x9ecbc8132ae2f1d7, 0x3149ba1dac77270d,
		0x70e2e076e30703c, 0x59bcc9659bc5296, 0x9ecbc8132ae2f1d7, 0xa01d30789bad7cf2,
		0xae03fe371981a0e0, 0x127e3883b8788934, 0xd0ac3d4c0a6fca32, 0x4252edb7,
	},
	{
		0xef923a7a1af78eab, 0x79163b1e1e9a9b18, 0xdf3b2aca6e1e4a30, 0x2193fb7620cbf23b,
		0x8b6a8ff06cda8302, 0x1a44469afd3e091f, 0x8b0449376612506, 0x2193fb7620cbf23b,
		0x8b6a8ff06cda8302, 0x1a44469afd3e091f, 0x8b0449376612506, 0xe9d9d41c32ad91d1,
		0xb44ab09f58e3c608, 0x19e9175f9fcf784, 0x839b3c9581b4a480, 0xebc34f3c,
	},
	{
		0x11df592596f41d88, 0x843ec0bce9042f9c, 0xcce2ea1e08b1eb30, 0x4d09e42f09cc3495,
		0x666236631b9f253b, 0xd28b3763cd02b6a3, 0x43b249e57c4d0c1b, 0x4d09e42f09cc3495,
		0x666236631b9f253b, 0xd28b3763cd02b6a3, 0x43b249e57c4d0c1b, 0x3887101c8adea101,
		0x8a9355d4efc91df0, 0x3e610944cc9fecfd, 0x5bf9eb60b08ac0ce, 0x26f2b463,
	},
	{
		0x831f448bdc5600b3, 0x62a24be3120a6919, 0x1b44098a41e010da, 0xdc07df53b949c6b,
		0xd2b11b2081aeb002, 0xd212b02c1b13f772, 0xc0bed297b4be1912, 0xdc07df53b949c6b,
		0xd2b11b2081aeb002, 0xd212b02c1b13f772, 0xc0bed297b4be1912, 0x682d3d2ad304e4af,
		0x40e9112a655437a1, 0x268b09f7ee09843f, 0x6b9698d43859ca47, 0xb042c047,
	},
	{
		0x3eca803e70304894, 0xd80de767e4a920a, 0xa51cfbb292efd53d, 0xd183dcda5f73edfa,
		0x3a93cbf40f30128c, 0x1a92544d0b41dbda, 0xaec2c4bee81975e1, 0xd183dcda5f73edfa,
		0x3a93cbf40f30128c, 0x1a92544d0b41dbda, 0xaec2c4bee81975e1, 0x5f91814d1126ba4b,
		0xf8ac57eee87fcf1f, 0xc55c644a5d0023cd, 0xadb761e827825ff2, 0xe73bb0a8,
	},
	{
		0x1b5a063fb4c7f9f1, 0x318dbc24af66dee9, 0x10ef7b32d5c719af, 0xb140a02ef5c97712,
		0xb7d00ef065b51b33, 0x635121d532897d98, 0x532daf21b312a6d6, 0xb140a02ef5c97712,
		0xb7d00ef065b51b33, 0x635121d532897d98, 0x532daf21b312a6d6, 0xc0b09b75d943910,
		0x8c84dfb5ef2a8e96, 0xe5c06034b0353433, 0x3170faf1c33a45dd, 0x91dfdd75,
	},
	{
		0xa0f10149a0e538d6, 0x69d008c20f87419f, 0x41b36376185b3e9e, 0x26b6689960ccf81d,
		0x55f23b27bb9efd94, 0x3a17f6166dd765db, 0xc891a8a62931e782, 0x26b6689960ccf81d,
		0x55f23b27bb9efd94, 0x3a17f6166dd765db, 0xc891a8a62931e782, 0x23852dc37ddd2607,
		0x8b7f1b1ec897829e, 0xd1d69452a54eed8a, 0x56431f2bd766ec24, 0xc87f95de,
	},
	{
		0xfb8d9c70660b910b, 0xa45b0cc3476bff1b, 0xb28d1996144f0207, 0x98ec31113e5e35d2,
		0x5e4aeb853f1b9aa7, 0xbcf5c8fe4465b7c8, 0xb1ea3a8243996f15, 0x98ec31113e5e35d2,
		0x5e4aeb853f1b9aa7, 0xbcf5c8fe4465b7c8, 0xb1ea3a8243996f15, 0xcabbccedb6407571,
		0xd1e40a84c445ec3a, 0x33302aa908cf4039, 0x9f15f79211b5cdf8, 0x3f5538ef,
	},
	{
		0x236827beae282a46, 0xe43970221139c946, 0x4f3ac6faa837a3aa, 0x71fec0f972248915,
		0x2170ec2061f24574, 0x9eb346b6caa36e82, 0x2908f0fdbca48e73, 0x71fec0f972248915,
		0x2170ec2061f24574, 0x9eb346b6caa36e82, 0x2908f0fdbca48e73, 0x8101c99f07c64abb,
		0xb9f4b02b1b6a96a7, 0x583a2b10cd222f88, 0x199dae4cf9db24c, 0x70eb1a1f,
	},
	{
		0xc385e435136ecf7c, 0xd9d17368ff6c4a08, 0x1b31eed4e5251a67, 0xdf01a322c43a6200,
		0x298b65a1714b5a7e, 0x933b83f0aedf23c, 0x157bcb44d63f765a, 0xdf01a322c43a6200,
		0x298b65a1714b5a7e, 0x933b83f0aedf23c, 0x157bcb44d63f765a, 0xd6e9fc7a272d8b51,
		0x3ee5073ef1a9b777, 0x63149e31fac02c59, 0x2f7979ff636ba1d8, 0xcfd63b83,
	},
	{
		0xe3f6828b6017086d, 0x21b4d1900554b3b0, 0xbef38be1809e24f1, 0xd93251758985ee6c,
		0x32a9e9f82ba2a932, 0x3822aacaa95f3329, 0xdb349b2f90a490d8, 0xd93251758985ee6c,
		0x32a9e9f82ba2a932, 0x3822aacaa95f3329, 0xdb349b2f90a490d8, 0x8d49194a894a19ca,
		0x79a78b06e42738e6, 0x7e0f1eda3d390c66, 0x1c291d7e641100a5, 0x894a52ef,
	},
	{
		0x851fff285561dca0, 0x4d1277d73cdf416f, 0x28ccffa61010ebe2, 0x77a4ccacd131d9ee,
		0xe1d08eeb2f0e29aa, 0x70b9e3051383fa45, 0x582d0120425caba, 0x77a4ccacd131d9ee,
		0xe1d08eeb2f0e29aa, 0x70b9e3051383fa45, 0x582d0120425caba, 0xa740eef1846e4564,
		0x572dddb74ac3ae00, 0xfdb5ca9579163bbd, 0xa649b9b799c615d2, 0x9cde6a54,
	},
	{
		0x61152a63595a96d9, 0xd1a3a91ef3a7ba45, 0x443b6bb4a493ad0c, 0xa154296d11362d06,
		0xd0f0bf1f1cb02fc1, 0xccb87e09309f90d1, 0xb24a8e4881911101, 0xa154296d11362d06,
		0xd0f0bf1f1cb02fc1, 0xccb87e09309f90d1, 0xb24a8e4881911101, 0x1a481b4528559f58,
		0xbf837a3150896995, 0x4989ef6b941a3757, 0x2e725ab72d0b2948, 0x6c4898d5,
	},
	{
		0x44473e03be306c88, 0x30097761f872472a, 0x9fd1b669bfad82d7, 0x3bab18b164396783,
		0x47e385ff9d4c06f, 0x18062081bf558df, 0x63416eb68f104a36, 0x3bab18b164396783,
		0x47e385ff9d4c06f, 0x18062081bf558df, 0x63416eb68f104a36, 0x4abda1560c47ac80,
		0x1ea0e63dc6587aee, 0x33ec79d92ebc1de, 0x94f9dccef771e048, 0x13e1978e,
	},
	{
		0x3ead5f21d344056, 0xfb6420393cfb05c3, 0x407932394cbbd303, 0xac059617f5906673,
		0x94d50d3dcd3069a7, 0x2b26c3b92dea0f0, 0x99b7374cc78fc3fb, 0xac059617f5906673,
		0x94d50d3dcd3069a7, 0x2b26c3b92dea0f0, 0x99b7374cc78fc3fb, 0x1a8e3c73cdd40ee8,
		0xcbb5fca06747f45b, 0xceec44238b291841, 0x28bf35cce9c90a25, 0x51b4ba8,
	},
	{
		0x6abbfde37ee03b5b, 0x83febf188d2cc113, 0xcda7b62d94d5b8ee, 0xa4375590b8ae7c82,
		0x168fd42f9ecae4ff, 0x23bbde43de2cb214, 0xa8c333112a243c8c, 0xa4375590b8ae7c82,
		0x168fd42f9ecae4ff, 0x23bbde43de2cb214, 0xa8c333112a243c8c, 0x10ac012e8c518b49,
		0x64a44605d8b29458, 0xa67e701d2a679075, 0x3a3a20f43ec92303, 0xb6b06e40,
	},
	{
		0x943e7ed63b3c080, 0x1ef207e9444ef7f8, 0xef4a9f9f8c6f9b4a, 0x6b54fc38d6a84108,
		0x32f4212a47a4665, 0x6b5a9a8f64ee1da6, 0x9f74e86c6da69421, 0x6b54fc38d6a84108,
		0x32f4212a47a4665, 0x6b5a9a8f64ee1da6, 0x9f74e86c6da69421, 0x946dd0cb30c1a08e,
		0xfdf376956907eaaa, 0xa59074c6eec03028, 0xb1a3abcf283f34ac, 0x240a2f2,
	},
	{
		0xd72ce05171ef8a1a, 0xc6bd6bd869203894, 0xc760e6396455d23a, 0xf86af0b40dcce7b,
		0x8d3c15d613394d3c, 0x491e400491cd4ece, 0x7c19d3530ea3547f, 0xf86af0b40dcce7b,
		0x8d3c15d613394d3c, 0x491e400491cd4ece, 0x7c19d3530ea3547f, 0x1362963a1dc32af9,
		0xfb9bc11762e1385c, 0x9e164ef1f5376083, 0x6c15819b5e828a7e, 0x5dcefc30,
	},
	{
		0x4182832b52d63735, 0x337097e123eea414, 0xb5a72ca0456df910, 0x7ebc034235bc122f,
		0xd9a7783d4edd8049, 0x5f8b04a15ae42361, 0xfc193363336453dd, 0x7ebc034235bc122f,
		0xd9a7783d4edd8049, 0x5f8b04a15ae42361, 0xfc193363336453dd, 0x9b6c50224ef8c4f8,
		0xba225c7942d16c3f, 0x6f6d55226a73c412, 0xabca061fe072152a, 0x7a48b105,
	},
	{
		0xd6cdae892584a2cb, 0x58de0fa4eca17dcd, 0x43df30b8f5f1cb00, 0x9e4ea5a4941e097d,
		0x547e048d5a9daaba, 0xeb6ecbb0b831d185, 0xe0168df5fad0c670, 0x9e4ea5a4941e097d,
		0x547e048d5a9daaba, 0xeb6ecbb0b831d185, 0xe0168df5fad0c670, 0xafa9705f98c2c96a,
		0x749436f48137a96b, 0x759c041fc21df486, 0xb23bf400107aa2ec, 0xfd55007b,
	},
	{
		0x5c8e90bc267c5ee4, 0xe9ae044075d992d9, 0xf234cbfd1f0a1e59, 0xce2744521944f14c,
		0x104f8032f99dc152, 0x4e7f425bfac67ca7, 0x9461b911a1c6d589, 0xce2744521944f14c,
		0x104f8032f99dc152, 0x4e7f425bfac67ca7, 0x9461b911a1c6d589, 0x5e5ecc726db8b60d,
		0xcce68b0586083b51, 0x8a7f8e54a9cba0fc, 0x42f010181d16f049, 0x6b95894c,
	},
	{
		0xbbd7f30ac310a6f3, 0xb23b570d2666685f, 0xfb13fb08c9814fe7, 0x4ee107042e512374,
		0x1e2c8c0d16097e13, 0x210c7500995aa0e6, 0x6c13190557106457, 0x4ee107042e512374,
		0x1e2c8c0d16097e13, 0x210c7500995aa0e6, 0x6c13190557106457, 0xa99b31c96777f381,
		0x8312ae8301d386c0, 0xed5042b2a4fa96a3, 0xd71d1bb23907fe97, 0x3360e827,
	},
	{
		0x36a097aa49519d97, 0x8204380a73c4065, 0x77c2004bdd9e276a, 0x6ee1f817ce0b7aee,
		0xe9dcb3507f0596ca, 0x6bc63c666b5100e2, 0xe0b056f1821752af, 0x6ee1f817ce0b7aee,
		0xe9dcb3507f0596ca, 0x6bc63c666b5100e2, 0xe0b056f1821752af, 0x8ea1114e60292678,
		0x904b80b46becc77, 0x46cd9bb6e9dff52f, 0x4c91e3b698355540, 0x45177e0b,
	},
	{
		0xdc78cb032c49217, 0x112464083f83e03a, 0x96ae53e28170c0f5, 0xd367ff54952a958,
		0xcdad930657371147, 0xaa24dc2a9573d5fe, 0xeb136daa89da5110, 0xd367ff54952a958,
		0xcdad930657371147, 0xaa24dc2a9573d5fe, 0xeb136daa89da5110, 0xde623005f6d46057,
		0xb50c0c92b95e9b7f, 0xa8aa54050b81c978, 0x573fb5c7895af9b5, 0x7c6fffe4,
	},
	{
		0x441593e0da922dfe, 0x936ef46061469b32, 0x204a1921197ddd87, 0x50d8a70e7a8d8f56,
		0x256d150ae75dab76, 0xe81f4c4a1989036a, 0xd0f8db365f9d7e00, 0x50d8a70e7a8d8f56,
		0x256d150ae75dab76, 0xe81f4c4a1989036a, 0xd0f8db365f9d7e00, 0x753d686677b14522,
		0x9f76e0cb6f2d0a66, 0xab14f95988ec0d39, 0x97621d9da9c9812f, 0xbbc78da4,
	},
	{
		0x2ba3883d71cc2133, 0x72f2bbb32bed1a3c, 0x27e1bd96d4843251, 0xa90f761e8db1543a,
		0xc339e23c09703cd8, 0xf0c6624c4b098fd3, 0x1bae2053e41fa4d9, 0xa90f761e8db1543a,
		0xc339e23c09703cd8, 0xf0c6624c4b098fd3, 0x1bae2053e41fa4d9, 0x3589e273c22ba059,
		0x63798246e5911a0b, 0x18e710ec268fc5dc, 0x714a122de1d074f3, 0xc5c25d39,
	},
	{
		0xf2b6d2adf8423600, 0x7514e2f016a48722, 0x43045743a50396ba, 0x23dacb811652ad4f,
		0xc982da480e0d4c7d, 0x3a9c8ed5a399d0a9, 0x951b8d084691d4e4, 0x23dacb811652ad4f,
		0xc982da480e0d4c7d, 0x3a9c8ed5a399d0a9, 0x951b8d084691d4e4, 0xd9f87b4988cff2f7,
		0x217a191d986aa3bc, 0x6ad23c56b480350, 0xdd78673938ceb2e7, 0xb6e5d06e,
	},
	{
		0x38fffe7f3680d63c, 0xd513325255a7a6d1, 0x31ed47790f6ca62f, 0xc801faaa0a2e331f,
		0x491dbc58279c7f88, 0x9c0178848321c97a, 0x9d934f814f4d6a3c, 0xc801faaa0a2e331f,
		0x491dbc58279c7f88, 0x9c0178848321c97a, 0x9d934f814f4d6a3c, 0x606a3e4fc8763192,
		0xbc15cb36a677ee84, 0x52d5904157e1fe71, 0x1588dd8b1145b79b, 0x6178504e,
	},
	{
		0xb7477bf0b9ce37c6, 0x63b1c580a7fd02a4, 0xf6433b9f10a5dac, 0x68dd76db9d64eca7,
		0x36297682b64b67, 0x42b192d71f414b7a, 0x79692cef44fa0206, 0x68dd76db9d64eca7,
		0x36297682b64b67, 0x42b192d71f414b7a, 0x79692cef44fa0206, 0xf0979252f4776d07,
		0x4b87cd4f1c9bbf52, 0x51b84bbc6312c710, 0x150720fbf85428a7, 0xbd4c3637,
	},
	{
		0x55bdb0e71e3edebd, 0xc7ab562bcf0568bc, 0x43166332f9ee684f, 0xb2e25964cd409117,
		0xa010599d6287c412, 0xfa5d6461e768dda2, 0xcb3ce74e8ec4f906, 0xb2e25964cd409117,
		0xa010599d6287c412, 0xfa5d6461e768dda2, 0xcb3ce74e8ec4f906, 0x6120abfd541a2610,
		0xaa88b148cc95794d, 0x2686ca35df6590e3, 0xc6b02d18616ce94d, 0x6e7ac474,
	},
	{
		0x782fa1b08b475e7, 0xfb7138951c61b23b, 0x9829105e234fb11e, 0x9a8c431f500ef06e,
		0xd848581a580b6c12, 0xfecfe11e13a2bdb4, 0x6c4fa0273d7db08c, 0x9a8c431f500ef06e,
		0xd848581a580b6c12, 0xfecfe11e13a2bdb4, 0x6c4fa0273d7db08c, 0x482f43bf5ae59fcb,
		0xf651fbca105d79e6, 0xf09f78695d865817, 0x7a99d0092085cf47, 0x1fb4b518,
	},
	{
		0xc5dc19b876d37a80, 0x15ffcff666cfd710, 0xe8c30c72003103e2, 0x7870765b470b2c5d,
		0x78a9103ff960d82, 0x7bb50ffc9fac74b3, 0x477e70ab2b347db2, 0x7870765b470b2c5d,
		0x78a9103ff960d82, 0x7bb50ffc9fac74b3, 0x477e70ab2b347db2, 0xa625238bdf7c07cf,
		0x1128d515174809f5, 0xb0f1647e82f45873, 0x17792d1c4f222c39, 0x31d13d6d,
	},
	{
		0x5e1141711d2d6706, 0xb537f6dee8de6933, 0x3af0a1fbbe027c54, 0xea349dbc16c2e441,
		0x38a7455b6a877547, 0x5f97b9750e365411, 0xe8cde7f93af49a3, 0xea349dbc16c2e441,
		0x38a7455b6a877547, 0x5f97b9750e365411, 0xe8cde7f93af49a3, 0xba101925ec1f7e26,
		0xd5e84cab8192c71e, 0xe256427726fdd633, 0xa4f38e2c6116890d, 0x26fa72e3,
	},
	{
		0x782edf6da001234f, 0xf48cbd5c66c48f3, 0x808754d1e64e2a32, 0x5d9dde77353b1a6d,
		0x11f58c54581fa8b1, 0xda90fa7c28c37478, 0x5e9a2eafc670a88a, 0x5d9dde77353b1a6d,
		0x11f58c54581fa8b1, 0xda90fa7c28c37478, 0x5e9a2eafc670a88a, 0xe35e1bc172e011ef,
		0xbf9255a4450ae7fe, 0x55f85194e26bc55f, 0x4f327873e14d0e54, 0x6a7433bf,
	},
	{
		0xd26285842ff04d44, 0x8f38d71341eacca9, 0x5ca436f4db7a883c, 0xbf41e5376b9f0eec,
		0x2252d21eb7e1c0e9, 0xf4b70a971855e732, 0x40c7695aa3662afd, 0xbf41e5376b9f0eec,
		0x2252d21eb7e1c0e9, 0xf4b70a971855e732, 0x40c7695aa3662afd, 0x770fe19e16ab73bb,
		0xd603ebda6393d749, 0xe58c62439aa50dbd, 0x96d51e5a02d2d7cf, 0x4e6df758,
	},
	{
		0xc6ab830865a6bae6, 0x6aa8e8dd4b98815c, 0xefe3846713c371e5, 0xa1924cbf0b5f9222,
		0x7f4872369c2b4258, 0xcd6da30530f3ea89, 0xb7f8b9a704e6cea1, 0xa1924cbf0b5f9222,
		0x7f4872369c2b4258, 0xcd6da30530f3ea89, 0xb7f8b9a704e6cea1, 0xfa06ff40433fd535,
		0xfb1c36fe8f0737f1, 0xbb7050561171f80, 0xb1bc23235935d897, 0xd57f63ea,
	},
	{
		0x44b3a1929232892, 0x61dca0e914fc217, 0xa607cc142096b964, 0xf7dbc8433c89b274,
		0x2f5f70581c9b7d32, 0x39bf5e5fec82dcca, 0x8ade56388901a619, 0xf7dbc8433c89b274,
		0x2f5f70581c9b7d32, 0x39bf5e5fec82dcca, 0x8ade56388901a619, 0xc1c6a725caab3ea9,
		0xc1c7906c2f80b898, 0x9c3871a04cc884e6, 0xdf01813cbbdf217f, 0x52ef73b3,
	},
	{
		0x4b603d7932a8de4f, 0xfae64c464b8a8f45, 0x8fafab75661d602a, 0x8ffe870ef4adc087,
		0x65bea2be41f55b54, 0x82f3503f636aef1, 0x5f78a282378b6bb0, 0x8ffe870ef4adc087,
		0x65bea2be41f55b54, 0x82f3503f636aef1, 0x5f78a282378b6bb0, 0x7bf2422c0beceddb,
		0x9d238d4780114bd, 0x7ad198311906597f, 0xec8f892c0422aca3, 0x3cb36c3,
	},
	{
		0x4ec0b54cf1566aff, 0x30d2c7269b206bf4, 0x77c22e82295e1061, 0x3df9b04434771542,
		0xfeddce785ccb661f, 0xa644aff716928297, 0xdd46aee73824b4ed, 0x3df9b04434771542,
		0xfeddce785ccb661f, 0xa644aff716928297, 0xdd46aee73824b4ed, 0xbf8d71879da29b02,
		0xfc82dccbfc8022a0, 0x31bfcd0d9f48d1d3, 0xc64ee24d0e7b5f8b, 0x72c39bea,
	},
	{
		0xed8b7a4b34954ff7, 0x56432de31f4ee757, 0x85bd3abaa572b155, 0x7d2c38a926dc1b88,
		0x5245b9eb4cd6791d, 0xfb53ab03b9ad0855, 0x3664026c8fc669d7, 0x7d2c38a926dc1b88,
		0x5245b9eb4cd6791d, 0xfb53ab03b9ad0855, 0x3664026c8fc669d7, 0x45024d5080bc196,
		0xb236ebec2cc2740, 0x27231ad0e3443be4, 0x145780b63f809250, 0xa65aa25c,
	},
	{
		0x5d28b43694176c26, 0x714cc8bc12d060ae, 0x3437726273a83fe6, 0x864b1b28ec16ea86,
		0x6a78a5a4039ec2b9, 0x8e959533e35a766, 0x347b7c22b75ae65f, 0x864b1b28ec16ea86,
		0x6a78a5a4039ec2b9, 0x8e959533e35a766, 0x347b7c22b75ae65f, 0x5005892bb61e647c,
		0xfe646519b4a1894d, 0xcd801026f74a8a53, 0x8713463e9a1ab9ce, 0x74740539,
	},
	{
		0x6a1ef3639e1d202e, 0x919bc1bd145ad928, 0x30f3f7e48c28a773, 0x2e8c49d7c7aaa527,
		0x5e2328fc8701db7c, 0x89ef1afca81f7de8, 0xb1857db11985d296, 0x2e8c49d7c7aaa527,
		0x5e2328fc8701db7c, 0x89ef1afca81f7de8, 0xb1857db11985d296, 0x17763d695f616115,
		0xb8f7bf1fcdc8322c, 0xcf0c61938ab07a27, 0x1122d3e6edb4e866, 0xc3ae3c26,
	},
	{
		0x159f4d9e0307b111, 0x3e17914a5675a0c, 0xaf849bd425047b51, 0x3b69edadf357432b,
		0x3a2e311c121e6bf2, 0x380fad1e288d57e5, 0xbf7c7e8ef0e3b83a, 0x3b69edadf357432b,
		0x3a2e311c121e6bf2, 0x380fad1e288d57e5, 0xbf7c7e8ef0e3b83a, 0x92966d5f4356ae9b,
		0x2a03fc66c4d6c036, 0x2516d8bddb0d5259, 0xb3ffe9737ff5090, 0xf29db8a2,
	},
	{
		0xcc0a840725a7e25b, 0x57c69454396e193a, 0x976eaf7eee0b4540, 0xcd7a46850b95e901,
		0xc57f7d060dda246f, 0x6b9406ead64079bf, 0x11b28e20a573b7bd, 0xcd7a46850b95e901,
		0xc57f7d060dda246f, 0x6b9406ead64079bf, 0x11b28e20a573b7bd, 0x2d6db356e9369ace,
		0xdc0afe10fba193, 0x5cdb10885dbbfce, 0x5c700e205782e35a, 0x1ef4cbf4,
	},
	{
		0xa2b27ee22f63c3f1, 0x9ebde0ce1b3976b2, 0x2fe6a92a257af308, 0x8c1df927a930af59,
		0xa462f4423c9e384e, 0x236542255b2ad8d9, 0x595d201a2c19d5bc, 0x8c1df927a930af59,
		0xa462f4423c9e384e, 0x236542255b2ad8d9, 0x595d201a2c19d5bc, 0x22c87d4604a67f3,
		0x585a06eb4bc44c4f, 0xb4175a7ac7eabcd8, 0xa457d3eeba14ab8c, 0xa9be6c41,
	},
	{
		0xd8f2f234899bcab3, 0xb10b037297c3a168, 0xdebea2c510ceda7f, 0x9498fefb890287ce,
		0xae68c2be5b1a69a6, 0x6189dfba34ed656c, 0x91658f95836e5206, 0x9498fefb890287ce,
		0xae68c2be5b1a69a6, 0x6189dfba34ed656c, 0x91658f95836e5206, 0xc0bb4fff32aecd4d,
		0x94125f505a50eef9, 0x6ac406e7cfbce5bb, 0x344a4b1dcdb7f5d8, 0xfa31801,
	},
	{
		0x584f28543864844f, 0xd7cee9fc2d46f20d, 0xa38dca5657387205, 0x7a0b6dbab9a14e69,
		0xc6d0a9d6b0e31ac4, 0xa674d85812c7cf6, 0x63538c0351049940, 0x7a0b6dbab9a14e69,
		0xc6d0a9d6b0e31ac4, 0xa674d85812c7cf6, 0x63538c0351049940, 0x9710e5f0bc93d1d,
		0xc2bea5bd7c54ddd4, 0x48739af2bed0d32d, 0xba2c4e09e21fba85, 0x8331c5d8,
	},
	{
		0xa94be46dd9aa41af, 0xa57e5b7723d3f9bd, 0x34bf845a52fd2f, 0x843b58463c8df0ae,
		0x74b258324e916045, 0xbdd7353230eb2b38, 0xfad31fced7abade5, 0x843b58463c8df0ae,
		0x74b258324e916045, 0xbdd7353230eb2b38, 0xfad31fced7abade5, 0x2436aeafb0046f85,
		0x65bc9af9e5e33161, 0x92733b1b3ae90628, 0xf48143eaf78a7a89, 0xe9876db8,
	},
	{
		0x9a87bea227491d20, 0xa468657e2b9c43e7, 0xaf9ba60db8d89ef7, 0xcc76f429ea7a12bb,
		0x5f30eaf2bb14870a, 0x434e824cb3e0cd11, 0x431a4d382e39d16e, 0xcc76f429ea7a12bb,
		0x5f30eaf2bb14870a, 0x434e824cb3e0cd11, 0x431a4d382e39d16e, 0x9e51f913c4773a8,
		0x32ab1925823d0add, 0x99c61b54c1d8f69d, 0x38cfb80f02b43b1f, 0x27b0604e,
	},
	{
		0x27688c24958d1a5c, 0xe3b4a1c9429cf253, 0x48a95811f70d64bc, 0x328063229db22884,
		0x67e9c95f8ba96028, 0x7c6bf01c60436075, 0xfa55161e7d9030b2, 0x328063229db22884,
		0x67e9c95f8ba96028, 0x7c6bf01c60436075, 0xfa55161e7d9030b2, 0xdadbc2f0dab91681,
		0xda39d7a4934ca11, 0x162e845d24c1b45c, 0xeb5b9dcd8c6ed31b, 0xdcec07f2,
	},
	{
		0x5d1d37790a1873ad, 0xed9cd4bcc5fa1090, 0xce51cde05d8cd96a, 0xf72c26e624407e66,
		0xa0eb541bdbc6d409, 0xc3f40a2f40b3b213, 0x6a784de68794492d, 0xf72c26e624407e66,
		0xa0eb541bdbc6d409, 0xc3f40a2f40b3b213, 0x6a784de68794492d, 0x10a38a23dbef7937,
		0x6a5560f853252278, 0xc3387bbf3c7b82ba, 0xfbee7c12eb072805, 0xcff0a82a,
	},
	{
		0x1f03fd18b711eea9, 0x566d89b1946d381a, 0x6e96e83fc92563ab, 0x405f66cf8cae1a32,
		0xd7261740d8f18ce6, 0xfea3af64a413d0b2, 0xd64d1810e83520fe, 0x405f66cf8cae1a32,
		0xd7261740d8f18ce6, 0xfea3af64a413d0b2, 0xd64d1810e83520fe, 0xe1334a00a580c6e8,
		0x454049e1b52c15f, 0x8895d823d9778247, 0xefa7f2e88b826618, 0xfec83621,
	},
	{
		0xf0316f286cf527b6, 0xf84c29538de1aa5a, 0x7612ed3c923d4a71, 0xd4eccebe9393ee8a,
		0x2eb7867c2318cc59, 0x1ce621fd700fe396, 0x686450d7a346878a, 0xd4eccebe9393ee8a,
		0x2eb7867c2318cc59, 0x1ce621fd700fe396, 0x686450d7a346878a, 0x75a5f37579f8b4cb,
		0x500cc16eb6541dc7, 0xb7b02317b539d9a6, 0x3519ddff5bc20a29, 0x743d8dc,
	},
	{
		0x297008bcb3e3401d, 0x61a8e407f82b0c69, 0xa4a35bff0524fa0e, 0x7a61d8f552a53442,
		0x821d1d8d8cfacf35, 0x7cc06361b86d0559, 0x119b617a8c2be199, 0x7a61d8f552a53442,
		0x821d1d8d8cfacf35, 0x7cc06361b86d0559, 0x119b617a8c2be199, 0x2996487da6721759,
		0x61a901376070b91d, 0xd88dee12ae9c9b3c, 0x5665491be1fa53a7, 0x64d41d26,
	},
	{
		0x43c6252411ee3be, 0xb4ca1b8077777168, 0x2746dc3f7da1737f, 0x2247a4b2058d1c50,
		0x1b3fa184b1d7bcc0, 0xdeb85613995c06ed, 0xcbe1d957485a3ccd, 0x2247a4b2058d1c50,
		0x1b3fa184b1d7bcc0, 0xdeb85613995c06ed, 0xcbe1d957485a3ccd, 0xdfe241f8f33c96b6,
		0x6597eb05019c2109, 0xda344b2a63a219cf, 0x79b8e3887612378a, 0xacd90c81,
	},
	{
		0xce38a9a54fad6599, 0x6d6f4a90b9e8755e, 0xc3ecc79ff105de3f, 0xe8b9ee96efa2d0e,
		0x90122905c4ab5358, 0x84f80c832d71979c, 0x229310f3ffbbf4c6, 0xe8b9ee96efa2d0e,
		0x90122905c4ab5358, 0x84f80c832d71979c, 0x229310f3ffbbf4c6, 0xcc9eb42100cd63a7,
		0x7a283f2f3da7b9f, 0x359b061d314e7a72, 0xd0d959720028862, 0x7c746a4b,
	},
	{
		0x270a9305fef70cf, 0x600193999d884f3a, 0xf4d49eae09ed8a1, 0x2e091b85660f1298,
		0xbfe37fae1cdd64c9, 0x8dddfbab930f6494, 0x2ccf4b08f5d417a, 0x2e091b85660f1298,
		0xbfe37fae1cdd64c9, 0x8dddfbab930f6494, 0x2ccf4b08f5d417a, 0x365c2ee85582fe6,
		0xdee027bcd36db62a, 0xb150994d3c7e5838, 0xfdfd1a0e692e436d, 0xb1047e99,
	},
	{
		0xe71be7c28e84d119, 0xeb6ace59932736e6, 0x70c4397807ba12c5, 0x7a9d77781ac53509,
		0x4489c3ccfda3b39c, 0xfa722d4f243b4964, 0x25f15800bffdd122, 0x7a9d77781ac53509,
		0x4489c3ccfda3b39c, 0xfa722d4f243b4964, 0x25f15800bffdd122, 0xed85e4157fbd3297,
		0xaab1967227d59efd, 0x2199631212eb3839, 0x3e4c19359aae1cc2, 0xd1fd1068,
	},
	{
		0xb5b58c24b53aaa19, 0xd2a6ab0773dd897f, 0xef762fe01ecb5b97, 0x9deefbcfa4cab1f1,
		0xb58f5943cd2492ba, 0xa96dcc4d1f4782a7, 0x102b62a82309dde5, 0x9deefbcfa4cab1f1,
		0xb58f5943cd2492ba, 0xa96dcc4d1f4782a7, 0x102b62a82309dde5, 0x35fe52684763b338,
		0xafe2616651eaad1f, 0x43e38715bdfa05e7, 0x83c9ba83b5ec4a40, 0x56486077,
	},
	{
		0x44dd59bd301995cf, 0x3ccabd76493ada1a, 0x540db4c87d55ef23, 0xcfc6d7adda35797,
		0x14c7d1f32332cf03, 0x2d553ffbff3be99d, 0xc91c4ee0cb563182, 0xcfc6d7adda35797,
		0x14c7d1f32332cf03, 0x2d553ffbff3be99d, 0xc91c4ee0cb563182, 0x9aa5e507f49136f0,
		0x760c5dd1a82c4888, 0xbeea7e974a1cfb5c, 0x640b247774fe4bf7, 0x6069be80,
	},
	{
		0xb4d4789eb6f2630b, 0xbf6973263ce8ef0e, 0xd1c75c50844b9d3, 0xbce905900c1ec6ea,
		0xc30f304f4045487d, 0xa5c550166b3a142b, 0x2f482b4e35327287, 0xbce905900c1ec6ea,
		0xc30f304f4045487d, 0xa5c550166b3a142b, 0x2f482b4e35327287, 0x15b21ddddf355438,
		0x496471fa3006bab, 0x2a8fd458d06c1a32, 0xdb91e8ae812f0b8d, 0x2078359b,
	},
	{
		0x12807833c463737c, 0x58e927ea3b3776b4, 0x72dd20ef1c2f8ad0, 0x910b610de7a967bf,
		0x801bc862120f6bf5, 0x9653efeed5897681, 0xf5367ff83e9ebbb3, 0x910b610de7a967bf,
		0x801bc862120f6bf5, 0x9653efeed5897681, 0xf5367ff83e9ebbb3, 0xcf56d489afd1b0bf,
		0xc7c793715cae3de8, 0x631f91d64abae47c, 0x5f1f42fb14a444a2, 0x9ea21004,
	},
	{
		0xe88419922b87176f, 0xbcf32f41a7ddbf6f, 0xd6ebefd8085c1a0f, 0xd1d44fe99451ef72,
		0xec951ba8e51e3545, 0xc0ca86b360746e96, 0xaa679cc066a8040b, 0xd1d44fe99451ef72,
		0xec951ba8e51e3545, 0xc0ca86b360746e96, 0xaa679cc066a8040b, 0x51065861ece6ffc1,
		0x76777368a2997e11, 0x87f278f46731100c, 0xbbaa4140bdba4527, 0x9c9cfe88,
	},
	{
		0x105191e0ec8f7f60, 0x5918dbfcca971e79, 0x6b285c8a944767b9, 0xd3e86ac4f5eccfa4,
		0xe5399df2b106ca1, 0x814aadfacd217f1d, 0x2754e3def1c405a9, 0xd3e86ac4f5eccfa4,
		0xe5399df2b106ca1, 0x814aadfacd217f1d, 0x2754e3def1c405a9, 0x99290323b9f06e74,
		0xa9782e043f271461, 0x13c8b3b8c275a860, 0x6038d620e581e9e7, 0xb70a6ddd,
	},
	{
		0xa5b88bf7399a9f07, 0xfca3ddfd96461cc4, 0xebe738fdc0282fc6, 0x69afbc800606d0fb,
		0x6104b97a9db12df7, 0xfcc09198bb90bf9f, 0xc5e077e41a65ba91, 0x69afbc800606d0fb,
		0x6104b97a9db12df7, 0xfcc09198bb90bf9f, 0xc5e077e41a65ba91, 0xdb261835ee8aa08e,
		0xdb0ee662e5796dc9, 0xfc1880ecec499e5f, 0x648866fbe1502034, 0xdea37298,
	},
	{
		0xd08c3f5747d84f50, 0x4e708b27d1b6f8ac, 0x70f70fd734888606, 0x909ae019d761d019,
		0x368bf4aab1b86ef9, 0x308bd616d5460239, 0x4fd33269f76783ea, 0x909ae019d761d019,
		0x368bf4aab1b86ef9, 0x308bd616d5460239, 0x4fd33269f76783ea, 0x7d53b37c19713eab,
		0x6bba6eabda58a897, 0x91abb50efc116047, 0x4e902f347e0e0e35, 0x8f480819,
	},
	{
		0x2f72d12a40044b4b, 0x889689352fec53de, 0xf03e6ad87eb2f36, 0xef79f28d874b9e2d,
		0xb512089e8e63b76c, 0x24dc06833bf193a9, 0x3c23308ba8e99d7e, 0xef79f28d874b9e2d,
		0xb512089e8e63b76c, 0x24dc06833bf193a9, 0x3c23308ba8e99d7e, 0x5ceff7b85cacefb7,
		0xef390338898cd73, 0xb12967d7d2254f54, 0xde874cbd8aef7b75, 0x30b3b16,
	},
	{
		0xaa1f61fdc5c2e11e, 0xc2c56cd11277ab27, 0xa1e73069fdf1f94f, 0x8184bab36bb79df0,
		0xc81929ce8655b940, 0x301b11bf8a4d8ce8, 0x73126fd45ab75de9, 0x8184bab36bb79df0,
		0xc81929ce8655b940, 0x301b11bf8a4d8ce8, 0x73126fd45ab75de9, 0x4bd6f76e4888229a,
		0x9aae355b54a756d5, 0xca3de9726f6e99d5, 0x83f80cac5bc36852, 0xf31bc4e8,
	},
	{
		0x9489b36fe2246244, 0x3355367033be74b8, 0x5f57c2277cbce516, 0xbc61414f9802ecaf,
		0x8edd1e7a50562924, 0x48f4ab74a35e95f2, 0xcc1afcfd99a180e7, 0xbc61414f9802ecaf,
		0x8edd1e7a50562924, 0x48f4ab74a35e95f2, 0xcc1afcfd99a180e7, 0x517dd5e3acf66110,
		0x7dd3ad9e8978b30d, 0x1f6d5dfc70de812b, 0x947daaba6441aaf3, 0x419f953b,
	},
	{
		0x358d7c0476a044cd, 0xe0b7b47bcbd8854f, 0xffb42ec696705519, 0xd45e44c263e95c38,
		0xdf61db53923ae3b1, 0xf2bc948cc4fc027c, 0x8a8000c6066772a3, 0xd45e44c263e95c38,
		0xdf61db53923ae3b1, 0xf2bc948cc4fc027c, 0x8a8000c6066772a3, 0x9fd93c942d31fa17,
		0xd7651ecebe09cbd3, 0x68682cefb6a6f165, 0x541eb99a2dcee40e, 0x20e9e76d,
	},
	{
		0xb0c48df14275265a, 0x9da4448975905efa, 0xd716618e414ceb6d, 0x30e888af70df1e56,
		0x4bee54bd47274f69, 0x178b4059e1a0afe5, 0x6e2c96b7f58e5178, 0x30e888af70df1e56,
		0x4bee54bd47274f69, 0x178b4059e1a0afe5, 0x6e2c96b7f58e5178, 0xbb429d3b9275e9bc,
		0xc198013f09cafdc6, 0xec0a6ee4fb5de348, 0x744e1e8ed2eb1eb0, 0x646f0ff8,
	},
	{
		0xdaa70bb300956588, 0x410ea6883a240c6d, 0xf5c8239fb5673eb3, 0x8b1d7bb4903c105f,
		0xcfb1c322b73891d4, 0x5f3b792b22f07297, 0xfd64061f8be86811, 0x8b1d7bb4903c105f,
		0xcfb1c322b73891d4, 0x5f3b792b22f07297, 0xfd64061f8be86811, 0x1d2db712921cfc2b,
		0xcd1b2b2f2cee18ae, 0x6b6f8790dc7feb09, 0x46c179efa3f0f518, 0xeeb7eca8,
	},
	{
		0x4ec97a20b6c4c7c2, 0x5913b1cd454f29fd, 0xa9629f9daf06d685, 0x852c9499156a8f3,
		0x3a180a6abfb79016, 0x9fc3c4764037c3c9, 0x2890c42fc0d972cf, 0x852c9499156a8f3,
		0x3a180a6abfb79016, 0x9fc3c4764037c3c9, 0x2890c42fc0d972cf, 0x1f92231d4e537651,
		0xfab8bb07aa54b7b9, 0xe05d2d771c485ed4, 0xd50b34bf808ca731, 0x8112bb9,
	},
	{
		0x5c3323628435a2e8, 0x1bea45ce9e72a6e3, 0x904f0a7027ddb52e, 0x939f31de14dcdc7b,
		0xa68fdf4379df068, 0xf169e1f0b835279d, 0x7498e432f9619b27, 0x939f31de14dcdc7b,
		0xa68fdf4379df068, 0xf169e1f0b835279d, 0x7498e432f9619b27, 0x1aa2a1f11088e785,
		0xd6ad72f45729de78, 0x9a63814157c80267, 0x55538e35c648e435, 0x85a6d477,
	},
	{
		0xc1ef26bea260abdb, 0x6ee423f2137f9280, 0xdf2118b946ed0b43, 0x11b87fb1b900cc39,
		0xe33e59b90dd815b1, 0xaa6cb5c4bafae741, 0x739699951ca8c713, 0x11b87fb1b900cc39,
		0xe33e59b90dd815b1, 0xaa6cb5c4bafae741, 0x739699951ca8c713, 0x2b4389a967310077,
		0x1d5382568a31c2c9, 0x55d1e787fbe68991, 0x277c254bc31301e7, 0x56f76c84,
	},
	{
		0x6be7381b115d653a, 0xed046190758ea511, 0xde6a45ffc3ed1159, 0xa64760e4041447d0,
		0xe3eac49f3e0c5109, 0xdd86c4d4cb6258e2, 0xefa9857afd046c7f, 0xa64760e4041447d0,
		0xe3eac49f3e0c5109, 0xdd86c4d4cb6258e2, 0xefa9857afd046c7f, 0xfab793dae8246f16,
		0xc9e3b121b31d094c, 0xa2a0f55858465226, 0xdba6f0ff39436344, 0x9af45d55,
	},
	{
		0xae3eece1711b2105, 0x14fd3f4027f81a4a, 0xabb7e45177d151db, 0x501f3e9b18861e44,
		0x465201170074e7d8, 0x96d5c91970f2cb12, 0x40fd28c43506c95d, 0x501f3e9b18861e44,
		0x465201170074e7d8, 0x96d5c91970f2cb12, 0x40fd28c43506c95d, 0xe86c4b07802aaff3,
		0xf317d14112372a70, 0x641b13e587711650, 0x4915421ab1090eaa, 0xd1c33760,
	},
	{
		0x376c28588b8fb389, 0x6b045e84d8491ed2, 0x4e857effb7d4e7dc, 0x154dd79fd2f984b4,
		0xf11171775622c1c3, 0x1fbe30982e78e6f0, 0xa460a15dcf327e44, 0x154dd79fd2f984b4,
		0xf11171775622c1c3, 0x1fbe30982e78e6f0, 0xa460a15dcf327e44, 0xf359e0900cc3d582,
		0x7e11070447976d00, 0x324e6daf276ea4b5, 0x7aa6e2df0cc94fa2, 0xc56bbf69,
	},
	{
		0x58d943503bb6748f, 0x419c6c8e88ac70f6, 0x586760cbf3d3d368, 0xb7e164979d5ccfc1,
		0x12cb4230d26bf286, 0xf1bf910d44bd84cb, 0xb32c24c6a40272, 0xb7e164979d5ccfc1,
		0x12cb4230d26bf286, 0xf1bf910d44bd84cb, 0xb32c24c6a40272, 0x11ed12e34c48c039,
		0xb0c2538e51d0a6ac, 0x4269bb773e1d553a, 0xe35a9dbabd34867, 0xabecfb9b,
	},
	{
		0xdfff5989f5cfd9a1, 0xbcee2e7ea3a96f83, 0x681c7874adb29017, 0x3ff6c8ac7c36b63a,
		0x48bc8831d849e326, 0x30b078e76b0214e2, 0x42954e6ad721b920, 0x3ff6c8ac7c36b63a,
		0x48bc8831d849e326, 0x30b078e76b0214e2, 0x42954e6ad721b920, 0xf9aeb33d164b4472,
		0x7b353b110831dbdc, 0x16f64c82f44ae17b, 0xb71244cc164b3b2b, 0x8de13255,
	},
	{
		0x7fb19eb1a496e8f5, 0xd49e5dfdb5c0833f, 0xc0d5d7b2f7c48dc7, 0x1a57313a32f22dde,
		0x30af46e49850bf8b, 0xaa0fe8d12f808f83, 0x443e31d70873bb6b, 0x1a57313a32f22dde,
		0x30af46e49850bf8b, 0xaa0fe8d12f808f83, 0x443e31d70873bb6b, 0xbbeb67c49c9fdc13,
		0x18f1e2a88f59f9d5, 0xfb1b05038e5def11, 0xd0450b5ce4c39c52, 0xa98ee299,
	},
	{
		0x5dba5b0dadccdbaa, 0x4ba8da8ded87fcdc, 0xf693fdd25badf2f0, 0xe9029e6364286587,
		0xae69f49ecb46726c, 0x18e002679217c405, 0xbd6d66e85332ae9f, 0xe9029e6364286587,
		0xae69f49ecb46726c, 0x18e002679217c405, 0xbd6d66e85332ae9f, 0x6bf330b1c353dd2a,
		0x74e9f2e71e3a4152, 0x3f85560b50f6c413, 0xd33a52a47eaed2b4, 0x3015f556,
	},
	{
		0x688bef4b135a6829, 0x8d31d82abcd54e8e, 0xf95f8a30d55036d7, 0x3d8c90e27aa2e147,
		0x2ec937ce0aa236b4, 0x89b563996d3a0b78, 0x39b02413b23c3f08, 0x3d8c90e27aa2e147,
		0x2ec937ce0aa236b4, 0x89b563996d3a0b78, 0x39b02413b23c3f08, 0x8d475a2e64faf2d2,
		0x48567f7dca46ecaf, 0x254cda08d5f87a6d, 0xec6ae9f729c47039, 0x5a430e29,
	},
	{
		0xd8323be05433a412, 0x8d48fa2b2b76141d, 0x3d346f23978336a5, 0x4d50c7537562033f,
		0x57dc7625b61dfe89, 0x9723a9f4c08ad93a, 0x5309596f48ab456b, 0x4d50c7537562033f,
		0x57dc7625b61dfe89, 0x9723a9f4c08ad93a, 0x5309596f48ab456b, 0x7e453088019d220f,
		0x8776067ba6ab9714, 0x67e1d06bd195de39, 0x74a1a32f8994b918, 0x2797add0,
	},
	{
		0x3b5404278a55a7fc, 0x23ca0b327c2d0a81, 0xa6d65329571c892c, 0x45504801e0e6066b,
		0x86e6c6d6152a3d04, 0x4f3db1c53eca2952, 0xd24d69b3e9ef10f3, 0x45504801e0e6066b,
		0x86e6c6d6152a3d04, 0x4f3db1c53eca2952, 0xd24d69b3e9ef10f3, 0x93a0de2219e66a70,
		0x8932c7115ccb1f8a, 0x5ef503fdf2841a8c, 0x38064dd9efa80a41, 0x27d55016,
	},
	{
		0x2a96a3f96c5e9bbc, 0x8caf8566e212dda8, 0x904de559ca16e45e, 0xf13bc2d9c2fe222e,
		0xbe4ccec9a6cdccfd, 0x37b2cbdd973a3ac9, 0x7b3223cd9c9497be, 0xf13bc2d9c2fe222e,
		0xbe4ccec9a6cdccfd, 0x37b2cbdd973a3ac9, 0x7b3223cd9c9497be, 0xd5904440f376f889,
		0x62b13187699c473c, 0x4751b89251f26726, 0x9500d84fa3a61ba8, 0x84945a82,
	},
	{
		0x22bebfdcc26d18ff, 0x4b4d8dcb10807ba1, 0x40265eee30c6b896, 0x3752b423073b119a,
		0x377dc5eb7c662bdb, 0x2b9f07f93a6c25b9, 0x96f24ede2bdc0718, 0x3752b423073b119a,
		0x377dc5eb7c662bdb, 0x2b9f07f93a6c25b9, 0x96f24ede2bdc0718, 0xf7699b12c31417bd,
		0x17b366f401c58b2, 0xbf60188d5f437b37, 0x484436e56df17f04, 0x3ef7e224,
	},
	{
		0x627a2249ec6bbcc2, 0xc0578b462a46735a, 0x4974b8ee1c2d4f1f, 0xebdbb918eb6d837f,
		0x8fb5f218dd84147c, 0xc77dd1f881df2c54, 0x62eac298ec226dc3, 0xebdbb918eb6d837f,
		0x8fb5f218dd84147c, 0xc77dd1f881df2c54, 0x62eac298ec226dc3, 0x43eded83c4b60bd0,
		0x9a0a403b5487503b, 0x25f305d9147f0bda, 0x3ad417f511bc1e64, 0x35ed8dc8,
	},
	{
		0x3abaf1667ba2f3e0, 0xee78476b5eeadc1, 0x7e56ac0a6ca4f3f4, 0xf1b9b413df9d79ed,
		0xa7621b6fd02db503, 0xd92f7ba9928a4ffe, 0x53f56babdcae96a6, 0xf1b9b413df9d79ed,
		0xa7621b6fd02db503, 0xd92f7ba9928a4ffe, 0x53f56babdcae96a6, 0x5302b89fc48713ab,
		0xd03e3b04dbe7a2f2, 0xfa74ef8af6d376a7, 0x103c8cdea1050ef2, 0x6a75e43d,
	},
	{
		0x3931ac68c5f1b2c9, 0xefe3892363ab0fb0, 0x40b707268337cd36, 0xa53a6b64b1ac85c9,
		0xd50e7f86ee1b832b, 0x7bab08fdd26ba0a4, 0x7587743c18fe2475, 0xa53a6b64b1ac85c9,
		0xd50e7f86ee1b832b, 0x7bab08fdd26ba0a4, 0x7587743c18fe2475, 0xe3b5d5d490cf5761,
		0xdfc053f7d065edd5, 0x42ffd8d5fb70129f, 0x599ca38677cccdc3, 0x235d9805,
	},
	{
		0xb98fb0606f416754, 0x46a6e5547ba99c1e, 0xc909d82112a8ed2, 0xdbfaae9642b3205a,
		0xf676a1339402bcb9, 0xf4f12a5b1ac11f29, 0x7db8bad81249dee4, 0xdbfaae9642b3205a,
		0xf676a1339402bcb9, 0xf4f12a5b1ac11f29, 0x7db8bad81249dee4, 0xb26e46f2da95922e,
		0x2aaedd5e12e3c611, 0xa0e2d9082966074, 0xc64da8a167add63d, 0xf7d69572,
	},
	{
		0x7f7729a33e58fcc4, 0x2e4bc1e7a023ead4, 0xe707008ea7ca6222, 0x47418a71800334a0,
		0xd10395d8fc64d8a4, 0x8257a30062cb66f, 0x6786f9b2dc1ff18a, 0x47418a71800334a0,
		0xd10395d8fc64d8a4, 0x8257a30062cb66f, 0x6786f9b2dc1ff18a, 0x5633f437bb2f180f,
		0xe5a3a405737d22d6, 0xca0ff1ef6f7f0b74, 0xd0ae600684b16df8, 0xbacd0199,
	},
	{
		0x42a0aa9ce82848b3, 0x57232730e6bee175, 0xf89bb3f370782031, 0xcaa33cf9b4f6619c,
		0xb2c8648ad49c209f, 0x9e89ece0712db1c0, 0x101d8274a711a54b, 0xcaa33cf9b4f6619c,
		0xb2c8648ad49c209f, 0x9e89ece0712db1c0, 0x101d8274a711a54b, 0x538e79f1e70135cd,
		0xe1f5a76f983c844e, 0x653c082fd66088fc, 0x1b9c9b464b654958, 0xe428f50e,
	},
	{
		0x6b2c6d38408a4889, 0xde3ef6f68fb25885, 0x20754f456c203361, 0x941f5023c0c943f9,
		0xdfdeb9564fd66f24, 0x2140cec706b9d406, 0x7b22429b131e9c72, 0x941f5023c0c943f9,
		0xdfdeb9564fd66f24, 0x2140cec706b9d406, 0x7b22429b131e9c72, 0x94215c22eb940f45,
		0xd28b9ed474f7249a, 0x6f25e88f2fbf9f56, 0xb6718f9e605b38ac, 0x81eaaad3,
	},
	{
		0x930380a3741e862a, 0x348d28638dc71658, 0x89dedcfd1654ea0d, 0x7e7f61684080106,
		0x837ace9794582976, 0x5ac8ca76a357eb1b, 0x32b58308625661fb, 0x7e7f61684080106,
		0x837ace9794582976, 0x5ac8ca76a357eb1b, 0x32b58308625661fb, 0xc09705c4572025d9,
		0xf9187f6af0291303, 0x1c0edd8ee4b02538, 0xe6cb105daa0578a, 0xaddbd3e3,
	},
	{
		0x94808b5d2aa25f9a, 0xcec72968128195e0, 0xd9f4da2bdc1e130f, 0x272d8dd74f3006cc,
		0xec6c2ad1ec03f554, 0x4ad276b249a5d5dd, 0x549a22a17c0cde12, 0x272d8dd74f3006cc,
		0xec6c2ad1ec03f554, 0x4ad276b249a5d5dd, 0x549a22a17c0cde12, 0x602119cb824d7cde,
		0xf4d3cef240ef35fa, 0xe889895e01911bc7, 0x785a7e5ac20e852b, 0xe66dbca0,
	},
	{
		0xb31abb08ae6e3d38, 0x9eb9a95cbd9e8223, 0x8019e79b7ee94ea9, 0x7b2271a7a3248e22,
		0x3b4f700e5a0ba523, 0x8ebc520c227206fe, 0xda3f861490f5d291, 0x7b2271a7a3248e22,
		0x3b4f700e5a0ba523, 0x8ebc520c227206fe, 0xda3f861490f5d291, 0xd08a689f9f3aa60e,
		0x547c1b97a068661f, 0x4b15a67fa29172f0, 0xeaf40c085191d80f, 0xafe11fd5,
	},
	{
		0xdccb5534a893ea1a, 0xce71c398708c6131, 0xfe2396315457c164, 0x3f1229f4d0fd96fb,
		0x33130aa5fa9d43f2, 0xe42693d5b34e63ab, 0x2f4ef2be67f62104, 0x3f1229f4d0fd96fb,
		0x33130aa5fa9d43f2, 0xe42693d5b34e63ab, 0x2f4ef2be67f62104, 0x372e5153516e37b9,
		0xaf9ec142ab12cc86, 0x777920c09345e359, 0xe7c4a383bef8adc6, 0xa71a406f,
	},
	{
		0x6369163565814de6, 0x8feb86fb38d08c2f, 0x4976933485cc9a20, 0x7d3e82d5ba29a90d,
		0xd5983cc93a9d126a, 0x37e9dfd950e7b692, 0x80673be6a7888b87, 0x7d3e82d5ba29a90d,
		0xd5983cc93a9d126a, 0x37e9dfd950e7b692, 0x80673be6a7888b87, 0x57f732dc600808bc,
		0x59477199802cc78b, 0xf824810eb8f2c2de, 0xc4a3437f05b3b61c, 0x9d90eaf5,
	},
	{
		0xedee4ff253d9f9b3, 0x96ef76fb279ef0ad, 0xa4d204d179db2460, 0x1f3dcdfa513512d6,
		0x4dc7ec07283117e4, 0x4438bae88ae28bf9, 0xaa7eae72c9244a0d, 0x1f3dcdfa513512d6,
		0x4dc7ec07283117e4, 0x4438bae88ae28bf9, 0xaa7eae72c9244a0d, 0xb9aedc8d3ecc72df,
		0xb75a8eb090a77d62, 0x6b15677f9cd91507, 0x51d8282cb3a9ddbf, 0x6665db10,
	},
	{
		0x941993df6e633214, 0x929bc1beca5b72c6, 0x141fc52b8d55572d, 0xb3b782ad308f21ed,
		0x4f2676485041dee0, 0xbfe279aed5cb4bc8, 0x2a62508a467a22ff, 0xb3b782ad308f21ed,
		0x4f2676485041dee0, 0xbfe279aed5cb4bc8, 0x2a62508a467a22ff, 0xe74d29eab742385d,
		0x56b05cd90ecfc293, 0xc603728ea73f8844, 0x8638fcd21bc692c4, 0x9c977cbf,
	},
	{
		0x859838293f64cd4c, 0x484403b39d44ad79, 0xbf674e64d64b9339, 0x44d68afda9568f08,
		0x478568ed51ca1d65, 0x679c204ad3d9e766, 0xb28e788878488dc1, 0x44d68afda9568f08,
		0x478568ed51ca1d65, 0x679c204ad3d9e766, 0xb28e788878488dc1, 0xd001a84d3a84fae6,
		0xd376958fe4cb913e, 0x17435277e36c86f0, 0x23657b263c347aa6, 0xee83ddd4,
	},
	{
		0xc19b5648e0d9f555, 0x328e47b2b7562993, 0xe756b92ba4bd6a51, 0xc3314e362764ddb8,
		0x6481c084ee9ec6b5, 0xede23fb9a251771, 0xbd617f2643324590, 0xc3314e362764ddb8,
		0x6481c084ee9ec6b5, 0xede23fb9a251771, 0xbd617f2643324590, 0xd2d30c9b95e030f5,
		0x8a517312ffc5795e, 0x8b1f325033bd535e, 0x3ee6e867e03f2892, 0x26519cc,
	},
	{
		0xf963b63b9006c248, 0x9e9bf727ffaa00bc, 0xc73bacc75b917e3a, 0x2c6aa706129cc54c,
		0x17a706f59a49f086, 0xc7c1eec455217145, 0x6adfdc6e07602d42, 0x2c6aa706129cc54c,
		0x17a706f59a49f086, 0xc7c1eec455217145, 0x6adfdc6e07602d42, 0xfb75fca30d848dd2,
		0x5228c9ed14653ed4, 0x953958910153b1a2, 0xa430103a24f42a5d, 0xa485a53f,
	},
	{
		0x6a8aa0852a8c1f3b, 0xc8f1e5e206a21016, 0x2aa554aed1ebb524, 0xfc3e3c322cd5d89b,
		0xb7e3911dc2bd4ebb, 0xfcd6da5e5fae833a, 0x51ed3c41f87f9118, 0xfc3e3c322cd5d89b,
		0xb7e3911dc2bd4ebb, 0xfcd6da5e5fae833a, 0x51ed3c41f87f9118, 0xf31750cbc19c420a,
		0x186dab1abada1d86, 0xca7f88cb894b3cd7, 0x2859eeb1c373790c, 0xf62bc412,
	},
	{
		0x740428b4d45e5fb8, 0x4c95a4ce922cb0a5, 0xe99c3ba78feae796, 0x914f1ea2fdcebf5c,
		0x9566453c07cd0601, 0x9841bf66d0462cd, 0x79140c1c18536aeb, 0x914f1ea2fdcebf5c,
		0x9566453c07cd0601, 0x9841bf66d0462cd, 0x79140c1c18536aeb, 0xa963b930b05820c2,
		0x6a7d9fa0c8c45153, 0x64214c40d07cf39b, 0x7057daf1d806c014, 0x8975a436,
	},
	{
		0x658b883b3a872b86, 0x2f0e303f0f64827a, 0x975337e23dc45e1, 0x99468a917986162b,
		0x7b31434aac6e0af0, 0xf6915c1562c7d82f, 0xe4071d82a6dd71db, 0x99468a917986162b,
		0x7b31434aac6e0af0, 0xf6915c1562c7d82f, 0xe4071d82a6dd71db, 0x5f5331f077b5d996,
		0x7b314ba21b747a4f, 0x5a73cb9521da17f5, 0x12ed435fae286d86, 0x94ff7f41,
	},
	{
		0x6df0a977da5d27d4, 0x891dd0e7cb19508, 0xfd65434a0b71e680, 0x8799e4740e573c50,
		0x9e739b52d0f341e8, 0xcdfd34ba7d7b03eb, 0x5061812ce6c88499, 0x8799e4740e573c50,
		0x9e739b52d0f341e8, 0xcdfd34ba7d7b03eb, 0x5061812ce6c88499, 0x612b8d8f2411dc5c,
		0x878bd883d29c7787, 0x47a846727182bb, 0xec4949508c8b3b9a, 0x760aa031,
	},
	{
		0xa900275464ae07ef, 0x11f2cfda34beb4a3, 0x9abf91e5a1c38e4, 0x8063d80ab26f3d6d,
		0x4177b4b9b4f0393f, 0x6de42ba8672b9640, 0xd0bccdb72c51c18, 0x8063d80ab26f3d6d,
		0x4177b4b9b4f0393f, 0x6de42ba8672b9640, 0xd0bccdb72c51c18, 0xaf3f611b7f22cf12,
		0x3863c41492645755, 0x928c7a616a8f14f9, 0xa82c78eb2eadc58b, 0x3bda76df,
	},
	{
		0x810bc8aa0c40bcb0, 0x448a019568d01441, 0xf60ec52f60d3aeae, 0x52c44837aa6dfc77,
		0x15d8d8fccdd6dc5b, 0x345b793ccfa93055, 0x932160fe802ca975, 0x52c44837aa6dfc77,
		0x15d8d8fccdd6dc5b, 0x345b793ccfa93055, 0x932160fe802ca975, 0xa624b0dd93fc18cd,
		0xd955b254c2037f1e, 0xe540533d370a664c, 0x2ba4ec12514e9d7, 0x498e2e65,
	},
	{
		0x22036327deb59ed7, 0xadc05ceb97026a02, 0x48bff0654262672b, 0xc791b313aba3f258,
		0x443c7757a4727bee, 0xe30e4b2372171bdf, 0xf3db986c4156f3cb, 0xc791b313aba3f258,
		0x443c7757a4727bee, 0xe30e4b2372171bdf, 0xf3db986c4156f3cb, 0xa939aefab97c6e15,
		0xdbeb8acf1d5b0e6c, 0x1e0eab667a795bba, 0x80dd539902df4d50, 0xd38deb48,
	},
	{
		0x7d14dfa9772b00c8, 0x595735efc7eeaed7, 0x29872854f94c3507, 0xbc241579d8348401,
		0x16dc832804d728f0, 0xe9cc71ae64e3f09e, 0xbef634bc978bac31, 0xbc241579d8348401,
		0x16dc832804d728f0, 0xe9cc71ae64e3f09e, 0xbef634bc978bac31, 0x7f64b1fa2a9129e,
		0x71d831bd530ac7f3, 0xc7ad0a8a6d5be6f1, 0x82a7d3a815c7aaab, 0x82b3fb6b,
	},
	{
		0x2d777cddb912675d, 0x278d7b10722a13f9, 0xf5c02bfb7cc078af, 0x4283001239888836,
		0xf44ca39a6f79db89, 0xed186122d71bcc9f, 0x8620017ab5f3ba3b, 0x4283001239888836,
		0xf44ca39a6f79db89, 0xed186122d71bcc9f, 0x8620017ab5f3ba3b, 0xe787472187f176c,
		0x267e64c4728cf181, 0xf1ba4b3007c15e30, 0x8e3a75d5b02ecfc0, 0xe500e25f,
	},
	{
		0xf2ec98824e8aa613, 0x5eb7e3fb53fe3bed, 0x12c22860466e1dd4, 0x374dd4288e0b72e5,
		0xff8916db706c0df4, 0xcb1a9e85de5e4b8d, 0xd4d12afb67a27659, 0x374dd4288e0b72e5,
		0xff8916db706c0df4, 0xcb1a9e85de5e4b8d, 0xd4d12afb67a27659, 0xfeb69095d1ba175a,
		0xe2003aab23a47fad, 0x8163a3ecab894b49, 0x46d356674ce041f6, 0xbd2bb07c,
	},
	{
		0x5e763988e21f487f, 0x24189de8065d8dc5, 0xd1519d2403b62aa0, 0x9136456740119815,
		0x4d8ff7733b27eb83, 0xea3040bc0c717ef8, 0x7617ab400dfadbc, 0x9136456740119815,
		0x4d8ff7733b27eb83, 0xea3040bc0c717ef8, 0x7617ab400dfadbc, 0xfb336770c10b17a1,
		0x6123b68b5b31f151, 0x1e147d5f295eccf2, 0x9ecbb1333556f977, 0x3a2b431d,
	},
	{
		0x48949dc327bb96ad, 0xe1fd21636c5c50b4, 0x3f6eb7f13a8712b4, 0x14cf7f02dab0eee8,
		0x6d01750605e89445, 0x4f1cf4006e613b78, 0x57c40c4db32bec3b, 0x14cf7f02dab0eee8,
		0x6d01750605e89445, 0x4f1cf4006e613b78, 0x57c40c4db32bec3b, 0x1fde5a347f4a326e,
		0xcb5a54308adb0e3f, 0x14994b2ba447a23c, 0x7067d0abb4257b68, 0x7322a83d,
	},
	{
		0xb7c4209fb24a85c5, 0xb35feb319c79ce10, 0xf0d3de191833b922, 0x570d62758ddf6397,
		0x5e0204fb68a7b800, 0x4383a9236f8b5a2b, 0x7bc1a64641d803a4, 0x570d62758ddf6397,
		0x5e0204fb68a7b800, 0x4383a9236f8b5a2b, 0x7bc1a64641d803a4, 0x5434d61285099f7a,
		0xd49449aacdd5dd67, 0x97855ba0e9a7d75d, 0xda67328062f3a62f, 0xa645ca1c,
	},
	{
		0x9c9e5be0943d4b05, 0xb73dc69e45201cbb, 0xaab17180bfe5083d, 0xc738a77a9a55f0e2,
		0x705221addedd81df, 0xfd9bd8d397abcfa3, 0x8ccf0004aa86b795, 0xc738a77a9a55f0e2,
		0x705221addedd81df, 0xfd9bd8d397abcfa3, 0x8ccf0004aa86b795, 0x2bb5db2280068206,
		0x8c22d29f307a01d, 0x274a22de02f473c8, 0xb8791870f4268182, 0x8909a45a,
	},
	{
		0x3898bca4dfd6638d, 0xf911ff35efef0167, 0x24bdf69e5091fc88, 0x9b82567ab6560796,
		0x891b69462b41c224, 0x8eccc7e4f3af3b51, 0x381e54c3c8f1c7d0, 0x9b82567ab6560796,
		0x891b69462b41c224, 0x8eccc7e4f3af3b51, 0x381e54c3c8f1c7d0, 0xc80fbc489a558a55,
		0x1ba88e062a663af7, 0xaf7b1ef1c0116303, 0xbd20e1a5a6b1a0cd, 0xbd30074c,
	},
	{
		0x5b5d2557400e68e7, 0x98d610033574cee, 0xdfd08772ce385deb, 0x3c13e894365dc6c2,
		0x26fc7bbcda3f0ef, 0xdbb71106cdbfea36, 0x785239a742c6d26d, 0x3c13e894365dc6c2,
		0x26fc7bbcda3f0ef, 0xdbb71106cdbfea36, 0x785239a742c6d26d, 0xf810c415ae05b2f4,
		0xbb9b9e7398526088, 0x70128f1bf830a32b, 0xbcc73f82b6410899, 0xc17cf001,
	},
	{
		0xa927ed8b2bf09bb6, 0x606e52f10ae94eca, 0x71c2203feb35a9ee, 0x6e65ec14a8fb565,
		0x34bff6f2ee5a7f79, 0x2e329a5be2c011b, 0x73161c93331b14f9, 0x6e65ec14a8fb565,
		0x34bff6f2ee5a7f79, 0x2e329a5be2c011b, 0x73161c93331b14f9, 0x15d13f2408aecf88,
		0x9f5b61b8a4b55b31, 0x8fe25a43b296dba6, 0xbdad03b7300f284e, 0x26ffd25a,
	},
	{
		0x8d25746414aedf28, 0x34b1629d28b33d3a, 0x4d5394aea5f82d7b, 0x379f76458a3c8957,
		0x79dd080f9843af77, 0xc46f0a7847f60c1d, 0xaf1579c5797703cc, 0x379f76458a3c8957,
		0x79dd080f9843af77, 0xc46f0a7847f60c1d, 0xaf1579c5797703cc, 0x8b7d31f338755c14,
		0x2eff97679512aaa8, 0xdf07d68e075179ed, 0xc8fa6c7a729e7f1f, 0xf1d8ce3c,
	},
	{
		0xb5bbdb73458712f2, 0x1ff887b3c2a35137, 0x7f7231f702d0ace9, 0x1e6f0910c3d25bd8,
		0xad9e250862102467, 0x1c842a07abab30cd, 0xcd8124176bac01ac, 0x1e6f0910c3d25bd8,
		0xad9e250862102467, 0x1c842a07abab30cd, 0xcd8124176bac01ac, 0xea6ebe7a79b67edc,
		0x73f598ac9db26713, 0x4f4e72d7460b8fc, 0x365dc4b9fdf13f21, 0x3ee8fb17,
	},
	{
		0x3d32a26e3ab9d254, 0xfc4070574dc30d3a, 0xf02629579c2b27c9, 0xb1cf09b0184a4834,
		0x5c03db48eb6cc159, 0xf18c7fcf34d1df47, 0xdfb043419ecf1fa9, 0xb1cf09b0184a4834,
		0x5c03db48eb6cc159, 0xf18c7fcf34d1df47, 0xdfb043419ecf1fa9, 0xdcd78d13f9ca658f,
		0x4355d408ffe8e49f, 0x81eefee908b593b4, 0x590c213c20e981a3, 0xa77acc2a,
	},
	{
		0x9371d3c35fa5e9a5, 0x42967cf4d01f30, 0x652d1eeae704145c, 0xceaf1a0d15234f15,
		0x1450a54e45ba9b9, 0x65e9c1fd885aa932, 0x354d4bc034ba8cbe, 0xceaf1a0d15234f15,
		0x1450a54e45ba9b9, 0x65e9c1fd885aa932, 0x354d4bc034ba8cbe, 0x8fd4ff484c08fb4b,
		0xbf46749866f69ba0, 0xcf1c21ede82c9477, 0x4217548c43da109, 0xf4556dee,
	},
	{
		0xcbaa3cb8f64f54e0, 0x76c3b48ee5c08417, 0x9f7d24e87e61ce9, 0x85b8e53f22e19507,
		0xbb57137739ca486b, 0xc77f131cca38f761, 0xc56ac3cf275be121, 0x85b8e53f22e19507,
		0xbb57137739ca486b, 0xc77f131cca38f761, 0xc56ac3cf275be121, 0x9ec1a6c9109d2685,
		0x3dad0922e76afdb0, 0xfd58cbf952958103, 0x7b04c908e78639a1, 0xde287a64,
	},
	{
		0xb2e23e8116c2ba9f, 0x7e4d9c0060101151, 0x3310da5e5028f367, 0xadc52dddb76f6e5e,
		0x4aad4e925a962b68, 0x204b79b7f7168e64, 0xdf29ed6671c36952, 0xadc52dddb76f6e5e,
		0x4aad4e925a962b68, 0x204b79b7f7168e64, 0xdf29ed6671c36952, 0xe02927cac396d210,
		0x5d500e71742b638a, 0x5c9998af7f27b124, 0x3fba9a2573dc2f7, 0x878e55b9,
	},
	{
		0x8aa77f52d7868eb9, 0x4d55bd587584e6e2, 0xd2db37041f495f5, 0xce030d15b5fe2f4,
		0x86b4a7a0780c2431, 0xee070a9ae5b51db7, 0xedc293d9595be5d8, 0xce030d15b5fe2f4,
		0x86b4a7a0780c2431, 0xee070a9ae5b51db7, 0xedc293d9595be5d8, 0x3dfc5ec108260a2b,
		0x8afe28c7123bf4e2, 0xda82ef38023a7a5f, 0x3e1f77b0174b77c3, 0x7648486,
	},
	{
		0x858fea922c7fe0c3, 0xcfe8326bf733bc6f, 0x4e5e2018cf8f7dfc, 0x64fd1bc011e5bab7,
		0x5c9e858728015568, 0x97ac42c2b00b29b1, 0x7f89caf08c109aee, 0x64fd1bc011e5bab7,
		0x5c9e858728015568, 0x97ac42c2b00b29b1, 0x7f89caf08c109aee, 0x9a8af34fd0e9dacf,
		0xbbc54161aa1507e0, 0x7cda723ccbbfe5ee, 0x2c289d839fb93f58, 0x57ac0fb1,
	},
	{
		0x46ef25fdec8392b1, 0xe48d7b6d42a5cd35, 0x56a6fe1c175299ca, 0xfdfa836b41dcef62,
		0x2f8db8030e847e1b, 0x5ba0a49ac4f9b0f8, 0xdae897ed3e3fce44, 0xfdfa836b41dcef62,
		0x2f8db8030e847e1b, 0x5ba0a49ac4f9b0f8, 0xdae897ed3e3fce44, 0x9c432e31aef626e7,
		0x9a36e1c6cd6e3dd, 0x5095a167c34d19d, 0xa70005cfa6babbea, 0xd01967ca,
	},
	{
		0x8d078f726b2df464, 0xb50ee71cdcabb299, 0xf4af300106f9c7ba, 0x7d222caae025158a,
		0xcc028d5fd40241b9, 0xdd42515b639e6f97, 0xe08e86531a58f87f, 0x7d222caae025158a,
		0xcc028d5fd40241b9, 0xdd42515b639e6f97, 0xe08e86531a58f87f, 0xd93612c835b37d7b,
		0x91dd61729b2fa7f4, 0xba765a1bdda09db7, 0x55258b451b2b1297, 0x96ecdf74,
	},
	{
		0x35ea86e6960ca950, 0x34fe1fe234fc5c76, 0xa00207a3dc2a72b7, 0x80395e48739e1a67,
		0x74a67d8f7f43c3d7, 0xdd2bdd1d62246c6e, 0xa1f44298ba80acf6, 0x80395e48739e1a67,
		0x74a67d8f7f43c3d7, 0xdd2bdd1d62246c6e, 0xa1f44298ba80acf6, 0xad86d86c187bf38,
		0x26feea1f2eee240d, 0xed7f1fd066b23897, 0xa768cf1e0fbb502, 0x779f5506,
	},
	{
		0x8aee9edbc15dd011, 0x51f5839dc8462695, 0xb2213e17c37dca2d, 0x133b299a939745c5,
		0x796e2aac053f52b3, 0xe8d9fe1521a4a222, 0x819a8863e5d1c290, 0x133b299a939745c5,
		0x796e2aac053f52b3, 0xe8d9fe1521a4a222, 0x819a8863e5d1c290, 0xc0737f0fe34d36ad,
		0xe6d6d4a267a5cc31, 0x98300a7911674c23, 0xbef189661c257098, 0x3c94c2de,
	},
	{
		0xc3e142ba98432dda, 0x911d060cab126188, 0xb753fbfa8365b844, 0xfd1a9ba5e71b08a2,
		0x7ac0dc2ed7778533, 0xb543161ff177188a, 0x492fc08a6186f3f4, 0xfd1a9ba5e71b08a2,
		0x7ac0dc2ed7778533, 0xb543161ff177188a, 0x492fc08a6186f3f4, 0xfc4745f516afd3b6,
		0x88c30370a53080e, 0x65a1bb34abc465e2, 0xabbd14662911c8b3, 0x39f98faf,
	},
	{
		0x123ba6b99c8cd8db, 0x448e582672ee07c4, 0xcebe379292db9e65, 0x938f5bbab544d3d6,
		0xd2a95f9f2d376d73, 0x68b2f16149e81aa3, 0xad7e32f82d86c79d, 0x938f5bbab544d3d6,
		0xd2a95f9f2d376d73, 0x68b2f16149e81aa3, 0xad7e32f82d86c79d, 0x4574015ae8626ce2,
		0x455aa6137386a582, 0x658ad2542e8ec20, 0xe31d7be2ca35d00, 0x7af31199,
	},
	{
		0xba87acef79d14f53, 0xb3e0fcae63a11558, 0xd5ac313a593a9f45, 0xeea5f5a9f74af591,
		0x578710bcc36fbea2, 0x7a8393432188931d, 0x705cfc5ec7cc172, 0xeea5f5a9f74af591,
		0x578710bcc36fbea2, 0x7a8393432188931d, 0x705cfc5ec7cc172, 0xda85ebe5fc427976,
		0xbfa5c7a454df54c8, 0x4632b72a81bf66d2, 0x5dd72877db539ee2, 0xe341a9d6,
	},
	{
		0xbcd3957d5717dc3, 0x2da746741b03a007, 0x873816f4b1ece472, 0x2b826f1a2c08c289,
		0xda50f56863b55e74, 0xb18712f6b3eed83b, 0xbdc7cc05ab4c685f, 0x2b826f1a2c08c289,
		0xda50f56863b55e74, 0xb18712f6b3eed83b, 0xbdc7cc05ab4c685f, 0x9e45fb833d1b0af,
		0xd7213081db29d82e, 0xd2a6b6c6a09ed55e, 0x98a7686cba323ca9, 0xca24aeeb,
	},
	{
		0x61442ff55609168e, 0x6447c5fc76e8c9cf, 0x6a846de83ae15728, 0xeffc2663cffc777f,
		0x93214f8f463afbed, 0xa156ef06066f4e4e, 0xa407b6ed8769d51e, 0xeffc2663cffc777f,
		0x93214f8f463afbed, 0xa156ef06066f4e4e, 0xa407b6ed8769d51e, 0xbb2f9ed29745c02a,
		0x981eecd435b36ad9, 0x461a5a05fb9cdff4, 0xbd6cb2a87b9f910c, 0xb2252b57,
	},
	{
		0xdbe4b1b2d174757f, 0x506512da18712656, 0x6857f3e0b8dd95f, 0x5a4fc2728a9bb671,
		0xebb971522ec38759, 0x1a5a093e6cf1f72b, 0x729b057fe784f504, 0x5a4fc2728a9bb671,
		0xebb971522ec38759, 0x1a5a093e6cf1f72b, 0x729b057fe784f504, 0x71fcbf42a767f9cf,
		0x114cfe772da6cdd, 0x60cdf9cb629d9d7a, 0xe270d10ad088b24e, 0x72c81da1,
	},
	{
		0x531e8e77b363161c, 0xeece0b43e2dae030, 0x8294b82c78f34ed1, 0xe777b1fd580582f2,
		0x7b880f58da112699, 0x562c6b189a6333f4, 0x139d64f88a611d4, 0xe777b1fd580582f2,
		0x7b880f58da112699, 0x562c6b189a6333f4, 0x139d64f88a611d4, 0x53d8ef17eda64fa4,
		0xbf3eded14dc60a04, 0x2b5c559cf5ec07c5, 0x8895f7339d03a48a, 0x6b9fce95,
	},
	{
		0xf71e9c926d711e2b, 0xd77af2853a4ceaa1, 0x9aa0d6d76a36fae7, 0xdd16cd0fbc08393,
		0x29a414a5d8c58962, 0x72793d8d1022b5b2, 0x2e8e69cf7cbffdf0, 0xdd16cd0fbc08393,
		0x29a414a5d8c58962, 0x72793d8d1022b5b2, 0x2e8e69cf7cbffdf0, 0x3721c0473aa99c9a,
		0x1cff4ed9c31cd91c, 0x4990735033cc482b, 0x7fdf8c701c72f577, 0x19399857,
	},
	{
		0xcb20ac28f52df368, 0xe6705ee7880996de, 0x9b665cc3ec6972f2, 0x4260e8c254e9924b,
		0xf197a6eb4591572d, 0x8e867ff0fb7ab27c, 0xf95502fb503efaf3, 0x4260e8c254e9924b,
		0xf197a6eb4591572d, 0x8e867ff0fb7ab27c, 0xf95502fb503efaf3, 0x30c41876b08e3e22,
		0x958e2419e3cd22f4, 0xf0f3aa1fe119a107, 0x481662310a379100, 0x3c57a994,
	},
	{
		0xe4a794b4acb94b55, 0x89795358057b661b, 0x9c4cdcec176d7a70, 0x4890a83ee435bc8b,
		0xd8c1c00fceb00914, 0x9e7111ba234f900f, 0xeb8dbab364d8b604, 0x4890a83ee435bc8b,
		0xd8c1c00fceb00914, 0x9e7111ba234f900f, 0xeb8dbab364d8b604, 0xb3261452963eebb,
		0x6cf94b02792c4f95, 0xd88fa815ef1e8fc, 0x2d687af66604c73, 0xc053e729,
	},
	{
		0xcb942e91443e7208, 0xe335de8125567c2a, 0xd4d74d268b86df1f, 0x8ba0fdd2ffc8b239,
		0xf413b366c1ffe02f, 0xc05b2717c59a8a28, 0x981188eab4fcc8fb, 0x8ba0fdd2ffc8b239,
		0xf413b366c1ffe02f, 0xc05b2717c59a8a28, 0x981188eab4fcc8fb, 0xe563f49a1d9072ba,
		0x3c6a3aa4a26367dc, 0xba0db13448653f34, 0x31065d756074d7d6, 0x51cbbba7,
	},
	{
		0xecca7563c203f7ba, 0x177ae2423ef34bb2, 0xf60b7243400c5731, 0xcf1edbfe7330e94e,
		0x881945906bcb3cc6, 0x4acf0293244855da, 0x65ae042c1c2a28c2, 0xcf1edbfe7330e94e,
		0x881945906bcb3cc6, 0x4acf0293244855da, 0x65ae042c1c2a28c2, 0xb25fa0a1cab33559,
		0xd98e8daa28124131, 0xfce17f50b9c351b3, 0x3f995ccf7386864b, 0x1acde79a,
	},
	{
		0x1652cb940177c8b5, 0x8c4fe7d85d2a6d6d, 0xf6216ad097e54e72, 0xf6521b912b368ae6,
		0xa9fe4eff81d03e73, 0xd6f623629f80d1a3, 0x2b9604f32cb7dc34, 0xf6521b912b368ae6,
		0xa9fe4eff81d03e73, 0xd6f623629f80d1a3, 0x2b9604f32cb7dc34, 0x2a43d84dcf59c7e2,
		0xd0a197c70c5dae0b, 0x6e84d4bbc71d76a0, 0xc7e94620378c6cb2, 0x2d160d13,
	},
	{
		0x31fed0fc04c13ce8, 0x3d5d03dbf7ff240a, 0x727c5c9b51581203, 0x6b5ffc1f54fecb29,
		0xa8e8e7ad5b9a21d9, 0xc4d5a32cd6aac22d, 0xd7e274ad22d4a79a, 0x6b5ffc1f54fecb29,
		0xa8e8e7ad5b9a21d9, 0xc4d5a32cd6aac22d, 0xd7e274ad22d4a79a, 0x368841ea5731a112,
		0xfeaf7bc2e73ca48f, 0x636fb272e9ea1f6, 0x5d9cb7580c3f6207, 0x787f5801,
	},
	{
		0xe7b668947590b9b3, 0xbaa41ad32938d3fa, 0xabcbc8d4ca4b39e4, 0x381ee1b7ea534f4e,
		0xda3759828e3de429, 0x3e015d76729f9955, 0xcbbec51a6485fbde, 0x381ee1b7ea534f4e,
		0xda3759828e3de429, 0x3e015d76729f9955, 0xcbbec51a6485fbde, 0x9b86605281f20727,
		0xfc6fcf508676982a, 0x3b135f7a813a1040, 0xd3a4706bea1db9c9, 0xc9629828,
	},
	{
		0x1de2119923e8ef3c, 0x6ab27c096cf2fe14, 0x8c3658edca958891, 0x4cc8ed3ada5f0f2,
		0x4a496b77c1f1c04e, 0x9085b0a862084201, 0xa1894bde9e3dee21, 0x4cc8ed3ada5f0f2,
		0x4a496b77c1f1c04e, 0x9085b0a862084201, 0xa1894bde9e3dee21, 0x367fb472dc5b277d,
		0x7d39ccca16fc6745, 0x763f988d70db9106, 0xa8b66f7fecb70f02, 0xbe139231,
	},
	{
		0x1269df1e69e14fa7, 0x992f9d58ac5041b7, 0xe97fcf695a7cbbb4, 0xe5d0549802d15008,
		0x424c134ecd0db834, 0x6fc44fd91be15c6c, 0xa1a5ef95d50e537d, 0xe5d0549802d15008,
		0x424c134ecd0db834, 0x6fc44fd91be15c6c, 0xa1a5ef95d50e537d, 0xd1e3daf5d05f5308,
		0x4c7f81600eaa1327, 0x109d1b8d1f9d0d2b, 0x871e8699e0aeb862, 0x7df699ef,
	},
	{
		0x820826d7aba567ff, 0x1f73d28e036a52f3, 0x41c4c5a73f3b0893, 0xaa0d74d4a98db89b,
		0x36fd486d07c56e1d, 0xd0ad23cbb6660d8a, 0x1264a84665b35e19, 0xaa0d74d4a98db89b,
		0x36fd486d07c56e1d, 0xd0ad23cbb6660d8a, 0x1264a84665b35e19, 0x789682bf7d781b33,
		0x6bfa6abd2fb5722d, 0x6779cb3623d33900, 0x435ca5214e1ee5f0, 0x8ce6b96d,
	},
	{
		0xffe0547e4923cef9, 0x3534ed49b9da5b02, 0x548a273700fba03d, 0x28ac84ca70958f7e,
		0xd8ae575a68faa731, 0x2aaaee9b9dcffd4c, 0x6c7faab5c285c6da, 0x28ac84ca70958f7e,
		0xd8ae575a68faa731, 0x2aaaee9b9dcffd4c, 0x6c7faab5c285c6da, 0x45d94235f99ba78f,
		0xab5ea16f39497f5b, 0xfb4d6c86fccbdca3, 0x8104e6310a5fd2c7, 0x6f9ed99c,
	},
	{
		0x72da8d1b11d8bc8b, 0xba94b56b91b681c6, 0x4e8cc51bd9b0fc8c, 0x43505ed133be672a,
		0xe8f2f9d973c2774e, 0x677b9b9c7cad6d97, 0x4e1f5d56ef17b906, 0x43505ed133be672a,
		0xe8f2f9d973c2774e, 0x677b9b9c7cad6d97, 0x4e1f5d56ef17b906, 0xeea3a6038f983767,
		0x87109f077f86db01, 0xecc1ca41f74d61cc, 0x34a87e86e83bed17, 0xe0244796,
	},
	{
		0xd62ab4e3f88fc797, 0xea86c7aeb6283ae4, 0xb5b93e09a7fe465, 0x4344a1a0134afe2,
		0xff5c17f02b62341d, 0x3214c6a587ce4644, 0xa905e7ed0629d05c, 0x4344a1a0134afe2,
		0xff5c17f02b62341d, 0x3214c6a587ce4644, 0xa905e7ed0629d05c, 0xb5c72690cd716e82,
		0x7c6097649e6ebe7b, 0x7ceee8c6e56a4dcd, 0x80ca849dc53eb9e4, 0x4ccf7e75,
	},
	{
		0xd0f06c28c7b36823, 0x1008cb0874de4bb8, 0xd6c7ff816c7a737b, 0x489b697fe30aa65f,
		0x4da0fb621fdc7817, 0xdc43583b82c58107, 0x4b0261debdec3cd6, 0x489b697fe30aa65f,
		0x4da0fb621fdc7817, 0xdc43583b82c58107, 0x4b0261debdec3cd6, 0xa9748d7b6c0e016c,
		0x7e8828f7ba4b034b, 0xda0fa54348a2512a, 0xebf9745c0962f9ad, 0x915cef86,
	},
	{
		0x99b7042460d72ec6, 0x2a53e5e2b8e795c2, 0x53a78132d9e1b3e3, 0xc043e67e6fc64118,
		0xff0abfe926d844d3, 0xf2a9fe5db2e910fe, 0xce352cdc84a964dd, 0xc043e67e6fc64118,
		0xff0abfe926d844d3, 0xf2a9fe5db2e910fe, 0xce352cdc84a964dd, 0xb89bc028aa5e6063,
		0xa354e7fdac04459c, 0x68d6547e6e980189, 0xc968dddfd573773e, 0x5cb59482,
	},
	{
		0x4f4dfcfc0ec2bae5, 0x841233148268a1b8, 0x9248a76ab8be0d3, 0x334c5a25b5903a8c,
		0x4c94fef443122128, 0x743e7d8454655c40, 0x1ab1e6d1452ae2cd, 0x334c5a25b5903a8c,
		0x4c94fef443122128, 0x743e7d8454655c40, 0x1ab1e6d1452ae2cd, 0xfec766de4a8e476c,
		0xcc0929da9567e71b, 0x5f9ef5b5f150c35a, 0x87659cabd649768f, 0x6ca3f532,
	},
	{
		0xfe86bf9d4422b9ae, 0xebce89c90641ef9c, 0x1c84e2292c0b5659, 0x8bde625a10a8c50d,
		0xeb8271ded1f79a0b, 0x14dc6844f0de7a3c, 0xf85b2f9541e7e6da, 0x8bde625a10a8c50d,
		0xeb8271ded1f79a0b, 0x14dc6844f0de7a3c, 0xf85b2f9541e7e6da, 0x2fe22cfd1683b961,
		0xea1d75c5b7aa01ca, 0x9eef60a44876bb95, 0x950c818e505c6f7f, 0xe24f3859,
	},
	{
		0xa90d81060932dbb0, 0x8acfaa88c5fbe92b, 0x7c6f3447e90f7f3f, 0xdd52fc14c8dd3143,
		0x1bc7508516e40628, 0x3059730266ade626, 0xffa526822f391c2, 0xdd52fc14c8dd3143,
		0x1bc7508516e40628, 0x3059730266ade626, 0xffa526822f391c2, 0xe25232d7afc8a406,
		0xd2b8a5a3f3b5f670, 0x6630f33edb7dfe32, 0xc71250ba68c4ea86, 0xadf5a9c7,
	},
	{
		0x17938a1b0e7f5952, 0x22cadd2f56f8a4be, 0x84b0d1183d5ed7c1, 0xc1336b92fef91bf6,
		0x80332a3945f33fa9, 0xa0f68b86f726ff92, 0xa3db5282cf5f4c0b, 0xc1336b92fef91bf6,
		0x80332a3945f33fa9, 0xa0f68b86f726ff92, 0xa3db5282cf5f4c0b, 0x82640b6fc4916607,
		0x2dc2a3aa1a894175, 0x8b4c852bdee7cc9, 0x10b9d0a08b55ff83, 0x32264b75,
	},
	{
		0xde9e0cb0e16f6e6d, 0x238e6283aa4f6594, 0x4fb9c914c2f0a13b, 0x497cb912b670f3b,
		0xd963a3f02ff4a5b6, 0x4fccefae11b50391, 0x42ba47db3f7672f, 0x497cb912b670f3b,
		0xd963a3f02ff4a5b6, 0x4fccefae11b50391, 0x42ba47db3f7672f, 0x1d6b655a1889feef,
		0x5f319abf8fafa19f, 0x715c2e49deb14620, 0x8d9153082ecdcea4, 0xa64b3376,
	},
	{
		0x6d4b876d9b146d1a, 0xaab2d64ce8f26739, 0xd315f93600e83fe5, 0x2fe9fabdbe7fdd4,
		0x755db249a2d81a69, 0xf27929f360446d71, 0x79a1bf957c0c1b92, 0x2fe9fabdbe7fdd4,
		0x755db249a2d81a69, 0xf27929f360446d71, 0x79a1bf957c0c1b92, 0x3c8a28d4c936c9cd,
		0xdf0d3d13b2c6a902, 0xc76702dd97cd2edd, 0x1aa220f7be16517, 0xd33890e,
	},
	{
		0xe698fa3f54e6ea22, 0xbd28e20e7455358c, 0x9ace161f6ea76e66, 0xd53fb7e3c93a9e4,
		0x737ae71b051bf108, 0x7ac71feb84c2df42, 0x3d8075cd293a15b4, 0xd53fb7e3c93a9e4,
		0x737ae71b051bf108, 0x7ac71feb84c2df42, 0x3d8075cd293a15b4, 0xbf8cee5e095d8a7c,
		0xe7086b3c7608143a, 0xe55b0c2fa938d70c, 0xfffb5f58e643649c, 0x926d4b63,
	},
	{
		0x7bc0deed4fb349f7, 0x1771aff25dc722fa, 0x19ff0644d9681917, 0xcf7d7f25bd70cd2c,
		0x9464ed9baeb41b4f, 0xb9064f5c3cb11b71, 0x237e39229b012b20, 0xcf7d7f25bd70cd2c,
		0x9464ed9baeb41b4f, 0xb9064f5c3cb11b71, 0x237e39229b012b20, 0xdd54d3f5d982dffe,
		0x7fc7562dbfc81dbf, 0x5b0dd1924f70945, 0xf1760537d8261135, 0xd51ba539,
	},
	{
		0xdb4b15e88533f622, 0x256d6d2419b41ce9, 0x9d7c5378396765d5, 0x9040e5b936b8661b,
		0x276e08fa53ac27fd, 0x8c944d39c2bdd2cc, 0xe2514c9802a5743c, 0x9040e5b936b8661b,
		0x276e08fa53ac27fd, 0x8c944d39c2bdd2cc, 0xe2514c9802a5743c, 0xe82107b11ac90386,
		0x7d6a22bc35055e6, 0xfd6ea9d1c438d8ae, 0xbe6015149e981553, 0x7f37636d,
	},
	{
		0x922834735e86ecb2, 0x363382685b88328e, 0xe9c92960d7144630, 0x8431b1bfd0a2379c,
		0x90383913aea283f9, 0xa6163831eb4924d2, 0x5f3921b4f9084aee, 0x8431b1bfd0a2379c,
		0x90383913aea283f9, 0xa6163831eb4924d2, 0x5f3921b4f9084aee, 0x7a70061a1473e579,
		0x5b19d80dcd2c6331, 0x6196b97931faad27, 0x869bf6828e237c3f, 0xb98026c0,
	},
	{
		0x30f1d72c812f1eb8, 0xb567cd4a69cd8989, 0x820b6c992a51f0bc, 0xc54677a80367125e,
		0x3204fbdba462e606, 0x8563278afc9eae69, 0x262147dd4bf7e566, 0xc54677a80367125e,
		0x3204fbdba462e606, 0x8563278afc9eae69, 0x262147dd4bf7e566, 0x2178b63e7ee2d230,
		0xe9c61ad81f5bff26, 0x9af7a81b3c501eca, 0x44104a3859f0238f, 0xb877767e,
	},
	{
		0x168884267f3817e9, 0x5b376e050f637645, 0x1c18314abd34497a, 0x9598f6ab0683fcc2,
		0x1c805abf7b80e1ee, 0xdec9ac42ee0d0f32, 0x8cd72e3912d24663, 0x9598f6ab0683fcc2,
		0x1c805abf7b80e1ee, 0xdec9ac42ee0d0f32, 0x8cd72e3912d24663, 0x1f025d405f1c1d87,
		0xbf7b6221e1668f8f, 0x52316f64e692dbb0, 0x7bf43df61ec51b39, 0xaefae77,
	},
	{
		0x82e78596ee3e56a7, 0x25697d9c87f30d98, 0x7600a8342834924d, 0x6ba372f4b7ab268b,
		0x8c3237cf1fe243df, 0x3833fc51012903df, 0x8e31310108c5683f, 0x6ba372f4b7ab268b,
		0x8c3237cf1fe243df, 0x3833fc51012903df, 0x8e31310108c5683f, 0x126593715c2de429,
		0x48ca8f35a3f54b90, 0xb9322b632f4f8b0, 0x926bb169b7337693, 0xf686911,
	},
	{
		0xaa2d6cf22e3cc252, 0x9b4dec4f5e179f16, 0x76fb0fba1d99a99a, 0x9a62af3dbba140da,
		0x27857ea044e9dfc1, 0x33abce9da2272647, 0xb22a7993aaf32556, 0x9a62af3dbba140da,
		0x27857ea044e9dfc1, 0x33abce9da2272647, 0xb22a7993aaf32556, 0xbf8f88f8019bedf0,
		0xed2d7f01fb273905, 0x6b45f15901b481cd, 0xf88ebb413ba6a8d5, 0x3deadf12,
	},
	{
		0x7bf5ffd7f69385c7, 0xfc077b1d8bc82879, 0x9c04e36f9ed83a24, 0x82065c62e6582188,
		0x8ef787fd356f5e43, 0x2922e53e36e17dfa, 0x9805f223d385010b, 0x82065c62e6582188,
		0x8ef787fd356f5e43, 0x2922e53e36e17dfa, 0x9805f223d385010b, 0x692154f3491b787d,
		0xe7e64700e414fbf, 0x757d4d4ab65069a0, 0xcd029446a8e348e2, 0xccf02a4e,
	},
	{
		0xe89c8ff9f9c6e34b, 0xf54c0f669a49f6c4, 0xfc3e46f5d846adef, 0x22f2aa3df2221cc,
		0xf66fea90f5d62174, 0xb75defaeaa1dd2a7, 0x9b994cd9a7214fd5, 0x22f2aa3df2221cc,
		0xf66fea90f5d62174, 0xb75defaeaa1dd2a7, 0x9b994cd9a7214fd5, 0xfac675a31804b773,
		0x98bcb3b820c50fc6, 0xe14af64d28cf0885, 0x27466fbd2b360eb5, 0x176c1722,
	},
	{
		0xa18fbcdccd11e1f4, 0x8248216751dfd65e, 0x40c089f208d89d7c, 0x229b79ab69ae97d,
		0xa87aabc2ec26e582, 0xbe2b053721eb26d2, 0x10febd7f0c3d6fcb, 0x229b79ab69ae97d,
		0xa87aabc2ec26e582, 0xbe2b053721eb26d2, 0x10febd7f0c3d6fcb, 0x9cc5b9b2f6e3bf7b,
		0x655d8495fe624a86, 0x6381a9f3d1f2bd7e, 0x79ebabbfc25c83e2, 0x26f82ad,
	},
	{
		0x2d54f40cc4088b17, 0x59d15633b0cd1399, 0xa8cc04bb1bffd15b, 0xd332cdb073d8dc46,
		0x272c56466868cb46, 0x7e7fcbe35ca6c3f3, 0xee8f51e5a70399d4, 0xd332cdb073d8dc46,
		0x272c56466868cb46, 0x7e7fcbe35ca6c3f3, 0xee8f51e5a70399d4, 0x16737a9c7581fe7b,
		0xed04bf52f4b75dcb, 0x9707ffb36bd30c1a, 0x1390f236fdc0de3e, 0xb5244f42,
	},
	{
		0x69276946cb4e87c7, 0x62bdbe6183be6fa9, 0x3ba9773dac442a1a, 0x702e2afc7f5a1825,
		0x8c49b11ea8151fdc, 0xcaf3fef61f5a86fa, 0xef0b2ee8649d7272, 0x702e2afc7f5a1825,
		0x8c49b11ea8151fdc, 0xcaf3fef61f5a86fa, 0xef0b2ee8649d7272, 0x9e34a4e08d9441e1,
		0x7bdc0cd64d5af533, 0xa926b14d99e3d868, 0xfca923a17788cce4, 0x49a689e5,
	},
	{
		0x668174a3f443df1d, 0x407299392da1ce86, 0xc2a3f7d7f2c5be28, 0xa590b202a7a5807b,
		0x968d2593f7ccb54e, 0x9dd8d669e3e95dec, 0xee0cc5dd58b6e93a, 0xa590b202a7a5807b,
		0x968d2593f7ccb54e, 0x9dd8d669e3e95dec, 0xee0cc5dd58b6e93a, 0xac65d5a9466fb483,
		0x221be538b2c9d806, 0x5cbe9441784f9fd9, 0xd4c7d5d6e3c122b8, 0x59fcdd3,
	},
	{
		0x5e29be847bd5046, 0xb561c7f19c8f80c3, 0x5e5abd5021ccaeaf, 0x7432d63888e0c306,
		0x74bbceeed479cb71, 0x6471586599575fdf, 0x6a859ad23365cba2, 0x7432d63888e0c306,
		0x74bbceeed479cb71, 0x6471586599575fdf, 0x6a859ad23365cba2, 0xf9ceec84acd18dcc,
		0x74a242ff1907437c, 0xf70890194e1ee913, 0x777dfcb4bb01f0ba, 0x4f4b04e9,
	},
	{
		0xcd0d79f2164da014, 0x4c386bb5c5d6ca0c, 0x8e771b03647c3b63, 0x69db23875cb0b715,
		0xada8dd91504ae37f, 0x46bf18dbf045ed6a, 0xe1b5f67b0645ab63, 0x69db23875cb0b715,
		0xada8dd91504ae37f, 0x46bf18dbf045ed6a, 0xe1b5f67b0645ab63, 0x877be8f5dcddff4,
		0x6d471b5f9ca2e2d1, 0x802c86d6f495b9bb, 0xa1f9b9b22b3be704, 0x8b00f891,
	},
	{
		0xe0e6fc0b1628af1d, 0x29be5fb4c27a2949, 0x1c3f781a604d3630, 0xc4af7faf883033aa,
		0x9bd296c4e9453cac, 0xca45426c1f7e33f9, 0xa6bbdcf7074d40c5, 0xc4af7faf883033aa,
		0x9bd296c4e9453cac, 0xca45426c1f7e33f9, 0xa6bbdcf7074d40c5, 0xe13a005d7142733b,
		0xc02b7925c5eeefaf, 0xd39119a60441e2d5, 0x3c24c710df8f4d43, 0x16e114f3,
	},
	{
		0x2058927664adfd93, 0x6e8f968c7963baa5, 0xaf3dced6fff7c394, 0x42e34cf3d53c7876,
		0x9cddbb26424dc5e, 0x64f6340a6d8eddad, 0x2196e488eb2a3a4b, 0x42e34cf3d53c7876,
		0x9cddbb26424dc5e, 0x64f6340a6d8eddad, 0x2196e488eb2a3a4b, 0xc9e9da25911a16fd,
		0xe21b4683f3e196a8, 0xcb80bf1a4c6fdbb4, 0x53792e9b3c3e67f8, 0xd6b6dadc,
	},
	{
		0xdc107285fd8e1af7, 0xa8641a0609321f3f, 0xdb06e89ffdc54466, 0xbcc7a81ed5432429,
		0xb6d7bdc6ad2e81f1, 0x93605ec471aa37db, 0xa2a73f8a85a8e397, 0xbcc7a81ed5432429,
		0xb6d7bdc6ad2e81f1, 0x93605ec471aa37db, 0xa2a73f8a85a8e397, 0x10a012b8ca7ac24b,
		0xaac5fd63351595cf, 0x5bb4c648a226dea0, 0x9d11ecb2b5c05c5f, 0x897e20ac,
	},
	{
		0xfbba1afe2e3280f1, 0x755a5f392f07fce, 0x9e44a9a15402809a, 0x6226a32e25099848,
		0xea895661ecf53004, 0x4d7e0158db2228b9, 0xe5a7d82922f69842, 0x6226a32e25099848,
		0xea895661ecf53004, 0x4d7e0158db2228b9, 0xe5a7d82922f69842, 0x2cea7713b69840ca,
		0x18de7b9ae938375b, 0xf127cca08f3cc665, 0xb1c22d727665ad2, 0xf996e05d,
	},
	{
		0xbfa10785ddc1011b, 0xb6e1c4d2f670f7de, 0x517d95604e4fcc1f, 0xca6552a0dfb82c73,
		0xb024cdf09e34ba07, 0x66cd8c5a95d7393b, 0xe3939acf790d4a74, 0xca6552a0dfb82c73,
		0xb024cdf09e34ba07, 0x66cd8c5a95d7393b, 0xe3939acf790d4a74, 0x97827541a1ef051e,
		0xac2fce47ebe6500c, 0xb3f06d3bddf3bd6a, 0x1d74afb25e1ce5fe, 0xc4306af6,
	},
	{
		0x534cc35f0ee1eb4e, 0xb703820f1f3b3dce, 0x884aa164cf22363, 0xf14ef7f47d8a57a3,
		0x80d1f86f2e061d7c, 0x401d6c2f151b5a62, 0xe988460224108944, 0xf14ef7f47d8a57a3,
		0x80d1f86f2e061d7c, 0x401d6c2f151b5a62, 0xe988460224108944, 0x7804d4135f68cd19,
		0x5487b4b39e69fe8e, 0x8cc5999015358a27, 0x8f3729b61c2d5601, 0x6dcad433,
	},
	{
		0x7ca6e3933995dac, 0xfd118c77daa8188, 0x3aceb7b5e7da6545, 0xc8389799445480db,
		0x5389f5df8aacd50d, 0xd136581f22fab5f, 0xc2f31f85991da417, 0xc8389799445480db,
		0x5389f5df8aacd50d, 0xd136581f22fab5f, 0xc2f31f85991da417, 0xaefbf9ff84035a43,
		0x8accbaf44adadd7c, 0xe57f3657344b67f5, 0x21490e5e8abdec51, 0x3c07374d,
	},
	{
		0xf0d6044f6efd7598, 0xe044d6ba4369856e, 0x91968e4f8c8a1a4c, 0x70bd1968996bffc2,
		0x4c613de5d8ab32ac, 0xfe1f4f97206f79d8, 0xac0434f2c4e213a9, 0x70bd1968996bffc2,
		0x4c613de5d8ab32ac, 0xfe1f4f97206f79d8, 0xac0434f2c4e213a9, 0x7490e9d82cfe22ca,
		0x5fbbf7f987454238, 0xc39e0dc8368ce949, 0x22201d3894676c71, 0xf0f4602c,
	},
	{
		0x3d69e52049879d61, 0x76610636ea9f74fe, 0xe9bf5602f89310c0, 0x8eeb177a86053c11,
		0xe390122c345f34a2, 0x1e30e47afbaaf8d6, 0x7b892f68e5f91732, 0x8eeb177a86053c11,
		0xe390122c345f34a2, 0x1e30e47afbaaf8d6, 0x7b892f68e5f91732, 0xb87922525fa44158,
		0xf440a1ee1a1a766b, 0xee8efad279d08c5c, 0x421f910c5b60216e, 0x3e1ea071,
	},
	{
		0x79da242a16acae31, 0x183c5f438e29d40, 0x6d351710ae92f3de, 0x27233b28b5b11e9b,
		0xc7dfe8988a942700, 0x570ed11c4abad984, 0x4b4c04632f48311a, 0x27233b28b5b11e9b,
		0xc7dfe8988a942700, 0x570ed11c4abad984, 0x4b4c04632f48311a, 0x12f33235442cbf9,
		0xa35315ca0b5b8cdb, 0xd8abde62ead5506b, 0xfc0fcf8478ad5266, 0x67580f0c,
	},
	{
		0x461c82656a74fb57, 0xd84b491b275aa0f7, 0x8f262cb29a6eb8b2, 0x49fa3070bc7b06d0,
		0xf12ed446bd0c0539, 0x6d43ac5d1dd4b240, 0x7609524fe90bec93, 0x49fa3070bc7b06d0,
		0xf12ed446bd0c0539, 0x6d43ac5d1dd4b240, 0x7609524fe90bec93, 0x391c2b2e076ec241,
		0xf5e62deda7839f7b, 0x3c7b3186a10d870f, 0x77ef4f2cba4f1005, 0x4e109454,
	},
	{
		0x53c1a66d0b13003, 0x731f060e6fe797fc, 0xdaa56811791371e3, 0x57466046cf6896ed,
		0x8ac37e0e8b25b0c6, 0x3e6074b52ad3cf18, 0xaa491ce7b45db297, 0x57466046cf6896ed,
		0x8ac37e0e8b25b0c6, 0x3e6074b52ad3cf18, 0xaa491ce7b45db297, 0xf7a9227c5e5e22c3,
		0x3d92e0841e29ce28, 0x2d30da5b2859e59d, 0xff37fa1c9cbfafc2, 0x88a474a7,
	},
	{
		0xd3a2efec0f047e9, 0x1cabce58853e58ea, 0x7a17b2eae3256be4, 0xc2dcc9758c910171,
		0xcb5cddaeff4ddb40, 0x5d7cc5869baefef1, 0x9644c5853af9cfeb, 0xc2dcc9758c910171,
		0xcb5cddaeff4ddb40, 0x5d7cc5869baefef1, 0x9644c5853af9cfeb, 0x255c968184694ee1,
		0x4e4d726eda360927, 0x7d27dd5b6d100377, 0x9a300e2020ddea2c, 0x5b5bedd,
	},
	{
		0x43c64d7484f7f9b2, 0x5da002b64aafaeb7, 0xb576c1e45800a716, 0x3ee84d3d5b4ca00b,
		0x5cbc6d701894c3f9, 0xd9e946f5ae1ca95, 0x24ca06e67f0b1833, 0x3ee84d3d5b4ca00b,
		0x5cbc6d701894c3f9, 0xd9e946f5ae1ca95, 0x24ca06e67f0b1833, 0x3413d46b4152650e,
		0xcbdfdbc2ab516f9c, 0x2aad8acb739e0c6c, 0x2bfc950d9f9fa977, 0x1aaddfa7,
	},
	{
		0xa7dec6ad81cf7fa1, 0x180c1ab708683063, 0x95e0fd7008d67cff, 0x6b11c5073687208,
		0x7e0a57de0d453f3, 0xe48c267d4f646867, 0x2168e9136375f9cb, 0x6b11c5073687208,
		0x7e0a57de0d453f3, 0xe48c267d4f646867, 0x2168e9136375f9cb, 0x64da194aeeea7fdf,
		0xa3b9f01fa5885678, 0xc316f8ee2eb2bd17, 0xa7e4d80f83e4427f, 0x5be07fd8,
	},
	{
		0x5408a1df99d4aff, 0xb9565e588740f6bd, 0xabf241813b08006e, 0x7da9e81d89fda7ad,
		0x274157cabe71440d, 0x2c22d9a480b331f7, 0xe835c8ac746472d5, 0x7da9e81d89fda7ad,
		0x274157cabe71440d, 0x2c22d9a480b331f7, 0xe835c8ac746472d5, 0x2038ce817a201ae4,
		0x46f3289dfe1c5e40, 0x435578a42d4b7c56, 0xf96d9f409fcf561, 0xcbca8606,
	},
	{
		0xa8b27a6bcaeeed4b, 0xaec1eeded6a87e39, 0x9daf246d6fed8326, 0xd45a938b79f54e8f,
		0x366b219d6d133e48, 0x5b14be3c25c49405, 0xfdd791d48811a572, 0xd45a938b79f54e8f,
		0x366b219d6d133e48, 0x5b14be3c25c49405, 0xfdd791d48811a572, 0x3de67b8d9e95d335,
		0x903c01307cfbeed5, 0xaf7d65f32274f1d1, 0x4dba141b5fc03c42, 0xbde64d01,
	},
	{
		0x9a952a8246fdc269, 0xd0dcfcac74ef278c, 0x250f7139836f0f1f, 0xc83d3c5f4e5f0320,
		0x694e7adeb2bf32e5, 0x7ad09538a3da27f5, 0x2b5c18f934aa5303, 0xc83d3c5f4e5f0320,
		0x694e7adeb2bf32e5, 0x7ad09538a3da27f5, 0x2b5c18f934aa5303, 0xc4dad7703d34326e,
		0x825569e2bcdc6a25, 0xb83d267709ca900d, 0x44ed05151f5d74e6, 0xee90cf33,
	},
	{
		0xc930841d1d88684f, 0x5eb66eb18b7f9672, 0xe455d413008a2546, 0xbc271bc0df14d647,
		0xb071100a9ff2edbb, 0x2b1a4c1cc31a119a, 0xb5d7caa1bd946cef, 0xbc271bc0df14d647,
		0xb071100a9ff2edbb, 0x2b1a4c1cc31a119a, 0xb5d7caa1bd946cef, 0xe02623ae10f4aadd,
		0xd79f600389cd06fd, 0x1e8da7965303e62b, 0x86f50e10eeab0925, 0x4305c3ce,
	},
	{
		0x94dc6971e3cf071a, 0x994c7003b73b2b34, 0xea16e85978694e5, 0x336c1b59a1fc19f6,
		0xc173acaecc471305, 0xdb1267d24f3f3f36, 0xe9a5ee98627a6e78, 0x336c1b59a1fc19f6,
		0xc173acaecc471305, 0xdb1267d24f3f3f36, 0xe9a5ee98627a6e78, 0x718f334204305ae5,
		0xe3b53c148f98d22c, 0xa184012df848926, 0x6e96386127d51183, 0x4b3a1d76,
	},
	{
		0x7fc98006e25cac9, 0x77fee0484cda86a7, 0x376ec3d447060456, 0x84064a6dcf916340,
		0xfbf55a26790e0ebb, 0x2e7f84151c31a5c2, 0x9f7f6d76b950f9bf, 0x84064a6dcf916340,
		0xfbf55a26790e0ebb, 0x2e7f84151c31a5c2, 0x9f7f6d76b950f9bf, 0x125e094fbee2b146,
		0x5706aa72b2eef7c2, 0x1c4a2daa905ee66e, 0x83d48029b5451694, 0xa8bb6d80,
	},
	{
		0xbd781c4454103f6, 0x612197322f49c931, 0xb9cf17fd7e5462d5, 0xe38e526cd3324364,
		0x85f2b63a5b5e840a, 0x485d7cef5aaadd87, 0xd2b837a462f6db6d, 0xe38e526cd3324364,
		0x85f2b63a5b5e840a, 0x485d7cef5aaadd87, 0xd2b837a462f6db6d, 0x3e41cef031520d9a,
		0x82df73902d7f67e, 0x3ba6fd54c15257cb, 0x22f91f079be42d40, 0x1f9fa607,
	},
	{
		0xda60e6b14479f9df, 0x3bdccf69ece16792, 0x18ebf45c4fecfdc9, 0x16818ee9d38c6664,
		0x5519fa9a1e35a329, 0xcbd0001e4b08ed8, 0x41a965e37a0c731b, 0x16818ee9d38c6664,
		0x5519fa9a1e35a329, 0xcbd0001e4b08ed8, 0x41a965e37a0c731b, 0x66e7b5dcca1ca28f,
		0x963b2d993614347d, 0x9b6fc6f41d411106, 0xaaaecaccf7848c0c, 0x8d0e4ed2,
	},
	{
		0x4ca56a348b6c4d3, 0x60618537c3872514, 0x2fbb9f0e65871b09, 0x30278016830ddd43,
		0xf046646d9012e074, 0xc62a5804f6e7c9da, 0x98d51f5830e2bc1e, 0x30278016830ddd43,
		0xf046646d9012e074, 0xc62a5804f6e7c9da, 0x98d51f5830e2bc1e, 0x7b2cbe5d37e3f29e,
		0x7b8c3ed50bda4aa0, 0x3ea60cc24639e038, 0xf7706de9fb0b5801, 0x1bf31347,
	},
	{
		0xebd22d4b70946401, 0x6863602bf7139017, 0xc0b1ac4e11b00666, 0x7d2782b82bd494b6,
		0x97159ba1c26b304b, 0x42b3b0fd431b2ac2, 0xfaa81f82691c830c, 0x7d2782b82bd494b6,
		0x97159ba1c26b304b, 0x42b3b0fd431b2ac2, 0xfaa81f82691c830c, 0x7cc6449234c7e185,
		0xaeaa6fa643ca86a5, 0x1412db1c0f2e0133, 0x4df2fe3e4072934f, 0x1ae3fc5b,
	},
	{
		0x3cc4693d6cbcb0c, 0x501689ea1c70ffa, 0x10a4353e9c89e364, 0x58c8aba7475e2d95,
		0x3e2f291698c9427a, 0xe8710d19c9de9e41, 0x65dda22eb04cf953, 0x58c8aba7475e2d95,
		0x3e2f291698c9427a, 0xe8710d19c9de9e41, 0x65dda22eb04cf953, 0xd7729c48c250cffa,
		0xef76162b2ddfba4b, 0x52371e17f4d51f6d, 0xddd002112ff0c833, 0x459c3930,
	},
	{
		0x38908e43f7ba5ef0, 0x1ab035d4e7781e76, 0x41d133e8c0a68ff7, 0xd1090893afaab8bc,
		0x96c4fe6922772807, 0x4522426c2b4205eb, 0xefad99a1262e7e0d, 0xd1090893afaab8bc,
		0x96c4fe6922772807, 0x4522426c2b4205eb, 0xefad99a1262e7e0d, 0xc7696029abdb465e,
		0x4e18eaf03d517651, 0xd006bced54c86ac8, 0x4330326d1021860c, 0xe00c4184,
	},
	{
		0x34983ccc6aa40205, 0x21802cad34e72bc4, 0x1943e8fb3c17bb8, 0xfc947167f69c0da5,
		0xae79cfdb91b6f6c1, 0x7b251d04c26cbda3, 0x128a33a79060d25e, 0xfc947167f69c0da5,
		0xae79cfdb91b6f6c1, 0x7b251d04c26cbda3, 0x128a33a79060d25e, 0x1eca842dbfe018dd,
		0x50a4cd2ee0ba9c63, 0xc2f5c97d8399682f, 0x3f929fc7cbe8ecbb, 0xffc7a781,
	},
	{
		0x86215c45dcac9905, 0xea546afe851cae4b, 0xd85b6457e489e374, 0xb7609c8e70386d66,
		0x36e6ccc278d1636d, 0x2f873307c08e6a1c, 0x10f252a758505289, 0xb7609c8e70386d66,
		0x36e6ccc278d1636d, 0x2f873307c08e6a1c, 0x10f252a758505289, 0xc8977646e81ab4b6,
		0x8017b745cd80213b, 0x960687db359bea0, 0xef4a470660799488, 0x6a125480,
	},
	{
		0x420fc255c38db175, 0xd503cd0f3c1208d1, 0xd4684e74c825a0bc, 0x4c10537443152f3d,
		0x720451d3c895e25d, 0xaff60c4d11f513fd, 0x881e8d6d2d5fb953, 0x4c10537443152f3d,
		0x720451d3c895e25d, 0xaff60c4d11f513fd, 0x881e8d6d2d5fb953, 0x9dec034a043f1f55,
		0xe27a0c22e7bfb39d, 0x2220b959128324, 0x53240272152dbd8b, 0x88a1512b,
	},
	{
		0x1d7a31f5bc8fe2f9, 0x4763991092dcf836, 0xed695f55b97416f4, 0xf265edb0c1c411d7,
		0x30e1e9ec5262b7e6, 0xc2c3ba061ce7957a, 0xd975f93b89a16409, 0xf265edb0c1c411d7,
		0x30e1e9ec5262b7e6, 0xc2c3ba061ce7957a, 0xd975f93b89a16409, 0xe9d703123f43450a,
		0x41383fedfed67c82, 0x6e9f43ecbbbd6004, 0xc7ccd23a24e77b8, 0x549bbbe5,
	},
	{
		0x94129a84c376a26e, 0xc245e859dc231933, 0x1b8f74fecf917453, 0xe9369d2e9007e74b,
		0xb1375915d1136052, 0x926c2021fe1d2351, 0x1d943addaaa2e7e6, 0xe9369d2e9007e74b,
		0xb1375915d1136052, 0x926c2021fe1d2351, 0x1d943addaaa2e7e6, 0xf5f515869c246738,
		0x7e309cd0e1c0f2a0, 0x153c3c36cf523e3b, 0x4931c66872ea6758, 0xc133d38c,
	},
	{
		0x1d3a9809dab05c8d, 0xadddeb4f71c93e8, 0xef342eb36631edb, 0x301d7a61c4b3dbca,
		0x861336c3f0552d61, 0x12c6db947471300f, 0xa679ef0ed761deb9, 0x301d7a61c4b3dbca,
		0x861336c3f0552d61, 0x12c6db947471300f, 0xa679ef0ed761deb9, 0x5f713b720efcd147,
		0x37ac330a333aa6b, 0x3309dc9ec1616eef, 0x52301d7a908026b5, 0xfcace348,
	},
	{
		0x90fa3ccbd60848da, 0xdfa6e0595b569e11, 0xe585d067a1f5135d, 0x6cef866ec295abea,
		0xc486c0d9214beb2d, 0xd6e490944d5fe100, 0x59df3175d72c9f38, 0x6cef866ec295abea,
		0xc486c0d9214beb2d, 0xd6e490944d5fe100, 0x59df3175d72c9f38, 0x3f23aeb4c04d1443,
		0x9bf0515cd8d24770, 0x958554f60ccaade2, 0x5182863c90132fe8, 0xed7b6f9a,
	},
	{
		0x2dbb4fc71b554514, 0x9650e04b86be0f82, 0x60f2304fba9274d3, 0xfcfb9443e997cab,
		0xf13310d96dec2772, 0x709cad2045251af2, 0xafd0d30cc6376dad, 0xfcfb9443e997cab,
		0xf13310d96dec2772, 0x709cad2045251af2, 0xafd0d30cc6376dad, 0x59d4bed30d550d0d,
		0x58006d4e22d8aad1, 0xeee12d2362d1f13b, 0x35cf1d7faaf1d228, 0x6d907dda,
	},
	{
		0xb98bf4274d18374a, 0x1b669fd4c7f9a19a, 0xb1f5972b88ba2b7a, 0x73119c99e6d508be,
		0x5d4036a187735385, 0x8fa66e192fd83831, 0x2abf64b6b592ed57, 0x73119c99e6d508be,
		0x5d4036a187735385, 0x8fa66e192fd83831, 0x2abf64b6b592ed57, 0xd4501f95dd84b08c,
		0xbf1552439c8bea02, 0x4f56fe753ba7e0ba, 0x4ca8d35cc058cfcd, 0x7a4d48d5,
	},
	{
		0xd6781d0b5e18eb68, 0xb992913cae09b533, 0x58f6021caaee3a40, 0xaafcb77497b5a20b,
		0x411819e5e79b77a3, 0xbd779579c51c77ce, 0x58d11f5dcf5d075d, 0xaafcb77497b5a20b,
		0x411819e5e79b77a3, 0xbd779579c51c77ce, 0x58d11f5dcf5d075d, 0x9eae76cde1cb4233,
		0x32fe25a9bf657970, 0x1c0c807948edb06a, 0xb8f29a3dfaee254d, 0xe686f3db,
	},
	{
		0x226651cf18f4884c, 0x595052a874f0f51c, 0xc9b75162b23bab42, 0x3f44f873be4812ec,
		0x427662c1dbfaa7b2, 0xa207ff9638fb6558, 0xa738d919e45f550f, 0x3f44f873be4812ec,
		0x427662c1dbfaa7b2, 0xa207ff9638fb6558, 0xa738d919e45f550f, 0xcb186ea05717e7d6,
		0x1ca7d68a5871fdc1, 0x5d4c119ea8ef3750, 0x72b6a10fa2ff9406, 0xcce7c55,
	},
	{
		0xa734fb047d3162d6, 0xe523170d240ba3a5, 0x125a6972809730e8, 0xd396a297799c24a1,
		0x8fee992e3069bad5, 0x2e3a01b0697ccf57, 0xee9c7390bd901cfa, 0xd396a297799c24a1,
		0x8fee992e3069bad5, 0x2e3a01b0697ccf57, 0xee9c7390bd901cfa, 0x56f2d9da0af28af2,
		0x3fdd37b2fe8437cb, 0x3d13eeeb60d6aec0, 0x2432ae62e800a5ce, 0xf58b96b,
	},
	{
		0xc6df6364a24f75a3, 0xc294e2c84c4f5df8, 0xa88df65c6a89313b, 0x895fe8443183da74,
		0xc7f2f6f895a67334, 0xa0d6b6a506691d31, 0x24f51712b459a9f0, 0x895fe8443183da74,
		0xc7f2f6f895a67334, 0xa0d6b6a506691d31, 0x24f51712b459a9f0, 0x173a699481b9e088,
		0x1dee9b77bcbf45d3, 0x32b98a646a8667d0, 0x3adcd4ee28f42a0e, 0x1bbf6f60,
	},
	{
		0xd8d1364c1fbcd10, 0x2d7cc7f54832deaa, 0x4e22c876a7c57625, 0xa3d5d1137d30c4bd,
		0x1e7d706a49bdfb9e, 0xc63282b20ad86db2, 0xaec97fa07916bfd6, 0xa3d5d1137d30c4bd,
		0x1e7d706a49bdfb9e, 0xc63282b20ad86db2, 0xaec97fa07916bfd6, 0x7c9ba3e52d44f73e,
		0xaf62fd245811185d, 0x8a9d2dacd8737652, 0xbd2cce277d5fbec0, 0xce5e0cc2,
	},
	{
		0xaae06f9146db885f, 0x3598736441e280d9, 0xfba339b117083e55, 0xb22bf08d9f8aecf7,
		0xc182730de337b922, 0x2b9adc87a0450a46, 0x192c29a9cfc00aad, 0xb22bf08d9f8aecf7,
		0xc182730de337b922, 0x2b9adc87a0450a46, 0x192c29a9cfc00aad, 0x9fd733f1d84a59d9,
		0xd86bd5c9839ace15, 0xaf20b57303172876, 0x9f63cb7161b5364c, 0x584cfd6f,
	},
	{
		0x8955ef07631e3bcc, 0x7d70965ea3926f83, 0x39aed4134f8b2db6, 0x882efc2561715a9c,
		0xef8132a18a540221, 0xb20a3c87a8c257c1, 0xf541b8628fad6c23, 0x882efc2561715a9c,
		0xef8132a18a540221, 0xb20a3c87a8c257c1, 0xf541b8628fad6c23, 0x9552aed57a6e0467,
		0x4d9fdd56867611a7, 0xc330279bf23b9eab, 0x44dbbaea2fcb8eba, 0x8f9bbc33,
	},
	{
		0xad611c609cfbe412, 0xd3c00b18bf253877, 0x90b2172e1f3d0bfd, 0x371a98b2cb084883,
		0x33a2886ee9f00663, 0xbe9568818ed6e6bd, 0xf244a0fa2673469a, 0x371a98b2cb084883,
		0x33a2886ee9f00663, 0xbe9568818ed6e6bd, 0xf244a0fa2673469a, 0xb447050bd3e559e9,
		0xd3b695dae7a13383, 0xded0bb65be471188, 0xca3c7a2b78922cae, 0xd7640d95,
	},
	{
		0xd5339adc295d5d69, 0xb633cc1dcb8b586a, 0xee84184cf5b1aeaf, 0x89f3aab99afbd636,
		0xf420e004f8148b9a, 0x6818073faa797c7c, 0xdd3b4e21cbbf42ca, 0x89f3aab99afbd636,
		0xf420e004f8148b9a, 0x6818073faa797c7c, 0xdd3b4e21cbbf42ca, 0x6a2b7db261164844,
		0xcbead63d1895852a, 0x93d37e1eae05e2f9, 0x5d06db2703fbc3ae, 0x3d12a2b,
	},
	{
		0x40d0aeff521375a8, 0x77ba1ad7ecebd506, 0x547c6f1a7d9df427, 0x21c2be098327f49b,
		0x7e035065ac7bbef5, 0x6d7348e63023fb35, 0x9d427dc1b67c3830, 0x21c2be098327f49b,
		0x7e035065ac7bbef5, 0x6d7348e63023fb35, 0x9d427dc1b67c3830, 0x4e3d018a43858341,
		0xcf924bb44d6b43c5, 0x4618b6a26e3446ae, 0x54d3013fac3ed469, 0xaaeafed0,
	},
	{
		0x8b2d54ae1a3df769, 0x11e7adaee3216679, 0x3483781efc563e03, 0x9d097dd3152ab107,
		0x51e21d24126e8563, 0xcba56cac884a1354, 0x39abb1b595f0a977, 0x9d097dd3152ab107,
		0x51e21d24126e8563, 0xcba56cac884a1354, 0x39abb1b595f0a977, 0x81e6dd1c1109848f,
		0x1644b209826d7b15, 0x6ac67e4e4b4812f0, 0xb3a9f5622c935bf7, 0x95b9b814,
	},
	{
		0x99c175819b4eae28, 0x932e8ff9f7a40043, 0xec78dcab07ca9f7c, 0xc1a78b82ba815b74,
		0x458cbdfc82eb322a, 0x17f4a192376ed8d7, 0x6f9e92968bc8ccef, 0xc1a78b82ba815b74,
		0x458cbdfc82eb322a, 0x17f4a192376ed8d7, 0x6f9e92968bc8ccef, 0x93e098c333b39905,
		0xd59b1cace44b7fdc, 0xf7a64ed78c64c7c5, 0x7c6eca5dd87ec1ce, 0x45fbe66e,
	},
	{
		0x2a418335779b82fc, 0xaf0295987849a76b, 0xc12bc5ff0213f46e, 0x5aeead8d6cb25bb9,
		0x739315f7743ec3ff, 0x9ab48d27111d2dcc, 0x5b87bd35a975929b, 0x5aeead8d6cb25bb9,
		0x739315f7743ec3ff, 0x9ab48d27111d2dcc, 0x5b87bd35a975929b, 0xc3dd8d6d95a46bb3,
		0x7bf9093215a4f483, 0xcb557d6ed84285bd, 0xdaf58422f261fdb5, 0xb4baa7a8,
	},
	{
		0x3b1fc6a3d279e67d, 0x70ea1e49c226396, 0x25505adcf104697c, 0xba1ffba29f0367aa,
		0xa20bec1dd15a8b6c, 0xe9bf61d2dab0f774, 0xf4f35bf5870a049c, 0xba1ffba29f0367aa,
		0xa20bec1dd15a8b6c, 0xe9bf61d2dab0f774, 0xf4f35bf5870a049c, 0x26787efa5b92385,
		0x3d9533590ce30b59, 0xa4da3e40530a01d4, 0x6395deaefb70067c, 0x83e962fe,
	},
	{
		0xd97eacdf10f1c3c9, 0xb54f4654043a36e0, 0xb128f6eb09d1234, 0xd8ad7ec84a9c9aa2,
		0xe256cffed11f69e6, 0x2cf65e4958ad5bda, 0xcfbf9b03245989a7, 0xd8ad7ec84a9c9aa2,
		0xe256cffed11f69e6, 0x2cf65e4958ad5bda, 0xcfbf9b03245989a7, 0x9fa51e6686cf4444,
		0x9425c117a34609d5, 0xb25f7e2c6f30e96, 0xea5477c3f2b5afd1, 0xaac3531c,
	},
	{
		0x293a5c1c4e203cd4, 0x6b3329f1c130cefe, 0xf2e32f8ec76aac91, 0x361e0a62c8187bff,
		0x6089971bb84d7133, 0x93df7741588dd50b, 0xc2a9b6abcd1d80b1, 0x361e0a62c8187bff,
		0x6089971bb84d7133, 0x93df7741588dd50b, 0xc2a9b6abcd1d80b1, 0x4d2f86869d79bc59,
		0x85cd24d8aa570ff, 0xb0dcf6ef0e94bbb5, 0x2037c69aa7a78421, 0x2b1db7cc,
	},
	{
		0x4290e018ffaedde7, 0xa14948545418eb5e, 0x72d851b202284636, 0x4ec02f3d2f2b23f2,
		0xab3580708aa7c339, 0xcdce066fbab3f65, 0xd8ed3ecf3c7647b9, 0x4ec02f3d2f2b23f2,
		0xab3580708aa7c339, 0xcdce066fbab3f65, 0xd8ed3ecf3c7647b9, 0x6d2204b3e31f344a,
		0x61a4d87f80ee61d7, 0x446c43dbed4b728f, 0x73130ac94f58747e, 0xcf00cd31,
	},
	{
		0xf919a59cbde8bf2f, 0xa56d04203b2dc5a5, 0x38b06753ac871e48, 0xc2c9fc637dbdfcfa,
		0x292ab8306d149d75, 0x7f436b874b9ffc07, 0xa5b56b0129218b80, 0xc2c9fc637dbdfcfa,
		0x292ab8306d149d75, 0x7f436b874b9ffc07, 0xa5b56b0129218b80, 0x9188f7bdc47ec050,
		0xcfe9345d03a15ade, 0x40b520fb2750c49e, 0xc2e83d343968af2e, 0x7d3c43b8,
	},
	{
		0x1d70a3f5521d7fa4, 0xfb97b3fdc5891965, 0x299d49bbbe3535af, 0xe1a8286a7d67946e,
		0x52bd956f047b298, 0xcbd74332dd4204ac, 0x12b5be7752721976, 0xe1a8286a7d67946e,
		0x52bd956f047b298, 0xcbd74332dd4204ac, 0x12b5be7752721976, 0x278426e27f6204b6,
		0x932ca7a7cd610181, 0x41647321f0a5914d, 0x48f4aa61a0ae80db, 0xcbd5fac6,
	},
	{
		0x6af98d7b656d0d7c, 0xd2e99ae96d6b5c0c, 0xf63bd1603ef80627, 0xbde51033ac0413f8,
		0xbc0272f691aec629, 0x6204332651bebc44, 0x1cbf00de026ea9bd, 0xbde51033ac0413f8,
		0xbc0272f691aec629, 0x6204332651bebc44, 0x1cbf00de026ea9bd, 0xb9c7ed6a75f3ff1e,
		0x7e310b76a5808e4f, 0xacbbd1aad5531885, 0xfc245f2473adeb9c, 0x76d0fec4,
	},
	{
		0x395b7a8adb96ab75, 0x582df7165b20f4a, 0xe52bd30e9ff657f9, 0x6c71064996cbec8b,
		0x352c535edeefcb89, 0xac7f0aba15cd5ecd, 0x3aba1ca8353e5c60, 0x6c71064996cbec8b,
		0x352c535edeefcb89, 0xac7f0aba15cd5ecd, 0x3aba1ca8353e5c60, 0x5c30a288a80ce646,
		0xc2940488b6617674, 0x925f8cc66b370575, 0xaa65d1283b9bb0ef, 0x405e3402,
	},
	{
		0x3822dd82c7df012f, 0xb9029b40bd9f122b, 0xfd25b988468266c4, 0x43e47bd5bab1e0ef,
		0x4a71f363421f282f, 0x880b2f32a2b4e289, 0x1299d4eda9d3eadf, 0x43e47bd5bab1e0ef,
		0x4a71f363421f282f, 0x880b2f32a2b4e289, 0x1299d4eda9d3eadf, 0xd713a40226f5564,
		0x4d8d34fedc769406, 0xa85001b29cd9cac3, 0xcae92352a41fd2b0, 0xc732c481,
	},
	{
		0x79f7efe4a80b951a, 0xdd3a3fddfc6c9c41, 0xab4c812f9e27aa40, 0x832954ec9d0de333,
		0x94c390aa9bcb6b8a, 0xf3b32afdc1f04f82, 0xd229c3b72e4b9a74, 0x832954ec9d0de333,
		0x94c390aa9bcb6b8a, 0xf3b32afdc1f04f82, 0xd229c3b72e4b9a74, 0x1d11860d7ed624a6,
		0xcadee20b3441b984, 0x75307079bf306f7b, 0x87902aa3b9753ba4, 0xa8d123c9,
	},
	{
		0xae6e59f5f055921a, 0xe9d9b7bf68e82, 0x5ce4e4a5b269cc59, 0x4960111789727567,
		0x149b8a37c7125ab6, 0x78c7a13ab9749382, 0x1c61131260ca151a, 0x4960111789727567,
		0x149b8a37c7125ab6, 0x78c7a13ab9749382, 0x1c61131260ca151a, 0x1e93276b35c309a0,
		0x2618f56230acde58, 0xaf61130a18e4febf, 0x7145deb18e89befe, 0x1e80ad7d,
	},
	{
		0x8959dbbf07387d36, 0xb4658afce48ea35d, 0x8f3f82437d8cb8d6, 0x6566d74954986ba5,
		0x99d5235cc82519a7, 0x257a23805c2d825, 0xad75ccb968e93403, 0x6566d74954986ba5,
		0x99d5235cc82519a7, 0x257a23805c2d825, 0xad75ccb968e93403, 0xb45bd4cf78e11f7f,
		0x80c5536bdc487983, 0xa4fd76ecbf018c8a, 0x3b9dac78a7a70d43, 0x52aeb863,
	},
	{
		0x4739613234278a49, 0x99ea5bcd340bf663, 0x258640912e712b12, 0xc8a2827404991402,
		0x7ee5e78550f02675, 0x2ec53952db5ac662, 0x1526405a9df6794b, 0xc8a2827404991402,
		0x7ee5e78550f02675, 0x2ec53952db5ac662, 0x1526405a9df6794b, 0xeddc6271170c5e1f,
		0xf5a85f986001d9d6, 0x95427c677bf58d58, 0x53ed666dfa85cb29, 0xef7c0c18,
	},
	{
		0x420e6c926bc54841, 0x96dbbf6f4e7c75cd, 0xd8d40fa70c3c67bb, 0x3edbc10e4bfee91b,
		0xf0d681304c28ef68, 0x77ea602029aaaf9c, 0x90f070bd24c8483c, 0x3edbc10e4bfee91b,
		0xf0d681304c28ef68, 0x77ea602029aaaf9c, 0x90f070bd24c8483c, 0x28bc8e41e08ceb86,
		0x1eb56e48a65691ef, 0x9fea5301c9202f0e, 0x3fcb65091aa9f135, 0xb6ad4b68,
	},
	{
		0xc8601bab561bc1b7, 0x72b26272a0ff869a, 0x56fdfc986d6bc3c4, 0x83707730cad725d4,
		0xc9ca88c3a779674a, 0xe1c696fbbd9aa933, 0x723f3baab1c17a45, 0x83707730cad725d4,
		0xc9ca88c3a779674a, 0xe1c696fbbd9aa933, 0x723f3baab1c17a45, 0xf82abc7a1d851682,
		0x30683836818e857d, 0x78bfa3e89a5ab23f, 0x6928234482b31817, 0xc1e46b17,
	},
	{
		0xb2d294931a0e20eb, 0x284ffd9a0815bc38, 0x1f8a103aac9bbe6, 0x1ef8e98e1ea57269,
		0x5971116272f45a8b, 0x187ad68ce95d8eac, 0xe94e93ee4e8ecaa6, 0x1ef8e98e1ea57269,
		0x5971116272f45a8b, 0x187ad68ce95d8eac, 0xe94e93ee4e8ecaa6, 0xa0ff2a58611838b5,
		0xb01e03849bfbae6f, 0xd081e202e28ea3ab, 0x51836bcee762bf13, 0x57b8df25,
	},
	{
		0x7966f53c37b6c6d7, 0x8e6abcfb3aa2b88f, 0x7f2e5e0724e5f345, 0x3eeb60c3f5f8143d,
		0xa25aec05c422a24f, 0xb026b03ad3cca4db, 0xe6e030028cc02a02, 0x3eeb60c3f5f8143d,
		0xa25aec05c422a24f, 0xb026b03ad3cca4db, 0xe6e030028cc02a02, 0x16fe679338b34bfc,
		0xc1be385b5c8a9de4, 0x65af5df6567530eb, 0xed3b303df4dc6335, 0xe9fa36d6,
	},
	{
		0xbe9bb0abd03b7368, 0x13bca93a3031be55, 0xe864f4f52b55b472, 0x36a8d13a2cbb0939,
		0x254ac73907413230, 0x73520d1522315a70, 0x8c9fdb5cf1e1a507, 0x36a8d13a2cbb0939,
		0x254ac73907413230, 0x73520d1522315a70, 0x8c9fdb5cf1e1a507, 0xb3640570b926886,
		0xfba2344ee87f7bab, 0xde57341ab448df05, 0x385612ee094fa977, 0x8f8daefc,
	},
	{
		0xa08d128c5f1649be, 0xa8166c3dbbe19aad, 0xcb9f914f829ec62c, 0x5b2b7ca856fad1c3,
		0x8093022d682e375d, 0xea5d163ba7ea231f, 0xd6181d012c0de641, 0x5b2b7ca856fad1c3,
		0x8093022d682e375d, 0xea5d163ba7ea231f, 0xd6181d012c0de641, 0xe7d40d0ab8b08159,
		0x2e82320f51b3a67e, 0x27c2e356ea0b63a3, 0x58842d01a2b1d077, 0x6e1bb7e,
	},
	{
		0x7c386f0ffe0465ac, 0x530419c9d843dbf3, 0x7450e3a4f72b8d8c, 0x48b218e3b721810d,
		0xd3757ac8609bc7fc, 0x111ba02a88aefc8, 0xe86343137d3bfc2a, 0x48b218e3b721810d,
		0xd3757ac8609bc7fc, 0x111ba02a88aefc8, 0xe86343137d3bfc2a, 0x44ad26b51661b507,
		0xdb1268670274f51e, 0x62a5e75beae875f3, 0xe266e7a44c5f28c6, 0xfd0076f0,
	},
	{
		0xbb362094e7ef4f8, 0xff3c2a48966f9725, 0x55152803acd4a7fe, 0x15747d8c505ffd00,
		0x438a15f391312cd6, 0xe46ca62c26d821f5, 0xbe78d74c9f79cb44, 0x15747d8c505ffd00,
		0x438a15f391312cd6, 0xe46ca62c26d821f5, 0xbe78d74c9f79cb44, 0xa8aa19f3aa59f09a,
		0xeffb3cddab2c9267, 0xd78e41ad97cb16a5, 0xace6821513527d32, 0x899b17b6,
	},
	{
		0xcd80dea24321eea4, 0x52b4fdc8130c2b15, 0xf3ea100b154bfb82, 0xd9ccef1d4be46988,
		0x5ede0c4e383a5e66, 0xda69683716a54d1e, 0xbfc3fdf02d242d24, 0xd9ccef1d4be46988,
		0x5ede0c4e383a5e66, 0xda69683716a54d1e, 0xbfc3fdf02d242d24, 0x20ed30274651b3f5,
		0x4c659824169e86c6, 0x637226dae5b52a0e, 0x7e050dbd1c71dc7f, 0xe3e84e31,
	},
	{
		0xd599a04125372c3a, 0x313136c56a56f363, 0x1e993c3677625832, 0x2870a99c76a587a4,
		0x99f74cc0b182dda4, 0x8a5e895b2f0ca7b6, 0x3d78882d5e0bb1dc, 0x2870a99c76a587a4,
		0x99f74cc0b182dda4, 0x8a5e895b2f0ca7b6, 0x3d78882d5e0bb1dc, 0xf466123732a3e25e,
		0xaca5e59716a40e50, 0x261d2e7383d0e686, 0xce9362d6a42c15a7, 0xeef79b6b,
	},
	{
		0xdbbf541e9dfda0a, 0x1479fceb6db4f844, 0x31ab576b59062534, 0xa3335c417687cf3a,
		0x92ff114ac45cda75, 0xc3b8a627384f13b5, 0xc4f25de33de8b3f7, 0xa3335c417687cf3a,
		0x92ff114ac45cda75, 0xc3b8a627384f13b5, 0xc4f25de33de8b3f7, 0xeacbf520578c5964,
		0x4cb19c5ab24f3215, 0xe7d8a6f67f0c6e7, 0x325c2413eb770ada, 0x868e3315,
	},
	{
		0xc2ee3288be4fe2bf, 0xc65d2f5ddf32b92, 0xaf6ecdf121ba5485, 0xc7cd48f7abf1fe59,
		0xce600656ace6f53a, 0x8a94a4381b108b34, 0xf9d1276c64bf59fb, 0xc7cd48f7abf1fe59,
		0xce600656ace6f53a, 0x8a94a4381b108b34, 0xf9d1276c64bf59fb, 0x219ce70ff5a112a5,
		0xe6026c576e2d28d7, 0xb8e467f25015e3a6, 0x950cb904f37af710, 0x4639a426,
	},
	{
		0xd86603ced1ed4730, 0xf9de718aaada7709, 0xdb8b9755194c6535, 0xd803e1eead47604c,
		0xad00f7611970a71b, 0xbc50036b16ce71f5, 0xafba96210a2ca7d6, 0xd803e1eead47604c,
		0xad00f7611970a71b, 0xbc50036b16ce71f5, 0xafba96210a2ca7d6, 0x28f7a7be1d6765f0,
		0x97bd888b93938c68, 0x6ad41d1b407ded49, 0xb9bfec098dc543e4, 0xf3213646,
	},
	{
		0x915263c671b28809, 0xa815378e7ad762fd, 0xabec6dc9b669f559, 0xd17c928c5342477f,
		0x745130b795254ad5, 0x8c5db926fe88f8ba, 0x742a95c953e6d974, 0xd17c928c5342477f,
		0x745130b795254ad5, 0x8c5db926fe88f8ba, 0x742a95c953e6d974, 0x279db8057b5d3e96,
		0x98168411565b4ec4, 0x50a72c54fa1125fa, 0x27766a635db73638, 0x17f148e9,
	},
	{
		0x2b67cdd38c307a5e, 0xcb1d45bb5c9fe1c, 0x800baf2a02ec18ad, 0x6531c1fe32bcb417,
		0x8c970d8df8cdbeb4, 0x917ba5fc67e72b40, 0x4b65e4e263e0a426, 0x6531c1fe32bcb417,
		0x8c970d8df8cdbeb4, 0x917ba5fc67e72b40, 0x4b65e4e263e0a426, 0xe0de33ce88a8b3a9,
		0xf8ef98a437e16b08, 0xa5162c0c7c5f7b62, 0xdbdac43361b2b881, 0xbfd94880,
	},
	{
		0x2d107419073b9cd0, 0xa96db0740cef8f54, 0xec41ee91b3ecdc1b, 0xffe319654c8e7ebc,
		0x6a67b8f13ead5a72, 0x6dd10a34f80d532f, 0x6e9cfaece9fbca4, 0xffe319654c8e7ebc,
		0x6a67b8f13ead5a72, 0x6dd10a34f80d532f, 0x6e9cfaece9fbca4, 0xb4468eb6a30aa7e9,
		0xe87995bee483222a, 0xd036c2c90c609391, 0x853306e82fa32247, 0xbb1fa7f3,
	},
	{
		0xf3e9487ec0e26dfc, 0x1ab1f63224e837fa, 0x119983bb5a8125d8, 0x8950cfcf4bdf622c,
		0x8847dca82efeef2f, 0x646b75b026708169, 0x21cab4b1687bd8b, 0x8950cfcf4bdf622c,
		0x8847dca82efeef2f, 0x646b75b026708169, 0x21cab4b1687bd8b, 0x243b489a9eae6231,
		0x5f3e634c4b779876, 0xff8abd1548eaf646, 0xc7962f5f0151914b, 0x88816b1,
	},
	{
		0x1160987c8fe86f7d, 0x879e6db1481eb91b, 0xd7dcb802bfe6885d, 0x14453b5cc3d82396,
		0x4ef700c33ed278bc, 0x1639c72ffc00d12e, 0xfb140ee6155f700d, 0x14453b5cc3d82396,
		0x4ef700c33ed278bc, 0x1639c72ffc00d12e, 0xfb140ee6155f700d, 0x2e6b5c96a6620862,
		0xa1f136998cbe19c, 0x74e058a3b6c5a712, 0x93dcf6bd33928b17, 0x5c2faeb3,
	},
	{
		0xeab8112c560b967b, 0x97f550b58e89dbae, 0x846ed506d304051f, 0x276aa37744b5a028,
		0x8c10800ee90ea573, 0xe6e57d2b33a1e0b7, 0x91f83563cd3b9dda, 0x276aa37744b5a028,
		0x8c10800ee90ea573, 0xe6e57d2b33a1e0b7, 0x91f83563cd3b9dda, 0xafbb4739570738a1,
		0x440ba98da5d8f69, 0xfde4e9b0eda20350, 0xe67dfa5a2138fa1, 0x51b5fc6f,
	},
	{
		0x1addcf0386d35351, 0xb5f436561f8f1484, 0x85d38e22181c9bb1, 0xff5c03f003c1fefe,
		0xe1098670afe7ff6, 0xea445030cf86de19, 0xf155c68b5c2967f8, 0xff5c03f003c1fefe,
		0xe1098670afe7ff6, 0xea445030cf86de19, 0xf155c68b5c2967f8, 0x95d31b145dbb2e9e,
		0x914fe1ca3deb3265, 0x6066020b1358ccc1, 0xc74bb7e2dee15036, 0x33d94752,
	},
	{
		0xd445ba84bf803e09, 0x1216c2497038f804, 0x2293216ea2237207, 0xe2164451c651adfb,
		0xb2534e65477f9823, 0x4d70691a69671e34, 0x15be4963dbde8143, 0xe2164451c651adfb,
		0xb2534e65477f9823, 0x4d70691a69671e34, 0x15be4963dbde8143, 0x762e75c406c5e9a3,
		0x7b7579f7e0356841, 0x480533eb066dfce5, 0x90ae14ea6bfeb4ae, 0xb0c92948,
	},
	{
		0x37235a096a8be435, 0xd9b73130493589c2, 0x3b1024f59378d3be, 0xad159f542d81f04e,
		0x49626a97a946096, 0xd8d3998bf09fd304, 0xd127a411eae69459, 0xad159f542d81f04e,
		0x49626a97a946096, 0xd8d3998bf09fd304, 0xd127a411eae69459, 0x8f3253c4eb785a7b,
		0x4049062f37e62397, 0xb9fa04d3b670e5c1, 0x1211a7967ac9350f, 0xc7171590,
	},
	{
		0x763ad6ea2fe1c99d, 0xcf7af5368ac1e26b, 0x4d5e451b3bb8d3d4, 0x3712eb913d04e2f2,
		0x2f9500d319c84d89, 0x4ac6eb21a8cf06f9, 0x7d1917afcde42744, 0x3712eb913d04e2f2,
		0x2f9500d319c84d89, 0x4ac6eb21a8cf06f9, 0x7d1917afcde42744, 0x6b58604b5dd10903,
		0xc4288dfbc1e319fc, 0x230f75ca96817c6e, 0x8894cba3b763756c, 0x240a67fb,
	},
	{
		0xea627fc84cd1b857, 0x85e372494520071f, 0x69ec61800845780b, 0xa3c1c5ca1b0367,
		0xeb6933997272bb3d, 0x76a72cb62692a655, 0x140bb5531edf756e, 0xa3c1c5ca1b0367,
		0xeb6933997272bb3d, 0x76a72cb62692a655, 0x140bb5531edf756e, 0x8d0d8067d1c925f4,
		0x7b3fa56d8d77a10c, 0x2bd00287b0946d88, 0xf08c8e4bd65b8970, 0xe1843cd5,
	},
	{
		0x1f2ffd79f2cdc0c8, 0x726a1bc31b337aaa, 0x678b7f275ef96434, 0x5aa82bfaa99d3978,
		0xc18f96cade5ce18d, 0x38404491f9e34c03, 0x891fb8926ba0418c, 0x5aa82bfaa99d3978,
		0xc18f96cade5ce18d, 0x38404491f9e34c03, 0x891fb8926ba0418c, 0xe5f69a6398114c15,
		0x7b8ded3623bc6b1d, 0x2f3e5c5da5ff70e8, 0x1ab142addea6a9ec, 0xfda1452b,
	},
	{
		0x39a9e146ec4b3210, 0xf63f75802a78b1ac, 0xe2e22539c94741c3, 0x8b305d532e61226e,
		0xcaeae80da2ea2e, 0x88a6289a76ac684e, 0x8ce5b5f9df1cbd85, 0x8b305d532e61226e,
		0xcaeae80da2ea2e, 0x88a6289a76ac684e, 0x8ce5b5f9df1cbd85, 0x8ae1fc4798e00d57,
		0xe7164b8fb364fc46, 0x6a978c9bd3a66943, 0xef10d5ae4dd08dc, 0xa2cad330,
	},
	{
		0x74cba303e2dd9d6d, 0x692699b83289fad1, 0xdfb9aa7874678480, 0x751390a8a5c41bdc,
		0x6ee5fbf87605d34, 0x6ca73f610f3a8f7c, 0xe898b3c996570ad, 0x751390a8a5c41bdc,
		0x6ee5fbf87605d34, 0x6ca73f610f3a8f7c, 0xe898b3c996570ad, 0x98168a5858fc7110,
		0x6f987fa27aa0daa2, 0xf25e3e180d4b36a3, 0xd0b03495aeb1be8a, 0x53467e16,
	},
	{
		0x4cbc2b73a43071e0, 0x56c5db4c4ca4e0b7, 0x1b275a162f46bd3d, 0xb87a326e413604bf,
		0xd8f9a5fa214b03ab, 0x8a8bb8265771cf88, 0xa655319054f6e70f, 0xb87a326e413604bf,
		0xd8f9a5fa214b03ab, 0x8a8bb8265771cf88, 0xa655319054f6e70f, 0xb499cb8e65a9af44,
		0xbee7fafcc8307491, 0x5d2e55fa9b27cda2, 0x63b120f5fb2d6ee5, 0xda14a8d0,
	},
	{
		0x875638b9715d2221, 0xd9ba0615c0c58740, 0x616d4be2dfe825aa, 0x5df25f13ea7bc284,
		0x165edfaafd2598fb, 0xaf7215c5c718c696, 0xe9f2f9ca655e769, 0x5df25f13ea7bc284,
		0x165edfaafd2598fb, 0xaf7215c5c718c696, 0xe9f2f9ca655e769, 0xe459cfcb565d3d2d,
		0x41d032631be2418a, 0xc505db05fd946f60, 0x54990394a714f5de, 0x67333551,
	},
	{
		0xfb686b2782994a8d, 0xedee60693756bb48, 0xe6bc3cae0ded2ef5, 0x58eb4d03b2c3ddf5,
		0x6d2542995f9189f1, 0xc0beec58a5f5fea2, 0xed67436f42e2a78b, 0x58eb4d03b2c3ddf5,
		0x6d2542995f9189f1, 0xc0beec58a5f5fea2, 0xed67436f42e2a78b, 0xdfec763cdb2b5193,
		0x724a8d5345bd2d6, 0x94d4fd1b81457c23, 0x28e87c50cdede453, 0xa0ebd66e,
	},
	{
		0xab21d81a911e6723, 0x4c31b07354852f59, 0x835da384c9384744, 0x7f759dddc6e8549a,
		0x616dd0ca022c8735, 0x94717ad4bc15ceb3, 0xf66c7be808ab36e, 0x7f759dddc6e8549a,
		0x616dd0ca022c8735, 0x94717ad4bc15ceb3, 0xf66c7be808ab36e, 0xaf8286b550b2f4b7,
		0x745bd217d20a9f40, 0xc73bfb9c5430f015, 0x55e65922666e3fc2, 0x4b769593,
	},
	{
		0x33d013cc0cd46ecf, 0x3de726423aea122c, 0x116af51117fe21a9, 0xf271ba474edc562d,
		0xe6596e67f9dd3ebd, 0xc0a288edf808f383, 0xb3def70681c6babc, 0xf271ba474edc562d,
		0xe6596e67f9dd3ebd, 0xc0a288edf808f383, 0xb3def70681c6babc, 0x7da7864e9989b095,
		0xbf2f8718693cd8a1, 0x264a9144166da776, 0x61ad90676870beb6, 0x6aa75624,
	},
	{
		0x8ca92c7cd39fae5d, 0x317e620e1bf20f1, 0x4f0b33bf2194b97f, 0x45744afcf131dbee,
		0x97222392c2559350, 0x498a19b280c6d6ed, 0x83ac2c36acdb8d49, 0x45744afcf131dbee,
		0x97222392c2559350, 0x498a19b280c6d6ed, 0x83ac2c36acdb8d49, 0x7a69645c294daa62,
		0xabe9d2be8275b3d2, 0x39542019de371085, 0x7f4efac8488cd6ad, 0x602a3f96,
	},
	{
		0xfdde3b03f018f43e, 0x38f932946c78660, 0xc84084ce946851ee, 0xb6dd09ba7851c7af,
		0x570de4e1bb13b133, 0xc4e784eb97211642, 0x8285a7fcdcc7c58d, 0xb6dd09ba7851c7af,
		0x570de4e1bb13b133, 0xc4e784eb97211642, 0x8285a7fcdcc7c58d, 0xd421f47990da899b,
		0x8aed409c997eaa13, 0x7a045929c2e29ccf, 0xb373682a6202c86b, 0xcd183c4d,
	},
	{
		0x9c8502050e9c9458, 0xd6d2a1a69964beb9, 0x1675766f480229b5, 0x216e1d6c86cb524c,
		0xd01cf6fd4f4065c0, 0xfffa4ec5b482ea0f, 0xa0e20ee6a5404ac1, 0x216e1d6c86cb524c,
		0xd01cf6fd4f4065c0, 0xfffa4ec5b482ea0f, 0xa0e20ee6a5404ac1, 0xc1b037e4eebaf85e,
		0x634e3d7c3ebf89eb, 0xbcda972358c67d1, 0xfd1352181e5b8578, 0x960a4d07,
	},
	{
		0x348176ca2fa2fdd2, 0x3a89c514cc360c2d, 0x9f90b8afb318d6d0, 0xbceee07c11a9ac30,
		0x2e2d47dff8e77eb7, 0x11a394cd7b6d614a, 0x1d7c41d54e15cb4a, 0xbceee07c11a9ac30,
		0x2e2d47dff8e77eb7, 0x11a394cd7b6d614a, 0x1d7c41d54e15cb4a, 0x15baa5ae7312b0fc,
		0xf398f596cc984635, 0x8ab8fdf87a6788e8, 0xb2b5c1234ab47e2, 0x9ae998c4,
	},
	{
		0x4a3d3dfbbaea130b, 0x4e221c920f61ed01, 0x553fd6cd1304531f, 0xbd2b31b5608143fe,
		0xab717a10f2554853, 0x293857f04d194d22, 0xd51be8fa86f254f0, 0xbd2b31b5608143fe,
		0xab717a10f2554853, 0x293857f04d194d22, 0xd51be8fa86f254f0, 0x1eee39e07686907e,
		0x639039fe0e8d3052, 0xd6ec1470cef97ff, 0x370c82b860034f0f, 0x74e2179d,
	},
	{
		0xb371f768cdf4edb9, 0xbdef2ace6d2de0f0, 0xe05b4100f7f1baec, 0xb9e0d415b4ebd534,
		0xc97c2a27efaa33d7, 0x591cdb35f84ef9da, 0xa57d02d0e8e3756c, 0xb9e0d415b4ebd534,
		0xc97c2a27efaa33d7, 0x591cdb35f84ef9da, 0xa57d02d0e8e3756c, 0x23f55f12d7c5c87b,
		0x4c7ca0fe23221101, 0xdbc3020480334564, 0xd985992f32c236b1, 0xee9bae25,
	},
	{
		0x7a1d2e96934f61f, 0xeb1760ae6af7d961, 0x887eb0da063005df, 0x2228d6725e31b8ab,
		0x9b98f7e4d0142e70, 0xb6a8c2115b8e0fe7, 0xb591e2f5ab9b94b1, 0x2228d6725e31b8ab,
		0x9b98f7e4d0142e70, 0xb6a8c2115b8e0fe7, 0xb591e2f5ab9b94b1, 0x6c1feaa8065318e0,
		0x4e7e2ca21c2e81fb, 0xe9fe5d8ce7993c45, 0xee411fa2f12cf8df, 0xb66edf10,
	},
	{
		0x8be53d466d4728f2, 0x86a5ac8e0d416640, 0x984aa464cdb5c8bb, 0x87049e68f5d38e59,
		0x7d8ce44ec6bd7751, 0xcc28d08ab414839c, 0x6c8f0bd34fe843e3, 0x87049e68f5d38e59,
		0x7d8ce44ec6bd7751, 0xcc28d08ab414839c, 0x6c8f0bd34fe843e3, 0xb8496dcdc01f3e47,
		0x2f03125c282ac26, 0x82a8797ba3f5ef07, 0x7c977a4d10bf52b8, 0xd6209737,
	},
	{
		0x829677eb03abf042, 0x43cad004b6bc2c0, 0xf2f224756803971a, 0x98d0dbf796480187,
		0xfbcb5f3e1bef5742, 0x5af2a0463bf6e921, 0xad9555bf0120b3a3, 0x98d0dbf796480187,
		0xfbcb5f3e1bef5742, 0x5af2a0463bf6e921, 0xad9555bf0120b3a3, 0x283e39b3dc99f447,
		0xbedaa1a4a0250c28, 0x9d50546624ff9a57, 0x4abaf523d1c090f6, 0xb994a88,
	},
	{
		0x754435bae3496fc, 0x5707fc006f094dcf, 0x8951c86ab19d8e40, 0x57c5208e8f021a77,
		0xf7653fbb69cd9276, 0xa484410af21d75cb, 0xf19b6844b3d627e8, 0x57c5208e8f021a77,
		0xf7653fbb69cd9276, 0xa484410af21d75cb, 0xf19b6844b3d627e8, 0xf37400fc3ffd9514,
		0x36ae0d821734edfd, 0x5f37820af1f1f306, 0xbe637d40e6a5ad0, 0xa05d43c0,
	},
	{
		0xfda9877ea8e3805f, 0x31e868b6ffd521b7, 0xb08c90681fb6a0fd, 0x68110a7f83f5d3ff,
		0x6d77e045901b85a8, 0x84ef681113036d8b, 0x3b9f8e3928f56160, 0x68110a7f83f5d3ff,
		0x6d77e045901b85a8, 0x84ef681113036d8b, 0x3b9f8e3928f56160, 0xfc8b7f56c130835,
		0xa11f3e800638e841, 0xd9572267f5cf28c1, 0x7897c8149803f2aa, 0xc79f73a8,
	},
	{
		0x2e36f523ca8f5eb5, 0x8b22932f89b27513, 0x331cd6ecbfadc1bb, 0xd1bfe4df12b04cbf,
		0xf58c17243fd63842, 0x3a453cdba80a60af, 0x5737b2ca7470ea95, 0xd1bfe4df12b04cbf,
		0xf58c17243fd63842, 0x3a453cdba80a60af, 0x5737b2ca7470ea95, 0x54d44a3f4477030c,
		0x8168e02d4869aa7f, 0x77f383a17778559d, 0x95e1737d77a268fc, 0xa490aff5,
	},
	{
		0x21a378ef76828208, 0xa5c13037fa841da2, 0x506d22a53fbe9812, 0x61c9c95d91017da5,
		0x16f7c83ba68f5279, 0x9c0619b0808d05f7, 0x83c117ce4e6b70a3, 0x61c9c95d91017da5,
		0x16f7c83ba68f5279, 0x9c0619b0808d05f7, 0x83c117ce4e6b70a3, 0xcfb4c8af7fd01413,
		0xfdef04e602e72296, 0xed6124d337889b1, 0x4919c86707b830da, 0xdfad65b4,
	},
	{
		0xccdd5600054b16ca, 0xf78846e84204cb7b, 0x1f9faec82c24eac9, 0x58634004c7b2d19a,
		0x24bb5f51ed3b9073, 0x46409de018033d00, 0x4a9805eed5ac802e, 0x58634004c7b2d19a,
		0x24bb5f51ed3b9073, 0x46409de018033d00, 0x4a9805eed5ac802e, 0xe18de8db306baf82,
		0x46bbf75f1fa025ff, 0x5faf2fb09be09487, 0x3fbc62bd4e558fb3, 0x1d07dfb,
	},
	{
		0x7854468f4e0cabd0, 0x3a3f6b4f098d0692, 0xae2423ec7799d30d, 0x29c3529eb165eeba,
		0x443de3703b657c35, 0x66acbce31ae1bc8d, 0x1acc99effe1d547e, 0x29c3529eb165eeba,
		0x443de3703b657c35, 0x66acbce31ae1bc8d, 0x1acc99effe1d547e, 0xcf07f8a57906573d,
		0x31bafb0bbb9a86e7, 0x40c69492702a9346, 0x7df61fdaa0b858af, 0x416df9a0,
	},
	{
		0x7f88db5346d8f997, 0x88eac9aacc653798, 0x68a4d0295f8eefa1, 0xae59ca86f4c3323d,
		0x25906c09906d5c4c, 0x8dd2aa0c0a6584ae, 0x232a7d96b38f40e9, 0xae59ca86f4c3323d,
		0x25906c09906d5c4c, 0x8dd2aa0c0a6584ae, 0x232a7d96b38f40e9, 0x8986ee00a2ed0042,
		0xc49ae7e428c8a7d1, 0xb7dd8280713ac9c2, 0xe018720aed1ebc28, 0x1f8fb9cc,
	},
	{
		0xbb3fb5fb01d60fcf, 0x1b7cc0847a215eb6, 0x1246c994437990a1, 0xd4edc954c07cd8f3,
		0x224f47e7c00a30ab, 0xd5ad7ad7f41ef0c6, 0x59e089281d869fd7, 0xd4edc954c07cd8f3,
		0x224f47e7c00a30ab, 0xd5ad7ad7f41ef0c6, 0x59e089281d869fd7, 0xf29340d07a14b6f1,
		0xc87c5ef76d9c4ef3, 0x463118794193a9a, 0x2922dcb0540f0dbc, 0x7abf48e3,
	},
	{
		0x2e783e1761acd84d, 0x39158042bac975a0, 0x1cd21c5a8071188d, 0xb1b7ec44f9302176,
		0x5cb476450dc0c297, 0xdc5ef652521ef6a2, 0x3cc79a9e334e1f84, 0xb1b7ec44f9302176,
		0x5cb476450dc0c297, 0xdc5ef652521ef6a2, 0x3cc79a9e334e1f84, 0x769e2a283dbcc651,
		0x9f24b105c8511d3f, 0xc31c15575de2f27e, 0xecfecf32c3ae2d66, 0xdea4e3dd,
	},
	{
		0x392058251cf22acc, 0x944ec4475ead4620, 0xb330a10b5cb94166, 0x54bc9bee7cbe1767,
		0x485820bdbe442431, 0x54d6120ea2972e90, 0xf437a0341f29b72a, 0x54bc9bee7cbe1767,
		0x485820bdbe442431, 0x54d6120ea2972e90, 0xf437a0341f29b72a, 0x8f30885c784d5704,
		0xaa95376b16c7906a, 0xe826928cfaf93dc3, 0x20e8f54d1c16d7d8, 0xc6064f22,
	},
	{
		0xadf5c1e5d6419947, 0x2a9747bc659d28aa, 0x95c5b8cb1f5d62c, 0x80973ea532b0f310,
		0xa471829aa9c17dd9, 0xc2ff3479394804ab, 0x6bf44f8606753636, 0x80973ea532b0f310,
		0xa471829aa9c17dd9, 0xc2ff3479394804ab, 0x6bf44f8606753636, 0x5184d2973e6dd827,
		0x121b96369a332d9a, 0x5c25d3475ab69e50, 0x26d2961d62884168, 0x743bed9c,
	},
	{
		0x6bc1db2c2bee5aba, 0xe63b0ed635307398, 0x7b2eca111f30dbbc, 0x230d2b3e47f09830,
		0xec8624a821c1caf4, 0xea6ec411cdbf1cb1, 0x5f38ae82af364e27, 0x230d2b3e47f09830,
		0xec8624a821c1caf4, 0xea6ec411cdbf1cb1, 0x5f38ae82af364e27, 0xa519ef515ea7187c,
		0x6bad5efa7ebae05f, 0x748abacb11a74a63, 0xa28eef963d1396eb, 0xfce254d5,
	},
	{
		0xb00f898229efa508, 0x83b7590ad7f6985c, 0x2780e70a0592e41d, 0x7122413bdbc94035,
		0xe7f90fae33bf7763, 0x4b6bd0fb30b12387, 0x557359c0c44f48ca, 0x7122413bdbc94035,
		0xe7f90fae33bf7763, 0x4b6bd0fb30b12387, 0x557359c0c44f48ca, 0xd5656c3d6bc5f0d,
		0x983ff8e5e784da99, 0x628479671b445bf, 0xe179a1e27ce68f5d, 0xe47ec9d1,
	},
	{
		0xb56eb769ce0d9a8c, 0xce196117bfbcaf04, 0xb26c3c3797d66165, 0x5ed12338f630ab76,
		0xfab19fcb319116d, 0x167f5f42b521724b, 0xc4aa56c409568d74, 0x5ed12338f630ab76,
		0xfab19fcb319116d, 0x167f5f42b521724b, 0xc4aa56c409568d74, 0x75fff4b42f8e9778,
		0x94218f94710c1ea3, 0xb7b05efb738b06a6, 0x83fff2deabf9cd3, 0x334a145c,
	},
	{
		0x70c0637675b94150, 0x259e1669305b0a15, 0x46e1dd9fd387a58d, 0xfca4e5bc9292788e,
		0xcd509dc1facce41c, 0xbbba575a59d82fe, 0x4e2e71c15b45d4d3, 0xfca4e5bc9292788e,
		0xcd509dc1facce41c, 0xbbba575a59d82fe, 0x4e2e71c15b45d4d3, 0x5dc54582ead999c,
		0x72612d1571963c6f, 0x30318a9d2d3d1829, 0x785dd00f4cc9c9a0, 0xadec1e3c,
	},
	{
		0x74c0b8a6821faafe, 0xabac39d7491370e7, 0xfaf0b2a48a4e6aed, 0x967e970df9673d2a,
		0xd465247cffa415c0, 0x33a1df0ca1107722, 0x49fc2a10adce4a32, 0x967e970df9673d2a,
		0xd465247cffa415c0, 0x33a1df0ca1107722, 0x49fc2a10adce4a32, 0xc5707e079a284308,
		0x573028266635dda6, 0xf786f5eee6127fa0, 0xb30d79cebfb51266, 0xf6a9fbf8,
	},
	{
		0x5fb5e48ac7b7fa4f, 0xa96170f08f5acbc7, 0xbbf5c63d4f52a1e5, 0x6cc09e60700563e9,
		0xd18f23221e964791, 0xffc23eeef7af26eb, 0x693a954a3622a315, 0x815308a32a9b0daf,
		0xefb2ab27bf6fd0bd, 0x9f1ffc0986111118, 0xf9a3aa1778ea3985, 0x698fe54b2b93933b,
		0xdacc2b28404d0f10, 0x815308a32a9b0daf, 0xefb2ab27bf6fd0bd, 0x5398210c,
	},
}

func check(t *testing.T, expected, actual uint64, offset, length int) {
	if expected != actual {
		t.Errorf("ERROR, offset: %d, length: %d, expected: %x, but got %x\n", offset, length, expected, actual)
	}
}

func test(t *testing.T, expected []uint64, offset, length int) {
	u := Hash128(data[offset : offset+length])
	v := Hash128Seed(data[offset:offset+length], seed128)
	check(t, expected[0], Hash64(data[offset:offset+length]), offset, length)
	check(t, expected[15], uint64(Hash32(data[offset:offset+length])), offset, length)
	check(t, expected[1], Hash64WithSeed(data[offset:offset+length], seed0), offset, length)
	check(t, expected[2], Hash64WithSeeds(data[offset:offset+length], seed0, seed1), offset, length)
	check(t, expected[3], u.Low, offset, length)
	check(t, expected[4], u.High, offset, length)
	check(t, expected[5], v.Low, offset, length)
	check(t, expected[6], v.High, offset, length)
}

func TestCityHash(t *testing.T) {
	setup()
	i := 0
	for ; i < testSize-1; i++ {
		test(t, testdata[i], i*i, i)
	}
	test(t, testdata[i], 0, dataSize)
}

func BenchmarkCityHash32(b *testing.B) {
	setup()
	b.ResetTimer()
	b.ReportAllocs()
	b.SetBytes(1024)

	for i := 0; i < b.N; i++ {
		Hash32(data[:1024])
	}
}

func BenchmarkCityHash64(b *testing.B) {
	setup()
	b.ResetTimer()
	b.ReportAllocs()
	b.SetBytes(1024)

	for i := 0; i < b.N; i++ {
		Hash64(data[:1024])
	}
}

func BenchmarkCityHash128(b *testing.B) {
	setup()
	b.ResetTimer()

	b.ReportAllocs()
	b.SetBytes(1024)

	for i := 0; i < b.N; i++ {
		Hash128(data[:1024])
	}
}

func BenchmarkCityHash64Small(b *testing.B) {
	setup()
	b.ResetTimer()
	b.ReportAllocs()
	b.SetBytes(10)

	for i := 0; i < b.N; i++ {
		Hash64(data[:10])
	}
}

func BenchmarkCityHash128Small(b *testing.B) {
	setup()
	b.ResetTimer()

	b.ReportAllocs()
	b.SetBytes(10)

	for i := 0; i < b.N; i++ {
		Hash128(data[:10])
	}
}

func TestRotate(t *testing.T) {
	t.Run("32", func(t *testing.T) {
		if rotate32(100, 0) != 100 {
			t.Error("unexpected rot64")
		}
	})
	t.Run("64", func(t *testing.T) {
		if rot64(100, 0) != 100 {
			t.Error("unexpected rot64")
		}
	})
}

func TestClickhouse(t *testing.T) {
	for _, tt := range []struct {
		Value    string
		Expected uint64
	}{
		{Value: "", Expected: 11160318154034397263},
		{Value: "Moscow", Expected: 12507901496292878638},
		{Value: "CH", Expected: 15020278857692564095},
		{Value: "ClickHouse", Expected: 12904064065176299341},

		{
			Value:    "ClickHouseIsAnOpenSource",
			Expected: 15757221730003458568,
		},
		{
			Value:    "ClickHouseIsAnOpenSourceAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
			Expected: 11821112606263625207,
		},
		{
			Value:    "ClickHouseIsAnOpenSourceAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
			Expected: 6686391753024203911,
		},
		{
			Value:    "ClickHouse is an open-source, high performance columnar OLAP database management system for real-time analytics using SQL",
			Expected: 12510537841872258940,
		},
	} {
		v := CH64([]byte(tt.Value))
		if v != tt.Expected {
			t.Errorf("CH64(%q) %d (got) != %d (expected)", tt.Value, v, tt.Expected)
		}
	}
}
