// Copyright 2021 The go-github AUTHORS. All rights reserved.
//
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package github

import "testing"

func TestPackageRegistry_Marshal(t *testing.T) {
	t.Parallel()
	testJSONMarshal(t, &PackageRegistry{}, "{}")

	o := &PackageRegistry{
		AboutURL: Ptr("aurl"),
		Name:     Ptr("name"),
		Type:     Ptr("type"),
		URL:      Ptr("url"),
		Vendor:   Ptr("vendor"),
	}
	want := `{
		"about_url": "aurl",
		"name": "name",
		"type": "type",
		"url": "url",
		"vendor": "vendor"
	}`

	testJSONMarshal(t, o, want)
}

func TestPackageFile_Marshal(t *testing.T) {
	t.Parallel()
	testJSONMarshal(t, &PackageFile{}, "{}")

	o := &PackageFile{
		DownloadURL: Ptr("durl"),
		ID:          Ptr(int64(1)),
		Name:        Ptr("name"),
		SHA256:      Ptr("sha256"),
		SHA1:        Ptr("sha1"),
		MD5:         Ptr("md5"),
		ContentType: Ptr("ct"),
		State:       Ptr("state"),
		Author: &User{
			Login:           Ptr("l"),
			ID:              Ptr(int64(1)),
			URL:             Ptr("u"),
			AvatarURL:       Ptr("a"),
			GravatarID:      Ptr("g"),
			Name:            Ptr("n"),
			Company:         Ptr("c"),
			Blog:            Ptr("b"),
			Location:        Ptr("l"),
			Email:           Ptr("e"),
			Hireable:        Ptr(true),
			Bio:             Ptr("b"),
			TwitterUsername: Ptr("t"),
			PublicRepos:     Ptr(1),
			Followers:       Ptr(1),
			Following:       Ptr(1),
			CreatedAt:       &Timestamp{referenceTime},
			SuspendedAt:     &Timestamp{referenceTime},
		},
		Size:      Ptr(int64(1)),
		CreatedAt: &Timestamp{referenceTime},
		UpdatedAt: &Timestamp{referenceTime},
	}

	want := `{
		"download_url": "durl",
		"id": 1,
		"name": "name",
		"sha256": "sha256",
		"sha1": "sha1",
		"md5": "md5",
		"content_type": "ct",
		"state": "state",
		"author": {
			"login": "l",
			"id": 1,
			"avatar_url": "a",
			"gravatar_id": "g",
			"name": "n",
			"company": "c",
			"blog": "b",
			"location": "l",
			"email": "e",
			"hireable": true,
			"bio": "b",
			"twitter_username": "t",
			"public_repos": 1,
			"followers": 1,
			"following": 1,
			"created_at": ` + referenceTimeStr + `,
			"suspended_at": ` + referenceTimeStr + `,
			"url": "u"
		},
		"size": 1,
		"created_at": ` + referenceTimeStr + `,
		"updated_at": ` + referenceTimeStr + `
	}`

	testJSONMarshal(t, o, want)
}

func TestPackageRelease_Marshal(t *testing.T) {
	t.Parallel()
	testJSONMarshal(t, &PackageRelease{}, "{}")

	o := &PackageRelease{
		URL:             Ptr("url"),
		HTMLURL:         Ptr("hurl"),
		ID:              Ptr(int64(1)),
		TagName:         Ptr("tn"),
		TargetCommitish: Ptr("tcs"),
		Name:            Ptr("name"),
		Draft:           Ptr(true),
		Author: &User{
			Login:           Ptr("l"),
			ID:              Ptr(int64(1)),
			URL:             Ptr("u"),
			AvatarURL:       Ptr("a"),
			GravatarID:      Ptr("g"),
			Name:            Ptr("n"),
			Company:         Ptr("c"),
			Blog:            Ptr("b"),
			Location:        Ptr("l"),
			Email:           Ptr("e"),
			Hireable:        Ptr(true),
			Bio:             Ptr("b"),
			TwitterUsername: Ptr("t"),
			PublicRepos:     Ptr(1),
			Followers:       Ptr(1),
			Following:       Ptr(1),
			CreatedAt:       &Timestamp{referenceTime},
			SuspendedAt:     &Timestamp{referenceTime},
		},
		Prerelease:  Ptr(true),
		CreatedAt:   &Timestamp{referenceTime},
		PublishedAt: &Timestamp{referenceTime},
	}

	want := `{
		"url": "url",
		"html_url": "hurl",
		"id": 1,
		"tag_name": "tn",
		"target_commitish": "tcs",
		"name": "name",
		"draft": true,
		"author": {
			"login": "l",
			"id": 1,
			"avatar_url": "a",
			"gravatar_id": "g",
			"name": "n",
			"company": "c",
			"blog": "b",
			"location": "l",
			"email": "e",
			"hireable": true,
			"bio": "b",
			"twitter_username": "t",
			"public_repos": 1,
			"followers": 1,
			"following": 1,
			"created_at": ` + referenceTimeStr + `,
			"suspended_at": ` + referenceTimeStr + `,
			"url": "u"
		},
		"prerelease": true,
		"created_at": ` + referenceTimeStr + `,
		"published_at": ` + referenceTimeStr + `
	}`

	testJSONMarshal(t, o, want)
}

func TestPackageVersion_Marshal(t *testing.T) {
	t.Parallel()
	testJSONMarshal(t, &PackageVersion{}, "{}")

	o := &PackageVersion{
		ID:       Ptr(int64(1)),
		Version:  Ptr("ver"),
		Summary:  Ptr("sum"),
		Body:     Ptr("body"),
		BodyHTML: Ptr("btnhtml"),
		Release: &PackageRelease{
			URL:             Ptr("url"),
			HTMLURL:         Ptr("hurl"),
			ID:              Ptr(int64(1)),
			TagName:         Ptr("tn"),
			TargetCommitish: Ptr("tcs"),
			Name:            Ptr("name"),
			Draft:           Ptr(true),
			Author: &User{
				Login:           Ptr("l"),
				ID:              Ptr(int64(1)),
				URL:             Ptr("u"),
				AvatarURL:       Ptr("a"),
				GravatarID:      Ptr("g"),
				Name:            Ptr("n"),
				Company:         Ptr("c"),
				Blog:            Ptr("b"),
				Location:        Ptr("l"),
				Email:           Ptr("e"),
				Hireable:        Ptr(true),
				Bio:             Ptr("b"),
				TwitterUsername: Ptr("t"),
				PublicRepos:     Ptr(1),
				Followers:       Ptr(1),
				Following:       Ptr(1),
				CreatedAt:       &Timestamp{referenceTime},
				SuspendedAt:     &Timestamp{referenceTime},
			},
			Prerelease:  Ptr(true),
			CreatedAt:   &Timestamp{referenceTime},
			PublishedAt: &Timestamp{referenceTime},
		},
		Manifest:        Ptr("mani"),
		HTMLURL:         Ptr("hurl"),
		TagName:         Ptr("tn"),
		TargetCommitish: Ptr("tcs"),
		TargetOID:       Ptr("tid"),
		Draft:           Ptr(true),
		Prerelease:      Ptr(true),
		CreatedAt:       &Timestamp{referenceTime},
		UpdatedAt:       &Timestamp{referenceTime},
		PackageFiles: []*PackageFile{
			{
				DownloadURL: Ptr("durl"),
				ID:          Ptr(int64(1)),
				Name:        Ptr("name"),
				SHA256:      Ptr("sha256"),
				SHA1:        Ptr("sha1"),
				MD5:         Ptr("md5"),
				ContentType: Ptr("ct"),
				State:       Ptr("state"),
				Author: &User{
					Login:           Ptr("l"),
					ID:              Ptr(int64(1)),
					URL:             Ptr("u"),
					AvatarURL:       Ptr("a"),
					GravatarID:      Ptr("g"),
					Name:            Ptr("n"),
					Company:         Ptr("c"),
					Blog:            Ptr("b"),
					Location:        Ptr("l"),
					Email:           Ptr("e"),
					Hireable:        Ptr(true),
					Bio:             Ptr("b"),
					TwitterUsername: Ptr("t"),
					PublicRepos:     Ptr(1),
					Followers:       Ptr(1),
					Following:       Ptr(1),
					CreatedAt:       &Timestamp{referenceTime},
					SuspendedAt:     &Timestamp{referenceTime},
				},
				Size:      Ptr(int64(1)),
				CreatedAt: &Timestamp{referenceTime},
				UpdatedAt: &Timestamp{referenceTime},
			},
		},
		Author: &User{
			Login:           Ptr("l"),
			ID:              Ptr(int64(1)),
			URL:             Ptr("u"),
			AvatarURL:       Ptr("a"),
			GravatarID:      Ptr("g"),
			Name:            Ptr("n"),
			Company:         Ptr("c"),
			Blog:            Ptr("b"),
			Location:        Ptr("l"),
			Email:           Ptr("e"),
			Hireable:        Ptr(true),
			Bio:             Ptr("b"),
			TwitterUsername: Ptr("t"),
			PublicRepos:     Ptr(1),
			Followers:       Ptr(1),
			Following:       Ptr(1),
			CreatedAt:       &Timestamp{referenceTime},
			SuspendedAt:     &Timestamp{referenceTime},
		},
		InstallationCommand: Ptr("ic"),
	}

	want := `{
		"id": 1,
		"version": "ver",
		"summary": "sum",
		"body": "body",
		"body_html": "btnhtml",
		"release": {
			"url": "url",
			"html_url": "hurl",
			"id": 1,
			"tag_name": "tn",
			"target_commitish": "tcs",
			"name": "name",
			"draft": true,
			"author": {
				"login": "l",
				"id": 1,
				"avatar_url": "a",
				"gravatar_id": "g",
				"name": "n",
				"company": "c",
				"blog": "b",
				"location": "l",
				"email": "e",
				"hireable": true,
				"bio": "b",
				"twitter_username": "t",
				"public_repos": 1,
				"followers": 1,
				"following": 1,
				"created_at": ` + referenceTimeStr + `,
				"suspended_at": ` + referenceTimeStr + `,
				"url": "u"
			},
			"prerelease": true,
			"created_at": ` + referenceTimeStr + `,
			"published_at": ` + referenceTimeStr + `
		},
		"manifest": "mani",
		"html_url": "hurl",
		"tag_name": "tn",
		"target_commitish": "tcs",
		"target_oid": "tid",
		"draft": true,
		"prerelease": true,
		"created_at": ` + referenceTimeStr + `,
		"updated_at": ` + referenceTimeStr + `,
		"package_files": [
			{
				"download_url": "durl",
				"id": 1,
				"name": "name",
				"sha256": "sha256",
				"sha1": "sha1",
				"md5": "md5",
				"content_type": "ct",
				"state": "state",
				"author": {
					"login": "l",
					"id": 1,
					"avatar_url": "a",
					"gravatar_id": "g",
					"name": "n",
					"company": "c",
					"blog": "b",
					"location": "l",
					"email": "e",
					"hireable": true,
					"bio": "b",
					"twitter_username": "t",
					"public_repos": 1,
					"followers": 1,
					"following": 1,
					"created_at": ` + referenceTimeStr + `,
					"suspended_at": ` + referenceTimeStr + `,
					"url": "u"
				},
				"size": 1,
				"created_at": ` + referenceTimeStr + `,
				"updated_at": ` + referenceTimeStr + `
			}
		],
		"author": {
			"login": "l",
			"id": 1,
			"avatar_url": "a",
			"gravatar_id": "g",
			"name": "n",
			"company": "c",
			"blog": "b",
			"location": "l",
			"email": "e",
			"hireable": true,
			"bio": "b",
			"twitter_username": "t",
			"public_repos": 1,
			"followers": 1,
			"following": 1,
			"created_at": ` + referenceTimeStr + `,
			"suspended_at": ` + referenceTimeStr + `,
			"url": "u"
		},
		"installation_command": "ic"
	}`

	testJSONMarshal(t, o, want)
}

func TestPackage_Marshal(t *testing.T) {
	t.Parallel()
	testJSONMarshal(t, &Package{}, "{}")

	o := &Package{
		ID:          Ptr(int64(1)),
		Name:        Ptr("name"),
		PackageType: Ptr("pt"),
		HTMLURL:     Ptr("hurl"),
		CreatedAt:   &Timestamp{referenceTime},
		UpdatedAt:   &Timestamp{referenceTime},
		Visibility:  Ptr("private"),
		Owner: &User{
			Login:           Ptr("l"),
			ID:              Ptr(int64(1)),
			URL:             Ptr("u"),
			AvatarURL:       Ptr("a"),
			GravatarID:      Ptr("g"),
			Name:            Ptr("n"),
			Company:         Ptr("c"),
			Blog:            Ptr("b"),
			Location:        Ptr("l"),
			Email:           Ptr("e"),
			Hireable:        Ptr(true),
			Bio:             Ptr("b"),
			TwitterUsername: Ptr("t"),
			PublicRepos:     Ptr(1),
			Followers:       Ptr(1),
			Following:       Ptr(1),
			CreatedAt:       &Timestamp{referenceTime},
			SuspendedAt:     &Timestamp{referenceTime},
		},
		PackageVersion: &PackageVersion{
			ID:       Ptr(int64(1)),
			Version:  Ptr("ver"),
			Summary:  Ptr("sum"),
			Body:     Ptr("body"),
			BodyHTML: Ptr("btnhtml"),
			Release: &PackageRelease{
				URL:             Ptr("url"),
				HTMLURL:         Ptr("hurl"),
				ID:              Ptr(int64(1)),
				TagName:         Ptr("tn"),
				TargetCommitish: Ptr("tcs"),
				Name:            Ptr("name"),
				Draft:           Ptr(true),
				Author: &User{
					Login:           Ptr("l"),
					ID:              Ptr(int64(1)),
					URL:             Ptr("u"),
					AvatarURL:       Ptr("a"),
					GravatarID:      Ptr("g"),
					Name:            Ptr("n"),
					Company:         Ptr("c"),
					Blog:            Ptr("b"),
					Location:        Ptr("l"),
					Email:           Ptr("e"),
					Hireable:        Ptr(true),
					Bio:             Ptr("b"),
					TwitterUsername: Ptr("t"),
					PublicRepos:     Ptr(1),
					Followers:       Ptr(1),
					Following:       Ptr(1),
					CreatedAt:       &Timestamp{referenceTime},
					SuspendedAt:     &Timestamp{referenceTime},
				},
				Prerelease:  Ptr(true),
				CreatedAt:   &Timestamp{referenceTime},
				PublishedAt: &Timestamp{referenceTime},
			},
			Manifest:        Ptr("mani"),
			HTMLURL:         Ptr("hurl"),
			TagName:         Ptr("tn"),
			TargetCommitish: Ptr("tcs"),
			TargetOID:       Ptr("tid"),
			Draft:           Ptr(true),
			Prerelease:      Ptr(true),
			CreatedAt:       &Timestamp{referenceTime},
			UpdatedAt:       &Timestamp{referenceTime},
			PackageFiles: []*PackageFile{
				{
					DownloadURL: Ptr("durl"),
					ID:          Ptr(int64(1)),
					Name:        Ptr("name"),
					SHA256:      Ptr("sha256"),
					SHA1:        Ptr("sha1"),
					MD5:         Ptr("md5"),
					ContentType: Ptr("ct"),
					State:       Ptr("state"),
					Author: &User{
						Login:           Ptr("l"),
						ID:              Ptr(int64(1)),
						URL:             Ptr("u"),
						AvatarURL:       Ptr("a"),
						GravatarID:      Ptr("g"),
						Name:            Ptr("n"),
						Company:         Ptr("c"),
						Blog:            Ptr("b"),
						Location:        Ptr("l"),
						Email:           Ptr("e"),
						Hireable:        Ptr(true),
						Bio:             Ptr("b"),
						TwitterUsername: Ptr("t"),
						PublicRepos:     Ptr(1),
						Followers:       Ptr(1),
						Following:       Ptr(1),
						CreatedAt:       &Timestamp{referenceTime},
						SuspendedAt:     &Timestamp{referenceTime},
					},
					Size:      Ptr(int64(1)),
					CreatedAt: &Timestamp{referenceTime},
					UpdatedAt: &Timestamp{referenceTime},
				},
			},
			Author: &User{
				Login:           Ptr("l"),
				ID:              Ptr(int64(1)),
				URL:             Ptr("u"),
				AvatarURL:       Ptr("a"),
				GravatarID:      Ptr("g"),
				Name:            Ptr("n"),
				Company:         Ptr("c"),
				Blog:            Ptr("b"),
				Location:        Ptr("l"),
				Email:           Ptr("e"),
				Hireable:        Ptr(true),
				Bio:             Ptr("b"),
				TwitterUsername: Ptr("t"),
				PublicRepos:     Ptr(1),
				Followers:       Ptr(1),
				Following:       Ptr(1),
				CreatedAt:       &Timestamp{referenceTime},
				SuspendedAt:     &Timestamp{referenceTime},
			},
			InstallationCommand: Ptr("ic"),
		},
		Registry: &PackageRegistry{
			AboutURL: Ptr("aurl"),
			Name:     Ptr("name"),
			Type:     Ptr("type"),
			URL:      Ptr("url"),
			Vendor:   Ptr("vendor"),
		},
	}

	want := `{
		"id": 1,
		"name": "name",
		"package_type": "pt",
		"html_url": "hurl",
		"created_at": ` + referenceTimeStr + `,
		"updated_at": ` + referenceTimeStr + `,
		"visibility": "private",
		"owner": {
			"login": "l",
			"id": 1,
			"avatar_url": "a",
			"gravatar_id": "g",
			"name": "n",
			"company": "c",
			"blog": "b",
			"location": "l",
			"email": "e",
			"hireable": true,
			"bio": "b",
			"twitter_username": "t",
			"public_repos": 1,
			"followers": 1,
			"following": 1,
			"created_at": ` + referenceTimeStr + `,
			"suspended_at": ` + referenceTimeStr + `,
			"url": "u"
		},
		"package_version": {
			"id": 1,
			"version": "ver",
			"summary": "sum",
			"body": "body",
			"body_html": "btnhtml",
			"release": {
				"url": "url",
				"html_url": "hurl",
				"id": 1,
				"tag_name": "tn",
				"target_commitish": "tcs",
				"name": "name",
				"draft": true,
				"author": {
					"login": "l",
					"id": 1,
					"avatar_url": "a",
					"gravatar_id": "g",
					"name": "n",
					"company": "c",
					"blog": "b",
					"location": "l",
					"email": "e",
					"hireable": true,
					"bio": "b",
					"twitter_username": "t",
					"public_repos": 1,
					"followers": 1,
					"following": 1,
					"created_at": ` + referenceTimeStr + `,
					"suspended_at": ` + referenceTimeStr + `,
					"url": "u"
				},
				"prerelease": true,
				"created_at": ` + referenceTimeStr + `,
				"published_at": ` + referenceTimeStr + `
			},
			"manifest": "mani",
			"html_url": "hurl",
			"tag_name": "tn",
			"target_commitish": "tcs",
			"target_oid": "tid",
			"draft": true,
			"prerelease": true,
			"created_at": ` + referenceTimeStr + `,
			"updated_at": ` + referenceTimeStr + `,
			"package_files": [
				{
					"download_url": "durl",
					"id": 1,
					"name": "name",
					"sha256": "sha256",
					"sha1": "sha1",
					"md5": "md5",
					"content_type": "ct",
					"state": "state",
					"author": {
						"login": "l",
						"id": 1,
						"avatar_url": "a",
						"gravatar_id": "g",
						"name": "n",
						"company": "c",
						"blog": "b",
						"location": "l",
						"email": "e",
						"hireable": true,
						"bio": "b",
						"twitter_username": "t",
						"public_repos": 1,
						"followers": 1,
						"following": 1,
						"created_at": ` + referenceTimeStr + `,
						"suspended_at": ` + referenceTimeStr + `,
						"url": "u"
					},
					"size": 1,
					"created_at": ` + referenceTimeStr + `,
					"updated_at": ` + referenceTimeStr + `
				}
			],
			"author": {
				"login": "l",
				"id": 1,
				"avatar_url": "a",
				"gravatar_id": "g",
				"name": "n",
				"company": "c",
				"blog": "b",
				"location": "l",
				"email": "e",
				"hireable": true,
				"bio": "b",
				"twitter_username": "t",
				"public_repos": 1,
				"followers": 1,
				"following": 1,
				"created_at": ` + referenceTimeStr + `,
				"suspended_at": ` + referenceTimeStr + `,
				"url": "u"
			},
			"installation_command": "ic"
		},
		"registry": {
			"about_url": "aurl",
			"name": "name",
			"type": "type",
			"url": "url",
			"vendor": "vendor"
		}
	}`

	testJSONMarshal(t, o, want)
}
