package russian_vocab_test

import (
	"testing"

	"github.com/kljensen/snowball/russian"
)

// Test a large set of words for which we know
// the correct stemmed form.
//
func Test_LargeVocabulary(t *testing.T) {
	testCases := []struct {
		in  string
		out string
	}{
		{"а", "а"},
		{"абиссинию", "абиссин"},
		{"абонемента", "абонемент"},
		{"абонировался", "абонирова"},
		{"абонируйся", "абонир"},
		{"абонируюсь", "абонир"},
		{"абрикосы", "абрикос"},
		{"август", "август"},
		{"августа", "август"},
		{"августе", "август"},
		{"авдотье", "авдот"},
		{"авдотьей", "авдот"},
		{"авдотьи", "авдот"},
		{"авдотью", "авдот"},
		{"авдотья", "авдот"},
		{"авенантненькая", "авенантненьк"},
		{"аверка", "аверк"},
		{"аверкой", "аверк"},
		{"авиатор", "авиатор"},
		{"авиаторов", "авиатор"},
		{"авиация", "авиац"},
		{"авось", "ав"},
		{"авраама", "авраам"},
		{"австрии", "австр"},
		{"австрийский", "австрийск"},
		{"автобиографию", "автобиограф"},
		{"автографом", "автограф"},
		{"автомобили", "автомоб"},
		{"автомобиль", "автомобил"},
		{"автор", "автор"},
		{"авторам", "автор"},
		{"авторитет", "авторитет"},
		{"авторитета", "авторитет"},
		{"авторитеты", "авторитет"},
		{"авторов", "автор"},
		{"автором", "автор"},
		{"ага", "аг"},
		{"агадир", "агадир"},
		{"агамемнон", "агамемнон"},
		{"агаповну", "агаповн"},
		{"агафье", "агаф"},
		{"агафьей", "агаф"},
		{"агафьи", "агаф"},
		{"агафью", "агаф"},
		{"агафья", "агаф"},
		{"агента", "агент"},
		{"агонии", "агон"},
		{"агроном", "агрон"},
		{"агрономии", "агроном"},
		{"ад", "ад"},
		{"ада", "ад"},
		{"адам", "ад"},
		{"адамант", "адамант"},
		{"адвокат", "адвокат"},
		{"адвокатом", "адвокат"},
		{"адвокатского", "адвокатск"},
		{"адвокатской", "адвокатск"},
		{"адвокатскую", "адвокатск"},
		{"административный", "административн"},
		{"административных", "административн"},
		{"администратор", "администратор"},
		{"адмиралы", "адмирал"},
		{"адом", "ад"},
		{"адрес", "адрес"},
		{"адреса", "адрес"},
		{"адресный", "адресн"},
		{"адресовано", "адресова"},
		{"адресом", "адрес"},
		{"адресу", "адрес"},
		{"адрианополя", "адрианопол"},
		{"адриатической", "адриатическ"},
		{"адская", "адск"},
		{"адский", "адск"},
		{"ажно", "ажн"},
		{"азарта", "азарт"},
		{"азартом", "азарт"},
		{"азбуку", "азбук"},
		{"азвольте", "азвольт"},
		{"азиатской", "азиатск"},
		{"азиатчина", "азиатчин"},
		{"азии", "аз"},
		{"азию", "аз"},
		{"азраил", "азра"},
		{"азы", "аз"},
		// {"аи", "а"},
		// {"ай", "а"},
		{"айвазовского", "айвазовск"},
		{"айда", "айд"},
		{"ак", "ак"},
		{"академии", "академ"},
		{"академию", "академ"},
		{"акации", "акац"},
		{"акацию", "акац"},
		{"акварелью", "акварел"},
		{"аккомпанемент", "аккомпанемент"},
		{"аккомпанировал", "аккомпанирова"},
		{"аккорд", "аккорд"},
		{"аккорда", "аккорд"},
		{"аккордом", "аккорд"},
		{"аккуратнее", "аккуратн"},
		{"аккуратно", "аккуратн"},
		{"аккуратного", "аккуратн"},
		{"аккуратный", "аккуратн"},
		{"акмеизм", "акмеизм"},
		{"аксельбантами", "аксельбант"},
		{"аксинье", "аксин"},
		{"аксиома", "аксиом"},
		{"акт", "акт"},
		{"акта", "акт"},
		{"акте", "акт"},
		{"актер", "актер"},
		{"актерам", "актер"},
		{"актером", "актер"},
		{"актеры", "актер"},
		{"активной", "активн"},
		{"актриса", "актрис"},
		{"актрисой", "актрис"},
		{"актрисы", "актрис"},
		{"акулина", "акулин"},
		{"акулине", "акулин"},
		{"акулиной", "акулин"},
		{"акулину", "акулин"},
		{"акулины", "акулин"},
		{"акурат", "акурат"},
		{"акушером", "акушер"},
		{"акцентом", "акцент"},
		{"акцизе", "акциз"},
		{"акции", "акц"},
		{"акций", "акц"},
		{"акциями", "акц"},
		{"акциях", "акц"},
		{"ал", "ал"},
		{"алая", "ал"},
		{"алгебра", "алгебр"},
		{"алгеброй", "алгебр"},
		{"алгебру", "алгебр"},
		{"алебастра", "алебастр"},
		{"алеко", "алек"},
		{"александр", "александр"},
		{"александра", "александр"},
		{"александринского", "александринск"},
		{"александрович", "александрович"},
		{"александровна", "александровн"},
		{"александровне", "александровн"},
		{"александровной", "александровн"},
		{"александровну", "александровн"},
		{"александровны", "александровн"},
		{"александром", "александр"},
		{"александру", "александр"},
		{"алексевна", "алексевн"},
		{"алексеев", "алексе"},
		{"алексеева", "алексеев"},
		{"алексеевич", "алексеевич"},
		{"алексеевича", "алексеевич"},
		{"алексеевичем", "алексеевич"},
		{"алексееву", "алексеев"},
		{"алексеевым", "алексеев"},
		{"алексеевыми", "алексеев"},
		{"алексеич", "алексеич"},
		{"алексеичем", "алексеич"},
		{"алексей", "алекс"},
		{"алексея", "алексе"},
		{"алела", "алел"},
		{"алена", "ал"},
		{"алене", "ален"},
		{"алену", "ален"},
		{"алены", "ал"},
		{"аленькие", "аленьк"},
		{"алеши", "алеш"},
		{"алешка", "алешк"},
		{"алеют", "алеют"},
		{"алея", "але"},
		{"али", "ал"},
		{"аллах", "алл"},
		{"аллегории", "аллегор"},
		{"аллее", "алл"},
		{"аллеи", "алл"},
		{"аллею", "алл"},
		{"аллея", "алле"},
		{"аллеям", "алле"},
		{"алмаз", "алмаз"},
		{"алмаза", "алмаз"},
		{"алмазах", "алмаз"},
		{"алмазы", "алмаз"},
		{"алой", "ал"},
		{"алом", "ал"},
		{"алоэ", "алоэ"},
		{"алтарь", "алтар"},
		{"алтарю", "алтар"},
		{"алтаря", "алтар"},
		{"алчущим", "алчущ"},
		{"алые", "ал"},
		{"алый", "ал"},
		{"алым", "ал"},
		{"алыми", "ал"},
		{"алых", "ал"},
		{"аль", "ал"},
		{"альбертыч", "альбертыч"},
		{"альбом", "альб"},
		{"альбомами", "альбом"},
		{"альмы", "альм"},
		{"альтшауер", "альтшауер"},
		{"альянов", "альян"},
		{"амалией", "амал"},
		{"амалии", "амал"},
		{"амалию", "амал"},
		{"амалия", "амал"},
		{"амаль", "амал"},
		{"амбар", "амбар"},
		{"амбара", "амбар"},
		{"амбарах", "амбар"},
		{"амбары", "амбар"},
		{"амбиций", "амбиц"},
		{"амбицию", "амбиц"},
		{"америка", "америк"},
		{"американским", "американск"},
		{"америке", "америк"},
		{"америки", "америк"},
		{"америку", "америк"},
		{"аметист", "аметист"},
		{"аминь", "амин"},
		{"амням", "амн"},
		{"ан", "ан"},
		{"анализ", "анализ"},
		{"анализа", "анализ"},
		{"анализировал", "анализирова"},
		{"анализирующей", "анализир"},
		{"анализом", "анализ"},
		{"анализу", "анализ"},
		{"аналоем", "анало"},
		{"анамнясь", "анамн"},
		{"ананас", "ананас"},
		{"ананасы", "ананас"},
		{"анастасий", "анастас"},
		{"анатомии", "анатом"},
		{"анатомию", "анатом"},
		{"ангажемент", "ангажемент"},
		{"ангары", "ангар"},
		{"ангел", "ангел"},
		{"ангела", "ангел"},
		{"ангелины", "ангелин"},
		{"ангелов", "ангел"},
		{"ангелом", "ангел"},
		{"ангелочка", "ангелочк"},
		{"ангелу", "ангел"},
		{"ангелы", "ангел"},
		{"ангельски", "ангельск"},
		{"ангельские", "ангельск"},
		{"ангельской", "ангельск"},
		{"англии", "англ"},
		{"английская", "английск"},
		{"английски", "английск"},
		{"английский", "английск"},
		{"английского", "английск"},
		{"английском", "английск"},
		{"английскому", "английск"},
		{"англичан", "англича"},
		{"англичанах", "англичан"},
		{"англичане", "англичан"},
		{"англичанин", "англичанин"},
		{"англию", "англ"},
		{"андрее", "андр"},
		{"андреев", "андре"},
		{"андреевич", "андреевич"},
		{"андреевна", "андреевн"},
		{"андреевне", "андреевн"},
		{"андреевной", "андреевн"},
		{"андреевну", "андреевн"},
		{"андреевны", "андреевн"},
		{"андреевым", "андреев"},
		{"андреем", "андре"},
		{"андреич", "андреич"},
		{"андреича", "андреич"},
		{"андреичем", "андреич"},
		{"андреичу", "андреич"},
		{"андрей", "андр"},
		{"андрею", "андр"},
		{"андрея", "андре"},
		{"андрюша", "андрюш"},
		{"андрюшанчик", "андрюшанчик"},
		{"андрюше", "андрюш"},
		{"андрюшей", "андрюш"},
		{"андрюши", "андрюш"},
		{"андрюшка", "андрюшк"},
		{"андрюшке", "андрюшк"},
		{"андрюшку", "андрюшк"},
		{"андрюшу", "андрюш"},
		{"ане", "ан"},
		{"аней", "ан"},
		{"анекдот", "анекдот"},
		{"анекдоты", "анекдот"},
		{"анечка", "анечк"},
		{"ани", "ан"},
		{"анисье", "анис"},
		{"анисьей", "анис"},
		{"анисьи", "анис"},
		{"аниська", "аниськ"},
		{"анисью", "анис"},
		{"анисья", "анис"},
		{"анна", "ан"},
		{"анне", "ан"},
		{"анной", "ан"},
		{"анну", "ан"},
		{"анны", "ан"},
		{"антверпен", "антверп"},
		{"антип", "антип"},
		{"антипа", "антип"},
		{"антипка", "антипк"},
		{"антипке", "антипк"},
		{"антипу", "антип"},
		{"антон", "антон"},
		{"антонов", "антон"},
		{"антракт", "антракт"},
		{"антракте", "антракт"},
		{"анфилада", "анфилад"},
		{"анфиса", "анфис"},
		{"анфисе", "анфис"},
		{"анчоусы", "анчоус"},
		{"аню", "ан"},
		{"аня", "ан"},
		{"апатии", "апат"},
		{"апатичен", "апатич"},
		{"апатически", "апатическ"},
		{"апатический", "апатическ"},
		{"апатическим", "апатическ"},
		{"апатию", "апат"},
		{"апатия", "апат"},
		{"апелляции", "апелляц"},
		{"апельсинничаешь", "апельсиннича"},
		{"апельсинов", "апельсин"},
		{"аплодирует", "аплодир"},
		{"аплодируя", "аплодиру"},
		{"аполлона", "аполлон"},
		{"апоплексический", "апоплексическ"},
		{"апоплексию", "апоплекс"},
		{"аппетита", "аппетит"},
		{"аппетитно", "аппетитн"},
		{"аппетитом", "аппетит"},
		{"апраксина", "апраксин"},
		{"апреле", "апрел"},
		{"апрель", "апрел"},
		{"апрельских", "апрельск"},
		{"апреля", "апрел"},
		{"аптека", "аптек"},
		{"аптекарь", "аптекар"},
		{"аптеке", "аптек"},
		{"аптеки", "аптек"},
		{"аптеку", "аптек"},
		{"аптечного", "аптечн"},
		{"апухтинские", "апухтинск"},
		{"ар", "ар"},
		{"аравийской", "аравийск"},
		{"арапка", "арапк"},
		{"арапку", "арапк"},
		{"арбуз", "арбуз"},
		{"арбузы", "арбуз"},
		{"аргументом", "аргумент"},
		{"арена", "ар"},
		{"арендатором", "арендатор"},
		{"аренду", "аренд"},
		{"аренды", "аренд"},
		{"арене", "арен"},
		{"арену", "арен"},
		{"арены", "ар"},
		{"арест", "арест"},
		{"арестант", "арестант"},
		{"арестанта", "арестант"},
		{"арестантов", "арестант"},
		{"арестантских", "арестантск"},
		{"арестантской", "арестантск"},
		{"арестанту", "арестант"},
		{"арестовать", "арестова"},
		{"арестовывают", "арестовыва"},
		{"арестуйте", "арест"},
		{"арестуют", "арест"},
		{"аресты", "арест"},
		{"арий", "ар"},
		{"аринушка", "аринушк"},
		{"аристократическим", "аристократическ"},
		{"аристократическими", "аристократическ"},
		{"аристократических", "аристократическ"},
		{"аристократического", "аристократическ"},
		{"аристократическом", "аристократическ"},
		{"аристократическою", "аристократическ"},
		{"арифметика", "арифметик"},
		{"арифметике", "арифметик"},
		{"арифметику", "арифметик"},
		{"ариша", "ариш"},
		{"арию", "ар"},
		{"аркадий", "аркад"},
		{"аркадина", "аркадин"},
		{"аркадиной", "аркадин"},
		{"аркадию", "аркад"},
		{"аркадия", "аркад"},
		{"аркой", "арк"},
		{"аркою", "арк"},
		{"армейский", "армейск"},
		{"армией", "арм"},
		{"армии", "арм"},
		{"армию", "арм"},
		{"армиями", "арм"},
		{"армяк", "армяк"},
		{"армяками", "армяк"},
		{"армяке", "армяк"},
		{"армячишка", "армячишк"},
		{"аромата", "аромат"},
		{"аромате", "аромат"},
		{"ароматном", "ароматн"},
		{"ароматные", "ароматн"},
		{"ароматным", "ароматн"},
		{"ароматом", "аромат"},
		{"ароматы", "аромат"},
		{"артели", "артел"},
		{"артельных", "артельн"},
		{"артельщик", "артельщик"},
		{"артельщика", "артельщик"},
		{"артельщиком", "артельщик"},
		{"артельщику", "артельщик"},
		{"артемий", "артем"},
		{"артемью", "артем"},
		{"артист", "артист"},
		{"артиста", "артист"},
		{"артистически", "артистическ"},
		{"артистической", "артистическ"},
		{"артистка", "артистк"},
		{"артистке", "артистк"},
		{"артисткой", "артистк"},
		{"артистов", "артист"},
		{"артисты", "артист"},
		{"артуром", "артур"},
		{"арфа", "арф"},
		{"арфам", "арф"},
		{"арфе", "арф"},
		{"арфы", "арф"},
		{"архангел", "архангел"},
		{"архангельск", "архангельск"},
		{"археолог", "археолог"},
		{"архив", "арх"},
		{"архива", "архив"},
		{"архимедова", "архимедов"},
		{"архитектор", "архитектор"},
		{"архитектором", "архитектор"},
		{"архитектору", "архитектор"},
		{"архитектура", "архитектур"},
		{"аршин", "аршин"},
		{"аршина", "аршин"},
		{"аршине", "аршин"},
		{"асессором", "асессор"},
		{"асессорше", "асессорш"},
		{"аскет", "аскет"},
		{"ассигнацию", "ассигнац"},
		{"ассигнациями", "ассигнац"},
		{"ассоциациях", "ассоциац"},
		{"астма", "астм"},
		{"астрахани", "астрахан"},
		{"астраханские", "астраханск"},
		{"астров", "астр"},
		{"астрова", "астров"},
		{"астрову", "астров"},
		{"астровым", "астров"},
		{"астрономии", "астроном"},
		{"ась", "а"},
		{"атакует", "атак"},
		{"атласам", "атлас"},
		{"атласная", "атласн"},
		{"атласном", "атласн"},
		{"атласным", "атласн"},
		{"атласными", "атласн"},
		{"атмосфере", "атмосфер"},
		{"атмосферы", "атмосфер"},
		{"атомах", "атом"},
		{"атомов", "атом"},
		{"атрибуты", "атрибут"},
		{"аттестат", "аттестат"},
		{"аттестате", "аттестат"},
		// {"ау", "а"},
		{"аудитории", "аудитор"},
		{"аукцион", "аукцион"},
		{"аукциона", "аукцион"},
		{"ауле", "аул"},
		{"аус", "аус"},
		{"афанасий", "афанас"},
		{"афанасию", "афанас"},
		{"афанасия", "афанас"},
		{"афанасьев", "афанас"},
		{"афанасьевич", "афанасьевич"},
		{"афанасьевича", "афанасьевич"},
		{"афанасьевичу", "афанасьевич"},
		{"афиша", "афиш"},
		{"африке", "африк"},
		{"африки", "африк"},
		{"африку", "африк"},
		{"афросиньюшка", "афросиньюшк"},
		{"афросиньюшки", "афросиньюшк"},
		{"аффектацией", "аффектац"},
		{"ах", "ах"},
		{"ахали", "аха"},
		{"аханья", "ахан"},
		{"ахают", "аха"},
		{"ахиллес", "ахиллес"},
		{"ахиллесовской", "ахиллесовск"},
		{"ахиллесу", "ахиллес"},
		{"ахиллов", "ахилл"},
		{"ахматовой", "ахматов"},
		{"ахнул", "ахнул"},
		{"ахнула", "ахнул"},
		{"ахнули", "ахнул"},
		{"ахнуть", "ахнут"},
		{"ацтеки", "ацтек"},
		{"аэроплан", "аэропла"},
		{"аэроплана", "аэропла"},
		{"б", "б"},
		{"ба", "ба"},
		{"баб", "баб"},
		{"баба", "баб"},
		{"бабенка", "бабенк"},
		{"бабками", "бабк"},
		{"бабой", "баб"},
		{"бабок", "бабок"},
		{"бабочка", "бабочк"},
		{"бабочки", "бабочк"},
		{"бабочку", "бабочк"},
		{"бабу", "баб"},
		{"бабушек", "бабушек"},
		{"бабушка", "бабушк"},
		{"бабушке", "бабушк"},
		{"бабушки", "бабушк"},
		{"бабушкин", "бабушкин"},
		{"бабушкина", "бабушкин"},
		{"бабушкиной", "бабушкин"},
		{"бабушкой", "бабушк"},
		{"бабушку", "бабушк"},
		{"бабы", "баб"},
		{"бабье", "баб"},
		{"бабьей", "баб"},
		{"бабьи", "баб"},
		{"бабьим", "баб"},
		{"бабья", "баб"},
		{"багаж", "багаж"},
		{"багрово", "багров"},
		{"багровой", "багров"},
		{"багровому", "багров"},
		{"багровые", "багров"},
		{"багровым", "багров"},
		{"багряный", "багрян"},
		{"багряным", "багрян"},
		{"бадья", "бад"},
		{"бает", "бает"},
		{"базар", "базар"},
		{"базарах", "базар"},
		{"базаре", "базар"},
		{"базилик", "базилик"},
		{"баиньки", "баиньк"},
		{"бай", "ба"},
		{"байро", "байр"},
		{"байрон", "байрон"},
		{"байрона", "байрон"},
		{"бакалеева", "бакалеев"},
		{"бакалееву", "бакалеев"},
		{"бакенами", "бакен"},
		{"бакенбард", "бакенбард"},
		{"бакенбарда", "бакенбард"},
		{"бакенбардами", "бакенбард"},
		{"бакенбарду", "бакенбард"},
		{"бакенбарды", "бакенбард"},
		{"бал", "бал"},
		{"бала", "бал"},
		{"балаган", "балага"},
		{"балаганы", "балага"},
		{"балалайка", "балалайк"},
		{"балалайками", "балалайк"},
		{"балами", "бал"},
		{"балах", "бал"},
		{"бале", "бал"},
		{"балерины", "балерин"},
		{"балета", "балет"},
		{"балете", "балет"},
		{"балканах", "балкан"},
		{"балканы", "балка"},
		{"балке", "балк"},
		{"балкон", "балкон"},
		{"балкона", "балкон"},
		{"балконами", "балкон"},
		{"балконах", "балкон"},
		{"балконе", "балкон"},
		{"балконом", "балкон"},
		{"баллада", "баллад"},
		{"баллотироваться", "баллотирова"},
		{"баловал", "балова"},
		{"балованых", "балован"},
		{"баловать", "балова"},
		{"баловень", "баловен"},
		{"баловника", "баловник"},
		{"баловню", "баловн"},
		{"баловство", "баловств"},
		{"балочов", "балоч"},
		{"балу", "бал"},
		{"балуй", "бал"},
		{"балующий", "бал"},
		{"бальзак", "бальзак"},
		{"бальтазару", "бальтазар"},
		{"банальной", "банальн"},
		{"бане", "бан"},
		{"бани", "бан"},
		{"банк", "банк"},
		{"банках", "банк"},
		{"банке", "банк"},
		{"банки", "банк"},
		{"банкирскую", "банкирск"},
		{"банкирша", "банкирш"},
		{"банков", "банк"},
		{"банковский", "банковск"},
		{"банку", "банк"},
		{"баночку", "баночк"},
		{"барабан", "бараба"},
		{"барабанил", "барабан"},
		{"барабанит", "барабан"},
		{"барабанить", "барабан"},
		{"бараков", "барак"},
		{"баран", "бара"},
		{"барана", "бара"},
		{"бараний", "баран"},
		{"баранина", "баранин"},
		{"бараниной", "баранин"},
		{"баранину", "баранин"},
		{"баранины", "баранин"},
		{"бараном", "баран"},
		{"бараньим", "баран"},
		{"барашками", "барашк"},
		{"барашком", "барашк"},
		{"барже", "барж"},
		{"барин", "барин"},
		{"барина", "барин"},
		{"барине", "барин"},
		{"барином", "барин"},
		{"барину", "барин"},
		{"барканом", "баркан"},
		{"барках", "барк"},
		{"барки", "барк"},
		{"бармгерциге", "бармгерциг"},
		{"барон", "барон"},
		{"барона", "барон"},
		{"бароне", "барон"},
		{"бароном", "барон"},
		{"барону", "барон"},
		{"бароны", "барон"},
		{"барская", "барск"},
		{"барские", "барск"},
		{"барский", "барск"},
		{"барских", "барск"},
		{"барского", "барск"},
		{"барское", "барск"},
		{"барской", "барск"},
		{"барском", "барск"},
		{"барскую", "барск"},
		{"барство", "барств"},
		{"бартола", "бартол"},
		{"бархат", "бархат"},
		{"бархата", "бархат"},
		{"бархате", "бархат"},
		{"бархатную", "бархатн"},
		{"бархатные", "бархатн"},
		{"бархатный", "бархатн"},
		{"бархатных", "бархатн"},
		{"бархату", "бархат"},
		{"бархатцами", "бархатц"},
		{"барчонка", "барчонк"},
		{"барчонком", "барчонк"},
		{"барчонок", "барчонок"},
		{"барщина", "барщин"},
		{"барщину", "барщин"},
		{"бары", "бар"},
		{"барыни", "барын"},
		{"барынин", "барынин"},
		{"барыню", "барын"},
		{"барыня", "барын"},
		{"барыш", "барыш"},
		{"барыше", "барыш"},
		{"барышень", "барышен"},
		{"барышне", "барышн"},
		{"барышней", "барышн"},
		{"барышни", "барышн"},
		{"барышниной", "барышнин"},
		{"барышню", "барышн"},
		{"барышня", "барышн"},
		{"барышнями", "барышн"},
		{"барышом", "барыш"},
		{"бас", "бас"},
		{"баск", "баск"},
		{"баском", "баск"},
		{"басманной", "басма"},
		{"басни", "басн"},
		{"баснословных", "баснословн"},
		{"басом", "бас"},
		{"баста", "баст"},
		{"басурманов", "басурман"},
		{"батальонных", "батальон"},
		{"батальоны", "батальон"},
		{"батареи", "батар"},
		{"батарейный", "батарейн"},
		{"батарейным", "батарейн"},
		{"батарею", "батар"},
		{"батарея", "батаре"},
		{"батистовый", "батистов"},
		{"батюшка", "батюшк"},
		{"батюшке", "батюшк"},
		{"батюшки", "батюшк"},
		{"батюшков", "батюшк"},
		{"батюшкова", "батюшков"},
		{"батюшкой", "батюшк"},
		{"бах", "бах"},
		{"бахрома", "бахром"},
		{"бахроме", "бахром"},
		{"бахромой", "бахром"},
		{"бахрому", "бахр"},
		{"бахромы", "бахром"},
		{"бахуса", "бахус"},
		{"бац", "бац"},
		{"башен", "баш"},
		{"башка", "башк"},
		{"башлыком", "башлык"},
		{"башмаках", "башмак"},
		{"башмаки", "башмак"},
		{"башмачки", "башмачк"},
		{"башмачонки", "башмачонк"},
		{"башне", "башн"},
		{"башни", "башн"},
		{"баюкает", "баюка"},
		{"бдение", "бден"},
		{"бдительного", "бдительн"},
		{"беато", "беат"},
		{"беатриче", "беатрич"},
		{"бег", "бег"},
		{"бега", "бег"},
		{"бегавшая", "бега"},
		{"бегавшими", "бега"},
		{"бегает", "бега"},
		{"бегаешь", "бега"},
		{"бегай", "бега"},
		{"бегал", "бега"},
		{"бегала", "бега"},
		{"бегали", "бега"},
		{"бегать", "бега"},
		{"бегах", "бег"},
		{"бегаю", "бега"},
		{"бегают", "бега"},
		{"бегающих", "бега"},
		{"бегая", "бег"},
		{"беги", "бег"},
		{"бегите", "бег"},
		{"бегло", "бегл"},
		{"беглый", "бегл"},
		{"беглым", "бегл"},
		{"беглых", "бегл"},
		{"бегом", "бег"},
		{"беготне", "беготн"},
		{"беготней", "беготн"},
		{"беготни", "беготн"},
		{"беготня", "беготн"},
		{"бегство", "бегств"},
		{"бегу", "бег"},
		{"бегуны", "бегун"},
		{"бегут", "бегут"},
		{"бегущего", "бегущ"},
		{"бегущий", "бегущ"},
		{"бегущую", "бегущ"},
		{"бед", "бед"},
		{"беда", "бед"},
		{"бедам", "бед"},
		{"беде", "бед"},
		{"беден", "бед"},
		{"бедная", "бедн"},
		{"беднее", "бедн"},
		{"беднейшее", "бедн"},
		{"беднейшей", "бедн"},
		{"беднейшие", "бедн"},
		{"беднейшую", "бедн"},
		{"беднел", "беднел"},
		{"бедненького", "бедненьк"},
		{"бедненькому", "бедненьк"},
		{"бедно", "бедн"},
		{"бедного", "бедн"},
		{"бедное", "бедн"},
		{"бедной", "бедн"},
		{"бедном", "бедн"},
		{"бедному", "бедн"},
		{"бедности", "бедност"},
		{"бедность", "бедност"},
		{"бедностью", "бедност"},
		{"бедною", "бедн"},
		{"бедную", "бедн"},
		{"бедны", "бедн"},
		{"бедные", "бедн"},
		{"бедный", "бедн"},
		{"бедным", "бедн"},
		{"бедных", "бедн"},
		{"бедняжка", "бедняжк"},
		{"бедняжки", "бедняжк"},
		{"бедняжку", "бедняжк"},
		{"бедняк", "бедняк"},
		{"бедняки", "бедняк"},
		{"бедовали", "бедова"},
		{"бедой", "бед"},
		{"бедр", "бедр"},
		{"бедрам", "бедр"},
		{"бедро", "бедр"},
		{"бедственное", "бедствен"},
		{"бедствий", "бедств"},
		{"бедствия", "бедств"},
		{"бедствиями", "бедств"},
		{"бедствует", "бедств"},
		{"беду", "бед"},
		{"беды", "бед"},
		{"бежавших", "бежа"},
		{"бежал", "бежа"},
		{"бежала", "бежа"},
		{"бежало", "бежа"},
		{"бежать", "бежа"},
		{"бежим", "беж"},
		{"бежит", "беж"},
		{"бежишь", "беж"},
		{"без", "без"},
		{"безбожная", "безбожн"},
		{"безбожник", "безбожник"},
		{"безбожника", "безбожник"},
		{"безбожникам", "безбожник"},
		{"безбожно", "безбожн"},
		{"безбожные", "безбожн"},
		{"безбольно", "безбольн"},
		{"безбородкин", "безбородкин"},
		{"безбрежной", "безбрежн"},
		{"безверие", "безвер"},
		{"безвестные", "безвестн"},
		{"безвестный", "безвестн"},
		{"безвкусица", "безвкусиц"},
		{"безвкусно", "безвкусн"},
		{"безвозвратно", "безвозвратн"},
		{"безвольно", "безвольн"},
		{"безвреден", "безвред"},
		{"безвыездно", "безвыездн"},
		{"безвыходная", "безвыходн"},
		{"безвыходное", "безвыходн"},
		{"безграмотно", "безграмотн"},
		{"безграмотным", "безграмотн"},
		{"безграничная", "безграничн"},
		{"безграничнейший", "безграничн"},
		{"безгранично", "безграничн"},
		{"безграничного", "безграничн"},
		{"безграничной", "безграничн"},
		{"безграничном", "безграничн"},
		{"безгрешным", "безгрешн"},
		{"бездарно", "бездарн"},
		{"бездарности", "бездарн"},
		{"бездарность", "бездарн"},
		{"бездарных", "бездарн"},
		{"бездействии", "бездейств"},
		{"бездействию", "бездейств"},
		{"бездействия", "бездейств"},
		{"безделицу", "безделиц"},
		{"безделицы", "безделиц"},
		{"безделку", "безделк"},
		{"бездн", "бездн"},
		{"бездна", "бездн"},
		{"бездне", "бездн"},
		{"бездной", "бездн"},
		{"бездну", "бездн"},
		{"бездны", "бездн"},
		{"бездомна", "бездомн"},
		{"бездомней", "бездомн"},
		{"бездомным", "бездомн"},
		{"бездонна", "бездон"},
		{"бездонной", "бездон"},
		{"бездонностью", "бездон"},
		{"бездонную", "бездон"},
		{"бездонным", "бездон"},
		{"бездонных", "бездон"},
		{"бездорожье", "бездорож"},
		{"бездорожью", "бездорож"},
		{"бездушном", "бездушн"},
		{"бездыханна", "бездыхан"},
		{"бездыханном", "бездыха"},
		{"бездыханность", "бездыхан"},
		{"безжалостно", "безжалостн"},
		{"безжалостный", "безжалостн"},
		{"беззаботен", "беззабот"},
		{"беззаботная", "беззаботн"},
		{"беззаботно", "беззаботн"},
		{"беззаботной", "беззаботн"},
		{"беззаботность", "беззаботн"},
		{"беззаботный", "беззаботн"},
		{"беззаботным", "беззаботн"},
		{"беззаботными", "беззаботн"},
		{"беззаветно", "беззаветн"},
		{"беззаветный", "беззаветн"},
		{"беззакатного", "беззакатн"},
		{"беззакатный", "беззакатн"},
		{"беззащитная", "беззащитн"},
		{"беззащитную", "беззащитн"},
		{"беззащитные", "беззащитн"},
		{"беззвездной", "беззвездн"},
		{"беззвездный", "беззвездн"},
		{"безземельная", "безземельн"},
		{"беззлобен", "беззлоб"},
		{"беззубый", "беззуб"},
		{"беззубым", "беззуб"},
		{"безличное", "безличн"},
		{"безличности", "безличн"},
		{"безличный", "безличн"},
		{"безлюдном", "безлюдн"},
		{"безлюдность", "безлюдн"},
		{"безлюдьи", "безлюд"},
		{"безмерно", "безмерн"},
		{"безмерными", "безмерн"},
		{"безмозглая", "безмозгл"},
		{"безмолвен", "безмолв"},
		{"безмолвного", "безмолвн"},
		{"безмолвное", "безмолвн"},
		{"безмолвны", "безмолвн"},
		{"безмолвные", "безмолвн"},
		{"безмолвный", "безмолвн"},
		{"безмолвным", "безмолвн"},
		{"безмолвных", "безмолвн"},
		{"безмолвствует", "безмолвств"},
		{"безмужних", "безмужн"},
		{"безмятежно", "безмятежн"},
		{"безмятежной", "безмятежн"},
		{"безмятежность", "безмятежн"},
		{"безмятежный", "безмятежн"},
		{"безнадежная", "безнадежн"},
		{"безнадежней", "безнадежн"},
		{"безнадежно", "безнадежн"},
		{"безнадежного", "безнадежн"},
		{"безнадежной", "безнадежн"},
		{"безнадежности", "безнадежн"},
		{"безнадежность", "безнадежн"},
		{"безнадежный", "безнадежн"},
		{"безнаказанно", "безнаказа"},
		{"безначальной", "безначальн"},
		{"безначальный", "безначальн"},
		{"безносым", "безнос"},
		{"безнравственная", "безнравствен"},
		{"безнравственно", "безнравствен"},
		{"безнравственны", "безнравствен"},
		{"безнравственный", "безнравствен"},
		{"безнравственным", "безнравствен"},
		{"безо", "без"},
		{"безобидно", "безобидн"},
		{"безобидного", "безобидн"},
		{"безоблачного", "безоблачн"},
		{"безоблачность", "безоблачн"},
		{"безоблачный", "безоблачн"},
		{"безобразен", "безобраз"},
		{"безобразие", "безобраз"},
		{"безобразию", "безобраз"},
		{"безобразия", "безобраз"},
		{"безобразна", "безобразн"},
		{"безобразная", "безобразн"},
		{"безобразнее", "безобразн"},
		{"безобразнейший", "безобразн"},
		{"безобразнейшим", "безобразн"},
		{"безобразник", "безобразник"},
		{"безобразничал", "безобразнича"},
		{"безобразничать", "безобразнича"},
		{"безобразно", "безобразн"},
		{"безобразного", "безобразн"},
		{"безобразное", "безобразн"},
		{"безобразной", "безобразн"},
		{"безобразном", "безобразн"},
		{"безобразною", "безобразн"},
		{"безобразную", "безобразн"},
		{"безобразные", "безобразн"},
		{"безобразный", "безобразн"},
		{"безобразным", "безобразн"},
		{"безобразными", "безобразн"},
		{"безобразных", "безобразн"},
		{"безобразью", "безобраз"},
		{"безопасным", "безопасн"},
		{"безответная", "безответн"},
		{"безответного", "безответн"},
		{"безответный", "безответн"},
		{"безотрадно", "безотрадн"},
		{"безотрадной", "безотрадн"},
		{"безотрадным", "безотрадн"},
		{"безотчетно", "безотчетн"},
		{"безотчетной", "безотчетн"},
		{"безотчетности", "безотчетн"},
		{"безотчетным", "безотчетн"},
		{"безошибочно", "безошибочн"},
		{"безошибочною", "безошибочн"},
		{"безполезны", "безполезн"},
		{"безрадостной", "безрадостн"},
		{"безрадостность", "безрадостн"},
		{"безразличный", "безразличн"},
		{"безрассудно", "безрассудн"},
		{"безрассудное", "безрассудн"},
		{"безрассудным", "безрассудн"},
		{"безропотно", "безропотн"},
		{"безукоризненна", "безукоризнен"},
		{"безукоризненно", "безукоризнен"},
		{"безукоризненное", "безукоризнен"},
		{"безумен", "безум"},
		{"безумие", "безум"},
		{"безумна", "безумн"},
		{"безумная", "безумн"},
		{"безумней", "безумн"},
		{"безумно", "безумн"},
		{"безумного", "безумн"},
		{"безумное", "безумн"},
		{"безумной", "безумн"},
		{"безумном", "безумн"},
		{"безумную", "безумн"},
		{"безумный", "безумн"},
		{"безумным", "безумн"},
		{"безумными", "безумн"},
		{"безумных", "безумн"},
		{"безумствовать", "безумствова"},
		{"безумствует", "безумств"},
		{"безумца", "безумц"},
		{"безумцы", "безумц"},
		{"безумья", "безум"},
		{"безупречная", "безупречн"},
		{"безусловно", "безусловн"},
		{"безутешная", "безутешн"},
		{"безучастия", "безучаст"},
		{"безучастно", "безучастн"},
		{"безучастным", "безучастн"},
		{"безызвестно", "безызвестн"},
		{"безыменные", "безымен"},
		{"безысходна", "безысходн"},
		{"безысходней", "безысходн"},
		{"безысходность", "безысходн"},
		{"безысходные", "безысходн"},
		{"безысходным", "безысходн"},
		{"бей", "бе"},
		{"бейся", "бе"},
		{"бейте", "бейт"},
		{"бекетов", "бекет"},
		{"бекешах", "бекеш"},
		{"бел", "бел"},
		{"бела", "бел"},
		{"белая", "бел"},
		{"белее", "бел"},
		{"белеет", "белеет"},
		{"белели", "белел"},
		{"беленькая", "беленьк"},
		{"беленькие", "беленьк"},
		{"беленький", "беленьк"},
		{"беленьким", "беленьк"},
		{"беленького", "беленьк"},
		{"белесым", "белес"},
		{"белея", "беле"},
		{"белизна", "белизн"},
		{"белизной", "белизн"},
		{"белизны", "белизн"},
		{"белильню", "белильн"},
		{"белинского", "белинск"},
		{"белится", "бел"},
		{"беллетрист", "беллетрист"},
		{"беллетристов", "беллетрист"},
		{"беллетристом", "беллетрист"},
		{"беллетристу", "беллетрист"},
		{"белобрысые", "белобрыс"},
		{"белобрысыми", "белобрыс"},
		{"беловатой", "беловат"},
		{"беловодовым", "беловодов"},
		{"белого", "бел"},
		{"белое", "бел"},
		{"белой", "бел"},
		{"белокурая", "белокур"},
		{"белокуренькая", "белокуреньк"},
		{"белокуренькое", "белокуреньк"},
		{"белокурою", "белокур"},
		{"белокурые", "белокур"},
		{"белокурый", "белокур"},
		{"белокурыми", "белокур"},
		{"белом", "бел"},
		{"белому", "бел"},
		{"белоручка", "белоручк"},
		{"белоснежная", "белоснежн"},
		{"белоснежной", "белоснежн"},
		{"белою", "бел"},
		{"белую", "бел"},
		{"белые", "бел"},
		{"белый", "бел"},
		{"белым", "бел"},
		{"белыми", "бел"},
		{"белых", "бел"},
		{"бельведерского", "бельведерск"},
		{"бельгию", "бельг"},
		{"белье", "бел"},
		{"бельем", "бел"},
		{"бельмеса", "бельмес"},
		{"белья", "бел"},
		{"бензол", "бензол"},
		{"берг", "берг"},
		{"бергом", "берг"},
		{"бердичеве", "бердичев"},
		{"берег", "берег"},
		{"берега", "берег"},
		{"береги", "берег"},
		{"берегись", "берег"},
		{"берегите", "берег"},
		{"берегитесь", "берег"},
		{"берегла", "берегл"},
		{"береглась", "берегл"},
		{"берегли", "берегл"},
		{"берегов", "берег"},
		{"берегся", "берег"},
		{"берегу", "берег"},
		{"бередило", "беред"},
		{"бережет", "бережет"},
		{"бережете", "бережет"},
		{"бережешь", "бережеш"},
		{"бережно", "бережн"},
		{"березками", "березк"},
		{"березку", "березк"},
		{"березняк", "березняк"},
		{"березняка", "березняк"},
		{"березовая", "березов"},
		{"березовую", "березов"},
		{"березы", "берез"},
		{"берем", "бер"},
		{"беременна", "беремен"},
		{"берет", "берет"},
		{"берете", "берет"},
		{"берется", "берет"},
		{"беречь", "береч"},
		{"беречься", "береч"},
		{"берешь", "береш"},
		{"бери", "бер"},
		{"берите", "бер"},
		{"берлин", "берлин"},
		{"берлога", "берлог"},
		{"берлоге", "берлог"},
		{"беру", "бер"},
		{"берусь", "бер"},
		{"берут", "берут"},
		{"берутся", "берут"},
		{"беря", "бер"},
		{"бес", "бес"},
		{"бесед", "бесед"},
		{"беседах", "бесед"},
		{"беседе", "бесед"},
		{"беседка", "беседк"},
		{"беседке", "беседк"},
		{"беседки", "беседк"},
		{"беседку", "беседк"},
		{"беседовавшим", "беседова"},
		{"беседовал", "беседова"},
		{"беседовала", "беседова"},
		{"беседой", "бесед"},
		{"беседу", "бесед"},
		{"беседует", "бесед"},
		{"беседуют", "бесед"},
		{"бесенок", "бесенок"},
		{"бесился", "бес"},
		{"бесится", "бес"},
		{"бесконечна", "бесконечн"},
		{"бесконечная", "бесконечн"},
		{"бесконечно", "бесконечн"},
		{"бесконечного", "бесконечн"},
		{"бесконечное", "бесконечн"},
		{"бесконечной", "бесконечн"},
		{"бесконечном", "бесконечн"},
		{"бесконечному", "бесконечн"},
		{"бесконечности", "бесконечн"},
		{"бесконечность", "бесконечн"},
		{"бесконечною", "бесконечн"},
		{"бесконечную", "бесконечн"},
		{"бесконечные", "бесконечн"},
		{"бесконечный", "бесконечн"},
		{"бесконечным", "бесконечн"},
		{"бесконечных", "бесконечн"},
		{"бескорыстно", "бескорыстн"},
		{"бескорыстное", "бескорыстн"},
		{"бескрайною", "бескрайн"},
		{"бескровно", "бескровн"},
		{"бескровных", "бескровн"},
		{"бесноватыми", "бесноват"},
		{"беспамятная", "беспамятн"},
		{"беспамятный", "беспамятн"},
		{"беспамятстве", "беспамятств"},
		{"беспамятство", "беспамятств"},
		{"бесперечь", "беспереч"},
		{"беспечальное", "беспечальн"},
		{"беспечнее", "беспечн"},
		{"беспечно", "беспечн"},
		{"беспечное", "беспечн"},
		{"беспечности", "беспечн"},
		{"беспечность", "беспечн"},
		{"беспечный", "беспечн"},
		{"беспечными", "беспечн"},
		{"бесплодная", "бесплодн"},
		{"бесплодного", "бесплодн"},
		{"бесплодные", "бесплодн"},
		{"бесплодных", "бесплодн"},
		{"бесповоротно", "бесповоротн"},
		{"бесподобно", "бесподобн"},
		{"беспокоен", "беспоко"},
		{"беспокоил", "беспоко"},
		{"беспокоила", "беспоко"},
		{"беспокоилась", "беспоко"},
		{"беспокоили", "беспоко"},
		{"беспокоило", "беспоко"},
		{"беспокоился", "беспоко"},
		{"беспокоим", "беспоко"},
		{"беспокоит", "беспоко"},
		{"беспокоитесь", "беспоко"},
		{"беспокоить", "беспоко"},
		{"беспокоиться", "беспоко"},
		{"беспокоишься", "беспоко"},
		{"беспокой", "беспок"},
		{"беспокойная", "беспокойн"},
		{"беспокойнее", "беспокойн"},
		{"беспокойно", "беспокойн"},
		{"беспокойного", "беспокойн"},
		{"беспокойное", "беспокойн"},
		{"беспокойной", "беспокойн"},
		{"беспокойны", "беспокойн"},
		{"беспокойный", "беспокойн"},
		{"беспокойным", "беспокойн"},
		{"беспокойных", "беспокойн"},
		{"беспокойства", "беспокойств"},
		{"беспокойстве", "беспокойств"},
		{"беспокойство", "беспокойств"},
		{"беспокойством", "беспокойств"},
		{"беспокойся", "беспок"},
		{"беспокойте", "беспокойт"},
		{"беспокойтесь", "беспокойт"},
		{"беспокою", "беспок"},
		{"беспокоюсь", "беспок"},
		{"беспокоят", "беспоко"},
		{"бесполезная", "бесполезн"},
		{"бесполезно", "бесполезн"},
		{"бесполезное", "бесполезн"},
		{"бесполезной", "бесполезн"},
		{"бесполезности", "бесполезн"},
		{"бесполезную", "бесполезн"},
		{"бесполезны", "бесполезн"},
		{"бесполезный", "бесполезн"},
		{"бесполезных", "бесполезн"},
		{"бесполый", "беспол"},
		{"беспомощен", "беспомощ"},
		{"беспомощно", "беспомощн"},
		{"беспомощное", "беспомощн"},
		{"беспомощном", "беспомощн"},
		{"беспомощности", "беспомощн"},
		{"беспомощность", "беспомощн"},
		{"беспорядка", "беспорядк"},
		{"беспорядках", "беспорядк"},
		{"беспорядке", "беспорядк"},
		{"беспорядков", "беспорядк"},
		{"беспорядок", "беспорядок"},
		{"беспорядочная", "беспорядочн"},
		{"беспорядочно", "беспорядочн"},
		{"беспорядочному", "беспорядочн"},
		{"беспорядочную", "беспорядочн"},
		{"беспошлинный", "беспошлин"},
		{"беспощадно", "беспощадн"},
		{"беспощадной", "беспощадн"},
		{"беспредельная", "беспредельн"},
		{"беспредельно", "беспредельн"},
		{"беспредельной", "беспредельн"},
		{"беспредельном", "беспредельн"},
		{"беспредельную", "беспредельн"},
		{"беспредметная", "беспредметн"},
		{"беспрекословно", "беспрекословн"},
		{"беспременно", "беспремен"},
		{"беспрерывная", "беспрерывн"},
		{"беспрерывно", "беспрерывн"},
		{"беспрерывного", "беспрерывн"},
		{"беспрерывной", "беспрерывн"},
		{"беспрерывную", "беспрерывн"},
		{"беспрерывные", "беспрерывн"},
		{"беспрерывный", "беспрерывн"},
		{"беспрерывными", "беспрерывн"},
		{"беспрерывных", "беспрерывн"},
		{"беспрестанно", "беспреста"},
		{"беспримерно", "беспримерн"},
		{"беспристрастно", "беспристрастн"},
		{"бесприютным", "бесприютн"},
		{"беспутной", "беспутн"},
		{"бессвязные", "бессвязн"},
		{"бессвязных", "бессвязн"},
		{"бессердечие", "бессердеч"},
		{"бессилен", "бессил"},
		{"бессилие", "бессил"},
		{"бессилии", "бессил"},
		{"бессилия", "бессил"},
		{"бессильно", "бессильн"},
		{"бессильного", "бессильн"},
		{"бессильное", "бессильн"},
		{"бессильною", "бессильн"},
		{"бессильны", "бессильн"},
		{"бессильный", "бессильн"},
		{"бессильных", "бессильн"},
		{"бессилья", "бессил"},
		{"бесследно", "бесследн"},
		{"бессловесный", "бессловесн"},
		{"бессловного", "бессловн"},
		{"бессменно", "бессмен"},
		{"бессмертной", "бессмертн"},
		{"бессмертную", "бессмертн"},
		{"бессмертный", "бессмертн"},
		{"бессмертья", "бессмерт"},
		{"бессмысленная", "бессмыслен"},
		{"бессмысленно", "бессмыслен"},
		{"бессмысленной", "бессмыслен"},
		{"бессмысленность", "бессмыслен"},
		{"бессмысленные", "бессмыслен"},
		{"бессмысленный", "бессмыслен"},
		{"бессмысленным", "бессмыслен"},
		{"бессмысленными", "бессмыслен"},
		{"бессмыслица", "бессмыслиц"},
		{"бессмыслице", "бессмыслиц"},
		{"бессмыслицей", "бессмыслиц"},
		{"бессмыслицу", "бессмыслиц"},
		{"бесснежная", "бесснежн"},
		{"бессовестно", "бессовестн"},
		{"бессознательная", "бессознательн"},
		{"бессознательно", "бессознательн"},
		{"бессознательной", "бессознательн"},
		{"бессознательном", "бессознательн"},
		{"бессознательным", "бессознательн"},
		{"бессоницы", "бессониц"},
		{"бессонная", "бессон"},
		{"бессонницы", "бессонниц"},
		{"бессонной", "бессон"},
		{"бессонные", "бессон"},
		{"бессонный", "бессон"},
		{"бессонных", "бессон"},
		{"бесспорно", "бесспорн"},
		{"бесспорное", "бесспорн"},
		{"бесспорной", "бесспорн"},
		{"бесстрастен", "бесстраст"},
		{"бесстрастная", "бесстрастн"},
		{"бесстрастно", "бесстрастн"},
		{"бесстрастной", "бесстрастн"},
		{"бесстрашная", "бесстрашн"},
		{"бесстрашней", "бесстрашн"},
		{"бесстыдник", "бесстыдник"},
		{"бесстыдники", "бесстыдник"},
		{"бесстыдно", "бесстыдн"},
		{"бесстыдные", "бесстыдн"},
		{"бесстыдным", "бесстыдн"},
		{"бессчетно", "бессчетн"},
		{"бестактно", "бестактн"},
		{"бестия", "бест"},
		{"бестолковую", "бестолков"},
		{"бестолковые", "бестолков"},
		{"бестолковых", "бестолков"},
		{"бестужеву", "бестужев"},
		{"бесхарактерности", "бесхарактерн"},
		{"бесцветен", "бесцвет"},
		{"бесцветно", "бесцветн"},
		{"бесцветной", "бесцветн"},
		{"бесцветны", "бесцветн"},
		{"бесцветных", "бесцветн"},
		{"бесцельная", "бесцельн"},
		{"бесцельно", "бесцельн"},
		{"бесцельный", "бесцельн"},
		{"бесцельных", "бесцельн"},
		{"бесценное", "бесцен"},
		{"бесценный", "бесцен"},
		{"бесцеремонно", "бесцеремон"},
		{"бесчеловечия", "бесчеловеч"},
		{"бесчеловечно", "бесчеловечн"},
		{"бесчестие", "бесчест"},
		{"бесчестит", "бесчест"},
		{"бесчестная", "бесчестн"},
		{"бесчестном", "бесчестн"},
		{"бесчестные", "бесчестн"},
		{"бесчестным", "бесчестн"},
		{"бесчестье", "бесчест"},
		{"бесчестья", "бесчест"},
		{"бесчинства", "бесчинств"},
		{"бесчисленно", "бесчислен"},
		{"бесчисленного", "бесчислен"},
		{"бесчисленных", "бесчислен"},
		{"бесчувствен", "бесчувств"},
		{"бесчувственна", "бесчувствен"},
		{"бесчувственная", "бесчувствен"},
		{"бесчувственного", "бесчувствен"},
		{"бесчувственному", "бесчувствен"},
		{"бесчувственный", "бесчувствен"},
		{"бесчувственным", "бесчувствен"},
		{"бесчувствия", "бесчувств"},
		{"бесшумно", "бесшумн"},
		{"бетховен", "бетхов"},
		{"бешеная", "бешен"},
		{"бешеного", "бешен"},
		{"бешеной", "бешен"},
		{"бешеном", "бешен"},
		{"бешеною", "бешен"},
		{"бешенства", "бешенств"},
		{"бешенстве", "бешенств"},
		{"бешенство", "бешенств"},
		{"бешенством", "бешенств"},
		{"бешеные", "бешен"},
		{"бешеным", "бешен"},
		{"библейские", "библейск"},
		{"библиотек", "библиотек"},
		{"библиотека", "библиотек"},
		{"библиотекарем", "библиотекар"},
		{"библиотеке", "библиотек"},
		{"библию", "библ"},
		{"библия", "библ"},
		{"бивал", "бива"},
		{"бивали", "бива"},
		{"бившего", "бивш"},
		{"бивший", "бивш"},
		{"биение", "биен"},
		{"биением", "биен"},
		{"биению", "биен"},
		{"биения", "биен"},
		{"бил", "бил"},
		{"била", "бил"},
		{"билась", "бил"},
		{"билет", "билет"},
		{"билета", "билет"},
		{"билетами", "билет"},
		{"билетах", "билет"},
		{"билетик", "билетик"},
		{"билетика", "билетик"},
		{"билетиками", "билетик"},
		{"билетов", "билет"},
		{"билетом", "билет"},
		{"билету", "билет"},
		{"билеты", "билет"},
		{"били", "бил"},
		{"биллиард", "биллиард"},
		{"биллиарда", "биллиард"},
		{"биллиарде", "биллиард"},
		{"било", "бил"},
		{"билось", "бил"},
		{"бился", "бил"},
		{"биль", "бил"},
		{"бильярд", "бильярд"},
		{"бильярде", "бильярд"},
		{"бильярдной", "бильярдн"},
		{"бильярдную", "бильярдн"},
		{"бильярдный", "бильярдн"},
		{"бинокле", "бинокл"},
		{"бинокль", "бинокл"},
		{"биография", "биограф"},
		{"бирже", "бирж"},
		{"биржи", "бирж"},
		{"бис", "бис"},
		{"бисер", "бисер"},
		{"бисером", "бисер"},
		{"бисквитов", "бисквит"},
		{"бисквиты", "бисквит"},
		{"битв", "битв"},
		{"битве", "битв"},
		{"битву", "битв"},
		{"битвы", "битв"},
		{"битком", "битк"},
		{"битое", "бит"},
		{"биты", "бит"},
		{"битый", "бит"},
		{"бить", "бит"},
		{"битье", "бит"},
		{"биться", "бит"},
		{"бифштекс", "бифштекс"},
		{"бифштекса", "бифштекс"},
		{"бицепсы", "бицепс"},
		{"бич", "бич"},
		{"бичами", "бич"},
		{"бичурину", "бичурин"},
		{"бишь", "биш"},
		{"бла", "бла"},
		{"благ", "благ"},
		{"блага", "благ"},
		{"благим", "благ"},
		{"благих", "благ"},
		{"благо", "благ"},
		{"благовещение", "благовещен"},
		{"благовиднее", "благовидн"},
		{"благоволите", "благовол"},
		{"благовония", "благовон"},
		{"благовонной", "благовон"},
		{"благовоспитанным", "благовоспита"},
		{"благовоспитанных", "благовоспита"},
		{"благоговеете", "благоговеет"},
		{"благоговейно", "благоговейн"},
		{"благоговейного", "благоговейн"},
		{"благоговейным", "благоговейн"},
		{"благоговела", "благоговел"},
		{"благоговение", "благоговен"},
		{"благоговением", "благоговен"},
		{"благоговеть", "благоговет"},
		{"благоговею", "благогов"},
		{"благодарен", "благодар"},
		{"благодари", "благодар"},
		{"благодарил", "благодар"},
		{"благодарила", "благодар"},
		{"благодарили", "благодар"},
		{"благодарим", "благодар"},
		{"благодарить", "благодар"},
		{"благодарна", "благодарн"},
		{"благодарная", "благодарн"},
		{"благодарно", "благодарн"},
		{"благодарном", "благодарн"},
		{"благодарности", "благодарн"},
		{"благодарностию", "благодарн"},
		{"благодарность", "благодарн"},
		{"благодарностью", "благодарн"},
		{"благодарю", "благодар"},
		{"благодаря", "благодар"},
		{"благодати", "благодат"},
		{"благодать", "благода"},
		{"благодетелей", "благодетел"},
		{"благодетелем", "благодетел"},
		{"благодетели", "благодетел"},
		{"благодетель", "благодетел"},
		{"благодетельной", "благодетельн"},
		{"благодетельные", "благодетельн"},
		{"благодетельствовал", "благодетельствова"},
		{"благодетельствовать", "благодетельствова"},
		{"благодетельствует", "благодетельств"},
		{"благодетеля", "благодетел"},
		{"благодеяние", "благодеян"},
		{"благодеяний", "благодеян"},
		{"благодеяния", "благодеян"},
		{"благодушном", "благодушн"},
		{"благолепие", "благолеп"},
		{"благонадежный", "благонадежн"},
		{"благонамереннейший", "благонамерен"},
		{"благонамеренного", "благонамерен"},
		{"благонравной", "благонравн"},
		{"благонравны", "благонравн"},
		{"благополучно", "благополучн"},
		{"благоприобретенные", "благоприобретен"},
		{"благоприобретенным", "благоприобретен"},
		{"благоприятного", "благоприятн"},
		{"благоприятное", "благоприятн"},
		{"благоприятный", "благоприятн"},
		{"благоприятствовавшие", "благоприятствова"},
		{"благоразумие", "благоразум"},
		{"благоразумия", "благоразум"},
		{"благоразумная", "благоразумн"},
		{"благоразумнейшей", "благоразумн"},
		{"благоразумно", "благоразумн"},
		{"благоразумных", "благоразумн"},
		{"благороден", "благород"},
		{"благородие", "благород"},
		{"благородная", "благородн"},
		{"благороднее", "благородн"},
		{"благороднейшая", "благородн"},
		{"благороднейший", "благородн"},
		{"благородно", "благородн"},
		{"благородного", "благородн"},
		{"благородное", "благородн"},
		{"благородной", "благородн"},
		{"благородном", "благородн"},
		{"благородною", "благородн"},
		{"благородны", "благородн"},
		{"благородные", "благородн"},
		{"благородный", "благородн"},
		{"благородным", "благородн"},
		{"благородных", "благородн"},
		{"благородства", "благородств"},
		{"благородстве", "благородств"},
		{"благородство", "благородств"},
		{"благородством", "благородств"},
		{"благосклонно", "благосклон"},
		{"благосклонной", "благосклон"},
		{"благосклонный", "благосклон"},
		{"благословением", "благословен"},
		{"благословениями", "благословен"},
		{"благословенно", "благословен"},
		{"благословенной", "благословен"},
		{"благословенном", "благословен"},
		{"благословенный", "благословен"},
		{"благослови", "благослов"},
		{"благословил", "благослов"},
		{"благословила", "благослов"},
		{"благословили", "благослов"},
		{"благословит", "благослов"},
		{"благословить", "благослов"},
		{"благословишь", "благослов"},
		{"благословлю", "благословл"},
		{"благословляет", "благословля"},
		{"благословляла", "благословля"},
		{"благословлять", "благословля"},
		{"благословляю", "благословля"},
		{"благословляя", "благословл"},
		{"благословясь", "благослов"},
		{"благословят", "благослов"},
		{"благосостояние", "благосостоян"},
		{"благосостояния", "благосостоян"},
		{"благости", "благост"},
		{"благость", "благост"},
		{"благотворители", "благотворител"},
		{"благотворительности", "благотворительн"},
		{"благотворительность", "благотворительн"},
		{"благотворны", "благотворн"},
		{"благотворный", "благотворн"},
		{"благоугодным", "благоугодн"},
		{"благоусмотрение", "благоусмотрен"},
		{"благоуханное", "благоуха"},
		{"благоуханный", "благоуха"},
		{"благоуханных", "благоуха"},
		{"благоуханье", "благоухан"},
		{"благоухающая", "благоуха"},
		{"благоухающею", "благоуха"},
		{"благочестиво", "благочестив"},
		{"блаженно", "блажен"},
		{"блаженное", "блажен"},
		{"блаженной", "блажен"},
		{"блаженные", "блажен"},
		{"блаженный", "блажен"},
		{"блаженным", "блажен"},
		{"блаженных", "блажен"},
		{"блаженства", "блаженств"},
		{"блаженстве", "блаженств"},
		{"блаженство", "блаженств"},
		{"блаженствовал", "блаженствова"},
		{"блаженствовали", "блаженствова"},
		{"блаженствовать", "блаженствова"},
		{"блаженством", "блаженств"},
		{"блаженствую", "блаженств"},
		{"блаженствуют", "блаженств"},
		{"блажной", "блажн"},
		{"блажь", "блаж"},
		{"блажью", "блаж"},
		{"бланбеками", "бланбек"},
		{"бланбеки", "бланбек"},
		{"бледен", "блед"},
		{"бледна", "бледн"},
		{"бледная", "бледн"},
		{"бледневший", "бледневш"},
		{"бледнее", "бледн"},
		{"бледнеет", "бледнеет"},
		{"бледнела", "бледнел"},
		{"бледненькое", "бледненьк"},
		{"бледнеть", "бледнет"},
		{"бледнеют", "бледнеют"},
		{"бледнея", "бледне"},
		{"бледно", "бледн"},
		{"бледного", "бледн"},
		{"бледное", "бледн"},
		{"бледной", "бледн"},
		{"бледном", "бледн"},
		{"бледность", "бледност"},
		{"бледночернильным", "бледночернильн"},
		{"бледную", "бледн"},
		{"бледны", "бледн"},
		{"бледные", "бледн"},
		{"бледный", "бледн"},
		{"бледным", "бледн"},
		{"бледными", "бледн"},
		{"бледных", "бледн"},
		{"блеклыми", "блекл"},
		{"блеск", "блеск"},
		{"блеска", "блеск"},
		{"блеске", "блеск"},
		{"блеском", "блеск"},
		{"блеснет", "блеснет"},
		{"блесну", "блесн"},
		{"блеснувшим", "блеснувш"},
		{"блеснул", "блеснул"},
		{"блеснула", "блеснул"},
		{"блеснули", "блеснул"},
		{"блеснуло", "блеснул"},
		{"блеснуть", "блеснут"},
		{"блестели", "блестел"},
		{"блестело", "блестел"},
		{"блестит", "блест"},
		{"блестками", "блестк"},
		{"блестят", "блест"},
		{"блестящей", "блестя"},
		{"блестящем", "блестя"},
		{"блестящему", "блестя"},
		{"блестящею", "блестя"},
		{"блестящие", "блестя"},
		{"блестящий", "блестя"},
		{"блестящим", "блестя"},
		{"блестящих", "блестя"},
		{"блестящую", "блестя"},
		{"блещет", "блещет"},
		{"блещешь", "блещеш"},
		{"блещут", "блещут"},
		{"блещущей", "блещущ"},
		{"блещущий", "блещущ"},
		{"блещущим", "блещущ"},
		{"блеющие", "блеющ"},
		{"ближайшее", "ближайш"},
		{"ближайшем", "ближайш"},
		{"ближайшие", "ближайш"},
		{"ближайший", "ближайш"},
		{"ближайшую", "ближайш"},
		{"ближе", "ближ"},
		{"ближней", "ближн"},
		{"ближний", "ближн"},
		{"ближним", "ближн"},
		{"близ", "близ"},
		{"близится", "близ"},
		{"близка", "близк"},
		{"близки", "близк"},
		{"близкие", "близк"},
		{"близким", "близк"},
		{"близкими", "близк"},
		{"близких", "близк"},
		{"близко", "близк"},
		{"близкого", "близк"},
		{"близкое", "близк"},
		{"близкой", "близк"},
		{"близком", "близк"},
		{"близкую", "близк"},
		{"близлежащие", "близлежа"},
		{"близок", "близок"},
		{"близорука", "близорук"},
		{"близость", "близост"},
		{"близь", "близ"},
		{"блин", "блин"},
		{"блинах", "блин"},
		{"блинов", "блин"},
		{"блины", "блин"},
		{"блиставшего", "блиста"},
		{"блиставшее", "блиста"},
		{"блистал", "блиста"},
		{"блистали", "блиста"},
		{"блистало", "блиста"},
		{"блистанье", "блистан"},
		{"блистательной", "блистательн"},
		{"блистательным", "блистательн"},
		{"блистательных", "блистательн"},
		{"блистать", "блиста"},
		{"блистают", "блиста"},
		{"блистающий", "блиста"},
		{"блистающими", "блиста"},
		{"бло", "бло"},
		{"блок", "блок"},
		{"блонд", "блонд"},
		{"блондах", "блонд"},
		{"блондин", "блондин"},
		{"блондинка", "блондинк"},
		{"блондинки", "блондинк"},
		{"блондины", "блондин"},
		{"блонды", "блонд"},
		{"блохи", "блох"},
		{"блудницу", "блудниц"},
		{"блуждает", "блужда"},
		{"блуждаете", "блужда"},
		{"блуждал", "блужда"},
		{"блуждали", "блужда"},
		{"блуждающий", "блужда"},
		{"блуждая", "блужд"},
		{"блузе", "блуз"},
		{"блузу", "блуз"},
		{"блю", "блю"},
		{"блюд", "блюд"},
		{"блюда", "блюд"},
		{"блюдами", "блюд"},
		{"блюде", "блюд"},
		{"блюдечками", "блюдечк"},
		{"блюдечке", "блюдечк"},
		{"блюдечко", "блюдечк"},
		{"блюдо", "блюд"},
		{"блюдом", "блюд"},
		{"блюдце", "блюдц"},
		{"блюсти", "блюст"},
		{"блюстителя", "блюстител"},
		{"бо", "бо"},
		{"бобах", "боб"},
		{"бобик", "бобик"},
		{"бобика", "бобик"},
		{"бобику", "бобик"},
		{"бобов", "боб"},
		{"бобровых", "бобров"},
		{"бобы", "боб"},
		{"бог", "бог"},
		{"бога", "бог"},
		{"богат", "богат"},
		{"богата", "богат"},
		{"богатая", "богат"},
		{"богатеет", "богатеет"},
		{"богатеют", "богатеют"},
		{"богато", "богат"},
		{"богатого", "богат"},
		{"богатой", "богат"},
		{"богатом", "богат"},
		{"богатому", "богат"},
		{"богатства", "богатств"},
		{"богатство", "богатств"},
		{"богатую", "богат"},
		{"богаты", "богат"},
		{"богатые", "богат"},
		{"богатый", "богат"},
		{"богатым", "богат"},
		{"богатыре", "богатыр"},
		{"богатырь", "богатыр"},
		{"богатырях", "богатыр"},
		{"богатых", "богат"},
		{"богач", "богач"},
		{"богаче", "богач"},
		{"богачи", "богач"},
		{"богачом", "богач"},
		{"богданович", "богданович"},
		{"богданыч", "богданыч"},
		{"боге", "бог"},
		{"богема", "богем"},
		{"богемии", "богем"},
		{"богемскую", "богемск"},
		{"боги", "бог"},
		{"богиня", "богин"},
		{"богом", "бог"},
		{"богомолье", "богомол"},
		{"богомольно", "богомольн"},
		{"богомольной", "богомольн"},
		{"богомольные", "богомольн"},
		{"богородица", "богородиц"},
		{"богородицу", "богородиц"},
		{"богохульник", "богохульник"},
		{"богу", "бог"},
		{"бодр", "бодр"},
		{"бодрая", "бодр"},
		{"бодрился", "бодр"},
		{"бодро", "бодр"},
		{"бодрое", "бодр"},
		{"бодрой", "бодр"},
		{"бодром", "бодр"},
		{"бодрости", "бодрост"},
		{"бодрость", "бодрост"},
		{"бодростью", "бодрост"},
		{"бодрствовал", "бодрствова"},
		{"бодрствовали", "бодрствова"},
		{"бодрствую", "бодрств"},
		{"бодрый", "бодр"},
		{"бодрым", "бодр"},
		{"бодрых", "бодр"},
		{"бодягой", "бодяг"},
		{"боевой", "боев"},
		{"боем", "бо"},
		{"боец", "боец"},
		{"боже", "бож"},
		{"божества", "божеств"},
		{"божественно", "божествен"},
		{"божественной", "божествен"},
		{"божественным", "божествен"},
		{"божественных", "божествен"},
		{"божество", "божеств"},
		{"божие", "бож"},
		{"божией", "бож"},
		{"божиим", "бож"},
		{"божий", "бож"},
		{"божилась", "бож"},
		{"божится", "бож"},
		{"божию", "бож"},
		{"божия", "бож"},
		{"божьего", "бож"},
		{"божьей", "бож"},
		{"божьим", "бож"},
		{"божья", "бож"},
		{"боитесь", "бо"},
		{"боится", "бо"},
		{"боишься", "бо"},
		{"бой", "бо"},
		{"бойкая", "бойк"},
		{"бойкие", "бойк"},
		{"бойкий", "бойк"},
		{"бойким", "бойк"},
		{"бойко", "бойк"},
		{"бойкого", "бойк"},
		{"бойкое", "бойк"},
		{"бойни", "бойн"},
		{"бойся", "бо"},
		{"бойтесь", "бойт"},
		{"бок", "бок"},
		{"бока", "бок"},
		{"бокал", "бока"},
		{"бокала", "бока"},
		{"бокалах", "бокал"},
		{"бокале", "бокал"},
		{"бокалу", "бокал"},
		{"бокам", "бок"},
		{"боками", "бок"},
		{"боки", "бок"},
		{"бокля", "бокл"},
		{"боков", "бок"},
		{"бокового", "боков"},
		{"боковой", "боков"},
		{"боковом", "боков"},
		{"боковую", "боков"},
		{"боком", "бок"},
		{"боку", "бок"},
		{"болван", "болва"},
		{"болваном", "болван"},
		{"болвану", "болван"},
		{"боле", "бол"},
		{"более", "бол"},
		{"болеет", "болеет"},
		{"болезная", "болезн"},
		{"болезней", "болезн"},
		{"болезненная", "болезнен"},
		{"болезненно", "болезнен"},
		{"болезненного", "болезнен"},
		{"болезненное", "болезнен"},
		{"болезненной", "болезнен"},
		{"болезненном", "болезнен"},
		{"болезненность", "болезнен"},
		{"болезненную", "болезнен"},
		{"болезненные", "болезнен"},
		{"болезненный", "болезнен"},
		{"болезненным", "болезнен"},
		{"болезненных", "болезнен"},
		{"болезни", "болезн"},
		{"болезнию", "болезн"},
		{"болезнь", "болезн"},
		{"болезнью", "болезн"},
		{"болезнями", "болезн"},
		{"болезнях", "болезн"},
		{"болел", "болел"},
		{"болела", "болел"},
		{"болели", "болел"},
		{"болело", "болел"},
		{"болен", "бол"},
		{"болеть", "болет"},
		{"болеют", "болеют"},
		{"боли", "бол"},
		{"болит", "бол"},
		{"болосы", "болос"},
		{"болот", "болот"},
		{"болота", "болот"},
		{"болотистым", "болотист"},
		{"болотной", "болотн"},
		{"болотные", "болотн"},
		{"болотный", "болотн"},
		{"болото", "болот"},
		{"болотом", "болот"},
		{"болтает", "болта"},
		{"болтается", "болта"},
		{"болтай", "болта"},
		{"болтал", "болта"},
		{"болталась", "болта"},
		{"болтали", "болта"},
		{"болтались", "болта"},
		{"болтался", "болта"},
		{"болтать", "болта"},
		{"болтаю", "болта"},
		{"болтают", "болта"},
		{"болтаются", "болта"},
		{"болтаясь", "болт"},
		{"болтлив", "болтл"},
		{"болтлива", "болтлив"},
		{"болтливая", "болтлив"},
		{"болтливы", "болтлив"},
		{"болтовней", "болтовн"},
		{"болтовни", "болтовн"},
		{"болтовню", "болтовн"},
		{"болтовня", "болтовн"},
		{"болтун", "болтун"},
		{"болтунишки", "болтунишк"},
		{"болтушка", "болтушк"},
		{"боль", "бол"},
		{"больна", "больн"},
		{"больная", "больн"},
		{"больнее", "больн"},
		{"больней", "больн"},
		{"больниц", "больниц"},
		{"больница", "больниц"},
		{"больнице", "больниц"},
		{"больницу", "больниц"},
		{"больницы", "больниц"},
		{"больно", "больн"},
		{"больного", "больн"},
		{"больное", "больн"},
		{"больной", "больн"},
		{"больном", "больн"},
		{"больному", "больн"},
		{"больною", "больн"},
		{"больную", "больн"},
		{"больны", "больн"},
		{"больные", "больн"},
		{"больным", "больн"},
		{"больными", "больн"},
		{"больных", "больн"},
		{"большая", "больш"},
		{"больше", "больш"},
		{"большего", "больш"},
		{"большеголовый", "большеголов"},
		{"большее", "больш"},
		{"большей", "больш"},
		{"большем", "больш"},
		{"большею", "больш"},
		{"большие", "больш"},
		{"большим", "больш"},
		{"большими", "больш"},
		{"большинстве", "большинств"},
		{"большинство", "большинств"},
		{"большинством", "большинств"},
		{"больших", "больш"},
		{"большого", "больш"},
		{"большое", "больш"},
		{"большой", "больш"},
		{"большом", "больш"},
		{"большому", "больш"},
		{"большою", "больш"},
		{"большую", "больш"},
		{"болью", "бол"},
		{"болят", "бол"},
		{"бомбами", "бомб"},
		{"бонжур", "бонжур"},
		{"бонне", "бон"},
		{"бор", "бор"},
		{"бореля", "борел"},
		{"борется", "борет"},
		{"борис", "борис"},
		{"бориса", "борис"},
		{"борисович", "борисович"},
		{"борису", "борис"},
		{"бормотал", "бормота"},
		{"бормотала", "бормота"},
		{"бормотали", "бормота"},
		{"бормотание", "бормотан"},
		{"бормотаний", "бормотан"},
		{"бормочет", "бормочет"},
		{"бормочешь", "бормочеш"},
		{"боров", "бор"},
		{"борода", "бород"},
		{"бороде", "бород"},
		{"бородкой", "бородк"},
		{"бородой", "бород"},
		{"бороду", "бород"},
		{"бороды", "бород"},
		{"бороздою", "борозд"},
		{"боролась", "борол"},
		{"боролись", "борол"},
		{"боролось", "борол"},
		{"боролся", "борол"},
		{"бороться", "борот"},
		{"борт", "борт"},
		{"борта", "борт"},
		{"бортов", "борт"},
		{"борты", "борт"},
		{"борцов", "борц"},
		{"борьба", "борьб"},
		{"борьбах", "борьб"},
		{"борьбе", "борьб"},
		{"борьбой", "борьб"},
		{"борьбу", "борьб"},
		{"борьбы", "борьб"},
		{"борются", "борют"},
		{"борясь", "бор"},
		{"босая", "бос"},
		{"босенькая", "босеньк"},
		{"босиком", "босик"},
		{"боскете", "боскет"},
		{"босой", "бос"},
		{"босоногих", "босоног"},
		{"бостон", "бостон"},
		{"босу", "бос"},
		{"босые", "бос"},
		{"ботаник", "ботаник"},
		{"ботвинью", "ботвин"},
		{"ботвинья", "ботвин"},
		{"ботинками", "ботинк"},
		{"ботинках", "ботинк"},
		{"ботинки", "ботинк"},
		{"ботиночки", "ботиночк"},
		{"бочка", "бочк"},
		{"бочки", "бочк"},
		{"бочкой", "бочк"},
		{"бочком", "бочк"},
		{"бочонка", "бочонк"},
		{"бочонок", "бочонок"},
		{"бою", "бо"},
		{"боюсь", "бо"},
		{"боя", "бо"},
		{"боявшиеся", "боя"},
		{"боявшийся", "боя"},
		{"боязливее", "боязлив"},
		{"боязливо", "боязлив"},
		{"боязливый", "боязлив"},
		{"боязни", "боязн"},
		{"боязнь", "боязн"},
		{"боязнью", "боязн"},
		{"боялась", "боя"},
		{"боялись", "боя"},
		{"боялся", "боя"},
		{"боясь", "бо"},
		{"боятся", "бо"},
		{"бояться", "боя"},
		{"браво", "брав"},
		{"бравое", "брав"},
		{"брак", "брак"},
		{"брака", "брак"},
		{"браке", "брак"},
		{"браки", "брак"},
		{"браком", "брак"},
		{"браку", "брак"},
		{"брал", "брал"},
		{"брала", "брал"},
		{"бралась", "брал"},
		{"брали", "брал"},
		{"брался", "брал"},
		{"бранда", "бранд"},
		{"брани", "бран"},
		{"бранил", "бран"},
		{"бранила", "бран"},
		{"бранили", "бран"},
		{"бранились", "бран"},
		{"бранит", "бран"},
		{"браните", "бран"},
		{"бранится", "бран"},
		{"бранить", "бран"},
		{"браниться", "бран"},
		{"бранчив", "бранч"},
		{"брань", "бран"},
		{"бранью", "бран"},
		{"бранюсь", "бран"},
		{"бранясь", "бран"},
		{"бранят", "бран"},
		{"браслет", "браслет"},
		{"браслете", "браслет"},
		{"браслетом", "браслет"},
		{"браслеты", "браслет"},
		{"брат", "брат"},
		{"брата", "брат"},
		{"брате", "брат"},
		{"братец", "братец"},
		{"братией", "брат"},
		{"братишка", "братишк"},
		{"братом", "брат"},
		{"братской", "братск"},
		{"братства", "братств"},
		{"брату", "брат"},
		{"братца", "братц"},
		{"братце", "братц"},
		{"братцем", "братц"},
		{"братцу", "братц"},
		{"братцы", "братц"},
		{"брать", "брат"},
		{"братьев", "брат"},
		{"братьи", "брат"},
		{"браться", "брат"},
		{"братья", "брат"},
		{"братьям", "брат"},
		{"брачной", "брачн"},
		{"брачные", "брачн"},
		{"бревна", "бревн"},
		{"брег", "брег"},
		{"бред", "бред"},
		{"бреда", "бред"},
		{"бреде", "бред"},
		{"бредил", "бред"},
		{"бредила", "бред"},
		{"бредит", "бред"},
		{"бредите", "бред"},
		{"бредить", "бред"},
		{"бредишь", "бред"},
		{"бредом", "бред"},
		{"бреду", "бред"},
		{"бредят", "бред"},
		{"бреется", "бреет"},
		{"брежжит", "брежж"},
		{"брежу", "бреж"},
		{"брезгает", "брезга"},
		{"брезгайте", "брезга"},
		{"брезгливости", "брезглив"},
		{"брезговал", "брезгова"},
		{"брезжит", "брезж"},
		{"брел", "брел"},
		{"брелок", "брелок"},
		{"брелоков", "брелок"},
		{"бременем", "бремен"},
		{"бремени", "бремен"},
		{"бремя", "брем"},
		{"бренно", "брен"},
		{"бренный", "брен"},
		{"бренчало", "бренча"},
		{"бренчу", "бренч"},
		{"бретер", "бретер"},
		{"бригаде", "бригад"},
		{"бригадой", "бригад"},
		{"бригаду", "бригад"},
		{"бриллиант", "бриллиант"},
		{"бриллианта", "бриллиант"},
		{"бриллиантовый", "бриллиантов"},
		{"брился", "брил"},
		{"брит", "брит"},
		{"бритая", "брит"},
		{"бритва", "бритв"},
		{"бритвы", "бритв"},
		{"бритой", "брит"},
		{"брить", "брит"},
		{"бриться", "брит"},
		{"бровей", "бров"},
		{"брови", "бров"},
		{"бровь", "бров"},
		{"бровью", "бров"},
		{"бровями", "бров"},
		{"бровях", "бров"},
		{"бродивших", "брод"},
		{"бродил", "брод"},
		{"бродила", "брод"},
		{"бродили", "брод"},
		{"бродит", "брод"},
		{"бродить", "брод"},
		{"бродишь", "брод"},
		{"бродяга", "бродяг"},
		{"бродяги", "бродяг"},
		{"бродягу", "бродяг"},
		{"бродяжничай", "бродяжнича"},
		{"бродяжничества", "бродяжничеств"},
		{"бродят", "брод"},
		{"бродячие", "бродяч"},
		{"бронза", "бронз"},
		{"бронзовую", "бронзов"},
		{"бросает", "броса"},
		{"бросаете", "броса"},
		{"бросаетесь", "броса"},
		{"бросается", "броса"},
		{"бросал", "броса"},
		{"бросала", "броса"},
		{"бросалась", "броса"},
		{"бросались", "броса"},
		{"бросало", "броса"},
		{"бросался", "броса"},
		{"бросать", "броса"},
		{"бросаю", "броса"},
		{"бросают", "броса"},
		{"бросаются", "броса"},
		{"бросающееся", "броса"},
		{"бросающую", "броса"},
		{"бросая", "брос"},
		{"бросаясь", "брос"},
		{"бросив", "брос"},
		{"бросившись", "брос"},
		{"бросил", "брос"},
		{"бросила", "брос"},
		{"бросилась", "брос"},
		{"бросили", "брос"},
		{"бросились", "брос"},
		{"бросило", "брос"},
		{"бросилось", "брос"},
		{"бросился", "брос"},
		{"бросим", "брос"},
		{"бросит", "брос"},
		{"бросится", "брос"},
		{"бросить", "брос"},
		{"броситься", "брос"},
		{"бросишь", "брос"},
		{"брось", "бро"},
		{"бросьте", "бросьт"},
		{"бросят", "брос"},
		{"бросятся", "брос"},
		{"брошен", "брош"},
		{"брошена", "брош"},
		{"брошенного", "брошен"},
		{"брошенное", "брошен"},
		{"брошенном", "брошен"},
		{"брошенный", "брошен"},
		{"брошено", "брош"},
		{"брошены", "брош"},
		{"брошка", "брошк"},
		{"брошкой", "брошк"},
		{"брошу", "брош"},
		{"брошусь", "брош"},
		{"брошь", "брош"},
		{"брошюр", "брошюр"},
		{"брошюру", "брошюр"},
		{"брошюры", "брошюр"},
		{"бррр", "бррр"},
		{"брудершафт", "брудершафт"},
		{"брусничной", "брусничн"},
		{"брут", "брут"},
		{"брызг", "брызг"},
		{"брызгает", "брызга"},
		{"брызги", "брызг"},
		{"брызжут", "брызжут"},
		{"брызнув", "брызнув"},
		{"брызнул", "брызнул"},
		{"брызнула", "брызнул"},
		{"брызнули", "брызнул"},
		{"брызнут", "брызнут"},
		{"брюжжит", "брюжж"},
		{"брюзгами", "брюзг"},
		{"брюзгливая", "брюзглив"},
		{"брюзгливо", "брюзглив"},
		{"брюзгливою", "брюзглив"},
		{"брюзгливые", "брюзглив"},
		{"брюзжащие", "брюзжа"},
		{"брюзжит", "брюзж"},
		{"брюках", "брюк"},
		{"брюквой", "брюкв"},
		{"брюки", "брюк"},
		{"брюнет", "брюнет"},
		{"брюнеты", "брюнет"},
		{"брюсову", "брюсов"},
		{"брюхе", "брюх"},
		{"брюхо", "брюх"},
		{"брюшком", "брюшк"},
		{"брякает", "бряка"},
		{"брякну", "брякн"},
		{"брякнул", "брякнул"},
		{"брякнула", "брякнул"},
		{"брякнуть", "брякнут"},
		{"бряцает", "бряца"},
		{"бубен", "буб"},
		{"бубенчик", "бубенчик"},
		{"бубенчики", "бубенчик"},
		{"бубенчуками", "бубенчук"},
		{"бубновый", "бубнов"},
		{"бубны", "бубн"},
		{"бугорках", "бугорк"},
		{"бугорки", "бугорк"},
		{"буде", "буд"},
		{"будем", "буд"},
		{"будемте", "будемт"},
		{"будет", "будет"},
		{"будете", "будет"},
		{"будешь", "будеш"},
		{"буди", "буд"},
		{"будил", "буд"},
		{"будила", "буд"},
		{"будили", "буд"},
		{"будило", "буд"},
		{"будируя", "будиру"},
		{"будит", "буд"},
		{"будить", "буд"},
		{"буднее", "будн"},
		{"будней", "будн"},
		{"будни", "будн"},
		{"будничное", "будничн"},
		{"будничной", "будничн"},
		{"будничном", "будничн"},
		{"будничные", "будничн"},
		{"будничными", "будничн"},
		{"будничных", "будничн"},
		{"будням", "будн"},
		{"буднях", "будн"},
		{"будок", "будок"},
		{"будоражишь", "будораж"},
		{"будочник", "будочник"},
		{"будочника", "будочник"},
		{"будто", "будт"},
		{"буду", "буд"},
		{"будуар", "будуар"},
		{"будуаров", "будуар"},
		{"будут", "будут"},
		{"будучи", "будуч"},
		{"будущая", "будущ"},
		{"будущего", "будущ"},
		{"будущее", "будущ"},
		{"будущей", "будущ"},
		{"будущем", "будущ"},
		{"будущему", "будущ"},
		{"будущие", "будущ"},
		{"будущий", "будущ"},
		{"будущим", "будущ"},
		{"будущих", "будущ"},
		{"будущности", "будущн"},
		{"будущность", "будущн"},
		{"будущую", "будущ"},
		{"будь", "буд"},
		{"будьте", "будьт"},
		{"будя", "буд"},
		{"будят", "буд"},
		{"буеракам", "буерак"},
		{"бузиной", "бузин"},
		{"буйная", "буйн"},
		{"буйно", "буйн"},
		{"буйного", "буйн"},
		{"буйной", "буйн"},
		{"буйном", "буйн"},
		{"буйному", "буйн"},
		{"буйный", "буйн"},
		{"буйным", "буйн"},
		{"буйных", "буйн"},
		{"буйственных", "буйствен"},
		{"буйство", "буйств"},
		{"буйством", "буйств"},
		{"букашками", "букашк"},
		{"буква", "букв"},
		{"буквально", "буквальн"},
		{"буквальное", "буквальн"},
		{"буквами", "букв"},
		{"буквой", "букв"},
		{"букву", "букв"},
		{"буквы", "букв"},
		{"букет", "букет"},
		{"букета", "букет"},
		{"букетами", "букет"},
		{"букетом", "букет"},
		{"букеты", "букет"},
		{"буки", "бук"},
		{"букинистами", "букинист"},
		{"булавки", "булавк"},
		{"булавку", "булавк"},
		{"булавой", "булав"},
		{"булка", "булк"},
		{"булки", "булк"},
		{"булку", "булк"},
		{"булок", "булок"},
		{"булочек", "булочек"},
		{"булочками", "булочк"},
		{"булочник", "булочник"},
		{"булочную", "булочн"},
		{"бултых", "булт"},
		{"буль", "бул"},
		{"бульвар", "бульвар"},
		{"бульвара", "бульвар"},
		{"бульваре", "бульвар"},
		{"бульвару", "бульвар"},
		{"бульдоге", "бульдог"},
		{"бульон", "бульон"},
		{"бумаг", "бумаг"},
		{"бумага", "бумаг"},
		{"бумагам", "бумаг"},
		{"бумагами", "бумаг"},
		{"бумагах", "бумаг"},
		{"бумаге", "бумаг"},
		{"бумаги", "бумаг"},
		{"бумагой", "бумаг"},
		{"бумагу", "бумаг"},
		{"бумажечки", "бумажечк"},
		{"бумажка", "бумажк"},
		{"бумажками", "бумажк"},
		{"бумажке", "бумажк"},
		{"бумажки", "бумажк"},
		{"бумажку", "бумажк"},
		{"бумажник", "бумажник"},
		{"бумажнике", "бумажник"},
		{"бумажной", "бумажн"},
		{"бумажный", "бумажн"},
		{"бумажным", "бумажн"},
		{"бумбия", "бумб"},
		{"бунтовать", "бунтова"},
		{"бунтующее", "бунт"},
		{"бунчуки", "бунчук"},
		{"буран", "бура"},
		{"бурду", "бурд"},
		{"буре", "бур"},
		{"бурей", "бур"},
		{"буржуазного", "буржуазн"},
		{"бури", "бур"},
		{"буркалы", "буркал"},
		{"бурмейстер", "бурмейстер"},
		{"бурно", "бурн"},
		{"бурного", "бурн"},
		{"бурное", "бурн"},
		{"бурною", "бурн"},
		{"бурнус", "бурнус"},
		{"бурнусик", "бурнусик"},
		{"бурнусике", "бурнусик"},
		{"бурный", "бурн"},
		{"бурными", "бурн"},
		{"бурных", "бурн"},
		{"бурого", "бур"},
		{"бурой", "бур"},
		{"бурсак", "бурсак"},
		{"бурш", "бурш"},
		{"бурша", "бурш"},
		{"бурый", "бур"},
		{"бурь", "бур"},
		{"бурьян", "бурья"},
		{"бурьяне", "бурьян"},
		{"бурьяном", "бурьян"},
		{"бурю", "бур"},
		{"буря", "бур"},
		{"бурями", "бур"},
		{"бурях", "бур"},
		{"бутончик", "бутончик"},
		{"бутошник", "бутошник"},
		{"бутылка", "бутылк"},
		{"бутылками", "бутылк"},
		{"бутылки", "бутылк"},
		{"бутылкой", "бутылк"},
		{"бутылкою", "бутылк"},
		{"бутылку", "бутылк"},
		{"бутылок", "бутылок"},
		{"бутылочки", "бутылочк"},
		{"бутылочку", "бутылочк"},
		{"бутылочным", "бутылочн"},
		{"буфера", "буфер"},
		{"буфет", "буфет"},
		{"буфета", "буфет"},
		{"буфете", "буфет"},
		{"буфету", "буфет"},
		{"буфетчик", "буфетчик"},
		{"буфетчика", "буфетчик"},
		{"буфеты", "буфет"},
		{"буффон", "буффон"},
		{"буха", "бух"},
		{"бухте", "бухт"},
		{"бушевало", "бушева"},
		{"бушует", "буш"},
		{"бушующих", "буш"},
		{"буян", "буя"},
		{"буянил", "буян"},
		{"буянила", "буян"},
		{"бы", "бы"},
		{"быв", "быв"},
		{"бываем", "быва"},
		{"бывает", "быва"},
		{"бываете", "быва"},
		{"бываешь", "быва"},
		{"бывал", "быва"},
		{"бывала", "быва"},
		{"бывали", "быва"},
		{"бывало", "быва"},
		{"бывалое", "бывал"},
		{"бывалую", "бывал"},
		{"бывалые", "бывал"},
		{"бывалый", "бывал"},
		{"бывалых", "бывал"},
		{"бывать", "быва"},
		{"бываю", "быва"},
		{"бывают", "быва"},
		{"бывшая", "бывш"},
		{"бывшего", "бывш"},
		{"бывшее", "бывш"},
		{"бывшей", "бывш"},
		{"бывшем", "бывш"},
		{"бывшему", "бывш"},
		{"бывши", "бывш"},
		{"бывшие", "бывш"},
		{"бывший", "бывш"},
		{"бывшим", "бывш"},
		{"бывшими", "бывш"},
		{"бывших", "бывш"},
		{"бывшую", "бывш"},
		{"был", "был"},
		{"была", "был"},
		{"былая", "был"},
		{"были", "был"},
		{"былинки", "былинк"},
		{"было", "был"},
		{"былое", "был"},
		{"былой", "был"},
		{"былые", "был"},
		{"былым", "был"},
		{"былых", "был"},
		{"быльем", "был"},
		{"быстрая", "быстр"},
		{"быстрее", "быстр"},
		{"быстрей", "быстр"},
		{"быстрины", "быстрин"},
		{"быстро", "быстр"},
		{"быстрого", "быстр"},
		{"быстрой", "быстр"},
		{"быстроте", "быстрот"},
		{"быстротой", "быстрот"},
		{"быстротою", "быстрот"},
		{"быстроту", "быстрот"},
		{"быстрые", "быстр"},
		{"быстрый", "быстр"},
		{"быстрым", "быстр"},
		{"быт", "быт"},
		{"быта", "быт"},
		{"быте", "быт"},
		{"бытие", "быт"},
		{"бытийственных", "бытийствен"},
		{"бытия", "быт"},
		{"бытность", "бытност"},
		{"быту", "быт"},
		{"быть", "быт"},
		{"бытьи", "быт"},
		{"бытья", "быт"},
		{"бьемся", "бьем"},
		{"бьет", "бьет"},
		{"бьется", "бьет"},
		{"бьешься", "бьеш"},
		{"бьюсь", "бью"},
		{"бьют", "бьют"},
		{"бьются", "бьют"},
		{"бьющаяся", "бьющ"},
		{"бьющимся", "бьющ"},
		{"бюджета", "бюджет"},
		{"бюджету", "бюджет"},
		{"бюргера", "бюргер"},
		{"бюргеров", "бюргер"},
		{"бюргером", "бюргер"},
		{"бюргерские", "бюргерск"},
		{"бюргерскими", "бюргерск"},
		{"бюро", "бюр"},
		{"бюст", "бюст"},
		{"бюста", "бюст"},
		{"в", "в"},
		{"вавиловка", "вавиловк"},
		{"вагнера", "вагнер"},
		{"вагон", "вагон"},
		{"вагона", "вагон"},
		{"вагоне", "вагон"},
		{"вагонов", "вагон"},
		{"вагоном", "вагон"},
		{"вагоны", "вагон"},
		{"важная", "важн"},
		{"важнее", "важн"},
		{"важнейшие", "важн"},
		{"важнейшими", "важн"},
		{"важничал", "важнича"},
		{"важно", "важн"},
		{"важного", "важн"},
		{"важное", "важн"},
		{"важной", "важн"},
		{"важном", "важн"},
		{"важному", "важн"},
		{"важности", "важност"},
		{"важностию", "важност"},
		{"важность", "важност"},
		{"важностью", "важност"},
		{"важную", "важн"},
		{"важны", "важн"},
		{"важные", "важн"},
		{"важный", "важн"},
		{"важным", "важн"},
		{"важных", "важн"},
		{"вазах", "ваз"},
		{"вазы", "ваз"},
		{"вакса", "вакс"},
		{"вакханка", "вакханк"},
		{"вал", "вал"},
		{"валандался", "валанда"},
		{"валентина", "валентин"},
		{"валериановых", "валерианов"},
		{"валерию", "валер"},
		{"валетами", "валет"},
		{"вали", "вал"},
		{"валил", "вал"},
		{"валился", "вал"},
		{"валится", "вал"},
		{"валов", "вал"},
		{"вальдшнепа", "вальдшнеп"},
		{"вальс", "вальс"},
		{"вальса", "вальс"},
		{"вальсе", "вальс"},
		{"вальсишку", "вальсишк"},
		{"вальтера", "вальтер"},
		{"валяется", "валя"},
		{"валялась", "валя"},
		{"валялись", "валя"},
		{"валялось", "валя"},
		{"валялся", "валя"},
		{"валять", "валя"},
		{"валяются", "валя"},
		{"вам", "вам"},
		{"вами", "вам"},
		{"вампир", "вампир"},
		{"вампирственного", "вампирствен"},
		{"ване", "ван"},
		{"ваней", "ван"},
		{"ванечки", "ванечк"},
		{"ванечкину", "ванечкин"},
		{"ванечку", "ванечк"},
		{"вани", "ван"},
		{"ванили", "ван"},
		{"ваниль", "ванил"},
		{"ванилью", "ванил"},
		{"ванну", "ван"},
		{"ванька", "ваньк"},
		{"ваню", "ван"},
		{"ванюша", "ванюш"},
		{"ванюше", "ванюш"},
		{"ванюши", "ванюш"},
		{"ванюшу", "ванюш"},
		{"ваня", "ван"},
		{"варвара", "варвар"},
		{"варваром", "варвар"},
		{"варвару", "варвар"},
		{"варе", "вар"},
		{"варений", "варен"},
		{"вареники", "вареник"},
		{"вареников", "вареник"},
		{"вареного", "варен"},
		{"вареной", "варен"},
		{"вареную", "варен"},
		{"варенц", "варенц"},
		{"варенье", "варен"},
		{"варенью", "варен"},
		{"варенья", "варен"},
		{"вареньями", "варен"},
		{"вари", "вар"},
		{"вариаций", "вариац"},
		{"варили", "вар"},
		{"варились", "вар"},
		{"варин", "варин"},
		{"варит", "вар"},
		{"варить", "вар"},
		{"варшава", "варшав"},
		{"варшаве", "варшав"},
		{"варшавой", "варшав"},
		{"варшавский", "варшавск"},
		{"варшаву", "варшав"},
		{"варьяции", "варьяц"},
		{"варьяциях", "варьяц"},
		{"варю", "вар"},
		{"варя", "вар"},
		{"варяга", "варяг"},
		{"варят", "вар"},
		{"вас", "вас"},
		{"василий", "васил"},
		{"василиса", "василис"},
		{"василисе", "василис"},
		{"васильев", "васил"},
		{"васильева", "васильев"},
		{"васильевич", "васильевич"},
		{"васильевна", "васильевн"},
		{"васильевну", "васильевн"},
		{"васильевны", "васильевн"},
		{"васильевский", "васильевск"},
		{"васильевского", "васильевск"},
		{"васильевском", "васильевск"},
		{"васильевскому", "васильевск"},
		{"васильевым", "васильев"},
		{"васильем", "васил"},
		{"васильич", "васильич"},
		{"васильичем", "васильич"},
		{"васька", "васьк"},
		{"ваське", "васьк"},
		{"васьки", "васьк"},
		{"ватага", "ватаг"},
		{"ватерлоо", "ватерло"},
		{"ватным", "ватн"},
		{"ватой", "ват"},
		{"ваточной", "ваточн"},
		{"ваточном", "ваточн"},
		{"ватрушки", "ватрушк"},
		{"ватрушку", "ватрушк"},
		{"вату", "ват"},
		{"ваты", "ват"},
		{"вафля", "вафл"},
		{"вахрушин", "вахрушин"},
		{"вахрушина", "вахрушин"},
		{"ваш", "ваш"},
		{"ваша", "ваш"},
		{"ваше", "ваш"},
		{"вашего", "ваш"},
		{"вашей", "ваш"},
		{"вашем", "ваш"},
		{"вашему", "ваш"},
		{"вашею", "ваш"},
		{"ваши", "ваш"},
		{"вашим", "ваш"},
		{"вашими", "ваш"},
		{"ваших", "ваш"},
		{"вашу", "ваш"},
		{"вбегает", "вбега"},
		{"вбежала", "вбежа"},
		{"вбежит", "вбеж"},
		{"вбивали", "вбива"},
		{"вбил", "вбил"},
		{"вблизи", "вблиз"},
		{"вбок", "вбок"},
		{"ввела", "ввел"},
		{"ввели", "ввел"},
		{"ввергался", "вверга"},
		{"ввергающий", "вверга"},
		{"вверенный", "вверен"},
		{"вверить", "ввер"},
		{"ввернет", "ввернет"},
		{"вверну", "вверн"},
		{"ввернул", "ввернул"},
		{"ввернуть", "ввернут"},
		{"вверх", "вверх"},
		{"вверху", "вверх"},
		{"вверяет", "вверя"},
		{"ввести", "ввест"},
		{"ввиду", "ввид"},
		{"вводила", "ввод"},
		{"вводить", "ввод"},
		{"ввожу", "ввож"},
		{"ввозят", "ввоз"},
		{"ввысь", "ввы"},
		{"ввязался", "ввяза"},
		{"вглубь", "вглуб"},
		{"вглядевшись", "вглядевш"},
		{"вгляделись", "вглядел"},
		{"вглядись", "вгляд"},
		{"вглядывается", "вглядыва"},
		{"вглядывалась", "вглядыва"},
		{"вглядывался", "вглядыва"},
		{"вглядываться", "вглядыва"},
		{"вглядываясь", "вглядыв"},
		{"вдаваться", "вдава"},
		{"вдавленного", "вдавлен"},
		{"вдалась", "вдал"},
		{"вдалеке", "вдалек"},
		{"вдали", "вдал"},
		{"вдаль", "вдал"},
		{"вдвое", "вдво"},
		{"вдвоем", "вдво"},
		{"вдвойне", "вдвойн"},
		{"вдеваешь", "вдева"},
		{"вдесятеро", "вдесятер"},
		{"вдобавок", "вдобавок"},
		{"вдова", "вдов"},
		{"вдове", "вдов"},
		{"вдовеет", "вдовеет"},
		{"вдовец", "вдовец"},
		{"вдовий", "вдов"},
		{"вдовиц", "вдовиц"},
		{"вдовица", "вдовиц"},
		{"вдовой", "вдов"},
		{"вдову", "вдов"},
		{"вдовы", "вдов"},
		{"вдогонку", "вдогонк"},
		{"вдоль", "вдол"},
		{"вдохновение", "вдохновен"},
		{"вдохновением", "вдохновен"},
		{"вдохновении", "вдохновен"},
		{"вдохновению", "вдохновен"},
		{"вдохновенно", "вдохновен"},
		{"вдохновенное", "вдохновен"},
		{"вдохновенный", "вдохновен"},
		{"вдохновенными", "вдохновен"},
		{"вдохновенных", "вдохновен"},
		{"вдохновенье", "вдохновен"},
		{"вдохновеньем", "вдохновен"},
		{"вдохнул", "вдохнул"},
		{"вдохнуть", "вдохнут"},
		{"вдребезги", "вдребезг"},
		{"вдруг", "вдруг"},
		{"вдругорядь", "вдругоряд"},
		{"вдумайтесь", "вдума"},
		{"вдуматься", "вдума"},
		{"вдумчиво", "вдумчив"},
		{"вдумчивое", "вдумчив"},
		{"вдумывалась", "вдумыва"},
		{"вдумывался", "вдумыва"},
		{"вдумываясь", "вдумыв"},
		{"ве", "ве"},
		{"ведаем", "веда"},
		{"ведает", "веда"},
		{"ведаете", "веда"},
		{"ведаешь", "веда"},
		{"ведай", "веда"},
		{"ведал", "веда"},
		{"ведать", "веда"},
		{"ведаю", "веда"},
		{"ведая", "вед"},
		{"ведению", "веден"},
		{"ведения", "веден"},
		{"веденной", "веден"},
		{"ведет", "ведет"},
		{"ведешь", "ведеш"},
		{"веди", "вед"},
		{"ведите", "вед"},
		{"ведома", "ведом"},
		{"ведомо", "ведом"},
		{"ведомости", "ведом"},
		{"ведомостях", "ведом"},
		{"ведомству", "ведомств"},
		{"ведрами", "ведр"},
		{"ведре", "ведр"},
		{"ведро", "ведр"},
		{"ведром", "ведр"},
		{"веду", "вед"},
		{"ведут", "ведут"},
		{"ведущая", "ведущ"},
		{"ведущей", "ведущ"},
		{"ведущие", "ведущ"},
		{"ведущим", "ведущ"},
		{"ведь", "вед"},
		{"ведьма", "ведьм"},
		{"ведьмой", "ведьм"},
		{"ведьмы", "ведьм"},
		{"ведя", "вед"},
		{"веером", "веер"},
		{"веет", "веет"},
		{"вежды", "вежд"},
		{"вежлив", "вежл"},
		{"вежливо", "вежлив"},
		{"вежливой", "вежлив"},
		{"вежливости", "вежлив"},
		{"вежливость", "вежлив"},
		{"вежливый", "вежлив"},
		{"вежливым", "вежлив"},
		{"везде", "везд"},
		{"везет", "везет"},
		{"везешь", "везеш"},
		{"везла", "везл"},
		{"везти", "везт"},
		{"везу", "вез"},
		{"везувии", "везув"},
		{"везущих", "везущ"},
		{"век", "век"},
		{"века", "век"},
		{"векам", "век"},
		{"веками", "век"},
		{"веках", "век"},
		{"веке", "век"},
		{"веки", "век"},
		{"веков", "век"},
		{"вековечная", "вековечн"},
		{"вековечные", "вековечн"},
		{"вековой", "веков"},
		{"вековою", "веков"},
		{"вековые", "веков"},
		{"векселей", "вексел"},
		{"векселек", "векселек"},
		{"вексель", "вексел"},
		{"векселя", "вексел"},
		{"веку", "век"},
		{"вел", "вел"},
		{"вела", "вел"},
		{"велел", "велел"},
		{"велела", "велел"},
		{"велели", "велел"},
		{"велению", "велен"},
		{"велено", "вел"},
		{"велеть", "велет"},
		{"вели", "вел"},
		{"велик", "велик"},
		{"велика", "велик"},
		{"великана", "велика"},
		{"великанами", "великан"},
		{"великанов", "великан"},
		{"великаны", "велика"},
		{"великая", "велик"},
		{"велики", "велик"},
		{"великие", "велик"},
		{"великий", "велик"},
		{"великим", "велик"},
		{"великих", "велик"},
		{"великого", "велик"},
		{"великодушен", "великодуш"},
		{"великодушия", "великодуш"},
		{"великодушная", "великодушн"},
		{"великодушнее", "великодушн"},
		{"великодушнейший", "великодушн"},
		{"великодушно", "великодушн"},
		{"великодушного", "великодушн"},
		{"великодушны", "великодушн"},
		{"великодушные", "великодушн"},
		{"великодушный", "великодушн"},
		{"великодушных", "великодушн"},
		{"великое", "велик"},
		{"великой", "велик"},
		{"великолепий", "великолеп"},
		{"великолепна", "великолепн"},
		{"великолепная", "великолепн"},
		{"великолепнее", "великолепн"},
		{"великолепнейшие", "великолепн"},
		{"великолепние", "великолепн"},
		{"великолепно", "великолепн"},
		{"великолепного", "великолепн"},
		{"великолепное", "великолепн"},
		{"великолепной", "великолепн"},
		{"великолепную", "великолепн"},
		{"великолепные", "великолепн"},
		{"великолепный", "великолепн"},
		{"великолепных", "великолепн"},
		{"великом", "велик"},
		{"великому", "велик"},
		{"великонек", "великонек"},
		{"великую", "велик"},
		{"велит", "вел"},
		{"велите", "вел"},
		{"величава", "величав"},
		{"величавого", "величав"},
		{"величавое", "величав"},
		{"величавой", "величав"},
		{"величайшая", "величайш"},
		{"величайшего", "величайш"},
		{"величайшее", "величайш"},
		{"величайшей", "величайш"},
		{"величайшем", "величайш"},
		{"величайшему", "величайш"},
		{"величайший", "величайш"},
		{"величайшим", "величайш"},
		{"величал", "велича"},
		{"величают", "велича"},
		{"величественная", "величествен"},
		{"величественно", "величествен"},
		{"величество", "величеств"},
		{"величие", "велич"},
		{"величий", "велич"},
		{"величина", "величин"},
		{"величине", "величин"},
		{"величиной", "величин"},
		{"величины", "величин"},
		{"величье", "велич"},
		{"велишь", "вел"},
		{"велосипеды", "велосипед"},
		{"велось", "вел"},
		{"вельможа", "вельмож"},
		{"вельможе", "вельмож"},
		{"вельможей", "вельмож"},
		{"велю", "вел"},
		{"велят", "вел"},
		{"венгерский", "венгерск"},
		{"вене", "вен"},
		{"венерических", "венерическ"},
		{"венец", "венец"},
		{"венецейской", "венецейск"},
		{"венецианская", "венецианск"},
		{"венецианской", "венецианск"},
		{"венеция", "венец"},
		{"веник", "веник"},
		{"веника", "веник"},
		{"веничек", "веничек"},
		{"веничком", "веничк"},
		{"венка", "венк"},
		{"венками", "венк"},
		{"венке", "венк"},
		{"венки", "венк"},
		{"венком", "венк"},
		{"венок", "венок"},
		{"венца", "венц"},
		{"венце", "венц"},
		{"венцу", "венц"},
		{"венчаемся", "венча"},
		{"венчал", "венча"},
		{"венчались", "венча"},
		{"венчался", "венча"},
		{"венчальный", "венчальн"},
		{"венчаться", "венча"},
		{"венчают", "венча"},
		{"вены", "вен"},
		{"вер", "вер"},
		{"вера", "вер"},
		{"верблюды", "верблюд"},
		{"вербы", "верб"},
		{"вере", "вер"},
		{"веревках", "веревк"},
		{"веревке", "веревк"},
		{"веревки", "веревк"},
		{"веревкой", "веревк"},
		{"веревкою", "веревк"},
		{"веревочку", "веревочк"},
		{"веревочное", "веревочн"},
		{"верен", "вер"},
		{"вереница", "верениц"},
		{"вереницей", "верениц"},
		{"вереницу", "верениц"},
		{"веретена", "верет"},
		{"верил", "вер"},
		{"верила", "вер"},
		{"верили", "вер"},
		{"верит", "вер"},
		{"верите", "вер"},
		{"верится", "вер"},
		{"верить", "вер"},
		{"веришь", "вер"},
		{"верна", "верн"},
		{"верная", "верн"},
		{"вернее", "верн"},
		{"верней", "верн"},
		{"вернем", "верн"},
		{"вернет", "вернет"},
		{"вернется", "вернет"},
		{"вернешь", "вернеш"},
		{"вернешься", "вернеш"},
		{"верни", "верн"},
		{"вернись", "верн"},
		{"вернитесь", "верн"},
		{"верно", "верн"},
		{"верного", "верн"},
		{"верное", "верн"},
		{"верной", "верн"},
		{"верному", "верн"},
		{"верности", "верност"},
		{"верность", "верност"},
		{"верностью", "верност"},
		{"верною", "верн"},
		{"вернув", "вернув"},
		{"вернувшись", "вернувш"},
		{"вернулась", "вернул"},
		{"вернулось", "вернул"},
		{"вернулся", "вернул"},
		{"вернусь", "верн"},
		{"вернутся", "вернут"},
		{"вернуть", "вернут"},
		{"вернуться", "вернут"},
		{"верную", "верн"},
		{"верны", "верн"},
		{"верные", "верн"},
		{"верный", "верн"},
		{"верным", "верн"},
		{"верных", "верн"},
		{"веровавший", "верова"},
		{"веровал", "верова"},
		{"веровала", "верова"},
		{"верований", "верован"},
		{"веровать", "верова"},
		{"верой", "вер"},
		{"вероломна", "вероломн"},
		{"вероломство", "вероломств"},
		{"вероломством", "вероломств"},
		{"верою", "вер"},
		{"вероятнее", "вероятн"},
		{"вероятно", "вероятн"},
		{"вероятности", "вероятн"},
		{"вероятность", "вероятн"},
		{"вероятны", "вероятн"},
		{"версия", "верс"},
		{"верст", "верст"},
		{"верстах", "верст"},
		{"версте", "верст"},
		{"верстовых", "верстов"},
		{"версту", "верст"},
		{"версты", "верст"},
		{"вертел", "вертел"},
		{"вертелся", "вертел"},
		{"вертеп", "вертеп"},
		{"вертеры", "вертер"},
		{"вертеть", "вертет"},
		{"верти", "верт"},
		{"вертись", "верт"},
		{"вертит", "верт"},
		{"вертится", "верт"},
		{"вертишься", "верт"},
		{"вертлявая", "вертляв"},
		{"вертун", "вертун"},
		{"вертя", "верт"},
		{"вертясь", "верт"},
		{"вертят", "верт"},
		{"веру", "вер"},
		{"верует", "вер"},
		{"веруете", "верует"},
		{"веруется", "вер"},
		{"веруешь", "веруеш"},
		{"веруй", "вер"},
		{"верую", "вер"},
		{"верующей", "вер"},
		{"верующие", "вер"},
		{"верующий", "вер"},
		{"верующим", "вер"},
		{"веруя", "веру"},
		{"верфям", "верф"},
		{"верх", "верх"},
		{"верхлева", "верхлев"},
		{"верхлеве", "верхлев"},
		{"верхлево", "верхлев"},
		{"верхлевский", "верхлевск"},
		{"верхлевского", "верхлевск"},
		{"верхнего", "верхн"},
		{"верхнее", "верхн"},
		{"верхней", "верхн"},
		{"верхнем", "верхн"},
		{"верхнею", "верхн"},
		{"верхние", "верхн"},
		{"верхний", "верхн"},
		{"верхнюю", "верхн"},
		{"верхняя", "верхн"},
		{"верховий", "верхов"},
		{"верховой", "верхов"},
		{"верховскому", "верховск"},
		{"верхом", "верх"},
		{"верху", "верх"},
		{"верхушки", "верхушк"},
		{"верхушку", "верхушк"},
		{"вершать", "верша"},
		{"вершина", "вершин"},
		{"вершинах", "вершин"},
		{"вершинин", "вершинин"},
		{"вершинина", "вершинин"},
		{"вершинину", "вершинин"},
		{"вершинины", "вершинин"},
		{"вершининым", "вершинин"},
		{"вершининых", "вершинин"},
		{"вершины", "вершин"},
		{"вершков", "вершк"},
		{"вершок", "вершок"},
		{"веры", "вер"},
		{"верь", "вер"},
		{"верьте", "верьт"},
		{"верю", "вер"},
		{"веря", "вер"},
		{"верят", "вер"},
		{"вес", "вес"},
		{"весах", "вес"},
		{"весел", "весел"},
		{"весела", "весел"},
		{"веселая", "весел"},
		{"веселее", "весел"},
		{"веселей", "весел"},
		{"веселея", "веселе"},
		{"веселила", "весел"},
		{"веселимся", "весел"},
		{"веселиться", "весел"},
		{"весело", "весел"},
		{"веселого", "весел"},
		{"веселое", "весел"},
		{"веселой", "весел"},
		{"веселости", "весел"},
		{"веселость", "весел"},
		{"веселы", "весел"},
		{"веселые", "весел"},
		{"веселый", "весел"},
		{"веселым", "весел"},
		{"веселых", "весел"},
		{"веселье", "весел"},
		{"весельем", "весел"},
		{"весельчаком", "весельчак"},
		{"веселья", "весел"},
		{"весеннею", "весен"},
		{"весенние", "весен"},
		{"весенний", "весен"},
		{"весенняя", "весен"},
		{"весит", "вес"},
		{"весла", "весл"},
		{"весле", "весл"},
		{"весло", "весл"},
		{"весна", "весн"},
		{"весне", "весн"},
		{"весной", "весн"},
		{"весну", "весн"},
		{"весны", "весн"},
		{"весталку", "весталк"},
		{"вести", "вест"},
		{"вестник", "вестник"},
		{"весты", "вест"},
		{"весть", "вест"},
		{"вестью", "вест"},
		{"вестями", "вест"},
		{"весу", "вес"},
		{"весы", "вес"},
		{"весь", "ве"},
		{"весьма", "весьм"},
		{"ветвей", "ветв"},
		{"ветви", "ветв"},
		{"ветвью", "ветв"},
		{"ветер", "ветер"},
		{"ветерка", "ветерк"},
		{"ветерком", "ветерк"},
		{"ветерок", "ветерок"},
		{"ветка", "ветк"},
		{"ветках", "ветк"},
		{"ветке", "ветк"},
		{"ветки", "ветк"},
		{"веткой", "ветк"},
		{"ветку", "ветк"},
		{"ветлы", "ветл"},
		{"веток", "веток"},
		{"ветр", "ветр"},
		{"ветра", "ветр"},
		{"ветреная", "ветрен"},
		{"ветреность", "ветрен"},
		{"ветреный", "ветрен"},
		{"ветровые", "ветров"},
		{"ветром", "ветр"},
		{"ветру", "ветр"},
		{"ветха", "ветх"},
		{"ветхая", "ветх"},
		{"ветхий", "ветх"},
		{"ветхим", "ветх"},
		{"ветхих", "ветх"},
		{"ветхом", "ветх"},
		{"ветчиной", "ветчин"},
		{"ветчины", "ветчин"},
		{"вечер", "вечер"},
		{"вечера", "вечер"},
		{"вечерам", "вечер"},
		{"вечерах", "вечер"},
		{"вечере", "вечер"},
		{"вечереет", "вечереет"},
		{"вечерело", "вечерел"},
		{"вечерком", "вечерк"},
		{"вечерне", "вечерн"},
		{"вечернего", "вечерн"},
		{"вечернее", "вечерн"},
		{"вечерней", "вечерн"},
		{"вечернем", "вечерн"},
		{"вечерние", "вечерн"},
		{"вечерний", "вечерн"},
		{"вечерних", "вечерн"},
		{"вечеров", "вечер"},
		{"вечерок", "вечерок"},
		{"вечером", "вечер"},
		{"вечеру", "вечер"},
		{"вечная", "вечн"},
		{"вечно", "вечн"},
		{"вечное", "вечн"},
		{"вечной", "вечн"},
		{"вечном", "вечн"},
		{"вечному", "вечн"},
		{"вечности", "вечност"},
		{"вечность", "вечност"},
		{"вечную", "вечн"},
		{"вечные", "вечн"},
		{"вечный", "вечн"},
		{"вечным", "вечн"},
		{"вечными", "вечн"},
		{"вечных", "вечн"},
		{"вешает", "веша"},
		{"вешал", "веша"},
		{"вешались", "веша"},
		{"вешаться", "веша"},
		{"вешаются", "веша"},
		{"веше", "веш"},
		{"веши", "веш"},
		{"вешней", "вешн"},
		{"вешнею", "вешн"},
		{"вешний", "вешн"},
		{"вешняя", "вешн"},
		{"вещам", "вещ"},
		{"вещами", "вещ"},
		{"вещания", "вещан"},
		{"вещах", "вещ"},
		{"вещей", "вещ"},
		{"веществ", "веществ"},
		{"вещественные", "веществен"},
		{"вещи", "вещ"},
		{"вещим", "вещ"},
		{"вещих", "вещ"},
		{"вещица", "вещиц"},
		{"вещицы", "вещиц"},
		{"вещь", "вещ"},
		{"вещью", "вещ"},
		{"веял", "вея"},
		{"веяло", "вея"},
		{"веянье", "веян"},
		{"веяньем", "веян"},
		{"взад", "взад"},
		{"взаимно", "взаимн"},
		{"взаимного", "взаимн"},
		{"взаимное", "взаимн"},
		{"взаимной", "взаимн"},
		{"взаимном", "взаимн"},
		{"взаимности", "взаимност"},
		{"взаимность", "взаимност"},
		{"взаимностью", "взаимност"},
		{"взаимною", "взаимн"},
		{"взаимные", "взаимн"},
		{"взаимными", "взаимн"},
		{"взаимных", "взаимн"},
		{"взаймы", "взайм"},
		{"взамен", "взам"},
		{"взаперти", "взаперт"},
		{"взаправду", "взаправд"},
		{"взапуски", "взапуск"},
		{"взбегал", "взбега"},
		{"взбежав", "взбежа"},
		{"взбежать", "взбежа"},
		{"взбесившийся", "взбес"},
		{"взбесился", "взбес"},
		{"взбесит", "взбес"},
		{"взбесятся", "взбес"},
		{"взбешенный", "взбешен"},
		{"взбирались", "взбира"},
		{"взбираться", "взбира"},
		{"взбираясь", "взбир"},
		{"взбитые", "взбит"},
		{"взбитыми", "взбит"},
		{"взбороздив", "взборозд"},
		{"взбунтоваться", "взбунтова"},
		{"взбунтуется", "взбунт"},
		{"взведет", "взведет"},
		{"взведете", "взведет"},
		{"взвел", "взвел"},
		{"взвела", "взвел"},
		{"взвесил", "взвес"},
		{"взвесила", "взвес"},
		{"взвешивает", "взвешива"},
		{"взвешивал", "взвешива"},
		{"взвешивая", "взвешив"},
		{"взвивается", "взвива"},
		{"взвизги", "взвизг"},
		{"взвизгнул", "взвизгнул"},
		{"взвизгнула", "взвизгнул"},
		{"взвизгов", "взвизг"},
		{"взвизгом", "взвизг"},
		{"взвилась", "взвил"},
		{"взвился", "взвил"},
		{"взвод", "взвод"},
		{"взводимых", "взводим"},
		{"взводом", "взвод"},
		{"взволновал", "взволнова"},
		{"взволновалась", "взволнова"},
		{"взволновало", "взволнова"},
		{"взволновался", "взволнова"},
		{"взволнован", "взволнова"},
		{"взволнована", "взволнова"},
		{"взволнованная", "взволнова"},
		{"взволнованно", "взволнова"},
		{"взволнованное", "взволнова"},
		{"взволнованный", "взволнова"},
		{"взволнованных", "взволнова"},
		{"взволнованы", "взволнова"},
		{"взволновать", "взволнова"},
		{"взволнует", "взволн"},
		{"взвыл", "взвыл"},
		{"взгляд", "взгляд"},
		{"взгляда", "взгляд"},
		{"взглядами", "взгляд"},
		{"взгляде", "взгляд"},
		{"взглядов", "взгляд"},
		{"взглядом", "взгляд"},
		{"взгляду", "взгляд"},
		{"взгляды", "взгляд"},
		{"взглядывает", "взглядыва"},
		{"взглядывал", "взглядыва"},
		{"взглядывала", "взглядыва"},
		{"взглядывали", "взглядыва"},
		{"взглядываниями", "взглядыван"},
		{"взглядывают", "взглядыва"},
		{"взглядывая", "взглядыв"},
		{"взглянет", "взглянет"},
		{"взглянешь", "взглянеш"},
		{"взгляни", "взглян"},
		{"взгляните", "взглян"},
		{"взгляну", "взглян"},
		{"взглянув", "взглянув"},
		{"взглянул", "взглянул"},
		{"взглянула", "взглянул"},
		{"взглянули", "взглянул"},
		{"взглянуть", "взглянут"},
		{"вздернет", "вздернет"},
		{"вздернута", "вздернут"},
		{"вздор", "вздор"},
		{"вздоре", "вздор"},
		{"вздорная", "вздорн"},
		{"вздорные", "вздорн"},
		{"вздорными", "вздорн"},
		{"вздорожают", "вздорожа"},
		{"вздором", "вздор"},
		{"вздору", "вздор"},
		{"вздох", "вздох"},
		{"вздохами", "вздох"},
		{"вздохи", "вздох"},
		{"вздохнет", "вздохнет"},
		{"вздохни", "вздохн"},
		{"вздохну", "вздохн"},
		{"вздохнув", "вздохнув"},
		{"вздохнул", "вздохнул"},
		{"вздохнула", "вздохнул"},
		{"вздохнули", "вздохнул"},
		{"вздохнуть", "вздохнут"},
		{"вздохом", "вздох"},
		{"вздрагивавшие", "вздрагива"},
		{"вздрагивает", "вздрагива"},
		{"вздрагивал", "вздрагива"},
		{"вздрагивала", "вздрагива"},
		{"вздрагивали", "вздрагива"},
		{"вздрагиваньям", "вздрагиван"},
		{"вздрагивают", "вздрагива"},
		{"вздрагивающей", "вздрагива"},
		{"вздрагивающими", "вздрагива"},
		{"вздрагивая", "вздрагив"},
		{"вздремнет", "вздремнет"},
		{"вздремнул", "вздремнул"},
		{"вздремнула", "вздремнул"},
		{"вздрогнет", "вздрогнет"},
		{"вздрогнув", "вздрогнув"},
		{"вздрогнул", "вздрогнул"},
		{"вздрогнула", "вздрогнул"},
		{"вздрогнули", "вздрогнул"},
		{"вздрогнут", "вздрогнут"},
		{"вздули", "вздул"},
		{"вздулись", "вздул"},
		{"вздумавший", "вздума"},
		{"вздумает", "вздума"},
		{"вздумается", "вздума"},
		{"вздумал", "вздума"},
		{"вздумала", "вздума"},
		{"вздумали", "вздума"},
		{"вздумалось", "вздума"},
		{"вздымается", "вздыма"},
		{"вздымаются", "вздыма"},
		{"вздыхает", "вздыха"},
		{"вздыхал", "вздыха"},
		{"вздыхала", "вздыха"},
		{"вздыхатель", "вздыхател"},
		{"вздыхать", "вздыха"},
		{"вздыхают", "вздыха"},
		{"вздыхающем", "вздыха"},
		{"вздыхая", "вздых"},
		{"взирая", "взир"},
		{"взламывать", "взламыва"},
		{"взлет", "взлет"},
		{"взлетающего", "взлета"},
		{"взлетел", "взлетел"},
		{"взлетит", "взлет"},
		{"взлечу", "взлеч"},
		{"взломом", "взлом"},
		{"взлохмачены", "взлохмач"},
		{"взмахнет", "взмахнет"},
		{"взмахни", "взмахн"},
		{"взмахнув", "взмахнув"},
		{"взмахнул", "взмахнул"},
		{"взмахом", "взмах"},
		{"взметутся", "взметут"},
		{"взнос", "взнос"},
		{"взобраться", "взобра"},
		{"взобьет", "взобьет"},
		{"взовьется", "взовьет"},
		{"взойдет", "взойдет"},
		{"взойду", "взойд"},
		{"взойдя", "взойд"},
		{"взойти", "взойт"},
		{"взор", "взор"},
		{"взора", "взор"},
		{"взоре", "взор"},
		{"взоров", "взор"},
		{"взором", "взор"},
		{"взору", "взор"},
		{"взоры", "взор"},
		{"взошел", "взошел"},
		{"взошла", "взошл"},
		{"взошло", "взошл"},
		{"взрослые", "взросл"},
		{"взрослый", "взросл"},
		{"взрослыми", "взросл"},
		{"взрослых", "взросл"},
		{"взрыв", "взрыв"},
		{"взрыва", "взрыв"},
		{"взрывов", "взрыв"},
		{"взрывом", "взрыв"},
		{"взрывы", "взрыв"},
		{"взъерошенный", "взъерошен"},
		{"взывает", "взыва"},
		{"взывал", "взыва"},
		{"взывала", "взыва"},
		{"взывают", "взыва"},
		{"взывая", "взыв"},
		{"взыграет", "взыгра"},
		{"взыскание", "взыскан"},
		{"взыскании", "взыскан"},
		{"взысканий", "взыскан"},
		{"взысканию", "взыскан"},
		{"взыскания", "взыскан"},
		{"взысканиях", "взыскан"},
		{"взыскательна", "взыскательн"},
		{"взыскательнее", "взыскательн"},
		{"взыскательному", "взыскательн"},
		{"взыскать", "взыска"},
		{"взыскивать", "взыскива"},
		{"взыскиваю", "взыскива"},
		{"взыщите", "взыщ"},
		{"взяв", "взяв"},
		{"взявши", "взявш"},
		{"взял", "взял"},
		{"взяла", "взял"},
		{"взялась", "взял"},
		{"взяли", "взял"},
		{"взялись", "взял"},
		{"взяло", "взял"},
		{"взялось", "взял"},
		{"взялся", "взял"},
		{"взятки", "взятк"},
		{"взяток", "взяток"},
		{"взяточник", "взяточник"},
		{"взяточника", "взяточник"},
		{"взяточников", "взяточник"},
		{"взяты", "взят"},
		{"взятые", "взят"},
		{"взятых", "взят"},
		{"взять", "взят"},
		{"взяться", "взят"},
		{"вив", "вив"},
		{"вид", "вид"},
		{"вида", "вид"},
		{"видавший", "вида"},
		{"видал", "вида"},
		{"видала", "вида"},
		{"видалась", "вида"},
		{"видали", "вида"},
		{"видались", "вида"},
		{"видано", "вида"},
		{"видать", "вида"},
		{"видаться", "вида"},
		{"видах", "вид"},
		{"виде", "вид"},
		{"видевший", "видевш"},
		{"видевших", "видевш"},
		{"видел", "видел"},
		{"видела", "видел"},
		{"видели", "видел"},
		{"виделись", "видел"},
		{"видело", "видел"},
		{"виделся", "видел"},
		{"видемши", "видемш"},
		{"виден", "вид"},
		{"видена", "вид"},
		{"видение", "виден"},
		{"видением", "виден"},
		{"видений", "виден"},
		{"видения", "виден"},
		{"видениям", "виден"},
		{"видениями", "виден"},
		{"виденною", "виден"},
		{"виденный", "виден"},
		{"виденных", "виден"},
		{"виденье", "виден"},
		{"виденья", "виден"},
		{"видеть", "видет"},
		{"видеться", "видет"},
		{"видим", "вид"},
		{"видимая", "видим"},
		{"видимо", "видим"},
		{"видимое", "видим"},
		{"видимой", "видим"},
		{"видимом", "видим"},
		{"видимому", "видим"},
		{"видимою", "видим"},
		{"видимся", "вид"},
		{"видимый", "видим"},
		{"видимым", "видим"},
		{"видимых", "видим"},
		{"видит", "вид"},
		{"видите", "вид"},
		{"видится", "вид"},
		{"видишь", "вид"},
		{"видишься", "вид"},
		{"видна", "видн"},
		{"видная", "видн"},
		{"виднелась", "виднел"},
		{"виднелись", "виднел"},
		{"виднелось", "виднел"},
		{"видно", "видн"},
		{"видного", "видн"},
		{"видною", "видн"},
		{"видны", "видн"},
		{"видов", "вид"},
		{"видоизменения", "видоизменен"},
		{"видоизменяющаяся", "видоизменя"},
		{"видом", "вид"},
		{"виду", "вид"},
		{"виды", "вид"},
		{"видывал", "видыва"},
		{"видывали", "видыва"},
		{"видя", "вид"},
		{"видят", "вид"},
		{"видятся", "вид"},
		{"видящий", "видя"},
		{"вижу", "виж"},
		{"вижусь", "виж"},
		{"визави", "визав"},
		{"визг", "визг"},
		{"визгливо", "визглив"},
		{"визгливый", "визглив"},
		{"визгом", "визг"},
		{"визжала", "визжа"},
		{"визжаль", "визжал"},
		{"визжат", "визжат"},
		{"визжать", "визжа"},
		{"визжит", "визж"},
		{"визит", "виз"},
		{"визита", "визит"},
		{"визитами", "визит"},
		{"визите", "виз"},
		{"визитик", "визитик"},
		{"визитная", "визитн"},
		{"визитную", "визитн"},
		{"визитный", "визитн"},
		{"визитных", "визитн"},
		{"визитов", "визит"},
		{"визитом", "визит"},
		{"визиты", "визит"},
		{"вилами", "вил"},
		{"виланда", "виланд"},
		{"вилась", "вил"},
		{"вилка", "вилк"},
		{"вилки", "вилк"},
		{"вилкой", "вилк"},
		{"вилку", "вилк"},
		{"виллу", "вилл"},
		{"вилой", "вил"},
		{"вильне", "вильн"},
		{"виляй", "виля"},
		{"виляя", "вил"},
		{"вин", "вин"},
		{"вина", "вин"},
		{"вине", "вин"},
		{"вини", "вин"},
		{"винили", "вин"},
		{"вините", "вин"},
		{"винительный", "винительн"},
		{"винить", "вин"},
		{"винишь", "вин"},
		{"винные", "вин"},
		{"винным", "вин"},
		{"вино", "вин"},
		{"виноват", "виноват"},
		{"виновата", "виноват"},
		{"виноватая", "виноват"},
		{"виноватее", "виноват"},
		{"виновато", "виноват"},
		{"виноватого", "виноват"},
		{"виноватою", "виноват"},
		{"виноваты", "виноват"},
		{"виноватым", "виноват"},
		{"виновен", "винов"},
		{"виновника", "виновник"},
		{"виновником", "виновник"},
		{"виновница", "виновниц"},
		{"виновным", "виновн"},
		{"виновных", "виновн"},
		{"виноград", "виноград"},
		{"винограда", "виноград"},
		{"виноградные", "виноградн"},
		{"виноградом", "виноград"},
		{"вином", "вин"},
		{"винт", "винт"},
		{"винта", "винт"},
		{"винтами", "винт"},
		{"винтовой", "винтов"},
		{"винтообразной", "винтообразн"},
		{"винтообразный", "винтообразн"},
		{"винты", "винт"},
		{"вину", "вин"},
		{"винца", "винц"},
		{"винчи", "винч"},
		{"вины", "вин"},
		{"виню", "вин"},
		{"винюсь", "вин"},
		{"виртуоз", "виртуоз"},
		{"виртуозом", "виртуоз"},
		{"висевшие", "висевш"},
		{"висел", "висел"},
		{"висели", "висел"},
		{"висело", "висел"},
		{"висеть", "висет"},
		{"виси", "вис"},
		{"висит", "вис"},
		{"вискам", "виск"},
		{"висках", "виск"},
		{"виски", "виск"},
		{"виску", "виск"},
		{"висла", "висл"},
		{"вислой", "висл"},
		{"вислу", "висл"},
		{"виснет", "виснет"},
		{"висок", "висок"},
		{"висячую", "висяч"},
		{"висящие", "вися"},
		{"висящий", "вися"},
		{"висящих", "вися"},
		{"витиеватой", "витиеват"},
		{"витиеватым", "витиеват"},
		{"витое", "вит"},
		{"витринах", "витрин"},
		{"вифании", "вифан"},
		{"вифлеема", "вифлеем"},
		{"вихор", "вихор"},
		{"вихрастый", "вихраст"},
		{"вихре", "вихр"},
		{"вихревое", "вихрев"},
		{"вихрем", "вихр"},
		{"вихри", "вихр"},
		{"вихров", "вихр"},
		{"вихры", "вихр"},
		{"вихрь", "вихр"},
		{"вице", "виц"},
		{"вицмундир", "вицмундир"},
		{"вицмундира", "вицмундир"},
		{"вицмундире", "вицмундир"},
		{"вишен", "виш"},
		{"вишеньем", "вишен"},
		{"вишере", "вишер"},
		{"вишневого", "вишнев"},
		{"вишневому", "вишнев"},
		{"вишневые", "вишнев"},
		{"вишневый", "вишнев"},
		{"вишни", "вишн"},
		{"вишню", "вишн"},
		{"вишня", "вишн"},
		{"вишь", "виш"},
		{"виясь", "ви"},
		{"вкладывались", "вкладыва"},
		{"включая", "включ"},
		{"вколачивали", "вколачива"},
		{"вколотить", "вколот"},
		{"вконец", "вконец"},
		{"вкопанная", "вкопа"},
		{"вкопанный", "вкопа"},
		{"вкось", "вко"},
		{"вкрадчиво", "вкрадчив"},
		{"вкрадывается", "вкрадыва"},
		{"вкралась", "вкрал"},
		{"вкралось", "вкрал"},
		{"вкривь", "вкрив"},
		{"вкруг", "вкруг"},
		{"вкус", "вкус"},
		{"вкуса", "вкус"},
		{"вкусе", "вкус"},
		{"вкусил", "вкус"},
		{"вкусит", "вкус"},
		{"вкусная", "вкусн"},
		{"вкусно", "вкусн"},
		{"вкусное", "вкусн"},
		{"вкусный", "вкусн"},
		{"вкусом", "вкус"},
		{"вкусу", "вкус"},
		{"вкушать", "вкуша"},
		{"вл", "вл"},
		{"влагает", "влага"},
		{"влагала", "влага"},
		{"влаги", "влаг"},
		{"влагой", "влаг"},
		{"владевшие", "владевш"},
		{"владеет", "владеет"},
		{"владел", "владел"},
		{"владела", "владел"},
		{"владелец", "владелец"},
		{"владельцу", "владельц"},
		{"владетельного", "владетельн"},
		{"владеть", "владет"},
		{"владею", "влад"},
		{"владея", "владе"},
		{"владимир", "владимир"},
		{"владимира", "владимир"},
		{"владимирка", "владимирк"},
		{"владимирке", "владимирк"},
		{"владимирович", "владимирович"},
		{"владимиру", "владимир"},
		{"владычества", "владычеств"},
		{"владычествовал", "владычествова"},
		{"владычествовать", "владычествова"},
		{"владычица", "владычиц"},
		{"владычицу", "владычиц"},
		{"влажна", "влажн"},
		{"влажны", "влажн"},
		{"влажные", "влажн"},
		{"влажный", "влажн"},
		{"властвовала", "властвова"},
		{"властвовать", "властвова"},
		{"властвует", "властв"},
		{"властей", "власт"},
		{"властелин", "властелин"},
		{"властен", "власт"},
		{"власти", "власт"},
		{"властительно", "властительн"},
		{"властному", "властн"},
		{"властный", "властн"},
		{"власть", "власт"},
		{"влачат", "влачат"},
		{"влево", "влев"},
		{"влез", "влез"},
		{"влезал", "влеза"},
		{"влезая", "влез"},
		{"влезет", "влезет"},
		{"влезешь", "влезеш"},
		{"влезли", "влезл"},
		{"влезть", "влезт"},
		{"влезут", "влезут"},
		{"влекли", "влекл"},
		{"влекло", "влекл"},
		{"влекома", "влеком"},
		{"влекущая", "влекущ"},
		{"влекущий", "влекущ"},
		{"влетел", "влетел"},
		{"влетела", "влетел"},
		{"влетели", "влетел"},
		{"влетит", "влет"},
		{"влечение", "влечен"},
		{"влечению", "влечен"},
		{"влечения", "влечен"},
		{"влеченье", "влечен"},
		{"влечет", "влечет"},
		{"влечешь", "влечеш"},
		{"вливается", "влива"},
		{"вливалась", "влива"},
		{"вливают", "влива"},
		{"влил", "влил"},
		{"влиявшие", "влия"},
		{"влияло", "влия"},
		{"влияние", "влиян"},
		{"влиянием", "влиян"},
		{"влияний", "влиян"},
		{"влиянию", "влиян"},
		{"влияния", "влиян"},
		{"вложением", "вложен"},
		{"вложил", "влож"},
		{"вложили", "влож"},
		{"вложить", "влож"},
		{"влопался", "влопа"},
		{"влюбилась", "влюб"},
		{"влюбился", "влюб"},
		{"влюбитесь", "влюб"},
		{"влюбиться", "влюб"},
		{"влюблен", "влюбл"},
		{"влюблена", "влюбл"},
		{"влюбленно", "влюблен"},
		{"влюбленного", "влюблен"},
		{"влюбленное", "влюблен"},
		{"влюбленной", "влюблен"},
		{"влюбленности", "влюблен"},
		{"влюбленность", "влюблен"},
		{"влюбленные", "влюблен"},
		{"влюбленный", "влюблен"},
		{"влюбленным", "влюблен"},
		{"влюбленными", "влюблен"},
		{"влюблены", "влюбл"},
		{"влюблюсь", "влюбл"},
		{"влюбляется", "влюбля"},
		{"влюблялись", "влюбля"},
		{"влюбляться", "влюбля"},
		{"влюбляются", "влюбля"},
		{"влюбляясь", "влюбл"},
		{"вместе", "вмест"},
		{"вместо", "вмест"},
		{"вмешалась", "вмеша"},
		{"вмешался", "вмеша"},
		{"вмещал", "вмеща"},
		{"внаем", "вна"},
		{"внакидку", "внакидк"},
		{"внакладочку", "внакладочк"},
		{"вначале", "вначал"},
		{"вне", "вне"},
		{"внезапная", "внезапн"},
		{"внезапно", "внезапн"},
		{"внезапного", "внезапн"},
		{"внезапное", "внезапн"},
		{"внезапной", "внезапн"},
		{"внезапном", "внезапн"},
		{"внезапному", "внезапн"},
		{"внезапною", "внезапн"},
		{"внезапную", "внезапн"},
		{"внезапные", "внезапн"},
		{"внезапный", "внезапн"},
		{"внезапным", "внезапн"},
		{"внезапными", "внезапн"},
		{"внезапных", "внезапн"},
		{"внемлю", "внемл"},
		{"внемлют", "внемлют"},
		{"внес", "внес"},
		{"внесенные", "внесен"},
		{"внеси", "внес"},
		{"внесла", "внесл"},
		{"внесли", "внесл"},
		{"внести", "внест"},
		{"внесу", "внес"},
		{"внеся", "вне"},
		{"внешне", "внешн"},
		{"внешней", "внешн"},
		{"внешний", "внешн"},
		{"внешним", "внешн"},
		{"внешних", "внешн"},
		{"внешнюю", "внешн"},
		{"внешняя", "внешн"},
		{"вниз", "вниз"},
		{"внизу", "вниз"},
		{"вникаешь", "вника"},
		{"вникал", "вника"},
		{"вникать", "вника"},
		{"вникая", "вник"},
		{"вникните", "вникн"},
		{"вникнул", "вникнул"},
		{"вникнуть", "вникнут"},
		{"внимай", "внима"},
		{"внимал", "внима"},
		{"внимание", "вниман"},
		{"вниманием", "вниман"},
		{"внимании", "вниман"},
		{"внимания", "вниман"},
		{"вниманье", "вниман"},
		{"вниманью", "вниман"},
		{"вниманья", "вниман"},
		{"внимателен", "внимател"},
		{"внимательнее", "внимательн"},
		{"внимательно", "внимательн"},
		{"внимательные", "внимательн"},
		{"внимательным", "внимательн"},
		{"внимать", "внима"},
		{"внимаю", "внима"},
		{"внимая", "вним"},
		{"внове", "внов"},
		{"вновь", "внов"},
		{"вносит", "внос"},
		{"вносите", "внос"},
		{"вносить", "внос"},
		{"вносишь", "внос"},
		{"внося", "вно"},
		{"внука", "внук"},
		{"внуки", "внук"},
		{"внутренне", "внутрен"},
		{"внутреннего", "внутрен"},
		{"внутреннее", "внутрен"},
		{"внутренней", "внутрен"},
		{"внутреннею", "внутрен"},
		{"внутренние", "внутрен"},
		{"внутренний", "внутрен"},
		{"внутренним", "внутрен"},
		{"внутренних", "внутрен"},
		{"внутренно", "внутрен"},
		{"внутренности", "внутрен"},
		{"внутреннюю", "внутрен"},
		{"внутренняя", "внутрен"},
		{"внутри", "внутр"},
		{"внутрь", "внутр"},
		{"внучата", "внучат"},
		{"внучка", "внучк"},
		{"внушает", "внуша"},
		{"внушал", "внуша"},
		{"внушало", "внуша"},
		{"внушают", "внуша"},
		{"внушению", "внушен"},
		{"внушения", "внушен"},
		{"внушенная", "внушен"},
		{"внушенным", "внушен"},
		{"внушив", "внуш"},
		{"внушил", "внуш"},
		{"внушите", "внуш"},
		{"внушительнее", "внушительн"},
		{"внушительно", "внушительн"},
		{"внушительный", "внушительн"},
		{"внушительным", "внушительн"},
		{"внушить", "внуш"},
		{"внятен", "внят"},
		{"внятно", "внятн"},
		{"внятной", "внятн"},
		{"внятным", "внятн"},
		{"во", "во"},
		{"вобла", "вобл"},
		{"вовек", "вовек"},
		{"вовлечен", "вовлеч"},
		{"вовремя", "воврем"},
		{"вовсе", "вовс"},
		{"вогнать", "вогна"},
		{"вод", "вод"},
		{"вода", "вод"},
		{"водворения", "водворен"},
		{"водворилась", "водвор"},
		{"водворилось", "водвор"},
		{"водворить", "водвор"},
		{"водворялся", "водворя"},
		{"водворят", "водвор"},
		{"воде", "вод"},
		{"водевиля", "водевил"},
		{"водил", "вод"},
		{"водила", "вод"},
		{"водили", "вод"},
		{"водились", "вод"},
		{"водилось", "вод"},
		{"водит", "вод"},
		{"водится", "вод"},
		{"водить", "вод"},
		{"водицы", "водиц"},
		{"водка", "водк"},
		{"водки", "водк"},
		{"водкой", "водк"},
		{"водку", "водк"},
		{"водной", "водн"},
		{"водовоз", "водовоз"},
		{"водоворот", "водоворот"},
		{"водоеме", "водоем"},
		{"водой", "вод"},
		{"водопад", "водопад"},
		{"водопады", "водопад"},
		{"водочки", "водочк"},
		{"водочку", "водочк"},
		{"водою", "вод"},
		{"воду", "вод"},
		{"воды", "вод"},
		{"водяная", "водян"},
		{"водянистый", "водянист"},
		{"водянистым", "водянист"},
		{"водяного", "водян"},
		{"водяной", "водян"},
		{"водяную", "водян"},
		{"водяные", "водян"},
		{"водят", "вод"},
		{"воедино", "воедин"},
		{"военного", "воен"},
		{"военное", "воен"},
		{"военной", "воен"},
		{"военною", "воен"},
		{"военную", "воен"},
		{"военные", "воен"},
		{"военный", "воен"},
		{"военным", "воен"},
		{"военных", "воен"},
		{"воет", "воет"},
		{"вожак", "вожак"},
		{"вожди", "вожд"},
		{"вождь", "вожд"},
		{"вожжи", "вожж"},
		{"воз", "воз"},
		{"воза", "воз"},
		{"возалкал", "возалка"},
		{"возами", "воз"},
		{"возблагодарили", "возблагодар"},
		{"возблагодарю", "возблагодар"},
		{"возбудил", "возбуд"},
		{"возбудила", "возбуд"},
		{"возбудило", "возбуд"},
		{"возбудимы", "возбудим"},
		{"возбудить", "возбуд"},
		{"возбудят", "возбуд"},
		{"возбуждает", "возбужда"},
		{"возбуждаете", "возбужда"},
		{"возбуждал", "возбужда"},
		{"возбуждались", "возбужда"},
		{"возбуждало", "возбужда"},
		{"возбуждать", "возбужда"},
		{"возбуждая", "возбужд"},
		{"возбуждении", "возбужден"},
		{"возбуждения", "возбужден"},
		{"возбужденно", "возбужден"},
		{"возбужденное", "возбужден"},
		{"возбужденном", "возбужден"},
		{"возбужденный", "возбужден"},
		{"возведен", "возвед"},
		{"возведении", "возведен"},
		{"возвел", "возвел"},
		{"возвела", "возвел"},
		{"возвестили", "возвест"},
		{"возвещено", "возвещ"},
		{"возврат", "возврат"},
		{"возврата", "возврат"},
		{"возвратившихся", "возврат"},
		{"возвратила", "возврат"},
		{"возвратилась", "возврат"},
		{"возвратился", "возврат"},
		{"возвратите", "возврат"},
		{"возвратится", "возврат"},
		{"возвратить", "возврат"},
		{"возвратиться", "возврат"},
		{"возвратишь", "возврат"},
		{"возвратного", "возвратн"},
		{"возвратный", "возвратн"},
		{"возвратясь", "возврат"},
		{"возвращавшийся", "возвраща"},
		{"возвращается", "возвраща"},
		{"возвращаешься", "возвраща"},
		{"возвращал", "возвраща"},
		{"возвращалась", "возвраща"},
		{"возвращались", "возвраща"},
		{"возвращалось", "возвраща"},
		{"возвращался", "возвраща"},
		{"возвращать", "возвраща"},
		{"возвращаться", "возвраща"},
		{"возвращаюсь", "возвраща"},
		{"возвращаются", "возвраща"},
		{"возвращающегося", "возвраща"},
		{"возвращаясь", "возвра"},
		{"возвращение", "возвращен"},
		{"возвращении", "возвращен"},
		{"возвращения", "возвращен"},
		{"возвращенья", "возвращен"},
		{"возвращу", "возвращ"},
		{"возвысив", "возвыс"},
		{"возвысился", "возвыс"},
		{"возвысить", "возвыс"},
		{"возвышались", "возвыша"},
		{"возвышалось", "возвыша"},
		{"возвышался", "возвыша"},
		{"возвышать", "возвыша"},
		{"возвышаться", "возвыша"},
		{"возвышая", "возвыш"},
		{"возвышаясь", "возвыш"},
		{"возвышенная", "возвышен"},
		{"возвышенный", "возвышен"},
		{"возглаголет", "возглаголет"},
		{"возглаголят", "возглагол"},
		{"возглас", "возглас"},
		{"возгласы", "возглас"},
		{"воздвиглась", "воздвигл"},
		{"воздвигнутых", "воздвигнут"},
		{"воздел", "воздел"},
		{"возделывал", "возделыва"},
		{"воздержание", "воздержан"},
		{"воздержнее", "воздержн"},
		{"воздусех", "воздусех"},
		{"воздух", "воздух"},
		{"воздуха", "воздух"},
		{"воздухе", "воздух"},
		{"воздухом", "воздух"},
		{"воздуху", "воздух"},
		{"воздушном", "воздушн"},
		{"воздушные", "воздушн"},
		{"воздушный", "воздушн"},
		{"воздушных", "воздушн"},
		{"воздыханий", "воздыхан"},
		{"возжаждешь", "возжаждеш"},
		{"воззвал", "воззва"},
		{"воззвание", "воззван"},
		{"воззрение", "воззрен"},
		{"воззрения", "воззрен"},
		{"воззреньях", "воззрен"},
		{"возила", "воз"},
		{"возили", "воз"},
		{"возились", "воз"},
		{"возился", "воз"},
		{"возить", "воз"},
		{"возиться", "воз"},
		{"возишься", "воз"},
		{"возле", "возл"},
		{"возлелеянную", "возлелея"},
		{"возложил", "возлож"},
		{"возлюби", "возлюб"},
		{"возлюбила", "возлюб"},
		{"возлюбишь", "возлюб"},
		{"возлюбленного", "возлюблен"},
		{"возлюбленной", "возлюблен"},
		{"возлюбленные", "возлюблен"},
		{"возлюбленный", "возлюблен"},
		{"возлюблял", "возлюбля"},
		{"возмездие", "возмезд"},
		{"возмездия", "возмезд"},
		{"возможна", "возможн"},
		{"возможно", "возможн"},
		{"возможности", "возможн"},
		{"возможность", "возможн"},
		{"возможную", "возможн"},
		{"возможным", "возможн"},
		{"возможных", "возможн"},
		{"возмутившее", "возмут"},
		{"возмутился", "возмут"},
		{"возмутительно", "возмутительн"},
		{"возмутителям", "возмутител"},
		{"возмутить", "возмут"},
		{"возмутишь", "возмут"},
		{"возмущает", "возмуща"},
		{"возмущающих", "возмуща"},
		{"возмущаясь", "возмущ"},
		{"возмущен", "возмущ"},
		{"возмущено", "возмущ"},
		{"вознаградить", "вознаград"},
		{"вознаграждать", "вознагражда"},
		{"вознаграждая", "вознагражд"},
		{"вознаграждение", "вознагражден"},
		{"вознаграждением", "вознагражден"},
		{"вознамерился", "вознамер"},
		{"возненавидел", "возненавидел"},
		{"возненавидела", "возненавидел"},
		{"возненавидит", "возненавид"},
		{"возненавидишь", "возненавид"},
		{"возненавижу", "возненавиж"},
		{"вознесенском", "вознесенск"},
		{"вознесены", "вознес"},
		{"вознесу", "вознес"},
		{"возни", "возн"},
		{"возник", "возник"},
		{"возникает", "возника"},
		{"возникал", "возника"},
		{"возникала", "возника"},
		{"возникать", "возника"},
		{"возникающей", "возника"},
		{"возникла", "возникл"},
		{"возникло", "возникл"},
		{"возникнет", "возникнет"},
		{"возникновения", "возникновен"},
		{"возникшего", "возникш"},
		{"возносилась", "вознос"},
		{"возносится", "вознос"},
		{"вознося", "возн"},
		{"возню", "возн"},
		{"возня", "возн"},
		{"возобновились", "возобнов"},
		{"возобновить", "возобнов"},
		{"возобновляет", "возобновля"},
		{"возобновлялась", "возобновля"},
		{"возов", "воз"},
		{"возом", "воз"},
		{"возопил", "возоп"},
		{"возражает", "возража"},
		{"возражал", "возража"},
		{"возражений", "возражен"},
		{"возражения", "возражен"},
		{"возразил", "возраз"},
		{"возразила", "возраз"},
		{"возразить", "возраз"},
		{"возраст", "возраст"},
		{"возраста", "возраст"},
		{"возраставшего", "возраста"},
		{"возраставшее", "возраста"},
		{"возраставшим", "возраста"},
		{"возрастало", "возраста"},
		{"возрастающей", "возраста"},
		{"возрастая", "возраст"},
		{"возрасте", "возраст"},
		{"возрастом", "возраст"},
		{"возрасты", "возраст"},
		{"возродить", "возрод"},
		{"возрождалась", "возрожда"},
		{"возрождение", "возрожден"},
		{"возрождению", "возрожден"},
		{"возросла", "возросл"},
		{"возросло", "возросл"},
		{"возу", "воз"},
		{"возымел", "возымел"},
		{"возыметь", "возымет"},
		{"возьмем", "возьм"},
		{"возьмет", "возьмет"},
		{"возьмете", "возьмет"},
		{"возьметесь", "возьмет"},
		{"возьмется", "возьмет"},
		{"возьмешь", "возьмеш"},
		{"возьми", "возьм"},
		{"возьмите", "возьм"},
		{"возьму", "возьм"},
		{"возьмут", "возьмут"},
		{"возьмутся", "возьмут"},
		{"возят", "воз"},
		{"воин", "воин"},
		{"воина", "воин"},
		{"воину", "воин"},
		{"воины", "воин"},
		{"воистину", "воистин"},
		{"вой", "во"},
		{"войдем", "войд"},
		{"войдет", "войдет"},
		{"войдешь", "войдеш"},
		{"войди", "войд"},
		{"войдите", "войд"},
		{"войду", "войд"},
		{"войдут", "войдут"},
		{"войдя", "войд"},
		{"войлок", "войлок"},
		{"войлоками", "войлок"},
		{"войлоке", "войлок"},
		{"войн", "войн"},
		{"война", "войн"},
		{"войнами", "войн"},
		{"войне", "войн"},
		{"войницкая", "войницк"},
		{"войницкии", "войницк"},
		{"войницкий", "войницк"},
		{"войницким", "войницк"},
		{"войницкого", "войницк"},
		{"войницкому", "войницк"},
		{"войну", "войн"},
		{"войны", "войн"},
		{"войска", "войск"},
		{"войско", "войск"},
		{"войском", "войск"},
		{"войти", "войт"},
		{"вокзал", "вокза"},
		{"вокзале", "вокзал"},
		{"вокзалу", "вокзал"},
		{"вокруг", "вокруг"},
		{"вол", "вол"},
		{"волге", "волг"},
		{"волгу", "волг"},
		{"воле", "вол"},
		{"волей", "вол"},
		{"волен", "вол"},
		{"волею", "вол"},
		{"воли", "вол"},
		{"волк", "волк"},
		{"волки", "волк"},
		{"волков", "волк"},
		{"волн", "волн"},
		{"волна", "волн"},
		{"волнами", "волн"},
		{"волнах", "волн"},
		{"волнение", "волнен"},
		{"волнением", "волнен"},
		{"волнении", "волнен"},
		{"волнений", "волнен"},
		{"волнения", "волнен"},
		{"волнениями", "волнен"},
		{"волненье", "волнен"},
		{"волненьем", "волнен"},
		{"волненья", "волнен"},
		{"волновала", "волнова"},
		{"волновалась", "волнова"},
		{"волновали", "волнова"},
		{"волновались", "волнова"},
		{"волновало", "волнова"},
		{"волновался", "волнова"},
		{"волновать", "волнова"},
		{"волноваться", "волнова"},
		{"волной", "волн"},
		{"волною", "волн"},
		{"волну", "волн"},
		{"волнуем", "волну"},
		{"волнует", "волн"},
		{"волнуете", "волнует"},
		{"волнуетесь", "волнует"},
		{"волнуется", "волн"},
		{"волнуешь", "волнуеш"},
		{"волнуешься", "волнуеш"},
		{"волнуйся", "волн"},
		{"волнуйтесь", "волн"},
		{"волнуюсь", "волн"},
		{"волнуют", "волн"},
		{"волнуются", "волн"},
		{"волнующей", "волн"},
		{"волнующейся", "волн"},
		{"волнующуюся", "волн"},
		{"волнуясь", "волну"},
		{"волны", "волн"},
		{"воло", "вол"},
		{"волокита", "волокит"},
		{"волокиты", "волокит"},
		{"волоком", "волок"},
		{"волом", "вол"},
		{"волос", "волос"},
		{"волоса", "волос"},
		{"волосам", "волос"},
		{"волосами", "волос"},
		{"волосатые", "волосат"},
		{"волосах", "волос"},
		{"волосенки", "волосенк"},
		{"волосики", "волосик"},
		{"волоска", "волоск"},
		{"волоски", "волоск"},
		{"волости", "волост"},
		{"волосы", "волос"},
		{"волочит", "волоч"},
		{"волчком", "волчк"},
		{"волчок", "волчок"},
		{"волчьим", "волч"},
		{"волшебная", "волшебн"},
		{"волшебника", "волшебник"},
		{"волшебниц", "волшебниц"},
		{"волшебница", "волшебниц"},
		{"волшебницы", "волшебниц"},
		{"волшебного", "волшебн"},
		{"волшебной", "волшебн"},
		{"волшебные", "волшебн"},
		{"волшебств", "волшебств"},
		{"воль", "вол"},
		{"вольна", "вольн"},
		{"вольно", "вольн"},
		{"вольнодумные", "вольнодумн"},
		{"вольнодумцам", "вольнодумц"},
		{"вольное", "вольн"},
		{"вольной", "вольн"},
		{"вольном", "вольн"},
		{"вольному", "вольн"},
		{"вольною", "вольн"},
		{"вольные", "вольн"},
		{"вольный", "вольн"},
		{"вольным", "вольн"},
		{"вольтер", "вольтер"},
		{"волю", "вол"},
		{"воля", "вол"},
		{"вон", "вон"},
		{"вона", "вон"},
		{"вонзает", "вонза"},
		{"вонзай", "вонза"},
		{"вонзал", "вонза"},
		{"вонзила", "вонз"},
		{"вонзился", "вонз"},
		{"вонзить", "вонз"},
		{"вони", "вон"},
		{"вонь", "вон"},
		{"вонью", "вон"},
		{"вонючего", "вонюч"},
		{"вонючих", "вонюч"},
		{"воображает", "вообража"},
		{"воображаешь", "вообража"},
		{"воображал", "вообража"},
		{"воображала", "вообража"},
		{"воображали", "вообража"},
		{"воображать", "вообража"},
		{"воображаю", "вообража"},
		{"воображают", "вообража"},
		{"воображая", "воображ"},
		{"воображение", "воображен"},
		{"воображением", "воображен"},
		{"воображении", "воображен"},
		{"воображению", "воображен"},
		{"воображения", "воображен"},
		{"воображениями", "воображен"},
		{"воображениях", "воображен"},
		{"вообрази", "вообраз"},
		{"вообразил", "вообраз"},
		{"вообразила", "вообраз"},
		{"вообразили", "вообраз"},
		{"вообразилось", "вообраз"},
		{"вообразился", "вообраз"},
		{"вообразись", "вообраз"},
		{"вообразит", "вообраз"},
		{"вообразите", "вообраз"},
		{"вообразится", "вообраз"},
		{"вообразить", "вообраз"},
		{"вообще", "вообщ"},
		{"вооружался", "вооружа"},
		{"вооружаться", "вооружа"},
		{"вооружаясь", "вооруж"},
		{"вооружений", "вооружен"},
		{"вооруженная", "вооружен"},
		{"вооруженный", "вооружен"},
		{"вооружились", "вооруж"},
		{"вооружиться", "вооруж"},
		{"вопил", "воп"},
		{"вопила", "воп"},
		{"вопиющее", "вопиющ"},
		{"вопиющим", "вопиющ"},
		{"вопле", "вопл"},
		{"воплей", "вопл"},
		{"воплем", "вопл"},
		{"вопли", "вопл"},
		{"воплотился", "воплот"},
		{"воплотить", "воплот"},
		{"воплощался", "воплоща"},
		{"воплощение", "воплощен"},
		{"воплощенного", "воплощен"},
		{"вопль", "вопл"},
		{"вопля", "вопл"},
		{"воплям", "вопл"},
		{"воплями", "вопл"},
		{"вопреки", "вопрек"},
		{"вопрос", "вопрос"},
		{"вопроса", "вопрос"},
		{"вопросам", "вопрос"},
		{"вопросами", "вопрос"},
		{"вопросах", "вопрос"},
		{"вопросе", "вопрос"},
		{"вопросик", "вопросик"},
		{"вопросительно", "вопросительн"},
		{"вопросительного", "вопросительн"},
		{"вопросительные", "вопросительн"},
		{"вопросительный", "вопросительн"},
		{"вопросов", "вопрос"},
		{"вопросом", "вопрос"},
		{"вопросу", "вопрос"},
		{"вопросы", "вопрос"},
		{"вопрошаю", "вопроша"},
		{"вопрошающей", "вопроша"},
		{"вопрошающий", "вопроша"},
		{"вопрошающих", "вопроша"},
		{"вопьется", "вопьет"},
		{"вор", "вор"},
		{"вора", "вор"},
		{"ворвавшись", "ворва"},
		{"ворвалась", "ворва"},
		{"ворвался", "ворва"},
		{"ворвется", "ворвет"},
		{"воробей", "вороб"},
		{"воробушек", "воробушек"},
		{"воробушка", "воробушк"},
		{"воров", "вор"},
		{"вороваль", "воровал"},
		{"воровать", "ворова"},
		{"воровка", "воровк"},
		{"воровки", "воровк"},
		{"воровских", "воровск"},
		{"ворожим", "ворож"},
		{"вором", "вор"},
		{"ворон", "ворон"},
		{"вороная", "ворон"},
		{"воронку", "воронк"},
		{"вороны", "ворон"},
		{"воронье", "ворон"},
		{"воронья", "ворон"},
		{"ворот", "ворот"},
		{"ворота", "ворот"},
		{"воротам", "ворот"},
		{"воротами", "ворот"},
		{"воротах", "ворот"},
		{"вороти", "ворот"},
		{"воротившийся", "ворот"},
		{"воротившись", "ворот"},
		{"воротила", "ворот"},
		{"воротилась", "ворот"},
		{"воротились", "ворот"},
		{"воротилось", "ворот"},
		{"воротился", "ворот"},
		{"воротись", "ворот"},
		{"воротитесь", "ворот"},
		{"воротится", "ворот"},
		{"воротить", "ворот"},
		{"воротиться", "ворот"},
		{"воротишь", "ворот"},
		{"воротишься", "ворот"},
		{"воротник", "воротник"},
		{"воротников", "воротник"},
		{"воротником", "воротник"},
		{"воротничках", "воротничк"},
		{"воротнички", "воротничк"},
		{"воротничок", "воротничок"},
		{"воротясь", "ворот"},
		{"воротятся", "ворот"},
		{"ворохом", "ворох"},
		{"ворочает", "вороча"},
		{"ворочается", "вороча"},
		{"ворочаешься", "вороча"},
		{"ворочал", "вороча"},
		{"ворочала", "вороча"},
		{"ворочались", "вороча"},
		{"ворочало", "вороча"},
		{"ворочалось", "вороча"},
		{"ворочался", "вороча"},
		{"ворочать", "вороча"},
		{"ворочаться", "вороча"},
		{"ворочающими", "вороча"},
		{"ворочаясь", "вороч"},
		{"ворочу", "вороч"},
		{"ворочусь", "вороч"},
		{"ворует", "вор"},
		{"воруете", "ворует"},
		{"ворующего", "вор"},
		{"ворча", "ворч"},
		{"ворчавшую", "ворча"},
		{"ворчал", "ворча"},
		{"ворчала", "ворча"},
		{"ворчанье", "ворчан"},
		{"ворчаньем", "ворчан"},
		{"ворчать", "ворча"},
		{"ворчит", "ворч"},
		{"ворчишь", "ворч"},
		{"ворчливые", "ворчлив"},
		{"ворчу", "ворч"},
		{"воры", "вор"},
		{"восемнадцати", "восемнадцат"},
		{"восемнадцать", "восемнадца"},
		{"восемь", "восем"},
		{"восемьдесят", "восемьдес"},
		{"восемьсот", "восемьсот"},
		{"воск", "воск"},
		{"воскликнул", "воскликнул"},
		{"воскликнула", "воскликнул"},
		{"восклицал", "восклица"},
		{"восклицала", "восклица"},
		{"восклицание", "восклицан"},
		{"восклицанием", "восклицан"},
		{"восклицании", "восклицан"},
		{"восклицанию", "восклицан"},
		{"восклицания", "восклицан"},
		{"восклицаниями", "восклицан"},
		{"восклицанья", "восклицан"},
		{"восклицая", "восклиц"},
		{"восклоняясь", "восклон"},
		{"воскрес", "воскрес"},
		{"воскресать", "воскреса"},
		{"воскресение", "воскресен"},
		{"воскресении", "воскресен"},
		{"воскресения", "воскресен"},
		{"воскресенье", "воскресен"},
		{"воскресенью", "воскресен"},
		{"воскресенья", "воскресен"},
		{"воскресеньям", "воскресен"},
		{"воскресил", "воскрес"},
		{"воскресила", "воскрес"},
		{"воскресить", "воскрес"},
		{"воскресла", "воскресл"},
		{"воскресло", "воскресл"},
		{"воскреснет", "воскреснет"},
		{"воскреснув", "воскреснув"},
		{"воскресную", "воскресн"},
		{"воскресный", "воскресн"},
		{"воскресных", "воскресн"},
		{"воскресший", "воскресш"},
		{"воскресят", "воскрес"},
		{"воскрешает", "воскреша"},
		{"воскрешало", "воскреша"},
		{"вослед", "вослед"},
		{"воспаление", "воспален"},
		{"воспаления", "воспален"},
		{"воспаленно", "воспален"},
		{"воспаленного", "воспален"},
		{"воспаленный", "воспален"},
		{"воспаленным", "воспален"},
		{"воспаленных", "воспален"},
		{"воспитаешь", "воспита"},
		{"воспитался", "воспита"},
		{"воспитан", "воспита"},
		{"воспитана", "воспита"},
		{"воспитание", "воспитан"},
		{"воспитанием", "воспитан"},
		{"воспитанию", "воспитан"},
		{"воспитания", "воспитан"},
		{"воспитанная", "воспита"},
		{"воспитанные", "воспита"},
		{"воспитанный", "воспита"},
		{"воспитанных", "воспита"},
		{"воспитаны", "воспита"},
		{"воспитаться", "воспита"},
		{"воспитывалась", "воспитыва"},
		{"воспитывался", "воспитыва"},
		{"воспитывать", "воспитыва"},
		{"воспитывая", "воспитыв"},
		{"воспламенившись", "воспламен"},
		{"воспламенился", "воспламен"},
		{"воспламенялся", "воспламеня"},
		{"воспользовавшись", "воспользова"},
		{"воспользовались", "воспользова"},
		{"воспользовался", "воспользова"},
		{"воспользоваться", "воспользова"},
		{"воспользуется", "воспольз"},
		{"воспоминание", "воспоминан"},
		{"воспоминанием", "воспоминан"},
		{"воспоминании", "воспоминан"},
		{"воспоминаний", "воспоминан"},
		{"воспоминанию", "воспоминан"},
		{"воспоминания", "воспоминан"},
		{"воспоминаниями", "воспоминан"},
		{"воспоминаниях", "воспоминан"},
		{"воспоминанье", "воспоминан"},
		{"воспоминанья", "воспоминан"},
		{"воспоследует", "воспослед"},
		{"воспрещено", "воспрещ"},
		{"восприимчив", "восприимч"},
		{"восприимчивая", "восприимчив"},
		{"восприимчивом", "восприимчив"},
		{"восприимчивость", "восприимчив"},
		{"восприимчивый", "восприимчив"},
		{"восприняла", "восприня"},
		{"воспринятию", "воспринят"},
		{"воспроизводится", "воспроизвод"},
		{"воспротивиться", "воспротив"},
		{"воспрянуть", "воспрянут"},
		{"восскорбел", "восскорбел"},
		{"восстав", "восста"},
		{"восставал", "восстава"},
		{"восставшего", "восста"},
		{"восстал", "восста"},
		{"восстали", "восста"},
		{"восстанет", "восстанет"},
		{"восстановил", "восстанов"},
		{"восстановила", "восстанов"},
		{"восстановили", "восстанов"},
		{"восстановить", "восстанов"},
		{"восстановлен", "восстановл"},
		{"восстановляло", "восстановля"},
		{"восстановляя", "восстановл"},
		{"восстанут", "восстанут"},
		{"восстанья", "восстан"},
		{"востер", "востер"},
		{"восток", "восток"},
		{"востока", "восток"},
		{"восторг", "восторг"},
		{"восторга", "восторг"},
		{"восторгам", "восторг"},
		{"восторгами", "восторг"},
		{"восторгаться", "восторга"},
		{"восторге", "восторг"},
		{"восторги", "восторг"},
		{"восторгом", "восторг"},
		{"восторженная", "восторжен"},
		{"восторженно", "восторжен"},
		{"восторженного", "восторжен"},
		{"восторженное", "восторжен"},
		{"восторженность", "восторжен"},
		{"восторженностью", "восторжен"},
		{"восторженною", "восторжен"},
		{"восторженный", "восторжен"},
		{"восторженным", "восторжен"},
		{"восторженными", "восторжен"},
		{"восторженных", "восторжен"},
		{"восточной", "восточн"},
		{"восточный", "восточн"},
		{"востреньким", "востреньк"},
		{"востренькое", "востреньк"},
		{"востро", "востр"},
		{"востроногий", "востроног"},
		{"вострые", "востр"},
		{"вострым", "востр"},
		{"вострыми", "востр"},
		{"восхвалить", "восхвал"},
		{"восхитительная", "восхитительн"},
		{"восхищался", "восхища"},
		{"восхищаться", "восхища"},
		{"восхищение", "восхищен"},
		{"восхищенного", "восхищен"},
		{"восхищеньи", "восхищен"},
		{"восходил", "восход"},
		{"восходит", "восход"},
		{"восходить", "восход"},
		{"восходном", "восходн"},
		{"восхождения", "восхожден"},
		{"восьмерка", "восьмерк"},
		{"восьми", "восьм"},
		{"восьмидесяти", "восьмидесят"},
		{"восьмидесятых", "восьмидесят"},
		{"восьмилетнего", "восьмилетн"},
		{"восьмого", "восьм"},
		{"восьмой", "восьм"},
		{"восьмом", "восьм"},
		{"восьмому", "восьм"},
		{"восьмую", "восьм"},
		{"вот", "вот"},
		{"воткнул", "воткнул"},
		{"воткнут", "воткнут"},
		{"вотчине", "вотчин"},
		{"воцарилась", "воцар"},
		{"воцарилось", "воцар"},
		{"воцаряется", "воцаря"},
		{"вочеловечить", "вочеловеч"},
		{"вошедшая", "вошедш"},
		{"вошедшего", "вошедш"},
		{"вошедшему", "вошедш"},
		{"вошедший", "вошедш"},
		{"вошедшими", "вошедш"},
		{"вошедших", "вошедш"},
		{"вошедшую", "вошедш"},
		{"вошел", "вошел"},
		{"вошла", "вошл"},
		{"вошли", "вошл"},
		{"вошло", "вошл"},
		{"вошь", "вош"},
		{"воюет", "воюет"},
		{"воющую", "воющ"},
		{"воя", "во"},
		{"вояж", "вояж"},
		{"вояжа", "вояж"},
		{"вояжем", "вояж"},
		{"вояжирует", "вояжир"},
		{"впавшими", "впавш"},
		{"впадавшего", "впада"},
		{"впадает", "впада"},
		{"впадал", "впада"},
		{"впадала", "впада"},
		{"впадая", "впад"},
		{"впадешь", "впадеш"},
		{"впал", "впал"},
		{"впала", "впал"},
		{"впали", "впал"},
		{"впасть", "впаст"},
		{"впервой", "вперв"},
		{"впервые", "вперв"},
		{"вперегонку", "вперегонк"},
		{"вперед", "вперед"},
		{"впереди", "вперед"},
		{"вперяя", "впер"},
		{"вперяясь", "впер"},
		{"впечатление", "впечатлен"},
		{"впечатлении", "впечатлен"},
		{"впечатлений", "впечатлен"},
		{"впечатлению", "впечатлен"},
		{"впечатления", "впечатлен"},
		{"впечатлительное", "впечатлительн"},
		{"впечатлительностью", "впечатлительн"},
		{"впивался", "впива"},
		{"впиваясь", "впив"},
		{"впилась", "впил"},
		{"впился", "впил"},
		{"вплелись", "вплел"},
		{"вплести", "вплест"},
		{"вплетаю", "вплета"},
		{"вплоть", "вплот"},
		{"вповалку", "вповалк"},
		{"вполглаза", "вполглаз"},
		{"вполголоса", "вполголос"},
		{"вползает", "вполза"},
		{"вползало", "вполза"},
		{"вполне", "вполн"},
		{"вполовину", "вполовин"},
		{"вполуоборот", "вполуоборот"},
		{"впопад", "впопад"},
		{"впопыхах", "впопых"},
		{"впору", "впор"},
		{"впоследствии", "впоследств"},
		{"впотьмах", "впотьм"},
		{"вправду", "вправд"},
		{"вправе", "вправ"},
		{"вправо", "вправ"},
		{"впредь", "впред"},
		{"впридачу", "впридач"},
		{"впрок", "впрок"},
		{"впрочем", "впроч"},
		{"впрягают", "впряга"},
		{"впряжена", "впряж"},
		{"впрямь", "впрям"},
		{"впускает", "впуска"},
		{"впускала", "впуска"},
		{"впускать", "впуска"},
		{"впустить", "впуст"},
		{"впустят", "впуст"},
		{"впятеро", "впятер"},
		{"вр", "вр"},
		{"враг", "враг"},
		{"врага", "враг"},
		{"врагами", "враг"},
		{"враги", "враг"},
		{"врагини", "врагин"},
		{"врагов", "враг"},
		{"врагом", "враг"},
		{"врагу", "враг"},
		{"вражда", "вражд"},
		{"враждебно", "враждебн"},
		{"враждебного", "враждебн"},
		{"враждебное", "враждебн"},
		{"враждебную", "враждебн"},
		{"враждебны", "враждебн"},
		{"враждебные", "враждебн"},
		{"враждебными", "враждебн"},
		{"враждебных", "враждебн"},
		{"враждой", "вражд"},
		{"вражды", "вражд"},
		{"вражеским", "вражеск"},
		{"вражеских", "вражеск"},
		{"вражий", "враж"},
		{"вражьей", "враж"},
		{"вражьим", "враж"},
		{"враздробь", "враздроб"},
		{"вразумите", "вразум"},
		{"вразумихин", "вразумихин"},
		{"врали", "врал"},
		{"вранье", "вран"},
		{"вранью", "вран"},
		{"врасплох", "врасплох"},
		{"врать", "врат"},
		{"врач", "врач"},
		{"врача", "врач"},
		{"врачах", "врач"},
		{"врачебному", "врачебн"},
		{"врачу", "врач"},
		{"вращает", "враща"},
		{"вращается", "враща"},
		{"вращалось", "враща"},
		{"вревской", "вревск"},
		{"вред", "вред"},
		{"вреда", "вред"},
		{"вреден", "вред"},
		{"вредить", "вред"},
		{"вредна", "вредн"},
		{"вредная", "вредн"},
		{"вредно", "вредн"},
		{"вредны", "вредн"},
		{"вредные", "вредн"},
		{"вредных", "вредн"},
		{"вредят", "вред"},
		{"врезался", "вреза"},
		{"врезывается", "врезыва"},
		{"врем", "врем"},
		{"времен", "врем"},
		{"времена", "врем"},
		{"временам", "времен"},
		{"временем", "времен"},
		{"времени", "времен"},
		{"временно", "времен"},
		{"временного", "времен"},
		{"временное", "времен"},
		{"временной", "времен"},
		{"временном", "времен"},
		{"временным", "времен"},
		{"временных", "времен"},
		{"времечко", "времечк"},
		{"время", "врем"},
		{"врет", "врет"},
		{"врете", "врет"},
		{"врешь", "вреш"},
		{"ври", "ври"},
		{"вроде", "врод"},
		{"врожденная", "врожден"},
		{"врожденную", "врожден"},
		{"врожденных", "врожден"},
		{"врозь", "вроз"},
		{"вросла", "вросл"},
		{"вру", "вру"},
		{"врубелем", "врубел"},
		{"врубель", "врубел"},
		{"врубеля", "врубел"},
		{"врут", "врут"},
		{"врученных", "вручен"},
		{"вручил", "вруч"},
		{"вручить", "вруч"},
		{"врывавшихся", "врыва"},
		{"врывается", "врыва"},
		{"врывалось", "врыва"},
		{"врываясь", "врыв"},
		{"врыть", "врыт"},
		{"вряд", "вряд"},
		{"всажу", "всаж"},
		{"все", "все"},
		{"всеблагое", "всеблаг"},
		{"всевдонимом", "всевдоним"},
		{"всевозможные", "всевозможн"},
		{"всевозможных", "всевозможн"},
		{"всевышнего", "всевышн"},
		{"всевышнему", "всевышн"},
		{"всегда", "всегд"},
		{"всегдашнее", "всегдашн"},
		{"всегдашний", "всегдашн"},
		{"всегдашним", "всегдашн"},
		{"всегдашнюю", "всегдашн"},
		{"всегдашняя", "всегдашн"},
		{"всего", "всег"},
		{"вседержителя", "вседержител"},
		{"вседневная", "вседневн"},
		{"всеевропейской", "всеевропейск"},
		{"всезнание", "всезнан"},
		{"всезнании", "всезнан"},
		{"всезнающей", "всезна"},
		{"всезнающие", "всезна"},
		{"всей", "все"},
		{"вселенная", "вселен"},
		{"вселенной", "вселен"},
		{"вселенской", "вселенск"},
		{"вселились", "всел"},
		{"вселился", "всел"},
		{"вселитературно", "вселитературн"},
		{"вселить", "всел"},
		{"вселявшиеся", "вселя"},
		{"вселяется", "вселя"},
		{"всем", "всем"},
		{"всеми", "всем"},
		{"всемирной", "всемирн"},
		{"всемирную", "всемирн"},
		{"всемирный", "всемирн"},
		{"всему", "всем"},
		{"всенаполняющего", "всенаполня"},
		{"всенародно", "всенародн"},
		{"всенижайший", "всенижайш"},
		{"всеной", "всен"},
		{"всенощной", "всенощн"},
		{"всеобщего", "всеобщ"},
		{"всеобщее", "всеобщ"},
		{"всеобщей", "всеобщ"},
		{"всеобщих", "всеобщ"},
		{"всеобъемлемостью", "всеобъемлем"},
		{"всеоружии", "всеоруж"},
		{"всепоглощающий", "всепоглоща"},
		{"всесветными", "всесветн"},
		{"всесметающая", "всесмета"},
		{"всесторонней", "всесторон"},
		{"всеуслышание", "всеуслышан"},
		{"всех", "всех"},
		{"всецело", "всецел"},
		{"всею", "все"},
		{"вскакивает", "вскакива"},
		{"вскакивал", "вскакива"},
		{"вскакивала", "вскакива"},
		{"вскакивают", "вскакива"},
		{"вскакивая", "вскакив"},
		{"вскачь", "вскач"},
		{"вскинув", "вскинув"},
		{"вскинул", "вскинул"},
		{"вскинулась", "вскинул"},
		{"вскинулся", "вскинул"},
		{"вскинутом", "вскинут"},
		{"вскинутые", "вскинут"},
		{"вскипев", "вскип"},
		{"вскипевшего", "вскипевш"},
		{"вскипел", "вскипел"},
		{"вскипело", "вскипел"},
		{"всклокоченный", "всклокочен"},
		{"всклоченного", "всклочен"},
		{"вское", "вско"},
		{"всколосится", "всколос"},
		{"всколыхнулись", "всколыхнул"},
		{"вскользь", "вскольз"},
		{"вскоре", "вскор"},
		{"вскочив", "вскоч"},
		{"вскочившую", "вскоч"},
		{"вскочил", "вскоч"},
		{"вскочила", "вскоч"},
		{"вскочило", "вскоч"},
		{"вскочит", "вскоч"},
		{"вскочишь", "вскоч"},
		{"вскрикивает", "вскрикива"},
		{"вскрикивала", "вскрикива"},
		{"вскрикивания", "вскрикиван"},
		{"вскрикивают", "вскрикива"},
		{"вскрикивая", "вскрикив"},
		{"вскрикнет", "вскрикнет"},
		{"вскрикну", "вскрикн"},
		{"вскрикнул", "вскрикнул"},
		{"вскрикнула", "вскрикнул"},
		{"вскрикнули", "вскрикнул"},
		{"вскрикнут", "вскрикнут"},
		{"вскрикнуть", "вскрикнут"},
		{"вскричал", "вскрича"},
		{"вскричала", "вскрича"},
		{"вскружить", "вскруж"},
		{"вскрывался", "вскрыва"},
		{"вскрывать", "вскрыва"},
		{"вскрылась", "вскрыл"},
		{"вскрыть", "вскрыт"},
		{"всласть", "всласт"},
		{"вслед", "вслед"},
		{"вследствие", "вследств"},
		{"вслух", "вслух"},
		{"вслушивается", "вслушива"},
		{"вслушивалась", "вслушива"},
		{"вслушивался", "вслушива"},
		{"вслушиваться", "вслушива"},
		{"вслушиваясь", "вслушив"},
		{"всматривается", "всматрива"},
		{"всматривался", "всматрива"},
		{"всматриваться", "всматрива"},
		{"всматриваясь", "всматрив"},
		{"всплескивая", "всплескив"},
		{"всплеснув", "всплеснув"},
		{"всплеснул", "всплеснул"},
		{"всплеснула", "всплеснул"},
		{"всплеснулась", "всплеснул"},
		{"всплывало", "всплыва"},
		{"всплывут", "всплывут"},
		{"всплыла", "всплыл"},
		{"вспоенная", "вспоен"},
		{"всполохнутый", "всполохнут"},
		{"всполошив", "всполош"},
		{"всполошился", "всполош"},
		{"вспоминаем", "вспомина"},
		{"вспоминает", "вспомина"},
		{"вспоминается", "вспомина"},
		{"вспоминаешь", "вспомина"},
		{"вспоминай", "вспомина"},
		{"вспоминайте", "вспомина"},
		{"вспоминал", "вспомина"},
		{"вспоминала", "вспомина"},
		{"вспоминалась", "вспомина"},
		{"вспоминали", "вспомина"},
		{"вспоминать", "вспомина"},
		{"вспоминаю", "вспомина"},
		{"вспоминая", "вспомин"},
		{"вспомни", "вспомн"},
		{"вспомнив", "вспомн"},
		{"вспомнил", "вспомн"},
		{"вспомнила", "вспомн"},
		{"вспомнилась", "вспомн"},
		{"вспомнили", "вспомн"},
		{"вспомнились", "вспомн"},
		{"вспомнилось", "вспомн"},
		{"вспомнился", "вспомн"},
		{"вспомнит", "вспомн"},
		{"вспомните", "вспомн"},
		{"вспомнить", "вспомн"},
		{"вспомнишь", "вспомн"},
		{"вспомню", "вспомн"},
		{"вспомнят", "вспомн"},
		{"вспоможение", "вспоможен"},
		{"вспоможении", "вспоможен"},
		{"вспоможения", "вспоможен"},
		{"вспорхнулся", "вспорхнул"},
		{"вспотевшее", "вспотевш"},
		{"вспрыгивает", "вспрыгива"},
		{"вспрыгнул", "вспрыгнул"},
		{"вспрянула", "вспрянул"},
		{"вспугнул", "вспугнул"},
		{"вспугнуть", "вспугнут"},
		{"вспухла", "вспухл"},
		{"вспухшею", "вспухш"},
		{"вспылив", "вспыл"},
		{"вспылил", "вспыл"},
		{"вспыльчива", "вспыльчив"},
		{"вспыльчиво", "вспыльчив"},
		{"вспыльчивые", "вспыльчив"},
		{"вспылю", "вспыл"},
		{"вспыхивает", "вспыхива"},
		{"вспыхивал", "вспыхива"},
		{"вспыхивала", "вспыхива"},
		{"вспыхивают", "вспыхива"},
		{"вспыхнет", "вспыхнет"},
		{"вспыхнув", "вспыхнув"},
		{"вспыхнул", "вспыхнул"},
		{"вспыхнула", "вспыхнул"},
		{"вспыхнули", "вспыхнул"},
		{"вспыхнуло", "вспыхнул"},
		{"вспыхнуть", "вспыхнут"},
		{"вспышкам", "вспышк"},
		{"вспышки", "вспышк"},
		{"вспышку", "вспышк"},
		{"вспять", "вспят"},
		{"встав", "встав"},
		{"вставай", "встава"},
		{"вставайте", "встава"},
		{"вставал", "встава"},
		{"вставала", "встава"},
		{"вставали", "встава"},
		{"вставать", "встава"},
		{"вставая", "встав"},
		{"вставил", "встав"},
		{"вставить", "встав"},
		{"вставлял", "вставля"},
		{"вставши", "вставш"},
		{"вставшие", "вставш"},
		{"вставший", "вставш"},
		{"встает", "встает"},
		{"встаете", "встает"},
		{"встаешь", "встаеш"},
		{"встал", "встал"},
		{"встала", "встал"},
		{"встали", "встал"},
		{"встанемте", "встанемт"},
		{"встанет", "встанет"},
		{"встанешь", "встанеш"},
		{"встану", "встан"},
		{"встанут", "встанут"},
		{"встань", "встан"},
		{"встаньте", "встаньт"},
		{"встарь", "встар"},
		{"встать", "встат"},
		{"встаю", "вста"},
		{"встают", "встают"},
		{"встревожен", "встревож"},
		{"встревоженная", "встревожен"},
		{"встревоженно", "встревожен"},
		{"встревоженного", "встревожен"},
		{"встревоженную", "встревожен"},
		{"встревоженный", "встревожен"},
		{"встревоженным", "встревожен"},
		{"встревоженными", "встревожен"},
		{"встревожившись", "встревож"},
		{"встревожилась", "встревож"},
		{"встревожило", "встревож"},
		{"встревожился", "встревож"},
		{"встревожить", "встревож"},
		{"встрепанный", "встрепа"},
		{"встрепенется", "встрепенет"},
		{"встрепенулась", "встрепенул"},
		{"встрепенулись", "встрепенул"},
		{"встрепенулся", "встрепенул"},
		{"встрепенутся", "встрепенут"},
		{"встретив", "встрет"},
		{"встретившегося", "встрет"},
		{"встретивший", "встрет"},
		{"встретившись", "встрет"},
		{"встретил", "встрет"},
		{"встретила", "встрет"},
		{"встретили", "встрет"},
		{"встретились", "встрет"},
		{"встретилось", "встрет"},
		{"встретился", "встрет"},
		{"встретимся", "встрет"},
		{"встретит", "встрет"},
		{"встретите", "встрет"},
		{"встретится", "встрет"},
		{"встретить", "встрет"},
		{"встретиться", "встрет"},
		{"встретишь", "встрет"},
		{"встретя", "встрет"},
		{"встретятся", "встрет"},
		{"встреч", "встреч"},
		{"встреча", "встреч"},
		{"встречавшемуся", "встреча"},
		{"встречавшиеся", "встреча"},
		{"встречаемся", "встреча"},
		{"встречает", "встреча"},
		{"встречается", "встреча"},
		{"встречаешь", "встреча"},
		{"встречал", "встреча"},
		{"встречала", "встреча"},
		{"встречалась", "встреча"},
		{"встречали", "встреча"},
		{"встречались", "встреча"},
		{"встречалось", "встреча"},
		{"встречался", "встреча"},
		{"встречать", "встреча"},
		{"встречаться", "встреча"},
		{"встречах", "встреч"},
		{"встречаю", "встреча"},
		{"встречаюсь", "встреча"},
		{"встречают", "встреча"},
		{"встречаются", "встреча"},
		{"встречаясь", "встреч"},
		{"встрече", "встреч"},
		{"встречен", "встреч"},
		{"встреченных", "встречен"},
		{"встречи", "встреч"},
		{"встречного", "встречн"},
		{"встречной", "встречн"},
		{"встречному", "встречн"},
		{"встречные", "встречн"},
		{"встречный", "встречн"},
		{"встречным", "встречн"},
		{"встречных", "встречн"},
		{"встречу", "встреч"},
		{"встряхивает", "встряхива"},
		{"встряхну", "встряхн"},
		{"встряхнул", "встряхнул"},
		{"встряхнулся", "встряхнул"},
		{"встряхнуть", "встряхнут"},
		{"встряхнуться", "встряхнут"},
		{"вступает", "вступа"},
		{"вступала", "вступа"},
		{"вступаю", "вступа"},
		{"вступают", "вступа"},
		{"вступая", "вступ"},
		{"вступив", "вступ"},
		{"вступившая", "вступ"},
		{"вступивши", "вступ"},
		{"вступил", "вступ"},
		{"вступила", "вступ"},
		{"вступилась", "вступ"},
		{"вступили", "вступ"},
		{"вступился", "вступ"},
		{"вступить", "вступ"},
		{"вступление", "вступлен"},
		{"вступления", "вступлен"},
		{"всунув", "всунув"},
		{"всунул", "всунул"},
		{"всунуть", "всунут"},
		{"всхлипнут", "всхлипнут"},
		{"всхлипывает", "всхлипыва"},
		{"всхлипывал", "всхлипыва"},
		{"всхлипывала", "всхлипыва"},
		{"всхлипывать", "всхлипыва"},
		{"всхлипывая", "всхлипыв"},
		{"всходила", "всход"},
		{"всходили", "всход"},
		{"всходить", "всход"},
		{"всю", "всю"},
		{"всюду", "всюд"},
		{"вся", "вся"},
		{"всяк", "всяк"},
		{"всякая", "всяк"},
		{"всякие", "всяк"},
		{"всякий", "всяк"},
		{"всяким", "всяк"},
		{"всякими", "всяк"},
		{"всяких", "всяк"},
		{"всякого", "всяк"},
		{"всякое", "всяк"},
		{"всякой", "всяк"},
		{"всяком", "всяк"},
		{"всякому", "всяк"},
		{"всякою", "всяк"},
		{"всяку", "всяк"},
		{"всякую", "всяк"},
		{"всячески", "всяческ"},
		{"втайне", "втайн"},
		{"втащила", "втащ"},
		{"втащили", "втащ"},
		{"втеснилось", "втесн"},
		{"втираться", "втира"},
		{"втихомолку", "втихомолк"},
		{"втолковать", "втолкова"},
		{"втоптала", "втопта"},
		{"вторая", "втор"},
		{"вторгается", "вторга"},
		{"вторично", "вторичн"},
		{"вторичный", "вторичн"},
		{"вторник", "вторник"},
		{"второго", "втор"},
		{"второе", "втор"},
		{"второй", "втор"},
		{"втором", "втор"},
		{"второпях", "второп"},
		{"второстепенную", "второстепен"},
		{"вторую", "втор"},
		{"вторые", "втор"},
		{"вторым", "втор"},
		{"вторых", "втор"},
		{"втрое", "втро"},
		{"втуне", "втун"},
		{"втыкается", "втыка"},
		{"втягивает", "втягива"},
		{"втягивание", "втягиван"},
		{"втягивать", "втягива"},
		{"втягивающее", "втягива"},
		{"втянет", "втянет"},
		{"втянется", "втянет"},
		{"втянув", "втянув"},
		{"втянул", "втянул"},
		{"вуали", "вуа"},
		{"вуалью", "вуал"},
		{"вулкана", "вулка"},
		{"вулканических", "вулканическ"},
		{"вулканической", "вулканическ"},
		{"вход", "вход"},
		{"входа", "вход"},
		{"входе", "вход"},
		{"входи", "вход"},
		{"входившего", "вход"},
		{"входившим", "вход"},
		{"входил", "вход"},
		{"входила", "вход"},
		{"входили", "вход"},
		{"входило", "вход"},
		{"входит", "вход"},
		{"входите", "вход"},
		{"входить", "вход"},
		{"входя", "вход"},
		{"входят", "вход"},
		{"входящие", "входя"},
		{"входящий", "входя"},
		{"входящим", "входя"},
		{"входящих", "входя"},
		{"входящую", "входя"},
		{"вхожу", "вхож"},
		{"вцепилась", "вцеп"},
		{"вцепился", "вцеп"},
		{"вцепится", "вцеп"},
		{"вчера", "вчер"},
		{"вчерашнего", "вчерашн"},
		{"вчерашнее", "вчерашн"},
		{"вчерашней", "вчерашн"},
		{"вчерашнем", "вчерашн"},
		{"вчерашнему", "вчерашн"},
		{"вчерашние", "вчерашн"},
		{"вчерашний", "вчерашн"},
		{"вчерашним", "вчерашн"},
		{"вчерашних", "вчерашн"},
		{"вчерашнюю", "вчерашн"},
		{"вчерашняя", "вчерашн"},
		{"вчерне", "вчерн"},
		{"вчетверо", "вчетвер"},
		{"вчуже", "вчуж"},
		{"вшей", "вше"},
		{"вши", "вши"},
		{"въелась", "въел"},
		{"въелись", "въел"},
		{"въехал", "въеха"},
		{"въехали", "въеха"},
		{"въехать", "въеха"},
		{"вы", "вы"},
		{"выбегает", "выбега"},
		{"выбегали", "выбега"},
		{"выбежавшего", "выбежа"},
		{"выбежал", "выбежа"},
		{"выбежала", "выбежа"},
		{"выбежали", "выбежа"},
		{"выбежит", "выбеж"},
		{"выбежишь", "выбеж"},
		{"выберет", "выберет"},
		{"выберу", "выбер"},
		{"выберут", "выберут"},
		{"выбивался", "выбива"},
		{"выбивая", "выбив"},
		{"выбилась", "выб"},
		{"выбирает", "выбира"},
		{"выбирай", "выбира"},
		{"выбирать", "выбира"},
		{"выбираться", "выбира"},
		{"выбираю", "выбира"},
		{"выбирают", "выбира"},
		{"выбирая", "выбир"},
		{"выбить", "выб"},
		{"выборам", "выбор"},
		{"выборгской", "выборгск"},
		{"выборгскую", "выборгск"},
		{"выборе", "выбор"},
		{"выборы", "выбор"},
		{"выбрав", "выбра"},
		{"выбрал", "выбра"},
		{"выбрала", "выбра"},
		{"выбрали", "выбра"},
		{"выбрался", "выбра"},
		{"выбран", "выбра"},
		{"выбрана", "выбра"},
		{"выбрасывает", "выбрасыва"},
		{"выбрасывать", "выбрасыва"},
		{"выбрать", "выбра"},
		{"выбраться", "выбра"},
		{"выбрился", "выбр"},
		{"выбрит", "выбр"},
		{"выбрито", "выбрит"},
		{"выбритый", "выбрит"},
		{"выбритыми", "выбрит"},
		{"выбросил", "выброс"},
		{"выбросить", "выброс"},
		{"выбрось", "выбр"},
		{"выбросьте", "выбросьт"},
		{"выброшенные", "выброшен"},
		{"выведает", "выведа"},
		{"выведать", "выведа"},
		{"выведения", "выведен"},
		{"выведет", "выведет"},
		{"выведешь", "выведеш"},
		{"выведи", "вывед"},
		{"выведу", "вывед"},
		{"выведут", "выведут"},
		{"вывезенному", "вывезен"},
		{"вывезенным", "вывезен"},
		{"вывезти", "вывезт"},
		{"вывел", "вывел"},
		{"вывела", "вывел"},
		{"вывели", "вывел"},
		{"вывело", "вывел"},
		{"вывернул", "вывернул"},
		{"вывернула", "вывернул"},
		{"вывернулся", "вывернул"},
		{"вывернутыми", "вывернут"},
		{"вывернуться", "вывернут"},
		{"выверт", "выверт"},
		{"вывеска", "вывеск"},
		{"вывескам", "вывеск"},
		{"вывески", "вывеск"},
		{"вывести", "вывест"},
		{"выветриваемся", "выветрива"},
		{"выветрившегося", "выветр"},
		{"вывешено", "вывеш"},
		{"вывод", "вывод"},
		{"вывода", "вывод"},
		{"выводил", "вывод"},
		{"выводили", "вывод"},
		{"выводило", "вывод"},
		{"выводить", "вывод"},
		{"выводов", "вывод"},
		{"выводом", "вывод"},
		{"выводят", "вывод"},
		{"вывожу", "вывож"},
		{"вывороти", "выворот"},
		{"выворотив", "выворот"},
		{"выворотил", "выворот"},
		{"выгадаешь", "выгада"},
		{"выгадать", "выгада"},
		{"выгладит", "выглад"},
		{"выглядишь", "выгляд"},
		{"выглядывавшую", "выглядыва"},
		{"выглядывает", "выглядыва"},
		{"выглядывал", "выглядыва"},
		{"выглядывала", "выглядыва"},
		{"выглядывали", "выглядыва"},
		{"выглядывало", "выглядыва"},
		{"выглядывая", "выглядыв"},
		{"выглянет", "выглянет"},
		{"выглянув", "выглянув"},
		{"выглянула", "выглянул"},
		{"выглянуло", "выглянул"},
		{"выгнал", "выгна"},
		{"выгнали", "выгна"},
		{"выгнанная", "выгна"},
		{"выгнать", "выгна"},
		{"выгнутою", "выгнут"},
		{"выговаривает", "выговарива"},
		{"выговаривал", "выговарива"},
		{"выговаривала", "выговарива"},
		{"выговаривало", "выговарива"},
		{"выговаривалось", "выговарива"},
		{"выговаривать", "выговарива"},
		{"выговариваться", "выговарива"},
		{"выговаривая", "выговарив"},
		{"выговора", "выговор"},
		{"выговорив", "выговор"},
		{"выговорил", "выговор"},
		{"выговорила", "выговор"},
		{"выговорилось", "выговор"},
		{"выговорить", "выговор"},
		{"выговорят", "выговор"},
		{"выгода", "выгод"},
		{"выгодами", "выгод"},
		{"выгодах", "выгод"},
		{"выгоднее", "выгодн"},
		{"выгодно", "выгодн"},
		{"выгодное", "выгодн"},
		{"выгодной", "выгодн"},
		{"выгодном", "выгодн"},
		{"выгодную", "выгодн"},
		{"выгоду", "выгод"},
		{"выгоды", "выгод"},
		{"выгонишь", "выгон"},
		{"выгоню", "выгон"},
		{"выгоняют", "выгоня"},
		{"выгорит", "выгор"},
		{"выгребла", "выгребл"},
		{"выдавай", "выдава"},
		{"выдавалась", "выдава"},
		{"выдавали", "выдава"},
		{"выдавались", "выдава"},
		{"выдавать", "выдава"},
		{"выдавая", "выдав"},
		{"выдавилась", "выдав"},
		{"выдавливалась", "выдавлива"},
		{"выдадите", "выдад"},
		{"выдает", "выда"},
		{"выдаете", "выда"},
		{"выдал", "выда"},
		{"выдала", "выда"},
		{"выдалась", "выда"},
		{"выдали", "выда"},
		{"выданное", "выда"},
		{"выданы", "выда"},
		{"выдаст", "выдаст"},
		{"выдастся", "выдаст"},
		{"выдать", "выда"},
		{"выдают", "выда"},
		{"выдаются", "выда"},
		{"выдающеюся", "выда"},
		{"выдвинул", "выдвинул"},
		{"выделанно", "выдела"},
		{"выделанным", "выдела"},
		{"выделка", "выделк"},
		{"выделывать", "выделыва"},
		{"выделывая", "выделыв"},
		{"выдергивает", "выдергива"},
		{"выдержав", "выдержа"},
		{"выдержал", "выдержа"},
		{"выдержала", "выдержа"},
		{"выдержали", "выдержа"},
		{"выдержать", "выдержа"},
		{"выдерживает", "выдержива"},
		{"выдерживал", "выдержива"},
		{"выдерживаю", "выдержива"},
		{"выдерживают", "выдержива"},
		{"выдерживая", "выдержив"},
		{"выдержит", "выдерж"},
		{"выдержите", "выдерж"},
		{"выдержишь", "выдерж"},
		{"выдержу", "выдерж"},
		{"выдернул", "выдернул"},
		{"выдернула", "выдернул"},
		{"выдохся", "выдох"},
		{"выдрал", "выдра"},
		{"выдрессированная", "выдрессирова"},
		{"выдумав", "выдума"},
		{"выдумаем", "выдума"},
		{"выдумает", "выдума"},
		{"выдумаешь", "выдума"},
		{"выдумай", "выдума"},
		{"выдумал", "выдума"},
		{"выдумала", "выдума"},
		{"выдумалась", "выдума"},
		{"выдумали", "выдума"},
		{"выдуманная", "выдума"},
		{"выдумать", "выдума"},
		{"выдумка", "выдумк"},
		{"выдумки", "выдумк"},
		{"выдумку", "выдумк"},
		{"выдумок", "выдумок"},
		{"выдумывал", "выдумыва"},
		{"выдумывала", "выдумыва"},
		{"выдумывать", "выдумыва"},
		{"выдумываю", "выдумыва"},
		{"выдумывают", "выдумыва"},
		{"выдь", "выд"},
		{"выдьте", "выдьт"},
		{"выедем", "выед"},
		{"выеду", "выед"},
		{"выезда", "выезд"},
		{"выездные", "выездн"},
		{"выездных", "выездн"},
		{"выезжаем", "выезжа"},
		{"выезжает", "выезжа"},
		{"выезжаете", "выезжа"},
		{"выезжал", "выезжа"},
		{"выезжала", "выезжа"},
		{"выезжать", "выезжа"},
		{"выезжают", "выезжа"},
		{"выехал", "выеха"},
		{"выехала", "выеха"},
		{"выехали", "выеха"},
		{"выехать", "выеха"},
		{"выждал", "выжда"},
		{"выждать", "выжда"},
		{"выжечь", "выжеч"},
		{"выжжены", "выжж"},
		{"выжига", "выжиг"},
		{"выжидал", "выжида"},
		{"выжидая", "выжид"},
		{"выжил", "выж"},
		{"выжили", "выж"},
		{"выжимали", "выжима"},
		{"выжмет", "выжмет"},
		{"вызвав", "вызва"},
		{"вызвал", "вызва"},
		{"вызвала", "вызва"},
		{"вызвали", "вызва"},
		{"вызвало", "вызва"},
		{"вызвался", "вызва"},
		{"вызванные", "вызва"},
		{"вызвать", "вызва"},
		{"выздоравливать", "выздоравлива"},
		{"выздоравливаю", "выздоравлива"},
		{"выздоравливая", "выздоравлив"},
		{"выздоровевший", "выздоровевш"},
		{"выздоровеешь", "выздоровееш"},
		{"выздоровел", "выздоровел"},
		{"выздороветь", "выздоровет"},
		{"выздоровление", "выздоровлен"},
		{"выздоровлению", "выздоровлен"},
		{"выздоровления", "выздоровлен"},
		{"выздоровлю", "выздоровл"},
		{"вызов", "выз"},
		{"вызова", "вызов"},
		{"вызовет", "вызовет"},
		{"вызывает", "вызыва"},
		{"вызывала", "вызыва"},
		{"вызывали", "вызыва"},
		{"вызывают", "вызыва"},
		{"вызывающее", "вызыва"},
		{"вызывающею", "вызыва"},
		{"вызывающий", "вызыва"},
		{"вызывающим", "вызыва"},
		{"вызывая", "вызыв"},
		{"выиграет", "выигра"},
		{"выиграл", "выигра"},
		{"выиграна", "выигра"},
		{"выиграть", "выигра"},
		{"вый", "вы"},
		{"выйдем", "выйд"},
		{"выйдемте", "выйдемт"},
		{"выйдет", "выйдет"},
		{"выйдешь", "выйдеш"},
		{"выйди", "выйд"},
		{"выйду", "выйд"},
		{"выйдут", "выйдут"},
		{"выйдя", "выйд"},
		{"выйти", "выйт"},
		{"выказал", "выказа"},
		{"выказана", "выказа"},
		{"выказывает", "выказыва"},
		{"выказывал", "выказыва"},
		{"выказывались", "выказыва"},
		{"выказывалось", "выказыва"},
		{"выказывать", "выказыва"},
		{"выкатила", "выкат"},
		{"выкатились", "выкат"},
		{"выкладку", "выкладк"},
		{"выкладывал", "выкладыва"},
		{"выковывал", "выковыва"},
		{"выколачивать", "выколачива"},
		{"выколоти", "выколот"},
		{"выколочу", "выколоч"},
		{"выкопал", "выкопа"},
		{"выкопать", "выкопа"},
		{"выкрасили", "выкрас"},
		{"выкрикивал", "выкрикива"},
		{"выкрикивала", "выкрикива"},
		{"выкричишься", "выкрич"},
		{"выкупала", "выкупа"},
		{"выкупила", "выкуп"},
		{"выкупить", "выкуп"},
		{"выкупленное", "выкуплен"},
		{"выкупленные", "выкуплен"},
		{"выкуплю", "выкупл"},
		{"выкуренная", "выкурен"},
		{"выкуривает", "выкурива"},
		{"выкурить", "выкур"},
		{"выл", "выл"},
		{"выла", "выл"},
		{"вылазки", "вылазк"},
		{"вылакать", "вылака"},
		{"вылез", "вылез"},
		{"вылезали", "вылеза"},
		{"вылезая", "вылез"},
		{"вылепленными", "вылеплен"},
		{"вылериановые", "вылерианов"},
		{"вылетали", "вылета"},
		{"вылетел", "вылетел"},
		{"вылетели", "вылетел"},
		{"вылетело", "вылетел"},
		{"вылетят", "вылет"},
		{"вылечитесь", "вылеч"},
		{"вылечить", "вылеч"},
		{"вылечу", "вылеч"},
		{"вылил", "выл"},
		{"вылитую", "вылит"},
		{"вылить", "выл"},
		{"выложи", "вылож"},
		{"выложил", "вылож"},
		{"выложила", "вылож"},
		{"выложит", "вылож"},
		{"выложить", "вылож"},
		{"вылупила", "вылуп"},
		{"вылью", "выл"},
		{"вымазав", "вымаза"},
		{"выманивать", "выманива"},
		{"выманила", "выман"},
		{"вымела", "вымел"},
		{"вымерло", "вымерл"},
		{"выметают", "вымета"},
		{"вымети", "вымет"},
		{"вымолвила", "вымолв"},
		{"вымолвить", "вымолв"},
		{"вымочит", "вымоч"},
		{"вымоют", "вымоют"},
		{"вымпелами", "вымпел"},
		{"вымыв", "вым"},
		{"вымыл", "вым"},
		{"вымыла", "вым"},
		{"вымыли", "вым"},
		{"вымылся", "вым"},
		{"вымысла", "вымысл"},
		{"вымыслом", "вымысл"},
		{"вымыты", "вымыт"},
		{"вымытые", "вымыт"},
		{"вымытый", "вымыт"},
		{"вымытыми", "вымыт"},
		{"вымыть", "вым"},
		{"вымыться", "вым"},
		{"вынес", "вынес"},
		{"вынесем", "вынес"},
		{"вынесенного", "вынесен"},
		{"вынесет", "вынесет"},
		{"вынеси", "вынес"},
		{"вынесла", "вынесл"},
		{"вынесли", "вынесл"},
		{"вынести", "вынест"},
		{"вынесу", "вынес"},
		{"вынет", "вынет"},
		{"вынимает", "вынима"},
		{"вынимала", "вынима"},
		{"вынимать", "вынима"},
		{"вынимая", "выним"},
		{"выносившие", "вынос"},
		{"выносил", "вынос"},
		{"выносила", "вынос"},
		{"выносили", "вынос"},
		{"выносились", "вынос"},
		{"выносился", "вынос"},
		{"выносимая", "выносим"},
		{"выносит", "вынос"},
		{"выносите", "вынос"},
		{"выносить", "вынос"},
		{"выношу", "вынош"},
		{"вынув", "вынув"},
		{"вынудила", "вынуд"},
		{"вынужденное", "вынужден"},
		{"вынул", "вынул"},
		{"вынула", "вынул"},
		{"вынули", "вынул"},
		{"вынулось", "вынул"},
		{"вынуто", "вынут"},
		{"вынутым", "вынут"},
		{"вынуть", "вынут"},
		{"выпадает", "выпада"},
		{"выпадал", "выпада"},
		{"выпадала", "выпада"},
		{"выпадало", "выпада"},
		{"выпадении", "выпаден"},
		{"выпадет", "выпадет"},
		{"выпал", "выпа"},
		{"выпала", "выпа"},
		{"выпали", "выпа"},
		{"выпало", "выпа"},
		{"выпачкаешь", "выпачка"},
		{"выпачкался", "выпачка"},
		{"выпачкан", "выпачка"},
		{"выпачканного", "выпачка"},
		{"выпевала", "выпева"},
		{"выпевать", "выпева"},
		{"выпей", "вып"},
		{"выпейте", "вып"},
		{"выпивает", "выпива"},
		{"выпивается", "выпива"},
		{"выпивалось", "выпива"},
		{"выпивательными", "выпивательн"},
		{"выпивку", "выпивк"},
		{"выпивши", "вып"},
		{"выпил", "вып"},
		{"выпила", "вып"},
		{"выпили", "вып"},
		{"выпиливает", "выпилива"},
		{"выписали", "выписа"},
		{"выписались", "выписа"},
		{"выписать", "выписа"},
		{"выписки", "выписк"},
		{"выписывал", "выписыва"},
		{"выписывать", "выписыва"},
		{"выписываю", "выписыва"},
		{"выпит", "вып"},
		{"выпита", "выпит"},
		{"выпито", "выпит"},
		{"выпитого", "выпит"},
		{"выпитое", "выпит"},
		{"выпитою", "выпит"},
		{"выпитым", "выпит"},
		{"выпить", "вып"},
		{"выплакав", "выплака"},
		{"выплакала", "выплака"},
		{"выплакаться", "выплака"},
		{"выплакивает", "выплакива"},
		{"выплатил", "выплат"},
		{"выплатится", "выплат"},
		{"выплачу", "выплач"},
		{"выплеснет", "выплеснет"},
		{"выплыли", "выпл"},
		{"выполнит", "выполн"},
		{"выправился", "выправ"},
		{"выпрашивать", "выпрашива"},
		{"выпроводил", "выпровод"},
		{"выпросила", "выпрос"},
		{"выпросили", "выпрос"},
		{"выпросит", "выпрос"},
		{"выпросить", "выпрос"},
		{"выпрошу", "выпрош"},
		{"выпрыгнули", "выпрыгнул"},
		{"выпрыгнуть", "выпрыгнут"},
		{"выпрямилась", "выпрям"},
		{"выпрямились", "выпрям"},
		{"выпрямлялась", "выпрямля"},
		{"выпрямляя", "выпрямл"},
		{"выпрямляясь", "выпрямл"},
		{"выпукло", "выпукл"},
		{"выпуклостию", "выпукл"},
		{"выпуклою", "выпукл"},
		{"выпуклую", "выпукл"},
		{"выпуклые", "выпукл"},
		{"выпускает", "выпуска"},
		{"выпускать", "выпуска"},
		{"выпуская", "выпуск"},
		{"выпуске", "выпуск"},
		{"выпустив", "выпуст"},
		{"выпустил", "выпуст"},
		{"выпустила", "выпуст"},
		{"выпустили", "выпуст"},
		{"выпустит", "выпуст"},
		{"выпустить", "выпуст"},
		{"выпустят", "выпуст"},
		{"выпутаться", "выпута"},
		{"выпутывался", "выпутыва"},
		{"выпуча", "выпуч"},
		{"выпучив", "выпуч"},
		{"выпучил", "выпуч"},
		{"выпущен", "выпущ"},
		{"выпущенные", "выпущен"},
		{"выпущены", "выпущ"},
		{"выпущу", "выпущ"},
		{"выпытает", "выпыта"},
		{"выпытала", "выпыта"},
		{"выпытываешь", "выпытыва"},
		{"выпытывала", "выпытыва"},
		{"выпытывая", "выпытыв"},
		{"выпьем", "вып"},
		{"выпьет", "выпьет"},
		{"выпьешь", "выпьеш"},
		{"выпью", "вып"},
		{"выпьют", "выпьют"},
		{"выработал", "выработа"},
		{"выработалась", "выработа"},
		{"выработался", "выработа"},
		{"выработанности", "выработан"},
		{"выработанные", "выработа"},
		{"выработать", "выработа"},
		{"выработаться", "выработа"},
		{"выработывается", "выработыва"},
		{"выработывались", "выработыва"},
		{"выработывалось", "выработыва"},
		{"выражавшими", "выража"},
		{"выражаете", "выража"},
		{"выражаетесь", "выража"},
		{"выражается", "выража"},
		{"выражал", "выража"},
		{"выражала", "выража"},
		{"выражалась", "выража"},
		{"выражали", "выража"},
		{"выражало", "выража"},
		{"выражалось", "выража"},
		{"выражался", "выража"},
		{"выражать", "выража"},
		{"выражаться", "выража"},
		{"выражая", "выраж"},
		{"выражение", "выражен"},
		{"выражением", "выражен"},
		{"выражений", "выражен"},
		{"выражению", "выражен"},
		{"выражения", "выражен"},
		{"выражениях", "выражен"},
		{"выражено", "выраж"},
		{"выраженье", "выражен"},
		{"выраженьем", "выражен"},
		{"выраженьице", "выраженьиц"},
		{"выразил", "выраз"},
		{"выразила", "выраз"},
		{"выразилась", "выраз"},
		{"выразились", "выраз"},
		{"выразило", "выраз"},
		{"выразилось", "выраз"},
		{"выразился", "выраз"},
		{"выразительная", "выразительн"},
		{"выразительно", "выразительн"},
		{"выразительное", "выразительн"},
		{"выразительные", "выразительн"},
		{"выразится", "выраз"},
		{"выразить", "выраз"},
		{"выразиться", "выраз"},
		{"вырастало", "выраста"},
		{"вырастают", "выраста"},
		{"вырастет", "вырастет"},
		{"вырасти", "выраст"},
		{"вырвавшаяся", "вырва"},
		{"вырвавшемся", "вырва"},
		{"вырвавшею", "вырва"},
		{"вырвавшихся", "вырва"},
		{"вырвал", "вырва"},
		{"вырвала", "вырва"},
		{"вырвалась", "вырва"},
		{"вырвали", "вырва"},
		{"вырвались", "вырва"},
		{"вырвалось", "вырва"},
		{"вырвался", "вырва"},
		{"вырванного", "вырва"},
		{"вырвать", "вырва"},
		{"вырваться", "вырва"},
		{"вырвется", "вырвет"},
		{"вырву", "вырв"},
		{"вырезай", "выреза"},
		{"вырезана", "выреза"},
		{"вырезать", "выреза"},
		{"вырождение", "вырожден"},
		{"вырождением", "вырожден"},
		{"вырождения", "вырожден"},
		{"выронил", "вырон"},
		{"выронила", "вырон"},
		{"вырос", "вырос"},
		{"выросла", "выросл"},
		{"выросли", "выросл"},
		{"выросло", "выросл"},
		{"выросшей", "выросш"},
		{"вырубить", "выруб"},
		{"выругает", "выруга"},
		{"выругать", "выруга"},
		{"выручает", "выруча"},
		{"выручена", "выруч"},
		{"вырученные", "выручен"},
		{"выручи", "выруч"},
		{"выручил", "выруч"},
		{"выручим", "выруч"},
		{"выручит", "выруч"},
		{"выручки", "выручк"},
		{"вырывавшиеся", "вырыва"},
		{"вырывается", "вырыва"},
		{"вырывалась", "вырыва"},
		{"вырывали", "вырыва"},
		{"вырывались", "вырыва"},
		{"вырывалось", "вырыва"},
		{"вырываются", "вырыва"},
		{"вырывая", "вырыв"},
		{"высасывал", "высасыва"},
		{"высасывала", "высасыва"},
		{"высвободив", "высвобод"},
		{"высвободил", "высвобод"},
		{"высвободился", "высвобод"},
		{"высвобождаете", "высвобожда"},
		{"выселить", "высел"},
		{"выселки", "выселк"},
		{"выселков", "выселк"},
		{"высечет", "высечет"},
		{"высидел", "высидел"},
		{"выскабливаешь", "выскаблива"},
		{"выскажет", "выскажет"},
		{"выскажется", "выскажет"},
		{"выскажу", "выскаж"},
		{"выскажусь", "выскаж"},
		{"высказала", "высказа"},
		{"высказалась", "высказа"},
		{"высказались", "высказа"},
		{"высказался", "высказа"},
		{"высказанных", "высказа"},
		{"высказать", "высказа"},
		{"высказаться", "высказа"},
		{"высказываете", "высказыва"},
		{"высказывается", "высказыва"},
		{"высказывала", "высказыва"},
		{"высказывались", "высказыва"},
		{"высказывало", "высказыва"},
		{"высказывалось", "высказыва"},
		{"высказывать", "высказыва"},
		{"высказываться", "высказыва"},
		{"высказываю", "высказыва"},
		{"высказываясь", "высказыв"},
		{"выскользало", "выскольза"},
		{"выскользнули", "выскользнул"},
		{"выскочившего", "выскоч"},
		{"выскочил", "выскоч"},
		{"выскочила", "выскоч"},
		{"выскочит", "выскоч"},
		{"выскочить", "выскоч"},
		{"выскочке", "выскочк"},
		{"выскочки", "выскочк"},
		{"выскребывались", "выскребыва"},
		{"выслали", "высла"},
		{"выслать", "высла"},
		{"выслежу", "выслеж"},
		{"выслужил", "выслуж"},
		{"выслужиться", "выслуж"},
		{"выслушав", "выслуша"},
		{"выслушает", "выслуша"},
		{"выслушаете", "выслуша"},
		{"выслушайте", "выслуша"},
		{"выслушал", "выслуша"},
		{"выслушала", "выслуша"},
		{"выслушать", "выслуша"},
		{"выслушивал", "выслушива"},
		{"выслушивали", "выслушива"},
		{"высматривал", "высматрива"},
		{"высматривать", "высматрива"},
		{"высматривая", "высматрив"},
		{"высморкается", "высморка"},
		{"высморкался", "высморка"},
		{"высмотреть", "высмотрет"},
		{"высовывалась", "высовыва"},
		{"высок", "высок"},
		{"высокая", "высок"},
		{"высокенькая", "высокеньк"},
		{"высокенькие", "высокеньк"},
		{"высоки", "высок"},
		{"высокие", "высок"},
		{"высокий", "высок"},
		{"высоким", "высок"},
		{"высокими", "высок"},
		{"высоких", "высок"},
		{"высоко", "высок"},
		{"высокоблагородие", "высокоблагород"},
		{"высокого", "высок"},
		{"высокое", "высок"},
		{"высокой", "высок"},
		{"высоком", "высок"},
		{"высокомерием", "высокомер"},
		{"высокомерия", "высокомер"},
		{"высокомерно", "высокомерн"},
		{"высокомерного", "высокомерн"},
		{"высокомерные", "высокомерн"},
		{"высокомерный", "высокомерн"},
		{"высокомерным", "высокомерн"},
		{"высокому", "высок"},
		{"высокопарные", "высокопарн"},
		{"высокородие", "высокород"},
		{"высокоуважаемая", "высокоуважа"},
		{"высокую", "высок"},
		{"высоте", "высот"},
		{"высоту", "высот"},
		{"высоты", "высот"},
		{"высох", "высох"},
		{"высохла", "высохл"},
		{"высохли", "высохл"},
		{"высохло", "высохл"},
		{"высохшем", "высохш"},
		{"высохшею", "высохш"},
		{"высочайшей", "высочайш"},
		{"высочайшим", "высочайш"},
		{"выспался", "выспа"},
		{"выспится", "высп"},
		{"высплюсь", "выспл"},
		{"выспросил", "выспрос"},
		{"выставив", "выстав"},
		{"выставил", "выстав"},
		{"выставила", "выстав"},
		{"выставилась", "выстав"},
		{"выставили", "выстав"},
		{"выставит", "выстав"},
		{"выставлена", "выставл"},
		{"выставлены", "выставл"},
		{"выставлю", "выставл"},
		{"выставляется", "выставля"},
		{"выставлял", "выставля"},
		{"выставлять", "выставля"},
		{"выстоять", "выстоя"},
		{"выстрадав", "выстрада"},
		{"выстрадал", "выстрада"},
		{"выстрел", "выстрел"},
		{"выстрелив", "выстрел"},
		{"выстрелила", "выстрел"},
		{"выстрелить", "выстрел"},
		{"выстрелишь", "выстрел"},
		{"выстрелю", "выстрел"},
		{"выстриженными", "выстрижен"},
		{"выстроен", "выстро"},
		{"выстроить", "выстро"},
		{"выступал", "выступа"},
		{"выступать", "выступа"},
		{"выступают", "выступа"},
		{"выступившей", "выступ"},
		{"выступившие", "выступ"},
		{"выступившими", "выступ"},
		{"выступил", "выступ"},
		{"выступила", "выступ"},
		{"выступили", "выступ"},
		{"выступить", "выступ"},
		{"выступление", "выступлен"},
		{"высунув", "высунув"},
		{"высунулась", "высунул"},
		{"высунуть", "высунут"},
		{"высушила", "высуш"},
		{"высушит", "высуш"},
		{"высчитал", "высчита"},
		{"высшего", "высш"},
		{"высшее", "высш"},
		{"высшей", "высш"},
		{"высшему", "высш"},
		{"высший", "высш"},
		{"высших", "высш"},
		{"высшую", "высш"},
		{"высылают", "высыла"},
		{"высыпает", "высыпа"},
		{"высыпал", "высыпа"},
		{"высыпали", "высыпа"},
		{"высь", "вы"},
		{"вытараща", "вытаращ"},
		{"вытаращены", "вытаращ"},
		{"вытаращив", "вытаращ"},
		{"вытаращил", "вытаращ"},
		{"вытаращила", "вытаращ"},
		{"вытаскивает", "вытаскива"},
		{"вытаскивала", "вытаскива"},
		{"вытаскивать", "вытаскива"},
		{"вытаскивая", "вытаскив"},
		{"вытащена", "вытащ"},
		{"вытащил", "вытащ"},
		{"вытащила", "вытащ"},
		{"вытащим", "выта"},
		{"вытащите", "вытащ"},
		{"вытащить", "вытащ"},
		{"вытащу", "вытащ"},
		{"вытер", "вытер"},
		{"вытерла", "вытерл"},
		{"вытерпел", "вытерпел"},
		{"вытерпела", "вытерпел"},
		{"вытерпели", "вытерпел"},
		{"вытерпеть", "вытерпет"},
		{"вытерпит", "вытерп"},
		{"вытерпливающего", "вытерплива"},
		{"вытерплю", "вытерпл"},
		{"вытирает", "вытира"},
		{"вытирать", "вытира"},
		{"вытирая", "вытир"},
		{"вытолкай", "вытолка"},
		{"выторговать", "выторгова"},
		{"выточен", "выточ"},
		{"выточена", "выточ"},
		{"выточенными", "выточен"},
		{"выточилась", "выточ"},
		{"выть", "выт"},
		{"вытягиваться", "вытягива"},
		{"вытягивая", "вытягив"},
		{"вытягушкин", "вытягушкин"},
		{"вытянет", "вытянет"},
		{"вытянув", "вытянув"},
		{"вытянул", "вытянул"},
		{"вытянулись", "вытянул"},
		{"выудить", "выуд"},
		{"выученного", "выучен"},
		{"выучи", "выуч"},
		{"выучивал", "выучива"},
		{"выучил", "выуч"},
		{"выучила", "выуч"},
		{"выучилась", "выуч"},
		{"выучился", "выуч"},
		{"выучиться", "выуч"},
		{"выучку", "выучк"},
		{"выучу", "выуч"},
		{"выхватив", "выхват"},
		{"выхватил", "выхват"},
		{"выхватила", "выхват"},
		{"выхлопотанной", "выхлопота"},
		{"выхлопотать", "выхлопота"},
		{"выхлопочет", "выхлопочет"},
		{"выход", "выход"},
		{"выхода", "выход"},
		{"выходе", "выход"},
		{"выходи", "выход"},
		{"выходившая", "выход"},
		{"выходившем", "выход"},
		{"выходившему", "выход"},
		{"выходивших", "выход"},
		{"выходил", "выход"},
		{"выходила", "выход"},
		{"выходили", "выход"},
		{"выходило", "выход"},
		{"выходим", "выход"},
		{"выходит", "выход"},
		{"выходите", "выход"},
		{"выходить", "выход"},
		{"выходишь", "выход"},
		{"выходками", "выходк"},
		{"выходке", "выходк"},
		{"выходкой", "выходк"},
		{"выходку", "выходк"},
		{"выходной", "выходн"},
		{"выходов", "выход"},
		{"выходом", "выход"},
		{"выходу", "выход"},
		{"выходя", "выход"},
		{"выходят", "выход"},
		{"выходящего", "выходя"},
		{"выходящее", "выходя"},
		{"выходящие", "выходя"},
		{"выхожу", "выхож"},
		{"выцвело", "выцвел"},
		{"выцветшим", "выцветш"},
		{"выцветших", "выцветш"},
		{"вычеркнуть", "вычеркнут"},
		{"вычертил", "вычерт"},
		{"вычета", "вычет"},
		{"вычетом", "вычет"},
		{"вычинила", "вычин"},
		{"вычислять", "вычисля"},
		{"вычистить", "вычист"},
		{"вычитал", "вычита"},
		{"вычитание", "вычитан"},
		{"вычитанные", "вычита"},
		{"вычитают", "вычита"},
		{"вычитывая", "вычитыв"},
		{"вычищенную", "вычищен"},
		{"вычищенный", "вычищен"},
		{"вычищенных", "вычищен"},
		{"вычищены", "вычищ"},
		{"вышвырнул", "вышвырнул"},
		{"выше", "выш"},
		{"вышедшего", "вышедш"},
		{"вышедши", "вышедш"},
		{"вышедший", "вышедш"},
		{"вышедшую", "вышедш"},
		{"вышел", "вышел"},
		{"вышесказанными", "вышесказа"},
		{"вышеупомянутою", "вышеупомянут"},
		{"вышиб", "вышиб"},
		{"вышивает", "вышива"},
		{"вышиваньем", "вышиван"},
		{"вышивая", "вышив"},
		{"вышине", "вышин"},
		{"вышитыми", "вышит"},
		{"вышла", "вышл"},
		{"вышлем", "вышл"},
		{"вышли", "вышл"},
		{"вышло", "вышл"},
		{"вышлю", "вышл"},
		{"выщипала", "выщипа"},
		{"выяснения", "выяснен"},
		{"выяснилась", "выясн"},
		{"выяснилось", "выясн"},
		{"выясняться", "выясня"},
		{"вьется", "вьет"},
		{"вьюг", "вьюг"},
		{"вьюга", "вьюг"},
		{"вьюгам", "вьюг"},
		{"вьюге", "вьюг"},
		{"вьюги", "вьюг"},
		{"вьюгой", "вьюг"},
		{"вьюн", "вьюн"},
		{"вьются", "вьют"},
		{"вьющимися", "вьющ"},
		{"вьющихся", "вьющ"},
		{"вяжет", "вяжет"},
		{"вяжутся", "вяжут"},
		{"вяз", "вяз"},
		{"вязанки", "вязанк"},
		{"вязанная", "вяза"},
		{"вязать", "вяза"},
		{"вяземского", "вяземск"},
		{"вязка", "вязк"},
		{"вязниковых", "вязников"},
		{"вязнут", "вязнут"},
		{"вязь", "вяз"},
		{"вялая", "вял"},
		{"вяло", "вял"},
		{"вялого", "вял"},
		{"вялой", "вял"},
		{"вялою", "вял"},
		{"вялый", "вял"},
		{"вялым", "вял"},
		{"вялыми", "вял"},
		{"вянут", "вянут"},
		{"вячеславу", "вячеслав"},
		{"г", "г"},
		{"га", "га"},
		{"гавана", "гава"},
		{"гавани", "гаван"},
		{"гаванской", "гаванск"},
		{"гаврилович", "гаврилович"},
		{"гавриловича", "гаврилович"},
		{"гавриловичу", "гаврилович"},
		{"гаврилыч", "гаврилыч"},
		{"гаврилыча", "гаврилыч"},
		{"гаврилычу", "гаврилыч"},
		{"гаги", "гаг"},
		{"гадай", "гада"},
		{"гадал", "гада"},
		{"гадала", "гада"},
		{"гаданьи", "гадан"},
		{"гадать", "гада"},
		{"гадают", "гада"},
		{"гадая", "гад"},
		{"гаденькая", "гаденьк"},
		{"гадки", "гадк"},
		{"гадкий", "гадк"},
		{"гадко", "гадк"},
		{"гадкое", "гадк"},
		{"гадкой", "гадк"},
		{"гадкую", "гадк"},
		{"гадов", "гад"},
		{"гадок", "гадок"},
		{"гадостны", "гадостн"},
		{"гадость", "гадост"},
		{"гадят", "гад"},
		{"гаев", "га"},
		{"гаева", "гаев"},
		{"гаеве", "гаев"},
		{"гаеву", "гаев"},
		{"гаже", "гаж"},
		{"газели", "газел"},
		{"газет", "газет"},
		{"газета", "газет"},
		{"газетам", "газет"},
		{"газетах", "газет"},
		{"газете", "газет"},
		{"газетку", "газетк"},
		{"газетной", "газетн"},
		{"газетную", "газетн"},
		{"газетный", "газетн"},
		{"газетой", "газет"},
		{"газету", "газет"},
		{"газеты", "газет"},
		{"газом", "газ"},
		{"газы", "газ"},
		{"гайдамаков", "гайдамак"},
		{"галатея", "галате"},
		{"галдеж", "галдеж"},
		{"галдели", "галдел"},
		{"галерее", "галер"},
		{"галереей", "галере"},
		{"галереи", "галер"},
		{"галерею", "галер"},
		{"галерея", "галере"},
		{"галереях", "галере"},
		{"галилея", "галиле"},
		{"галицийских", "галицийск"},
		{"галка", "галк"},
		{"галла", "галл"},
		{"галлереи", "галлер"},
		{"галлы", "галл"},
		{"галлюсинацию", "галлюсинац"},
		{"галлюцинации", "галлюцинац"},
		{"галстук", "галстук"},
		{"галстука", "галстук"},
		{"галстуке", "галстук"},
		{"галстучек", "галстучек"},
		{"галчат", "галчат"},
		{"гам", "гам"},
		{"гамбринусе", "гамбринус"},
		{"гамлет", "гамлет"},
		{"гамлета", "гамлет"},
		{"гамлетовского", "гамлетовск"},
		{"гамом", "гам"},
		{"гаму", "гам"},
		{"ганц", "ганц"},
		{"гарантировали", "гарантирова"},
		{"гари", "гар"},
		{"гармонией", "гармон"},
		{"гармонии", "гармон"},
		{"гармоника", "гармоник"},
		{"гармонике", "гармоник"},
		{"гармоники", "гармоник"},
		{"гармоникой", "гармоник"},
		{"гармонировавшие", "гармонирова"},
		{"гармонировал", "гармонирова"},
		{"гармонировало", "гармонирова"},
		{"гармонического", "гармоническ"},
		{"гармония", "гармон"},
		{"гарнитур", "гарнитур"},
		{"гарнитуром", "гарнитур"},
		{"гарпагон", "гарпагон"},
		{"гаруса", "гарус"},
		{"гарью", "гар"},
		{"гаси", "гас"},
		{"гасил", "гас"},
		{"гасли", "гасл"},
		{"гасло", "гасл"},
		{"гаснет", "гаснет"},
		{"гасну", "гасн"},
		{"гаснул", "гаснул"},
		{"гаснут", "гаснут"},
		{"гаснуть", "гаснут"},
		{"гаснущая", "гаснущ"},
		{"гастроном", "гастрон"},
		{"гастрономической", "гастрономическ"},
		{"гастрономическом", "гастрономическ"},
		{"гатей", "гат"},
		{"гатчинских", "гатчинск"},
		{"гвалт", "гвалт"},
		{"гвоздем", "гвозд"},
		{"гвозди", "гвозд"},
		{"гвоздиками", "гвоздик"},
		{"гвоздичку", "гвоздичк"},
		{"гвоздочками", "гвоздочк"},
		{"гвоздь", "гвозд"},
		{"гвоздями", "гвозд"},
		{"гг", "гг"},
		{"где", "где"},
		{"ге", "ге"},
		{"гедрюкт", "гедрюкт"},
		{"гей", "ге"},
		{"гелиотропом", "гелиотроп"},
		{"гельд", "гельд"},
		{"геморрой", "геморр"},
		{"генерал", "генера"},
		{"генерала", "генера"},
		{"генералишка", "генералишк"},
		{"генералишку", "генералишк"},
		{"генералов", "генерал"},
		{"генералом", "генерал"},
		{"генералу", "генерал"},
		{"генералы", "генерал"},
		{"генеральские", "генеральск"},
		{"генеральский", "генеральск"},
		{"гениальной", "гениальн"},
		{"гениальности", "гениальн"},
		{"гениальные", "гениальн"},
		{"гениальный", "гениальн"},
		{"гениальным", "гениальн"},
		{"гением", "ген"},
		{"гении", "ген"},
		{"гений", "ген"},
		{"генриет", "генриет"},
		{"генуя", "гену"},
		{"географической", "географическ"},
		{"географию", "географ"},
		{"география", "географ"},
		{"геологические", "геологическ"},
		{"геометра", "геометр"},
		{"геометрия", "геометр"},
		{"георгий", "георг"},
		{"герани", "геран"},
		{"геранью", "геран"},
		{"геранями", "геран"},
		{"герасимович", "герасимович"},
		{"герасимовича", "герасимович"},
		{"герасимовичу", "герасимович"},
		{"герасимыч", "герасимыч"},
		{"герасимычу", "герасимыч"},
		{"гербовая", "гербов"},
		{"гербовой", "гербов"},
		{"гербовыми", "гербов"},
		{"гердера", "гердер"},
		{"геркулан", "геркула"},
		{"геркулесовыми", "геркулесов"},
		{"германии", "герман"},
		{"германию", "герман"},
		{"германские", "германск"},
		{"героев", "геро"},
		{"героем", "геро"},
		{"героинь", "героин"},
		{"герой", "гер"},
		{"геройски", "геройск"},
		{"геройства", "геройств"},
		{"герою", "гер"},
		{"героя", "геро"},
		{"герр", "герр"},
		{"гертруды", "гертруд"},
		{"герца", "герц"},
		{"герцом", "герц"},
		{"гершеля", "гершел"},
		{"гете", "гет"},
		{"гиацинты", "гиацинт"},
		{"гибели", "гибел"},
		{"гибель", "гибел"},
		{"гибельно", "гибельн"},
		{"гибельной", "гибельн"},
		{"гибельный", "гибельн"},
		{"гибелью", "гибел"},
		{"гибки", "гибк"},
		{"гибли", "гибл"},
		{"гибнет", "гибнет"},
		{"гибнешь", "гибнеш"},
		{"гибну", "гибн"},
		{"гибнут", "гибнут"},
		{"гибнуть", "гибнут"},
		{"гибок", "гибок"},
		{"гигантской", "гигантск"},
		{"гиганты", "гигант"},
		{"гимн", "гимн"},
		{"гимна", "гимн"},
		{"гимназии", "гимназ"},
		{"гимназист", "гимназист"},
		{"гимназистами", "гимназист"},
		{"гимназическом", "гимназическ"},
		{"гимназию", "гимназ"},
		{"гимнастика", "гимнастик"},
		{"гимнастикой", "гимнастик"},
		{"гимнастику", "гимнастик"},
		{"гимны", "гимн"},
		{"гиппиус", "гиппиус"},
		{"гирлянды", "гирлянд"},
		{"гиря", "гир"},
		{"гитара", "гитар"},
		{"гитаре", "гитар"},
		{"гитарные", "гитарн"},
		{"гитарой", "гитар"},
		{"гитару", "гитар"},
		{"глав", "глав"},
		{"глава", "глав"},
		{"главе", "глав"},
		{"главная", "главн"},
		{"главнейшая", "главн"},
		{"главнейшего", "главн"},
		{"главнейшие", "главн"},
		{"главнейшим", "главн"},
		{"главнейших", "главн"},
		{"главного", "главн"},
		{"главное", "главн"},
		{"главной", "главн"},
		{"главном", "главн"},
		{"главною", "главн"},
		{"главную", "главн"},
		{"главные", "главн"},
		{"главный", "главн"},
		{"главным", "главн"},
		{"главных", "главн"},
		{"главой", "глав"},
		{"главу", "глав"},
		{"главы", "глав"},
		{"глагол", "глагол"},
		{"глади", "глад"},
		{"гладиатор", "гладиатор"},
		{"гладил", "глад"},
		{"гладила", "глад"},
		{"гладит", "глад"},
		{"гладить", "глад"},
		{"гладкие", "гладк"},
		{"гладкий", "гладк"},
		{"гладко", "гладк"},
		{"гладковыбритым", "гладковыбрит"},
		{"гладкой", "гладк"},
		{"гладкую", "гладк"},
		{"гладь", "глад"},
		{"гладьте", "гладьт"},
		{"глаженье", "глажен"},
		{"глаз", "глаз"},
		{"глаза", "глаз"},
		{"глазам", "глаз"},
		{"глазами", "глаз"},
		{"глазах", "глаз"},
		{"глазела", "глазел"},
		{"глазенками", "глазенк"},
		{"глазеночками", "глазеночк"},
		{"глазеющих", "глазеющ"},
		{"глазея", "глазе"},
		{"глазищи", "глазищ"},
		{"глазками", "глазк"},
		{"глазки", "глазк"},
		{"глазок", "глазок"},
		{"глазом", "глаз"},
		{"глазу", "глаз"},
		{"глас", "глас"},
		{"гласила", "глас"},
		{"гласное", "гласн"},
		{"гласности", "гласност"},
		{"гласом", "глас"},
		{"гласят", "глас"},
		{"глашатаи", "глашата"},
		{"глин", "глин"},
		{"глиной", "глин"},
		{"глину", "глин"},
		{"глины", "глин"},
		{"глиняной", "глинян"},
		{"глиняный", "глинян"},
		{"глиняными", "глинян"},
		{"глиссе", "глисс"},
		{"глобус", "глобус"},
		{"гложет", "гложет"},
		{"гложут", "гложут"},
		{"глотает", "глота"},
		{"глотками", "глотк"},
		{"глотков", "глотк"},
		{"глотку", "глотк"},
		{"глотнул", "глотнул"},
		{"глоток", "глоток"},
		{"глохнут", "глохнут"},
		{"глубже", "глубж"},
		{"глуби", "глуб"},
		{"глубина", "глубин"},
		{"глубине", "глубин"},
		{"глубиной", "глубин"},
		{"глубину", "глубин"},
		{"глубины", "глубин"},
		{"глубок", "глубок"},
		{"глубока", "глубок"},
		{"глубокая", "глубок"},
		{"глубоки", "глубок"},
		{"глубокие", "глубок"},
		{"глубокий", "глубок"},
		{"глубоким", "глубок"},
		{"глубоких", "глубок"},
		{"глубоко", "глубок"},
		{"глубокого", "глубок"},
		{"глубокое", "глубок"},
		{"глубокой", "глубок"},
		{"глубоком", "глубок"},
		{"глубокомысленно", "глубокомыслен"},
		{"глубокомысленное", "глубокомыслен"},
		{"глубокую", "глубок"},
		{"глубочайшего", "глубочайш"},
		{"глубочайшее", "глубочайш"},
		{"глубочайшем", "глубочайш"},
		{"глубочайшие", "глубочайш"},
		{"глубь", "глуб"},
		{"глумитесь", "глум"},
		{"глуп", "глуп"},
		{"глупа", "глуп"},
		{"глупая", "глуп"},
		{"глупее", "глуп"},
		{"глупей", "глуп"},
		{"глупейшего", "глуп"},
		{"глупейшею", "глуп"},
		{"глупейшие", "глуп"},
		{"глупейшими", "глуп"},
		{"глупенькие", "глупеньк"},
		{"глупенькое", "глупеньк"},
		{"глупо", "глуп"},
		{"глуповат", "глуповат"},
		{"глуповатый", "глуповат"},
		{"глупого", "глуп"},
		{"глупое", "глуп"},
		{"глупой", "глуп"},
		{"глупом", "глуп"},
		{"глупому", "глуп"},
		{"глупопоспешной", "глупопоспешн"},
		{"глупостей", "глупост"},
		{"глупости", "глупост"},
		{"глупостию", "глупост"},
		{"глупость", "глупост"},
		{"глупостью", "глупост"},
		{"глупостям", "глупост"},
		{"глупостях", "глупост"},
		{"глупою", "глуп"},
		{"глупую", "глуп"},
		{"глупца", "глупц"},
		{"глупы", "глуп"},
		{"глупые", "глуп"},
		{"глупый", "глуп"},
		{"глупым", "глуп"},
		{"глупыми", "глуп"},
		{"глупых", "глуп"},
		{"глух", "глух"},
		{"глуха", "глух"},
		{"глухари", "глухар"},
		{"глухая", "глух"},
		{"глухи", "глух"},
		{"глухие", "глух"},
		{"глухим", "глух"},
		{"глухими", "глух"},
		{"глухих", "глух"},
		{"глухо", "глух"},
		{"глухого", "глух"},
		{"глухое", "глух"},
		{"глухой", "глух"},
		{"глухом", "глух"},
		{"глухонемая", "глухонем"},
		{"глухота", "глухот"},
		{"глухую", "глух"},
		{"глуше", "глуш"},
		{"глуши", "глуш"},
		{"глушит", "глуш"},
		{"глушь", "глуш"},
		{"глыба", "глыб"},
		{"глыбам", "глыб"},
		{"глядевшему", "глядевш"},
		{"глядел", "глядел"},
		{"глядела", "глядел"},
		{"глядели", "глядел"},
		{"глядеть", "глядет"},
		{"гляди", "гляд"},
		{"глядись", "гляд"},
		{"глядит", "гляд"},
		{"глядите", "гляд"},
		{"глядится", "гляд"},
		{"глядишь", "гляд"},
		{"глядь", "гляд"},
		{"глядя", "гляд"},
		{"глядясь", "гляд"},
		{"глядят", "гляд"},
		{"глядящий", "глядя"},
		{"гляжу", "гляж"},
		{"гляжусь", "гляж"},
		{"глянет", "глянет"},
		{"глянец", "глянец"},
		{"глянув", "глянув"},
		{"глянул", "глянул"},
		{"глянула", "глянул"},
		{"глянцевитой", "глянцевит"},
		{"гм", "гм"},
		{"гнал", "гнал"},
		{"гнала", "гнал"},
		{"гнать", "гнат"},
		{"гнев", "гнев"},
		{"гнева", "гнев"},
		{"гневается", "гнева"},
		{"гневайся", "гнева"},
		{"гневаться", "гнева"},
		{"гневно", "гневн"},
		{"гневной", "гневн"},
		{"гневность", "гневност"},
		{"гневный", "гневн"},
		{"гневными", "гневн"},
		{"гневом", "гнев"},
		{"гневу", "гнев"},
		{"гнедой", "гнед"},
		{"гнезд", "гнезд"},
		{"гнезда", "гнезд"},
		{"гнездится", "гнезд"},
		{"гнездо", "гнезд"},
		{"гнет", "гнет"},
		{"гнета", "гнет"},
		{"гнетет", "гнетет"},
		{"гнетете", "гнетет"},
		{"гнетом", "гнет"},
		{"гнешь", "гнеш"},
		{"гниения", "гниен"},
		{"гниет", "гниет"},
		{"гниешь", "гниеш"},
		{"гнилого", "гнил"},
		{"гнилое", "гнил"},
		{"гнилой", "гнил"},
		{"гнилом", "гнил"},
		{"гнилую", "гнил"},
		{"гнилыми", "гнил"},
		{"гниль", "гнил"},
		{"гнилью", "гнил"},
		{"гнусавой", "гнусав"},
		{"гнусно", "гнусн"},
		{"гнусное", "гнусн"},
		{"гнусном", "гнусн"},
		{"гнусному", "гнусн"},
		{"гнусность", "гнусност"},
		{"гнусные", "гнусн"},
		{"гнусный", "гнусн"},
		{"гнусным", "гнусн"},
		{"гнусными", "гнусн"},
		{"гнусных", "гнусн"},
		{"го", "го"},
		{"говеть", "говет"},
		{"говор", "говор"},
		{"говоренного", "говорен"},
		{"говорено", "говор"},
		{"говори", "говор"},
		{"говорившего", "говор"},
		{"говорившие", "говор"},
		{"говоривших", "говор"},
		{"говорившую", "говор"},
		{"говорил", "говор"},
		{"говорила", "говор"},
		{"говорили", "говор"},
		{"говорило", "говор"},
		{"говорилось", "говор"},
		{"говориль", "говорил"},
		{"говорим", "говор"},
		{"говорит", "говор"},
		{"говорите", "говор"},
		{"говорится", "говор"},
		{"говорить", "говор"},
		{"говориться", "говор"},
		{"говоришь", "говор"},
		{"говорю", "говор"},
		{"говоря", "говор"},
		{"говорят", "говор"},
		{"говорящ", "говорящ"},
		{"говорящее", "говоря"},
		{"говорящей", "говоря"},
		{"говорящие", "говоря"},
		{"говорящий", "говоря"},
		{"говорящим", "говоря"},
		{"говорящих", "говоря"},
		{"говядина", "говядин"},
		{"говядиной", "говядин"},
		{"говядину", "говядин"},
		{"говядины", "говядин"},
		{"гоголевский", "гоголевск"},
		{"гоголя", "гогол"},
		{"год", "год"},
		{"года", "год"},
		{"годам", "год"},
		{"годами", "год"},
		{"годах", "год"},
		{"годен", "год"},
		{"годика", "годик"},
		{"година", "годин"},
		{"годину", "годин"},
		{"годится", "год"},
		{"годичный", "годичн"},
		{"годишься", "год"},
		{"годной", "годн"},
		{"годные", "годн"},
		{"годов", "год"},
		{"годовой", "годов"},
		{"годовщиной", "годовщин"},
		{"годовым", "годов"},
		{"годом", "год"},
		{"годочек", "годочек"},
		{"году", "год"},
		{"годы", "год"},
		{"годятся", "год"},
		{"голая", "гол"},
		{"голгофу", "голгоф"},
		{"голи", "гол"},
		{"голиков", "голик"},
		{"голландская", "голландск"},
		{"голландских", "голландск"},
		{"голландцем", "голландц"},
		{"голо", "гол"},
		{"голов", "гол"},
		{"голова", "голов"},
		{"головами", "голов"},
		{"головах", "голов"},
		{"голове", "голов"},
		{"головка", "головк"},
		{"головке", "головк"},
		{"головку", "головк"},
		{"головлей", "головл"},
		{"головного", "головн"},
		{"головной", "головн"},
		{"головном", "головн"},
		{"головною", "головн"},
		{"головную", "головн"},
		{"головные", "головн"},
		{"головных", "головн"},
		{"головой", "голов"},
		{"головокружений", "головокружен"},
		{"головою", "голов"},
		{"голову", "голов"},
		{"головушка", "головушк"},
		{"головы", "голов"},
		{"голод", "голод"},
		{"голода", "голод"},
		{"голоден", "голод"},
		{"голодная", "голодн"},
		{"голодно", "голодн"},
		{"голодного", "голодн"},
		{"голодное", "голодн"},
		{"голодной", "голодн"},
		{"голодному", "голодн"},
		{"голодные", "голодн"},
		{"голодный", "голодн"},
		{"голодных", "голодн"},
		{"голоду", "голод"},
		{"голой", "гол"},
		{"голос", "голос"},
		{"голоса", "голос"},
		{"голосами", "голос"},
		{"голосе", "голос"},
		{"голосил", "голос"},
		{"голосила", "голос"},
		{"голосистый", "голосист"},
		{"голоском", "голоск"},
		{"голосов", "голос"},
		{"голосок", "голосок"},
		{"голосом", "голос"},
		{"голосу", "голос"},
		{"голосят", "голос"},
		{"голубая", "голуб"},
		{"голубиная", "голубин"},
		{"голубиную", "голубин"},
		{"голубка", "голубк"},
		{"голубоватый", "голубоват"},
		{"голубоватым", "голубоват"},
		{"голубое", "голуб"},
		{"голубой", "голуб"},
		{"голубом", "голуб"},
		{"голубоокий", "голубоок"},
		{"голубою", "голуб"},
		{"голубушка", "голубушк"},
		{"голубую", "голуб"},
		{"голубчик", "голубчик"},
		{"голубые", "голуб"},
		{"голубым", "голуб"},
		{"голубыми", "голуб"},
		{"голубых", "голуб"},
		{"голубь", "голуб"},
		{"голубятни", "голубятн"},
		{"голубятню", "голубятн"},
		{"голую", "гол"},
		{"голы", "гол"},
		{"голые", "гол"},
		{"голый", "гол"},
		{"голым", "гол"},
		{"голыми", "гол"},
		{"голых", "гол"},
		{"голь", "гол"},
		{"гольштейн", "гольштейн"},
		{"гом", "гом"},
		{"гомера", "гомер"},
		{"гомеридами", "гомерид"},
		{"гомозится", "гомоз"},
		{"гомозиться", "гомоз"},
		{"гонга", "гонг"},
		{"гондол", "гондол"},
		{"гондоле", "гондол"},
		{"гонение", "гонен"},
		{"гонений", "гонен"},
		{"гони", "гон"},
		{"гоним", "гон"},
		{"гонима", "гоним"},
		{"гонимого", "гоним"},
		{"гонимый", "гоним"},
		{"гонимых", "гоним"},
		{"гонит", "гон"},
		{"гоните", "гон"},
		{"гонорарий", "гонорар"},
		{"гончаров", "гончар"},
		{"гонявшийся", "гоня"},
		{"гоняется", "гоня"},
		{"гоняла", "гоня"},
		{"гонялся", "гоня"},
		{"гонят", "гон"},
		{"гонять", "гоня"},
		{"гоп", "гоп"},
		{"гор", "гор"},
		{"гора", "гор"},
		{"горазд", "горазд"},
		{"гораздо", "горазд"},
		{"горам", "гор"},
		{"горами", "гор"},
		{"горах", "гор"},
		{"горбатая", "горбат"},
		{"горбиться", "горб"},
		{"горд", "горд"},
		{"горда", "горд"},
		{"гордая", "горд"},
		{"горделиво", "горделив"},
		{"гордилась", "горд"},
		{"гордились", "горд"},
		{"гордился", "горд"},
		{"гордится", "горд"},
		{"гордиться", "горд"},
		{"гордо", "горд"},
		{"гордого", "горд"},
		{"гордой", "горд"},
		{"гордом", "горд"},
		{"гордому", "горд"},
		{"гордости", "гордост"},
		{"гордостию", "гордост"},
		{"гордость", "гордост"},
		{"гордостью", "гордост"},
		{"гордою", "горд"},
		{"горды", "горд"},
		{"гордые", "горд"},
		{"гордый", "горд"},
		{"гордым", "горд"},
		{"гордыне", "гордын"},
		{"гордых", "горд"},
		{"гордясь", "горд"},
		{"гордятся", "горд"},
		{"гордячка", "гордячк"},
		{"горе", "гор"},
		{"горевал", "горева"},
		{"горевать", "горева"},
		{"горевший", "горевш"},
		{"горевшим", "горевш"},
		{"горел", "горел"},
		{"горела", "горел"},
		{"горели", "горел"},
		{"горелки", "горелк"},
		{"горело", "горел"},
		{"горем", "гор"},
		{"горемыка", "горемык"},
		{"горение", "горен"},
		{"горением", "горен"},
		{"горенки", "горенк"},
		{"горенье", "горен"},
		{"горестно", "горестн"},
		{"горестное", "горестн"},
		{"горестной", "горестн"},
		{"горестью", "горест"},
		{"горестях", "горест"},
		{"гореть", "горет"},
		{"горечи", "гореч"},
		{"горечь", "гореч"},
		{"горечью", "гореч"},
		{"горжусь", "горж"},
		{"гори", "гор"},
		{"горизонт", "горизонт"},
		{"горизонта", "горизонт"},
		{"горизонтально", "горизонтальн"},
		{"горизонте", "горизонт"},
		{"горизонтом", "горизонт"},
		{"горизонты", "горизонт"},
		{"гористую", "горист"},
		{"горит", "гор"},
		{"горке", "горк"},
		{"горла", "горл"},
		{"горланили", "горлан"},
		{"горланит", "горлан"},
		{"горле", "горл"},
		{"горло", "горл"},
		{"горловую", "горлов"},
		{"горлом", "горл"},
		{"горлу", "горл"},
		{"горлышко", "горлышк"},
		{"горний", "горн"},
		{"горнист", "горнист"},
		{"горниста", "горнист"},
		{"горницу", "горниц"},
		{"горницы", "горниц"},
		{"горничная", "горничн"},
		{"горничной", "горничн"},
		{"горничных", "горничн"},
		{"горном", "горн"},
		{"горный", "горн"},
		{"горным", "горн"},
		{"горными", "горн"},
		{"город", "город"},
		{"города", "город"},
		{"городам", "город"},
		{"городах", "город"},
		{"городе", "город"},
		{"городите", "город"},
		{"городишке", "городишк"},
		{"городишко", "городишк"},
		{"городишком", "городишк"},
		{"городке", "городк"},
		{"городничего", "городнич"},
		{"городничий", "городнич"},
		{"городового", "городов"},
		{"городовой", "городов"},
		{"городовому", "городов"},
		{"городовые", "городов"},
		{"городок", "городок"},
		{"городом", "город"},
		{"городская", "городск"},
		{"городского", "городск"},
		{"городское", "городск"},
		{"городской", "городск"},
		{"городском", "городск"},
		{"городскую", "городск"},
		{"городу", "город"},
		{"городят", "город"},
		{"горой", "гор"},
		{"гороскоп", "гороскоп"},
		{"горох", "горох"},
		{"горохе", "горох"},
		{"гороховой", "горохов"},
		{"гороховую", "горохов"},
		{"гороховый", "горохов"},
		{"горохом", "горох"},
		{"гороху", "горох"},
		{"горошек", "горошек"},
		{"горошина", "горошин"},
		{"горошком", "горошк"},
		{"горсточку", "горсточк"},
		{"горстью", "горст"},
		{"гортани", "гортан"},
		{"гортанные", "горта"},
		{"гору", "гор"},
		{"горчица", "горчиц"},
		{"горше", "горш"},
		{"горшками", "горшк"},
		{"горшки", "горшк"},
		{"горшков", "горшк"},
		{"горшочки", "горшочк"},
		{"горы", "гор"},
		{"горькая", "горьк"},
		{"горькие", "горьк"},
		{"горький", "горьк"},
		{"горьким", "горьк"},
		{"горьких", "горьк"},
		{"горько", "горьк"},
		{"горькой", "горьк"},
		{"горьком", "горьк"},
		{"горькою", "горьк"},
		{"горькую", "горьк"},
		{"горю", "гор"},
		{"горюй", "горю"},
		{"горюнова", "горюнов"},
		{"горюновы", "горюнов"},
		{"горюновым", "горюнов"},
		{"горючей", "горюч"},
		{"горючий", "горюч"},
		{"горючими", "горюч"},
		{"горя", "гор"},
		{"горят", "гор"},
		{"горяча", "горяч"},
		{"горячась", "горяч"},
		{"горячая", "горяч"},
		{"горячего", "горяч"},
		{"горячей", "горяч"},
		{"горячему", "горяч"},
		{"горячешных", "горячешн"},
		{"горячею", "горяч"},
		{"горячившегося", "горяч"},
		{"горячие", "горяч"},
		{"горячий", "горяч"},
		{"горячил", "горяч"},
		{"горячились", "горяч"},
		{"горячился", "горяч"},
		{"горячим", "горяч"},
		{"горячими", "горяч"},
		{"горячись", "горяч"},
		{"горячится", "горяч"},
		{"горячих", "горяч"},
		{"горячишься", "горяч"},
		{"горячка", "горячк"},
		{"горячке", "горячк"},
		{"горячки", "горячк"},
		{"горячку", "горячк"},
		{"горячности", "горячн"},
		{"горячо", "горяч"},
		{"горячусь", "горяч"},
		{"горячую", "горяч"},
		{"горящее", "горя"},
		{"горящей", "горя"},
		{"горящий", "горя"},
		{"горящим", "горя"},
		{"горящими", "горя"},
		{"госпитале", "госпитал"},
		{"госпитальный", "госпитальн"},
		{"госпитальных", "госпитальн"},
		{"господ", "господ"},
		{"господа", "господ"},
		{"господам", "господ"},
		{"господах", "господ"},
		{"господи", "господ"},
		{"господин", "господин"},
		{"господина", "господин"},
		{"господине", "господин"},
		{"господином", "господин"},
		{"господину", "господин"},
		{"господним", "господн"},
		{"господня", "господн"},
		{"господняя", "господн"},
		{"господом", "господ"},
		{"господская", "господск"},
		{"господский", "господск"},
		{"господского", "господск"},
		{"господской", "господск"},
		{"господствовавшей", "господствова"},
		{"господствовала", "господствова"},
		{"господствующею", "господств"},
		{"господствующие", "господств"},
		{"господствующим", "господств"},
		{"господу", "господ"},
		{"господь", "господ"},
		{"госпожа", "госпож"},
		{"госпожам", "госпож"},
		{"госпоже", "госпож"},
		{"госпожи", "госпож"},
		{"госпожинок", "госпожинок"},
		{"госпожой", "госпож"},
		{"госпожу", "госпож"},
		{"гостей", "гост"},
		{"гостем", "гост"},
		{"гостеприимном", "гостеприимн"},
		{"гостеприимный", "гостеприимн"},
		{"гости", "гост"},
		{"гостивший", "гост"},
		{"гостила", "гост"},
		{"гостиная", "гостин"},
		{"гостинец", "гостинец"},
		{"гостиница", "гостиниц"},
		{"гостинице", "гостиниц"},
		{"гостиницу", "гостиниц"},
		{"гостиницы", "гостиниц"},
		{"гостиного", "гостин"},
		{"гостиной", "гостин"},
		{"гостиную", "гостин"},
		{"гостинцу", "гостинц"},
		{"гостинцы", "гостинц"},
		{"гостиный", "гостин"},
		{"гостиных", "гостин"},
		{"гостит", "гост"},
		{"гостить", "гост"},
		{"гость", "гост"},
		{"гостья", "гост"},
		{"гостю", "гост"},
		{"гостя", "гост"},
		{"гостям", "гост"},
		{"гостями", "гост"},
		{"гостях", "гост"},
		{"государственная", "государствен"},
		{"государственное", "государствен"},
		{"государственный", "государствен"},
		{"государственным", "государствен"},
		{"государственных", "государствен"},
		{"государыня", "государын"},
		{"государь", "государ"},
		{"государю", "государ"},
		{"гот", "гот"},
		{"готов", "гот"},
		{"готова", "готов"},
		{"готовая", "готов"},
		{"готовившиеся", "готов"},
		{"готовившийся", "готов"},
		{"готовил", "готов"},
		{"готовила", "готов"},
		{"готовилась", "готов"},
		{"готовили", "готов"},
		{"готовились", "готов"},
		{"готовился", "готов"},
		{"готовит", "готов"},
		{"готовитесь", "готов"},
		{"готовится", "готов"},
		{"готовить", "готов"},
		{"готовиться", "готов"},
		{"готовлюсь", "готовл"},
		{"готовность", "готовн"},
		{"готово", "готов"},
		{"готового", "готов"},
		{"готовое", "готов"},
		{"готовой", "готов"},
		{"готовом", "готов"},
		{"готовую", "готов"},
		{"готовы", "готов"},
		{"готовые", "готов"},
		{"готовый", "готов"},
		{"готовыми", "готов"},
		{"готовь", "готов"},
		{"готовясь", "готов"},
		{"готовят", "готов"},
		{"готовятся", "готов"},
		{"готовящегося", "готовя"},
		{"гофкригсрат", "гофкригсрат"},
		{"гр", "гр"},
		{"грабеж", "грабеж"},
		{"грабежа", "грабеж"},
		{"грабежах", "грабеж"},
		{"грабеже", "грабеж"},
		{"грабежей", "грабеж"},
		{"грабежом", "грабеж"},
		{"грабил", "граб"},
		{"грабит", "граб"},
		{"грабителя", "грабител"},
		{"грабят", "граб"},
		{"гравюр", "гравюр"},
		{"гравюры", "гравюр"},
		{"град", "град"},
		{"граденаплем", "граденапл"},
		{"градирен", "градир"},
		{"градуса", "градус"},
		{"градусник", "градусник"},
		{"градусов", "градус"},
		{"градусы", "градус"},
		{"грады", "град"},
		{"гражданин", "гражданин"},
		{"гражданина", "гражданин"},
		{"гражданские", "гражданск"},
		{"гражданский", "гражданск"},
		{"гражданских", "гражданск"},
		{"гражданской", "гражданск"},
		{"гражданском", "гражданск"},
		{"гражданскому", "гражданск"},
		{"грамматике", "грамматик"},
		{"грамматику", "грамматик"},
		{"грамота", "грамот"},
		{"грамотеи", "грамот"},
		{"грамотность", "грамотн"},
		{"грамотный", "грамотн"},
		{"грамотных", "грамотн"},
		{"грамоты", "грамот"},
		{"гранат", "гранат"},
		{"грандиозного", "грандиозн"},
		{"грандиозное", "грандиозн"},
		{"грандиозные", "грандиозн"},
		{"грани", "гран"},
		{"гранит", "гран"},
		{"гранитные", "гранитн"},
		{"границ", "границ"},
		{"граница", "границ"},
		{"границей", "границ"},
		{"границею", "границ"},
		{"границу", "границ"},
		{"границы", "границ"},
		{"гранпасьянс", "гранпасьянс"},
		{"графа", "граф"},
		{"графин", "графин"},
		{"графина", "графин"},
		{"графине", "графин"},
		{"графини", "графин"},
		{"графинина", "графинин"},
		{"графином", "графин"},
		{"графинчик", "графинчик"},
		{"графов", "граф"},
		{"графский", "графск"},
		{"графского", "графск"},
		{"графской", "графск"},
		{"графскому", "графск"},
		{"графу", "граф"},
		{"графы", "граф"},
		{"грации", "грац"},
		{"грациозно", "грациозн"},
		{"грациозности", "грациозн"},
		{"грациозную", "грациозн"},
		{"грациозны", "грациозн"},
		{"грациозный", "грациозн"},
		{"грация", "грац"},
		{"гребенки", "гребенк"},
		{"гребенку", "гребенк"},
		{"гребень", "гребен"},
		{"греет", "греет"},
		{"грез", "грез"},
		{"греза", "грез"},
		{"грезами", "грез"},
		{"грезах", "грез"},
		{"грезил", "грез"},
		{"грезилась", "грез"},
		{"грезилось", "грез"},
		{"грезит", "грез"},
		{"грезится", "грез"},
		{"грезить", "грез"},
		{"грезишь", "грез"},
		{"грезы", "грез"},
		{"грезятся", "грез"},
		{"гремел", "гремел"},
		{"гремела", "гремел"},
		{"гремело", "гремел"},
		{"гремит", "грем"},
		{"гренадера", "гренадер"},
		{"гренадеры", "гренадер"},
		{"греть", "грет"},
		{"грех", "грех"},
		{"греха", "грех"},
		{"грехам", "грех"},
		{"грехе", "грех"},
		{"грехи", "грех"},
		{"грехов", "грех"},
		{"греховной", "греховн"},
		{"греховные", "греховн"},
		{"грехом", "грех"},
		{"гречневой", "гречнев"},
		{"гречонка", "гречонк"},
		{"грешен", "греш"},
		{"греши", "греш"},
		{"грешили", "греш"},
		{"грешить", "греш"},
		{"грешник", "грешник"},
		{"грешница", "грешниц"},
		{"грешницу", "грешниц"},
		{"грешно", "грешн"},
		{"грешной", "грешн"},
		{"грешную", "грешн"},
		{"грешные", "грешн"},
		{"грешный", "грешн"},
		{"грибами", "гриб"},
		{"грибов", "гриб"},
		{"грива", "грив"},
		{"гривен", "грив"},
		{"гривенник", "гривенник"},
		{"гривенниками", "гривенник"},
		{"гривенники", "гривенник"},
		{"гривнами", "гривн"},
		{"гривне", "гривн"},
		{"гривну", "гривн"},
		{"гривны", "гривн"},
		{"григория", "григор"},
		{"григорьевич", "григорьевич"},
		{"григорьевича", "григорьевич"},
		{"григорьевичем", "григорьевич"},
		{"гримасой", "гримас"},
		{"гримасу", "гримас"},
		{"гриф", "гриф"},
		{"гриша", "гриш"},
		{"гриши", "гриш"},
		{"гроб", "гроб"},
		{"гроба", "гроб"},
		{"гробам", "гроб"},
		{"гробах", "гроб"},
		{"гробе", "гроб"},
		{"гробниц", "гробниц"},
		{"гробницей", "гробниц"},
		{"гробов", "гроб"},
		{"гробовая", "гробов"},
		{"гробовой", "гробов"},
		{"гробовою", "гробов"},
		{"гробовые", "гробов"},
		{"гробом", "гроб"},
		{"гробу", "гроб"},
		{"гроз", "гроз"},
		{"гроза", "гроз"},
		{"грозе", "гроз"},
		{"грозил", "гроз"},
		{"грозила", "гроз"},
		{"грозило", "гроз"},
		{"грозим", "гроз"},
		{"грозись", "гроз"},
		{"грозит", "гроз"},
		{"грозится", "гроз"},
		{"грозишь", "гроз"},
		{"грозная", "грозн"},
		{"грозней", "грозн"},
		{"грозно", "грозн"},
		{"грозное", "грозн"},
		{"грозной", "грозн"},
		{"грозном", "грозн"},
		{"грозному", "грозн"},
		{"грозны", "грозн"},
		{"грозные", "грозн"},
		{"грозный", "грозн"},
		{"грозным", "грозн"},
		{"грозных", "грозн"},
		{"грозовая", "грозов"},
		{"грозовой", "грозов"},
		{"грозовые", "грозов"},
		{"грозовых", "грозов"},
		{"грозой", "гроз"},
		{"грозы", "гроз"},
		{"грозя", "гроз"},
		{"грозясь", "гроз"},
		{"грозят", "гроз"},
		{"грозящие", "грозя"},
		{"гром", "гром"},
		{"грома", "гром"},
		{"громад", "громад"},
		{"громада", "громад"},
		{"громаднейшего", "громадн"},
		{"громадного", "громадн"},
		{"громадное", "громадн"},
		{"громадной", "громадн"},
		{"громадном", "громадн"},
		{"громадную", "громадн"},
		{"громадные", "громадн"},
		{"громадный", "громадн"},
		{"громадным", "громадн"},
		{"громадных", "громадн"},
		{"громам", "гром"},
		{"громами", "гром"},
		{"громит", "гром"},
		{"громкая", "громк"},
		{"громкий", "громк"},
		{"громким", "громк"},
		{"громкими", "громк"},
		{"громко", "громк"},
		{"громкое", "громк"},
		{"громком", "громк"},
		{"громовая", "громов"},
		{"громовой", "громов"},
		{"громовые", "громов"},
		{"громовым", "громов"},
		{"громоздить", "громозд"},
		{"громоздкий", "громоздк"},
		{"громоздко", "громоздк"},
		{"громом", "гром"},
		{"громче", "громч"},
		{"громыханье", "громыхан"},
		{"громя", "гром"},
		{"грот", "грот"},
		{"грохнулась", "грохнул"},
		{"грохнулся", "грохнул"},
		{"грохольского", "грохольск"},
		{"грохот", "грохот"},
		{"грохотом", "грохот"},
		{"грош", "грош"},
		{"гроша", "грош"},
		{"грошей", "грош"},
		{"грошик", "грошик"},
		{"грошовый", "грошов"},
		{"грошовых", "грошов"},
		{"гру", "гру"},
		{"груб", "груб"},
		{"груба", "груб"},
		{"грубая", "груб"},
		{"грубее", "груб"},
		{"грубеешь", "грубееш"},
		{"грубейшим", "груб"},
		{"грубить", "груб"},
		{"грубиян", "грубия"},
		{"грубиянит", "грубиян"},
		{"грубо", "груб"},
		{"грубоват", "грубоват"},
		{"грубовато", "грубоват"},
		{"грубого", "груб"},
		{"грубое", "груб"},
		{"грубой", "груб"},
		{"грубом", "груб"},
		{"грубости", "грубост"},
		{"грубостию", "грубост"},
		{"грубость", "грубост"},
		{"грубостью", "грубост"},
		{"грубою", "груб"},
		{"грубые", "груб"},
		{"грубый", "груб"},
		{"грубым", "груб"},
		{"грубыми", "груб"},
		{"грубых", "груб"},
		{"грудами", "груд"},
		{"грудах", "груд"},
		{"груде", "груд"},
		{"груди", "груд"},
		{"грудная", "грудн"},
		{"грудной", "грудн"},
		{"грудные", "грудн"},
		{"грудных", "грудн"},
		{"грудой", "груд"},
		{"груду", "груд"},
		{"груды", "груд"},
		{"грудь", "груд"},
		{"грудью", "груд"},
		{"груз", "груз"},
		{"груза", "груз"},
		{"грузди", "грузд"},
		{"грунт", "грунт"},
		{"группа", "групп"},
		{"группами", "групп"},
		{"группы", "групп"},
		{"грустен", "груст"},
		{"грусти", "груст"},
		{"грустим", "груст"},
		{"грустит", "груст"},
		{"грустишь", "груст"},
		{"грустная", "грустн"},
		{"грустнее", "грустн"},
		{"грустно", "грустн"},
		{"грустного", "грустн"},
		{"грустное", "грустн"},
		{"грустной", "грустн"},
		{"грустном", "грустн"},
		{"грустною", "грустн"},
		{"грустную", "грустн"},
		{"грустные", "грустн"},
		{"грустный", "грустн"},
		{"грустным", "грустн"},
		{"грустных", "грустн"},
		{"грусть", "груст"},
		{"грустью", "груст"},
		{"грустя", "груст"},
		{"грустят", "груст"},
		{"груше", "груш"},
		{"грушевой", "грушев"},
		{"груши", "груш"},
		{"грущу", "грущ"},
		{"грызет", "грызет"},
		{"грызла", "грызл"},
		{"грызло", "грызл"},
		{"гряд", "гряд"},
		{"грядами", "гряд"},
		{"грядущего", "грядущ"},
		{"грядущее", "грядущ"},
		{"грядущей", "грядущ"},
		{"грядущем", "грядущ"},
		{"грядущие", "грядущ"},
		{"грядущий", "грядущ"},
		{"грядущим", "грядущ"},
		{"грядущих", "грядущ"},
		{"гряды", "гряд"},
		{"грязи", "гряз"},
		{"грязна", "грязн"},
		{"грязная", "грязн"},
		{"грязно", "грязн"},
		{"грязное", "грязн"},
		{"грязной", "грязн"},
		{"грязном", "грязн"},
		{"грязнотцой", "грязнотц"},
		{"грязною", "грязн"},
		{"грязную", "грязн"},
		{"грязны", "грязн"},
		{"грязные", "грязн"},
		{"грязный", "грязн"},
		{"грязными", "грязн"},
		{"грязных", "грязн"},
		{"грязня", "грязн"},
		{"грязь", "гряз"},
		{"грязью", "гряз"},
		{"грянет", "грянет"},
		{"грянул", "грянул"},
		{"грянулась", "грянул"},
		{"грянули", "грянул"},
		{"грянуло", "грянул"},
		{"губ", "губ"},
		{"губа", "губ"},
		{"губам", "губ"},
		{"губами", "губ"},
		{"губах", "губ"},
		{"губернатор", "губернатор"},
		{"губернатора", "губернатор"},
		{"губернаторе", "губернатор"},
		{"губернатором", "губернатор"},
		{"губернаторстве", "губернаторств"},
		{"губернатору", "губернатор"},
		{"губернии", "губерн"},
		{"губерний", "губерн"},
		{"губернию", "губерн"},
		{"губерния", "губерн"},
		{"губерниям", "губерн"},
		{"губернский", "губернск"},
		{"губернским", "губернск"},
		{"губернских", "губернск"},
		{"губернского", "губернск"},
		{"губернском", "губернск"},
		{"губил", "губ"},
		{"губили", "губ"},
		{"губите", "губ"},
		{"губительнее", "губительн"},
		{"губительный", "губительн"},
		{"губительных", "губительн"},
		{"губить", "губ"},
		{"губишь", "губ"},
		{"губка", "губк"},
		{"губки", "губк"},
		{"губкой", "губк"},
		{"губой", "губ"},
		{"губок", "губок"},
		{"губу", "губ"},
		{"губы", "губ"},
		{"губят", "губ"},
		{"гувернантка", "гувернантк"},
		{"гувернантках", "гувернантк"},
		{"гувернантки", "гувернантк"},
		{"гувернанткой", "гувернантк"},
		{"гудел", "гудел"},
		{"гудело", "гудел"},
		{"гудки", "гудк"},
		{"гудут", "гудут"},
		{"гуды", "гуд"},
		{"гул", "гул"},
		{"гуле", "гул"},
		{"гулкий", "гулк"},
		{"гулом", "гул"},
		{"гульбе", "гульб"},
		{"гуляет", "гуля"},
		{"гулякой", "гуляк"},
		{"гулял", "гуля"},
		{"гуляла", "гуля"},
		{"гуляли", "гуля"},
		{"гулянье", "гулян"},
		{"гуляньями", "гулян"},
		{"гулять", "гуля"},
		{"гуляю", "гуля"},
		{"гуляют", "гуля"},
		{"гуляющей", "гуля"},
		{"гуляя", "гул"},
		{"гумаге", "гумаг"},
		{"гумагу", "гумаг"},
		{"гуманистический", "гуманистическ"},
		{"гуманитетом", "гуманитет"},
		{"гуманная", "гума"},
		{"гуманно", "гума"},
		{"гуманного", "гума"},
		{"гуманном", "гума"},
		{"гуманности", "гуман"},
		{"гуманные", "гума"},
		{"гуманный", "гума"},
		{"гуммиластик", "гуммиластик"},
		{"гунял", "гуня"},
		{"гурьбой", "гурьб"},
		{"гусак", "гусак"},
		{"гусаки", "гусак"},
		{"гусар", "гусар"},
		{"гусара", "гусар"},
		{"гусарские", "гусарск"},
		{"гусарское", "гусарск"},
		{"гусей", "гус"},
		{"гуси", "гус"},
		{"гусиных", "гусин"},
		{"густая", "густ"},
		{"густейшими", "густ"},
		{"густо", "густ"},
		{"густой", "густ"},
		{"густоту", "густот"},
		{"густою", "густ"},
		{"густые", "густ"},
		{"густым", "густ"},
		{"густыми", "густ"},
		{"гусыни", "гусын"},
		{"гуся", "гу"},
		{"гущей", "гущ"},
		{"гюго", "гюг"},
		{"д", "д"},
		{"да", "да"},
		{"дав", "дав"},
		{"давай", "дава"},
		{"давайте", "дава"},
		{"давал", "дава"},
		{"давала", "дава"},
		{"давалася", "дава"},
		{"давали", "дава"},
		{"давался", "дава"},
		{"давать", "дава"},
		{"давая", "дав"},
		{"даве", "дав"},
		{"давеча", "давеч"},
		{"давешнего", "давешн"},
		{"давешнее", "давешн"},
		{"давешней", "давешн"},
		{"давешнему", "давешн"},
		{"давешний", "давешн"},
		{"давешних", "давешн"},
		{"давешнюю", "давешн"},
		{"давешняя", "давешн"},
		{"давившее", "дав"},
		{"давившей", "дав"},
		{"давила", "дав"},
		{"давило", "дав"},
		{"давит", "дав"},
		{"давить", "дав"},
		{"давка", "давк"},
		{"давке", "давк"},
		{"давлением", "давлен"},
		{"давней", "давн"},
		{"давненько", "давненьк"},
		{"давнишнее", "давнишн"},
		{"давнишней", "давнишн"},
		{"давнишние", "давнишн"},
		{"давнишний", "давнишн"},
		{"давнишних", "давнишн"},
		{"давнишнюю", "давнишн"},
		{"давно", "давн"},
		{"давным", "давн"},
		{"давняя", "давн"},
		{"давя", "дав"},
		{"давящим", "давя"},
		{"дагестана", "дагеста"},
		{"дадим", "дад"},
		{"дадите", "дад"},
		{"дадут", "дадут"},
		{"дает", "дает"},
		{"даете", "дает"},
		{"дается", "дает"},
		{"даже", "даж"},
		{"дай", "да"},
		{"дайте", "дайт"},
		{"дал", "дал"},
		{"дала", "дал"},
		{"далась", "дал"},
		{"далее", "дал"},
		{"далей", "дал"},
		{"далек", "далек"},
		{"далека", "далек"},
		{"далекая", "далек"},
		{"далекие", "далек"},
		{"далекий", "далек"},
		{"далеким", "далек"},
		{"далеких", "далек"},
		{"далеко", "далек"},
		{"далекое", "далек"},
		{"далекой", "далек"},
		{"далеком", "далек"},
		{"далекую", "далек"},
		{"далече", "далеч"},
		{"дали", "дал"},
		{"дались", "дал"},
		{"дало", "дал"},
		{"далось", "дал"},
		{"дался", "дал"},
		{"даль", "дал"},
		{"дальнего", "дальн"},
		{"дальней", "дальн"},
		{"дальнейшего", "дальн"},
		{"дальнейшее", "дальн"},
		{"дальнейшей", "дальн"},
		{"дальнейшем", "дальн"},
		{"дальнейшему", "дальн"},
		{"дальнейшею", "дальн"},
		{"дальнейшие", "дальн"},
		{"дальнейших", "дальн"},
		{"дальнему", "дальн"},
		{"дальние", "дальн"},
		{"дальний", "дальн"},
		{"дальних", "дальн"},
		{"дальный", "дальн"},
		{"дальня", "дальн"},
		{"дальняя", "дальн"},
		{"дальше", "дальш"},
		{"далью", "дал"},
		{"дам", "дам"},
		{"дама", "дам"},
		{"дамам", "дам"},
		{"дамами", "дам"},
		{"дамах", "дам"},
		{"даме", "дам"},
		{"дамой", "дам"},
		{"дамою", "дам"},
		{"дамские", "дамск"},
		{"дамскую", "дамск"},
		{"дамся", "дам"},
		{"даму", "дам"},
		{"дамы", "дам"},
		{"дана", "дан"},
		{"данного", "дан"},
		{"данное", "дан"},
		{"данному", "дан"},
		{"данную", "дан"},
		{"данным", "дан"},
		{"дано", "дан"},
		{"дант", "дант"},
		{"данта", "дант"},
		{"дантово", "дантов"},
		{"дань", "дан"},
		{"дар", "дар"},
		{"дарами", "дар"},
		{"дарвина", "дарвин"},
		{"дареную", "дарен"},
		{"дарил", "дар"},
		{"дарить", "дар"},
		{"дармоедка", "дармоедк"},
		{"дармоедом", "дармоед"},
		{"даров", "дар"},
		{"дарований", "дарован"},
		{"дарования", "дарован"},
		{"даровщинку", "даровщинк"},
		{"даром", "дар"},
		{"дарь", "дар"},
		{"дарье", "дар"},
		{"дарьи", "дар"},
		{"дарья", "дар"},
		{"дарят", "дар"},
		{"даст", "даст"},
		{"дать", "дат"},
		{"дач", "дач"},
		{"дача", "дач"},
		{"дачами", "дач"},
		{"дачах", "дач"},
		{"даче", "дач"},
		{"дачей", "дач"},
		{"дачи", "дач"},
		{"дачник", "дачник"},
		{"дачники", "дачник"},
		{"дачников", "дачник"},
		{"дачном", "дачн"},
		{"дачные", "дачн"},
		{"дачу", "дач"},
		{"дашенька", "дашеньк"},
		{"дашеньки", "дашеньк"},
		{"дашеньку", "дашеньк"},
		{"дашь", "даш"},
		{"даю", "да"},
		{"дают", "дают"},
		{"два", "два"},
		{"двадцати", "двадцат"},
		{"двадцатилетней", "двадцатилетн"},
		{"двадцатирублевые", "двадцатирублев"},
		{"двадцатисемилетний", "двадцатисемилетн"},
		{"двадцатый", "двадцат"},
		{"двадцать", "двадца"},
		{"двадцатью", "двадцат"},
		{"дважды", "дважд"},
		{"две", "две"},
		{"двенадцати", "двенадцат"},
		{"двенадцатого", "двенадцат"},
		{"двенадцатом", "двенадцат"},
		{"двенадцатую", "двенадцат"},
		{"двенадцатый", "двенадцат"},
		{"двенадцать", "двенадца"},
		{"двенадцатью", "двенадцат"},
		{"дверах", "двер"},
		{"дверей", "двер"},
		{"дверец", "дверец"},
		{"двери", "двер"},
		{"дверными", "дверн"},
		{"дверь", "двер"},
		{"дверьми", "дверьм"},
		{"дверью", "двер"},
		{"дверям", "двер"},
		{"дверями", "двер"},
		{"дверях", "двер"},
		{"двести", "двест"},
		{"двигавшимся", "двига"},
		{"двигает", "двига"},
		{"двигается", "двига"},
		{"двигал", "двига"},
		{"двигалась", "двига"},
		{"двигали", "двига"},
		{"двигались", "двига"},
		{"двигалось", "двига"},
		{"двигался", "двига"},
		{"двигатель", "двигател"},
		{"двигателя", "двигател"},
		{"двигать", "двига"},
		{"двигаться", "двига"},
		{"двигают", "двига"},
		{"двигая", "двиг"},
		{"двигаясь", "двиг"},
		{"движение", "движен"},
		{"движением", "движен"},
		{"движении", "движен"},
		{"движений", "движен"},
		{"движению", "движен"},
		{"движения", "движен"},
		{"движениях", "движен"},
		{"движенье", "движен"},
		{"движеньем", "движен"},
		{"движенья", "движен"},
		{"движеньях", "движен"},
		{"движет", "движет"},
		{"движется", "движет"},
		{"движешься", "движеш"},
		{"движимости", "движим"},
		{"движимый", "движим"},
		{"движись", "движ"},
		{"движутся", "движут"},
		{"движущее", "движущ"},
		{"движущееся", "движущ"},
		{"движущиеся", "движущ"},
		{"движущихся", "движущ"},
		{"двинется", "двинет"},
		{"двинулась", "двинул"},
		{"двинули", "двинул"},
		{"двинулись", "двинул"},
		{"двинулся", "двинул"},
		{"двинутся", "двинут"},
		{"двинуть", "двинут"},
		{"двинуться", "двинут"},
		{"двое", "дво"},
		{"двоеверьем", "двоевер"},
		{"двои", "дво"},
		{"двоим", "дво"},
		{"двоих", "дво"},
		{"двойку", "двойк"},
		{"двойник", "двойник"},
		{"двойной", "двойн"},
		{"двойные", "двойн"},
		{"двойным", "двойн"},
		{"двойных", "двойн"},
		{"двор", "двор"},
		{"двора", "двор"},
		{"дворам", "двор"},
		{"дворах", "двор"},
		{"дворе", "двор"},
		{"дворец", "дворец"},
		{"дворик", "дворик"},
		{"дворне", "дворн"},
		{"дворней", "дворн"},
		{"дворни", "дворн"},
		{"дворник", "дворник"},
		{"дворника", "дворник"},
		{"дворнике", "дворник"},
		{"дворники", "дворник"},
		{"дворников", "дворник"},
		{"дворникова", "дворников"},
		{"дворником", "дворник"},
		{"дворнику", "дворник"},
		{"дворницкой", "дворницк"},
		{"дворницкую", "дворницк"},
		{"дворню", "дворн"},
		{"дворня", "дворн"},
		{"дворняжка", "дворняжк"},
		{"дворняжки", "дворняжк"},
		{"дворов", "двор"},
		{"дворовый", "дворов"},
		{"двором", "двор"},
		{"двору", "двор"},
		{"дворца", "дворц"},
		{"дворце", "дворц"},
		{"дворцовой", "дворцов"},
		{"дворцовым", "дворцов"},
		{"дворцы", "дворц"},
		{"дворян", "дворя"},
		{"дворяне", "дворян"},
		{"дворянин", "дворянин"},
		{"дворянина", "дворянин"},
		{"дворянке", "дворянк"},
		{"дворянки", "дворянк"},
		{"дворянская", "дворянск"},
		{"дворянские", "дворянск"},
		{"дворянский", "дворянск"},
		{"дворянскими", "дворянск"},
		{"дворянского", "дворянск"},
		{"дворянской", "дворянск"},
		{"дворянском", "дворянск"},
		{"двугривенного", "двугривен"},
		{"двугривенные", "двугривен"},
		{"двугривенный", "двугривен"},
		{"двугривенных", "двугривен"},
		{"двукратного", "двукратн"},
		{"двуличного", "двуличн"},
		{"двум", "двум"},
		{"двумя", "двум"},
		{"двусмысленного", "двусмыслен"},
		{"двусмысленности", "двусмыслен"},
		{"двусмысленную", "двусмыслен"},
		{"двусмысленных", "двусмыслен"},
		{"двух", "двух"},
		{"двухкопеечника", "двухкопеечник"},
		{"двухмачтовый", "двухмачтов"},
		{"двухминутной", "двухминутн"},
		{"двухнедельное", "двухнедельн"},
		{"двухчасовое", "двухчасов"},
		{"двухэтажном", "двухэтажн"},
		{"де", "де"},
		{"дебаркадера", "дебаркадер"},
		{"дебатирован", "дебатирова"},
		{"дебош", "дебош"},
		{"дебютировала", "дебютирова"},
		{"дев", "дев"},
		{"дева", "дев"},
		{"деваешь", "дева"},
		{"девал", "дева"},
		{"девам", "дев"},
		{"девать", "дева"},
		{"деваться", "дева"},
		{"деверь", "девер"},
		{"девизами", "девиз"},
		{"девиц", "девиц"},
		{"девица", "девиц"},
		{"девицами", "девиц"},
		{"девицах", "девиц"},
		{"девице", "девиц"},
		{"девицей", "девиц"},
		{"девицу", "девиц"},
		{"девицы", "девиц"},
		{"девически", "девическ"},
		{"девические", "девическ"},
		{"девический", "девическ"},
		{"девического", "девическ"},
		{"девической", "девическ"},
		{"девичий", "девич"},
		{"девичьего", "девич"},
		{"девичьей", "девич"},
		{"девичьем", "девич"},
		{"девичью", "девич"},
		{"девка", "девк"},
		{"девкам", "девк"},
		{"девке", "девк"},
		{"девки", "девк"},
		{"девкой", "девк"},
		{"девой", "дев"},
		{"девок", "девок"},
		{"девочек", "девочек"},
		{"девочка", "девочк"},
		{"девочкам", "девочк"},
		{"девочками", "девочк"},
		{"девочке", "девочк"},
		{"девочки", "девочк"},
		{"девочкой", "девочк"},
		{"девочку", "девочк"},
		{"девственна", "девствен"},
		{"девственного", "девствен"},
		{"девственное", "девствен"},
		{"девственность", "девствен"},
		{"девушек", "девушек"},
		{"девушка", "девушк"},
		{"девушке", "девушк"},
		{"девушки", "девушк"},
		{"девушкой", "девушк"},
		{"девушку", "девушк"},
		{"девчонка", "девчонк"},
		{"девчонки", "девчонк"},
		{"девчонку", "девчонк"},
		{"девчоночку", "девчоночк"},
		{"девчурочка", "девчурочк"},
		{"девы", "дев"},
		{"девяносто", "девян"},
		{"девяти", "девят"},
		{"девятнадцатый", "девятнадцат"},
		{"девятнадцать", "девятнадца"},
		{"девятого", "девят"},
		{"девятом", "девят"},
		{"девятый", "девят"},
		{"девятым", "девят"},
		{"девять", "девя"},
		{"девятьсот", "девятьсот"},
		{"деготь", "дегот"},
		{"дегтем", "дегт"},
		{"дед", "дед"},
		{"деда", "дед"},
		{"дедах", "дед"},
		{"дедов", "дед"},
		{"дедом", "дед"},
		{"деду", "дед"},
		{"дедушка", "дедушк"},
		{"дедушке", "дедушк"},
		{"дедушки", "дедушк"},
		{"деды", "дед"},
		{"дезертировать", "дезертирова"},
		{"действенный", "действен"},
		{"действие", "действ"},
		{"действием", "действ"},
		{"действий", "действ"},
		{"действительная", "действительн"},
		{"действительнее", "действительн"},
		{"действительно", "действительн"},
		{"действительной", "действительн"},
		{"действительности", "действительн"},
		{"действительность", "действительн"},
		{"действительностью", "действительн"},
		{"действительные", "действительн"},
		{"действительный", "действительн"},
		{"действительным", "действительн"},
		{"действительных", "действительн"},
		{"действия", "действ"},
		{"действиях", "действ"},
		{"действйтельности", "действйтельн"},
		{"действовал", "действова"},
		{"действовало", "действова"},
		{"действовать", "действова"},
		{"действует", "действ"},
		{"действуют", "действ"},
		{"действующие", "действ"},
		{"действуя", "действу"},
		{"декабре", "декабр"},
		{"декабрь", "декабр"},
		{"декабрьский", "декабрьск"},
		{"декабря", "декабр"},
		{"декадент", "декадент"},
		{"декадентах", "декадент"},
		{"декадентский", "декадентск"},
		{"декадентское", "декадентск"},
		{"декан", "дека"},
		{"декламирует", "декламир"},
		{"декламируете", "декламирует"},
		{"декламируя", "декламиру"},
		{"декларируя", "деклариру"},
		{"декорации", "декорац"},
		{"декораций", "декорац"},
		{"декорация", "декорац"},
		{"дел", "дел"},
		{"дела", "дел"},
		{"делаем", "дела"},
		{"делает", "дела"},
		{"делаете", "дела"},
		{"делается", "дела"},
		{"делаешь", "дела"},
		{"делай", "дела"},
		{"делайте", "дела"},
		{"делал", "дела"},
		{"делала", "дела"},
		{"делалась", "дела"},
		{"делали", "дела"},
		{"делались", "дела"},
		{"делалось", "дела"},
		{"делаль", "делал"},
		{"делам", "дел"},
		{"делами", "дел"},
		{"делась", "дел"},
		{"делать", "дела"},
		{"делах", "дел"},
		{"делаю", "дела"},
		{"делаюсь", "дела"},
		{"делают", "дела"},
		{"делающим", "дела"},
		{"делая", "дел"},
		{"делаясь", "дел"},
		{"деле", "дел"},
		{"деления", "делен"},
		{"делец", "делец"},
		{"деликатен", "деликат"},
		{"деликатная", "деликатн"},
		{"деликатнейшим", "деликатн"},
		{"деликатно", "деликатн"},
		{"деликатности", "деликатн"},
		{"деликатный", "деликатн"},
		{"деликатным", "деликатн"},
		{"деликатными", "деликатн"},
		{"делила", "дел"},
		{"делился", "дел"},
		{"делись", "дел"},
		{"делит", "дел"},
		{"делишкам", "делишк"},
		{"делишки", "делишк"},
		{"дело", "дел"},
		{"деловая", "делов"},
		{"деловитости", "деловит"},
		{"деловитость", "деловит"},
		{"делового", "делов"},
		{"деловой", "делов"},
		{"деловую", "делов"},
		{"деловые", "делов"},
		{"деловым", "делов"},
		{"деловых", "делов"},
		{"делом", "дел"},
		{"делось", "дел"},
		{"делу", "дел"},
		{"делывал", "делыва"},
		{"дельное", "дельн"},
		{"дельному", "дельн"},
		{"дельный", "дельн"},
		{"дельцам", "дельц"},
		{"дельце", "дельц"},
		{"дельцу", "дельц"},
		{"дементьева", "дементьев"},
		{"демисезон", "демисезон"},
		{"демка", "демк"},
		{"демон", "демон"},
		{"демона", "демон"},
		{"демонизм", "демонизм"},
		{"демонов", "демон"},
		{"демонологии", "демонолог"},
		{"демоном", "демон"},
		{"демонским", "демонск"},
		{"демонстрации", "демонстрац"},
		{"денег", "денег"},
		{"денежного", "денежн"},
		{"денежной", "денежн"},
		{"денежные", "денежн"},
		{"денежный", "денежн"},
		{"денек", "денек"},
		{"денем", "ден"},
		{"дение", "ден"},
		{"денницу", "денниц"},
		{"денно", "ден"},
		{"дену", "ден"},
		{"денщик", "денщик"},
		{"денщиков", "денщик"},
		{"денщику", "денщик"},
		{"день", "ден"},
		{"деньгам", "деньг"},
		{"деньгами", "деньг"},
		{"деньгах", "деньг"},
		{"деньги", "деньг"},
		{"деньжонки", "деньжонк"},
		{"денька", "деньк"},
		{"деньские", "деньск"},
		{"деньской", "деньск"},
		{"департамент", "департамент"},
		{"департамента", "департамент"},
		{"департаментах", "департамент"},
		{"департаменте", "департамент"},
		{"депутата", "депутат"},
		{"депутатами", "депутат"},
		{"депутатов", "депутат"},
		{"депутаты", "депутат"},
		{"дер", "дер"},
		{"дергает", "дерга"},
		{"дергайте", "дерга"},
		{"дергала", "дерга"},
		{"дергали", "дерга"},
		{"дергать", "дерга"},
		{"дергая", "дерг"},
		{"дерев", "дер"},
		{"дерева", "дерев"},
		{"дереве", "дерев"},
		{"деревенели", "деревенел"},
		{"деревенская", "деревенск"},
		{"деревенские", "деревенск"},
		{"деревенский", "деревенск"},
		{"деревенскими", "деревенск"},
		{"деревенских", "деревенск"},
		{"деревенского", "деревенск"},
		{"деревенское", "деревенск"},
		{"деревенской", "деревенск"},
		{"деревень", "деревен"},
		{"деревеньках", "деревеньк"},
		{"деревне", "деревн"},
		{"деревней", "деревн"},
		{"деревни", "деревн"},
		{"деревню", "деревн"},
		{"деревня", "деревн"},
		{"деревням", "деревн"},
		{"дерево", "дерев"},
		{"деревом", "дерев"},
		{"дереву", "дерев"},
		{"деревушку", "деревушк"},
		{"деревцо", "деревц"},
		{"деревьев", "дерев"},
		{"деревья", "дерев"},
		{"деревьям", "дерев"},
		{"деревьями", "дерев"},
		{"деревьях", "дерев"},
		{"деревянная", "деревя"},
		{"деревянное", "деревя"},
		{"деревянной", "деревя"},
		{"деревянною", "деревя"},
		{"деревянную", "деревя"},
		{"деревянные", "деревя"},
		{"деревянный", "деревя"},
		{"деревянным", "деревя"},
		{"деревянными", "деревя"},
		{"деревянных", "деревя"},
		{"дерем", "дер"},
		{"дерет", "дерет"},
		{"держа", "держ"},
		{"державный", "державн"},
		{"держал", "держа"},
		{"держала", "держа"},
		{"держалась", "держа"},
		{"держали", "держа"},
		{"держалось", "держа"},
		{"держась", "держ"},
		{"держат", "держат"},
		{"держать", "держа"},
		{"держаться", "держа"},
		{"держи", "держ"},
		{"держим", "держ"},
		{"держись", "держ"},
		{"держит", "держ"},
		{"держите", "держ"},
		{"держится", "держ"},
		{"держишь", "держ"},
		{"держу", "держ"},
		{"дерзкие", "дерзк"},
		{"дерзкий", "дерзк"},
		{"дерзким", "дерзк"},
		{"дерзких", "дерзк"},
		{"дерзко", "дерзк"},
		{"дерзкого", "дерзк"},
		{"дерзкой", "дерзк"},
		{"дерзкую", "дерзк"},
		{"дерзок", "дерзок"},
		{"дерзостей", "дерзост"},
		{"дерзости", "дерзост"},
		{"дерзостно", "дерзостн"},
		{"дерзостные", "дерзостн"},
		{"дерзость", "дерзост"},
		{"дерзостью", "дерзост"},
		{"дери", "дер"},
		{"дериганов", "дериган"},
		{"дернет", "дернет"},
		{"дернул", "дернул"},
		{"дернула", "дернул"},
		{"дернули", "дернул"},
		{"дернуло", "дернул"},
		{"дернуть", "дернут"},
		{"дерутся", "дерут"},
		{"десертом", "десерт"},
		{"дескать", "деска"},
		{"десницей", "десниц"},
		{"деспот", "деспот"},
		{"деспотизм", "деспотизм"},
		{"деспотически", "деспотическ"},
		{"деспотическому", "деспотическ"},
		{"десяти", "десят"},
		{"десятилетнею", "десятилетн"},
		{"десятилетнюю", "десятилетн"},
		{"десятин", "десятин"},
		{"десятине", "десятин"},
		{"десятину", "десятин"},
		{"десятирублевую", "десятирублев"},
		{"десятирублевый", "десятирублев"},
		{"десятка", "десятк"},
		{"десятки", "десятк"},
		{"десятков", "десятк"},
		{"десятого", "десят"},
		{"десятое", "десят"},
		{"десятом", "десят"},
		{"десятую", "десят"},
		{"десятый", "десят"},
		{"десять", "деся"},
		{"детали", "дета"},
		{"детей", "дет"},
		{"дети", "дет"},
		{"детками", "детк"},
		{"детки", "детк"},
		{"деток", "деток"},
		{"деточек", "деточек"},
		{"деточка", "деточк"},
		{"деточки", "деточк"},
		{"детская", "детск"},
		{"детски", "детск"},
		{"детские", "детск"},
		{"детский", "детск"},
		{"детским", "детск"},
		{"детскими", "детск"},
		{"детских", "детск"},
		{"детского", "детск"},
		{"детское", "детск"},
		{"детской", "детск"},
		{"детском", "детск"},
		{"детскому", "детск"},
		{"детскою", "детск"},
		{"детскую", "детск"},
		{"детства", "детств"},
		{"детстве", "детств"},
		{"детство", "детств"},
		{"деть", "дет"},
		{"детьми", "детьм"},
		{"деться", "дет"},
		{"детям", "дет"},
		{"детях", "дет"},
		{"дешевле", "дешевл"},
		{"дешево", "дешев"},
		{"дешевую", "дешев"},
		{"дешевый", "дешев"},
		{"деятелен", "деятел"},
		{"деятели", "деятел"},
		{"деятельная", "деятельн"},
		{"деятельно", "деятельн"},
		{"деятельности", "деятельн"},
		{"деятельность", "деятельн"},
		{"деятельностью", "деятельн"},
		{"деятельною", "деятельн"},
		{"деятельные", "деятельн"},
		{"деятельным", "деятельн"},
		{"джентльмен", "джентльм"},
		{"джентльмена", "джентльм"},
		{"джентльменничанье", "джентльменничан"},
		{"джентльменски", "джентльменск"},
		{"джентльмены", "джентльм"},
		{"ди", "ди"},
		{"диалектики", "диалектик"},
		{"диапазона", "диапазон"},
		{"диван", "дива"},
		{"дивана", "дива"},
		{"диванам", "диван"},
		{"диване", "диван"},
		{"диванов", "диван"},
		{"диваном", "диван"},
		{"дивану", "диван"},
		{"диванчик", "диванчик"},
		{"диваны", "дива"},
		{"дивизионно", "дивизион"},
		{"дивились", "див"},
		{"дивился", "див"},
		{"дивись", "див"},
		{"дивитесь", "див"},
		{"дивиться", "див"},
		{"дивлюсь", "дивл"},
		{"дивная", "дивн"},
		{"дивно", "дивн"},
		{"дивное", "дивн"},
		{"дивную", "дивн"},
		{"дивные", "дивн"},
		{"дивный", "дивн"},
		{"дивным", "дивн"},
		{"диво", "див"},
		{"диву", "див"},
		{"дивясь", "див"},
		{"дикарь", "дикар"},
		{"дикая", "дик"},
		{"дикие", "дик"},
		{"дикий", "дик"},
		{"диким", "дик"},
		{"дико", "дик"},
		{"диковина", "диковин"},
		{"дикого", "дик"},
		{"дикое", "дик"},
		{"дикой", "дик"},
		{"диком", "дик"},
		{"дикости", "дикост"},
		{"дикость", "дикост"},
		{"дикою", "дик"},
		{"диктовала", "диктова"},
		{"диктовать", "диктова"},
		{"диктовку", "диктовк"},
		{"диктует", "дикт"},
		{"дикую", "дик"},
		{"дилетантизма", "дилетантизм"},
		{"дилижанс", "дилижанс"},
		{"динамит", "динам"},
		{"диораме", "диорам"},
		{"диплом", "дипл"},
		{"дипломат", "дипломат"},
		{"дипломатическим", "дипломатическ"},
		{"дипломаты", "дипломат"},
		{"директор", "директор"},
		{"директора", "директор"},
		{"директрисы", "директрис"},
		{"дирижера", "дирижер"},
		{"диск", "диск"},
		{"диссертацией", "диссертац"},
		{"диссонанс", "диссонанс"},
		{"дитюся", "дит"},
		{"дитя", "дит"},
		{"дифтерит", "дифтер"},
		{"дичей", "дич"},
		{"дичи", "дич"},
		{"дичившийся", "дич"},
		{"дичь", "дич"},
		{"диэта", "диэт"},
		{"дланью", "длан"},
		{"длилась", "длил"},
		{"длились", "длил"},
		{"длилось", "длил"},
		{"длился", "длил"},
		{"длинная", "длин"},
		{"длиннее", "длин"},
		{"длиннейшим", "длин"},
		{"длинно", "длин"},
		{"длинноволосый", "длинноволос"},
		{"длинного", "длин"},
		{"длинное", "длин"},
		{"длинной", "длин"},
		{"длинному", "длин"},
		{"длинною", "длин"},
		{"длинную", "длин"},
		{"длинны", "длин"},
		{"длинные", "длин"},
		{"длинный", "длин"},
		{"длинным", "длин"},
		{"длинными", "длин"},
		{"длинных", "длин"},
		{"длиной", "длин"},
		{"длину", "длин"},
		{"длится", "длит"},
		{"для", "для"},
		{"длятся", "длят"},
		{"дмитриев", "дмитр"},
		{"дмитрием", "дмитр"},
		{"дмитрий", "дмитр"},
		{"дмитрия", "дмитр"},
		{"дна", "дна"},
		{"дне", "дне"},
		{"дневная", "дневн"},
		{"дневник", "дневник"},
		{"дневника", "дневник"},
		{"дневного", "дневн"},
		{"дневное", "дневн"},
		{"дневной", "дневн"},
		{"дневном", "дневн"},
		{"дневному", "дневн"},
		{"дневных", "дневн"},
		{"дней", "дне"},
		{"днем", "днем"},
		{"дни", "дни"},
		{"днища", "днищ"},
		{"дно", "дно"},
		{"дном", "дном"},
		{"дню", "дню"},
		{"дня", "дня"},
		{"дням", "дням"},
		{"днями", "дням"},
		{"днях", "днях"},
		{"до", "до"},
		{"добавил", "добав"},
		{"добавила", "добав"},
		{"добежав", "добежа"},
		{"доберусь", "добер"},
		{"добиваешься", "добива"},
		{"добивай", "добива"},
		{"добивались", "добива"},
		{"добивался", "добива"},
		{"добиваться", "добива"},
		{"добиваются", "добива"},
		{"добилась", "доб"},
		{"добился", "доб"},
		{"добирается", "добира"},
		{"добирались", "добира"},
		{"добирался", "добира"},
		{"добираясь", "добир"},
		{"добиться", "доб"},
		{"доблести", "доблест"},
		{"доблестно", "доблестн"},
		{"доблестях", "доблест"},
		{"добр", "добр"},
		{"добра", "добр"},
		{"добрался", "добра"},
		{"добраться", "добра"},
		{"добрая", "добр"},
		{"добре", "добр"},
		{"добрее", "добр"},
		{"добрей", "добр"},
		{"добрейший", "добр"},
		{"добрели", "добрел"},
		{"добренький", "добреньк"},
		{"добреющий", "добреющ"},
		{"добро", "добр"},
		{"добровольно", "добровольн"},
		{"добровольную", "добровольн"},
		{"доброго", "добр"},
		{"добродетели", "добродетел"},
		{"добродетель", "добродетел"},
		{"добродетельна", "добродетельн"},
		{"добродетельная", "добродетельн"},
		{"добродетельно", "добродетельн"},
		{"добродетельного", "добродетельн"},
		{"добродетельной", "добродетельн"},
		{"добродетелям", "добродетел"},
		{"добродетелями", "добродетел"},
		{"добродушием", "добродуш"},
		{"добродушнейшим", "добродушн"},
		{"добродушно", "добродушн"},
		{"добродушного", "добродушн"},
		{"добродушное", "добродушн"},
		{"добродушным", "добродушн"},
		{"доброе", "добр"},
		{"доброй", "добр"},
		{"добролюбов", "добролюб"},
		{"добром", "добр"},
		{"доброму", "добр"},
		{"добросовестная", "добросовестн"},
		{"добросовестный", "добросовестн"},
		{"доброте", "доброт"},
		{"добротой", "доброт"},
		{"доброту", "доброт"},
		{"доброты", "доброт"},
		{"доброю", "добр"},
		{"добру", "добр"},
		{"добрую", "добр"},
		{"добры", "добр"},
		{"добрые", "добр"},
		{"добрый", "добр"},
		{"добрым", "добр"},
		{"добрыми", "добр"},
		{"добрыни", "добрын"},
		{"добрынин", "добрынин"},
		{"добрых", "добр"},
		{"добряк", "добряк"},
		{"добудет", "добудет"},
		{"добываемых", "добыва"},
		{"добываете", "добыва"},
		{"добывал", "добыва"},
		{"добывали", "добыва"},
		{"добывание", "добыван"},
		{"добыванья", "добыван"},
		{"добывать", "добыва"},
		{"добываю", "добыва"},
		{"добывают", "добыва"},
		{"добывающим", "добыва"},
		{"добывая", "добыв"},
		{"добыл", "доб"},
		{"добытые", "добыт"},
		{"добыть", "доб"},
		{"добычу", "добыч"},
		{"добьется", "добьет"},
		{"добьюсь", "доб"},
		{"добьют", "добьют"},
		{"довариваете", "доварива"},
		{"доварит", "довар"},
		{"доведенные", "доведен"},
		{"доведенных", "доведен"},
		{"доведены", "довед"},
		{"доведете", "доведет"},
		{"доведите", "довед"},
		{"доведу", "довед"},
		{"довез", "довез"},
		{"довезем", "довез"},
		{"довезу", "довез"},
		{"довел", "довел"},
		{"довела", "довел"},
		{"довели", "довел"},
		{"доверенности", "доверен"},
		{"доверенность", "доверен"},
		{"доверенностью", "доверен"},
		{"доверии", "довер"},
		{"доверилась", "довер"},
		{"доверителю", "доверител"},
		{"довериться", "довер"},
		{"доверия", "довер"},
		{"доверчив", "доверч"},
		{"доверчива", "доверчив"},
		{"доверчиво", "доверчив"},
		{"доверчивого", "доверчив"},
		{"доверчивой", "доверчив"},
		{"доверчивости", "доверчив"},
		{"доверчивость", "доверчив"},
		{"доверчивостью", "доверчив"},
		{"доверчивый", "доверчив"},
		{"довершению", "довершен"},
		{"доверши", "доверш"},
		{"довершили", "доверш"},
		{"доверьтесь", "доверьт"},
		{"доверявшего", "доверя"},
		{"доверял", "доверя"},
		{"доверялась", "доверя"},
		{"доверялся", "доверя"},
		{"доверясь", "довер"},
		{"доверяю", "доверя"},
		{"доверяя", "довер"},
		{"довести", "довест"},
		{"доводами", "довод"},
		{"доводил", "довод"},
		{"доводит", "довод"},
		{"доводить", "довод"},
		{"доводишь", "довод"},
		{"доводы", "довод"},
		{"доволен", "довол"},
		{"довольна", "довольн"},
		{"довольно", "довольн"},
		{"довольны", "довольн"},
		{"довольный", "довольн"},
		{"довольным", "довольн"},
		{"довольных", "довольн"},
		{"довольства", "довольств"},
		{"довольство", "довольств"},
		{"довольствовался", "довольствова"},
		{"довольством", "довольств"},
		{"довоспиталась", "довоспита"},
		{"доврались", "довра"},
		{"довремся", "довр"},
		{"догадавшись", "догада"},
		{"догадается", "догада"},
		{"догадалась", "догада"},
		{"догадался", "догада"},
		{"догадаться", "догада"},
		{"догадаются", "догада"},
		{"догадка", "догадк"},
		{"догадками", "догадк"},
		{"догадках", "догадк"},
		{"догадки", "догадк"},
		{"догадок", "догадок"},
		{"догадывалась", "догадыва"},
		{"догадывался", "догадыва"},
		{"догадываюсь", "догадыва"},
		{"догадываясь", "догадыв"},
		{"доглядит", "догляд"},
		{"догматы", "догмат"},
		{"догнав", "догна"},
		{"догнал", "догна"},
		{"догнала", "догна"},
		{"договариваешь", "договарива"},
		{"договаривала", "договарива"},
		{"договаривали", "договарива"},
		{"договаривать", "договарива"},
		{"договаривая", "договарив"},
		{"договорил", "договор"},
		{"договорила", "договор"},
		{"договорилась", "договор"},
		{"договорились", "договор"},
		{"договорился", "договор"},
		{"договорить", "договор"},
		{"догонишь", "догон"},
		{"догонял", "догоня"},
		{"догонять", "догоня"},
		{"догоняют", "догоня"},
		{"догоравшего", "догора"},
		{"догоравший", "догора"},
		{"догорает", "догора"},
		{"догорал", "догора"},
		{"догорала", "догора"},
		{"догоревшей", "догоревш"},
		{"догорел", "догорел"},
		{"догорим", "догор"},
		{"доделал", "додела"},
		{"додумалась", "додума"},
		{"додумался", "додума"},
		{"додумывался", "додумыва"},
		{"доедаем", "доеда"},
		{"доедали", "доеда"},
		{"доедешь", "доедеш"},
		{"доеду", "доед"},
		{"доел", "доел"},
		{"доели", "доел"},
		{"доехали", "доеха"},
		{"доехать", "доеха"},
		{"дожа", "дож"},
		{"дождавшись", "дожда"},
		{"дождалась", "дожда"},
		{"дождался", "дожда"},
		{"дождаться", "дожда"},
		{"дождевом", "дождев"},
		{"дождевою", "дождев"},
		{"дождевые", "дождев"},
		{"дождей", "дожд"},
		{"дождем", "дожд"},
		{"дождемся", "дожд"},
		{"дождется", "дождет"},
		{"дождешься", "дождеш"},
		{"дожди", "дожд"},
		{"дождик", "дождик"},
		{"дождичек", "дождичек"},
		{"дождливую", "дождлив"},
		{"дождливых", "дождлив"},
		{"дождь", "дожд"},
		{"дождю", "дожд"},
		{"дождя", "дожд"},
		{"дождями", "дожд"},
		{"дожей", "дож"},
		{"дожив", "дож"},
		{"доживает", "дожива"},
		{"доживал", "дожива"},
		{"доживать", "дожива"},
		{"доживающая", "дожива"},
		{"доживем", "дожив"},
		{"дожидается", "дожида"},
		{"дожидался", "дожида"},
		{"дожидаться", "дожида"},
		{"дожидаюсь", "дожида"},
		{"дожидаются", "дожида"},
		{"дожидаясь", "дожид"},
		{"дожил", "дож"},
		{"дожила", "дож"},
		{"дожить", "дож"},
		{"дозваться", "дозва"},
		{"дозволение", "дозволен"},
		{"дозволили", "дозвол"},
		{"дозвольте", "дозвольт"},
		{"дозволю", "дозвол"},
		{"дозволяется", "дозволя"},
		{"дозвонишься", "дозвон"},
		{"доискался", "доиска"},
		{"доискивался", "доискива"},
		{"доискиваться", "доискива"},
		{"доить", "до"},
		{"доищусь", "доищ"},
		{"дойдем", "дойд"},
		{"дойдет", "дойдет"},
		{"дойдешь", "дойдеш"},
		{"дойду", "дойд"},
		{"дойдут", "дойдут"},
		{"дойдя", "дойд"},
		{"доймут", "доймут"},
		{"дойти", "дойт"},
		{"дока", "док"},
		{"докажет", "докажет"},
		{"докажете", "докажет"},
		{"докажу", "докаж"},
		{"доказав", "доказа"},
		{"доказал", "доказа"},
		{"доказала", "доказа"},
		{"доказали", "доказа"},
		{"доказана", "доказа"},
		{"доказано", "доказа"},
		{"доказательств", "доказательств"},
		{"доказательства", "доказательств"},
		{"доказательство", "доказательств"},
		{"доказательством", "доказательств"},
		{"доказать", "доказа"},
		{"доказывает", "доказыва"},
		{"доказываете", "доказыва"},
		{"доказывается", "доказыва"},
		{"доказывал", "доказыва"},
		{"доказывали", "доказыва"},
		{"доказывалось", "доказыва"},
		{"доказывать", "доказыва"},
		{"докам", "док"},
		{"доканчивает", "доканчива"},
		{"доканчивай", "доканчива"},
		{"доклад", "доклад"},
		{"доклада", "доклад"},
		{"докладах", "доклад"},
		{"докладную", "докладн"},
		{"докладом", "доклад"},
		{"докладу", "доклад"},
		{"докладывает", "докладыва"},
		{"докладывал", "докладыва"},
		{"докладывать", "докладыва"},
		{"докликался", "доклика"},
		{"доколе", "докол"},
		{"доконал", "докона"},
		{"доконало", "докона"},
		{"докончив", "доконч"},
		{"докончил", "доконч"},
		{"докончить", "доконч"},
		{"доктор", "доктор"},
		{"доктора", "доктор"},
		{"доктором", "доктор"},
		{"докторской", "докторск"},
		{"доктору", "доктор"},
		{"доктрин", "доктрин"},
		{"документ", "документ"},
		{"документа", "документ"},
		{"документам", "документ"},
		{"документы", "документ"},
		{"докучливого", "докучлив"},
		{"докучливых", "докучлив"},
		{"докучной", "докучн"},
		{"докучным", "докучн"},
		{"дол", "дол"},
		{"долг", "долг"},
		{"долга", "долг"},
		{"долгая", "долг"},
		{"долге", "долг"},
		{"долги", "долг"},
		{"долгие", "долг"},
		{"долгий", "долг"},
		{"долгим", "долг"},
		{"долгих", "долг"},
		{"долго", "долг"},
		{"долгов", "долг"},
		{"долговой", "долгов"},
		{"долговом", "долгов"},
		{"долговременной", "долговремен"},
		{"долговременную", "долговремен"},
		{"долговязо", "долговяз"},
		{"долгого", "долг"},
		{"долгогривых", "долгогрив"},
		{"долгое", "долг"},
		{"долгожданный", "долгожда"},
		{"долгой", "долг"},
		{"долголетний", "долголетн"},
		{"долгом", "долг"},
		{"долгу", "долг"},
		{"доле", "дол"},
		{"долее", "дол"},
		{"долетал", "долета"},
		{"долетают", "долета"},
		{"долетели", "долетел"},
		{"должен", "долж"},
		{"должна", "должн"},
		{"должником", "должник"},
		{"должно", "должн"},
		{"должностей", "должност"},
		{"должности", "должност"},
		{"должность", "должност"},
		{"должны", "должн"},
		{"должным", "должн"},
		{"должок", "должок"},
		{"доли", "дол"},
		{"долин", "долин"},
		{"долинам", "долин"},
		{"долинах", "долин"},
		{"долине", "долин"},
		{"долинный", "долин"},
		{"долины", "долин"},
		{"долог", "долог"},
		{"доложил", "долож"},
		{"доложили", "долож"},
		{"доложился", "долож"},
		{"доложить", "долож"},
		{"долой", "дол"},
		{"долу", "дол"},
		{"долы", "дол"},
		{"дольнего", "дольн"},
		{"дольше", "дольш"},
		{"долю", "дол"},
		{"доля", "дол"},
		{"дом", "дом"},
		{"дома", "дом"},
		{"домам", "дом"},
		{"домами", "дом"},
		{"домах", "дом"},
		{"домашнего", "домашн"},
		{"домашнее", "домашн"},
		{"домашней", "домашн"},
		{"домашнем", "домашн"},
		{"домашнему", "домашн"},
		{"домашние", "домашн"},
		{"домашний", "домашн"},
		{"домашним", "домашн"},
		{"домашними", "домашн"},
		{"домашних", "домашн"},
		{"домашнюю", "домашн"},
		{"домашняя", "домашн"},
		{"доме", "дом"},
		{"домелю", "домел"},
		{"домик", "домик"},
		{"домика", "домик"},
		{"домики", "домик"},
		{"домов", "дом"},
		{"домовитой", "домовит"},
		{"домового", "домов"},
		{"домовой", "домов"},
		{"домовому", "домов"},
		{"домовый", "домов"},
		{"домой", "дом"},
		{"домом", "дом"},
		{"домоседа", "домосед"},
		{"домоседы", "домосед"},
		{"домохозяйка", "домохозяйк"},
		{"дому", "дом"},
		{"домчавшись", "домча"},
		{"домчался", "домча"},
		{"домчимся", "домч"},
		{"домы", "дом"},
		{"дон", "дон"},
		{"донельзя", "донельз"},
		{"донес", "донес"},
		{"донесем", "донес"},
		{"донесение", "донесен"},
		{"донесению", "донесен"},
		{"донесет", "донесет"},
		{"донеси", "донес"},
		{"донесла", "донесл"},
		{"донесли", "донесл"},
		{"донести", "донест"},
		{"донесу", "донес"},
		{"донжуаны", "донжуа"},
		{"донизу", "дониз"},
		{"донна", "дон"},
		{"доннерветтеров", "доннерветтер"},
		{"доном", "дон"},
		{"донос", "донос"},
		{"доноса", "донос"},
		{"доносилась", "донос"},
		{"доносились", "донос"},
		{"доносился", "донос"},
		{"доносится", "донос"},
		{"доносить", "донос"},
		{"доносу", "донос"},
		{"доносы", "донос"},
		{"доносятся", "донос"},
		{"доношу", "донош"},
		{"донского", "донск"},
		{"донцем", "донц"},
		{"доныне", "донын"},
		{"допечет", "допечет"},
		{"допивая", "допив"},
		{"допилась", "доп"},
		{"дописать", "дописа"},
		{"дописывает", "дописыва"},
		{"дописывать", "дописыва"},
		{"допитый", "допит"},
		{"доплыл", "допл"},
		{"доподлинно", "доподлин"},
		{"дополнения", "дополнен"},
		{"дополняет", "дополня"},
		{"дополняла", "дополня"},
		{"допотопных", "допотопн"},
		{"допрашиваемого", "допрашива"},
		{"допрашивает", "допрашива"},
		{"допрашивайся", "допрашива"},
		{"допрашивал", "допрашива"},
		{"допрашивала", "допрашива"},
		{"допрашивались", "допрашива"},
		{"допрашивать", "допрашива"},
		{"допрашивающих", "допрашива"},
		{"допрос", "допрос"},
		{"допроса", "допрос"},
		{"допросах", "допрос"},
		{"допросика", "допросик"},
		{"допросим", "допрос"},
		{"допросить", "допрос"},
		{"допроситься", "допрос"},
		{"допросишься", "допрос"},
		{"допросов", "допрос"},
		{"допросчика", "допросчик"},
		{"допросы", "допрос"},
		{"допрошу", "допрош"},
		{"допускает", "допуска"},
		{"допускаете", "допуска"},
		{"допускается", "допуска"},
		{"допускай", "допуска"},
		{"допускал", "допуска"},
		{"допускали", "допуска"},
		{"допускалось", "допуска"},
		{"допускать", "допуска"},
		{"допускаю", "допуска"},
		{"допустил", "допуст"},
		{"допустила", "допуст"},
		{"допустили", "допуст"},
		{"допустим", "допуст"},
		{"допустит", "допуст"},
		{"допустить", "допуст"},
		{"допущу", "допущ"},
		{"допытывались", "допытыва"},
		{"допьяна", "допья"},
		{"допятилась", "допят"},
		{"дорисовывать", "дорисовыва"},
		{"дорн", "дорн"},
		{"дорна", "дорн"},
		{"дорном", "дорн"},
		{"дорну", "дорн"},
		{"дорог", "дорог"},
		{"дорога", "дорог"},
		{"дорогам", "дорог"},
		{"дорогами", "дорог"},
		{"дорогах", "дорог"},
		{"дорогая", "дорог"},
		{"дороге", "дорог"},
		{"дороги", "дорог"},
		{"дорогие", "дорог"},
		{"дорогим", "дорог"},
		{"дорогих", "дорог"},
		{"дорого", "дор"},
		{"дороговизне", "дороговизн"},
		{"дорогое", "дорог"},
		{"дорогой", "дорог"},
		{"дорогом", "дорог"},
		{"дорогою", "дорог"},
		{"дорогу", "дорог"},
		{"дорогую", "дорог"},
		{"дородный", "дородн"},
		{"дорожа", "дорож"},
		{"дорожат", "дорожат"},
		{"дороже", "дорож"},
		{"дорожил", "дорож"},
		{"дорожили", "дорож"},
		{"дорожит", "дорож"},
		{"дорожите", "дорож"},
		{"дорожка", "дорожк"},
		{"дорожке", "дорожк"},
		{"дорожки", "дорожк"},
		{"дорожкой", "дорожк"},
		{"дорожку", "дорожк"},
		{"дорожная", "дорожн"},
		{"дорожное", "дорожн"},
		{"дорожной", "дорожн"},
		{"дорожном", "дорожн"},
		{"дорожному", "дорожн"},
		{"дорожные", "дорожн"},
		{"дорожный", "дорожн"},
		{"дорожу", "дорож"},
		{"досада", "досад"},
		{"досаде", "досад"},
		{"досадливо", "досадлив"},
		{"досадливого", "досадлив"},
		{"досадно", "досадн"},
		{"досадный", "досадн"},
		{"досадовал", "досадова"},
		{"досадой", "досад"},
		{"досаду", "досад"},
		{"досадую", "досад"},
		{"досадуя", "досаду"},
		{"досады", "досад"},
		{"досаждают", "досажда"},
		{"досаждающее", "досажда"},
		{"доселе", "досел"},
		{"доселева", "доселев"},
		{"доска", "доск"},
		{"досказал", "досказа"},
		{"досказала", "досказа"},
		{"досказано", "досказа"},
		{"досказать", "досказа"},
		{"доскам", "доск"},
		{"доскачет", "доскачет"},
		{"доске", "доск"},
		{"доски", "доск"},
		{"досконально", "доскональн"},
		{"доскою", "доск"},
		{"доску", "доск"},
		{"доследить", "дослед"},
		{"дослушал", "дослуша"},
		{"дослушала", "дослуша"},
		{"дослушали", "дослуша"},
		{"дослушивает", "дослушива"},
		{"дослушивать", "дослушива"},
		{"досмеивался", "досмеива"},
		{"досок", "досок"},
		{"доспех", "доспех"},
		{"доставай", "достава"},
		{"доставал", "достава"},
		{"доставала", "достава"},
		{"доставало", "достава"},
		{"доставалось", "достава"},
		{"доставался", "достава"},
		{"доставать", "достава"},
		{"доставаться", "достава"},
		{"доставил", "достав"},
		{"доставили", "достав"},
		{"доставит", "достав"},
		{"доставить", "достав"},
		{"доставляет", "доставля"},
		{"доставлял", "доставля"},
		{"доставляло", "доставля"},
		{"доставшейся", "доста"},
		{"доставшихся", "доста"},
		{"доставшуюся", "доста"},
		{"достает", "доста"},
		{"достается", "доста"},
		{"достал", "доста"},
		{"достала", "доста"},
		{"достали", "доста"},
		{"достались", "доста"},
		{"достало", "доста"},
		{"досталось", "доста"},
		{"достался", "доста"},
		{"достанет", "достанет"},
		{"достанется", "достанет"},
		{"достанешь", "достанеш"},
		{"достану", "достан"},
		{"достанут", "достанут"},
		{"достанутся", "достанут"},
		{"достань", "достан"},
		{"достатком", "достатк"},
		{"достаточная", "достаточн"},
		{"достаточно", "достаточн"},
		{"достаточном", "достаточн"},
		{"достаточными", "достаточн"},
		{"достать", "доста"},
		{"достают", "доста"},
		{"достаются", "доста"},
		{"достающеюся", "доста"},
		{"достиг", "достиг"},
		{"достигаешь", "достига"},
		{"достигал", "достига"},
		{"достигали", "достига"},
		{"достигало", "достига"},
		{"достигают", "достига"},
		{"достигли", "достигл"},
		{"достигло", "достигл"},
		{"достигнет", "достигнет"},
		{"достигнешь", "достигнеш"},
		{"достигну", "достигн"},
		{"достигнув", "достигнув"},
		{"достигнутого", "достигнут"},
		{"достигнуть", "достигнут"},
		{"достижении", "достижен"},
		{"достичь", "достич"},
		{"достодолжного", "достодолжн"},
		{"достоевский", "достоевск"},
		{"достоин", "достоин"},
		{"достоинств", "достоинств"},
		{"достоинства", "достоинств"},
		{"достоинствами", "достоинств"},
		{"достоинстве", "достоинств"},
		{"достоинство", "достоинств"},
		{"достоинством", "достоинств"},
		{"достоинству", "достоинств"},
		{"достойная", "достойн"},
		{"достойней", "достойн"},
		{"достойнейшей", "достойн"},
		{"достойно", "достойн"},
		{"достойное", "достойн"},
		{"достойной", "достойн"},
		{"достойный", "достойн"},
		{"достойным", "достойн"},
		{"достойных", "достойн"},
		{"достояньем", "достоян"},
		{"достроил", "достро"},
		{"достроился", "достро"},
		{"дострочу", "достроч"},
		{"доступ", "доступ"},
		{"доступна", "доступн"},
		{"доступной", "доступн"},
		{"доступном", "доступн"},
		{"доступны", "доступн"},
		{"доступных", "доступн"},
		{"достучался", "достуча"},
		{"досуги", "досуг"},
		{"досчитал", "досчита"},
		{"дотащился", "дотащ"},
		{"дотащиться", "дотащ"},
		{"дотла", "дотл"},
		{"дотоле", "дотол"},
		{"дотолкалась", "дотолка"},
		{"дотрогиваясь", "дотрогив"},
		{"дотронется", "дотронет"},
		{"дотронулся", "дотронул"},
		{"дотягивает", "дотягива"},
		{"дотянет", "дотянет"},
		{"дотянешь", "дотянеш"},
		{"дотянул", "дотянул"},
		{"дотянуть", "дотянут"},
		{"доучился", "доуч"},
		{"дохленьких", "дохленьк"},
		{"дохнет", "дохнет"},
		{"дохнул", "дохнул"},
		{"дохнула", "дохнул"},
		{"дохнут", "дохнут"},
		{"дохнуть", "дохнут"},
		{"доход", "доход"},
		{"дохода", "доход"},
		{"доходил", "доход"},
		{"доходила", "доход"},
		{"доходили", "доход"},
		{"доходило", "доход"},
		{"доходит", "доход"},
		{"доходов", "доход"},
		{"доходу", "доход"},
		{"доходцу", "доходц"},
		{"доходы", "доход"},
		{"доходя", "доход"},
		{"доходят", "доход"},
		{"доцента", "доцент"},
		{"дочел", "дочел"},
		{"дочерей", "дочер"},
		{"дочери", "дочер"},
		{"дочерью", "дочер"},
		{"дочерях", "дочер"},
		{"дочесть", "дочест"},
		{"дочиста", "дочист"},
		{"дочитав", "дочита"},
		{"дочитала", "дочита"},
		{"дочитать", "дочита"},
		{"дочитывал", "дочитыва"},
		{"дочитывали", "дочитыва"},
		{"дочитывать", "дочитыва"},
		{"дочка", "дочк"},
		{"дочке", "дочк"},
		{"дочки", "дочк"},
		{"дочку", "дочк"},
		{"дочла", "дочл"},
		{"дочли", "дочл"},
		{"дочь", "доч"},
		{"дошедшее", "дошедш"},
		{"дошедшие", "дошедш"},
		{"дошел", "дошел"},
		{"дошила", "дош"},
		{"дошла", "дошл"},
		{"дошли", "дошл"},
		{"дошло", "дошл"},
		{"дощечка", "дощечк"},
		{"дощечке", "дощечк"},
		{"дощечки", "дощечк"},
		{"дощечку", "дощечк"},
		{"драгоценнейшие", "драгоцен"},
		{"драгоценнейший", "драгоцен"},
		{"драгоценно", "драгоцен"},
		{"драгоценное", "драгоцен"},
		{"драгоценность", "драгоцен"},
		{"драгоценные", "драгоцен"},
		{"драгоценный", "драгоцен"},
		{"драгоценным", "драгоцен"},
		{"драгоценных", "драгоцен"},
		{"драгунские", "драгунск"},
		{"драдедамовой", "драдедамов"},
		{"драдедамовом", "драдедамов"},
		{"драдедамовый", "драдедамов"},
		{"дразни", "дразн"},
		{"дразнил", "дразн"},
		{"дразнила", "дразн"},
		{"дразнили", "дразн"},
		{"дразнит", "дразн"},
		{"дразните", "дразн"},
		{"дразнить", "дразн"},
		{"дразнишь", "дразн"},
		{"дразня", "дразн"},
		{"дразнят", "дразн"},
		{"драка", "драк"},
		{"драке", "драк"},
		{"драки", "драк"},
		{"дракон", "дракон"},
		{"драку", "драк"},
		{"драл", "драл"},
		{"драли", "драл"},
		{"дрались", "драл"},
		{"дрался", "драл"},
		{"драма", "драм"},
		{"драмы", "драм"},
		{"дранье", "дран"},
		{"драньем", "дран"},
		{"драпированы", "драпирова"},
		{"драпировка", "драпировк"},
		{"драпируется", "драпир"},
		{"драпируюсь", "драпир"},
		{"драпри", "драпр"},
		{"драть", "драт"},
		{"драться", "драт"},
		{"дребедень", "дребеден"},
		{"дребезжащему", "дребезжа"},
		{"дребезжащим", "дребезжа"},
		{"древ", "древ"},
		{"древен", "древ"},
		{"древесных", "древесн"},
		{"древко", "древк"},
		{"древнее", "древн"},
		{"древней", "древн"},
		{"древнейших", "древн"},
		{"древнем", "древн"},
		{"древни", "древн"},
		{"древние", "древн"},
		{"древний", "древн"},
		{"древних", "древн"},
		{"древности", "древност"},
		{"дремал", "дрема"},
		{"дремать", "дрема"},
		{"дремлет", "дремлет"},
		{"дремлется", "дремлет"},
		{"дремлешь", "дремлеш"},
		{"дремлю", "дремл"},
		{"дремлют", "дремлют"},
		{"дремлющей", "дремлющ"},
		{"дремлющим", "дремлющ"},
		{"дремлющими", "дремлющ"},
		{"дремлющую", "дремлющ"},
		{"дремные", "дремн"},
		{"дремота", "дремот"},
		{"дремоте", "дремот"},
		{"дремотные", "дремотн"},
		{"дремотой", "дремот"},
		{"дремоту", "дремот"},
		{"дремоты", "дремот"},
		{"дрему", "дрем"},
		{"дремучий", "дремуч"},
		{"дремучих", "дремуч"},
		{"дроби", "дроб"},
		{"дробил", "дроб"},
		{"дробились", "дроб"},
		{"дробный", "дробн"},
		{"дров", "дров"},
		{"дрова", "дров"},
		{"дровами", "дров"},
		{"дрогнет", "дрогнет"},
		{"дрогнувшие", "дрогнувш"},
		{"дрогнувших", "дрогнувш"},
		{"дрогнул", "дрогнул"},
		{"дрогнула", "дрогнул"},
		{"дрогнули", "дрогнул"},
		{"дрогнуло", "дрогнул"},
		{"дрожа", "дрож"},
		{"дрожавшее", "дрожа"},
		{"дрожавшей", "дрожа"},
		{"дрожавшею", "дрожа"},
		{"дрожавшие", "дрожа"},
		{"дрожавших", "дрожа"},
		{"дрожавшую", "дрожа"},
		{"дрожал", "дрожа"},
		{"дрожала", "дрожа"},
		{"дрожали", "дрожа"},
		{"дрожало", "дрожа"},
		{"дрожаньем", "дрожан"},
		{"дрожат", "дрожат"},
		{"дрожать", "дрожа"},
		{"дрожащая", "дрожа"},
		{"дрожащей", "дрожа"},
		{"дрожащею", "дрожа"},
		{"дрожащим", "дрожа"},
		{"дрожащими", "дрожа"},
		{"дрожащую", "дрожа"},
		{"дрожек", "дрожек"},
		{"дрожи", "дрож"},
		{"дрожит", "дрож"},
		{"дрожите", "дрож"},
		{"дрожишь", "дрож"},
		{"дрожки", "дрожк"},
		{"дрожу", "дрож"},
		{"дрожь", "дрож"},
		{"дрожью", "дрож"},
		{"друг", "друг"},
		{"друга", "друг"},
		{"другая", "друг"},
		{"друге", "друг"},
		{"другие", "друг"},
		{"другим", "друг"},
		{"другими", "друг"},
		{"других", "друг"},
		{"другого", "друг"},
		{"другое", "друг"},
		{"другой", "друг"},
		{"другом", "друг"},
		{"другому", "друг"},
		{"другою", "друг"},
		{"другу", "друг"},
		{"другую", "друг"},
		{"дружб", "дружб"},
		{"дружба", "дружб"},
		{"дружбе", "дружб"},
		{"дружбой", "дружб"},
		{"дружбу", "дружб"},
		{"дружбы", "дружб"},
		{"дружелюбно", "дружелюбн"},
		{"дружелюбной", "дружелюбн"},
		{"дружески", "дружеск"},
		{"дружеский", "дружеск"},
		{"дружеским", "дружеск"},
		{"дружескими", "дружеск"},
		{"дружеских", "дружеск"},
		{"дружеского", "дружеск"},
		{"дружеской", "дружеск"},
		{"дружескому", "дружеск"},
		{"дружескую", "дружеск"},
		{"дружества", "дружеств"},
		{"дружественные", "дружествен"},
		{"дружественным", "дружествен"},
		{"дружил", "друж"},
		{"дружился", "друж"},
		{"дружить", "друж"},
		{"дружище", "дружищ"},
		{"дружке", "дружк"},
		{"дружкой", "дружк"},
		{"дружна", "дружн"},
		{"дружно", "дружн"},
		{"дружной", "дружн"},
		{"дружном", "дружн"},
		{"дружную", "дружн"},
		{"дружочек", "дружочек"},
		{"друзей", "друз"},
		{"друзья", "друз"},
		{"друзьям", "друз"},
		{"друзьями", "друз"},
		{"дрыхнет", "дрыхнет"},
		{"дрыхнешь", "дрыхнеш"},
		{"дрыхни", "дрыхн"},
		{"дрыхнут", "дрыхнут"},
		{"дрыхнуть", "дрыхнут"},
		{"дряблое", "дрябл"},
		{"дряблый", "дрябл"},
		{"дряблым", "дрябл"},
		{"дряблыми", "дрябл"},
		{"дрязг", "дрязг"},
		{"дряни", "дрян"},
		{"дрянная", "дрян"},
		{"дряннейшего", "дрян"},
		{"дрянно", "дрян"},
		{"дрянного", "дрян"},
		{"дрянное", "дрян"},
		{"дрянной", "дрян"},
		{"дрянном", "дрян"},
		{"дрянные", "дрян"},
		{"дрянным", "дрян"},
		{"дрянных", "дрян"},
		{"дрянцо", "дрянц"},
		{"дрянь", "дрян"},
		{"дрянью", "дрян"},
		{"дряхлом", "дряхл"},
		{"дряхлую", "дряхл"},
		{"дряхлый", "дряхл"},
		{"ду", "ду"},
		{"дуб", "дуб"},
		{"дубе", "дуб"},
		{"дубняк", "дубняк"},
		{"дубняком", "дубняк"},
		{"дубовые", "дубов"},
		{"дубы", "дуб"},
		{"дубье", "дуб"},
		{"дуга", "дуг"},
		{"дуги", "дуг"},
		{"дугой", "дуг"},
		{"дугообразны", "дугообразн"},
		{"дует", "дует"},
		{"дуется", "дует"},
		{"дузе", "дуз"},
		{"дуклиду", "дуклид"},
		{"дум", "дум"},
		{"дума", "дум"},
		{"думаем", "дума"},
		{"думает", "дума"},
		{"думаете", "дума"},
		{"думаешь", "дума"},
		{"думай", "дума"},
		{"думайте", "дума"},
		{"думал", "дума"},
		{"думала", "дума"},
		{"думали", "дума"},
		{"думалось", "дума"},
		{"думам", "дум"},
		{"думать", "дума"},
		{"думах", "дум"},
		{"думаю", "дума"},
		{"думают", "дума"},
		{"думающих", "дума"},
		{"думая", "дум"},
		{"думу", "дум"},
		{"думы", "дум"},
		{"дуне", "дун"},
		{"дуней", "дун"},
		{"дунечка", "дунечк"},
		{"дунечке", "дунечк"},
		{"дунечки", "дунечк"},
		{"дунечкино", "дунечкин"},
		{"дунечкинова", "дунечкинов"},
		{"дунечкиной", "дунечкин"},
		{"дунечкины", "дунечкин"},
		{"дунечкиных", "дунечкин"},
		{"дунечкой", "дунечк"},
		{"дунечку", "дунечк"},
		{"дуни", "дун"},
		{"дунул", "дунул"},
		{"дунуло", "дунул"},
		{"дуню", "дун"},
		{"дуня", "дун"},
		{"дуняша", "дуняш"},
		{"дуняше", "дуняш"},
		{"дуняшей", "дуняш"},
		{"дуняши", "дуняш"},
		{"дуняшу", "дуняш"},
		{"дупелей", "дупел"},
		{"дуплет", "дуплет"},
		{"дуплетом", "дуплет"},
		{"дур", "дур"},
		{"дура", "дур"},
		{"дурак", "дурак"},
		{"дурака", "дурак"},
		{"дураками", "дурак"},
		{"дураки", "дурак"},
		{"дураков", "дурак"},
		{"дураком", "дурак"},
		{"дураку", "дурак"},
		{"дурацкая", "дурацк"},
		{"дурацки", "дурацк"},
		{"дурацкие", "дурацк"},
		{"дурачила", "дурач"},
		{"дурачится", "дурач"},
		{"дурачить", "дурач"},
		{"дурачке", "дурачк"},
		{"дурачки", "дурачк"},
		{"дурачок", "дурачок"},
		{"дуре", "дур"},
		{"дурен", "дур"},
		{"дурман", "дурма"},
		{"дурмана", "дурма"},
		{"дурмане", "дурман"},
		{"дурна", "дурн"},
		{"дурная", "дурн"},
		{"дурно", "дурн"},
		{"дурного", "дурн"},
		{"дурное", "дурн"},
		{"дурной", "дурн"},
		{"дурною", "дурн"},
		{"дурнушка", "дурнушк"},
		{"дурную", "дурн"},
		{"дурные", "дурн"},
		{"дурочке", "дурочк"},
		{"дуррак", "дуррак"},
		{"дуру", "дур"},
		{"дурында", "дурынд"},
		{"дурь", "дур"},
		{"дуся", "ду"},
		{"дуться", "дут"},
		{"дух", "дух"},
		{"духа", "дух"},
		{"духами", "дух"},
		{"духах", "дух"},
		{"духе", "дух"},
		{"духи", "дух"},
		{"духов", "дух"},
		{"духовник", "духовник"},
		{"духовно", "духовн"},
		{"духовному", "духовн"},
		{"духовным", "духовн"},
		{"духом", "дух"},
		{"духота", "духот"},
		{"духоте", "духот"},
		{"духоту", "духот"},
		{"духоты", "духот"},
		{"духу", "дух"},
		{"дучяша", "дучяш"},
		{"душ", "душ"},
		{"душа", "душ"},
		{"душами", "душ"},
		{"душат", "душат"},
		{"душе", "душ"},
		{"душевная", "душевн"},
		{"душевного", "душевн"},
		{"душевное", "душевн"},
		{"душевной", "душевн"},
		{"душевном", "душевн"},
		{"душевные", "душевн"},
		{"душевный", "душевн"},
		{"душевным", "душевн"},
		{"душевных", "душевн"},
		{"душегубец", "душегубец"},
		{"душегубства", "душегубств"},
		{"душегубцы", "душегубц"},
		{"душен", "душ"},
		{"душенька", "душеньк"},
		{"душеньку", "душеньк"},
		{"душечка", "душечк"},
		{"души", "душ"},
		{"душил", "душ"},
		{"душистая", "душист"},
		{"душистою", "душист"},
		{"душистым", "душист"},
		{"душистыми", "душист"},
		{"душит", "душ"},
		{"душите", "душ"},
		{"душить", "душ"},
		{"душкин", "душкин"},
		{"душкина", "душкин"},
		{"душкину", "душкин"},
		{"душна", "душн"},
		{"душно", "душн"},
		{"душное", "душн"},
		{"душный", "душн"},
		{"душным", "душн"},
		{"душных", "душн"},
		{"душой", "душ"},
		{"душонка", "душонк"},
		{"душою", "душ"},
		{"душу", "душ"},
		{"дуэли", "дуэл"},
		{"дуэль", "дуэл"},
		{"дуют", "дуют"},
		{"дщерь", "дщер"},
		{"дым", "дым"},
		{"дыма", "дым"},
		{"дыме", "дым"},
		{"дымилась", "дым"},
		{"дымился", "дым"},
		{"дымится", "дым"},
		{"дымка", "дымк"},
		{"дымно", "дымн"},
		{"дымной", "дымн"},
		{"дымные", "дымн"},
		{"дымный", "дымн"},
		{"дымных", "дымн"},
		{"дымом", "дым"},
		{"дыму", "дым"},
		{"дымят", "дым"},
		{"дымящимся", "дымя"},
		{"дыр", "дыр"},
		{"дыра", "дыр"},
		{"дырах", "дыр"},
		{"дыре", "дыр"},
		{"дырке", "дырк"},
		{"дырочки", "дырочк"},
		{"дыру", "дыр"},
		{"дыры", "дыр"},
		{"дырявая", "дыряв"},
		{"дырявые", "дыряв"},
		{"дырявый", "дыряв"},
		{"дырявым", "дыряв"},
		{"дыхание", "дыхан"},
		{"дыхания", "дыхан"},
		{"дыханье", "дыхан"},
		{"дыша", "дыш"},
		{"дышал", "дыша"},
		{"дышала", "дыша"},
		{"дышали", "дыша"},
		{"дышало", "дыша"},
		{"дышат", "дышат"},
		{"дышать", "дыша"},
		{"дыши", "дыш"},
		{"дышит", "дыш"},
		{"дышится", "дыш"},
		{"дышишь", "дыш"},
		{"дышлом", "дышл"},
		{"дышу", "дыш"},
		{"дьявол", "дьявол"},
		{"дьявола", "дьявол"},
		{"дьяволе", "дьявол"},
		{"дьяволом", "дьявол"},
		{"дьяволу", "дьявол"},
		{"дьявольская", "дьявольск"},
		{"дьявольский", "дьявольск"},
		{"дьякон", "дьякон"},
		{"дьячка", "дьячк"},
		{"дьячок", "дьячок"},
		{"дюжего", "дюж"},
		{"дюжина", "дюжин"},
		{"дюжинная", "дюжин"},
		{"дюжину", "дюжин"},
		{"дюжины", "дюжин"},
		{"дюссотов", "дюссот"},
		{"дяде", "дяд"},
		{"дядей", "дяд"},
		{"дядек", "дядек"},
		{"дядечка", "дядечк"},
		{"дяди", "дяд"},
		{"дядин", "дядин"},
		{"дядьки", "дядьк"},
		{"дядьшка", "дядьшк"},
		{"дядьшки", "дядьшк"},
		{"дядю", "дяд"},
		{"дядя", "дяд"},
		{"е", "е"},
		{"евангелие", "евангел"},
		{"евангелии", "евангел"},
		{"евангелия", "евангел"},
		{"евг", "евг"},
		{"евгений", "евген"},
		{"евгению", "евген"},
		{"евнух", "евнух"},
		{"еврей", "евр"},
		{"еврейский", "еврейск"},
		{"еврейского", "еврейск"},
		{"евреями", "евре"},
		{"европе", "европ"},
		{"европейские", "европейск"},
		{"европою", "европ"},
		{"европу", "европ"},
		{"евстигней", "евстигн"},
		{"евстигнея", "евстигне"},
		{"евших", "евш"},
		{"египет", "египет"},
		{"египетские", "египетск"},
		{"египетскими", "египетск"},
		{"египетского", "египетск"},
		{"египетскую", "египетск"},
		{"египта", "египт"},
		{"египте", "египт"},
		{"его", "ег"},
		{"егоровна", "егоровн"},
		{"егору", "егор"},
		{"еда", "ед"},
		{"едва", "едв"},
		{"еде", "ед"},
		{"едем", "ед"},
		{"едемте", "едемт"},
		{"едет", "едет"},
		{"едете", "едет"},
		{"едешь", "едеш"},
		{"единичное", "единичн"},
		{"единичных", "единичн"},
		{"единоверцев", "единоверц"},
		{"единовременного", "единовремен"},
		{"единогласно", "единогласн"},
		{"единогласных", "единогласн"},
		{"единого", "един"},
		{"единой", "един"},
		{"единому", "един"},
		{"единородна", "единородн"},
		{"единородную", "единородн"},
		{"единства", "единств"},
		{"единственная", "единствен"},
		{"единственно", "единствен"},
		{"единственное", "единствен"},
		{"единственной", "единствен"},
		{"единственному", "единствен"},
		{"единственною", "единствен"},
		{"единственную", "единствен"},
		{"единственные", "единствен"},
		{"единственный", "единствен"},
		{"единственным", "единствен"},
		{"единый", "един"},
		{"единым", "един"},
		{"едите", "ед"},
		{"едко", "едк"},
		{"едкого", "едк"},
		{"едкой", "едк"},
		{"едкою", "едк"},
		{"еду", "ед"},
		{"едут", "едут"},
		{"едучи", "едуч"},
		{"едущая", "едущ"},
		{"едят", "ед"},
		// {"ее", "е"},
		{"ежегодно", "ежегодн"},
		{"ежедневная", "ежедневн"},
		{"ежедневно", "ежедневн"},
		{"ежедневного", "ежедневн"},
		{"ежедневное", "ежедневн"},
		{"ежедневной", "ежедневн"},
		{"ежедневном", "ежедневн"},
		{"ежедневный", "ежедневн"},
		{"ежедневным", "ежедневн"},
		{"ежедневных", "ежедневн"},
		{"ежели", "ежел"},
		{"ежемесячного", "ежемесячн"},
		{"ежеминутно", "ежеминутн"},
		{"ежеминутным", "ежеминутн"},
		{"ежеминутных", "ежеминутн"},
		{"еженедельная", "еженедельн"},
		{"еженочной", "еженочн"},
		{"ежечасные", "ежечасн"},
		{"езди", "езд"},
		{"ездивший", "езд"},
		{"ездил", "езд"},
		{"ездила", "езд"},
		{"ездили", "езд"},
		{"ездит", "езд"},
		{"ездите", "езд"},
		{"ездить", "езд"},
		{"ездиют", "ездиют"},
		{"ездой", "езд"},
		{"езды", "езд"},
		{"ездят", "езд"},
		{"езжал", "езжа"},
		{"езжали", "езжа"},
		{"езжу", "езж"},
		// {"ей", "е"},
		{"екатерингоф", "екатерингоф"},
		{"екатерингофе", "екатерингоф"},
		{"екатеринина", "екатеринин"},
		{"екатерининского", "екатерининск"},
		{"ектеньи", "ектен"},
		{"ел", "ел"},
		{"ела", "ел"},
		{"елагин", "елагин"},
		{"еле", "ел"},
		{"елей", "ел"},
		{"елена", "ел"},
		{"елене", "елен"},
		{"елену", "елен"},
		{"елены", "ел"},
		{"елец", "елец"},
		{"ели", "ел"},
		{"елисаветграде", "елисаветград"},
		{"елисейские", "елисейск"},
		{"елка", "елк"},
		{"елку", "елк"},
		{"еловая", "елов"},
		{"еловую", "елов"},
		{"ель", "ел"},
		{"ельце", "ельц"},
		{"ем", "ем"},
		{"емеле", "емел"},
		{"ему", "ем"},
		{"енотовых", "енотов"},
		{"епиходов", "епиход"},
		{"епиходова", "епиходов"},
		{"епиходову", "епиходов"},
		{"епоходов", "епоход"},
		{"ерани", "еран"},
		{"ерань", "еран"},
		{"еранью", "еран"},
		{"ермак", "ермак"},
		{"ермака", "ермак"},
		{"ермолай", "ермола"},
		{"ермолках", "ермолк"},
		{"ермолке", "ермолк"},
		{"ерошил", "ерош"},
		{"ерунду", "ерунд"},
		{"еруслан", "ерусла"},
		{"ерша", "ерш"},
		{"ершей", "ерш"},
		{"ершика", "ершик"},
		{"если", "есл"},
		{"есмь", "есм"},
		{"ест", "ест"},
		{"естественная", "естествен"},
		{"естественно", "естествен"},
		{"естественного", "естествен"},
		{"естественное", "естествен"},
		{"естественной", "естествен"},
		{"естественном", "естествен"},
		{"естественному", "естествен"},
		{"естественнонаучные", "естественнонаучн"},
		{"естественности", "естествен"},
		{"естественною", "естествен"},
		{"естественную", "естествен"},
		{"естественные", "естествен"},
		{"естественный", "естествен"},
		{"естественным", "естествен"},
		{"естественных", "естествен"},
		{"есть", "ест"},
		{"есь", "е"},
		{"ет", "ет"},
		{"ефим", "еф"},
		{"ефимьюшка", "ефимьюшк"},
		{"ефтова", "ефтов"},
		{"ефтом", "ефт"},
		{"ефту", "ефт"},
		{"ехав", "еха"},
		{"ехавшему", "еха"},
		{"ехал", "еха"},
		{"ехала", "еха"},
		{"ехали", "еха"},
		{"ехать", "еха"},
		{"ешь", "еш"},
		{"ешьте", "ешьт"},
		{"еще", "ещ"},
		// {"ею", "е"},
		{"ж", "ж"},
		{"жаба", "жаб"},
		{"жаворонка", "жаворонк"},
		{"жаворонков", "жаворонк"},
		{"жадная", "жадн"},
		{"жадно", "жадн"},
		{"жадности", "жадност"},
		{"жадностию", "жадност"},
		{"жадностью", "жадност"},
		{"жадные", "жадн"},
		{"жадный", "жадн"},
		{"жадным", "жадн"},
		{"жадными", "жадн"},
		{"жадных", "жадн"},
		{"жажда", "жажд"},
		{"жаждал", "жажда"},
		{"жаждало", "жажда"},
		{"жаждет", "жаждет"},
		{"жаждой", "жажд"},
		{"жажду", "жажд"},
		{"жаждущего", "жаждущ"},
		{"жаждущей", "жаждущ"},
		{"жаждущим", "жаждущ"},
		{"жаждущую", "жаждущ"},
		{"жажды", "жажд"},
		{"жакетку", "жакетк"},
		{"жал", "жал"},
		{"жала", "жал"},
		{"жалась", "жал"},
		{"жалей", "жал"},
		{"жалела", "жалел"},
		{"жалели", "жалел"},
		{"жалеть", "жалет"},
		{"жалею", "жал"},
		{"жалеют", "жалеют"},
		{"жалея", "жале"},
		{"жали", "жал"},
		{"жались", "жал"},
		{"жалиться", "жал"},
		{"жалка", "жалк"},
		{"жалкая", "жалк"},
		{"жалкие", "жалк"},
		{"жалкий", "жалк"},
		{"жалким", "жалк"},
		{"жалкими", "жалк"},
		{"жалких", "жалк"},
		{"жалко", "жалк"},
		{"жалкого", "жалк"},
		{"жалкое", "жалк"},
		{"жалкой", "жалк"},
		{"жалком", "жалк"},
		{"жало", "жал"},
		{"жалоб", "жалоб"},
		{"жалобились", "жалоб"},
		{"жалобная", "жалобн"},
		{"жалобно", "жалобн"},
		{"жалобный", "жалобн"},
		{"жалобным", "жалобн"},
		{"жалобы", "жалоб"},
		{"жаловалась", "жалова"},
		{"жаловались", "жалова"},
		{"жаловался", "жалова"},
		{"жалование", "жалован"},
		{"жалованье", "жалован"},
		{"жалованья", "жалован"},
		{"жаловаться", "жалова"},
		{"жалок", "жалок"},
		{"жалости", "жалост"},
		{"жалость", "жалост"},
		{"жалостью", "жалост"},
		{"жался", "жал"},
		{"жалуемся", "жалу"},
		{"жалуетесь", "жалует"},
		{"жалуется", "жал"},
		{"жалуйся", "жал"},
		{"жалуюсь", "жал"},
		{"жаль", "жал"},
		{"жан", "жан"},
		{"жандарма", "жандарм"},
		{"жандармы", "жандарм"},
		{"жанлис", "жанлис"},
		{"жар", "жар"},
		{"жара", "жар"},
		{"жаргон", "жаргон"},
		{"жаре", "жар"},
		{"жареная", "жарен"},
		{"жареной", "жарен"},
		{"жарила", "жар"},
		{"жарится", "жар"},
		{"жарища", "жарищ"},
		{"жаркая", "жарк"},
		{"жаркие", "жарк"},
		{"жаркий", "жарк"},
		{"жарким", "жарк"},
		{"жарких", "жарк"},
		{"жарко", "жарк"},
		{"жаркого", "жарк"},
		{"жаркое", "жарк"},
		{"жаркой", "жарк"},
		{"жаркую", "жарк"},
		{"жарок", "жарок"},
		{"жаром", "жар"},
		{"жару", "жар"},
		{"жарче", "жарч"},
		{"жги", "жги"},
		{"жгла", "жгла"},
		{"жгли", "жгли"},
		{"жгло", "жгло"},
		{"жгут", "жгут"},
		{"жгучая", "жгуч"},
		{"жгуче", "жгуч"},
		{"жгучего", "жгуч"},
		{"жгучее", "жгуч"},
		{"жгучие", "жгуч"},
		{"жгучий", "жгуч"},
		{"жгучим", "жгуч"},
		{"жгучих", "жгуч"},
		{"жгучую", "жгуч"},
		{"ждал", "ждал"},
		{"ждала", "ждал"},
		{"ждали", "ждал"},
		{"ждало", "ждал"},
		{"ждать", "ждат"},
		{"ждем", "ждем"},
		{"ждет", "ждет"},
		{"ждете", "ждет"},
		{"ждешь", "ждеш"},
		{"жди", "жди"},
		{"ждите", "ждит"},
		{"жду", "жду"},
		{"ждут", "ждут"},
		{"ждущие", "ждущ"},
		{"ждущую", "ждущ"},
		{"жды", "жды"},
		{"же", "же"},
		{"жевал", "жева"},
		{"жеваное", "жеван"},
		{"жевать", "жева"},
		{"жег", "жег"},
		{"жезл", "жезл"},
		{"желаем", "жела"},
		{"желаемого", "жела"},
		{"желаемое", "жела"},
		{"желает", "жела"},
		{"желаете", "жела"},
		{"желаешь", "жела"},
		{"желай", "жела"},
		{"желал", "жела"},
		{"желала", "жела"},
		{"желали", "жела"},
		{"желаль", "желал"},
		{"желание", "желан"},
		{"желанием", "желан"},
		{"желании", "желан"},
		{"желаний", "желан"},
		{"желанию", "желан"},
		{"желания", "желан"},
		{"желаниям", "желан"},
		{"желанная", "жела"},
		{"желанное", "жела"},
		{"желанным", "жела"},
		{"желанье", "желан"},
		{"желанья", "желан"},
		{"желательно", "желательн"},
		{"желать", "жела"},
		{"желаю", "жела"},
		{"желающего", "жела"},
		{"желая", "жел"},
		{"железа", "желез"},
		{"железная", "железн"},
		{"железней", "железн"},
		{"железнодорожных", "железнодорожн"},
		{"железной", "железн"},
		{"железном", "железн"},
		{"железною", "железн"},
		{"железную", "железн"},
		{"железные", "железн"},
		{"железный", "железн"},
		{"железным", "железн"},
		{"железными", "железн"},
		{"железо", "желез"},
		{"железом", "желез"},
		{"желоб", "желоб"},
		{"желобом", "желоб"},
		{"желт", "желт"},
		{"желтенький", "желтеньк"},
		{"желтенькими", "желтеньк"},
		{"желтеют", "желтеют"},
		{"желтизной", "желтизн"},
		{"желтизны", "желтизн"},
		{"желтоватая", "желтоват"},
		{"желтовато", "желтоват"},
		{"желтоватой", "желтоват"},
		{"желтоватые", "желтоват"},
		{"желтого", "желт"},
		{"желтое", "желт"},
		{"желтой", "желт"},
		{"желтом", "желт"},
		{"желтому", "желт"},
		{"желтою", "желт"},
		{"желтые", "желт"},
		{"желтый", "желт"},
		{"желтым", "желт"},
		{"желтыми", "желт"},
		{"желтых", "желт"},
		{"желудком", "желудк"},
		{"желудок", "желудок"},
		{"желудочка", "желудочк"},
		{"желчен", "желч"},
		{"желчи", "желч"},
		{"желчная", "желчн"},
		{"желчно", "желчн"},
		{"желчное", "желчн"},
		{"желчный", "желчн"},
		{"желчь", "желч"},
		{"жеманства", "жеманств"},
		{"жемчуг", "жемчуг"},
		{"жемчуга", "жемчуг"},
		{"жемчугами", "жемчуг"},
		{"жемчуге", "жемчуг"},
		{"жемчугом", "жемчуг"},
		{"жемчугу", "жемчуг"},
		{"жемчужин", "жемчужин"},
		{"жемчужина", "жемчужин"},
		{"жемчужной", "жемчужн"},
		{"жемчужный", "жемчужн"},
		{"жен", "жен"},
		{"жена", "жен"},
		{"женам", "жен"},
		{"женами", "жен"},
		{"женат", "женат"},
		{"женаты", "женат"},
		{"жене", "жен"},
		{"женив", "жен"},
		{"женился", "жен"},
		{"женись", "жен"},
		{"женитесь", "жен"},
		{"женится", "жен"},
		{"женить", "жен"},
		{"женитьба", "женитьб"},
		{"женитьбе", "женитьб"},
		{"женитьбу", "женитьб"},
		{"женитьбы", "женитьб"},
		{"жениться", "жен"},
		{"жених", "жен"},
		{"жениха", "жених"},
		{"женихами", "жених"},
		{"женихи", "жених"},
		{"женихов", "жених"},
		{"жениховское", "жениховск"},
		{"женихом", "жених"},
		{"жениху", "жених"},
		{"женишка", "женишк"},
		{"женишок", "женишок"},
		{"женишь", "жен"},
		{"женишься", "жен"},
		{"женой", "жен"},
		{"женская", "женск"},
		{"женски", "женск"},
		{"женские", "женск"},
		{"женский", "женск"},
		{"женских", "женск"},
		{"женского", "женск"},
		{"женское", "женск"},
		{"женской", "женск"},
		{"женскому", "женск"},
		{"женскою", "женск"},
		{"женскую", "женск"},
		{"женственной", "женствен"},
		{"жену", "жен"},
		{"женщин", "женщин"},
		{"женщина", "женщин"},
		{"женщинам", "женщин"},
		{"женщинами", "женщин"},
		{"женщинах", "женщин"},
		{"женщине", "женщин"},
		{"женщиной", "женщин"},
		{"женщиною", "женщин"},
		{"женщину", "женщин"},
		{"женщины", "женщин"},
		{"жены", "жен"},
		{"женюсь", "жен"},
		{"женят", "жен"},
		{"женятся", "жен"},
		{"жерди", "жерд"},
		{"жердями", "жерд"},
		{"жернова", "жернов"},
		{"жертв", "жертв"},
		{"жертва", "жертв"},
		{"жертвами", "жертв"},
		{"жертвах", "жертв"},
		{"жертвовать", "жертвова"},
		{"жертвой", "жертв"},
		{"жертвочка", "жертвочк"},
		{"жертву", "жертв"},
		{"жертвует", "жертв"},
		{"жертвую", "жертв"},
		{"жертвы", "жертв"},
		{"жест", "жест"},
		{"жеста", "жест"},
		{"жестами", "жест"},
		{"жесте", "жест"},
		{"жести", "жест"},
		{"жестки", "жестк"},
		{"жесткие", "жестк"},
		{"жесткий", "жестк"},
		{"жестким", "жестк"},
		{"жесткими", "жестк"},
		{"жестко", "жестк"},
		{"жесткой", "жестк"},
		{"жесткости", "жесткост"},
		{"жесткость", "жесткост"},
		{"жесткую", "жестк"},
		{"жесток", "жесток"},
		{"жестокая", "жесток"},
		{"жестокие", "жесток"},
		{"жестокий", "жесток"},
		{"жестоких", "жесток"},
		{"жестоко", "жесток"},
		{"жестокой", "жесток"},
		{"жестоком", "жесток"},
		{"жестокости", "жесток"},
		{"жестокость", "жесток"},
		{"жестом", "жест"},
		{"жесту", "жест"},
		{"жесты", "жест"},
		{"жестяной", "жестян"},
		{"жестяном", "жестян"},
		{"жетон", "жетон"},
		{"жечь", "жеч"},
		{"жженкой", "жженк"},
		{"жжет", "жжет"},
		{"жив", "жив"},
		{"жива", "жив"},
		{"живая", "жив"},
		{"живее", "жив"},
		{"живей", "жив"},
		{"живем", "жив"},
		{"живет", "живет"},
		{"живете", "живет"},
		{"живется", "живет"},
		{"живешь", "живеш"},
		{"живи", "жив"},
		{"живите", "жив"},
		{"живишь", "жив"},
		{"живности", "живност"},
		{"живо", "жив"},
		{"живого", "жив"},
		{"живое", "жив"},
		{"живой", "жив"},
		{"живом", "жив"},
		{"живописец", "живописец"},
		{"живописи", "живопис"},
		{"живописно", "живописн"},
		{"живописной", "живописн"},
		{"живописных", "живописн"},
		{"живописцу", "живописц"},
		{"живопись", "живоп"},
		{"живописью", "живопис"},
		{"живости", "живост"},
		{"живость", "живост"},
		{"живостью", "живост"},
		{"животного", "животн"},
		{"животное", "животн"},
		{"животной", "животн"},
		{"животный", "животн"},
		{"животными", "животн"},
		{"животных", "животн"},
		{"животрепещущей", "животрепещущ"},
		{"животрепещущею", "животрепещущ"},
		{"живу", "жив"},
		{"живут", "живут"},
		{"живуча", "живуч"},
		{"живучи", "живуч"},
		{"живущего", "живущ"},
		{"живущей", "живущ"},
		{"живущий", "живущ"},
		{"живущими", "живущ"},
		{"живущих", "живущ"},
		{"живущую", "живущ"},
		{"живую", "жив"},
		{"жившим", "живш"},
		{"живших", "живш"},
		{"живы", "жив"},
		{"живые", "жив"},
		{"живым", "жив"},
		{"живыми", "жив"},
		{"живых", "жив"},
		{"живьем", "жив"},
		{"живя", "жив"},
		{"жид", "жид"},
		{"жиденькие", "жиденьк"},
		{"жиденький", "жиденьк"},
		{"жиденькой", "жиденьк"},
		{"жидким", "жидк"},
		{"жиду", "жид"},
		{"жиды", "жид"},
		{"жизней", "жизн"},
		{"жизненная", "жизнен"},
		{"жизненного", "жизнен"},
		{"жизненное", "жизнен"},
		{"жизненной", "жизнен"},
		{"жизненную", "жизнен"},
		{"жизненные", "жизнен"},
		{"жизненный", "жизнен"},
		{"жизненных", "жизнен"},
		{"жизни", "жизн"},
		{"жизнию", "жизн"},
		{"жизнь", "жизн"},
		{"жизнью", "жизн"},
		{"жил", "жил"},
		{"жила", "жил"},
		{"жилам", "жил"},
		{"жилах", "жил"},
		{"жилет", "жилет"},
		{"жилета", "жилет"},
		{"жилете", "жилет"},
		{"жилетка", "жилетк"},
		{"жилетке", "жилетк"},
		{"жилетку", "жилетк"},
		{"жилетный", "жилетн"},
		{"жилету", "жилет"},
		{"жилец", "жилец"},
		{"жили", "жил"},
		{"жилица", "жилиц"},
		{"жилицу", "жилиц"},
		{"жилища", "жилищ"},
		{"жилище", "жилищ"},
		{"жилками", "жилк"},
		{"жилку", "жилк"},
		{"жило", "жил"},
		{"жилы", "жил"},
		{"жилье", "жил"},
		{"жильца", "жильц"},
		{"жильцам", "жильц"},
		{"жильцах", "жильц"},
		{"жильцов", "жильц"},
		{"жильцом", "жильц"},
		{"жильцу", "жильц"},
		{"жильцы", "жильц"},
		{"жилья", "жил"},
		{"жир", "жир"},
		{"жирна", "жирн"},
		{"жирно", "жирн"},
		{"жирного", "жирн"},
		{"жирное", "жирн"},
		{"жирной", "жирн"},
		{"жирные", "жирн"},
		{"жирный", "жирн"},
		{"жирных", "жирн"},
		{"жирок", "жирок"},
		{"жиром", "жир"},
		{"жиру", "жир"},
		{"житейские", "житейск"},
		{"житейских", "житейск"},
		{"житейского", "житейск"},
		{"житейское", "житейск"},
		{"житейской", "житейск"},
		{"жителей", "жител"},
		{"жители", "жител"},
		{"жительства", "жительств"},
		{"жительстве", "жительств"},
		{"жить", "жит"},
		{"житье", "жит"},
		{"житья", "жит"},
		{"жмет", "жмет"},
		{"жмется", "жмет"},
		{"жмешь", "жмеш"},
		{"жмут", "жмут"},
		{"жнивью", "жнив"},
		{"жнитво", "жнитв"},
		{"жнут", "жнут"},
		{"жорж", "жорж"},
		{"жоржа", "жорж"},
		{"жребий", "жреб"},
		{"жребия", "жреб"},
		{"жрецы", "жрец"},
		{"жрешь", "жреш"},
		{"жуан", "жуа"},
		{"жуаном", "жуан"},
		{"жувеневских", "жувеневск"},
		{"жужжала", "жужжа"},
		{"жужжаньем", "жужжан"},
		{"жужжат", "жужжат"},
		{"жужжащий", "жужжа"},
		{"жужжит", "жужж"},
		{"жужжите", "жужж"},
		{"жук", "жук"},
		{"жуков", "жук"},
		{"жуковский", "жуковск"},
		{"журавли", "журавл"},
		{"журавлиным", "журавлин"},
		{"журавля", "журавл"},
		{"журил", "жур"},
		{"журнал", "журна"},
		{"журнала", "журна"},
		{"журнале", "журнал"},
		{"журналистом", "журналист"},
		{"журналов", "журнал"},
		{"журналы", "журнал"},
		{"журнальной", "журнальн"},
		{"журнальце", "журнальц"},
		{"журчанье", "журчан"},
		{"журчаньем", "журчан"},
		{"журчит", "журч"},
		{"жутко", "жутк"},
		{"жуткость", "жуткост"},
		{"жучка", "жучк"},
		{"жующие", "жующ"},
		{"з", "з"},
		{"за", "за"},
		{"заалели", "заалел"},
		{"заалелся", "заалел"},
		{"заарестовать", "заарестова"},
		{"забавах", "забав"},
		{"забавляет", "забавля"},
		{"забавляло", "забавля"},
		{"забавляюсь", "забавля"},
		{"забавник", "забавник"},
		{"забавника", "забавник"},
		{"забавный", "забавн"},
		{"забаву", "забав"},
		{"забалканский", "забалканск"},
		{"забастовки", "забастовк"},
		{"забвение", "забвен"},
		{"забвению", "забвен"},
		{"забвенье", "забвен"},
		{"забвенья", "забвен"},
		{"забегает", "забега"},
		{"забегаешь", "забега"},
		{"забегал", "забега"},
		{"забегала", "забега"},
		{"забегать", "забега"},
		{"забегаю", "забега"},
		{"забегают", "забега"},
		{"забегая", "забег"},
		{"забегу", "забег"},
		{"забедневшее", "забедневш"},
		{"забежали", "забежа"},
		{"забежать", "забежа"},
		{"забежит", "забеж"},
		{"забелеет", "забелеет"},
		{"забелелась", "забелел"},
		{"заберется", "заберет"},
		{"заберите", "забер"},
		{"заберут", "заберут"},
		{"забилась", "заб"},
		{"забился", "заб"},
		{"забирает", "забира"},
		{"забирается", "забира"},
		{"забирали", "забира"},
		{"забирался", "забира"},
		{"забираясь", "забир"},
		{"забит", "заб"},
		{"забита", "забит"},
		{"забитым", "забит"},
		{"забитых", "забит"},
		{"забить", "заб"},
		{"забияки", "забияк"},
		{"заблаговременно", "заблаговремен"},
		{"заблагорассудила", "заблагорассуд"},
		{"заблагорассудилось", "заблагорассуд"},
		{"заблагорассудит", "заблагорассуд"},
		{"заблестят", "заблест"},
		{"заблистала", "заблиста"},
		{"заблистали", "заблиста"},
		{"заблудился", "заблуд"},
		{"заблудитесь", "заблуд"},
		{"заблуждение", "заблужден"},
		{"заблуждении", "заблужден"},
		{"заболевала", "заболева"},
		{"заболеваю", "заболева"},
		{"заболеете", "заболеет"},
		{"заболел", "заболел"},
		{"заболела", "заболел"},
		{"заболею", "забол"},
		{"заболтавшегося", "заболта"},
		{"заболтался", "заболта"},
		{"забор", "забор"},
		{"забора", "забор"},
		{"заборами", "забор"},
		{"заборе", "забор"},
		{"заборов", "забор"},
		{"забором", "забор"},
		{"заборы", "забор"},
		{"забот", "забот"},
		{"забота", "забот"},
		{"заботам", "забот"},
		{"заботами", "забот"},
		{"заботах", "забот"},
		{"заботе", "забот"},
		{"заботились", "забот"},
		{"заботилось", "забот"},
		{"заботился", "забот"},
		{"заботимся", "забот"},
		{"заботит", "забот"},
		{"заботится", "забот"},
		{"заботиться", "забот"},
		{"заботишься", "забот"},
		{"заботливая", "заботлив"},
		{"заботливо", "заботлив"},
		{"заботливое", "заботлив"},
		{"заботливой", "заботлив"},
		{"заботливости", "заботлив"},
		{"заботливостию", "заботлив"},
		{"заботливость", "заботлив"},
		{"заботливостью", "заботлив"},
		{"заботливую", "заботлив"},
		{"заботливы", "заботлив"},
		{"заботливый", "заботлив"},
		{"заботливым", "заботлив"},
		{"заботливыми", "заботлив"},
		{"заботой", "забот"},
		{"заботою", "забот"},
		{"заботу", "забот"},
		{"заботы", "забот"},
		{"заботясь", "забот"},
		{"забочусь", "забоч"},
		{"забрал", "забра"},
		{"забрала", "забра"},
		{"забрали", "забра"},
		{"забрались", "забра"},
		{"забрасывал", "забрасыва"},
		{"забрать", "забра"},
		{"забрежжит", "забрежж"},
		{"забрезжилось", "забрезж"},
		{"забрезжит", "забрезж"},
		{"забрел", "забрел"},
		{"забродит", "заброд"},
		{"забросил", "заброс"},
		{"забросить", "заброс"},
		{"заброшенная", "заброшен"},
		{"заброшенной", "заброшен"},
		{"забубенного", "забубен"},
		{"забудем", "забуд"},
		{"забудемся", "забуд"},
		{"забудет", "забудет"},
		{"забудется", "забудет"},
		{"забудешь", "забудеш"},
		{"забуду", "забуд"},
		{"забудут", "забудут"},
		{"забудь", "забуд"},
		{"забудьте", "забудьт"},
		{"забулдыга", "забулдыг"},
		{"забуянила", "забуян"},
		{"забыв", "заб"},
		{"забывает", "забыва"},
		{"забываете", "забыва"},
		{"забывай", "забыва"},
		{"забывайте", "забыва"},
		{"забывал", "забыва"},
		{"забывала", "забыва"},
		{"забывалась", "забыва"},
		{"забывали", "забыва"},
		{"забывался", "забыва"},
		{"забывать", "забыва"},
		{"забываться", "забыва"},
		{"забываю", "забыва"},
		{"забывают", "забыва"},
		{"забывая", "забыв"},
		{"забывчив", "забывч"},
		{"забывчивости", "забывчив"},
		{"забывшему", "заб"},
		{"забывшие", "заб"},
		{"забывший", "заб"},
		{"забывшись", "заб"},
		{"забыл", "заб"},
		{"забыла", "заб"},
		{"забылась", "заб"},
		{"забыли", "заб"},
		{"забылись", "заб"},
		{"забылось", "заб"},
		{"забылся", "заб"},
		{"забыт", "заб"},
		{"забыто", "забыт"},
		{"забытого", "забыт"},
		{"забытое", "забыт"},
		{"забытой", "забыт"},
		{"забытом", "забыт"},
		{"забытую", "забыт"},
		{"забыты", "забыт"},
		{"забытые", "забыт"},
		{"забытый", "забыт"},
		{"забытым", "забыт"},
		{"забытых", "забыт"},
		{"забыть", "заб"},
		{"забытье", "забыт"},
		{"забытьи", "забыт"},
		{"забыться", "заб"},
		{"забытья", "забыт"},
		{"забьется", "забьет"},
		{"забьются", "забьют"},
		{"завадских", "завадск"},
		{"завален", "завал"},
		{"завалена", "завал"},
		{"завалившись", "завал"},
		{"завалилось", "завал"},
		{"завалился", "завал"},
		{"завалинке", "завалинк"},
		{"завалиться", "завал"},
		{"завалявшиеся", "заваля"},
		{"завалялось", "заваля"},
		{"заваривать", "заварива"},
		{"заведен", "завед"},
		{"заведение", "заведен"},
		{"заведении", "заведен"},
		{"заведений", "заведен"},
		{"заведения", "заведен"},
		{"заведениями", "заведен"},
		{"заведениях", "заведен"},
		{"заведет", "заведет"},
		{"заведется", "заведет"},
		{"заведи", "завед"},
		{"заведите", "завед"},
		{"заведовал", "заведова"},
		{"заведовают", "заведова"},
		{"заведу", "завед"},
		{"заведутся", "заведут"},
		{"заведую", "завед"},
		{"завезти", "завезт"},
		{"завезу", "завез"},
		{"завел", "завел"},
		{"завелась", "завел"},
		{"завелись", "завел"},
		{"завелся", "завел"},
		{"завернул", "завернул"},
		{"завернулась", "завернул"},
		{"завернута", "завернут"},
		{"завернутою", "завернут"},
		{"завернуты", "завернут"},
		{"завернуть", "завернут"},
		{"завернуться", "завернут"},
		{"завертелись", "завертел"},
		{"завертелось", "завертел"},
		{"завертываться", "завертыва"},
		{"завертываясь", "завертыв"},
		{"завершила", "заверш"},
		{"завершители", "завершител"},
		{"завеса", "завес"},
		{"завесила", "завес"},
		{"завести", "завест"},
		{"завесу", "завес"},
		{"завет", "завет"},
		{"завета", "завет"},
		{"заветного", "заветн"},
		{"заветное", "заветн"},
		{"заветной", "заветн"},
		{"заветный", "заветн"},
		{"заветных", "заветн"},
		{"заветов", "завет"},
		{"заветом", "завет"},
		{"заветы", "завет"},
		{"завещал", "завеща"},
		{"завещала", "завеща"},
		{"завещание", "завещан"},
		{"завещании", "завещан"},
		{"завещанию", "завещан"},
		{"завещания", "завещан"},
		{"завещанного", "завеща"},
		{"завещанных", "завеща"},
		{"завивала", "завива"},
		{"завивать", "завива"},
		{"завидели", "завидел"},
		{"завидит", "завид"},
		{"завидно", "завидн"},
		{"завидовать", "завидова"},
		{"завидуешь", "завидуеш"},
		{"завидую", "завид"},
		{"завидя", "завид"},
		{"завизжала", "завизжа"},
		{"завираюсь", "завира"},
		{"зависело", "зависел"},
		{"зависит", "завис"},
		{"зависти", "завист"},
		{"завистлив", "завистл"},
		{"завистливо", "завистлив"},
		{"завистливого", "завистлив"},
		{"завистливы", "завистлив"},
		{"зависть", "завист"},
		{"завистью", "завист"},
		{"зависящим", "завися"},
		{"зависящими", "завися"},
		{"завитой", "завит"},
		{"завитые", "завит"},
		{"завитых", "завит"},
		{"завладев", "завлад"},
		{"завлек", "завлек"},
		{"завлекал", "завлека"},
		{"завлекала", "завлека"},
		{"завлекли", "завлекл"},
		{"завлечен", "завлеч"},
		{"завод", "завод"},
		{"завода", "завод"},
		{"заводах", "завод"},
		{"заводе", "завод"},
		{"заводил", "завод"},
		{"заводила", "завод"},
		{"заводит", "завод"},
		{"заводится", "завод"},
		{"заводить", "завод"},
		{"заводские", "заводск"},
		{"заводы", "завод"},
		{"завоет", "завоет"},
		{"завожу", "завож"},
		{"завозилась", "завоз"},
		{"заволакивало", "заволакива"},
		{"заволновалась", "заволнова"},
		{"заволокла", "заволокл"},
		{"заволокло", "заволокл"},
		{"завопил", "завоп"},
		{"завопила", "завоп"},
		{"завопит", "завоп"},
		{"завороженный", "заворожен"},
		{"заворотила", "заворот"},
		{"заворчал", "заворча"},
		{"заврался", "завра"},
		{"завсегда", "завсегд"},
		{"завтра", "завтр"},
		{"завтрак", "завтрак"},
		{"завтрака", "завтрак"},
		{"завтракает", "завтрака"},
		{"завтракал", "завтрака"},
		{"завтракать", "завтрака"},
		{"завтракают", "завтрака"},
		{"завтраком", "завтрак"},
		{"завтраму", "завтрам"},
		{"завтрашнем", "завтрашн"},
		{"завтрашнему", "завтрашн"},
		{"завтрашний", "завтрашн"},
		{"завтрему", "завтр"},
		{"завывавший", "завыва"},
		{"завывал", "завыва"},
		{"завываниями", "завыван"},
		{"завыванье", "завыван"},
		{"завяжется", "завяжет"},
		{"завяжу", "завяж"},
		{"завяз", "завяз"},
		{"завязалась", "завяза"},
		{"завязались", "завяза"},
		{"завязалось", "завяза"},
		{"завязан", "завяза"},
		{"завязанный", "завяза"},
		{"завязанными", "завяза"},
		{"завязнет", "завязнет"},
		{"загадал", "загада"},
		{"загадала", "загада"},
		{"загадать", "загада"},
		{"загадка", "загадк"},
		{"загадками", "загадк"},
		{"загадки", "загадк"},
		{"загадкой", "загадк"},
		{"загадку", "загадк"},
		{"загадок", "загадок"},
		{"загадочно", "загадочн"},
		{"загадочное", "загадочн"},
		{"загадочной", "загадочн"},
		{"загадочному", "загадочн"},
		{"загадочные", "загадочн"},
		{"загадочный", "загадочн"},
		{"загадочным", "загадочн"},
		{"загадывает", "загадыва"},
		{"загадываний", "загадыван"},
		{"загар", "загар"},
		{"загвоздка", "загвоздк"},
		{"заглавие", "заглав"},
		{"заглавием", "заглав"},
		{"заглавия", "заглав"},
		{"заглавном", "заглавн"},
		{"загладилось", "заглад"},
		{"загладится", "заглад"},
		{"загладить", "заглад"},
		{"заглохло", "заглохл"},
		{"заглушает", "заглуша"},
		{"заглушал", "заглуша"},
		{"заглушала", "заглуша"},
		{"заглушат", "заглушат"},
		{"заглушенный", "заглушен"},
		{"заглушенных", "заглушен"},
		{"заглушило", "заглуш"},
		{"заглушит", "заглуш"},
		{"заглушить", "заглуш"},
		{"заглушон", "заглушон"},
		{"заглядевшись", "заглядевш"},
		{"загляделась", "заглядел"},
		{"загляделся", "заглядел"},
		{"заглядывавшей", "заглядыва"},
		{"заглядывает", "заглядыва"},
		{"заглядывал", "заглядыва"},
		{"заглядывала", "заглядыва"},
		{"заглядывать", "заглядыва"},
		{"заглядывая", "заглядыв"},
		{"заглянет", "заглянет"},
		{"заглянете", "заглянет"},
		{"загляну", "заглян"},
		{"заглянув", "заглянув"},
		{"заглянул", "заглянул"},
		{"заглянула", "заглянул"},
		{"заглянуть", "заглянут"},
		{"загнан", "загна"},
		{"загнанная", "загна"},
		{"загниют", "загниют"},
		{"загнутым", "загнут"},
		{"заговаривай", "заговарива"},
		{"заговаривал", "заговарива"},
		{"заговаривала", "заговарива"},
		{"заговаривался", "заговарива"},
		{"заговаривать", "заговарива"},
		{"заговариваю", "заговарива"},
		{"заговаривая", "заговарив"},
		{"заговелись", "заговел"},
		{"заговенья", "заговен"},
		{"заговор", "заговор"},
		{"заговорил", "заговор"},
		{"заговорила", "заговор"},
		{"заговорили", "заговор"},
		{"заговорило", "заговор"},
		{"заговорит", "заговор"},
		{"заговорить", "заговор"},
		{"заговорщик", "заговорщик"},
		{"заговорщиков", "заговорщик"},
		{"заговорю", "заговор"},
		{"заговорят", "заговор"},
		{"заголосил", "заголос"},
		{"загорается", "загора"},
		{"загораживаемый", "загоражива"},
		{"загораживают", "загоражива"},
		{"загораживая", "загоражив"},
		{"загорались", "загора"},
		{"загоралось", "загора"},
		{"загораются", "загора"},
		{"загордится", "загорд"},
		{"загоревшаяся", "загоревш"},
		{"загоревшейся", "загоревш"},
		{"загоревшеюся", "загоревш"},
		{"загоревшимися", "загоревш"},
		{"загорелась", "загорел"},
		{"загорели", "загорел"},
		{"загорелись", "загорел"},
		{"загорелой", "загорел"},
		{"загорелось", "загорел"},
		{"загорелую", "загорел"},
		{"загорелые", "загорел"},
		{"загорись", "загор"},
		{"загородившем", "загород"},
		{"загородил", "загород"},
		{"загородили", "загород"},
		{"загородных", "загородн"},
		{"загорожена", "загорож"},
		{"загороженные", "загорожен"},
		{"загорячился", "загоряч"},
		{"заготовил", "заготов"},
		{"заготовлю", "заготовл"},
		{"заградил", "заград"},
		{"заградить", "заград"},
		{"заграничная", "заграничн"},
		{"заграничный", "заграничн"},
		{"заграничных", "заграничн"},
		{"загреб", "загреб"},
		{"загремел", "загремел"},
		{"загремели", "загремел"},
		{"загромождена", "загроможд"},
		{"загромождены", "загроможд"},
		{"загрубевшими", "загрубевш"},
		{"загрубели", "загрубел"},
		{"загрызет", "загрызет"},
		{"загрязнилось", "загрязн"},
		{"загрязнился", "загрязн"},
		{"загрязнить", "загрязн"},
		{"загубила", "загуб"},
		{"загудели", "загудел"},
		{"загулял", "загуля"},
		{"задаваемые", "задава"},
		{"задаваемым", "задава"},
		{"задавайте", "задава"},
		{"задавал", "задава"},
		{"задавала", "задава"},
		{"задавали", "задава"},
		{"задавать", "задава"},
		{"задавая", "задав"},
		{"задавила", "задав"},
		{"задавит", "задав"},
		{"задавлен", "задавл"},
		{"задавят", "задав"},
		{"задайте", "зада"},
		{"задал", "зада"},
		{"задала", "зада"},
		{"задало", "зада"},
		{"задам", "зад"},
		{"задаром", "задар"},
		{"задаст", "задаст"},
		{"задатка", "задатк"},
		{"задатки", "задатк"},
		{"задатку", "задатк"},
		{"задаток", "задаток"},
		{"задать", "зада"},
		{"задач", "задач"},
		{"задача", "задач"},
		{"задачею", "задач"},
		{"задачи", "задач"},
		{"задачу", "задач"},
		{"задаю", "зада"},
		{"задвигались", "задвига"},
		{"задвигаются", "задвига"},
		{"задвижкой", "задвижк"},
		{"задвижку", "задвижк"},
		{"задвинет", "задвинет"},
		{"задворками", "задворк"},
		{"задворки", "задворк"},
		{"задебренные", "задебрен"},
		{"задевает", "задева"},
		{"задевающего", "задева"},
		{"задевая", "задев"},
		{"задевши", "задевш"},
		{"задел", "задел"},
		{"задела", "задел"},
		{"задели", "задел"},
		{"задело", "задел"},
		{"заденет", "заденет"},
		{"задержал", "задержа"},
		{"задержали", "задержа"},
		{"задержало", "задержа"},
		{"задержать", "задержа"},
		{"задерживаем", "задержива"},
		{"задерживай", "задержива"},
		{"задерживать", "задержива"},
		{"задерживаю", "задержива"},
		{"задержкам", "задержк"},
		{"задержками", "задержк"},
		{"задержу", "задерж"},
		{"задернется", "задернет"},
		{"задернул", "задернул"},
		{"задетым", "задет"},
		{"задеть", "задет"},
		{"задираете", "задира"},
		{"задирательное", "задирательн"},
		{"задирать", "задира"},
		{"задней", "задн"},
		{"заднем", "задн"},
		{"задний", "задн"},
		{"задних", "задн"},
		{"задобрить", "задобр"},
		{"задок", "задок"},
		{"задолжал", "задолжа"},
		{"задом", "зад"},
		{"задор", "задор"},
		{"задора", "задор"},
		{"задохнулся", "задохнул"},
		{"задразнили", "задразн"},
		{"задремавший", "задрема"},
		{"задремал", "задрема"},
		{"задремали", "задрема"},
		{"задремать", "задрема"},
		{"задремлет", "задремлет"},
		{"задремлешь", "задремлеш"},
		{"задрожав", "задрожа"},
		{"задрожал", "задрожа"},
		{"задрожала", "задрожа"},
		{"задрожали", "задрожа"},
		{"задрожало", "задрожа"},
		{"задрожит", "задрож"},
		{"задул", "задул"},
		{"задумавшись", "задума"},
		{"задумает", "задума"},
		{"задумаетесь", "задума"},
		{"задумай", "задума"},
		{"задумайте", "задума"},
		{"задумал", "задума"},
		{"задумала", "задума"},
		{"задумалась", "задума"},
		{"задумались", "задума"},
		{"задумался", "задума"},
		{"задумана", "задума"},
		{"задуманного", "задума"},
		{"задуманное", "задума"},
		{"задуматься", "задума"},
		{"задумаюсь", "задума"},
		{"задумаются", "задума"},
		{"задумчив", "задумч"},
		{"задумчивая", "задумчив"},
		{"задумчивее", "задумчив"},
		{"задумчиво", "задумчив"},
		{"задумчивое", "задумчив"},
		{"задумчивой", "задумчив"},
		{"задумчивом", "задумчив"},
		{"задумчивосгь", "задумчивосг"},
		{"задумчивости", "задумчив"},
		{"задумчивость", "задумчив"},
		{"задумчивостью", "задумчив"},
		{"задумчивую", "задумчив"},
		{"задумчивы", "задумчив"},
		{"задумчивые", "задумчив"},
		{"задумчивый", "задумчив"},
		{"задумчивым", "задумчив"},
		{"задумчивыми", "задумчив"},
		{"задумчивых", "задумчив"},
		{"задумывается", "задумыва"},
		{"задумывайтесь", "задумыва"},
		{"задумывалась", "задумыва"},
		{"задумывались", "задумыва"},
		{"задумывался", "задумыва"},
		{"задумываться", "задумыва"},
		{"задумываются", "задумыва"},
		{"задумываясь", "задумыв"},
		{"задурманился", "задурман"},
		{"задушевный", "задушевн"},
		{"задушил", "задуш"},
		{"задушила", "задуш"},
		{"задушило", "задуш"},
		{"задушить", "задуш"},
		{"зады", "зад"},
		{"задымились", "задым"},
		{"задымится", "задым"},
		{"задыхавшаяся", "задыха"},
		{"задыхается", "задыха"},
		{"задыхаешься", "задыха"},
		{"задыхалась", "задыха"},
		{"задыхался", "задыха"},
		{"задыхаться", "задыха"},
		{"задыхаюсь", "задыха"},
		{"задыхающимся", "задыха"},
		{"задыхающуюся", "задыха"},
		{"задыхаясь", "задых"},
		{"заедает", "заеда"},
		{"заедать", "заеда"},
		{"заедены", "заед"},
		{"заеду", "заед"},
		{"заезжая", "заезж"},
		{"заела", "заел"},
		{"заем", "за"},
		{"заемного", "заемн"},
		{"заемное", "заемн"},
		{"заемному", "заемн"},
		{"заемные", "заемн"},
		{"заехал", "заеха"},
		{"заехали", "заеха"},
		{"заехать", "заеха"},
		{"зажав", "зажа"},
		{"зажаждал", "зажажда"},
		{"зажал", "зажа"},
		{"зажаты", "зажат"},
		{"зажатый", "зажат"},
		{"зажглись", "зажгл"},
		{"зажгло", "зажгл"},
		{"зажгут", "зажгут"},
		{"заждались", "зажда"},
		{"зажег", "зажег"},
		{"зажечь", "зажеч"},
		{"зажжена", "зажж"},
		{"зажженный", "зажжен"},
		{"зажженных", "зажжен"},
		{"заживаются", "зажива"},
		{"заживем", "зажив"},
		{"заживет", "заживет"},
		{"заживо", "зажив"},
		{"зажигает", "зажига"},
		{"зажигал", "зажига"},
		{"зажигали", "зажига"},
		{"зажигательное", "зажигательн"},
		{"зажигать", "зажига"},
		{"зажигая", "зажиг"},
		{"зажила", "заж"},
		{"зажили", "заж"},
		{"зажило", "заж"},
		{"зажимая", "зажим"},
		{"зажирел", "зажирел"},
		{"зажиточном", "зажиточн"},
		{"зажмет", "зажмет"},
		{"зажмешь", "зажмеш"},
		{"зажмуренные", "зажмурен"},
		{"зажмурив", "зажмур"},
		{"зажмуриваясь", "зажмурив"},
		{"зажмурил", "зажмур"},
		{"зажмурился", "зажмур"},
		{"зажмуря", "зажмур"},
		{"зажужжала", "зажужжа"},
		{"зажужжали", "зажужжа"},
		{"зазвать", "зазва"},
		{"зазвенел", "зазвенел"},
		{"зазвенело", "зазвенел"},
		{"зазвенят", "зазвен"},
		{"зазвучал", "зазвуча"},
		{"зазвучала", "зазвуча"},
		{"зазвучат", "зазвучат"},
		{"зазвучит", "зазвуч"},
		{"зазовет", "зазовет"},
		{"зазору", "зазор"},
		{"зазубринки", "зазубринк"},
		{"заиграет", "заигра"},
		{"заиграешь", "заигра"},
		{"заиграл", "заигра"},
		{"заиграла", "заигра"},
		{"заиграли", "заигра"},
		{"заиграть", "заигра"},
		{"заиграют", "заигра"},
		{"заигрывал", "заигрыва"},
		{"заикается", "заика"},
		{"заикаться", "заика"},
		{"заикаются", "заика"},
		{"заикаясь", "заик"},
		{"заикнется", "заикнет"},
		{"заикнитесь", "заикн"},
		{"заикнулась", "заикнул"},
		{"заимодавец", "заимодавец"},
		{"заинтересовала", "заинтересова"},
		{"заинтересовали", "заинтересова"},
		{"заинтересовало", "заинтересова"},
		{"заинтересовался", "заинтересова"},
		{"заинтересован", "заинтересова"},
		{"заинтересованы", "заинтересова"},
		{"заинтересовать", "заинтересова"},
		{"заискать", "заиска"},
		{"заискивала", "заискива"},
		{"заискиванье", "заискиван"},
		{"заискиваю", "заискива"},
		{"зайдем", "зайд"},
		{"зайдет", "зайдет"},
		{"зайдете", "зайдет"},
		{"зайдешь", "зайдеш"},
		{"зайди", "зайд"},
		{"зайдите", "зайд"},
		{"зайду", "зайд"},
		{"зайдя", "зайд"},
		{"займа", "займ"},
		{"займемся", "займ"},
		{"займет", "займет"},
		{"займется", "займет"},
		{"займешь", "займеш"},
		{"займите", "займ"},
		{"займусь", "займ"},
		{"займут", "займут"},
		{"займутся", "займут"},
		{"зайти", "зайт"},
		{"зайца", "зайц"},
		{"зайцами", "зайц"},
		{"закажу", "закаж"},
		{"заказ", "заказ"},
		{"заказал", "заказа"},
		{"заказали", "заказа"},
		{"заказанные", "заказа"},
		{"заказать", "заказа"},
		{"заказу", "заказ"},
		{"заказывал", "заказыва"},
		{"заказывала", "заказыва"},
		{"заказывались", "заказыва"},
		{"заказывать", "заказыва"},
		{"закалена", "закал"},
		{"закаливая", "закалив"},
		{"заканчивается", "заканчива"},
		{"заканчивать", "заканчива"},
		{"закапали", "закапа"},
		{"закарабкался", "закарабка"},
		{"закат", "закат"},
		{"заката", "закат"},
		{"закатал", "заката"},
		{"закате", "закат"},
		{"закатившееся", "закат"},
		{"закатившись", "закат"},
		{"закатилась", "закат"},
		{"закатилось", "закат"},
		{"закатился", "закат"},
		{"закатим", "закат"},
		{"закатной", "закатн"},
		{"закатном", "закатн"},
		{"закатный", "закатн"},
		{"закатных", "закатн"},
		{"закатом", "закат"},
		{"закату", "закат"},
		{"закатывалось", "закатыва"},
		{"закачал", "закача"},
		{"закивала", "закива"},
		{"закинем", "закин"},
		{"закинув", "закинув"},
		{"закинул", "закинул"},
		{"закинула", "закинул"},
		{"закинулось", "закинул"},
		{"закинуто", "закинут"},
		{"закинуть", "закинут"},
		{"закипел", "закипел"},
		{"закипела", "закипел"},
		{"закипели", "закипел"},
		{"закипело", "закипел"},
		{"закипит", "закип"},
		{"заклад", "заклад"},
		{"заклада", "заклад"},
		{"закладе", "заклад"},
		{"закладка", "закладк"},
		{"закладной", "закладн"},
		{"закладом", "заклад"},
		{"закладу", "заклад"},
		{"закладчик", "закладчик"},
		{"закладчика", "закладчик"},
		{"закладчиками", "закладчик"},
		{"закладчики", "закладчик"},
		{"закладчиков", "закладчик"},
		{"закладчицы", "закладчиц"},
		{"заклады", "заклад"},
		{"закладываемое", "закладыва"},
		{"закладывай", "закладыва"},
		{"закладывал", "закладыва"},
		{"закладывала", "закладыва"},
		{"закладывать", "закладыва"},
		{"закладывают", "закладыва"},
		{"заклинаю", "заклина"},
		{"заключает", "заключа"},
		{"заключаете", "заключа"},
		{"заключается", "заключа"},
		{"заключало", "заключа"},
		{"заключалось", "заключа"},
		{"заключать", "заключа"},
		{"заключаю", "заключа"},
		{"заключен", "заключ"},
		{"заключение", "заключен"},
		{"заключением", "заключен"},
		{"заключении", "заключен"},
		{"заключению", "заключен"},
		{"заключения", "заключен"},
		{"заключениям", "заключен"},
		{"заключив", "заключ"},
		{"заключил", "заключ"},
		{"заключила", "заключ"},
		{"заключили", "заключ"},
		{"заключить", "заключ"},
		{"заклятье", "заклят"},
		{"заколдован", "заколдова"},
		{"заколдованных", "заколдова"},
		{"заколот", "заколот"},
		{"заколотила", "заколот"},
		{"заколотить", "заколот"},
		{"заколют", "заколют"},
		{"закон", "закон"},
		{"закона", "закон"},
		{"законам", "закон"},
		{"законно", "закон"},
		{"законного", "закон"},
		{"законное", "закон"},
		{"законном", "закон"},
		{"законному", "закон"},
		{"законности", "закон"},
		{"законность", "закон"},
		{"законною", "закон"},
		{"законную", "закон"},
		{"законный", "закон"},
		{"законным", "закон"},
		{"законными", "закон"},
		{"законных", "закон"},
		{"законов", "закон"},
		{"законодатели", "законодател"},
		{"законодатель", "законодател"},
		{"законом", "закон"},
		{"закону", "закон"},
		{"законфузившись", "законфуз"},
		{"законфузилась", "законфуз"},
		{"законфузились", "законфуз"},
		{"законфузился", "законфуз"},
		{"законфузиться", "законфуз"},
		{"законченный", "закончен"},
		{"законченных", "закончен"},
		{"закончился", "законч"},
		{"закончить", "законч"},
		{"законы", "закон"},
		{"закопал", "закопа"},
		{"закоптелая", "закоптел"},
		{"закопченного", "закопчен"},
		{"закоренелою", "закоренел"},
		{"закоробившись", "закороб"},
		{"закорузлую", "закорузл"},
		{"закорузлый", "закорузл"},
		{"закоснелость", "закоснел"},
		{"закоулки", "закоулк"},
		{"закоченели", "закоченел"},
		{"закоченелый", "закоченел"},
		{"закрадывается", "закрадыва"},
		{"закрасить", "закрас"},
		{"закраснелась", "закраснел"},
		{"закрепить", "закреп"},
		{"закричал", "закрича"},
		{"закричала", "закрича"},
		{"закричали", "закрича"},
		{"закричат", "закричат"},
		{"закричать", "закрича"},
		{"закричите", "закрич"},
		{"закричу", "закрич"},
		{"закровянилось", "закровян"},
		{"закроет", "закроет"},
		{"закроется", "закроет"},
		{"закрой", "закр"},
		{"закройся", "закр"},
		{"закрою", "закр"},
		{"закроют", "закроют"},
		{"закругленной", "закруглен"},
		{"закружилась", "закруж"},
		{"закружился", "закруж"},
		{"закружите", "закруж"},
		{"закружитесь", "закруж"},
		{"закружится", "закруж"},
		{"закрутишься", "закрут"},
		{"закрутят", "закрут"},
		{"закручивая", "закручив"},
		{"закрыв", "закр"},
		{"закрывает", "закрыва"},
		{"закрываешь", "закрыва"},
		{"закрывай", "закрыва"},
		{"закрывал", "закрыва"},
		{"закрывали", "закрыва"},
		{"закрывать", "закрыва"},
		{"закрывая", "закрыв"},
		{"закрывшись", "закр"},
		{"закрыл", "закр"},
		{"закрыла", "закр"},
		{"закрылась", "закр"},
		{"закрыли", "закр"},
		{"закрылись", "закр"},
		{"закрыт", "закр"},
		{"закрыта", "закрыт"},
		{"закрытое", "закрыт"},
		{"закрыты", "закрыт"},
		{"закрытый", "закрыт"},
		{"закрытыми", "закрыт"},
		{"закрытых", "закрыт"},
		{"закудахтал", "закудахта"},
		{"закулисную", "закулисн"},
		{"закупить", "закуп"},
		{"закупка", "закупк"},
		{"закупками", "закупк"},
		{"закупкой", "закупк"},
		{"закупоривать", "закупорива"},
		{"закупорилось", "закупор"},
		{"закуражился", "закураж"},
		{"закурив", "закур"},
		{"закуривает", "закурива"},
		{"закуривать", "закурива"},
		{"закурил", "закур"},
		{"закусив", "закус"},
		{"закусил", "закус"},
		{"закусить", "закус"},
		{"закуска", "закуск"},
		{"закусками", "закуск"},
		{"закуске", "закуск"},
		{"закуски", "закуск"},
		{"закуску", "закуск"},
		{"закусывал", "закусыва"},
		{"закусывала", "закусыва"},
		{"закусывать", "закусыва"},
		{"закутав", "закута"},
		{"закутавшись", "закута"},
		{"закутал", "закута"},
		{"закуталась", "закута"},
		{"закутался", "закута"},
		{"закутанный", "закута"},
		{"закутанным", "закута"},
		{"закутаться", "закута"},
		{"зал", "зал"},
		{"зала", "зал"},
		{"залает", "зала"},
		{"залах", "зал"},
		{"залаяла", "залая"},
		{"зале", "зал"},
		{"залег", "залег"},
		{"залегла", "залегл"},
		{"залежавшейся", "залежа"},
		{"залежался", "залежа"},
		{"залежаться", "залежа"},
		{"залез", "залез"},
		{"залезал", "залеза"},
		{"залезли", "залезл"},
		{"залезть", "залезт"},
		{"залепетала", "залепета"},
		{"залепила", "залеп"},
		{"залетает", "залета"},
		{"залетал", "залета"},
		{"залетела", "залетел"},
		{"залетело", "залетел"},
		{"залечить", "залеч"},
		{"залив", "зал"},
		{"залива", "залив"},
		{"заливалась", "залива"},
		{"заливались", "залива"},
		{"заливаться", "залива"},
		{"заливаются", "залива"},
		{"заливаясь", "залив"},
		{"заливистый", "заливист"},
		{"заливного", "заливн"},
		{"заливные", "заливн"},
		{"залившею", "зал"},
		{"залившись", "зал"},
		{"залила", "зал"},
		{"залилась", "зал"},
		{"залились", "зал"},
		{"залился", "зал"},
		{"залитая", "залит"},
		{"залито", "залит"},
		{"залитое", "залит"},
		{"залитые", "залит"},
		{"залитый", "залит"},
		{"залить", "зал"},
		{"залог", "залог"},
		{"залогом", "залог"},
		{"заложена", "залож"},
		{"заложенном", "заложен"},
		{"заложенный", "заложен"},
		{"заложив", "залож"},
		{"заложил", "залож"},
		{"заложила", "залож"},
		{"заложили", "залож"},
		{"заложит", "залож"},
		{"заложить", "залож"},
		{"заложу", "залож"},
		{"залой", "зал"},
		{"заломив", "залом"},
		{"заломившаяся", "залом"},
		{"заломлены", "заломл"},
		{"залпом", "залп"},
		{"залу", "зал"},
		{"залучить", "залуч"},
		{"залы", "зал"},
		{"зальдевших", "зальдевш"},
		{"зальет", "зальет"},
		{"зальется", "зальет"},
		{"зальешь", "зальеш"},
		{"зальюсь", "зал"},
		{"замай", "зама"},
		{"замайте", "зама"},
		{"заманит", "заман"},
		{"заманить", "заман"},
		{"заманишь", "заман"},
		{"заманчивей", "заманчив"},
		{"заманчиво", "заманчив"},
		{"замараться", "замара"},
		{"замарашки", "замарашк"},
		{"замасленной", "замаслен"},
		{"замасленную", "замаслен"},
		{"замасленные", "замаслен"},
		{"замасленных", "замаслен"},
		{"замахал", "замаха"},
		{"замахивается", "замахива"},
		{"замахиваясь", "замахив"},
		{"замахнется", "замахнет"},
		{"замахнула", "замахнул"},
		{"замахнулась", "замахнул"},
		{"замашками", "замашк"},
		{"замашкой", "замашк"},
		{"замедленного", "замедлен"},
		{"замедлил", "замедл"},
		{"замедляет", "замедля"},
		{"замедляй", "замедля"},
		{"замелькал", "замелька"},
		{"замелькали", "замелька"},
		{"заменил", "замен"},
		{"заменились", "замен"},
		{"заменился", "замен"},
		{"заменить", "замен"},
		{"замену", "замен"},
		{"заменю", "замен"},
		{"заменя", "замен"},
		{"заменяет", "заменя"},
		{"заменялась", "заменя"},
		{"замер", "замер"},
		{"замереть", "замерет"},
		{"замерзнет", "замерзнет"},
		{"замерзнуть", "замерзнут"},
		{"замерла", "замерл"},
		{"замерли", "замерл"},
		{"замерло", "замерл"},
		{"замертво", "замертв"},
		{"замести", "замест"},
		{"заместо", "замест"},
		{"заметен", "замет"},
		{"заметенной", "заметен"},
		{"заметив", "замет"},
		{"заметил", "замет"},
		{"заметила", "замет"},
		{"заметили", "замет"},
		{"заметим", "замет"},
		{"заметит", "замет"},
		{"заметить", "замет"},
		{"заметишь", "замет"},
		{"заметку", "заметк"},
		{"заметливы", "заметлив"},
		{"заметна", "заметн"},
		{"заметная", "заметн"},
		{"заметно", "заметн"},
		{"заметного", "заметн"},
		{"заметное", "заметн"},
		{"заметной", "заметн"},
		{"заметном", "заметн"},
		{"заметною", "заметн"},
		{"заметную", "заметн"},
		{"заметны", "заметн"},
		{"заметным", "заметн"},
		{"заметов", "замет"},
		{"заметова", "заметов"},
		{"заметовские", "заметовск"},
		{"заметову", "заметов"},
		{"заметовым", "заметов"},
		{"заметок", "заметок"},
		{"заметь", "замет"},
		{"заметьте", "заметьт"},
		{"заметят", "замет"},
		{"замечаемый", "замеча"},
		{"замечает", "замеча"},
		{"замечаете", "замеча"},
		{"замечается", "замеча"},
		{"замечаешь", "замеча"},
		{"замечай", "замеча"},
		{"замечал", "замеча"},
		{"замечала", "замеча"},
		{"замечали", "замеча"},
		{"замечалось", "замеча"},
		{"замечание", "замечан"},
		{"замечанием", "замечан"},
		{"замечанию", "замечан"},
		{"замечания", "замечан"},
		{"замечаниями", "замечан"},
		{"замечателен", "замечател"},
		{"замечательная", "замечательн"},
		{"замечательнейшая", "замечательн"},
		{"замечательнейший", "замечательн"},
		{"замечательно", "замечательн"},
		{"замечательного", "замечательн"},
		{"замечательное", "замечательн"},
		{"замечательном", "замечательн"},
		{"замечательные", "замечательн"},
		{"замечательными", "замечательн"},
		{"замечать", "замеча"},
		{"замечаю", "замеча"},
		{"замечают", "замеча"},
		{"замечая", "замеч"},
		{"замечен", "замеч"},
		{"замеченной", "замечен"},
		{"замеченным", "замечен"},
		{"замечтает", "замечта"},
		{"замечтал", "замечта"},
		{"замечталась", "замечта"},
		{"замечтался", "замечта"},
		{"замешается", "замеша"},
		{"замешался", "замеша"},
		{"замешан", "замеша"},
		{"замешаны", "замеша"},
		{"замешательство", "замешательств"},
		{"замешательством", "замешательств"},
		{"замешкали", "замешка"},
		{"замешкались", "замешка"},
		{"замешкался", "замешка"},
		{"замигал", "замига"},
		{"замирает", "замира"},
		{"замирала", "замира"},
		{"замирало", "замира"},
		{"замиранием", "замиран"},
		{"замирающие", "замира"},
		{"замирающий", "замира"},
		{"замирающим", "замира"},
		{"замирая", "замир"},
		{"замка", "замк"},
		{"замке", "замк"},
		{"замки", "замк"},
		{"замкну", "замкн"},
		{"замкнут", "замкнут"},
		{"замкнутую", "замкнут"},
		{"замкнутых", "замкнут"},
		{"замкнуть", "замкнут"},
		{"замнется", "замнет"},
		{"замогильный", "замогильн"},
		{"замок", "замок"},
		{"замолит", "замол"},
		{"замолк", "замолк"},
		{"замолкла", "замолкл"},
		{"замолкли", "замолкл"},
		{"замолкло", "замолкл"},
		{"замолкнут", "замолкнут"},
		{"замолкнуть", "замолкнут"},
		{"замолчал", "замолча"},
		{"замолчала", "замолча"},
		{"замолчали", "замолча"},
		{"замолчат", "замолчат"},
		{"замолчать", "замолча"},
		{"замолчи", "замолч"},
		{"замолчит", "замолч"},
		{"замолчу", "замолч"},
		{"заморишь", "замор"},
		{"заморские", "заморск"},
		{"заморской", "заморск"},
		{"заморскую", "заморск"},
		{"замочились", "замоч"},
		{"замочился", "замоч"},
		{"замочной", "замочн"},
		{"замочную", "замочн"},
		{"замою", "зам"},
		{"замрет", "замрет"},
		{"замри", "замр"},
		{"замрут", "замрут"},
		{"замуж", "замуж"},
		{"замужем", "замуж"},
		{"замужества", "замужеств"},
		{"замужестве", "замужеств"},
		{"замужеству", "замужеств"},
		{"замужнюю", "замужн"},
		{"замучаешься", "замуча"},
		{"замучай", "замуча"},
		{"замученные", "замучен"},
		{"замучил", "замуч"},
		{"замучила", "замуч"},
		{"замучилась", "замуч"},
		{"замучили", "замуч"},
		{"замучились", "замуч"},
		{"замучился", "замуч"},
		{"замучит", "замуч"},
		{"замучить", "замуч"},
		{"замшевые", "замшев"},
		{"замшевый", "замшев"},
		{"замыкалась", "замыка"},
		{"замыкалова", "замыкалов"},
		{"замыкаю", "замыка"},
		{"замыкая", "замык"},
		{"замыкаясь", "замык"},
		{"замысел", "замысел"},
		{"замысла", "замысл"},
		{"замыслишь", "замысл"},
		{"замыслов", "замысл"},
		{"замыслы", "замысл"},
		{"замыть", "зам"},
		{"замышляла", "замышля"},
		{"замышлялась", "замышля"},
		{"замяла", "замя"},
		{"замялся", "замя"},
		{"замямлил", "замямл"},
		{"замять", "замя"},
		{"занавес", "занавес"},
		{"занавесей", "занавес"},
		{"занавеска", "занавеск"},
		{"занавесками", "занавеск"},
		{"занавески", "занавеск"},
		{"занавеску", "занавеск"},
		{"занавесок", "занавесок"},
		{"занавесом", "занавес"},
		{"занавесы", "занавес"},
		{"занавесью", "занавес"},
		{"занды", "занд"},
		{"занес", "занес"},
		{"занесен", "занес"},
		{"занесенной", "занесен"},
		{"занесенными", "занесен"},
		{"занести", "занест"},
		{"занимавшая", "занима"},
		{"занимаемая", "занима"},
		{"занимает", "занима"},
		{"занимаетесь", "занима"},
		{"занимается", "занима"},
		{"занимаешься", "занима"},
		{"занимай", "занима"},
		{"занимайся", "занима"},
		{"занимайтесь", "занима"},
		{"занимал", "занима"},
		{"занимала", "занима"},
		{"занималась", "занима"},
		{"занимали", "занима"},
		{"занимались", "занима"},
		{"занимало", "занима"},
		{"занимался", "занима"},
		{"занимательным", "занимательн"},
		{"занимать", "занима"},
		{"заниматься", "занима"},
		{"занимаю", "занима"},
		{"занимаюсь", "занима"},
		{"занимаются", "занима"},
		{"занимающаяся", "занима"},
		{"заново", "занов"},
		{"занозила", "заноз"},
		{"заносил", "занос"},
		{"заносчивая", "заносчив"},
		{"заносчиво", "заносчив"},
		{"заносчивого", "заносчив"},
		{"заносчивость", "заносчив"},
		{"заносчивы", "заносчив"},
		{"заносчивым", "заносчив"},
		{"занося", "зан"},
		{"заношенная", "заношен"},
		{"заношенного", "заношен"},
		{"заношенные", "заношен"},
		{"занявший", "заня"},
		{"занял", "заня"},
		{"заняла", "заня"},
		{"занялась", "заня"},
		{"занялись", "заня"},
		{"заняло", "заня"},
		{"занялся", "заня"},
		{"занят", "зан"},
		{"занята", "занят"},
		{"занятая", "занят"},
		{"занятие", "занят"},
		{"занятии", "занят"},
		{"занятий", "занят"},
		{"занятия", "занят"},
		{"занятиям", "занят"},
		{"занято", "занят"},
		{"занятого", "занят"},
		{"заняты", "занят"},
		{"занятые", "занят"},
		{"занятый", "занят"},
		{"занять", "заня"},
		{"заняться", "заня"},
		{"заодно", "заодн"},
		{"заорал", "заора"},
		{"заочно", "заочн"},
		{"западают", "запада"},
		{"западник", "западник"},
		{"западню", "западн"},
		{"запамятовал", "запамятова"},
		{"запамятовали", "запамятова"},
		{"запарился", "запар"},
		{"запас", "запас"},
		{"запаса", "запас"},
		{"запасе", "запас"},
		{"запасем", "запас"},
		{"запасными", "запасн"},
		{"запасом", "запас"},
		{"запасы", "запас"},
		{"запах", "зап"},
		{"запахе", "запах"},
		{"запахло", "запахл"},
		{"запахов", "запах"},
		{"запахом", "запах"},
		{"запачкала", "запачка"},
		{"запачкался", "запачка"},
		{"запачканная", "запачка"},
		{"запачканное", "запачка"},
		{"запачканные", "запачка"},
		{"запачкаться", "запачка"},
		{"запашке", "запашк"},
		{"запевает", "запева"},
		{"запевали", "запева"},
		{"запевания", "запеван"},
		{"запевая", "запев"},
		{"запекаться", "запека"},
		{"запеклась", "запекл"},
		{"запеклись", "запекл"},
		{"запекшейся", "запекш"},
		{"запекшимися", "запекш"},
		{"запекшихся", "запекш"},
		{"запел", "запел"},
		{"запела", "запел"},
		{"запели", "запел"},
		{"запендю", "запенд"},
		{"запенившимися", "запен"},
		{"запер", "запер"},
		{"заперев", "запер"},
		{"запереть", "заперет"},
		{"запереться", "заперет"},
		{"заперла", "заперл"},
		{"заперлась", "заперл"},
		{"заперли", "заперл"},
		{"заперся", "запер"},
		{"заперт", "заперт"},
		{"заперта", "заперт"},
		{"запертая", "заперт"},
		{"заперто", "заперт"},
		{"запертой", "заперт"},
		{"запертую", "заперт"},
		{"заперты", "заперт"},
		{"запертых", "заперт"},
		{"запершись", "заперш"},
		{"запечатал", "запечата"},
		{"запечатанную", "запечата"},
		{"запивает", "запива"},
		{"запивать", "запива"},
		{"запил", "зап"},
		{"запинаются", "запина"},
		{"запинаясь", "запин"},
		{"запинкой", "запинк"},
		{"запираемы", "запираем"},
		{"запирает", "запира"},
		{"запираете", "запира"},
		{"запирается", "запира"},
		{"запираешь", "запира"},
		{"запиралась", "запира"},
		{"запирали", "запира"},
		{"запирался", "запира"},
		{"запирать", "запира"},
		{"запираю", "запира"},
		{"запирают", "запира"},
		{"запираются", "запира"},
		{"записали", "записа"},
		{"записались", "записа"},
		{"записан", "записа"},
		{"записанными", "записа"},
		{"записано", "записа"},
		{"записаны", "записа"},
		{"записать", "записа"},
		{"записка", "записк"},
		{"записками", "записк"},
		{"записках", "записк"},
		{"записке", "записк"},
		{"записки", "записк"},
		{"запиской", "записк"},
		{"записку", "записк"},
		{"записная", "записн"},
		{"записной", "записн"},
		{"записную", "записн"},
		{"записок", "записок"},
		{"записочку", "записочк"},
		{"записывает", "записыва"},
		{"записывания", "записыван"},
		{"записыванья", "записыван"},
		{"записывать", "записыва"},
		{"записываю", "записыва"},
		{"записывают", "записыва"},
		{"записывая", "записыв"},
		{"запихивать", "запихива"},
		{"запихивая", "запихив"},
		{"запишем", "запиш"},
		{"запиши", "запиш"},
		{"запишу", "запиш"},
		{"заплакал", "заплака"},
		{"заплакала", "заплака"},
		{"заплакали", "заплака"},
		{"заплаканное", "заплака"},
		{"заплаканной", "заплака"},
		{"заплаканы", "заплака"},
		{"заплакать", "заплака"},
		{"заплатами", "заплат"},
		{"заплатах", "заплат"},
		{"заплати", "заплат"},
		{"заплатил", "заплат"},
		{"заплатила", "заплат"},
		{"заплатили", "заплат"},
		{"заплатим", "заплат"},
		{"заплатит", "заплат"},
		{"заплатите", "заплат"},
		{"заплатить", "заплат"},
		{"заплатишь", "заплат"},
		{"заплатки", "заплатк"},
		{"заплаты", "заплат"},
		{"заплачем", "заплач"},
		{"заплачет", "заплачет"},
		{"заплачешь", "заплачеш"},
		{"заплачу", "заплач"},
		{"заплачут", "заплачут"},
		{"заплевавший", "заплева"},
		{"заплеванному", "заплева"},
		{"заплеванный", "заплева"},
		{"заплесневела", "заплесневел"},
		{"заплесть", "заплест"},
		{"заплетены", "заплет"},
		{"заплутался", "заплута"},
		{"заплывали", "заплыва"},
		{"заплывшие", "запл"},
		{"заплыли", "запл"},
		{"заплясало", "запляса"},
		{"запнулась", "запнул"},
		{"запнулся", "запнул"},
		{"заподозрили", "заподозр"},
		{"заподозрить", "заподозр"},
		{"запоем", "запо"},
		{"запоет", "запоет"},
		{"запоешь", "запоеш"},
		{"запоздало", "запозда"},
		{"запоздалый", "запоздал"},
		{"запоздалым", "запоздал"},
		{"запоздать", "запозда"},
		{"запой", "зап"},
		{"заполнить", "заполн"},
		{"заполняя", "заполн"},
		{"заполонила", "заполон"},
		{"заполонить", "заполон"},
		{"запомнивши", "запомн"},
		{"запомнил", "запомн"},
		{"запомнили", "запомн"},
		{"запомним", "запомн"},
		{"запомните", "запомн"},
		{"запомнить", "запомн"},
		{"запомню", "запомн"},
		{"запомнят", "запомн"},
		{"запонка", "запонк"},
		{"запонки", "запонк"},
		{"запор", "запор"},
		{"запора", "запор"},
		{"запоре", "запор"},
		{"запорет", "запорет"},
		{"запороши", "запорош"},
		{"запорошит", "запорош"},
		{"запоя", "запо"},
		{"заправить", "заправ"},
		{"заправленную", "заправлен"},
		{"заправляет", "заправля"},
		{"запрег", "запрег"},
		{"запредельных", "запредельн"},
		{"запрет", "запрет"},
		{"запретив", "запрет"},
		{"запретил", "запрет"},
		{"запретишь", "запрет"},
		{"запреты", "запрет"},
		{"запрещает", "запреща"},
		{"запрещала", "запреща"},
		{"запрещение", "запрещен"},
		{"запрещения", "запрещен"},
		{"запрещениям", "запрещен"},
		{"запрещено", "запрещ"},
		{"заприте", "запр"},
		{"запритесь", "запр"},
		{"запропастившуюся", "запропаст"},
		{"запропастились", "запропаст"},
		{"запросила", "запрос"},
		{"запросится", "запрос"},
		{"запросто", "запрост"},
		{"запросят", "запрос"},
		{"запружена", "запруж"},
		{"запрыгает", "запрыга"},
		{"запрыгал", "запрыга"},
		{"запрыгала", "запрыга"},
		{"запрыгали", "запрыга"},
		{"запрягут", "запрягут"},
		{"запряженная", "запряжен"},
		{"запряженной", "запряжен"},
		{"запрячь", "запряч"},
		{"запугала", "запуга"},
		{"запуганной", "запуга"},
		{"запуганным", "запуга"},
		{"запугать", "запуга"},
		{"запускает", "запуска"},
		{"запускал", "запуска"},
		{"запускать", "запуска"},
		{"запустив", "запуст"},
		{"запустил", "запуст"},
		{"запутавшись", "запута"},
		{"запутает", "запута"},
		{"запутаешься", "запута"},
		{"запуталось", "запута"},
		{"запутанного", "запута"},
		{"запутанный", "запута"},
		{"запутываться", "запутыва"},
		{"запутывая", "запутыв"},
		{"запутываясь", "запутыв"},
		{"запушенным", "запушен"},
		{"запущенностью", "запущен"},
		{"запущенную", "запущен"},
		{"запущенный", "запущен"},
		{"запущенным", "запущен"},
		{"запущенных", "запущен"},
		{"запущено", "запущ"},
		{"запылавшие", "запыла"},
		{"запылали", "запыла"},
		{"запыленное", "запылен"},
		{"запыленной", "запылен"},
		{"запыленному", "запылен"},
		{"запыленную", "запылен"},
		{"запыленные", "запылен"},
		{"запылены", "запыл"},
		{"запылилась", "запыл"},
		{"запылилось", "запыл"},
		{"запыхавшись", "запыха"},
		{"запыхалась", "запыха"},
		{"запьешь", "запьеш"},
		{"запью", "зап"},
		{"запястье", "запяст"},
		{"запястья", "запяст"},
		{"запятнаны", "запятна"},
		{"зарабатывал", "зарабатыва"},
		{"заработает", "заработа"},
		{"заработал", "заработа"},
		{"заработались", "заработа"},
		{"заработался", "заработа"},
		{"заработать", "заработа"},
		{"заработка", "заработк"},
		{"заработком", "заработк"},
		{"заражал", "заража"},
		{"заражалась", "заража"},
		{"заражались", "заража"},
		{"заражаются", "заража"},
		{"заражая", "зараж"},
		{"зараженного", "заражен"},
		{"зараженные", "заражен"},
		{"зараженный", "заражен"},
		{"зараз", "зараз"},
		{"заразил", "зараз"},
		{"заразила", "зараз"},
		{"заразилась", "зараз"},
		{"заразили", "зараз"},
		{"заразителен", "заразител"},
		{"заразительная", "заразительн"},
		{"заразительно", "заразительн"},
		{"заразительны", "заразительн"},
		{"заразы", "зараз"},
		{"зарайский", "зарайск"},
		{"зарайских", "зарайск"},
		{"зарайского", "зарайск"},
		{"заранее", "заран"},
		{"зарапортовался", "зарапортова"},
		{"зарастает", "зараста"},
		{"зарделось", "зардел"},
		{"заре", "зар"},
		{"зарева", "зарев"},
		{"заревам", "зарев"},
		{"заревая", "зарев"},
		{"зареве", "зарев"},
		{"заревел", "заревел"},
		{"зарево", "зарев"},
		{"заревое", "зарев"},
		{"заревом", "зарев"},
		{"заревою", "зарев"},
		{"заревые", "зарев"},
		{"зарежет", "зарежет"},
		{"зарежь", "зареж"},
		{"зарезал", "зареза"},
		{"зарезала", "зареза"},
		{"зарезу", "зарез"},
		{"зарей", "зар"},
		{"заречная", "заречн"},
		{"заречной", "заречн"},
		{"зарею", "зар"},
		{"заржавленным", "заржавлен"},
		{"заржет", "заржет"},
		{"зари", "зар"},
		{"зарницы", "зарниц"},
		{"зарницына", "зарницын"},
		{"зарницыной", "зарницын"},
		{"зародилась", "зарод"},
		{"зародились", "зарод"},
		{"зародыш", "зародыш"},
		{"зарождаются", "зарожда"},
		{"зарождение", "зарожден"},
		{"зарождения", "зарожден"},
		{"заройся", "зар"},
		{"зароненной", "заронен"},
		{"заронился", "зарон"},
		{"заросла", "заросл"},
		{"заросший", "заросш"},
		{"заросшими", "заросш"},
		{"зароют", "зароют"},
		{"зарубежной", "зарубежн"},
		{"зарубить", "заруб"},
		{"зарывшись", "зар"},
		{"зарыдала", "зарыда"},
		{"зарыдаю", "зарыда"},
		{"зарыдают", "зарыда"},
		{"зарылись", "зар"},
		{"зарыт", "зар"},
		{"зарыто", "зарыт"},
		{"зарыть", "зар"},
		{"зарыться", "зар"},
		{"зарычал", "зарыча"},
		{"зарычи", "зарыч"},
		{"зарю", "зар"},
		{"заря", "зар"},
		{"заряда", "заряд"},
		{"зарядили", "заряд"},
		{"заряжанье", "заряжан"},
		{"засади", "засад"},
		{"засаленной", "засален"},
		{"засаленном", "засален"},
		{"засаленный", "засален"},
		{"засасывает", "засасыва"},
		{"засверкав", "засверка"},
		{"засверкавшими", "засверка"},
		{"засверкал", "засверка"},
		{"засверкали", "засверка"},
		{"засверкало", "засверка"},
		{"засветили", "засвет"},
		{"засветились", "засвет"},
		{"засветилось", "засвет"},
		{"засветло", "засветл"},
		{"засвидетельствовал", "засвидетельствова"},
		{"засвидетельствовала", "засвидетельствова"},
		{"засвидетельствован", "засвидетельствова"},
		{"засвидетельствована", "засвидетельствова"},
		{"засвидетельствование", "засвидетельствован"},
		{"засвидетельствовано", "засвидетельствова"},
		{"засвидетельствовать", "засвидетельствова"},
		{"засевались", "засева"},
		{"заседание", "заседан"},
		{"засеку", "засек"},
		{"засела", "засел"},
		{"заселен", "засел"},
		{"засели", "засел"},
		{"заселилось", "засел"},
		{"засело", "засел"},
		{"засиделся", "засидел"},
		{"засидится", "засид"},
		{"засипел", "засипел"},
		{"засияет", "засия"},
		{"засиял", "засия"},
		{"засияла", "засия"},
		{"засияли", "засия"},
		{"засияло", "засия"},
		{"заскакать", "заскака"},
		{"заскрежетав", "заскрежета"},
		{"заскрежетал", "заскрежета"},
		{"заскрипит", "заскрип"},
		{"заслоненная", "заслонен"},
		{"заслонил", "заслон"},
		{"заслонилась", "заслон"},
		{"заслонилось", "заслон"},
		{"заслонявший", "заслоня"},
		{"заслонялось", "заслоня"},
		{"заслуга", "заслуг"},
		{"заслуги", "заслуг"},
		{"заслуженная", "заслужен"},
		{"заслуженного", "заслужен"},
		{"заслуженной", "заслужен"},
		{"заслуженном", "заслужен"},
		{"заслуженным", "заслужен"},
		{"заслужил", "заслуж"},
		{"заслужу", "заслуж"},
		{"заслушивался", "заслушива"},
		{"заслыша", "заслыш"},
		{"заслышавший", "заслыша"},
		{"засматриваться", "засматрива"},
		{"засмеется", "засмеет"},
		{"засмеют", "засмеют"},
		{"засмеются", "засмеют"},
		{"засмеявшись", "засмея"},
		{"засмеялась", "засмея"},
		{"засмеялись", "засмея"},
		{"засмеялся", "засмея"},
		{"засмеяться", "засмея"},
		{"засмотрится", "засмотр"},
		{"заснет", "заснет"},
		{"засну", "засн"},
		{"заснувши", "заснувш"},
		{"заснул", "заснул"},
		{"заснула", "заснул"},
		{"заснули", "заснул"},
		{"заснуть", "заснут"},
		{"засовы", "засов"},
		{"засохла", "засохл"},
		{"засохли", "засохл"},
		{"засохло", "засохл"},
		{"засохшею", "засохш"},
		{"засохшим", "засохш"},
		{"засохшими", "засохш"},
		{"засохших", "засохш"},
		{"заспав", "заспа"},
		{"заспанная", "заспа"},
		{"заспанные", "заспа"},
		{"заспаны", "заспа"},
		{"заспишься", "засп"},
		{"заставал", "застава"},
		{"заставала", "застава"},
		{"заставил", "застав"},
		{"заставили", "застав"},
		{"заставило", "застав"},
		{"заставит", "застав"},
		{"заставить", "застав"},
		{"заставленное", "заставлен"},
		{"заставлю", "заставл"},
		{"заставлявших", "заставля"},
		{"заставляет", "заставля"},
		{"заставлял", "заставля"},
		{"заставляла", "заставля"},
		{"заставляли", "заставля"},
		{"заставляю", "заставля"},
		{"заставляют", "заставля"},
		{"заставляя", "заставл"},
		{"заставой", "застав"},
		{"заставу", "застав"},
		{"заставы", "застав"},
		{"заставят", "застав"},
		{"застал", "заста"},
		{"застала", "заста"},
		{"застали", "заста"},
		{"застанем", "застан"},
		{"застанет", "застанет"},
		{"застанешь", "застанеш"},
		{"застану", "застан"},
		{"застарелой", "застарел"},
		{"застать", "заста"},
		{"застегивался", "застегива"},
		{"застегивание", "застегиван"},
		{"застегивать", "застегива"},
		{"застегивая", "застегив"},
		{"застегнут", "застегнут"},
		{"застегнутое", "застегнут"},
		{"застелют", "застелют"},
		{"застенчив", "застенч"},
		{"застенчива", "застенчив"},
		{"застенчиво", "застенчив"},
		{"застенчивости", "застенчив"},
		{"застенчивость", "застенчив"},
		{"застенчивы", "застенчив"},
		{"застиг", "застиг"},
		{"застилает", "застила"},
		{"застойки", "застойк"},
		{"застойкой", "застойк"},
		{"застольных", "застольн"},
		{"застонал", "застона"},
		{"застонет", "застонет"},
		{"застрахованы", "застрахова"},
		{"застраховать", "застрахова"},
		{"застрелившиеся", "застрел"},
		{"застрелил", "застрел"},
		{"застрелился", "застрел"},
		{"застрелись", "застрел"},
		{"застрелитесь", "застрел"},
		{"застрелится", "застрел"},
		{"застрелиться", "застрел"},
		{"застрелюсь", "застрел"},
		{"застряло", "застря"},
		{"застрянет", "застрянет"},
		{"застукало", "застука"},
		{"заступаться", "заступа"},
		{"заступила", "заступ"},
		{"заступило", "заступ"},
		{"заступитесь", "заступ"},
		{"заступиться", "заступ"},
		{"заступом", "заступ"},
		{"застучал", "застуча"},
		{"застучали", "застуча"},
		{"застучало", "застуча"},
		{"застучат", "застучат"},
		{"застучит", "застуч"},
		{"застывала", "застыва"},
		{"застывало", "застыва"},
		{"застывая", "застыв"},
		{"застыдилась", "застыд"},
		{"застыдился", "застыд"},
		{"застыдят", "застыд"},
		{"застыла", "заст"},
		{"застыли", "заст"},
		{"застынет", "застынет"},
		{"засудят", "засуд"},
		{"засуетилась", "засует"},
		{"засуетились", "засует"},
		{"засуетился", "засует"},
		{"засуетятся", "засует"},
		{"засунув", "засунув"},
		{"засунул", "засунул"},
		{"засунула", "засунул"},
		{"засуха", "засух"},
		{"засухи", "засух"},
		{"засуху", "засух"},
		{"засученными", "засучен"},
		{"засучились", "засуч"},
		{"засыпает", "засыпа"},
		{"засыпал", "засыпа"},
		{"засыпала", "засыпа"},
		{"засыпанная", "засыпа"},
		{"засыпаю", "засыпа"},
		{"засыпают", "засыпа"},
		{"засыпи", "засып"},
		{"засыплет", "засыплет"},
		{"засыхавшая", "засыха"},
		{"засядем", "засяд"},
		{"затаенной", "затаен"},
		{"затаившейся", "зата"},
		{"затаившиеся", "зата"},
		{"затаившимся", "зата"},
		{"затаил", "зата"},
		{"затараторила", "затаратор"},
		{"затасканной", "затаска"},
		{"затасканную", "затаска"},
		{"затвердил", "затверд"},
		{"затворе", "затвор"},
		{"затворена", "затвор"},
		{"затвори", "затвор"},
		{"затворив", "затвор"},
		{"затворилась", "затвор"},
		{"затворить", "затвор"},
		{"затворничества", "затворничеств"},
		{"затворю", "затвор"},
		{"затворяет", "затворя"},
		{"затворяется", "затворя"},
		{"затворялась", "затворя"},
		{"затворят", "затвор"},
		{"затворяя", "затвор"},
		{"затевать", "затева"},
		{"затеем", "зате"},
		{"затеет", "затеет"},
		{"затеешь", "затееш"},
		{"затеи", "зат"},
		{"затей", "зат"},
		{"затекает", "затека"},
		{"затем", "зат"},
		{"затемнен", "затемн"},
		{"затепли", "затепл"},
		{"затеплят", "затепл"},
		{"затерлось", "затерл"},
		{"затерся", "затер"},
		{"затертого", "затерт"},
		{"затертый", "затерт"},
		{"затерявшийся", "затеря"},
		{"затерял", "затеря"},
		{"затерянная", "затеря"},
		{"затерянный", "затеря"},
		{"затеряться", "затеря"},
		{"затесало", "затеса"},
		{"затеснились", "затесн"},
		{"затеял", "затея"},
		{"затеяли", "затея"},
		{"затеялись", "затея"},
		{"затеями", "зате"},
		{"затеять", "затея"},
		{"затих", "зат"},
		{"затихать", "затиха"},
		{"затихла", "затихл"},
		{"затихли", "затихл"},
		{"затихло", "затихл"},
		{"затишье", "затиш"},
		{"затишьев", "затиш"},
		{"затишьях", "затиш"},
		{"заткнет", "заткнет"},
		{"заткни", "заткн"},
		{"заткнутые", "заткнут"},
		{"заткнутым", "заткнут"},
		{"затмевал", "затмева"},
		{"затмение", "затмен"},
		{"затмений", "затмен"},
		{"зато", "зат"},
		{"затолпилось", "затолп"},
		{"затопав", "затопа"},
		{"затопал", "затопа"},
		{"затопишь", "затоп"},
		{"затопчет", "затопчет"},
		{"заторелым", "заторел"},
		{"заторопившись", "затороп"},
		{"заторопилась", "затороп"},
		{"заторопился", "затороп"},
		{"заторопясь", "затороп"},
		{"затосковал", "затоскова"},
		{"затравленная", "затравлен"},
		{"затратили", "затрат"},
		{"затревожилась", "затревож"},
		{"затревожит", "затревож"},
		{"затрепетал", "затрепета"},
		{"затрепетала", "затрепета"},
		{"затрепетали", "затрепета"},
		{"затрепещет", "затрепещет"},
		{"затрется", "затрет"},
		{"затрогивать", "затрогива"},
		{"затронет", "затронет"},
		{"затронули", "затронул"},
		{"затруднение", "затруднен"},
		{"затруднении", "затруднен"},
		{"затруднений", "затруднен"},
		{"затруднения", "затруднен"},
		{"затруднилась", "затрудн"},
		{"затруднительно", "затруднительн"},
		{"затруднительное", "затруднительн"},
		{"затрудняется", "затрудня"},
		{"затруднялась", "затрудня"},
		{"затруднялись", "затрудня"},
		{"затруднялся", "затрудня"},
		{"затрудняясь", "затрудн"},
		{"затряс", "затряс"},
		{"затряслась", "затрясл"},
		{"затряслись", "затрясл"},
		{"затрясусь", "затряс"},
		{"затуманила", "затуман"},
		{"затуманилась", "затуман"},
		{"затуманит", "затуман"},
		{"затушено", "затуш"},
		{"затушить", "затуш"},
		{"затушу", "затуш"},
		{"затыкать", "затыка"},
		{"затылка", "затылк"},
		{"затылке", "затылк"},
		{"затылок", "затылок"},
		{"затягивает", "затягива"},
		{"затягивая", "затягив"},
		{"затянувшийся", "затянувш"},
		{"затянула", "затянул"},
		{"затянули", "затянул"},
		{"затянулось", "затянул"},
		{"затянут", "затянут"},
		{"заунывную", "заунывн"},
		{"заурядным", "заурядн"},
		{"заутрене", "заутрен"},
		{"заутрени", "заутрен"},
		{"заученное", "заучен"},
		{"заученную", "заучен"},
		{"заученный", "заучен"},
		{"заучены", "зауч"},
		{"заучусь", "зауч"},
		{"зафилософствовался", "зафилософствова"},
		{"захар", "захар"},
		{"захара", "захар"},
		{"захарка", "захарк"},
		{"захарке", "захарк"},
		{"захаров", "захар"},
		{"захарович", "захарович"},
		{"захаровича", "захарович"},
		{"захаровичем", "захарович"},
		{"захаровых", "захаров"},
		{"захаром", "захар"},
		{"захару", "захар"},
		{"захарыч", "захарыч"},
		{"захарыча", "захарыч"},
		{"захвалить", "захвал"},
		{"захватив", "захват"},
		{"захватившие", "захват"},
		{"захватил", "захват"},
		{"захватила", "захват"},
		{"захватили", "захват"},
		{"захватило", "захват"},
		{"захватит", "захват"},
		{"захватить", "захват"},
		{"захватывает", "захватыва"},
		{"захватывало", "захватыва"},
		{"захватывают", "захватыва"},
		{"захваченный", "захвачен"},
		{"захворал", "захвора"},
		{"захворала", "захвора"},
		{"захихикал", "захихика"},
		{"захлебнулся", "захлебнул"},
		{"захлебываясь", "захлебыв"},
		{"захлестнула", "захлестнул"},
		{"захлестнуло", "захлестнул"},
		{"захлопнула", "захлопнул"},
		{"захлопнулась", "захлопнул"},
		{"захлопывала", "захлопыва"},
		{"захлопывая", "захлопыв"},
		{"захмелевший", "захмелевш"},
		{"захныкал", "захныка"},
		{"заходил", "заход"},
		{"заходила", "заход"},
		{"заходили", "заход"},
		{"заходило", "заход"},
		{"заходит", "заход"},
		{"заходить", "заход"},
		{"заходя", "заход"},
		{"заходящее", "заходя"},
		{"заходящим", "заходя"},
		{"захождения", "захожден"},
		{"захожу", "захож"},
		{"захотел", "захотел"},
		{"захотела", "захотел"},
		{"захотели", "захотел"},
		{"захотелось", "захотел"},
		{"захотеть", "захотет"},
		{"захотите", "захот"},
		{"захотят", "захот"},
		{"захохотал", "захохота"},
		{"захохотала", "захохота"},
		{"захохотали", "захохота"},
		{"захохочет", "захохочет"},
		{"захочет", "захочет"},
		{"захочется", "захочет"},
		{"захочешь", "захочеш"},
		{"захочу", "захоч"},
		{"захрипел", "захрипел"},
		{"захромала", "захрома"},
		{"захрустел", "захрустел"},
		{"зацелованный", "зацелова"},
		{"зацепите", "зацеп"},
		{"зацепиться", "зацеп"},
		{"зачало", "зача"},
		{"зачастую", "зачаст"},
		{"зачатия", "зачат"},
		{"зачатками", "зачатк"},
		{"зачах", "зач"},
		{"зачахла", "зачахл"},
		{"зачахну", "зачахн"},
		{"зачем", "зач"},
		{"зачеркивает", "зачеркива"},
		{"зачеркнет", "зачеркнет"},
		{"зачерпнула", "зачерпнул"},
		{"зачешется", "зачешет"},
		{"зачешутся", "зачешут"},
		{"зачислен", "зачисл"},
		{"зачитался", "зачита"},
		{"зачитывался", "зачитыва"},
		{"зачтено", "зачт"},
		{"зашаталась", "зашата"},
		{"зашаталось", "зашата"},
		{"зашевелились", "зашевел"},
		{"зашевелит", "зашевел"},
		{"зашевелится", "зашевел"},
		{"зашевелят", "зашевел"},
		{"зашевелятся", "зашевел"},
		{"зашел", "зашел"},
		{"зашептал", "зашепта"},
		{"зашептала", "зашепта"},
		{"зашептались", "зашепта"},
		{"зашивались", "зашива"},
		{"зашипел", "зашипел"},
		{"зашипит", "зашип"},
		{"зашить", "заш"},
		{"зашла", "зашл"},
		{"зашлепал", "зашлепа"},
		{"зашли", "зашл"},
		{"зашло", "зашл"},
		{"заштопала", "заштопа"},
		{"заштопать", "заштопа"},
		{"зашумит", "зашум"},
		{"зашумят", "зашум"},
		{"защелкал", "защелка"},
		{"защемило", "защем"},
		{"защита", "защит"},
		{"защите", "защ"},
		{"защити", "защит"},
		{"защитит", "защит"},
		{"защитите", "защит"},
		{"защитить", "защит"},
		{"защититься", "защит"},
		{"защитник", "защитник"},
		{"защиту", "защит"},
		{"защиты", "защит"},
		{"защищавших", "защища"},
		{"защищает", "защища"},
		{"защищал", "защища"},
		{"защищала", "защища"},
		{"защищался", "защища"},
		{"защищать", "защища"},
		{"заявил", "заяв"},
		{"заявила", "заяв"},
		{"заявить", "заяв"},
		{"заявлениях", "заявлен"},
		{"заявлено", "заявл"},
		{"заявляет", "заявля"},
		{"заячья", "заяч"},
		{"звал", "звал"},
		{"звала", "звал"},
		{"звали", "звал"},
		{"зваль", "звал"},
		{"звание", "зван"},
		{"званием", "зван"},
		{"звания", "зван"},
		{"званых", "зван"},
		{"званья", "зван"},
		{"звать", "зват"},
		{"звезд", "звезд"},
		{"звезда", "звезд"},
		{"звездам", "звезд"},
		{"звездами", "звезд"},
		{"звездах", "звезд"},
		{"звезде", "звезд"},
		{"звездной", "звездн"},
		{"звездой", "звезд"},
		{"звездочка", "звездочк"},
		{"звездою", "звезд"},
		{"звезду", "звезд"},
		{"звезды", "звезд"},
		{"звена", "звен"},
		{"звенела", "звенел"},
		{"звенели", "звенел"},
		{"звенело", "звенел"},
		{"звенит", "звен"},
		{"звено", "звен"},
		{"звену", "звен"},
		{"звеньев", "звен"},
		{"звенья", "звен"},
		{"звеня", "звен"},
		{"звенят", "звен"},
		{"звенящий", "звеня"},
		{"зверей", "звер"},
		{"зверем", "звер"},
		{"звери", "звер"},
		{"звериного", "зверин"},
		{"звериной", "зверин"},
		{"звериную", "зверин"},
		{"звериный", "зверин"},
		{"зверок", "зверок"},
		{"зверская", "зверск"},
		{"зверские", "зверск"},
		{"зверского", "зверск"},
		{"зверском", "зверск"},
		{"зверства", "зверств"},
		{"зверь", "звер"},
		{"зверю", "звер"},
		{"зверя", "звер"},
		{"звон", "звон"},
		{"звоне", "звон"},
		{"звонил", "звон"},
		{"звонили", "звон"},
		{"звонит", "звон"},
		{"звонить", "звон"},
		{"звонка", "звонк"},
		{"звонки", "звонк"},
		{"звонкие", "звонк"},
		{"звонкий", "звонк"},
		{"звонким", "звонк"},
		{"звонко", "звонк"},
		{"звонкой", "звонк"},
		{"звонком", "звонк"},
		{"звонок", "звонок"},
		{"звоном", "звон"},
		{"звонят", "звон"},
		{"звук", "звук"},
		{"звука", "звук"},
		{"звукам", "звук"},
		{"звуках", "звук"},
		{"звуки", "звук"},
		{"звуков", "звук"},
		{"звуком", "звук"},
		{"звуку", "звук"},
		{"звучал", "звуча"},
		{"звучала", "звуча"},
		{"звучали", "звуча"},
		{"звучало", "звуча"},
		{"звучат", "звучат"},
		{"звучащий", "звуча"},
		{"звучащим", "звуча"},
		{"звучен", "звуч"},
		{"звучит", "звуч"},
		{"звучным", "звучн"},
		{"звякание", "звякан"},
		{"звякают", "звяка"},
		{"звякнул", "звякнул"},
		{"зги", "зги"},
		{"зготовляй", "зготовля"},
		{"здание", "здан"},
		{"здания", "здан"},
		{"зданиях", "здан"},
		{"зданье", "здан"},
		{"зданья", "здан"},
		{"здесь", "зде"},
		{"здеся", "зде"},
		{"здешнего", "здешн"},
		{"здешней", "здешн"},
		{"здешнем", "здешн"},
		{"здешнему", "здешн"},
		{"здешнею", "здешн"},
		{"здешние", "здешн"},
		{"здешний", "здешн"},
		{"здешним", "здешн"},
		{"здешних", "здешн"},
		{"здешнюю", "здешн"},
		{"здоров", "здор"},
		{"здорова", "здоров"},
		{"здоровается", "здорова"},
		{"здоровалась", "здорова"},
		{"здороваться", "здорова"},
		{"здоровая", "здоров"},
		{"здоровее", "здоров"},
		{"здоровенькие", "здоровеньк"},
		{"здоровенького", "здоровеньк"},
		{"здоровехонек", "здоровехонек"},
		{"здорово", "здоров"},
		{"здорового", "здоров"},
		{"здоровой", "здоров"},
		{"здоровому", "здоров"},
		{"здоровы", "здоров"},
		{"здоровые", "здоров"},
		{"здоровый", "здоров"},
		{"здоровым", "здоров"},
		{"здоровыми", "здоров"},
		{"здоровье", "здоров"},
		{"здоровьем", "здоров"},
		{"здоровью", "здоров"},
		{"здоровья", "здоров"},
		{"здравие", "здрав"},
		{"здраво", "здрав"},
		{"здравого", "здрав"},
		{"здравое", "здрав"},
		{"здравом", "здрав"},
		{"здравому", "здрав"},
		{"здравствовать", "здравствова"},
		{"здравствует", "здравств"},
		{"здравствуй", "здравств"},
		{"здравствуйте", "здравств"},
		{"здравый", "здрав"},
		{"здсеь", "здсе"},
		{"зе", "зе"},
		{"зевавшему", "зева"},
		{"зевает", "зева"},
		{"зевак", "зевак"},
		{"зевал", "зева"},
		{"зевать", "зева"},
		{"зевая", "зев"},
		{"зевес", "зевес"},
		{"зевнет", "зевнет"},
		{"зевнул", "зевнул"},
		{"зевнуть", "зевнут"},
		{"зевотой", "зевот"},
		{"зевоту", "зевот"},
		{"зевоты", "зевот"},
		{"зейн", "зейн"},
		{"зейнрок", "зейнрок"},
		{"зелена", "зел"},
		{"зеленая", "зелен"},
		{"зеленеет", "зеленеет"},
		{"зелененькими", "зелененьк"},
		{"зелененькую", "зелененьк"},
		{"зеленеют", "зеленеют"},
		{"зелени", "зелен"},
		{"зеленные", "зелен"},
		{"зелено", "зел"},
		{"зеленоватые", "зеленоват"},
		{"зеленоватым", "зеленоват"},
		{"зеленого", "зелен"},
		{"зеленой", "зелен"},
		{"зеленом", "зелен"},
		{"зеленую", "зелен"},
		{"зеленщик", "зеленщик"},
		{"зеленщика", "зеленщик"},
		{"зеленщики", "зеленщик"},
		{"зеленщику", "зеленщик"},
		{"зелены", "зел"},
		{"зеленые", "зелен"},
		{"зеленый", "зелен"},
		{"зеленым", "зелен"},
		{"зелеными", "зелен"},
		{"зеленых", "зелен"},
		{"зелень", "зелен"},
		{"зеленью", "зелен"},
		{"земель", "земел"},
		{"земле", "земл"},
		{"земледелие", "земледел"},
		{"землей", "земл"},
		{"землетрясений", "землетрясен"},
		{"землетрясению", "землетрясен"},
		{"землею", "земл"},
		{"земли", "земл"},
		{"землю", "земл"},
		{"земля", "земл"},
		{"земляк", "земляк"},
		{"земляка", "земляк"},
		{"земляку", "земляк"},
		{"землям", "земл"},
		{"землянику", "земляник"},
		{"землянке", "землянк"},
		{"земная", "земн"},
		{"земно", "земн"},
		{"земного", "земн"},
		{"земное", "земн"},
		{"земной", "земн"},
		{"земном", "земн"},
		{"земные", "земн"},
		{"земского", "земск"},
		{"земской", "земск"},
		{"земском", "земск"},
		{"земскую", "земск"},
		{"земстве", "земств"},
		{"земчуг", "земчуг"},
		{"зените", "зен"},
		{"зеницах", "зениц"},
		{"зеркал", "зерка"},
		{"зеркала", "зерка"},
		{"зеркалах", "зеркал"},
		{"зеркале", "зеркал"},
		{"зеркало", "зерка"},
		{"зеркалом", "зеркал"},
		{"зеркальной", "зеркальн"},
		{"зеркальный", "зеркальн"},
		{"зеркальце", "зеркальц"},
		{"зеркальцем", "зеркальц"},
		{"зерна", "зерн"},
		{"зерно", "зерн"},
		{"зигзагами", "зигзаг"},
		{"зигзагом", "зигзаг"},
		{"зигфрид", "зигфрид"},
		{"зима", "зим"},
		{"зиме", "зим"},
		{"зимнее", "зимн"},
		{"зимней", "зимн"},
		{"зимнем", "зимн"},
		{"зимние", "зимн"},
		{"зимний", "зимн"},
		{"зимних", "зимн"},
		{"зимой", "зим"},
		{"зиму", "зим"},
		{"зимы", "зим"},
		{"зинаиду", "зинаид"},
		{"зияющие", "зия"},
		{"зла", "зла"},
		{"злаки", "злак"},
		{"златая", "злат"},
		{"златит", "злат"},
		{"злато", "злат"},
		{"златом", "злат"},
		{"златым", "злат"},
		{"златятся", "злат"},
		{"злая", "зла"},
		{"злее", "зле"},
		// {"злейший", "злейш"},
		{"злился", "злил"},
		{"злит", "злит"},
		{"злитесь", "злит"},
		{"злится", "злит"},
		{"злиться", "злит"},
		{"злишься", "злиш"},
		{"зло", "зло"},
		{"злоба", "злоб"},
		{"злобе", "злоб"},
		{"злобен", "злоб"},
		{"злобная", "злобн"},
		{"злобно", "злобн"},
		{"злобного", "злобн"},
		{"злобное", "злобн"},
		{"злобной", "злобн"},
		{"злобною", "злобн"},
		{"злобные", "злобн"},
		{"злобный", "злобн"},
		{"злобным", "злобн"},
		{"злобных", "злобн"},
		{"злобой", "злоб"},
		{"злобою", "злоб"},
		{"злобствуют", "злобств"},
		{"злобу", "злоб"},
		{"злобы", "злоб"},
		{"зловещая", "зловещ"},
		{"зловещее", "зловещ"},
		{"зловещей", "зловещ"},
		{"зловещие", "зловещ"},
		{"зловещий", "зловещ"},
		{"зловещим", "зловещ"},
		{"зловили", "злов"},
		{"зловредную", "зловредн"},
		{"злодее", "злод"},
		{"злодеев", "злоде"},
		{"злодей", "злод"},
		{"злодейка", "злодейк"},
		{"злодейскою", "злодейск"},
		{"злодейств", "злодейств"},
		{"злодейство", "злодейств"},
		{"злодея", "злоде"},
		{"злодеяние", "злодеян"},
		{"злое", "зло"},
		{"злой", "зло"},
		{"злом", "злом"},
		{"злонамеренная", "злонамерен"},
		{"злорадно", "злорадн"},
		{"злорадством", "злорадств"},
		{"злословил", "злослов"},
		{"злословия", "злослов"},
		{"злости", "злост"},
		{"злостных", "злостн"},
		{"злость", "злост"},
		{"злостью", "злост"},
		{"злосчастный", "злосчастн"},
		{"злоупотреблении", "злоупотреблен"},
		{"злоупотреблений", "злоупотреблен"},
		{"злоупотреблять", "злоупотребля"},
		{"злу", "злу"},
		{"злую", "злу"},
		{"злы", "злы"},
		{"злые", "злы"},
		{"злым", "злым"},
		{"злыми", "злым"},
		{"злых", "злых"},
		{"злюсь", "злю"},
		{"злющая", "злющ"},
		{"злющей", "злющ"},
		{"злясь", "зля"},
		{"змее", "зме"},
		{"змеей", "зме"},
		{"змеею", "зме"},
		{"змеи", "зме"},
		{"змеилась", "зме"},
		{"змеились", "зме"},
		{"змеиный", "змеин"},
		{"змеится", "зме"},
		{"змей", "зме"},
		{"змею", "зме"},
		{"змея", "зме"},
		{"змеящихся", "змея"},
		{"знавал", "знава"},
		{"знавали", "знава"},
		{"знавшие", "знавш"},
		{"знавший", "знавш"},
		{"знаем", "зна"},
		{"знает", "знает"},
		{"знаете", "знает"},
		{"знаешь", "знаеш"},
		{"знай", "зна"},
		{"знайте", "знайт"},
		{"знак", "знак"},
		{"знака", "знак"},
		{"знаками", "знак"},
		{"знаки", "знак"},
		{"знаков", "знак"},
		{"знаком", "знак"},
		{"знакома", "знаком"},
		{"знакомая", "знаком"},
		{"знакомиться", "знаком"},
		{"знакомо", "знаком"},
		{"знакомого", "знаком"},
		{"знакомое", "знаком"},
		{"знакомой", "знаком"},
		{"знакомом", "знаком"},
		{"знакомому", "знаком"},
		{"знакомств", "знакомств"},
		{"знакомства", "знакомств"},
		{"знакомстве", "знакомств"},
		{"знакомство", "знакомств"},
		{"знакомству", "знакомств"},
		{"знакомую", "знаком"},
		{"знакомы", "знаком"},
		{"знакомые", "знаком"},
		{"знакомый", "знаком"},
		{"знакомым", "знаком"},
		{"знакомыми", "знаком"},
		{"знакомых", "знаком"},
		{"знакомя", "знаком"},
		{"знал", "знал"},
		{"знала", "знал"},
		{"знали", "знал"},
		{"зналь", "знал"},
		{"знамен", "знам"},
		{"знамена", "знам"},
		{"знаменательно", "знаменательн"},
		{"знаменательного", "знаменательн"},
		{"знаменательные", "знаменательн"},
		{"знаменательный", "знаменательн"},
		{"знаменательным", "знаменательн"},
		{"знаменательных", "знаменательн"},
		{"знамением", "знамен"},
		{"знамений", "знамен"},
		{"знаменит", "знамен"},
		{"знаменитейший", "знаменит"},
		{"знаменитой", "знаменит"},
		{"знаменитости", "знаменит"},
		{"знаменитость", "знаменит"},
		{"знаменитый", "знаменит"},
		{"знамения", "знамен"},
		{"знаменовало", "знаменова"},
		{"знамо", "знам"},
		{"знамя", "знам"},
		{"знание", "знан"},
		{"знанием", "знан"},
		{"знаний", "знан"},
		{"знания", "знан"},
		{"знаниями", "знан"},
		{"знаньем", "знан"},
		{"знанья", "знан"},
		{"знатно", "знатн"},
		{"знатной", "знатн"},
		{"знатность", "знатност"},
		{"знатный", "знатн"},
		{"знатных", "знатн"},
		{"знатока", "знаток"},
		{"знать", "знат"},
		{"знаться", "знат"},
		{"значат", "значат"},
		{"значение", "значен"},
		{"значением", "значен"},
		{"значении", "значен"},
		{"значения", "значен"},
		{"значенье", "значен"},
		{"значила", "знач"},
		{"значили", "знач"},
		{"значило", "знач"},
		{"значит", "знач"},
		{"значительная", "значительн"},
		{"значительно", "значительн"},
		{"значительное", "значительн"},
		{"значительной", "значительн"},
		{"значительном", "значительн"},
		{"значительному", "значительн"},
		{"значительную", "значительн"},
		{"значительные", "значительн"},
		{"значительный", "значительн"},
		{"значительным", "значительн"},
		{"значительных", "значительн"},
		{"значить", "знач"},
		{"значишь", "знач"},
		{"значу", "знач"},
		{"знаю", "зна"},
		{"знают", "знают"},
		{"знающая", "знающ"},
		{"знающему", "знающ"},
		{"знающий", "знающ"},
		{"знающими", "знающ"},
		{"зная", "зна"},
		{"знобить", "зноб"},
		{"зное", "зно"},
		{"зноем", "зно"},
		{"зной", "зно"},
		{"знойкову", "знойков"},
		{"знойной", "знойн"},
		{"знойною", "знойн"},
		{"знойный", "знойн"},
		{"зною", "зно"},
		{"зноя", "зно"},
		{"зов", "зов"},
		{"зова", "зов"},
		{"зовем", "зов"},
		{"зовет", "зовет"},
		{"зовете", "зовет"},
		{"зовешь", "зовеш"},
		{"зови", "зов"},
		{"зовом", "зов"},
		{"зову", "зов"},
		{"зовут", "зовут"},
		{"зовущих", "зовущ"},
		{"зол", "зол"},
		{"зола", "зол"},
		{"золе", "зол"},
		{"золовка", "золовк"},
		{"золой", "зол"},
		{"золота", "золот"},
		{"золотая", "золот"},
		{"золоте", "золот"},
		{"золотеет", "золотеет"},
		{"золотистые", "золотист"},
		{"золотистый", "золотист"},
		{"золотистым", "золотист"},
		{"золотит", "золот"},
		{"золотится", "золот"},
		{"золотить", "золот"},
		{"золотника", "золотник"},
		{"золото", "золот"},
		{"золотоглавый", "золотоглав"},
		{"золотого", "золот"},
		{"золотое", "золот"},
		{"золотой", "золот"},
		{"золотокудрый", "золотокудр"},
		{"золотом", "золот"},
		{"золотопромышленника", "золотопромышленник"},
		{"золотопромышленность", "золотопромышлен"},
		{"золотою", "золот"},
		{"золоту", "золот"},
		{"золотухой", "золотух"},
		{"золотушный", "золотушн"},
		{"золотую", "золот"},
		{"золотые", "золот"},
		{"золотым", "золот"},
		{"золотыми", "золот"},
		{"золотых", "золот"},
		{"золу", "зол"},
		{"зонтик", "зонтик"},
		{"зонтика", "зонтик"},
		{"зонтиком", "зонтик"},
		{"зоргенфрею", "зоргенфр"},
		{"зори", "зор"},
		{"зоркая", "зорк"},
		{"зорким", "зорк"},
		{"зорко", "зорк"},
		{"зоркого", "зорк"},
		{"зорком", "зорк"},
		{"зорче", "зорч"},
		{"зорь", "зор"},
		{"зорю", "зор"},
		{"зосимов", "зосим"},
		{"зосимова", "зосимов"},
		{"зосимову", "зосимов"},
		{"зосимовым", "зосимов"},
		{"зрак", "зрак"},
		{"зрачки", "зрачк"},
		{"зреет", "зреет"},
		{"зрели", "зрел"},
		{"зрелище", "зрелищ"},
		{"зрелого", "зрел"},
		{"зрелой", "зрел"},
		{"зрелости", "зрелост"},
		{"зрелость", "зрелост"},
		{"зрелостью", "зрелост"},
		{"зрелый", "зрел"},
		{"зрелым", "зрел"},
		{"зрение", "зрен"},
		{"зрению", "зрен"},
		{"зрения", "зрен"},
		{"зренье", "зрен"},
		{"зреют", "зреют"},
		{"зримей", "зрим"},
		{"зримый", "зрим"},
		{"зрит", "зрит"},
		{"зрителей", "зрител"},
		{"зрители", "зрител"},
		{"зритель", "зрител"},
		{"зрителями", "зрител"},
		{"зришь", "зриш"},
		{"зря", "зря"},
		{"зуб", "зуб"},
		{"зубам", "зуб"},
		{"зубами", "зуб"},
		{"зубах", "зуб"},
		{"зубки", "зубк"},
		{"зубов", "зуб"},
		{"зубок", "зубок"},
		{"зубоскалить", "зубоскал"},
		{"зубчатой", "зубчат"},
		{"зубчатою", "зубчат"},
		{"зубчатый", "зубчат"},
		{"зубы", "зуб"},
		{"зудят", "зуд"},
		{"зурна", "зурн"},
		{"зурны", "зурн"},
		{"зыбкий", "зыбк"},
		{"зыбкой", "зыбк"},
		{"зыбком", "зыбк"},
		{"зыблется", "зыблет"},
		{"зятем", "зят"},
		{"и", "и"},
		{"ибо", "иб"},
		{"ибсен", "ибс"},
		{"ибсена", "ибс"},
		{"иван", "ива"},
		{"ивана", "ива"},
		{"иване", "иван"},
		{"иванов", "иван"},
		{"иванове", "иванов"},
		{"иванович", "иванович"},
		{"ивановича", "иванович"},
		{"ивановичу", "иванович"},
		{"ивановна", "ивановн"},
		{"ивановне", "ивановн"},
		{"ивановной", "ивановн"},
		{"ивановну", "ивановн"},
		{"ивановны", "ивановн"},
		{"ивановой", "иванов"},
		{"ивановскую", "ивановск"},
		{"иванову", "иванов"},
		{"иваном", "иван"},
		{"иванонва", "иванонв"},
		{"ивану", "иван"},
		{"иваныч", "иваныч"},
		{"иваныча", "иваныч"},
		{"иванычем", "иваныч"},
		{"иванычу", "иваныч"},
		{"ига", "иг"},
		{"игла", "игл"},
		{"иглой", "игл"},
		{"иглу", "игл"},
		{"иглы", "игл"},
		{"игнатьевич", "игнатьевич"},
		{"игнатьевича", "игнатьевич"},
		{"игнатьевичем", "игнатьевич"},
		{"игнатьевна", "игнатьевн"},
		{"игнатьевны", "игнатьевн"},
		{"игнатьич", "игнатьич"},
		{"игнатьича", "игнатьич"},
		{"игнашка", "игнашк"},
		{"иго", "иг"},
		{"иголка", "иголк"},
		{"иголочки", "иголочк"},
		{"игом", "иг"},
		{"игра", "игр"},
		{"играем", "игра"},
		{"играет", "игра"},
		{"играете", "игра"},
		{"играешь", "игра"},
		{"играешься", "игра"},
		{"играй", "игра"},
		{"играйте", "игра"},
		{"играл", "игра"},
		{"играла", "игра"},
		{"играли", "игра"},
		{"играло", "игра"},
		{"играль", "играл"},
		{"играми", "игр"},
		{"играть", "игра"},
		{"играх", "игр"},
		{"играю", "игра"},
		{"играют", "игра"},
		{"играючи", "играюч"},
		{"играющем", "игра"},
		{"играющею", "игра"},
		{"играющих", "игра"},
		{"играя", "игр"},
		{"игре", "игр"},
		{"игривая", "игрив"},
		{"игривее", "игрив"},
		{"игривенькая", "игривеньк"},
		{"игривого", "игрив"},
		{"игривости", "игрив"},
		{"игривостию", "игрив"},
		{"игривую", "игрив"},
		{"игривый", "игрив"},
		{"игрой", "игр"},
		{"игрок", "игрок"},
		{"игру", "игр"},
		{"игрушечку", "игрушечк"},
		{"игрушки", "игрушк"},
		{"игры", "игр"},
		{"идеал", "идеа"},
		{"идеала", "идеа"},
		{"идеалам", "идеал"},
		{"идеале", "идеал"},
		{"идеален", "идеал"},
		{"идеализм", "идеализм"},
		{"идеалист", "идеалист"},
		{"идеалов", "идеал"},
		{"идеалом", "идеал"},
		{"идеалу", "идеал"},
		{"идеалы", "идеал"},
		{"идеальная", "идеальн"},
		{"идеально", "идеальн"},
		{"идеальный", "идеальн"},
		{"идее", "ид"},
		{"идеей", "иде"},
		{"идеи", "ид"},
		{"идей", "ид"},
		{"идейка", "идейк"},
		{"идейку", "идейк"},
		{"идейных", "идейн"},
		{"идем", "ид"},
		{"идемте", "идемт"},
		{"идет", "идет"},
		{"идете", "идет"},
		{"идеть", "идет"},
		{"идешь", "идеш"},
		{"идею", "ид"},
		{"идея", "иде"},
		{"иди", "ид"},
		{"идиллией", "идилл"},
		{"идиллий", "идилл"},
		{"идиот", "идиот"},
		{"идиотка", "идиотк"},
		{"идиотством", "идиотств"},
		{"идите", "ид"},
		{"идола", "идол"},
		{"идолов", "идол"},
		{"идолом", "идол"},
		{"идолу", "идол"},
		{"идти", "идт"},
		{"иду", "ид"},
		{"идут", "идут"},
		{"идучи", "идуч"},
		{"идущего", "идущ"},
		{"идущей", "идущ"},
		{"идущему", "идущ"},
		{"идущие", "идущ"},
		{"идущим", "идущ"},
		{"идущими", "идущ"},
		{"идущих", "идущ"},
		{"идущую", "идущ"},
		{"идя", "ид"},
		{"иезуитов", "иезуит"},
		{"иене", "иен"},
		{"иерей", "иер"},
		{"иероглифов", "иероглиф"},
		{"иерусалим", "иерусал"},
		{"иерусалима", "иерусалим"},
		{"ижицы", "ижиц"},
		{"из", "из"},
		{"изб", "изб"},
		{"изба", "изб"},
		{"избавил", "избав"},
		{"избавились", "избав"},
		{"избавите", "избав"},
		{"избавить", "избав"},
		{"избавлен", "избавл"},
		{"избавление", "избавлен"},
		{"избавленным", "избавлен"},
		{"избавлялись", "избавля"},
		{"избавляя", "избавл"},
		{"избавь", "избав"},
		{"избаловали", "избалова"},
		{"избаловались", "избалова"},
		{"избаловался", "избалова"},
		{"избалован", "избалова"},
		{"избалованы", "избалова"},
		{"избах", "изб"},
		{"избе", "изб"},
		{"избегайте", "избега"},
		{"избегал", "избега"},
		{"избегала", "избега"},
		{"избегали", "избега"},
		{"избегать", "избега"},
		{"избегая", "избег"},
		{"избегнул", "избегнул"},
		{"избегнуть", "избегнут"},
		{"избежать", "избежа"},
		{"изберет", "изберет"},
		{"изберешь", "избереш"},
		{"избил", "изб"},
		{"избитая", "избит"},
		{"избито", "избит"},
		{"избитой", "избит"},
		{"избитую", "избит"},
		{"избитых", "избит"},
		{"избрав", "избра"},
		{"избрал", "избра"},
		{"избрала", "избра"},
		{"избранной", "избра"},
		{"избранном", "избра"},
		{"избранные", "избра"},
		{"избранный", "избра"},
		{"избранных", "избра"},
		{"избранья", "избран"},
		{"избрать", "избра"},
		{"избу", "изб"},
		{"избудешь", "избудеш"},
		{"избушке", "избушк"},
		{"избы", "изб"},
		{"избытка", "избытк"},
		{"избытке", "избытк"},
		{"избыток", "избыток"},
		{"избыть", "изб"},
		{"изведав", "изведа"},
		{"изведал", "изведа"},
		{"изведала", "изведа"},
		{"изведаны", "изведа"},
		{"изведать", "изведа"},
		{"изведешь", "изведеш"},
		{"извел", "извел"},
		{"изверг", "изверг"},
		{"изверга", "изверг"},
		{"извергнете", "извергнет"},
		{"извергнуть", "извергнут"},
		{"изверившись", "извер"},
		{"изверились", "извер"},
		{"известен", "извест"},
		{"известившись", "извест"},
		{"известие", "извест"},
		{"известием", "извест"},
		{"известий", "извест"},
		{"известия", "извест"},
		{"известиям", "извест"},
		{"известиями", "извест"},
		{"известка", "известк"},
		{"известке", "известк"},
		{"известковую", "известков"},
		{"известная", "известн"},
		{"известно", "известн"},
		{"известного", "известн"},
		{"известное", "известн"},
		{"известной", "известн"},
		{"известном", "известн"},
		{"известному", "известн"},
		{"известности", "известн"},
		{"известность", "известн"},
		{"известною", "известн"},
		{"известную", "известн"},
		{"известны", "известн"},
		{"известные", "известн"},
		{"известный", "известн"},
		{"известным", "известн"},
		{"известными", "известн"},
		{"известных", "известн"},
		{"известью", "извест"},
		{"известясь", "извест"},
		{"извечно", "извечн"},
		{"извечной", "извечн"},
		{"извещение", "извещен"},
		{"извещены", "извещ"},
		{"извив", "изв"},
		{"извиваясь", "извив"},
		{"извивом", "извив"},
		{"извилистой", "извилист"},
		{"извинение", "извинен"},
		{"извинения", "извинен"},
		{"извини", "извин"},
		{"извинилась", "извин"},
		{"извинили", "извин"},
		{"извинился", "извин"},
		{"извините", "извин"},
		{"извинить", "извин"},
		{"извиниться", "извин"},
		{"извинюсь", "извин"},
		{"извиняешься", "извиня"},
		{"извинялся", "извиня"},
		{"извинят", "извин"},
		{"извиняться", "извиня"},
		{"извиняюсь", "извиня"},
		{"извиняются", "извиня"},
		{"извиняя", "извин"},
		{"извиняясь", "извин"},
		{"извлек", "извлек"},
		{"извлекал", "извлека"},
		{"извлекли", "извлекл"},
		{"извлеку", "извлек"},
		{"извлечь", "извлеч"},
		{"извне", "извн"},
		{"изводят", "извод"},
		{"извозчик", "извозчик"},
		{"извозчика", "извозчик"},
		{"извозчикам", "извозчик"},
		{"извозчике", "извозчик"},
		{"извозчики", "извозчик"},
		{"извозчиков", "извозчик"},
		{"извозчиком", "извозчик"},
		{"извозчику", "извозчик"},
		{"изволив", "извол"},
		{"изволил", "извол"},
		{"изволила", "извол"},
		{"изволили", "извол"},
		{"изволит", "извол"},
		{"изволите", "извол"},
		{"изволь", "извол"},
		{"извольте", "извольт"},
		{"изволят", "извол"},
		{"изворачивался", "изворачива"},
		{"изворачиваться", "изворачива"},
		{"изворотиться", "изворот"},
		{"изворотливости", "изворотлив"},
		{"изворотов", "изворот"},
		{"извращенном", "извращен"},
		{"изгадил", "изгад"},
		{"изгадили", "изгад"},
		{"изгадит", "изгад"},
		{"изгиба", "изгиб"},
		{"изгибе", "изгиб"},
		{"изгладилась", "изглад"},
		{"изгладилось", "изглад"},
		{"изгнанник", "изгнанник"},
		{"изголовье", "изголов"},
		{"изголовья", "изголов"},
		{"изгонишь", "изгон"},
		{"изгоняется", "изгоня"},
		{"изготовилась", "изготов"},
		{"изготовить", "изготов"},
		{"изготовлять", "изготовля"},
		{"издавать", "издава"},
		{"издавна", "издавн"},
		{"издалека", "издалек"},
		{"издали", "изда"},
		{"издало", "изда"},
		{"издание", "издан"},
		{"издании", "издан"},
		{"изданию", "издан"},
		{"издания", "издан"},
		{"изданьица", "изданьиц"},
		{"издателей", "издател"},
		{"издатели", "издател"},
		{"издательской", "издательск"},
		{"издательство", "издательств"},
		{"издателям", "издател"},
		{"издать", "изда"},
		{"издевается", "издева"},
		{"издеваться", "издева"},
		{"изделия", "издел"},
		{"издерганы", "издерга"},
		{"издержанного", "издержа"},
		{"издержек", "издержек"},
		{"издержка", "издержк"},
		{"издержкам", "издержк"},
		{"издержками", "издержк"},
		{"издержке", "издержк"},
		{"издержки", "издержк"},
		{"издохнем", "издохн"},
		{"издохнуть", "издохнут"},
		{"издрогнувший", "издрогнувш"},
		{"издрогшая", "издрогш"},
		{"издыхании", "издыхан"},
		{"издыхающей", "издыха"},
		{"изжарил", "изжар"},
		{"изжога", "изжог"},
		{"иззяб", "иззяб"},
		{"излагаете", "излага"},
		{"излагал", "излага"},
		{"излагать", "излага"},
		{"излагающего", "излага"},
		{"излагая", "излаг"},
		{"излер", "излер"},
		{"излечивать", "излечива"},
		{"излив", "изл"},
		{"изливавшейся", "излива"},
		{"изливается", "излива"},
		{"изливали", "излива"},
		{"изливались", "излива"},
		{"излить", "изл"},
		{"излишек", "излишек"},
		{"излишне", "излишн"},
		{"излишней", "излишн"},
		{"излишнею", "излишн"},
		{"излишним", "излишн"},
		{"излишнюю", "излишн"},
		{"излияний", "излиян"},
		{"излияния", "излиян"},
		{"изловили", "излов"},
		{"изловчусь", "изловч"},
		{"изложением", "изложен"},
		{"изложения", "изложен"},
		{"изложил", "излож"},
		{"изложили", "излож"},
		{"изложить", "излож"},
		{"изложу", "излож"},
		{"изломал", "излома"},
		{"изломала", "излома"},
		{"изломалась", "излома"},
		{"изломались", "излома"},
		{"изломан", "излома"},
		{"изломанной", "излома"},
		{"изломанными", "излома"},
		{"изломанных", "излома"},
		{"изломано", "излома"},
		{"изломаться", "излома"},
		{"излученье", "излучен"},
		{"измайлов", "измайл"},
		{"измалетства", "измалетств"},
		{"измельчал", "измельча"},
		{"измен", "изм"},
		{"измена", "изм"},
		{"изменение", "изменен"},
		{"изменению", "изменен"},
		{"изменив", "измен"},
		{"изменившимся", "измен"},
		{"изменил", "измен"},
		{"изменилась", "измен"},
		{"изменили", "измен"},
		{"изменились", "измен"},
		{"изменилось", "измен"},
		{"изменился", "измен"},
		{"изменит", "измен"},
		{"изменится", "измен"},
		{"изменить", "измен"},
		{"измениться", "измен"},
		{"изменнику", "изменник"},
		{"изменница", "изменниц"},
		{"изменю", "измен"},
		{"изменюсь", "измен"},
		{"изменяем", "изменя"},
		{"изменяет", "изменя"},
		{"изменяется", "изменя"},
		{"изменяла", "изменя"},
		{"изменят", "измен"},
		{"изменятся", "измен"},
		{"изменять", "изменя"},
		{"изменяющий", "изменя"},
		{"изменяя", "измен"},
		{"измерили", "измер"},
		{"измерит", "измер"},
		{"измерить", "измер"},
		{"измерял", "измеря"},
		{"измеряла", "измеря"},
		{"измеряли", "измеря"},
		{"измеряя", "измер"},
		{"измокшее", "измокш"},
		{"измокшем", "измокш"},
		{"изморились", "измор"},
		{"измучена", "измуч"},
		{"измученная", "измучен"},
		{"измученное", "измучен"},
		{"измученному", "измучен"},
		{"измученный", "измучен"},
		{"измучившееся", "измуч"},
		{"измучившись", "измуч"},
		{"измучил", "измуч"},
		{"измучилась", "измуч"},
		{"измучили", "измуч"},
		{"измучились", "измуч"},
		{"измучило", "измуч"},
		{"измучился", "измуч"},
		{"измучиться", "измуч"},
		{"измучишь", "измуч"},
		{"измучусь", "измуч"},
		{"измята", "измят"},
		{"измятая", "измят"},
		{"измято", "измят"},
		{"измятую", "измят"},
		{"измятые", "измят"},
		{"измятый", "измят"},
		{"изнеженному", "изнежен"},
		{"изнеженные", "изнежен"},
		{"изнеженным", "изнежен"},
		{"изнемог", "изнемог"},
		{"изнемогает", "изнемога"},
		{"изнемогал", "изнемога"},
		{"изнемогала", "изнемога"},
		{"изнемогаю", "изнемога"},
		{"изнемогают", "изнемога"},
		{"изнемогая", "изнемог"},
		{"изнемогли", "изнемогл"},
		{"изнеможение", "изнеможен"},
		{"изнеможении", "изнеможен"},
		{"изнеможения", "изнеможен"},
		{"изнемочь", "изнемоч"},
		{"износишь", "износ"},
		{"изношенная", "изношен"},
		{"изношенного", "изношен"},
		{"изношенный", "изношен"},
		{"изношенными", "изношен"},
		{"изнурения", "изнурен"},
		{"изнуренное", "изнурен"},
		{"изнутри", "изнутр"},
		{"изо", "из"},
		{"изобличена", "изоблич"},
		{"изображавших", "изобража"},
		{"изображайте", "изобража"},
		{"изображал", "изобража"},
		{"изображать", "изобража"},
		{"изображают", "изобража"},
		{"изображен", "изображ"},
		{"изображена", "изображ"},
		{"изображение", "изображен"},
		{"изображу", "изображ"},
		{"изобрази", "изобраз"},
		{"изобразилось", "изобраз"},
		{"изобразительнее", "изобразительн"},
		{"изобразить", "изобраз"},
		{"изобразят", "изобраз"},
		{"изобретал", "изобрета"},
		{"изобретатель", "изобретател"},
		{"изобретений", "изобретен"},
		{"изобретенном", "изобретен"},
		{"изобьешь", "изобьеш"},
		{"изовраться", "изовра"},
		{"изодранные", "изодра"},
		{"изодранными", "изодра"},
		{"изойти", "изойт"},
		{"изорвал", "изорва"},
		{"изорваль", "изорвал"},
		{"изорванное", "изорва"},
		{"изошла", "изошл"},
		{"изощрены", "изощр"},
		{"изощряется", "изощря"},
		{"израсходовано", "израсходова"},
		{"изредка", "изредк"},
		{"изрек", "изрек"},
		{"изрекаете", "изрека"},
		{"изрекал", "изрека"},
		{"изрекала", "изрека"},
		{"изтопталь", "изтоптал"},
		{"изувера", "изувер"},
		{"изуверство", "изуверств"},
		{"изукрашенною", "изукрашен"},
		{"изумил", "изум"},
		{"изумились", "изум"},
		{"изумился", "изум"},
		{"изумительно", "изумительн"},
		{"изумительного", "изумительн"},
		{"изумительной", "изумительн"},
		{"изумительною", "изумительн"},
		{"изумительные", "изумительн"},
		{"изумительный", "изумительн"},
		{"изумится", "изум"},
		{"изумить", "изум"},
		{"изумлен", "изумл"},
		{"изумление", "изумлен"},
		{"изумлением", "изумлен"},
		{"изумлении", "изумлен"},
		{"изумлению", "изумлен"},
		{"изумления", "изумлен"},
		{"изумленный", "изумлен"},
		{"изумленными", "изумлен"},
		{"изумленных", "изумлен"},
		{"изуродованный", "изуродова"},
		{"изустный", "изустн"},
		{"изучавший", "изуча"},
		{"изучал", "изуча"},
		{"изучать", "изуча"},
		{"изучая", "изуч"},
		{"изучение", "изучен"},
		{"изученно", "изучен"},
		{"изучил", "изуч"},
		{"изучила", "изуч"},
		{"изучить", "изуч"},
		{"изъездить", "изъезд"},
		{"изъявил", "изъяв"},
		{"изъявили", "изъяв"},
		{"изъявления", "изъявлен"},
		{"изъявлял", "изъявля"},
		{"изъяснил", "изъясн"},
		{"изъяснить", "изъясн"},
		{"изъясняю", "изъясня"},
		{"изъятья", "изъят"},
		{"изысканная", "изыска"},
		{"изысканно", "изыска"},
		{"изыскать", "изыска"},
		{"изюминки", "изюминк"},
		{"изюму", "изюм"},
		{"изяществом", "изяществ"},
		{"изящна", "изящн"},
		{"изящная", "изящн"},
		{"изящно", "изящн"},
		{"изящного", "изящн"},
		{"изящной", "изящн"},
		{"изящную", "изящн"},
		{"изящны", "изящн"},
		{"изящные", "изящн"},
		{"изящный", "изящн"},
		{"изящным", "изящн"},
		{"изящных", "изящн"},
		{"иисус", "иисус"},
		{"иисусу", "иисус"},
		{"икнув", "икнув"},
		{"иконе", "икон"},
		{"иконостас", "иконостас"},
		{"икону", "икон"},
		{"иконы", "икон"},
		{"ил", "ил"},
		{"или", "ил"},
		{"илиаду", "илиад"},
		{"илистом", "илист"},
		{"иль", "ил"},
		{"илье", "ил"},
		{"ильей", "ил"},
		{"ильи", "ил"},
		{"ильин", "ильин"},
		{"ильина", "ильин"},
		{"ильине", "ильин"},
		{"ильинична", "ильиничн"},
		{"ильинская", "ильинск"},
		{"ильинские", "ильинск"},
		{"ильинским", "ильинск"},
		{"ильинскими", "ильинск"},
		{"ильинских", "ильинск"},
		{"ильинской", "ильинск"},
		{"ильинскую", "ильинск"},
		{"ильич", "ильич"},
		{"ильича", "ильич"},
		{"ильиче", "ильич"},
		{"ильичом", "ильич"},
		{"ильичу", "ильич"},
		{"илью", "ил"},
		{"илья", "ил"},
		{"илюша", "илюш"},
		{"илюше", "илюш"},
		{"илюши", "илюш"},
		{"илюшу", "илюш"},
		{"им", "им"},
		{"иматру", "иматр"},
		{"имевшая", "имевш"},
		{"имевшего", "имевш"},
		{"имевшей", "имевш"},
		{"имевшему", "имевш"},
		{"имевшею", "имевш"},
		{"имевшиеся", "имевш"},
		{"имеем", "име"},
		{"имеет", "имеет"},
		{"имеете", "имеет"},
		{"имеется", "имеет"},
		{"имеешь", "имееш"},
		{"имей", "им"},
		{"имейте", "им"},
		{"имел", "имел"},
		{"имела", "имел"},
		{"имели", "имел"},
		{"имелись", "имел"},
		{"имело", "имел"},
		{"имелось", "имел"},
		{"имен", "им"},
		{"имена", "им"},
		{"именами", "имен"},
		{"именем", "имен"},
		{"имени", "имен"},
		{"имение", "имен"},
		{"имением", "имен"},
		{"имении", "имен"},
		{"именин", "именин"},
		{"именинам", "именин"},
		{"именинах", "именин"},
		{"именинники", "именинник"},
		{"именинница", "именинниц"},
		{"именинницей", "именинниц"},
		{"именины", "именин"},
		{"имению", "имен"},
		{"имения", "имен"},
		{"именно", "имен"},
		{"именьишко", "именьишк"},
		{"иметь", "имет"},
		{"имею", "им"},
		{"имеют", "имеют"},
		{"имеющего", "имеющ"},
		{"имеющему", "имеющ"},
		{"имеющий", "имеющ"},
		{"имеющих", "имеющ"},
		{"имея", "име"},
		{"ими", "им"},
		{"импровизирует", "импровизир"},
		{"имущества", "имуществ"},
		{"имущественные", "имуществен"},
		{"имущество", "имуществ"},
		{"имя", "им"},
		{"иначе", "инач"},
		{"иная", "ин"},
		{"инда", "инд"},
		{"индеек", "индеек"},
		{"индейка", "индейк"},
		{"индейки", "индейк"},
		{"индейку", "индейк"},
		{"индейский", "индейск"},
		{"индивидуализма", "индивидуализм"},
		{"индивидууму", "индивидуум"},
		{"индийского", "индийск"},
		{"индифферентизм", "индифферентизм"},
		{"индию", "инд"},
		{"инеем", "ине"},
		{"иней", "ин"},
		{"инициалы", "инициал"},
		{"инкрустацией", "инкрустац"},
		{"ино", "ин"},
		{"иногда", "иногд"},
		{"иного", "ин"},
		{"иное", "ин"},
		{"иной", "ин"},
		{"инока", "инок"},
		{"иному", "ин"},
		{"иностранец", "иностранец"},
		{"иностранка", "иностранк"},
		{"иностранного", "иностра"},
		{"иностранной", "иностра"},
		{"иностранные", "иностра"},
		{"иностранцам", "иностранц"},
		{"иностранцами", "иностранц"},
		{"иностранцем", "иностранц"},
		{"иностранцы", "иностранц"},
		{"иною", "ин"},
		{"инспектором", "инспектор"},
		{"инстанции", "инстанц"},
		{"инстанцию", "инстанц"},
		{"инстинкт", "инстинкт"},
		{"инстинкта", "инстинкт"},
		{"инстинктами", "инстинкт"},
		{"инстинктивно", "инстинктивн"},
		{"инстинктивное", "инстинктивн"},
		{"инстинктом", "инстинкт"},
		{"инстинкту", "инстинкт"},
		{"институт", "институт"},
		{"института", "институт"},
		{"институте", "институт"},
		{"инструкцию", "инструкц"},
		{"инструкция", "инструкц"},
		{"инструмент", "инструмент"},
		{"инструмента", "инструмент"},
		{"инструментами", "инструмент"},
		{"инструменте", "инструмент"},
		{"инструментом", "инструмент"},
		{"интегральное", "интегральн"},
		{"интеллигента", "интеллигент"},
		{"интеллигентных", "интеллигентн"},
		{"интеллигенцией", "интеллигенц"},
		{"интеллигенции", "интеллигенц"},
		{"интендантов", "интендант"},
		{"интерес", "интерес"},
		{"интереса", "интерес"},
		{"интересах", "интерес"},
		{"интересе", "интерес"},
		{"интересен", "интерес"},
		{"интересная", "интересн"},
		{"интереснейшем", "интересн"},
		{"интереснейшую", "интересн"},
		{"интересно", "интересн"},
		{"интересного", "интересн"},
		{"интересное", "интересн"},
		{"интересной", "интересн"},
		{"интересном", "интересн"},
		{"интересны", "интересн"},
		{"интересные", "интересн"},
		{"интересный", "интересн"},
		{"интересным", "интересн"},
		{"интересных", "интересн"},
		{"интересов", "интерес"},
		{"интересовал", "интересова"},
		{"интересовала", "интересова"},
		{"интересовало", "интересова"},
		{"интересовался", "интересова"},
		{"интересовать", "интересова"},
		{"интересоваться", "интересова"},
		{"интересом", "интерес"},
		{"интересует", "интерес"},
		{"интересуете", "интересует"},
		{"интересуетесь", "интересует"},
		{"интересуется", "интерес"},
		{"интересуюсь", "интерес"},
		{"интересуются", "интерес"},
		{"интересуясь", "интересу"},
		{"интересы", "интерес"},
		{"интимном", "интимн"},
		{"интимностей", "интимн"},
		{"интонацией", "интонац"},
		{"интонации", "интонац"},
		{"интриги", "интриг"},
		{"интриговать", "интригова"},
		{"интригует", "интриг"},
		{"ину", "ин"},
		{"иную", "ин"},
		{"инфлюэнца", "инфлюэнц"},
		{"информацию", "информац"},
		{"иные", "ин"},
		{"иным", "ин"},
		{"иными", "ин"},
		{"иных", "ин"},
		{"иоаннова", "иоаннов"},
		{"иоган", "иога"},
		{"ипохондрией", "ипохондр"},
		{"ипохондрик", "ипохондрик"},
		{"ипохондрику", "ипохондрик"},
		{"ипохондрическое", "ипохондрическ"},
		{"ипохондрию", "ипохондр"},
		{"ирина", "ирин"},
		{"ирине", "ирин"},
		{"ириной", "ирин"},
		{"ирину", "ирин"},
		{"ирины", "ирин"},
		{"ирис", "ирис"},
		{"ирисы", "ирис"},
		{"иронией", "ирон"},
		{"иронически", "ироническ"},
		{"иронический", "ироническ"},
		{"иртыша", "иртыш"},
		{"исай", "иса"},
		{"исая", "ис"},
		{"искажал", "искажа"},
		{"искажать", "искажа"},
		{"искажая", "искаж"},
		{"искажение", "искажен"},
		{"искажением", "искажен"},
		{"искажению", "искажен"},
		{"искаженным", "искажен"},
		{"искажены", "искаж"},
		{"исказила", "исказ"},
		{"исказили", "исказ"},
		{"исказилось", "исказ"},
		{"исказить", "исказ"},
		{"искал", "иска"},
		{"искала", "иска"},
		{"искалеченной", "искалечен"},
		{"искалеченным", "искалечен"},
		{"искали", "иска"},
		{"искам", "иск"},
		{"исканий", "искан"},
		{"искания", "искан"},
		{"искательниц", "искательниц"},
		{"искать", "иска"},
		{"исключая", "исключ"},
		{"исключением", "исключен"},
		{"исключений", "исключен"},
		{"исключения", "исключен"},
		{"исключениями", "исключен"},
		{"исключительно", "исключительн"},
		{"исключительное", "исключительн"},
		{"исключительном", "исключительн"},
		{"исключительный", "исключительн"},
		{"исковеркана", "исковерка"},
		{"исковерканную", "исковерка"},
		{"исковеркано", "исковерка"},
		{"искони", "искон"},
		{"искоренено", "искорен"},
		{"искоренить", "искорен"},
		{"искореняет", "искореня"},
		{"искоса", "искос"},
		{"искр", "искр"},
		{"искра", "искр"},
		{"искрами", "искр"},
		{"искренен", "искрен"},
		{"искренна", "искрен"},
		{"искренне", "искрен"},
		{"искреннего", "искрен"},
		{"искреннее", "искрен"},
		{"искренней", "искрен"},
		{"искреннему", "искрен"},
		{"искренние", "искрен"},
		{"искренний", "искрен"},
		{"искренним", "искрен"},
		{"искренно", "искрен"},
		{"искренности", "искрен"},
		{"искренность", "искрен"},
		{"искренностью", "искрен"},
		{"искренняя", "искрен"},
		{"искривившеюся", "искрив"},
		{"искривившимися", "искрив"},
		{"искривившимся", "искрив"},
		{"искривила", "искрив"},
		{"искривилось", "искрив"},
		{"искривленном", "искривлен"},
		{"искривленною", "искривлен"},
		{"искривленный", "искривлен"},
		{"искривленным", "искривлен"},
		{"искривлено", "искривл"},
		{"искрились", "искр"},
		{"искрой", "искр"},
		{"искрометное", "искрометн"},
		{"искры", "искр"},
		{"искупался", "искупа"},
		{"искупит", "искуп"},
		{"искупительный", "искупительн"},
		{"искупителя", "искупител"},
		{"искупить", "искуп"},
		{"искупление", "искуплен"},
		{"искусно", "искусн"},
		{"искусным", "искусн"},
		{"искусными", "искусн"},
		{"искусств", "искусств"},
		{"искусства", "искусств"},
		{"искусствах", "искусств"},
		{"искусстве", "искусств"},
		{"искусственная", "искусствен"},
		{"искусственно", "искусствен"},
		{"искусственной", "искусствен"},
		{"искусство", "искусств"},
		{"искусству", "искусств"},
		{"искушенье", "искушен"},
		{"испакостили", "испак"},
		{"испанец", "испанец"},
		{"испанию", "испан"},
		{"испанке", "испанк"},
		{"испанскую", "испанск"},
		{"испарения", "испарен"},
		{"испарениями", "испарен"},
		{"испачкав", "испачка"},
		{"испачкался", "испачка"},
		{"испейте", "исп"},
		{"испекли", "испекл"},
		{"испекут", "испекут"},
		{"испепелить", "испепел"},
		{"испепеляешь", "испепеля"},
		{"испепеляющие", "испепеля"},
		{"испечет", "испечет"},
		{"исписанная", "исписа"},
		{"исписанной", "исписа"},
		{"исписанную", "исписа"},
		{"исписаны", "исписа"},
		{"испитое", "испит"},
		{"испитым", "испит"},
		{"испить", "исп"},
		{"исповеди", "исповед"},
		{"исповедовал", "исповедова"},
		{"исповедовала", "исповедова"},
		{"исповедовать", "исповедова"},
		{"исповедывал", "исповедыва"},
		{"исповедь", "исповед"},
		{"исповедью", "исповед"},
		{"исподлобья", "исподлоб"},
		{"исподтишка", "исподтишк"},
		{"исполинский", "исполинск"},
		{"исполнен", "исполн"},
		{"исполнена", "исполн"},
		{"исполнение", "исполнен"},
		{"исполнением", "исполнен"},
		{"исполнении", "исполнен"},
		{"исполнения", "исполнен"},
		{"исполненного", "исполнен"},
		{"исполненное", "исполнен"},
		{"исполненной", "исполнен"},
		{"исполненный", "исполнен"},
		{"исполненных", "исполнен"},
		{"исполнено", "исполн"},
		{"исполнены", "исполн"},
		{"исполнив", "исполн"},
		{"исполнил", "исполн"},
		{"исполнила", "исполн"},
		{"исполнились", "исполн"},
		{"исполнилось", "исполн"},
		{"исполнимость", "исполним"},
		{"исполнит", "исполн"},
		{"исполните", "исполн"},
		{"исполнитель", "исполнител"},
		{"исполнителя", "исполнител"},
		{"исполнится", "исполн"},
		{"исполнить", "исполн"},
		{"исполниться", "исполн"},
		{"исполнишь", "исполн"},
		{"исполню", "исполн"},
		{"исполняет", "исполня"},
		{"исполнял", "исполня"},
		{"исполняла", "исполня"},
		{"исполнять", "исполня"},
		{"исполняю", "исполня"},
		{"исполняют", "исполня"},
		{"исполняя", "исполн"},
		{"использован", "использова"},
		{"использование", "использован"},
		{"испортил", "испорт"},
		{"испортилось", "испорт"},
		{"испортите", "испорт"},
		{"испортить", "испорт"},
		{"испортишь", "испорт"},
		{"испорчен", "испорч"},
		{"испорченные", "испорчен"},
		{"испорченный", "испорчен"},
		{"испорчу", "испорч"},
		{"испошлился", "испошл"},
		{"исправился", "исправ"},
		{"исправимая", "исправим"},
		{"исправить", "исправ"},
		{"исправишь", "исправ"},
		{"исправлением", "исправлен"},
		{"исправления", "исправлен"},
		{"исправляет", "исправля"},
		{"исправлять", "исправля"},
		{"исправника", "исправник"},
		{"исправнику", "исправник"},
		{"исправно", "исправн"},
		{"исправным", "исправн"},
		{"исправных", "исправн"},
		{"испрашивать", "испрашива"},
		{"испросив", "испрос"},
		{"испуг", "испуг"},
		{"испуга", "испуг"},
		{"испугавшись", "испуга"},
		{"испугается", "испуга"},
		{"испугал", "испуга"},
		{"испугалась", "испуга"},
		{"испугали", "испуга"},
		{"испугались", "испуга"},
		{"испугался", "испуга"},
		{"испугаль", "испугал"},
		{"испуган", "испуга"},
		{"испуганная", "испуга"},
		{"испуганно", "испуга"},
		{"испуганного", "испуга"},
		{"испуганное", "испуга"},
		{"испуганной", "испуга"},
		{"испуганном", "испуга"},
		{"испуганные", "испуга"},
		{"испуганный", "испуга"},
		{"испуганным", "испуга"},
		{"испуганными", "испуга"},
		{"испуганы", "испуга"},
		{"испугать", "испуга"},
		{"испугаться", "испуга"},
		{"испугаюсь", "испуга"},
		{"испуге", "испуг"},
		{"испугом", "испуг"},
		{"испугу", "испуг"},
		{"испужаешь", "испужа"},
		{"испужались", "испужа"},
		{"испускала", "испуска"},
		{"испуская", "испуск"},
		{"испустил", "испуст"},
		{"испытав", "испыта"},
		{"испытавшей", "испыта"},
		{"испытавший", "испыта"},
		{"испытал", "испыта"},
		{"испытала", "испыта"},
		{"испытание", "испытан"},
		{"испытаний", "испытан"},
		{"испытания", "испытан"},
		{"испытанные", "испыта"},
		{"испытанный", "испыта"},
		{"испытать", "испыта"},
		{"испытаю", "испыта"},
		{"испытующии", "испытующ"},
		{"испытующим", "испыт"},
		{"испытывает", "испытыва"},
		{"испытывал", "испытыва"},
		{"испытывала", "испытыва"},
		{"испытывали", "испытыва"},
		{"испытывать", "испытыва"},
		{"испытываю", "испытыва"},
		{"исследования", "исследован"},
		{"исследованы", "исследова"},
		{"исследую", "исслед"},
		{"иссосали", "иссоса"},
		{"иссохшее", "иссохш"},
		{"иссохшей", "иссохш"},
		{"иссохшие", "иссохш"},
		{"иссохшими", "иссохш"},
		{"иссохших", "иссохш"},
		{"исстрадаешься", "исстрада"},
		{"исстрадались", "исстрада"},
		{"исстрадался", "исстрада"},
		{"исступление", "исступлен"},
		{"исступлении", "исступлен"},
		{"исступлений", "исступлен"},
		{"исступления", "исступлен"},
		{"исступленно", "исступлен"},
		{"исступленном", "исступлен"},
		{"исступленному", "исступлен"},
		{"истаскался", "истаска"},
		{"истасканное", "истаска"},
		{"истасканные", "истаска"},
		{"истасканных", "истаска"},
		{"истаял", "истая"},
		{"истерзал", "истерза"},
		{"истерзала", "истерза"},
		{"истерзали", "истерза"},
		{"истерзан", "истерза"},
		{"истерзана", "истерза"},
		{"истерзанной", "истерза"},
		{"истерзанный", "истерза"},
		{"истерика", "истерик"},
		{"истерике", "истерик"},
		{"истерически", "истерическ"},
		{"истерическим", "истерическ"},
		{"истерических", "истерическ"},
		{"истерического", "истерическ"},
		{"истеричны", "истеричн"},
		{"истечении", "истечен"},
		{"истин", "истин"},
		{"истина", "истин"},
		{"истинам", "истин"},
		{"истине", "истин"},
		{"истинно", "истин"},
		{"истинное", "истин"},
		{"истинной", "истин"},
		{"истинном", "истин"},
		{"истинные", "истин"},
		{"истинный", "истин"},
		{"истинным", "истин"},
		{"истинных", "истин"},
		{"истиной", "истин"},
		{"истину", "истин"},
		{"истины", "истин"},
		{"истолкование", "истолкован"},
		{"истолкования", "истолкован"},
		{"истолкователя", "истолковател"},
		{"истома", "истом"},
		{"истоме", "истом"},
		{"истомилась", "истом"},
		{"исторг", "исторг"},
		{"историей", "истор"},
		{"истории", "истор"},
		{"историк", "историк"},
		{"историческим", "историческ"},
		{"исторического", "историческ"},
		{"историю", "истор"},
		{"история", "истор"},
		{"источник", "источник"},
		{"источника", "источник"},
		{"источнике", "источник"},
		{"источники", "источник"},
		{"источников", "источник"},
		{"истощался", "истоща"},
		{"истощена", "истощ"},
		{"истощенный", "истощен"},
		{"истощив", "истощ"},
		{"истощилась", "истощ"},
		{"истощился", "истощ"},
		{"истощится", "истощ"},
		{"истратила", "истрат"},
		{"истратили", "истрат"},
		{"истраченный", "истрачен"},
		{"истраченных", "истрачен"},
		{"истребил", "истреб"},
		{"истребить", "истреб"},
		{"истребленных", "истреблен"},
		{"истребленья", "истреблен"},
		{"истребляли", "истребля"},
		{"истреблять", "истребля"},
		{"истрепанная", "истрепа"},
		{"истрепанное", "истрепа"},
		{"истрепанном", "истрепа"},
		{"истрепанных", "истрепа"},
		{"истязал", "истяза"},
		{"истязаний", "истязан"},
		{"истязания", "истязан"},
		{"исход", "исход"},
		{"исхода", "исход"},
		{"исходе", "исход"},
		{"исходить", "исход"},
		{"исходя", "исход"},
		{"исхудавшем", "исхуда"},
		{"исхудалом", "исхудал"},
		{"исцеленья", "исцелен"},
		{"исцеляющую", "исцеля"},
		{"исчахнет", "исчахнет"},
		{"исчез", "исчез"},
		{"исчезает", "исчеза"},
		{"исчезал", "исчеза"},
		{"исчезала", "исчеза"},
		{"исчезали", "исчеза"},
		{"исчезало", "исчеза"},
		{"исчезать", "исчеза"},
		{"исчезают", "исчеза"},
		{"исчезла", "исчезл"},
		{"исчезли", "исчезл"},
		{"исчезло", "исчезл"},
		{"исчезнет", "исчезнет"},
		{"исчезнете", "исчезнет"},
		{"исчезни", "исчезн"},
		{"исчезнут", "исчезнут"},
		{"исчерпаешь", "исчерпа"},
		{"исчерпаны", "исчерпа"},
		{"исчерченных", "исчерчен"},
		{"исчисление", "исчислен"},
		{"исчислил", "исчисл"},
		{"исчислить", "исчисл"},
		{"исшаркавшегося", "исшарка"},
		{"ит", "ит"},
		{"итак", "итак"},
		{"итал", "ита"},
		{"италии", "итал"},
		{"италию", "итал"},
		{"итальянски", "итальянск"},
		{"итальянские", "итальянск"},
		{"итальянский", "итальянск"},
		{"итог", "итог"},
		{"итога", "итог"},
		{"итоги", "итог"},
		{"итого", "ит"},
		{"иуда", "иуд"},
		{"иудеев", "иуде"},
		{"иудеи", "иуд"},
		{"их", "их"},
		{"ихнего", "ихн"},
		{"ихнее", "ихн"},
		{"ихней", "ихн"},
		{"ихние", "ихн"},
		{"ихнюю", "ихн"},
		{"ихняя", "ихн"},
		{"ишь", "иш"},
		{"ища", "ищ"},
		{"ищем", "ищ"},
		{"ищет", "ищет"},
		{"ищете", "ищет"},
		{"ищешь", "ищеш"},
		{"ищи", "ищ"},
		{"ищите", "ищ"},
		{"ищу", "ищ"},
		{"ищут", "ищут"},
		{"ищущего", "ищущ"},
		{"ищущие", "ищущ"},
		{"июль", "июл"},
		{"июля", "июл"},
		{"июне", "июн"},
		{"июнь", "июн"},
		{"июня", "июн"},
		{"й", "й"},
		{"йодоформ", "йодоформ"},
		{"к", "к"},
		{"ка", "ка"},
		{"кабак", "кабак"},
		{"кабака", "кабак"},
		{"кабаке", "кабак"},
		{"кабаки", "кабак"},
		{"кабалу", "кабал"},
		{"кабацкой", "кабацк"},
		{"кабачного", "кабачн"},
		{"кабачным", "кабачн"},
		{"кабинет", "кабинет"},
		{"кабинета", "кабинет"},
		{"кабинете", "кабинет"},
		{"кабинетов", "кабинет"},
		{"кабинетом", "кабинет"},
		{"кабинету", "кабинет"},
		{"каблук", "каблук"},
		{"каблуками", "каблук"},
		{"каблуки", "каблук"},
		{"каблучки", "каблучк"},
		{"кабули", "кабул"},
		{"кабы", "каб"},
		{"кавалер", "кавалер"},
		{"кавалера", "кавалер"},
		{"кавалерии", "кавалер"},
		{"кавалеров", "кавалер"},
		{"кавалеру", "кавалер"},
		{"кавалеры", "кавалер"},
		{"кавалькада", "кавалькад"},
		{"каватину", "каватин"},
		{"каватины", "каватин"},
		{"кавказе", "кавказ"},
		{"кавказское", "кавказск"},
		{"кавычках", "кавычк"},
		{"кавычки", "кавычк"},
		{"кадил", "кад"},
		{"кадили", "кад"},
		{"кадило", "кад"},
		{"кадилом", "кадил"},
		{"кадильный", "кадильн"},
		{"кадки", "кадк"},
		{"кадочка", "кадочк"},
		{"кадочку", "кадочк"},
		{"каждая", "кажд"},
		{"каждого", "кажд"},
		{"каждое", "кажд"},
		{"каждой", "кажд"},
		{"каждом", "кажд"},
		{"каждому", "кажд"},
		{"каждою", "кажд"},
		{"кажду", "кажд"},
		{"каждую", "кажд"},
		{"каждый", "кажд"},
		{"каждым", "кажд"},
		{"каждыми", "кажд"},
		{"кажет", "кажет"},
		{"кажетесь", "кажет"},
		{"кажется", "кажет"},
		{"кажешься", "кажеш"},
		{"кажись", "каж"},
		{"кажусь", "каж"},
		{"кажутся", "кажут"},
		{"кажущуюся", "кажущ"},
		{"казавшеюся", "каза"},
		{"казавшийся", "каза"},
		{"казак", "казак"},
		{"казал", "каза"},
		{"казалась", "каза"},
		{"казались", "каза"},
		{"казалось", "каза"},
		{"казался", "каза"},
		{"казанскою", "казанск"},
		{"казарм", "казарм"},
		{"казармах", "казарм"},
		{"казармой", "казарм"},
		{"казармы", "казарм"},
		{"казать", "каза"},
		{"казаться", "каза"},
		{"казачка", "казачк"},
		{"казачку", "казачк"},
		{"казачок", "казачок"},
		{"казенная", "казен"},
		{"казенного", "казен"},
		{"казенное", "казен"},
		{"казенной", "казен"},
		{"казенную", "казен"},
		{"казенные", "казен"},
		{"казенщину", "казенщин"},
		{"казимир", "казимир"},
		{"казне", "казн"},
		{"казней", "казн"},
		{"казненных", "казнен"},
		{"казнены", "казн"},
		{"казни", "казн"},
		{"казнил", "казн"},
		{"казнила", "казн"},
		{"казнит", "казн"},
		{"казните", "казн"},
		{"казнить", "казн"},
		{"казну", "казн"},
		{"казны", "казн"},
		{"казнь", "казн"},
		{"казнью", "казн"},
		{"казнят", "казн"},
		{"казуистика", "казуистик"},
		{"казуистики", "казуистик"},
		{"казуистику", "казуистик"},
		{"казус", "казус"},
		{"кайма", "кайм"},
		{"как", "как"},
		{"какая", "как"},
		{"какие", "как"},
		{"каким", "как"},
		{"какими", "как"},
		{"каких", "как"},
		{"како", "как"},
		{"каков", "как"},
		{"какова", "каков"},
		{"каково", "каков"},
		{"каковы", "каков"},
		{"каковые", "каков"},
		{"какого", "как"},
		{"какое", "как"},
		{"какой", "как"},
		{"каком", "как"},
		{"какому", "как"},
		{"какою", "как"},
		{"кактусами", "кактус"},
		{"кактусов", "кактус"},
		{"каку", "как"},
		{"какую", "как"},
		{"кал", "кал"},
		{"каламбуром", "каламбур"},
		{"каланча", "каланч"},
		{"каланчой", "каланч"},
		{"калачей", "калач"},
		{"калачом", "калач"},
		{"калашный", "калашн"},
		{"калеб", "калеб"},
		{"калебов", "калеб"},
		{"калек", "калек"},
		{"калека", "калек"},
		{"календарем", "календар"},
		{"календарь", "календар"},
		{"календаря", "календар"},
		{"калеными", "кален"},
		{"калигула", "калигул"},
		{"калинникова", "калинников"},
		{"калитка", "калитк"},
		{"калитке", "калитк"},
		{"калитки", "калитк"},
		{"калитку", "калитк"},
		{"калиту", "калит"},
		{"калош", "калош"},
		{"калошах", "калош"},
		{"калоши", "калош"},
		{"камелии", "камел"},
		{"камелий", "камел"},
		{"камелиями", "камел"},
		{"каменистом", "каменист"},
		{"каменистый", "каменист"},
		{"каменная", "камен"},
		{"каменного", "камен"},
		{"каменное", "камен"},
		{"каменной", "камен"},
		{"каменным", "камен"},
		{"каменных", "камен"},
		{"каменщик", "каменщик"},
		{"камень", "камен"},
		{"каменья", "камен"},
		{"камер", "камер"},
		{"камердинер", "камердинер"},
		{"камердинером", "камердинер"},
		{"камешек", "камешек"},
		{"камешкам", "камешк"},
		{"камешки", "камешк"},
		{"камином", "камин"},
		{"камне", "камн"},
		{"камней", "камн"},
		{"камнем", "камн"},
		{"камни", "камн"},
		{"камню", "камн"},
		{"камня", "камн"},
		{"камням", "камн"},
		{"камнях", "камн"},
		{"каморка", "каморк"},
		{"каморке", "каморк"},
		{"каморки", "каморк"},
		{"каморку", "каморк"},
		{"кампанил", "кампан"},
		{"камушками", "камушк"},
		{"камфарой", "камфар"},
		{"камыш", "камыш"},
		{"камыша", "камыш"},
		{"камыши", "камыш"},
		{"канав", "кана"},
		{"канавами", "канав"},
		{"канаве", "канав"},
		{"канавкой", "канавк"},
		{"канаву", "канав"},
		{"канавы", "канав"},
		{"канал", "кана"},
		{"канала", "кана"},
		{"каналы", "канал"},
		{"каналью", "канал"},
		{"каналья", "канал"},
		{"канареек", "канареек"},
		{"канарейка", "канарейк"},
		{"канарейками", "канарейк"},
		{"канарейках", "канарейк"},
		{"канарейки", "канарейк"},
		{"канат", "канат"},
		{"канве", "канв"},
		{"канву", "канв"},
		{"канвы", "канв"},
		{"кандалов", "кандал"},
		{"канделябры", "канделябр"},
		{"канет", "канет"},
		{"канешь", "канеш"},
		{"каникулы", "каникул"},
		{"канкан", "канка"},
		{"канканную", "канка"},
		{"канул", "канул"},
		{"канула", "канул"},
		{"канцелярии", "канцеляр"},
		{"канцелярист", "канцелярист"},
		{"канцеляристов", "канцелярист"},
		{"канцеляристы", "канцелярист"},
		{"канцелярию", "канцеляр"},
		{"канцелярские", "канцелярск"},
		{"канцелярскими", "канцелярск"},
		{"канцоной", "канцон"},
		{"капали", "капа"},
		{"капель", "капел"},
		{"капельки", "капельк"},
		{"капельку", "капельк"},
		{"капернаумов", "капернаум"},
		{"капернаумова", "капернаумов"},
		{"капернаумову", "капернаумов"},
		{"капернаумовых", "капернаумов"},
		{"капитал", "капита"},
		{"капитала", "капита"},
		{"капиталисты", "капиталист"},
		{"капиталов", "капитал"},
		{"капиталом", "капитал"},
		{"капиталу", "капитал"},
		{"капиталы", "капитал"},
		{"капитальная", "капитальн"},
		{"капитального", "капитальн"},
		{"капитальное", "капитальн"},
		{"капитальном", "капитальн"},
		{"капитальному", "капитальн"},
		{"капитальный", "капитальн"},
		{"капитан", "капита"},
		{"капитолии", "капитол"},
		{"капитэн", "капитэн"},
		{"капкан", "капка"},
		{"капле", "капл"},
		{"каплей", "капл"},
		{"каплет", "каплет"},
		{"капли", "капл"},
		{"каплю", "капл"},
		{"каплют", "каплют"},
		{"капля", "капл"},
		{"каплям", "капл"},
		{"каплями", "капл"},
		{"капоте", "капот"},
		{"каприз", "каприз"},
		{"каприза", "каприз"},
		{"капризам", "каприз"},
		{"капризах", "каприз"},
		{"капризен", "каприз"},
		{"капризная", "капризн"},
		{"капризница", "капризниц"},
		{"капризничай", "капризнича"},
		{"капризничал", "капризнича"},
		{"капризничать", "капризнича"},
		{"капризно", "капризн"},
		{"капризный", "капризн"},
		{"капризным", "капризн"},
		{"капризов", "каприз"},
		{"капризу", "каприз"},
		{"капризы", "каприз"},
		{"капсюль", "капсюл"},
		{"капуста", "капуст"},
		{"капустой", "капуст"},
		{"капусту", "капуст"},
		{"капусты", "капуст"},
		{"кара", "кар"},
		{"карабкался", "карабка"},
		{"карабкаясь", "карабк"},
		{"караван", "карава"},
		{"карамзин", "карамзин"},
		{"карандаш", "карандаш"},
		{"карандашей", "карандаш"},
		{"карандаши", "карандаш"},
		{"карандашиком", "карандашик"},
		{"карандашом", "карандаш"},
		{"карать", "кара"},
		{"караул", "караул"},
		{"караулила", "караул"},
		{"караулить", "караул"},
		{"карашо", "караш"},
		{"карающего", "кара"},
		{"кардамонову", "кардамонов"},
		{"карет", "карет"},
		{"карета", "карет"},
		{"каретах", "карет"},
		{"карете", "карет"},
		{"каретного", "каретн"},
		{"каретное", "каретн"},
		{"каретой", "карет"},
		{"карету", "карет"},
		{"кареты", "карет"},
		{"карий", "кар"},
		{"карих", "кар"},
		{"каркают", "карка"},
		{"карлик", "карлик"},
		{"карловны", "карловн"},
		{"карлосу", "карлос"},
		{"карль", "карл"},
		{"карман", "карма"},
		{"кармана", "карма"},
		{"карманам", "карман"},
		{"карманах", "карман"},
		{"кармане", "карман"},
		{"карманном", "карма"},
		{"карманные", "карма"},
		{"карманный", "карма"},
		{"карману", "карман"},
		{"карманы", "карма"},
		{"кармен", "карм"},
		{"карменситы", "карменсит"},
		{"карнизов", "карниз"},
		{"карп", "карп"},
		{"карпатах", "карпат"},
		{"карпович", "карпович"},
		{"карт", "карт"},
		{"карта", "карт"},
		{"картавя", "картав"},
		{"картам", "карт"},
		{"картами", "карт"},
		{"картах", "карт"},
		{"карте", "карт"},
		{"картежник", "картежник"},
		{"картин", "картин"},
		{"картина", "картин"},
		{"картинами", "картин"},
		{"картине", "картин"},
		{"картинка", "картинк"},
		{"картинки", "картинк"},
		{"картинно", "картин"},
		{"картиной", "картин"},
		{"картинок", "картинок"},
		{"картину", "картин"},
		{"картины", "картин"},
		{"картишки", "картишк"},
		{"картограмме", "картограмм"},
		{"картограммой", "картограмм"},
		{"картограмму", "картограмм"},
		{"картограммы", "картограмм"},
		{"картой", "карт"},
		{"картонки", "картонк"},
		{"картонку", "картонк"},
		{"картофелем", "картофел"},
		{"картофель", "картофел"},
		{"картофелю", "картофел"},
		{"картофеля", "картофел"},
		{"карточек", "карточек"},
		{"карточка", "карточк"},
		{"карточку", "карточк"},
		{"карту", "карт"},
		{"картуз", "картуз"},
		{"картуза", "картуз"},
		{"карты", "карт"},
		{"кары", "кар"},
		{"карьера", "карьер"},
		{"карьере", "карьер"},
		{"карьерой", "карьер"},
		{"карьеру", "карьер"},
		{"карьеры", "карьер"},
		{"касавшемуся", "каса"},
		{"касавшийся", "каса"},
		{"касается", "каса"},
		{"касалась", "каса"},
		{"касалось", "каса"},
		{"касался", "каса"},
		{"касаются", "каса"},
		{"касающейся", "каса"},
		{"касающемся", "каса"},
		{"касающихся", "каса"},
		{"касаясь", "кас"},
		{"каске", "каск"},
		{"каскетку", "каскетк"},
		{"каски", "каск"},
		{"кастаньет", "кастаньет"},
		{"кастелянши", "кастелянш"},
		{"кастрюль", "кастрюл"},
		{"кастрюлю", "кастрюл"},
		{"кастрюлями", "кастрюл"},
		{"катавшийся", "ката"},
		{"катается", "ката"},
		{"катай", "ката"},
		{"катастроф", "катастроф"},
		{"катастрофа", "катастроф"},
		{"катастрофе", "катастроф"},
		{"катастрофой", "катастроф"},
		{"катастрофы", "катастроф"},
		{"кататься", "ката"},
		{"кате", "кат"},
		{"катей", "кат"},
		{"катерина", "катерин"},
		{"катерине", "катерин"},
		{"катериной", "катерин"},
		{"катериною", "катерин"},
		{"катерину", "катерин"},
		{"катерины", "катерин"},
		{"катехизис", "катехизис"},
		{"кати", "кат"},
		{"катила", "кат"},
		{"катился", "кат"},
		{"катит", "кат"},
		{"каторгами", "каторг"},
		{"каторге", "каторг"},
		{"каторги", "каторг"},
		{"каторгу", "каторг"},
		{"каторжник", "каторжник"},
		{"каторжного", "каторжн"},
		{"каторжной", "каторжн"},
		{"каторжные", "каторжн"},
		{"каторжный", "каторжн"},
		{"каторжных", "каторжн"},
		{"катышек", "катышек"},
		{"катю", "кат"},
		{"катя", "кат"},
		{"катят", "кат"},
		{"катятся", "кат"},
		{"каучука", "каучук"},
		{"кафедру", "кафедр"},
		{"кафедры", "кафедр"},
		{"кафешантана", "кафешанта"},
		{"кафтан", "кафта"},
		{"кафтана", "кафта"},
		{"кафтане", "кафтан"},
		{"кафтанов", "кафтан"},
		{"кафэ", "кафэ"},
		{"кацавеек", "кацавеек"},
		{"кацавейка", "кацавейк"},
		{"качавший", "кача"},
		{"качает", "кача"},
		{"качается", "кача"},
		{"качал", "кача"},
		{"качала", "кача"},
		{"качали", "кача"},
		{"качалка", "качалк"},
		{"качался", "кача"},
		{"качанье", "качан"},
		{"качаться", "кача"},
		{"качающихся", "кача"},
		{"качая", "кач"},
		{"качели", "качел"},
		{"качелях", "качел"},
		{"качеств", "качеств"},
		{"качества", "качеств"},
		{"качествами", "качеств"},
		{"качествах", "качеств"},
		{"качестве", "качеств"},
		{"качнув", "качнув"},
		{"качнулся", "качнул"},
		{"качнусь", "качн"},
		{"каша", "каш"},
		{"кашей", "каш"},
		{"кашель", "кашел"},
		{"каши", "каш"},
		{"кашин", "кашин"},
		{"кашлем", "кашл"},
		{"кашля", "кашл"},
		{"кашляет", "кашля"},
		{"кашлял", "кашля"},
		{"кашляла", "кашля"},
		{"кашлянув", "кашлянув"},
		{"кашлянул", "кашлянул"},
		{"кашлять", "кашля"},
		{"кашляю", "кашля"},
		{"кашляя", "кашл"},
		{"каштановые", "каштанов"},
		{"каюсь", "ка"},
		{"каюта", "кают"},
		{"каюте", "кают"},
		{"каюту", "кают"},
		{"каяться", "кая"},
		{"квадратный", "квадратн"},
		{"квадраты", "квадрат"},
		{"квартал", "кварта"},
		{"квартала", "кварта"},
		{"квартале", "квартал"},
		{"квартального", "квартальн"},
		{"квартальной", "квартальн"},
		{"квартальному", "квартальн"},
		{"квартальные", "квартальн"},
		{"квартальный", "квартальн"},
		{"квартальными", "квартальн"},
		{"квартальных", "квартальн"},
		{"кварташками", "кварташк"},
		{"квартир", "квартир"},
		{"квартира", "квартир"},
		{"квартирам", "квартир"},
		{"квартирами", "квартир"},
		{"квартирах", "квартир"},
		{"квартире", "квартир"},
		{"квартиркам", "квартирк"},
		{"квартирку", "квартирк"},
		{"квартирная", "квартирн"},
		{"квартирных", "квартирн"},
		{"квартировать", "квартирова"},
		{"квартирой", "квартир"},
		{"квартиру", "квартир"},
		{"квартиры", "квартир"},
		{"квас", "квас"},
		{"квасом", "квас"},
		{"квасу", "квас"},
		{"квасцов", "квасц"},
		{"квасы", "квас"},
		{"квентин", "квентин"},
		{"кверху", "кверх"},
		{"келье", "кел"},
		{"келья", "кел"},
		{"кем", "кем"},
		{"кеплеровы", "кеплеров"},
		{"керченские", "керченск"},
		{"кивает", "кива"},
		{"кивал", "кива"},
		{"кивать", "кива"},
		{"кивая", "кив"},
		{"кивера", "кивер"},
		{"кивнув", "кивнув"},
		{"кивнул", "кивнул"},
		{"кивнула", "кивнул"},
		{"кидает", "кида"},
		{"кидается", "кида"},
		{"кидал", "кида"},
		{"кидала", "кида"},
		{"кидать", "кида"},
		{"кидаются", "кида"},
		{"кидаясь", "кид"},
		{"киев", "ки"},
		{"киева", "киев"},
		{"киеве", "киев"},
		{"киевский", "киевск"},
		{"кий", "ки"},
		{"кильватерной", "кильватерн"},
		{"кинематографе", "кинематограф"},
		{"кинжал", "кинжа"},
		{"кинжалы", "кинжал"},
		{"кину", "кин"},
		{"кинув", "кинув"},
		{"кинул", "кинул"},
		{"кинулась", "кинул"},
		{"кинулись", "кинул"},
		{"кинулся", "кинул"},
		{"кинуто", "кинут"},
		{"кинутого", "кинут"},
		{"кинутые", "кинут"},
		{"кинуть", "кинут"},
		{"кинуться", "кинут"},
		{"киот", "киот"},
		{"киотом", "киот"},
		{"киоты", "киот"},
		{"кипарисный", "кипарисн"},
		{"кипарисы", "кипарис"},
		{"кипел", "кипел"},
		{"кипела", "кипел"},
		{"кипело", "кипел"},
		{"кипение", "кипен"},
		{"кипеть", "кипет"},
		{"кипит", "кип"},
		{"кипсэком", "кипсэк"},
		{"кипучая", "кипуч"},
		{"кипучими", "кипуч"},
		{"кипят", "кип"},
		{"кипятком", "кипятк"},
		{"кипятят", "кипят"},
		{"кирбитьевна", "кирбитьевн"},
		{"кирбитьевне", "кирбитьевн"},
		{"кире", "кир"},
		{"кирка", "кирк"},
		{"киркой", "кирк"},
		{"киркою", "кирк"},
		{"кирпич", "кирпич"},
		{"кирпича", "кирпич"},
		{"кирпичей", "кирпич"},
		{"кирпичик", "кирпичик"},
		{"кирпичиков", "кирпичик"},
		{"кирпичный", "кирпичн"},
		{"кирпичных", "кирпичн"},
		{"кирсановский", "кирсановск"},
		{"кисеей", "кисе"},
		{"кисейная", "кисейн"},
		{"кисейными", "кисейн"},
		{"киселе", "кисел"},
		{"кисель", "кисел"},
		{"киселя", "кисел"},
		{"кислая", "кисл"},
		{"кисло", "кисл"},
		{"кислое", "кисл"},
		{"киснет", "киснет"},
		{"киссинген", "киссинг"},
		{"кистей", "кист"},
		{"кисти", "кист"},
		{"кисть", "кист"},
		{"кистью", "кист"},
		{"китайских", "китайск"},
		{"китайского", "китайск"},
		{"китайцами", "китайц"},
		{"китах", "кит"},
		{"кихотов", "кихот"},
		{"кичке", "кичк"},
		{"кичливости", "кичлив"},
		{"кишат", "кишат"},
		{"кишела", "кишел"},
		{"кишели", "кишел"},
		{"кишки", "кишк"},
		{"клавикорды", "клавикорд"},
		{"клавиш", "клавиш"},
		{"клавиши", "клавиш"},
		{"клад", "клад"},
		{"кладби", "кладб"},
		{"кладбища", "кладбищ"},
		{"кладбище", "кладбищ"},
		{"кладбищенской", "кладбищенск"},
		{"кладбищу", "кладбищ"},
		{"кладет", "кладет"},
		{"кладете", "кладет"},
		{"кладите", "клад"},
		{"кладку", "кладк"},
		{"кладовой", "кладов"},
		{"кладовую", "кладов"},
		{"кладом", "клад"},
		{"кладу", "клад"},
		{"кладут", "кладут"},
		{"кладя", "клад"},
		{"клал", "клал"},
		{"клала", "клал"},
		{"кланявшиеся", "кланя"},
		{"кланяется", "кланя"},
		{"кланялась", "кланя"},
		{"кланялись", "кланя"},
		{"кланялся", "кланя"},
		{"кланяться", "кланя"},
		{"кланяются", "кланя"},
		{"кланяясь", "клан"},
		{"кларнет", "кларнет"},
		{"класса", "класс"},
		{"классах", "класс"},
		{"классе", "класс"},
		{"класть", "класт"},
		{"клеверными", "клеверн"},
		{"клеверных", "клеверн"},
		{"клевета", "клевет"},
		{"клевете", "клевет"},
		{"клеветник", "клеветник"},
		{"клеветой", "клевет"},
		{"клевету", "клевет"},
		{"клеветы", "клевет"},
		{"клевещете", "клевещет"},
		{"клевещут", "клевещут"},
		{"клеенкой", "клеенк"},
		{"клеенчатый", "клеенчат"},
		{"клейменые", "клеймен"},
		{"клеймила", "клейм"},
		{"клеймили", "клейм"},
		{"клеймы", "клейм"},
		{"клеймят", "клейм"},
		{"клекот", "клекот"},
		{"клен", "клен"},
		{"кленовые", "кленов"},
		{"клены", "клен"},
		{"клетка", "клетк"},
		{"клетке", "клетк"},
		{"клетки", "клетк"},
		{"клетку", "клетк"},
		{"клетушка", "клетушк"},
		{"клетушке", "клетушк"},
		{"клетчатых", "клетчат"},
		{"клешни", "клешн"},
		{"клещах", "клещ"},
		{"кликал", "клика"},
		{"кликами", "клик"},
		{"кликать", "клика"},
		{"клике", "клик"},
		{"клики", "клик"},
		{"кликни", "кликн"},
		{"кликну", "кликн"},
		{"кликнул", "кликнул"},
		{"кликнуть", "кликнут"},
		{"климат", "климат"},
		{"климата", "климат"},
		{"климате", "климат"},
		{"климатические", "климатическ"},
		{"клинком", "клинк"},
		{"клинок", "клинок"},
		{"клич", "клич"},
		{"кличет", "кличет"},
		{"кличешь", "кличеш"},
		{"кличь", "клич"},
		{"клоак", "клоак"},
		{"клоакам", "клоак"},
		{"клоаки", "клоак"},
		{"клобука", "клобук"},
		{"клок", "клок"},
		{"клокотало", "клокота"},
		{"клокочут", "клокочут"},
		{"клонил", "клон"},
		{"клонилась", "клон"},
		{"клонились", "клон"},
		{"клонило", "клон"},
		{"клонится", "клон"},
		{"клоня", "клон"},
		{"клонясь", "клон"},
		{"клонятся", "клон"},
		{"клопа", "клоп"},
		{"клопов", "клоп"},
		{"клопом", "клоп"},
		{"клопшток", "клопшток"},
		{"клопы", "клоп"},
		{"клочка", "клочк"},
		{"клочками", "клочк"},
		{"клочке", "клочк"},
		{"клочки", "клочк"},
		{"клочком", "клочк"},
		{"клочок", "клочок"},
		{"клуб", "клуб"},
		{"клубами", "клуб"},
		{"клубе", "клуб"},
		{"клубный", "клубн"},
		{"клубов", "клуб"},
		{"клубок", "клубок"},
		{"клубы", "клуб"},
		{"клумбами", "клумб"},
		{"клюках", "клюк"},
		{"клюквы", "клюкв"},
		{"клюкою", "клюк"},
		{"ключ", "ключ"},
		{"ключа", "ключ"},
		{"ключами", "ключ"},
		{"ключей", "ключ"},
		{"ключи", "ключ"},
		{"ключик", "ключик"},
		{"ключниц", "ключниц"},
		{"ключница", "ключниц"},
		{"ключом", "ключ"},
		{"клюют", "клюют"},
		{"кля", "кля"},
		{"кляла", "клял"},
		{"клялись", "клял"},
		{"клялся", "клял"},
		{"клянет", "клянет"},
		{"клянется", "клянет"},
		{"клянешь", "клянеш"},
		{"кляни", "клян"},
		{"клянусь", "клян"},
		{"кляня", "клян"},
		{"клясть", "кляст"},
		{"клясться", "кляст"},
		{"клятвам", "клятв"},
		{"клятвы", "клятв"},
		{"кляча", "кляч"},
		{"клячи", "кляч"},
		{"клячонка", "клячонк"},
		{"клячу", "кляч"},
		{"кн", "кн"},
		{"книг", "книг"},
		{"книга", "книг"},
		{"книгам", "книг"},
		{"книгами", "книг"},
		{"книгах", "книг"},
		{"книге", "книг"},
		{"книги", "книг"},
		{"книгой", "книг"},
		{"книгопродавец", "книгопродавец"},
		{"книгопродавцы", "книгопродавц"},
		{"книгу", "книг"},
		{"книжек", "книжек"},
		{"книжечку", "книжечк"},
		{"книжка", "книжк"},
		{"книжками", "книжк"},
		{"книжках", "книжк"},
		{"книжке", "книжк"},
		{"книжки", "книжк"},
		{"книжкой", "книжк"},
		{"книжку", "книжк"},
		{"книжной", "книжн"},
		{"книжные", "книжн"},
		{"книжный", "книжн"},
		{"книжонки", "книжонк"},
		{"кнопа", "кноп"},
		{"кнопками", "кнопк"},
		{"кнопку", "кнопк"},
		{"кнут", "кнут"},
		{"кнута", "кнут"},
		{"кнутами", "кнут"},
		{"кнутов", "кнут"},
		{"кнутом", "кнут"},
		{"кнуту", "кнут"},
		{"кнуты", "кнут"},
		{"княгини", "княгин"},
		{"княгиня", "княгин"},
		{"княжеская", "княжеск"},
		{"княжеский", "княжеск"},
		{"княжеском", "княжеск"},
		{"княжна", "княжн"},
		{"княжны", "княжн"},
		{"княжьего", "княж"},
		{"князей", "княз"},
		{"князь", "княз"},
		{"князька", "князьк"},
		{"князья", "княз"},
		{"князю", "княз"},
		{"князя", "княз"},
		{"ко", "ко"},
		{"кобелева", "кобелев"},
		{"кобыле", "кобыл"},
		{"кобыленка", "кобыленк"},
		{"кобыленке", "кобыленк"},
		{"кобыленку", "кобыленк"},
		{"кобылиц", "кобылиц"},
		{"кобылица", "кобылиц"},
		{"кобылятниковым", "кобылятников"},
		{"ков", "ков"},
		{"коварнее", "коварн"},
		{"коварно", "коварн"},
		{"коварную", "коварн"},
		{"коварные", "коварн"},
		{"коварства", "коварств"},
		{"коварство", "коварств"},
		{"коварству", "коварств"},
		{"кове", "ков"},
		{"ковер", "ковер"},
		{"коверкала", "коверка"},
		{"коврам", "ковр"},
		{"коврами", "ковр"},
		{"коврах", "ковр"},
		{"ковре", "ковр"},
		{"коврик", "коврик"},
		{"ковров", "ковр"},
		{"ковром", "ковр"},
		{"ковру", "ковр"},
		{"ковры", "ковр"},
		{"ковчег", "ковчег"},
		{"ковчега", "ковчег"},
		{"ковчеге", "ковчег"},
		{"ковша", "ковш"},
		{"ковы", "ков"},
		{"ковылем", "ковыл"},
		{"ковыли", "ков"},
		{"ковыль", "ковыл"},
		{"ковырять", "ковыря"},
		{"когда", "когд"},
		{"кого", "ког"},
		{"когса", "когс"},
		{"когти", "когт"},
		{"когтит", "когт"},
		{"когтями", "когт"},
		{"когтях", "когт"},
		{"кодексе", "кодекс"},
		{"кое", "ко"},
		{"коего", "ко"},
		{"коему", "ко"},
		{"кожаное", "кожан"},
		{"кожаном", "кожан"},
		{"кожаным", "кожан"},
		{"коже", "кож"},
		{"кожи", "кож"},
		{"коз", "коз"},
		{"коза", "коз"},
		{"козе", "коз"},
		{"козел", "козел"},
		{"козеля", "козел"},
		{"козла", "козл"},
		{"козловых", "козлов"},
		{"козлу", "козл"},
		{"козни", "козн"},
		{"козоедова", "козоедов"},
		{"козы", "коз"},
		{"козырев", "козыр"},
		{"козыри", "козыр"},
		{"козырьком", "козырьк"},
		{"козьего", "коз"},
		{"коим", "ко"},
		{"коих", "ко"},
		{"кой", "ко"},
		{"кокардой", "кокард"},
		{"кокетку", "кокетк"},
		{"кокетливо", "кокетлив"},
		{"кокетничает", "кокетнича"},
		{"кокетничаете", "кокетнича"},
		{"кокетничала", "кокетнича"},
		{"кокетничать", "кокетнича"},
		{"кокетничаю", "кокетнича"},
		{"кокетства", "кокетств"},
		{"кокетстве", "кокетств"},
		{"кокетство", "кокетств"},
		{"кокетству", "кокетств"},
		{"кол", "кол"},
		{"кола", "кол"},
		{"колбасница", "колбасниц"},
		{"колбасной", "колбасн"},
		{"колбасы", "колбас"},
		{"колдовское", "колдовск"},
		{"колдовства", "колдовств"},
		{"колдун", "колдун"},
		{"колдуна", "колдун"},
		{"колдунья", "колдун"},
		{"колдуют", "колд"},
		{"коле", "кол"},
		{"колебалась", "колеба"},
		{"колебании", "колебан"},
		{"колебания", "колебан"},
		{"колеблется", "колеблет"},
		{"колеблющейся", "колеблющ"},
		{"колеблющемся", "колеблющ"},
		{"колебля", "колебл"},
		{"колеблясь", "колебл"},
		{"колее", "кол"},
		{"колеи", "кол"},
		{"колен", "кол"},
		{"колена", "кол"},
		{"коленами", "колен"},
		{"коленах", "колен"},
		{"колене", "колен"},
		{"коленей", "колен"},
		{"колени", "колен"},
		{"коленках", "коленк"},
		{"коленке", "коленк"},
		{"коленки", "коленк"},
		{"коленкоровые", "коленкоров"},
		{"коленку", "коленк"},
		{"колено", "кол"},
		{"коленопреклоненным", "коленопреклонен"},
		{"коленочках", "коленочк"},
		{"коленяк", "коленяк"},
		{"коленям", "колен"},
		{"коленями", "колен"},
		{"коленях", "колен"},
		{"колес", "колес"},
		{"колеса", "колес"},
		{"колесами", "колес"},
		{"колесах", "колес"},
		{"колеснице", "колесниц"},
		{"колесо", "колес"},
		{"колец", "колец"},
		{"колечище", "колечищ"},
		{"колечках", "колечк"},
		{"колечко", "колечк"},
		{"колечком", "колечк"},
		{"колею", "кол"},
		{"колеями", "коле"},
		{"коли", "кол"},
		{"количестве", "количеств"},
		{"количество", "количеств"},
		{"количеством", "количеств"},
		{"колко", "колк"},
		{"коллежский", "коллежск"},
		{"коллежским", "коллежск"},
		{"коллежского", "коллежск"},
		{"коллежской", "коллежск"},
		{"коллежскую", "коллежск"},
		{"коллекцию", "коллекц"},
		{"коллекция", "коллекц"},
		{"колода", "колод"},
		{"колодезь", "колодез"},
		{"колодец", "колодец"},
		{"колодник", "колодник"},
		{"колодой", "колод"},
		{"колоду", "колод"},
		{"колок", "колок"},
		{"колокол", "колокол"},
		{"колокола", "колокол"},
		{"колоколов", "колокол"},
		{"колокольни", "колокольн"},
		{"колокольный", "колокольн"},
		{"колокольным", "колокольн"},
		{"колокольня", "колокольн"},
		{"колокольцы", "колокольц"},
		{"колокольчик", "колокольчик"},
		{"колокольчика", "колокольчик"},
		{"колокольчики", "колокольчик"},
		{"колокольчиков", "колокольчик"},
		{"колокольчику", "колокольчик"},
		{"колола", "колол"},
		{"кололись", "колол"},
		{"коломенских", "коломенск"},
		{"колонии", "колон"},
		{"колоннами", "колон"},
		{"колонной", "колон"},
		{"колонны", "колон"},
		{"колорит", "колор"},
		{"колорита", "колорит"},
		{"колоритом", "колорит"},
		{"колос", "колос"},
		{"колосья", "колос"},
		{"колотилины", "колотилин"},
		{"колотит", "колот"},
		{"колотится", "колот"},
		{"колотить", "колот"},
		{"колотиться", "колот"},
		{"колотушку", "колотушк"},
		{"колоть", "колот"},
		{"колоться", "колот"},
		{"колотятся", "колот"},
		{"колпак", "колпак"},
		{"колпаке", "колпак"},
		{"колпаки", "колпак"},
		{"колпаком", "колпак"},
		{"колпине", "колпин"},
		{"колумба", "колумб"},
		{"колхидой", "колхид"},
		{"колчан", "колча"},
		{"колыбель", "колыбел"},
		{"колыбелью", "колыбел"},
		{"колымаге", "колымаг"},
		{"колымягин", "колымягин"},
		{"колыхалась", "колыха"},
		{"колыхаться", "колыха"},
		{"колыхаясь", "колых"},
		{"коль", "кол"},
		{"колька", "кольк"},
		{"кольца", "кольц"},
		{"кольце", "кольц"},
		{"кольцо", "кольц"},
		{"кольцом", "кольц"},
		{"кольцы", "кольц"},
		{"кольчугу", "кольчуг"},
		{"колю", "кол"},
		{"колючие", "колюч"},
		{"колючий", "колюч"},
		{"колючих", "колюч"},
		{"коля", "кол"},
		{"коляска", "коляск"},
		{"коляске", "коляск"},
		{"коляски", "коляск"},
		{"коляской", "коляск"},
		{"коляску", "коляск"},
		{"колясочке", "колясочк"},
		{"колясочкой", "колясочк"},
		{"колясочку", "колясочк"},
		{"ком", "ком"},
		{"команда", "команд"},
		{"команде", "команд"},
		{"командир", "командир"},
		{"командиров", "командир"},
		{"командировку", "командировк"},
		{"командиром", "командир"},
		{"командовал", "командова"},
		{"командор", "командор"},
		{"командора", "командор"},
		{"комары", "комар"},
		{"комбинаций", "комбинац"},
		{"комедии", "комед"},
		{"комедию", "комед"},
		{"комедия", "комед"},
		{"комет", "комет"},
		{"комета", "комет"},
		{"кометы", "комет"},
		{"комизм", "комизм"},
		{"комик", "комик"},
		{"комиссии", "комисс"},
		{"комиссию", "комисс"},
		{"комиссия", "комисс"},
		{"комическая", "комическ"},
		{"комические", "комическ"},
		{"комической", "комическ"},
		{"комическую", "комическ"},
		{"комкая", "комк"},
		{"комком", "комк"},
		{"комку", "комк"},
		{"коммерческий", "коммерческ"},
		{"коммиссаржевской", "коммиссаржевск"},
		{"коммун", "коммун"},
		{"коммуна", "коммун"},
		{"коммуне", "коммун"},
		{"коммуну", "коммун"},
		{"коммуны", "коммун"},
		{"комнат", "комнат"},
		{"комната", "комнат"},
		{"комнатам", "комнат"},
		{"комнатах", "комнат"},
		{"комнате", "комнат"},
		{"комнатка", "комнатк"},
		{"комнатке", "комнатк"},
		{"комнатки", "комнатк"},
		{"комнатку", "комнатк"},
		{"комнатном", "комнатн"},
		{"комнату", "комнат"},
		{"комнаты", "комнат"},
		{"комод", "комод"},
		{"комода", "комод"},
		{"комоде", "комод"},
		{"комоду", "комод"},
		{"комок", "комок"},
		{"компанейские", "компанейск"},
		{"компанией", "компан"},
		{"компании", "компан"},
		{"компанию", "компан"},
		{"компания", "компан"},
		{"компаньоном", "компаньон"},
		{"компаньоны", "компаньон"},
		{"комплекции", "комплекц"},
		{"комплимент", "комплимент"},
		{"комплимента", "комплимент"},
		{"комплиментов", "комплимент"},
		{"комплименты", "комплимент"},
		{"компромиссов", "компромисс"},
		{"кому", "ком"},
		{"комфорт", "комфорт"},
		{"комфорта", "комфорт"},
		{"комфорте", "комфорт"},
		{"комфортно", "комфортн"},
		{"конверт", "конверт"},
		{"конвоем", "конво"},
		{"конвойного", "конвойн"},
		{"конвойный", "конвойн"},
		{"конвульсии", "конвульс"},
		{"конго", "конг"},
		{"конгрессов", "конгресс"},
		{"кондитера", "кондитер"},
		{"кондитерской", "кондитерск"},
		{"кондрашки", "кондрашк"},
		{"кондуктор", "кондуктор"},
		{"коне", "кон"},
		{"коней", "кон"},
		{"конец", "конец"},
		{"конечно", "конечн"},
		{"конечные", "конечн"},
		{"конник", "конник"},
		{"конница", "конниц"},
		{"консервативное", "консервативн"},
		{"консервативные", "консервативн"},
		{"консерватории", "консерватор"},
		{"консилиум", "консилиум"},
		{"конская", "конск"},
		{"конский", "конск"},
		{"конских", "конск"},
		{"конспиративным", "конспиративн"},
		{"констан", "конста"},
		{"константин", "константин"},
		{"константина", "константин"},
		{"константином", "константин"},
		{"константинополь", "константинопол"},
		{"константину", "константин"},
		{"конституцией", "конституц"},
		{"конституций", "конституц"},
		{"консультации", "консультац"},
		{"контор", "контор"},
		{"контора", "контор"},
		{"конторах", "контор"},
		{"конторе", "контор"},
		{"конторка", "конторк"},
		{"конторскую", "конторск"},
		{"контору", "контор"},
		{"конторщик", "конторщик"},
		{"конторщика", "конторщик"},
		{"конторы", "контор"},
		{"контрабас", "контрабас"},
		{"контракт", "контракт"},
		{"контракта", "контракт"},
		{"контракте", "контракт"},
		{"контрактом", "контракт"},
		{"контракту", "контракт"},
		{"контракты", "контракт"},
		{"контральтом", "контральт"},
		{"контрдансов", "контрданс"},
		{"контролем", "контрол"},
		{"контру", "контр"},
		{"конур", "конур"},
		{"конуре", "конур"},
		{"конуру", "конур"},
		{"конуры", "конур"},
		{"конус", "конус"},
		{"конфекты", "конфект"},
		{"конфетами", "конфет"},
		{"конфузился", "конфуз"},
		{"конфузится", "конфуз"},
		{"конфузишься", "конфуз"},
		{"конфузливо", "конфузлив"},
		{"конфузливы", "конфузлив"},
		{"конфузьтесь", "конфузьт"},
		{"конфузятся", "конфуз"},
		{"конца", "конц"},
		{"концам", "конц"},
		{"концах", "конц"},
		{"конце", "конц"},
		{"концентрировалась", "концентрирова"},
		{"концентрических", "концентрическ"},
		{"концепция", "концепц"},
		{"концерт", "концерт"},
		{"концертах", "концерт"},
		{"концерте", "концерт"},
		{"концов", "конц"},
		{"концом", "конц"},
		{"концу", "конц"},
		{"концы", "конц"},
		{"кончается", "конча"},
		{"кончалась", "конча"},
		{"кончались", "конча"},
		{"кончалось", "конча"},
		{"кончался", "конча"},
		{"кончатся", "кончат"},
		{"кончать", "конча"},
		{"кончаю", "конча"},
		{"кончаются", "конча"},
		{"кончен", "конч"},
		{"кончена", "конч"},
		{"кончено", "конч"},
		{"кончив", "конч"},
		{"кончивших", "конч"},
		{"кончик", "кончик"},
		{"кончике", "кончик"},
		{"кончики", "кончик"},
		{"кончиком", "кончик"},
		{"кончил", "конч"},
		{"кончила", "конч"},
		{"кончилась", "конч"},
		{"кончили", "конч"},
		{"кончились", "конч"},
		{"кончилось", "конч"},
		{"кончился", "конч"},
		{"кончим", "конч"},
		{"кончит", "конч"},
		{"кончится", "конч"},
		{"кончить", "конч"},
		{"кончиться", "конч"},
		{"кончишь", "конч"},
		{"кончу", "конч"},
		{"конь", "кон"},
		{"коньке", "коньк"},
		{"коньяк", "коньяк"},
		{"коньяком", "коньяк"},
		{"коньяку", "коньяк"},
		{"коньячком", "коньячк"},
		{"коню", "кон"},
		{"конюшен", "конюш"},
		{"конюшенную", "конюшен"},
		{"конюшне", "конюшн"},
		{"конюшни", "конюшн"},
		{"конюшню", "конюшн"},
		{"конюшня", "конюшн"},
		{"коня", "кон"},
		{"копаю", "копа"},
		{"копают", "копа"},
		{"копеек", "копеек"},
		{"копеечки", "копеечк"},
		{"копейка", "копейк"},
		{"копейками", "копейк"},
		{"копейке", "копейк"},
		{"копейки", "копейк"},
		{"копейкой", "копейк"},
		{"копейку", "копейк"},
		{"копен", "коп"},
		{"коперник", "коперник"},
		{"коперника", "коперник"},
		{"копи", "коп"},
		{"копии", "коп"},
		{"копит", "коп"},
		{"копить", "коп"},
		{"копиться", "коп"},
		{"копию", "коп"},
		{"копна", "копн"},
		{"копоть", "копот"},
		{"копошились", "копош"},
		{"копыт", "коп"},
		{"копыта", "копыт"},
		{"копытами", "копыт"},
		{"копытом", "копыт"},
		{"корабли", "корабл"},
		{"корабль", "корабл"},
		{"кораблю", "корабл"},
		{"корабля", "корабл"},
		{"кораллов", "коралл"},
		{"кордегардии", "кордегард"},
		{"корделией", "кордел"},
		{"корделии", "кордел"},
		{"корделия", "кордел"},
		{"коренной", "корен"},
		{"коренные", "корен"},
		{"коренным", "корен"},
		{"коренными", "корен"},
		{"коренных", "корен"},
		{"кореньями", "корен"},
		{"корешки", "корешк"},
		{"корзине", "корзин"},
		{"корзинку", "корзинк"},
		{"корзиной", "корзин"},
		{"корзиночкой", "корзиночк"},
		{"корзину", "корзин"},
		{"корзины", "корзин"},
		{"коридор", "коридор"},
		{"коридора", "коридор"},
		{"коридоре", "коридор"},
		{"коридоров", "коридор"},
		{"коридором", "коридор"},
		{"коридору", "коридор"},
		{"корит", "кор"},
		{"корица", "кориц"},
		{"корицей", "кориц"},
		{"корицу", "кориц"},
		{"корицы", "кориц"},
		{"коричневого", "коричнев"},
		{"коричневое", "коричнев"},
		{"коричневой", "коричнев"},
		{"коричневыми", "коричнев"},
		{"корка", "корк"},
		{"корки", "корк"},
		{"кормами", "корм"},
		{"корми", "корм"},
		{"кормил", "корм"},
		{"кормилец", "кормилец"},
		{"кормили", "корм"},
		{"кормилица", "кормилиц"},
		{"кормит", "корм"},
		{"кормить", "корм"},
		{"кормишь", "корм"},
		{"кормление", "кормлен"},
		{"кормят", "корм"},
		{"корней", "корн"},
		{"корнелия", "корнел"},
		{"корнем", "корн"},
		{"корнета", "корнет"},
		{"корнету", "корнет"},
		{"корни", "корн"},
		{"корню", "корн"},
		{"коробит", "короб"},
		{"коробить", "короб"},
		{"коробиться", "короб"},
		{"коробка", "коробк"},
		{"коробке", "коробк"},
		{"коробки", "коробк"},
		{"коробку", "коробк"},
		{"коробочками", "коробочк"},
		{"коробочке", "коробочк"},
		{"коробочки", "коробочк"},
		{"коробясь", "короб"},
		{"корова", "коров"},
		{"коровам", "коров"},
		{"корове", "коров"},
		{"коровник", "коровник"},
		{"корову", "коров"},
		{"коровы", "коров"},
		{"коровье", "коров"},
		{"коровьими", "коров"},
		{"корок", "корок"},
		{"королева", "королев"},
		{"королевой", "королев"},
		{"король", "корол"},
		{"короля", "корол"},
		{"короне", "корон"},
		{"корону", "корон"},
		{"короны", "корон"},
		{"коротаю", "корота"},
		{"коротенькие", "коротеньк"},
		{"коротенький", "коротеньк"},
		{"коротеньким", "коротеньк"},
		{"коротенькой", "коротеньк"},
		{"коротеньком", "коротеньк"},
		{"коротки", "коротк"},
		{"короткий", "коротк"},
		{"короткими", "коротк"},
		{"коротких", "коротк"},
		{"коротко", "коротк"},
		{"короткого", "коротк"},
		{"короткое", "коротк"},
		{"коротком", "коротк"},
		{"короче", "короч"},
		{"корпию", "корп"},
		{"корпус", "корпус"},
		{"корпуса", "корпус"},
		{"корпусом", "корпус"},
		{"корреджиевой", "корреджиев"},
		{"корректна", "корректн"},
		{"корректуру", "корректур"},
		{"корреспонденции", "корреспонденц"},
		{"корреспонденция", "корреспонденц"},
		{"корточках", "корточк"},
		{"корчаги", "корчаг"},
		{"корчит", "корч"},
		{"коршун", "коршун"},
		{"коршуну", "коршун"},
		{"коры", "кор"},
		{"корыстолюбивых", "корыстолюбив"},
		{"корысть", "корыст"},
		{"корыта", "корыт"},
		{"корытами", "корыт"},
		{"корыте", "корыт"},
		{"корыто", "корыт"},
		{"корь", "кор"},
		{"кос", "кос"},
		{"коса", "кос"},
		{"косами", "кос"},
		{"косвенно", "косвен"},
		{"косенький", "косеньк"},
		{"косится", "кос"},
		{"косичку", "косичк"},
		{"космос", "космос"},
		{"коснел", "коснел"},
		{"косности", "косност"},
		{"косноязычен", "косноязыч"},
		{"косноязычная", "косноязычн"},
		{"косноязычное", "косноязычн"},
		{"косноязычные", "косноязычн"},
		{"коснувшись", "коснувш"},
		{"коснулись", "коснул"},
		{"коснулось", "коснул"},
		{"коснулся", "коснул"},
		{"коснуться", "коснут"},
		{"косо", "кос"},
		{"косоворотке", "косоворотк"},
		{"косоворотку", "косоворотк"},
		{"косой", "кос"},
		{"косоплеткой", "косоплетк"},
		{"косою", "кос"},
		{"косте", "кост"},
		{"костей", "кост"},
		{"костер", "костер"},
		{"кости", "кост"},
		{"костлявой", "костляв"},
		{"косточкам", "косточк"},
		{"косточкой", "косточк"},
		{"косточку", "косточк"},
		{"костра", "костр"},
		{"кострами", "костр"},
		{"костров", "костр"},
		{"кость", "кост"},
		{"костюм", "костюм"},
		{"костюма", "костюм"},
		{"костюмах", "костюм"},
		{"костюме", "костюм"},
		{"костюмов", "костюм"},
		{"костюмом", "костюм"},
		{"костюму", "костюм"},
		{"костюшку", "костюшк"},
		{"костя", "кост"},
		{"костяной", "костян"},
		{"костяшек", "костяшек"},
		{"косы", "кос"},
		{"косым", "кос"},
		{"косынка", "косынк"},
		{"косынкой", "косынк"},
		{"косынку", "косынк"},
		{"косынок", "косынок"},
		{"косыночка", "косыночк"},
		{"косыночки", "косыночк"},
		{"косыночку", "косыночк"},
		{"косяком", "косяк"},
		{"косясь", "кос"},
		{"косят", "кос"},
		{"косятся", "кос"},
		{"кот", "кот"},
		{"кота", "кот"},
		{"котенка", "котенк"},
		{"котенком", "котенк"},
		{"котеночка", "котеночк"},
		{"котлет", "котлет"},
		{"котлетам", "котлет"},
		{"котлетки", "котлетк"},
		{"котлеты", "котлет"},
		{"котомку", "котомк"},
		{"которая", "котор"},
		{"которого", "котор"},
		{"которое", "котор"},
		{"которой", "котор"},
		{"котором", "котор"},
		{"которому", "котор"},
		{"которою", "котор"},
		{"которую", "котор"},
		{"которые", "котор"},
		{"который", "котор"},
		{"которым", "котор"},
		{"которыми", "котор"},
		{"которых", "котор"},
		{"коттедж", "коттедж"},
		{"коттень", "коттен"},
		{"коты", "кот"},
		{"кофе", "коф"},
		{"кофеем", "кофе"},
		{"кофеи", "коф"},
		{"кофей", "коф"},
		{"кофейник", "кофейник"},
		{"кофейника", "кофейник"},
		{"кофейником", "кофейник"},
		{"кофейнику", "кофейник"},
		{"кофейной", "кофейн"},
		{"кофею", "коф"},
		{"кофеями", "кофе"},
		{"кофий", "коф"},
		{"кофточка", "кофточк"},
		{"кофточки", "кофточк"},
		{"кофточку", "кофточк"},
		{"кох", "кох"},
		{"коха", "кох"},
		{"кохане", "кохан"},
		{"кохом", "кох"},
		{"кочевые", "кочев"},
		{"коченеют", "коченеют"},
		{"кочергой", "кочерг"},
		{"кочкам", "кочк"},
		{"кочки", "кочк"},
		{"кочующих", "коч"},
		{"кошачьи", "кошач"},
		{"кошек", "кошек"},
		{"кошелек", "кошелек"},
		{"кошелька", "кошельк"},
		{"кошельке", "кошельк"},
		{"кошемар", "кошемар"},
		{"кошка", "кошк"},
		{"кошки", "кошк"},
		{"кошкой", "кошк"},
		{"кошмар", "кошмар"},
		{"кощунственных", "кощунствен"},
		{"кра", "кра"},
		{"краб", "краб"},
		{"краденое", "краден"},
		{"крадеными", "краден"},
		{"крадешь", "крадеш"},
		{"краев", "кра"},
		{"краем", "кра"},
		{"краешком", "краешк"},
		{"кража", "краж"},
		{"краи", "кра"},
		{"край", "кра"},
		{"крайне", "крайн"},
		{"крайнее", "крайн"},
		{"крайней", "крайн"},
		{"крайний", "крайн"},
		{"крайних", "крайн"},
		{"крайностей", "крайност"},
		{"крайности", "крайност"},
		{"крайность", "крайност"},
		{"крайностях", "крайност"},
		{"крайнюю", "крайн"},
		{"крако", "крак"},
		{"крал", "крал"},
		{"кран", "кран"},
		{"крапиве", "крапив"},
		{"крапивой", "крапив"},
		{"крапиву", "крапив"},
		{"крапинками", "крапинк"},
		{"краса", "крас"},
		{"красавец", "красавец"},
		{"красавиц", "красавиц"},
		{"красавица", "красавиц"},
		{"красавицам", "красавиц"},
		{"красавице", "красавиц"},
		{"красавицей", "красавиц"},
		{"красавицы", "красавиц"},
		{"красавцем", "красавц"},
		{"красе", "крас"},
		{"красен", "крас"},
		{"красив", "крас"},
		{"красива", "красив"},
		{"красивая", "красив"},
		{"красивее", "красив"},
		{"красивенькая", "красивеньк"},
		{"красиво", "красив"},
		{"красивое", "красив"},
		{"красивой", "красив"},
		{"красивом", "красив"},
		{"красивую", "красив"},
		{"красивы", "красив"},
		{"красивые", "красив"},
		{"красивый", "красив"},
		{"красивыми", "красив"},
		{"красивых", "красив"},
		{"красил", "крас"},
		{"красили", "крас"},
		{"красильщик", "красильщик"},
		{"красильщика", "красильщик"},
		{"красильщике", "красильщик"},
		{"красильщики", "красильщик"},
		{"красильщиков", "красильщик"},
		{"красит", "крас"},
		{"красить", "крас"},
		{"краска", "краск"},
		{"красками", "краск"},
		{"красках", "краск"},
		{"краске", "краск"},
		{"краски", "краск"},
		{"краской", "краск"},
		{"краскою", "краск"},
		{"краску", "краск"},
		{"красная", "красн"},
		{"краснеет", "краснеет"},
		{"краснеешь", "краснееш"},
		{"краснел", "краснел"},
		{"краснела", "краснел"},
		{"красненькие", "красненьк"},
		{"красненький", "красненьк"},
		{"красненькой", "красненьк"},
		{"красненькую", "красненьк"},
		{"краснеть", "краснет"},
		{"краснею", "красн"},
		{"краснеющим", "краснеющ"},
		{"краснея", "красне"},
		{"красно", "красн"},
		{"красноватое", "красноват"},
		{"красноваты", "красноват"},
		{"красноватые", "красноват"},
		{"красноватый", "красноват"},
		{"красноватыми", "красноват"},
		{"красного", "красн"},
		{"красное", "красн"},
		{"красной", "красн"},
		{"красном", "красн"},
		{"красному", "красн"},
		{"красноречиво", "красноречив"},
		{"красноречивое", "красноречив"},
		{"красноречивые", "красноречив"},
		{"красноречие", "краснореч"},
		{"краснощекая", "краснощек"},
		{"красною", "красн"},
		{"красную", "красн"},
		{"красны", "красн"},
		{"красные", "красн"},
		{"красный", "красн"},
		{"красным", "красн"},
		{"красными", "красн"},
		{"красных", "красн"},
		{"красовался", "красова"},
		{"красой", "крас"},
		{"красок", "красок"},
		{"красот", "красот"},
		{"красота", "красот"},
		{"красоте", "красот"},
		{"красотой", "красот"},
		{"красотою", "красот"},
		{"красоту", "красот"},
		{"красоты", "красот"},
		{"красочны", "красочн"},
		{"красть", "краст"},
		{"красу", "крас"},
		{"красы", "крас"},
		{"кратер", "кратер"},
		{"краткие", "кратк"},
		{"кратко", "кратк"},
		{"кратковременного", "кратковремен"},
		{"краткости", "краткост"},
		{"краткую", "кратк"},
		{"кратчайшим", "кратчайш"},
		{"крахмалит", "крахмал"},
		{"крашеные", "крашен"},
		{"крашеный", "крашен"},
		{"краю", "кра"},
		{"края", "кра"},
		{"краям", "кра"},
		{"краях", "кра"},
		{"кредит", "кред"},
		{"кредитивные", "кредитивн"},
		{"кредитками", "кредитк"},
		{"кредитке", "кредитк"},
		{"кредитки", "кредитк"},
		{"кредитку", "кредитк"},
		{"кредитный", "кредитн"},
		{"кредитными", "кредитн"},
		{"кредитовать", "кредитова"},
		{"кредиток", "кредиток"},
		{"кредиторшей", "кредиторш"},
		{"кредиту", "кредит"},
		{"кремля", "кремл"},
		{"кремнист", "кремнист"},
		{"кренделей", "крендел"},
		{"крендельков", "крендельк"},
		{"кренделями", "крендел"},
		{"крепилась", "креп"},
		{"крепили", "креп"},
		{"крепился", "креп"},
		{"крепитесь", "креп"},
		{"крепится", "креп"},
		{"крепка", "крепк"},
		{"крепкие", "крепк"},
		{"крепкий", "крепк"},
		{"крепким", "крепк"},
		{"крепко", "крепк"},
		{"крепкое", "крепк"},
		{"крепкой", "крепк"},
		{"крепкую", "крепк"},
		{"крепнет", "крепнет"},
		{"крепок", "крепок"},
		{"крепости", "крепост"},
		{"крепостник", "крепостник"},
		{"крепостники", "крепостник"},
		{"крепостного", "крепостн"},
		{"крепостном", "крепостн"},
		{"крепостные", "крепостн"},
		{"крепостным", "крепостн"},
		{"крепостных", "крепостн"},
		{"крепость", "крепост"},
		{"крепостью", "крепост"},
		{"крепчал", "крепча"},
		{"крепче", "крепч"},
		{"крепь", "креп"},
		{"крепясь", "креп"},
		{"кресел", "кресел"},
		{"кресла", "кресл"},
		{"креслами", "кресл"},
		{"креслах", "кресл"},
		{"кресле", "кресл"},
		{"кресло", "кресл"},
		{"креслом", "кресл"},
		{"крест", "крест"},
		{"креста", "крест"},
		{"крестами", "крест"},
		{"кресте", "крест"},
		{"крестик", "крестик"},
		{"крестил", "крест"},
		{"крестила", "крест"},
		{"крестилась", "крест"},
		{"крестили", "крест"},
		{"крестился", "крест"},
		{"крестин", "крестин"},
		{"крестины", "крестин"},
		{"крестит", "крест"},
		{"крестится", "крест"},
		{"крестить", "крест"},
		{"крестненький", "крестненьк"},
		{"крестную", "крестн"},
		{"крестным", "крестн"},
		{"крестов", "крест"},
		{"крестовском", "крестовск"},
		{"крестовые", "крестов"},
		{"крестом", "крест"},
		{"кресту", "крест"},
		{"кресты", "крест"},
		{"крестьян", "крестья"},
		{"крестьянам", "крестьян"},
		{"крестьянами", "крестьян"},
		{"крестьяне", "крестьян"},
		{"крестьянин", "крестьянин"},
		{"крестьянина", "крестьянин"},
		{"крестьянка", "крестьянк"},
		{"крестьянская", "крестьянск"},
		{"крестьянских", "крестьянск"},
		{"крестьянской", "крестьянск"},
		{"крестясь", "крест"},
		{"крещены", "крещ"},
		{"крещенье", "крещен"},
		{"крещенью", "крещен"},
		{"кривая", "крив"},
		{"кривизну", "кривизн"},
		{"кривизны", "кривизн"},
		{"кривила", "крив"},
		{"кривились", "крив"},
		{"криво", "крив"},
		{"кривой", "крив"},
		{"кривом", "крив"},
		{"кривы", "крив"},
		{"кривыми", "крив"},
		{"кризис", "кризис"},
		{"крик", "крик"},
		{"крика", "крик"},
		{"криками", "крик"},
		{"криках", "крик"},
		{"крике", "крик"},
		{"крики", "крик"},
		{"крикливо", "криклив"},
		{"крикнет", "крикнет"},
		{"крикнешь", "крикнеш"},
		{"крикнул", "крикнул"},
		{"крикнула", "крикнул"},
		{"крикнуть", "крикнут"},
		{"криков", "крик"},
		{"криком", "крик"},
		{"крику", "крик"},
		{"кринолине", "кринолин"},
		{"кристаль", "кристал"},
		{"кристально", "кристальн"},
		{"критик", "критик"},
		{"критики", "критик"},
		{"критиков", "критик"},
		{"критиковать", "критикова"},
		{"крича", "крич"},
		{"кричал", "крича"},
		{"кричала", "крича"},
		{"кричали", "крича"},
		{"кричаль", "кричал"},
		{"кричат", "кричат"},
		{"кричать", "крича"},
		{"кричащем", "крича"},
		{"кричи", "крич"},
		{"кричит", "крич"},
		{"кричите", "крич"},
		{"кричишь", "крич"},
		{"кричу", "крич"},
		{"кров", "кров"},
		{"кровавая", "кровав"},
		{"кровавиться", "кровав"},
		{"кровавого", "кровав"},
		{"кровавой", "кровав"},
		{"кровавый", "кровав"},
		{"кровавых", "кровав"},
		{"кровати", "кроват"},
		{"кроватка", "кроватк"},
		{"кроватке", "кроватк"},
		{"кровать", "крова"},
		{"кроватью", "кроват"},
		{"кроватями", "кроват"},
		{"кровель", "кровел"},
		{"крови", "кров"},
		{"кровию", "кров"},
		{"кровлей", "кровл"},
		{"кровлю", "кровл"},
		{"кровля", "кровл"},
		{"кровлями", "кровл"},
		{"кровная", "кровн"},
		{"кровное", "кровн"},
		{"кровные", "кровн"},
		{"кровный", "кровн"},
		{"кровом", "кров"},
		{"кровообращение", "кровообращен"},
		{"кровопроливцы", "кровопроливц"},
		{"кровопролитий", "кровопролит"},
		{"кровотечение", "кровотечен"},
		{"кровь", "кров"},
		{"кровью", "кров"},
		{"кроется", "кроет"},
		{"кроила", "кро"},
		{"кроить", "кро"},
		{"крокета", "крокет"},
		{"крокодилов", "крокодил"},
		{"кроликов", "кролик"},
		{"кроме", "кром"},
		{"кроне", "крон"},
		{"крот", "крот"},
		{"кроткая", "кротк"},
		{"кроткие", "кротк"},
		{"кроткий", "кротк"},
		{"кротким", "кротк"},
		{"кроткими", "кротк"},
		{"кротких", "кротк"},
		{"кротко", "кротк"},
		{"кроткого", "кротк"},
		{"кроткое", "кротк"},
		{"кроток", "кроток"},
		{"кротости", "кротост"},
		{"кротостию", "кротост"},
		{"кротость", "кротост"},
		{"кротостью", "кротост"},
		{"крошат", "крошат"},
		{"крошеные", "крошен"},
		{"крошечная", "крошечн"},
		{"крошечное", "крошечн"},
		{"крошечную", "крошечн"},
		{"крошечные", "крошечн"},
		{"крошить", "крош"},
		{"крошка", "крошк"},
		{"крошки", "крошк"},
		{"кру", "кру"},
		{"круазе", "круаз"},
		{"круг", "круг"},
		{"круга", "круг"},
		{"кругам", "круг"},
		{"кругами", "круг"},
		{"кругах", "круг"},
		{"круге", "круг"},
		{"круги", "круг"},
		{"кругла", "кругл"},
		{"круглая", "кругл"},
		{"кругленькие", "кругленьк"},
		{"круглого", "кругл"},
		{"круглое", "кругл"},
		{"круглой", "кругл"},
		{"круглолицая", "круглолиц"},
		{"круглом", "кругл"},
		{"круглую", "кругл"},
		{"круглые", "кругл"},
		{"круглый", "кругл"},
		{"круглым", "кругл"},
		{"круглыми", "кругл"},
		{"круглых", "кругл"},
		{"кругов", "круг"},
		{"круговорот", "круговорот"},
		{"круговорота", "круговорот"},
		{"кругом", "круг"},
		{"кругу", "круг"},
		{"кружатся", "кружат"},
		{"кружащаяся", "кружа"},
		{"кружащийся", "кружа"},
		{"кружев", "круж"},
		{"кружева", "кружев"},
		{"кружевами", "кружев"},
		{"кружевах", "кружев"},
		{"кружевною", "кружевн"},
		{"кружевных", "кружевн"},
		{"кружево", "кружев"},
		{"кружевцом", "кружевц"},
		{"круженье", "кружен"},
		{"круженьем", "кружен"},
		{"круженья", "кружен"},
		{"кружи", "круж"},
		{"кружилась", "круж"},
		{"кружились", "круж"},
		{"кружит", "круж"},
		{"кружится", "круж"},
		{"кружить", "круж"},
		{"кружиться", "круж"},
		{"кружках", "кружк"},
		{"кружке", "кружк"},
		{"кружки", "кружк"},
		{"кружку", "кружк"},
		{"кружок", "кружок"},
		{"крупно", "крупн"},
		{"крупного", "крупн"},
		{"крупной", "крупн"},
		{"крупные", "крупн"},
		{"крупным", "крупн"},
		{"крупными", "крупн"},
		{"крупных", "крупн"},
		{"крупой", "круп"},
		{"крупы", "круп"},
		{"крутая", "крут"},
		{"крутившимся", "крут"},
		{"крутизну", "крутизн"},
		{"крутились", "крут"},
		{"крутился", "крут"},
		{"круто", "крут"},
		{"крутого", "крут"},
		{"крутой", "крут"},
		{"круторогий", "круторог"},
		{"крутым", "крут"},
		{"крутыми", "крут"},
		{"крутясь", "крут"},
		{"кручами", "круч"},
		{"круче", "круч"},
		{"кручи", "круч"},
		{"кручину", "кручин"},
		{"круша", "круш"},
		{"крушить", "круш"},
		{"крыл", "крыл"},
		{"крыла", "крыл"},
		{"крылами", "крыл"},
		{"крылатое", "крылат"},
		{"крылатый", "крылат"},
		{"крылах", "крыл"},
		{"крылий", "крыл"},
		{"крылись", "крыл"},
		{"крыло", "крыл"},
		{"крылова", "крылов"},
		{"крылом", "крыл"},
		{"крыльев", "крыл"},
		{"крыльем", "крыл"},
		{"крыльца", "крыльц"},
		{"крыльце", "крыльц"},
		{"крыльцо", "крыльц"},
		{"крыльцом", "крыльц"},
		{"крыльцу", "крыльц"},
		{"крылья", "крыл"},
		{"крыльями", "крыл"},
		{"крыльях", "крыл"},
		{"крым", "крым"},
		{"крыма", "крым"},
		{"крыс", "крыс"},
		{"крысах", "крыс"},
		{"крысиную", "крысин"},
		{"крысой", "крыс"},
		{"крысы", "крыс"},
		{"крытая", "крыт"},
		{"крытой", "крыт"},
		{"крытые", "крыт"},
		{"крыш", "крыш"},
		{"крыше", "крыш"},
		{"крышей", "крыш"},
		{"крыши", "крыш"},
		{"крышку", "крышк"},
		{"крышу", "крыш"},
		{"крюк", "крюк"},
		{"крюков", "крюк"},
		{"крюком", "крюк"},
		{"крюку", "крюк"},
		{"крючка", "крючк"},
		{"крючках", "крючк"},
		{"крючке", "крючк"},
		{"крючком", "крючк"},
		{"крючкотворец", "крючкотворец"},
		{"крючок", "крючок"},
		{"крючочки", "крючочк"},
		{"крякнет", "крякнет"},
		{"крякнул", "крякнул"},
		{"кряхтела", "кряхтел"},
		{"кряхтит", "кряхт"},
		{"кряхтите", "кряхт"},
		{"кряхтя", "кряхт"},
		{"кстати", "кстат"},
		{"ксюша", "ксюш"},
		{"кто", "кто"},
		{"кувшинами", "кувшин"},
		{"кувыркнулся", "кувыркнул"},
		{"кувырком", "кувырк"},
		{"куда", "куд"},
		{"кудахтает", "кудахта"},
		{"кудахтал", "кудахта"},
		{"кудахтанье", "кудахтан"},
		{"кудахтаньем", "кудахтан"},
		{"кудахтающие", "кудахта"},
		{"кудель", "кудел"},
		{"кудрей", "кудр"},
		{"кудри", "кудр"},
		{"кудрино", "кудрин"},
		{"кудрявой", "кудряв"},
		{"кудрями", "кудр"},
		{"куды", "куд"},
		{"кузины", "кузин"},
		{"кузнец", "кузнец"},
		{"кузнеца", "кузнец"},
		{"кузнецов", "кузнец"},
		{"кузнецу", "кузнец"},
		{"кузнечики", "кузнечик"},
		{"кузница", "кузниц"},
		{"кузовковы", "кузовков"},
		{"кузьку", "кузьк"},
		{"кузьминична", "кузьминичн"},
		{"куклы", "кукл"},
		{"куколку", "куколк"},
		{"кукушка", "кукушк"},
		{"кулак", "кулак"},
		{"кулаками", "кулак"},
		{"кулаке", "кулак"},
		{"кулаки", "кулак"},
		{"кулаком", "кулак"},
		{"кулачонками", "кулачонк"},
		{"кулебяк", "кулебяк"},
		{"кулебяка", "кулебяк"},
		{"кулебяки", "кулебяк"},
		{"куликовом", "куликов"},
		{"куликовым", "куликов"},
		{"кулинарной", "кулинарн"},
		{"кулиса", "кулис"},
		{"кулыгин", "кулыгин"},
		{"кулыгина", "кулыгин"},
		{"кулыгину", "кулыгин"},
		{"кулыгиным", "кулыгин"},
		{"кулькова", "кульков"},
		{"кульком", "кульк"},
		{"культурные", "культурн"},
		{"кум", "кум"},
		{"кума", "кум"},
		{"кумачах", "кумач"},
		{"куме", "кум"},
		{"кумир", "кумир"},
		{"кумирню", "кумирн"},
		{"кумиром", "кумир"},
		{"кумиры", "кумир"},
		{"кумой", "кум"},
		{"кумом", "кум"},
		{"куму", "кум"},
		{"кумы", "кум"},
		{"купала", "купа"},
		{"купалась", "купа"},
		{"купались", "купа"},
		{"купальне", "купальн"},
		{"купальню", "купальн"},
		{"купальня", "купальн"},
		{"купания", "купан"},
		{"купаньем", "купан"},
		{"купаться", "купа"},
		{"купец", "купец"},
		{"купеческие", "купеческ"},
		{"купеческих", "купеческ"},
		{"купи", "куп"},
		{"купидонов", "купидон"},
		{"купил", "куп"},
		{"купила", "куп"},
		{"купит", "куп"},
		{"купить", "куп"},
		{"купишь", "куп"},
		{"куплен", "купл"},
		{"куплена", "купл"},
		{"купленное", "куплен"},
		{"купленную", "куплен"},
		{"купленные", "куплен"},
		{"купленными", "куплен"},
		{"куплено", "купл"},
		{"куплю", "купл"},
		{"купол", "купол"},
		{"куполом", "купол"},
		{"купоны", "купон"},
		{"купца", "купц"},
		{"купцам", "купц"},
		{"купцах", "купц"},
		{"купцов", "купц"},
		{"купцу", "купц"},
		{"купцы", "купц"},
		{"купчик", "купчик"},
		{"купчиха", "купчих"},
		{"кур", "кур"},
		{"куражился", "кураж"},
		{"курами", "кур"},
		{"куранты", "курант"},
		{"куреньями", "курен"},
		{"курил", "кур"},
		{"курилась", "кур"},
		{"курился", "кур"},
		{"куриная", "курин"},
		{"куриную", "курин"},
		{"курит", "кур"},
		{"курите", "кур"},
		{"курить", "кур"},
		{"курица", "куриц"},
		{"курице", "куриц"},
		{"курицей", "куриц"},
		{"курицу", "куриц"},
		{"курносое", "курнос"},
		{"курносые", "курнос"},
		{"курок", "курок"},
		{"куролесил", "куролес"},
		{"куропатки", "куропатк"},
		{"курс", "курс"},
		{"курса", "курс"},
		{"курсистка", "курсистк"},
		{"курске", "курск"},
		{"куртины", "куртин"},
		{"куртке", "куртк"},
		{"куртки", "куртк"},
		{"куртку", "куртк"},
		{"курточки", "курточк"},
		{"курточку", "курточк"},
		{"курчавою", "курчав"},
		{"курчавый", "курчав"},
		{"куры", "кур"},
		{"курьез", "курьез"},
		{"курьезу", "курьез"},
		{"курьер", "курьер"},
		{"курьера", "курьер"},
		{"курьерский", "курьерск"},
		{"курьеру", "курьер"},
		{"курю", "кур"},
		{"куря", "кур"},
		{"курятникам", "курятник"},
		{"курятники", "курятник"},
		{"курящейся", "куря"},
		{"курящих", "куря"},
		{"кусали", "куса"},
		{"кусая", "кус"},
		{"куска", "куск"},
		{"кускам", "куск"},
		{"куски", "куск"},
		{"куском", "куск"},
		{"кусок", "кусок"},
		{"кусочек", "кусочек"},
		{"кусочка", "кусочк"},
		{"кусочкам", "кусочк"},
		{"кусочками", "кусочк"},
		{"кусочки", "кусочк"},
		{"куст", "куст"},
		{"кустам", "куст"},
		{"кустами", "куст"},
		{"кустарник", "кустарник"},
		{"кустах", "куст"},
		{"кусте", "куст"},
		{"кустика", "кустик"},
		{"кустов", "куст"},
		{"кустом", "куст"},
		{"кусты", "куст"},
		{"кутает", "кута"},
		{"кутались", "кута"},
		{"кутаясь", "кут"},
		{"кутит", "кут"},
		{"кутьи", "кут"},
		{"кутью", "кут"},
		{"кутья", "кут"},
		{"кухарка", "кухарк"},
		{"кухарками", "кухарк"},
		{"кухарках", "кухарк"},
		{"кухарке", "кухарк"},
		{"кухарки", "кухарк"},
		{"кухаркой", "кухарк"},
		{"кухарку", "кухарк"},
		{"кухарок", "кухарок"},
		{"кухмистерской", "кухмистерск"},
		{"кухне", "кухн"},
		{"кухней", "кухн"},
		{"кухни", "кухн"},
		{"кухню", "кухн"},
		{"кухня", "кухн"},
		{"кухням", "кухн"},
		{"кухонной", "кухон"},
		{"кухонный", "кухон"},
		{"куцой", "куц"},
		{"куча", "куч"},
		{"куче", "куч"},
		{"кучей", "куч"},
		{"кучер", "кучер"},
		{"кучера", "кучер"},
		{"кучерам", "кучер"},
		{"кучерами", "кучер"},
		{"кучере", "кучер"},
		{"кучером", "кучер"},
		{"кучеру", "кучер"},
		{"кучи", "куч"},
		{"кучка", "кучк"},
		{"кучками", "кучк"},
		{"кучки", "кучк"},
		{"кучку", "кучк"},
		{"кучу", "куч"},
		{"кушает", "куша"},
		{"кушай", "куша"},
		{"кушак", "кушак"},
		{"кушаком", "кушак"},
		{"кушал", "куша"},
		{"кушала", "куша"},
		{"кушало", "куша"},
		{"кушанье", "кушан"},
		{"кушанья", "кушан"},
		{"кушать", "куша"},
		{"кушаю", "куша"},
		{"кушают", "куша"},
		{"кушетке", "кушетк"},
		{"кушетку", "кушетк"},
		{"куши", "куш"},
		{"куще", "кущ"},
		{"кующей", "кующ"},
		{"кхе", "кхе"},
		{"кхи", "кхи"},
		{"л", "л"},
		{"ла", "ла"},
		{"лабаз", "лабаз"},
		{"лабазника", "лабазник"},
		{"лабазнике", "лабазник"},
		{"лабазнику", "лабазник"},
		{"лабиринт", "лабиринт"},
		{"лавиза", "лавиз"},
		{"лавизе", "лавиз"},
		{"лавизу", "лавиз"},
		{"лавизы", "лавиз"},
		{"лавину", "лавин"},
		{"лавкам", "лавк"},
		{"лавках", "лавк"},
		{"лавке", "лавк"},
		{"лавки", "лавк"},
		{"лавкой", "лавк"},
		{"лавку", "лавк"},
		{"лавочек", "лавочек"},
		{"лавочка", "лавочк"},
		{"лавочкам", "лавочк"},
		{"лавочках", "лавочк"},
		{"лавочке", "лавочк"},
		{"лавочки", "лавочк"},
		{"лавочку", "лавочк"},
		{"лавочник", "лавочник"},
		{"лавочники", "лавочник"},
		{"лавочнику", "лавочник"},
		{"лавочные", "лавочн"},
		{"лавр", "лавр"},
		{"лаврентий", "лаврент"},
		{"лаврентия", "лаврент"},
		{"лавром", "лавр"},
		{"лавры", "лавр"},
		{"лагере", "лагер"},
		{"лагерь", "лагер"},
		{"лагуне", "лагун"},
		{"лагуны", "лагун"},
		{"лад", "лад"},
		{"ладан", "лада"},
		{"ладах", "лад"},
		{"ладил", "лад"},
		{"ладить", "лад"},
		{"ладно", "ладн"},
		{"ладони", "ладон"},
		{"ладонь", "ладон"},
		{"ладонью", "ладон"},
		{"ладонями", "ладон"},
		{"лады", "лад"},
		{"ладью", "лад"},
		{"лае", "ла"},
		{"лаем", "ла"},
		{"лает", "лает"},
		{"лаетесь", "лает"},
		{"лается", "лает"},
		{"лазаревич", "лазаревич"},
		{"лазарь", "лазар"},
		{"лазаря", "лазар"},
		{"лазил", "лаз"},
		{"лазить", "лаз"},
		{"лазури", "лазур"},
		{"лазурной", "лазурн"},
		{"лазурь", "лазур"},
		{"лай", "ла"},
		{"лайдак", "лайдак"},
		{"лайтесь", "лайт"},
		{"лакедемонов", "лакедемон"},
		{"лакее", "лак"},
		{"лакеев", "лаке"},
		{"лакеи", "лак"},
		{"лакей", "лак"},
		{"лакейским", "лакейск"},
		{"лакейской", "лакейск"},
		{"лакейскому", "лакейск"},
		{"лакейскую", "лакейск"},
		{"лакейщину", "лакейщин"},
		{"лакею", "лак"},
		{"лакея", "лаке"},
		{"лакеям", "лаке"},
		{"лакированные", "лакирова"},
		{"лакомств", "лакомств"},
		{"лампа", "ламп"},
		{"лампада", "лампад"},
		{"лампадки", "лампадк"},
		{"лампадой", "лампад"},
		{"лампе", "ламп"},
		{"лампой", "ламп"},
		{"лампу", "ламп"},
		{"лампы", "ламп"},
		{"лангвагеном", "лангваген"},
		{"лангвагену", "лангваген"},
		{"ландышах", "ландыш"},
		{"ландышей", "ландыш"},
		{"ландыши", "ландыш"},
		{"ланитам", "ланит"},
		{"ланиты", "ланит"},
		{"лапах", "лап"},
		{"лаптев", "лапт"},
		{"лаптей", "лапт"},
		{"лапу", "лап"},
		{"лапши", "лапш"},
		{"лапшу", "лапш"},
		{"ларец", "ларец"},
		{"ларцов", "ларц"},
		{"лас", "лас"},
		{"ласк", "ласк"},
		{"ласка", "ласк"},
		{"ласкает", "ласка"},
		{"ласкал", "ласка"},
		{"ласкала", "ласка"},
		{"ласкалась", "ласка"},
		{"ласками", "ласк"},
		{"ласкательный", "ласкательн"},
		{"ласках", "ласк"},
		{"ласкаю", "ласка"},
		{"ласкают", "ласка"},
		{"ласкаются", "ласка"},
		{"ласкающая", "ласка"},
		{"ласкающем", "ласка"},
		{"ласкающую", "ласка"},
		{"лаская", "ласк"},
		{"ласкаясь", "ласк"},
		{"ласке", "ласк"},
		{"ласки", "ласк"},
		{"ласков", "ласк"},
		{"ласковая", "ласков"},
		{"ласковее", "ласков"},
		{"ласково", "ласков"},
		{"ласкового", "ласков"},
		{"ласковость", "ласков"},
		{"ласковы", "ласков"},
		{"ласковые", "ласков"},
		{"ласковый", "ласков"},
		{"ласковым", "ласков"},
		{"ласковых", "ласков"},
		{"лаской", "ласк"},
		{"ласку", "ласк"},
		{"ласточка", "ласточк"},
		{"ласточкино", "ласточкин"},
		{"лат", "лат"},
		{"латах", "лат"},
		{"латник", "латник"},
		{"латыни", "латын"},
		{"латынскую", "латынск"},
		{"латынью", "латын"},
		{"латыши", "латыш"},
		{"лафитом", "лафит"},
		{"лачуг", "лачуг"},
		{"лая", "ла"},
		{"лаяла", "лая"},
		{"лаять", "лая"},
		{"лба", "лба"},
		{"лбов", "лбов"},
		{"лбом", "лбом"},
		{"лбу", "лбу"},
		{"лгал", "лгал"},
		{"лгала", "лгал"},
		{"лгали", "лгал"},
		{"лгать", "лгат"},
		{"лги", "лги"},
		{"лгу", "лгу"},
		{"лгунишка", "лгунишк"},
		{"лгунью", "лгун"},
		{"лгут", "лгут"},
		{"лгущие", "лгущ"},
		{"лебедей", "лебед"},
		{"лебеди", "лебед"},
		{"лебедь", "лебед"},
		{"лебезил", "лебез"},
		{"лебезятников", "лебезятник"},
		{"лебезятникова", "лебезятников"},
		{"лебезятникове", "лебезятников"},
		{"лебезятникову", "лебезятников"},
		{"лебезятниковым", "лебезятников"},
		{"левая", "лев"},
		{"левого", "лев"},
		{"левой", "лев"},
		{"левом", "лев"},
		{"левою", "лев"},
		{"левую", "лев"},
		{"левый", "лев"},
		{"левым", "лев"},
		{"лег", "лег"},
		{"легавая", "легав"},
		{"легион", "легион"},
		{"легиона", "легион"},
		{"легистраторша", "легистраторш"},
		{"легка", "легк"},
		{"легкая", "легк"},
		{"легкие", "легк"},
		{"легкий", "легк"},
		{"легким", "легк"},
		{"легкими", "легк"},
		{"легких", "легк"},
		{"легко", "легк"},
		{"легковерная", "легковерн"},
		{"легковерные", "легковерн"},
		{"легкого", "легк"},
		{"легкое", "легк"},
		{"легкой", "легк"},
		{"легком", "легк"},
		{"легкомыслен", "легкомысл"},
		{"легкомысленнее", "легкомыслен"},
		{"легкомысленно", "легкомыслен"},
		{"легкомысленного", "легкомыслен"},
		{"легкомысленное", "легкомыслен"},
		{"легкомысленной", "легкомыслен"},
		{"легкомысленных", "легкомыслен"},
		{"легкомыслие", "легкомысл"},
		{"легкомыслием", "легкомысл"},
		{"легкомыслии", "легкомысл"},
		{"легкомыслия", "легкомысл"},
		{"легкоперой", "легкопер"},
		{"легкою", "легк"},
		{"легкую", "легк"},
		{"легла", "легл"},
		{"легли", "легл"},
		{"легло", "легл"},
		{"легок", "легок"},
		{"легонько", "легоньк"},
		{"легче", "легч"},
		{"лед", "лед"},
		{"леденец", "леденец"},
		{"леденея", "ледене"},
		{"леденит", "леден"},
		{"леденцами", "леденц"},
		{"леденцах", "леденц"},
		{"леденящий", "леденя"},
		{"леди", "лед"},
		{"ледком", "ледк"},
		{"ледоход", "ледоход"},
		{"ледяная", "ледян"},
		{"лежа", "леж"},
		{"лежавшего", "лежа"},
		{"лежавшее", "лежа"},
		{"лежавшие", "лежа"},
		{"лежавший", "лежа"},
		{"лежавшим", "лежа"},
		{"лежавшими", "лежа"},
		{"лежавших", "лежа"},
		{"лежавшую", "лежа"},
		{"лежал", "лежа"},
		{"лежала", "лежа"},
		{"лежали", "лежа"},
		{"лежало", "лежа"},
		{"лежанке", "лежанк"},
		{"лежанки", "лежанк"},
		{"лежанку", "лежанк"},
		{"лежанок", "лежанок"},
		{"лежанье", "лежан"},
		{"лежаньем", "лежан"},
		{"лежанья", "лежан"},
		{"лежат", "лежат"},
		{"лежать", "лежа"},
		{"лежащего", "лежа"},
		{"лежащий", "лежа"},
		{"лежащими", "лежа"},
		{"лежащую", "лежа"},
		{"лежи", "леж"},
		{"лежит", "леж"},
		{"лежите", "леж"},
		{"лежишь", "леж"},
		{"лежу", "леж"},
		{"лез", "лез"},
		{"лезвие", "лезв"},
		{"лезвием", "лезв"},
		{"лезвию", "лезв"},
		{"лезгинку", "лезгинк"},
		{"лезет", "лезет"},
		{"лезешь", "лезеш"},
		{"лезть", "лезт"},
		{"лезут", "лезут"},
		{"лейтмотив", "лейтмот"},
		{"лейтмотивом", "лейтмотив"},
		{"лекарем", "лекар"},
		{"лекарств", "лекарств"},
		{"лекарства", "лекарств"},
		{"лекарствами", "лекарств"},
		{"лекарствах", "лекарств"},
		{"лекарственных", "лекарствен"},
		{"лекарство", "лекарств"},
		{"лексиконе", "лексикон"},
		{"лектор", "лектор"},
		{"лекции", "лекц"},
		{"лекций", "лекц"},
		{"лекцию", "лекц"},
		{"лелеемый", "лелеем"},
		{"лелеет", "лелеет"},
		{"лелей", "лел"},
		{"лелеял", "лелея"},
		{"лена", "лен"},
		{"лене", "лен"},
		{"лени", "лен"},
		{"ленив", "лен"},
		{"ленивая", "ленив"},
		{"ленивее", "ленив"},
		{"ленивец", "ленивец"},
		{"лениво", "ленив"},
		{"ленивого", "ленив"},
		{"ленивое", "ленив"},
		{"ленивой", "ленив"},
		{"ленивом", "ленив"},
		{"ленивою", "ленив"},
		{"ленивы", "ленив"},
		{"ленивые", "ленив"},
		{"ленивый", "ленив"},
		{"ленивых", "ленив"},
		{"ленился", "лен"},
		{"ленора", "ленор"},
		{"лености", "леност"},
		{"леночка", "леночк"},
		{"лентами", "лент"},
		{"лентах", "лент"},
		{"ленточек", "ленточек"},
		{"ленточке", "ленточк"},
		{"лентою", "лент"},
		{"ленты", "лент"},
		{"лентяй", "лентя"},
		{"лентяя", "лент"},
		{"лень", "лен"},
		{"ленью", "лен"},
		{"леню", "лен"},
		{"леня", "лен"},
		{"леонардо", "леонард"},
		{"леонид", "леонид"},
		{"леонида", "леонид"},
		{"лепестки", "лепестк"},
		{"лепет", "лепет"},
		{"лепетал", "лепета"},
		{"лепетала", "лепета"},
		{"лепетом", "лепет"},
		{"лепечет", "лепечет"},
		{"лепилась", "леп"},
		{"лепились", "леп"},
		{"лепно", "лепн"},
		{"лепят", "леп"},
		{"лепятся", "леп"},
		{"лермонтов", "лермонт"},
		{"лермонтова", "лермонтов"},
		{"лес", "лес"},
		{"леса", "лес"},
		{"лесами", "лес"},
		{"лесах", "лес"},
		{"лесе", "лес"},
		{"лесничестве", "лесничеств"},
		{"лесничество", "лесничеств"},
		{"лесничий", "леснич"},
		{"лесной", "лесн"},
		{"лесные", "лесн"},
		{"лесных", "лесн"},
		{"лесов", "лес"},
		{"лесок", "лесок"},
		{"лесом", "лес"},
		{"лести", "лест"},
		{"лестию", "лест"},
		{"лестниц", "лестниц"},
		{"лестница", "лестниц"},
		{"лестницами", "лестниц"},
		{"лестнице", "лестниц"},
		{"лестницей", "лестниц"},
		{"лестницу", "лестниц"},
		{"лестницы", "лестниц"},
		{"лестничестве", "лестничеств"},
		{"лестно", "лестн"},
		{"лестного", "лестн"},
		{"лестным", "лестн"},
		{"лестных", "лестн"},
		{"лесть", "лест"},
		{"лестью", "лест"},
		{"лесу", "лес"},
		{"лет", "лет"},
		{"лета", "лет"},
		{"летает", "лета"},
		{"летаешь", "лета"},
		{"летай", "лета"},
		{"летал", "лета"},
		{"летала", "лета"},
		{"летало", "лета"},
		{"летам", "лет"},
		{"летами", "лет"},
		{"летать", "лета"},
		{"летах", "лет"},
		{"летают", "лета"},
		{"летевшую", "летевш"},
		{"летел", "летел"},
		{"летела", "летел"},
		{"летели", "летел"},
		{"лететь", "летет"},
		{"лети", "лет"},
		{"летим", "лет"},
		{"летит", "лет"},
		{"летите", "лет"},
		{"летишь", "лет"},
		{"летнего", "летн"},
		{"летнее", "летн"},
		{"летней", "летн"},
		{"летнем", "летн"},
		{"летнему", "летн"},
		{"летние", "летн"},
		{"летний", "летн"},
		{"летних", "летн"},
		{"летнюю", "летн"},
		{"летняя", "летн"},
		{"лето", "лет"},
		{"летом", "лет"},
		{"летописей", "летопис"},
		{"лету", "лет"},
		{"летун", "летун"},
		{"летучая", "летуч"},
		{"летучей", "летуч"},
		{"летучим", "летуч"},
		{"летучих", "летуч"},
		{"летя", "лет"},
		{"летят", "лет"},
		{"лечат", "лечат"},
		{"лечебницу", "лечебниц"},
		{"лечению", "лечен"},
		{"лечения", "лечен"},
		{"лечи", "леч"},
		{"лечил", "леч"},
		{"лечилась", "леч"},
		{"лечит", "леч"},
		{"лечить", "леч"},
		{"лечиться", "леч"},
		{"лечишься", "леч"},
		{"лечь", "леч"},
		{"лешие", "леш"},
		{"леший", "леш"},
		{"лешим", "леш"},
		{"лжет", "лжет"},
		{"лжете", "лжет"},
		{"лжешь", "лжеш"},
		{"лжи", "лжи"},
		{"лжива", "лжив"},
		{"лживой", "лжив"},
		{"лживою", "лжив"},
		{"ли", "ли"},
		{"либерал", "либера"},
		{"либерализма", "либерализм"},
		{"либералы", "либерал"},
		{"либеральной", "либеральн"},
		{"либеральный", "либеральн"},
		{"либеральным", "либеральн"},
		{"либо", "либ"},
		{"ливень", "ливен"},
		{"ливингстона", "ливингстон"},
		{"ливрее", "ливр"},
		{"ливреи", "ливр"},
		{"ливрейном", "ливрейн"},
		{"ливрея", "ливре"},
		{"лида", "лид"},
		{"лидией", "лид"},
		{"лидинька", "лидиньк"},
		{"лидию", "лид"},
		{"лидия", "лид"},
		{"лидочка", "лидочк"},
		{"лидочке", "лидочк"},
		{"лидочкой", "лидочк"},
		{"лидочку", "лидочк"},
		{"лизавета", "лизавет"},
		{"лизавете", "лизавет"},
		{"лизаветин", "лизаветин"},
		{"лизаветина", "лизаветин"},
		{"лизаветой", "лизавет"},
		{"лизавету", "лизавет"},
		{"лизаветы", "лизавет"},
		{"лизнет", "лизнет"},
		{"лик", "лик"},
		{"лике", "лик"},
		{"ликов", "лик"},
		{"ликованье", "ликован"},
		{"ликом", "лик"},
		{"ликует", "лик"},
		{"ликург", "ликург"},
		{"ликургами", "ликург"},
		{"лил", "лил"},
		{"лилась", "лил"},
		{"лилии", "лил"},
		{"лилий", "лил"},
		{"лились", "лил"},
		{"лиллас", "лиллас"},
		{"лилово", "лилов"},
		{"лиловое", "лилов"},
		{"лиловые", "лилов"},
		{"лиловыми", "лилов"},
		{"лился", "лил"},
		{"лимон", "лимон"},
		{"лимонничаешь", "лимоннича"},
		{"лимоном", "лимон"},
		{"линейке", "линейк"},
		{"линейки", "линейк"},
		{"линейку", "линейк"},
		{"линией", "лин"},
		{"линии", "лин"},
		{"линий", "лин"},
		{"линию", "лин"},
		{"линия", "лин"},
		{"линор", "линор"},
		{"линялая", "линял"},
		{"линять", "линя"},
		{"лион", "лион"},
		{"липкий", "липк"},
		{"липким", "липк"},
		{"липкой", "липк"},
		{"липли", "липл"},
		{"липовая", "липов"},
		{"липового", "липов"},
		{"липовые", "липов"},
		{"липовым", "липов"},
		{"липовых", "липов"},
		{"липпевехзель", "липпевехзел"},
		{"липпи", "липп"},
		{"липу", "лип"},
		{"липы", "лип"},
		{"лира", "лир"},
		{"лирическая", "лирическ"},
		{"лисица", "лисиц"},
		{"лиссабонского", "лиссабонск"},
		{"лиссабонское", "лиссабонск"},
		{"лист", "лист"},
		{"листа", "лист"},
		{"листами", "лист"},
		{"листе", "лист"},
		{"листиках", "листик"},
		{"листиков", "листик"},
		{"листка", "листк"},
		{"листках", "листк"},
		{"листке", "листк"},
		{"листки", "листк"},
		{"листов", "лист"},
		{"листок", "листок"},
		{"листом", "лист"},
		{"листочки", "листочк"},
		{"листу", "лист"},
		{"листы", "лист"},
		{"листьев", "лист"},
		{"листья", "лист"},
		{"листьях", "лист"},
		{"лись", "ли"},
		{"литавр", "литавр"},
		{"литейной", "литейн"},
		{"литейном", "литейн"},
		{"литератор", "литератор"},
		{"литераторов", "литератор"},
		{"литератором", "литератор"},
		{"литераторы", "литератор"},
		{"литература", "литератур"},
		{"литературе", "литератур"},
		{"литературно", "литературн"},
		{"литературном", "литературн"},
		{"литературную", "литературн"},
		{"литературный", "литературн"},
		{"литературой", "литератур"},
		{"литературу", "литератур"},
		{"литературы", "литератур"},
		{"лития", "лит"},
		{"лить", "лит"},
		{"лифчик", "лифчик"},
		{"лихо", "лих"},
		{"лихом", "лих"},
		{"лихоманка", "лихоманк"},
		{"лихорадка", "лихорадк"},
		{"лихорадке", "лихорадк"},
		{"лихорадки", "лихорадк"},
		{"лихорадку", "лихорадк"},
		{"лихорадочная", "лихорадочн"},
		{"лихорадочно", "лихорадочн"},
		{"лихорадочного", "лихорадочн"},
		{"лихорадочное", "лихорадочн"},
		{"лихорадочной", "лихорадочн"},
		{"лихорадочность", "лихорадочн"},
		{"лихорадочную", "лихорадочн"},
		{"лихорадочный", "лихорадочн"},
		{"лихорадочным", "лихорадочн"},
		{"лихорадочных", "лихорадочн"},
		{"лиц", "лиц"},
		{"лица", "лиц"},
		{"лицами", "лиц"},
		{"лицах", "лиц"},
		{"лице", "лиц"},
		{"лицезреть", "лицезрет"},
		{"лицемерный", "лицемерн"},
		{"лицо", "лиц"},
		{"лицом", "лиц"},
		{"лицу", "лиц"},
		{"личек", "личек"},
		{"личика", "личик"},
		{"личике", "личик"},
		{"личико", "личик"},
		{"личиком", "личик"},
		{"личиной", "личин"},
		{"личная", "личн"},
		{"лично", "личн"},
		{"личного", "личн"},
		{"личной", "личн"},
		{"личном", "личн"},
		{"личному", "личн"},
		{"личности", "личност"},
		{"личность", "личност"},
		{"личностью", "личност"},
		{"личную", "личн"},
		{"личные", "личн"},
		{"личный", "личн"},
		{"личным", "личн"},
		{"личных", "личн"},
		{"лишает", "лиша"},
		{"лишали", "лиша"},
		{"лишений", "лишен"},
		{"лишениями", "лишен"},
		{"лишенное", "лишен"},
		{"лишенной", "лишен"},
		{"лишенною", "лишен"},
		{"лишенные", "лишен"},
		{"лишено", "лиш"},
		{"лишившийся", "лиш"},
		{"лишил", "лиш"},
		{"лишились", "лиш"},
		{"лишился", "лиш"},
		{"лишить", "лиш"},
		{"лишком", "лишк"},
		{"лишнего", "лишн"},
		{"лишнее", "лишн"},
		{"лишней", "лишн"},
		{"лишние", "лишн"},
		{"лишний", "лишн"},
		{"лишним", "лишн"},
		{"лишних", "лишн"},
		{"лишняя", "лишн"},
		{"лишу", "лиш"},
		{"лишь", "лиш"},
		{"ло", "ло"},
		{"лоб", "лоб"},
		{"лобзаний", "лобзан"},
		{"лобызает", "лобыза"},
		{"ловелас", "ловелас"},
		{"лови", "лов"},
		{"ловил", "лов"},
		{"ловит", "лов"},
		{"ловить", "лов"},
		{"ловкая", "ловк"},
		{"ловкий", "ловк"},
		{"ловко", "ловк"},
		{"ловкость", "ловкост"},
		{"ловкую", "ловк"},
		{"ловлю", "ловл"},
		{"ловушка", "ловушк"},
		{"ловушкой", "ловушк"},
		{"ловушку", "ловушк"},
		{"ловя", "лов"},
		{"ловят", "лов"},
		{"лога", "лог"},
		{"логика", "логик"},
		{"логики", "логик"},
		{"логикой", "логик"},
		{"логиныч", "логиныч"},
		{"логическая", "логическ"},
		{"логически", "логическ"},
		{"логическим", "логическ"},
		{"логической", "логическ"},
		{"логично", "логичн"},
		{"логичны", "логичн"},
		{"лодка", "лодк"},
		{"лодке", "лодк"},
		{"лодки", "лодк"},
		{"лодку", "лодк"},
		{"лодочник", "лодочник"},
		{"лодочника", "лодочник"},
		{"лодочнику", "лодочник"},
		{"ложа", "лож"},
		{"ложась", "лож"},
		{"ложатся", "ложат"},
		{"ложе", "лож"},
		{"ложек", "ложек"},
		{"ложем", "лож"},
		{"ложементов", "ложемент"},
		{"ложечка", "ложечк"},
		{"ложечки", "ложечк"},
		{"ложечкой", "ложечк"},
		{"ложечку", "ложечк"},
		{"ложи", "лож"},
		{"ложилась", "лож"},
		{"ложились", "лож"},
		{"ложилось", "лож"},
		{"ложился", "лож"},
		{"ложись", "лож"},
		{"ложитесь", "лож"},
		{"ложится", "лож"},
		{"ложиться", "лож"},
		{"ложка", "ложк"},
		{"ложки", "ложк"},
		{"ложку", "ложк"},
		{"ложного", "ложн"},
		{"ложное", "ложн"},
		{"ложные", "ложн"},
		{"ложу", "лож"},
		{"ложь", "лож"},
		{"ложью", "лож"},
		{"лоз", "лоз"},
		{"лозунг", "лозунг"},
		{"лозунгом", "лозунг"},
		{"локон", "локон"},
		{"локончиками", "локончик"},
		{"локончики", "локончик"},
		{"локоть", "локот"},
		{"локтей", "локт"},
		{"локтем", "локт"},
		{"локти", "локт"},
		{"локтя", "локт"},
		{"локтями", "локт"},
		{"локтях", "локт"},
		{"лом", "лом"},
		{"ломает", "лома"},
		{"ломаешь", "лома"},
		{"ломайтесь", "лома"},
		{"ломал", "лома"},
		{"ломала", "лома"},
		{"ломаль", "ломал"},
		{"ломаной", "ломан"},
		{"ломать", "лома"},
		{"ломаться", "лома"},
		{"ломаю", "лома"},
		{"ломают", "лома"},
		{"ломаются", "лома"},
		{"ломая", "лом"},
		{"ломбард", "ломбард"},
		{"ломберном", "ломберн"},
		{"ломберный", "ломберн"},
		{"ломберными", "ломберн"},
		{"ломились", "лом"},
		{"ломится", "лом"},
		{"ломка", "ломк"},
		{"ломки", "ломк"},
		{"ломовою", "ломов"},
		{"ломовых", "ломов"},
		{"ломом", "лом"},
		{"ломоть", "ломот"},
		{"ломтем", "ломт"},
		{"ломтю", "ломт"},
		{"ломят", "лом"},
		{"лона", "лон"},
		{"лондон", "лондон"},
		{"лондона", "лондон"},
		{"лондоне", "лондон"},
		{"лоне", "лон"},
		{"лоно", "лон"},
		{"лопастей", "лопаст"},
		{"лопасти", "лопаст"},
		{"лопатками", "лопатк"},
		{"лопатой", "лопат"},
		{"лопахин", "лопахин"},
		{"лопахина", "лопахин"},
		{"лопахину", "лопахин"},
		{"лопахиным", "лопахин"},
		{"лопнет", "лопнет"},
		{"лопнувшей", "лопнувш"},
		{"лопнула", "лопнул"},
		{"лопнуло", "лопнул"},
		{"лорд", "лорд"},
		{"лордом", "лорд"},
		{"лорнет", "лорнет"},
		{"лорнеткой", "лорнетк"},
		{"лорнетку", "лорнетк"},
		{"лорнировала", "лорнирова"},
		{"лоси", "лос"},
		{"лоск", "лоск"},
		{"лоскутка", "лоскутк"},
		{"лоскутки", "лоскутк"},
		{"лоскутков", "лоскутк"},
		{"лоскутное", "лоскутн"},
		{"лоскуток", "лоскуток"},
		{"лоскуты", "лоскут"},
		{"лоскутья", "лоскут"},
		{"лоснилась", "лосн"},
		{"лоснятся", "лосн"},
		{"лоснящиеся", "лосня"},
		{"лососина", "лососин"},
		{"лососиной", "лососин"},
		{"лота", "лот"},
		{"лотереей", "лотере"},
		{"лотереи", "лотер"},
		{"лотерею", "лотер"},
		{"лотках", "лотк"},
		{"лото", "лот"},
		{"лохматый", "лохмат"},
		{"лохмотник", "лохмотник"},
		{"лохмотников", "лохмотник"},
		{"лохмотьев", "лохмот"},
		{"лохмотья", "лохмот"},
		{"лохмотьям", "лохмот"},
		{"лохмотьях", "лохмот"},
		{"лошадей", "лошад"},
		{"лошаденка", "лошаденк"},
		{"лошаденке", "лошаденк"},
		{"лошаденку", "лошаденк"},
		{"лошади", "лошад"},
		{"лошадиное", "лошадин"},
		{"лошадке", "лошадк"},
		{"лошадки", "лошадк"},
		{"лошадку", "лошадк"},
		{"лошадь", "лошад"},
		{"лошадьми", "лошадьм"},
		{"лошадью", "лошад"},
		{"лошадям", "лошад"},
		{"лошадях", "лошад"},
		{"луг", "луг"},
		{"луга", "луг"},
		{"лугов", "луг"},
		{"лугом", "луг"},
		{"лугу", "луг"},
		{"лужа", "луж"},
		{"луже", "луж"},
		{"лужи", "луж"},
		{"лужин", "лужин"},
		{"лужина", "лужин"},
		{"лужине", "лужин"},
		{"лужинская", "лужинск"},
		{"лужину", "лужин"},
		{"лужиным", "лужин"},
		{"лужица", "лужиц"},
		{"лужице", "лужиц"},
		{"лужу", "луж"},
		{"луиза", "луиз"},
		{"лук", "лук"},
		{"лука", "лук"},
		{"лукав", "лука"},
		{"лукавая", "лукав"},
		{"лукавил", "лукав"},
		{"лукавит", "лукав"},
		{"лукавить", "лукав"},
		{"лукавлю", "лукавл"},
		{"лукаво", "лукав"},
		{"лукавого", "лукав"},
		{"лукавое", "лукав"},
		{"лукавой", "лукав"},
		{"лукавою", "лукав"},
		{"лукавства", "лукавств"},
		{"лукавство", "лукавств"},
		{"лукавством", "лукавств"},
		{"лукавые", "лукав"},
		{"лукавый", "лукав"},
		{"лукавым", "лукав"},
		{"лукавыми", "лукав"},
		{"лукавых", "лукав"},
		{"лукавь", "лукав"},
		{"луковицу", "луковиц"},
		{"луком", "лук"},
		{"лукоморья", "лукомор"},
		{"луку", "лук"},
		{"луна", "лун"},
		{"лунатизм", "лунатизм"},
		{"луне", "лун"},
		{"лунная", "лун"},
		{"лунного", "лун"},
		{"лунной", "лун"},
		{"лунном", "лун"},
		{"лунному", "лун"},
		{"лунные", "лун"},
		{"лунный", "лун"},
		{"лунным", "лун"},
		{"луной", "лун"},
		{"луну", "лун"},
		{"луны", "лун"},
		{"лупить", "луп"},
		{"луч", "луч"},
		{"луча", "луч"},
		{"лучам", "луч"},
		{"лучами", "луч"},
		{"лучах", "луч"},
		{"лучезарнее", "лучезарн"},
		{"лучей", "луч"},
		{"лучи", "луч"},
		{"лучинку", "лучинк"},
		{"лучину", "лучин"},
		{"лучины", "лучин"},
		{"лучистые", "лучист"},
		{"лучистыми", "лучист"},
		{"лучом", "луч"},
		{"лучшая", "лучш"},
		{"лучше", "лучш"},
		{"лучшего", "лучш"},
		{"лучшее", "лучш"},
		{"лучшей", "лучш"},
		{"лучшему", "лучш"},
		{"лучшие", "лучш"},
		{"лучший", "лучш"},
		{"лучшим", "лучш"},
		{"лучших", "лучш"},
		{"лучшую", "лучш"},
		{"лущить", "лущ"},
		{"лчать", "лчат"},
		{"лый", "лы"},
		{"лысая", "лыс"},
		{"лысина", "лысин"},
		{"лысине", "лысин"},
		{"лысиной", "лысин"},
		{"лысину", "лысин"},
		{"лысый", "лыс"},
		{"лысым", "лыс"},
		{"ль", "ль"},
		{"львиного", "львин"},
		{"львиное", "львин"},
		{"львице", "львиц"},
		{"львицей", "львиц"},
		{"львов", "львов"},
		{"львович", "львович"},
		{"львовна", "львовн"},
		{"льву", "льву"},
		{"львы", "львы"},
		{"льда", "льда"},
		{"льдины", "льдин"},
		{"льдом", "льдом"},
		{"льет", "льет"},
		{"льется", "льет"},
		{"льзя", "льзя"},
		{"льнет", "льнет"},
		{"льном", "льном"},
		{"льстецам", "льстец"},
		{"льстиво", "льстив"},
		{"льстивое", "льстив"},
		{"льстивый", "льстив"},
		{"льстил", "льстил"},
		{"льшая", "льша"},
		{"льшим", "льшим"},
		{"льших", "льших"},
		{"льшую", "льшу"},
		{"льщу", "льщу"},
		{"льюиса", "льюис"},
		{"льют", "льют"},
		{"лю", "лю"},
		{"люба", "люб"},
		{"любви", "любв"},
		{"любезен", "любез"},
		{"любезна", "любезн"},
		{"любезнее", "любезн"},
		{"любезнейшая", "любезн"},
		{"любезнейший", "любезн"},
		{"любезно", "любезн"},
		{"любезной", "любезн"},
		{"любезностей", "любезн"},
		{"любезность", "любезн"},
		{"любезностью", "любезн"},
		{"любезностями", "любезн"},
		{"любезны", "любезн"},
		{"любезные", "любезн"},
		{"любезный", "любезн"},
		{"любезным", "любезн"},
		{"любезных", "любезн"},
		{"любека", "любек"},
		{"люби", "люб"},
		{"любивший", "люб"},
		{"любил", "люб"},
		{"любила", "люб"},
		{"любили", "люб"},
		{"любило", "люб"},
		{"любим", "люб"},
		{"любимая", "любим"},
		{"любимец", "любимец"},
		{"любимице", "любимиц"},
		{"любимо", "любим"},
		{"любимого", "любим"},
		{"любимое", "любим"},
		{"любимой", "любим"},
		{"любимую", "любим"},
		{"любимца", "любимц"},
		{"любимцу", "любимц"},
		{"любимые", "любим"},
		{"любимый", "любим"},
		{"любимым", "любим"},
		{"любимых", "любим"},
		{"любит", "люб"},
		{"любите", "люб"},
		{"любитель", "любител"},
		{"любителя", "любител"},
		{"любить", "люб"},
		{"любишь", "люб"},
		{"люблю", "любл"},
		{"любо", "люб"},
		{"любовалась", "любова"},
		{"любовались", "любова"},
		{"любовался", "любова"},
		{"любоваться", "любова"},
		{"любови", "любов"},
		{"любовник", "любовник"},
		{"любовника", "любовник"},
		{"любовники", "любовник"},
		{"любовником", "любовник"},
		{"любовница", "любовниц"},
		{"любовницей", "любовниц"},
		{"любовницы", "любовниц"},
		{"любовно", "любовн"},
		{"любовные", "любовн"},
		{"любовных", "любовн"},
		{"любовь", "любов"},
		{"любовью", "любов"},
		{"любого", "люб"},
		{"любое", "люб"},
		{"любом", "люб"},
		{"любопытен", "любопыт"},
		{"любопытная", "любопытн"},
		{"любопытнейшими", "любопытн"},
		{"любопытно", "любопытн"},
		{"любопытного", "любопытн"},
		{"любопытной", "любопытн"},
		{"любопытную", "любопытн"},
		{"любопытные", "любопытн"},
		{"любопытный", "любопытн"},
		{"любопытным", "любопытн"},
		{"любопытными", "любопытн"},
		{"любопытных", "любопытн"},
		{"любопытства", "любопытств"},
		{"любопытство", "любопытств"},
		{"любопытством", "любопытств"},
		{"любопытствую", "любопытств"},
		{"любуется", "люб"},
		{"любуйся", "люб"},
		{"любуясь", "любу"},
		{"любя", "люб"},
		{"любят", "люб"},
		{"любящего", "любя"},
		{"любящее", "любя"},
		{"любящей", "любя"},
		{"любящие", "любя"},
		{"любящим", "любя"},
		{"любящимися", "любя"},
		{"любящихся", "любя"},
		{"люд", "люд"},
		{"люда", "люд"},
		{"людвиговна", "людвиговн"},
		{"людвиговной", "людвиговн"},
		{"людвиговну", "людвиговн"},
		{"людей", "люд"},
		{"люди", "люд"},
		{"людовике", "людовик"},
		{"людом", "люд"},
		{"людская", "людск"},
		{"людски", "людск"},
		{"людские", "людск"},
		{"людских", "людск"},
		{"людской", "людск"},
		{"людскому", "людск"},
		{"людскую", "людск"},
		{"людьми", "людьм"},
		{"людям", "люд"},
		{"людях", "люд"},
		{"люли", "люл"},
		{"люстра", "люстр"},
		{"лютни", "лютн"},
		{"лютой", "лют"},
		{"лютые", "лют"},
		{"люциферова", "люциферов"},
		{"ля", "ля"},
		{"ляг", "ляг"},
		{"лягаев", "ляга"},
		{"лягается", "ляга"},
		{"лягаться", "ляга"},
		{"лягнул", "лягнул"},
		{"лягу", "ляг"},
		{"лягут", "лягут"},
		{"лягушек", "лягушек"},
		{"лядащая", "ляда"},
		{"ляжет", "ляжет"},
		{"ляжете", "ляжет"},
		{"ляжешь", "ляжеш"},
		{"лязбиля", "лязбил"},
		{"лязг", "лязг"},
		{"лязгают", "лязга"},
		{"ляп", "ляп"},
		{"м", "м"},
		{"ма", "ма"},
		{"мавзолей", "мавзол"},
		{"мага", "маг"},
		{"магазин", "магазин"},
		{"магазина", "магазин"},
		{"магазинам", "магазин"},
		{"магазине", "магазин"},
		{"магазинов", "магазин"},
		{"магазины", "магазин"},
		{"магарыч", "магарыч"},
		{"магдалина", "магдалин"},
		{"магистром", "магистр"},
		{"магического", "магическ"},
		{"магнатские", "магнатск"},
		{"магнетизер", "магнетизер"},
		{"магомет", "магомет"},
		{"магометами", "магомет"},
		{"магометом", "магомет"},
		{"мадам", "мад"},
		{"мадемуазель", "мадемуазел"},
		{"мадера", "мадер"},
		{"мадеры", "мадер"},
		{"мадонн", "мадон"},
		{"мадонне", "мадон"},
		{"мадонну", "мадон"},
		{"мадонны", "мадон"},
		{"мае", "ма"},
		{"маевские", "маевск"},
		{"мается", "мает"},
		{"маешься", "маеш"},
		{"мажет", "мажет"},
		{"мажордом", "мажорд"},
		{"мажордомы", "мажордом"},
		{"мажу", "маж"},
		{"мазали", "маза"},
		{"мази", "маз"},
		{"мазилкой", "мазилк"},
		{"мазнул", "мазнул"},
		{"мазурика", "мазурик"},
		{"мазурка", "мазурк"},
		{"мазурке", "мазурк"},
		{"мазурки", "мазурк"},
		{"мазурку", "мазурк"},
		{"мазью", "маз"},
		{"май", "ма"},
		{"майков", "майк"},
		{"майор", "майор"},
		{"майора", "майор"},
		{"майором", "майор"},
		{"майская", "майск"},
		{"майские", "майск"},
		{"майскими", "майск"},
		{"майских", "майск"},
		{"мак", "мак"},
		{"макароны", "макарон"},
		{"маклашиных", "маклашин"},
		{"маклера", "маклер"},
		{"маковой", "маков"},
		{"маковый", "маков"},
		{"маку", "мак"},
		{"макушка", "макушк"},
		{"макушке", "макушк"},
		{"мал", "мал"},
		{"мала", "мал"},
		{"маланья", "малан"},
		{"малейшая", "мал"},
		{"малейшего", "мал"},
		{"малейшей", "мал"},
		{"малейшем", "мал"},
		{"малейшему", "мал"},
		{"малейший", "мал"},
		{"малейших", "мал"},
		{"малейшую", "мал"},
		{"маленькая", "маленьк"},
		{"маленькие", "маленьк"},
		{"маленький", "маленьк"},
		{"маленьким", "маленьк"},
		{"маленькими", "маленьк"},
		{"маленьких", "маленьк"},
		{"маленько", "маленьк"},
		{"маленького", "маленьк"},
		{"маленькое", "маленьк"},
		{"маленькой", "маленьк"},
		{"маленьком", "маленьк"},
		{"маленькому", "маленьк"},
		{"маленькою", "маленьк"},
		{"маленькую", "маленьк"},
		{"малина", "малин"},
		{"малинки", "малинк"},
		{"малиновая", "малинов"},
		{"малинового", "малинов"},
		{"малиной", "малин"},
		{"малину", "малин"},
		{"малины", "малин"},
		{"малицкое", "малицк"},
		{"мало", "мал"},
		{"маловажное", "маловажн"},
		{"маловажной", "маловажн"},
		{"малого", "мал"},
		{"малограмотный", "малограмотн"},
		{"малодейственных", "малодействен"},
		{"малодушие", "малодуш"},
		{"малодушии", "малодуш"},
		{"малодушия", "малодуш"},
		{"малодушного", "малодушн"},
		{"малодушной", "малодушн"},
		{"малое", "мал"},
		{"малой", "мал"},
		{"малолетним", "малолетн"},
		{"малолетними", "малолетн"},
		{"малолетными", "малолетн"},
		{"малом", "мал"},
		{"малому", "мал"},
		{"малоподвижная", "малоподвижн"},
		{"малостоящее", "малостоя"},
		{"малость", "малост"},
		{"малую", "мал"},
		{"малые", "мал"},
		{"малый", "мал"},
		{"малым", "мал"},
		{"малыми", "мал"},
		{"малых", "мал"},
		{"мальски", "мальск"},
		{"мальчик", "мальчик"},
		{"мальчика", "мальчик"},
		{"мальчике", "мальчик"},
		{"мальчики", "мальчик"},
		{"мальчиком", "мальчик"},
		{"мальчику", "мальчик"},
		{"мальчишек", "мальчишек"},
		{"мальчишески", "мальчишеск"},
		{"мальчишескою", "мальчишеск"},
		{"мальчишечка", "мальчишечк"},
		{"мальчишка", "мальчишк"},
		{"мальчишкам", "мальчишк"},
		{"мальчишками", "мальчишк"},
		{"мальчишке", "мальчишк"},
		{"мальчишки", "мальчишк"},
		{"мальчишков", "мальчишк"},
		{"мальчишкой", "мальчишк"},
		{"мальчишку", "мальчишк"},
		{"мальчонком", "мальчонк"},
		{"мальчугана", "мальчуга"},
		{"малютки", "малютк"},
		{"малюток", "малюток"},
		{"малявочка", "малявочк"},
		{"малявочкой", "малявочк"},
		{"маляре", "маляр"},
		{"маляры", "маляр"},
		{"мама", "мам"},
		{"мамай", "мама"},
		{"мамасю", "мамас"},
		{"мамася", "мам"},
		{"мамаша", "мамаш"},
		{"мамаше", "мамаш"},
		{"мамашей", "мамаш"},
		{"мамаши", "мамаш"},
		{"мамашину", "мамашин"},
		{"мамашу", "мамаш"},
		{"маме", "мам"},
		{"маменька", "маменьк"},
		{"маменьки", "маменьк"},
		{"маменькой", "маменьк"},
		{"маменьку", "маменьк"},
		{"мамками", "мамк"},
		{"мамочка", "мамочк"},
		{"маму", "мам"},
		{"мамы", "мам"},
		{"ман", "ман"},
		{"манго", "манг"},
		{"мандолина", "мандолин"},
		{"мандолине", "мандолин"},
		{"манер", "манер"},
		{"манера", "манер"},
		{"манерами", "манер"},
		{"манерах", "манер"},
		{"манерой", "манер"},
		{"манером", "манер"},
		{"манеру", "манер"},
		{"манеры", "манер"},
		{"манжет", "манжет"},
		{"манжетах", "манжет"},
		{"мани", "ман"},
		{"манил", "ман"},
		{"манила", "ман"},
		{"манило", "ман"},
		{"манир", "манир"},
		{"манит", "ман"},
		{"манифест", "манифест"},
		{"манишка", "манишк"},
		{"манишки", "манишк"},
		{"манишку", "манишк"},
		{"манкировал", "манкирова"},
		{"манкировали", "манкирова"},
		{"манкировать", "манкирова"},
		{"манкируете", "манкирует"},
		{"мансардах", "мансард"},
		{"мансарде", "мансард"},
		{"мантилий", "мантил"},
		{"мантильи", "мантил"},
		{"мантильке", "мантильк"},
		{"мантильку", "мантильк"},
		{"мантилью", "мантил"},
		{"манто", "мант"},
		{"манфредами", "манфред"},
		{"манящие", "маня"},
		{"маргарита", "маргарит"},
		{"марево", "марев"},
		{"марии", "мар"},
		{"марина", "марин"},
		{"марине", "марин"},
		{"мариновали", "маринова"},
		{"марину", "марин"},
		{"марины", "марин"},
		{"марионетками", "марионетк"},
		{"марию", "мар"},
		{"мария", "мар"},
		{"марк", "марк"},
		{"маркизы", "маркиз"},
		{"мармелад", "мармелад"},
		{"мармеладов", "мармелад"},
		{"мармеладова", "мармеладов"},
		{"мармеладову", "мармеладов"},
		{"мармеладовым", "мармеладов"},
		{"мармеладовых", "мармеладов"},
		{"марсово", "марсов"},
		{"март", "март"},
		{"марта", "март"},
		{"марте", "март"},
		{"марфа", "марф"},
		{"марфе", "марф"},
		{"марфой", "марф"},
		{"марфу", "марф"},
		{"марфы", "марф"},
		{"марш", "марш"},
		{"марширующих", "маршир"},
		{"марье", "мар"},
		{"марьей", "мар"},
		{"марьи", "мар"},
		{"марью", "мар"},
		{"марья", "мар"},
		{"марьяж", "марьяж"},
		{"маска", "маск"},
		{"маскарад", "маскарад"},
		{"маскарада", "маскарад"},
		{"маски", "маск"},
		{"маской", "маск"},
		{"маску", "маск"},
		{"масла", "масл"},
		{"масленица", "маслениц"},
		{"масленице", "маслениц"},
		{"масленицы", "маслениц"},
		{"маслеными", "маслен"},
		{"масло", "масл"},
		{"маслом", "масл"},
		{"маслу", "масл"},
		{"масляною", "маслян"},
		{"масок", "масок"},
		{"масса", "масс"},
		{"массивная", "массивн"},
		{"массивный", "массивн"},
		{"массивных", "массивн"},
		{"массимо", "массим"},
		{"массис", "массис"},
		{"массою", "масс"},
		{"массу", "масс"},
		{"массы", "масс"},
		{"мастей", "маст"},
		{"мастер", "мастер"},
		{"мастерица", "мастериц"},
		{"мастеровому", "мастеров"},
		{"мастеровым", "мастеров"},
		{"мастеровых", "мастеров"},
		{"мастером", "мастер"},
		{"мастерская", "мастерск"},
		{"мастерски", "мастерск"},
		{"мастерских", "мастерск"},
		{"мастерское", "мастерск"},
		{"мастерской", "мастерск"},
		{"мастеру", "мастер"},
		{"масштабе", "масштаб"},
		{"матвевна", "матвевн"},
		{"матвеевич", "матвеевич"},
		{"матвеевича", "матвеевич"},
		{"матвеевичем", "матвеевич"},
		{"матвеевичу", "матвеевич"},
		{"матвеевна", "матвеевн"},
		{"матвеевне", "матвеевн"},
		{"матвеевной", "матвеевн"},
		{"матвеевну", "матвеевн"},
		{"матвеевны", "матвеевн"},
		{"матвеем", "матве"},
		{"матвеич", "матвеич"},
		{"матвеича", "матвеич"},
		{"матвеичем", "матвеич"},
		{"матвей", "матв"},
		{"математики", "математик"},
		{"математику", "математик"},
		{"математически", "математическ"},
		{"математический", "математическ"},
		{"математических", "математическ"},
		{"математической", "математическ"},
		{"математическую", "математическ"},
		{"матерей", "матер"},
		{"матери", "матер"},
		{"материал", "материа"},
		{"материала", "материа"},
		{"материалов", "материал"},
		{"материалом", "материал"},
		{"материалу", "материал"},
		{"материалы", "материал"},
		{"материальной", "материальн"},
		{"материальную", "материальн"},
		{"материальные", "материальн"},
		{"материальный", "материальн"},
		{"материальных", "материальн"},
		{"материею", "матер"},
		{"материи", "матер"},
		{"материй", "матер"},
		{"материнским", "материнск"},
		{"материнское", "материнск"},
		{"материю", "матер"},
		{"материя", "матер"},
		{"матерчатое", "матерчат"},
		{"матерь", "матер"},
		{"матерью", "матер"},
		{"матерьялистских", "матерьялистск"},
		{"матовому", "матов"},
		{"матовый", "матов"},
		{"матовых", "матов"},
		{"матом", "мат"},
		{"матрена", "матр"},
		{"матрос", "матрос"},
		{"матросы", "матрос"},
		{"матушка", "матушк"},
		{"матушки", "матушк"},
		{"матушку", "матушк"},
		{"матчиш", "матчиш"},
		{"мать", "мат"},
		{"махал", "маха"},
		{"махать", "маха"},
		{"махая", "мах"},
		{"махнешь", "махнеш"},
		{"махнув", "махнув"},
		{"махнул", "махнул"},
		{"махнула", "махнул"},
		{"махов", "мах"},
		{"махочкие", "махочк"},
		{"махочкую", "махочк"},
		{"маху", "мах"},
		{"мачеха", "мачех"},
		{"мачехе", "мачех"},
		{"мачехи", "мачех"},
		{"мачехой", "мачех"},
		{"мачтам", "мачт"},
		{"мачтах", "мачт"},
		{"маша", "маш"},
		{"маше", "маш"},
		{"машей", "маш"},
		{"машенька", "машеньк"},
		{"машеньки", "машеньк"},
		{"машенькой", "машеньк"},
		{"машет", "машет"},
		{"маши", "маш"},
		{"машин", "машин"},
		{"машина", "машин"},
		{"машинально", "машинальн"},
		{"машине", "машин"},
		{"машинистом", "машинист"},
		{"машинку", "машинк"},
		{"машину", "машин"},
		{"машины", "машин"},
		{"машка", "машк"},
		{"машу", "маш"},
		{"мая", "ма"},
		{"маяк", "маяк"},
		{"маяки", "маяк"},
		{"маятник", "маятник"},
		{"маятника", "маятник"},
		{"маятником", "маятник"},
		{"маяться", "мая"},
		{"маячишь", "маяч"},
		{"мгла", "мгла"},
		{"мгле", "мгле"},
		{"мглист", "мглист"},
		{"мглистой", "мглист"},
		{"мглистом", "мглист"},
		{"мглистый", "мглист"},
		{"мглой", "мгло"},
		{"мглою", "мгло"},
		{"мглу", "мглу"},
		{"мглы", "мглы"},
		{"мгновение", "мгновен"},
		{"мгновением", "мгновен"},
		{"мгновений", "мгновен"},
		{"мгновения", "мгновен"},
		{"мгновениями", "мгновен"},
		{"мгновенно", "мгновен"},
		{"мгновенного", "мгновен"},
		{"мгновенное", "мгновен"},
		{"мгновенной", "мгновен"},
		{"мгновенный", "мгновен"},
		{"мгновенным", "мгновен"},
		{"мгновенье", "мгновен"},
		{"ме", "ме"},
		{"мебели", "мебел"},
		{"мебель", "мебел"},
		{"мебельном", "мебельн"},
		{"мебельщику", "мебельщик"},
		{"меблированная", "меблирова"},
		{"меблированные", "меблирова"},
		{"меда", "мед"},
		{"медали", "меда"},
		{"медаль", "медал"},
		{"медальон", "медальон"},
		{"медальоне", "медальон"},
		{"медалях", "медал"},
		{"медведе", "медвед"},
		{"медведей", "медвед"},
		{"медведенка", "медведенк"},
		{"медведенко", "медведенк"},
		{"медведенком", "медведенк"},
		{"медведенку", "медведенк"},
		{"медведь", "медвед"},
		{"медведя", "медвед"},
		{"медвежью", "медвеж"},
		{"меди", "мед"},
		{"медикаменты", "медикамент"},
		{"медики", "медик"},
		{"медиком", "медик"},
		{"медицине", "медицин"},
		{"медицинский", "медицинск"},
		{"медицинское", "медицинск"},
		{"медицину", "медицин"},
		{"медицины", "медицин"},
		{"медленная", "медлен"},
		{"медленнее", "медлен"},
		{"медленно", "медлен"},
		{"медленного", "медлен"},
		{"медленное", "медлен"},
		{"медленною", "медлен"},
		{"медленный", "медлен"},
		{"медленным", "медлен"},
		{"медленных", "медлен"},
		{"медли", "медл"},
		{"медлил", "медл"},
		{"медлила", "медл"},
		{"медлит", "медл"},
		{"медлительно", "медлительн"},
		{"медлительностью", "медлительн"},
		{"медлю", "медл"},
		{"медля", "медл"},
		{"медно", "медн"},
		{"медного", "медн"},
		{"медной", "медн"},
		{"медном", "медн"},
		{"медную", "медн"},
		{"медные", "медн"},
		{"медный", "медн"},
		{"медными", "медн"},
		{"медных", "медн"},
		{"медовые", "медов"},
		{"медовых", "медов"},
		{"меду", "мед"},
		{"меды", "мед"},
		{"медь", "мед"},
		{"медью", "мед"},
		{"меж", "меж"},
		{"между", "межд"},
		{"межу", "меж"},
		{"мездровых", "мездров"},
		{"мейербер", "мейербер"},
		{"мел", "мел"},
		{"меланхолик", "меланхолик"},
		{"меланхолики", "меланхолик"},
		{"меланхолически", "меланхолическ"},
		{"меланхолический", "меланхолическ"},
		{"меланхолической", "меланхолическ"},
		{"меланхоличный", "меланхоличн"},
		{"меланхолия", "меланхол"},
		{"меленьким", "меленьк"},
		{"мелет", "мелет"},
		{"мелете", "мелет"},
		{"мелеют", "мелеют"},
		{"мели", "мел"},
		{"мелка", "мелк"},
		{"мелкая", "мелк"},
		{"мелкие", "мелк"},
		{"мелкий", "мелк"},
		{"мелким", "мелк"},
		{"мелкими", "мелк"},
		{"мелких", "мелк"},
		{"мелко", "мелк"},
		{"мелкого", "мелк"},
		{"мелкое", "мелк"},
		{"мелкой", "мелк"},
		{"мелкоозабоченные", "мелкоозабочен"},
		{"мелкопоместный", "мелкопоместн"},
		{"мелкою", "мелк"},
		{"мелодией", "мелод"},
		{"мелодраме", "мелодрам"},
		{"мелом", "мел"},
		{"мелочам", "мелоч"},
		{"мелочах", "мелоч"},
		{"мелочей", "мелоч"},
		{"мелочи", "мелоч"},
		{"мелочной", "мелочн"},
		{"мелочною", "мелочн"},
		{"мелочную", "мелочн"},
		{"мелочных", "мелочн"},
		{"мелочь", "мелоч"},
		{"мелочью", "мелоч"},
		{"мелькавшие", "мелька"},
		{"мелькавшим", "мелька"},
		{"мелькавших", "мелька"},
		{"мелькает", "мелька"},
		{"мелькал", "мелька"},
		{"мелькала", "мелька"},
		{"мелькалась", "мелька"},
		{"мелькали", "мелька"},
		{"мелькало", "мелька"},
		{"мелькают", "мелька"},
		{"мелькающем", "мелька"},
		{"мелькающие", "мелька"},
		{"мелькнет", "мелькнет"},
		{"мелькнувшей", "мелькнувш"},
		{"мелькнувшем", "мелькнувш"},
		{"мелькнувший", "мелькнувш"},
		{"мелькнул", "мелькнул"},
		{"мелькнула", "мелькнул"},
		{"мелькнуло", "мелькнул"},
		{"мелькнуть", "мелькнут"},
		{"мельком", "мельк"},
		{"мельник", "мельник"},
		{"мельниц", "мельниц"},
		{"мельнице", "мельниц"},
		{"мельницей", "мельниц"},
		{"мельницы", "мельниц"},
		{"мельничного", "мельничн"},
		{"мельчайшей", "мельчайш"},
		{"мельчайшие", "мельчайш"},
		{"мельчайших", "мельчайш"},
		{"мельчал", "мельча"},
		{"мелю", "мел"},
		{"мена", "мен"},
		{"мене", "мен"},
		{"менее", "мен"},
		{"ментоны", "ментон"},
		{"меньшая", "меньш"},
		{"меньше", "меньш"},
		{"меньшее", "меньш"},
		{"меньшей", "меньш"},
		{"меньшие", "меньш"},
		{"меньшинство", "меньшинств"},
		{"меньшого", "меньш"},
		{"меньшой", "меньш"},
		{"меньшую", "меньш"},
		{"меня", "мен"},
		{"меняет", "меня"},
		{"меняетесь", "меня"},
		{"меняется", "меня"},
		{"меняешь", "меня"},
		{"менял", "меня"},
		{"менялась", "меня"},
		{"менялись", "меня"},
		{"менялы", "менял"},
		{"менять", "меня"},
		{"меняясь", "мен"},
		{"мер", "мер"},
		{"мера", "мер"},
		{"мерам", "мер"},
		{"мере", "мер"},
		{"мерещилась", "мерещ"},
		{"мерещились", "мерещ"},
		{"мерещилось", "мерещ"},
		{"мерещился", "мерещ"},
		{"мерещится", "мерещ"},
		{"мерещиться", "мерещ"},
		{"мерещутся", "мерещут"},
		{"мерзавец", "мерзавец"},
		{"мерзавцы", "мерзавц"},
		{"мерзкий", "мерзк"},
		{"мерзко", "мерзк"},
		{"мерзкой", "мерзк"},
		{"мерзкую", "мерзк"},
		{"мерзлого", "мерзл"},
		{"мерзнуть", "мерзнут"},
		{"мерзок", "мерзок"},
		{"мерзости", "мерзост"},
		{"мерзость", "мерзост"},
		{"мерке", "мерк"},
		{"мерки", "мерк"},
		{"меркнет", "меркнет"},
		{"мерку", "мерк"},
		{"мерлехлюндии", "мерлехлюнд"},
		{"мерно", "мерн"},
		{"мерное", "мерн"},
		{"мерные", "мерн"},
		{"мерный", "мерн"},
		{"мерным", "мерн"},
		{"мерой", "мер"},
		{"мертв", "мертв"},
		{"мертва", "мертв"},
		{"мертвая", "мертв"},
		{"мертвее", "мертв"},
		{"мертвец", "мертвец"},
		{"мертвеца", "мертвец"},
		{"мертвецам", "мертвец"},
		{"мертвецами", "мертвец"},
		{"мертвецах", "мертвец"},
		{"мертвецу", "мертвец"},
		{"мертвецы", "мертвец"},
		{"мертвечинкой", "мертвечинк"},
		{"мертво", "мертв"},
		{"мертвого", "мертв"},
		{"мертвое", "мертв"},
		{"мертвой", "мертв"},
		{"мертвому", "мертв"},
		{"мертвую", "мертв"},
		{"мертвые", "мертв"},
		{"мертвый", "мертв"},
		{"мертвым", "мертв"},
		{"мертвых", "мертв"},
		{"мертвящей", "мертвя"},
		{"меру", "мер"},
		{"мерцает", "мерца"},
		{"мерцание", "мерцан"},
		{"мерцанье", "мерцан"},
		{"мерцаньем", "мерцан"},
		{"мерцающий", "мерца"},
		{"мерцающими", "мерца"},
		{"меры", "мер"},
		{"меря", "мер"},
		{"меряясь", "мер"},
		{"мессианическую", "мессианическ"},
		{"мессины", "мессин"},
		{"мессия", "месс"},
		{"мессы", "месс"},
		{"мест", "мест"},
		{"места", "мест"},
		{"местам", "мест"},
		{"местами", "мест"},
		{"местах", "мест"},
		{"месте", "мест"},
		{"местечка", "местечк"},
		{"местечках", "местечк"},
		{"мести", "мест"},
		{"местности", "местност"},
		{"местность", "местност"},
		{"место", "мест"},
		{"местов", "мест"},
		{"местом", "мест"},
		{"местоположение", "местоположен"},
		{"месту", "мест"},
		{"месть", "мест"},
		{"местью", "мест"},
		{"месяц", "месяц"},
		{"месяца", "месяц"},
		{"месяце", "месяц"},
		{"месяцев", "месяц"},
		{"месяцем", "месяц"},
		{"месяцу", "месяц"},
		{"месяцы", "месяц"},
		{"метал", "мета"},
		{"металась", "мета"},
		{"металл", "металл"},
		{"металлическая", "металлическ"},
		{"металлической", "металлическ"},
		{"метался", "мета"},
		{"метаний", "метан"},
		{"метать", "мета"},
		{"метаться", "мета"},
		{"метафизике", "метафизик"},
		{"метели", "метел"},
		{"метелицей", "метелиц"},
		{"метель", "метел"},
		{"метельную", "метельн"},
		{"метелью", "метел"},
		{"метеоры", "метеор"},
		{"метешь", "метеш"},
		{"мети", "мет"},
		{"метит", "мет"},
		{"меткое", "метк"},
		{"метлинскому", "метлинск"},
		{"метлой", "метл"},
		{"метнула", "метнул"},
		{"метнулись", "метнул"},
		{"метнулся", "метнул"},
		{"метод", "метод"},
		{"метода", "метод"},
		{"методически", "методическ"},
		{"метр", "метр"},
		{"мету", "мет"},
		{"мех", "мех"},
		{"меха", "мех"},
		{"механизм", "механизм"},
		{"механики", "механик"},
		{"механически", "механическ"},
		{"мехмет", "мехмет"},
		{"мехов", "мех"},
		{"меховая", "мехов"},
		{"меховою", "мехов"},
		{"меховую", "мехов"},
		{"мехом", "мех"},
		{"меч", "меч"},
		{"мечами", "меч"},
		{"мече", "меч"},
		{"мечет", "мечет"},
		{"мечетесь", "мечет"},
		{"мечи", "меч"},
		{"мечом", "меч"},
		{"мечта", "мечт"},
		{"мечтавшую", "мечта"},
		{"мечтает", "мечта"},
		{"мечтается", "мечта"},
		{"мечтаешь", "мечта"},
		{"мечтал", "мечта"},
		{"мечтала", "мечта"},
		{"мечтали", "мечта"},
		{"мечталось", "мечта"},
		{"мечтам", "мечт"},
		{"мечтами", "мечт"},
		{"мечтаний", "мечтан"},
		{"мечтания", "мечтан"},
		{"мечтаниями", "мечтан"},
		{"мечтаниях", "мечтан"},
		{"мечтанье", "мечтан"},
		{"мечтанья", "мечтан"},
		{"мечтатель", "мечтател"},
		{"мечтательница", "мечтательниц"},
		{"мечтательно", "мечтательн"},
		{"мечтательностию", "мечтательн"},
		{"мечтательность", "мечтательн"},
		{"мечтательных", "мечтательн"},
		{"мечтателям", "мечтател"},
		{"мечтать", "мечта"},
		{"мечтах", "мечт"},
		{"мечтаю", "мечта"},
		{"мечте", "мечт"},
		{"мечтой", "мечт"},
		{"мечту", "мечт"},
		{"мечты", "мечт"},
		{"мечусь", "меч"},
		{"мечут", "мечут"},
		{"мечутся", "мечут"},
		{"мешавшее", "меша"},
		{"мешавшийся", "меша"},
		{"мешавших", "меша"},
		{"мешаем", "меша"},
		{"мешает", "меша"},
		{"мешаете", "меша"},
		{"мешается", "меша"},
		{"мешай", "меша"},
		{"мешайте", "меша"},
		{"мешал", "меша"},
		{"мешала", "меша"},
		{"мешалась", "меша"},
		{"мешали", "меша"},
		{"мешались", "меша"},
		{"мешало", "меша"},
		{"мешать", "меша"},
		{"мешаться", "меша"},
		{"мешаю", "меша"},
		{"мешают", "меша"},
		{"мешаются", "меша"},
		{"мешающего", "меша"},
		{"мешающую", "меша"},
		{"мешая", "меш"},
		{"мешаясь", "меш"},
		{"мешкать", "мешка"},
		{"мешкая", "мешк"},
		{"мешке", "мешк"},
		{"мешки", "мешк"},
		{"мешок", "мешок"},
		{"мещан", "меща"},
		{"мещане", "мещан"},
		{"мещанин", "мещанин"},
		{"мещанина", "мещанин"},
		{"мещанинишка", "мещанинишк"},
		{"мещанинишкой", "мещанинишк"},
		{"мещанинишку", "мещанинишк"},
		{"мещанином", "мещанин"},
		{"мещанка", "мещанк"},
		{"мещанок", "мещанок"},
		{"мещаночка", "мещаночк"},
		{"мещански", "мещанск"},
		{"мещанской", "мещанск"},
		{"ми", "ми"},
		{"миг", "миг"},
		{"мига", "миг"},
		{"мигавшие", "мига"},
		{"мигает", "мига"},
		{"мигал", "мига"},
		{"мигали", "мига"},
		{"мигать", "мига"},
		{"мигают", "мига"},
		{"мигающего", "мига"},
		{"мигая", "миг"},
		{"миги", "миг"},
		{"мигнет", "мигнет"},
		{"мигни", "мигн"},
		{"мигнувший", "мигнувш"},
		{"мигнул", "мигнул"},
		{"мигнуть", "мигнут"},
		{"мигом", "миг"},
		{"мигрень", "мигрен"},
		{"мизерном", "мизерн"},
		{"мизерные", "мизерн"},
		{"мизерных", "мизерн"},
		{"мизинца", "мизинц"},
		{"микельанджело", "микельанджел"},
		{"миколай", "микола"},
		{"миколаю", "микола"},
		{"миколая", "микол"},
		{"миколка", "миколк"},
		{"миколке", "миколк"},
		{"миколки", "миколк"},
		{"миколкину", "миколкин"},
		{"миколкой", "миколк"},
		{"миколку", "миколк"},
		{"микробами", "микроб"},
		{"микроскопические", "микроскопическ"},
		{"микроскопических", "микроскопическ"},
		{"микстуру", "микстур"},
		{"мил", "мил"},
		{"мила", "мил"},
		{"милашка", "милашк"},
		{"милая", "мил"},
		{"милее", "мил"},
		{"милейшая", "мил"},
		{"милейший", "мил"},
		{"миленькая", "миленьк"},
		{"миленькие", "миленьк"},
		{"миленький", "миленьк"},
		{"милитриса", "милитрис"},
		{"милитрисе", "милитрис"},
		{"миллиарды", "миллиард"},
		{"миллион", "миллион"},
		{"миллиона", "миллион"},
		{"миллионами", "миллион"},
		{"миллионер", "миллионер"},
		{"миллионером", "миллионер"},
		{"миллионная", "миллион"},
		{"миллионов", "миллион"},
		{"миллиону", "миллион"},
		{"миллионы", "миллион"},
		{"мило", "мил"},
		{"миловать", "милова"},
		{"милого", "мил"},
		{"милое", "мил"},
		{"милой", "мил"},
		{"милосерд", "милосерд"},
		{"милосердии", "милосерд"},
		{"милосердию", "милосерд"},
		{"милосердия", "милосерд"},
		{"милосердному", "милосердн"},
		{"милосердные", "милосердн"},
		{"милосердный", "милосердн"},
		{"милосердным", "милосердн"},
		{"милосердые", "милосерд"},
		{"милосердый", "милосерд"},
		{"милости", "милост"},
		{"милостив", "милост"},
		{"милостивая", "милостив"},
		{"милостивее", "милостив"},
		{"милостивое", "милостив"},
		{"милостивый", "милостив"},
		{"милостисдарь", "милостисдар"},
		{"милостию", "милост"},
		{"милостыни", "милостын"},
		{"милостыню", "милостын"},
		{"милость", "милост"},
		{"милочка", "милочк"},
		{"милую", "мил"},
		{"милы", "мил"},
		{"милые", "мил"},
		{"милый", "мил"},
		{"милым", "мил"},
		{"мильон", "мильон"},
		{"милюков", "милюк"},
		{"милютины", "милютин"},
		{"миме", "мим"},
		{"мимики", "мимик"},
		{"мимо", "мим"},
		{"мимолетное", "мимолетн"},
		{"мимолетной", "мимолетн"},
		{"мимолетных", "мимолетн"},
		{"мимоходом", "мимоход"},
		{"миндаль", "миндал"},
		{"миндалю", "миндал"},
		{"миндаля", "миндал"},
		{"минералами", "минерал"},
		{"минет", "минет"},
		{"министерства", "министерств"},
		{"министерстве", "министерств"},
		{"министр", "министр"},
		{"министра", "министр"},
		{"министрам", "министр"},
		{"министром", "министр"},
		{"министру", "министр"},
		{"миновав", "минова"},
		{"миновал", "минова"},
		{"миновала", "минова"},
		{"миновалась", "минова"},
		{"миновались", "минова"},
		{"миновало", "минова"},
		{"миновалось", "минова"},
		{"миновать", "минова"},
		{"миной", "мин"},
		{"мину", "мин"},
		{"минувшего", "минувш"},
		{"минувшее", "минувш"},
		{"минувшем", "минувш"},
		{"минувших", "минувш"},
		{"минует", "мин"},
		{"минул", "минул"},
		{"минусом", "минус"},
		{"минут", "минут"},
		{"минута", "минут"},
		{"минутами", "минут"},
		{"минутах", "минут"},
		{"минуте", "минут"},
		{"минутки", "минутк"},
		{"минутку", "минутк"},
		{"минутно", "минутн"},
		{"минутного", "минутн"},
		{"минутное", "минутн"},
		{"минутны", "минутн"},
		{"минутный", "минутн"},
		{"минутой", "минут"},
		{"минуток", "минуток"},
		{"минуточку", "минуточк"},
		{"минутою", "минут"},
		{"минуту", "минут"},
		{"минуты", "минут"},
		{"минуют", "мин"},
		{"минуя", "мину"},
		{"мины", "мин"},
		{"миньятюрных", "миньятюрн"},
		{"мир", "мир"},
		{"мира", "мир"},
		{"мираж", "мираж"},
		{"миражами", "мираж"},
		{"миражи", "мираж"},
		{"мире", "мир"},
		{"мирен", "мир"},
		{"мирилась", "мир"},
		{"миримся", "мир"},
		{"мирит", "мир"},
		{"мирится", "мир"},
		{"мирить", "мир"},
		{"мириться", "мир"},
		{"мирно", "мирн"},
		{"мирного", "мирн"},
		{"мирное", "мирн"},
		{"мирной", "мирн"},
		{"мирный", "мирн"},
		{"мирным", "мирн"},
		{"мирных", "мирн"},
		{"миров", "мир"},
		{"мировая", "миров"},
		{"мирового", "миров"},
		{"мировое", "миров"},
		{"мировой", "миров"},
		{"мировою", "миров"},
		{"мировую", "миров"},
		{"мировых", "миров"},
		{"миролюбивого", "миролюбив"},
		{"миром", "мир"},
		{"миртов", "мирт"},
		{"мирты", "мирт"},
		{"миру", "мир"},
		{"миры", "мир"},
		{"мирянам", "мирян"},
		{"мирятся", "мир"},
		{"мисс", "мисс"},
		{"мистик", "мистик"},
		{"мистифицировать", "мистифицирова"},
		{"мистически", "мистическ"},
		{"мистическое", "мистическ"},
		{"мистическом", "мистическ"},
		{"митреем", "митре"},
		{"митрей", "митр"},
		{"митрея", "митре"},
		{"митрофаниевском", "митрофаниевск"},
		{"митька", "митьк"},
		{"митьки", "митьк"},
		{"митькой", "митьк"},
		{"митьку", "митьк"},
		{"михаил", "миха"},
		{"михаила", "миха"},
		{"михайлов", "михайл"},
		{"михайлович", "михайлович"},
		{"михайловна", "михайловн"},
		{"михайловну", "михайловн"},
		{"михайловны", "михайловн"},
		{"михайловским", "михайловск"},
		{"михайлыч", "михайлыч"},
		{"михайлычем", "михайлыч"},
		{"михеем", "михе"},
		{"михей", "мих"},
		{"михею", "мих"},
		{"михея", "михе"},
		{"мицкевича", "мицкевич"},
		{"мичману", "мичман"},
		{"миша", "миш"},
		{"мише", "миш"},
		{"мишель", "мишел"},
		{"мишу", "миш"},
		{"мишуры", "мишур"},
		{"младая", "млад"},
		{"младенец", "младенец"},
		{"младенца", "младенц"},
		{"младенцев", "младенц"},
		{"младенцем", "младенц"},
		{"младенцу", "младенц"},
		{"младенческий", "младенческ"},
		{"младенческой", "младенческ"},
		{"младость", "младост"},
		{"младшая", "младш"},
		{"младшей", "младш"},
		{"младшему", "младш"},
		{"младший", "младш"},
		{"младшую", "младш"},
		{"младые", "млад"},
		{"млет", "млет"},
		{"мне", "мне"},
		{"мнение", "мнен"},
		{"мнением", "мнен"},
		{"мнении", "мнен"},
		{"мнений", "мнен"},
		{"мнению", "мнен"},
		{"мнения", "мнен"},
		{"мнет", "мнет"},
		{"мнителен", "мнител"},
		{"мнительно", "мнительн"},
		{"мнительность", "мнительн"},
		{"мнительный", "мнительн"},
		{"многие", "мног"},
		{"многим", "мног"},
		{"многими", "мног"},
		{"многих", "мног"},
		{"много", "мног"},
		{"многого", "мног"},
		{"многодумный", "многодумн"},
		{"многое", "мног"},
		{"многознаменателен", "многознаменател"},
		{"многозначительны", "многозначительн"},
		{"многоколонный", "многоколон"},
		{"многократно", "многократн"},
		{"многократное", "многократн"},
		{"многолюдный", "многолюдн"},
		{"многолюдству", "многолюдств"},
		{"многом", "мног"},
		{"многому", "мног"},
		{"многообещавшее", "многообеща"},
		{"многообещающих", "многообеща"},
		{"многообразным", "многообразн"},
		{"многопенный", "многопен"},
		{"многоразличны", "многоразличн"},
		{"многоразличных", "многоразличн"},
		{"многоречиво", "многоречив"},
		{"многословные", "многословн"},
		{"многостороннее", "многосторон"},
		{"многотрудной", "многотрудн"},
		{"многотрудный", "многотрудн"},
		{"многоуважаемая", "многоуважа"},
		{"многоуважаемой", "многоуважа"},
		{"многоуважаемою", "многоуважа"},
		{"многоуважаемый", "многоуважа"},
		{"многоцветными", "многоцветн"},
		{"многочисленнее", "многочислен"},
		{"многочисленнейшее", "многочислен"},
		{"многочисленной", "многочислен"},
		{"многочисленные", "многочислен"},
		{"многочисленными", "многочислен"},
		{"многочисленных", "многочислен"},
		{"многоэтажный", "многоэтажн"},
		{"многоярусный", "многоярусн"},
		{"множественном", "множествен"},
		{"множество", "множеств"},
		{"множеством", "множеств"},
		{"множит", "множ"},
		{"мнози", "мноз"},
		{"мной", "мно"},
		{"мною", "мно"},
		{"мня", "мня"},
		{"мо", "мо"},
		{"мов", "мов"},
		{"мовщина", "мовщин"},
		{"мог", "мог"},
		{"могил", "мог"},
		{"могила", "мог"},
		{"могиле", "могил"},
		{"могилка", "могилк"},
		{"могилке", "могилк"},
		{"могилки", "могилк"},
		{"могилкой", "могилк"},
		{"могилой", "могил"},
		{"могилою", "могил"},
		{"могилу", "могил"},
		{"могилы", "могил"},
		{"могильная", "могильн"},
		{"могильной", "могильн"},
		{"могильный", "могильн"},
		{"могильными", "могильн"},
		{"могильщик", "могильщик"},
		{"могла", "могл"},
		{"могли", "могл"},
		{"могло", "могл"},
		{"могу", "мог"},
		{"могут", "могут"},
		{"могуче", "могуч"},
		{"могучей", "могуч"},
		{"могучие", "могуч"},
		{"могучий", "могуч"},
		{"могучих", "могуч"},
		{"могущее", "могущ"},
		{"могущество", "могуществ"},
		{"могущие", "могущ"},
		{"мода", "мод"},
		{"моде", "мод"},
		{"модели", "модел"},
		{"моделью", "модел"},
		{"модистки", "модистк"},
		{"модная", "модн"},
		{"модничаньем", "модничан"},
		{"модно", "модн"},
		{"модное", "модн"},
		{"модной", "модн"},
		{"модный", "модн"},
		{"модным", "модн"},
		{"мое", "мо"},
		{"моего", "мо"},
		{"моей", "мо"},
		{"моем", "мо"},
		{"моему", "мо"},
		{"моет", "моет"},
		{"моется", "моет"},
		{"моею", "мо"},
		{"може", "мож"},
		{"можедом", "можед"},
		{"можем", "мож"},
		{"может", "может"},
		{"можете", "может"},
		{"можешь", "можеш"},
		{"можно", "можн"},
		{"мозаи", "моза"},
		{"мозг", "мозг"},
		{"мозга", "мозг"},
		{"мозги", "мозг"},
		{"мозгу", "мозг"},
		{"мозоль", "мозол"},
		{"мои", "мо"},
		{"моим", "мо"},
		{"моими", "мо"},
		{"моисеич", "моисеич"},
		{"моих", "мо"},
		{"мой", "мо"},
		{"мокка", "мокк"},
		{"мокрая", "мокр"},
		{"мокро", "мокр"},
		{"мокрое", "мокр"},
		{"мокрой", "мокр"},
		{"мокрота", "мокрот"},
		{"мокроту", "мокрот"},
		{"мокрую", "мокр"},
		{"мокры", "мокр"},
		{"мокрые", "мокр"},
		{"мокрый", "мокр"},
		{"мокрым", "мокр"},
		{"мокрыми", "мокр"},
		{"мокрых", "мокр"},
		{"мол", "мол"},
		{"молва", "молв"},
		{"молвив", "молв"},
		{"молвил", "молв"},
		{"молвит", "молв"},
		{"молвой", "молв"},
		{"молву", "молв"},
		{"молвы", "молв"},
		{"молебен", "молеб"},
		{"моли", "мол"},
		{"молил", "мол"},
		{"молила", "мол"},
		{"молилась", "мол"},
		{"молились", "мол"},
		{"молилось", "мол"},
		{"молился", "мол"},
		{"молимся", "мол"},
		{"молись", "мол"},
		{"молитв", "молитв"},
		{"молитва", "молитв"},
		{"молитвах", "молитв"},
		{"молитвенное", "молитвен"},
		{"молитвенный", "молитвен"},
		{"молитвой", "молитв"},
		{"молитву", "молитв"},
		{"молитвы", "молитв"},
		{"молитесь", "мол"},
		{"молить", "мол"},
		{"молиться", "мол"},
		{"молишься", "мол"},
		{"молнией", "молн"},
		{"молниеносного", "молниеносн"},
		{"молниеносных", "молниеносн"},
		{"молнии", "молн"},
		{"молний", "молн"},
		{"молния", "молн"},
		{"молньей", "молн"},
		{"молньи", "молн"},
		{"молод", "молод"},
		{"молода", "молод"},
		{"молодая", "молод"},
		{"молодеет", "молодеет"},
		{"молодежи", "молодеж"},
		{"молодежь", "молодеж"},
		{"молоденькая", "молоденьк"},
		{"молоденькое", "молоденьк"},
		{"молоденькой", "молоденьк"},
		{"молодец", "молодец"},
		{"молодо", "молод"},
		{"молодого", "молод"},
		{"молодое", "молод"},
		{"молодой", "молод"},
		{"молодом", "молод"},
		{"молодому", "молод"},
		{"молодости", "молод"},
		{"молодость", "молод"},
		{"молодою", "молод"},
		{"молодую", "молод"},
		{"молодца", "молодц"},
		{"молодцевато", "молодцеват"},
		{"молодцы", "молодц"},
		{"молоды", "молод"},
		{"молодые", "молод"},
		{"молодым", "молод"},
		{"молодыми", "молод"},
		{"молодых", "молод"},
		{"моложавее", "моложав"},
		{"моложавом", "моложав"},
		{"моложе", "молож"},
		{"молока", "молок"},
		{"молоко", "молок"},
		{"молоком", "молок"},
		{"молокосос", "молокосос"},
		{"молокососа", "молокосос"},
		{"молокососы", "молокосос"},
		{"молола", "молол"},
		{"молот", "молот"},
		{"молотком", "молотк"},
		{"молоть", "молот"},
		{"молочнице", "молочниц"},
		{"молочный", "молочн"},
		{"молочным", "молочн"},
		{"молочных", "молочн"},
		{"молча", "молч"},
		{"молчавшего", "молча"},
		{"молчал", "молча"},
		{"молчала", "молча"},
		{"молчали", "молча"},
		{"молчалива", "молчалив"},
		{"молчаливая", "молчалив"},
		{"молчаливее", "молчалив"},
		{"молчаливо", "молчалив"},
		{"молчаливость", "молчалив"},
		{"молчаливую", "молчалив"},
		{"молчаливые", "молчалив"},
		{"молчаливый", "молчалив"},
		{"молчаливым", "молчалив"},
		{"молчание", "молчан"},
		{"молчанием", "молчан"},
		{"молчании", "молчан"},
		{"молчанию", "молчан"},
		{"молчания", "молчан"},
		{"молчановка", "молчановк"},
		{"молчат", "молчат"},
		{"молчать", "молча"},
		{"молчи", "молч"},
		{"молчим", "молч"},
		{"молчит", "молч"},
		{"молчите", "молч"},
		{"молчишь", "молч"},
		{"молчу", "молч"},
		{"моль", "мол"},
		{"мольба", "мольб"},
		{"мольбами", "мольб"},
		{"мольбе", "мольб"},
		{"мольбой", "мольб"},
		{"молюсь", "мол"},
		{"моля", "мол"},
		{"молясь", "мол"},
		{"молящий", "моля"},
		{"момент", "момент"},
		{"момента", "момент"},
		{"моменты", "момент"},
		{"монастыре", "монастыр"},
		{"монастырем", "монастыр"},
		{"монастырский", "монастырск"},
		{"монастырь", "монастыр"},
		{"монах", "мон"},
		{"монахам", "монах"},
		{"монахи", "монах"},
		{"монахов", "монах"},
		{"монаху", "монах"},
		{"монашку", "монашк"},
		{"монблан", "монбла"},
		{"монета", "монет"},
		{"монетку", "монетк"},
		{"монетой", "монет"},
		{"монетчиков", "монетчик"},
		{"монеты", "монет"},
		{"монисто", "монист"},
		{"монологам", "монолог"},
		{"монологах", "монолог"},
		{"монологи", "монолог"},
		{"мономана", "монома"},
		{"мономании", "мономан"},
		{"мономанию", "мономан"},
		{"мономания", "мономан"},
		{"мономанов", "мономан"},
		{"мономаны", "монома"},
		{"монополию", "монопол"},
		{"монотонно", "монотон"},
		{"монотонной", "монотон"},
		{"монотонные", "монотон"},
		{"монтировало", "монтирова"},
		{"монумент", "монумент"},
		{"монументально", "монументальн"},
		{"монументальных", "монументальн"},
		{"мопассан", "мопасса"},
		{"мораль", "морал"},
		{"морально", "моральн"},
		{"моргающими", "морга"},
		{"морген", "морг"},
		{"морде", "морд"},
		{"морду", "морд"},
		{"морды", "морд"},
		{"море", "мор"},
		{"морем", "мор"},
		{"моркови", "морков"},
		{"морковь", "морков"},
		{"морковью", "морков"},
		{"моровой", "моров"},
		{"мороженое", "морожен"},
		{"мороженым", "морожен"},
		{"мороз", "мороз"},
		{"мороза", "мороз"},
		{"морозами", "мороз"},
		{"морозе", "мороз"},
		{"морозное", "морозн"},
		{"морозном", "морозн"},
		{"морозный", "морозн"},
		{"морозным", "морозн"},
		{"морозцы", "морозц"},
		{"морозы", "мороз"},
		{"морочил", "мороч"},
		{"морочите", "мороч"},
		{"морская", "морск"},
		{"морские", "морск"},
		{"морским", "морск"},
		{"морских", "морск"},
		{"морского", "морск"},
		{"морское", "морск"},
		{"морской", "морск"},
		{"морском", "морск"},
		{"морскую", "морск"},
		{"морфием", "морф"},
		{"морфий", "морф"},
		{"морщась", "морщ"},
		{"морщатся", "морщат"},
		{"морщила", "морщ"},
		{"морщились", "морщ"},
		{"морщился", "морщ"},
		{"морщин", "морщин"},
		{"морщинами", "морщин"},
		{"морщинах", "морщин"},
		{"морщинистое", "морщинист"},
		{"морщинки", "морщинк"},
		{"морщиной", "морщин"},
		{"морщины", "морщин"},
		{"морщитесь", "морщ"},
		{"морщится", "морщ"},
		{"морю", "мор"},
		{"моря", "мор"},
		{"мосеич", "мосеич"},
		{"москва", "москв"},
		{"москве", "москв"},
		{"москви", "москв"},
		{"москвой", "москв"},
		{"москву", "москв"},
		{"москвы", "москв"},
		{"московские", "московск"},
		{"московским", "московск"},
		{"московских", "московск"},
		{"московского", "московск"},
		{"московской", "московск"},
		{"московском", "московск"},
		{"московскою", "московск"},
		{"мост", "мост"},
		{"моста", "мост"},
		{"мостах", "мост"},
		{"мосте", "мост"},
		{"мостик", "мостик"},
		{"мосткам", "мостк"},
		{"мостки", "мостк"},
		{"мостов", "мост"},
		{"мостовая", "мостов"},
		{"мостовой", "мостов"},
		{"мостовую", "мостов"},
		{"мостовых", "мостов"},
		{"мостом", "мост"},
		{"мосту", "мост"},
		{"мосты", "мост"},
		{"мосье", "мос"},
		{"мотать", "мота"},
		{"мотаю", "мота"},
		{"мотают", "мота"},
		{"мотив", "мот"},
		{"мотор", "мотор"},
		{"мотька", "мотьк"},
		{"мох", "мох"},
		{"мохнатое", "мохнат"},
		{"мохнатой", "мохнат"},
		{"мохнатую", "мохнат"},
		{"моцарта", "моцарт"},
		{"моцион", "моцион"},
		{"моциона", "моцион"},
		{"мочальное", "мочальн"},
		{"моченье", "мочен"},
		{"мочи", "моч"},
		{"мочили", "моч"},
		{"мочками", "мочк"},
		{"мочь", "моч"},
		{"мошенник", "мошенник"},
		{"мошенника", "мошенник"},
		{"мошенники", "мошенник"},
		{"мошенников", "мошенник"},
		{"мошенническая", "мошенническ"},
		{"мошенническим", "мошенническ"},
		{"мошенничеств", "мошенничеств"},
		{"мощные", "мощн"},
		{"мою", "мо"},
		{"моя", "мо"},
		{"мрак", "мрак"},
		{"мрака", "мрак"},
		{"мраке", "мрак"},
		{"мраков", "мрак"},
		{"мраком", "мрак"},
		{"мрамора", "мрамор"},
		{"мраморной", "мраморн"},
		{"мраморном", "мраморн"},
		{"мрачен", "мрач"},
		{"мрачились", "мрач"},
		{"мрачить", "мрач"},
		{"мрачна", "мрачн"},
		{"мрачная", "мрачн"},
		{"мрачнее", "мрачн"},
		{"мрачней", "мрачн"},
		{"мрачно", "мрачн"},
		{"мрачного", "мрачн"},
		{"мрачное", "мрачн"},
		{"мрачной", "мрачн"},
		{"мрачном", "мрачн"},
		{"мрачною", "мрачн"},
		{"мрачную", "мрачн"},
		{"мрачные", "мрачн"},
		{"мрачный", "мрачн"},
		{"мрачным", "мрачн"},
		{"мрачных", "мрачн"},
		{"мстил", "мстил"},
		{"мстителен", "мстител"},
		{"мститель", "мстител"},
		{"мстительной", "мстительн"},
		{"мстить", "мстит"},
		{"мстят", "мстят"},
		{"мсье", "мсье"},
		{"му", "му"},
		{"мудр", "мудр"},
		{"мудрая", "мудр"},
		{"мудрено", "мудр"},
		{"мудреной", "мудрен"},
		{"мудреному", "мудрен"},
		{"мудреные", "мудрен"},
		{"мудреным", "мудрен"},
		{"мудреными", "мудрен"},
		{"мудрецов", "мудрец"},
		{"мудрецы", "мудрец"},
		{"мудрил", "мудр"},
		{"мудрить", "мудр"},
		{"мудро", "мудр"},
		{"мудрое", "мудр"},
		{"мудрость", "мудрост"},
		{"мудрствуйте", "мудрств"},
		{"мудрые", "мудр"},
		{"муж", "муж"},
		{"мужа", "муж"},
		{"муже", "муж"},
		{"мужей", "муж"},
		{"мужем", "муж"},
		{"мужества", "мужеств"},
		{"мужественно", "мужествен"},
		{"мужественного", "мужествен"},
		{"мужественное", "мужествен"},
		{"мужественности", "мужествен"},
		{"мужественность", "мужествен"},
		{"мужественный", "мужествен"},
		{"мужественным", "мужествен"},
		{"мужеством", "мужеств"},
		{"мужик", "мужик"},
		{"мужика", "мужик"},
		{"мужикам", "мужик"},
		{"мужиками", "мужик"},
		{"мужиках", "мужик"},
		{"мужики", "мужик"},
		{"мужиков", "мужик"},
		{"мужиком", "мужик"},
		{"мужику", "мужик"},
		{"мужичка", "мужичк"},
		{"мужичком", "мужичк"},
		{"мужичок", "мужичок"},
		{"мужичья", "мужич"},
		{"мужнего", "мужн"},
		{"мужнина", "мужнин"},
		{"мужниного", "мужнин"},
		{"мужниной", "мужнин"},
		{"мужские", "мужск"},
		{"мужского", "мужск"},
		{"мужской", "мужск"},
		{"мужскому", "мужск"},
		{"мужу", "муж"},
		{"мужчин", "мужчин"},
		{"мужчина", "мужчин"},
		{"мужчине", "мужчин"},
		{"мужчиной", "мужчин"},
		{"мужчины", "мужчин"},
		{"мужья", "муж"},
		{"мужьям", "муж"},
		{"мужьями", "муж"},
		{"муза", "муз"},
		{"музе", "муз"},
		{"музее", "муз"},
		{"музы", "муз"},
		{"музыка", "музык"},
		{"музыкальной", "музыкальн"},
		{"музыкальном", "музыкальн"},
		{"музыкальный", "музыкальн"},
		{"музыкант", "музыкант"},
		{"музыкантам", "музыкант"},
		{"музыкантов", "музыкант"},
		{"музыканты", "музыкант"},
		{"музыке", "музык"},
		{"музыки", "музык"},
		{"музыкой", "музык"},
		{"музыку", "музык"},
		{"мук", "мук"},
		{"мука", "мук"},
		{"муками", "мук"},
		{"муке", "мук"},
		{"муки", "мук"},
		{"мукой", "мук"},
		{"муку", "мук"},
		{"мундир", "мундир"},
		{"мундире", "мундир"},
		{"мундиров", "мундир"},
		{"мундштук", "мундштук"},
		{"муравей", "мурав"},
		{"муравейником", "муравейник"},
		{"муравьи", "мурав"},
		{"муравьиною", "муравьин"},
		{"мурашиной", "мурашин"},
		{"мурашки", "мурашк"},
		{"мурлыкающем", "мурлыка"},
		{"мурлычу", "мурлыч"},
		{"мурлычут", "мурлычут"},
		{"муромца", "муромц"},
		{"мус", "мус"},
		{"мускул", "мускул"},
		{"мускулистых", "мускулист"},
		{"мускулов", "мускул"},
		{"мускулы", "мускул"},
		{"мускульная", "мускульн"},
		{"мускульной", "мускульн"},
		{"мускульном", "мускульн"},
		{"муссинским", "муссинск"},
		{"муссинских", "муссинск"},
		{"мусье", "мус"},
		{"мутилось", "мут"},
		{"мутит", "мут"},
		{"мутится", "мут"},
		{"мутить", "мут"},
		{"мутна", "мутн"},
		{"мутно", "мутн"},
		{"мутны", "мутн"},
		{"мутные", "мутн"},
		{"мутный", "мутн"},
		{"мутным", "мутн"},
		{"муть", "мут"},
		{"мутят", "мут"},
		{"мух", "мух"},
		{"муха", "мух"},
		{"мухи", "мух"},
		{"мухояров", "мухояр"},
		{"мухоярова", "мухояров"},
		{"муху", "мух"},
		{"мучает", "муча"},
		{"мучаете", "муча"},
		{"мучаетесь", "муча"},
		{"мучается", "муча"},
		{"мучаешь", "муча"},
		{"мучаешься", "муча"},
		{"мучай", "муча"},
		{"мучайся", "муча"},
		{"мучайте", "муча"},
		{"мучась", "муч"},
		{"мучат", "мучат"},
		{"мучаю", "муча"},
		{"мучаюсь", "муча"},
		{"мучают", "муча"},
		{"мучая", "муч"},
		{"мучаясь", "муч"},
		{"мучение", "мучен"},
		{"мучением", "мучен"},
		{"мучений", "мучен"},
		{"мученичество", "мученичеств"},
		{"мучения", "мучен"},
		{"мучениях", "мучен"},
		{"мученье", "мучен"},
		{"мученья", "мучен"},
		{"мученьях", "мучен"},
		{"мучил", "муч"},
		{"мучила", "муч"},
		{"мучилась", "муч"},
		{"мучили", "муч"},
		{"мучились", "муч"},
		{"мучило", "муч"},
		{"мучился", "муч"},
		{"мучимый", "мучим"},
		{"мучит", "муч"},
		{"мучителей", "мучител"},
		{"мучители", "мучител"},
		{"мучительная", "мучительн"},
		{"мучительнее", "мучительн"},
		{"мучительнейшее", "мучительн"},
		{"мучительницы", "мучительниц"},
		{"мучительно", "мучительн"},
		{"мучительного", "мучительн"},
		{"мучительное", "мучительн"},
		{"мучительной", "мучительн"},
		{"мучительном", "мучительн"},
		{"мучительною", "мучительн"},
		{"мучительную", "мучительн"},
		{"мучительны", "мучительн"},
		{"мучительные", "мучительн"},
		{"мучительный", "мучительн"},
		{"мучительным", "мучительн"},
		{"мучительных", "мучительн"},
		{"мучителя", "мучител"},
		{"мучитесь", "муч"},
		{"мучится", "муч"},
		{"мучить", "муч"},
		{"мучиться", "муч"},
		{"мучишь", "муч"},
		{"мучишься", "муч"},
		{"мучнистой", "мучнист"},
		{"мучной", "мучн"},
		{"мучть", "мучт"},
		{"мучусь", "муч"},
		{"мучь", "муч"},
		{"мучьте", "мучьт"},
		{"мучьтесь", "мучьт"},
		{"мхов", "мхов"},
		{"мчалась", "мчал"},
		{"мчались", "мчал"},
		{"мчатся", "мчат"},
		{"мчаться", "мчат"},
		{"мчит", "мчит"},
		{"мчится", "мчит"},
		{"мщение", "мщен"},
		{"мщения", "мщен"},
		{"мы", "мы"},
		{"мыкается", "мыка"},
		{"мыкать", "мыка"},
		{"мыкаться", "мыка"},
		{"мыкаю", "мыка"},
		{"мыла", "мыл"},
		{"мыли", "мыл"},
		{"мыло", "мыл"},
		{"мылом", "мыл"},
		{"мыльный", "мыльн"},
		{"мыслей", "мысл"},
		{"мысленно", "мыслен"},
		{"мысленный", "мыслен"},
		{"мысли", "мысл"},
		{"мыслил", "мысл"},
		{"мыслит", "мысл"},
		{"мыслителей", "мыслител"},
		{"мыслитель", "мыслител"},
		{"мыслительная", "мыслительн"},
		{"мыслительной", "мыслительн"},
		{"мыслителя", "мыслител"},
		{"мыслителям", "мыслител"},
		{"мыслить", "мысл"},
		{"мыслию", "мысл"},
		{"мысль", "мысл"},
		{"мыслью", "мысл"},
		{"мыслю", "мысл"},
		{"мыслям", "мысл"},
		{"мыслями", "мысл"},
		{"мыслят", "мысл"},
		{"мыслях", "мысл"},
		{"мыслящие", "мысля"},
		{"мыслящим", "мысля"},
		{"мыты", "мыт"},
		{"мыть", "мыт"},
		{"мытье", "мыт"},
		{"мытья", "мыт"},
		{"мычит", "мыч"},
		{"мышами", "мыш"},
		{"мышей", "мыш"},
		{"мыши", "мыш"},
		{"мышкой", "мышк"},
		{"мышку", "мышк"},
		{"мышления", "мышлен"},
		{"мышонка", "мышонк"},
		{"мышц", "мышц"},
		{"мышь", "мыш"},
		{"мышью", "мыш"},
		{"мэри", "мэр"},
		{"мюнхенский", "мюнхенск"},
		{"мягка", "мягк"},
		{"мягкая", "мягк"},
		{"мягки", "мягк"},
		{"мягкие", "мягк"},
		{"мягкий", "мягк"},
		{"мягким", "мягк"},
		{"мягкими", "мягк"},
		{"мягких", "мягк"},
		{"мягко", "мягк"},
		{"мягкого", "мягк"},
		{"мягкое", "мягк"},
		{"мягкой", "мягк"},
		{"мягком", "мягк"},
		{"мягкости", "мягкост"},
		{"мягкость", "мягкост"},
		{"мягкостью", "мягкост"},
		{"мягок", "мягок"},
		{"мягче", "мягч"},
		{"мяса", "мяс"},
		{"мясистым", "мясист"},
		{"мясник", "мясник"},
		{"мясника", "мясник"},
		{"мясники", "мясник"},
		{"мяснику", "мясник"},
		{"мясное", "мясн"},
		{"мясной", "мясн"},
		{"мясо", "мяс"},
		{"мясоед", "мясоед"},
		{"мятеж", "мятеж"},
		{"мятежи", "мятеж"},
		{"мятежно", "мятежн"},
		{"мятежность", "мятежн"},
		{"мятежный", "мятежн"},
		{"мятежным", "мятежн"},
		{"мятежными", "мятежн"},
		{"мятежных", "мятежн"},
		{"мятой", "мят"},
		{"мяты", "мят"},
		{"мячик", "мячик"},
		{"н", "н"},
		{"на", "на"},
		{"набавил", "набав"},
		{"набалдашником", "набалдашник"},
		{"набат", "набат"},
		{"набата", "набат"},
		{"набегал", "набега"},
		{"набегала", "набега"},
		{"набегами", "набег"},
		{"набегают", "набега"},
		{"набегающих", "набега"},
		{"набежал", "набежа"},
		{"набежали", "набежа"},
		{"набекрень", "набекрен"},
		{"набережной", "набережн"},
		{"набережную", "набережн"},
		{"набережные", "набережн"},
		{"набережных", "набережн"},
		{"наберется", "наберет"},
		{"наберу", "набер"},
		{"наберут", "наберут"},
		{"набивает", "набива"},
		{"набивать", "набива"},
		{"набирается", "набира"},
		{"набиралось", "набира"},
		{"набит", "наб"},
		{"набита", "набит"},
		{"набитая", "набит"},
		{"набито", "набит"},
		{"набитую", "набит"},
		{"набитый", "набит"},
		{"наблюдавший", "наблюда"},
		{"наблюдает", "наблюда"},
		{"наблюдай", "наблюда"},
		{"наблюдайте", "наблюда"},
		{"наблюдал", "наблюда"},
		{"наблюдала", "наблюда"},
		{"наблюдали", "наблюда"},
		{"наблюдатель", "наблюдател"},
		{"наблюдательный", "наблюдательн"},
		{"наблюдателя", "наблюдател"},
		{"наблюдать", "наблюда"},
		{"наблюдают", "наблюда"},
		{"наблюдаются", "наблюда"},
		{"наблюдая", "наблюд"},
		{"наблюдение", "наблюден"},
		{"наблюдению", "наблюден"},
		{"наблюдения", "наблюден"},
		{"наблюдениям", "наблюден"},
		{"наблюдениями", "наблюден"},
		{"набожных", "набожн"},
		{"набок", "набок"},
		{"наболевшие", "наболевш"},
		{"наболело", "наболел"},
		{"наболит", "набол"},
		{"наборным", "наборн"},
		{"набравшуюся", "набра"},
		{"набрал", "набра"},
		{"набрался", "набра"},
		{"набрасывала", "набрасыва"},
		{"набрасывалась", "набрасыва"},
		{"набрать", "набра"},
		{"набраться", "набра"},
		{"набрел", "набрел"},
		{"набросана", "наброса"},
		{"набросать", "наброса"},
		{"набросаю", "наброса"},
		{"набросившись", "наброс"},
		{"набросил", "наброс"},
		{"набросилась", "наброс"},
		{"набросился", "наброс"},
		{"набросить", "наброс"},
		{"наброситься", "наброс"},
		{"наброску", "наброск"},
		{"наваждения", "наважден"},
		{"наваленные", "навален"},
		{"навалено", "навал"},
		{"навалил", "навал"},
		{"навалили", "навал"},
		{"наварю", "навар"},
		{"наведаться", "наведа"},
		{"наведаюсь", "наведа"},
		{"наведены", "навед"},
		{"наведывайтесь", "наведыва"},
		{"наведывалась", "наведыва"},
		{"наведывался", "наведыва"},
		{"навек", "навек"},
		{"навеки", "навек"},
		{"навел", "навел"},
		{"навела", "навел"},
		{"навели", "навел"},
		{"навело", "навел"},
		{"наверно", "наверн"},
		{"наверное", "наверн"},
		{"навернулись", "навернул"},
		{"навертел", "навертел"},
		{"наверх", "наверх"},
		{"наверху", "наверх"},
		{"наверчено", "наверч"},
		{"навесами", "навес"},
		{"навеселе", "навесел"},
		{"навесов", "навес"},
		{"навесом", "навес"},
		{"навести", "навест"},
		{"навестил", "навест"},
		{"навестить", "навест"},
		{"наветам", "навет"},
		{"навешаны", "навеша"},
		{"навещает", "навеща"},
		{"навеянной", "навея"},
		{"навзничь", "навзнич"},
		{"навзрыд", "навзрыд"},
		{"навин", "навин"},
		{"нависали", "нависа"},
		{"нависла", "нависл"},
		{"навлечь", "навлеч"},
		{"наводил", "навод"},
		{"наводила", "навод"},
		{"наводило", "навод"},
		{"наводит", "навод"},
		{"наводите", "навод"},
		{"наводить", "навод"},
		{"наводнение", "наводнен"},
		{"наводнениями", "наводнен"},
		{"наводя", "навод"},
		{"навозе", "навоз"},
		{"наволочками", "наволочк"},
		{"наволочки", "наволочк"},
		{"навострив", "навостр"},
		{"навострил", "навостр"},
		{"навострила", "навостр"},
		{"навострились", "навостр"},
		{"наврал", "навра"},
		{"наврала", "навра"},
		{"навсегда", "навсегд"},
		{"навстречу", "навстреч"},
		{"навыворот", "навыворот"},
		{"навык", "навык"},
		{"навыкате", "навыкат"},
		{"навязал", "навяза"},
		{"навязала", "навяза"},
		{"навязался", "навяза"},
		{"навязчивая", "навязчив"},
		{"навязывал", "навязыва"},
		{"навязывались", "навязыва"},
		{"навязывать", "навязыва"},
		{"навязываться", "навязыва"},
		{"наг", "наг"},
		{"нагибается", "нагиба"},
		{"нагибалась", "нагиба"},
		{"нагибаться", "нагиба"},
		{"нагибаясь", "нагиб"},
		{"нагло", "нагл"},
		{"наглого", "нагл"},
		{"наглости", "наглост"},
		{"наглость", "наглост"},
		{"наглостью", "наглост"},
		{"наглухо", "наглух"},
		{"наглые", "нагл"},
		{"наглый", "нагл"},
		{"наглым", "нагл"},
		{"наглядел", "наглядел"},
		{"нагляделась", "наглядел"},
		{"наглядно", "наглядн"},
		{"нагнал", "нагна"},
		{"нагнется", "нагнет"},
		{"нагнись", "нагн"},
		{"нагнув", "нагнув"},
		{"нагнувшись", "нагнувш"},
		{"нагнул", "нагнул"},
		{"нагнула", "нагнул"},
		{"нагнулся", "нагнул"},
		{"нагнусь", "нагн"},
		{"нагнуться", "нагнут"},
		{"наговариваешь", "наговарива"},
		{"наговорил", "наговор"},
		{"наговорила", "наговор"},
		{"наговорить", "наговор"},
		{"наговориться", "наговор"},
		{"нагой", "наг"},
		{"нагонит", "нагон"},
		{"нагорном", "нагорн"},
		{"наготове", "наготов"},
		{"наготой", "нагот"},
		{"наград", "наград"},
		{"награда", "наград"},
		{"наградишь", "наград"},
		{"наградой", "наград"},
		{"награду", "наград"},
		{"награды", "наград"},
		{"награждать", "награжда"},
		{"награждая", "награжд"},
		{"награждения", "награжден"},
		{"награждено", "награжд"},
		{"нагреется", "нагреет"},
		{"нагроможденной", "нагроможден"},
		{"нагрубила", "нагруб"},
		{"нагрубит", "нагруб"},
		{"нагрудник", "нагрудник"},
		{"нагружать", "нагружа"},
		{"нагрязнили", "нагрязн"},
		{"нагрянет", "нагрянет"},
		{"нагрянул", "нагрянул"},
		{"над", "над"},
		{"надавал", "надава"},
		{"надает", "нада"},
		{"надарит", "надар"},
		{"надбавляет", "надбавля"},
		{"надвигается", "надвига"},
		{"надвинулись", "надвинул"},
		{"надвое", "надв"},
		{"надворного", "надворн"},
		{"надворному", "надворн"},
		{"надворный", "надворн"},
		{"надворным", "надворн"},
		{"надвяжем", "надвяж"},
		{"надвяжу", "надвяж"},
		{"надвязать", "надвяза"},
		{"надев", "над"},
		{"надевает", "надева"},
		{"надеваешь", "надева"},
		{"надевала", "надева"},
		{"надевали", "надева"},
		{"надевать", "надева"},
		{"надевают", "надева"},
		{"надевая", "надев"},
		{"надеемся", "наде"},
		{"надеетесь", "надеет"},
		{"надеется", "надеет"},
		{"надежд", "надежд"},
		{"надежда", "надежд"},
		{"надеждами", "надежд"},
		{"надеждах", "надежд"},
		{"надежде", "надежд"},
		{"надеждой", "надежд"},
		{"надеждою", "надежд"},
		{"надежду", "надежд"},
		{"надежды", "надежд"},
		{"надежная", "надежн"},
		{"надежный", "надежн"},
		{"надейтесь", "над"},
		{"надел", "надел"},
		{"надела", "надел"},
		{"наделает", "надела"},
		{"наделал", "надела"},
		{"наделала", "надела"},
		{"наделают", "надела"},
		{"надели", "надел"},
		{"наденет", "наденет"},
		{"наденете", "наденет"},
		{"наденешь", "наденеш"},
		{"надену", "наден"},
		{"надень", "наден"},
		{"наденьте", "наденьт"},
		{"надеру", "надер"},
		{"надет", "надет"},
		{"надета", "надет"},
		{"надето", "надет"},
		{"надетое", "надет"},
		{"надетой", "надет"},
		{"надеты", "надет"},
		{"надеть", "надет"},
		{"надеюсь", "над"},
		{"надеются", "надеют"},
		{"надеялась", "надея"},
		{"надеялись", "надея"},
		{"надеялся", "надея"},
		{"надеясь", "наде"},
		{"надеяться", "надея"},
		{"надзиратель", "надзирател"},
		{"надзирательницей", "надзирательниц"},
		{"надзирателю", "надзирател"},
		{"надзирателя", "надзирател"},
		{"надзором", "надзор"},
		{"надивиться", "надив"},
		{"надлежащем", "надлежа"},
		{"надлежащими", "надлежа"},
		{"надменен", "надмен"},
		{"надменная", "надмен"},
		{"надменно", "надмен"},
		{"надменной", "надмен"},
		{"надменном", "надмен"},
		{"надменность", "надмен"},
		{"надменный", "надмен"},
		{"надменным", "надмен"},
		{"надо", "над"},
		{"надобен", "надоб"},
		{"надобилось", "надоб"},
		{"надобно", "надобн"},
		{"надобностей", "надобн"},
		{"надобности", "надобн"},
		{"надобность", "надобн"},
		{"надобны", "надобн"},
		{"надоедаем", "надоеда"},
		{"надоедал", "надоеда"},
		{"надоедала", "надоеда"},
		{"надоедаю", "надоеда"},
		{"надоедлив", "надоедл"},
		{"надоедят", "надоед"},
		{"надоел", "надоел"},
		{"надоела", "надоел"},
		{"надоели", "надоел"},
		{"надоело", "надоел"},
		{"надоем", "надо"},
		{"надоест", "надоест"},
		{"надоесть", "надоест"},
		{"надолго", "надолг"},
		{"надорвала", "надорва"},
		{"надорвана", "надорва"},
		{"надорванный", "надорва"},
		{"надоть", "надот"},
		{"надписана", "надписа"},
		{"надпись", "надп"},
		{"надписью", "надпис"},
		{"надругается", "надруга"},
		{"надругательство", "надругательств"},
		{"надрывает", "надрыва"},
		{"надрываются", "надрыва"},
		{"надрывающимся", "надрыва"},
		{"надрываясь", "надрыв"},
		{"надсаждаясь", "надсажд"},
		{"надседаются", "надседа"},
		{"надсона", "надсон"},
		{"надстройками", "надстройк"},
		{"надтреснутый", "надтреснут"},
		{"надув", "надув"},
		{"надувает", "надува"},
		{"надувание", "надуван"},
		{"надувать", "надува"},
		{"надует", "над"},
		{"надуется", "над"},
		{"надуешь", "надуеш"},
		{"надула", "надул"},
		{"надуло", "надул"},
		{"надулся", "надул"},
		{"надумавшись", "надума"},
		{"надумаем", "надума"},
		{"надумаете", "надума"},
		{"надумается", "надума"},
		{"надумал", "надума"},
		{"надумался", "надума"},
		{"надуто", "надут"},
		{"надутого", "надут"},
		{"надуть", "надут"},
		{"надушенный", "надушен"},
		{"надушил", "надуш"},
		{"надуют", "над"},
		{"надышаться", "надыша"},
		{"наедине", "наедин"},
		{"наездники", "наездник"},
		{"наездницы", "наездниц"},
		{"наемной", "наемн"},
		{"наехали", "наеха"},
		{"наживать", "нажива"},
		{"наживешь", "наживеш"},
		{"нажил", "наж"},
		{"нажимала", "нажима"},
		{"нажимали", "нажима"},
		{"нажить", "наж"},
		{"нажми", "нажм"},
		{"назад", "назад"},
		{"назади", "назад"},
		{"назвал", "назва"},
		{"назвала", "назва"},
		{"назвали", "назва"},
		{"назван", "назва"},
		{"название", "назван"},
		{"названием", "назван"},
		{"названии", "назван"},
		{"названий", "назван"},
		{"названия", "назван"},
		{"названья", "назван"},
		{"назвать", "назва"},
		{"наземь", "назем"},
		{"назидание", "назидан"},
		{"назидания", "назидан"},
		{"назидательно", "назидательн"},
		{"назло", "назл"},
		{"назначавшуюся", "назнача"},
		{"назначаемые", "назнача"},
		{"назначал", "назнача"},
		{"назначалась", "назнача"},
		{"назначали", "назнача"},
		{"назначались", "назнача"},
		{"назначался", "назнача"},
		{"назначать", "назнача"},
		{"назначаю", "назнача"},
		{"назначение", "назначен"},
		{"назначением", "назначен"},
		{"назначении", "назначен"},
		{"назначения", "назначен"},
		{"назначенного", "назначен"},
		{"назначенною", "назначен"},
		{"назначенную", "назначен"},
		{"назначенным", "назначен"},
		{"назначено", "назнач"},
		{"назначены", "назнач"},
		{"назначил", "назнач"},
		{"назначила", "назнач"},
		{"назначить", "назнач"},
		{"назначу", "назнач"},
		{"назовет", "назовет"},
		{"назовете", "назовет"},
		{"назовешь", "назовеш"},
		{"назови", "назов"},
		{"назовут", "назовут"},
		{"назойливому", "назойлив"},
		{"назойливость", "назойлив"},
		{"назойливостью", "назойлив"},
		{"называвшихся", "называ"},
		{"называем", "называ"},
		{"называемая", "называ"},
		{"называемое", "называ"},
		{"называемой", "называ"},
		{"называемую", "называ"},
		{"называемые", "называ"},
		{"называемый", "называ"},
		{"называет", "называ"},
		{"называете", "называ"},
		{"называется", "называ"},
		{"называешь", "называ"},
		{"называл", "называ"},
		{"называла", "называ"},
		{"называли", "называ"},
		{"называлось", "называ"},
		{"называть", "называ"},
		{"называться", "называ"},
		{"называю", "называ"},
		{"называют", "называ"},
		{"называя", "назыв"},
		{"назюзюкался", "назюзюка"},
		{"наибесполезнейшую", "наибесполезн"},
		{"наибеспрерывнейшие", "наибеспрерывн"},
		{"наиблагороднейшего", "наиблагородн"},
		{"наиболее", "наибол"},
		{"наивен", "наив"},
		{"наивная", "наивн"},
		{"наивно", "наивн"},
		{"наивности", "наивност"},
		{"наивною", "наивн"},
		{"наивные", "наивн"},
		{"наивный", "наивн"},
		{"наивным", "наивн"},
		{"наивных", "наивн"},
		{"наигрывает", "наигрыва"},
		{"наигрывая", "наигрыв"},
		{"наизнанку", "наизнанк"},
		{"наизусть", "наизуст"},
		{"наименьшим", "наименьш"},
		{"наиполезнейший", "наиполезн"},
		{"наиприличнейшая", "наиприличн"},
		{"наискось", "наиск"},
		{"наитий", "наит"},
		{"наихитрейшим", "наихитр"},
		{"найдем", "найд"},
		{"найден", "найд"},
		{"найдена", "найд"},
		{"найдено", "найд"},
		{"найдены", "найд"},
		{"найдет", "найдет"},
		{"найдете", "найдет"},
		{"найдется", "найдет"},
		{"найдешь", "найдеш"},
		{"найди", "найд"},
		{"найдите", "найд"},
		{"найду", "найд"},
		{"найдут", "найдут"},
		{"найдутся", "найдут"},
		{"найдя", "найд"},
		{"найма", "найм"},
		{"наймет", "наймет"},
		{"наймешь", "наймеш"},
		{"найми", "найм"},
		{"наймите", "найм"},
		{"найму", "найм"},
		{"найти", "найт"},
		{"накажет", "накажет"},
		{"наказал", "наказа"},
		{"наказали", "наказа"},
		{"наказана", "наказа"},
		{"наказание", "наказан"},
		{"наказания", "наказан"},
		{"наказанье", "наказан"},
		{"наказать", "наказа"},
		{"наказов", "наказ"},
		{"наказывает", "наказыва"},
		{"наказывай", "наказыва"},
		{"наказывал", "наказыва"},
		{"наказывали", "наказыва"},
		{"накануне", "наканун"},
		{"накапало", "накапа"},
		{"накаркают", "накарка"},
		{"накидывает", "накидыва"},
		{"накинет", "накинет"},
		{"накинете", "накинет"},
		{"накинув", "накинув"},
		{"накинул", "накинул"},
		{"накинула", "накинул"},
		{"накинулась", "накинул"},
		{"накинулся", "накинул"},
		{"накинута", "накинут"},
		{"накинутом", "накинут"},
		{"накинутся", "накинут"},
		{"накипала", "накипа"},
		{"накипевшую", "накипевш"},
		{"накипят", "накип"},
		{"накладет", "накладет"},
		{"накладывает", "накладыва"},
		{"накладывала", "накладыва"},
		{"накладывать", "накладыва"},
		{"наклевывалась", "наклевыва"},
		{"наклеенное", "наклеен"},
		{"наклеены", "накле"},
		{"наклепал", "наклепа"},
		{"наклонен", "наклон"},
		{"наклоненная", "наклонен"},
		{"наклоненной", "наклонен"},
		{"наклонив", "наклон"},
		{"наклонившегося", "наклон"},
		{"наклонившись", "наклон"},
		{"наклонил", "наклон"},
		{"наклонила", "наклон"},
		{"наклонилась", "наклон"},
		{"наклонился", "наклон"},
		{"наклонится", "наклон"},
		{"наклониться", "наклон"},
		{"наклонностей", "наклон"},
		{"наклонности", "наклон"},
		{"наклонность", "наклон"},
		{"наклонностях", "наклон"},
		{"наклонною", "наклон"},
		{"наклонны", "наклон"},
		{"наклоном", "наклон"},
		{"наклонялся", "наклоня"},
		{"наклонясь", "наклон"},
		{"наклоняя", "наклон"},
		{"наклоняясь", "наклон"},
		{"наколем", "накол"},
		{"наколет", "наколет"},
		{"наконец", "наконец"},
		{"накопившиеся", "накоп"},
		{"накопилось", "накоп"},
		{"накоплялась", "накопля"},
		{"накорми", "накорм"},
		{"накормила", "накорм"},
		{"накормить", "накорм"},
		{"накормлю", "накормл"},
		{"накормят", "накорм"},
		{"накось", "нак"},
		{"накошенною", "накошен"},
		{"накрахмаленные", "накрахмален"},
		{"накрахмалены", "накрахмал"},
		{"накрепко", "накрепк"},
		{"накрест", "накрест"},
		{"накричит", "накрич"},
		{"накроют", "накроют"},
		{"накрывавшегося", "накрыва"},
		{"накрывает", "накрыва"},
		{"накрываете", "накрыва"},
		{"накрывай", "накрыва"},
		{"накрывался", "накрыва"},
		{"накрывают", "накрыва"},
		{"накрыл", "накр"},
		{"накрыла", "накр"},
		{"накрыли", "накр"},
		{"накрылся", "накр"},
		{"накрыт", "накр"},
		{"накрытый", "накрыт"},
		{"накрыть", "накр"},
		{"накуплю", "накупл"},
		{"накурено", "накур"},
		{"накурил", "накур"},
		{"накуролесили", "накуролес"},
		{"налагает", "налага"},
		{"налагал", "налага"},
		{"налагают", "налага"},
		{"наладили", "налад"},
		{"наладит", "налад"},
		{"налгал", "налга"},
		{"налево", "налев"},
		{"налег", "налег"},
		{"налегая", "налег"},
		{"налегке", "налегк"},
		{"належал", "належа"},
		{"налезло", "налезл"},
		{"налепит", "налеп"},
		{"налета", "налет"},
		{"налетевшим", "налетевш"},
		{"налетел", "налетел"},
		{"налетели", "налетел"},
		{"налетело", "налетел"},
		{"налетит", "налет"},
		{"наливает", "налива"},
		{"наливается", "налива"},
		{"наливай", "налива"},
		{"наливал", "налива"},
		{"наливала", "налива"},
		{"наливая", "налив"},
		{"наливка", "наливк"},
		{"наливной", "наливн"},
		{"нализаться", "нализа"},
		{"налил", "нал"},
		{"налиновал", "налинова"},
		{"налитыми", "налит"},
		{"налить", "нал"},
		{"налицо", "налиц"},
		{"наличности", "наличн"},
		{"наличных", "наличн"},
		{"наложенное", "наложен"},
		{"наложил", "налож"},
		{"наложила", "налож"},
		{"наложницей", "наложниц"},
		{"налью", "нал"},
		{"налюбуется", "налюб"},
		{"налюбуешься", "налюбуеш"},
		{"налюбуются", "налюб"},
		{"нам", "нам"},
		{"намазанных", "намаза"},
		{"намахивается", "намахива"},
		{"намахнулся", "намахнул"},
		{"намачивая", "намачив"},
		{"намашет", "намашет"},
		{"намедни", "намедн"},
		{"намеднишнему", "намеднишн"},
		{"намек", "намек"},
		{"намека", "намек"},
		{"намекавший", "намека"},
		{"намекает", "намека"},
		{"намекаете", "намека"},
		{"намекал", "намека"},
		{"намекала", "намека"},
		{"намекам", "намек"},
		{"намеками", "намек"},
		{"намекать", "намека"},
		{"намекают", "намека"},
		{"намеке", "намек"},
		{"намеки", "намек"},
		{"намекнул", "намекнул"},
		{"намеков", "намек"},
		{"намеком", "намек"},
		{"намелет", "намелет"},
		{"намелко", "намелк"},
		{"наменял", "наменя"},
		{"намеревался", "намерева"},
		{"намереваясь", "намерев"},
		{"намерен", "намер"},
		{"намерена", "намер"},
		{"намерение", "намерен"},
		{"намерением", "намерен"},
		{"намерении", "намерен"},
		{"намерений", "намерен"},
		{"намерению", "намерен"},
		{"намерения", "намерен"},
		{"намерениям", "намерен"},
		{"намерениями", "намерен"},
		{"намерениях", "намерен"},
		{"намеренно", "намерен"},
		{"намерены", "намер"},
		{"намерила", "намер"},
		{"наметался", "намета"},
		{"наметил", "намет"},
		{"намечтать", "намечта"},
		{"нами", "нам"},
		{"намок", "намок"},
		{"наморщивши", "наморщ"},
		{"намотал", "намота"},
		{"намочил", "намоч"},
		{"нанес", "нанес"},
		{"нанесенная", "нанесен"},
		{"нанесенной", "нанесен"},
		{"нанесет", "нанесет"},
		{"нанести", "нанест"},
		{"нанимаете", "нанима"},
		{"нанимал", "нанима"},
		{"нанималась", "нанима"},
		{"нанимать", "нанима"},
		{"нанимают", "нанима"},
		{"нанкового", "нанков"},
		{"наносил", "нанос"},
		{"наносит", "нанос"},
		{"наносный", "наносн"},
		{"наносным", "наносн"},
		{"нанял", "наня"},
		{"наняла", "наня"},
		{"наняли", "наня"},
		{"нанялся", "наня"},
		{"нанята", "нанят"},
		{"нанятой", "нанят"},
		{"нанять", "наня"},
		{"наобещал", "наобеща"},
		{"наоборот", "наоборот"},
		{"наобум", "наобум"},
		{"наотмашь", "наотмаш"},
		{"наотрез", "наотрез"},
		{"нападает", "напада"},
		{"нападал", "напада"},
		{"нападая", "напад"},
		{"нападение", "нападен"},
		{"напал", "напа"},
		{"напали", "напа"},
		{"напасешься", "напасеш"},
		{"напасть", "напаст"},
		{"напев", "нап"},
		{"напева", "напев"},
		{"напевает", "напева"},
		{"напевах", "напев"},
		{"напевают", "напева"},
		{"напевая", "напев"},
		{"напевы", "напев"},
		{"напекли", "напекл"},
		{"напекут", "напекут"},
		{"напел", "напел"},
		{"напер", "напер"},
		{"наперво", "наперв"},
		{"наперебив", "напереб"},
		{"наперед", "наперед"},
		{"наперекор", "наперекор"},
		{"наперерыв", "наперер"},
		{"напечатали", "напечата"},
		{"напечатана", "напечата"},
		{"напечатанным", "напечата"},
		{"напечатано", "напечата"},
		{"напечатать", "напечата"},
		{"напечет", "напечет"},
		{"напивались", "напива"},
		{"напивался", "напива"},
		{"напиваться", "напива"},
		{"напиваюсь", "напива"},
		{"напившись", "нап"},
		{"напилась", "нап"},
		{"напился", "нап"},
		{"напирали", "напира"},
		{"напирать", "напира"},
		{"напирают", "напира"},
		{"напирая", "напир"},
		{"написав", "написа"},
		{"написал", "написа"},
		{"написала", "написа"},
		{"написали", "написа"},
		{"написанная", "написа"},
		{"написанного", "написа"},
		{"написанное", "написа"},
		{"написанной", "написа"},
		{"написанную", "написа"},
		{"написанный", "написа"},
		{"написано", "написа"},
		{"написаны", "написа"},
		{"написать", "написа"},
		{"напитанная", "напита"},
		{"напитку", "напитк"},
		{"напиток", "напиток"},
		{"напитывается", "напитыва"},
		{"напиться", "нап"},
		{"напиханном", "напиха"},
		{"напишет", "напишет"},
		{"напишешь", "напишеш"},
		{"напиши", "напиш"},
		{"напишите", "напиш"},
		{"напишу", "напиш"},
		{"напишут", "напишут"},
		{"наплакались", "наплака"},
		{"наплевал", "наплева"},
		{"наплевать", "наплева"},
		{"наплывают", "наплыва"},
		{"наплясался", "напляса"},
		{"наподличал", "наподлича"},
		{"напоенного", "напоен"},
		{"напоил", "напо"},
		{"напоила", "напо"},
		{"напоили", "напо"},
		{"напой", "нап"},
		{"напоказ", "напоказ"},
		{"наполеон", "наполеон"},
		{"наполеона", "наполеон"},
		{"наполеонами", "наполеон"},
		{"наполеоновской", "наполеоновск"},
		{"наполеоном", "наполеон"},
		{"наползают", "наполза"},
		{"наползет", "наползет"},
		{"наполнена", "наполн"},
		{"наполненная", "наполнен"},
		{"наполненной", "наполнен"},
		{"наполненные", "наполнен"},
		{"наполненный", "наполнен"},
		{"наполненным", "наполнен"},
		{"наполненных", "наполнен"},
		{"наполнено", "наполн"},
		{"наполнены", "наполн"},
		{"наполнил", "наполн"},
		{"наполнилась", "наполн"},
		{"наполнился", "наполн"},
		{"наполнит", "наполн"},
		{"наполнится", "наполн"},
		{"наполнить", "наполн"},
		{"наполню", "наполн"},
		{"наполняемы", "наполняем"},
		{"наполняется", "наполня"},
		{"наполнял", "наполня"},
		{"наполнялись", "наполня"},
		{"наполняло", "наполня"},
		{"наполнялось", "наполня"},
		{"наполнялся", "наполня"},
		{"наполнять", "наполня"},
		{"наполняют", "наполня"},
		{"наполняя", "наполн"},
		{"наполовину", "наполовин"},
		{"напомадил", "напомад"},
		{"напомадился", "напомад"},
		{"напомаженных", "напомажен"},
		{"напоминавших", "напомина"},
		{"напоминает", "напомина"},
		{"напоминай", "напомина"},
		{"напоминал", "напомина"},
		{"напоминало", "напомина"},
		{"напоминания", "напоминан"},
		{"напоминать", "напомина"},
		{"напоминаю", "напомина"},
		{"напоминают", "напомина"},
		{"напоминающий", "напомина"},
		{"напоминая", "напомин"},
		{"напомни", "напомн"},
		{"напомнив", "напомн"},
		{"напомнил", "напомн"},
		{"напомнила", "напомн"},
		{"напомнилась", "напомн"},
		{"напомнило", "напомн"},
		{"напомнит", "напомн"},
		{"напомните", "напомн"},
		{"напомнить", "напомн"},
		{"напор", "напор"},
		{"напора", "напор"},
		{"напорется", "напорет"},
		{"напою", "нап"},
		{"напоят", "напо"},
		{"направил", "направ"},
		{"направились", "направ"},
		{"направился", "направ"},
		{"направить", "направ"},
		{"направиться", "направ"},
		{"направлен", "направл"},
		{"направление", "направлен"},
		{"направлений", "направлен"},
		{"направлению", "направлен"},
		{"направления", "направлен"},
		{"направленных", "направлен"},
		{"направлено", "направл"},
		{"направлю", "направл"},
		{"направляется", "направля"},
		{"направляют", "направля"},
		{"направляясь", "направл"},
		{"направо", "направ"},
		{"направятся", "направ"},
		{"напрасна", "напрасн"},
		{"напрасная", "напрасн"},
		{"напрасно", "напрасн"},
		{"напрасной", "напрасн"},
		{"напрасные", "напрасн"},
		{"напрасный", "напрасн"},
		{"напрасных", "напрасн"},
		{"напрашиваетесь", "напрашива"},
		{"напрашивается", "напрашива"},
		{"напрашиваются", "напрашива"},
		{"например", "например"},
		{"напролет", "напролет"},
		{"напросто", "напрост"},
		{"напротив", "напрот"},
		{"напрягает", "напряга"},
		{"напрягал", "напряга"},
		{"напрягались", "напряга"},
		{"напрягаю", "напряга"},
		{"напрягая", "напряг"},
		{"напрягаясь", "напряг"},
		{"напрягутся", "напрягут"},
		{"напрягши", "напрягш"},
		{"напряжение", "напряжен"},
		{"напряжением", "напряжен"},
		{"напряжения", "напряжен"},
		{"напряженная", "напряжен"},
		{"напряженно", "напряжен"},
		{"напряженное", "напряжен"},
		{"напряженной", "напряжен"},
		{"напряженном", "напряжен"},
		{"напряженный", "напряжен"},
		{"напряженным", "напряжен"},
		{"напряжены", "напряж"},
		{"напугал", "напуга"},
		{"напугали", "напуга"},
		{"напугались", "напуга"},
		{"напугана", "напуга"},
		{"напуганные", "напуга"},
		{"напугано", "напуга"},
		{"напугать", "напуга"},
		{"напудренные", "напудрен"},
		{"напужались", "напужа"},
		{"напускное", "напускн"},
		{"напускные", "напускн"},
		{"напускным", "напускн"},
		{"напустило", "напуст"},
		{"напустился", "напуст"},
		{"напутает", "напута"},
		{"напутаешь", "напута"},
		{"напутствие", "напутств"},
		{"напутствии", "напутств"},
		{"напущенные", "напущен"},
		{"напьется", "напьет"},
		{"напьюсь", "нап"},
		{"наравне", "наравн"},
		{"нараспашку", "нараспашк"},
		{"нарастала", "нараста"},
		{"нарастали", "нараста"},
		{"нарастание", "нарастан"},
		{"нарастающие", "нараста"},
		{"нарах", "нар"},
		{"нарву", "нарв"},
		{"нарезался", "нареза"},
		{"нарисовался", "нарисова"},
		{"нарисованного", "нарисова"},
		{"нарисованные", "нарисова"},
		{"нарисовано", "нарисова"},
		{"нарисовать", "нарисова"},
		{"наркотических", "наркотическ"},
		{"народ", "народ"},
		{"народа", "народ"},
		{"народе", "народ"},
		{"народец", "народец"},
		{"народницей", "народниц"},
		{"народной", "народн"},
		{"народный", "народн"},
		{"народов", "народ"},
		{"народовольческого", "народовольческ"},
		{"народом", "народ"},
		{"народонаселения", "народонаселен"},
		{"народу", "народ"},
		{"народы", "народ"},
		{"народятся", "народ"},
		{"нарочно", "нарочн"},
		{"наруже", "наруж"},
		{"наружная", "наружн"},
		{"наружно", "наружн"},
		{"наружного", "наружн"},
		{"наружное", "наружн"},
		{"наружной", "наружн"},
		{"наружности", "наружн"},
		{"наружность", "наружн"},
		{"наружностью", "наружн"},
		{"наружную", "наружн"},
		{"наружные", "наружн"},
		{"наружу", "наруж"},
		{"нарукавнички", "нарукавничк"},
		{"нарумяненную", "нарумянен"},
		{"нарушаемого", "наруша"},
		{"нарушает", "наруша"},
		{"нарушал", "наруша"},
		{"нарушалась", "наруша"},
		{"нарушали", "наруша"},
		{"нарушало", "наруша"},
		{"нарушалось", "наруша"},
		{"нарушать", "наруша"},
		{"нарушая", "наруш"},
		{"нарушена", "наруш"},
		{"нарушение", "нарушен"},
		{"нарушения", "нарушен"},
		{"нарушено", "наруш"},
		{"нарушил", "наруш"},
		{"нарушилось", "наруш"},
		{"нарушился", "наруш"},
		{"нарушит", "наруш"},
		{"нарушителей", "нарушител"},
		{"нарушится", "наруш"},
		{"нарушить", "наруш"},
		{"нарушу", "наруш"},
		{"нарцизов", "нарциз"},
		{"нарыв", "нар"},
		{"нарывавший", "нарыва"},
		{"нарывает", "нарыва"},
		{"наряд", "наряд"},
		{"наряда", "наряд"},
		{"наряде", "наряд"},
		{"наряди", "наряд"},
		{"нарядить", "наряд"},
		{"нарядная", "нарядн"},
		{"нарядно", "нарядн"},
		{"нарядной", "нарядн"},
		{"нарядом", "наряд"},
		{"наряжается", "наряжа"},
		{"наряжалась", "наряжа"},
		{"наряжаются", "наряжа"},
		{"нас", "нас"},
		{"насадил", "насад"},
		{"насадим", "насад"},
		{"насвистывает", "насвистыва"},
		{"наседка", "наседк"},
		{"наседке", "наседк"},
		{"наседки", "наседк"},
		{"наседку", "наседк"},
		{"насекомое", "насеком"},
		{"насел", "насел"},
		{"население", "населен"},
		{"населения", "населен"},
		{"населенная", "населен"},
		{"населенные", "населен"},
		{"населились", "насел"},
		{"населило", "насел"},
		{"насердке", "насердк"},
		{"насидишься", "насид"},
		{"насилие", "насил"},
		{"насилием", "насил"},
		{"насилия", "насил"},
		{"насиловать", "насилова"},
		{"насилу", "насил"},
		{"насильно", "насильн"},
		{"насильственно", "насильствен"},
		{"насильственной", "насильствен"},
		{"насильственною", "насильствен"},
		{"насильственные", "насильствен"},
		{"насильственным", "насильствен"},
		{"насилья", "насил"},
		{"насказала", "насказа"},
		{"насквозь", "насквоз"},
		{"насколько", "наскольк"},
		{"наскоро", "наскор"},
		{"наскочила", "наскоч"},
		{"наскучивало", "наскучива"},
		{"наскучил", "наскуч"},
		{"наскучу", "наскуч"},
		{"наслаждается", "наслажда"},
		{"наслаждайся", "наслажда"},
		{"наслаждайтесь", "наслажда"},
		{"наслаждалась", "наслажда"},
		{"наслаждались", "наслажда"},
		{"наслаждался", "наслажда"},
		{"наслаждаться", "наслажда"},
		{"наслаждаюсь", "наслажда"},
		{"наслаждаются", "наслажда"},
		{"наслаждаясь", "наслажд"},
		{"наслаждение", "наслажден"},
		{"наслаждением", "наслажден"},
		{"наслаждений", "наслажден"},
		{"наслаждения", "наслажден"},
		{"наследникам", "наследник"},
		{"наследники", "наследник"},
		{"наследнику", "наследник"},
		{"наследовавшие", "наследова"},
		{"наследовал", "наследова"},
		{"наследства", "наследств"},
		{"наследственной", "наследствен"},
		{"наследство", "наследств"},
		{"наследству", "наследств"},
		{"наслышан", "наслыша"},
		{"наслышка", "наслышк"},
		{"насмеемся", "насме"},
		{"насмерть", "насмерт"},
		{"насмехается", "насмеха"},
		{"насмехаюсь", "насмеха"},
		{"насмешек", "насмешек"},
		{"насмешил", "насмеш"},
		{"насмешите", "насмеш"},
		{"насмешка", "насмешк"},
		{"насмешками", "насмешк"},
		{"насмешки", "насмешк"},
		{"насмешкой", "насмешк"},
		{"насмешку", "насмешк"},
		{"насмешлив", "насмешл"},
		{"насмешливая", "насмешлив"},
		{"насмешливее", "насмешлив"},
		{"насмешливо", "насмешлив"},
		{"насмешливого", "насмешлив"},
		{"насмешливое", "насмешлив"},
		{"насмешливой", "насмешлив"},
		{"насмешливости", "насмешлив"},
		{"насмешливый", "насмешлив"},
		{"насмешливыми", "насмешлив"},
		{"насмешливых", "насмешлив"},
		{"насмеюсь", "насм"},
		{"насмеялся", "насмея"},
		{"насмотрелся", "насмотрел"},
		{"насолить", "насол"},
		{"наспал", "наспа"},
		{"наставал", "настава"},
		{"наставало", "настава"},
		{"наставил", "настав"},
		{"наставительно", "наставительн"},
		{"наставлений", "наставлен"},
		{"наставлениях", "наставлен"},
		{"наставлено", "наставл"},
		{"наставляй", "наставля"},
		{"наставница", "наставниц"},
		{"настает", "наста"},
		{"настаивавший", "настаива"},
		{"настаиваем", "настаива"},
		{"настаивает", "настаива"},
		{"настаиваете", "настаива"},
		{"настаивал", "настаива"},
		{"настаивала", "настаива"},
		{"настаивать", "настаива"},
		{"настаиваю", "настаива"},
		{"настал", "наста"},
		{"настала", "наста"},
		{"настали", "наста"},
		{"настало", "наста"},
		{"настанет", "настанет"},
		{"настанут", "настанут"},
		{"настасье", "настас"},
		{"настасьей", "настас"},
		{"настасьи", "настас"},
		{"настасью", "настас"},
		{"настасьюшка", "настасьюшк"},
		{"настасья", "настас"},
		{"настежь", "настеж"},
		{"настенька", "настеньк"},
		{"настигающую", "настига"},
		{"настлали", "настла"},
		{"настоенной", "настоен"},
		{"настоит", "насто"},
		{"настоится", "насто"},
		{"настойчива", "настойчив"},
		{"настойчиво", "настойчив"},
		{"настойчивого", "настойчив"},
		{"настойчивое", "настойчив"},
		{"настойчивости", "настойчив"},
		{"настойчивость", "настойчив"},
		{"настойчивостью", "настойчив"},
		{"настойчивые", "настойчив"},
		{"настойчивый", "настойчив"},
		{"настойчивым", "настойчив"},
		{"настолько", "настольк"},
		{"настоял", "настоя"},
		{"настояла", "настоя"},
		{"настоянию", "настоян"},
		{"настояно", "настоя"},
		{"настоятельно", "настоятельн"},
		{"настоятельную", "настоятельн"},
		{"настоящая", "настоя"},
		{"настоящего", "настоя"},
		{"настоящее", "настоя"},
		{"настоящей", "настоя"},
		{"настоящем", "настоя"},
		{"настоящему", "настоя"},
		{"настоящею", "настоя"},
		{"настоящие", "настоя"},
		{"настоящий", "настоя"},
		{"настоящим", "настоя"},
		{"настоящих", "настоя"},
		{"настоящую", "настоя"},
		{"настраивает", "настраива"},
		{"настраивается", "настраива"},
		{"настрого", "настр"},
		{"настроение", "настроен"},
		{"настроением", "настроен"},
		{"настроении", "настроен"},
		{"настроению", "настроен"},
		{"настроения", "настроен"},
		{"настроено", "настро"},
		{"настроены", "настро"},
		{"настроим", "настро"},
		{"настроит", "настро"},
		{"настроить", "настро"},
		{"настройщика", "настройщик"},
		{"настрочить", "настроч"},
		{"настудишь", "настуд"},
		{"настудят", "настуд"},
		{"наступает", "наступа"},
		{"наступала", "наступа"},
		{"наступают", "наступа"},
		{"наступивший", "наступ"},
		{"наступил", "наступ"},
		{"наступила", "наступ"},
		{"наступили", "наступ"},
		{"наступило", "наступ"},
		{"наступит", "наступ"},
		{"наступлением", "наступлен"},
		{"насущного", "насущн"},
		{"насущной", "насущн"},
		{"насущные", "насущн"},
		{"насущным", "насущн"},
		{"насущными", "насущн"},
		{"насчет", "насчет"},
		{"насылает", "насыла"},
		{"насылая", "насыл"},
		{"насыпь", "насып"},
		{"насыпью", "насып"},
		{"насядет", "насядет"},
		{"наталия", "натал"},
		{"натальи", "натал"},
		{"наталья", "натал"},
		{"наташа", "наташ"},
		{"наташе", "наташ"},
		{"наташей", "наташ"},
		{"наташи", "наташ"},
		{"наташу", "наташ"},
		{"натащил", "натащ"},
		{"натащить", "натащ"},
		{"нате", "нат"},
		{"натекла", "натекл"},
		{"натерпелась", "натерпел"},
		{"натерпелся", "натерпел"},
		{"натертому", "натерт"},
		{"наткнулся", "наткнул"},
		{"натолкнуть", "натолкнут"},
		{"натолочь", "натолоч"},
		{"натопленных", "натоплен"},
		{"натоптали", "натопта"},
		{"наточи", "наточ"},
		{"натруженный", "натружен"},
		{"натугой", "натуг"},
		{"натура", "натур"},
		{"натурализме", "натурализм"},
		{"натуральнее", "натуральн"},
		{"натурально", "натуральн"},
		{"натуральности", "натуральн"},
		{"натуральную", "натуральн"},
		{"натуральны", "натуральн"},
		{"натуре", "натур"},
		{"натурой", "натур"},
		{"натуру", "натур"},
		{"натуры", "натур"},
		{"натыкается", "натыка"},
		{"натыкался", "натыка"},
		{"натягивает", "натягива"},
		{"натягивать", "натягива"},
		{"натяжек", "натяжек"},
		{"натянул", "натянул"},
		{"натянулись", "натянул"},
		{"натянутая", "натянут"},
		{"натянуто", "натянут"},
		{"натянутый", "натянут"},
		{"наугад", "наугад"},
		{"наук", "наук"},
		{"наука", "наук"},
		{"науках", "наук"},
		{"науке", "наук"},
		{"науки", "наук"},
		{"наукой", "наук"},
		{"науку", "наук"},
		{"наумыч", "наумыч"},
		{"наутро", "наутр"},
		{"научи", "науч"},
		{"научил", "науч"},
		{"научила", "науч"},
		{"научило", "науч"},
		{"научился", "науч"},
		{"научимся", "науч"},
		{"научит", "науч"},
		{"научите", "науч"},
		{"научить", "науч"},
		{"научные", "научн"},
		{"научных", "научн"},
		{"научусь", "науч"},
		{"наущения", "наущен"},
		{"нафталина", "нафталин"},
		{"нафталином", "нафталин"},
		{"нахал", "наха"},
		{"нахально", "нахальн"},
		{"нахальное", "нахальн"},
		{"нахальный", "нахальн"},
		{"нахальным", "нахальн"},
		{"нахальства", "нахальств"},
		{"нахальство", "нахальств"},
		{"нахальством", "нахальств"},
		{"нахватает", "нахвата"},
		{"нахватают", "нахвата"},
		{"нахлебник", "нахлебник"},
		{"нахлестался", "нахлеста"},
		{"нахлобучив", "нахлобуч"},
		{"нахмуренно", "нахмурен"},
		{"нахмуренной", "нахмурен"},
		{"нахмуренные", "нахмурен"},
		{"нахмуренный", "нахмурен"},
		{"нахмуренным", "нахмурен"},
		{"нахмуренными", "нахмурен"},
		{"нахмурив", "нахмур"},
		{"нахмурившись", "нахмур"},
		{"нахмурил", "нахмур"},
		{"нахмурила", "нахмур"},
		{"нахмурился", "нахмур"},
		{"нахмурясь", "нахмур"},
		{"находившейся", "наход"},
		{"находил", "наход"},
		{"находила", "наход"},
		{"находилась", "наход"},
		{"находили", "наход"},
		{"находились", "наход"},
		{"находило", "наход"},
		{"находилось", "наход"},
		{"находился", "наход"},
		{"находит", "наход"},
		{"находите", "наход"},
		{"находится", "наход"},
		{"находить", "наход"},
		{"находиться", "наход"},
		{"находишь", "наход"},
		{"находку", "находк"},
		{"находчивости", "находчив"},
		{"находя", "наход"},
		{"находясь", "наход"},
		{"находят", "наход"},
		{"находятся", "наход"},
		{"находящий", "находя"},
		{"нахожу", "нахож"},
		{"нацарапает", "нацарапа"},
		{"нации", "нац"},
		{"наций", "нац"},
		{"национальности", "национальн"},
		{"национальные", "национальн"},
		{"нацию", "нац"},
		{"нациями", "нац"},
		{"начавшегося", "нача"},
		{"начавшей", "нача"},
		{"начавши", "нача"},
		{"начавший", "нача"},
		{"начал", "нача"},
		{"начала", "нача"},
		{"началами", "начал"},
		{"началась", "нача"},
		{"начале", "начал"},
		{"начали", "нача"},
		{"начались", "нача"},
		{"начало", "нача"},
		{"началом", "начал"},
		{"началось", "нача"},
		{"начался", "нача"},
		{"начальник", "начальник"},
		{"начальника", "начальник"},
		{"начальнике", "начальник"},
		{"начальники", "начальник"},
		{"начальником", "начальник"},
		{"начальнику", "начальник"},
		{"начальница", "начальниц"},
		{"начальницей", "начальниц"},
		{"начальницу", "начальниц"},
		{"начальства", "начальств"},
		{"начальство", "начальств"},
		{"начальством", "начальств"},
		{"начальству", "начальств"},
		{"начатки", "начатк"},
		{"начатое", "начат"},
		{"начатые", "начат"},
		{"начать", "нача"},
		{"начерно", "начерн"},
		{"начертал", "начерта"},
		{"начертана", "начерта"},
		{"начертанной", "начерта"},
		{"начертанный", "начерта"},
		{"начертанных", "начерта"},
		{"начертать", "начерта"},
		{"начертил", "начерт"},
		{"начесть", "начест"},
		{"начинавшаяся", "начина"},
		{"начинавшее", "начина"},
		{"начинавшейся", "начина"},
		{"начинавший", "начина"},
		{"начинаем", "начина"},
		{"начинает", "начина"},
		{"начинаете", "начина"},
		{"начинается", "начина"},
		{"начинаешь", "начина"},
		{"начинай", "начина"},
		{"начинайте", "начина"},
		{"начинал", "начина"},
		{"начинала", "начина"},
		{"начиналась", "начина"},
		{"начинали", "начина"},
		{"начинались", "начина"},
		{"начинало", "начина"},
		{"начиналось", "начина"},
		{"начинался", "начина"},
		{"начинаний", "начинан"},
		{"начинать", "начина"},
		{"начинаю", "начина"},
		{"начинают", "начина"},
		{"начинающая", "начина"},
		{"начинающего", "начина"},
		{"начинающий", "начина"},
		{"начинающимся", "начина"},
		{"начиная", "начин"},
		{"начиненные", "начинен"},
		{"начинки", "начинк"},
		{"начинкой", "начинк"},
		{"начисто", "начист"},
		{"начистоту", "начистот"},
		{"начнем", "начн"},
		{"начнет", "начнет"},
		{"начнете", "начнет"},
		{"начнется", "начнет"},
		{"начнешь", "начнеш"},
		{"начни", "начн"},
		{"начну", "начн"},
		{"начнут", "начнут"},
		{"начнутся", "начнут"},
		{"начудила", "начуд"},
		{"наш", "наш"},
		{"наша", "наш"},
		{"наше", "наш"},
		{"нашего", "наш"},
		{"нашей", "наш"},
		{"нашел", "нашел"},
		{"нашелся", "нашел"},
		{"нашем", "наш"},
		{"нашему", "наш"},
		{"нашептывает", "нашептыва"},
		{"нашептывал", "нашептыва"},
		{"нашествие", "нашеств"},
		{"нашествий", "нашеств"},
		{"наши", "наш"},
		{"нашим", "наш"},
		{"нашими", "наш"},
		{"наших", "наш"},
		{"нашла", "нашл"},
		{"нашлась", "нашл"},
		{"нашли", "нашл"},
		{"нашлись", "нашл"},
		{"нашло", "нашл"},
		{"нашлось", "нашл"},
		{"нашу", "наш"},
		{"нащипанными", "нащипа"},
		{"нащупав", "нащупа"},
		{"наяву", "наяв"},
		{"наяды", "наяд"},
		{"не", "не"},
		{"неаккуратна", "неаккуратн"},
		{"неаккуратно", "неаккуратн"},
		{"неаполитанский", "неаполитанск"},
		{"неба", "неб"},
		{"небе", "неб"},
		{"небеленая", "небелен"},
		{"небес", "небес"},
		{"небеса", "небес"},
		{"небесам", "небес"},
		{"небесами", "небес"},
		{"небесах", "небес"},
		{"небесная", "небесн"},
		{"небесно", "небесн"},
		{"небесное", "небесн"},
		{"небесной", "небесн"},
		{"небесном", "небесн"},
		{"небесные", "небесн"},
		{"небесных", "небесн"},
		{"небесполезно", "небесполезн"},
		{"неблаговоления", "неблаговолен"},
		{"неблагодарен", "неблагодар"},
		{"неблагодарно", "неблагодарн"},
		{"неблагодарности", "неблагодарн"},
		{"неблагодарность", "неблагодарн"},
		{"неблагодарностью", "неблагодарн"},
		{"неблагодарные", "неблагодарн"},
		{"неблагодарный", "неблагодарн"},
		{"неблагонадежность", "неблагонадежн"},
		{"неблагонамеренных", "неблагонамерен"},
		{"неблагополучно", "неблагополучн"},
		{"неблагоприятному", "неблагоприятн"},
		{"неблагородный", "неблагородн"},
		{"неблагородными", "неблагородн"},
		{"неблагородство", "неблагородств"},
		{"неблагосклонно", "неблагосклон"},
		{"небледнеющих", "небледнеющ"},
		{"небо", "неб"},
		{"небогат", "небогат"},
		{"небогатый", "небогат"},
		{"небожитель", "небожител"},
		{"небойкие", "небойк"},
		{"небойсь", "неб"},
		{"небольшая", "небольш"},
		{"небольшие", "небольш"},
		{"небольшим", "небольш"},
		{"небольшими", "небольш"},
		{"небольшого", "небольш"},
		{"небольшое", "небольш"},
		{"небольшой", "небольш"},
		{"небольшому", "небольш"},
		{"небольшую", "небольш"},
		{"небом", "неб"},
		{"небосклон", "небосклон"},
		{"небосклону", "небосклон"},
		{"небось", "неб"},
		{"небрежен", "небреж"},
		{"небрежнее", "небрежн"},
		{"небрежно", "небрежн"},
		{"небрежное", "небрежн"},
		{"небрежной", "небрежн"},
		{"небрежности", "небрежн"},
		{"небрежностию", "небрежн"},
		{"небрежностью", "небрежн"},
		{"небрежною", "небрежн"},
		{"небрежные", "небрежн"},
		{"небрежный", "небрежн"},
		{"небрежным", "небрежн"},
		{"небритая", "небрит"},
		{"небритую", "небрит"},
		{"небритый", "небрит"},
		{"небу", "неб"},
		{"небывалая", "небывал"},
		{"небывалого", "небывал"},
		{"небывалое", "небывал"},
		{"небывалой", "небывал"},
		{"небывалые", "небывал"},
		{"небывалый", "небывал"},
		{"небывалым", "небывал"},
		{"небывалыми", "небывал"},
		{"небывалых", "небывал"},
		{"небывальщину", "небывальщин"},
		{"небылицу", "небылиц"},
		{"небытия", "небыт"},
		{"нева", "нев"},
		{"неважно", "неважн"},
		{"неважное", "неважн"},
		{"неважной", "неважн"},
		{"неважны", "неважн"},
		{"неважным", "неважн"},
		{"невдомек", "невдомек"},
		{"неве", "нев"},
		{"неведении", "неведен"},
		{"неведения", "неведен"},
		{"неведом", "невед"},
		{"неведомая", "неведом"},
		{"неведомо", "неведом"},
		{"неведомой", "неведом"},
		{"неведомою", "неведом"},
		{"неведомый", "неведом"},
		{"неведомым", "неведом"},
		{"неведомыми", "неведом"},
		{"неведомых", "неведом"},
		{"невежд", "невежд"},
		{"невежды", "невежд"},
		{"невежества", "невежеств"},
		{"невежество", "невежеств"},
		{"невежи", "невеж"},
		{"невежливая", "невежлив"},
		{"невежливо", "невежлив"},
		{"невежливою", "невежлив"},
		{"невелик", "невелик"},
		{"невелика", "невелик"},
		{"невелико", "невелик"},
		{"невеликодушно", "невеликодушн"},
		{"неверие", "невер"},
		{"неверна", "неверн"},
		{"неверно", "неверн"},
		{"неверного", "неверн"},
		{"неверною", "неверн"},
		{"неверный", "неверн"},
		{"неверных", "неверн"},
		{"невероятно", "невероятн"},
		{"невероятности", "невероятн"},
		{"невероятные", "невероятн"},
		{"невероятным", "невероятн"},
		{"неверующий", "невер"},
		{"неверующих", "невер"},
		{"неверьем", "невер"},
		{"невеселая", "невесел"},
		{"невесело", "невесел"},
		{"невеселые", "невесел"},
		{"невеста", "невест"},
		{"невесте", "невест"},
		{"невестка", "невестк"},
		{"невестки", "невестк"},
		{"невестой", "невест"},
		{"невесту", "невест"},
		{"невесты", "невест"},
		{"невесть", "невест"},
		{"невзгод", "невзгод"},
		{"невзначай", "невзнача"},
		{"невзрачный", "невзрачн"},
		{"невиданно", "невида"},
		{"невиданного", "невида"},
		{"невиданной", "невида"},
		{"невиданные", "невида"},
		{"невидимая", "невидим"},
		{"невидимо", "невидим"},
		{"невидимой", "невидим"},
		{"невидимую", "невидим"},
		{"невидимые", "невидим"},
		{"невидимый", "невидим"},
		{"невинен", "невин"},
		{"невинна", "невин"},
		{"невинная", "невин"},
		{"невинно", "невин"},
		{"невинного", "невин"},
		{"невинном", "невин"},
		{"невинному", "невин"},
		{"невинности", "невин"},
		{"невинность", "невин"},
		{"невинностью", "невин"},
		{"невинные", "невин"},
		{"невинный", "невин"},
		{"невинных", "невин"},
		{"невиновных", "невиновн"},
		{"невменяем", "невменя"},
		{"невнимания", "невниман"},
		{"невнимательным", "невнимательн"},
		{"невнятны", "невнятн"},
		{"неводу", "невод"},
		{"невозвратимо", "невозвратим"},
		{"невозвратимый", "невозвратим"},
		{"невозвратно", "невозвратн"},
		{"невозвратного", "невозвратн"},
		{"невозвратном", "невозвратн"},
		{"невозвратные", "невозвратн"},
		{"невозвратным", "невозвратн"},
		{"невоздержания", "невоздержан"},
		{"невозможен", "невозмож"},
		{"невозможно", "невозможн"},
		{"невозможное", "невозможн"},
		{"невозможному", "невозможн"},
		{"невозможности", "невозможн"},
		{"невозможную", "невозможн"},
		{"невозможный", "невозможн"},
		{"невозможным", "невозможн"},
		{"невозможных", "невозможн"},
		{"невозмутимая", "невозмутим"},
		{"невозмутимее", "невозмутим"},
		{"невозмутимо", "невозмутим"},
		{"невозмутимого", "невозмутим"},
		{"невозмутимое", "невозмутим"},
		{"невозмутимый", "невозмутим"},
		{"невозмутимым", "невозмутим"},
		{"невой", "нев"},
		{"неволе", "невол"},
		{"неволей", "невол"},
		{"неволит", "невол"},
		{"невольно", "невольн"},
		{"невольного", "невольн"},
		{"невольное", "невольн"},
		{"невольной", "невольн"},
		{"невольном", "невольн"},
		{"невольною", "невольн"},
		{"невольный", "невольн"},
		{"невольных", "невольн"},
		{"невообразимо", "невообразим"},
		{"невоскресшему", "невоскресш"},
		{"невоспитанный", "невоспита"},
		{"невою", "нев"},
		{"невпопад", "невпопад"},
		{"невредим", "невред"},
		{"невредимо", "невредим"},
		{"невредимого", "невредим"},
		{"невредимый", "невредим"},
		{"невский", "невск"},
		{"невскому", "невск"},
		{"неву", "нев"},
		{"невы", "нев"},
		{"невыгодно", "невыгодн"},
		{"невыкупленные", "невыкуплен"},
		{"невыносим", "невынос"},
		{"невыносима", "невыносим"},
		{"невыносимая", "невыносим"},
		{"невыносимо", "невыносим"},
		{"невыносимое", "невыносим"},
		{"невыносимы", "невыносим"},
		{"невыносимый", "невыносим"},
		{"невыносимыми", "невыносим"},
		{"невыразимая", "невыразим"},
		{"невыразимо", "невыразим"},
		{"невыразимого", "невыразим"},
		{"невыразимое", "невыразим"},
		{"невыразимом", "невыразим"},
		{"невыразимым", "невыразим"},
		{"невысказанном", "невысказа"},
		{"невысоко", "невысок"},
		{"невысокому", "невысок"},
		{"невыстоявшеюся", "невыстоя"},
		{"невысыхаемым", "невысыха"},
		{"невыторгованный", "невыторгова"},
		{"нег", "нег"},
		{"негаданно", "негада"},
		{"негаданной", "негада"},
		{"негде", "негд"},
		{"неге", "нег"},
		{"неги", "нег"},
		{"негибнущая", "негибнущ"},
		{"неглубокой", "неглубок"},
		{"неглуп", "неглуп"},
		{"неглупая", "неглуп"},
		{"неглупый", "неглуп"},
		{"него", "нег"},
		{"неговорлив", "неговорл"},
		{"негодная", "негодн"},
		{"негодном", "негодн"},
		{"негодный", "негодн"},
		{"негодным", "негодн"},
		{"негодование", "негодован"},
		{"негодованием", "негодован"},
		{"негодовании", "негодован"},
		{"негодования", "негодован"},
		{"негодуя", "негоду"},
		{"негодяем", "негодя"},
		{"негодяй", "негодя"},
		{"негодяя", "негод"},
		{"негой", "нег"},
		{"негоцианта", "негоциант"},
		{"неграциозными", "неграциозн"},
		{"негреющий", "негреющ"},
		{"негреющим", "негреющ"},
		{"негромкие", "негромк"},
		{"негромко", "негромк"},
		{"негры", "негр"},
		{"негустые", "негуст"},
		{"недавнего", "недавн"},
		{"недавнее", "недавн"},
		{"недавней", "недавн"},
		{"недавнем", "недавн"},
		{"недавнему", "недавн"},
		{"недавний", "недавн"},
		{"недавних", "недавн"},
		{"недавно", "недавн"},
		{"недавнюю", "недавн"},
		{"недавняя", "недавн"},
		{"недалеко", "недалек"},
		{"недалечко", "недалечк"},
		{"недальней", "недальн"},
		{"недальный", "недальн"},
		{"недаром", "недар"},
		{"недвижимо", "недвижим"},
		{"недворянина", "недворянин"},
		{"неделе", "недел"},
		{"недели", "недел"},
		{"неделикатно", "неделикатн"},
		{"неделикатным", "неделикатн"},
		{"неделовитостью", "неделовит"},
		{"неделовых", "неделов"},
		{"недель", "недел"},
		{"недельку", "недельк"},
		{"неделю", "недел"},
		{"неделя", "недел"},
		{"неделям", "недел"},
		{"недетски", "недетск"},
		{"недетской", "недетск"},
		{"недлинный", "недлин"},
		{"недобор", "недобор"},
		{"недобровольно", "недобровольн"},
		{"недоброе", "недобр"},
		{"недоброжелателен", "недоброжелател"},
		{"недоброжелательством", "недоброжелательств"},
		{"недоваренную", "недоварен"},
		{"недоварены", "недовар"},
		{"недоверием", "недовер"},
		{"недоверия", "недовер"},
		{"недоверчив", "недоверч"},
		{"недоверчива", "недоверчив"},
		{"недоверчиво", "недоверчив"},
		{"недоверчивости", "недоверчив"},
		{"недоверчивость", "недоверчив"},
		{"недоверчивостью", "недоверчив"},
		{"недоверчивые", "недоверчив"},
		{"недоверчивый", "недоверчив"},
		{"недоверчивым", "недоверчив"},
		{"недоволен", "недовол"},
		{"недовольна", "недовольн"},
		{"недовольная", "недовольн"},
		{"недовольны", "недовольн"},
		{"недовольный", "недовольн"},
		{"недовольным", "недовольн"},
		{"недовольства", "недовольств"},
		{"недовольство", "недовольств"},
		{"недогадливый", "недогадлив"},
		{"недоглядел", "недоглядел"},
		{"недогляди", "недогляд"},
		{"недоглядишь", "недогляд"},
		{"недозрелости", "недозрел"},
		{"недоимках", "недоимк"},
		{"недоимки", "недоимк"},
		{"недоконченное", "недокончен"},
		{"недолго", "недолг"},
		{"недоносков", "недоноск"},
		{"недоразумения", "недоразумен"},
		{"недорого", "недор"},
		{"недосказанного", "недосказа"},
		{"недосказанное", "недосказа"},
		{"недоставало", "недостава"},
		{"недостает", "недоста"},
		{"недостало", "недоста"},
		{"недостатка", "недостатк"},
		{"недостатками", "недостатк"},
		{"недостатков", "недостатк"},
		{"недостатком", "недостатк"},
		{"недостаток", "недостаток"},
		{"недостаточно", "недостаточн"},
		{"недостижимой", "недостижим"},
		{"недостоин", "недостоин"},
		{"недостойна", "недостойн"},
		{"недостойному", "недостойн"},
		{"недостойною", "недостойн"},
		{"недостойны", "недостойн"},
		{"недостойный", "недостойн"},
		{"недостойных", "недостойн"},
		{"недоступна", "недоступн"},
		{"недоступно", "недоступн"},
		{"недоступного", "недоступн"},
		{"недоступность", "недоступн"},
		{"недоступную", "недоступн"},
		{"недоступный", "недоступн"},
		{"недосягаем", "недосяга"},
		{"недосягаемо", "недосягаем"},
		{"недосягаемым", "недосяга"},
		{"недосягаемых", "недосяга"},
		{"недотепа", "недотеп"},
		{"недотрогой", "недотрог"},
		{"недоумевал", "недоумева"},
		{"недоумевали", "недоумева"},
		{"недоумевая", "недоумев"},
		{"недоумение", "недоумен"},
		{"недоумением", "недоумен"},
		{"недоумении", "недоумен"},
		{"недоумений", "недоумен"},
		{"недоумения", "недоумен"},
		{"недоумениям", "недоумен"},
		{"недоуменьем", "недоумен"},
		{"недоучившихся", "недоуч"},
		{"недочитанная", "недочита"},
		{"недр", "недр"},
		{"недрах", "недр"},
		{"недремлющая", "недремлющ"},
		{"недрогнувший", "недрогнувш"},
		{"недруги", "недруг"},
		{"недружелюбным", "недружелюбн"},
		{"недуг", "недуг"},
		{"недугами", "недуг"},
		{"недугом", "недуг"},
		{"недурна", "недурн"},
		{"недурно", "недурн"},
		{"нее", "не"},
		{"неестественно", "неестествен"},
		{"неестественною", "неестествен"},
		{"неестественных", "неестествен"},
		{"нежа", "неж"},
		{"нежат", "нежат"},
		{"нежданно", "нежда"},
		{"нежданной", "нежда"},
		{"нежданные", "нежда"},
		{"нежданный", "нежда"},
		{"нежели", "нежел"},
		{"нежен", "неж"},
		{"неженского", "неженск"},
		{"неживой", "нежив"},
		{"неживою", "нежив"},
		{"нежинский", "нежинск"},
		{"нежит", "неж"},
		{"нежить", "неж"},
		{"нежливенькая", "нежливеньк"},
		{"нежна", "нежн"},
		{"нежная", "нежн"},
		{"нежнее", "нежн"},
		{"нежней", "нежн"},
		{"нежнейшего", "нежн"},
		{"нежнейшей", "нежн"},
		{"нежнейшими", "нежн"},
		{"нежно", "нежн"},
		{"нежного", "нежн"},
		{"нежное", "нежн"},
		{"нежной", "нежн"},
		{"нежном", "нежн"},
		{"нежности", "нежност"},
		{"нежность", "нежност"},
		{"нежностью", "нежност"},
		{"нежною", "нежн"},
		{"нежную", "нежн"},
		{"нежны", "нежн"},
		{"нежные", "нежн"},
		{"нежный", "нежн"},
		{"нежным", "нежн"},
		{"нежных", "нежн"},
		{"незабвенная", "незабвен"},
		{"незабвенной", "незабвен"},
		{"незабываемую", "незабыва"},
		{"незабываемые", "незабыва"},
		{"незабываемых", "незабыва"},
		{"независимо", "независим"},
		{"независимое", "независим"},
		{"независимости", "независим"},
		{"независимость", "независим"},
		{"независимую", "независим"},
		{"независимым", "независим"},
		{"независящим", "независя"},
		{"незадолго", "незадолг"},
		{"незаживающие", "незажива"},
		{"незаконная", "незакон"},
		{"незаконною", "незакон"},
		{"незаметнее", "незаметн"},
		{"незаметно", "незаметн"},
		{"незаметное", "незаметн"},
		{"незаметной", "незаметн"},
		{"незаметную", "незаметн"},
		{"незапамятные", "незапамятн"},
		{"незапамятных", "незапамятн"},
		{"незапертую", "незаперт"},
		{"незапятнанном", "незапятна"},
		{"незаработанном", "незаработа"},
		{"незаработанный", "незаработа"},
		{"незаслуженного", "незаслужен"},
		{"незаслуженным", "незаслужен"},
		{"незастенчивые", "незастенчив"},
		{"незатейливой", "незатейлив"},
		{"незачем", "незач"},
		{"незваному", "незван"},
		{"незваный", "незван"},
		{"незвучно", "незвучн"},
		{"нездешней", "нездешн"},
		{"нездешнею", "нездешн"},
		{"нездешний", "нездешн"},
		{"нездешними", "нездешн"},
		{"нездоров", "нездор"},
		{"нездорова", "нездоров"},
		{"нездоровая", "нездоров"},
		{"нездоровилось", "нездоров"},
		{"нездоровится", "нездоров"},
		{"нездорово", "нездоров"},
		{"нездоровы", "нездоров"},
		{"нездоровье", "нездоров"},
		{"нездоровьем", "нездоров"},
		{"неземная", "неземн"},
		{"неземного", "неземн"},
		{"неземной", "неземн"},
		{"неземные", "неземн"},
		{"незлобный", "незлобн"},
		{"незнаемою", "незна"},
		{"незнаком", "незнак"},
		{"незнакомец", "незнакомец"},
		{"незнакомого", "незнаком"},
		{"незнакомое", "незнаком"},
		{"незнакомой", "незнаком"},
		{"незнакомом", "незнаком"},
		{"незнакомому", "незнаком"},
		{"незнакомству", "незнакомств"},
		{"незнакомую", "незнаком"},
		{"незнакомцами", "незнакомц"},
		{"незнакомые", "незнаком"},
		{"незнакомый", "незнаком"},
		{"незнакомым", "незнаком"},
		{"незнакомыми", "незнаком"},
		{"незнание", "незнан"},
		{"незнанию", "незнан"},
		{"незначащие", "незнача"},
		{"незначительна", "незначительн"},
		{"незначительное", "незначительн"},
		{"незначительные", "незначительн"},
		{"незрим", "незр"},
		{"незримо", "незрим"},
		{"незримые", "незрим"},
		{"незримый", "незрим"},
		{"незримым", "незрим"},
		{"незыблемое", "незыблем"},
		{"неизбежная", "неизбежн"},
		{"неизбежно", "неизбежн"},
		{"неизбежного", "неизбежн"},
		{"неизбежное", "неизбежн"},
		{"неизбежность", "неизбежн"},
		{"неизбежною", "неизбежн"},
		{"неизбежные", "неизбежн"},
		{"неизбежный", "неизбежн"},
		{"неизбежным", "неизбежн"},
		{"неизбежными", "неизбежн"},
		{"неизбежных", "неизбежн"},
		{"неизведанных", "неизведа"},
		{"неизвестен", "неизвест"},
		{"неизвестная", "неизвестн"},
		{"неизвестно", "неизвестн"},
		{"неизвестное", "неизвестн"},
		{"неизвестном", "неизвестн"},
		{"неизвестности", "неизвестн"},
		{"неизвестность", "неизвестн"},
		{"неизвестные", "неизвестн"},
		{"неизвестный", "неизвестн"},
		{"неизвестным", "неизвестн"},
		{"неизвестными", "неизвестн"},
		{"неизвестных", "неизвестн"},
		{"неизгладимо", "неизгладим"},
		{"неизгладимым", "неизгладим"},
		{"неизлечима", "неизлечим"},
		{"неизлечимую", "неизлечим"},
		{"неизлечимые", "неизлечим"},
		{"неизменной", "неизмен"},
		{"неизменную", "неизмен"},
		{"неизменный", "неизмен"},
		{"неизменным", "неизмен"},
		{"неизменяющейся", "неизменя"},
		{"неизмеримою", "неизмерим"},
		{"неизреченной", "неизречен"},
		{"неизъяснимо", "неизъясним"},
		{"неизъяснимое", "неизъясним"},
		{"неизъяснимых", "неизъясним"},
		{"неимением", "неимен"},
		{"неимоверно", "неимоверн"},
		{"неимоверные", "неимоверн"},
		{"неимоверным", "неимоверн"},
		{"неинтересно", "неинтересн"},
		{"неинтересны", "неинтересн"},
		{"неискусно", "неискусн"},
		{"неисполненные", "неисполнен"},
		{"неисправим", "неисправ"},
		{"неисправимый", "неисправим"},
		{"неисправимым", "неисправим"},
		{"неисправной", "неисправн"},
		{"неистинном", "неистин"},
		{"неистовая", "неистов"},
		{"неистово", "неистов"},
		{"неистовое", "неистов"},
		{"неистовой", "неистов"},
		{"неистовые", "неистов"},
		{"неистовым", "неистов"},
		{"неистощимой", "неистощим"},
		{"неистощимых", "неистощим"},
		{"неистребима", "неистребим"},
		{"неистребимого", "неистребим"},
		{"неисходно", "неисходн"},
		{"неисходный", "неисходн"},
		{"ней", "не"},
		{"нейдем", "нейд"},
		{"нейдет", "нейдет"},
		{"нейдут", "нейдут"},
		{"нейрастения", "нейрастен"},
		{"нейти", "нейт"},
		{"некий", "нек"},
		{"неких", "нек"},
		{"некогда", "некогд"},
		{"некого", "нек"},
		{"некоего", "неко"},
		{"некоммерческих", "некоммерческ"},
		{"некому", "нек"},
		{"неконченные", "некончен"},
		{"неконченный", "некончен"},
		{"некормленая", "некормлен"},
		{"некоторая", "некотор"},
		{"некоторого", "некотор"},
		{"некоторое", "некотор"},
		{"некоторой", "некотор"},
		{"некотором", "некотор"},
		{"некоторому", "некотор"},
		{"некоторою", "некотор"},
		{"некоторую", "некотор"},
		{"некоторые", "некотор"},
		{"некоторый", "некотор"},
		{"некоторым", "некотор"},
		{"некоторыми", "некотор"},
		{"некоторых", "некотор"},
		{"некошенном", "некошен"},
		{"некрасива", "некрасив"},
		{"некрасиво", "некрасив"},
		{"некрасивой", "некрасив"},
		{"некрасивый", "некрасив"},
		{"некрасивым", "некрасив"},
		{"некраснеющих", "некраснеющ"},
		{"некрасова", "некрасов"},
		{"некрашеный", "некрашен"},
		{"некрепкий", "некрепк"},
		{"некрепок", "некрепок"},
		{"некстати", "некстат"},
		{"некто", "нект"},
		{"некуда", "некуд"},
		{"некупленных", "некуплен"},
		{"некую", "нек"},
		{"неладно", "неладн"},
		{"неласково", "неласков"},
		{"неласковый", "неласков"},
		{"нелегкая", "нелегк"},
		{"нелегко", "нелегк"},
		{"нелегкую", "нелегк"},
		{"нелепа", "нелеп"},
		{"нелепее", "нелеп"},
		{"нелепей", "нелеп"},
		{"нелепейшая", "нелеп"},
		{"нелепо", "нелеп"},
		{"нелепого", "нелеп"},
		{"нелепое", "нелеп"},
		{"нелепости", "нелеп"},
		{"нелепость", "нелеп"},
		{"нелепый", "нелеп"},
		{"нелепым", "нелеп"},
		{"нелживо", "нелжив"},
		{"нелицемерной", "нелицемерн"},
		{"неловкая", "неловк"},
		{"неловкий", "неловк"},
		{"неловким", "неловк"},
		{"неловко", "неловк"},
		{"неловкого", "неловк"},
		{"неловкое", "неловк"},
		{"неловкости", "неловк"},
		{"неловкость", "неловк"},
		{"неловкостью", "неловк"},
		{"неловок", "неловок"},
		{"неложно", "неложн"},
		{"нельзя", "нельз"},
		{"нельки", "нельк"},
		{"нелюбимый", "нелюбим"},
		{"нелюбовь", "нелюбов"},
		{"нем", "нем"},
		{"немалая", "немал"},
		{"немало", "нема"},
		{"немедленно", "немедлен"},
		{"немедленного", "немедлен"},
		{"немедленной", "немедлен"},
		{"немедленном", "немедлен"},
		{"немедленным", "немедлен"},
		{"немеет", "немеет"},
		{"немезида", "немезид"},
		{"немели", "немел"},
		{"немеркнущей", "немеркнущ"},
		{"немеркнущий", "немеркнущ"},
		{"немец", "немец"},
		{"немецкая", "немецк"},
		{"немецки", "немецк"},
		{"немецкие", "немецк"},
		{"немецкий", "немецк"},
		{"немецким", "немецк"},
		{"немецких", "немецк"},
		{"немецкого", "немецк"},
		{"немецкой", "немецк"},
		{"немецком", "немецк"},
		{"немецкую", "немецк"},
		{"немилосердно", "немилосердн"},
		{"неминуемая", "неминуем"},
		{"неминуемо", "неминуем"},
		{"неминуемое", "неминуем"},
		{"неминуемою", "неминуем"},
		{"неминучий", "неминуч"},
		{"немка", "немк"},
		{"немке", "немк"},
		{"немки", "немк"},
		{"немку", "немк"},
		{"немногие", "немног"},
		{"немногими", "немног"},
		{"немногих", "немног"},
		{"немного", "немн"},
		{"немножечко", "немножечк"},
		{"немножко", "немножк"},
		{"немо", "нем"},
		{"немое", "нем"},
		{"немой", "нем"},
		{"немок", "немок"},
		{"немолодой", "немолод"},
		{"немолодому", "немолод"},
		{"немолоды", "немолод"},
		{"немолодых", "немолод"},
		{"немолчный", "немолчн"},
		{"немом", "нем"},
		{"немота", "немот"},
		{"немотствует", "немотств"},
		{"немочь", "немоч"},
		{"немощами", "немощ"},
		{"немощеного", "немощен"},
		{"немощеными", "немощен"},
		{"нему", "нем"},
		{"немудреная", "немудрен"},
		{"немудрено", "немудр"},
		{"немудреный", "немудрен"},
		{"немудрый", "немудр"},
		{"немую", "нем"},
		{"немца", "немц"},
		{"немцами", "немц"},
		{"немцев", "немц"},
		{"немцем", "немц"},
		{"немцу", "немц"},
		{"немцы", "немц"},
		{"немые", "нем"},
		{"немым", "нем"},
		{"немыслимо", "немыслим"},
		{"немыслимы", "немыслим"},
		{"немытого", "немыт"},
		{"немытую", "немыт"},
		{"немытый", "немыт"},
		{"немых", "нем"},
		{"ненавидел", "ненавидел"},
		{"ненавидела", "ненавидел"},
		{"ненавидели", "ненавидел"},
		{"ненавидеть", "ненавидет"},
		{"ненавидит", "ненавид"},
		{"ненавидите", "ненавид"},
		{"ненавидишь", "ненавид"},
		{"ненавидя", "ненавид"},
		{"ненавидят", "ненавид"},
		{"ненавижу", "ненавиж"},
		{"ненавистен", "ненавист"},
		{"ненависти", "ненавист"},
		{"ненавистию", "ненавист"},
		{"ненавистна", "ненавистн"},
		{"ненавистная", "ненавистн"},
		{"ненавистничают", "ненавистнича"},
		{"ненавистно", "ненавистн"},
		{"ненавистного", "ненавистн"},
		{"ненавистное", "ненавистн"},
		{"ненавистною", "ненавистн"},
		{"ненавистным", "ненавистн"},
		{"ненавистными", "ненавистн"},
		{"ненависть", "ненавист"},
		{"ненавистью", "ненавист"},
		{"ненаглядная", "ненаглядн"},
		{"ненаглядную", "ненаглядн"},
		{"ненаглядный", "ненаглядн"},
		{"ненаглядным", "ненаглядн"},
		{"ненадежно", "ненадежн"},
		{"ненадежных", "ненадежн"},
		{"ненадолго", "ненадолг"},
		{"ненароком", "ненарок"},
		{"ненарушимая", "ненарушим"},
		{"ненарушимо", "ненарушим"},
		{"ненарушимого", "ненарушим"},
		{"ненастной", "ненастн"},
		{"ненастный", "ненастн"},
		{"ненастья", "ненаст"},
		{"ненасытимое", "ненасытим"},
		{"ненасытимый", "ненасытим"},
		{"ненасытность", "ненасытн"},
		{"ненасытную", "ненасытн"},
		{"ненатурально", "ненатуральн"},
		{"ненормальности", "ненормальн"},
		{"ненормальным", "ненормальн"},
		{"ненужная", "ненужн"},
		{"ненужно", "ненужн"},
		{"ненужного", "ненужн"},
		{"ненужной", "ненужн"},
		{"ненужность", "ненужн"},
		{"ненужную", "ненужн"},
		{"ненужны", "ненужн"},
		{"ненужные", "ненужн"},
		{"ненужный", "ненужн"},
		{"ненужным", "ненужн"},
		{"необдуманно", "необдума"},
		{"необдуманное", "необдума"},
		{"необитаемою", "необита"},
		{"необитаемыми", "необита"},
		{"необозримая", "необозрим"},
		{"необозримо", "необозрим"},
		{"необозримого", "необозрим"},
		{"необозримой", "необозрим"},
		{"необразованная", "необразова"},
		{"необразованный", "необразова"},
		{"необузданной", "необузда"},
		{"необходим", "необход"},
		{"необходима", "необходим"},
		{"необходимейшие", "необходим"},
		{"необходимо", "необходим"},
		{"необходимого", "необходим"},
		{"необходимое", "необходим"},
		{"необходимости", "необходим"},
		{"необходимостию", "необходим"},
		{"необходимость", "необходим"},
		{"необходимостью", "необходим"},
		{"необходимою", "необходим"},
		{"необходимы", "необходим"},
		{"необходимые", "необходим"},
		{"необходимыми", "необходим"},
		{"необъяснимо", "необъясним"},
		{"необъяснимое", "необъясним"},
		{"необъяснимые", "необъясним"},
		{"необъяснимым", "необъясним"},
		{"необъятная", "необъятн"},
		{"необъятно", "необъятн"},
		{"необъятного", "необъятн"},
		{"необъятное", "необъятн"},
		{"необъятном", "необъятн"},
		{"необъятною", "необъятн"},
		{"необъятную", "необъятн"},
		{"необъятные", "необъятн"},
		{"необыденное", "необыден"},
		{"необыкновенная", "необыкновен"},
		{"необыкновенно", "необыкновен"},
		{"необыкновенного", "необыкновен"},
		{"необыкновенное", "необыкновен"},
		{"необыкновенном", "необыкновен"},
		{"необыкновенному", "необыкновен"},
		{"необыкновенною", "необыкновен"},
		{"необыкновенную", "необыкновен"},
		{"необыкновенные", "необыкновен"},
		{"необыкновенный", "необыкновен"},
		{"необыкновенным", "необыкновен"},
		{"необыкновенных", "необыкновен"},
		{"необычайнейшее", "необычайн"},
		{"необычайное", "необычайн"},
		{"необычайной", "необычайн"},
		{"необычайности", "необычайн"},
		{"необычайный", "необычайн"},
		{"необычайным", "необычайн"},
		{"необычный", "необычн"},
		{"неоднократно", "неоднократн"},
		{"неоднократный", "неоднократн"},
		{"неодобрение", "неодобрен"},
		{"неодолимая", "неодолим"},
		{"неодушевленный", "неодушевлен"},
		{"неожиданная", "неожида"},
		{"неожиданнейшим", "неожидан"},
		{"неожиданно", "неожида"},
		{"неожиданного", "неожида"},
		{"неожиданное", "неожида"},
		{"неожиданному", "неожида"},
		{"неожиданность", "неожидан"},
		{"неожиданною", "неожида"},
		{"неожиданную", "неожида"},
		{"неожиданны", "неожидан"},
		{"неожиданные", "неожида"},
		{"неожиданный", "неожида"},
		{"неожиданным", "неожида"},
		{"неожиданными", "неожида"},
		{"неоконченная", "неокончен"},
		{"неокрепших", "неокрепш"},
		{"неопасен", "неопас"},
		{"неопасно", "неопасн"},
		{"неописанна", "неописан"},
		{"неописанная", "неописа"},
		{"неописанном", "неописа"},
		{"неописанный", "неописа"},
		{"неопределенно", "неопределен"},
		{"неопределенное", "неопределен"},
		{"неопределенной", "неопределен"},
		{"неопределенною", "неопределен"},
		{"неопределенные", "неопределен"},
		{"неопределенным", "неопределен"},
		{"неопределенными", "неопределен"},
		{"неопределенных", "неопределен"},
		{"неопровержимой", "неопровержим"},
		{"неопрятен", "неопрят"},
		{"неопрятнее", "неопрятн"},
		{"неопрятном", "неопрятн"},
		{"неопрятный", "неопрятн"},
		{"неопытного", "неопытн"},
		{"неопытному", "неопытн"},
		{"неопытности", "неопытн"},
		{"неопытные", "неопытн"},
		{"неопытный", "неопытн"},
		{"неосновательны", "неосновательн"},
		{"неоспоримый", "неоспорим"},
		{"неосторожнее", "неосторожн"},
		{"неосторожно", "неосторожн"},
		{"неосторожного", "неосторожн"},
		{"неосторожное", "неосторожн"},
		{"неосторожности", "неосторожн"},
		{"неосторожность", "неосторожн"},
		{"неосторожны", "неосторожн"},
		{"неосторожным", "неосторожн"},
		{"неостроумен", "неостроум"},
		{"неосуществима", "неосуществим"},
		{"неосуществимым", "неосуществим"},
		{"неотвратимо", "неотвратим"},
		{"неотвратимый", "неотвратим"},
		{"неотвязная", "неотвязн"},
		{"неотвязный", "неотвязн"},
		{"неотвязчивую", "неотвязчив"},
		{"неотвязчивый", "неотвязчив"},
		{"неотделанную", "неотдела"},
		{"неотесанный", "неотеса"},
		{"неоткуда", "неоткуд"},
		{"неотлагательное", "неотлагательн"},
		{"неотлагательные", "неотлагательн"},
		{"неотлагательным", "неотлагательн"},
		{"неотлучно", "неотлучн"},
		{"неотразим", "неотраз"},
		{"неотразимо", "неотразим"},
		{"неотразимого", "неотразим"},
		{"неотразимое", "неотразим"},
		{"неотразимою", "неотразим"},
		{"неотразимые", "неотразим"},
		{"неотразимый", "неотразим"},
		{"неотразимых", "неотразим"},
		{"неотсрочимый", "неотсрочим"},
		{"неотступная", "неотступн"},
		{"неотступнее", "неотступн"},
		{"неотступно", "неотступн"},
		{"неотступный", "неотступн"},
		{"неотъемлемо", "неотъемлем"},
		{"неохота", "неохот"},
		{"неохотно", "неохотн"},
		{"неохотой", "неохот"},
		{"неохотою", "неохот"},
		{"неоцененных", "неоценен"},
		{"непарадных", "непарадн"},
		{"неплотно", "неплотн"},
		{"непобедимой", "непобедим"},
		{"непобедимый", "непобедим"},
		{"непобедимым", "непобедим"},
		{"неповинен", "неповин"},
		{"неповоротливой", "неповоротлив"},
		{"неповоротливом", "неповоротлив"},
		{"неповторимая", "неповторим"},
		{"непогибшим", "непогибш"},
		{"непогоды", "непогод"},
		{"неподалеку", "неподалек"},
		{"неподвижен", "неподвиж"},
		{"неподвижна", "неподвижн"},
		{"неподвижная", "неподвижн"},
		{"неподвижнее", "неподвижн"},
		{"неподвижно", "неподвижн"},
		{"неподвижного", "неподвижн"},
		{"неподвижное", "неподвижн"},
		{"неподвижной", "неподвижн"},
		{"неподвижности", "неподвижн"},
		{"неподвижность", "неподвижн"},
		{"неподвижностью", "неподвижн"},
		{"неподвижною", "неподвижн"},
		{"неподвижные", "неподвижн"},
		{"неподвижный", "неподвижн"},
		{"неподвижным", "неподвижн"},
		{"неподвижными", "неподвижн"},
		{"неподдельного", "неподдельн"},
		{"неподдельное", "неподдельн"},
		{"неподкупная", "неподкупн"},
		{"неподкупном", "неподкупн"},
		{"неподобно", "неподобн"},
		{"неподражаем", "неподража"},
		{"неподражаемо", "неподражаем"},
		{"неподходящие", "неподходя"},
		{"непозволительно", "непозволительн"},
		{"непоказанной", "непоказа"},
		{"непокладные", "непокладн"},
		{"непокойна", "непокойн"},
		{"непокойно", "непокойн"},
		{"непоколебимее", "непоколебим"},
		{"непоколебимой", "непоколебим"},
		{"непоколебимыми", "непоколебим"},
		{"непокорных", "непокорн"},
		{"непокрытой", "непокрыт"},
		{"неполнота", "неполнот"},
		{"неполный", "неполн"},
		{"неположенные", "неположен"},
		{"непомерной", "непомерн"},
		{"непомерный", "непомерн"},
		{"непомерным", "непомерн"},
		{"непонимании", "непониман"},
		{"непонятая", "непонят"},
		{"непонятливости", "непонятлив"},
		{"непонятливый", "непонятлив"},
		{"непонятна", "непонятн"},
		{"непонятная", "непонятн"},
		{"непонятно", "непонятн"},
		{"непонятного", "непонятн"},
		{"непонятное", "непонятн"},
		{"непонятному", "непонятн"},
		{"непонятный", "непонятн"},
		{"непонятным", "непонятн"},
		{"непорядком", "непорядк"},
		{"непорядочно", "непорядочн"},
		{"непосвященные", "непосвящен"},
		{"непосильной", "непосильн"},
		{"непослушны", "непослушн"},
		{"непосредственного", "непосредствен"},
		{"непосредственное", "непосредствен"},
		{"непосредственной", "непосредствен"},
		{"непостижима", "непостижим"},
		{"непостижимо", "непостижим"},
		{"непостижимый", "непостижим"},
		{"непота", "непот"},
		{"непотребного", "непотребн"},
		{"непохожее", "непохож"},
		{"непохожей", "непохож"},
		{"непочтительна", "непочтительн"},
		{"непочтительностию", "непочтительн"},
		{"неправда", "неправд"},
		{"неправду", "неправд"},
		{"неправедные", "неправедн"},
		{"неправильного", "неправильн"},
		{"неправильное", "неправильн"},
		{"неправильной", "неправильн"},
		{"неправильности", "неправильн"},
		{"неправильность", "неправильн"},
		{"неправильный", "неправильн"},
		{"неправленный", "неправлен"},
		{"неправые", "неправ"},
		{"непредвиденных", "непредвиден"},
		{"непреклонна", "непреклон"},
		{"непреклонная", "непреклон"},
		{"непрекращающегося", "непрекраща"},
		{"непреложным", "непреложн"},
		{"непремен", "непрем"},
		{"непременно", "непремен"},
		{"непременною", "непремен"},
		{"непреодолимое", "непреодолим"},
		{"непрерывно", "непрерывн"},
		{"непрерывное", "непрерывн"},
		{"непрерывной", "непрерывн"},
		{"непрерывный", "непрерывн"},
		{"непрерывным", "непрерывн"},
		{"непрестанной", "непреста"},
		{"непривилегированные", "непривилегирова"},
		{"непривлекательной", "непривлекательн"},
		{"непривычки", "непривычк"},
		{"непривычного", "непривычн"},
		{"непривычное", "непривычн"},
		{"непривычным", "непривычн"},
		{"непривычных", "непривычн"},
		{"неприготовленную", "неприготовлен"},
		{"непризнанный", "непризна"},
		{"непризнанным", "непризна"},
		{"неприкосновенное", "неприкосновен"},
		{"неприкосновенность", "неприкосновен"},
		{"неприличия", "неприлич"},
		{"неприлично", "неприличн"},
		{"неприличное", "неприличн"},
		{"неприличном", "неприличн"},
		{"неприличности", "неприличн"},
		{"неприличным", "неприличн"},
		{"неприметнее", "неприметн"},
		{"неприметно", "неприметн"},
		{"непримиримых", "непримирим"},
		{"непринужденности", "непринужден"},
		{"непринужденный", "непринужден"},
		{"неприступная", "неприступн"},
		{"неприступны", "неприступн"},
		{"неприступные", "неприступн"},
		{"непритворенную", "непритворен"},
		{"непритворно", "непритворн"},
		{"непритворной", "непритворн"},
		{"непритворным", "непритворн"},
		{"непричастен", "непричаст"},
		{"непричесанной", "непричеса"},
		{"неприязненно", "неприязнен"},
		{"неприязнь", "неприязн"},
		{"неприятель", "неприятел"},
		{"неприятельский", "неприятельск"},
		{"неприятельским", "неприятельск"},
		{"неприятен", "неприят"},
		{"неприятнейшее", "неприятн"},
		{"неприятно", "неприятн"},
		{"неприятного", "неприятн"},
		{"неприятное", "неприятн"},
		{"неприятной", "неприятн"},
		{"неприятном", "неприятн"},
		{"неприятному", "неприятн"},
		{"неприятностей", "неприятн"},
		{"неприятности", "неприятн"},
		{"неприятностию", "неприятн"},
		{"неприятность", "неприятн"},
		{"неприятностях", "неприятн"},
		{"неприятные", "неприятн"},
		{"неприятный", "неприятн"},
		{"неприятным", "неприятн"},
		{"неприятными", "неприятн"},
		{"непробудней", "непробудн"},
		{"непробудно", "непробудн"},
		{"непробудной", "непробудн"},
		{"непробудную", "непробудн"},
		{"непробудные", "непробудн"},
		{"непробудный", "непробудн"},
		{"непроглядном", "непроглядн"},
		{"непроглядный", "непроглядн"},
		{"непродажных", "непродажн"},
		{"непродолжительный", "непродолжительн"},
		{"непроезжий", "непроезж"},
		{"непролазная", "непролазн"},
		{"непроницаема", "непроницаем"},
		{"непроницаемости", "непроницаем"},
		{"непроницаемы", "непроницаем"},
		{"непроницаемым", "непроница"},
		{"непроницаемых", "непроница"},
		{"непростительно", "непростительн"},
		{"непростительной", "непростительн"},
		{"непростительную", "непростительн"},
		{"непростительные", "непростительн"},
		{"непроходимая", "непроходим"},
		{"непрочен", "непроч"},
		{"непрошеные", "непрошен"},
		{"непрошеных", "непрошен"},
		{"непрядва", "непрядв"},
		{"непрядвой", "непрядв"},
		{"непутное", "непутн"},
		{"неравенством", "неравенств"},
		{"неравна", "неравн"},
		{"неравнодушен", "неравнодуш"},
		{"нерадению", "нераден"},
		{"нерадостно", "нерадостн"},
		{"неразборчиво", "неразборчив"},
		{"неразвернувшийся", "неразвернувш"},
		{"неразвитость", "неразвит"},
		{"неразвиты", "неразвит"},
		{"неразгаданного", "неразгада"},
		{"неразговорчив", "неразговорч"},
		{"неразговорчивы", "неразговорчив"},
		{"неразделенную", "неразделен"},
		{"нераздельности", "нераздельн"},
		{"неразлучна", "неразлучн"},
		{"неразлучны", "неразлучн"},
		{"неразрезанною", "неразреза"},
		{"неразрешаемые", "неразреша"},
		{"неразрешенных", "неразрешен"},
		{"неразрешим", "неразреш"},
		{"неразрешимому", "неразрешим"},
		{"неразрешимы", "неразрешим"},
		{"неразрешимый", "неразрешим"},
		{"неразрываемую", "неразрыва"},
		{"неразрывно", "неразрывн"},
		{"неразрывного", "неразрывн"},
		{"неразрывную", "неразрывн"},
		{"нерасположение", "нерасположен"},
		{"нерасчетливо", "нерасчетлив"},
		{"нерв", "нерв"},
		{"нервами", "нерв"},
		{"нервическая", "нервическ"},
		{"нервический", "нервическ"},
		{"нервическим", "нервическ"},
		{"нервическое", "нервическ"},
		{"нервная", "нервн"},
		{"нервно", "нервн"},
		{"нервное", "нервн"},
		{"нервной", "нервн"},
		{"нервною", "нервн"},
		{"нервны", "нервн"},
		{"нервный", "нервн"},
		{"нервным", "нервн"},
		{"нервных", "нервн"},
		{"нервов", "нерв"},
		{"нервы", "нерв"},
		{"нередко", "нередк"},
		{"нерешенное", "нерешен"},
		{"нерешимости", "нерешим"},
		{"нерешительно", "нерешительн"},
		{"нерешительном", "нерешительн"},
		{"нерешительности", "нерешительн"},
		{"нерешительность", "нерешительн"},
		{"нерешительностью", "нерешительн"},
		{"нерешительны", "нерешительн"},
		{"неровно", "неровн"},
		{"неровности", "неровн"},
		{"нерукотворный", "нерукотворн"},
		{"неряхой", "нерях"},
		{"нес", "нес"},
		{"несбывшееся", "несб"},
		{"несбыточной", "несбыточн"},
		{"несбыточною", "несбыточн"},
		{"несбыточную", "несбыточн"},
		{"несбыточные", "несбыточн"},
		{"несбыточный", "несбыточн"},
		{"несбыточных", "несбыточн"},
		{"несвежем", "несвеж"},
		{"несвежи", "несвеж"},
		{"несвежую", "несвеж"},
		{"несветский", "несветск"},
		{"несвойственным", "несвойствен"},
		{"несвязанный", "несвяза"},
		{"несвязно", "несвязн"},
		{"несвязный", "несвязн"},
		{"несгораемые", "несгора"},
		{"несдобровать", "несдоброва"},
		{"несерьезным", "несерьезн"},
		{"несессер", "несессер"},
		{"несессеры", "несессер"},
		{"несет", "несет"},
		{"несете", "несет"},
		{"несется", "несет"},
		{"несешь", "несеш"},
		{"неси", "нес"},
		{"несказа", "несказ"},
		{"несказанно", "несказа"},
		{"нескладная", "нескладн"},
		{"нескладно", "нескладн"},
		{"нескладном", "нескладн"},
		{"несколькими", "нескольк"},
		{"нескольких", "нескольк"},
		{"несколько", "нескольк"},
		{"нескольку", "нескольк"},
		{"нескончаемой", "несконча"},
		{"нескончаемому", "несконча"},
		{"нескончаемый", "несконча"},
		{"нескончаемым", "несконча"},
		{"нескромности", "нескромн"},
		{"нескромный", "нескромн"},
		{"нескрываемая", "нескрыва"},
		{"нескрываемым", "нескрыва"},
		{"нескучно", "нескучн"},
		{"несла", "несл"},
		{"неслабый", "неслаб"},
		{"неслась", "несл"},
		{"несли", "несл"},
		{"неслись", "несл"},
		{"неслиянности", "неслиян"},
		{"несло", "несл"},
		{"несловоохотлив", "несловоохотл"},
		{"неслось", "несл"},
		{"неслушавшие", "неслуша"},
		{"неслыханно", "неслыха"},
		{"неслыханное", "неслыха"},
		{"неслыханной", "неслыха"},
		{"неслыханном", "неслыха"},
		{"неслыханному", "неслыха"},
		{"неслыханные", "неслыха"},
		{"неслыханным", "неслыха"},
		{"неслыханными", "неслыха"},
		{"неслыханных", "неслыха"},
		{"неслышно", "неслышн"},
		{"неслышным", "неслышн"},
		{"несмелые", "несмел"},
		{"несметные", "несметн"},
		{"несметный", "несметн"},
		{"несмотря", "несмотр"},
		{"несмыкаемости", "несмыкаем"},
		{"несносна", "несносн"},
		{"несносно", "несносн"},
		{"несносное", "несносн"},
		{"несносную", "несносн"},
		{"несносный", "несносн"},
		{"несовершенно", "несовершен"},
		{"несовершеннолетнее", "несовершеннолетн"},
		{"несовершеннолетний", "несовершеннолетн"},
		{"несовершенств", "несовершенств"},
		{"несогласие", "несоглас"},
		{"несогласимых", "несогласим"},
		{"несогласию", "несоглас"},
		{"несомненно", "несомнен"},
		{"несомненного", "несомнен"},
		{"несомненною", "несомнен"},
		{"несомненный", "несомнен"},
		{"несомненным", "несомнен"},
		{"несомненных", "несомнен"},
		{"несообразный", "несообразн"},
		{"несообщителен", "несообщител"},
		{"несоразмерно", "несоразмерн"},
		{"несостоятельны", "несостоятельн"},
		{"неспешно", "неспешн"},
		{"неспешные", "неспешн"},
		{"неспешный", "неспешн"},
		{"неспешным", "неспешн"},
		{"неспокойна", "неспокойн"},
		{"неспокойно", "неспокойн"},
		{"неспособность", "неспособн"},
		{"неспособным", "неспособн"},
		{"несправедлива", "несправедлив"},
		{"несправедливая", "несправедлив"},
		{"несправедливо", "несправедлив"},
		{"несправедливого", "несправедлив"},
		{"несправедливое", "несправедлив"},
		{"несправедливость", "несправедлив"},
		{"несправедливостью", "несправедлив"},
		{"несправедливую", "несправедлив"},
		{"несправедливы", "несправедлив"},
		{"несправедливые", "несправедлив"},
		{"несравненном", "несравнен"},
		{"несравненный", "несравнен"},
		{"нессиметрично", "нессиметричн"},
		{"нестерпим", "нестерп"},
		{"нестерпима", "нестерпим"},
		{"нестерпимая", "нестерпим"},
		{"нестерпимо", "нестерпим"},
		{"нестерпимого", "нестерпим"},
		{"нестерпимой", "нестерпим"},
		{"нестерпимом", "нестерпим"},
		{"нестерпимый", "нестерпим"},
		{"нести", "нест"},
		{"нестиранного", "нестира"},
		{"нестройном", "нестройн"},
		{"несу", "нес"},
		{"несут", "несут"},
		{"несущейся", "несущ"},
		{"несуществующий", "несуществ"},
		{"несущие", "несущ"},
		{"несущиеся", "несущ"},
		{"несущий", "несущ"},
		{"несущихся", "несущ"},
		{"несч", "несч"},
		{"несчастен", "несчаст"},
		{"несчастие", "несчаст"},
		{"несчастии", "несчаст"},
		{"несчастий", "несчаст"},
		{"несчастию", "несчаст"},
		{"несчастия", "несчаст"},
		{"несчастиях", "несчаст"},
		{"несчастлив", "несчастл"},
		{"несчастлива", "несчастлив"},
		{"несчастливейшим", "несчастлив"},
		{"несчастливый", "несчастлив"},
		{"несчастна", "несчастн"},
		{"несчастная", "несчастн"},
		{"несчастнее", "несчастн"},
		{"несчастненькой", "несчастненьк"},
		{"несчастного", "несчастн"},
		{"несчастное", "несчастн"},
		{"несчастной", "несчастн"},
		{"несчастном", "несчастн"},
		{"несчастному", "несчастн"},
		{"несчастною", "несчастн"},
		{"несчастную", "несчастн"},
		{"несчастны", "несчастн"},
		{"несчастные", "несчастн"},
		{"несчастный", "несчастн"},
		{"несчастным", "несчастн"},
		{"несчастными", "несчастн"},
		{"несчастных", "несчастн"},
		{"несчастье", "несчаст"},
		{"несчастьем", "несчаст"},
		{"несчастью", "несчаст"},
		{"несчастья", "несчаст"},
		{"несчастьями", "несчаст"},
		{"несчетные", "несчетн"},
		{"несытым", "несыт"},
		{"неся", "не"},
		{"нет", "нет"},
		{"нетактично", "нетактичн"},
		{"нетвердо", "нетверд"},
		{"нетвердой", "нетверд"},
		{"нетвердым", "нетверд"},
		{"нетвердыми", "нетверд"},
		{"нетверез", "нетверез"},
		{"нетерепении", "нетерепен"},
		{"нетерпелив", "нетерпел"},
		{"нетерпеливо", "нетерпелив"},
		{"нетерпеливого", "нетерпелив"},
		{"нетерпеливое", "нетерпелив"},
		{"нетерпеливой", "нетерпелив"},
		{"нетерпеливому", "нетерпелив"},
		{"нетерпеливы", "нетерпелив"},
		{"нетерпеливый", "нетерпелив"},
		{"нетерпеливым", "нетерпелив"},
		{"нетерпение", "нетерпен"},
		{"нетерпением", "нетерпен"},
		{"нетерпении", "нетерпен"},
		{"нетерпения", "нетерпен"},
		{"нетопленную", "нетоплен"},
		{"неторопливо", "нетороплив"},
		{"нетрезвом", "нетрезв"},
		{"нетронутую", "нетронут"},
		{"нетронутый", "нетронут"},
		{"нету", "нет"},
		{"нетяжко", "нетяжк"},
		{"неуважения", "неуважен"},
		{"неуважительный", "неуважительн"},
		{"неуверенном", "неуверен"},
		{"неувядаемая", "неувяда"},
		{"неувядающая", "неувяда"},
		{"неувядающим", "неувяда"},
		{"неувязанное", "неувяза"},
		{"неугодлив", "неугодл"},
		{"неугодливая", "неугодлив"},
		{"неугодно", "неугодн"},
		{"неугомонный", "неугомон"},
		{"неудавшейся", "неуда"},
		{"неудавшиеся", "неуда"},
		{"неудавшуюся", "неуда"},
		{"неудач", "неудач"},
		{"неудача", "неудач"},
		{"неудачами", "неудач"},
		{"неудаче", "неудач"},
		{"неудачи", "неудач"},
		{"неудачная", "неудачн"},
		{"неудачно", "неудачн"},
		{"неудержимее", "неудержим"},
		{"неудержимо", "неудержим"},
		{"неудержимого", "неудержим"},
		{"неудержимую", "неудержим"},
		{"неудержимым", "неудержим"},
		{"неудобно", "неудобн"},
		{"неудобной", "неудобн"},
		{"неудобным", "неудобн"},
		{"неудобств", "неудобств"},
		{"неудобства", "неудобств"},
		{"неудобствами", "неудобств"},
		{"неудобстве", "неудобств"},
		{"неудобству", "неудобств"},
		{"неудовлетворенность", "неудовлетворен"},
		{"неудовлетворительными", "неудовлетворительн"},
		{"неудовольствие", "неудовольств"},
		{"неудовольствием", "неудовольств"},
		{"неудовольствия", "неудовольств"},
		{"неужели", "неужел"},
		{"неужель", "неужел"},
		{"неузнаваемого", "неузнава"},
		{"неузнаваемой", "неузнава"},
		{"неузнанной", "неузна"},
		{"неуклонно", "неуклон"},
		{"неуклюж", "неуклюж"},
		{"неуклюжая", "неуклюж"},
		{"неуклюже", "неуклюж"},
		{"неуклюжее", "неуклюж"},
		{"неуклюжий", "неуклюж"},
		{"неуклюжим", "неуклюж"},
		{"неукоснительно", "неукоснительн"},
		{"неукрашенное", "неукрашен"},
		{"неукротимей", "неукротим"},
		{"неуловимо", "неуловим"},
		{"неуловимое", "неуловим"},
		{"неуловимый", "неуловим"},
		{"неуложенное", "неуложен"},
		{"неумело", "неумел"},
		{"неумелые", "неумел"},
		{"неумен", "неум"},
		{"неумением", "неумен"},
		{"неуменьем", "неумен"},
		{"неуменья", "неумен"},
		{"неумеренная", "неумерен"},
		{"неумеренно", "неумерен"},
		{"неумеренное", "неумерен"},
		{"неуместною", "неуместн"},
		{"неумирающего", "неумира"},
		{"неумной", "неумн"},
		{"неумолим", "неумол"},
		{"неумолима", "неумолим"},
		{"неумолимо", "неумолим"},
		{"неумолимый", "неумолим"},
		{"неумолкаемый", "неумолка"},
		{"неумолкающей", "неумолка"},
		{"неумолчной", "неумолчн"},
		{"неумолчные", "неумолчн"},
		{"неумытого", "неумыт"},
		{"неумытый", "неумыт"},
		{"неумышленно", "неумышлен"},
		{"неумышленное", "неумышлен"},
		{"неуныло", "неун"},
		{"неурожае", "неурожа"},
		{"неурожай", "неурожа"},
		{"неурядиц", "неурядиц"},
		{"неуследимый", "неуследим"},
		{"неуспеха", "неуспех"},
		{"неустанный", "неуста"},
		{"неустающие", "неуста"},
		{"неустойки", "неустойк"},
		{"неустранимых", "неустраним"},
		{"неустрашимо", "неустрашим"},
		{"неустройств", "неустройств"},
		{"неусыпно", "неусыпн"},
		{"неутешными", "неутешн"},
		{"неутолимую", "неутолим"},
		{"неутомимая", "неутомим"},
		{"неутомимо", "неутомим"},
		{"неутомимым", "неутомим"},
		{"неученых", "неучен"},
		{"неучтиво", "неучтив"},
		{"неучтивости", "неучтив"},
		{"неуютно", "неуютн"},
		{"нехитрого", "нехитр"},
		{"нехождению", "нехожден"},
		{"нехороши", "нехорош"},
		{"нехорошие", "нехорош"},
		{"нехороших", "нехорош"},
		{"нехорошо", "нехорош"},
		{"нехоть", "нехот"},
		{"нехотя", "нехот"},
		{"нехристь", "нехрист"},
		{"нецеремонные", "нецеремон"},
		{"нечальны", "нечальн"},
		{"нечаянно", "нечая"},
		{"нечаянного", "нечая"},
		{"нечаянностей", "нечаян"},
		{"нечаянный", "нечая"},
		{"нечаянным", "нечая"},
		{"нечего", "неч"},
		{"нечеловеческих", "нечеловеческ"},
		{"нечем", "неч"},
		{"нечему", "неч"},
		{"нечесаную", "нечесан"},
		{"нечесаный", "нечесан"},
		{"нечет", "нечет"},
		{"нечиновных", "нечиновн"},
		{"нечисто", "нечист"},
		{"нечистота", "нечистот"},
		{"нечистоту", "нечистот"},
		{"нечистоты", "нечистот"},
		{"нечистым", "нечист"},
		{"нечищенным", "нечищен"},
		{"нечищеные", "нечищен"},
		{"нечопорно", "нечопорн"},
		{"нечто", "нечт"},
		{"нечувствительно", "нечувствительн"},
		{"нечуток", "нечуток"},
		{"нешто", "нешт"},
		{"нештопанного", "нештопа"},
		{"нештопаных", "нештопан"},
		{"нешуточному", "нешуточн"},
		{"нею", "не"},
		{"неявившихся", "неяв"},
		{"неярки", "неярк"},
		{"неяркий", "неярк"},
		{"неясна", "неясн"},
		{"неясная", "неясн"},
		{"неясно", "неясн"},
		{"неясное", "неясн"},
		{"неясном", "неясн"},
		{"неясному", "неясн"},
		{"неясные", "неясн"},
		{"неясный", "неясн"},
		{"неясным", "неясн"},
		{"неясных", "неясн"},
		{"ни", "ни"},
		{"нибудь", "нибуд"},
		{"ниве", "нив"},
		{"нивы", "нив"},
		{"нигде", "нигд"},
		{"нигилизм", "нигилизм"},
		{"нигилист", "нигилист"},
		{"нигилистов", "нигилист"},
		{"нигилисты", "нигилист"},
		{"ниже", "ниж"},
		{"нижеподписавшийся", "нижеподписа"},
		{"нижнего", "нижн"},
		{"нижнее", "нижн"},
		{"нижней", "нижн"},
		{"нижнем", "нижн"},
		{"нижнему", "нижн"},
		{"нижнею", "нижн"},
		{"нижний", "нижн"},
		{"нижних", "нижн"},
		{"нижнюю", "нижн"},
		{"нижняя", "нижн"},
		{"низать", "низа"},
		{"низведении", "низведен"},
		{"низвела", "низвел"},
		{"низводила", "низвод"},
		{"низенькие", "низеньк"},
		{"низенький", "низеньк"},
		{"низка", "низк"},
		{"низкая", "низк"},
		{"низки", "низк"},
		{"низкие", "низк"},
		{"низкий", "низк"},
		{"низким", "низк"},
		{"низких", "низк"},
		{"низко", "низк"},
		{"низкого", "низк"},
		{"низкое", "низк"},
		{"низкой", "низк"},
		{"низком", "низк"},
		{"низкую", "низк"},
		{"низменный", "низмен"},
		{"низок", "низок"},
		{"низости", "низост"},
		{"низость", "низост"},
		{"низшего", "низш"},
		{"низшее", "низш"},
		{"низшем", "низш"},
		{"низший", "низш"},
		{"никак", "никак"},
		{"никакая", "никак"},
		{"никакие", "никак"},
		{"никаким", "никак"},
		{"никакими", "никак"},
		{"никаких", "никак"},
		{"никакого", "никак"},
		{"никакое", "никак"},
		{"никакой", "никак"},
		{"никанора", "никанор"},
		{"никем", "ник"},
		{"никита", "никит"},
		{"никите", "ник"},
		{"никитича", "никитич"},
		{"никитой", "никит"},
		{"никиту", "никит"},
		{"никиты", "никит"},
		{"никифоровна", "никифоровн"},
		{"никли", "никл"},
		{"никнет", "никнет"},
		{"никогда", "никогд"},
		{"никого", "ник"},
		{"никодим", "никод"},
		{"никодима", "никодим"},
		{"никодиме", "никодим"},
		{"никодимом", "никодим"},
		{"никодиму", "никодим"},
		{"никоим", "нико"},
		{"николавна", "николавн"},
		{"николаевич", "николаевич"},
		{"николаевича", "николаевич"},
		{"николаевичу", "николаевич"},
		{"николаевна", "николаевн"},
		{"николаевне", "николаевн"},
		{"николаевной", "николаевн"},
		{"николаевну", "николаевн"},
		{"николаевских", "николаевск"},
		{"николаевском", "николаевск"},
		{"николаем", "никола"},
		{"николаичем", "николаич"},
		{"николай", "никола"},
		{"николашка", "николашк"},
		{"николашке", "николашк"},
		{"николаю", "никола"},
		{"николая", "никол"},
		{"николы", "никол"},
		{"никому", "ник"},
		{"никто", "никт"},
		{"никуда", "никуд"},
		{"нил", "нил"},
		{"ним", "ним"},
		{"нимало", "нима"},
		{"нимбы", "нимб"},
		{"ними", "ним"},
		{"нимфа", "нимф"},
		{"нина", "нин"},
		{"нине", "нин"},
		{"ниной", "нин"},
		{"нину", "нин"},
		{"нины", "нин"},
		{"ниоткуда", "ниоткуд"},
		{"нипочем", "нипоч"},
		{"нисколько", "нискольк"},
		{"ниспосланное", "ниспосла"},
		{"нисходила", "нисход"},
		{"нисходит", "нисход"},
		{"нисходят", "нисход"},
		{"нити", "нит"},
		{"нитка", "нитк"},
		{"нитками", "нитк"},
		{"нитки", "нитк"},
		{"ниткой", "нитк"},
		{"нитку", "нитк"},
		{"ниток", "ниток"},
		{"ниточками", "ниточк"},
		{"ниточке", "ниточк"},
		{"нить", "нит"},
		{"нитью", "нит"},
		{"нитяной", "нитян"},
		{"нитяный", "нитян"},
		{"них", "них"},
		{"ниц", "ниц"},
		{"ницце", "ницц"},
		{"ницше", "ницш"},
		{"ничего", "нич"},
		{"ничегошечко", "ничегошечк"},
		{"ничей", "нич"},
		{"ничем", "нич"},
		{"ничему", "нич"},
		{"ничком", "ничк"},
		{"ничто", "ничт"},
		{"ничтожен", "ничтож"},
		{"ничтожества", "ничтожеств"},
		{"ничтожество", "ничтожеств"},
		{"ничтожная", "ничтожн"},
		{"ничтожнейшее", "ничтожн"},
		{"ничтожнейшем", "ничтожн"},
		{"ничтожнейший", "ничтожн"},
		{"ничтожного", "ничтожн"},
		{"ничтожной", "ничтожн"},
		{"ничтожностию", "ничтожн"},
		{"ничтожною", "ничтожн"},
		{"ничтожны", "ничтожн"},
		{"ничтожные", "ничтожн"},
		{"ничтожный", "ничтожн"},
		{"ничтожным", "ничтожн"},
		{"ничтожных", "ничтожн"},
		{"ничуть", "ничут"},
		{"ничьего", "нич"},
		{"ничьей", "нич"},
		{"ничьи", "нич"},
		{"ничьим", "нич"},
		{"ничьих", "нич"},
		{"ничья", "нич"},
		{"нише", "ниш"},
		{"нищ", "нищ"},
		{"нищая", "нищ"},
		{"нищего", "нищ"},
		{"нищей", "нищ"},
		{"нищему", "нищ"},
		{"нищенские", "нищенск"},
		{"нищенской", "нищенск"},
		{"нищета", "нищет"},
		{"нищете", "нищет"},
		{"нищетой", "нищет"},
		{"нищету", "нищет"},
		{"нищеты", "нищет"},
		{"нищие", "нищ"},
		{"нищий", "нищ"},
		{"нищим", "нищ"},
		{"нищими", "нищ"},
		{"нищих", "нищ"},
		{"нмением", "нмен"},
		{"нная", "нна"},
		{"нной", "нно"},
		{"ннх", "ннх"},
		{"нных", "нных"},
		{"но", "но"},
		{"новая", "нов"},
		{"новейшая", "нов"},
		{"новейшее", "нов"},
		{"новейший", "нов"},
		{"новейшими", "нов"},
		{"новейших", "нов"},
		{"новенькие", "новеньк"},
		{"новеньким", "новеньк"},
		{"новенького", "новеньк"},
		{"новенькое", "новеньк"},
		{"новехонькая", "новехоньк"},
		{"новизне", "новизн"},
		{"новичка", "новичк"},
		{"новичкам", "новичк"},
		{"новички", "новичк"},
		{"новичок", "новичок"},
		{"ново", "нов"},
		{"нового", "нов"},
		{"новодевичий", "новодевич"},
		{"новое", "нов"},
		{"новой", "нов"},
		{"новом", "нов"},
		{"новому", "нов"},
		{"новорожденный", "новорожден"},
		{"новоселье", "новосел"},
		{"новостей", "новост"},
		{"новости", "новост"},
		{"новость", "новост"},
		{"новостью", "новост"},
		{"новостями", "новост"},
		{"новостях", "новост"},
		{"новою", "нов"},
		{"новоявленный", "новоявлен"},
		{"новую", "нов"},
		{"новые", "нов"},
		{"новый", "нов"},
		{"новым", "нов"},
		{"новыми", "нов"},
		{"новых", "нов"},
		{"новь", "нов"},
		{"новью", "нов"},
		{"ног", "ног"},
		{"нога", "ног"},
		{"ногам", "ног"},
		{"ногами", "ног"},
		{"ногах", "ног"},
		{"ноге", "ног"},
		{"ноги", "ног"},
		{"ного", "ног"},
		{"ногой", "ног"},
		{"ноготками", "ноготк"},
		{"ногою", "ног"},
		{"ногтем", "ногт"},
		{"ногти", "ногт"},
		{"ногтями", "ногт"},
		{"ногу", "ног"},
		{"ноев", "но"},
		{"ноет", "ноет"},
		{"нож", "нож"},
		{"ножа", "нож"},
		{"ножами", "нож"},
		{"ноже", "нож"},
		{"ножей", "нож"},
		{"ножи", "нож"},
		{"ножик", "ножик"},
		{"ножичек", "ножичек"},
		{"ножищами", "ножищ"},
		{"ножка", "ножк"},
		{"ножками", "ножк"},
		{"ножки", "ножк"},
		{"ножницами", "ножниц"},
		{"ножнички", "ножничк"},
		{"ножом", "нож"},
		{"ноль", "нол"},
		{"номер", "номер"},
		{"номере", "номер"},
		{"номинально", "номинальн"},
		{"ноне", "нон"},
		{"норе", "нор"},
		{"норке", "норк"},
		{"норки", "норк"},
		{"норма", "норм"},
		{"нормален", "нормал"},
		{"нормальная", "нормальн"},
		{"нормально", "нормальн"},
		{"нормальное", "нормальн"},
		{"нормальной", "нормальн"},
		{"нормальному", "нормальн"},
		{"нормальную", "нормальн"},
		{"нормальный", "нормальн"},
		{"нормальным", "нормальн"},
		{"норму", "норм"},
		{"нормы", "норм"},
		{"норов", "нор"},
		{"норовил", "норов"},
		{"норовит", "норов"},
		{"норы", "нор"},
		{"нос", "нос"},
		{"носа", "нос"},
		{"носами", "нос"},
		{"носе", "нос"},
		{"носики", "носик"},
		{"носил", "нос"},
		{"носила", "нос"},
		{"носилась", "нос"},
		{"носили", "нос"},
		{"носились", "нос"},
		{"носило", "нос"},
		{"носился", "нос"},
		{"носиль", "носил"},
		{"носильщики", "носильщик"},
		{"носит", "нос"},
		{"носитесь", "нос"},
		{"носится", "нос"},
		{"носить", "нос"},
		{"носиться", "нос"},
		{"носишь", "нос"},
		{"носка", "носк"},
		{"носками", "носк"},
		{"носке", "носк"},
		{"носки", "носк"},
		{"носков", "носк"},
		{"носком", "носк"},
		{"носовой", "носов"},
		{"носок", "носок"},
		{"носом", "нос"},
		{"носу", "нос"},
		{"носы", "нос"},
		{"носясь", "нос"},
		{"носят", "нос"},
		{"носятся", "нос"},
		{"носящих", "нося"},
		{"нот", "нот"},
		{"нота", "нот"},
		{"нотами", "нот"},
		{"ноте", "нот"},
		{"нотка", "нотк"},
		{"нотке", "нотк"},
		{"нотки", "нотк"},
		{"нотой", "нот"},
		{"ноту", "нот"},
		{"нотунг", "нотунг"},
		{"ноты", "нот"},
		{"ночам", "ноч"},
		{"ночами", "ноч"},
		{"ночах", "ноч"},
		{"ночевавший", "ночева"},
		{"ночевал", "ночева"},
		{"ночевать", "ночева"},
		{"ночевывал", "ночевыва"},
		{"ночей", "ноч"},
		{"ночи", "ноч"},
		{"ночлег", "ночлег"},
		{"ночная", "ночн"},
		{"ночно", "ночн"},
		{"ночного", "ночн"},
		{"ночное", "ночн"},
		{"ночной", "ночн"},
		{"ночном", "ночн"},
		{"ночною", "ночн"},
		{"ночную", "ночн"},
		{"ночные", "ночн"},
		{"ночным", "ночн"},
		{"ночными", "ночн"},
		{"ночных", "ночн"},
		{"ночуем", "ночу"},
		{"ночует", "ноч"},
		{"ночуете", "ночует"},
		{"ночуешь", "ночуеш"},
		{"ночую", "ноч"},
		{"ночь", "ноч"},
		{"ночью", "ноч"},
		{"ноша", "нош"},
		{"ношей", "нош"},
		{"ношу", "нош"},
		{"ношусь", "нош"},
		{"нощи", "нощ"},
		{"нощно", "нощн"},
		{"ноют", "ноют"},
		{"ноябре", "ноябр"},
		{"ноябрь", "ноябр"},
		{"ноября", "ноябр"},
		{"нрав", "нрав"},
		{"нрава", "нрав"},
		{"нравах", "нрав"},
		{"нраве", "нрав"},
		{"нравилась", "нрав"},
		{"нравились", "нрав"},
		{"нравилось", "нрав"},
		{"нравился", "нрав"},
		{"нравитесь", "нрав"},
		{"нравится", "нрав"},
		{"нравиться", "нрав"},
		{"нравов", "нрав"},
		{"нравом", "нрав"},
		{"нравоучения", "нравоучен"},
		{"нравственная", "нравствен"},
		{"нравственно", "нравствен"},
		{"нравственного", "нравствен"},
		{"нравственное", "нравствен"},
		{"нравственной", "нравствен"},
		{"нравственном", "нравствен"},
		{"нравственности", "нравствен"},
		{"нравственность", "нравствен"},
		{"нравственною", "нравствен"},
		{"нравственную", "нравствен"},
		{"нравственные", "нравствен"},
		{"нравственным", "нравствен"},
		{"нравственными", "нравствен"},
		{"нравственных", "нравствен"},
		{"нраву", "нрав"},
		{"нравы", "нрав"},
		{"нравятся", "нрав"},
		{"ну", "ну"},
		{"нудная", "нудн"},
		{"нудные", "нудн"},
		{"нудный", "нудн"},
		{"нужда", "нужд"},
		{"нуждается", "нужда"},
		{"нуждался", "нужда"},
		{"нуждах", "нужд"},
		{"нуждаюсь", "нужда"},
		{"нуждаются", "нужда"},
		{"нуждаясь", "нужд"},
		{"нужде", "нужд"},
		{"нуждой", "нужд"},
		{"нужду", "нужд"},
		{"нужды", "нужд"},
		{"нужен", "нуж"},
		{"нужна", "нужн"},
		{"нужная", "нужн"},
		{"нужнее", "нужн"},
		{"нужно", "нужн"},
		{"нужное", "нужн"},
		{"нужной", "нужн"},
		{"нужную", "нужн"},
		{"нужны", "нужн"},
		{"нужные", "нужн"},
		{"нужным", "нужн"},
		{"нужных", "нужн"},
		{"нулю", "нул"},
		{"нумер", "нумер"},
		{"нумера", "нумер"},
		{"нумерам", "нумер"},
		{"нумерами", "нумер"},
		{"нумерах", "нумер"},
		{"нумере", "нумер"},
		{"нумеров", "нумер"},
		{"нумером", "нумер"},
		{"нутру", "нутр"},
		{"ны", "ны"},
		{"ные", "ны"},
		{"ныне", "нын"},
		{"нынешнего", "нынешн"},
		{"нынешнее", "нынешн"},
		{"нынешней", "нынешн"},
		{"нынешнем", "нынешн"},
		{"нынешнему", "нынешн"},
		{"нынешние", "нынешн"},
		{"нынешний", "нынешн"},
		{"нынешнюю", "нынешн"},
		{"нынче", "нынч"},
		{"ньютон", "ньютон"},
		{"ньютоновы", "ньютонов"},
		{"нюхает", "нюха"},
		{"нюхала", "нюха"},
		{"нюханье", "нюхан"},
		{"нюхая", "нюх"},
		{"няне", "нян"},
		{"няней", "нян"},
		{"нянек", "нянек"},
		{"нянечка", "нянечк"},
		{"няни", "нян"},
		{"нянчась", "нянч"},
		{"нянчил", "нянч"},
		{"нянчились", "нянч"},
		{"нянчусь", "нянч"},
		{"нянька", "няньк"},
		{"няньками", "няньк"},
		{"няньке", "няньк"},
		{"няньки", "няньк"},
		{"няньку", "няньк"},
		{"няня", "нян"},
		{"нях", "нях"},
		{"о", "о"},
		{"оазисе", "оазис"},
		{"об", "об"},
		{"оба", "об"},
		{"обагри", "обагр"},
		{"обагрившая", "обагр"},
		{"обагрит", "обагр"},
		{"обанкрутится", "обанкрут"},
		{"обаяние", "обаян"},
		{"обаянием", "обаян"},
		{"обаянию", "обаян"},
		{"обаяния", "обаян"},
		{"обаяньем", "обаян"},
		{"обаятельная", "обаятельн"},
		{"обаятельного", "обаятельн"},
		{"обаятельное", "обаятельн"},
		{"обаятельной", "обаятельн"},
		{"обаятельный", "обаятельн"},
		{"обвалиться", "обвал"},
		{"обварил", "обвар"},
		{"обвел", "обвел"},
		{"обвела", "обвел"},
		{"обветшал", "обветша"},
		{"обвешивают", "обвешива"},
		{"обвив", "обв"},
		{"обвивал", "обвива"},
		{"обвивали", "обвива"},
		{"обвиваясь", "обвив"},
		{"обвила", "обв"},
		{"обвились", "обв"},
		{"обвинение", "обвинен"},
		{"обвинении", "обвинен"},
		{"обвинений", "обвинен"},
		{"обвинению", "обвинен"},
		{"обвинения", "обвинен"},
		{"обвиненного", "обвинен"},
		{"обвинив", "обвин"},
		{"обвинил", "обвин"},
		{"обвинила", "обвин"},
		{"обвинительные", "обвинительн"},
		{"обвинить", "обвин"},
		{"обвинишь", "обвин"},
		{"обвиняете", "обвиня"},
		{"обвиняешь", "обвиня"},
		{"обвинял", "обвиня"},
		{"обвиняли", "обвиня"},
		{"обвинять", "обвиня"},
		{"обвиняю", "обвиня"},
		{"обвиняющим", "обвиня"},
		{"обвитый", "обвит"},
		{"обводила", "обвод"},
		{"обводили", "обвод"},
		{"обводить", "обвод"},
		{"обводят", "обвод"},
		{"обворожительно", "обворожительн"},
		{"обворожительными", "обворожительн"},
		{"обвязал", "обвяза"},
		{"обвязано", "обвяза"},
		{"обвязаны", "обвяза"},
		{"обглоданной", "обглода"},
		{"обдавало", "обдава"},
		{"обдадут", "обдадут"},
		{"обдает", "обда"},
		{"обдаешь", "обда"},
		{"обдал", "обда"},
		{"обдать", "обда"},
		{"обделаешь", "обдела"},
		{"обделать", "обдела"},
		{"обдергивались", "обдергива"},
		{"обдерет", "обдерет"},
		{"обдирать", "обдира"},
		{"обдует", "обд"},
		{"обдуло", "обдул"},
		{"обдумав", "обдума"},
		{"обдумаем", "обдума"},
		{"обдумайте", "обдума"},
		{"обдумал", "обдума"},
		{"обдумала", "обдума"},
		{"обдуман", "обдума"},
		{"обдуманно", "обдума"},
		{"обдумать", "обдума"},
		{"обдумывает", "обдумыва"},
		{"обдумывал", "обдумыва"},
		{"обдумывая", "обдумыв"},
		{"обе", "об"},
		{"обегал", "обега"},
		{"обегая", "обег"},
		{"обед", "обед"},
		{"обеда", "обед"},
		{"обедаем", "обеда"},
		{"обедает", "обеда"},
		{"обедаешь", "обеда"},
		{"обедай", "обеда"},
		{"обедал", "обеда"},
		{"обедала", "обеда"},
		{"обедали", "обеда"},
		{"обедать", "обеда"},
		{"обедах", "обед"},
		{"обедаю", "обеда"},
		{"обедают", "обеда"},
		{"обедая", "обед"},
		{"обеде", "обед"},
		{"обеденной", "обеден"},
		{"обедне", "обедн"},
		{"обедневший", "обедневш"},
		{"обедней", "обедн"},
		{"обедни", "обедн"},
		{"обедню", "обедн"},
		{"обедня", "обедн"},
		{"обедов", "обед"},
		{"обедом", "обед"},
		{"обеду", "обед"},
		{"обеды", "обед"},
		{"обежал", "обежа"},
		{"обежит", "обеж"},
		{"обездоленный", "обездолен"},
		{"обезличивают", "обезличива"},
		{"обезображеннее", "обезображен"},
		{"обезображенной", "обезображен"},
		{"обезображено", "обезображ"},
		{"обезобразили", "обезобраз"},
		{"обезоруженный", "обезоружен"},
		{"обезуме", "обезум"},
		{"обезумел", "обезумел"},
		{"обеим", "обе"},
		{"обеими", "обе"},
		{"обеих", "обе"},
		{"оберегаемая", "оберега"},
		{"оберегали", "оберега"},
		{"оберегать", "оберега"},
		{"оберегаю", "оберега"},
		{"оберечь", "обереч"},
		{"оберешься", "обереш"},
		{"обернется", "обернет"},
		{"обернув", "обернув"},
		{"обернувшись", "обернувш"},
		{"обернул", "обернул"},
		{"обернулась", "обернул"},
		{"обернулись", "обернул"},
		{"обернулось", "обернул"},
		{"обернулся", "обернул"},
		{"обернуты", "обернут"},
		{"обернуться", "обернут"},
		{"обертках", "обертк"},
		{"обескуражены", "обескураж"},
		{"обеспечена", "обеспеч"},
		{"обеспечение", "обеспечен"},
		{"обеспеченный", "обеспечен"},
		{"обеспечено", "обеспеч"},
		{"обеспечить", "обеспеч"},
		{"обеспокоена", "обеспоко"},
		{"обеспокоенная", "обеспокоен"},
		{"обеспокоил", "обеспоко"},
		{"обеспокоила", "обеспоко"},
		{"обеспокоили", "обеспоко"},
		{"обеспокоить", "обеспоко"},
		{"обеспокою", "обеспок"},
		{"обеспокоятся", "обеспоко"},
		{"обессиление", "обессилен"},
		{"обессиленный", "обессилен"},
		{"обессилил", "обессил"},
		{"обессилили", "обессил"},
		{"обетова", "обетов"},
		{"обетование", "обетован"},
		{"обетованной", "обетова"},
		{"обеты", "обет"},
		{"обещает", "обеща"},
		{"обещаете", "обеща"},
		{"обещаетесь", "обеща"},
		{"обещай", "обеща"},
		{"обещал", "обеща"},
		{"обещала", "обеща"},
		{"обещали", "обеща"},
		{"обещались", "обеща"},
		{"обещался", "обеща"},
		{"обещание", "обещан"},
		{"обещанием", "обещан"},
		{"обещаний", "обещан"},
		{"обещанию", "обещан"},
		{"обещания", "обещан"},
		{"обещаниях", "обещан"},
		{"обещанья", "обещан"},
		{"обещать", "обеща"},
		{"обещаюсь", "обеща"},
		{"обещая", "обещ"},
		{"обещаясь", "обещ"},
		{"обжег", "обжег"},
		{"обжегся", "обжег"},
		{"обжечь", "обжеч"},
		{"обживаясь", "обжив"},
		{"обжигаем", "обжига"},
		{"обжигаешь", "обжига"},
		{"обжигательная", "обжигательн"},
		{"обжог", "обжог"},
		{"обжора", "обжор"},
		{"обивал", "обива"},
		{"обивать", "обива"},
		{"обид", "обид"},
		{"обида", "обид"},
		{"обидевшись", "обидевш"},
		{"обидел", "обидел"},
		{"обидела", "обидел"},
		{"обиделась", "обидел"},
		{"обиделись", "обидел"},
		{"обиделся", "обидел"},
		{"обидеть", "обидет"},
		{"обидеться", "обидет"},
		{"обиднее", "обидн"},
		{"обидно", "обидн"},
		{"обидное", "обидн"},
		{"обидной", "обидн"},
		{"обидном", "обидн"},
		{"обидному", "обидн"},
		{"обидны", "обидн"},
		{"обидой", "обид"},
		{"обиду", "обид"},
		{"обидчивость", "обидчив"},
		{"обидчиком", "обидчик"},
		{"обидчику", "обидчик"},
		{"обиды", "обид"},
		{"обидятся", "обид"},
		{"обижает", "обижа"},
		{"обижаете", "обижа"},
		{"обижаетесь", "обижа"},
		{"обижается", "обижа"},
		{"обижаешься", "обижа"},
		{"обижайтесь", "обижа"},
		{"обижалась", "обижа"},
		{"обижался", "обижа"},
		{"обижать", "обижа"},
		{"обижаться", "обижа"},
		{"обижаю", "обижа"},
		{"обижают", "обижа"},
		{"обижаются", "обижа"},
		{"обижена", "обиж"},
		{"обиженная", "обижен"},
		{"обиженно", "обижен"},
		{"обиженный", "обижен"},
		{"обиженным", "обижен"},
		{"обилие", "обил"},
		{"обилием", "обил"},
		{"обилии", "обил"},
		{"обилия", "обил"},
		{"обильно", "обильн"},
		{"обильное", "обильн"},
		{"обильный", "обильн"},
		{"обильным", "обильн"},
		{"обиняков", "обиняк"},
		{"обираешь", "обира"},
		{"обирать", "обира"},
		{"обираю", "обира"},
		{"обит", "об"},
		{"обитавшие", "обита"},
		{"обитатели", "обитател"},
		{"обитатель", "обитател"},
		{"обитая", "обит"},
		{"обитые", "обит"},
		{"обитый", "обит"},
		{"обитым", "обит"},
		{"обиходе", "обиход"},
		{"обкрадывал", "обкрадыва"},
		{"облагодетельствовал", "облагодетельствова"},
		{"облагодетельствована", "облагодетельствова"},
		{"облагодетельствованных", "облагодетельствова"},
		{"облагороженному", "облагорожен"},
		{"облагороженных", "облагорожен"},
		{"обладателем", "обладател"},
		{"облак", "облак"},
		{"облака", "облак"},
		{"облаками", "облак"},
		{"облаках", "облак"},
		{"облаке", "облак"},
		{"облако", "облак"},
		{"облаков", "облак"},
		{"облаком", "облак"},
		{"обласкала", "обласка"},
		{"обласкали", "обласка"},
		{"обласкают", "обласка"},
		{"областей", "област"},
		{"области", "област"},
		{"область", "област"},
		{"облачка", "облачк"},
		{"облачко", "облачк"},
		{"облегчаемы", "облегчаем"},
		{"облегчает", "облегча"},
		{"облегчаете", "облегча"},
		{"облегчается", "облегча"},
		{"облегчал", "облегча"},
		{"облегчалась", "облегча"},
		{"облегчающих", "облегча"},
		{"облегчить", "облегч"},
		{"облезлый", "облезл"},
		{"облекать", "облека"},
		{"облекла", "облекл"},
		{"обленилась", "облен"},
		{"обленился", "облен"},
		{"облепил", "облеп"},
		{"облепленный", "облеплен"},
		{"облетели", "облетел"},
		{"облечена", "облеч"},
		{"обливает", "облива"},
		{"обливала", "облива"},
		{"обливалось", "облива"},
		{"обливаюсь", "облива"},
		{"обливая", "облив"},
		{"обливаясь", "облив"},
		{"облигаций", "облигац"},
		{"облизываясь", "облизыв"},
		{"облик", "облик"},
		{"облике", "облик"},
		{"облилось", "обл"},
		{"облита", "облит"},
		{"облитой", "облит"},
		{"облитый", "облит"},
		{"облить", "обл"},
		{"облиться", "обл"},
		{"обличайте", "облича"},
		{"обличало", "облича"},
		{"обличат", "обличат"},
		{"обличать", "облича"},
		{"обличают", "облича"},
		{"обличающее", "облича"},
		{"обличающие", "облича"},
		{"обличению", "обличен"},
		{"обличения", "обличен"},
		{"обличенного", "обличен"},
		{"обличенных", "обличен"},
		{"обличий", "облич"},
		{"обличили", "облич"},
		{"обличителем", "обличител"},
		{"обличители", "обличител"},
		{"обличить", "облич"},
		{"обличье", "облич"},
		{"облобызал", "облобыза"},
		{"обложив", "облож"},
		{"обложил", "облож"},
		{"обложила", "облож"},
		{"обложит", "облож"},
		{"облокотившись", "облокот"},
		{"облокотилась", "облокот"},
		{"облокотился", "облокот"},
		{"облокотиться", "облокот"},
		{"облокотясь", "облокот"},
		{"обломав", "облома"},
		{"обломаешь", "облома"},
		{"обломками", "обломк"},
		{"обломов", "облом"},
		{"обломова", "обломов"},
		{"обломове", "обломов"},
		{"обломовка", "обломовк"},
		{"обломовках", "обломовк"},
		{"обломовке", "обломовк"},
		{"обломовки", "обломовк"},
		{"обломовкой", "обломовк"},
		{"обломовку", "обломовк"},
		{"обломовская", "обломовск"},
		{"обломовский", "обломовск"},
		{"обломовских", "обломовск"},
		{"обломовского", "обломовск"},
		{"обломовской", "обломовск"},
		{"обломовском", "обломовск"},
		{"обломовскому", "обломовск"},
		{"обломовскую", "обломовск"},
		{"обломову", "обломов"},
		{"обломовцам", "обломовц"},
		{"обломовцев", "обломовц"},
		{"обломовцы", "обломовц"},
		{"обломовщина", "обломовщин"},
		{"обломовы", "обломов"},
		{"обломовым", "обломов"},
		{"обломовых", "обломов"},
		{"обломок", "обломок"},
		{"облупит", "облуп"},
		{"обмакнешь", "обмакнеш"},
		{"обмакнул", "обмакнул"},
		{"обмакнуть", "обмакнут"},
		{"обман", "обма"},
		{"обмана", "обма"},
		{"обманет", "обманет"},
		{"обманете", "обманет"},
		{"обманом", "обман"},
		{"обманувшие", "обманувш"},
		{"обманул", "обманул"},
		{"обманула", "обманул"},
		{"обманули", "обманул"},
		{"обманут", "обманут"},
		{"обманутое", "обманут"},
		{"обманутые", "обманут"},
		{"обмануть", "обманут"},
		{"обмануться", "обманут"},
		{"обманчив", "обманч"},
		{"обманчивый", "обманчив"},
		{"обманщика", "обманщик"},
		{"обманщицы", "обманщиц"},
		{"обманывает", "обманыва"},
		{"обманываете", "обманыва"},
		{"обманываешь", "обманыва"},
		{"обманывал", "обманыва"},
		{"обманывали", "обманыва"},
		{"обманывать", "обманыва"},
		{"обманываю", "обманыва"},
		{"обманывают", "обманыва"},
		{"обманывающих", "обманыва"},
		{"обмахивая", "обмахив"},
		{"обмен", "обм"},
		{"обмена", "обм"},
		{"обмененном", "обменен"},
		{"обменялись", "обменя"},
		{"обменялся", "обменя"},
		{"обмер", "обмер"},
		{"обмеривали", "обмерива"},
		{"обмеривают", "обмерива"},
		{"обмеривая", "обмерив"},
		{"обмерить", "обмер"},
		{"обмести", "обмест"},
		{"обметает", "обмета"},
		{"обмоет", "обмоет"},
		{"обморок", "обморок"},
		{"обморока", "обморок"},
		{"обмороками", "обморок"},
		{"обмороке", "обморок"},
		{"обмороки", "обморок"},
		{"обмороком", "обморок"},
		{"обмороку", "обморок"},
		{"обмундировку", "обмундировк"},
		{"обмывает", "обмыва"},
		{"обмывать", "обмыва"},
		{"обмызганный", "обмызга"},
		{"обнажает", "обнажа"},
		{"обнажаться", "обнажа"},
		{"обнаженная", "обнажен"},
		{"обнаженную", "обнажен"},
		{"обнаженные", "обнажен"},
		{"обнаженных", "обнажен"},
		{"обнажил", "обнаж"},
		{"обнаружен", "обнаруж"},
		{"обнаружена", "обнаруж"},
		{"обнаружением", "обнаружен"},
		{"обнаруживал", "обнаружива"},
		{"обнаруживать", "обнаружива"},
		{"обнаруживая", "обнаружив"},
		{"обнаружил", "обнаруж"},
		{"обнаружили", "обнаруж"},
		{"обнаружило", "обнаруж"},
		{"обнаружилось", "обнаруж"},
		{"обнаружится", "обнаруж"},
		{"обнаружить", "обнаруж"},
		{"обнаружиться", "обнаруж"},
		{"обнеряшиться", "обнеряш"},
		{"обнес", "обнес"},
		{"обнесенный", "обнесен"},
		{"обнимает", "обнима"},
		{"обнимаешь", "обнима"},
		{"обнимал", "обнима"},
		{"обнимала", "обнима"},
		{"обнимались", "обнима"},
		{"обнимать", "обнима"},
		{"обнимаю", "обнима"},
		{"обнимаются", "обнима"},
		{"обнимая", "обним"},
		{"обнимемся", "обним"},
		{"обнимет", "обнимет"},
		{"обними", "обн"},
		{"обновившимся", "обнов"},
		{"обновился", "обнов"},
		{"обновить", "обнов"},
		{"обновки", "обновк"},
		{"обновления", "обновлен"},
		{"обновленного", "обновлен"},
		{"обновлять", "обновля"},
		{"обноски", "обноск"},
		{"обнюхает", "обнюха"},
		{"обняв", "обня"},
		{"обнявшись", "обня"},
		{"обнял", "обня"},
		{"обняла", "обня"},
		{"обнялись", "обня"},
		{"обнять", "обня"},
		{"обо", "об"},
		{"обобрал", "обобра"},
		{"обовьет", "обовьет"},
		{"обовьют", "обовьют"},
		{"ободком", "ободк"},
		{"ободранный", "ободра"},
		{"ободрано", "ободра"},
		{"ободренная", "ободрен"},
		{"ободрившийся", "ободр"},
		{"ободрившись", "ободр"},
		{"ободрил", "ободр"},
		{"ободрилась", "ободр"},
		{"ободрило", "ободр"},
		{"ободрился", "ободр"},
		{"ободрись", "ободр"},
		{"ободрите", "ободр"},
		{"ободритесь", "ободр"},
		{"ободрить", "ободр"},
		{"ободриться", "ободр"},
		{"ободряла", "ободря"},
		{"ободряться", "ободря"},
		{"обоего", "обо"},
		{"обожаемого", "обожа"},
		{"обожает", "обожа"},
		{"обожал", "обожа"},
		{"обожанием", "обожан"},
		{"обожания", "обожан"},
		{"обожающие", "обожа"},
		{"обожая", "обож"},
		{"обожги", "обожг"},
		{"обожгли", "обожгл"},
		{"обожгло", "обожгл"},
		{"обожгу", "обожг"},
		{"обожжен", "обожж"},
		{"обожженный", "обожжен"},
		{"обоз", "обоз"},
		{"обозвав", "обозва"},
		{"обозвал", "обозва"},
		{"обознать", "обозна"},
		{"обозначавшие", "обознача"},
		{"обозначается", "обознача"},
		{"обозначалась", "обознача"},
		{"обозначаются", "обознача"},
		{"обозначена", "обознач"},
		{"обозначено", "обознач"},
		{"обозначился", "обознач"},
		{"обозначить", "обознач"},
		{"обозники", "обозник"},
		{"обозреть", "обозрет"},
		{"обозу", "обоз"},
		{"обои", "обо"},
		{"обоим", "обо"},
		{"обоими", "обо"},
		{"обоих", "обо"},
		{"обой", "об"},
		{"обойдет", "обойдет"},
		{"обойдется", "обойдет"},
		{"обойдешь", "обойдеш"},
		{"обойди", "обойд"},
		{"обойдусь", "обойд"},
		{"обойдутся", "обойдут"},
		{"обойдя", "обойд"},
		{"обойной", "обойн"},
		{"обойти", "обойт"},
		{"обойтись", "обойт"},
		{"обокрали", "обокра"},
		{"обокрасть", "обокраст"},
		{"оболочке", "оболочк"},
		{"оболочку", "оболочк"},
		{"обольет", "обольет"},
		{"обольется", "обольет"},
		{"обольстит", "обольст"},
		{"обольстительнее", "обольстительн"},
		{"обольстительной", "обольстительн"},
		{"обольстительный", "обольстительн"},
		{"обольют", "обольют"},
		{"обомлел", "обомлел"},
		{"обомлели", "обомлел"},
		{"обопрется", "обопрет"},
		{"оборачивается", "оборачива"},
		{"оборачивайте", "оборачива"},
		{"оборачивались", "оборачива"},
		{"оборачивался", "оборачива"},
		{"оборачивая", "оборачив"},
		{"оборачиваясь", "оборачив"},
		{"оборвавший", "оборва"},
		{"оборвал", "оборва"},
		{"оборвала", "оборва"},
		{"оборвалась", "оборва"},
		{"оборвалось", "оборва"},
		{"оборванец", "оборванец"},
		{"оборванной", "оборва"},
		{"оборванный", "оборва"},
		{"оборванца", "оборванц"},
		{"оборванцем", "оборванц"},
		{"оборвать", "оборва"},
		{"оборвет", "оборвет"},
		{"оборвется", "оборвет"},
		{"оборвешься", "оборвеш"},
		{"оборви", "оборв"},
		{"оборвыш", "оборвыш"},
		{"обороняется", "обороня"},
		{"оборот", "оборот"},
		{"оборота", "оборот"},
		{"оборотившийся", "оборот"},
		{"оборотившись", "оборот"},
		{"оборотил", "оборот"},
		{"оборотилась", "оборот"},
		{"оборотился", "оборот"},
		{"оборотится", "оборот"},
		{"оборотиться", "оборот"},
		{"оборотне", "оборотн"},
		{"оборотнем", "оборотн"},
		{"оборотной", "оборотн"},
		{"оборотням", "оборотн"},
		{"оборотом", "оборот"},
		{"оборотясь", "оборот"},
		{"оботрет", "оботрет"},
		{"обошел", "обошел"},
		{"обошла", "обошл"},
		{"обошлась", "обошл"},
		{"обошлись", "обошл"},
		{"обошлось", "обошл"},
		{"обошьют", "обошьют"},
		{"обоюдное", "обоюдн"},
		{"обоюдных", "обоюдн"},
		{"обоями", "обо"},
		{"обоях", "обо"},
		{"обрадовавшись", "обрадова"},
		{"обрадовал", "обрадова"},
		{"обрадовалась", "обрадова"},
		{"обрадовались", "обрадова"},
		{"обрадовался", "обрадова"},
		{"обрадованная", "обрадова"},
		{"обрадованного", "обрадова"},
		{"обрадованный", "обрадова"},
		{"обрадуемся", "обраду"},
		{"обрадуется", "обрад"},
		{"обрадуются", "обрад"},
		{"образ", "образ"},
		{"образа", "образ"},
		{"образами", "образ"},
		{"образе", "образ"},
		{"образина", "образин"},
		{"образов", "образ"},
		{"образовал", "образова"},
		{"образовались", "образова"},
		{"образовалось", "образова"},
		{"образован", "образова"},
		{"образование", "образован"},
		{"образованием", "образован"},
		{"образовании", "образован"},
		{"образованию", "образован"},
		{"образования", "образован"},
		{"образованная", "образова"},
		{"образованного", "образова"},
		{"образованной", "образова"},
		{"образованном", "образова"},
		{"образованность", "образован"},
		{"образованны", "образован"},
		{"образованные", "образова"},
		{"образованный", "образова"},
		{"образованных", "образова"},
		{"образок", "образок"},
		{"образом", "образ"},
		{"образу", "образ"},
		{"образует", "образ"},
		{"образумить", "образум"},
		{"образумиться", "образум"},
		{"образцами", "образц"},
		{"образцовая", "образцов"},
		{"образцовое", "образцов"},
		{"образцовый", "образцов"},
		{"образцу", "образц"},
		{"обрамлен", "обрамл"},
		{"обрамлена", "обрамл"},
		{"обрамленное", "обрамлен"},
		{"обратив", "обрат"},
		{"обратившись", "обрат"},
		{"обратил", "обрат"},
		{"обратила", "обрат"},
		{"обратилась", "обрат"},
		{"обратили", "обрат"},
		{"обратились", "обрат"},
		{"обратило", "обрат"},
		{"обратилось", "обрат"},
		{"обратился", "обрат"},
		{"обратим", "обрат"},
		{"обратит", "обрат"},
		{"обратится", "обрат"},
		{"обратить", "обрат"},
		{"обратиться", "обрат"},
		{"обратно", "обратн"},
		{"обратясь", "обрат"},
		{"обратят", "обрат"},
		{"обратятся", "обрат"},
		{"обращает", "обраща"},
		{"обращается", "обраща"},
		{"обращайте", "обраща"},
		{"обращайтесь", "обраща"},
		{"обращал", "обраща"},
		{"обращалась", "обраща"},
		{"обращали", "обраща"},
		{"обращались", "обраща"},
		{"обращался", "обраща"},
		{"обращать", "обраща"},
		{"обращаться", "обраща"},
		{"обращаюсь", "обраща"},
		{"обращаются", "обраща"},
		{"обращающийся", "обраща"},
		{"обращая", "обра"},
		{"обращаясь", "обра"},
		{"обращен", "обращ"},
		{"обращена", "обращ"},
		{"обращение", "обращен"},
		{"обращением", "обращен"},
		{"обращении", "обращен"},
		{"обращений", "обращен"},
		{"обращения", "обращен"},
		{"обращенная", "обращен"},
		{"обращенном", "обращен"},
		{"обращенному", "обращен"},
		{"обращенных", "обращен"},
		{"обращены", "обращ"},
		{"обращу", "обращ"},
		{"обращусь", "обращ"},
		{"обреешься", "обрееш"},
		{"обрезал", "обреза"},
		{"обрезки", "обрезк"},
		{"обрекает", "обрека"},
		{"обрел", "обрел"},
		{"обречен", "обреч"},
		{"обречена", "обреч"},
		{"обреченного", "обречен"},
		{"обреченные", "обречен"},
		{"обреченными", "обречен"},
		{"обречь", "обреч"},
		{"обрисовывается", "обрисовыва"},
		{"обритый", "обрит"},
		{"обритым", "обрит"},
		{"обробел", "обробел"},
		{"обробела", "обробел"},
		{"оброк", "оброк"},
		{"оброка", "оброк"},
		{"оброке", "оброк"},
		{"оброку", "оброк"},
		{"обронил", "оброн"},
		{"обросший", "обросш"},
		{"оброчных", "оброчн"},
		{"обрубила", "обруб"},
		{"обрубленный", "обрублен"},
		{"обрубок", "обрубок"},
		{"обругает", "обруга"},
		{"обругал", "обруга"},
		{"обругала", "обруга"},
		{"обругали", "обруга"},
		{"обругать", "обруга"},
		{"обругаться", "обруга"},
		{"обругаю", "обруга"},
		{"обругают", "обруга"},
		{"обруч", "обруч"},
		{"обручен", "обруч"},
		{"обручился", "обруч"},
		{"обручится", "обруч"},
		{"обручке", "обручк"},
		{"обрушившиеся", "обруш"},
		{"обрушила", "обруш"},
		{"обрушилась", "обруш"},
		{"обрушились", "обруш"},
		{"обрушится", "обруш"},
		{"обрыв", "обр"},
		{"обрыва", "обрыв"},
		{"обрывается", "обрыва"},
		{"обрывалась", "обрыва"},
		{"обрываю", "обрыва"},
		{"обрывая", "обрыв"},
		{"обрываясь", "обрыв"},
		{"обрывках", "обрывк"},
		{"обрывке", "обрывк"},
		{"обрывки", "обрывк"},
		{"обрывок", "обрывок"},
		{"обрывы", "обрыв"},
		{"обрюзг", "обрюзг"},
		{"обряд", "обряд"},
		{"обряда", "обряд"},
		{"обрядах", "обряд"},
		{"обрядов", "обряд"},
		{"обряды", "обряд"},
		{"обрящете", "обрящет"},
		{"обсаженный", "обсажен"},
		{"обсасывая", "обсасыв"},
		{"обслуживания", "обслуживан"},
		{"обставленная", "обставлен"},
		{"обставленный", "обставлен"},
		{"обстала", "обста"},
		{"обстановив", "обстанов"},
		{"обстановка", "обстановк"},
		{"обстановке", "обстановк"},
		{"обстановки", "обстановк"},
		{"обстановкой", "обстановк"},
		{"обстановку", "обстановк"},
		{"обстоятельно", "обстоятельн"},
		{"обстоятельное", "обстоятельн"},
		{"обстоятельную", "обстоятельн"},
		{"обстоятельным", "обстоятельн"},
		{"обстоятельств", "обстоятельств"},
		{"обстоятельства", "обстоятельств"},
		{"обстоятельствам", "обстоятельств"},
		{"обстоятельствами", "обстоятельств"},
		{"обстоятельствах", "обстоятельств"},
		{"обстоятельстве", "обстоятельств"},
		{"обстоятельство", "обстоятельств"},
		{"обстоятельством", "обстоятельств"},
		{"обстоятельству", "обстоятельств"},
		{"обстреливать", "обстрелива"},
		{"обстриженными", "обстрижен"},
		{"обстроилась", "обстро"},
		{"обструганная", "обструга"},
		{"обступили", "обступ"},
		{"обступило", "обступ"},
		{"обступят", "обступ"},
		{"обсудил", "обсуд"},
		{"обсудим", "обсуд"},
		{"обсудите", "обсуд"},
		{"обсудить", "обсуд"},
		{"обсудят", "обсуд"},
		{"обсуждение", "обсужден"},
		{"обсуживался", "обсужива"},
		{"обсыплет", "обсыплет"},
		{"обтерев", "обтер"},
		{"обтереть", "обтерет"},
		{"обтерпелась", "обтерпел"},
		{"обтертую", "обтерт"},
		{"обтесанный", "обтеса"},
		{"обтирал", "обтира"},
		{"обтирала", "обтира"},
		{"обтирая", "обтир"},
		{"обтирку", "обтирк"},
		{"обтяжку", "обтяжк"},
		{"обуви", "обув"},
		{"обугленный", "обуглен"},
		{"обуется", "об"},
		{"обузу", "обуз"},
		{"обусловлена", "обусловл"},
		{"обут", "обут"},
		{"обутым", "обут"},
		{"обухом", "обух"},
		{"обучался", "обуча"},
		{"обучение", "обучен"},
		{"обуют", "об"},
		{"обуянный", "обуя"},
		{"обхватив", "обхват"},
		{"обхватил", "обхват"},
		{"обхватила", "обхват"},
		{"обхватили", "обхват"},
		{"обхватывает", "обхватыва"},
		{"обход", "обход"},
		{"обходившую", "обход"},
		{"обходил", "обход"},
		{"обходили", "обход"},
		{"обходилось", "обход"},
		{"обходился", "обход"},
		{"обходит", "обход"},
		{"обходится", "обход"},
		{"обходиться", "обход"},
		{"обходом", "обход"},
		{"обходя", "обход"},
		{"обходятся", "обход"},
		{"обхождении", "обхожден"},
		{"обшаривал", "обшарива"},
		{"обшарканные", "обшарка"},
		{"обшарканными", "обшарка"},
		{"обшивала", "обшива"},
		{"обширен", "обшир"},
		{"обширного", "обширн"},
		{"обширную", "обширн"},
		{"обширные", "обширн"},
		{"обширный", "обширн"},
		{"обшит", "обш"},
		{"обшлагом", "обшлаг"},
		{"обшмыганные", "обшмыга"},
		{"общая", "общ"},
		{"общего", "общ"},
		{"общее", "общ"},
		{"общей", "общ"},
		{"общем", "общ"},
		{"общему", "общ"},
		{"общества", "обществ"},
		{"обществе", "обществ"},
		{"общественного", "обществен"},
		{"общественное", "обществен"},
		{"общественной", "обществен"},
		{"общественному", "обществен"},
		{"общественный", "обществен"},
		{"общественным", "обществен"},
		{"общественных", "обществен"},
		{"общество", "обществ"},
		{"обществом", "обществ"},
		{"обществу", "обществ"},
		{"общечеловеческая", "общечеловеческ"},
		{"общие", "общ"},
		{"общий", "общ"},
		{"общим", "общ"},
		{"общих", "общ"},
		{"общую", "общ"},
		{"объедаете", "объеда"},
		{"объездили", "объезд"},
		{"объезжать", "объезжа"},
		{"объем", "объ"},
		{"объемистый", "объемист"},
		{"объемлет", "объемлет"},
		{"объемля", "объемл"},
		{"объемом", "объем"},
		{"объехать", "объеха"},
		{"объявив", "объяв"},
		{"объявил", "объяв"},
		{"объявила", "объяв"},
		{"объявили", "объяв"},
		{"объявились", "объяв"},
		{"объявился", "объяв"},
		{"объявим", "объяв"},
		{"объявит", "объяв"},
		{"объявите", "объяв"},
		{"объявится", "объяв"},
		{"объявить", "объяв"},
		{"объявлен", "объявл"},
		{"объявление", "объявлен"},
		{"объявлены", "объявл"},
		{"объявляет", "объявля"},
		{"объявляется", "объявля"},
		{"объявляйте", "объявля"},
		{"объявлял", "объявля"},
		{"объявляю", "объявля"},
		{"объявляют", "объявля"},
		{"объявляя", "объявл"},
		{"объял", "объя"},
		{"объяснение", "объяснен"},
		{"объяснением", "объяснен"},
		{"объяснений", "объяснен"},
		{"объяснению", "объяснен"},
		{"объяснения", "объяснен"},
		{"объяснениями", "объяснен"},
		{"объясненья", "объяснен"},
		{"объяснил", "объясн"},
		{"объяснила", "объясн"},
		{"объяснили", "объясн"},
		{"объяснились", "объясн"},
		{"объяснилось", "объясн"},
		{"объяснимся", "объясн"},
		{"объяснимым", "объясним"},
		{"объяснись", "объясн"},
		{"объяснит", "объясн"},
		{"объясните", "объясн"},
		{"объяснитесь", "объясн"},
		{"объяснить", "объясн"},
		{"объясниться", "объясн"},
		{"объяснишься", "объясн"},
		{"объясню", "объясн"},
		{"объяснюсь", "объясн"},
		{"объясняемую", "объясня"},
		{"объясняет", "объясня"},
		{"объясняется", "объясня"},
		{"объясняешь", "объясня"},
		{"объяснял", "объясня"},
		{"объяснять", "объясня"},
		{"объясняться", "объясня"},
		{"объясняю", "объясня"},
		{"объясняя", "объясн"},
		{"объясняясь", "объясн"},
		{"объятие", "объят"},
		{"объятии", "объят"},
		{"объятий", "объят"},
		{"объятия", "объят"},
		{"объятиях", "объят"},
		{"объятый", "объят"},
		{"объятья", "объят"},
		{"объятьях", "объят"},
		{"обывательская", "обывательск"},
		{"обывательской", "обывательск"},
		{"обывательскую", "обывательск"},
		{"обыденной", "обыден"},
		{"обыденною", "обыден"},
		{"обыденную", "обыден"},
		{"обыденных", "обыден"},
		{"обыкновение", "обыкновен"},
		{"обыкновению", "обыкновен"},
		{"обыкновения", "обыкновен"},
		{"обыкновенная", "обыкновен"},
		{"обыкновеннейших", "обыкновен"},
		{"обыкновенно", "обыкновен"},
		{"обыкновенного", "обыкновен"},
		{"обыкновенное", "обыкновен"},
		{"обыкновенной", "обыкновен"},
		{"обыкновенном", "обыкновен"},
		{"обыкновенною", "обыкновен"},
		{"обыкновенную", "обыкновен"},
		{"обыкновенны", "обыкновен"},
		{"обыкновенные", "обыкновен"},
		{"обыкновенный", "обыкновен"},
		{"обыкновенным", "обыкновен"},
		{"обыкновенных", "обыкновен"},
		{"обыск", "обыск"},
		{"обыски", "обыск"},
		{"обыскивает", "обыскива"},
		{"обыскивайте", "обыскива"},
		{"обыскивать", "обыскива"},
		{"обыском", "обыск"},
		{"обычаев", "обыча"},
		{"обычаи", "обыча"},
		{"обычай", "обыча"},
		{"обычаю", "обыча"},
		{"обычное", "обычн"},
		{"обычной", "обычн"},
		{"обычную", "обычн"},
		{"обычные", "обычн"},
		{"обычным", "обычн"},
		{"обычными", "обычн"},
		{"обычных", "обычн"},
		{"обыщем", "обыщ"},
		{"обыщите", "обыщ"},
		{"обьятиях", "обьят"},
		{"обяжете", "обяжет"},
		{"обязан", "обяза"},
		{"обязана", "обяза"},
		{"обязанной", "обяза"},
		{"обязанностей", "обязан"},
		{"обязанности", "обязан"},
		{"обязанность", "обязан"},
		{"обязанностью", "обязан"},
		{"обязанностях", "обязан"},
		{"обязанною", "обяза"},
		{"обязанным", "обяза"},
		{"обязаны", "обяза"},
		{"обязательная", "обязательн"},
		{"обязательной", "обязательн"},
		{"обязательны", "обязательн"},
		{"обязательный", "обязательн"},
		{"обязательных", "обязательн"},
		{"обязательств", "обязательств"},
		{"обязательства", "обязательств"},
		{"обязательствах", "обязательств"},
		{"обязательство", "обязательств"},
		{"овал", "ова"},
		{"овальной", "овальн"},
		{"овальными", "овальн"},
		{"овацию", "овац"},
		{"овес", "овес"},
		{"овечку", "овечк"},
		{"овеют", "овеют"},
		{"овеяла", "овея"},
		{"овин", "овин"},
		{"овином", "овин"},
		{"овладевает", "овладева"},
		{"овладевала", "овладева"},
		{"овладевали", "овладева"},
		{"овладевало", "овладева"},
		{"овладевать", "овладева"},
		{"овладевшей", "овладевш"},
		{"овладел", "овладел"},
		{"овладела", "овладел"},
		{"овладели", "овладел"},
		{"овладело", "овладел"},
		{"овощами", "овощ"},
		{"овощи", "овощ"},
		{"овощные", "овощн"},
		{"овраг", "овраг"},
		{"оврага", "овраг"},
		{"оврагам", "овраг"},
		{"овраге", "овраг"},
		{"оврагом", "овраг"},
		{"овса", "овс"},
		{"овсе", "овс"},
		{"овца", "овц"},
		{"овцой", "овц"},
		{"овцы", "овц"},
		{"овчинин", "овчинин"},
		{"овчинину", "овчинин"},
		{"огарка", "огарк"},
		{"огарками", "огарк"},
		{"огарки", "огарк"},
		{"огарок", "огарок"},
		{"огибавшую", "огиба"},
		{"огибает", "огиба"},
		{"огибали", "огиба"},
		{"огибая", "огиб"},
		{"огласку", "огласк"},
		{"оглоблю", "оглобл"},
		{"оглобля", "оглобл"},
		{"оглохла", "оглохл"},
		{"оглушение", "оглушен"},
		{"оглушенный", "оглушен"},
		{"оглушила", "оглуш"},
		{"оглушили", "оглуш"},
		{"оглядев", "огляд"},
		{"оглядевшись", "оглядевш"},
		{"оглядел", "оглядел"},
		{"оглядела", "оглядел"},
		{"огляделся", "оглядел"},
		{"оглядеть", "оглядет"},
		{"оглядимся", "огляд"},
		{"оглядки", "оглядк"},
		{"оглядкой", "оглядк"},
		{"оглядывает", "оглядыва"},
		{"оглядывается", "оглядыва"},
		{"оглядывайся", "оглядыва"},
		{"оглядывал", "оглядыва"},
		{"оглядывала", "оглядыва"},
		{"оглядывался", "оглядыва"},
		{"оглядывание", "оглядыван"},
		{"оглядывать", "оглядыва"},
		{"оглядывают", "оглядыва"},
		{"оглядываются", "оглядыва"},
		{"оглядывая", "оглядыв"},
		{"оглядываяась", "оглядывая"},
		{"оглядываясь", "оглядыв"},
		{"оглянемся", "оглян"},
		{"оглянется", "оглянет"},
		{"оглянись", "оглян"},
		{"оглянитесь", "оглян"},
		{"оглянувшись", "оглянувш"},
		{"оглянула", "оглянул"},
		{"оглянулась", "оглянул"},
		{"оглянулись", "оглянул"},
		{"оглянулся", "оглянул"},
		{"оглянуться", "оглянут"},
		{"огне", "огн"},
		{"огневая", "огнев"},
		{"огневицей", "огневиц"},
		{"огневой", "огнев"},
		{"огневые", "огнев"},
		{"огневым", "огнев"},
		{"огней", "огн"},
		{"огнекрасная", "огнекрасн"},
		{"огнекрасные", "огнекрасн"},
		{"огнем", "огн"},
		{"огненного", "огнен"},
		{"огненной", "огнен"},
		{"огненные", "огнен"},
		{"огненным", "огнен"},
		{"огненных", "огнен"},
		{"огни", "огн"},
		{"огнь", "огн"},
		{"огню", "огн"},
		{"огня", "огн"},
		{"огнями", "огн"},
		{"огнях", "огн"},
		{"ого", "ог"},
		{"оговор", "оговор"},
		{"огонек", "огонек"},
		{"огонь", "огон"},
		{"огонька", "огоньк"},
		{"огоньки", "огоньк"},
		{"огоньком", "огоньк"},
		{"огород", "огород"},
		{"огорода", "огород"},
		{"огородам", "огород"},
		{"огородами", "огород"},
		{"огороде", "огород"},
		{"огородить", "огород"},
		{"огородишком", "огородишк"},
		{"огородницы", "огородниц"},
		{"огородом", "огород"},
		{"огороду", "огород"},
		{"огороды", "огород"},
		{"огорошат", "огорошат"},
		{"огорошивает", "огорошива"},
		{"огорошить", "огорош"},
		{"огорчать", "огорча"},
		{"огорчен", "огорч"},
		{"огорчением", "огорчен"},
		{"огорчения", "огорчен"},
		{"огорчениями", "огорчен"},
		{"огорченный", "огорчен"},
		{"огорчены", "огорч"},
		{"огорчил", "огорч"},
		{"огорчился", "огорч"},
		{"огорчить", "огорч"},
		{"ограбил", "ограб"},
		{"ограбили", "ограб"},
		{"ограбить", "ограб"},
		{"ограблен", "ограбл"},
		{"ограбленной", "ограблен"},
		{"ограбленным", "ограблен"},
		{"ограда", "оград"},
		{"оградах", "оград"},
		{"ограде", "оград"},
		{"оградить", "оград"},
		{"оградой", "оград"},
		{"ограду", "оград"},
		{"ограды", "оград"},
		{"оградясь", "оград"},
		{"огражу", "ограж"},
		{"ограниченные", "ограничен"},
		{"ограничивался", "ограничива"},
		{"ограничилось", "огранич"},
		{"ограничился", "огранич"},
		{"ограничится", "огранич"},
		{"ограничить", "огранич"},
		{"огрел", "огрел"},
		{"огромная", "огромн"},
		{"огромней", "огромн"},
		{"огромнейшие", "огромн"},
		{"огромнейшими", "огромн"},
		{"огромного", "огромн"},
		{"огромное", "огромн"},
		{"огромной", "огромн"},
		{"огромном", "огромн"},
		{"огромному", "огромн"},
		{"огромною", "огромн"},
		{"огромные", "огромн"},
		{"огромный", "огромн"},
		{"огромным", "огромн"},
		{"огромных", "огромн"},
		{"огрызался", "огрыза"},
		{"огрызаться", "огрыза"},
		{"огурец", "огурец"},
		{"огурцами", "огурц"},
		{"огурцов", "огурц"},
		{"огурцы", "огурц"},
		{"огурчик", "огурчик"},
		{"одарен", "одар"},
		{"одаренные", "одарен"},
		{"одев", "од"},
		{"одевается", "одева"},
		{"одеваешься", "одева"},
		{"одевайся", "одева"},
		{"одевала", "одева"},
		{"одевалась", "одева"},
		{"одевали", "одева"},
		{"одевался", "одева"},
		{"одевать", "одева"},
		{"одеваться", "одева"},
		{"одевают", "одева"},
		{"одевающаяся", "одева"},
		{"одеваясь", "одев"},
		{"одевшись", "одевш"},
		{"одежа", "одеж"},
		{"одежда", "одежд"},
		{"одежде", "одежд"},
		{"одеждой", "одежд"},
		{"одежду", "одежд"},
		{"одежды", "одежд"},
		{"одежке", "одежк"},
		{"одежонку", "одежонк"},
		{"одеколону", "одеколон"},
		{"одел", "одел"},
		{"оделась", "одел"},
		{"оделся", "одел"},
		{"оденется", "оденет"},
		{"оденешь", "оденеш"},
		{"одену", "оден"},
		{"оденусь", "оден"},
		{"оденут", "оденут"},
		{"одеревенелыми", "одеревенел"},
		{"одержим", "одерж"},
		{"одессе", "одесс"},
		{"одессы", "одесс"},
		{"одет", "одет"},
		{"одета", "одет"},
		{"одетая", "одет"},
		{"одетого", "одет"},
		{"одетой", "одет"},
		{"одетому", "одет"},
		{"одеты", "одет"},
		{"одетые", "одет"},
		{"одетый", "одет"},
		{"одетым", "одет"},
		{"одеть", "одет"},
		{"одеться", "одет"},
		{"одеяла", "одея"},
		{"одеяло", "одея"},
		{"одеялом", "одеял"},
		{"одеяние", "одеян"},
		{"один", "один"},
		{"одинаким", "одинак"},
		{"одинакова", "одинаков"},
		{"одинаково", "одинаков"},
		{"одинаковое", "одинаков"},
		{"одинаковы", "одинаков"},
		{"одинаковым", "одинаков"},
		{"одинехонька", "одинехоньк"},
		{"одинешенька", "одинешеньк"},
		{"одиннадцати", "одиннадцат"},
		{"одиннадцатой", "одиннадцат"},
		{"одиннадцатом", "одиннадцат"},
		{"одиннадцатый", "одиннадцат"},
		{"одиннадцать", "одиннадца"},
		{"одинок", "одинок"},
		{"одинока", "одинок"},
		{"одинокий", "одинок"},
		{"одиноким", "одинок"},
		{"одиноко", "одинок"},
		{"одинокой", "одинок"},
		{"одиноком", "одинок"},
		{"одинокому", "одинок"},
		{"одиночества", "одиночеств"},
		{"одиночестве", "одиночеств"},
		{"одиночество", "одиночеств"},
		{"одиночку", "одиночк"},
		{"одиночное", "одиночн"},
		{"одичал", "одича"},
		{"одичалая", "одичал"},
		{"одна", "одн"},
		{"однажды", "однажд"},
		{"однако", "однак"},
		{"однакож", "однакож"},
		{"однех", "однех"},
		{"одни", "одн"},
		{"одним", "одн"},
		{"одними", "одн"},
		{"одних", "одн"},
		{"одно", "одн"},
		{"одноважды", "одноважд"},
		{"одновременно", "одновремен"},
		{"одного", "одн"},
		{"однозвучно", "однозвучн"},
		{"однозвучны", "однозвучн"},
		{"однозвучным", "однозвучн"},
		{"одноименной", "одноимен"},
		{"одной", "одн"},
		{"однократное", "однократн"},
		{"одном", "одн"},
		{"одному", "одн"},
		{"однообразие", "однообраз"},
		{"однообразием", "однообраз"},
		{"однообразия", "однообраз"},
		{"однообразно", "однообразн"},
		{"однообразное", "однообразн"},
		{"однообразною", "однообразн"},
		{"однообразную", "однообразн"},
		{"однообразны", "однообразн"},
		{"однообразный", "однообразн"},
		{"однообразьи", "однообраз"},
		{"одностороннего", "односторон"},
		{"односторонний", "односторон"},
		{"односторонним", "односторон"},
		{"одноцвет", "одноцвет"},
		{"одною", "одн"},
		{"одну", "одн"},
		{"одобрение", "одобрен"},
		{"одобрением", "одобрен"},
		{"одобрительно", "одобрительн"},
		{"одобрить", "одобр"},
		{"одобряла", "одобря"},
		{"одолеваемый", "одолева"},
		{"одолевает", "одолева"},
		{"одолевала", "одолева"},
		{"одолевают", "одолева"},
		{"одолел", "одолел"},
		{"одолела", "одолел"},
		{"одолели", "одолел"},
		{"одолело", "одолел"},
		{"одолеть", "одолет"},
		{"одолжайтесь", "одолжа"},
		{"одолжение", "одолжен"},
		{"одолжением", "одолжен"},
		{"одолжите", "одолж"},
		{"одонцовым", "одонцов"},
		{"одр", "одр"},
		{"одре", "одр"},
		{"одумается", "одума"},
		{"одумался", "одума"},
		{"одурачить", "одурач"},
		{"одуревшими", "одуревш"},
		{"одуреешь", "одурееш"},
		{"одурения", "одурен"},
		{"одурманения", "одурманен"},
		{"одурманивающий", "одурманива"},
		{"одутловатым", "одутловат"},
		{"одушевлена", "одушевл"},
		{"одушевление", "одушевлен"},
		{"одушевлением", "одушевлен"},
		{"одушевленные", "одушевлен"},
		{"оды", "од"},
		{"одышка", "одышк"},
		{"одышки", "одышк"},
		{"ожерельем", "ожерел"},
		{"ожесточением", "ожесточен"},
		{"ожесточенная", "ожесточен"},
		{"ожесточенное", "ожесточен"},
		{"оживали", "ожива"},
		{"оживет", "оживет"},
		{"оживилась", "ожив"},
		{"оживили", "ожив"},
		{"оживить", "ожив"},
		{"оживление", "оживлен"},
		{"оживлением", "оживлен"},
		{"оживленнее", "оживлен"},
		{"оживленно", "оживлен"},
		{"оживленном", "оживлен"},
		{"оживленностью", "оживлен"},
		{"оживленный", "оживлен"},
		{"оживлю", "оживл"},
		{"оживляет", "оживля"},
		{"оживлялась", "оживля"},
		{"оживлялись", "оживля"},
		{"оживлялся", "оживля"},
		{"оживляющих", "оживля"},
		{"оживляясь", "оживл"},
		{"оживут", "оживут"},
		{"ожидавшему", "ожида"},
		{"ожидавший", "ожида"},
		{"ожидает", "ожида"},
		{"ожидаете", "ожида"},
		{"ожидается", "ожида"},
		{"ожидал", "ожида"},
		{"ожидала", "ожида"},
		{"ожидали", "ожида"},
		{"ожидание", "ожидан"},
		{"ожиданием", "ожидан"},
		{"ожидании", "ожидан"},
		{"ожиданий", "ожидан"},
		{"ожидания", "ожидан"},
		{"ожидать", "ожида"},
		{"ожидающий", "ожида"},
		{"ожидающую", "ожида"},
		{"ожидая", "ожид"},
		{"ожидовел", "ожидовел"},
		{"ожил", "ож"},
		{"ожиревший", "ожиревш"},
		{"оз", "оз"},
		{"озаботится", "озабот"},
		{"озабочен", "озабоч"},
		{"озабоченнее", "озабочен"},
		{"озабоченно", "озабочен"},
		{"озабоченную", "озабочен"},
		{"озабоченный", "озабочен"},
		{"озабоченным", "озабочен"},
		{"озадачен", "озадач"},
		{"озадаченный", "озадачен"},
		{"озадачила", "озадач"},
		{"озаренная", "озарен"},
		{"озарены", "озар"},
		{"озарил", "озар"},
		{"озарила", "озар"},
		{"озарили", "озар"},
		{"озарилось", "озар"},
		{"озарился", "озар"},
		{"озарим", "озар"},
		{"озарит", "озар"},
		{"озарялись", "озаря"},
		{"озаряющего", "озаря"},
		{"оземь", "озем"},
		{"озер", "озер"},
		{"озера", "озер"},
		{"озерах", "озер"},
		{"озере", "озер"},
		{"озерна", "озерн"},
		{"озеро", "озер"},
		{"озером", "озер"},
		{"озеру", "озер"},
		{"озими", "оз"},
		{"озимь", "озим"},
		{"озирается", "озира"},
		{"озирал", "озира"},
		{"озирала", "озира"},
		{"озиралась", "озира"},
		{"озирался", "озира"},
		{"озираньем", "озиран"},
		{"озираясь", "озир"},
		{"озлились", "озл"},
		{"озлился", "озл"},
		{"озлобившись", "озлоб"},
		{"озлобленные", "озлоблен"},
		{"ознакомиться", "ознаком"},
		{"ознаменовало", "ознаменова"},
		{"ознаменовалось", "ознаменова"},
		{"означавшие", "означа"},
		{"означает", "означа"},
		{"означается", "означа"},
		{"означенного", "означен"},
		{"озноб", "озноб"},
		{"озноба", "озноб"},
		{"ознобил", "озноб"},
		{"озябла", "озябл"},
		{"озябшие", "озябш"},
		{"озябший", "озябш"},
		{"озябшими", "озябш"},
		// {"ой", "о"},
		{"ока", "ок"},
		{"окажется", "окажет"},
		{"оказалась", "оказа"},
		{"оказалось", "оказа"},
		{"оказался", "оказа"},
		{"оказанных", "оказа"},
		{"оказано", "оказа"},
		{"оказать", "оказа"},
		{"оказии", "оказ"},
		{"оказываемым", "оказыва"},
		{"оказывает", "оказыва"},
		{"оказывается", "оказыва"},
		{"оказывала", "оказыва"},
		{"оказывалась", "оказыва"},
		{"оказывали", "оказыва"},
		{"оказывалось", "оказыва"},
		{"оказывают", "оказыва"},
		{"окаймляющими", "окаймля"},
		{"окаменеет", "окаменеет"},
		{"окаменел", "окаменел"},
		{"окаменелая", "окаменел"},
		{"окаменелость", "окаменел"},
		{"окаменелых", "окаменел"},
		{"окаменяющий", "окаменя"},
		{"оканчивалось", "оканчива"},
		{"окарикатурить", "окарикатур"},
		{"окатил", "окат"},
		{"окаянная", "окая"},
		{"окаянного", "окая"},
		{"окаянные", "окая"},
		{"окаянный", "окая"},
		{"океан", "океа"},
		{"океане", "океан"},
		{"океаны", "океа"},
		{"окидывает", "окидыва"},
		{"окидывая", "окидыв"},
		{"окинет", "окинет"},
		{"окинешь", "окинеш"},
		{"окинув", "окинув"},
		{"окинул", "окинул"},
		{"оклад", "оклад"},
		{"окладов", "оклад"},
		{"оклеветал", "оклевета"},
		{"оклеветали", "оклевета"},
		{"оклеивали", "оклеива"},
		{"окликает", "оклика"},
		{"окликать", "оклика"},
		{"окликнул", "окликнул"},
		{"окликнули", "окликнул"},
		{"окликнуть", "окликнут"},
		{"окна", "окн"},
		{"окнам", "окн"},
		{"окнами", "окн"},
		{"окнах", "окн"},
		{"окне", "окн"},
		{"окно", "окн"},
		{"окном", "окн"},
		{"окну", "окн"},
		{"око", "ок"},
		{"оков", "ок"},
		{"окованная", "окова"},
		{"окованный", "окова"},
		{"оковы", "оков"},
		{"оковывает", "оковыва"},
		{"оковывать", "оковыва"},
		{"околдован", "околдова"},
		{"околдованная", "околдова"},
		{"околицей", "околиц"},
		{"около", "окол"},
		{"околотке", "околотк"},
		{"околотку", "околотк"},
		{"околоток", "околоток"},
		{"оком", "ок"},
		{"окон", "окон"},
		{"оконной", "окон"},
		{"оконные", "окон"},
		{"окончание", "окончан"},
		{"окончании", "окончан"},
		{"окончания", "окончан"},
		{"окончательнее", "окончательн"},
		{"окончательно", "окончательн"},
		{"окончательного", "окончательн"},
		{"окончательное", "окончательн"},
		{"окончательной", "окончательн"},
		{"окончательному", "окончательн"},
		{"окончательные", "окончательн"},
		{"окончательный", "окончательн"},
		{"окончательным", "окончательн"},
		{"окончательных", "окончательн"},
		{"окончен", "оконч"},
		{"окончив", "оконч"},
		{"окончил", "оконч"},
		{"окончила", "оконч"},
		{"окончит", "оконч"},
		{"окончится", "оконч"},
		{"окончить", "оконч"},
		{"окопать", "окопа"},
		{"окорока", "окорок"},
		{"окороков", "окорок"},
		{"окостенел", "окостенел"},
		{"окостенела", "окостенел"},
		{"окостенелый", "окостенел"},
		{"окоченели", "окоченел"},
		{"окоченил", "окочен"},
		{"окошечко", "окошечк"},
		{"окошка", "окошк"},
		{"окошке", "окошк"},
		{"окошки", "окошк"},
		{"окошко", "окошк"},
		{"окошку", "окошк"},
		{"окраинах", "окраин"},
		{"окраску", "окраск"},
		{"окрашена", "окраш"},
		{"окрашены", "окраш"},
		{"окрест", "окрест"},
		{"окрестив", "окрест"},
		{"окрестил", "окрест"},
		{"окрестности", "окрестн"},
		{"окрестность", "окрестн"},
		{"окрестный", "окрестн"},
		{"окрестных", "окрестн"},
		{"окрик", "окрик"},
		{"окрика", "окрик"},
		{"окровавленного", "окровавлен"},
		{"окровавленную", "окровавлен"},
		{"окроме", "окром"},
		{"окромя", "окром"},
		{"окрошки", "окрошк"},
		{"округлилась", "округл"},
		{"округлости", "округл"},
		{"округляли", "округля"},
		{"округлять", "округля"},
		{"окружавшее", "окружа"},
		{"окружавшей", "окружа"},
		{"окружали", "окружа"},
		{"окружающего", "окружа"},
		{"окружающее", "окружа"},
		{"окружающей", "окружа"},
		{"окружающему", "окружа"},
		{"окружающие", "окружа"},
		{"окружающий", "окружа"},
		{"окружающим", "окружа"},
		{"окружающих", "окружа"},
		{"окружающую", "окружа"},
		{"окружены", "окруж"},
		{"окружила", "окруж"},
		{"окружили", "окруж"},
		{"окружит", "окруж"},
		{"окружить", "окруж"},
		{"окружном", "окружн"},
		{"окрыляли", "окрыля"},
		{"октавой", "октав"},
		{"октаву", "октав"},
		{"октябрь", "октябр"},
		{"октябрьском", "октябрьск"},
		{"октябрьскую", "октябрьск"},
		{"октября", "октябр"},
		{"окует", "ок"},
		{"окунулась", "окунул"},
		{"окуня", "окун"},
		{"окупаются", "окупа"},
		{"окутали", "окута"},
		{"окуталось", "окута"},
		{"оле", "ол"},
		{"оледенела", "оледенел"},
		{"оледенивший", "оледен"},
		{"оледенил", "оледен"},
		{"оледенило", "оледен"},
		{"олей", "ол"},
		{"олени", "олен"},
		{"оленька", "оленьк"},
		{"олечка", "олечк"},
		{"олешкин", "олешкин"},
		{"оли", "ол"},
		{"олимпийские", "олимпийск"},
		{"олифе", "олиф"},
		{"олицетворенный", "олицетворен"},
		{"олово", "олов"},
		{"олух", "олух"},
		{"олухи", "олух"},
		{"олухов", "олух"},
		{"ольга", "ольг"},
		{"ольге", "ольг"},
		{"ольги", "ольг"},
		{"ольгиной", "ольгин"},
		{"ольгой", "ольг"},
		{"ольгу", "ольг"},
		{"ольхи", "ольх"},
		{"ольхой", "ольх"},
		{"олюшка", "олюшк"},
		{"олюшкой", "олюшк"},
		{"оля", "ол"},
		{"омаров", "омар"},
		{"омбрельке", "омбрельк"},
		{"омерзение", "омерзен"},
		{"омерзением", "омерзен"},
		{"омерзения", "омерзен"},
		{"омерзительно", "омерзительн"},
		{"омрачалось", "омрача"},
		{"омрачался", "омрача"},
		{"омрачение", "омрачен"},
		{"омрачением", "омрачен"},
		{"омрачения", "омрачен"},
		{"омрачилось", "омрач"},
		{"ому", "ом"},
		{"омут", "омут"},
		{"омута", "омут"},
		{"омуте", "омут"},
		{"омываемых", "омыва"},
		{"омыт", "ом"},
		{"омытый", "омыт"},
		{"он", "он"},
		{"она", "он"},
		{"онемев", "онем"},
		{"онемела", "онемел"},
		{"онемели", "онемел"},
		{"онемелое", "онемел"},
		{"онемение", "онемен"},
		{"онеметь", "онемет"},
		{"онерами", "онер"},
		{"они", "он"},
		{"онисим", "онис"},
		{"онисимовна", "онисимовн"},
		{"онисимовны", "онисимовн"},
		{"онисиму", "онисим"},
		{"оно", "он"},
		{"оного", "он"},
		{"оную", "он"},
		{"оны", "он"},
		{"опадающий", "опада"},
		{"опал", "опа"},
		{"опаленная", "опален"},
		{"опаленные", "опален"},
		{"опаленный", "опален"},
		{"опалит", "опал"},
		{"опамятовавшись", "опамятова"},
		{"опары", "опар"},
		{"опасался", "опаса"},
		{"опасаясь", "опас"},
		{"опасен", "опас"},
		{"опасение", "опасен"},
		{"опасением", "опасен"},
		{"опасений", "опасен"},
		{"опасения", "опасен"},
		{"опасениях", "опасен"},
		{"опасливо", "опаслив"},
		{"опасна", "опасн"},
		{"опасная", "опасн"},
		{"опаснее", "опасн"},
		{"опасно", "опасн"},
		{"опасного", "опасн"},
		{"опасное", "опасн"},
		{"опасностей", "опасн"},
		{"опасности", "опасн"},
		{"опасность", "опасн"},
		{"опасностями", "опасн"},
		{"опасные", "опасн"},
		{"опасный", "опасн"},
		{"опасным", "опасн"},
		{"опасных", "опасн"},
		{"опеки", "опек"},
		{"опекой", "опек"},
		{"опеку", "опек"},
		{"опекун", "опекун"},
		{"опекуна", "опекун"},
		{"опекуном", "опекун"},
		{"опера", "опер"},
		{"операции", "операц"},
		{"операцию", "операц"},
		{"опере", "опер"},
		{"опередил", "оперед"},
		{"опередить", "оперед"},
		{"оперлась", "оперл"},
		{"оперном", "оперн"},
		{"оперся", "опер"},
		{"оперу", "опер"},
		{"опершись", "оперш"},
		{"оперы", "опер"},
		{"опечаленный", "опечален"},
		{"опечалилась", "опечал"},
		{"опечалится", "опечал"},
		{"опечаток", "опечаток"},
		{"опиваете", "опива"},
		{"опирается", "опира"},
		{"опираясь", "опир"},
		{"описав", "описа"},
		{"описал", "описа"},
		{"описание", "описан"},
		{"описанием", "описан"},
		{"описании", "описан"},
		{"описаний", "описан"},
		{"описанная", "описа"},
		{"описанное", "описа"},
		{"описано", "описа"},
		{"описать", "описа"},
		{"описи", "опис"},
		{"описывал", "описыва"},
		{"описывать", "описыва"},
		{"опишу", "опиш"},
		{"оплакала", "оплака"},
		{"оплакиваю", "оплакива"},
		{"оплевать", "оплева"},
		{"оплетая", "оплет"},
		{"оплетет", "оплетет"},
		{"оплетешь", "оплетеш"},
		{"оплеуха", "оплеух"},
		{"оплеуху", "оплеух"},
		{"оплодотворяется", "оплодотворя"},
		{"оподлит", "оподл"},
		{"опоздавший", "опозда"},
		{"опоздает", "опозда"},
		{"опоздал", "опозда"},
		{"опоздала", "опозда"},
		{"опоздали", "опозда"},
		{"опоздать", "опозда"},
		{"опозоренная", "опозорен"},
		{"опозорить", "опозор"},
		{"опозориться", "опозор"},
		{"опомнившись", "опомн"},
		{"опомнилась", "опомн"},
		{"опомнился", "опомн"},
		{"опомнись", "опомн"},
		{"опомнитесь", "опомн"},
		{"опомниться", "опомн"},
		{"опомнясь", "опомн"},
		{"опора", "опор"},
		{"опору", "опор"},
		{"опоры", "опор"},
		{"опосля", "опосл"},
		{"опочиет", "опочиет"},
		{"опошлить", "опошл"},
		{"оппозицию", "оппозиц"},
		{"оппонировать", "оппонирова"},
		{"оправдает", "оправда"},
		{"оправдан", "оправда"},
		{"оправдана", "оправда"},
		{"оправдание", "оправдан"},
		{"оправданию", "оправдан"},
		{"оправдания", "оправдан"},
		{"оправдательные", "оправдательн"},
		{"оправдать", "оправда"},
		{"оправдаю", "оправда"},
		{"оправдывавшее", "оправдыва"},
		{"оправдывавшим", "оправдыва"},
		{"оправдывается", "оправдыва"},
		{"оправдываешь", "оправдыва"},
		{"оправдывалась", "оправдыва"},
		{"оправдывали", "оправдыва"},
		{"оправдывались", "оправдыва"},
		{"оправдывался", "оправдыва"},
		{"оправдывать", "оправдыва"},
		{"оправдываться", "оправдыва"},
		{"оправдываюсь", "оправдыва"},
		{"оправдывают", "оправдыва"},
		{"оправдывая", "оправдыв"},
		{"оправе", "оправ"},
		{"оправив", "оправ"},
		{"оправившись", "оправ"},
		{"оправила", "оправ"},
		{"оправилась", "оправ"},
		{"оправился", "оправ"},
		{"оправиться", "оправ"},
		{"оправлюсь", "оправл"},
		{"оправляла", "оправля"},
		{"оправлялась", "оправля"},
		{"оправляться", "оправля"},
		{"оправляя", "оправл"},
		{"опрашивает", "опрашива"},
		{"опрашивали", "опрашива"},
		{"определен", "определ"},
		{"определения", "определен"},
		{"определенная", "определен"},
		{"определеннее", "определен"},
		{"определенно", "определен"},
		{"определенного", "определен"},
		{"определенное", "определен"},
		{"определенной", "определен"},
		{"определенности", "определен"},
		{"определенную", "определен"},
		{"определенные", "определен"},
		{"определенный", "определен"},
		{"определенных", "определен"},
		{"определив", "определ"},
		{"определившеюся", "определ"},
		{"определил", "определ"},
		{"определился", "определ"},
		{"определительно", "определительн"},
		{"определить", "определ"},
		{"определю", "определ"},
		{"определял", "определя"},
		{"определяя", "определ"},
		{"опредленным", "опредлен"},
		{"опричь", "оприч"},
		{"опробую", "опроб"},
		{"опровергает", "опроверга"},
		{"опровергал", "опроверга"},
		{"опровергнутых", "опровергнут"},
		{"опровергнуть", "опровергнут"},
		{"опрокидыванье", "опрокидыван"},
		{"опрокидываться", "опрокидыва"},
		{"опрокидываясь", "опрокидыв"},
		{"опрокинул", "опрокинул"},
		{"опрокинутого", "опрокинут"},
		{"опрокинутое", "опрокинут"},
		{"опрокинутою", "опрокинут"},
		{"опрокинутую", "опрокинут"},
		{"опрометчиво", "опрометчив"},
		{"опрометью", "опромет"},
		{"опросила", "опрос"},
		{"опротивели", "опротивел"},
		{"опрыскивает", "опрыскива"},
		{"опрятно", "опрятн"},
		{"опрятное", "опрятн"},
		{"опрятность", "опрятн"},
		{"оптический", "оптическ"},
		{"оптом", "опт"},
		{"опубликован", "опубликова"},
		{"опускаем", "опуска"},
		{"опускает", "опуска"},
		{"опускаете", "опуска"},
		{"опускается", "опуска"},
		{"опускаешься", "опуска"},
		{"опускал", "опуска"},
		{"опускала", "опуска"},
		{"опускалась", "опуска"},
		{"опускали", "опуска"},
		{"опускалось", "опуска"},
		{"опускаться", "опуска"},
		{"опускают", "опуска"},
		{"опуская", "опуск"},
		{"опускаясь", "опуск"},
		{"опустевший", "опустевш"},
		{"опустевшим", "опустевш"},
		{"опустеет", "опустеет"},
		{"опустел", "опустел"},
		{"опустели", "опустел"},
		{"опустело", "опустел"},
		{"опустелой", "опустел"},
		{"опусти", "опуст"},
		{"опустив", "опуст"},
		{"опустил", "опуст"},
		{"опустила", "опуст"},
		{"опустилась", "опуст"},
		{"опустили", "опуст"},
		{"опустились", "опуст"},
		{"опустился", "опуст"},
		{"опустись", "опуст"},
		{"опустите", "опуст"},
		{"опустится", "опуст"},
		{"опуститься", "опуст"},
		{"опустошает", "опустоша"},
		{"опустошаются", "опустоша"},
		{"опустошение", "опустошен"},
		{"опустошенной", "опустошен"},
		{"опустошенный", "опустошен"},
		{"опустя", "опуст"},
		{"опутавшие", "опута"},
		{"опухоли", "опухол"},
		{"опухоль", "опухол"},
		{"опушены", "опуш"},
		{"опущена", "опущ"},
		{"опущенная", "опущен"},
		{"опущенные", "опущен"},
		{"опущенный", "опущен"},
		{"опущенным", "опущен"},
		{"опущенными", "опущен"},
		{"опущены", "опущ"},
		{"опыт", "оп"},
		{"опыта", "опыт"},
		{"опытах", "опыт"},
		{"опыте", "опыт"},
		{"опытная", "опытн"},
		{"опытнее", "опытн"},
		{"опытной", "опытн"},
		{"опытности", "опытн"},
		{"опытность", "опытн"},
		{"опытный", "опытн"},
		{"опытным", "опытн"},
		{"опыту", "опыт"},
		{"опыты", "опыт"},
		{"опьянев", "опьян"},
		{"опьяневший", "опьяневш"},
		{"опьянел", "опьянел"},
		{"опьяненные", "опьянен"},
		{"опьяненный", "опьянен"},
		{"опьянят", "опьян"},
		{"опять", "опя"},
		{"орава", "орав"},
		{"оракула", "оракул"},
		{"орали", "ора"},
		{"оранжа", "оранж"},
		{"оранжерею", "оранжер"},
		{"оранжереям", "оранжере"},
		{"оранжереях", "оранжере"},
		{"орарем", "орар"},
		{"оратор", "оратор"},
		{"оратора", "оратор"},
		{"ораторской", "ораторск"},
		{"ораторствовал", "ораторствова"},
		{"ораторствуешь", "ораторствуеш"},
		{"орбит", "орб"},
		{"органа", "орга"},
		{"организацию", "организац"},
		{"организм", "организм"},
		{"организмами", "организм"},
		{"организме", "организм"},
		{"органом", "орган"},
		{"органчиком", "органчик"},
		{"органщика", "органщик"},
		{"орда", "орд"},
		{"орден", "орд"},
		{"ордена", "орд"},
		{"орденом", "орден"},
		{"ордою", "орд"},
		{"ореол", "ореол"},
		{"орет", "орет"},
		{"орех", "орех"},
		{"орехами", "орех"},
		{"орехи", "орех"},
		{"орешки", "орешк"},
		{"оригиналы", "оригинал"},
		{"оригинальная", "оригинальн"},
		{"оригинально", "оригинальн"},
		{"оригинальности", "оригинальн"},
		{"оригинальность", "оригинальн"},
		{"оригинальный", "оригинальн"},
		{"оригинальным", "оригинальн"},
		{"оригинальных", "оригинальн"},
		{"оркестр", "оркестр"},
		{"оркестре", "оркестр"},
		{"орла", "орл"},
		{"орлий", "орл"},
		{"орлиный", "орлин"},
		{"орлиным", "орлин"},
		{"орлица", "орлиц"},
		{"орлы", "орл"},
		{"оробев", "ороб"},
		{"оробевшая", "оробевш"},
		{"оробеет", "оробеет"},
		{"оробела", "оробел"},
		{"орудий", "оруд"},
		{"орудия", "оруд"},
		{"оружие", "оруж"},
		{"оружием", "оруж"},
		{"орфографии", "орфограф"},
		{"оса", "ос"},
		{"осадила", "осад"},
		{"осадить", "осад"},
		{"осадка", "осадк"},
		{"осадой", "осад"},
		{"осадок", "осадок"},
		{"осаду", "осад"},
		{"осаждает", "осажда"},
		{"осанисто", "осанист"},
		{"осанистый", "осанист"},
		{"осанистым", "осанист"},
		{"осанка", "осанк"},
		{"осанкой", "осанк"},
		{"осведомилась", "осведом"},
		{"осведомился", "осведом"},
		{"осведомиться", "осведом"},
		{"освежает", "освежа"},
		{"освежалась", "освежа"},
		{"освежали", "освежа"},
		{"освежать", "освежа"},
		{"освежают", "освежа"},
		{"освежила", "освеж"},
		{"освежило", "освеж"},
		{"освежит", "освеж"},
		{"освежится", "освеж"},
		{"освежить", "освеж"},
		{"освежиться", "освеж"},
		{"осветил", "освет"},
		{"осветилось", "освет"},
		{"осветит", "освет"},
		{"осветленный", "осветлен"},
		{"осветят", "освет"},
		{"освещает", "освеща"},
		{"освещал", "освеща"},
		{"освещало", "освеща"},
		{"освещая", "освещ"},
		{"освещена", "освещ"},
		{"освещение", "освещен"},
		{"освещении", "освещен"},
		{"освещенная", "освещен"},
		{"освободи", "освобод"},
		{"освободившись", "освобод"},
		{"освободилось", "освобод"},
		{"освободить", "освобод"},
		{"освободиться", "освобод"},
		{"освободу", "освобод"},
		{"освободясь", "освобод"},
		{"освободят", "освобод"},
		{"освобождался", "освобожда"},
		{"освобождаются", "освобожда"},
		{"освобождения", "освобожден"},
		{"освятил", "освят"},
		{"оседает", "оседа"},
		{"оседлать", "оседла"},
		{"осеклась", "осекл"},
		{"осеклись", "осекл"},
		{"осекся", "осек"},
		{"осел", "осел"},
		{"осела", "осел"},
		{"оселся", "осел"},
		{"осени", "осен"},
		{"осенила", "осен"},
		{"осенило", "осен"},
		{"осенить", "осен"},
		{"осенней", "осен"},
		{"осенние", "осен"},
		{"осенний", "осен"},
		{"осенних", "осен"},
		{"осень", "осен"},
		{"осенью", "осен"},
		{"осеняет", "осеня"},
		{"осеняли", "осеня"},
		{"осенялись", "осеня"},
		{"осеняя", "осен"},
		{"осердитесь", "осерд"},
		{"осерчает", "осерча"},
		{"осетр", "осетр"},
		{"осетра", "осетр"},
		{"осетрина", "осетрин"},
		{"осетрины", "осетрин"},
		{"осечка", "осечк"},
		{"осиливал", "осилива"},
		{"осилил", "осил"},
		{"осилила", "осил"},
		{"осиновый", "осинов"},
		{"осину", "осин"},
		{"осипшим", "осипш"},
		{"осквернил", "оскверн"},
		{"осклабясь", "осклаб"},
		{"осколки", "осколк"},
		{"осколок", "осколок"},
		{"оскопления", "оскоплен"},
		{"оскорбил", "оскорб"},
		{"оскорбились", "оскорб"},
		{"оскорбило", "оскорб"},
		{"оскорбительнее", "оскорбительн"},
		{"оскорбительно", "оскорбительн"},
		{"оскорбительное", "оскорбительн"},
		{"оскорбить", "оскорб"},
		{"оскорблен", "оскорбл"},
		{"оскорблена", "оскорбл"},
		{"оскорбление", "оскорблен"},
		{"оскорблении", "оскорблен"},
		{"оскорблений", "оскорблен"},
		{"оскорбления", "оскорблен"},
		{"оскорбленная", "оскорблен"},
		{"оскорбленного", "оскорблен"},
		{"оскорбленное", "оскорблен"},
		{"оскорбленною", "оскорблен"},
		{"оскорбленный", "оскорблен"},
		{"оскорбленным", "оскорблен"},
		{"оскорблены", "оскорбл"},
		{"оскорблю", "оскорбл"},
		{"оскорбляет", "оскорбля"},
		{"оскорбляется", "оскорбля"},
		{"оскорблял", "оскорбля"},
		{"оскорблялась", "оскорбля"},
		{"оскорблять", "оскорбля"},
		{"оскорбляют", "оскорбля"},
		{"оскорбляясь", "оскорбл"},
		{"осла", "осл"},
		{"ослаб", "ослаб"},
		{"ослабевал", "ослабева"},
		{"ослабевали", "ослабева"},
		{"ослабевало", "ослабева"},
		{"ослабевший", "ослабевш"},
		{"ослабевшим", "ослабевш"},
		{"ослабел", "ослабел"},
		{"ослабели", "ослабел"},
		{"ослабело", "ослабел"},
		{"ослеп", "ослеп"},
		{"ослепил", "ослеп"},
		{"ослепила", "ослеп"},
		{"ослепительно", "ослепительн"},
		{"ослепительной", "ослепительн"},
		{"ослепительные", "ослепительн"},
		{"ослепительным", "ослепительн"},
		{"ослепительных", "ослепительн"},
		{"ослепла", "ослепл"},
		{"ослепление", "ослеплен"},
		{"ослепления", "ослеплен"},
		{"ослепленный", "ослеплен"},
		{"ослеплял", "ослепля"},
		{"ослеплялся", "ослепля"},
		{"ослепнув", "ослепнув"},
		{"ослепнуть", "ослепнут"},
		{"ослепшего", "ослепш"},
		{"ослепшие", "ослепш"},
		{"ослепший", "ослепш"},
		{"ослободился", "ослобод"},
		{"ослы", "осл"},
		{"ослышался", "ослыша"},
		{"осматривавшего", "осматрива"},
		{"осматривает", "осматрива"},
		{"осматривается", "осматрива"},
		{"осматривал", "осматрива"},
		{"осматривалась", "осматрива"},
		{"осматривался", "осматрива"},
		{"осматривать", "осматрива"},
		{"осматриваться", "осматрива"},
		{"осматриваю", "осматрива"},
		{"осматривая", "осматрив"},
		{"осматриваясь", "осматрив"},
		{"осмеливаюсь", "осмелива"},
		{"осмелилась", "осмел"},
		{"осмелились", "осмел"},
		{"осмелился", "осмел"},
		{"осмелитесь", "осмел"},
		{"осмелится", "осмел"},
		{"осмелиться", "осмел"},
		{"осмелишься", "осмел"},
		{"осмелюсь", "осмел"},
		{"осмеяно", "осмея"},
		{"осмеять", "осмея"},
		{"осмотр", "осмотр"},
		{"осмотрев", "осмотр"},
		{"осмотрел", "осмотрел"},
		{"осмотрела", "осмотрел"},
		{"осмотрелась", "осмотрел"},
		{"осмотрелся", "осмотрел"},
		{"осмотрено", "осмотр"},
		{"осмотреть", "осмотрет"},
		{"осмотрит", "осмотр"},
		{"осмотрительнее", "осмотрительн"},
		{"осмотрится", "осмотр"},
		{"осмыслив", "осмысл"},
		{"осмыслилась", "осмысл"},
		{"осмыслилось", "осмысл"},
		{"осмыслить", "осмысл"},
		{"оснеженном", "оснежен"},
		{"оснеженные", "оснежен"},
		{"оснежит", "оснеж"},
		{"основа", "основ"},
		{"основался", "основа"},
		{"основание", "основан"},
		{"основанием", "основан"},
		{"основании", "основан"},
		{"оснований", "основан"},
		{"основания", "основан"},
		{"основаниях", "основан"},
		{"основанная", "основа"},
		{"основанное", "основа"},
		{"основанный", "основа"},
		{"основано", "основа"},
		{"основатель", "основател"},
		{"основательно", "основательн"},
		{"основательные", "основательн"},
		{"основать", "основа"},
		{"основная", "основн"},
		{"основным", "основн"},
		{"основных", "основн"},
		{"основу", "основ"},
		{"основы", "основ"},
		{"основываете", "основыва"},
		{"основываясь", "основыв"},
		{"особа", "особ"},
		{"особая", "особ"},
		{"особенная", "особен"},
		{"особенно", "особен"},
		{"особенного", "особен"},
		{"особенное", "особен"},
		{"особенной", "особен"},
		{"особенном", "особен"},
		{"особенностей", "особен"},
		{"особенности", "особен"},
		{"особенность", "особен"},
		{"особенною", "особен"},
		{"особенную", "особен"},
		{"особенные", "особен"},
		{"особенный", "особен"},
		{"особенным", "особен"},
		{"особенных", "особен"},
		{"особо", "особ"},
		{"особого", "особ"},
		{"особое", "особ"},
		{"особой", "особ"},
		{"особом", "особ"},
		{"особому", "особ"},
		{"особу", "особ"},
		{"особую", "особ"},
		{"особы", "особ"},
		{"особый", "особ"},
		{"особых", "особ"},
		{"осой", "ос"},
		{"оспа", "осп"},
		{"оспаривает", "оспарива"},
		{"оспаривали", "оспарива"},
		{"оспаривать", "оспарива"},
		{"оспу", "осп"},
		{"осрамили", "осрам"},
		{"осрамлю", "осрамл"},
		{"оставай", "остава"},
		{"оставайся", "остава"},
		{"оставайтесь", "остава"},
		{"оставалась", "остава"},
		{"оставались", "остава"},
		{"оставалось", "остава"},
		{"оставался", "остава"},
		{"оставаться", "остава"},
		{"оставаясь", "остав"},
		{"оставив", "остав"},
		{"оставил", "остав"},
		{"оставила", "остав"},
		{"оставили", "остав"},
		{"оставим", "остав"},
		{"оставит", "остав"},
		{"оставите", "остав"},
		{"оставить", "остав"},
		{"оставишь", "остав"},
		{"оставлена", "оставл"},
		{"оставленную", "оставлен"},
		{"оставленные", "оставлен"},
		{"оставленный", "оставлен"},
		{"оставленных", "оставлен"},
		{"оставлю", "оставл"},
		{"оставляет", "оставля"},
		{"оставляй", "оставля"},
		{"оставляйте", "оставля"},
		{"оставлял", "оставля"},
		{"оставляли", "оставля"},
		{"оставлять", "оставля"},
		{"оставляю", "оставля"},
		{"оставляют", "оставля"},
		{"оставляя", "оставл"},
		{"оставшийся", "оста"},
		{"оставшимся", "оста"},
		{"оставшись", "оста"},
		{"оставшихся", "оста"},
		{"оставь", "остав"},
		{"оставьте", "оставьт"},
		{"оставят", "остав"},
		{"остаемся", "оста"},
		{"остается", "оста"},
		{"осталась", "оста"},
		{"остались", "оста"},
		{"осталось", "оста"},
		{"остался", "оста"},
		{"остальная", "остальн"},
		{"остального", "остальн"},
		{"остальное", "остальн"},
		{"остальной", "остальн"},
		{"остальном", "остальн"},
		{"остальную", "остальн"},
		{"остальные", "остальн"},
		{"остальным", "остальн"},
		{"остальных", "остальн"},
		{"останавливаемся", "останавлива"},
		{"останавливаетесь", "останавлива"},
		{"останавливается", "останавлива"},
		{"останавливайся", "останавлива"},
		{"останавливал", "останавлива"},
		{"останавливалась", "останавлива"},
		{"останавливались", "останавлива"},
		{"останавливался", "останавлива"},
		{"останавливать", "останавлива"},
		{"останавливаться", "останавлива"},
		{"останавливаются", "останавлива"},
		{"останавливающихся", "останавлива"},
		{"останавливая", "останавлив"},
		{"останавливаясь", "останавлив"},
		{"останемся", "остан"},
		{"останетесь", "останет"},
		{"останется", "останет"},
		{"останешься", "останеш"},
		{"останови", "останов"},
		{"остановив", "останов"},
		{"остановившегося", "останов"},
		{"остановившись", "останов"},
		{"остановил", "останов"},
		{"остановила", "останов"},
		{"остановилась", "останов"},
		{"остановили", "останов"},
		{"остановились", "останов"},
		{"остановилось", "останов"},
		{"остановился", "останов"},
		{"остановись", "останов"},
		{"остановит", "останов"},
		{"остановите", "останов"},
		{"остановитесь", "останов"},
		{"остановится", "останов"},
		{"остановить", "останов"},
		{"остановиться", "останов"},
		{"остановишь", "останов"},
		{"остановка", "остановк"},
		{"остановку", "остановк"},
		{"остановлюсь", "остановл"},
		{"остановясь", "останов"},
		{"остановят", "останов"},
		{"остановятся", "останов"},
		{"останусь", "остан"},
		{"останутся", "останут"},
		{"останься", "остан"},
		{"останьтесь", "останьт"},
		{"остатке", "остатк"},
		{"остатки", "остатк"},
		{"остаток", "остаток"},
		{"остаться", "оста"},
		{"остаюсь", "оста"},
		{"остаются", "оста"},
		{"остающимися", "оста"},
		{"остервенением", "остервенен"},
		{"остервенении", "остервенен"},
		{"остервенения", "остервенен"},
		{"остервенясь", "остервен"},
		{"остереглись", "остерегл"},
		{"остеречь", "остереч"},
		{"остзейскому", "остзейск"},
		{"остолбенел", "остолбенел"},
		{"остолбенелый", "остолбенел"},
		{"осторожен", "осторож"},
		{"осторожна", "осторожн"},
		{"осторожнее", "осторожн"},
		{"осторожно", "осторожн"},
		{"осторожного", "осторожн"},
		{"осторожности", "осторожн"},
		{"осторожностию", "осторожн"},
		{"осторожность", "осторожн"},
		{"осторожностью", "осторожн"},
		{"осторожною", "осторожн"},
		{"осторожны", "осторожн"},
		{"осторожный", "осторожн"},
		{"осторожным", "осторожн"},
		{"осторожными", "осторожн"},
		{"острастка", "острастк"},
		{"остри", "остр"},
		{"острием", "остр"},
		{"острия", "остр"},
		{"остров", "остр"},
		{"острова", "остров"},
		{"островах", "остров"},
		{"острове", "остров"},
		{"островов", "остров"},
		{"островок", "островок"},
		{"островского", "островск"},
		{"острову", "остров"},
		{"острог", "острог"},
		{"острога", "острог"},
		{"остроге", "острог"},
		{"острого", "остр"},
		{"острое", "остр"},
		{"острожной", "острожн"},
		{"острожных", "острожн"},
		{"острота", "острот"},
		{"остротами", "острот"},
		{"остротою", "острот"},
		{"остроты", "острот"},
		{"остроумие", "остроум"},
		{"остроумием", "остроум"},
		{"остроумии", "остроум"},
		{"остроумия", "остроум"},
		{"остроумная", "остроумн"},
		{"остроумнее", "остроумн"},
		{"остроумнейшим", "остроумн"},
		{"остроумно", "остроумн"},
		{"остроумного", "остроумн"},
		{"остроумны", "остроумн"},
		{"острые", "остр"},
		{"острый", "остр"},
		{"острым", "остр"},
		{"остудился", "остуд"},
		{"оступался", "оступа"},
		{"оступаясь", "оступ"},
		{"остыл", "ост"},
		{"остынет", "остынет"},
		{"осудили", "осуд"},
		{"осудит", "осуд"},
		{"осудить", "осуд"},
		{"осудят", "осуд"},
		{"осуждает", "осужда"},
		{"осуждайте", "осужда"},
		{"осуждала", "осужда"},
		{"осуждаю", "осужда"},
		{"осужден", "осужд"},
		{"осуждена", "осужд"},
		{"осужденные", "осужден"},
		{"осужденных", "осужден"},
		{"осунулось", "осунул"},
		{"осунулся", "осунул"},
		{"осушая", "осуш"},
		{"осуществилось", "осуществ"},
		{"осуществился", "осуществ"},
		{"осуществится", "осуществ"},
		{"осуществить", "осуществ"},
		{"осуществлялась", "осуществля"},
		{"осчастливил", "осчастлив"},
		{"осчастливили", "осчастлив"},
		{"осчастливленный", "осчастливлен"},
		{"осыпавшимися", "осыпа"},
		{"осыпает", "осыпа"},
		{"осыпается", "осыпа"},
		{"осыпала", "осыпа"},
		{"осыпалась", "осыпа"},
		{"осыпали", "осыпа"},
		{"осыпанье", "осыпан"},
		{"осыпать", "осыпа"},
		{"осыпка", "осыпк"},
		{"осыплет", "осыплет"},
		{"осьмерка", "осьмерк"},
		{"от", "от"},
		{"отбивался", "отбива"},
		{"отбивные", "отбивн"},
		{"отбился", "отб"},
		{"отбирал", "отбира"},
		{"отбить", "отб"},
		{"отблагодарила", "отблагодар"},
		{"отблеск", "отблеск"},
		{"отблеском", "отблеск"},
		{"отборно", "отборн"},
		{"отборный", "отборн"},
		{"отбрасывать", "отбрасыва"},
		{"отбросив", "отброс"},
		{"отбросил", "отброс"},
		{"отбросила", "отброс"},
		{"отброшенный", "отброшен"},
		{"отваги", "отваг"},
		{"отвагой", "отваг"},
		{"отвагу", "отваг"},
		{"отваживает", "отважива"},
		{"отважно", "отважн"},
		{"отважные", "отважн"},
		{"отважный", "отважн"},
		{"отвален", "отвал"},
		{"отваливать", "отвалива"},
		{"отвалил", "отвал"},
		{"отведает", "отведа"},
		{"отведать", "отведа"},
		{"отведет", "отведет"},
		{"отведу", "отвед"},
		{"отвезите", "отвез"},
		{"отвезли", "отвезл"},
		{"отвезти", "отвезт"},
		{"отвека", "отвек"},
		{"отвел", "отвел"},
		{"отвела", "отвел"},
		{"отвергался", "отверга"},
		{"отверделость", "отвердел"},
		{"отверженец", "отверженец"},
		{"отверженности", "отвержен"},
		{"отверзалась", "отверза"},
		{"отверзалось", "отверза"},
		{"отверзлась", "отверзл"},
		{"отверзший", "отверзш"},
		{"отвернется", "отвернет"},
		{"отвернувшееся", "отвернувш"},
		{"отвернувшись", "отвернувш"},
		{"отвернулась", "отвернул"},
		{"отвернулись", "отвернул"},
		{"отвернулся", "отвернул"},
		{"отвернуться", "отвернут"},
		{"отвертывалась", "отвертыва"},
		{"отвертываться", "отвертыва"},
		{"отвесной", "отвесн"},
		{"отвесов", "отвес"},
		{"отвести", "отвест"},
		{"ответ", "ответ"},
		{"ответа", "ответ"},
		{"ответе", "ответ"},
		{"ответив", "ответ"},
		{"ответил", "ответ"},
		{"ответила", "ответ"},
		{"ответили", "ответ"},
		{"ответит", "ответ"},
		{"ответите", "ответ"},
		{"ответить", "ответ"},
		{"ответишь", "ответ"},
		{"ответный", "ответн"},
		{"ответов", "ответ"},
		{"ответом", "ответ"},
		{"ответственности", "ответствен"},
		{"ответственность", "ответствен"},
		{"ответу", "ответ"},
		{"ответы", "ответ"},
		{"ответь", "ответ"},
		{"ответьте", "ответьт"},
		{"отвечает", "отвеча"},
		{"отвечаешь", "отвеча"},
		{"отвечай", "отвеча"},
		{"отвечайте", "отвеча"},
		{"отвечал", "отвеча"},
		{"отвечала", "отвеча"},
		{"отвечали", "отвеча"},
		{"отвечать", "отвеча"},
		{"отвечаю", "отвеча"},
		{"отвечая", "отвеч"},
		{"отвечу", "отвеч"},
		{"отвешивает", "отвешива"},
		{"отвинчу", "отвинч"},
		{"отвлекайся", "отвлека"},
		{"отвлекала", "отвлека"},
		{"отвлекали", "отвлека"},
		{"отвлеченен", "отвлечен"},
		{"отвлеченные", "отвлечен"},
		{"отвлеченных", "отвлечен"},
		{"отвлечь", "отвлеч"},
		{"отво", "отв"},
		{"отвода", "отвод"},
		{"отводил", "отвод"},
		{"отводили", "отвод"},
		{"отводило", "отвод"},
		{"отводит", "отвод"},
		{"отводить", "отвод"},
		{"отводу", "отвод"},
		{"отводя", "отвод"},
		{"отводят", "отвод"},
		{"отвожу", "отвож"},
		{"отворачивалась", "отворачива"},
		{"отворачиваться", "отворачива"},
		{"отворачивая", "отворачив"},
		{"отворачиваясь", "отворачив"},
		{"отворен", "отвор"},
		{"отворена", "отвор"},
		{"отворенной", "отворен"},
		{"отворенною", "отворен"},
		{"отворенную", "отворен"},
		{"отворенные", "отворен"},
		{"отворенных", "отворен"},
		{"отворены", "отвор"},
		{"отвори", "отвор"},
		{"отворив", "отвор"},
		{"отворившего", "отвор"},
		{"отворил", "отвор"},
		{"отворила", "отвор"},
		{"отворилась", "отвор"},
		{"отворили", "отвор"},
		{"отворились", "отвор"},
		{"отворит", "отвор"},
		{"отворите", "отвор"},
		{"отворить", "отвор"},
		{"отворотами", "отворот"},
		{"отворотил", "отворот"},
		{"отворотился", "отворот"},
		{"отворотит", "отворот"},
		{"отворотясь", "отворот"},
		{"отворю", "отвор"},
		{"отворявшийся", "отворя"},
		{"отворяет", "отворя"},
		{"отворяется", "отворя"},
		{"отворяешь", "отворя"},
		{"отворяй", "отворя"},
		{"отворяйте", "отворя"},
		{"отворял", "отворя"},
		{"отворялась", "отворя"},
		{"отворялись", "отворя"},
		{"отворяль", "отворял"},
		{"отворять", "отворя"},
		{"отворяться", "отворя"},
		{"отворяют", "отворя"},
		{"отворяются", "отворя"},
		{"отворяя", "отвор"},
		{"отвратил", "отврат"},
		{"отвратительна", "отвратительн"},
		{"отвратительная", "отвратительн"},
		{"отвратительнее", "отвратительн"},
		{"отвратительно", "отвратительн"},
		{"отвратительной", "отвратительн"},
		{"отвратительные", "отвратительн"},
		{"отвратительный", "отвратительн"},
		{"отвратительным", "отвратительн"},
		{"отвратить", "отврат"},
		{"отвращение", "отвращен"},
		{"отвращением", "отвращен"},
		{"отвращении", "отвращен"},
		{"отвращения", "отвращен"},
		{"отвращенье", "отвращен"},
		{"отвык", "отвык"},
		{"отвыкла", "отвыкл"},
		{"отвыкни", "отвыкн"},
		{"отвычки", "отвычк"},
		{"отвяжешься", "отвяжеш"},
		{"отвяжутся", "отвяжут"},
		{"отвязал", "отвяза"},
		{"отвязался", "отвяза"},
		{"отвязать", "отвяза"},
		{"отвязаться", "отвяза"},
		{"отгадать", "отгада"},
		{"отгадываний", "отгадыван"},
		{"отговаривался", "отговарива"},
		{"отговариваться", "отговарива"},
		{"отговорил", "отговор"},
		{"отговорился", "отговор"},
		{"отговорка", "отговорк"},
		{"отговорки", "отговорк"},
		{"отголосок", "отголосок"},
		{"отгони", "отгон"},
		{"отгонит", "отгон"},
		{"отгонишь", "отгон"},
		{"отгонял", "отгоня"},
		{"отгоняющее", "отгоня"},
		{"отгороженное", "отгорожен"},
		{"отгуляется", "отгуля"},
		{"отдававшаяся", "отдава"},
		{"отдавал", "отдава"},
		{"отдавала", "отдава"},
		{"отдавали", "отдава"},
		{"отдавался", "отдава"},
		{"отдавать", "отдава"},
		{"отдаваться", "отдава"},
		{"отдавят", "отдав"},
		{"отдадите", "отдад"},
		{"отдает", "отда"},
		{"отдаете", "отда"},
		{"отдается", "отда"},
		{"отдай", "отда"},
		{"отдайте", "отда"},
		{"отдайтесь", "отда"},
		{"отдал", "отда"},
		{"отдала", "отда"},
		{"отдалась", "отда"},
		{"отдалении", "отдален"},
		{"отдаленно", "отдален"},
		{"отдаленного", "отдален"},
		{"отдаленное", "отдален"},
		{"отдаленной", "отдален"},
		{"отдаленную", "отдален"},
		{"отдаленные", "отдален"},
		{"отдаленный", "отдален"},
		{"отдаленным", "отдален"},
		{"отдаленных", "отдален"},
		{"отдали", "отда"},
		{"отдалить", "отдал"},
		{"отдалял", "отдаля"},
		{"отдам", "отд"},
		{"отдан", "отда"},
		{"отданию", "отдан"},
		{"отдано", "отда"},
		{"отдаст", "отдаст"},
		{"отдать", "отда"},
		{"отдаться", "отда"},
		{"отдашь", "отдаш"},
		{"отдаю", "отда"},
		{"отдают", "отда"},
		{"отдаются", "отда"},
		{"отдающих", "отда"},
		{"отдающуюся", "отда"},
		{"отделан", "отдела"},
		{"отделанную", "отдела"},
		{"отделаться", "отдела"},
		{"отделаюсь", "отдела"},
		{"отделение", "отделен"},
		{"отделении", "отделен"},
		{"отделению", "отделен"},
		{"отделения", "отделен"},
		{"отделенная", "отделен"},
		{"отделено", "отдел"},
		{"отделенье", "отделен"},
		{"отделкой", "отделк"},
		{"отделку", "отделк"},
		{"отделом", "отдел"},
		{"отделываемой", "отделыва"},
		{"отделывается", "отделыва"},
		{"отделывали", "отделыва"},
		{"отделывать", "отделыва"},
		{"отделываться", "отделыва"},
		{"отделывают", "отделыва"},
		{"отдельная", "отдельн"},
		{"отдельно", "отдельн"},
		{"отдельное", "отдельн"},
		{"отдельной", "отдельн"},
		{"отдельные", "отдельн"},
		{"отдельных", "отдельн"},
		{"отделявший", "отделя"},
		{"отделяла", "отделя"},
		{"отделялась", "отделя"},
		{"отделялось", "отделя"},
		{"отделять", "отделя"},
		{"отделяющей", "отделя"},
		{"отдернул", "отдернул"},
		{"отдернула", "отдернул"},
		{"отдохнем", "отдохн"},
		{"отдохнет", "отдохнет"},
		{"отдохнешь", "отдохнеш"},
		{"отдохни", "отдохн"},
		{"отдохните", "отдохн"},
		{"отдохновенья", "отдохновен"},
		{"отдохну", "отдохн"},
		{"отдохнув", "отдохнув"},
		{"отдохнул", "отдохнул"},
		{"отдохнула", "отдохнул"},
		{"отдохнуть", "отдохнут"},
		{"отдых", "отд"},
		{"отдыха", "отдых"},
		{"отдыхает", "отдыха"},
		{"отдыхаете", "отдыха"},
		{"отдыхал", "отдыха"},
		{"отдыхала", "отдыха"},
		{"отдыхать", "отдыха"},
		{"отдыхаю", "отдыха"},
		{"отдыхая", "отдых"},
		{"отдыхиваясь", "отдыхив"},
		{"отдыхнувшись", "отдыхнувш"},
		{"отдыхнулся", "отдыхнул"},
		{"отдыхом", "отдых"},
		{"отдышалась", "отдыша"},
		{"отдышаться", "отдыша"},
		{"отекать", "отека"},
		{"отекшим", "отекш"},
		{"отель", "отел"},
		{"отеля", "отел"},
		{"отер", "отер"},
		{"отерла", "отерл"},
		{"отец", "отец"},
		{"отеческое", "отеческ"},
		{"отеческой", "отеческ"},
		{"отечества", "отечеств"},
		{"отечественные", "отечествен"},
		{"отечеству", "отечеств"},
		{"отжившее", "отж"},
		{"отжила", "отж"},
		{"отжило", "отж"},
		{"отзвук", "отзвук"},
		{"отзвучие", "отзвуч"},
		{"отзовется", "отзовет"},
		{"отзыв", "отз"},
		{"отзыва", "отзыв"},
		{"отзывается", "отзыва"},
		{"отзываешься", "отзыва"},
		{"отзываться", "отзыва"},
		{"отзывов", "отзыв"},
		{"отзывом", "отзыв"},
		{"отзыву", "отзыв"},
		{"отзывы", "отзыв"},
		{"отиравшую", "отира"},
		{"отирать", "отира"},
		{"отирая", "отир"},
		{"откажемся", "откаж"},
		{"откажет", "откажет"},
		{"откажетесь", "откажет"},
		{"откажется", "откажет"},
		{"откажите", "откаж"},
		{"откажитесь", "откаж"},
		{"откажу", "откаж"},
		{"откажусь", "откаж"},
		{"отказа", "отказ"},
		{"отказавшись", "отказа"},
		{"отказал", "отказа"},
		{"отказала", "отказа"},
		{"отказалась", "отказа"},
		{"отказали", "отказа"},
		{"отказались", "отказа"},
		{"отказался", "отказа"},
		{"отказано", "отказа"},
		{"отказать", "отказа"},
		{"отказаться", "отказа"},
		{"отказом", "отказ"},
		{"отказывает", "отказыва"},
		{"отказываете", "отказыва"},
		{"отказываетесь", "отказыва"},
		{"отказываешься", "отказыва"},
		{"отказывайтесь", "отказыва"},
		{"отказывал", "отказыва"},
		{"отказывался", "отказыва"},
		{"отказываться", "отказыва"},
		{"отказываюсь", "отказыва"},
		{"откапывать", "откапыва"},
		{"откармливались", "откармлива"},
		{"откашливаться", "откашлива"},
		{"откашлянулся", "откашлянул"},
		{"откидные", "откидн"},
		{"откидываясь", "откидыв"},
		{"откинув", "откинув"},
		{"откинула", "откинул"},
		{"откинутом", "откинут"},
		{"откладывал", "откладыва"},
		{"откладывать", "откладыва"},
		{"откладывая", "откладыв"},
		{"откланивалась", "откланива"},
		{"откланиваться", "откланива"},
		{"откланиваясь", "откланив"},
		{"откланялась", "откланя"},
		{"откланялся", "откланя"},
		{"откланяться", "откланя"},
		{"откликалось", "отклика"},
		{"откликаясь", "отклик"},
		{"отклики", "отклик"},
		{"откликнулась", "откликнул"},
		{"откликнулся", "откликнул"},
		{"отклонив", "отклон"},
		{"отклонить", "отклон"},
		{"отколотил", "отколот"},
		{"отколотили", "отколот"},
		{"откомандировала", "откомандирова"},
		{"откопал", "откопа"},
		{"откопать", "откопа"},
		{"откровенен", "откровен"},
		{"откровенна", "откровен"},
		{"откровеннее", "откровен"},
		{"откровеннейшим", "откровен"},
		{"откровенничает", "откровеннича"},
		{"откровенничал", "откровеннича"},
		{"откровенно", "откровен"},
		{"откровенное", "откровен"},
		{"откровенности", "откровен"},
		{"откровенностию", "откровен"},
		{"откровенность", "откровен"},
		{"откровенностью", "откровен"},
		{"откровенною", "откровен"},
		{"откровенны", "откровен"},
		{"откровенный", "откровен"},
		{"откровенных", "откровен"},
		{"откроет", "откроет"},
		{"откроется", "откроет"},
		{"открой", "откр"},
		{"откройся", "откр"},
		{"открою", "откр"},
		{"откроют", "откроют"},
		{"откроются", "откроют"},
		{"открыв", "откр"},
		{"открывает", "открыва"},
		{"открывается", "открыва"},
		{"открывай", "открыва"},
		{"открывал", "открыва"},
		{"открывалась", "открыва"},
		{"открывались", "открыва"},
		{"открывалось", "открыва"},
		{"открывался", "открыва"},
		{"открывать", "открыва"},
		{"открываться", "открыва"},
		{"открываю", "открыва"},
		{"открываются", "открыва"},
		{"открывающимся", "открыва"},
		{"открывая", "открыв"},
		{"открыл", "откр"},
		{"открыла", "откр"},
		{"открылась", "откр"},
		{"открыли", "откр"},
		{"открылись", "откр"},
		{"открылось", "откр"},
		{"открыт", "откр"},
		{"открыта", "открыт"},
		{"открытая", "открыт"},
		{"открытие", "открыт"},
		{"открытием", "открыт"},
		{"открытий", "открыт"},
		{"открытию", "открыт"},
		{"открытия", "открыт"},
		{"открыто", "открыт"},
		{"открытое", "открыт"},
		{"открытой", "открыт"},
		{"открытом", "открыт"},
		{"открытою", "открыт"},
		{"открытую", "открыт"},
		{"открыты", "открыт"},
		{"открытый", "открыт"},
		{"открытым", "открыт"},
		{"открытыми", "открыт"},
		{"открытых", "открыт"},
		{"открыть", "откр"},
		{"открыться", "откр"},
		{"откуда", "откуд"},
		{"откудова", "откудов"},
		{"откупа", "откуп"},
		{"откупам", "откуп"},
		{"откупиться", "откуп"},
		{"откупорил", "откупор"},
		{"откусила", "откус"},
		{"откусить", "откус"},
		{"откушать", "откуша"},
		{"откушенной", "откушен"},
		{"отлагается", "отлага"},
		{"отлагал", "отлага"},
		{"отлегло", "отлегл"},
		{"отлетевший", "отлетевш"},
		{"отлетел", "отлетел"},
		{"отлетела", "отлетел"},
		{"отлетит", "отлет"},
		{"отлива", "отлив"},
		{"отливались", "отлива"},
		{"отливало", "отлива"},
		{"отливать", "отлива"},
		{"отличаемся", "отлича"},
		{"отличает", "отлича"},
		{"отличались", "отлича"},
		{"отличалось", "отлича"},
		{"отличат", "отличат"},
		{"отличать", "отлича"},
		{"отличаются", "отлича"},
		{"отличие", "отлич"},
		{"отличительные", "отличительн"},
		{"отличить", "отлич"},
		{"отличная", "отличн"},
		{"отличнейший", "отличн"},
		{"отлично", "отличн"},
		{"отличное", "отличн"},
		{"отличной", "отличн"},
		{"отличном", "отличн"},
		{"отличную", "отличн"},
		{"отличные", "отличн"},
		{"отличный", "отличн"},
		{"отлогие", "отлог"},
		{"отлогих", "отлог"},
		{"отложат", "отложат"},
		{"отложенные", "отложен"},
		{"отложив", "отлож"},
		{"отложил", "отлож"},
		{"отложили", "отлож"},
		{"отложим", "отлож"},
		{"отложите", "отлож"},
		{"отложить", "отлож"},
		{"отложу", "отлож"},
		{"отломанными", "отлома"},
		{"отломок", "отломок"},
		{"отлучался", "отлуча"},
		{"отлучился", "отлуч"},
		{"отлучиться", "отлуч"},
		{"отлучке", "отлучк"},
		{"отлучусь", "отлуч"},
		{"отмалчивается", "отмалчива"},
		{"отмахивается", "отмахива"},
		{"отмахивался", "отмахива"},
		{"отмахиваются", "отмахива"},
		{"отмахиваясь", "отмахив"},
		{"отмахнувшись", "отмахнувш"},
		{"отмахнулась", "отмахнул"},
		{"отменили", "отмен"},
		{"отменить", "отмен"},
		{"отменном", "отмен"},
		{"отменные", "отмен"},
		{"отменный", "отмен"},
		{"отменят", "отмен"},
		{"отменяю", "отменя"},
		{"отметил", "отмет"},
		{"отметили", "отмет"},
		{"отметки", "отметк"},
		{"отметкою", "отметк"},
		{"отмеченный", "отмечен"},
		{"отмеченных", "отмечен"},
		{"отмечены", "отмеч"},
		{"отмоешь", "отмоеш"},
		{"отмыв", "отм"},
		{"отмывает", "отмыва"},
		{"отмывал", "отмыва"},
		{"отмывать", "отмыва"},
		{"отмыли", "отм"},
		{"отнекивалась", "отнекива"},
		{"отнекивался", "отнекива"},
		{"отнес", "отнес"},
		{"отнесет", "отнесет"},
		{"отнесла", "отнесл"},
		{"отнеслись", "отнесл"},
		{"отнесло", "отнесл"},
		{"отнесся", "отнес"},
		{"отнести", "отнест"},
		{"отнесу", "отнес"},
		{"отнимает", "отнима"},
		{"отнимается", "отнима"},
		{"отнимал", "отнима"},
		{"отнимала", "отнима"},
		{"отнимало", "отнима"},
		{"отнимаю", "отнима"},
		{"отнимая", "отним"},
		{"отнимет", "отнимет"},
		{"отнимите", "отним"},
		{"отниму", "отним"},
		{"отнимут", "отнимут"},
		{"относил", "относ"},
		{"относилась", "относ"},
		{"относились", "относ"},
		{"относительно", "относительн"},
		{"относительны", "относительн"},
		{"относитесь", "относ"},
		{"относится", "относ"},
		{"относиться", "относ"},
		{"относятся", "относ"},
		{"относящимся", "относя"},
		{"отношение", "отношен"},
		{"отношении", "отношен"},
		{"отношений", "отношен"},
		{"отношения", "отношен"},
		{"отношениях", "отношен"},
		{"отношусь", "отнош"},
		{"отныне", "отнын"},
		{"отнюдь", "отнюд"},
		{"отняв", "отня"},
		{"отнял", "отня"},
		{"отняли", "отня"},
		{"отнялись", "отня"},
		{"отнять", "отня"},
		{"ото", "от"},
		{"отобедаем", "отобеда"},
		{"отобедайте", "отобеда"},
		{"отобедал", "отобеда"},
		{"отобедать", "отобеда"},
		{"отобедаю", "отобеда"},
		{"отобрал", "отобра"},
		{"отобрать", "отобра"},
		{"отовсюду", "отовсюд"},
		{"отогнал", "отогна"},
		{"отогнать", "отогна"},
		{"отогрел", "отогрел"},
		{"отогрелось", "отогрел"},
		{"отодвигалась", "отодвига"},
		{"отодвигалось", "отодвига"},
		{"отодвигаться", "отодвига"},
		{"отодвигая", "отодвиг"},
		{"отодвигаясь", "отодвиг"},
		{"отодвинув", "отодвинув"},
		{"отодвинул", "отодвинул"},
		{"отодвинула", "отодвинул"},
		{"отодвинулось", "отодвинул"},
		{"отодвинулся", "отодвинул"},
		{"отодвинуть", "отодвинут"},
		{"отозвал", "отозва"},
		{"отозвалась", "отозва"},
		{"отозвалось", "отозва"},
		{"отозвался", "отозва"},
		{"отозваться", "отозва"},
		{"отойдет", "отойдет"},
		{"отойди", "отойд"},
		{"отойду", "отойд"},
		{"отойдя", "отойд"},
		{"отойти", "отойт"},
		{"отолстел", "отолстел"},
		{"отолщением", "отолщен"},
		{"отомкнув", "отомкнув"},
		{"отомкнула", "отомкнул"},
		{"отомстить", "отомст"},
		{"отопление", "отоплен"},
		{"отопрет", "отопрет"},
		{"отопрется", "отопрет"},
		{"оторвавшись", "оторва"},
		{"оторвала", "оторва"},
		{"оторвало", "оторва"},
		{"оторвать", "оторва"},
		{"оторваться", "оторва"},
		{"оторвет", "оторвет"},
		{"оторопев", "отороп"},
		{"оторопевший", "оторопевш"},
		{"оторопел", "оторопел"},
		{"оторопью", "отороп"},
		{"отослал", "отосла"},
		{"отослана", "отосла"},
		{"отошел", "отошел"},
		{"отошла", "отошл"},
		{"отошли", "отошл"},
		{"отошло", "отошл"},
		{"отощал", "отоща"},
		{"отпаривал", "отпарива"},
		{"отпевании", "отпеван"},
		{"отпейте", "отп"},
		{"отпер", "отпер"},
		{"отперла", "отперл"},
		{"отперли", "отперл"},
		{"отперта", "отперт"},
		{"отпертая", "отперт"},
		{"отперто", "отперт"},
		{"отпертой", "отперт"},
		{"отпертую", "отперт"},
		{"отпечаталась", "отпечата"},
		{"отпечатают", "отпечата"},
		{"отпив", "отп"},
		{"отпивая", "отпив"},
		{"отпил", "отп"},
		{"отпирает", "отпира"},
		{"отпирала", "отпира"},
		{"отпирать", "отпира"},
		{"отпирают", "отпира"},
		{"отписал", "отписа"},
		{"отпихнул", "отпихнул"},
		{"отпихнула", "отпихнул"},
		{"отпихнуть", "отпихнут"},
		{"отплясывал", "отплясыва"},
		{"отполированного", "отполирова"},
		{"отпор", "отпор"},
		{"отправившимся", "отправ"},
		{"отправил", "отправ"},
		{"отправила", "отправ"},
		{"отправилась", "отправ"},
		{"отправили", "отправ"},
		{"отправились", "отправ"},
		{"отправилось", "отправ"},
		{"отправился", "отправ"},
		{"отправимся", "отправ"},
		{"отправитесь", "отправ"},
		{"отправится", "отправ"},
		{"отправить", "отправ"},
		{"отправиться", "отправ"},
		{"отправишь", "отправ"},
		{"отправишься", "отправ"},
		{"отправлен", "отправл"},
		{"отправление", "отправлен"},
		{"отправлении", "отправлен"},
		{"отправлений", "отправлен"},
		{"отправления", "отправлен"},
		{"отправлениях", "отправлен"},
		{"отправленью", "отправлен"},
		{"отправлю", "отправл"},
		{"отправлюсь", "отправл"},
		{"отправляетесь", "отправля"},
		{"отправляется", "отправля"},
		{"отправляйся", "отправля"},
		{"отправляйтесь", "отправля"},
		{"отправляла", "отправля"},
		{"отправляли", "отправля"},
		{"отправлялись", "отправля"},
		{"отправлялось", "отправля"},
		{"отправлялся", "отправля"},
		{"отправляю", "отправля"},
		{"отправляюсь", "отправля"},
		{"отправляются", "отправля"},
		{"отправляющей", "отправля"},
		{"отправляясь", "отправл"},
		{"отправятся", "отправ"},
		{"отпраздновать", "отпразднова"},
		{"отпрыск", "отпрыск"},
		{"отпрыске", "отпрыск"},
		{"отпрыски", "отпрыск"},
		{"отпрыску", "отпрыск"},
		{"отпуск", "отпуск"},
		{"отпускает", "отпуска"},
		{"отпускай", "отпуска"},
		{"отпускала", "отпуска"},
		{"отпускалась", "отпуска"},
		{"отпускали", "отпуска"},
		{"отпускалось", "отпуска"},
		{"отпускать", "отпуска"},
		{"отпуская", "отпуск"},
		{"отпусти", "отпуст"},
		{"отпустил", "отпуст"},
		{"отпустит", "отпуст"},
		{"отпустить", "отпуст"},
		{"отпустят", "отпуст"},
		{"отпущен", "отпущ"},
		{"отпущено", "отпущ"},
		{"отрав", "отра"},
		{"отрава", "отрав"},
		{"отрави", "отрав"},
		{"отравил", "отрав"},
		{"отравилась", "отрав"},
		{"отравился", "отрав"},
		{"отравитесь", "отрав"},
		{"отравится", "отрав"},
		{"отравить", "отрав"},
		{"отравиться", "отрав"},
		{"отравлений", "отравлен"},
		{"отравленной", "отравлен"},
		{"отравленный", "отравлен"},
		{"отравленным", "отравлен"},
		{"отравлены", "отравл"},
		{"отравляешь", "отравля"},
		{"отравляла", "отравля"},
		{"отравлять", "отравля"},
		{"отрада", "отрад"},
		{"отрадно", "отрадн"},
		{"отрадного", "отрадн"},
		{"отражалось", "отража"},
		{"отражать", "отража"},
		{"отражаться", "отража"},
		{"отражаясь", "отраж"},
		{"отражение", "отражен"},
		{"отражением", "отражен"},
		{"отражены", "отраж"},
		{"отразились", "отраз"},
		{"отразилось", "отраз"},
		{"отразился", "отраз"},
		{"отразится", "отраз"},
		{"отрапортует", "отрапорт"},
		{"отрастил", "отраст"},
		{"отрезал", "отреза"},
		{"отрезала", "отреза"},
		{"отрезали", "отреза"},
		{"отрезанные", "отреза"},
		{"отрезанный", "отреза"},
		{"отрезать", "отреза"},
		{"отрезвел", "отрезвел"},
		{"отрезвился", "отрезв"},
		{"отрезвись", "отрезв"},
		{"отрезвится", "отрезв"},
		{"отрезвляется", "отрезвля"},
		{"отрезвлялась", "отрезвля"},
		{"отрезвляло", "отрезвля"},
		{"отрезвляться", "отрезвля"},
		{"отрезвляющее", "отрезвля"},
		{"отрезвляясь", "отрезвл"},
		{"отрезков", "отрезк"},
		{"отрекаетесь", "отрека"},
		{"отрекался", "отрека"},
		{"отрекаюсь", "отрека"},
		{"отрекомендовался", "отрекомендова"},
		{"отрекомендовать", "отрекомендова"},
		{"отрекомендоваться", "отрекомендова"},
		{"отрекся", "отрек"},
		{"отретировался", "отретирова"},
		{"отречение", "отречен"},
		{"отречения", "отречен"},
		{"отречься", "отреч"},
		{"отрешиться", "отреш"},
		{"отринь", "отрин"},
		{"отрицаем", "отрица"},
		{"отрицаете", "отрица"},
		{"отрицал", "отрица"},
		{"отрицания", "отрицан"},
		{"отрицатели", "отрицател"},
		{"отрицательная", "отрицательн"},
		{"отрицательно", "отрицательн"},
		{"отрицательный", "отрицательн"},
		{"отрицаю", "отрица"},
		{"отрицают", "отрица"},
		{"отрицая", "отриц"},
		{"отроду", "отрод"},
		{"отродясь", "отрод"},
		{"отрочества", "отрочеств"},
		{"отрочестве", "отрочеств"},
		{"отрубленную", "отрублен"},
		{"отрывает", "отрыва"},
		{"отрывается", "отрыва"},
		{"отрывал", "отрыва"},
		{"отрывала", "отрыва"},
		{"отрывать", "отрыва"},
		{"отрывая", "отрыв"},
		{"отрываясь", "отрыв"},
		{"отрывисто", "отрывист"},
		{"отрывистые", "отрывист"},
		{"отрывки", "отрывк"},
		{"отрывок", "отрывок"},
		{"отрывочно", "отрывочн"},
		{"отряд", "отряд"},
		{"отряс", "отряс"},
		{"отсвет", "отсвет"},
		{"отсветы", "отсвет"},
		{"отсидела", "отсидел"},
		{"отскакивавший", "отскакива"},
		{"отскочи", "отскоч"},
		{"отскочив", "отскоч"},
		{"отскочил", "отскоч"},
		{"отскочила", "отскоч"},
		{"отскочило", "отскоч"},
		{"отслужив", "отслуж"},
		{"отслужить", "отслуж"},
		{"отслужу", "отслуж"},
		{"отсохли", "отсохл"},
		{"отсрочивать", "отсрочива"},
		{"отсрочишь", "отсроч"},
		{"отставал", "отстава"},
		{"отставила", "отстав"},
		{"отставке", "отставк"},
		{"отставку", "отставк"},
		{"отставного", "отставн"},
		{"отставной", "отставн"},
		{"отставные", "отставн"},
		{"отставшие", "отста"},
		{"отставший", "отста"},
		{"отставшими", "отста"},
		{"отстает", "отста"},
		{"отстал", "отста"},
		{"отстала", "отста"},
		{"отстали", "отста"},
		{"отстало", "отста"},
		{"отсталого", "отстал"},
		{"отсталостью", "отстал"},
		{"отсталый", "отстал"},
		{"отсталых", "отстал"},
		{"отстанет", "отстанет"},
		{"отстану", "отстан"},
		{"отстанут", "отстанут"},
		{"отстань", "отстан"},
		{"отстаньте", "отстаньт"},
		{"отстать", "отста"},
		{"отстаю", "отста"},
		{"отстегнул", "отстегнул"},
		{"отстоял", "отстоя"},
		{"отстоялся", "отстоя"},
		{"отстранились", "отстран"},
		{"отстраняются", "отстраня"},
		{"отстраняя", "отстран"},
		{"отстраняясь", "отстран"},
		{"отстряпает", "отстряпа"},
		{"отступает", "отступа"},
		{"отступало", "отступа"},
		{"отступающих", "отступа"},
		{"отступая", "отступ"},
		{"отступив", "отступ"},
		{"отступил", "отступ"},
		{"отступила", "отступ"},
		{"отступили", "отступ"},
		{"отступило", "отступ"},
		{"отступилось", "отступ"},
		{"отступишься", "отступ"},
		{"отступлю", "отступл"},
		{"отступлюсь", "отступл"},
		{"отступятся", "отступ"},
		{"отсутствие", "отсутств"},
		{"отсутствием", "отсутств"},
		{"отсутствия", "отсутств"},
		{"отсутствующему", "отсутств"},
		{"отсутствующих", "отсутств"},
		{"отсчитал", "отсчита"},
		{"отсчитывать", "отсчитыва"},
		{"отсылал", "отсыла"},
		{"отсылать", "отсыла"},
		{"отсыпной", "отсыпн"},
		{"отсюда", "отсюд"},
		{"оттает", "отта"},
		{"отталкиваешь", "отталкива"},
		{"отталкивал", "отталкива"},
		{"отталкивала", "отталкива"},
		{"отталкивали", "отталкива"},
		{"отталкивало", "отталкива"},
		{"отталкивать", "отталкива"},
		{"отталкивающее", "отталкива"},
		{"отталкивающий", "отталкива"},
		{"отталкивающим", "отталкива"},
		{"отталкивая", "отталкив"},
		{"оттащить", "оттащ"},
		{"оттенка", "оттенк"},
		{"оттенком", "оттенк"},
		{"оттенок", "оттенок"},
		{"оттеняет", "оттеня"},
		{"оттеняется", "оттеня"},
		{"оттепель", "оттепел"},
		{"оттепелями", "оттепел"},
		{"оттер", "оттер"},
		{"оттерты", "оттерт"},
		{"оттого", "отт"},
		{"оттолкнет", "оттолкнет"},
		{"оттолкнешь", "оттолкнеш"},
		{"оттолкнув", "оттолкнув"},
		{"оттолкнул", "оттолкнул"},
		{"оттолкнула", "оттолкнул"},
		{"оттолкнутый", "оттолкнут"},
		{"оттопырившуюся", "оттопыр"},
		{"оттрепал", "оттрепа"},
		{"оттреплет", "оттреплет"},
		{"оттуда", "оттуд"},
		{"оттудова", "оттудов"},
		{"оттянулось", "оттянул"},
		{"отуманенный", "отуманен"},
		{"отуманивать", "отуманива"},
		{"отупения", "отупен"},
		{"отупеть", "отупет"},
		{"отупления", "отуплен"},
		{"отучены", "отуч"},
		{"отучит", "отуч"},
		{"отхаркивая", "отхаркив"},
		{"отхаркнулась", "отхаркнул"},
		{"отхватит", "отхват"},
		{"отхлебнул", "отхлебнул"},
		{"отхлебывая", "отхлебыв"},
		{"отхлеставшему", "отхлеста"},
		{"отхлестал", "отхлеста"},
		{"отхлынет", "отхлынет"},
		{"отхлынули", "отхлынул"},
		{"отходи", "отход"},
		{"отходил", "отход"},
		{"отходила", "отход"},
		{"отходит", "отход"},
		{"отходной", "отходн"},
		{"отходят", "отход"},
		{"отца", "отц"},
		{"отцам", "отц"},
		{"отцами", "отц"},
		{"отцах", "отц"},
		{"отцвели", "отцвел"},
		{"отцветшего", "отцветш"},
		{"отцветшими", "отцветш"},
		{"отце", "отц"},
		{"отцов", "отц"},
		{"отцовские", "отцовск"},
		{"отцовский", "отцовск"},
		{"отцовских", "отцовск"},
		{"отцовскую", "отцовск"},
		{"отцом", "отц"},
		{"отцу", "отц"},
		{"отцы", "отц"},
		{"отчаивайся", "отчаива"},
		{"отчаивался", "отчаива"},
		{"отчаиваться", "отчаива"},
		{"отчасти", "отчаст"},
		{"отчаявшийся", "отчая"},
		{"отчаялась", "отчая"},
		{"отчаялись", "отчая"},
		{"отчаяние", "отчаян"},
		{"отчаянием", "отчаян"},
		{"отчаянии", "отчаян"},
		{"отчаянию", "отчаян"},
		{"отчаяния", "отчаян"},
		{"отчаянная", "отчая"},
		{"отчаяннее", "отчая"},
		{"отчаянно", "отчая"},
		{"отчаянного", "отчая"},
		{"отчаянное", "отчая"},
		{"отчаянной", "отчая"},
		{"отчаянном", "отчая"},
		{"отчаянному", "отчая"},
		{"отчаянною", "отчая"},
		{"отчаянные", "отчая"},
		{"отчаянный", "отчая"},
		{"отчаянным", "отчая"},
		{"отчаянных", "отчая"},
		{"отчаянье", "отчаян"},
		{"отчаянью", "отчаян"},
		{"отчаянья", "отчаян"},
		{"отчаяться", "отчая"},
		{"отче", "отч"},
		{"отчего", "отч"},
		{"отчеканивая", "отчеканив"},
		{"отчеканила", "отчекан"},
		{"отчеканилась", "отчекан"},
		{"отчеканился", "отчекан"},
		{"отчество", "отчеств"},
		{"отчет", "отчет"},
		{"отчета", "отчет"},
		{"отчетливо", "отчетлив"},
		{"отчетливым", "отчетлив"},
		{"отчетов", "отчет"},
		{"отчетом", "отчет"},
		{"отчеты", "отчет"},
		{"отчизне", "отчизн"},
		{"отчизну", "отчизн"},
		{"отчий", "отч"},
		{"отчиной", "отчин"},
		{"отчистил", "отчист"},
		{"отчищенных", "отчищен"},
		{"отчуждения", "отчужден"},
		{"отшатнулась", "отшатнул"},
		{"отшатнулся", "отшатнул"},
		{"отшельник", "отшельник"},
		{"отшибло", "отшибл"},
		{"отшумела", "отшумел"},
		{"отщелкивая", "отщелкив"},
		{"отъезд", "отъезд"},
		{"отъезда", "отъезд"},
		{"отъезде", "отъезд"},
		{"отъездом", "отъезд"},
		{"отъезду", "отъезд"},
		{"отъезжают", "отъезжа"},
		{"отъезжающими", "отъезжа"},
		{"отъехав", "отъеха"},
		{"отъявленного", "отъявлен"},
		{"отъявленной", "отъявлен"},
		{"отъявленным", "отъявлен"},
		{"отыскав", "отыска"},
		{"отыскал", "отыска"},
		{"отыскала", "отыска"},
		{"отыскалась", "отыска"},
		{"отыскали", "отыска"},
		{"отыскались", "отыска"},
		{"отыскалось", "отыска"},
		{"отыскался", "отыска"},
		{"отыскание", "отыскан"},
		{"отыскать", "отыска"},
		{"отыскивает", "отыскива"},
		{"отыскиваете", "отыскива"},
		{"отыскивал", "отыскива"},
		{"отыскивали", "отыскива"},
		{"отыскивать", "отыскива"},
		{"отыскиваются", "отыскива"},
		{"отыскивая", "отыскив"},
		{"отыщется", "отыщет"},
		{"отыщешь", "отыщеш"},
		{"отыщи", "отыщ"},
		{"отягчены", "отягч"},
		{"отяжелевшую", "отяжелевш"},
		{"отяжелели", "отяжелел"},
		{"отяжелелой", "отяжелел"},
		{"офелию", "офел"},
		{"офицер", "офицер"},
		{"офицера", "офицер"},
		{"офицерами", "офицер"},
		{"офицеров", "офицер"},
		{"офицером", "офицер"},
		{"офицерская", "офицерск"},
		{"офицерских", "офицерск"},
		{"офицеру", "офицер"},
		{"офицеры", "офицер"},
		{"официально", "официальн"},
		{"официальное", "официальн"},
		{"официальном", "официальн"},
		{"официальный", "официальн"},
		{"официальным", "официальн"},
		{"официальных", "официальн"},
		{"офранцузят", "офранцуз"},
		{"ох", "ох"},
		{"охает", "оха"},
		{"охаете", "оха"},
		{"охала", "оха"},
		{"оханье", "охан"},
		{"оханья", "охан"},
		{"охапку", "охапк"},
		{"охая", "ох"},
		{"охватил", "охват"},
		{"охватила", "охват"},
		{"охватили", "охват"},
		{"охватило", "охват"},
		{"охватить", "охват"},
		{"охватывавшим", "охватыва"},
		{"охватывал", "охватыва"},
		{"охватывала", "охватыва"},
		{"охватывают", "охватыва"},
		{"охвачен", "охвач"},
		{"охваченный", "охвачен"},
		{"охи", "ох"},
		{"охладевших", "охладевш"},
		{"охлаждается", "охлажда"},
		{"охлаждался", "охлажда"},
		{"охлаждение", "охлажден"},
		{"охлаждений", "охлажден"},
		{"охмелел", "охмелел"},
		{"охмелия", "охмел"},
		{"охмеляющего", "охмеля"},
		{"охнет", "охнет"},
		{"охнуть", "охнут"},
		{"охо", "ох"},
		{"охота", "охот"},
		{"охоте", "охот"},
		{"охотку", "охотк"},
		{"охотливо", "охотлив"},
		{"охотнее", "охотн"},
		{"охотник", "охотник"},
		{"охотника", "охотник"},
		{"охотничья", "охотнич"},
		{"охотно", "охотн"},
		{"охотой", "охот"},
		{"охоту", "охот"},
		{"охоты", "охот"},
		{"охранению", "охранен"},
		{"охраню", "охран"},
		{"охраняет", "охраня"},
		{"охраняя", "охран"},
		{"охрипший", "охрипш"},
		{"охта", "охт"},
		{"оцененного", "оценен"},
		{"оцени", "оцен"},
		{"оценила", "оцен"},
		{"оценили", "оцен"},
		{"оценить", "оцен"},
		{"оценку", "оценк"},
		{"оцепенеет", "оцепенеет"},
		{"оцепенел", "оцепенел"},
		{"оцепенело", "оцепенел"},
		{"оцепенелой", "оцепенел"},
		{"оцепенении", "оцепенен"},
		{"оцепенения", "оцепенен"},
		{"оцепили", "оцеп"},
		{"оч", "оч"},
		{"очаг", "очаг"},
		{"очами", "оч"},
		{"очаровал", "очарова"},
		{"очарован", "очарова"},
		{"очарована", "очарова"},
		{"очарование", "очарован"},
		{"очарования", "очарован"},
		{"очарованной", "очарова"},
		{"очарованный", "очарова"},
		{"очарованье", "очарован"},
		{"очарованьи", "очарован"},
		{"очарованьям", "очарован"},
		{"очарователен", "очаровател"},
		{"очаровательна", "очаровательн"},
		{"очаровательная", "очаровательн"},
		{"очаровательнее", "очаровательн"},
		{"очаровательнейшая", "очаровательн"},
		{"очаровательное", "очаровательн"},
		{"очах", "оч"},
		{"очевидно", "очевидн"},
		{"очевидное", "очевидн"},
		{"очевидности", "очевидн"},
		{"очевидность", "очевидн"},
		{"очевидную", "очевидн"},
		{"очевидные", "очевидн"},
		{"очевидный", "очевидн"},
		{"очей", "оч"},
		{"очень", "очен"},
		{"очереди", "очеред"},
		{"очередь", "очеред"},
		{"очерк", "очерк"},
		{"очертаний", "очертан"},
		{"очертания", "очертан"},
		{"очертанье", "очертан"},
		{"очертил", "очерт"},
		{"очертя", "очерт"},
		{"очи", "оч"},
		{"очинил", "очин"},
		{"очините", "очин"},
		{"очинна", "очин"},
		{"очистительных", "очистительн"},
		{"очистить", "очист"},
		{"очистки", "очистк"},
		{"очищает", "очища"},
		{"очищать", "очища"},
		{"очищений", "очищен"},
		{"очию", "оч"},
		{"очками", "очк"},
		{"очках", "очк"},
		{"очки", "очк"},
		{"очнетесь", "очнет"},
		{"очнется", "очнет"},
		{"очнешься", "очнеш"},
		{"очнитесь", "очн"},
		{"очнувшийся", "очнувш"},
		{"очнувшись", "очнувш"},
		{"очнулась", "очнул"},
		{"очнулись", "очнул"},
		{"очнулся", "очнул"},
		{"очнусь", "очн"},
		{"очнуться", "очнут"},
		{"очумел", "очумел"},
		{"очутившиеся", "очут"},
		{"очутившийся", "очут"},
		{"очутилась", "очут"},
		{"очутились", "очут"},
		{"очутилось", "очут"},
		{"очутился", "очут"},
		{"очутиться", "очут"},
		{"очутясь", "очут"},
		{"ошеломлен", "ошеломл"},
		{"ошеломленная", "ошеломлен"},
		{"ошень", "ошен"},
		{"ошибаетесь", "ошиба"},
		{"ошибается", "ошиба"},
		{"ошибаешься", "ошиба"},
		{"ошибалась", "ошиба"},
		{"ошибались", "ошиба"},
		{"ошибался", "ошиба"},
		{"ошибаться", "ошиба"},
		{"ошибаюсь", "ошиба"},
		{"ошибаются", "ошиба"},
		{"ошибетесь", "ошибет"},
		{"ошибиться", "ошиб"},
		{"ошибка", "ошибк"},
		{"ошибке", "ошибк"},
		{"ошибки", "ошибк"},
		{"ошибкой", "ошибк"},
		{"ошибку", "ошибк"},
		{"ошиблась", "ошибл"},
		{"ошиблись", "ошибл"},
		{"ошибок", "ошибок"},
		{"ошибочка", "ошибочк"},
		{"ошибочно", "ошибочн"},
		{"ошибочное", "ошибочн"},
		{"ошибочных", "ошибочн"},
		{"ошибся", "ошиб"},
		{"ошибусь", "ошиб"},
		{"ощеришься", "ощер"},
		{"ощетинившейся", "ощетин"},
		{"ощетинился", "ощетин"},
		{"ощупав", "ощупа"},
		{"ощупал", "ощупа"},
		{"ощупом", "ощуп"},
		{"ощупывает", "ощупыва"},
		{"ощупывая", "ощупыв"},
		{"ощупью", "ощуп"},
		{"ощутил", "ощут"},
		{"ощутилась", "ощут"},
		{"ощутим", "ощут"},
		{"ощутительно", "ощутительн"},
		{"ощутить", "ощут"},
		{"ощущаете", "ощуща"},
		{"ощущается", "ощуща"},
		{"ощущал", "ощуща"},
		{"ощущала", "ощуща"},
		{"ощущать", "ощуща"},
		{"ощущаю", "ощуща"},
		{"ощущая", "ощущ"},
		{"ощущение", "ощущен"},
		{"ощущением", "ощущен"},
		{"ощущении", "ощущен"},
		{"ощущений", "ощущен"},
		{"ощущения", "ощущен"},
		{"п", "п"},
		{"па", "па"},
		{"пава", "пав"},
		{"павел", "павел"},
		{"павильон", "павильон"},
		{"павильонам", "павильон"},
		{"павла", "павл"},
		{"павлиний", "павлин"},
		{"павлиньи", "павлин"},
		{"павлиньим", "павлин"},
		{"павлович", "павлович"},
		{"павловна", "павловн"},
		{"павловне", "павловн"},
		{"павловной", "павловн"},
		{"павловну", "павловн"},
		{"павловны", "павловн"},
		{"павловцы", "павловц"},
		{"павлыч", "павлыч"},
		{"павлыча", "павлыч"},
		{"пагубная", "пагубн"},
		{"падает", "пада"},
		{"падай", "пада"},
		{"падал", "пада"},
		{"падала", "пада"},
		{"падаль", "падал"},
		{"падать", "пада"},
		{"падаю", "пада"},
		{"падают", "пада"},
		{"падающего", "пада"},
		{"падающие", "пада"},
		{"падеж", "падеж"},
		{"падение", "паден"},
		{"падением", "паден"},
		{"падении", "паден"},
		{"падений", "паден"},
		{"падения", "паден"},
		{"паденье", "паден"},
		{"паденья", "паден"},
		{"падет", "падет"},
		{"падут", "падут"},
		{"падучая", "падуч"},
		{"падчерицей", "падчериц"},
		{"падчерицы", "падчериц"},
		{"падшая", "падш"},
		{"падшей", "падш"},
		{"падшему", "падш"},
		{"падший", "падш"},
		{"падшим", "падш"},
		{"падших", "падш"},
		{"падшую", "падш"},
		{"паек", "паек"},
		{"пазухи", "пазух"},
		{"пазуху", "пазух"},
		{"пай", "па"},
		{"пакет", "пакет"},
		{"пакетом", "пакет"},
		{"пакеты", "пакет"},
		{"пакостей", "пакост"},
		{"пакостно", "пакостн"},
		{"пал", "пал"},
		{"пала", "пал"},
		{"паладином", "паладин"},
		{"палата", "палат"},
		{"палате", "палат"},
		{"палату", "палат"},
		{"палаты", "палат"},
		{"пале", "пал"},
		{"палевые", "палев"},
		{"палевых", "палев"},
		{"палец", "палец"},
		{"пали", "пал"},
		{"палил", "пал"},
		{"палим", "пал"},
		{"палима", "палим"},
		{"палимый", "палим"},
		{"палит", "пал"},
		{"палка", "палк"},
		{"палки", "палк"},
		{"палкой", "палк"},
		{"палку", "палк"},
		{"палладиумом", "палладиум"},
		{"палочкой", "палочк"},
		{"палочку", "палочк"},
		{"пальбу", "пальб"},
		{"пальмерстон", "пальмерстон"},
		{"пальмерстоном", "пальмерстон"},
		{"пальмой", "пальм"},
		{"пальмы", "пальм"},
		{"пальто", "пальт"},
		{"пальца", "пальц"},
		{"пальцам", "пальц"},
		{"пальцами", "пальц"},
		{"пальцах", "пальц"},
		{"пальце", "пальц"},
		{"пальцев", "пальц"},
		{"пальцем", "пальц"},
		{"пальцы", "пальц"},
		{"пальчик", "пальчик"},
		{"пальчика", "пальчик"},
		{"пальчики", "пальчик"},
		{"палящая", "паля"},
		{"палящих", "паля"},
		{"памятен", "памят"},
		{"памяти", "памят"},
		{"памятлив", "памятл"},
		{"памятниками", "памятник"},
		{"памятно", "памятн"},
		{"памятной", "памятн"},
		{"память", "памя"},
		{"памятью", "памят"},
		{"пан", "пан"},
		{"панаму", "панам"},
		{"пане", "пан"},
		{"панегирика", "панегирик"},
		{"панели", "панел"},
		{"пани", "пан"},
		{"панихид", "панихид"},
		{"панихиду", "панихид"},
		{"панихиды", "панихид"},
		{"панический", "паническ"},
		{"панического", "паническ"},
		{"панн", "пан"},
		{"панна", "пан"},
		{"панорама", "панорам"},
		{"панораму", "панорам"},
		{"панорамы", "панорам"},
		{"пансион", "пансион"},
		{"пансиона", "пансион"},
		{"пансионе", "пансион"},
		{"пансионерки", "пансионерк"},
		{"пансионной", "пансион"},
		{"панталон", "панталон"},
		{"панталонах", "панталон"},
		{"панталоны", "панталон"},
		{"пантелеевич", "пантелеевич"},
		{"пантелеевна", "пантелеевн"},
		{"пантелеич", "пантелеич"},
		{"пантера", "пантер"},
		{"панцырь", "панцыр"},
		{"папа", "пап"},
		{"папаша", "папаш"},
		{"папашей", "папаш"},
		{"папаши", "папаш"},
		{"папашу", "папаш"},
		{"папенька", "папеньк"},
		{"папеньке", "папеньк"},
		{"папеньки", "папеньк"},
		{"папенькой", "папеньк"},
		{"паперти", "паперт"},
		{"папильотках", "папильотк"},
		{"папирос", "папирос"},
		{"папиросами", "папирос"},
		{"папиросками", "папироск"},
		{"папироски", "папироск"},
		{"папироской", "папироск"},
		{"папироску", "папироск"},
		{"папиросочка", "папиросочк"},
		{"папиросочку", "папиросочк"},
		{"папиросочница", "папиросочниц"},
		{"папиросочницу", "папиросочниц"},
		{"папиросу", "папирос"},
		{"папиросы", "папирос"},
		{"папка", "папк"},
		{"папку", "папк"},
		{"папочка", "папочк"},
		{"папочку", "папочк"},
		{"папы", "пап"},
		{"папье", "пап"},
		{"пар", "пар"},
		{"пара", "пар"},
		{"параболу", "парабол"},
		{"парада", "парад"},
		{"парадная", "парадн"},
		{"парадное", "парадн"},
		{"парадному", "парадн"},
		{"парадных", "парадн"},
		{"параду", "парад"},
		{"парализованы", "парализова"},
		{"паралич", "паралич"},
		{"паралича", "паралич"},
		{"параллель", "параллел"},
		{"параллельно", "параллельн"},
		{"параша", "параш"},
		{"парашу", "параш"},
		{"пардон", "пардон"},
		{"паре", "пар"},
		{"парень", "парен"},
		{"пари", "пар"},
		{"париж", "париж"},
		{"парижа", "париж"},
		{"париже", "париж"},
		{"парижем", "париж"},
		{"парижских", "парижск"},
		{"парижскому", "парижск"},
		{"парикмахера", "парикмахер"},
		{"париком", "парик"},
		{"парили", "пар"},
		{"парит", "пар"},
		{"парк", "парк"},
		{"парка", "парк"},
		{"парке", "парк"},
		{"парку", "парк"},
		{"пармезаном", "пармезан"},
		{"парней", "парн"},
		{"парнем", "парн"},
		{"парнике", "парник"},
		{"парнишка", "парнишк"},
		{"парнишкой", "парнишк"},
		{"парню", "парн"},
		{"парня", "парн"},
		{"парой", "пар"},
		{"пароксизме", "пароксизм"},
		{"пароксизмы", "пароксизм"},
		{"паром", "пар"},
		{"пароход", "пароход"},
		{"парохода", "пароход"},
		{"пароходах", "пароход"},
		{"пароходе", "пароход"},
		{"пароходы", "пароход"},
		{"партере", "партер"},
		{"партией", "парт"},
		{"партии", "парт"},
		{"партию", "парт"},
		{"партия", "парт"},
		{"пару", "пар"},
		{"парус", "парус"},
		{"парусах", "парус"},
		{"парчами", "парч"},
		{"парче", "парч"},
		{"пары", "пар"},
		{"пас", "пас"},
		{"паску", "паск"},
		{"пасли", "пасл"},
		{"пасмурна", "пасмурн"},
		{"пасмурно", "пасмурн"},
		{"пасмурный", "пасмурн"},
		{"паспорт", "паспорт"},
		{"паспорта", "паспорт"},
		{"паспорте", "паспорт"},
		{"паспорту", "паспорт"},
		{"паспорты", "паспорт"},
		{"пассивно", "пассивн"},
		{"пастил", "паст"},
		{"пастух", "пастух"},
		{"пастухи", "пастух"},
		{"пастухом", "пастух"},
		{"пастуший", "пастуш"},
		{"пастушков", "пастушк"},
		{"пастырь", "пастыр"},
		{"пасть", "паст"},
		{"пастья", "паст"},
		{"пасхальной", "пасхальн"},
		{"пасхальный", "пасхальн"},
		{"пасхе", "пасх"},
		{"пасхи", "пасх"},
		{"пасху", "пасх"},
		{"пасьянс", "пасьянс"},
		{"патентованных", "патентова"},
		{"патер", "патер"},
		{"патетическая", "патетическ"},
		{"патетически", "патетическ"},
		{"патетических", "патетическ"},
		{"патетической", "патетическ"},
		{"патетическому", "патетическ"},
		{"патетическую", "патетическ"},
		{"патоки", "паток"},
		{"патриархальной", "патриархальн"},
		{"пауза", "пауз"},
		{"паузы", "пауз"},
		{"паук", "паук"},
		{"пауке", "паук"},
		{"пауки", "паук"},
		{"пауком", "паук"},
		{"паутина", "паутин"},
		{"паутиной", "паутин"},
		{"паутину", "паутин"},
		{"паутины", "паутин"},
		{"паф", "паф"},
		{"пафос", "пафос"},
		{"пахарь", "пахар"},
		{"пахать", "паха"},
		{"пахло", "пахл"},
		{"пахнет", "пахнет"},
		{"пахнул", "пахнул"},
		{"пахнуло", "пахнул"},
		{"пахнут", "пахнут"},
		{"пациент", "пациент"},
		{"пациента", "пациент"},
		{"пациентами", "пациент"},
		{"пациентов", "пациент"},
		{"паче", "пач"},
		{"пачками", "пачк"},
		{"пачки", "пачк"},
		{"пачку", "пачк"},
		{"пачтенье", "пачтен"},
		{"пачулями", "пачул"},
		{"пашен", "паш"},
		{"пашенька", "пашеньк"},
		{"пашеньке", "пашеньк"},
		{"пашеньки", "пашеньк"},
		{"пашенькой", "пашеньк"},
		{"пашка", "пашк"},
		{"пашой", "паш"},
		{"пашут", "пашут"},
		{"паях", "па"},
		{"паяц", "паяц"},
		{"паяца", "паяц"},
		{"пе", "пе"},
		{"певал", "пева"},
		{"певали", "пева"},
		{"певец", "певец"},
		{"певица", "певиц"},
		{"певицы", "певиц"},
		{"певуче", "певуч"},
		{"певучее", "певуч"},
		{"певучей", "певуч"},
		{"певучий", "певуч"},
		{"певучим", "певуч"},
		{"певучих", "певуч"},
		{"певца", "певц"},
		{"певцы", "певц"},
		{"певчий", "певч"},
		{"певчих", "певч"},
		{"певший", "певш"},
		{"пегашке", "пегашк"},
		{"пегашку", "пегашк"},
		{"пегую", "пег"},
		{"педагог", "педагог"},
		{"педагогическом", "педагогическ"},
		{"педагогов", "педагог"},
		{"педант", "педант"},
		{"педантизма", "педантизм"},
		{"педантизмом", "педантизм"},
		{"педантически", "педантическ"},
		{"педантических", "педантическ"},
		{"педантическое", "педантическ"},
		{"педантическую", "педантическ"},
		{"педанты", "педант"},
		{"пей", "пе"},
		{"пейзаж", "пейзаж"},
		{"пейзажа", "пейзаж"},
		{"пейзажах", "пейзаж"},
		{"пейзажей", "пейзаж"},
		{"пейзажи", "пейзаж"},
		{"пейзажист", "пейзажист"},
		{"пейсы", "пейс"},
		{"пейте", "пейт"},
		{"пекли", "пекл"},
		{"пеклись", "пекл"},
		{"пекут", "пекут"},
		{"пекущуюся", "пекущ"},
		{"пел", "пел"},
		{"пела", "пел"},
		{"пелагеи", "пелаг"},
		{"пелагея", "пелаге"},
		{"пеленами", "пелен"},
		{"пеленах", "пелен"},
		{"пеленой", "пелен"},
		{"пелериной", "пелерин"},
		{"пели", "пел"},
		{"пело", "пел"},
		{"пена", "пен"},
		{"пене", "пен"},
		{"пени", "пен"},
		{"пение", "пен"},
		{"пением", "пен"},
		{"пении", "пен"},
		{"пенится", "пен"},
		{"пения", "пен"},
		{"пенки", "пенк"},
		{"пенкин", "пенкин"},
		{"пенкиным", "пенкин"},
		{"пеной", "пен"},
		{"пеною", "пен"},
		{"пенсии", "пенс"},
		{"пенсион", "пенсион"},
		{"пенсиона", "пенсион"},
		{"пенсионе", "пенсион"},
		{"пенсионишко", "пенсионишк"},
		{"пенсионом", "пенсион"},
		{"пенсию", "пенс"},
		{"пень", "пен"},
		{"пенье", "пен"},
		{"пеньем", "пен"},
		{"пеньи", "пен"},
		{"пеняй", "пеня"},
		{"пеняйте", "пеня"},
		{"пенять", "пеня"},
		{"пепел", "пепел"},
		{"пепельницу", "пепельниц"},
		{"пепла", "пепл"},
		{"пеплом", "пепл"},
		{"пера", "пер"},
		{"первая", "перв"},
		{"первейшая", "перв"},
		{"первейшими", "перв"},
		{"первенец", "первенец"},
		{"первенство", "первенств"},
		{"первенствующей", "первенств"},
		{"первенствующую", "первенств"},
		{"первенца", "первенц"},
		{"первенце", "первенц"},
		{"перво", "перв"},
		{"первобытною", "первобытн"},
		{"первобытный", "первобытн"},
		{"первого", "перв"},
		{"первое", "перв"},
		{"первозданный", "первозда"},
		{"первой", "перв"},
		{"первом", "перв"},
		{"первому", "перв"},
		{"первоначальная", "первоначальн"},
		{"первоначально", "первоначальн"},
		{"первоначальное", "первоначальн"},
		{"первоначальную", "первоначальн"},
		{"первоначальные", "первоначальн"},
		{"первоначальный", "первоначальн"},
		{"первоначальных", "первоначальн"},
		{"первою", "перв"},
		{"первую", "перв"},
		{"первые", "перв"},
		{"первый", "перв"},
		{"первым", "перв"},
		{"первых", "перв"},
		{"пергаментным", "пергаментн"},
		{"пергаментом", "пергамент"},
		{"перебегал", "перебега"},
		{"перебегала", "перебега"},
		{"перебегающими", "перебега"},
		{"перебежала", "перебежа"},
		{"переберем", "перебер"},
		{"переберет", "переберет"},
		{"переберешься", "перебереш"},
		{"переберись", "перебер"},
		{"переберусь", "перебер"},
		{"переберут", "переберут"},
		{"перебиваете", "перебива"},
		{"перебивай", "перебива"},
		{"перебивал", "перебива"},
		{"перебивалась", "перебива"},
		{"перебиванья", "перебиван"},
		{"перебиваюсь", "перебива"},
		{"перебивая", "перебив"},
		{"перебившего", "переб"},
		{"перебил", "переб"},
		{"перебила", "переб"},
		{"перебили", "переб"},
		{"перебирает", "перебира"},
		{"перебирается", "перебира"},
		{"перебирай", "перебира"},
		{"перебирала", "перебира"},
		{"перебирать", "перебира"},
		{"перебирая", "перебир"},
		{"перебиты", "перебит"},
		{"перебить", "переб"},
		{"переболело", "переболел"},
		{"перебороло", "переборол"},
		{"перебором", "перебор"},
		{"перебрал", "перебра"},
		{"перебрала", "перебра"},
		{"перебранился", "перебран"},
		{"перебраны", "перебра"},
		{"перебрасывается", "перебрасыва"},
		{"перебудить", "перебуд"},
		{"перебывает", "перебыва"},
		{"перебывали", "перебыва"},
		{"перебывало", "перебыва"},
		{"перебьют", "перебьют"},
		{"перевале", "перевал"},
		{"переваливаться", "перевалива"},
		{"переваливаясь", "перевалив"},
		{"перевалился", "перевал"},
		{"переварилась", "перевар"},
		{"переварится", "перевар"},
		{"переварить", "перевар"},
		{"переведет", "переведет"},
		{"перевез", "перевез"},
		{"перевезенную", "перевезен"},
		{"перевезли", "перевезл"},
		{"перевезут", "перевезут"},
		{"перевел", "перевел"},
		{"перевелась", "перевел"},
		{"перевели", "перевел"},
		{"перевелись", "перевел"},
		{"перевернется", "перевернет"},
		{"перевернул", "перевернул"},
		{"перевернула", "перевернул"},
		{"перевернули", "перевернул"},
		{"перевернуло", "перевернул"},
		{"перевернулось", "перевернул"},
		{"перевернулся", "перевернул"},
		{"перевернуть", "перевернут"},
		{"перевертел", "перевертел"},
		{"перевертывается", "перевертыва"},
		{"перевертывать", "перевертыва"},
		{"перевертываясь", "перевертыв"},
		{"перевес", "перевес"},
		{"перевести", "перевест"},
		{"перевесть", "перевест"},
		{"перевиваясь", "перевив"},
		{"перевод", "перевод"},
		{"переводе", "перевод"},
		{"переводили", "перевод"},
		{"переводить", "перевод"},
		{"переводной", "переводн"},
		{"переводов", "перевод"},
		{"переводом", "перевод"},
		{"переводы", "перевод"},
		{"переводя", "перевод"},
		{"переводят", "перевод"},
		{"перевожу", "перевож"},
		{"перевозиться", "перевоз"},
		{"перевозка", "перевозк"},
		{"перевозки", "перевозк"},
		{"перевозят", "перевоз"},
		{"переворачивается", "переворачива"},
		{"переворачивался", "переворачива"},
		{"переворачивая", "переворачив"},
		{"переворот", "переворот"},
		{"переворотил", "переворот"},
		{"переворотов", "переворот"},
		{"переворочаешь", "перевороча"},
		{"переврала", "перевра"},
		{"перевязан", "перевяза"},
		{"перевязана", "перевяза"},
		{"перевязочным", "перевязочн"},
		{"перевязь", "перевяз"},
		{"переглотают", "переглота"},
		{"переглянулась", "переглянул"},
		{"переглянулся", "переглянул"},
		{"перегнувшись", "перегнувш"},
		{"переговаривал", "переговарива"},
		{"переговорено", "переговор"},
		{"переговорив", "переговор"},
		{"переговорил", "переговор"},
		{"переговорили", "переговор"},
		{"переговорить", "переговор"},
		{"переговоры", "переговор"},
		{"переговорю", "переговор"},
		{"перегорает", "перегора"},
		{"перегородке", "перегородк"},
		{"перегородки", "перегородк"},
		{"перегородкой", "перегородк"},
		{"перегородку", "перегородк"},
		{"перед", "перед"},
		{"передав", "переда"},
		{"передаваемая", "передава"},
		{"передаваемой", "передава"},
		{"передавал", "передава"},
		{"передавала", "передава"},
		{"передавали", "передава"},
		{"передавать", "передава"},
		{"передавая", "передав"},
		{"передадите", "передад"},
		{"передает", "переда"},
		{"передайте", "переда"},
		{"передал", "переда"},
		{"передала", "переда"},
		{"передали", "переда"},
		{"передам", "перед"},
		{"переданной", "переда"},
		{"передаст", "передаст"},
		{"передать", "переда"},
		{"передаю", "переда"},
		{"передвигая", "передвиг"},
		{"переделает", "передела"},
		{"переделаешь", "передела"},
		{"переделала", "передела"},
		{"переделано", "передела"},
		{"переделать", "передела"},
		{"переделки", "переделк"},
		{"переделку", "переделк"},
		{"переделывать", "переделыва"},
		{"передергивая", "передергив"},
		{"передернуло", "передернул"},
		{"передернут", "передернут"},
		{"передке", "передк"},
		{"передней", "передн"},
		{"переднюю", "передн"},
		{"передо", "перед"},
		{"передовые", "передов"},
		{"передовыми", "передов"},
		{"передовых", "передов"},
		{"передом", "перед"},
		{"передразниванье", "передразниван"},
		{"передразнил", "передразн"},
		{"передумав", "передума"},
		{"передумает", "передума"},
		{"передумал", "передума"},
		{"передумала", "передума"},
		{"передумано", "передума"},
		{"передумать", "передума"},
		{"передумывать", "передумыва"},
		{"передушил", "передуш"},
		{"переедем", "переед"},
		{"переедет", "переедет"},
		{"переедете", "переедет"},
		{"переедешь", "переедеш"},
		{"перееду", "переед"},
		{"переежаем", "переежа"},
		{"переезд", "переезд"},
		{"переезда", "переезд"},
		{"переезде", "переезд"},
		{"переездкой", "переездк"},
		{"переездом", "переезд"},
		{"переезжаем", "переезжа"},
		{"переезжает", "переезжа"},
		{"переезжай", "переезжа"},
		{"переезжайте", "переезжа"},
		{"переезжал", "переезжа"},
		{"переезжали", "переезжа"},
		{"переезжать", "переезжа"},
		{"переезжаю", "переезжа"},
		{"переезжают", "переезжа"},
		{"переезжая", "переезж"},
		{"переехал", "перееха"},
		{"переехала", "перееха"},
		{"переехали", "перееха"},
		{"переехать", "перееха"},
		{"переешь", "перееш"},
		{"переждав", "пережда"},
		{"переждал", "пережда"},
		{"переждать", "пережда"},
		{"пережевывает", "пережевыва"},
		{"переживает", "пережива"},
		{"переживала", "пережива"},
		{"переживаю", "пережива"},
		{"переживем", "пережив"},
		{"переживешь", "переживеш"},
		{"переживу", "пережив"},
		{"пережил", "переж"},
		{"пережит", "переж"},
		{"пережитых", "пережит"},
		{"пережить", "переж"},
		{"перезрелою", "перезрел"},
		{"переименованы", "переименова"},
		{"переимчивым", "переимчив"},
		{"переиначил", "переинач"},
		{"переиначить", "переинач"},
		{"перейдет", "перейдет"},
		{"перейдете", "перейдет"},
		{"перейдут", "перейдут"},
		{"перейдя", "перейд"},
		{"перейти", "перейт"},
		{"перекидывались", "перекидыва"},
		{"перекинет", "перекинет"},
		{"перекинул", "перекинул"},
		{"перекинуть", "перекинут"},
		{"перекладных", "перекладн"},
		{"перекладывал", "перекладыва"},
		{"перекладывала", "перекладыва"},
		{"перекликаются", "переклика"},
		{"перековать", "перекова"},
		{"перекосились", "перекос"},
		{"перекосилось", "перекос"},
		{"перекрестив", "перекрест"},
		{"перекрестившись", "перекрест"},
		{"перекрестила", "перекрест"},
		{"перекрестилась", "перекрест"},
		{"перекрестился", "перекрест"},
		{"перекрестись", "перекрест"},
		{"перекрестит", "перекрест"},
		{"перекрестка", "перекрестк"},
		{"перекрестках", "перекрестк"},
		{"перекрестке", "перекрестк"},
		{"перекрестку", "перекрестк"},
		{"перекрестный", "перекрестн"},
		{"перекресток", "перекресток"},
		{"перекрестясь", "перекрест"},
		{"перекрещивались", "перекрещива"},
		{"перекрещивания", "перекрещиван"},
		{"перекрещиваясь", "перекрещив"},
		{"перекрещу", "перекрещ"},
		{"перекупленном", "перекуплен"},
		{"перекусил", "перекус"},
		{"перелетная", "перелетн"},
		{"перелетные", "перелетн"},
		{"переливает", "перелива"},
		{"переливается", "перелива"},
		{"переливы", "перелив"},
		{"перелилась", "перел"},
		{"перелистывает", "перелистыва"},
		{"перелистывай", "перелистыва"},
		{"перелистывал", "перелистыва"},
		{"перелистывать", "перелистыва"},
		{"перелистывая", "перелистыв"},
		{"переложил", "перелож"},
		{"переложили", "перелож"},
		{"переложит", "перелож"},
		{"перелом", "перел"},
		{"перелома", "перелом"},
		{"переломали", "перелома"},
		{"переломаны", "перелома"},
		{"переломится", "перелом"},
		{"перельет", "перельет"},
		{"перемелется", "перемелет"},
		{"перемен", "перем"},
		{"перемена", "перем"},
		{"перемене", "перемен"},
		{"перемени", "перемен"},
		{"переменившемся", "перемен"},
		{"переменившимся", "перемен"},
		{"переменил", "перемен"},
		{"переменила", "перемен"},
		{"переменилась", "перемен"},
		{"переменили", "перемен"},
		{"переменились", "перемен"},
		{"переменилось", "перемен"},
		{"переменился", "перемен"},
		{"переменит", "перемен"},
		{"перемените", "перемен"},
		{"переменить", "перемен"},
		{"перемениться", "перемен"},
		{"переменного", "перемен"},
		{"перемены", "перем"},
		{"переменяете", "переменя"},
		{"переменят", "перемен"},
		{"переменятся", "перемен"},
		{"переменять", "переменя"},
		{"переменяя", "перемен"},
		{"перемешал", "перемеша"},
		{"перемешались", "перемеша"},
		{"перемешаны", "перемеша"},
		{"переминаясь", "перемин"},
		{"перемолвить", "перемолв"},
		{"перемолола", "перемолол"},
		{"перемыть", "перем"},
		{"перенес", "перенес"},
		{"перенесемся", "перенес"},
		{"перенесении", "перенесен"},
		{"перенесено", "перенес"},
		{"перенесет", "перенесет"},
		{"перенесешь", "перенесеш"},
		{"перенесла", "перенесл"},
		{"перенесли", "перенесл"},
		{"перенесся", "перенес"},
		{"перенести", "перенест"},
		{"перенестись", "перенест"},
		{"перенесть", "перенест"},
		{"перенесу", "перенес"},
		{"перенесут", "перенесут"},
		{"переносил", "перенос"},
		{"переносила", "перенос"},
		{"переносили", "перенос"},
		{"переносит", "перенос"},
		{"переносится", "перенос"},
		{"переносить", "перенос"},
		{"переносье", "перенос"},
		{"переночуют", "переноч"},
		{"переношу", "перенош"},
		{"перенял", "переня"},
		{"перенятым", "перенят"},
		{"переодеться", "переодет"},
		{"перепало", "перепа"},
		{"перепахали", "перепаха"},
		{"перепел", "перепел"},
		{"перепела", "перепел"},
		{"перепелки", "перепелк"},
		{"перепелов", "перепел"},
		{"переписал", "переписа"},
		{"переписана", "переписа"},
		{"переписать", "переписа"},
		{"переписка", "переписк"},
		{"переписывали", "переписыва"},
		{"переписыванье", "переписыван"},
		{"переписывать", "переписыва"},
		{"перепишет", "перепишет"},
		{"перепишете", "перепишет"},
		{"переплет", "переплет"},
		{"переплете", "переплет"},
		{"переплетом", "переплет"},
		{"переплыть", "перепл"},
		{"переползанье", "переползан"},
		{"переползанья", "переползан"},
		{"переполненном", "переполнен"},
		{"переполненный", "переполнен"},
		{"переполнено", "переполн"},
		{"переполнять", "переполня"},
		{"переправил", "переправ"},
		{"переправился", "переправ"},
		{"переправить", "переправ"},
		{"переправиться", "переправ"},
		{"перепродает", "перепрода"},
		{"перепродай", "перепрода"},
		{"перепугается", "перепуга"},
		{"перепугаешься", "перепуга"},
		{"перепугалась", "перепуга"},
		{"перепугались", "перепуга"},
		{"перепугался", "перепуга"},
		{"перепуганная", "перепуга"},
		{"перепуганные", "перепуга"},
		{"перепуганными", "перепуга"},
		{"перепутает", "перепута"},
		{"перепутаешь", "перепута"},
		{"перепуталось", "перепута"},
		{"перепутанные", "перепута"},
		{"перепьются", "перепьют"},
		{"переработалась", "переработа"},
		{"перерастает", "перераста"},
		{"перервал", "перерва"},
		{"перерву", "перерв"},
		{"перерезывала", "перерезыва"},
		{"переродило", "перерод"},
		{"перерождавшаяся", "перерожда"},
		{"перерождения", "перерожден"},
		{"перерыв", "перер"},
		{"перерывая", "перерыв"},
		{"перерыву", "перерыв"},
		{"перерывы", "перерыв"},
		{"переряженный", "переряжен"},
		{"пересветов", "пересвет"},
		{"пересекло", "пересекл"},
		{"пересел", "пересел"},
		{"переселился", "пересел"},
		{"переселимся", "пересел"},
		{"переселить", "пересел"},
		{"переселиться", "пересел"},
		{"переселяемся", "переселя"},
		{"пересиливает", "пересилива"},
		{"пересиливал", "пересилива"},
		{"пересиливая", "пересилив"},
		{"пересилил", "пересил"},
		{"пересилила", "пересил"},
		{"перескажу", "перескаж"},
		{"пересказал", "пересказа"},
		{"пересказать", "пересказа"},
		{"пересказывал", "пересказыва"},
		{"пересказывала", "пересказыва"},
		{"пересказывать", "пересказыва"},
		{"перескакивали", "перескакива"},
		{"перескакивать", "перескакива"},
		{"перескочить", "перескоч"},
		{"переславший", "пересла"},
		{"переслал", "пересла"},
		{"переслать", "пересла"},
		{"переслюнить", "переслюн"},
		{"пересмешку", "пересмешк"},
		{"пересмотрю", "пересмотр"},
		{"пересохли", "пересохл"},
		{"пересохло", "пересохл"},
		{"переспорил", "переспор"},
		{"переспрашиваемый", "переспрашива"},
		{"переспросил", "переспрос"},
		{"переспросила", "переспрос"},
		{"перессорились", "перессор"},
		{"перестав", "переста"},
		{"переставал", "перестава"},
		{"переставала", "перестава"},
		{"переставали", "перестава"},
		{"переставало", "перестава"},
		{"переставая", "перестав"},
		{"переставил", "перестав"},
		{"переставлял", "переставля"},
		{"перестает", "переста"},
		{"перестал", "переста"},
		{"перестала", "переста"},
		{"перестали", "переста"},
		{"перестало", "переста"},
		{"перестанем", "перестан"},
		{"перестанемте", "перестанемт"},
		{"перестанет", "перестанет"},
		{"перестанете", "перестанет"},
		{"перестанешь", "перестанеш"},
		{"перестану", "перестан"},
		{"перестанут", "перестанут"},
		{"перестань", "перестан"},
		{"перестаньте", "перестаньт"},
		{"перестать", "переста"},
		{"перестаю", "переста"},
		{"перестают", "переста"},
		{"перестрадать", "перестрада"},
		{"перестройки", "перестройк"},
		{"перестройку", "перестройк"},
		{"переступал", "переступа"},
		{"переступать", "переступа"},
		{"переступают", "переступа"},
		{"переступая", "переступ"},
		{"переступив", "переступ"},
		{"переступил", "переступ"},
		{"переступила", "переступ"},
		{"переступить", "переступ"},
		{"пересуды", "пересуд"},
		{"пересчитаешь", "пересчита"},
		{"пересчитал", "пересчита"},
		{"пересчитают", "пересчита"},
		{"пересчитывал", "пересчитыва"},
		{"пересылала", "пересыла"},
		{"пересылались", "пересыла"},
		{"пересыпать", "пересыпа"},
		{"перетаскано", "перетаска"},
		{"перетаскивать", "перетаскива"},
		{"перетасовка", "перетасовк"},
		{"перетащил", "перетащ"},
		{"перетащить", "перетащ"},
		{"переулка", "переулк"},
		{"переулкам", "переулк"},
		{"переулках", "переулк"},
		{"переулке", "переулк"},
		{"переулки", "переулк"},
		{"переулком", "переулк"},
		{"переулку", "переулк"},
		{"переулок", "переулок"},
		{"перехватила", "перехват"},
		{"перехватить", "перехват"},
		{"перехитренному", "перехитрен"},
		{"переход", "переход"},
		{"перехода", "переход"},
		{"переходил", "переход"},
		{"переходила", "переход"},
		{"переходили", "переход"},
		{"переходим", "переход"},
		{"переходит", "переход"},
		{"переходить", "переход"},
		{"переходя", "переход"},
		{"переходят", "переход"},
		{"переходящее", "переходя"},
		{"перехожу", "перехож"},
		{"перечислил", "перечисл"},
		{"перечитав", "перечита"},
		{"перечитал", "перечита"},
		{"перечница", "перечниц"},
		{"перешагнешь", "перешагнеш"},
		{"перешагнул", "перешагнул"},
		{"перешагнуть", "перешагнут"},
		{"перешедшее", "перешедш"},
		{"перешедший", "перешедш"},
		{"перешел", "перешел"},
		{"перешептал", "перешепта"},
		{"перешептываться", "перешептыва"},
		{"перешивал", "перешива"},
		{"перешить", "переш"},
		{"перешла", "перешл"},
		{"перешли", "перешл"},
		{"перешло", "перешл"},
		{"перещеголял", "перещеголя"},
		{"перил", "пер"},
		{"перила", "пер"},
		{"перилам", "перил"},
		{"перин", "перин"},
		{"перине", "перин"},
		{"перинное", "перин"},
		{"перины", "перин"},
		{"период", "период"},
		{"периодически", "периодическ"},
		{"периодического", "периодическ"},
		{"периодической", "периодическ"},
		{"периодическою", "периодическ"},
		{"периодическую", "периодическ"},
		{"периферию", "перифер"},
		{"перл", "перл"},
		{"перламутра", "перламутр"},
		{"перлы", "перл"},
		{"перо", "пер"},
		{"пером", "пер"},
		{"персидским", "персидск"},
		{"персидской", "персидск"},
		{"персидском", "персидск"},
		{"персики", "персик"},
		{"персон", "персон"},
		{"перспектива", "перспектив"},
		{"перспективе", "перспектив"},
		{"перспективой", "перспектив"},
		{"перспективу", "перспектив"},
		{"перспективы", "перспектив"},
		{"перст", "перст"},
		{"перстень", "перстен"},
		{"перстней", "перстн"},
		{"перстнем", "перстн"},
		{"перстнями", "перстн"},
		{"перстнях", "перстн"},
		{"перстом", "перст"},
		{"персты", "перст"},
		{"перу", "пер"},
		{"перуджийский", "перуджийск"},
		{"перуджино", "перуджин"},
		{"перуджия", "перудж"},
		{"перунами", "перун"},
		{"перцу", "перц"},
		{"перчатками", "перчатк"},
		{"перчатках", "перчатк"},
		{"перчатке", "перчатк"},
		{"перчатки", "перчатк"},
		{"перчатку", "перчатк"},
		{"перчаток", "перчаток"},
		{"першить", "перш"},
		{"перышко", "перышк"},
		{"перьев", "пер"},
		{"перья", "пер"},
		{"перьями", "пер"},
		{"пес", "пес"},
		{"песен", "пес"},
		{"песенка", "песенк"},
		{"песенки", "песенк"},
		{"песенку", "песенк"},
		{"песенники", "песенник"},
		{"песенников", "песенник"},
		{"песка", "песк"},
		{"песках", "песк"},
		{"песке", "песк"},
		{"пескину", "пескин"},
		{"песком", "песк"},
		{"песку", "песк"},
		{"песне", "песн"},
		{"песней", "песн"},
		{"песни", "песн"},
		{"песнию", "песн"},
		{"песнь", "песн"},
		{"песнью", "песн"},
		{"песню", "песн"},
		{"песня", "песн"},
		{"песнями", "песн"},
		{"песнях", "песн"},
		{"песок", "песок"},
		{"пестиком", "пестик"},
		{"пестрая", "пестр"},
		{"пестреют", "пестреют"},
		{"пестрили", "пестр"},
		{"пестрит", "пестр"},
		{"пестрой", "пестр"},
		{"пестротой", "пестрот"},
		{"пестрою", "пестр"},
		{"пеструшка", "пеструшк"},
		{"пестрые", "пестр"},
		{"пестрый", "пестр"},
		{"пестряков", "пестряк"},
		{"пестрякова", "пестряков"},
		{"пестряковым", "пестряков"},
		{"песчаной", "песчан"},
		{"песчаную", "песчан"},
		{"песчаные", "песчан"},
		{"песчинка", "песчинк"},
		{"петей", "пет"},
		{"петель", "петел"},
		{"петербург", "петербург"},
		{"петербурга", "петербург"},
		{"петербурге", "петербург"},
		{"петербургом", "петербург"},
		{"петербургская", "петербургск"},
		{"петербургские", "петербургск"},
		{"петербургский", "петербургск"},
		{"петербургским", "петербургск"},
		{"петербургских", "петербургск"},
		{"петербургское", "петербургск"},
		{"петербургской", "петербургск"},
		{"петербургском", "петербургск"},
		{"петербургскую", "петербургск"},
		{"петербуржцу", "петербуржц"},
		{"петле", "петл"},
		{"петлей", "петл"},
		{"петли", "петл"},
		{"петлицах", "петлиц"},
		{"петлице", "петлиц"},
		{"петлицу", "петлиц"},
		{"петлю", "петл"},
		{"петля", "петл"},
		{"петр", "петр"},
		{"петра", "петр"},
		{"петре", "петр"},
		{"петров", "петр"},
		{"петрович", "петрович"},
		{"петровича", "петрович"},
		{"петровиче", "петрович"},
		{"петровичем", "петрович"},
		{"петровичу", "петрович"},
		{"петровки", "петровк"},
		{"петровна", "петровн"},
		{"петровне", "петровн"},
		{"петровной", "петровн"},
		{"петровну", "петровн"},
		{"петровны", "петровн"},
		{"петровский", "петровск"},
		{"петровского", "петровск"},
		{"петроградское", "петроградск"},
		{"петроградском", "петроградск"},
		{"петром", "петр"},
		{"петру", "петр"},
		{"петруша", "петруш"},
		{"петрушка", "петрушк"},
		{"петрушке", "петрушк"},
		{"петрушку", "петрушк"},
		{"петруща", "петрущ"},
		{"петух", "петух"},
		{"петуха", "петух"},
		{"петухи", "петух"},
		{"петухом", "петух"},
		{"петушка", "петушк"},
		{"петь", "пет"},
		{"петя", "пет"},
		{"пехота", "пехот"},
		{"пехоте", "пехот"},
		{"пехотного", "пехотн"},
		{"печалей", "печал"},
		{"печален", "печал"},
		{"печали", "печа"},
		{"печалились", "печал"},
		{"печалить", "печал"},
		{"печалиться", "печал"},
		{"печаль", "печал"},
		{"печальна", "печальн"},
		{"печальная", "печальн"},
		{"печально", "печальн"},
		{"печального", "печальн"},
		{"печальное", "печальн"},
		{"печальной", "печальн"},
		{"печальном", "печальн"},
		{"печальною", "печальн"},
		{"печальный", "печальн"},
		{"печальным", "печальн"},
		{"печальных", "печальн"},
		{"печалью", "печал"},
		{"печалями", "печал"},
		{"печами", "печ"},
		{"печатаете", "печата"},
		{"печатаетесь", "печата"},
		{"печатал", "печата"},
		{"печататься", "печата"},
		{"печати", "печат"},
		{"печатными", "печатн"},
		{"печатных", "печатн"},
		{"печать", "печа"},
		{"печатью", "печат"},
		{"печени", "печен"},
		{"печений", "печен"},
		{"печенка", "печенк"},
		{"печенками", "печенк"},
		{"печеную", "печен"},
		{"печенья", "печен"},
		{"печет", "печет"},
		{"печи", "печ"},
		{"печке", "печк"},
		{"печки", "печк"},
		{"печкой", "печк"},
		{"печку", "печк"},
		{"печь", "печ"},
		{"пешкой", "пешк"},
		{"пешком", "пешк"},
		{"пешни", "пешн"},
		{"пещера", "пещер"},
		{"пещеры", "пещер"},
		{"пианино", "пианин"},
		{"пива", "пив"},
		{"пивал", "пива"},
		{"пивали", "пива"},
		{"пиво", "пив"},
		{"пивом", "пив"},
		{"пивца", "пивц"},
		{"пивцо", "пивц"},
		{"пигмалионом", "пигмалион"},
		{"пидерита", "пидерит"},
		{"пиджак", "пиджак"},
		{"пиджаке", "пиджак"},
		{"пиджаки", "пиджак"},
		{"пик", "пик"},
		{"пикирована", "пикирова"},
		{"пикированная", "пикирова"},
		{"пикированным", "пикирова"},
		{"пикнуть", "пикнут"},
		{"пил", "пил"},
		{"пила", "пил"},
		{"пили", "пил"},
		{"пилит", "пил"},
		{"пилот", "пилот"},
		{"пилюли", "пилюл"},
		{"пилюлю", "пилюл"},
		{"пион", "пион"},
		{"пир", "пир"},
		{"пира", "пир"},
		{"пирамидами", "пирамид"},
		{"пирамиду", "пирамид"},
		{"пирамиды", "пирамид"},
		{"пире", "пир"},
		{"пиров", "пир"},
		{"пирог", "пирог"},
		{"пирога", "пирог"},
		{"пироги", "пирог"},
		{"пирогов", "пирог"},
		{"пирогом", "пирог"},
		{"пирожка", "пирожк"},
		{"пирожки", "пирожк"},
		{"пирожное", "пирожн"},
		{"пирожных", "пирожн"},
		{"пиру", "пир"},
		{"пирушках", "пирушк"},
		{"пиры", "пир"},
		{"писал", "писа"},
		{"писала", "писа"},
		{"писали", "писа"},
		{"писало", "писа"},
		{"писалось", "писа"},
		{"писан", "писа"},
		{"писаний", "писан"},
		{"писанное", "писа"},
		{"писанный", "писа"},
		{"писаных", "писан"},
		{"писаньем", "писан"},
		{"писанья", "писан"},
		{"писаньям", "писан"},
		{"писарем", "писар"},
		{"писаришек", "писаришек"},
		{"писаришками", "писаришк"},
		{"писаришки", "писаришк"},
		{"писарь", "писар"},
		{"писаря", "писар"},
		{"писателей", "писател"},
		{"писателем", "писател"},
		{"писатели", "писател"},
		{"писатель", "писател"},
		{"писательницей", "писательниц"},
		{"писательство", "писательств"},
		{"писателю", "писател"},
		{"писателя", "писател"},
		{"писателям", "писател"},
		{"писателями", "писател"},
		{"писать", "писа"},
		{"писем", "пис"},
		{"писец", "писец"},
		{"писк", "писк"},
		{"пискариной", "пискарин"},
		{"пистолет", "пистолет"},
		{"писца", "писц"},
		{"писцов", "писц"},
		{"писцом", "писц"},
		{"писцы", "писц"},
		{"писывал", "писыва"},
		{"письма", "письм"},
		{"письмах", "письм"},
		{"письме", "письм"},
		{"письменно", "письмен"},
		{"письменном", "письмен"},
		{"письменному", "письмен"},
		{"письменный", "письмен"},
		{"письменным", "письмен"},
		{"письмецо", "письмец"},
		{"письмецом", "письмец"},
		{"письмо", "письм"},
		{"письмоводителем", "письмоводител"},
		{"письмоводитель", "письмоводител"},
		{"письмоводителю", "письмоводител"},
		{"письмоводителя", "письмоводител"},
		{"письмом", "письм"},
		{"письму", "письм"},
		{"питает", "пита"},
		{"питал", "пита"},
		{"питала", "пита"},
		{"питалась", "пита"},
		{"питали", "пита"},
		{"питание", "питан"},
		{"питать", "пита"},
		{"питаться", "пита"},
		{"питаю", "пита"},
		{"питая", "пит"},
		{"питаясь", "пит"},
		{"питейное", "питейн"},
		{"питейной", "питейн"},
		{"питер", "питер"},
		{"питере", "питер"},
		{"питером", "питер"},
		{"питии", "пит"},
		{"питомник", "питомник"},
		{"питомце", "питомц"},
		{"питомцы", "питомц"},
		{"пить", "пит"},
		{"питье", "пит"},
		{"питья", "пит"},
		{"пиф", "пиф"},
		{"пишем", "пиш"},
		{"пишет", "пишет"},
		{"пишете", "пишет"},
		{"пишется", "пишет"},
		{"пишешь", "пишеш"},
		{"пиши", "пиш"},
		{"пишите", "пиш"},
		{"пишу", "пиш"},
		{"пишут", "пишут"},
		{"пишутся", "пишут"},
		{"пишущим", "пишущ"},
		{"пища", "пищ"},
		{"пищали", "пища"},
		{"пищат", "пищат"},
		{"пище", "пищ"},
		{"пищи", "пищ"},
		{"пищик", "пищик"},
		{"пищика", "пищик"},
		{"пищиков", "пищик"},
		{"пищиком", "пищик"},
		{"пищику", "пищик"},
		{"пищу", "пищ"},
		{"пиявки", "пиявк"},
		{"пла", "пла"},
		{"плавает", "плава"},
		{"плаванье", "плаван"},
		{"плавать", "плава"},
		{"плавающих", "плава"},
		{"плавит", "плав"},
		{"плавно", "плавн"},
		{"плавное", "плавн"},
		{"плавный", "плавн"},
		{"плакавшие", "плака"},
		{"плакавших", "плака"},
		{"плакавшую", "плака"},
		{"плакал", "плака"},
		{"плакала", "плака"},
		{"плакали", "плака"},
		{"плакались", "плака"},
		{"плакало", "плака"},
		{"плакаль", "плакал"},
		{"плакать", "плака"},
		{"плакиды", "плакид"},
		{"плаксиво", "плаксив"},
		{"плакун", "плакун"},
		{"пламенем", "пламен"},
		{"пламени", "пламен"},
		{"пламенная", "пламен"},
		{"пламенной", "пламен"},
		{"пламенном", "пламен"},
		{"пламенные", "пламен"},
		{"пламень", "пламен"},
		{"пламя", "плам"},
		{"план", "план"},
		{"плана", "план"},
		{"планами", "план"},
		{"планах", "план"},
		{"плане", "план"},
		{"планет", "планет"},
		{"планета", "планет"},
		{"планетах", "планет"},
		{"планете", "планет"},
		{"планету", "планет"},
		{"планеты", "планет"},
		{"планом", "план"},
		{"планомерно", "планомерн"},
		{"плантатору", "плантатор"},
		{"плану", "план"},
		{"планы", "план"},
		{"пластинка", "пластинк"},
		{"пластырями", "пластыр"},
		{"плат", "плат"},
		{"платеже", "платеж"},
		{"плати", "плат"},
		{"платил", "плат"},
		{"платили", "плат"},
		{"платиль", "платил"},
		{"платим", "плат"},
		{"платит", "плат"},
		{"платите", "плат"},
		{"платить", "плат"},
		{"платишь", "плат"},
		{"платка", "платк"},
		{"платками", "платк"},
		{"платке", "платк"},
		{"платки", "платк"},
		{"платком", "платк"},
		{"платку", "платк"},
		{"платок", "платок"},
		{"платона", "платон"},
		{"платочек", "платочек"},
		{"платочком", "платочк"},
		{"платою", "плат"},
		{"плату", "плат"},
		{"платформу", "платформ"},
		{"платье", "плат"},
		{"платьев", "плат"},
		{"платьем", "плат"},
		{"платьица", "платьиц"},
		{"платьице", "платьиц"},
		{"платьишке", "платьишк"},
		{"платьишки", "платьишк"},
		{"платью", "плат"},
		{"платья", "плат"},
		{"платьями", "плат"},
		{"платьях", "плат"},
		{"платя", "плат"},
		{"платят", "плат"},
		{"плач", "плач"},
		{"плача", "плач"},
		{"плаче", "плач"},
		{"плачевный", "плачевн"},
		{"плачевным", "плачевн"},
		{"плачем", "плач"},
		{"плачет", "плачет"},
		{"плачете", "плачет"},
		{"плачешь", "плачеш"},
		{"плачу", "плач"},
		{"плачут", "плачут"},
		{"плачущее", "плачущ"},
		{"плачущим", "плачущ"},
		{"плачущих", "плачущ"},
		{"плачущую", "плачущ"},
		{"плачь", "плач"},
		{"плачьте", "плачьт"},
		{"плащ", "плащ"},
		{"плаща", "плащ"},
		{"плащах", "плащ"},
		{"плаще", "плащ"},
		{"плащом", "плащ"},
		{"пле", "пле"},
		{"плевал", "плева"},
		{"плеванье", "плеван"},
		{"плевны", "плевн"},
		{"плевок", "плевок"},
		{"плед", "плед"},
		{"пледом", "плед"},
		{"плелся", "плел"},
		{"племени", "племен"},
		{"племянника", "племянник"},
		{"племянниками", "племянник"},
		{"племянница", "племянниц"},
		{"плен", "плен"},
		{"пленительна", "пленительн"},
		{"пленительные", "пленительн"},
		{"пленная", "плен"},
		{"пленник", "пленник"},
		{"пленной", "плен"},
		{"плену", "плен"},
		{"пленя", "плен"},
		{"пленял", "пленя"},
		{"пленялись", "пленя"},
		{"пленяло", "пленя"},
		{"пленять", "пленя"},
		{"плеск", "плеск"},
		{"плесканье", "плескан"},
		{"плесну", "плесн"},
		{"плеснул", "плеснул"},
		{"плеснула", "плеснул"},
		{"плеснули", "плеснул"},
		{"плести", "плест"},
		{"плетеных", "плетен"},
		{"плетень", "плетен"},
		{"плетет", "плетет"},
		{"плети", "плет"},
		{"плетнем", "плетн"},
		{"плетни", "плетн"},
		{"плетня", "плетн"},
		{"плетут", "плетут"},
		{"плеч", "плеч"},
		{"плеча", "плеч"},
		{"плечам", "плеч"},
		{"плечами", "плеч"},
		{"плечах", "плеч"},
		{"плече", "плеч"},
		{"плечи", "плеч"},
		{"плечики", "плечик"},
		{"плечо", "плеч"},
		{"плечом", "плеч"},
		{"плечу", "плеч"},
		{"плешивым", "плешив"},
		{"плещет", "плещет"},
		{"плибьет", "плибьет"},
		{"плисовых", "плисов"},
		{"плита", "плит"},
		{"плитам", "плит"},
		{"плитами", "плит"},
		{"плитах", "плит"},
		{"плите", "плит"},
		{"плиты", "плит"},
		{"плод", "плод"},
		{"плодами", "плод"},
		{"плодить", "плод"},
		{"плодов", "плод"},
		{"плодотворная", "плодотворн"},
		{"плодотворной", "плодотворн"},
		{"плоды", "плод"},
		{"плоские", "плоск"},
		{"плоско", "плоск"},
		{"плоти", "плот"},
		{"плотин", "плотин"},
		{"плотине", "плотин"},
		{"плотнее", "плотн"},
		{"плотник", "плотник"},
		{"плотника", "плотник"},
		{"плотно", "плотн"},
		{"плотного", "плотн"},
		{"плотное", "плотн"},
		{"плотной", "плотн"},
		{"плотный", "плотн"},
		{"плотных", "плотн"},
		{"плоты", "плот"},
		{"плоть", "плот"},
		{"плох", "плох"},
		{"плоха", "плох"},
		{"плохая", "плох"},
		{"плохие", "плох"},
		{"плохих", "плох"},
		{"плохо", "плох"},
		{"плохонький", "плохоньк"},
		{"плохую", "плох"},
		{"плоше", "плош"},
		{"площади", "площад"},
		{"площадка", "площадк"},
		{"площадке", "площадк"},
		{"площадки", "площадк"},
		{"площадная", "площадн"},
		{"площадной", "площадн"},
		{"площадь", "площад"},
		{"площадям", "площад"},
		{"площадях", "площад"},
		{"плугом", "плуг"},
		{"плут", "плут"},
		{"плутоват", "плутоват"},
		{"плутоватою", "плутоват"},
		{"плутовка", "плутовк"},
		{"плутовская", "плутовск"},
		{"плутовски", "плутовск"},
		{"плутовскою", "плутовск"},
		{"плутовства", "плутовств"},
		{"плуты", "плут"},
		{"плывет", "плывет"},
		{"плыви", "плыв"},
		{"плыл", "плыл"},
		{"плыла", "плыл"},
		{"плыло", "плыл"},
		{"плыть", "плыт"},
		{"плэд", "плэд"},
		{"плюгавенький", "плюгавеньк"},
		{"плюй", "плю"},
		{"плюнет", "плюнет"},
		{"плюну", "плюн"},
		{"плюнул", "плюнул"},
		{"плюнуть", "плюнут"},
		{"плюнь", "плюн"},
		{"плюс", "плюс"},
		{"плюх", "плюх"},
		{"плющами", "плющ"},
		{"плющей", "плющ"},
		{"плющом", "плющ"},
		{"плюю", "плю"},
		{"плюют", "плюют"},
		{"пляс", "пляс"},
		{"плясала", "пляса"},
		{"плясать", "пляса"},
		{"пляске", "пляск"},
		{"пляски", "пляск"},
		{"плясуна", "плясун"},
		{"плясунья", "плясун"},
		{"пляшет", "пляшет"},
		{"пляши", "пляш"},
		{"пляшут", "пляшут"},
		{"пни", "пни"},
		{"по", "по"},
		{"побагровел", "побагровел"},
		{"побаивается", "побаива"},
		{"побаивались", "побаива"},
		{"побег", "побег"},
		{"побегом", "побег"},
		{"побегу", "побег"},
		{"побегут", "побегут"},
		{"побегушках", "побегушк"},
		{"победа", "побед"},
		{"победам", "побед"},
		{"победами", "побед"},
		{"победе", "побед"},
		{"победил", "побед"},
		{"победила", "побед"},
		{"победителем", "победител"},
		{"победителя", "победител"},
		{"победить", "побед"},
		{"победно", "победн"},
		{"победный", "победн"},
		{"победой", "побед"},
		{"победоносцев", "победоносц"},
		{"победоносцевым", "победоносцев"},
		{"победу", "побед"},
		{"победы", "побед"},
		{"побежал", "побежа"},
		{"побежала", "побежа"},
		{"побежали", "побежа"},
		{"побежало", "побежа"},
		{"побеждает", "побежда"},
		{"побеждал", "побежда"},
		{"побеждать", "побежда"},
		{"побеждающею", "побежда"},
		{"побеждая", "побежд"},
		{"побежден", "побежд"},
		{"побежденный", "побежден"},
		{"побеждены", "побежд"},
		{"побежит", "побеж"},
		{"побелевшею", "побелевш"},
		{"побелевшие", "побелевш"},
		{"побелел", "побелел"},
		{"побереги", "поберег"},
		{"поберегите", "поберег"},
		{"поберегся", "поберег"},
		{"поберечь", "побереч"},
		{"поберечься", "побереч"},
		{"побеспокоили", "побеспоко"},
		{"побеспокоить", "побеспоко"},
		{"побивали", "побива"},
		{"побили", "поб"},
		{"побираться", "побира"},
		{"поблагодари", "поблагодар"},
		{"поблагодарил", "поблагодар"},
		{"поблагодарила", "поблагодар"},
		{"поблагодарить", "поблагодар"},
		{"поблагодарю", "поблагодар"},
		{"побледнев", "побледн"},
		{"побледневшая", "побледневш"},
		{"побледневшими", "побледневш"},
		{"побледнеет", "побледнеет"},
		{"побледнел", "побледнел"},
		{"побледнела", "побледнел"},
		{"побледнели", "побледнел"},
		{"побледнело", "побледнел"},
		{"побледнеют", "побледнеют"},
		{"поблек", "поблек"},
		{"поблекла", "поблекл"},
		{"поблекли", "поблекл"},
		{"поблекло", "поблекл"},
		{"поблеклые", "поблекл"},
		{"поблекнет", "поблекнет"},
		{"поблекшей", "поблекш"},
		{"поблескивает", "поблескива"},
		{"поближе", "поближ"},
		{"поблизости", "поблиз"},
		{"пободрее", "пободр"},
		{"побоев", "побо"},
		{"побожился", "побож"},
		{"побожиться", "побож"},
		{"побои", "побо"},
		{"побоится", "побо"},
		{"побой", "поб"},
		{"побоку", "побок"},
		{"поболее", "побол"},
		{"поболтать", "поболта"},
		{"побольше", "побольш"},
		{"поборемся", "побор"},
		{"поборол", "поборол"},
		{"поборюсь", "побор"},
		{"побрал", "побра"},
		{"побранят", "побран"},
		{"побрезгуй", "побрезг"},
		{"побрился", "побр"},
		{"побродив", "поброд"},
		{"побросать", "поброса"},
		{"побудившие", "побуд"},
		{"побудила", "побуд"},
		{"побудили", "побуд"},
		{"побудило", "побуд"},
		{"побуду", "побуд"},
		{"побудь", "побуд"},
		{"побудьте", "побудьт"},
		{"побуждали", "побужда"},
		{"побуждение", "побужден"},
		{"побывали", "побыва"},
		{"побывать", "побыва"},
		{"побываю", "побыва"},
		{"побыл", "поб"},
		{"побыстрей", "побыстр"},
		{"побыть", "поб"},
		{"побьет", "побьет"},
		{"повадился", "повад"},
		{"повадки", "повадк"},
		{"повадятся", "повад"},
		{"поважнее", "поважн"},
		{"повалил", "повал"},
		{"повалилась", "повал"},
		{"повалило", "повал"},
		{"повалилось", "повал"},
		{"повалился", "повал"},
		{"повалится", "повал"},
		{"повалиться", "повал"},
		{"повальной", "повальн"},
		{"поваляться", "поваля"},
		{"повар", "повар"},
		{"повара", "повар"},
		{"поварам", "повар"},
		{"поваров", "повар"},
		{"поваром", "повар"},
		{"повару", "повар"},
		{"поведать", "поведа"},
		{"поведение", "поведен"},
		{"поведении", "поведен"},
		{"поведению", "поведен"},
		{"поведения", "поведен"},
		{"поведет", "поведет"},
		{"поведешь", "поведеш"},
		{"поведу", "повед"},
		{"поведут", "поведут"},
		{"повез", "повез"},
		{"повезет", "повезет"},
		{"повезли", "повезл"},
		{"повезут", "повезут"},
		{"повел", "повел"},
		{"повела", "повел"},
		{"повелевала", "повелева"},
		{"повелел", "повелел"},
		{"повеление", "повелен"},
		{"повелению", "повелен"},
		{"повелено", "повел"},
		{"повели", "повел"},
		{"повелитель", "повелител"},
		{"повелительно", "повелительн"},
		{"повелительным", "повелительн"},
		{"повело", "повел"},
		{"повелось", "повел"},
		{"повергает", "поверга"},
		{"повергался", "поверга"},
		{"поверенная", "поверен"},
		{"поверенного", "поверен"},
		{"поверенному", "поверен"},
		{"поверенный", "поверен"},
		{"поверенным", "поверен"},
		{"поверженный", "повержен"},
		{"поверженным", "повержен"},
		{"поверив", "повер"},
		{"поверил", "повер"},
		{"поверила", "повер"},
		{"поверили", "повер"},
		{"поверим", "повер"},
		{"поверит", "повер"},
		{"поверите", "повер"},
		{"поверить", "повер"},
		{"поверишь", "повер"},
		{"поверка", "поверк"},
		{"поверкою", "поверк"},
		{"повернись", "поверн"},
		{"поверну", "поверн"},
		{"повернувшийся", "повернувш"},
		{"повернувшись", "повернувш"},
		{"повернул", "повернул"},
		{"повернула", "повернул"},
		{"повернулась", "повернул"},
		{"повернули", "повернул"},
		{"повернулся", "повернул"},
		{"повернуть", "повернут"},
		{"повернуться", "повернут"},
		{"поверстный", "поверстн"},
		{"повертел", "повертел"},
		{"повертеться", "повертет"},
		{"повертит", "поверт"},
		{"повертывая", "повертыв"},
		{"поверх", "поверх"},
		{"поверхностно", "поверхностн"},
		{"поверхностным", "поверхностн"},
		{"поверхность", "поверхн"},
		{"поверь", "повер"},
		{"поверьте", "поверьт"},
		{"поверю", "повер"},
		{"поверяет", "поверя"},
		{"поверял", "поверя"},
		{"поверяла", "поверя"},
		{"поверят", "повер"},
		{"поверять", "поверя"},
		{"повеселевшим", "повеселевш"},
		{"повеселее", "повесел"},
		{"повеселеет", "повеселеет"},
		{"повеселел", "повеселел"},
		{"повеселела", "повеселел"},
		{"повеселели", "повеселел"},
		{"повеселело", "повеселел"},
		{"повеселись", "повесел"},
		{"повеселиться", "повесел"},
		{"повеселясь", "повесел"},
		{"повесив", "повес"},
		{"повесившиеся", "повес"},
		{"повесил", "повес"},
		{"повесить", "повес"},
		{"повеситься", "повес"},
		{"повествовала", "повествова"},
		{"повествований", "повествован"},
		{"повествовать", "повествова"},
		{"повествует", "повеств"},
		{"повести", "повест"},
		{"повестка", "повестк"},
		{"повестке", "повестк"},
		{"повестки", "повестк"},
		{"повестку", "повестк"},
		{"повесть", "повест"},
		{"повестям", "повест"},
		{"повестях", "повест"},
		{"повесьте", "повесьт"},
		{"повеся", "пов"},
		{"повечеру", "повечер"},
		{"повешенные", "повешен"},
		{"повеял", "повея"},
		{"повеяло", "повея"},
		{"повивальными", "повивальн"},
		{"повивальных", "повивальн"},
		{"повидавшись", "повида"},
		{"повидать", "повида"},
		{"повидаться", "повида"},
		{"повилика", "повилик"},
		{"повинился", "повин"},
		{"повинною", "повин"},
		{"повинных", "повин"},
		{"повиновались", "повинова"},
		{"повиновался", "повинова"},
		{"повинуйся", "повин"},
		{"повинуются", "повин"},
		{"повинуясь", "повину"},
		{"повинюсь", "повин"},
		{"повис", "повис"},
		{"повисали", "повиса"},
		{"повисла", "повисл"},
		{"повисли", "повисл"},
		{"повиснет", "повиснет"},
		{"повисшей", "повисш"},
		{"повисшими", "повисш"},
		{"повита", "повит"},
		{"повито", "повит"},
		{"повихнулся", "повихнул"},
		{"повлекли", "повлекл"},
		{"повлекло", "повлекл"},
		{"повлияло", "повлия"},
		{"повнимательнее", "повнимательн"},
		{"повод", "повод"},
		{"повода", "повод"},
		{"поводил", "повод"},
		{"поводом", "повод"},
		{"поводу", "повод"},
		{"поводя", "повод"},
		{"поворачивает", "поворачива"},
		{"поворачивается", "поворачива"},
		{"поворачивайся", "поворачива"},
		{"поворачивался", "поворачива"},
		{"поворачиваю", "поворачива"},
		{"поворачивая", "поворачив"},
		{"поворачиваясь", "поворачив"},
		{"поворот", "поворот"},
		{"поворота", "поворот"},
		{"повороте", "поворот"},
		{"поворотил", "поворот"},
		{"поворотили", "поворот"},
		{"поворотился", "поворот"},
		{"поворотить", "поворот"},
		{"поворотом", "поворот"},
		{"поворчав", "поворча"},
		{"повострее", "повостр"},
		{"повредил", "повред"},
		{"повредит", "повред"},
		{"повредить", "повред"},
		{"повредиться", "повред"},
		{"повсеместно", "повсеместн"},
		{"повсеместных", "повсеместн"},
		{"повстречался", "повстреча"},
		{"повстречаться", "повстреча"},
		{"повсюду", "повсюд"},
		{"повторение", "повторен"},
		{"повторения", "повторен"},
		{"повторенья", "повторен"},
		{"повтори", "повтор"},
		{"повторившееся", "повтор"},
		{"повторил", "повтор"},
		{"повторила", "повтор"},
		{"повторили", "повтор"},
		{"повторило", "повтор"},
		{"повторился", "повтор"},
		{"повторим", "повтор"},
		{"повторит", "повтор"},
		{"повторится", "повтор"},
		{"повторить", "повтор"},
		{"повторю", "повтор"},
		{"повторяет", "повторя"},
		{"повторяйте", "повторя"},
		{"повторял", "повторя"},
		{"повторяла", "повторя"},
		{"повторялись", "повторя"},
		{"повторялось", "повторя"},
		{"повторялся", "повторя"},
		{"повторять", "повторя"},
		{"повторяться", "повторя"},
		{"повторяю", "повторя"},
		{"повторяют", "повторя"},
		{"повторяются", "повторя"},
		{"повторяя", "повтор"},
		{"повыскочит", "повыскоч"},
		{"повысосет", "повысосет"},
		{"повыше", "повыш"},
		{"повышение", "повышен"},
		{"повяжется", "повяжет"},
		{"повязан", "повяза"},
		{"повязаны", "повяза"},
		{"повязки", "повязк"},
		{"повязкой", "повязк"},
		{"повязку", "повязк"},
		{"поганая", "поган"},
		{"поганую", "поган"},
		{"погас", "погас"},
		{"погасал", "погаса"},
		{"погасали", "погаса"},
		{"погасания", "погасан"},
		{"погаси", "погас"},
		{"погасил", "погас"},
		{"погасит", "погас"},
		{"погасла", "погасл"},
		{"погасли", "погасл"},
		{"погасло", "погасл"},
		{"погасшие", "погасш"},
		{"погиб", "погиб"},
		{"погибает", "погиба"},
		{"погибайте", "погиба"},
		{"погибал", "погиба"},
		{"погибало", "погиба"},
		{"погибают", "погиба"},
		{"погибающий", "погиба"},
		{"погибели", "погибел"},
		{"погибельные", "погибельн"},
		{"погибелью", "погибел"},
		{"погибла", "погибл"},
		{"погибли", "погибл"},
		{"погибло", "погибл"},
		{"погибнет", "погибнет"},
		{"погибну", "погибн"},
		{"погибнуть", "погибнут"},
		{"погибший", "погибш"},
		{"погибшим", "погибш"},
		{"погибшими", "погибш"},
		{"погладил", "поглад"},
		{"поглотила", "поглот"},
		{"поглотит", "поглот"},
		{"поглощаются", "поглоща"},
		{"поглощен", "поглощ"},
		{"поглощенные", "поглощен"},
		{"поглубже", "поглубж"},
		{"поглумился", "поглум"},
		{"поглупее", "поглуп"},
		{"поглупел", "поглупел"},
		{"поглупеть", "поглупет"},
		{"поглядев", "погляд"},
		{"поглядел", "поглядел"},
		{"поглядела", "поглядел"},
		{"поглядели", "поглядел"},
		{"поглядело", "поглядел"},
		{"поглядеть", "поглядет"},
		{"погляди", "погляд"},
		{"поглядим", "погляд"},
		{"поглядит", "погляд"},
		{"поглядите", "погляд"},
		{"поглядишь", "погляд"},
		{"поглядывает", "поглядыва"},
		{"поглядывал", "поглядыва"},
		{"поглядывала", "поглядыва"},
		{"поглядывали", "поглядыва"},
		{"поглядывая", "поглядыв"},
		{"погляжу", "погляж"},
		{"погнал", "погна"},
		{"погнался", "погна"},
		{"погнушались", "погнуша"},
		{"поговаривали", "поговарива"},
		{"поговаривать", "поговарива"},
		{"поговаривают", "поговарива"},
		{"поговори", "поговор"},
		{"поговорила", "поговор"},
		{"поговорили", "поговор"},
		{"поговорим", "поговор"},
		{"поговорит", "поговор"},
		{"поговорить", "поговор"},
		{"поговоришь", "поговор"},
		{"поговорю", "поговор"},
		{"погогочут", "погогочут"},
		{"погода", "погод"},
		{"погоде", "погод"},
		{"погоди", "погод"},
		{"погодили", "погод"},
		{"погодите", "погод"},
		{"погодой", "погод"},
		{"погоду", "погод"},
		{"погоды", "погод"},
		{"погодя", "погод"},
		{"погожу", "погож"},
		{"погони", "погон"},
		{"погонит", "погон"},
		{"погоняя", "погон"},
		{"погордиться", "погорд"},
		{"погорел", "погорел"},
		{"погорельцам", "погорельц"},
		{"погорельцев", "погорельц"},
		{"погорячатся", "погорячат"},
		{"погорячился", "погоряч"},
		{"погорячится", "погоряч"},
		{"погостить", "погост"},
		{"погреб", "погреб"},
		{"погреба", "погреб"},
		{"погребаете", "погреба"},
		{"погребался", "погреба"},
		{"погребальный", "погребальн"},
		{"погребальными", "погребальн"},
		{"погребе", "погреб"},
		{"погребена", "погреб"},
		{"погребла", "погребл"},
		{"погребло", "погребл"},
		{"погрозив", "погроз"},
		{"погрозил", "погроз"},
		{"погрозила", "погроз"},
		{"погрознее", "погрозн"},
		{"погромче", "погромч"},
		{"погружается", "погружа"},
		{"погружалась", "погружа"},
		{"погружались", "погружа"},
		{"погружался", "погружа"},
		{"погружаюсь", "погружа"},
		{"погружаясь", "погруж"},
		{"погружен", "погруж"},
		{"погружена", "погруж"},
		{"погруженная", "погружен"},
		{"погруженный", "погружен"},
		{"погрузил", "погруз"},
		{"погрузилась", "погруз"},
		{"погрузились", "погруз"},
		{"погрузилось", "погруз"},
		{"погрузился", "погруз"},
		{"погрузит", "погруз"},
		{"погрузится", "погруз"},
		{"погрузиться", "погруз"},
		{"погрузишь", "погруз"},
		{"погрузясь", "погруз"},
		{"погрустив", "погруст"},
		{"погрустим", "погруст"},
		{"погубил", "погуб"},
		{"погубило", "погуб"},
		{"погубит", "погуб"},
		{"погубить", "погуб"},
		{"погублю", "погубл"},
		{"погуляет", "погуля"},
		{"погуляй", "погуля"},
		{"погуляйте", "погуля"},
		{"погулял", "погуля"},
		{"погулять", "погуля"},
		{"погуще", "погущ"},
		{"под", "под"},
		{"подавай", "подава"},
		{"подавал", "подава"},
		{"подавала", "подава"},
		{"подавали", "подава"},
		{"подавался", "подава"},
		{"подавать", "подава"},
		{"подавая", "подав"},
		{"подавила", "подав"},
		{"подавить", "подав"},
		{"подавленной", "подавлен"},
		{"подавленный", "подавлен"},
		{"подавленным", "подавлен"},
		{"подавленными", "подавлен"},
		{"подавлю", "подавл"},
		{"подавлять", "подавля"},
		{"подавляются", "подавля"},
		{"подавляя", "подавл"},
		{"подавно", "подавн"},
		{"подагра", "подагр"},
		{"подагрика", "подагрик"},
		{"подагрой", "подагр"},
		{"подагру", "подагр"},
		{"подагры", "подагр"},
		{"подадим", "подад"},
		{"подадите", "подад"},
		{"подадут", "подадут"},
		{"подает", "пода"},
		{"подаешь", "пода"},
		{"подай", "пода"},
		{"подайте", "пода"},
		{"подал", "пода"},
		{"подала", "пода"},
		{"подалась", "пода"},
		{"подали", "пода"},
		{"подались", "пода"},
		{"подальше", "подальш"},
		{"подам", "под"},
		{"подан", "пода"},
		{"поданный", "пода"},
		{"подано", "пода"},
		{"поданы", "пода"},
		{"подаренное", "подарен"},
		{"подаренный", "подарен"},
		{"подари", "подар"},
		{"подарил", "подар"},
		{"подарила", "подар"},
		{"подарили", "подар"},
		{"подарите", "подар"},
		{"подарить", "подар"},
		{"подарка", "подарк"},
		{"подарки", "подарк"},
		{"подарков", "подарк"},
		{"подарок", "подарок"},
		{"подарю", "подар"},
		{"подаст", "подаст"},
		{"подати", "подат"},
		{"подать", "пода"},
		{"подаче", "подач"},
		{"подачки", "подачк"},
		{"подашь", "подаш"},
		{"подаю", "пода"},
		{"подают", "пода"},
		{"подающий", "пода"},
		{"подающим", "пода"},
		{"подаяние", "подаян"},
		{"подбегая", "подбег"},
		{"подбегут", "подбегут"},
		{"подбежал", "подбежа"},
		{"подбежала", "подбежа"},
		{"подбежали", "подбежа"},
		{"подберу", "подбер"},
		{"подбивался", "подбива"},
		{"подбирает", "подбира"},
		{"подбирала", "подбира"},
		{"подбирался", "подбира"},
		{"подбираю", "подбира"},
		{"подбирающийся", "подбира"},
		{"подбирая", "подбир"},
		{"подбитые", "подбит"},
		{"подбитыми", "подбит"},
		{"подбородка", "подбородк"},
		{"подбородком", "подбородк"},
		{"подбородок", "подбородок"},
		{"подбоченясь", "подбочен"},
		{"подбросить", "подброс"},
		{"подвал", "подва"},
		{"подвалы", "подвал"},
		{"подвальной", "подвальн"},
		{"подвальные", "подвальн"},
		{"подвальный", "подвальн"},
		{"подведет", "подведет"},
		{"подведешь", "подведеш"},
		{"подвел", "подвел"},
		{"подвела", "подвел"},
		{"подвели", "подвел"},
		{"подвергаемся", "подверга"},
		{"подвергается", "подверга"},
		{"подвергалось", "подверга"},
		{"подвергать", "подверга"},
		{"подвергаться", "подверга"},
		{"подвергаются", "подверга"},
		{"подвергнуться", "подвергнут"},
		{"подвернись", "подверн"},
		{"подвернувшемуся", "подвернувш"},
		{"подвернулась", "подвернул"},
		{"подвернулся", "подвернул"},
		{"подвести", "подвест"},
		{"подвиг", "подвиг"},
		{"подвига", "подвиг"},
		{"подвигается", "подвига"},
		{"подвигалась", "подвига"},
		{"подвигалось", "подвига"},
		{"подвигался", "подвига"},
		{"подвигах", "подвиг"},
		{"подвигая", "подвиг"},
		{"подвигаясь", "подвиг"},
		{"подвиги", "подвиг"},
		{"подвигла", "подвигл"},
		{"подвигов", "подвиг"},
		{"подвигом", "подвиг"},
		{"подвижника", "подвижник"},
		{"подвижною", "подвижн"},
		{"подвинет", "подвинет"},
		{"подвинув", "подвинув"},
		{"подвинул", "подвинул"},
		{"подвинулась", "подвинул"},
		{"подвинулись", "подвинул"},
		{"подвинулось", "подвинул"},
		{"подвинулся", "подвинул"},
		{"подвинуться", "подвинут"},
		{"подвластной", "подвластн"},
		{"подвода", "подвод"},
		{"подводах", "подвод"},
		{"подводил", "подвод"},
		{"подводит", "подвод"},
		{"подводить", "подвод"},
		{"подводя", "подвод"},
		{"подвожу", "подвож"},
		{"подворотню", "подворотн"},
		{"подвох", "подвох"},
		{"подвязанными", "подвяза"},
		{"подгадил", "подгад"},
		{"подгадить", "подгад"},
		{"подгибались", "подгиба"},
		{"подгибаться", "подгиба"},
		{"подгибая", "подгиб"},
		{"подглядели", "подглядел"},
		{"подглядывавшая", "подглядыва"},
		{"подгоняют", "подгоня"},
		{"подгорелое", "подгорел"},
		{"подгорит", "подгор"},
		{"подгорюнившись", "подгорюн"},
		{"подготовки", "подготовк"},
		{"подготовленного", "подготовлен"},
		{"подготовлено", "подготовл"},
		{"подготовлюсь", "подготовл"},
		{"подготовляет", "подготовля"},
		{"подготовлялось", "подготовля"},
		{"подгреб", "подгреб"},
		{"подгулял", "подгуля"},
		{"поддавалась", "поддава"},
		{"поддавался", "поддава"},
		{"поддадим", "поддад"},
		{"поддадитесь", "поддад"},
		{"поддаешься", "подда"},
		{"поддакивал", "поддакива"},
		{"поддакивала", "поддакива"},
		{"поддакивали", "поддакива"},
		{"поддакивают", "поддакива"},
		{"поддакивая", "поддакив"},
		{"поддакнул", "поддакнул"},
		{"поддалась", "подда"},
		{"поддали", "подда"},
		{"поддало", "подда"},
		{"поддамся", "подд"},
		{"поддаст", "поддаст"},
		{"поддаться", "подда"},
		{"поддеваете", "поддева"},
		{"поддевке", "поддевк"},
		{"поддел", "поддел"},
		{"подделаться", "поддела"},
		{"подделаю", "поддела"},
		{"подделывал", "подделыва"},
		{"подделывали", "подделыва"},
		{"подделывателей", "подделывател"},
		{"поддержания", "поддержан"},
		{"поддержать", "поддержа"},
		{"поддержи", "поддерж"},
		{"поддерживаемую", "поддержива"},
		{"поддерживает", "поддержива"},
		{"поддерживал", "поддержива"},
		{"поддерживала", "поддержива"},
		{"поддерживалась", "поддержива"},
		{"поддерживало", "поддержива"},
		{"поддерживать", "поддержива"},
		{"поддерживая", "поддержив"},
		{"поддержите", "поддерж"},
		{"поддержки", "поддержк"},
		{"поддразнивает", "поддразнива"},
		{"поддразнивал", "поддразнива"},
		{"поддразнивающие", "поддразнива"},
		{"подействовал", "подействова"},
		{"подействовала", "подействова"},
		{"подействовали", "подействова"},
		{"подействовало", "подействова"},
		{"подействовать", "подействова"},
		{"подействует", "подейств"},
		{"поделаешь", "подела"},
		{"поделится", "подел"},
		{"поделиться", "подел"},
		{"поделом", "подел"},
		{"подержанная", "подержа"},
		{"подери", "подер"},
		{"подернулась", "подернул"},
		{"подернулось", "подернул"},
		{"подернут", "подернут"},
		{"подернутое", "подернут"},
		{"подернутый", "подернут"},
		{"подернутым", "подернут"},
		{"подерутся", "подерут"},
		{"подешевле", "подешевл"},
		{"поджав", "поджа"},
		{"поджатым", "поджат"},
		{"поджег", "поджег"},
		{"поджигал", "поджига"},
		{"поджигающее", "поджига"},
		{"поджидал", "поджида"},
		{"поджидала", "поджида"},
		{"поджидали", "поджида"},
		{"поджидаю", "поджида"},
		{"поджимают", "поджима"},
		{"поджимая", "поджим"},
		{"подзадоривая", "подзадорив"},
		{"подзатыльника", "подзатыльник"},
		{"подземелье", "подземел"},
		{"подземной", "подземн"},
		{"подземный", "подземн"},
		{"подземных", "подземн"},
		{"подзывал", "подзыва"},
		{"подзывали", "подзыва"},
		{"поди", "под"},
		{"подивился", "подив"},
		{"подивитесь", "подив"},
		{"подите", "под"},
		{"подкараулил", "подкараул"},
		{"подкараульте", "подкараульт"},
		{"подкатил", "подкат"},
		{"подкинуть", "подкинут"},
		{"подкипятило", "подкипят"},
		{"подкладке", "подкладк"},
		{"подкладкой", "подкладк"},
		{"подкладку", "подкладк"},
		{"подкладывал", "подкладыва"},
		{"подкладывала", "подкладыва"},
		{"подкладывая", "подкладыв"},
		{"подков", "подк"},
		{"подколенки", "подколенк"},
		{"подколодная", "подколодн"},
		{"подкосились", "подкос"},
		{"подкрадется", "подкрадет"},
		{"подкрадутся", "подкрадут"},
		{"подкрепил", "подкреп"},
		{"подкреплена", "подкрепл"},
		{"подкрепляющим", "подкрепля"},
		{"подкупишь", "подкуп"},
		{"подлаживаясь", "подлажив"},
		{"подлая", "подл"},
		{"подле", "подл"},
		{"подлее", "подл"},
		{"подлейшего", "подл"},
		{"подлейший", "подл"},
		{"подлейшими", "подл"},
		{"подлейших", "подл"},
		{"подленькая", "подленьк"},
		{"подлец", "подлец"},
		{"подлеца", "подлец"},
		{"подлецов", "подлец"},
		{"подлецом", "подлец"},
		{"подлецу", "подлец"},
		{"подлецы", "подлец"},
		{"подливку", "подливк"},
		{"подлил", "подл"},
		{"подлиннее", "подлин"},
		{"подлинно", "подлин"},
		{"подличают", "подлича"},
		{"подло", "подл"},
		{"подлого", "подл"},
		{"подлое", "подл"},
		{"подложил", "подлож"},
		{"подлой", "подл"},
		{"подлости", "подлост"},
		{"подлость", "подлост"},
		{"подлые", "подл"},
		{"подлый", "подл"},
		{"подлым", "подл"},
		{"подлых", "подл"},
		{"подманить", "подман"},
		{"подмахнет", "подмахнет"},
		{"подмахни", "подмахн"},
		{"подмахнул", "подмахнул"},
		{"подмести", "подмест"},
		{"подметено", "подмет"},
		{"подметишь", "подмет"},
		{"подметки", "подметк"},
		{"подметок", "подметок"},
		{"подмигивать", "подмигива"},
		{"подмигивающего", "подмигива"},
		{"подмигивающий", "подмигива"},
		{"подмигивая", "подмигив"},
		{"подмигнув", "подмигнув"},
		{"подмигнул", "подмигнул"},
		{"подмонтироваться", "подмонтирова"},
		{"подмывает", "подмыва"},
		{"подмывало", "подмыва"},
		{"подмышки", "подмышк"},
		{"подмышкой", "подмышк"},
		{"поднадул", "поднадул"},
		{"поднадуть", "поднадут"},
		{"поднес", "поднес"},
		{"поднести", "поднест"},
		{"поднеся", "подн"},
		{"поднимавшийся", "поднима"},
		{"поднимает", "поднима"},
		{"поднимается", "поднима"},
		{"поднимай", "поднима"},
		{"поднимал", "поднима"},
		{"поднимала", "поднима"},
		{"поднималась", "поднима"},
		{"поднимали", "поднима"},
		{"поднимались", "поднима"},
		{"поднималось", "поднима"},
		{"поднимался", "поднима"},
		{"поднимать", "поднима"},
		{"подниматься", "поднима"},
		{"поднимаю", "поднима"},
		{"поднимают", "поднима"},
		{"поднимающихся", "поднима"},
		{"поднимая", "подним"},
		{"поднимаясь", "подним"},
		{"поднимем", "подним"},
		{"поднимет", "поднимет"},
		{"подними", "подн"},
		{"поднимут", "поднимут"},
		{"подноготную", "подноготн"},
		{"поднос", "поднос"},
		{"подноса", "поднос"},
		{"подносе", "поднос"},
		{"подносом", "поднос"},
		{"подносы", "поднос"},
		{"подносят", "поднос"},
		{"подняв", "подня"},
		{"поднявшаяся", "подня"},
		{"поднявши", "подня"},
		{"поднявшись", "подня"},
		{"поднявшую", "подня"},
		{"поднял", "подня"},
		{"подняла", "подня"},
		{"поднялась", "подня"},
		{"подняли", "подня"},
		{"поднялись", "подня"},
		{"поднялось", "подня"},
		{"поднялся", "подня"},
		{"поднят", "подн"},
		{"поднятым", "поднят"},
		{"поднять", "подня"},
		{"подняться", "подня"},
		{"подо", "под"},
		{"подобен", "подоб"},
		{"подобие", "подоб"},
		{"подобно", "подобн"},
		{"подобного", "подобн"},
		{"подобное", "подобн"},
		{"подобной", "подобн"},
		{"подобном", "подобн"},
		{"подобную", "подобн"},
		{"подобные", "подобн"},
		{"подобный", "подобн"},
		{"подобным", "подобн"},
		{"подобных", "подобн"},
		{"подобострастия", "подобостраст"},
		{"подобострастничал", "подобострастнича"},
		{"подобострастно", "подобострастн"},
		{"подобрав", "подобра"},
		{"подобрал", "подобра"},
		{"подобрала", "подобра"},
		{"подобрали", "подобра"},
		{"подобрался", "подобра"},
		{"подобраны", "подобра"},
		{"подобру", "подобр"},
		{"подогнутых", "подогнут"},
		{"подогревалась", "подогрева"},
		{"подогреть", "подогрет"},
		{"пододвинул", "пододвинул"},
		{"пододвинула", "пододвинул"},
		{"подожгла", "подожгл"},
		{"подождав", "подожда"},
		{"подождал", "подожда"},
		{"подождать", "подожда"},
		{"подождем", "подожд"},
		{"подождет", "подождет"},
		{"подождешь", "подождеш"},
		{"подожди", "подожд"},
		{"подождите", "подожд"},
		{"подожду", "подожд"},
		{"подожму", "подожм"},
		{"подозвала", "подозва"},
		{"подозревает", "подозрева"},
		{"подозреваете", "подозрева"},
		{"подозревай", "подозрева"},
		{"подозревал", "подозрева"},
		{"подозревала", "подозрева"},
		{"подозревали", "подозрева"},
		{"подозревать", "подозрева"},
		{"подозреваю", "подозрева"},
		{"подозревают", "подозрева"},
		{"подозревающего", "подозрева"},
		{"подозревая", "подозрев"},
		{"подозрение", "подозрен"},
		{"подозрением", "подозрен"},
		{"подозрении", "подозрен"},
		{"подозрений", "подозрен"},
		{"подозрению", "подозрен"},
		{"подозрения", "подозрен"},
		{"подозрениях", "подозрен"},
		{"подозрителен", "подозрител"},
		{"подозрительна", "подозрительн"},
		{"подозрительно", "подозрительн"},
		{"подозрительного", "подозрительн"},
		{"подозрительное", "подозрительн"},
		{"подозрительным", "подозрительн"},
		{"подозрительными", "подозрительн"},
		{"подозрительных", "подозрительн"},
		{"подойдет", "подойдет"},
		{"подойдешь", "подойдеш"},
		{"подойди", "подойд"},
		{"подойду", "подойд"},
		{"подойдя", "подойд"},
		{"подойти", "подойт"},
		{"подоконник", "подоконник"},
		{"подоконниках", "подоконник"},
		{"подол", "подол"},
		{"подолгу", "подолг"},
		{"подолом", "подол"},
		{"подольстилась", "подольст"},
		{"подольщаюсь", "подольща"},
		{"подонки", "подонк"},
		{"подонкихотствовать", "подонкихотствова"},
		{"подопрет", "подопрет"},
		{"подосадовал", "подосадова"},
		{"подослал", "подосла"},
		{"подоспеет", "подоспеет"},
		{"подоспела", "подоспел"},
		{"подоткнул", "подоткнул"},
		{"подох", "подох"},
		{"подошвы", "подошв"},
		{"подошедшего", "подошедш"},
		{"подошедшею", "подошедш"},
		{"подошедший", "подошедш"},
		{"подошел", "подошел"},
		{"подошла", "подошл"},
		{"подошли", "подошл"},
		{"подошло", "подошл"},
		{"подошлю", "подошл"},
		{"подпевает", "подпева"},
		{"подпевал", "подпева"},
		{"подпевать", "подпева"},
		{"подпер", "подпер"},
		{"подпереть", "подперет"},
		{"подпившая", "подп"},
		{"подпирал", "подпира"},
		{"подпираясь", "подпир"},
		{"подписал", "подписа"},
		{"подписала", "подписа"},
		{"подписан", "подписа"},
		{"подписана", "подписа"},
		{"подписано", "подписа"},
		{"подписать", "подписа"},
		{"подписи", "подпис"},
		{"подписка", "подписк"},
		{"подписки", "подписк"},
		{"подписку", "подписк"},
		{"подписывает", "подписыва"},
		{"подписывал", "подписыва"},
		{"подписывалась", "подписыва"},
		{"подписывались", "подписыва"},
		{"подписывать", "подписыва"},
		{"подписывают", "подписыва"},
		{"подписывая", "подписыв"},
		{"подпись", "подп"},
		{"подписью", "подпис"},
		{"подпишет", "подпишет"},
		{"подпишете", "подпишет"},
		{"подпишешь", "подпишеш"},
		{"подпишитесь", "подпиш"},
		{"подпишу", "подпиш"},
		{"подползает", "подполза"},
		{"подползи", "подполз"},
		{"подползла", "подползл"},
		{"подполковник", "подполковник"},
		{"подпольи", "подпол"},
		{"подпоручик", "подпоручик"},
		{"подпруга", "подпруг"},
		{"подпрыгивал", "подпрыгива"},
		{"подпрыгивать", "подпрыгива"},
		{"подпрыгивая", "подпрыгив"},
		{"подпрыгнула", "подпрыгнул"},
		{"подпускать", "подпуска"},
		{"подрагивали", "подрагива"},
		{"подражать", "подража"},
		{"подразделений", "подразделен"},
		{"подразделения", "подразделен"},
		{"подразнить", "подразн"},
		{"подразумевать", "подразумева"},
		{"подрал", "подра"},
		{"подрались", "подра"},
		{"подрался", "подра"},
		{"подрастают", "подраста"},
		{"подрезаны", "подреза"},
		{"подрезать", "подреза"},
		{"подробная", "подробн"},
		{"подробнее", "подробн"},
		{"подробничать", "подробнича"},
		{"подробно", "подробн"},
		{"подробностей", "подробн"},
		{"подробности", "подробн"},
		{"подробность", "подробн"},
		{"подробностью", "подробн"},
		{"подробностями", "подробн"},
		{"подробностях", "подробн"},
		{"подробную", "подробн"},
		{"подробные", "подробн"},
		{"подробный", "подробн"},
		{"подрос", "подрос"},
		{"подросли", "подросл"},
		{"подруг", "подруг"},
		{"подруга", "подруг"},
		{"подругам", "подруг"},
		{"подруге", "подруг"},
		{"подруги", "подруг"},
		{"подругою", "подруг"},
		{"подругу", "подруг"},
		{"подружились", "подруж"},
		{"подружится", "подруж"},
		{"подряд", "подряд"},
		{"подрядами", "подряд"},
		{"подряде", "подряд"},
		{"подрядчик", "подрядчик"},
		{"подрядчиками", "подрядчик"},
		{"подрядчиков", "подрядчик"},
		{"подряжают", "подряжа"},
		{"подсвечник", "подсвечник"},
		{"подсвечнике", "подсвечник"},
		{"подсвистывает", "подсвистыва"},
		{"подсекают", "подсека"},
		{"подсекли", "подсекл"},
		{"подсекло", "подсекл"},
		{"подсел", "подсел"},
		{"подскажи", "подскаж"},
		{"подсказанное", "подсказа"},
		{"подсказывает", "подсказыва"},
		{"подсказывала", "подсказыва"},
		{"подсказывать", "подсказыва"},
		{"подсказывая", "подсказыв"},
		{"подскакивала", "подскакива"},
		{"подскочил", "подскоч"},
		{"подскочит", "подскоч"},
		{"подслеповатыми", "подслеповат"},
		{"подслепые", "подслеп"},
		{"подслуживается", "подслужива"},
		{"подслушал", "подслуша"},
		{"подслушала", "подслуша"},
		{"подслушанные", "подслуша"},
		{"подслушивай", "подслушива"},
		{"подслушивайте", "подслушива"},
		{"подслушивал", "подслушива"},
		{"подслушивали", "подслушива"},
		{"подслушивать", "подслушива"},
		{"подсматривать", "подсматрива"},
		{"подсмеиваешься", "подсмеива"},
		{"подсмотреть", "подсмотрет"},
		{"подсобить", "подсоб"},
		{"подсочиненные", "подсочинен"},
		{"подставил", "подстав"},
		{"подставит", "подстав"},
		{"подставить", "подстав"},
		{"подставляет", "подставля"},
		{"подставлять", "подставля"},
		{"подставляя", "подставл"},
		{"подстерегает", "подстерега"},
		{"подстерегают", "подстерега"},
		{"подстерегающую", "подстерега"},
		{"подстеречь", "подстереч"},
		{"подстилая", "подстил"},
		{"подстрекательные", "подстрекательн"},
		{"подстреленный", "подстрелен"},
		{"подстрелили", "подстрел"},
		{"подстрелю", "подстрел"},
		{"подстригали", "подстрига"},
		{"подступившего", "подступ"},
		{"подступило", "подступ"},
		{"подсудимой", "подсудим"},
		{"подсудимому", "подсудим"},
		{"подсудимый", "подсудим"},
		{"подсудимых", "подсудим"},
		{"подсунул", "подсунул"},
		{"подсунули", "подсунул"},
		{"подталкивал", "подталкива"},
		{"подталкивать", "подталкива"},
		{"подтвердил", "подтверд"},
		{"подтвердила", "подтверд"},
		{"подтвердите", "подтверд"},
		{"подтвердить", "подтверд"},
		{"подтверждает", "подтвержда"},
		{"подтверждаешь", "подтвержда"},
		{"подтверждал", "подтвержда"},
		{"подтверждала", "подтвержда"},
		{"подтверждало", "подтвержда"},
		{"подтверждалось", "подтвержда"},
		{"подтверждать", "подтвержда"},
		{"подтверждаю", "подтвержда"},
		{"подтверждением", "подтвержден"},
		{"подтвержденное", "подтвержден"},
		{"подтвержу", "подтверж"},
		{"подтибрил", "подтибр"},
		{"подтыканной", "подтыка"},
		{"подтянуть", "подтянут"},
		{"подув", "подув"},
		{"подувания", "подуван"},
		{"подувая", "подув"},
		{"подул", "подул"},
		{"подумав", "подума"},
		{"подумаем", "подума"},
		{"подумает", "подума"},
		{"подумаете", "подума"},
		{"подумаешь", "подума"},
		{"подумай", "подума"},
		{"подумайте", "подума"},
		{"подумал", "подума"},
		{"подумала", "подума"},
		{"подумали", "подума"},
		{"подумалось", "подума"},
		{"подумать", "подума"},
		{"подумаю", "подума"},
		{"подумают", "подума"},
		{"подумывает", "подумыва"},
		{"подумывал", "подумыва"},
		{"подурнела", "подурнел"},
		{"подурнели", "подурнел"},
		{"подустроить", "подустро"},
		{"подушек", "подушек"},
		{"подушечку", "подушечк"},
		{"подушка", "подушк"},
		{"подушке", "подушк"},
		{"подушки", "подушк"},
		{"подушкой", "подушк"},
		{"подушку", "подушк"},
		{"подхватил", "подхват"},
		{"подхватила", "подхват"},
		{"подхватили", "подхват"},
		{"подхватило", "подхват"},
		{"подхватим", "подхват"},
		{"подхватит", "подхват"},
		{"подхватить", "подхват"},
		{"подхватывает", "подхватыва"},
		{"подхватывают", "подхватыва"},
		{"подхватывая", "подхватыв"},
		{"подхлопывать", "подхлопыва"},
		{"подходи", "подход"},
		{"подходившие", "подход"},
		{"подходил", "подход"},
		{"подходили", "подход"},
		{"подходило", "подход"},
		{"подходим", "подход"},
		{"подходит", "подход"},
		{"подходите", "подход"},
		{"подходить", "подход"},
		{"подходной", "подходн"},
		{"подходя", "подход"},
		{"подходят", "подход"},
		{"подходящего", "подходя"},
		{"подходящее", "подходя"},
		{"подходящей", "подходя"},
		{"подхожу", "подхож"},
		{"подчас", "подчас"},
		{"подчеркнула", "подчеркнул"},
		{"подчиненного", "подчинен"},
		{"подчиненные", "подчинен"},
		{"подчиненных", "подчинен"},
		{"подчинила", "подчин"},
		{"подчинились", "подчин"},
		{"подчинить", "подчин"},
		{"подчиниться", "подчин"},
		{"подчинялась", "подчиня"},
		{"подшиванье", "подшиван"},
		{"подшивать", "подшива"},
		{"подшучивают", "подшучива"},
		{"подъезда", "подъезд"},
		{"подъезжает", "подъезжа"},
		{"подъезжают", "подъезжа"},
		{"подъем", "подъ"},
		{"подъеме", "подъем"},
		{"подъемлют", "подъемлют"},
		{"подъемный", "подъемн"},
		{"подъехал", "подъеха"},
		{"подымавшеюся", "подыма"},
		{"подымавшийся", "подыма"},
		{"подымает", "подыма"},
		{"подымайся", "подыма"},
		{"подымал", "подыма"},
		{"подымалась", "подыма"},
		{"подыманием", "подыман"},
		{"подымать", "подыма"},
		{"подыматься", "подыма"},
		{"подымая", "подым"},
		{"подымаясь", "подым"},
		{"подымусь", "подым"},
		{"подыскивает", "подыскива"},
		{"подышать", "подыша"},
		{"подышит", "подыш"},
		{"подь", "под"},
		{"подьяческой", "подьяческ"},
		{"подьячий", "подьяч"},
		{"подьячим", "подьяч"},
		{"подьячих", "подьяч"},
		{"поедем", "поед"},
		{"поедемте", "поедемт"},
		{"поедет", "поедет"},
		{"поедете", "поедет"},
		{"поедешь", "поедеш"},
		{"поединке", "поединк"},
		{"поединку", "поединк"},
		{"поеду", "поед"},
		{"поедут", "поедут"},
		{"поедят", "поед"},
		{"поезд", "поезд"},
		{"поезда", "поезд"},
		{"поезде", "поезд"},
		{"поездка", "поездк"},
		{"поездке", "поездк"},
		{"поездки", "поездк"},
		{"поездку", "поездк"},
		{"поездом", "поезд"},
		{"поезду", "поезд"},
		{"поезжай", "поезжа"},
		{"поезжайте", "поезжа"},
		{"поела", "поел"},
		{"поем", "по"},
		{"поест", "поест"},
		{"поесть", "поест"},
		{"поет", "поет"},
		{"поете", "поет"},
		{"поется", "поет"},
		{"поехал", "поеха"},
		{"поехала", "поеха"},
		{"поехали", "поеха"},
		{"поехать", "поеха"},
		{"поешь", "поеш"},
		{"пожав", "пожа"},
		{"пожал", "пожа"},
		{"пожала", "пожа"},
		{"пожалев", "пожал"},
		{"пожалеет", "пожалеет"},
		{"пожалеешь", "пожалееш"},
		{"пожалей", "пожал"},
		{"пожалейте", "пожал"},
		{"пожалел", "пожалел"},
		{"пожалела", "пожалел"},
		{"пожалели", "пожалел"},
		{"пожали", "пожа"},
		{"пожаловал", "пожалова"},
		{"пожаловали", "пожалова"},
		{"пожаловался", "пожалова"},
		{"пожаловать", "пожалова"},
		{"пожаловаться", "пожалова"},
		{"пожалуй", "пожал"},
		{"пожалуйста", "пожалуйст"},
		{"пожалуйства", "пожалуйств"},
		{"пожалуйте", "пожал"},
		{"пожар", "пожар"},
		{"пожара", "пожар"},
		{"пожарах", "пожар"},
		{"пожаре", "пожар"},
		{"пожарищ", "пожарищ"},
		{"пожарная", "пожарн"},
		{"пожарные", "пожарн"},
		{"пожаров", "пожар"},
		{"пожаром", "пожар"},
		{"пожары", "пожар"},
		{"пожатие", "пожат"},
		{"пожатия", "пожат"},
		{"пожать", "пожа"},
		{"пожелав", "пожела"},
		{"пожелавшим", "пожела"},
		{"пожелаете", "пожела"},
		{"пожелал", "пожела"},
		{"пожелали", "пожела"},
		{"пожеланий", "пожелан"},
		{"пожелать", "пожела"},
		{"пожелаю", "пожела"},
		{"пожелают", "пожела"},
		{"пожелтевшую", "пожелтевш"},
		{"пожелтелая", "пожелтел"},
		{"пожелтели", "пожелтел"},
		{"пожертвовал", "пожертвова"},
		{"пожертвовала", "пожертвова"},
		{"пожертвованием", "пожертвован"},
		{"пожертвовать", "пожертвова"},
		{"пожертвует", "пожертв"},
		{"пожертвуете", "пожертвует"},
		{"пожертвуйте", "пожертв"},
		{"поживаете", "пожива"},
		{"поживаешь", "пожива"},
		{"поживее", "пожив"},
		{"поживем", "пожив"},
		{"поживешь", "поживеш"},
		{"пожилая", "пожил"},
		{"пожили", "пож"},
		{"пожилой", "пожил"},
		{"пожилые", "пожил"},
		{"пожилыми", "пожил"},
		{"пожилых", "пожил"},
		{"пожимает", "пожима"},
		{"пожимал", "пожима"},
		{"пожимала", "пожима"},
		{"пожимая", "пожим"},
		{"пожинает", "пожина"},
		{"пожирал", "пожира"},
		{"пожирали", "пожира"},
		{"пожирающая", "пожира"},
		{"пожирая", "пожир"},
		{"пожить", "пож"},
		{"пожмем", "пожм"},
		{"пожмет", "пожмет"},
		{"позабавится", "позабав"},
		{"позаботится", "позабот"},
		{"позаботься", "позабот"},
		{"позабыв", "позаб"},
		{"позабыл", "позаб"},
		{"позабытый", "позабыт"},
		{"позавидовали", "позавидова"},
		{"позавидовать", "позавидова"},
		{"позавтракал", "позавтрака"},
		{"позавтракали", "позавтрака"},
		{"позавтракать", "позавтрака"},
		{"позавтракаю", "позавтрака"},
		{"позавчера", "позавчер"},
		{"позади", "позад"},
		{"позаимствоваться", "позаимствова"},
		{"позаняться", "позаня"},
		{"позах", "поз"},
		{"позвал", "позва"},
		{"позвала", "позва"},
		{"позвали", "позва"},
		{"позваль", "позвал"},
		{"позвано", "позва"},
		{"позвать", "позва"},
		{"позволением", "позволен"},
		{"позволения", "позволен"},
		{"позволено", "позвол"},
		{"позволил", "позвол"},
		{"позволила", "позвол"},
		{"позволили", "позвол"},
		{"позволит", "позвол"},
		{"позволите", "позвол"},
		{"позволительна", "позволительн"},
		{"позволительная", "позволительн"},
		{"позволительно", "позволительн"},
		{"позволить", "позвол"},
		{"позволь", "позвол"},
		{"позвольте", "позвольт"},
		{"позволю", "позвол"},
		{"позволяет", "позволя"},
		{"позволяешь", "позволя"},
		{"позволял", "позволя"},
		{"позволят", "позвол"},
		{"позволять", "позволя"},
		{"позволяю", "позволя"},
		{"позволяют", "позволя"},
		{"позволяющей", "позволя"},
		{"позвонил", "позвон"},
		{"позвонила", "позвон"},
		{"позвякивают", "позвякива"},
		{"позднее", "поздн"},
		{"поздней", "поздн"},
		{"позднейшее", "поздн"},
		{"позднейшей", "поздн"},
		{"позднейшие", "поздн"},
		{"поздний", "поздн"},
		{"поздно", "поздн"},
		{"поздняя", "поздн"},
		{"поздоровавшись", "поздорова"},
		{"поздороваемся", "поздорова"},
		{"поздоровался", "поздорова"},
		{"поздороваться", "поздорова"},
		{"поздоровее", "поздоров"},
		{"поздорову", "поздоров"},
		{"поздравил", "поздрав"},
		{"поздравить", "поздрав"},
		{"поздравлений", "поздравлен"},
		{"поздравляем", "поздравля"},
		{"поздравляли", "поздравля"},
		{"поздравлять", "поздравля"},
		{"поздравляю", "поздравля"},
		{"поздравляют", "поздравля"},
		{"поздравь", "поздрав"},
		{"позе", "поз"},
		{"позевывая", "позевыв"},
		{"позже", "позж"},
		{"позицию", "позиц"},
		{"позлить", "позл"},
		{"познав", "позна"},
		{"познавала", "познава"},
		{"познавший", "позна"},
		{"познаем", "позна"},
		{"познай", "позна"},
		{"познакомил", "познаком"},
		{"познакомились", "познаком"},
		{"познакомился", "познаком"},
		{"познакомить", "познаком"},
		{"познакомиться", "познаком"},
		{"познакомлю", "познакомл"},
		{"познакомясь", "познаком"},
		{"познал", "позна"},
		{"познала", "позна"},
		{"познании", "познан"},
		{"познания", "познан"},
		{"познаниях", "познан"},
		{"позначительнее", "позначительн"},
		{"позовет", "позовет"},
		{"позови", "позов"},
		{"позову", "позов"},
		{"позовут", "позовут"},
		{"позолота", "позолот"},
		{"позолотой", "позолот"},
		{"позор", "позор"},
		{"позора", "позор"},
		{"позорище", "позорищ"},
		{"позорно", "позорн"},
		{"позорное", "позорн"},
		{"позорном", "позорн"},
		{"позорным", "позорн"},
		{"позором", "позор"},
		{"позу", "поз"},
		{"позы", "поз"},
		{"позыв", "поз"},
		{"позывов", "позыв"},
		{"позьми", "позьм"},
		{"пои", "по"},
		{"поиграть", "поигра"},
		{"поил", "по"},
		{"поила", "по"},
		{"поименно", "поимен"},
		{"поискав", "поиска"},
		{"поискал", "поиска"},
		{"поискать", "поиска"},
		{"поиски", "поиск"},
		{"поистине", "поистин"},
		{"поит", "по"},
		{"поить", "по"},
		{"поищи", "поищ"},
		{"поищите", "поищ"},
		{"поищу", "поищ"},
		{"пой", "по"},
		{"пойдем", "пойд"},
		{"пойдемте", "пойдемт"},
		{"пойдет", "пойдет"},
		{"пойдете", "пойдет"},
		{"пойдешь", "пойдеш"},
		{"пойду", "пойд"},
		{"пойдут", "пойдут"},
		{"поймав", "пойма"},
		{"поймает", "пойма"},
		{"поймаешь", "пойма"},
		{"поймайте", "пойма"},
		{"поймал", "пойма"},
		{"поймала", "пойма"},
		{"поймали", "пойма"},
		{"поймана", "пойма"},
		{"пойманной", "пойма"},
		{"пойманным", "пойма"},
		{"поймать", "пойма"},
		{"поймают", "пойма"},
		{"поймем", "пойм"},
		{"поймет", "поймет"},
		{"поймете", "поймет"},
		{"поймешь", "поймеш"},
		{"пойми", "пойм"},
		{"поймите", "пойм"},
		{"пойму", "пойм"},
		{"поймут", "поймут"},
		{"пойти", "пойт"},
		{"пока", "пок"},
		{"покажет", "покажет"},
		{"покажете", "покажет"},
		{"покажется", "покажет"},
		{"покажи", "покаж"},
		{"покажите", "покаж"},
		{"покажу", "покаж"},
		{"покажут", "покажут"},
		{"покажутся", "покажут"},
		{"показав", "показа"},
		{"показал", "показа"},
		{"показала", "показа"},
		{"показалась", "показа"},
		{"показали", "показа"},
		{"показались", "показа"},
		{"показалось", "показа"},
		{"показался", "показа"},
		{"показание", "показан"},
		{"показанием", "показан"},
		{"показании", "показан"},
		{"показания", "показан"},
		{"показаниями", "показан"},
		{"показанная", "показа"},
		{"показать", "показа"},
		{"показаться", "показа"},
		{"показывает", "показыва"},
		{"показываете", "показыва"},
		{"показывается", "показыва"},
		{"показывай", "показыва"},
		{"показывайся", "показыва"},
		{"показывайте", "показыва"},
		{"показывал", "показыва"},
		{"показывала", "показыва"},
		{"показывалась", "показыва"},
		{"показывали", "показыва"},
		{"показывало", "показыва"},
		{"показывался", "показыва"},
		{"показывать", "показыва"},
		{"показываю", "показыва"},
		{"показывают", "показыва"},
		{"показываются", "показыва"},
		{"показывая", "показыв"},
		{"покамест", "покамест"},
		{"покатаемся", "поката"},
		{"покатает", "поката"},
		{"покататься", "поката"},
		{"покатилась", "покат"},
		{"покатились", "покат"},
		{"покатился", "покат"},
		{"покатится", "покат"},
		{"покатости", "покат"},
		{"покатываться", "покатыва"},
		{"покатых", "покат"},
		{"покачав", "покача"},
		{"покачает", "покача"},
		{"покачал", "покача"},
		{"покачала", "покача"},
		{"покачивай", "покачива"},
		{"покачивая", "покачив"},
		{"покачнулись", "покачнул"},
		{"покачнулся", "покачнул"},
		{"покаюсь", "пока"},
		{"покаяния", "покаян"},
		{"покаяньи", "покаян"},
		{"покиванием", "покиван"},
		{"покидает", "покида"},
		{"покидаете", "покида"},
		{"покидал", "покида"},
		{"покидала", "покида"},
		{"покидали", "покида"},
		{"покидать", "покида"},
		{"покидаю", "покида"},
		{"покидая", "покид"},
		{"покинет", "покинет"},
		{"покинешь", "покинеш"},
		{"покину", "покин"},
		{"покинув", "покинув"},
		{"покинули", "покинул"},
		{"покинутой", "покинут"},
		{"покинуть", "покинут"},
		{"покладая", "поклад"},
		{"покладываться", "покладыва"},
		{"поклажа", "поклаж"},
		{"поклажу", "поклаж"},
		{"поклал", "покла"},
		{"поклон", "поклон"},
		{"поклонами", "поклон"},
		{"поклонение", "поклонен"},
		{"поклонением", "поклонен"},
		{"поклонения", "поклонен"},
		{"поклонившись", "поклон"},
		{"поклонилась", "поклон"},
		{"поклонился", "поклон"},
		{"поклонись", "поклон"},
		{"поклонится", "поклон"},
		{"поклониться", "поклон"},
		{"поклонник", "поклонник"},
		{"поклонника", "поклонник"},
		{"поклонникам", "поклонник"},
		{"поклонники", "поклонник"},
		{"поклонников", "поклонник"},
		{"поклонником", "поклонник"},
		{"поклонов", "поклон"},
		{"поклоном", "поклон"},
		{"поклоны", "поклон"},
		{"поклонюсь", "поклон"},
		{"поклоняется", "поклоня"},
		{"поклонялась", "поклоня"},
		{"поклонялся", "поклоня"},
		{"поклонясь", "поклон"},
		{"поклоняются", "поклоня"},
		{"поклоняясь", "поклон"},
		{"поклявшись", "покля"},
		{"поклялась", "покля"},
		{"поклялся", "покля"},
		{"поклянусь", "поклян"},
		{"покое", "пок"},
		{"покоем", "поко"},
		{"покоен", "поко"},
		{"покои", "поко"},
		{"покоившейся", "поко"},
		{"покоилась", "поко"},
		{"покоились", "поко"},
		{"покоит", "поко"},
		{"покоится", "поко"},
		{"покоить", "поко"},
		{"покоишься", "поко"},
		{"покой", "пок"},
		{"покойна", "покойн"},
		{"покойная", "покойн"},
		{"покойнее", "покойн"},
		{"покойник", "покойник"},
		{"покойника", "покойник"},
		{"покойнике", "покойник"},
		{"покойников", "покойник"},
		{"покойником", "покойник"},
		{"покойнику", "покойник"},
		{"покойница", "покойниц"},
		{"покойницу", "покойниц"},
		{"покойницы", "покойниц"},
		{"покойно", "покойн"},
		{"покойного", "покойн"},
		{"покойное", "покойн"},
		{"покойной", "покойн"},
		{"покойному", "покойн"},
		{"покойною", "покойн"},
		{"покойную", "покойн"},
		{"покойны", "покойн"},
		{"покойные", "покойн"},
		{"покойный", "покойн"},
		{"покойным", "покойн"},
		{"покойных", "покойн"},
		{"пококетничать", "пококетнича"},
		{"поколебавшись", "поколеба"},
		{"поколебалась", "поколеба"},
		{"поколебать", "поколеба"},
		{"поколением", "поколен"},
		{"поколений", "поколен"},
		{"поколения", "поколен"},
		{"поколениях", "поколен"},
		{"поколотил", "поколот"},
		{"поколотить", "поколот"},
		{"покомфортнее", "покомфортн"},
		{"покончен", "поконч"},
		{"поконченный", "покончен"},
		{"покончено", "поконч"},
		{"покончены", "поконч"},
		{"покончив", "поконч"},
		{"покончил", "поконч"},
		{"покончим", "поконч"},
		{"покончит", "поконч"},
		{"покончить", "поконч"},
		{"покорев", "покор"},
		{"покорению", "покорен"},
		{"покорилась", "покор"},
		{"покориться", "покор"},
		{"покормит", "покорм"},
		{"покорна", "покорн"},
		{"покорнейше", "покорн"},
		{"покорнейший", "покорн"},
		{"покорно", "покорн"},
		{"покорного", "покорн"},
		{"покорной", "покорн"},
		{"покорности", "покорн"},
		{"покорностию", "покорн"},
		{"покорность", "покорн"},
		{"покорностью", "покорн"},
		{"покорную", "покорн"},
		{"покорны", "покорн"},
		{"покорный", "покорн"},
		{"покорным", "покорн"},
		{"покорных", "покорн"},
		{"покоробило", "покороб"},
		{"покоробился", "покороб"},
		{"покороче", "покороч"},
		{"покорствуй", "покорств"},
		{"покоряется", "покоря"},
		{"покорять", "покоря"},
		{"покоряюсь", "покоря"},
		{"покосившись", "покос"},
		{"покосился", "покос"},
		{"покою", "пок"},
		{"покоюсь", "пок"},
		{"покоя", "поко"},
		{"покоятся", "поко"},
		{"покоях", "поко"},
		{"покоящимся", "покоя"},
		{"покрадут", "покрадут"},
		{"покраже", "покраж"},
		{"покражи", "покраж"},
		{"покраснев", "покрасн"},
		{"покрасневший", "покрасневш"},
		{"покраснеете", "покраснеет"},
		{"покраснел", "покраснел"},
		{"покраснела", "покраснел"},
		{"покрашенных", "покрашен"},
		{"покрепче", "покрепч"},
		{"покривившаяся", "покрив"},
		{"покривившимися", "покрив"},
		{"покрикивать", "покрикива"},
		{"покричит", "покрич"},
		{"покровителей", "покровител"},
		{"покровитель", "покровител"},
		{"покровительства", "покровительств"},
		{"покровительство", "покровительств"},
		{"покровительствовали", "покровительствова"},
		{"покровительством", "покровительств"},
		{"покровительствует", "покровительств"},
		{"покровы", "покров"},
		{"покроюсь", "покр"},
		{"покрупнее", "покрупн"},
		{"покрутившись", "покрут"},
		{"покрыв", "покр"},
		{"покрывает", "покрыва"},
		{"покрывается", "покрыва"},
		{"покрывал", "покрыва"},
		{"покрывала", "покрыва"},
		{"покрывало", "покрыва"},
		{"покрывалом", "покрывал"},
		{"покрываясь", "покрыв"},
		{"покрывший", "покр"},
		{"покрыл", "покр"},
		{"покрыла", "покр"},
		{"покрылись", "покр"},
		{"покрыта", "покрыт"},
		{"покрыто", "покрыт"},
		{"покрытый", "покрыт"},
		{"покрытыми", "покрыт"},
		{"покрытых", "покрыт"},
		{"покрыть", "покр"},
		{"покрышку", "покрышк"},
		{"покуда", "покуд"},
		{"покупает", "покупа"},
		{"покупается", "покупа"},
		{"покупал", "покупа"},
		{"покупалась", "покупа"},
		{"покупатели", "покупател"},
		{"покупать", "покупа"},
		{"покупками", "покупк"},
		{"покупке", "покупк"},
		{"покупки", "покупк"},
		{"покупкой", "покупк"},
		{"покупкою", "покупк"},
		{"покупку", "покупк"},
		{"покупные", "покупн"},
		{"покуражился", "покураж"},
		{"покурить", "покур"},
		{"покуситься", "покус"},
		{"покушает", "покуша"},
		{"покушается", "покуша"},
		{"покушался", "покуша"},
		{"покушение", "покушен"},
		{"покушений", "покушен"},
		{"покушения", "покушен"},
		{"покушениях", "покушен"},
		{"покушенья", "покушен"},
		{"пол", "пол"},
		{"пола", "пол"},
		{"полагает", "полага"},
		{"полагаете", "полага"},
		{"полагается", "полага"},
		{"полагал", "полага"},
		{"полагали", "полага"},
		{"полагать", "полага"},
		{"полагаю", "полага"},
		{"полагая", "полаг"},
		{"полакомиться", "полаком"},
		{"поласковее", "поласков"},
		{"полатве", "полатв"},
		{"полбутылки", "полбутылк"},
		{"полведра", "полведр"},
		{"полгода", "полгод"},
		{"полгорода", "полгород"},
		{"полдень", "полден"},
		{"полдневный", "полдневн"},
		{"полдня", "полдн"},
		{"полдороге", "полдорог"},
		{"полдюжины", "полдюжин"},
		{"поле", "пол"},
		{"полевых", "полев"},
		{"полегоньку", "полегоньк"},
		{"полегче", "полегч"},
		{"поледенело", "поледенел"},
		{"полежав", "полежа"},
		{"полежал", "полежа"},
		{"полежать", "полежа"},
		{"полежишь", "полеж"},
		{"полежу", "полеж"},
		{"полез", "полез"},
		{"полезай", "полеза"},
		{"полезен", "полез"},
		{"полезет", "полезет"},
		{"полезла", "полезл"},
		{"полезли", "полезл"},
		{"полезная", "полезн"},
		{"полезнее", "полезн"},
		{"полезнейшая", "полезн"},
		{"полезно", "полезн"},
		{"полезное", "полезн"},
		{"полезной", "полезн"},
		{"полезною", "полезн"},
		{"полезную", "полезн"},
		{"полезны", "полезн"},
		{"полезные", "полезн"},
		{"полезный", "полезн"},
		{"полезным", "полезн"},
		{"полей", "пол"},
		{"полем", "пол"},
		{"полен", "пол"},
		{"поленами", "полен"},
		{"поленился", "полен"},
		{"поленом", "полен"},
		{"поленька", "поленьк"},
		{"поленьке", "поленьк"},
		{"поленькой", "поленьк"},
		{"полет", "полет"},
		{"полете", "полет"},
		{"полетел", "полетел"},
		{"полетела", "полетел"},
		{"полетели", "полетел"},
		{"полетело", "полетел"},
		{"полететь", "полетет"},
		{"полетит", "полет"},
		{"полетишь", "полет"},
		{"полетов", "полет"},
		{"полетом", "полет"},
		{"полечивать", "полечива"},
		{"полечка", "полечк"},
		{"полечке", "полечк"},
		{"полечкой", "полечк"},
		{"полечку", "полечк"},
		{"полечу", "полеч"},
		{"полжизни", "полжизн"},
		{"ползает", "полза"},
		{"ползала", "полза"},
		{"ползало", "полза"},
		{"ползают", "полза"},
		{"ползет", "ползет"},
		{"ползучем", "ползуч"},
		{"ползущий", "ползущ"},
		{"ползущих", "ползущ"},
		{"ползя", "полз"},
		{"поливали", "полива"},
		{"поливающий", "полива"},
		{"полились", "пол"},
		{"полина", "полин"},
		{"полине", "полин"},
		{"полиной", "полин"},
		{"полину", "полин"},
		{"полинявшая", "полиня"},
		{"полинявшими", "полиня"},
		{"полиняли", "полиня"},
		{"полиняло", "полиня"},
		{"полинялые", "полинял"},
		{"полинялым", "полинял"},
		{"полинялых", "полинял"},
		{"полис", "полис"},
		{"политика", "политик"},
		{"политике", "политик"},
		{"политики", "политик"},
		{"политико", "политик"},
		{"политику", "политик"},
		{"политическая", "политическ"},
		{"политически", "политическ"},
		{"политические", "политическ"},
		{"политический", "политическ"},
		{"политического", "политическ"},
		{"политической", "политическ"},
		{"политическую", "политическ"},
		{"полицейские", "полицейск"},
		{"полицейский", "полицейск"},
		{"полицейскому", "полицейск"},
		{"полицианом", "полициан"},
		{"полицией", "полиц"},
		{"полиции", "полиц"},
		{"полицию", "полиц"},
		{"полиция", "полиц"},
		{"полицу", "полиц"},
		{"полишинель", "полишинел"},
		{"полк", "полк"},
		{"полками", "полк"},
		{"полкане", "полкан"},
		{"полках", "полк"},
		{"полке", "полк"},
		{"полки", "полк"},
		{"полковник", "полковник"},
		{"полковника", "полковник"},
		{"полковнику", "полковник"},
		{"полковничьем", "полковнич"},
		{"полковничья", "полковнич"},
		{"полководцем", "полководц"},
		{"полковое", "полков"},
		{"полковой", "полков"},
		{"полкомнаты", "полкомнат"},
		{"полку", "полк"},
		{"полмиллиона", "полмиллион"},
		{"полминуты", "полминут"},
		{"полна", "полн"},
		{"полная", "полн"},
		{"полнее", "полн"},
		{"полнейшего", "полн"},
		{"полнейшем", "полн"},
		{"полнеть", "полнет"},
		{"полно", "полн"},
		{"полноводной", "полноводн"},
		{"полного", "полн"},
		{"полное", "полн"},
		{"полной", "полн"},
		{"полнокровный", "полнокровн"},
		{"полном", "полн"},
		{"полностью", "полност"},
		{"полноте", "полнот"},
		{"полнотой", "полнот"},
		{"полноты", "полнот"},
		{"полночи", "полноч"},
		{"полночный", "полночн"},
		{"полночным", "полночн"},
		{"полночными", "полночн"},
		{"полночь", "полноч"},
		{"полною", "полн"},
		{"полную", "полн"},
		{"полны", "полн"},
		{"полные", "полн"},
		{"полный", "полн"},
		{"полным", "полн"},
		{"полными", "полн"},
		{"полных", "полн"},
		{"полов", "пол"},
		{"половецкий", "половецк"},
		{"половик", "половик"},
		{"половина", "половин"},
		{"половине", "половин"},
		{"половинки", "половинк"},
		{"половинку", "половинк"},
		{"половинных", "половин"},
		{"половиной", "половин"},
		{"половину", "половин"},
		{"половинчатой", "половинчат"},
		{"половины", "половин"},
		{"половой", "полов"},
		{"половому", "полов"},
		{"половую", "полов"},
		{"половчей", "половч"},
		{"половым", "полов"},
		{"полог", "полог"},
		{"пологой", "полог"},
		{"положа", "полож"},
		{"положат", "положат"},
		{"положен", "полож"},
		{"положена", "полож"},
		{"положение", "положен"},
		{"положением", "положен"},
		{"положении", "положен"},
		{"положений", "положен"},
		{"положению", "положен"},
		{"положения", "положен"},
		{"положениях", "положен"},
		{"положенного", "положен"},
		{"положенной", "положен"},
		{"положенные", "положен"},
		{"положенный", "положен"},
		{"положенными", "положен"},
		{"положено", "полож"},
		{"положи", "полож"},
		{"положив", "полож"},
		{"положил", "полож"},
		{"положила", "полож"},
		{"положили", "полож"},
		{"положим", "полож"},
		{"положись", "полож"},
		{"положит", "полож"},
		{"положительно", "положительн"},
		{"положительного", "положительн"},
		{"положительное", "положительн"},
		{"положительной", "положительн"},
		{"положительный", "положительн"},
		{"положительным", "положительн"},
		{"положительных", "положительн"},
		{"положится", "полож"},
		{"положить", "полож"},
		{"положиться", "полож"},
		{"положишь", "полож"},
		{"положу", "полож"},
		{"полой", "пол"},
		{"полок", "полок"},
		{"полом", "пол"},
		{"поломаться", "полома"},
		{"поломойная", "поломойн"},
		{"полон", "полон"},
		{"полоненная", "полонен"},
		{"полонский", "полонск"},
		{"полосами", "полос"},
		{"полосатой", "полосат"},
		{"полоскалась", "полоска"},
		{"полосками", "полоск"},
		{"полоски", "полоск"},
		{"полоской", "полоск"},
		{"полоскою", "полоск"},
		{"полоску", "полоск"},
		{"полоснет", "полоснет"},
		{"полосой", "полос"},
		{"полосою", "полос"},
		{"полости", "полост"},
		{"полость", "полост"},
		{"полосы", "полос"},
		{"полотенце", "полотенц"},
		{"полотенцем", "полотенц"},
		{"полотна", "полотн"},
		{"полотнами", "полотн"},
		{"полотне", "полотн"},
		{"полотнища", "полотнищ"},
		{"полотно", "полотн"},
		{"полоумная", "полоумн"},
		{"полоумного", "полоумн"},
		{"полоумный", "полоумн"},
		{"полоумным", "полоумн"},
		{"полощет", "полощет"},
		{"полощутся", "полощут"},
		{"полпивной", "полпивн"},
		{"полпивную", "полпивн"},
		{"полслова", "полслов"},
		{"полсотня", "полсотн"},
		{"полстакана", "полстака"},
		{"полстраницы", "полстраниц"},
		{"полсутки", "полсутк"},
		{"полтинник", "полтинник"},
		{"полтинники", "полтинник"},
		{"полтиннику", "полтинник"},
		{"полтиной", "полтин"},
		{"полтора", "полтор"},
		{"полторы", "полтор"},
		{"полу", "пол"},
		{"полубезумною", "полубезумн"},
		{"полубессмысленная", "полубессмыслен"},
		{"полубог", "полубог"},
		{"полубреде", "полубред"},
		{"полувеселый", "полувесел"},
		{"полугода", "полугод"},
		{"полугоры", "полугор"},
		{"полудворянских", "полудворянск"},
		{"полуденными", "полуден"},
		{"полудне", "полудн"},
		{"полудня", "полудн"},
		{"полудремоту", "полудремот"},
		{"полукругом", "полукруг"},
		{"полумглу", "полумгл"},
		{"полумертвый", "полумертв"},
		{"полумрак", "полумрак"},
		{"полумраке", "полумрак"},
		{"полуно", "полун"},
		{"полуночи", "полуноч"},
		{"полуночной", "полуночн"},
		{"полунощный", "полунощн"},
		{"полуотворенной", "полуотворен"},
		{"полуотворенную", "полуотворен"},
		{"полуотворенные", "полуотворен"},
		{"полуотворенный", "полуотворен"},
		{"полуоткрыты", "полуоткрыт"},
		{"полупоклоном", "полупоклон"},
		{"полупрезрение", "полупрезрен"},
		{"полупрезрением", "полупрезрен"},
		{"полупросыпаясь", "полупросып"},
		{"полупьяных", "полупьян"},
		{"полуразвалившиеся", "полуразвал"},
		{"полуразрушенные", "полуразрушен"},
		{"полуслове", "полуслов"},
		{"полусне", "полусн"},
		{"полусознанием", "полусознан"},
		{"полустрадный", "полустрадн"},
		{"полусумасшедшая", "полусумасшедш"},
		{"полусумасшедших", "полусумасшедш"},
		{"полутон", "полутон"},
		{"полуторы", "полутор"},
		{"полуформенной", "полуформен"},
		{"получаем", "получа"},
		{"получаемых", "получа"},
		{"получает", "получа"},
		{"получаете", "получа"},
		{"получается", "получа"},
		{"получаешь", "получа"},
		{"получал", "получа"},
		{"получала", "получа"},
		{"получало", "получа"},
		{"получалось", "получа"},
		{"получасу", "получас"},
		{"получать", "получа"},
		{"получаю", "получа"},
		{"получая", "получ"},
		{"получение", "получен"},
		{"получении", "получен"},
		{"получения", "получен"},
		{"полученное", "получен"},
		{"полученному", "получен"},
		{"полученные", "получен"},
		{"полученный", "получен"},
		{"полученными", "получен"},
		{"полученных", "получен"},
		{"получено", "получ"},
		{"получены", "получ"},
		{"получи", "получ"},
		{"получив", "получ"},
		{"получил", "получ"},
		{"получила", "получ"},
		{"получили", "получ"},
		{"получило", "получ"},
		{"получимши", "получимш"},
		{"получиновничьих", "получиновнич"},
		{"получит", "получ"},
		{"получите", "получ"},
		{"получить", "получ"},
		{"получишь", "получ"},
		{"получу", "получ"},
		{"получше", "получш"},
		{"полушепотом", "полушепот"},
		{"полуштоф", "полуштоф"},
		{"полушубком", "полушубк"},
		{"полушубок", "полушубок"},
		{"полчаса", "полчас"},
		{"полчасика", "полчасик"},
		{"полчища", "полчищ"},
		{"полшага", "полшаг"},
		{"полы", "пол"},
		{"полымем", "полым"},
		{"полымя", "полым"},
		{"полыни", "полын"},
		{"полынь", "полын"},
		{"полыхнет", "полыхнет"},
		{"польза", "польз"},
		{"пользовавшееся", "пользова"},
		{"пользовал", "пользова"},
		{"пользовалась", "пользова"},
		{"пользовалось", "пользова"},
		{"пользоваться", "пользова"},
		{"пользой", "польз"},
		{"пользу", "польз"},
		{"пользуетесь", "пользует"},
		{"пользуешься", "пользуеш"},
		{"пользуются", "польз"},
		{"пользуясь", "пользу"},
		{"пользы", "польз"},
		{"полька", "польк"},
		{"польку", "польк"},
		{"польски", "польск"},
		{"польские", "польск"},
		{"польскими", "польск"},
		{"польских", "польск"},
		{"польское", "польск"},
		{"польстилась", "польст"},
		{"польстит", "польст"},
		{"польше", "польш"},
		{"польши", "польш"},
		{"польшу", "польш"},
		{"польщено", "польщ"},
		{"полю", "пол"},
		{"полюбил", "полюб"},
		{"полюбила", "полюб"},
		{"полюбили", "полюб"},
		{"полюбим", "полюб"},
		{"полюбит", "полюб"},
		{"полюбите", "полюб"},
		{"полюбить", "полюб"},
		{"полюбишь", "полюб"},
		{"полюблю", "полюбл"},
		{"полюбопытствовал", "полюбопытствова"},
		{"полюбуйтесь", "полюб"},
		{"полюс", "полюс"},
		{"полюсах", "полюс"},
		{"поля", "пол"},
		{"поляк", "поляк"},
		{"поляки", "поляк"},
		{"поляко", "поляк"},
		{"поляков", "поляк"},
		{"полям", "пол"},
		{"полями", "пол"},
		{"поляне", "полян"},
		{"поляны", "поля"},
		{"полярный", "полярн"},
		{"полях", "пол"},
		{"полячишки", "полячишк"},
		{"полячка", "полячк"},
		{"полячков", "полячк"},
		{"полячку", "полячк"},
		{"полячок", "полячок"},
		{"помадки", "помадк"},
		{"помадой", "помад"},
		{"помаду", "помад"},
		{"помаленьку", "помаленьк"},
		{"помалу", "помал"},
		{"помахивая", "помахив"},
		{"помашет", "помашет"},
		{"помедли", "помедл"},
		{"помедлив", "помедл"},
		{"помедлить", "помедл"},
		{"помедля", "помедл"},
		{"помелом", "помел"},
		{"помене", "помен"},
		{"поменее", "помен"},
		{"поменьше", "поменьш"},
		{"поменялись", "поменя"},
		{"поменяться", "поменя"},
		{"помер", "помер"},
		{"померанцевой", "померанцев"},
		{"померанцевую", "померанцев"},
		{"померанцевые", "померанцев"},
		{"померанцы", "померанц"},
		{"помереть", "померет"},
		{"померещилось", "померещ"},
		{"померещился", "померещ"},
		{"померзло", "померзл"},
		{"померк", "померк"},
		{"померкал", "померка"},
		{"померкла", "померкл"},
		{"померкло", "померкл"},
		{"померкший", "померкш"},
		{"померла", "померл"},
		{"помертвев", "помертв"},
		{"помертвевшая", "помертвевш"},
		{"помертвевшей", "помертвевш"},
		{"помертвевшие", "помертвевш"},
		{"помертвевший", "помертвевш"},
		{"помертвевшими", "помертвевш"},
		{"помертвел", "помертвел"},
		{"помертвело", "помертвел"},
		{"померяемся", "померя"},
		{"поместившись", "помест"},
		{"поместил", "помест"},
		{"поместились", "помест"},
		{"поместился", "помест"},
		{"поместим", "помест"},
		{"поместительный", "поместительн"},
		{"поместить", "помест"},
		{"помесячно", "помесячн"},
		{"помехи", "помех"},
		{"помеченную", "помечен"},
		{"помечтаем", "помечта"},
		{"помешает", "помеша"},
		{"помешаете", "помеша"},
		{"помешал", "помеша"},
		{"помешала", "помеша"},
		{"помешалась", "помеша"},
		{"помешали", "помеша"},
		{"помешало", "помеша"},
		{"помешался", "помеша"},
		{"помешан", "помеша"},
		{"помешанная", "помеша"},
		{"помешанного", "помеша"},
		{"помешанной", "помеша"},
		{"помешанную", "помеша"},
		{"помешанные", "помеша"},
		{"помешанный", "помеша"},
		{"помешаны", "помеша"},
		{"помешательства", "помешательств"},
		{"помешательстве", "помешательств"},
		{"помешательству", "помешательств"},
		{"помешать", "помеша"},
		{"помешаю", "помеша"},
		{"помешают", "помеша"},
		{"помещается", "помеща"},
		{"помещалась", "помеща"},
		{"помещался", "помеща"},
		{"помещаются", "помеща"},
		{"помещение", "помещен"},
		{"помещений", "помещен"},
		{"помещению", "помещен"},
		{"помещения", "помещен"},
		{"помещик", "помещик"},
		{"помещика", "помещик"},
		{"помещиков", "помещик"},
		{"помещиком", "помещик"},
		{"помещику", "помещик"},
		{"помещица", "помещиц"},
		{"помещичьи", "помещич"},
		{"помещу", "помещ"},
		{"помещьчьих", "помещьч"},
		{"помилует", "помил"},
		{"помилуй", "помил"},
		{"помилуйте", "помил"},
		{"помимо", "помим"},
		{"помин", "помин"},
		{"поминай", "помина"},
		{"поминайте", "помина"},
		{"поминал", "помина"},
		{"поминала", "помина"},
		{"поминали", "помина"},
		{"поминать", "помина"},
		{"поминкам", "поминк"},
		{"поминках", "поминк"},
		{"поминки", "поминк"},
		{"поминок", "поминок"},
		{"помину", "помин"},
		{"поминутно", "поминутн"},
		{"поминутные", "поминутн"},
		{"помирает", "помира"},
		{"помирились", "помир"},
		{"помирись", "помир"},
		{"помирите", "помир"},
		{"помиритесь", "помир"},
		{"помириться", "помир"},
		{"помни", "помн"},
		{"помнил", "помн"},
		{"помнила", "помн"},
		{"помнили", "помн"},
		{"помнилось", "помн"},
		{"помним", "помн"},
		{"помнит", "помн"},
		{"помните", "помн"},
		{"помнится", "помн"},
		{"помнить", "помн"},
		{"помнишь", "помн"},
		{"помню", "помн"},
		{"помня", "помн"},
		{"помнят", "помн"},
		{"помнятся", "помн"},
		{"помнящего", "помня"},
		{"помнящей", "помня"},
		{"помнящую", "помня"},
		{"помог", "помог"},
		{"помогает", "помога"},
		{"помогаешь", "помога"},
		{"помогал", "помога"},
		{"помогала", "помога"},
		{"помогать", "помога"},
		{"помогаю", "помога"},
		{"помогают", "помога"},
		{"помогая", "помог"},
		{"помоги", "помог"},
		{"помогите", "помог"},
		{"помогла", "помогл"},
		{"помогли", "помогл"},
		{"помогло", "помогл"},
		{"помогу", "помог"},
		{"помогут", "помогут"},
		{"поможет", "поможет"},
		{"помойные", "помойн"},
		{"помойных", "помойн"},
		{"помолимся", "помол"},
		{"помолись", "помол"},
		{"помолитесь", "помол"},
		{"помолиться", "помол"},
		{"помолодела", "помолодел"},
		{"помолодело", "помолодел"},
		{"помоложе", "помолож"},
		{"помолчав", "помолча"},
		{"помолчал", "помолча"},
		{"помолчала", "помолча"},
		{"помолчали", "помолча"},
		{"помолчать", "помолча"},
		{"помолчи", "помолч"},
		{"помолчим", "помолч"},
		{"помолюсь", "помол"},
		{"поморщились", "поморщ"},
		{"поморщился", "поморщ"},
		{"помочах", "помоч"},
		{"помочи", "помоч"},
		{"помочил", "помоч"},
		{"помочь", "помоч"},
		{"помощи", "помощ"},
		{"помощию", "помощ"},
		{"помощник", "помощник"},
		{"помощнике", "помощник"},
		{"помощники", "помощник"},
		{"помощницей", "помощниц"},
		{"помощницы", "помощниц"},
		{"помощь", "помощ"},
		{"помощью", "помощ"},
		{"помоями", "помо"},
		{"помоях", "помо"},
		{"помрачался", "помрача"},
		{"помрачен", "помрач"},
		{"помрачение", "помрачен"},
		{"помрачения", "помрачен"},
		{"помрачилась", "помрач"},
		{"помрет", "помрет"},
		{"помудренее", "помудрен"},
		{"помутилось", "помут"},
		{"помутневший", "помутневш"},
		{"помучается", "помуча"},
		{"помучить", "помуч"},
		{"помыкают", "помыка"},
		{"помыслить", "помысл"},
		{"помыслишь", "помысл"},
		{"помыслов", "помысл"},
		{"помыслы", "помысл"},
		{"помышления", "помышлен"},
		{"помышленья", "помышлен"},
		{"помышлял", "помышля"},
		{"помягче", "помягч"},
		{"помянет", "помянет"},
		{"помяни", "помян"},
		{"помяните", "помян"},
		{"помянул", "помянул"},
		{"помянут", "помянут"},
		{"помянуть", "помянут"},
		{"помятый", "помят"},
		{"помять", "помя"},
		{"понаведаться", "понаведа"},
		{"понадеялась", "понадея"},
		{"понадеялся", "понадея"},
		{"понадеяться", "понадея"},
		{"понадобились", "понадоб"},
		{"понадобилось", "понадоб"},
		{"понадобился", "понадоб"},
		{"понадобится", "понадоб"},
		{"понадобиться", "понадоб"},
		{"понадоблюсь", "понадобл"},
		{"понадобятся", "понадоб"},
		{"понапрасну", "понапрасн"},
		{"понасажать", "понасажа"},
		{"понатужиться", "понатуж"},
		{"поневоле", "поневол"},
		{"понедельник", "понедельник"},
		{"понедельника", "понедельник"},
		{"понедельники", "понедельник"},
		{"понелепее", "понелеп"},
		{"понемногу", "понемног"},
		{"понемножку", "понемножк"},
		{"понес", "понес"},
		{"понесем", "понес"},
		{"понесет", "понесет"},
		{"понесется", "понесет"},
		{"понесли", "понесл"},
		{"понеслись", "понесл"},
		{"понесло", "понесл"},
		{"понесут", "понесут"},
		{"понижая", "пониж"},
		{"пониже", "пониж"},
		{"понижения", "понижен"},
		{"понизив", "пониз"},
		{"понизилась", "пониз"},
		{"поник", "поник"},
		{"поникнув", "поникнув"},
		{"понимавшая", "понима"},
		{"понимавший", "понима"},
		{"понимаем", "понима"},
		{"понимает", "понима"},
		{"понимаете", "понима"},
		{"понимаешь", "понима"},
		{"понимал", "понима"},
		{"понимала", "понима"},
		{"понимали", "понима"},
		{"понимание", "пониман"},
		{"понимании", "пониман"},
		{"пониманию", "пониман"},
		{"понимания", "пониман"},
		{"понимать", "понима"},
		{"понимаю", "понима"},
		{"понимают", "понима"},
		{"понимающею", "понима"},
		{"понимающие", "понима"},
		{"понимающим", "понима"},
		{"понимая", "поним"},
		{"понмю", "понм"},
		{"поновее", "понов"},
		{"поносил", "понос"},
		{"поношенное", "поношен"},
		{"поношенном", "поношен"},
		{"поношенные", "поношен"},
		{"поношенный", "поношен"},
		{"поношено", "понош"},
		{"понравилась", "понрав"},
		{"понравились", "понрав"},
		{"понравилось", "понрав"},
		{"понравился", "понрав"},
		{"понравится", "понрав"},
		{"понравиться", "понрав"},
		{"понудила", "понуд"},
		{"понуждай", "понужда"},
		{"понукал", "понука"},
		{"понуканий", "понукан"},
		{"понукая", "понук"},
		{"понурил", "понур"},
		{"поныне", "понын"},
		{"понюхав", "понюха"},
		{"понюхает", "понюха"},
		{"понюхай", "понюха"},
		{"понюхайте", "понюха"},
		{"понюхала", "понюха"},
		{"понюхать", "понюха"},
		{"поняв", "поня"},
		{"понявшая", "поня"},
		{"понявший", "поня"},
		{"понял", "поня"},
		{"поняла", "поня"},
		{"поняли", "поня"},
		{"понят", "пон"},
		{"понятен", "понят"},
		{"понятие", "понят"},
		{"понятием", "понят"},
		{"понятии", "понят"},
		{"понятий", "понят"},
		{"понятия", "понят"},
		{"понятиям", "понят"},
		{"понятливый", "понятлив"},
		{"понятна", "понятн"},
		{"понятная", "понятн"},
		{"понятнее", "понятн"},
		{"понятней", "понятн"},
		{"понятно", "понятн"},
		{"понятное", "понятн"},
		{"понятны", "понятн"},
		{"понятным", "понятн"},
		{"понятое", "понят"},
		{"понять", "поня"},
		{"пообедав", "пообеда"},
		{"пообедаем", "пообеда"},
		{"пообедаете", "пообеда"},
		{"пообедал", "пообеда"},
		{"пообедала", "пообеда"},
		{"пообедать", "пообеда"},
		{"пообедаю", "пообеда"},
		{"пообтерся", "пообтер"},
		{"поодаль", "поодал"},
		{"поочередно", "поочередн"},
		{"поощряли", "поощря"},
		{"поощряю", "поощря"},
		{"попав", "попа"},
		{"попавший", "попа"},
		{"попадавшиеся", "попада"},
		{"попадает", "попада"},
		{"попадаете", "попада"},
		{"попадается", "попада"},
		{"попадал", "попада"},
		{"попадались", "попада"},
		{"попадало", "попада"},
		{"попадалось", "попада"},
		{"попадать", "попада"},
		{"попадают", "попада"},
		{"попадаются", "попада"},
		{"попадая", "попад"},
		{"попадет", "попадет"},
		{"попадется", "попадет"},
		{"попадешь", "попадеш"},
		{"попадешься", "попадеш"},
		{"попал", "попа"},
		{"попала", "попа"},
		{"попалась", "попа"},
		{"попали", "попа"},
		{"попались", "попа"},
		{"попало", "попа"},
		{"попалось", "попа"},
		{"попался", "попа"},
		{"попами", "поп"},
		{"попасть", "попаст"},
		{"попахивает", "попахива"},
		{"поперек", "поперек"},
		{"попеременно", "поперемен"},
		{"поперечного", "поперечн"},
		{"поперечных", "поперечн"},
		{"попестрей", "попестр"},
		{"попечение", "попечен"},
		{"попираешь", "попира"},
		{"попираньи", "попиран"},
		{"пописываете", "пописыва"},
		{"поплавок", "поплавок"},
		{"поплакали", "поплака"},
		{"поплачет", "поплачет"},
		{"поплачут", "поплачут"},
		{"поплевывающих", "поплевыва"},
		{"поплевывая", "поплевыв"},
		{"поплелись", "поплел"},
		{"поплелся", "поплел"},
		{"поплотнее", "поплотн"},
		{"поплывут", "поплывут"},
		{"попов", "поп"},
		{"поповича", "попович"},
		{"попозже", "попозж"},
		{"попойки", "попойк"},
		{"пополам", "попол"},
		{"пополз", "пополз"},
		{"поползет", "поползет"},
		{"поползли", "поползл"},
		{"поползновения", "поползновен"},
		{"пополнела", "пополнел"},
		{"пополнит", "пополн"},
		{"пополудни", "пополудн"},
		{"попользоваться", "попользова"},
		{"попомните", "попомн"},
		{"попона", "попон"},
		{"попортились", "попорт"},
		{"попотчевать", "попотчева"},
		{"поправил", "поправ"},
		{"поправила", "поправ"},
		{"поправилась", "поправ"},
		{"поправилось", "поправ"},
		{"поправился", "поправ"},
		{"поправимое", "поправим"},
		{"поправит", "поправ"},
		{"поправить", "поправ"},
		{"поправиться", "поправ"},
		{"поправка", "поправк"},
		{"поправке", "поправк"},
		{"поправки", "поправк"},
		{"поправлю", "поправл"},
		{"поправляет", "поправля"},
		{"поправляется", "поправля"},
		{"поправлял", "поправля"},
		{"поправляют", "поправля"},
		{"поправляя", "поправл"},
		{"поправочки", "поправочк"},
		{"поправь", "поправ"},
		{"поправьте", "поправьт"},
		{"поправят", "поправ"},
		{"попрекаете", "попрека"},
		{"попрекаешь", "попрека"},
		{"попрекал", "попрека"},
		{"попрекала", "попрека"},
		{"попрекать", "попрека"},
		{"попрекнула", "попрекнул"},
		{"попреков", "попрек"},
		{"попридержать", "попридержа"},
		{"попридержу", "попридерж"},
		{"поприща", "поприщ"},
		{"поприще", "поприщ"},
		{"поприщина", "поприщин"},
		{"поприщу", "поприщ"},
		{"попробовал", "попробова"},
		{"попробовала", "попробова"},
		{"попробовать", "попробова"},
		{"попробуй", "попроб"},
		{"попробуйте", "попроб"},
		{"попробую", "попроб"},
		{"попрозорливее", "попрозорлив"},
		{"попроси", "попрос"},
		{"попросил", "попрос"},
		{"попросила", "попрос"},
		{"попросили", "попрос"},
		{"попросит", "попрос"},
		{"попросите", "попрос"},
		{"попросить", "попрос"},
		{"попроситься", "попрос"},
		{"попросишь", "попрос"},
		{"попросту", "попрост"},
		{"попросят", "попрос"},
		{"попрошу", "попрош"},
		{"попугать", "попуга"},
		{"попудриться", "попудр"},
		{"попустит", "попуст"},
		{"попутчиков", "попутчик"},
		{"попытался", "попыта"},
		{"попытать", "попыта"},
		{"попытаться", "попыта"},
		{"попытка", "попытк"},
		{"попытку", "попытк"},
		{"попыток", "попыток"},
		{"попятился", "попят"},
		{"пор", "пор"},
		{"пора", "пор"},
		{"поработает", "поработа"},
		{"поработать", "поработа"},
		{"порабощении", "порабощен"},
		{"поравнявшись", "поравня"},
		{"поравнялись", "поравня"},
		{"поравнялся", "поравня"},
		{"порадовалась", "порадова"},
		{"порадовался", "порадова"},
		{"порадуется", "порад"},
		{"поражает", "поража"},
		{"поражал", "поража"},
		{"поражало", "поража"},
		{"поражен", "пораж"},
		{"поражена", "пораж"},
		{"пораженная", "поражен"},
		{"пораженное", "поражен"},
		{"пораженному", "поражен"},
		{"пораженную", "поражен"},
		{"пораженные", "поражен"},
		{"пораженный", "поражен"},
		{"пораженным", "поражен"},
		{"поражены", "пораж"},
		{"пораженья", "поражен"},
		{"поразившая", "пораз"},
		{"поразил", "пораз"},
		{"поразила", "пораз"},
		{"поразили", "пораз"},
		{"поразило", "пораз"},
		{"поразительной", "поразительн"},
		{"поразить", "пораз"},
		{"пораньше", "пораньш"},
		{"пораспотрошили", "пораспотрош"},
		{"порастет", "порастет"},
		{"порах", "пор"},
		{"порвала", "порва"},
		{"порвался", "порва"},
		{"порванная", "порва"},
		{"порвать", "порва"},
		{"порвется", "порвет"},
		{"поре", "пор"},
		{"порешивший", "пореш"},
		{"порешил", "пореш"},
		{"порешили", "пореш"},
		{"порешите", "пореш"},
		{"порешить", "пореш"},
		{"порицал", "порица"},
		{"порицанию", "порицан"},
		{"порицать", "порица"},
		{"поровнявшись", "поровня"},
		{"поровнялся", "поровня"},
		{"порог", "порог"},
		{"порога", "порог"},
		{"порогами", "порог"},
		{"пороге", "порог"},
		{"пороги", "порог"},
		{"пород", "пород"},
		{"порода", "пород"},
		{"породе", "пород"},
		{"породить", "пород"},
		{"порождает", "порожда"},
		{"порождают", "порожда"},
		{"порожнее", "порожн"},
		{"порой", "пор"},
		{"порок", "порок"},
		{"пороках", "порок"},
		{"пороков", "порок"},
		{"пороку", "порок"},
		{"поросенка", "поросенк"},
		{"поросла", "поросл"},
		{"порослях", "поросл"},
		{"поросята", "поросят"},
		{"порох", "порох"},
		{"пороха", "порох"},
		{"пороховом", "порохов"},
		{"пороховые", "порохов"},
		{"порохом", "порох"},
		{"пороху", "порох"},
		{"порочен", "пороч"},
		{"порочна", "порочн"},
		{"порочного", "порочн"},
		{"порочной", "порочн"},
		{"порочные", "порочн"},
		{"порочный", "порочн"},
		{"порошками", "порошк"},
		{"порошком", "порошк"},
		{"порошок", "порошок"},
		{"порошочек", "порошочек"},
		{"порою", "пор"},
		{"портер", "портер"},
		{"портером", "портер"},
		{"портеру", "портер"},
		{"портила", "порт"},
		{"портили", "порт"},
		{"портит", "порт"},
		{"портить", "порт"},
		{"портмоне", "портмон"},
		{"портниха", "портних"},
		{"портнихе", "портних"},
		{"портнихи", "портних"},
		{"портного", "портн"},
		{"портной", "портн"},
		{"портному", "портн"},
		{"портным", "портн"},
		{"портными", "портн"},
		{"портрет", "портрет"},
		{"портрета", "портрет"},
		{"портретик", "портретик"},
		{"портреты", "портрет"},
		{"портфеля", "портфел"},
		{"портьер", "портьер"},
		{"пору", "пор"},
		{"порубки", "порубк"},
		{"поругались", "поруга"},
		{"поругание", "поруган"},
		{"поруганный", "поруга"},
		{"поруганье", "поруган"},
		{"поруганьи", "поруган"},
		{"порумянее", "порумян"},
		{"поручат", "поручат"},
		{"поручают", "поруча"},
		{"поручение", "поручен"},
		{"поручением", "поручен"},
		{"поручений", "поручен"},
		{"поручению", "поручен"},
		{"поручения", "поручен"},
		{"поручениями", "поручен"},
		{"порученному", "поручен"},
		{"поручено", "поруч"},
		{"поручившись", "поруч"},
		{"поручик", "поручик"},
		{"поручика", "поручик"},
		{"поручики", "поручик"},
		{"поручиков", "поручик"},
		{"поручикова", "поручиков"},
		{"поручиком", "поручик"},
		{"поручику", "поручик"},
		{"поручил", "поруч"},
		{"поручился", "поруч"},
		{"поручит", "поруч"},
		{"поручите", "поруч"},
		{"поручить", "поруч"},
		{"поручиться", "поруч"},
		{"поручу", "поруч"},
		{"порфирием", "порфир"},
		{"порфирии", "порфир"},
		{"порфирий", "порфир"},
		{"порфирию", "порфир"},
		{"порфирия", "порфир"},
		{"порхает", "порха"},
		{"порхала", "порха"},
		{"порхнула", "порхнул"},
		{"порхнули", "порхнул"},
		{"порцию", "порц"},
		{"порчи", "порч"},
		{"поры", "пор"},
		{"порыв", "пор"},
		{"порыва", "порыв"},
		{"порывается", "порыва"},
		{"порывались", "порыва"},
		{"порывалось", "порыва"},
		{"порывался", "порыва"},
		{"порывами", "порыв"},
		{"порываний", "порыван"},
		{"порываниями", "порыван"},
		{"порыванье", "порыван"},
		{"порываются", "порыва"},
		{"порыве", "порыв"},
		{"порывисто", "порывист"},
		{"порывистой", "порывист"},
		{"порывистую", "порывист"},
		{"порывистых", "порывист"},
		{"порывов", "порыв"},
		{"порывом", "порыв"},
		{"порывы", "порыв"},
		{"порылась", "пор"},
		{"порылся", "пор"},
		{"порыскать", "порыска"},
		{"порядка", "порядк"},
		{"порядке", "порядк"},
		{"порядки", "порядк"},
		{"порядков", "порядк"},
		{"порядком", "порядк"},
		{"порядку", "порядк"},
		{"порядок", "порядок"},
		{"порядочная", "порядочн"},
		{"порядочно", "порядочн"},
		{"порядочного", "порядочн"},
		{"порядочное", "порядочн"},
		{"порядочную", "порядочн"},
		{"порядочные", "порядочн"},
		{"порядочный", "порядочн"},
		{"порядочным", "порядочн"},
		{"порядочных", "порядочн"},
		{"посади", "посад"},
		{"посадив", "посад"},
		{"посадил", "посад"},
		{"посадила", "посад"},
		{"посадили", "посад"},
		{"посадит", "посад"},
		{"посадите", "посад"},
		{"посадить", "посад"},
		{"посадят", "посад"},
		{"посаженные", "посажен"},
		{"посаженный", "посажен"},
		{"посажу", "посаж"},
		{"посватал", "посвата"},
		{"посватался", "посвата"},
		{"посвататься", "посвата"},
		{"посвежее", "посвеж"},
		{"посвистывая", "посвистыв"},
		{"посвятил", "посвят"},
		{"посвятить", "посвят"},
		{"посвящается", "посвяща"},
		{"посвящал", "посвяща"},
		{"посвящают", "посвяща"},
		{"посвящена", "посвящ"},
		{"посвящением", "посвящен"},
		{"посев", "пос"},
		{"поседевшие", "поседевш"},
		{"посекут", "посекут"},
		{"поселились", "посел"},
		{"поселился", "посел"},
		{"поселить", "посел"},
		{"поселиться", "посел"},
		{"поселюсь", "посел"},
		{"поселятся", "посел"},
		{"посему", "пос"},
		{"посередине", "посередин"},
		{"посетило", "посет"},
		{"посетит", "посет"},
		{"посетителей", "посетител"},
		{"посетители", "посетител"},
		{"посетитель", "посетител"},
		{"посетителю", "посетител"},
		{"посетителя", "посетител"},
		{"посетителями", "посетител"},
		{"посетить", "посет"},
		{"посетят", "посет"},
		{"посечь", "посеч"},
		{"посещавшие", "посеща"},
		{"посещавших", "посеща"},
		{"посещаемых", "посеща"},
		{"посещайте", "посеща"},
		{"посещал", "посеща"},
		{"посещали", "посеща"},
		{"посещать", "посеща"},
		{"посещают", "посеща"},
		{"посещающие", "посеща"},
		{"посещение", "посещен"},
		{"посещением", "посещен"},
		{"посещении", "посещен"},
		{"посещения", "посещен"},
		{"посещениями", "посещен"},
		{"посеял", "посея"},
		{"посидев", "посид"},
		{"посидел", "посидел"},
		{"посидеть", "посидет"},
		{"посиди", "посид"},
		{"посидим", "посид"},
		{"посидит", "посид"},
		{"посидите", "посид"},
		{"посидишь", "посид"},
		{"посиживает", "посижива"},
		{"посижу", "посиж"},
		{"посильное", "посильн"},
		{"посильную", "посильн"},
		{"посильные", "посильн"},
		{"посимпатичнее", "посимпатичн"},
		{"посиневших", "посиневш"},
		{"поскакать", "поскака"},
		{"поскачет", "поскачет"},
		{"поскольку", "поскольк"},
		{"посконного", "поскон"},
		{"посконные", "поскон"},
		{"поскорее", "поскор"},
		{"поскорей", "поскор"},
		{"послав", "посла"},
		{"послал", "посла"},
		{"послала", "посла"},
		{"послали", "посла"},
		{"послан", "посла"},
		{"послания", "послан"},
		{"посланник", "посланник"},
		{"посланника", "посланник"},
		{"посланную", "посла"},
		{"послать", "посла"},
		{"после", "посл"},
		{"последки", "последк"},
		{"последнего", "последн"},
		{"последнее", "последн"},
		{"последней", "последн"},
		{"последнем", "последн"},
		{"последнему", "последн"},
		{"последнею", "последн"},
		{"последние", "последн"},
		{"последний", "последн"},
		{"последним", "последн"},
		{"последними", "последн"},
		{"последних", "последн"},
		{"последнюю", "последн"},
		{"последняя", "последн"},
		{"последовавшего", "последова"},
		{"последовавшее", "последова"},
		{"последовал", "последова"},
		{"последовала", "последова"},
		{"последовало", "последова"},
		{"последовательно", "последовательн"},
		{"последовательности", "последовательн"},
		{"последовать", "последова"},
		{"последствием", "последств"},
		{"последствий", "последств"},
		{"последствия", "последств"},
		{"последствиям", "последств"},
		{"последствиями", "последств"},
		{"последующее", "послед"},
		{"последующий", "послед"},
		{"последующим", "послед"},
		{"послезавтра", "послезавтр"},
		{"послеобеденного", "послеобеден"},
		{"послеобеденную", "послеобеден"},
		{"послеобеденный", "послеобеден"},
		{"пословица", "пословиц"},
		{"пословице", "пословиц"},
		{"послужила", "послуж"},
		{"послужили", "послуж"},
		{"послужило", "послуж"},
		{"послужит", "послуж"},
		{"послужить", "послуж"},
		{"послушаем", "послуша"},
		{"послушает", "послуша"},
		{"послушается", "послуша"},
		{"послушаешь", "послуша"},
		{"послушай", "послуша"},
		{"послушайте", "послуша"},
		{"послушайтесь", "послуша"},
		{"послушал", "послуша"},
		{"послушала", "послуша"},
		{"послушали", "послуша"},
		{"послушались", "послуша"},
		{"послушался", "послуша"},
		{"послушание", "послушан"},
		{"послушании", "послушан"},
		{"послушанию", "послушан"},
		{"послушать", "послуша"},
		{"послушаюсь", "послуша"},
		{"послушна", "послушн"},
		{"послушно", "послушн"},
		{"послушною", "послушн"},
		{"послушный", "послушн"},
		{"послушным", "послушн"},
		{"послушными", "послушн"},
		{"послушных", "послушн"},
		{"послышавшийся", "послыша"},
		{"послышалась", "послыша"},
		{"послышались", "послыша"},
		{"послышалось", "послыша"},
		{"послышался", "послыша"},
		{"послышатся", "послышат"},
		{"послышится", "послыш"},
		{"послышишь", "послыш"},
		{"посматривал", "посматрива"},
		{"посматривая", "посматрив"},
		{"посмеемся", "посме"},
		{"посмеет", "посмеет"},
		{"посмеется", "посмеет"},
		{"посмеивается", "посмеива"},
		{"посмеиваться", "посмеива"},
		{"посмел", "посмел"},
		{"посмели", "посмел"},
		{"посметь", "посмет"},
		{"посмешу", "посмеш"},
		{"посмеются", "посмеют"},
		{"посмеялись", "посмея"},
		{"посмеяться", "посмея"},
		{"посмирнее", "посмирн"},
		{"посмотрев", "посмотр"},
		{"посмотрел", "посмотрел"},
		{"посмотрела", "посмотрел"},
		{"посмотрели", "посмотрел"},
		{"посмотрелся", "посмотрел"},
		{"посмотреть", "посмотрет"},
		{"посмотри", "посмотр"},
		{"посмотрим", "посмотр"},
		{"посмотрит", "посмотр"},
		{"посмотрите", "посмотр"},
		{"посмотришь", "посмотр"},
		{"посмотрю", "посмотр"},
		{"посмотрят", "посмотр"},
		{"поснимали", "поснима"},
		{"пособию", "пособ"},
		{"пособия", "пособ"},
		{"посовестился", "посовест"},
		{"посовестится", "посовест"},
		{"посоветовал", "посоветова"},
		{"посоветовались", "посоветова"},
		{"посоветовать", "посоветова"},
		{"посоветоваться", "посоветова"},
		{"посоветуется", "посовет"},
		{"посоветуюсь", "посовет"},
		{"посолиднее", "посолидн"},
		{"посолить", "посол"},
		{"посоловелые", "посоловел"},
		{"посольства", "посольств"},
		{"посох", "посох"},
		{"поспал", "поспа"},
		{"поспала", "поспа"},
		{"поспевала", "поспева"},
		{"поспевать", "поспева"},
		{"поспевают", "поспева"},
		{"поспеет", "поспеет"},
		{"поспел", "поспел"},
		{"поспеть", "поспет"},
		{"поспешайте", "поспеша"},
		{"поспешающих", "поспеша"},
		{"поспешая", "поспеш"},
		{"поспешив", "поспеш"},
		{"поспешил", "поспеш"},
		{"поспешила", "поспеш"},
		{"поспешили", "поспеш"},
		{"поспешить", "поспеш"},
		{"поспешно", "поспешн"},
		{"поспешное", "поспешн"},
		{"поспешность", "поспешн"},
		{"поспешностью", "поспешн"},
		{"поспешные", "поспешн"},
		{"поспешу", "поспеш"},
		{"поспею", "посп"},
		{"поспи", "посп"},
		{"посплетничать", "посплетнича"},
		{"поспорил", "поспор"},
		{"посреди", "посред"},
		{"посредине", "посредин"},
		{"посредника", "посредник"},
		{"посредником", "посредник"},
		{"посредство", "посредств"},
		{"посредством", "посредств"},
		{"поссорились", "поссор"},
		{"поссорился", "поссор"},
		{"поссорить", "поссор"},
		{"поста", "пост"},
		{"поставив", "постав"},
		{"поставил", "постав"},
		{"поставила", "постав"},
		{"поставили", "постав"},
		{"поставило", "постав"},
		{"поставим", "постав"},
		{"поставит", "постав"},
		{"поставить", "постав"},
		{"поставлен", "поставл"},
		{"поставленная", "поставлен"},
		{"поставленное", "поставлен"},
		{"поставленный", "поставлен"},
		{"поставляла", "поставля"},
		{"поставцами", "поставц"},
		{"поставь", "постав"},
		{"поставьте", "поставьт"},
		{"поставят", "постав"},
		{"постараетесь", "постара"},
		{"постарается", "постара"},
		{"постаралась", "постара"},
		{"постарались", "постара"},
		{"постарался", "постара"},
		{"постараться", "постара"},
		{"постараюсь", "постара"},
		{"постаревшем", "постаревш"},
		{"постарел", "постарел"},
		{"постарела", "постарел"},
		{"постарели", "постарел"},
		{"постареть", "постарет"},
		{"постарше", "постарш"},
		{"постели", "постел"},
		{"постель", "постел"},
		{"постельке", "постельк"},
		{"постельку", "постельк"},
		{"постельное", "постельн"},
		{"постелью", "постел"},
		{"постелям", "постел"},
		{"постепенная", "постепен"},
		{"постепенно", "постепен"},
		{"постепенного", "постепен"},
		{"постепенное", "постепен"},
		{"постепенностью", "постепен"},
		{"постиг", "постиг"},
		{"постигать", "постига"},
		{"постигнет", "постигнет"},
		{"постигнешь", "постигнеш"},
		{"постигнуть", "постигнут"},
		{"постилает", "постила"},
		{"постилая", "постил"},
		{"постичь", "постич"},
		{"постлал", "постла"},
		{"постлан", "постла"},
		{"постлано", "постла"},
		{"постлать", "постла"},
		{"постного", "постн"},
		{"постный", "постн"},
		{"постным", "постн"},
		{"постой", "пост"},
		{"постойте", "постойт"},
		{"постольку", "постольк"},
		{"постом", "пост"},
		{"посторонился", "посторон"},
		{"посторонись", "посторон"},
		{"постороннего", "посторон"},
		{"постороннее", "посторон"},
		{"посторонней", "посторон"},
		{"постороннем", "посторон"},
		{"постороннему", "посторон"},
		{"посторонние", "посторон"},
		{"посторонний", "посторон"},
		{"посторонними", "посторон"},
		{"посторонних", "посторон"},
		{"посторонннй", "посторонн"},
		{"посторонняя", "посторон"},
		{"постояв", "постоя"},
		{"постоял", "постоя"},
		{"постояла", "постоя"},
		{"постояли", "постоя"},
		{"постоялом", "постоял"},
		{"постояннее", "постоя"},
		{"постоянно", "постоя"},
		{"постоянного", "постоя"},
		{"постоянное", "постоя"},
		{"постоянной", "постоя"},
		{"постоянною", "постоя"},
		{"постоянную", "постоя"},
		{"постоянные", "постоя"},
		{"постоянный", "постоя"},
		{"постоянным", "постоя"},
		{"постоянством", "постоянств"},
		{"постоять", "постоя"},
		{"пострадавшую", "пострада"},
		{"пострадайте", "пострада"},
		{"пострадать", "пострада"},
		{"постращать", "постраща"},
		{"пострел", "пострел"},
		{"постреленок", "постреленок"},
		{"постричься", "пострич"},
		{"построек", "построек"},
		{"построен", "постро"},
		{"построение", "построен"},
		{"построения", "построен"},
		{"построже", "построж"},
		{"построив", "постро"},
		{"построили", "постро"},
		{"построит", "постро"},
		{"построить", "постро"},
		{"постройка", "постройк"},
		{"постройки", "постройк"},
		{"постройкой", "постройк"},
		{"постройку", "постройк"},
		{"посту", "пост"},
		{"постукивал", "постукива"},
		{"постукиванье", "постукиван"},
		{"постукивают", "постукива"},
		{"постукивающим", "постукива"},
		{"поступаешь", "поступа"},
		{"поступал", "поступа"},
		{"поступали", "поступа"},
		{"поступать", "поступа"},
		{"поступаю", "поступа"},
		{"поступают", "поступа"},
		{"поступи", "поступ"},
		{"поступил", "поступ"},
		{"поступила", "поступ"},
		{"поступили", "поступ"},
		{"поступило", "поступ"},
		{"поступит", "поступ"},
		{"поступите", "поступ"},
		{"поступить", "поступ"},
		{"поступка", "поступк"},
		{"поступкам", "поступк"},
		{"поступками", "поступк"},
		{"поступках", "поступк"},
		{"поступке", "поступк"},
		{"поступки", "поступк"},
		{"поступков", "поступк"},
		{"поступком", "поступк"},
		{"поступку", "поступк"},
		{"поступлю", "поступл"},
		{"поступок", "поступок"},
		{"поступь", "поступ"},
		{"поступью", "поступ"},
		{"постучав", "постуча"},
		{"постучал", "постуча"},
		{"постучались", "постуча"},
		{"постучаться", "постуча"},
		{"постучите", "постуч"},
		{"постучу", "постуч"},
		{"постыдились", "постыд"},
		{"постыдился", "постыд"},
		{"постыдится", "постыд"},
		{"постыдного", "постыдн"},
		{"постылая", "постыл"},
		{"постыли", "пост"},
		{"постылом", "постыл"},
		{"постылый", "постыл"},
		{"посуда", "посуд"},
		{"посудинкой", "посудинк"},
		{"посудинку", "посудинк"},
		{"посудите", "посуд"},
		{"посудой", "посуд"},
		{"посуду", "посуд"},
		{"посуды", "посуд"},
		{"посуетиться", "посует"},
		{"посущественнее", "посуществен"},
		{"посылаем", "посыла"},
		{"посылает", "посыла"},
		{"посылается", "посыла"},
		{"посылаешь", "посыла"},
		{"посылал", "посыла"},
		{"посылала", "посыла"},
		{"посылали", "посыла"},
		{"посылались", "посыла"},
		{"посылалось", "посыла"},
		{"посылать", "посыла"},
		{"посылают", "посыла"},
		{"посылки", "посылк"},
		{"посыльный", "посыльн"},
		{"посыпались", "посыпа"},
		{"посягал", "посяга"},
		{"пот", "пот"},
		{"пота", "пот"},
		{"потаенною", "потаен"},
		{"потанчиков", "потанчик"},
		{"потапыча", "потапыч"},
		{"потаскали", "потаска"},
		{"потасканной", "потаска"},
		{"потаскун", "потаскун"},
		{"потаскуна", "потаскун"},
		{"поташ", "поташ"},
		{"потащил", "потащ"},
		{"потащила", "потащ"},
		{"потащили", "потащ"},
		{"потащился", "потащ"},
		{"поте", "пот"},
		{"потекла", "потекл"},
		{"потекло", "потекл"},
		{"потекут", "потекут"},
		{"потемневший", "потемневш"},
		{"потемневшую", "потемневш"},
		{"потемнее", "потемн"},
		{"потемнело", "потемнел"},
		{"потемок", "потемок"},
		{"потеплее", "потепл"},
		{"потере", "потер"},
		{"потерей", "потер"},
		{"потерею", "потер"},
		{"потери", "потер"},
		{"потерпели", "потерпел"},
		{"потерпите", "потерп"},
		{"потертом", "потерт"},
		{"потертым", "потерт"},
		{"потерь", "потер"},
		{"потеря", "потер"},
		{"потеряв", "потеря"},
		{"потерявший", "потеря"},
		{"потерявшийся", "потеря"},
		{"потерявшись", "потеря"},
		{"потеряет", "потеря"},
		{"потеряется", "потеря"},
		{"потеряешь", "потеря"},
		{"потерял", "потеря"},
		{"потеряла", "потеря"},
		{"потерялась", "потеря"},
		{"потеряли", "потеря"},
		{"потерялись", "потеря"},
		{"потерялся", "потеря"},
		{"потерян", "потеря"},
		{"потеряна", "потеря"},
		{"потерянная", "потеря"},
		{"потерянно", "потеря"},
		{"потерянный", "потеря"},
		{"потеряно", "потеря"},
		{"потеряны", "потеря"},
		{"потерять", "потеря"},
		{"потеснее", "потесн"},
		{"потехи", "потех"},
		{"потешаясь", "потеш"},
		{"потешится", "потеш"},
		{"потешить", "потеш"},
		{"потешником", "потешник"},
		{"потешно", "потешн"},
		{"потирает", "потира"},
		{"потирал", "потира"},
		{"потирания", "потиран"},
		{"потирая", "потир"},
		{"потихоньку", "потихоньк"},
		{"потише", "потиш"},
		{"поток", "поток"},
		{"потока", "поток"},
		{"потоками", "поток"},
		{"потоком", "поток"},
		{"потолка", "потолк"},
		{"потолки", "потолк"},
		{"потолковее", "потолков"},
		{"потолку", "потолк"},
		{"потолкуем", "потолку"},
		{"потолок", "потолок"},
		{"потолочь", "потолоч"},
		{"потолстеет", "потолстеет"},
		{"потом", "пот"},
		{"потомили", "потом"},
		{"потомки", "потомк"},
		{"потомков", "потомк"},
		{"потомок", "потомок"},
		{"потомству", "потомств"},
		{"потому", "пот"},
		{"потонет", "потонет"},
		{"потонуло", "потонул"},
		{"потонут", "потонут"},
		{"потонуть", "потонут"},
		{"поторапливайтесь", "потораплива"},
		{"поторговалась", "поторгова"},
		{"поторопился", "потороп"},
		{"потосковала", "потоскова"},
		{"потратить", "потрат"},
		{"потрафил", "потраф"},
		{"потрачено", "потрач"},
		{"потребления", "потреблен"},
		{"потребности", "потребн"},
		{"потребность", "потребн"},
		{"потребностью", "потребн"},
		{"потребностями", "потребн"},
		{"потребностях", "потребн"},
		{"потребовал", "потребова"},
		{"потребовала", "потребова"},
		{"потребовали", "потребова"},
		{"потребовалось", "потребова"},
		{"потребовать", "потребова"},
		{"потребу", "потреб"},
		{"потребует", "потреб"},
		{"потребуется", "потреб"},
		{"потребуешь", "потребуеш"},
		{"потребую", "потреб"},
		{"потревожить", "потревож"},
		{"потрепал", "потрепа"},
		{"потрепала", "потрепа"},
		{"потрепанным", "потрепа"},
		{"потрепать", "потрепа"},
		{"потрепещут", "потрепещут"},
		{"потрет", "потрет"},
		{"потрогивая", "потрогив"},
		{"потроха", "потрох"},
		{"потрохами", "потрох"},
		{"потрудился", "потруд"},
		{"потрудитесь", "потруд"},
		{"потрут", "потрут"},
		{"потряс", "потряс"},
		{"потрясаемый", "потряса"},
		{"потрясает", "потряса"},
		{"потрясающие", "потряса"},
		{"потрясающий", "потряса"},
		{"потрясая", "потряс"},
		{"потрясен", "потряс"},
		{"потрясений", "потрясен"},
		{"потрясенный", "потрясен"},
		{"потрясла", "потрясл"},
		{"потрясло", "потрясл"},
		{"поту", "пот"},
		{"потуг", "потуг"},
		{"потупив", "потуп"},
		{"потупившись", "потуп"},
		{"потупил", "потуп"},
		{"потупила", "потуп"},
		{"потупилась", "потуп"},
		{"потупленные", "потуплен"},
		{"потупляет", "потупля"},
		{"потупляясь", "потупл"},
		{"потупя", "потуп"},
		{"потупясь", "потуп"},
		{"потускли", "потускл"},
		{"потускнели", "потускнел"},
		{"потух", "потух"},
		{"потухал", "потуха"},
		{"потухший", "потухш"},
		{"потухшим", "потухш"},
		{"потухших", "потухш"},
		{"потушая", "потуш"},
		{"потушены", "потуш"},
		{"потуши", "потуш"},
		{"потчеванье", "потчеван"},
		{"потчеванья", "потчеван"},
		{"потчевать", "потчева"},
		{"потчую", "потч"},
		{"потягивается", "потягива"},
		{"потягиваться", "потягива"},
		{"потягиваясь", "потягив"},
		{"потянет", "потянет"},
		{"потяни", "потян"},
		{"потянул", "потянул"},
		{"потянулась", "потянул"},
		{"потянулись", "потянул"},
		{"потянуло", "потянул"},
		{"поубавил", "поубав"},
		{"поубрать", "поубра"},
		{"поудалее", "поудал"},
		{"поужинал", "поужина"},
		{"поужинать", "поужина"},
		{"поужинаю", "поужина"},
		{"поумнее", "поумн"},
		{"поумничала", "поумнича"},
		{"поутру", "поутр"},
		{"поучался", "поуча"},
		{"поучения", "поучен"},
		{"поучительно", "поучительн"},
		{"поучить", "поуч"},
		{"пофанфаронить", "пофанфарон"},
		{"пофилософствовал", "пофилософствова"},
		{"пофилософствовать", "пофилософствова"},
		{"пофилософствуем", "пофилософству"},
		{"пофилософствуешь", "пофилософствуеш"},
		{"пофорсить", "пофорс"},
		{"похаживая", "похажив"},
		{"похвал", "похва"},
		{"похвала", "похва"},
		{"похвалами", "похвал"},
		{"похвалил", "похвал"},
		{"похвалила", "похвал"},
		{"похвалить", "похвал"},
		{"похвалиться", "похвал"},
		{"похвалой", "похвал"},
		{"похвалы", "похвал"},
		{"похвальбы", "похвальб"},
		{"похвально", "похвальн"},
		{"похвальном", "похвальн"},
		{"похвальный", "похвальн"},
		{"похвастается", "похваста"},
		{"похвастать", "похваста"},
		{"похвать", "похва"},
		{"похитил", "похит"},
		{"похитителя", "похитител"},
		{"похищаю", "похища"},
		{"похищенных", "похищен"},
		{"похлопотать", "похлопота"},
		{"похлопочу", "похлопоч"},
		{"похлопывая", "похлопыв"},
		{"похмелье", "похмел"},
		{"похмельем", "похмел"},
		{"похмелья", "похмел"},
		{"поход", "поход"},
		{"похода", "поход"},
		{"походами", "поход"},
		{"походе", "поход"},
		{"походившему", "поход"},
		{"походил", "поход"},
		{"походила", "поход"},
		{"походили", "поход"},
		{"походило", "поход"},
		{"походимте", "походимт"},
		{"походит", "поход"},
		{"походите", "поход"},
		{"походить", "поход"},
		{"походка", "походк"},
		{"походкой", "походк"},
		{"походкою", "походк"},
		{"походку", "походк"},
		{"походной", "походн"},
		{"походы", "поход"},
		{"походя", "поход"},
		{"похож", "похож"},
		{"похожа", "похож"},
		{"похожая", "похож"},
		{"похождениях", "похожден"},
		{"похоже", "похож"},
		{"похожего", "похож"},
		{"похожее", "похож"},
		{"похожей", "похож"},
		{"похожем", "похож"},
		{"похожему", "похож"},
		{"похожи", "похож"},
		{"похожие", "похож"},
		{"похожий", "похож"},
		{"похожими", "похож"},
		{"похозяйничать", "похозяйнича"},
		{"похолодев", "похолод"},
		{"похолодел", "похолодел"},
		{"похолодела", "похолодел"},
		{"похолодели", "похолодел"},
		{"похолодело", "похолодел"},
		{"похорон", "похорон"},
		{"похоронами", "похорон"},
		{"похоронах", "похорон"},
		{"похоронен", "похорон"},
		{"похоронены", "похорон"},
		{"похоронив", "похорон"},
		{"похоронил", "похорон"},
		{"похоронила", "похорон"},
		{"похоронили", "похорон"},
		{"похоронить", "похорон"},
		{"похоронная", "похорон"},
		{"похоронных", "похорон"},
		{"похороны", "похорон"},
		{"похороню", "похорон"},
		{"похоронят", "похорон"},
		{"похорошела", "похорошел"},
		{"похоти", "похот"},
		{"похотливой", "похотлив"},
		{"похудевшая", "похудевш"},
		{"похудевшее", "похудевш"},
		{"похудеешь", "похудееш"},
		{"похудел", "похудел"},
		{"похудела", "похудел"},
		{"похудели", "похудел"},
		{"похудело", "похудел"},
		{"похуже", "похуж"},
		{"поцеловав", "поцелова"},
		{"поцеловал", "поцелова"},
		{"поцеловала", "поцелова"},
		{"поцеловать", "поцелова"},
		{"поцелуе", "поцелу"},
		{"поцелуев", "поцелу"},
		{"поцелуем", "поцелу"},
		{"поцелуемся", "поцелу"},
		{"поцелует", "поцел"},
		{"поцелуи", "поцелу"},
		{"поцелуй", "поцел"},
		{"поцелуйте", "поцел"},
		{"поцелую", "поцел"},
		{"поцелуя", "поцелу"},
		{"поцелуями", "поцелу"},
		{"почавкав", "почавка"},
		{"початая", "почат"},
		{"почаще", "почащ"},
		{"почва", "почв"},
		{"почве", "почв"},
		{"почву", "почв"},
		{"почвы", "почв"},
		{"почел", "почел"},
		{"почем", "поч"},
		{"почему", "поч"},
		{"почерк", "почерк"},
		{"почерневшее", "почерневш"},
		{"почернели", "почернел"},
		{"почернелых", "почернел"},
		{"почерпать", "почерпа"},
		{"почерпнешь", "почерпнеш"},
		{"почерпнутые", "почерпнут"},
		{"почерпнуть", "почерпнут"},
		{"почесал", "почеса"},
		{"почесть", "почест"},
		{"почестью", "почест"},
		{"почесывает", "почесыва"},
		{"почесывая", "почесыв"},
		{"почетно", "почетн"},
		{"почетном", "почетн"},
		{"почетным", "почетн"},
		{"почетных", "почетн"},
		{"почетом", "почет"},
		{"почивает", "почива"},
		{"почивал", "почива"},
		{"почивали", "почива"},
		{"почивать", "почива"},
		{"почивают", "почива"},
		{"почиет", "почиет"},
		{"почил", "поч"},
		{"починена", "почин"},
		{"починивала", "починива"},
		{"починили", "почин"},
		{"починить", "почин"},
		{"починки", "починк"},
		{"починкова", "починков"},
		{"починку", "починк"},
		{"починю", "почин"},
		{"почистить", "почист"},
		{"почитает", "почита"},
		{"почитаешь", "почита"},
		{"почитай", "почита"},
		{"почитал", "почита"},
		{"почитали", "почита"},
		{"почитатели", "почитател"},
		{"почитаю", "почита"},
		{"почитают", "почита"},
		{"почище", "почищ"},
		{"почищу", "почищ"},
		{"почиющий", "почиющ"},
		{"почли", "почл"},
		{"почмокавшись", "почмока"},
		{"почт", "почт"},
		{"почта", "почт"},
		{"почтальона", "почтальон"},
		{"почтальону", "почтальон"},
		{"почтамте", "почтамт"},
		{"почте", "почт"},
		{"почтение", "почтен"},
		{"почтении", "почтен"},
		{"почтенная", "почтен"},
		{"почтеннейшая", "почтен"},
		{"почтеннейшие", "почтен"},
		{"почтеннейший", "почтен"},
		{"почтеннейшим", "почтен"},
		{"почтенного", "почтен"},
		{"почтенное", "почтен"},
		{"почтенном", "почтен"},
		{"почтенному", "почтен"},
		{"почтенные", "почтен"},
		{"почтенный", "почтен"},
		{"почтенным", "почтен"},
		{"почти", "почт"},
		{"почтит", "почт"},
		{"почтительнейшие", "почтительн"},
		{"почтительно", "почтительн"},
		{"почтительной", "почтительн"},
		{"почтительном", "почтительн"},
		{"почтительность", "почтительн"},
		{"почтительною", "почтительн"},
		{"почтительные", "почтительн"},
		{"почтительным", "почтительн"},
		{"почтить", "почт"},
		{"почтмейстером", "почтмейстер"},
		{"почто", "почт"},
		{"почтовой", "почтов"},
		{"почтовом", "почтов"},
		{"почтовые", "почтов"},
		{"почтовый", "почтов"},
		{"почтовым", "почтов"},
		{"почтовых", "почтов"},
		{"почтой", "почт"},
		{"почту", "почт"},
		{"почты", "почт"},
		{"почувствовав", "почувствова"},
		{"почувствовал", "почувствова"},
		{"почувствовала", "почувствова"},
		{"почувствовать", "почувствова"},
		{"почувствует", "почувств"},
		{"почудилась", "почуд"},
		{"почудились", "почуд"},
		{"почудилось", "почуд"},
		{"почудится", "почуд"},
		{"почуя", "почу"},
		{"пошаливают", "пошалива"},
		{"пошарив", "пошар"},
		{"пошарил", "пошар"},
		{"пошатнется", "пошатнет"},
		{"пошатнулось", "пошатнул"},
		{"пошатывается", "пошатыва"},
		{"пошатываясь", "пошатыв"},
		{"пошевелив", "пошевел"},
		{"пошевеливаться", "пошевелива"},
		{"пошевелил", "пошевел"},
		{"пошевелила", "пошевел"},
		{"пошевелилась", "пошевел"},
		{"пошевелились", "пошевел"},
		{"пошевелилось", "пошевел"},
		{"пошевелился", "пошевел"},
		{"пошевелится", "пошевел"},
		{"пошевелить", "пошевел"},
		{"пошевелиться", "пошевел"},
		{"пошел", "пошел"},
		{"пошептал", "пошепта"},
		{"пошепчут", "пошепчут"},
		{"пошла", "пошл"},
		{"пошлейшая", "пошл"},
		{"пошлейшую", "пошл"},
		{"пошлем", "пошл"},
		{"пошленький", "пошленьк"},
		{"пошленького", "пошленьк"},
		{"пошленькой", "пошленьк"},
		{"пошлет", "пошлет"},
		{"пошлешь", "пошлеш"},
		{"пошли", "пошл"},
		{"пошлите", "пошл"},
		{"пошло", "пошл"},
		{"пошлое", "пошл"},
		{"пошлой", "пошл"},
		{"пошлом", "пошл"},
		{"пошлости", "пошлост"},
		{"пошлость", "пошлост"},
		{"пошлостью", "пошлост"},
		{"пошлые", "пошл"},
		{"пошлый", "пошл"},
		{"пошлых", "пошл"},
		{"пошлю", "пошл"},
		{"пошляками", "пошляк"},
		{"пошляков", "пошляк"},
		{"пошляком", "пошляк"},
		{"пошляческая", "пошляческ"},
		{"пошто", "пошт"},
		{"пошты", "пошт"},
		{"пошутил", "пошут"},
		{"пошутила", "пошут"},
		{"пошутили", "пошут"},
		{"пощаде", "пощад"},
		{"пощадите", "пощад"},
		{"пощадить", "пощад"},
		{"пощады", "пощад"},
		{"пощекотали", "пощекота"},
		{"пощекотать", "пощекота"},
		{"пощелкав", "пощелка"},
		{"пощелкивал", "пощелкива"},
		{"пощупать", "пощупа"},
		{"поэзии", "поэз"},
		{"поэзию", "поэз"},
		{"поэзия", "поэз"},
		{"поэма", "поэм"},
		{"поэму", "поэм"},
		{"поэмы", "поэм"},
		{"поэт", "поэт"},
		{"поэта", "поэт"},
		{"поэтах", "поэт"},
		{"поэтическая", "поэтическ"},
		{"поэтически", "поэтическ"},
		{"поэтические", "поэтическ"},
		{"поэтический", "поэтическ"},
		{"поэтическим", "поэтическ"},
		{"поэтических", "поэтическ"},
		{"поэтическое", "поэтическ"},
		{"поэтической", "поэтическ"},
		{"поэтична", "поэтичн"},
		{"поэтично", "поэтичн"},
		{"поэтов", "поэт"},
		{"поэтому", "поэт"},
		{"поэты", "поэт"},
		{"пою", "по"},
		{"поют", "поют"},
		{"поющую", "поющ"},
		{"появилась", "появ"},
		{"появились", "появ"},
		{"появился", "появ"},
		{"появится", "появ"},
		{"появиться", "появ"},
		{"появление", "появлен"},
		{"появлением", "появлен"},
		{"появлении", "появлен"},
		{"появления", "появлен"},
		{"появляется", "появля"},
		{"появлялась", "появля"},
		{"появлялось", "появля"},
		{"появлялся", "появля"},
		{"появляться", "появля"},
		{"поярче", "поярч"},
		{"пояс", "пояс"},
		{"пояса", "пояс"},
		{"поясе", "пояс"},
		{"пояснее", "поясн"},
		{"пояснений", "пояснен"},
		{"пояснить", "поясн"},
		{"поясом", "пояс"},
		{"поят", "по"},
		{"прав", "прав"},
		{"права", "прав"},
		{"правам", "прав"},
		{"правами", "прав"},
		{"правах", "прав"},
		{"правая", "прав"},
		{"правда", "правд"},
		{"правде", "правд"},
		{"правдиво", "правдив"},
		{"правдой", "правд"},
		{"правдоподобия", "правдоподоб"},
		{"правдоподобнее", "правдоподобн"},
		{"правдоподобным", "правдоподобн"},
		{"правдою", "правд"},
		{"правду", "правд"},
		{"правды", "правд"},
		{"праве", "прав"},
		{"праведная", "праведн"},
		{"праведным", "праведн"},
		{"праведными", "праведн"},
		{"правее", "прав"},
		{"правел", "правел"},
		{"правил", "прав"},
		{"правила", "прав"},
		{"правилам", "правил"},
		{"правилах", "правил"},
		{"правило", "прав"},
		{"правильнее", "правильн"},
		{"правильно", "правильн"},
		{"правильного", "правильн"},
		{"правильное", "правильн"},
		{"правильной", "правильн"},
		{"правильность", "правильн"},
		{"правильною", "правильн"},
		{"правильную", "правильн"},
		{"правильным", "правильн"},
		{"правит", "прав"},
		{"правительствующему", "правительств"},
		{"правительствующий", "правительств"},
		{"правителя", "правител"},
		{"правление", "правлен"},
		{"правнуки", "правнук"},
		{"право", "прав"},
		{"правовед", "правовед"},
		{"правого", "прав"},
		{"правой", "прав"},
		{"правом", "прав"},
		{"правому", "прав"},
		{"православной", "православн"},
		{"православному", "православн"},
		{"православную", "православн"},
		{"православный", "православн"},
		{"правосудие", "правосуд"},
		{"правою", "прав"},
		{"праву", "прав"},
		{"правую", "прав"},
		{"правы", "прав"},
		{"правые", "прав"},
		{"правый", "прав"},
		{"правым", "прав"},
		{"прадед", "прадед"},
		{"прадедов", "прадед"},
		{"прадедушки", "прадедушк"},
		{"празден", "празд"},
		{"праздная", "праздн"},
		{"праздник", "праздник"},
		{"праздника", "праздник"},
		{"праздникам", "праздник"},
		{"праздниками", "праздник"},
		{"праздниках", "праздник"},
		{"празднике", "праздник"},
		{"праздники", "праздник"},
		{"праздников", "праздник"},
		{"празднику", "праздник"},
		{"празднично", "праздничн"},
		{"праздничные", "праздничн"},
		{"праздничный", "праздничн"},
		{"праздничных", "праздничн"},
		{"праздно", "праздн"},
		{"праздного", "праздн"},
		{"праздное", "праздн"},
		{"праздной", "праздн"},
		{"празднолюбцами", "празднолюбц"},
		{"праздном", "праздн"},
		{"праздному", "праздн"},
		{"праздности", "праздност"},
		{"праздность", "праздност"},
		{"праздностью", "праздност"},
		{"праздные", "праздн"},
		{"праздный", "праздн"},
		{"праздных", "праздн"},
		{"празелень", "празелен"},
		{"практика", "практик"},
		{"практике", "практик"},
		{"практики", "практик"},
		{"практиковать", "практикова"},
		{"практику", "практик"},
		{"практические", "практическ"},
		{"практический", "практическ"},
		{"практических", "практическ"},
		{"практического", "практическ"},
		{"практическое", "практическ"},
		{"практической", "практическ"},
		{"практическую", "практическ"},
		{"праотцам", "праотц"},
		{"праотцев", "праотц"},
		{"прасковьи", "прасков"},
		{"прасковью", "прасков"},
		{"прасковья", "прасков"},
		{"прах", "прах"},
		{"праха", "прах"},
		{"прахе", "прах"},
		{"прахом", "прах"},
		{"прачки", "прачк"},
		{"прачку", "прачк"},
		{"преавенантненькая", "преавенантненьк"},
		{"преблагополучно", "преблагополучн"},
		{"пребудет", "пребудет"},
		{"пребуду", "пребуд"},
		{"пребывали", "пребыва"},
		{"пребывания", "пребыван"},
		{"пребывая", "пребыв"},
		{"превесело", "превесел"},
		{"превозмогало", "превозмога"},
		{"превозмочь", "превозмоч"},
		{"превозносили", "превознос"},
		{"превозносят", "превознос"},
		{"превосходительство", "превосходительств"},
		{"превосходительством", "превосходительств"},
		{"превосходительству", "превосходительств"},
		{"превосходная", "превосходн"},
		{"превосходнейшем", "превосходн"},
		{"превосходно", "превосходн"},
		{"превосходного", "превосходн"},
		{"превосходный", "превосходн"},
		{"превосходными", "превосходн"},
		{"превосходных", "превосходн"},
		{"превосходства", "превосходств"},
		{"превратил", "преврат"},
		{"превратила", "преврат"},
		{"превратилось", "преврат"},
		{"превратился", "преврат"},
		{"превратится", "преврат"},
		{"превратно", "превратн"},
		{"превращается", "превраща"},
		{"превращалась", "превраща"},
		{"превращался", "превраща"},
		{"превращать", "превраща"},
		{"превращаться", "превраща"},
		{"превращения", "превращен"},
		{"превышает", "превыша"},
		{"превышать", "превыша"},
		{"преглупейшею", "преглуп"},
		{"преград", "преград"},
		{"преграда", "преград"},
		{"преградах", "преград"},
		{"преградой", "преград"},
		{"преграды", "преград"},
		{"пред", "пред"},
		{"предавала", "предава"},
		{"предавать", "предава"},
		{"предаваться", "предава"},
		{"предал", "преда"},
		{"предала", "преда"},
		{"предам", "пред"},
		{"предамся", "пред"},
		{"предан", "преда"},
		{"предана", "преда"},
		{"предание", "предан"},
		{"преданий", "предан"},
		{"предания", "предан"},
		{"преданиями", "предан"},
		{"преданнейшая", "предан"},
		{"преданной", "преда"},
		{"преданности", "предан"},
		{"преданность", "предан"},
		{"преданностью", "предан"},
		{"преданную", "преда"},
		{"преданные", "преда"},
		{"преданный", "преда"},
		{"преданьям", "предан"},
		{"преданьях", "предан"},
		{"предаст", "предаст"},
		{"предатели", "предател"},
		{"предательство", "предательств"},
		{"предать", "преда"},
		{"предаться", "преда"},
		{"предварительно", "предварительн"},
		{"предварительные", "предварительн"},
		{"предварительных", "предварительн"},
		{"предвестником", "предвестник"},
		{"предвестье", "предвест"},
		{"предвечном", "предвечн"},
		{"предвещало", "предвеща"},
		{"предвзятому", "предвзят"},
		{"предвзятых", "предвзят"},
		{"предвидел", "предвидел"},
		{"предвидела", "предвидел"},
		{"предвидели", "предвидел"},
		{"предвиделось", "предвидел"},
		{"предвидений", "предвиден"},
		{"предвидения", "предвиден"},
		{"предвидеть", "предвидет"},
		{"предвидите", "предвид"},
		{"предвидится", "предвид"},
		{"предвидя", "предвид"},
		{"предвижу", "предвиж"},
		{"предводителя", "предводител"},
		{"предел", "предел"},
		{"предела", "предел"},
		{"пределов", "предел"},
		{"предзнаменование", "предзнаменован"},
		{"предисловие", "предислов"},
		{"предки", "предк"},
		{"предлагавший", "предлага"},
		{"предлагаем", "предлага"},
		{"предлагаемого", "предлага"},
		{"предлагает", "предлага"},
		{"предлагается", "предлага"},
		{"предлагаешь", "предлага"},
		{"предлагал", "предлага"},
		{"предлагала", "предлага"},
		{"предлагали", "предлага"},
		{"предлагать", "предлага"},
		{"предлагаю", "предлага"},
		{"предлагают", "предлага"},
		{"предлагая", "предлаг"},
		{"предлог", "предлог"},
		{"предлога", "предлог"},
		{"предлогам", "предлог"},
		{"предлогами", "предлог"},
		{"предлоги", "предлог"},
		{"предлогом", "предлог"},
		{"предложат", "предложат"},
		{"предложен", "предлож"},
		{"предложение", "предложен"},
		{"предложением", "предложен"},
		{"предложении", "предложен"},
		{"предложений", "предложен"},
		{"предложению", "предложен"},
		{"предложения", "предложен"},
		{"предложениями", "предложен"},
		{"предложил", "предлож"},
		{"предложила", "предлож"},
		{"предложит", "предлож"},
		{"предложите", "предлож"},
		{"предложить", "предлож"},
		{"предложу", "предлож"},
		{"предместье", "предмест"},
		{"предмет", "предмет"},
		{"предмета", "предмет"},
		{"предметам", "предмет"},
		{"предметах", "предмет"},
		{"предмете", "предмет"},
		{"предметов", "предмет"},
		{"предметом", "предмет"},
		{"предмету", "предмет"},
		{"предметы", "предмет"},
		{"предназначена", "предназнач"},
		{"предназначенное", "предназначен"},
		{"предназначенные", "предназначен"},
		{"преднамеренно", "преднамерен"},
		{"преднамеренном", "преднамерен"},
		{"предо", "пред"},
		{"предопределение", "предопределен"},
		{"предопределением", "предопределен"},
		{"предорогие", "предорог"},
		{"предоставив", "предостав"},
		{"предоставить", "предостав"},
		{"предоставляла", "предоставля"},
		{"предоставляю", "предоставля"},
		{"предостерегал", "предостерега"},
		{"предостерегаю", "предостерега"},
		{"предостережение", "предостережен"},
		{"предостережений", "предостережен"},
		{"предостережения", "предостережен"},
		{"предостеречь", "предостереч"},
		{"предосторожности", "предосторожн"},
		{"предотвращение", "предотвращен"},
		{"предохранял", "предохраня"},
		{"предписал", "предписа"},
		{"предписания", "предписан"},
		{"предписанную", "предписа"},
		{"предписанным", "предписа"},
		{"предписано", "предписа"},
		{"предписываю", "предписыва"},
		{"предполагавшийся", "предполага"},
		{"предполагавшихся", "предполага"},
		{"предполагает", "предполага"},
		{"предполагаете", "предполага"},
		{"предполагается", "предполага"},
		{"предполагаешь", "предполага"},
		{"предполагал", "предполага"},
		{"предполагали", "предполага"},
		{"предполагались", "предполага"},
		{"предполагалось", "предполага"},
		{"предполагать", "предполага"},
		{"предполагая", "предполаг"},
		{"предположение", "предположен"},
		{"предположением", "предположен"},
		{"предположении", "предположен"},
		{"предположения", "предположен"},
		{"предположено", "предполож"},
		{"предположи", "предполож"},
		{"предположив", "предполож"},
		{"предположили", "предполож"},
		{"предположите", "предполож"},
		{"предположительно", "предположительн"},
		{"предположить", "предполож"},
		{"предпослать", "предпосла"},
		{"предпочел", "предпочел"},
		{"предпочитают", "предпочита"},
		{"предпочитая", "предпочит"},
		{"предпочла", "предпочл"},
		{"предпочтение", "предпочтен"},
		{"предпочтет", "предпочтет"},
		{"предпримет", "предпримет"},
		{"предпринимайте", "предпринима"},
		{"предпринималась", "предпринима"},
		{"предпринимаю", "предпринима"},
		{"предпринять", "предприня"},
		{"предприятие", "предприят"},
		{"предприятием", "предприят"},
		{"предприятии", "предприят"},
		{"предприятию", "предприят"},
		{"предприятия", "предприят"},
		{"предприятьях", "предприят"},
		{"предрассудкам", "предрассудк"},
		{"предрассудками", "предрассудк"},
		{"предрассудках", "предрассудк"},
		{"предрассудке", "предрассудк"},
		{"предрассудки", "предрассудк"},
		{"предрассудков", "предрассудк"},
		{"предрассудок", "предрассудок"},
		{"предрассудочные", "предрассудочн"},
		{"предрассудочных", "предрассудочн"},
		{"предрек", "предрек"},
		{"председателем", "председател"},
		{"председатель", "председател"},
		{"председательствует", "председательств"},
		{"председателя", "председател"},
		{"предсказал", "предсказа"},
		{"предсказания", "предсказан"},
		{"предсказывая", "предсказыв"},
		{"предсмертного", "предсмертн"},
		{"предсмертный", "предсмертн"},
		{"представали", "представа"},
		{"представив", "представ"},
		{"представившейся", "представ"},
		{"представил", "представ"},
		{"представила", "представ"},
		{"представилась", "представ"},
		{"представили", "представ"},
		{"представилось", "представ"},
		{"представительницею", "представительниц"},
		{"представится", "представ"},
		{"представить", "представ"},
		{"представиться", "представ"},
		{"представление", "представлен"},
		{"представления", "представлен"},
		{"представленный", "представлен"},
		{"представлю", "представл"},
		{"представляем", "представля"},
		{"представляет", "представля"},
		{"представляетесь", "представля"},
		{"представляется", "представля"},
		{"представлял", "представля"},
		{"представляла", "представля"},
		{"представлялась", "представля"},
		{"представляли", "представля"},
		{"представлялись", "представля"},
		{"представлялось", "представля"},
		{"представлялся", "представля"},
		{"представлять", "представля"},
		{"представляться", "представля"},
		{"представляюсь", "представля"},
		{"представляя", "представл"},
		{"представь", "представ"},
		{"представьте", "представьт"},
		{"представят", "представ"},
		{"предстала", "предста"},
		{"предстанет", "предстанет"},
		{"предстать", "предста"},
		{"предстоит", "предсто"},
		{"предстоял", "предстоя"},
		{"предстояла", "предстоя"},
		{"предстояли", "предстоя"},
		{"предстояло", "предстоя"},
		{"предстоящего", "предстоя"},
		{"предстоящее", "предстоя"},
		{"предстоящей", "предстоя"},
		{"предстоящим", "предстоя"},
		{"предстоящих", "предстоя"},
		{"предубеждение", "предубежден"},
		{"предубеждений", "предубежден"},
		{"предубеждения", "предубежден"},
		{"предуведомить", "предуведом"},
		{"предуведомлены", "предуведомл"},
		{"предугадает", "предугада"},
		{"предугадала", "предугада"},
		{"предугадывает", "предугадыва"},
		{"предуготовительном", "предуготовительн"},
		{"предузнало", "предузна"},
		{"предупредив", "предупред"},
		{"предупредил", "предупред"},
		{"предупредила", "предупред"},
		{"предупредительна", "предупредительн"},
		{"предупредить", "предупред"},
		{"предупреждал", "предупрежда"},
		{"предупреждаю", "предупрежда"},
		{"предупреждена", "предупрежд"},
		{"предусмотрела", "предусмотрел"},
		{"предусмотрительностью", "предусмотрительн"},
		{"предчувствие", "предчувств"},
		{"предчувствием", "предчувств"},
		{"предчувствий", "предчувств"},
		{"предчувствию", "предчувств"},
		{"предчувствия", "предчувств"},
		{"предчувствиями", "предчувств"},
		{"предчувствовавшая", "предчувствова"},
		{"предчувствовал", "предчувствова"},
		{"предчувствовала", "предчувствова"},
		{"предчувствовалась", "предчувствова"},
		{"предчувствовались", "предчувствова"},
		{"предчувствовалось", "предчувствова"},
		{"предчувствовать", "предчувствова"},
		{"предчувствует", "предчувств"},
		{"предчувствую", "предчувств"},
		{"предчувствуют", "предчувств"},
		{"предчувствуя", "предчувству"},
		{"предшественником", "предшественник"},
		{"предшествовавшие", "предшествова"},
		{"предшествовавших", "предшествова"},
		{"предъявляет", "предъявля"},
		{"предыдущая", "предыдущ"},
		{"предыдущего", "предыдущ"},
		{"предыдущий", "предыдущ"},
		{"предыдущим", "предыдущ"},
		{"предыдущую", "предыдущ"},
		{"прежде", "прежд"},
		{"преждевременно", "преждевремен"},
		{"преждевременными", "преждевремен"},
		{"прежнего", "прежн"},
		{"прежнее", "прежн"},
		{"прежней", "прежн"},
		{"прежнему", "прежн"},
		{"прежнею", "прежн"},
		{"прежние", "прежн"},
		{"прежний", "прежн"},
		{"прежним", "прежн"},
		{"прежними", "прежн"},
		{"прежних", "прежн"},
		{"прежнюю", "прежн"},
		{"прежняя", "прежн"},
		{"президентом", "президент"},
		{"презиравшие", "презира"},
		{"презиравших", "презира"},
		{"презирает", "презира"},
		{"презираете", "презира"},
		{"презирал", "презира"},
		{"презирала", "презира"},
		{"презирали", "презира"},
		{"презирать", "презира"},
		{"презираю", "презира"},
		{"презирают", "презира"},
		{"презирающие", "презира"},
		{"презирая", "презир"},
		{"презревши", "презревш"},
		{"презрение", "презрен"},
		{"презрением", "презрен"},
		{"презрения", "презрен"},
		{"презренная", "презрен"},
		{"презренно", "презрен"},
		{"презренного", "презрен"},
		{"презренье", "презрен"},
		{"презренья", "презрен"},
		{"презрительнее", "презрительн"},
		{"презрительно", "презрительн"},
		{"презрительной", "презрительн"},
		{"презрительный", "презрительн"},
		{"презрительным", "презрительн"},
		{"презрительных", "презрительн"},
		{"преимуществ", "преимуществ"},
		{"преимущества", "преимуществ"},
		{"преимуществе", "преимуществ"},
		{"преимущественно", "преимуществен"},
		{"преимущество", "преимуществ"},
		{"преимуществом", "преимуществ"},
		{"преимуществу", "преимуществ"},
		{"преисподнего", "преисподн"},
		{"преисполнена", "преисполн"},
		{"преисполнила", "преисполн"},
		{"преклонив", "преклон"},
		{"преклонила", "преклон"},
		{"преклонился", "преклон"},
		{"преклониться", "преклон"},
		{"преклонный", "преклон"},
		{"преклоняться", "преклоня"},
		{"прекомические", "прекомическ"},
		{"прекословь", "прекослов"},
		{"прекрасен", "прекрас"},
		{"прекрасна", "прекрасн"},
		{"прекрасная", "прекрасн"},
		{"прекраснее", "прекрасн"},
		{"прекрасней", "прекрасн"},
		{"прекраснейшая", "прекрасн"},
		{"прекрасно", "прекрасн"},
		{"прекрасного", "прекрасн"},
		{"прекрасное", "прекрасн"},
		{"прекрасной", "прекрасн"},
		{"прекрасном", "прекрасн"},
		{"прекрасною", "прекрасн"},
		{"прекрасную", "прекрасн"},
		{"прекрасны", "прекрасн"},
		{"прекрасные", "прекрасн"},
		{"прекрасный", "прекрасн"},
		{"прекрасным", "прекрасн"},
		{"прекрасными", "прекрасн"},
		{"прекрасных", "прекрасн"},
		{"прекратив", "прекрат"},
		{"прекратились", "прекрат"},
		{"прекратило", "прекрат"},
		{"прекратить", "прекрат"},
		{"прекращалось", "прекраща"},
		{"прекращаю", "прекраща"},
		{"прела", "прел"},
		{"прелести", "прелест"},
		{"прелестная", "прелестн"},
		{"прелестней", "прелестн"},
		{"прелестной", "прелестн"},
		{"прелестные", "прелестн"},
		{"прелестный", "прелестн"},
		{"прелесть", "прелест"},
		{"прелестями", "прелест"},
		{"преломились", "прелом"},
		{"преломлялись", "преломля"},
		{"прельститься", "прельст"},
		{"прельщает", "прельща"},
		{"прелюбопытный", "прелюбопытн"},
		{"премило", "прем"},
		{"премудрости", "премудр"},
		{"премудрствующие", "премудрств"},
		{"премудрые", "премудр"},
		{"премудрых", "премудр"},
		{"пренаивно", "пренаивн"},
		{"пренебрег", "пренебрег"},
		{"пренебрегал", "пренебрега"},
		{"пренебрегать", "пренебрега"},
		{"пренебрегая", "пренебрег"},
		{"пренебрежение", "пренебрежен"},
		{"пренебрежением", "пренебрежен"},
		{"пренебрежения", "пренебрежен"},
		{"преобладали", "преоблада"},
		{"преобладало", "преоблада"},
		{"преображаются", "преобража"},
		{"преобразилась", "преобраз"},
		{"преобразилось", "преобраз"},
		{"преобразований", "преобразован"},
		{"преогромнейшему", "преогромн"},
		{"преодолела", "преодолел"},
		{"преодолеть", "преодолет"},
		{"преподаванию", "преподаван"},
		{"преподал", "препода"},
		{"преподана", "препода"},
		{"преподаю", "препода"},
		{"препочтенный", "препочтен"},
		{"препровождал", "препровожда"},
		{"препровожу", "препровож"},
		{"препьяные", "препьян"},
		{"препятствие", "препятств"},
		{"препятствий", "препятств"},
		{"препятствию", "препятств"},
		{"препятствия", "препятств"},
		{"препятствиями", "препятств"},
		{"препятствовала", "препятствова"},
		{"препятствуют", "препятств"},
		{"прервав", "прерва"},
		{"прервал", "прерва"},
		{"прервала", "прерва"},
		{"прервали", "прерва"},
		{"прервался", "прерва"},
		{"прерванных", "прерва"},
		{"прервано", "прерва"},
		{"прерий", "прер"},
		{"прерываемый", "прерыва"},
		{"прерывает", "прерыва"},
		{"прерывай", "прерыва"},
		{"прерывал", "прерыва"},
		{"прерывался", "прерыва"},
		{"прерывать", "прерыва"},
		{"прерываю", "прерыва"},
		{"прерывающее", "прерыва"},
		{"прерывающуюся", "прерыва"},
		{"прерывая", "прерыв"},
		{"прерывисто", "прерывист"},
		{"прерывистым", "прерывист"},
		{"пресвятая", "пресвят"},
		{"пресекла", "пресекл"},
		{"пресеклось", "пресекл"},
		{"пресекся", "пресек"},
		{"пресекшую", "пресекш"},
		{"пресинее", "пресин"},
		{"прескладной", "прескладн"},
		{"преследовал", "преследова"},
		{"преследовала", "преследова"},
		{"преследования", "преследован"},
		{"преследовать", "преследова"},
		{"преследует", "преслед"},
		{"преследуйте", "преслед"},
		{"преследуют", "преслед"},
		{"пресмыкаюсь", "пресмыка"},
		{"преспокойно", "преспокойн"},
		{"пресс", "пресс"},
		{"преставлю", "преставл"},
		{"престарелая", "престарел"},
		{"престарелому", "престарел"},
		{"престарелую", "престарел"},
		{"престоле", "престол"},
		{"престранная", "престра"},
		{"преступать", "преступа"},
		{"преступают", "преступа"},
		{"преступление", "преступлен"},
		{"преступлением", "преступлен"},
		{"преступлении", "преступлен"},
		{"преступлений", "преступлен"},
		{"преступлению", "преступлен"},
		{"преступления", "преступлен"},
		{"преступлениях", "преступлен"},
		{"преступленьице", "преступленьиц"},
		{"преступленья", "преступлен"},
		{"преступна", "преступн"},
		{"преступнее", "преступн"},
		{"преступник", "преступник"},
		{"преступника", "преступник"},
		{"преступникам", "преступник"},
		{"преступниками", "преступник"},
		{"преступнике", "преступник"},
		{"преступники", "преступник"},
		{"преступников", "преступник"},
		{"преступнику", "преступник"},
		{"преступница", "преступниц"},
		{"преступного", "преступн"},
		{"преступной", "преступн"},
		{"преступны", "преступн"},
		{"преступный", "преступн"},
		{"претендуете", "претендует"},
		{"претендующих", "претенд"},
		{"претендуя", "претенду"},
		{"претензии", "претенз"},
		{"претензий", "претенз"},
		{"претензию", "претенз"},
		{"претензия", "претенз"},
		{"претензиями", "претенз"},
		{"претерпели", "претерпел"},
		{"претит", "прет"},
		{"преткновения", "преткновен"},
		{"преткнулся", "преткнул"},
		{"претрудная", "претрудн"},
		{"претят", "прет"},
		{"преувеличенно", "преувеличен"},
		{"преувеличенного", "преувеличен"},
		{"преувеличенною", "преувеличен"},
		{"преувеличенный", "преувеличен"},
		{"преувеличено", "преувелич"},
		{"преувеличены", "преувелич"},
		{"преувеличиваете", "преувеличива"},
		{"преувеличивал", "преувеличива"},
		{"преувеличивающий", "преувеличива"},
		{"преувеличил", "преувелич"},
		{"преувеличили", "преувелич"},
		{"преувеличить", "преувелич"},
		{"преумножать", "преумножа"},
		{"преуспеяние", "преуспеян"},
		{"преуспеяния", "преуспеян"},
		{"прехитрейшее", "прехитр"},
		{"прехорошенькие", "прехорошеньк"},
		{"пречистенки", "пречистенк"},
		{"прещено", "прещ"},
		{"при", "при"},
		{"прибавив", "прибав"},
		{"прибавил", "прибав"},
		{"прибавила", "прибав"},
		{"прибавили", "прибав"},
		{"прибавило", "прибав"},
		{"прибавилось", "прибав"},
		{"прибавим", "прибав"},
		{"прибавит", "прибав"},
		{"прибавить", "прибав"},
		{"прибавишь", "прибав"},
		{"прибавке", "прибавк"},
		{"прибавкой", "прибавк"},
		{"прибавлена", "прибавл"},
		{"прибавление", "прибавлен"},
		{"прибавлением", "прибавлен"},
		{"прибавления", "прибавлен"},
		{"прибавлено", "прибавл"},
		{"прибавлю", "прибавл"},
		{"прибавляет", "прибавля"},
		{"прибавляешь", "прибавля"},
		{"прибавлял", "прибавля"},
		{"прибавляла", "прибавля"},
		{"прибавлять", "прибавля"},
		{"прибавляю", "прибавля"},
		{"прибавляют", "прибавля"},
		{"прибавляются", "прибавля"},
		{"прибавочные", "прибавочн"},
		{"прибавь", "прибав"},
		{"прибавьте", "прибавьт"},
		{"прибегал", "прибега"},
		{"прибегала", "прибега"},
		{"прибегают", "прибега"},
		{"прибегая", "прибег"},
		{"прибегнул", "прибегнул"},
		{"прибежавшей", "прибежа"},
		{"прибежал", "прибежа"},
		{"прибежала", "прибежа"},
		{"прибежали", "прибежа"},
		{"прибежит", "прибеж"},
		{"приберегла", "приберегл"},
		{"приберет", "приберет"},
		{"приберешь", "прибереш"},
		{"прибившему", "приб"},
		{"прибил", "приб"},
		{"прибили", "приб"},
		{"прибиль", "прибил"},
		{"прибирает", "прибира"},
		{"прибирали", "прибира"},
		{"прибирать", "прибира"},
		{"прибираться", "прибира"},
		{"прибитый", "прибит"},
		{"приближавшейся", "приближа"},
		{"приближается", "приближа"},
		{"приближалась", "приближа"},
		{"приближался", "приближа"},
		{"приближающееся", "приближа"},
		{"приближая", "приближ"},
		{"приближения", "приближен"},
		{"приблизив", "приблиз"},
		{"приблизилась", "приблиз"},
		{"приблизился", "приблиз"},
		{"приблизительно", "приблизительн"},
		{"прибой", "приб"},
		{"прибор", "прибор"},
		{"прибоя", "прибо"},
		{"прибрал", "прибра"},
		{"прибрано", "прибра"},
		{"прибраны", "прибра"},
		{"прибрать", "прибра"},
		{"прибрежный", "прибрежн"},
		{"прибрежья", "прибреж"},
		{"прибудет", "прибудет"},
		{"прибудут", "прибудут"},
		{"прибыв", "приб"},
		{"прибывает", "прибыва"},
		{"прибывшего", "приб"},
		{"прибыл", "приб"},
		{"прибыли", "приб"},
		{"прибытия", "прибыт"},
		{"прибыть", "приб"},
		{"прибьет", "прибьет"},
		{"прибьют", "прибьют"},
		{"привалило", "привал"},
		{"приватном", "приватн"},
		{"приватным", "приватн"},
		{"приведенное", "приведен"},
		{"приведены", "привед"},
		{"приведет", "приведет"},
		{"приведи", "привед"},
		{"приведу", "привед"},
		{"приведут", "приведут"},
		{"привез", "привез"},
		{"привезет", "привезет"},
		{"привезите", "привез"},
		{"привезла", "привезл"},
		{"привезли", "привезл"},
		{"привезти", "привезт"},
		{"привезть", "привезт"},
		{"привезу", "привез"},
		{"привезут", "привезут"},
		{"привел", "привел"},
		{"привела", "привел"},
		{"привели", "привел"},
		{"привело", "привел"},
		{"привелось", "привел"},
		{"привесить", "привес"},
		{"привести", "привест"},
		{"привет", "привет"},
		{"приветливо", "приветлив"},
		{"приветливым", "приветлив"},
		{"приветные", "приветн"},
		{"приветствие", "приветств"},
		{"приветствий", "приветств"},
		{"приветствует", "приветств"},
		{"приветствую", "приветств"},
		{"привечали", "привеча"},
		{"привешены", "привеш"},
		{"привидение", "привиден"},
		{"привидений", "привиден"},
		{"привидения", "привиден"},
		{"привил", "прив"},
		{"привилегию", "привилег"},
		{"привилегия", "привилег"},
		{"привить", "прив"},
		{"привлек", "привлек"},
		{"привлекает", "привлека"},
		{"привлекало", "привлека"},
		{"привлекательном", "привлекательн"},
		{"привлекать", "привлека"},
		{"привлекают", "привлека"},
		{"привлекло", "привлекл"},
		{"привлечь", "привлеч"},
		{"приводил", "привод"},
		{"приводила", "привод"},
		{"приводили", "привод"},
		{"приводит", "привод"},
		{"приводить", "привод"},
		{"приводят", "привод"},
		{"привозили", "привоз"},
		{"привозом", "привоз"},
		{"приволье", "привол"},
		{"приволью", "привол"},
		{"приволья", "привол"},
		{"привскочив", "привскоч"},
		{"привскочила", "привскоч"},
		{"привстав", "привста"},
		{"привставший", "привста"},
		{"привстал", "привста"},
		{"привстала", "привста"},
		{"привстанет", "привстанет"},
		{"привык", "привык"},
		{"привыкает", "привыка"},
		{"привыкали", "привыка"},
		{"привыкла", "привыкл"},
		{"привыкли", "привыкл"},
		{"привыкнет", "привыкнет"},
		{"привыкнете", "привыкнет"},
		{"привыкнешь", "привыкнеш"},
		{"привыкну", "привыкн"},
		{"привыкнуть", "привыкнут"},
		{"привыкши", "привыкш"},
		{"привыкшим", "привыкш"},
		{"привычек", "привычек"},
		{"привычка", "привычк"},
		{"привычками", "привычк"},
		{"привычках", "привычк"},
		{"привычке", "привычк"},
		{"привычки", "привычк"},
		{"привычкой", "привычк"},
		{"привычку", "привычк"},
		{"привычно", "привычн"},
		{"привычной", "привычн"},
		{"привычную", "привычн"},
		{"привычны", "привычн"},
		{"привычный", "привычн"},
		{"привычными", "привычн"},
		{"привяжусь", "привяж"},
		{"привяжутся", "привяжут"},
		{"привязал", "привяза"},
		{"привязалась", "привяза"},
		{"привязался", "привяза"},
		{"привязан", "привяза"},
		{"привязана", "привяза"},
		{"привязанности", "привязан"},
		{"привязанность", "привязан"},
		{"привязанностью", "привязан"},
		{"привязанностям", "привязан"},
		{"привязаны", "привяза"},
		{"привязаться", "привяза"},
		{"привязки", "привязк"},
		{"привязчивее", "привязчив"},
		{"привязываетесь", "привязыва"},
		{"пригвожден", "пригвожд"},
		{"приглаживает", "приглажива"},
		{"пригласил", "приглас"},
		{"пригласила", "приглас"},
		{"пригласили", "приглас"},
		{"пригласит", "приглас"},
		{"пригласите", "приглас"},
		{"пригласить", "приглас"},
		{"приглашаем", "приглаша"},
		{"приглашает", "приглаша"},
		{"приглашайте", "приглаша"},
		{"приглашал", "приглаша"},
		{"приглашала", "приглаша"},
		{"приглашалась", "приглаша"},
		{"приглашали", "приглаша"},
		{"приглашаль", "приглашал"},
		{"приглашать", "приглаша"},
		{"приглашаю", "приглаша"},
		{"приглашают", "приглаша"},
		{"приглашен", "приглаш"},
		{"приглашена", "приглаш"},
		{"приглашение", "приглашен"},
		{"приглашением", "приглашен"},
		{"приглашения", "приглашен"},
		{"приглашенных", "приглашен"},
		{"приглашены", "приглаш"},
		{"приглашенье", "приглашен"},
		{"приглядеть", "приглядет"},
		{"приглядится", "пригляд"},
		{"приглядывалась", "приглядыва"},
		{"приглядывался", "приглядыва"},
		{"приглядываться", "приглядыва"},
		{"приглядывая", "приглядыв"},
		{"приглядываясь", "приглядыв"},
		{"приглянуть", "приглянут"},
		{"пригнул", "пригнул"},
		{"пригнула", "пригнул"},
		{"пригнулись", "пригнул"},
		{"пригнулся", "пригнул"},
		{"приговаривал", "приговарива"},
		{"приговаривала", "приговарива"},
		{"приговаривая", "приговарив"},
		{"приговарнвал", "приговарнва"},
		{"приговор", "приговор"},
		{"приговора", "приговор"},
		{"приговоренного", "приговорен"},
		{"приговоренный", "приговорен"},
		{"приговорили", "приговор"},
		{"приговоркой", "приговорк"},
		{"приговоров", "приговор"},
		{"приговору", "приговор"},
		{"пригодилась", "пригод"},
		{"пригодитесь", "пригод"},
		{"пригодится", "пригод"},
		{"пригодное", "пригодн"},
		{"пригодятся", "пригод"},
		{"пригорка", "пригорк"},
		{"пригорки", "пригорк"},
		{"пригорку", "пригорк"},
		{"пригорок", "пригорок"},
		{"пригорюнившись", "пригорюн"},
		{"пригорюнился", "пригорюн"},
		{"приготовившись", "приготов"},
		{"приготовил", "приготов"},
		{"приготовилась", "приготов"},
		{"приготовился", "приготов"},
		{"приготовит", "приготов"},
		{"приготовить", "приготов"},
		{"приготовиться", "приготов"},
		{"приготовлен", "приготовл"},
		{"приготовление", "приготовлен"},
		{"приготовлений", "приготовлен"},
		{"приготовления", "приготовлен"},
		{"приготовлениях", "приготовлен"},
		{"приготовленный", "приготовлен"},
		{"приготовленным", "приготовлен"},
		{"приготовлено", "приготовл"},
		{"приготовлены", "приготовл"},
		{"приготовляй", "приготовля"},
		{"приготовлялось", "приготовля"},
		{"приготовлять", "приготовля"},
		{"приготовляться", "приготовля"},
		{"приготовляюсь", "приготовля"},
		{"приготовляясь", "приготовл"},
		{"приготовят", "приготов"},
		{"пригрезился", "пригрез"},
		{"пригрозили", "пригроз"},
		{"придававшая", "придава"},
		{"придавал", "придава"},
		{"придавала", "придава"},
		{"придавали", "придава"},
		{"придавало", "придава"},
		{"придавил", "придав"},
		{"придавили", "придав"},
		{"придавило", "придав"},
		{"придавим", "придав"},
		{"придавить", "придав"},
		{"придавлено", "придавл"},
		{"придавливая", "придавлив"},
		{"придает", "прида"},
		{"придаете", "прида"},
		{"придам", "прид"},
		{"приданный", "прида"},
		{"приданого", "придан"},
		{"приданое", "придан"},
		{"придаст", "придаст"},
		{"придаток", "придаток"},
		{"придать", "прида"},
		{"придачу", "придач"},
		{"придвигая", "придвиг"},
		{"придвинул", "придвинул"},
		{"придвинулась", "придвинул"},
		{"придворной", "придворн"},
		{"придем", "прид"},
		{"придержал", "придержа"},
		{"придержать", "придержа"},
		{"придержаться", "придержа"},
		{"придерживает", "придержива"},
		{"придерживал", "придержива"},
		{"придерживались", "придержива"},
		{"придерживать", "придержива"},
		{"придерживаться", "придержива"},
		{"придерживаюсь", "придержива"},
		{"придерживая", "придержив"},
		{"придерживаясь", "придержив"},
		{"придет", "придет"},
		{"придете", "придет"},
		{"придется", "придет"},
		{"придешь", "придеш"},
		{"приди", "прид"},
		{"придираетесь", "придира"},
		{"придираться", "придира"},
		{"придирки", "придирк"},
		{"придирчив", "придирч"},
		{"придите", "прид"},
		{"придорожной", "придорожн"},
		{"приду", "прид"},
		{"придумаем", "придума"},
		{"придумает", "придума"},
		{"придумай", "придума"},
		{"придумал", "придума"},
		{"придумала", "придума"},
		{"придуманное", "придума"},
		{"придуманных", "придума"},
		{"придумать", "придума"},
		{"придумывает", "придумыва"},
		{"придумывай", "придумыва"},
		{"придумывала", "придумыва"},
		{"придумывая", "придумыв"},
		{"придут", "придут"},
		{"придутся", "придут"},
		{"придя", "прид"},
		{"приедем", "приед"},
		{"приедет", "приедет"},
		{"приедете", "приедет"},
		{"приедешь", "приедеш"},
		{"приеду", "приед"},
		{"приедут", "приедут"},
		{"приезд", "приезд"},
		{"приезда", "приезд"},
		{"приезде", "приезд"},
		{"приездом", "приезд"},
		{"приезду", "приезд"},
		{"приезжавшие", "приезжа"},
		{"приезжаем", "приезжа"},
		{"приезжает", "приезжа"},
		{"приезжай", "приезжа"},
		{"приезжайте", "приезжа"},
		{"приезжал", "приезжа"},
		{"приезжала", "приезжа"},
		{"приезжали", "приезжа"},
		{"приезжать", "приезжа"},
		{"приезжают", "приезжа"},
		{"приезжая", "приезж"},
		{"приезжее", "приезж"},
		{"приезжие", "приезж"},
		{"приезжий", "приезж"},
		{"приезжих", "приезж"},
		{"прием", "при"},
		{"приема", "прием"},
		{"приемам", "прием"},
		{"приемами", "прием"},
		{"приеме", "прием"},
		{"приемлеши", "приемлеш"},
		{"приемлю", "приемл"},
		{"приемная", "приемн"},
		{"приемной", "приемн"},
		{"приемные", "приемн"},
		{"приемов", "прием"},
		{"приему", "при"},
		{"приемы", "прием"},
		{"приехав", "приеха"},
		{"приехавшем", "приеха"},
		{"приехал", "приеха"},
		{"приехала", "приеха"},
		{"приехали", "приеха"},
		{"приехать", "приеха"},
		{"прижав", "прижа"},
		{"прижавшись", "прижа"},
		{"прижал", "прижа"},
		{"прижала", "прижа"},
		{"прижалась", "прижа"},
		{"прижали", "прижа"},
		{"прижата", "прижат"},
		{"прижатыми", "прижат"},
		{"приживал", "прижива"},
		{"приживалы", "приживал"},
		{"прижила", "приж"},
		{"прижимается", "прижима"},
		{"прижимала", "прижима"},
		{"прижималась", "прижима"},
		{"прижимался", "прижима"},
		{"прижимаю", "прижима"},
		{"прижимая", "прижим"},
		{"прижимаясь", "прижим"},
		{"прижмешь", "прижмеш"},
		{"прижмись", "прижм"},
		{"прижму", "прижм"},
		{"призадумался", "призадума"},
		{"призван", "призва"},
		{"призвание", "призван"},
		{"призванием", "призван"},
		{"призвании", "призван"},
		{"призванию", "призван"},
		{"призванную", "призва"},
		{"призвать", "призва"},
		{"признав", "призна"},
		{"признавай", "признава"},
		{"признавайся", "признава"},
		{"признавал", "признава"},
		{"признавала", "признава"},
		{"признавалась", "признава"},
		{"признавалось", "признава"},
		{"признавался", "признава"},
		{"признаваться", "признава"},
		{"признавая", "признав"},
		{"признаваясь", "признав"},
		{"признает", "призна"},
		{"признаете", "призна"},
		{"признаетесь", "призна"},
		{"признайся", "призна"},
		{"признайтесь", "призна"},
		{"признак", "признак"},
		{"признака", "признак"},
		{"признакам", "признак"},
		{"признаками", "признак"},
		{"признаках", "признак"},
		{"признаки", "признак"},
		{"признаков", "признак"},
		{"признаком", "признак"},
		{"признал", "призна"},
		{"призналась", "призна"},
		{"признали", "призна"},
		{"признался", "призна"},
		{"признание", "признан"},
		{"признании", "признан"},
		{"признания", "признан"},
		{"признаниям", "признан"},
		{"признанную", "призна"},
		{"признанных", "призна"},
		{"признанья", "признан"},
		{"признательности", "признательн"},
		{"признательный", "признательн"},
		{"признать", "призна"},
		{"признаться", "призна"},
		{"признаю", "призна"},
		{"признаюсь", "призна"},
		{"призовет", "призовет"},
		{"призовешь", "призовеш"},
		{"призрак", "призрак"},
		{"призрака", "призрак"},
		{"призраками", "призрак"},
		{"призраках", "призрак"},
		{"призраки", "призрак"},
		{"призраков", "призрак"},
		{"призраку", "призрак"},
		{"призрачное", "призрачн"},
		{"призрение", "призрен"},
		{"призрения", "призрен"},
		{"призыв", "приз"},
		{"призывал", "призыва"},
		{"призывать", "призыва"},
		{"призывающе", "призывающ"},
		{"призывающий", "призыва"},
		{"призывая", "призыв"},
		{"призыве", "призыв"},
		{"призывном", "призывн"},
		{"призывных", "призывн"},
		{"приидет", "приидет"},
		{"прииди", "приид"},
		{"приидите", "приид"},
		{"приискал", "прииска"},
		{"приискания", "приискан"},
		{"приискать", "прииска"},
		{"приискивать", "приискива"},
		{"приищете", "приищет"},
		{"прийти", "прийт"},
		{"прикажет", "прикажет"},
		{"прикажете", "прикажет"},
		{"прикажешь", "прикажеш"},
		{"прикажите", "прикаж"},
		{"приказ", "приказ"},
		{"приказал", "приказа"},
		{"приказала", "приказа"},
		{"приказали", "приказа"},
		{"приказан", "приказа"},
		{"приказание", "приказан"},
		{"приказанием", "приказан"},
		{"приказании", "приказан"},
		{"приказанию", "приказан"},
		{"приказания", "приказан"},
		{"приказаниями", "приказан"},
		{"приказано", "приказа"},
		{"приказать", "приказа"},
		{"приказного", "приказн"},
		{"приказчик", "приказчик"},
		{"приказчика", "приказчик"},
		{"приказчики", "приказчик"},
		{"приказчиков", "приказчик"},
		{"приказывает", "приказыва"},
		{"приказываете", "приказыва"},
		{"приказывайте", "приказыва"},
		{"приказывал", "приказыва"},
		{"приказывала", "приказыва"},
		{"приказывали", "приказыва"},
		{"приказывать", "приказыва"},
		{"приказывающего", "приказыва"},
		{"прикапливал", "прикаплива"},
		{"прикасалось", "прикаса"},
		{"прикасаясь", "прикас"},
		{"прикидываются", "прикидыва"},
		{"прикинешься", "прикинеш"},
		{"прикинулся", "прикинул"},
		{"прикинуться", "прикинут"},
		{"прикладывал", "прикладыва"},
		{"прикладывали", "прикладыва"},
		{"прикладывая", "прикладыв"},
		{"приклеившись", "прикле"},
		{"приклонить", "приклон"},
		{"приключалась", "приключа"},
		{"приключение", "приключен"},
		{"приключений", "приключен"},
		{"приковалась", "прикова"},
		{"приковываться", "приковыва"},
		{"приколотить", "приколот"},
		{"прикомандировался", "прикомандирова"},
		{"прикосновением", "прикосновен"},
		{"прикосновения", "прикосновен"},
		{"прикоснулись", "прикоснул"},
		{"прикоснулось", "прикоснул"},
		{"прикоснулся", "прикоснул"},
		{"прикоснуться", "прикоснут"},
		{"прикраситься", "прикрас"},
		{"прикрасной", "прикрасн"},
		{"прикрепил", "прикреп"},
		{"прикрутит", "прикрут"},
		{"прикрыв", "прикр"},
		{"прикрывается", "прикрыва"},
		{"прикрывала", "прикрыва"},
		{"прикрывалось", "прикрыва"},
		{"прикрываясь", "прикрыв"},
		{"прикрыл", "прикр"},
		{"прикрыла", "прикр"},
		{"прикрыли", "прикр"},
		{"прикрытая", "прикрыт"},
		{"прикрытого", "прикрыт"},
		{"прикрытой", "прикрыт"},
		{"прикрытую", "прикрыт"},
		{"прикрытыми", "прикрыт"},
		{"прикрытых", "прикрыт"},
		{"прикрыть", "прикр"},
		{"прикупить", "прикуп"},
		{"прикусив", "прикус"},
		{"прилагает", "прилага"},
		{"приладил", "прилад"},
		{"приладиться", "прилад"},
		{"прилаженный", "прилажен"},
		{"прилаживалась", "прилажива"},
		{"прилаживать", "прилажива"},
		{"приласкав", "приласка"},
		{"приласкаешь", "приласка"},
		{"приласкать", "приласка"},
		{"прилег", "прилег"},
		{"прилегавший", "прилега"},
		{"прилежанием", "прилежан"},
		{"прилежно", "прилежн"},
		{"прилежным", "прилежн"},
		{"прилепившись", "прилеп"},
		{"прилепилась", "прилеп"},
		{"прилепить", "прилеп"},
		{"прилепливаются", "прилеплива"},
		{"прилеплялся", "прилепля"},
		{"прилетит", "прилет"},
		{"прилетят", "прилет"},
		{"прилечь", "прилеч"},
		{"прилив", "прил"},
		{"прилива", "прилив"},
		{"приливало", "прилива"},
		{"приливам", "прилив"},
		{"приливе", "прилив"},
		{"приливом", "прилив"},
		{"приливы", "прилив"},
		{"прилип", "прилип"},
		{"прилипчивая", "прилипчив"},
		{"прилипшие", "прилипш"},
		{"приличие", "прилич"},
		{"приличием", "прилич"},
		{"приличий", "прилич"},
		{"приличия", "прилич"},
		{"приличнее", "приличн"},
		{"приличненьким", "приличненьк"},
		{"прилично", "приличн"},
		{"приличного", "приличн"},
		{"приличной", "приличн"},
		{"приличном", "приличн"},
		{"приличною", "приличн"},
		{"приличную", "приличн"},
		{"приличные", "приличн"},
		{"приличный", "приличн"},
		{"приличным", "приличн"},
		{"приличных", "приличн"},
		{"приложа", "прилож"},
		{"приложат", "приложат"},
		{"приложена", "прилож"},
		{"приложением", "приложен"},
		{"приложи", "прилож"},
		{"приложив", "прилож"},
		{"приложил", "прилож"},
		{"приложила", "прилож"},
		{"приложить", "прилож"},
		{"прилуковой", "прилуков"},
		{"прильнет", "прильнет"},
		{"прильнув", "прильнув"},
		{"прильнул", "прильнул"},
		{"прильнула", "прильнул"},
		{"прильнуть", "прильнут"},
		{"прильпне", "прильпн"},
		{"прилягу", "приляг"},
		{"приляжет", "приляжет"},
		{"приманивала", "приманива"},
		{"примачивайте", "примачива"},
		{"примелькавшиеся", "примелька"},
		{"примем", "прим"},
		{"применены", "примен"},
		{"применилась", "примен"},
		{"применить", "примен"},
		{"применять", "применя"},
		{"пример", "пример"},
		{"примерами", "пример"},
		{"примерив", "пример"},
		{"примеривал", "примерива"},
		{"примеривая", "примерив"},
		{"примерит", "пример"},
		{"примерить", "пример"},
		{"примерно", "примерн"},
		{"примеров", "пример"},
		{"примером", "пример"},
		{"примеру", "пример"},
		{"примерчик", "примерчик"},
		{"примерчики", "примерчик"},
		{"примеры", "пример"},
		{"примеряет", "примеря"},
		{"примерять", "примеря"},
		{"примесью", "примес"},
		{"примет", "примет"},
		{"примета", "примет"},
		{"приметам", "примет"},
		{"примете", "примет"},
		{"приметесь", "примет"},
		{"приметил", "примет"},
		{"приметила", "примет"},
		{"приметить", "примет"},
		{"приметишь", "примет"},
		{"приметная", "приметн"},
		{"приметнее", "приметн"},
		{"приметно", "приметн"},
		{"приметною", "приметн"},
		{"приметную", "приметн"},
		{"приметным", "приметн"},
		{"приметными", "приметн"},
		{"примется", "примет"},
		{"примету", "примет"},
		{"приметы", "примет"},
		{"примечал", "примеча"},
		{"примечательную", "примечательн"},
		{"примечать", "примеча"},
		{"примешивается", "примешива"},
		{"примешивал", "примешива"},
		{"примешивалась", "примешива"},
		{"прими", "прим"},
		{"примирении", "примирен"},
		{"примирения", "примирен"},
		{"примирительный", "примирительн"},
		{"примирительным", "примирительн"},
		{"примирительных", "примирительн"},
		{"примирится", "примир"},
		{"примириться", "примир"},
		{"примите", "прим"},
		{"примолвив", "примолв"},
		{"примолвить", "примолв"},
		{"примолк", "примолк"},
		{"примолкла", "примолкл"},
		{"примолкли", "примолкл"},
		{"примочками", "примочк"},
		{"примочки", "примочк"},
		{"приму", "прим"},
		{"примут", "примут"},
		{"примутся", "примут"},
		{"примыкавшей", "примыка"},
		{"примятой", "примят"},
		{"принагнувшись", "принагнувш"},
		{"принадлежавшая", "принадлежа"},
		{"принадлежавшего", "принадлежа"},
		{"принадлежавшее", "принадлежа"},
		{"принадлежавший", "принадлежа"},
		{"принадлежал", "принадлежа"},
		{"принадлежала", "принадлежа"},
		{"принадлежали", "принадлежа"},
		{"принадлежало", "принадлежа"},
		{"принадлежат", "принадлежат"},
		{"принадлежащие", "принадлежа"},
		{"принадлежащий", "принадлежа"},
		{"принадлежащих", "принадлежа"},
		{"принадлежит", "принадлеж"},
		{"принадлежите", "принадлеж"},
		{"принадлежностей", "принадлежн"},
		{"принадлежности", "принадлежн"},
		{"принадлежностью", "принадлежн"},
		{"принадлежу", "принадлеж"},
		{"принарядиться", "принаряд"},
		{"принес", "принес"},
		{"принесенного", "принесен"},
		{"принесенное", "принесен"},
		{"принесенные", "принесен"},
		{"принесенный", "принесен"},
		{"принесенных", "принесен"},
		{"принесено", "принес"},
		{"принесены", "принес"},
		{"принесет", "принесет"},
		{"принесете", "принесет"},
		{"принесешь", "принесеш"},
		{"принеси", "принес"},
		{"принесите", "принес"},
		{"принесла", "принесл"},
		{"принесли", "принесл"},
		{"принесло", "принесл"},
		{"принести", "принест"},
		{"принесу", "принес"},
		{"принесут", "принесут"},
		{"принеся", "прин"},
		{"принижающее", "принижа"},
		{"принижение", "принижен"},
		{"приниженная", "принижен"},
		{"приниженное", "принижен"},
		{"приниженную", "принижен"},
		{"принижено", "приниж"},
		{"приник", "приник"},
		{"приникла", "приникл"},
		{"приникшей", "приникш"},
		{"принимавшие", "принима"},
		{"принимает", "принима"},
		{"принимаете", "принима"},
		{"принимается", "принима"},
		{"принимаешь", "принима"},
		{"принимай", "принима"},
		{"принимайте", "принима"},
		{"принимал", "принима"},
		{"принимала", "принима"},
		{"принималась", "принима"},
		{"принимали", "принима"},
		{"принимались", "принима"},
		{"принимало", "принима"},
		{"принимался", "принима"},
		{"принимать", "принима"},
		{"приниматься", "принима"},
		{"принимаю", "принима"},
		{"принимаюсь", "принима"},
		{"принимают", "принима"},
		{"принимающим", "принима"},
		{"принимая", "приним"},
		{"принимаясь", "приним"},
		{"приносил", "принос"},
		{"приносили", "принос"},
		{"приносились", "принос"},
		{"приносилось", "принос"},
		{"приносимые", "приносим"},
		{"приносит", "принос"},
		{"приносить", "принос"},
		{"приносят", "принос"},
		{"приношением", "приношен"},
		{"приношу", "принош"},
		{"принудив", "принуд"},
		{"принудила", "принуд"},
		{"принудили", "принуд"},
		{"принудить", "принуд"},
		{"принуждал", "принужда"},
		{"принужден", "принужд"},
		{"принуждена", "принужд"},
		{"принуждение", "принужден"},
		{"принуждений", "принужден"},
		{"принужденным", "принужден"},
		{"принц", "принц"},
		{"принцип", "принцип"},
		{"принципах", "принцип"},
		{"принципиально", "принципиальн"},
		{"принципу", "принцип"},
		{"принципы", "принцип"},
		{"приняв", "приня"},
		{"принявшие", "приня"},
		{"принял", "приня"},
		{"приняла", "приня"},
		{"принялась", "приня"},
		{"приняли", "приня"},
		{"принялись", "приня"},
		{"приняло", "приня"},
		{"принялся", "приня"},
		{"принят", "прин"},
		{"принятая", "принят"},
		{"принятию", "принят"},
		{"принято", "принят"},
		{"приняты", "принят"},
		{"принятый", "принят"},
		{"принятых", "принят"},
		{"принять", "приня"},
		{"приняться", "приня"},
		{"приобрел", "приобрел"},
		{"приобрела", "приобрел"},
		{"приобрести", "приобрест"},
		{"приобресть", "приобрест"},
		{"приобретается", "приобрета"},
		{"приобретал", "приобрета"},
		{"приобретала", "приобрета"},
		{"приобреталось", "приобрета"},
		{"приобретался", "приобрета"},
		{"приобретать", "приобрета"},
		{"приобретаю", "приобрета"},
		{"приобретают", "приобрета"},
		{"приобретая", "приобрет"},
		{"приобретение", "приобретен"},
		{"приобретению", "приобретен"},
		{"приобретенного", "приобретен"},
		{"приобретенной", "приобретен"},
		{"приобщались", "приобща"},
		{"приоделась", "приодел"},
		{"приостановились", "приостанов"},
		{"приостановился", "приостанов"},
		{"приотворена", "приотвор"},
		{"приотворенная", "приотворен"},
		{"приотворил", "приотвор"},
		{"приотворилась", "приотвор"},
		{"приоткрытой", "приоткрыт"},
		{"припав", "припа"},
		{"припадает", "припада"},
		{"припадая", "припад"},
		{"припадем", "припад"},
		{"припадка", "припадк"},
		{"припадках", "припадк"},
		{"припадке", "припадк"},
		{"припадки", "припадк"},
		{"припадков", "припадк"},
		{"припадком", "припадк"},
		{"припадок", "припадок"},
		{"припадочек", "припадочек"},
		{"припадочного", "припадочн"},
		{"припадочный", "припадочн"},
		{"припал", "припа"},
		{"припала", "припа"},
		{"припас", "припас"},
		{"припасами", "припас"},
		{"припасен", "припас"},
		{"припасов", "припас"},
		{"припасы", "припас"},
		{"припахивает", "припахива"},
		{"припевах", "припев"},
		{"припеваючи", "припеваюч"},
		{"припер", "припер"},
		{"приписанное", "приписа"},
		{"приписать", "приписа"},
		{"приписываемые", "приписыва"},
		{"приписывает", "приписыва"},
		{"приписывал", "приписыва"},
		{"приписывают", "приписыва"},
		{"приплюснут", "приплюснут"},
		{"приподнимайтесь", "приподнима"},
		{"приподниматься", "приподнима"},
		{"приподнимаются", "приподнима"},
		{"приподнимая", "приподним"},
		{"приподнимаясь", "приподним"},
		{"приподняв", "приподня"},
		{"приподнявшись", "приподня"},
		{"приподнял", "приподня"},
		{"приподняла", "приподня"},
		{"приподнялась", "приподня"},
		{"приподняли", "приподня"},
		{"приподнялись", "приподня"},
		{"приподнялся", "приподня"},
		{"приподнятые", "приподнят"},
		{"приподнять", "приподня"},
		{"приподняться", "приподня"},
		{"припомажен", "припомаж"},
		{"припоминаемое", "припомина"},
		{"припоминал", "припомина"},
		{"припоминала", "припомина"},
		{"припоминался", "припомина"},
		{"припоминать", "припомина"},
		{"припоминаться", "припомина"},
		{"припоминая", "припомин"},
		{"припомни", "припомн"},
		{"припомнив", "припомн"},
		{"припомнил", "припомн"},
		{"припомнила", "припомн"},
		{"припомнилась", "припомн"},
		{"припомнилось", "припомн"},
		{"припомнился", "припомн"},
		{"припомните", "припомн"},
		{"припомнить", "припомн"},
		{"припомнишь", "припомн"},
		{"припомню", "припомн"},
		{"припрыгнул", "припрыгнул"},
		{"припрятал", "припрята"},
		{"припрятано", "припрята"},
		{"припрятать", "припрята"},
		{"припускает", "припуска"},
		{"припутали", "припута"},
		{"припухлые", "припухл"},
		{"припухшее", "припухш"},
		{"припухшем", "припухш"},
		{"припухшею", "припухш"},
		{"припухшие", "припухш"},
		{"припухшими", "припухш"},
		{"прирастала", "прираста"},
		{"приращение", "приращен"},
		{"приревнует", "приревн"},
		{"природа", "природ"},
		{"природе", "природ"},
		{"природная", "природн"},
		{"природное", "природн"},
		{"природной", "природн"},
		{"природным", "природн"},
		{"природой", "природ"},
		{"природу", "природ"},
		{"природы", "природ"},
		{"прирожденный", "прирожден"},
		{"прирос", "прирос"},
		{"приросли", "приросл"},
		{"приросло", "приросл"},
		{"приручил", "прируч"},
		{"присвататься", "присвата"},
		{"присватывался", "присватыва"},
		{"присвистнул", "присвистнул"},
		{"присвоить", "присво"},
		{"присев", "прис"},
		{"приседает", "приседа"},
		{"приседала", "приседа"},
		{"приседать", "приседа"},
		{"приседая", "присед"},
		{"присел", "присел"},
		{"присела", "присел"},
		{"присесть", "присест"},
		{"прискорбием", "прискорб"},
		{"прискорбию", "прискорб"},
		{"прислал", "присла"},
		{"прислала", "присла"},
		{"прислали", "присла"},
		{"прислан", "присла"},
		{"присланные", "присла"},
		{"присланных", "присла"},
		{"прислать", "присла"},
		{"прислоненная", "прислонен"},
		{"прислонил", "прислон"},
		{"прислонилась", "прислон"},
		{"прислонился", "прислон"},
		{"прислонясь", "прислон"},
		{"прислуга", "прислуг"},
		{"прислуге", "прислуг"},
		{"прислуги", "прислуг"},
		{"прислугой", "прислуг"},
		{"прислуживала", "прислужива"},
		{"прислуживать", "прислужива"},
		{"прислужница", "прислужниц"},
		{"прислушавшись", "прислуша"},
		{"прислушался", "прислуша"},
		{"прислушать", "прислуша"},
		{"прислушивавшийся", "прислушива"},
		{"прислушивается", "прислушива"},
		{"прислушивалась", "прислушива"},
		{"прислушивались", "прислушива"},
		{"прислушивался", "прислушива"},
		{"прислушиваться", "прислушива"},
		{"прислушиваются", "прислушива"},
		{"прислушиваясь", "прислушив"},
		{"присматривать", "присматрива"},
		{"присмиреет", "присмиреет"},
		{"присмирел", "присмирел"},
		{"присмирела", "присмирел"},
		{"присмирели", "присмирел"},
		{"присмотрит", "присмотр"},
		{"присмотрите", "присмотр"},
		{"приснившийся", "присн"},
		{"приснилась", "присн"},
		{"приснились", "присн"},
		{"приснилось", "присн"},
		{"приснился", "присн"},
		{"приснится", "присн"},
		{"присовокупить", "присовокуп"},
		{"присочиним", "присочин"},
		{"приспела", "приспел"},
		{"приспособились", "приспособ"},
		{"приспособился", "приспособ"},
		{"приспособить", "приспособ"},
		{"приспособлено", "приспособл"},
		{"пристав", "приста"},
		{"пристава", "пристав"},
		{"приставай", "пристава"},
		{"приставал", "пристава"},
		{"приставала", "пристава"},
		{"приставания", "приставан"},
		{"приставать", "пристава"},
		{"приставил", "пристав"},
		{"приставить", "пристав"},
		{"приставлен", "приставл"},
		{"приставленными", "приставлен"},
		{"приставу", "пристав"},
		{"пристает", "приста"},
		{"пристаешь", "приста"},
		{"пристал", "приста"},
		{"пристала", "приста"},
		{"пристало", "приста"},
		{"пристальнее", "пристальн"},
		{"пристальней", "пристальн"},
		{"пристально", "пристальн"},
		{"пристальном", "пристальн"},
		{"пристальность", "пристальн"},
		{"пристальный", "пристальн"},
		{"пристальным", "пристальн"},
		{"пристальными", "пристальн"},
		{"пристанет", "пристанет"},
		{"пристани", "пристан"},
		{"пристанище", "пристанищ"},
		{"пристань", "пристан"},
		{"пристают", "приста"},
		{"пристрастно", "пристрастн"},
		{"пристроены", "пристро"},
		{"пристроил", "пристро"},
		{"пристроили", "пристро"},
		{"пристроились", "пристро"},
		{"пристройками", "пристройк"},
		{"пристукнуло", "пристукнул"},
		{"приступил", "приступ"},
		{"приступила", "приступ"},
		{"приступим", "приступ"},
		{"приступить", "приступ"},
		{"пристяжная", "пристяжн"},
		{"присужден", "присужд"},
		{"присуждено", "присужд"},
		{"присутственного", "присутствен"},
		{"присутственное", "присутствен"},
		{"присутственном", "присутствен"},
		{"присутственные", "присутствен"},
		{"присутственным", "присутствен"},
		{"присутственных", "присутствен"},
		{"присутствие", "присутств"},
		{"присутствием", "присутств"},
		{"присутствии", "присутств"},
		{"присутствия", "присутств"},
		{"присутствовавших", "присутствова"},
		{"присутствовал", "присутствова"},
		{"присутствовала", "присутствова"},
		{"присутствовало", "присутствова"},
		{"присутствовать", "присутствова"},
		{"присутствует", "присутств"},
		{"присутствующим", "присутств"},
		{"присутствующих", "присутств"},
		{"присылаемые", "присыла"},
		{"присылает", "присыла"},
		{"присылал", "присыла"},
		{"присылали", "присыла"},
		{"присылать", "присыла"},
		{"присылки", "присылк"},
		{"присягнул", "присягнул"},
		{"присягу", "присяг"},
		{"присядет", "присядет"},
		{"присяду", "присяд"},
		{"присядь", "присяд"},
		{"присядьте", "присядьт"},
		{"присяжного", "присяжн"},
		{"притаившись", "прита"},
		{"притаились", "прита"},
		{"притаился", "прита"},
		{"притаиться", "прита"},
		{"притащи", "притащ"},
		{"притащит", "притащ"},
		{"притащить", "притащ"},
		{"притащу", "притащ"},
		{"притаясь", "прит"},
		{"притворе", "притвор"},
		{"притворена", "притвор"},
		{"притворил", "притвор"},
		{"притворила", "притвор"},
		{"притворилась", "притвор"},
		{"притворили", "притвор"},
		{"притворился", "притвор"},
		{"притворить", "притвор"},
		{"притвориться", "притвор"},
		{"притворной", "притворн"},
		{"притворным", "притворн"},
		{"притворства", "притворств"},
		{"притворство", "притворств"},
		{"притворщик", "притворщик"},
		{"притворяется", "притворя"},
		{"притворяла", "притворя"},
		{"притворялись", "притворя"},
		{"притворяться", "притворя"},
		{"притворяясь", "притвор"},
		{"притих", "прит"},
		{"притихла", "притихл"},
		{"притихли", "притихл"},
		{"притихнет", "притихнет"},
		{"притом", "прит"},
		{"притон", "притон"},
		{"приторности", "приторн"},
		{"приторный", "приторн"},
		{"притрогивалась", "притрогива"},
		{"притупились", "притуп"},
		{"притупленными", "притуплен"},
		{"притуши", "притуш"},
		{"притягивал", "притягива"},
		{"притягивая", "притягив"},
		{"притязания", "притязан"},
		{"притянуть", "притянут"},
		{"приударил", "приудар"},
		{"приумножают", "приумножа"},
		{"приуныл", "приун"},
		{"приучи", "приуч"},
		{"приучил", "приуч"},
		{"приучили", "приуч"},
		{"прихлопнуть", "прихлопнут"},
		{"прихлынувшей", "прихлынувш"},
		{"прихлынуло", "прихлынул"},
		{"приход", "приход"},
		{"прихода", "приход"},
		{"приходи", "приход"},
		{"приходившая", "приход"},
		{"приходивший", "приход"},
		{"приходивших", "приход"},
		{"приходил", "приход"},
		{"приходила", "приход"},
		{"приходилась", "приход"},
		{"приходили", "приход"},
		{"приходило", "приход"},
		{"приходилось", "приход"},
		{"приходит", "приход"},
		{"приходите", "приход"},
		{"приходится", "приход"},
		{"приходить", "приход"},
		{"приходишь", "приход"},
		{"приходо", "приход"},
		{"приходом", "приход"},
		{"приходя", "приход"},
		{"приходят", "приход"},
		{"приходящих", "приходя"},
		{"прихожая", "прихож"},
		{"прихожей", "прихож"},
		{"прихожу", "прихож"},
		{"прихожую", "прихож"},
		{"прихотей", "прихот"},
		{"прихоти", "прихот"},
		{"прихотливо", "прихотлив"},
		{"прихотливой", "прихотлив"},
		{"прихотливый", "прихотлив"},
		{"прихрамывал", "прихрамыва"},
		{"прицеливается", "прицелива"},
		{"прицепилось", "прицеп"},
		{"прицепился", "прицеп"},
		{"прицепиться", "прицеп"},
		{"причалены", "причал"},
		{"причастись", "причаст"},
		{"причастных", "причастн"},
		{"причащение", "причащен"},
		{"причем", "прич"},
		{"причесал", "причеса"},
		{"причесалась", "причеса"},
		{"причесан", "причеса"},
		{"причесана", "причеса"},
		{"причесанные", "причеса"},
		{"причесанный", "причеса"},
		{"причесанным", "причеса"},
		{"причесать", "причеса"},
		{"прическа", "прическ"},
		{"прическе", "прическ"},
		{"прически", "прическ"},
		{"прической", "прическ"},
		{"причесывает", "причесыва"},
		{"причесывал", "причесыва"},
		{"причешутся", "причешут"},
		{"причин", "причин"},
		{"причина", "причин"},
		{"причинам", "причин"},
		{"причине", "причин"},
		{"причиной", "причин"},
		{"причиною", "причин"},
		{"причину", "причин"},
		{"причины", "причин"},
		{"причисленных", "причислен"},
		{"причислила", "причисл"},
		{"причисляешь", "причисля"},
		{"причитается", "причита"},
		{"причитал", "причита"},
		{"причитала", "причита"},
		{"причитаньях", "причитан"},
		{"причитающиеся", "причита"},
		{"причтется", "причтет"},
		{"причуд", "причуд"},
		{"причуды", "причуд"},
		{"пришедшего", "пришедш"},
		{"пришедши", "пришедш"},
		{"пришедшим", "пришедш"},
		{"пришедших", "пришедш"},
		{"пришел", "пришел"},
		{"пришелся", "пришел"},
		{"пришивать", "пришива"},
		{"пришивая", "пришив"},
		{"пришили", "приш"},
		{"пришить", "приш"},
		{"пришла", "пришл"},
		{"пришлем", "пришл"},
		{"пришлет", "пришлет"},
		{"пришли", "пришл"},
		{"пришлись", "пришл"},
		{"пришлите", "пришл"},
		{"пришло", "пришл"},
		{"пришлось", "пришл"},
		{"пришлю", "пришл"},
		{"пришлют", "пришлют"},
		{"пришоль", "пришол"},
		{"пришпорить", "пришпор"},
		{"прищелкивать", "прищелкива"},
		{"прищурившись", "прищур"},
		{"прищурил", "прищур"},
		{"прищурилась", "прищур"},
		{"прищурился", "прищур"},
		{"прищуря", "прищур"},
		{"приют", "приют"},
		{"приюта", "приют"},
		{"приютиться", "приют"},
		{"приютов", "приют"},
		{"приятелей", "приятел"},
		{"приятелем", "приятел"},
		{"приятели", "приятел"},
		{"приятель", "приятел"},
		{"приятельница", "приятельниц"},
		{"приятельницею", "приятельниц"},
		{"приятельницы", "приятельниц"},
		{"приятельские", "приятельск"},
		{"приятелю", "приятел"},
		{"приятеля", "приятел"},
		{"приятелям", "приятел"},
		{"приятелями", "приятел"},
		{"приятелях", "приятел"},
		{"приятен", "прият"},
		{"приятна", "приятн"},
		{"приятнее", "приятн"},
		{"приятнейшего", "приятн"},
		{"приятнейшие", "приятн"},
		{"приятно", "приятн"},
		{"приятного", "приятн"},
		{"приятное", "приятн"},
		{"приятной", "приятн"},
		{"приятном", "приятн"},
		{"приятности", "приятност"},
		{"приятною", "приятн"},
		{"приятную", "приятн"},
		{"приятные", "приятн"},
		{"приятный", "приятн"},
		{"приятным", "приятн"},
		{"приятными", "приятн"},
		{"приятных", "приятн"},
		{"про", "про"},
		{"пробавляться", "пробавля"},
		{"пробавляются", "пробавля"},
		{"пробалтывался", "пробалтыва"},
		{"пробегает", "пробега"},
		{"пробегал", "пробега"},
		{"пробегала", "пробега"},
		{"пробегающий", "пробега"},
		{"пробегая", "пробег"},
		{"пробежал", "пробежа"},
		{"пробежала", "пробежа"},
		{"пробежали", "пробежа"},
		{"пробежало", "пробежа"},
		{"пробежит", "пробеж"},
		{"пробиваем", "пробива"},
		{"пробивается", "пробива"},
		{"пробивалась", "пробива"},
		{"пробивался", "пробива"},
		{"пробивать", "пробива"},
		{"пробившись", "проб"},
		{"пробил", "проб"},
		{"пробили", "проб"},
		{"пробило", "проб"},
		{"пробирается", "пробира"},
		{"пробирались", "пробира"},
		{"пробирался", "пробира"},
		{"пробираться", "пробира"},
		{"пробираясь", "пробир"},
		{"пробить", "проб"},
		{"пробиться", "проб"},
		{"пробка", "пробк"},
		{"проблеск", "проблеск"},
		{"проблеска", "проблеск"},
		{"пробовал", "пробова"},
		{"пробовала", "пробова"},
		{"пробовали", "пробова"},
		{"пробои", "пробо"},
		{"пробок", "пробок"},
		{"проболтался", "проболта"},
		{"проболтаться", "проболта"},
		{"пробор", "пробор"},
		{"пробормотал", "пробормота"},
		{"пробормотала", "пробормота"},
		{"пробормотать", "пробормота"},
		{"пробором", "пробор"},
		{"пробочка", "пробочк"},
		{"пробравшись", "пробра"},
		{"пробралась", "пробра"},
		{"пробрался", "пробра"},
		{"пробу", "проб"},
		{"пробудет", "пробудет"},
		{"пробудившегося", "пробуд"},
		{"пробудился", "пробуд"},
		{"пробудись", "пробуд"},
		{"пробуду", "пробуд"},
		{"пробудь", "пробуд"},
		{"пробудясь", "пробуд"},
		{"пробуждавшимися", "пробужда"},
		{"пробуждались", "пробужда"},
		{"пробуждаться", "пробужда"},
		{"пробуждающейся", "пробужда"},
		{"пробуждения", "пробужден"},
		{"пробужденного", "пробужден"},
		{"пробужденные", "пробужден"},
		{"пробуя", "пробу"},
		{"пробывшее", "проб"},
		{"пробыл", "проб"},
		{"пробыть", "проб"},
		{"провале", "провал"},
		{"проваливается", "провалива"},
		{"провалившимся", "провал"},
		{"провалилась", "провал"},
		{"провалился", "провал"},
		{"провались", "прова"},
		{"провалиться", "провал"},
		{"провалишься", "провал"},
		{"провалом", "провал"},
		{"провалялся", "проваля"},
		{"провалятся", "провал"},
		{"проведать", "проведа"},
		{"проведем", "провед"},
		{"проведена", "провед"},
		{"проведение", "проведен"},
		{"проведенного", "проведен"},
		{"проведенный", "проведен"},
		{"проведет", "проведет"},
		{"проведу", "провед"},
		{"проведя", "провед"},
		{"провезут", "провезут"},
		{"провел", "провел"},
		{"провела", "провел"},
		{"проверил", "провер"},
		{"проверить", "провер"},
		{"провесная", "провесн"},
		{"провести", "провест"},
		{"проветрись", "проветр"},
		{"провиантский", "провиантск"},
		{"провиантского", "провиантск"},
		{"провиантскому", "провиантск"},
		{"провидел", "провидел"},
		{"провидение", "провиден"},
		{"провидением", "провиден"},
		{"провидения", "провиден"},
		{"провидит", "провид"},
		{"провижу", "провиж"},
		{"провизии", "провиз"},
		{"провизию", "провиз"},
		{"провинциалка", "провинциалк"},
		{"провинциальной", "провинциальн"},
		{"провинциальный", "провинциальн"},
		{"провинции", "провинц"},
		{"провинцию", "провинц"},
		{"провод", "провод"},
		{"провода", "провод"},
		{"проводи", "провод"},
		{"проводив", "провод"},
		{"проводил", "провод"},
		{"проводила", "провод"},
		{"проводили", "провод"},
		{"проводим", "провод"},
		{"проводимых", "проводим"},
		{"проводит", "провод"},
		{"проводите", "провод"},
		{"проводится", "провод"},
		{"проводить", "провод"},
		{"провожавшего", "провожа"},
		{"провожает", "провожа"},
		{"провожай", "провожа"},
		{"провожайте", "провожа"},
		{"провожал", "провожа"},
		{"провожала", "провожа"},
		{"провожали", "провожа"},
		{"провожать", "провожа"},
		{"провожаю", "провожа"},
		{"провожают", "провожа"},
		{"провожая", "провож"},
		{"провождение", "провожден"},
		{"провожу", "провож"},
		{"провозгласил", "провозглас"},
		{"провозили", "провоз"},
		{"провозит", "провоз"},
		{"провозишься", "провоз"},
		{"проволок", "проволок"},
		{"проворен", "провор"},
		{"проворковать", "проворкова"},
		{"проворная", "проворн"},
		{"проворно", "проворн"},
		{"проворное", "проворн"},
		{"проворным", "проворн"},
		{"проворонил", "проворон"},
		{"проворством", "проворств"},
		{"проворчав", "проворча"},
		{"проворчал", "проворча"},
		{"проворчат", "проворчат"},
		{"проврался", "провра"},
		{"провспоминаешь", "провспомина"},
		{"проглотил", "проглот"},
		{"проглотить", "проглот"},
		{"проглочу", "проглоч"},
		{"проглядели", "проглядел"},
		{"проглядывала", "проглядыва"},
		{"проглядывало", "проглядыва"},
		{"проглянула", "проглянул"},
		{"прогнал", "прогна"},
		{"прогнала", "прогна"},
		{"прогнали", "прогна"},
		{"прогналь", "прогнал"},
		{"прогнать", "прогна"},
		{"прогневали", "прогнева"},
		{"проговаривается", "проговарива"},
		{"проговариваться", "проговарива"},
		{"проговаривая", "проговарив"},
		{"проговорено", "проговор"},
		{"проговорив", "проговор"},
		{"проговорившись", "проговор"},
		{"проговорил", "проговор"},
		{"проговорила", "проговор"},
		{"проговорилось", "проговор"},
		{"проговорился", "проговор"},
		{"проговорится", "проговор"},
		{"проговориться", "проговор"},
		{"проговорюсь", "проговор"},
		{"проголодалась", "проголода"},
		{"прогонами", "прогон"},
		{"прогоните", "прогон"},
		{"прогоны", "прогон"},
		{"прогоню", "прогон"},
		{"прогоняли", "прогоня"},
		{"прогонят", "прогон"},
		{"прогонять", "прогоня"},
		{"программ", "программ"},
		{"программе", "программ"},
		{"программу", "программ"},
		{"прогресс", "прогресс"},
		{"прогресса", "прогресс"},
		{"прогрессивно", "прогрессивн"},
		{"прогрессивного", "прогрессивн"},
		{"прогрессивные", "прогрессивн"},
		{"прогрессиста", "прогрессист"},
		{"прогрессистов", "прогрессист"},
		{"прогрессисты", "прогрессист"},
		{"прогрессом", "прогресс"},
		{"прогрессу", "прогресс"},
		{"прогуливается", "прогулива"},
		{"прогуливаясь", "прогулив"},
		{"прогулка", "прогулк"},
		{"прогулках", "прогулк"},
		{"прогулке", "прогулк"},
		{"прогулки", "прогулк"},
		{"прогулок", "прогулок"},
		{"прогуляемся", "прогуля"},
		{"прогуляла", "прогуля"},
		{"прогуляться", "прогуля"},
		{"продав", "прода"},
		{"продавайте", "продава"},
		{"продавал", "продава"},
		{"продавало", "продава"},
		{"продавать", "продава"},
		{"продаваться", "продава"},
		{"продавец", "продавец"},
		{"продавить", "продав"},
		{"продавленном", "продавлен"},
		{"продавца", "продавц"},
		{"продадут", "продадут"},
		{"продаем", "прода"},
		{"продает", "прода"},
		{"продается", "прода"},
		{"продаешь", "прода"},
		{"продажей", "продаж"},
		{"продажи", "продаж"},
		{"продажной", "продажн"},
		{"продажу", "продаж"},
		{"продай", "прода"},
		{"продал", "прода"},
		{"продала", "прода"},
		{"продали", "прода"},
		{"продан", "прода"},
		{"проданный", "прода"},
		{"продано", "прода"},
		{"продаст", "продаст"},
		{"продать", "прода"},
		{"продашь", "продаш"},
		{"продают", "прода"},
		{"продаются", "прода"},
		{"продеваемой", "продева"},
		{"продеваемые", "продева"},
		{"продевала", "продева"},
		{"продевая", "продев"},
		{"проделка", "проделк"},
		{"проделки", "проделк"},
		{"продержали", "продержа"},
		{"продиктует", "продикт"},
		{"продиктую", "продикт"},
		{"продиралась", "продира"},
		{"продлил", "продл"},
		{"продлить", "продл"},
		{"продлятся", "продл"},
		{"продовольствие", "продовольств"},
		{"продовольствии", "продовольств"},
		{"продолговатым", "продолговат"},
		{"продолжавшему", "продолжа"},
		{"продолжавшийся", "продолжа"},
		{"продолжает", "продолжа"},
		{"продолжаете", "продолжа"},
		{"продолжается", "продолжа"},
		{"продолжай", "продолжа"},
		{"продолжайте", "продолжа"},
		{"продолжал", "продолжа"},
		{"продолжала", "продолжа"},
		{"продолжалась", "продолжа"},
		{"продолжали", "продолжа"},
		{"продолжалось", "продолжа"},
		{"продолжался", "продолжа"},
		{"продолжать", "продолжа"},
		{"продолжаться", "продолжа"},
		{"продолжаю", "продолжа"},
		{"продолжают", "продолжа"},
		{"продолжаются", "продолжа"},
		{"продолжая", "продолж"},
		{"продолжена", "продолж"},
		{"продолжение", "продолжен"},
		{"продолжением", "продолжен"},
		{"продолжения", "продолжен"},
		{"продолженье", "продолжен"},
		{"продолжил", "продолж"},
		{"продолжим", "продолж"},
		{"продолжительно", "продолжительн"},
		{"продолжительного", "продолжительн"},
		{"продолжительное", "продолжительн"},
		{"продолжительный", "продолжительн"},
		{"продолжительным", "продолжительн"},
		{"продолжится", "продолж"},
		{"продранные", "продра"},
		{"продранным", "продра"},
		{"продувная", "продувн"},
		{"продувной", "продувн"},
		{"продувных", "продувн"},
		{"продукт", "продукт"},
		{"продуктов", "продукт"},
		{"продукты", "продукт"},
		{"продумал", "продума"},
		{"продумала", "продума"},
		{"проедет", "проедет"},
		{"проедусь", "проед"},
		{"проезда", "проезд"},
		{"проезду", "проезд"},
		{"проезжавшей", "проезжа"},
		{"проезжавший", "проезжа"},
		{"проезжавшую", "проезжа"},
		{"проезжает", "проезжа"},
		{"проезжающих", "проезжа"},
		{"проезжая", "проезж"},
		{"проезжий", "проезж"},
		{"проект", "проект"},
		{"проекты", "проект"},
		{"проел", "проел"},
		{"проехал", "проеха"},
		{"проехала", "проеха"},
		{"проехали", "проеха"},
		{"проехать", "проеха"},
		{"прождала", "прожда"},
		{"прожег", "прожег"},
		{"прожектор", "прожектор"},
		{"прожжено", "прожж"},
		{"прожжет", "прожжет"},
		{"проживавшего", "прожива"},
		{"проживаем", "прожива"},
		{"проживает", "прожива"},
		{"проживаете", "прожива"},
		{"проживал", "прожива"},
		{"проживали", "прожива"},
		{"проживать", "прожива"},
		{"проживают", "прожива"},
		{"проживем", "прожив"},
		{"проживет", "проживет"},
		{"проживете", "проживет"},
		{"проживешь", "проживеш"},
		{"проживу", "прожив"},
		{"проживут", "проживут"},
		{"прожил", "прож"},
		{"прожила", "прож"},
		{"прожили", "прож"},
		{"прожита", "прожит"},
		{"прожитой", "прожит"},
		{"прожиток", "прожиток"},
		{"прожитую", "прожит"},
		{"прожить", "прож"},
		{"прожитье", "прожит"},
		{"прожорливым", "прожорлив"},
		{"прожужжит", "прожужж"},
		{"прозвал", "прозва"},
		{"прозвали", "прозва"},
		{"прозвание", "прозван"},
		{"прозвенели", "прозвенел"},
		{"прозвучала", "прозвуча"},
		{"прозвучали", "прозвуча"},
		{"прозвучит", "прозвуч"},
		{"прозе", "проз"},
		{"прозирать", "прозира"},
		{"прозорлив", "прозорл"},
		{"прозорливейший", "прозорлив"},
		{"прозоров", "прозор"},
		{"прозорова", "прозоров"},
		{"прозоровых", "прозоров"},
		{"прозрачная", "прозрачн"},
		{"прозрачное", "прозрачн"},
		{"прозрачной", "прозрачн"},
		{"прозрачный", "прозрачн"},
		{"прозрачным", "прозрачн"},
		{"прозревал", "прозрева"},
		{"прозрения", "прозрен"},
		{"прозреть", "прозрет"},
		{"прозябает", "прозяба"},
		{"прозябал", "прозяба"},
		{"проиграл", "проигра"},
		{"проиграла", "проигра"},
		{"проигрались", "проигра"},
		{"проигрался", "проигра"},
		{"проиграно", "проигра"},
		{"проигрыше", "проигрыш"},
		{"произведен", "произвед"},
		{"произведена", "произвед"},
		{"произведение", "произведен"},
		{"произведением", "произведен"},
		{"произведении", "произведен"},
		{"произведений", "произведен"},
		{"произведения", "произведен"},
		{"произведенное", "произведен"},
		{"произведенный", "произведен"},
		{"произведено", "произвед"},
		{"произведет", "произведет"},
		{"произвел", "произвел"},
		{"произвела", "произвел"},
		{"произвели", "произвел"},
		{"произвело", "произвел"},
		{"произвести", "произвест"},
		{"производивший", "производ"},
		{"производил", "производ"},
		{"производило", "производ"},
		{"производилось", "производ"},
		{"производился", "производ"},
		{"производит", "производ"},
		{"производительности", "производительн"},
		{"производить", "производ"},
		{"производишь", "производ"},
		{"производят", "производ"},
		{"произвол", "произвол"},
		{"произволу", "произвол"},
		{"произвольно", "произвольн"},
		{"произнес", "произнес"},
		{"произнесенные", "произнесен"},
		{"произнесено", "произнес"},
		{"произнесены", "произнес"},
		{"произнесла", "произнесл"},
		{"произнести", "произнест"},
		{"произносили", "произнос"},
		{"произносимые", "произносим"},
		{"произносит", "произнос"},
		{"произносить", "произнос"},
		{"произнося", "произн"},
		{"произносят", "произнос"},
		{"произойдет", "произойдет"},
		{"произойти", "произойт"},
		{"произошел", "произошел"},
		{"произошла", "произошл"},
		{"произошли", "произошл"},
		{"произошло", "произошл"},
		{"происходившую", "происход"},
		{"происходил", "происход"},
		{"происходила", "происход"},
		{"происходили", "происход"},
		{"происходило", "происход"},
		{"происходит", "происход"},
		{"происходить", "происход"},
		{"происходят", "происход"},
		{"происходящее", "происходя"},
		{"происхождение", "происхожден"},
		{"происхождения", "происхожден"},
		{"происшедшим", "происшедш"},
		{"происшедшую", "происшедш"},
		{"происшествие", "происшеств"},
		{"происшествием", "происшеств"},
		{"происшествии", "происшеств"},
		{"происшествий", "происшеств"},
		{"происшествия", "происшеств"},
		{"пройдем", "пройд"},
		{"пройдемте", "пройдемт"},
		{"пройденное", "пройден"},
		{"пройдет", "пройдет"},
		{"пройдете", "пройдет"},
		{"пройдется", "пройдет"},
		{"пройдешь", "пройдеш"},
		{"пройди", "пройд"},
		{"пройдите", "пройд"},
		{"пройдохе", "пройдох"},
		{"пройдохой", "пройдох"},
		{"пройду", "пройд"},
		{"пройдусь", "пройд"},
		{"пройдут", "пройдут"},
		{"пройдя", "пройд"},
		{"пройдясь", "пройд"},
		{"пройти", "пройт"},
		{"проказ", "проказ"},
		{"проказа", "проказ"},
		{"проказил", "проказ"},
		{"проказить", "проказ"},
		{"проказнику", "проказник"},
		{"прокаркал", "прокарка"},
		{"прокатимся", "прокат"},
		{"прокатимтесь", "прокатимт"},
		{"прокатитесь", "прокат"},
		{"прокатиться", "прокат"},
		{"проклинает", "проклина"},
		{"проклинал", "проклина"},
		{"проклинала", "проклина"},
		{"проклинали", "проклина"},
		{"проклинать", "проклина"},
		{"проклял", "прокля"},
		{"проклянешь", "проклянеш"},
		{"проклясть", "прокляст"},
		{"проклят", "прокл"},
		{"проклятая", "проклят"},
		{"проклятие", "проклят"},
		{"проклятием", "проклят"},
		{"проклятий", "проклят"},
		{"проклятию", "проклят"},
		{"проклятия", "проклят"},
		{"проклятиями", "проклят"},
		{"проклято", "проклят"},
		{"проклятого", "проклят"},
		{"проклятое", "проклят"},
		{"проклятой", "проклят"},
		{"проклятом", "проклят"},
		{"проклятую", "проклят"},
		{"прокляты", "проклят"},
		{"проклятые", "проклят"},
		{"проклятый", "проклят"},
		{"проклятым", "проклят"},
		{"проклятых", "проклят"},
		{"проклятье", "проклят"},
		{"проклятья", "проклят"},
		{"проконсула", "проконсул"},
		{"прокормить", "прокорм"},
		{"прокормиться", "прокорм"},
		{"прокормлю", "прокормл"},
		{"прокофий", "прокоф"},
		{"прокофьич", "прокофьич"},
		{"прокофьича", "прокофьич"},
		{"прокофьичем", "прокофьич"},
		{"прокричал", "прокрича"},
		{"прокричала", "прокрича"},
		{"проку", "прок"},
		{"прокурора", "прокурор"},
		{"пролегли", "пролегл"},
		{"пролежал", "пролежа"},
		{"пролежала", "пролежа"},
		{"пролежали", "пролежа"},
		{"пролежало", "пролежа"},
		{"пролежит", "пролеж"},
		{"пролез", "пролез"},
		{"пролезайте", "пролеза"},
		{"пролезают", "пролеза"},
		{"пролезая", "пролез"},
		{"пролезть", "пролезт"},
		{"пролей", "прол"},
		{"пролетает", "пролета"},
		{"пролетала", "пролета"},
		{"пролетарий", "пролетар"},
		{"пролетария", "пролетар"},
		{"пролетевший", "пролетевш"},
		{"пролетевших", "пролетевш"},
		{"пролетел", "пролетел"},
		{"пролетела", "пролетел"},
		{"пролетит", "пролет"},
		{"пролетки", "пролетк"},
		{"пролив", "прол"},
		{"проливать", "пролива"},
		{"проливают", "пролива"},
		{"проливая", "пролив"},
		{"проливной", "проливн"},
		{"пролил", "прол"},
		{"пролилась", "прол"},
		{"пролита", "пролит"},
		{"пролитая", "пролит"},
		{"пролитую", "пролит"},
		{"пролог", "пролог"},
		{"пролога", "пролог"},
		{"проложат", "проложат"},
		{"проложил", "пролож"},
		{"проложить", "пролож"},
		{"пролюбить", "пролюб"},
		{"промалчивал", "промалчива"},
		{"промах", "пром"},
		{"промаха", "промах"},
		{"промахнулся", "промахнул"},
		{"промаху", "промах"},
		{"промедленья", "промедлен"},
		{"промедлить", "промедл"},
		{"промеж", "промеж"},
		{"промежутками", "промежутк"},
		{"промежутках", "промежутк"},
		{"промежуток", "промежуток"},
		{"промежуточная", "промежуточн"},
		{"промелькнет", "промелькнет"},
		{"промелькнувшею", "промелькнувш"},
		{"промелькнула", "промелькнул"},
		{"промелькнуло", "промелькнул"},
		{"променял", "променя"},
		{"променяла", "променя"},
		{"променять", "променя"},
		{"променяю", "променя"},
		{"прометеев", "промете"},
		{"промозглым", "промозгл"},
		{"промозолил", "промозол"},
		{"промокший", "промокш"},
		{"промолвил", "промолв"},
		{"промолвила", "промолв"},
		{"промолвили", "промолв"},
		{"промолвить", "промолв"},
		{"промолчал", "промолча"},
		{"промолчала", "промолча"},
		{"промочит", "промоч"},
		{"промучился", "промуч"},
		{"промысл", "промысл"},
		{"промысла", "промысл"},
		{"промычал", "промыча"},
		{"промычит", "промыч"},
		{"промышленник", "промышленник"},
		{"промышленниками", "промышленник"},
		{"промышленников", "промышленник"},
		{"промышляет", "промышля"},
		{"промышляют", "промышля"},
		{"промышляющее", "промышля"},
		{"промямлил", "промямл"},
		{"пронес", "пронес"},
		{"пронесет", "пронесет"},
		{"пронесется", "пронесет"},
		{"пронесла", "пронесл"},
		{"пронеслась", "пронесл"},
		{"пронесли", "пронесл"},
		{"пронеслось", "пронесл"},
		{"пронесся", "пронес"},
		{"пронесу", "пронес"},
		{"пронзал", "пронза"},
		{"пронзали", "пронза"},
		{"пронзающий", "пронза"},
		{"пронзающим", "пронза"},
		{"пронзенная", "пронзен"},
		{"пронзенный", "пронзен"},
		{"пронзенными", "пронзен"},
		{"пронзи", "пронз"},
		{"пронзил", "пронз"},
		{"пронзили", "пронз"},
		{"пронзило", "пронз"},
		{"пронзиль", "пронзил"},
		{"пронзительные", "пронзительн"},
		{"пронзительным", "пронзительн"},
		{"пронзительных", "пронзительн"},
		{"пронизан", "прониза"},
		{"пронизывает", "пронизыва"},
		{"проник", "проник"},
		{"проникал", "проника"},
		{"проникать", "проника"},
		{"проникающим", "проника"},
		{"проникло", "проникл"},
		{"проникнет", "проникнет"},
		{"проникнув", "проникнув"},
		{"проникнут", "проникнут"},
		{"проникнута", "проникнут"},
		{"проникнуты", "проникнут"},
		{"проникнуть", "проникнут"},
		{"проникся", "проник"},
		{"проникшись", "проникш"},
		{"проницал", "проница"},
		{"проницательнее", "проницательн"},
		{"проницательного", "проницательн"},
		{"проницательности", "проницательн"},
		{"проницательным", "проницательн"},
		{"проницающим", "проница"},
		{"проносили", "пронос"},
		{"проносилось", "пронос"},
		{"проносит", "пронос"},
		{"проносить", "пронос"},
		{"проносящейся", "пронося"},
		{"пронюхал", "пронюха"},
		{"пронюхают", "пронюха"},
		{"пропаганд", "пропаганд"},
		{"пропаганде", "пропаганд"},
		{"пропагандировать", "пропагандирова"},
		{"пропагандного", "пропагандн"},
		{"пропадает", "пропада"},
		{"пропадай", "пропада"},
		{"пропадал", "пропада"},
		{"пропадала", "пропада"},
		{"пропадали", "пропада"},
		{"пропадало", "пропада"},
		{"пропадать", "пропада"},
		{"пропадают", "пропада"},
		{"пропадающие", "пропада"},
		{"пропадающих", "пропада"},
		{"пропадет", "пропадет"},
		{"пропадете", "пропадет"},
		{"пропадешь", "пропадеш"},
		{"пропаду", "пропад"},
		{"пропадут", "пропадут"},
		{"пропал", "пропа"},
		{"пропала", "пропа"},
		{"пропали", "пропа"},
		{"пропало", "пропа"},
		{"пропастей", "пропаст"},
		{"пропасти", "пропаст"},
		{"пропасть", "пропаст"},
		{"пропастью", "пропаст"},
		{"пропастями", "пропаст"},
		{"пропастях", "пропаст"},
		{"пропащего", "пропа"},
		{"пропащий", "пропа"},
		{"пропев", "проп"},
		{"пропекать", "пропека"},
		{"пропел", "пропел"},
		{"пропела", "пропел"},
		{"пропели", "пропел"},
		{"пропеллер", "пропеллер"},
		{"пропетая", "пропет"},
		{"пропивал", "пропива"},
		{"пропил", "проп"},
		{"прописала", "прописа"},
		{"прописанной", "прописа"},
		{"прописываешь", "прописыва"},
		{"пропитан", "пропита"},
		{"пропитанном", "пропита"},
		{"пропитано", "пропита"},
		{"пропитаны", "пропита"},
		{"проплакала", "проплака"},
		{"проплывет", "проплывет"},
		{"проплывший", "пропл"},
		{"проплыли", "пропл"},
		{"проповедовал", "проповедова"},
		{"проповедовали", "проповедова"},
		{"проповедует", "проповед"},
		{"проповедуется", "проповед"},
		{"пропоет", "пропоет"},
		{"проползет", "проползет"},
		{"пропускал", "пропуска"},
		{"пропускающем", "пропуска"},
		{"пропускающим", "пропуска"},
		{"пропуская", "пропуск"},
		{"пропусти", "пропуст"},
		{"пропустив", "пропуст"},
		{"пропустил", "пропуст"},
		{"пропустили", "пропуст"},
		{"пропустит", "пропуст"},
		{"пропустить", "пропуст"},
		{"пропустят", "пропуст"},
		{"пропущенная", "пропущен"},
		{"пропыленные", "пропылен"},
		{"пропьет", "пропьет"},
		{"пропятие", "пропят"},
		{"прорва", "прорв"},
		{"прорвалось", "прорва"},
		{"прорвался", "прорва"},
		{"проревел", "проревел"},
		{"прорезывались", "прорезыва"},
		{"прорепетировать", "прорепетирова"},
		{"прореха", "прорех"},
		{"прорехою", "прорех"},
		{"прореху", "прорех"},
		{"пророк", "пророк"},
		{"пророка", "пророк"},
		{"пророком", "пророк"},
		{"проронил", "пророн"},
		{"проронила", "пророн"},
		{"проронит", "пророн"},
		{"проронить", "пророн"},
		{"пророчат", "пророчат"},
		{"пророческая", "пророческ"},
		{"пророчески", "пророческ"},
		{"пророческий", "пророческ"},
		{"пророчества", "пророчеств"},
		{"пророчество", "пророчеств"},
		{"пророчит", "пророч"},
		{"прорубил", "проруб"},
		{"прорывает", "прорыва"},
		{"прорывами", "прорыв"},
		{"прорывать", "прорыва"},
		{"прорыве", "прорыв"},
		{"просватали", "просвата"},
		{"просветился", "просвет"},
		{"просветите", "просвет"},
		{"просветлевшее", "просветлевш"},
		{"просветлеет", "просветлеет"},
		{"просветлели", "просветлел"},
		{"просветлении", "просветлен"},
		{"просвечивалось", "просвечива"},
		{"просвещение", "просвещен"},
		{"просвещению", "просвещен"},
		{"просвещения", "просвещен"},
		{"просвещенной", "просвещен"},
		{"просвещенный", "просвещен"},
		{"просвещенья", "просвещен"},
		{"просвирне", "просвирн"},
		{"просеванье", "просеван"},
		{"проседью", "просед"},
		{"просеет", "просеет"},
		{"проселки", "проселк"},
		{"проселок", "проселок"},
		{"просидев", "просид"},
		{"просидел", "просидел"},
		{"просидела", "просидел"},
		{"просидели", "просидел"},
		{"просидеть", "просидет"},
		{"просиди", "просид"},
		{"просиживала", "просижива"},
		{"просиживаю", "просижива"},
		{"просил", "прос"},
		{"просила", "прос"},
		{"просилась", "прос"},
		{"просили", "прос"},
		{"просился", "прос"},
		{"просиль", "просил"},
		{"просим", "прос"},
		{"просимой", "просим"},
		{"просипел", "просипел"},
		{"просит", "прос"},
		{"просите", "прос"},
		{"просителей", "просител"},
		{"просители", "просител"},
		{"просительных", "просительн"},
		{"проситесь", "прос"},
		{"просится", "прос"},
		{"просить", "прос"},
		{"просишь", "прос"},
		{"просияла", "просия"},
		{"проскакать", "проскака"},
		{"проскакивать", "проскакива"},
		{"просквозит", "просквоз"},
		{"проскользнет", "проскользнет"},
		{"проскользнул", "проскользнул"},
		{"проскользнула", "проскользнул"},
		{"проскользнуть", "проскользнут"},
		{"проскрежетал", "проскрежета"},
		{"проскучает", "проскуча"},
		{"прославить", "прослав"},
		{"прославленный", "прославлен"},
		{"проследил", "прослед"},
		{"проследовали", "проследова"},
		{"прослезились", "прослез"},
		{"прослезился", "прослез"},
		{"прослезится", "прослез"},
		{"прослужил", "прослуж"},
		{"прослужили", "прослуж"},
		{"прослужит", "прослуж"},
		{"прослушал", "прослуша"},
		{"прослушали", "прослуша"},
		{"прослушать", "прослуша"},
		{"прослышал", "прослыша"},
		{"прослышали", "прослыша"},
		{"прослышать", "прослыша"},
		{"просмотрел", "просмотрел"},
		{"просмотреть", "просмотрет"},
		{"просмотришь", "просмотр"},
		{"проснетесь", "проснет"},
		{"проснется", "проснет"},
		{"проснешься", "проснеш"},
		{"проснись", "просн"},
		{"проснитесь", "просн"},
		{"проснувшаяся", "проснувш"},
		{"проснувшегося", "проснувш"},
		{"проснувшиеся", "проснувш"},
		{"проснувшись", "проснувш"},
		{"проснулась", "проснул"},
		{"проснулись", "проснул"},
		{"проснулся", "проснул"},
		{"проснусь", "просн"},
		{"проснуться", "проснут"},
		{"просо", "прос"},
		{"проспал", "проспа"},
		{"проспали", "проспа"},
		{"проспать", "проспа"},
		{"проспект", "проспект"},
		{"проспекта", "проспект"},
		{"проспекте", "проспект"},
		{"проспекту", "проспект"},
		{"просроченное", "просрочен"},
		{"просроченные", "просрочен"},
		{"просроченный", "просрочен"},
		{"просрочить", "просроч"},
		{"прост", "прост"},
		{"проста", "прост"},
		{"простаковых", "простаков"},
		{"простая", "прост"},
		{"простегивала", "простегива"},
		{"простейшей", "прост"},
		{"простейшем", "прост"},
		{"простейшим", "прост"},
		{"простенке", "простенк"},
		{"простенькое", "простеньк"},
		{"простер", "простер"},
		{"простерла", "простерл"},
		{"простерт", "простерт"},
		{"простертой", "простерт"},
		{"простерши", "простерш"},
		{"прости", "прост"},
		{"простившись", "прост"},
		{"простил", "прост"},
		{"простила", "прост"},
		{"простили", "прост"},
		{"простились", "прост"},
		{"простился", "прост"},
		{"простим", "прост"},
		{"простирает", "простира"},
		{"простиралось", "простира"},
		{"простирать", "простира"},
		{"простирая", "простир"},
		{"простит", "прост"},
		{"простите", "прост"},
		{"простительно", "простительн"},
		{"проститутка", "проститутк"},
		{"простить", "прост"},
		{"проститься", "прост"},
		{"простишь", "прост"},
		{"просто", "прост"},
		{"простоват", "простоват"},
		{"простоватого", "простоват"},
		{"простоватость", "простоват"},
		{"простоватый", "простоват"},
		{"простоволосая", "простоволос"},
		{"простоволосые", "простоволос"},
		{"простого", "прост"},
		{"простодушие", "простодуш"},
		{"простодушием", "простодуш"},
		{"простодушия", "простодуш"},
		{"простодушно", "простодушн"},
		{"простодушное", "простодушн"},
		{"простодушные", "простодушн"},
		{"простодушными", "простодушн"},
		{"простое", "прост"},
		{"простой", "прост"},
		{"простокваши", "простокваш"},
		{"простоквашу", "простокваш"},
		{"простолюдин", "простолюдин"},
		{"простом", "прост"},
		{"простонал", "простона"},
		{"простонало", "простона"},
		{"простонародный", "простонародн"},
		{"простонародье", "простонарод"},
		{"простонет", "простонет"},
		{"простор", "простор"},
		{"простора", "простор"},
		{"просторе", "простор"},
		{"просторно", "просторн"},
		{"просторное", "просторн"},
		{"просторную", "просторн"},
		{"просторные", "просторн"},
		{"просторный", "просторн"},
		{"простору", "простор"},
		{"простота", "простот"},
		{"простоте", "простот"},
		{"простотой", "простот"},
		{"простотою", "простот"},
		{"простоту", "простот"},
		{"простоты", "простот"},
		{"простою", "прост"},
		{"простоял", "простоя"},
		{"простояла", "простоя"},
		{"простоять", "простоя"},
		{"пространным", "простра"},
		{"пространств", "пространств"},
		{"пространства", "пространств"},
		{"пространство", "пространств"},
		{"пространствовал", "пространствова"},
		{"прострелено", "прострел"},
		{"прострет", "прострет"},
		{"простуда", "простуд"},
		{"простудилась", "простуд"},
		{"простудился", "простуд"},
		{"простудитесь", "простуд"},
		{"простудится", "простуд"},
		{"простудишь", "простуд"},
		{"простудишься", "простуд"},
		{"простуды", "простуд"},
		{"простуживайся", "простужива"},
		{"простужусь", "простуж"},
		{"проступали", "проступа"},
		{"проступок", "проступок"},
		{"простучат", "простучат"},
		{"просты", "прост"},
		{"простые", "прост"},
		{"простыл", "прост"},
		{"простыла", "прост"},
		{"простым", "прост"},
		{"простыми", "прост"},
		{"простыне", "простын"},
		{"простыней", "простын"},
		{"простыни", "простын"},
		{"простыню", "простын"},
		{"простыня", "простын"},
		{"простынях", "простын"},
		{"простых", "прост"},
		{"простят", "прост"},
		{"просунется", "просунет"},
		{"просунул", "просунул"},
		{"просунулась", "просунул"},
		{"просунуть", "просунут"},
		{"просушить", "просуш"},
		{"просыпается", "просыпа"},
		{"просыпаешься", "просыпа"},
		{"просыпалась", "просыпа"},
		{"просыпали", "просыпа"},
		{"просыпались", "просыпа"},
		{"просыпалось", "просыпа"},
		{"просыпался", "просыпа"},
		{"просыпаться", "просыпа"},
		{"просыпаюсь", "просыпа"},
		{"просыпаются", "просыпа"},
		{"просыпающегося", "просыпа"},
		{"просыплю", "просыпл"},
		{"просьб", "просьб"},
		{"просьба", "просьб"},
		{"просьбами", "просьб"},
		{"просьбе", "просьб"},
		{"просьбица", "просьбиц"},
		{"просьбой", "просьб"},
		{"просьбу", "просьб"},
		{"просьбы", "просьб"},
		{"прося", "про"},
		{"просясь", "прос"},
		{"просят", "прос"},
		{"просятся", "прос"},
		{"протанцевал", "протанцева"},
		{"протанцевала", "протанцева"},
		{"протанцевать", "протанцева"},
		{"протащила", "протащ"},
		{"протекал", "протека"},
		{"протекли", "протекл"},
		{"протекция", "протекц"},
		{"протекших", "протекш"},
		{"протер", "протер"},
		{"протереть", "протерет"},
		{"протерпела", "протерпел"},
		{"протертые", "протерт"},
		{"протеснилась", "протесн"},
		{"протеснились", "протесн"},
		{"протеснился", "протесн"},
		{"протеснявшихся", "протесня"},
		{"протеснятся", "протесн"},
		{"протесняясь", "протесн"},
		{"протест", "протест"},
		{"протестовал", "протестова"},
		{"протестованное", "протестова"},
		{"протестовать", "протестова"},
		{"протестом", "протест"},
		{"протестуй", "протест"},
		{"протестую", "протест"},
		{"протесты", "протест"},
		{"против", "прот"},
		{"противен", "против"},
		{"противиться", "против"},
		{"противная", "противн"},
		{"противнее", "противн"},
		{"противник", "противник"},
		{"противника", "противник"},
		{"противником", "противник"},
		{"противно", "противн"},
		{"противное", "противн"},
		{"противном", "противн"},
		{"противны", "противн"},
		{"противный", "противн"},
		{"противным", "противн"},
		{"противных", "противн"},
		{"противоестественном", "противоестествен"},
		{"противоположная", "противоположн"},
		{"противоположного", "противоположн"},
		{"противоположной", "противоположн"},
		{"противоположном", "противоположн"},
		{"противоположность", "противоположн"},
		{"противоположную", "противоположн"},
		{"противоположные", "противоположн"},
		{"противоположный", "противоположн"},
		{"противоположными", "противоположн"},
		{"противопоставить", "противопостав"},
		{"противоречие", "противореч"},
		{"противоречии", "противореч"},
		{"противоречий", "противореч"},
		{"противоречило", "противореч"},
		{"противоречит", "противореч"},
		{"противоречить", "противореч"},
		{"противоречия", "противореч"},
		{"противоречу", "противореч"},
		{"противоречь", "противореч"},
		{"противуположную", "противуположн"},
		{"противься", "против"},
		{"протискиваясь", "протискив"},
		{"протиснулся", "протиснул"},
		{"проткнет", "проткнет"},
		{"протокол", "протокол"},
		{"протолкнул", "протолкнул"},
		{"протолкуешь", "протолкуеш"},
		{"протопопов", "протопоп"},
		{"протопопова", "протопопов"},
		{"протопоповым", "протопопов"},
		{"протоптанной", "протопта"},
		{"проторила", "протор"},
		{"протягивает", "протягива"},
		{"протягивай", "протягива"},
		{"протягивайте", "протягива"},
		{"протягивал", "протягива"},
		{"протягивала", "протягива"},
		{"протягивались", "протягива"},
		{"протягивая", "протягив"},
		{"протяжен", "протяж"},
		{"протяжно", "протяжн"},
		{"протяжный", "протяжн"},
		{"протянет", "протянет"},
		{"протянешь", "протянеш"},
		{"протяните", "протян"},
		{"протяну", "протян"},
		{"протянув", "протянув"},
		{"протянувшеюся", "протянувш"},
		{"протянувшиеся", "протянувш"},
		{"протянул", "протянул"},
		{"протянула", "протянул"},
		{"протянулись", "протянул"},
		{"протянулся", "протянул"},
		{"протянут", "протянут"},
		{"протянута", "протянут"},
		{"протянутая", "протянут"},
		{"протянутой", "протянут"},
		{"протянутою", "протянут"},
		{"протянутую", "протянут"},
		{"протянутый", "протянут"},
		{"протянутыми", "протянут"},
		{"протянуть", "протянут"},
		{"протянуться", "протянут"},
		{"проулок", "проулок"},
		{"профессор", "профессор"},
		{"профессора", "профессор"},
		{"профессоров", "профессор"},
		{"профессором", "профессор"},
		{"профессорскую", "профессорск"},
		{"профессору", "профессор"},
		{"профессоршу", "профессорш"},
		{"профилем", "профил"},
		{"профиль", "профил"},
		{"профиля", "профил"},
		{"прохаживается", "прохажива"},
		{"прохаживался", "прохажива"},
		{"прохаживаться", "прохажива"},
		{"прохватила", "прохват"},
		{"прохватывали", "прохватыва"},
		{"прохлада", "прохлад"},
		{"прохладная", "прохладн"},
		{"прохладно", "прохладн"},
		{"прохладной", "прохладн"},
		{"прохладные", "прохладн"},
		{"прохладный", "прохладн"},
		{"прохладных", "прохладн"},
		{"прохладой", "прохлад"},
		{"прохлады", "прохлад"},
		{"проход", "проход"},
		{"прохода", "проход"},
		{"проходи", "проход"},
		{"проходившего", "проход"},
		{"проходившей", "проход"},
		{"проходившему", "проход"},
		{"проходил", "проход"},
		{"проходила", "проход"},
		{"проходили", "проход"},
		{"проходило", "проход"},
		{"проходимцев", "проходимц"},
		{"проходит", "проход"},
		{"проходите", "проход"},
		{"проходить", "проход"},
		{"проходишь", "проход"},
		{"проходная", "проходн"},
		{"проходной", "проходн"},
		{"проходу", "проход"},
		{"проходя", "проход"},
		{"проходят", "проход"},
		{"проходящих", "проходя"},
		{"прохожего", "прохож"},
		{"прохожей", "прохож"},
		{"прохожем", "прохож"},
		{"прохожему", "прохож"},
		{"прохожие", "прохож"},
		{"прохожий", "прохож"},
		{"прохожим", "прохож"},
		{"прохожих", "прохож"},
		{"прохожу", "прохож"},
		{"прохора", "прохор"},
		{"прохрипел", "прохрипел"},
		{"процветал", "процвета"},
		{"процветала", "процвета"},
		{"процветают", "процвета"},
		{"процедил", "процед"},
		{"процедур", "процедур"},
		{"процеживая", "процежив"},
		{"процент", "процент"},
		{"процентами", "процент"},
		{"процентные", "процентн"},
		{"процентов", "процент"},
		{"процентщица", "процентщиц"},
		{"процентщицу", "процентщиц"},
		{"процентщицы", "процентщиц"},
		{"проценты", "процент"},
		{"процесс", "процесс"},
		{"процесса", "процесс"},
		{"процессе", "процесс"},
		{"процессии", "процесс"},
		{"процессия", "процесс"},
		{"процессом", "процесс"},
		{"процессу", "процесс"},
		{"проч", "проч"},
		{"проча", "проч"},
		{"прочего", "проч"},
		{"прочее", "проч"},
		{"прочел", "прочел"},
		{"прочесть", "прочест"},
		{"прочие", "проч"},
		{"прочий", "проч"},
		{"прочим", "проч"},
		{"прочими", "проч"},
		{"прочитав", "прочита"},
		{"прочитайте", "прочита"},
		{"прочитал", "прочита"},
		{"прочитанного", "прочита"},
		{"прочитанное", "прочита"},
		{"прочитано", "прочита"},
		{"прочитать", "прочита"},
		{"прочитают", "прочита"},
		{"прочитывал", "прочитыва"},
		{"прочих", "проч"},
		{"прочла", "прочл"},
		{"прочно", "прочн"},
		{"прочного", "прочн"},
		{"прочной", "прочн"},
		{"прочность", "прочност"},
		{"прочтем", "прочт"},
		{"прочтении", "прочтен"},
		{"прочтения", "прочтен"},
		{"прочтет", "прочтет"},
		{"прочтете", "прочтет"},
		{"прочтешь", "прочтеш"},
		{"прочти", "прочт"},
		{"прочтите", "прочт"},
		{"прочту", "прочт"},
		{"прочтя", "прочт"},
		{"прочь", "проч"},
		{"прошагнул", "прошагнул"},
		{"прошедшая", "прошедш"},
		{"прошедшего", "прошедш"},
		{"прошедшее", "прошедш"},
		{"прошедшем", "прошедш"},
		{"прошедшие", "прошедш"},
		{"прошедших", "прошедш"},
		{"прошедшую", "прошедш"},
		{"прошел", "прошел"},
		{"прошелся", "прошел"},
		{"прошептал", "прошепта"},
		{"прошептала", "прошепта"},
		{"прошептались", "прошепта"},
		{"прошибет", "прошибет"},
		{"прошипел", "прошипел"},
		{"прошла", "прошл"},
		{"прошли", "прошл"},
		{"прошло", "прошл"},
		{"прошлого", "прошл"},
		{"прошлогоднего", "прошлогодн"},
		{"прошлогодний", "прошлогодн"},
		{"прошлогодним", "прошлогодн"},
		{"прошлогодних", "прошлогодн"},
		{"прошлое", "прошл"},
		{"прошлой", "прошл"},
		{"прошлом", "прошл"},
		{"прошлому", "прошл"},
		{"прошлую", "прошл"},
		{"прошлый", "прошл"},
		{"прошлым", "прошл"},
		{"прошлых", "прошл"},
		{"прошлялся", "прошля"},
		{"прошмыгнул", "прошмыгнул"},
		{"прошмыгнуть", "прошмыгнут"},
		{"прошу", "прош"},
		{"прошумевшего", "прошумевш"},
		{"прошумела", "прошумел"},
		{"прощаетесь", "проща"},
		{"прощается", "проща"},
		{"прощай", "проща"},
		{"прощайте", "проща"},
		{"прощал", "проща"},
		{"прощали", "проща"},
		{"прощались", "проща"},
		{"прощальный", "прощальн"},
		{"прощальным", "прощальн"},
		{"прощание", "прощан"},
		{"прощании", "прощан"},
		{"прощанию", "прощан"},
		{"прощанье", "прощан"},
		{"прощаться", "проща"},
		{"прощаю", "проща"},
		{"прощаюсь", "проща"},
		{"прощают", "проща"},
		{"прощаются", "проща"},
		{"прощая", "прощ"},
		{"прощаясь", "прощ"},
		{"проще", "прощ"},
		{"прощение", "прощен"},
		{"прощении", "прощен"},
		{"прощения", "прощен"},
		{"прощено", "прощ"},
		{"прощенья", "прощен"},
		{"прощипаны", "прощипа"},
		{"прощу", "прощ"},
		{"проявилась", "прояв"},
		{"проявилось", "прояв"},
		{"проявить", "прояв"},
		{"проявление", "проявлен"},
		{"проявлению", "проявлен"},
		{"проявления", "проявлен"},
		{"проявлениями", "проявлен"},
		{"проявляется", "проявля"},
		{"проявляла", "проявля"},
		{"проявлялась", "проявля"},
		{"проявлялись", "проявля"},
		{"проявлялось", "проявля"},
		{"прояснел", "прояснел"},
		{"прояснели", "прояснел"},
		{"прояснившийся", "проясн"},
		{"прояснилось", "проясн"},
		{"прояснить", "проясн"},
		{"проясняет", "проясня"},
		{"прояснялось", "проясня"},
		{"прояснялся", "проясня"},
		{"пруд", "пруд"},
		{"пруда", "пруд"},
		{"прудами", "пруд"},
		{"пруде", "пруд"},
		{"прудов", "пруд"},
		{"прудовой", "прудов"},
		{"пруды", "пруд"},
		{"пружин", "пружин"},
		{"пружинами", "пружин"},
		{"пружинах", "пружин"},
		{"пружиной", "пружин"},
		{"пружины", "пружин"},
		{"прусская", "прусск"},
		{"прусскую", "прусск"},
		{"прыг", "прыг"},
		{"прыгавший", "прыга"},
		{"прыгает", "прыга"},
		{"прыгай", "прыга"},
		{"прыгал", "прыга"},
		{"прыгала", "прыга"},
		{"прыгали", "прыга"},
		{"прыгало", "прыга"},
		{"прыганье", "прыган"},
		{"прыгать", "прыга"},
		{"прыгают", "прыга"},
		{"прыгая", "прыг"},
		{"прыгнул", "прыгнул"},
		{"прыгнуть", "прыгнут"},
		{"прыжка", "прыжк"},
		{"прыжком", "прыжк"},
		{"прыжок", "прыжок"},
		{"прынцессы", "прынцесс"},
		{"прыскается", "прыска"},
		{"прыснет", "прыснет"},
		{"прыснула", "прыснул"},
		{"прыснуть", "прыснут"},
		{"прыть", "прыт"},
		{"прытью", "прыт"},
		{"прядет", "прядет"},
		{"прядь", "пряд"},
		{"пряжку", "пряжк"},
		{"прям", "прям"},
		{"прямехонько", "прямехоньк"},
		{"прямились", "прям"},
		{"прямо", "прям"},
		{"прямого", "прям"},
		{"прямодушии", "прямодуш"},
		{"прямодушия", "прямодуш"},
		{"прямодушный", "прямодушн"},
		{"прямое", "прям"},
		{"прямой", "прям"},
		{"прямому", "прям"},
		{"прямою", "прям"},
		{"прямую", "прям"},
		{"прямые", "прям"},
		{"прямым", "прям"},
		{"прямыми", "прям"},
		{"прямых", "прям"},
		{"пряничного", "пряничн"},
		{"пряно", "прян"},
		{"пряное", "прян"},
		{"пряности", "пряност"},
		{"прянул", "прянул"},
		{"пряный", "прян"},
		{"прятал", "прята"},
		{"прятала", "прята"},
		{"пряталась", "прята"},
		{"прятались", "прята"},
		{"прятался", "прята"},
		{"прятать", "прята"},
		{"прятаться", "прята"},
		{"пряча", "пряч"},
		{"прячась", "пряч"},
		{"прячет", "прячет"},
		{"прячется", "прячет"},
		{"прячешь", "прячеш"},
		{"прячут", "прячут"},
		{"прячутся", "прячут"},
		{"прячущегося", "прячущ"},
		{"пса", "пса"},
		{"псами", "псам"},
		{"психически", "психическ"},
		{"психоз", "психоз"},
		{"психологии", "психолог"},
		{"психологическая", "психологическ"},
		{"психологически", "психологическ"},
		{"психологические", "психологическ"},
		{"психологический", "психологическ"},
		{"психологическим", "психологическ"},
		{"психологическое", "психологическ"},
		{"психологической", "психологическ"},
		{"психологическую", "психологическ"},
		{"психологию", "психолог"},
		{"психология", "психолог"},
		{"психологов", "психолог"},
		{"психологом", "психолог"},
		{"психопат", "психопат"},
		{"псом", "псом"},
		{"псу", "псу"},
		{"псы", "псы"},
		{"птенцов", "птенц"},
		{"птиц", "птиц"},
		{"птица", "птиц"},
		{"птицами", "птиц"},
		{"птицах", "птиц"},
		{"птице", "птиц"},
		{"птицей", "птиц"},
		{"птицу", "птиц"},
		{"птицы", "птиц"},
		{"птичек", "птичек"},
		{"птичий", "птич"},
		{"птичка", "птичк"},
		{"птички", "птичк"},
		{"птичку", "птичк"},
		{"птичницы", "птичниц"},
		{"птичьего", "птич"},
		{"птичьем", "птич"},
		{"птичьи", "птич"},
		{"пуантов", "пуант"},
		{"публика", "публик"},
		{"публике", "публик"},
		{"публики", "публик"},
		{"публикой", "публик"},
		{"публику", "публик"},
		{"публичная", "публичн"},
		{"публичного", "публичн"},
		{"публичное", "публичн"},
		{"публичной", "публичн"},
		{"публичную", "публичн"},
		{"публичные", "публичн"},
		{"публичный", "публичн"},
		{"публичных", "публичн"},
		{"пугает", "пуга"},
		{"пугаете", "пуга"},
		{"пугается", "пуга"},
		{"пугаешь", "пуга"},
		{"пугай", "пуга"},
		{"пугайся", "пуга"},
		{"пугайтесь", "пуга"},
		{"пугал", "пуга"},
		{"пугала", "пуга"},
		{"пугали", "пуга"},
		{"пугало", "пуга"},
		{"пугался", "пуга"},
		{"пугать", "пуга"},
		{"пугаться", "пуга"},
		{"пугаюсь", "пуга"},
		{"пугают", "пуга"},
		{"пугающая", "пуга"},
		{"пугающий", "пуга"},
		{"пугающим", "пуга"},
		{"пугаясь", "пуг"},
		{"пугливая", "пуглив"},
		{"пугливее", "пуглив"},
		{"пугливо", "пуглив"},
		{"пугливого", "пуглив"},
		{"пугливые", "пуглив"},
		{"пугливых", "пуглив"},
		{"пуговиц", "пуговиц"},
		{"пуговицами", "пуговиц"},
		{"пуговицы", "пуговиц"},
		{"пуговкой", "пуговк"},
		{"пуд", "пуд"},
		{"пуда", "пуд"},
		{"пудами", "пуд"},
		{"пудинг", "пудинг"},
		{"пудов", "пуд"},
		{"пудрится", "пудр"},
		{"пудрой", "пудр"},
		{"пужливые", "пужлив"},
		{"пузатый", "пузат"},
		{"пузатых", "пузат"},
		{"пузырек", "пузырек"},
		{"пузырь", "пузыр"},
		{"пуль", "пул"},
		{"пульс", "пульс"},
		{"пульса", "пульс"},
		{"пульхерией", "пульхер"},
		{"пульхерии", "пульхер"},
		{"пульхерию", "пульхер"},
		{"пульхерия", "пульхер"},
		{"пулю", "пул"},
		{"пуля", "пул"},
		{"пулями", "пул"},
		{"пулярку", "пулярк"},
		{"пункт", "пункт"},
		{"пункта", "пункт"},
		{"пунктам", "пункт"},
		{"пунктах", "пункт"},
		{"пункте", "пункт"},
		{"пунктов", "пункт"},
		{"пунктом", "пункт"},
		{"пункту", "пункт"},
		{"пункты", "пункт"},
		{"пуншем", "пунш"},
		{"пуншу", "пунш"},
		{"пуп", "пуп"},
		{"пуританский", "пуританск"},
		{"пурпурово", "пурпуров"},
		{"пускает", "пуска"},
		{"пускаете", "пуска"},
		{"пускаешь", "пуска"},
		{"пускай", "пуска"},
		{"пускайте", "пуска"},
		{"пускал", "пуска"},
		{"пускала", "пуска"},
		{"пускалась", "пуска"},
		{"пускали", "пуска"},
		{"пускался", "пуска"},
		{"пускать", "пуска"},
		{"пускаться", "пуска"},
		{"пускают", "пуска"},
		{"пуст", "пуст"},
		{"пуста", "пуст"},
		{"пустая", "пуст"},
		{"пустейшем", "пуст"},
		{"пустейших", "пуст"},
		{"пусти", "пуст"},
		{"пустив", "пуст"},
		{"пустил", "пуст"},
		{"пустила", "пуст"},
		{"пустилась", "пуст"},
		{"пустили", "пуст"},
		{"пустились", "пуст"},
		{"пустило", "пуст"},
		{"пустился", "пуст"},
		{"пустим", "пуст"},
		{"пустит", "пуст"},
		{"пустите", "пуст"},
		{"пустить", "пуст"},
		{"пуститься", "пуст"},
		{"пусто", "пуст"},
		{"пустого", "пуст"},
		{"пустое", "пуст"},
		{"пустой", "пуст"},
		{"пустом", "пуст"},
		{"пустому", "пуст"},
		{"пустота", "пустот"},
		{"пустоте", "пустот"},
		{"пустоту", "пустот"},
		{"пустоты", "пустот"},
		{"пустоши", "пустош"},
		{"пустошь", "пустош"},
		{"пустою", "пуст"},
		{"пустую", "пуст"},
		{"пусты", "пуст"},
		{"пустые", "пуст"},
		{"пустым", "пуст"},
		{"пустыми", "пуст"},
		{"пустыне", "пустын"},
		{"пустыней", "пустын"},
		{"пустыни", "пустын"},
		{"пустынна", "пустын"},
		{"пустынного", "пустын"},
		{"пустынной", "пустын"},
		{"пустынном", "пустын"},
		{"пустынную", "пустын"},
		{"пустынные", "пустын"},
		{"пустынный", "пустын"},
		{"пустынным", "пустын"},
		{"пустынь", "пустын"},
		{"пустыню", "пустын"},
		{"пустырей", "пустыр"},
		{"пустырем", "пустыр"},
		{"пустырь", "пустыр"},
		{"пустых", "пуст"},
		{"пусть", "пуст"},
		{"пустякам", "пустяк"},
		{"пустяками", "пустяк"},
		{"пустяках", "пустяк"},
		{"пустяки", "пустяк"},
		{"пустяков", "пустяк"},
		{"пустят", "пуст"},
		{"пустячков", "пустячк"},
		{"пустяшная", "пустяшн"},
		{"пустяшное", "пустяшн"},
		{"пустяшным", "пустяшн"},
		{"путал", "пута"},
		{"путали", "пута"},
		{"путались", "пута"},
		{"путаница", "путаниц"},
		{"путаницу", "путаниц"},
		{"путанно", "пута"},
		{"путаться", "пута"},
		{"путаю", "пута"},
		{"путаются", "пута"},
		{"путая", "пут"},
		{"путеводителем", "путеводител"},
		{"путеводительный", "путеводительн"},
		{"путеводной", "путеводн"},
		{"путей", "пут"},
		{"путем", "пут"},
		{"путешественники", "путешественник"},
		{"путешественников", "путешественник"},
		{"путешествие", "путешеств"},
		{"путешествии", "путешеств"},
		{"путешествий", "путешеств"},
		{"путешествия", "путешеств"},
		{"путешествиях", "путешеств"},
		{"путешествует", "путешеств"},
		{"пути", "пут"},
		{"путилки", "путилк"},
		{"путилкой", "путилк"},
		{"путилку", "путилк"},
		{"путник", "путник"},
		{"путнику", "путник"},
		{"путное", "путн"},
		{"путной", "путн"},
		{"путь", "пут"},
		{"путями", "пут"},
		{"путях", "пут"},
		{"пуф", "пуф"},
		{"пух", "пух"},
		{"пуха", "пух"},
		{"пухленькие", "пухленьк"},
		{"пухлое", "пухл"},
		{"пухлых", "пухл"},
		{"пуховые", "пухов"},
		{"пуховым", "пухов"},
		{"пухом", "пух"},
		{"пучат", "пучат"},
		{"пучине", "пучин"},
		{"пучину", "пучин"},
		{"пучком", "пучк"},
		{"пушечный", "пушечн"},
		{"пушистые", "пушист"},
		{"пушистый", "пушист"},
		{"пушка", "пушк"},
		{"пушки", "пушк"},
		{"пушкин", "пушкин"},
		{"пушкина", "пушкин"},
		{"пушкинское", "пушкинск"},
		{"пушкиным", "пушкин"},
		{"пушкиньянца", "пушкиньянц"},
		{"пушку", "пушк"},
		{"пушок", "пушок"},
		{"пущай", "пуща"},
		{"пуще", "пущ"},
		{"пущенное", "пущен"},
		{"пущенными", "пущен"},
		{"пущу", "пущ"},
		{"пхайло", "пхайл"},
		{"пчелка", "пчелк"},
		{"пчеловодство", "пчеловодств"},
		{"пчелы", "пчел"},
		{"пшеница", "пшениц"},
		{"пшеницын", "пшеницын"},
		{"пшеницына", "пшеницын"},
		{"пшеницыной", "пшеницын"},
		{"пшеницыну", "пшеницын"},
		{"пшеницыных", "пшеницын"},
		{"пшла", "пшла"},
		{"пшол", "пшол"},
		{"пыжикова", "пыжиков"},
		{"пыл", "пыл"},
		{"пылает", "пыла"},
		{"пылал", "пыла"},
		{"пылала", "пыла"},
		{"пылают", "пыла"},
		{"пылающим", "пыла"},
		{"пылающими", "пыла"},
		{"пылая", "пыл"},
		{"пыли", "пыл"},
		{"пылили", "пыл"},
		{"пылинки", "пылинк"},
		{"пылинку", "пылинк"},
		{"пылким", "пылк"},
		{"пылко", "пылк"},
		{"пылкой", "пылк"},
		{"пылом", "пыл"},
		{"пылу", "пыл"},
		{"пыль", "пыл"},
		{"пыльная", "пыльн"},
		{"пыльного", "пыльн"},
		{"пыльной", "пыльн"},
		{"пыльном", "пыльн"},
		{"пыльную", "пыльн"},
		{"пыльные", "пыльн"},
		{"пыльный", "пыльн"},
		{"пыльными", "пыльн"},
		{"пылью", "пыл"},
		{"пылят", "пыл"},
		{"пытала", "пыта"},
		{"пытался", "пыта"},
		{"пытая", "пыт"},
		{"пытаясь", "пыт"},
		{"пытка", "пытк"},
		{"пытки", "пытк"},
		{"пыткой", "пытк"},
		{"пытку", "пытк"},
		{"пытливо", "пытлив"},
		{"пытливого", "пытлив"},
		{"пытливости", "пытлив"},
		{"пытливый", "пытлив"},
		{"пыток", "пыток"},
		{"пыхтя", "пыхт"},
		{"пышная", "пышн"},
		{"пышнейшее", "пышн"},
		{"пышно", "пышн"},
		{"пышной", "пышн"},
		{"пышном", "пышн"},
		{"пышную", "пышн"},
		{"пышные", "пышн"},
		{"пышных", "пышн"},
		{"пышут", "пышут"},
		{"пьан", "пьан"},
		{"пьедестал", "пьедеста"},
		{"пьем", "пьем"},
		{"пьер", "пьер"},
		{"пьеса", "пьес"},
		{"пьесах", "пьес"},
		{"пьесе", "пьес"},
		{"пьесу", "пьес"},
		{"пьесы", "пьес"},
		{"пьет", "пьет"},
		{"пьете", "пьет"},
		{"пьешь", "пьеш"},
		{"пью", "пью"},
		{"пьют", "пьют"},
		{"пьющей", "пьющ"},
		{"пьющий", "пьющ"},
		{"пьющих", "пьющ"},
		{"пьян", "пьян"},
		{"пьяна", "пьян"},
		{"пьяная", "пьян"},
		{"пьяненькие", "пьяненьк"},
		{"пьяненький", "пьяненьк"},
		{"пьяненькой", "пьяненьк"},
		{"пьянею", "пьян"},
		{"пьянея", "пьяне"},
		{"пьяниц", "пьяниц"},
		{"пьяница", "пьяниц"},
		{"пьяницей", "пьяниц"},
		{"пьяницу", "пьяниц"},
		{"пьяницы", "пьяниц"},
		{"пьяного", "пьян"},
		{"пьяной", "пьян"},
		{"пьяном", "пьян"},
		{"пьяному", "пьян"},
		{"пьянства", "пьянств"},
		{"пьянство", "пьянств"},
		{"пьянствовал", "пьянствова"},
		{"пьянствовать", "пьянствова"},
		{"пьянствует", "пьянств"},
		{"пьянствуй", "пьянств"},
		{"пьяную", "пьян"},
		{"пьяны", "пьян"},
		{"пьяные", "пьян"},
		{"пьяный", "пьян"},
		{"пьяным", "пьян"},
		{"пьяных", "пьян"},
		{"пэб", "пэб"},
		{"пялит", "пял"},
		{"пялить", "пял"},
		{"пясту", "пяст"},
		{"пятак", "пятак"},
		{"пятака", "пятак"},
		{"пятаками", "пятак"},
		{"пятаки", "пятак"},
		{"пятаков", "пятак"},
		{"пятам", "пят"},
		{"пятачка", "пятачк"},
		{"пятачку", "пятачк"},
		{"пятачок", "пятачок"},
		{"пятая", "пят"},
		{"пятен", "пят"},
		{"пятеро", "пятер"},
		{"пяти", "пят"},
		{"пятидесяти", "пятидесят"},
		{"пятидесятилетнему", "пятидесятилетн"},
		{"пятидесятилетнею", "пятидесятилетн"},
		{"пятидесятирублевую", "пятидесятирублев"},
		{"пятидесятью", "пятидесят"},
		{"пятилетнего", "пятилетн"},
		{"пятилетняя", "пятилетн"},
		{"пятиминутным", "пятиминутн"},
		{"пятипроцентные", "пятипроцентн"},
		{"пятипроцентный", "пятипроцентн"},
		{"пятипроцентных", "пятипроцентн"},
		{"пятипудовая", "пятипудов"},
		{"пятисот", "пятисот"},
		{"пятистах", "пятист"},
		{"пятить", "пят"},
		{"пятиэтажного", "пятиэтажн"},
		{"пятками", "пятк"},
		{"пятки", "пятк"},
		{"пятна", "пятн"},
		{"пятнадцати", "пятнадцат"},
		{"пятнадцатилетнюю", "пятнадцатилетн"},
		{"пятнадцатом", "пятнадцат"},
		{"пятнадцать", "пятнадца"},
		{"пятнам", "пятн"},
		{"пятнами", "пятн"},
		{"пятнах", "пятн"},
		{"пятница", "пятниц"},
		{"пятнице", "пятниц"},
		{"пятницу", "пятниц"},
		{"пятницы", "пятниц"},
		{"пятно", "пятн"},
		{"пятном", "пятн"},
		{"пятнышка", "пятнышк"},
		{"пятнышко", "пятнышк"},
		{"пятого", "пят"},
		{"пятое", "пят"},
		{"пятой", "пят"},
		{"пятом", "пят"},
		{"пяточками", "пяточк"},
		{"пятою", "пят"},
		{"пятую", "пят"},
		{"пятый", "пят"},
		{"пять", "пят"},
		{"пятьдесят", "пятьдес"},
		{"пятьсот", "пятьсот"},
		{"пятью", "пят"},
		{"пятясь", "пят"},
		{"р", "р"},
		{"ра", "ра"},
		{"раб", "раб"},
		{"раба", "раб"},
		{"рабам", "раб"},
		{"рабами", "раб"},
		{"рабой", "раб"},
		{"рабом", "раб"},
		{"работ", "работ"},
		{"работа", "работ"},
		{"работавшая", "работа"},
		{"работавши", "работа"},
		{"работавший", "работа"},
		{"работаем", "работа"},
		{"работает", "работа"},
		{"работаете", "работа"},
		{"работаешь", "работа"},
		{"работай", "работа"},
		{"работал", "работа"},
		{"работала", "работа"},
		{"работали", "работа"},
		{"работам", "работ"},
		{"работами", "работ"},
		{"работать", "работа"},
		{"работах", "работ"},
		{"работаю", "работа"},
		{"работают", "работа"},
		{"работе", "работ"},
		{"работник", "работник"},
		{"работника", "работник"},
		{"работниках", "работник"},
		{"работники", "работник"},
		{"работников", "работник"},
		{"работнику", "работник"},
		{"работой", "работ"},
		{"работу", "работ"},
		{"работы", "работ"},
		{"рабочие", "рабоч"},
		{"рабочий", "рабоч"},
		{"рабочим", "рабоч"},
		{"рабочих", "рабоч"},
		{"рабочую", "рабоч"},
		{"рабская", "рабск"},
		{"рабски", "рабск"},
		{"рабских", "рабск"},
		{"рабстве", "рабств"},
		{"рабы", "раб"},
		{"равенна", "равен"},
		{"равеннских", "равеннск"},
		{"равенства", "равенств"},
		{"равенство", "равенств"},
		{"равна", "равн"},
		{"равнина", "равнин"},
		{"равнины", "равнин"},
		{"равно", "равн"},
		{"равновесие", "равновес"},
		{"равновесии", "равновес"},
		{"равновесия", "равновес"},
		{"равнодушен", "равнодуш"},
		{"равнодушие", "равнодуш"},
		{"равнодушием", "равнодуш"},
		{"равнодушии", "равнодуш"},
		{"равнодушию", "равнодуш"},
		{"равнодушия", "равнодуш"},
		{"равнодушна", "равнодушн"},
		{"равнодушная", "равнодушн"},
		{"равнодушно", "равнодушн"},
		{"равнодушного", "равнодушн"},
		{"равнодушное", "равнодушн"},
		{"равнодушны", "равнодушн"},
		{"равнодушный", "равнодушн"},
		{"равное", "равн"},
		{"равномерно", "равномерн"},
		{"равносильное", "равносильн"},
		{"равны", "равн"},
		{"равным", "равн"},
		{"равных", "равн"},
		{"равняется", "равня"},
		{"равняла", "равня"},
		{"равняли", "равня"},
		{"равнять", "равня"},
		{"равняю", "равня"},
		{"рагулиным", "рагулин"},
		{"рад", "рад"},
		{"рада", "рад"},
		{"радехонек", "радехонек"},
		{"ради", "рад"},
		{"радикально", "радикальн"},
		{"радикальный", "радикальн"},
		{"радиус", "радиус"},
		{"радищев", "радищ"},
		{"радо", "рад"},
		{"радовавшемуся", "радова"},
		{"радовались", "радова"},
		{"радовался", "радова"},
		{"радоваться", "радова"},
		{"радостей", "радост"},
		{"радости", "радост"},
		{"радостн", "радостн"},
		{"радостная", "радостн"},
		{"радостно", "радостн"},
		{"радостного", "радостн"},
		{"радостном", "радостн"},
		{"радостный", "радостн"},
		{"радостным", "радостн"},
		{"радостных", "радостн"},
		{"радость", "радост"},
		{"радостью", "радост"},
		{"радостями", "радост"},
		{"радостях", "радост"},
		{"радуется", "рад"},
		{"радуешься", "радуеш"},
		{"радужного", "радужн"},
		{"радужные", "радужн"},
		{"радушие", "радуш"},
		{"радушием", "радуш"},
		{"радушия", "радуш"},
		{"радушно", "радушн"},
		{"радушный", "радушн"},
		{"радуюсь", "рад"},
		{"радуясь", "раду"},
		{"рады", "рад"},
		{"раз", "раз"},
		{"раза", "раз"},
		{"разбавлять", "разбавля"},
		{"разбегаются", "разбега"},
		{"разбей", "разб"},
		{"разберет", "разберет"},
		{"разберешь", "разбереш"},
		{"разбери", "разбер"},
		{"разберу", "разбер"},
		{"разберут", "разберут"},
		{"разберутся", "разберут"},
		{"разбивал", "разбива"},
		{"разбивалась", "разбива"},
		{"разбиваются", "разбива"},
		{"разбивающее", "разбива"},
		{"разбившимся", "разб"},
		{"разбил", "разб"},
		{"разбила", "разб"},
		{"разбилась", "разб"},
		{"разбили", "разб"},
		{"разбился", "разб"},
		{"разбирал", "разбира"},
		{"разбирала", "разбира"},
		{"разбиранья", "разбиран"},
		{"разбирать", "разбира"},
		{"разбирая", "разбир"},
		{"разбит", "разб"},
		{"разбито", "разбит"},
		{"разбитого", "разбит"},
		{"разбитое", "разбит"},
		{"разбитой", "разбит"},
		{"разбитою", "разбит"},
		{"разбитую", "разбит"},
		{"разбитым", "разбит"},
		{"разбить", "разб"},
		{"разблаготворились", "разблаготвор"},
		{"разбогатевший", "разбогатевш"},
		{"разбогател", "разбогател"},
		{"разбогатеть", "разбогатет"},
		{"разбой", "разб"},
		{"разбойник", "разбойник"},
		{"разбойника", "разбойник"},
		{"разбойниках", "разбойник"},
		{"разбойники", "разбойник"},
		{"разбойников", "разбойник"},
		{"разбойником", "разбойник"},
		{"разбойничий", "разбойнич"},
		{"разбойную", "разбойн"},
		{"разболелась", "разболел"},
		{"разболелся", "разболел"},
		{"разболтал", "разболта"},
		{"разболтали", "разболта"},
		{"разболтался", "разболта"},
		{"разбор", "разбор"},
		{"разборчив", "разборч"},
		{"разборы", "разбор"},
		{"разбранил", "разбран"},
		{"разбрасывается", "разбрасыва"},
		{"разбредутся", "разбредут"},
		{"разбрелись", "разбрел"},
		{"разбросала", "разброса"},
		{"разбросались", "разброса"},
		{"разбросанные", "разброса"},
		{"разбросанный", "разброса"},
		{"разбросанных", "разброса"},
		{"разбросано", "разброса"},
		{"разбросаны", "разброса"},
		{"разбуди", "разбуд"},
		{"разбудил", "разбуд"},
		{"разбудила", "разбуд"},
		{"разбудили", "разбуд"},
		{"разбудило", "разбуд"},
		{"разбудит", "разбуд"},
		{"разбудить", "разбуд"},
		{"разбудишь", "разбуд"},
		{"разваливается", "развалива"},
		{"развалившуюся", "развал"},
		{"развалилась", "развал"},
		{"развалились", "развал"},
		{"развалилось", "развал"},
		{"развалился", "развал"},
		{"развалинами", "развалин"},
		{"развалинах", "развалин"},
		{"развалине", "развалин"},
		{"разве", "разв"},
		{"разведу", "развед"},
		{"разведывать", "разведыва"},
		{"развел", "развел"},
		{"развели", "развел"},
		{"развелось", "развел"},
		{"разверзается", "разверза"},
		{"разверзалась", "разверза"},
		{"развернув", "развернув"},
		{"развернувший", "развернувш"},
		{"развернул", "развернул"},
		{"развернула", "развернул"},
		{"развернулись", "развернул"},
		{"развернутое", "развернут"},
		{"развернутую", "развернут"},
		{"развернуты", "развернут"},
		{"развернутые", "развернут"},
		{"развернуть", "развернут"},
		{"развертывает", "развертыва"},
		{"развертывая", "развертыв"},
		{"развеселившись", "развесел"},
		{"развеселился", "развесел"},
		{"развеселить", "развесел"},
		{"развеселясь", "развесел"},
		{"развести", "развест"},
		{"развешивает", "развешива"},
		{"развешивал", "развешива"},
		{"развешивать", "развешива"},
		{"развивает", "развива"},
		{"развиваете", "развива"},
		{"развивается", "развива"},
		{"развивал", "развива"},
		{"развивалась", "развива"},
		{"развивалось", "развива"},
		{"развивать", "развива"},
		{"развиваю", "развива"},
		{"развиваются", "развива"},
		{"развившись", "разв"},
		{"развилась", "разв"},
		{"развились", "разв"},
		{"развился", "разв"},
		{"развинтился", "развинт"},
		{"развит", "разв"},
		{"развитие", "развит"},
		{"развитием", "развит"},
		{"развитии", "развит"},
		{"развитий", "развит"},
		{"развитию", "развит"},
		{"развития", "развит"},
		{"развитой", "развит"},
		{"развитость", "развит"},
		{"развитый", "развит"},
		{"развитым", "развит"},
		{"развить", "разв"},
		{"развлекайтесь", "развлека"},
		{"развлекать", "развлека"},
		{"развлекись", "развлек"},
		{"развлечения", "развлечен"},
		{"развлечь", "развлеч"},
		{"разводить", "развод"},
		{"разводя", "развод"},
		{"развозимыми", "развозим"},
		{"разворчатся", "разворчат"},
		{"разврат", "разврат"},
		{"разврата", "разврат"},
		{"разврате", "разврат"},
		{"развратен", "разврат"},
		{"развратник", "развратник"},
		{"развратника", "развратник"},
		{"развратничает", "развратнича"},
		{"развратной", "развратн"},
		{"развратный", "развратн"},
		{"развратным", "развратн"},
		{"развратятся", "разврат"},
		{"развращение", "развращен"},
		{"развращенный", "развращен"},
		{"развяжешься", "развяжеш"},
		{"развяжите", "развяж"},
		{"развязал", "развяза"},
		{"развязала", "развяза"},
		{"развязался", "развяза"},
		{"развязаны", "развяза"},
		{"развязать", "развяза"},
		{"развязен", "развяз"},
		{"развязка", "развязк"},
		{"развязке", "развязк"},
		{"развязки", "развязк"},
		{"развязку", "развязк"},
		{"развязная", "развязн"},
		{"развязнее", "развязн"},
		{"развязно", "развязн"},
		{"развязности", "развязн"},
		{"развязным", "развязн"},
		{"развязных", "развязн"},
		{"развязывать", "развязыва"},
		{"разгадал", "разгада"},
		{"разгадать", "разгада"},
		{"разгадка", "разгадк"},
		{"разгадке", "разгадк"},
		{"разгадки", "разгадк"},
		{"разгадку", "разгадк"},
		{"разгаре", "разгар"},
		{"разгибать", "разгиба"},
		{"разгладились", "разглад"},
		{"разгладился", "разглад"},
		{"разглаживается", "разглажива"},
		{"разглядев", "разгляд"},
		{"разглядел", "разглядел"},
		{"разглядела", "разглядел"},
		{"разглядели", "разглядел"},
		{"разглядеть", "разглядет"},
		{"разгляди", "разгляд"},
		{"разглядит", "разгляд"},
		{"разглядишь", "разгляд"},
		{"разглядывавший", "разглядыва"},
		{"разглядывает", "разглядыва"},
		{"разглядываете", "разглядыва"},
		{"разглядывал", "разглядыва"},
		{"разглядывать", "разглядыва"},
		{"разглядывая", "разглядыв"},
		{"разгневанная", "разгнева"},
		{"разговаривавшие", "разговарива"},
		{"разговаривает", "разговарива"},
		{"разговаривал", "разговарива"},
		{"разговаривала", "разговарива"},
		{"разговаривали", "разговарива"},
		{"разговаривать", "разговарива"},
		{"разговаривают", "разговарива"},
		{"разговаривая", "разговарив"},
		{"разговеемся", "разгове"},
		{"разговенья", "разговен"},
		{"разговор", "разговор"},
		{"разговора", "разговор"},
		{"разговорам", "разговор"},
		{"разговорами", "разговор"},
		{"разговорах", "разговор"},
		{"разговоре", "разговор"},
		{"разговорился", "разговор"},
		{"разговоров", "разговор"},
		{"разговором", "разговор"},
		{"разговору", "разговор"},
		{"разговорчивым", "разговорчив"},
		{"разговоры", "разговор"},
		{"разговорят", "разговор"},
		{"разгонял", "разгоня"},
		{"разгорался", "разгора"},
		{"разгоревшееся", "разгоревш"},
		{"разгорелось", "разгорел"},
		{"разгорится", "разгор"},
		{"разгороженную", "разгорожен"},
		{"разгорячаясь", "разгоряч"},
		{"разгорячились", "разгоряч"},
		{"разгорячился", "разгоряч"},
		{"разгула", "разгул"},
		{"разгуливала", "разгулива"},
		{"разгульная", "разгульн"},
		{"разгульного", "разгульн"},
		{"разгулялась", "разгуля"},
		{"разгулялось", "разгуля"},
		{"раздавались", "раздава"},
		{"раздавалось", "раздава"},
		{"раздавался", "раздава"},
		{"раздаваться", "раздава"},
		{"раздавил", "раздав"},
		{"раздавила", "раздав"},
		{"раздавили", "раздав"},
		{"раздавлен", "раздавл"},
		{"раздавлена", "раздавл"},
		{"раздавленного", "раздавлен"},
		{"раздавленный", "раздавлен"},
		{"раздается", "разда"},
		{"раздала", "разда"},
		{"раздалась", "разда"},
		{"раздались", "разда"},
		{"раздалось", "разда"},
		{"раздался", "разда"},
		{"раздарить", "раздар"},
		{"раздастся", "раздаст"},
		{"раздаются", "разда"},
		{"раздвигать", "раздвига"},
		{"раздвигаются", "раздвига"},
		{"раздвигая", "раздвиг"},
		{"раздвинув", "раздвинув"},
		{"раздвинул", "раздвинул"},
		{"раздвинулась", "раздвинул"},
		{"раздвинулись", "раздвинул"},
		{"раздвинуто", "раздвинут"},
		{"раздев", "разд"},
		{"раздевала", "раздева"},
		{"раздевался", "раздева"},
		{"раздевать", "раздева"},
		{"раздевают", "раздева"},
		{"раздеваясь", "раздев"},
		{"раздевшись", "раздевш"},
		{"раздел", "раздел"},
		{"разделаешься", "раздела"},
		{"разделался", "раздела"},
		{"разделенная", "разделен"},
		{"разделив", "раздел"},
		{"разделил", "раздел"},
		{"разделила", "раздел"},
		{"разделит", "раздел"},
		{"разделить", "раздел"},
		{"раздельнее", "раздельн"},
		{"раздельно", "раздельн"},
		{"разделявший", "разделя"},
		{"разделяет", "разделя"},
		{"разделял", "разделя"},
		{"разделяла", "разделя"},
		{"разделялась", "разделя"},
		{"разделяю", "разделя"},
		{"разделяются", "разделя"},
		{"разделяющего", "разделя"},
		{"разденусь", "разден"},
		{"раздетого", "раздет"},
		{"раздетые", "раздет"},
		{"раздирающим", "раздира"},
		{"раздольем", "раздол"},
		{"раздольи", "раздол"},
		{"раздор", "раздор"},
		{"раздоры", "раздор"},
		{"раздосадованный", "раздосадова"},
		{"раздражает", "раздража"},
		{"раздражается", "раздража"},
		{"раздражайтесь", "раздража"},
		{"раздражал", "раздража"},
		{"раздражались", "раздража"},
		{"раздражался", "раздража"},
		{"раздражать", "раздража"},
		{"раздражают", "раздража"},
		{"раздражаются", "раздража"},
		{"раздражающие", "раздража"},
		{"раздражающим", "раздража"},
		{"раздражаясь", "раздраж"},
		{"раздражен", "раздраж"},
		{"раздражена", "раздраж"},
		{"раздражение", "раздражен"},
		{"раздражением", "раздражен"},
		{"раздражении", "раздражен"},
		{"раздражению", "раздражен"},
		{"раздражения", "раздражен"},
		{"раздраженная", "раздражен"},
		{"раздраженнее", "раздражен"},
		{"раздраженно", "раздражен"},
		{"раздраженного", "раздражен"},
		{"раздраженное", "раздражен"},
		{"раздраженную", "раздражен"},
		{"раздраженным", "раздражен"},
		{"раздражившись", "раздраж"},
		{"раздражил", "раздраж"},
		{"раздражилась", "раздраж"},
		{"раздражило", "раздраж"},
		{"раздражит", "раздраж"},
		{"раздражителен", "раздражител"},
		{"раздражительная", "раздражительн"},
		{"раздражительнее", "раздражительн"},
		{"раздражительно", "раздражительн"},
		{"раздражительное", "раздражительн"},
		{"раздражительном", "раздражительн"},
		{"раздражительностию", "раздражительн"},
		{"раздражительную", "раздражительн"},
		{"раздражительны", "раздражительн"},
		{"раздражительные", "раздражительн"},
		{"раздражительный", "раздражительн"},
		{"раздражительным", "раздражительн"},
		{"раздражить", "раздраж"},
		{"раздразнить", "раздразн"},
		{"раздроблен", "раздробл"},
		{"раздроблено", "раздробл"},
		{"раздробляется", "раздробля"},
		{"раздувает", "раздува"},
		{"раздувала", "раздува"},
		{"раздувающий", "раздува"},
		{"раздувая", "раздув"},
		{"раздуем", "разду"},
		{"раздулась", "раздул"},
		{"раздумал", "раздума"},
		{"раздумался", "раздума"},
		{"раздумчиво", "раздумчив"},
		{"раздумывал", "раздумыва"},
		{"раздумывали", "раздумыва"},
		{"раздумываться", "раздумыва"},
		{"раздумывая", "раздумыв"},
		{"раздумье", "раздум"},
		{"раздумью", "раздум"},
		{"раздумья", "раздум"},
		{"раздуть", "раздут"},
		{"раздушенный", "раздушен"},
		{"разевая", "разев"},
		{"разжал", "разжа"},
		{"разжалобил", "разжалоб"},
		{"разжалобясь", "разжалоб"},
		{"разжалована", "разжалова"},
		{"разжигает", "разжига"},
		{"раззадорило", "раззадор"},
		{"разик", "разик"},
		{"разини", "разин"},
		{"разинув", "разинув"},
		{"разинул", "разинул"},
		{"разинули", "разинул"},
		{"разинутым", "разинут"},
		{"разинутыми", "разинут"},
		{"разиня", "разин"},
		{"разит", "раз"},
		{"разлакомившийся", "разлаком"},
		{"разлакомившись", "разлаком"},
		{"разлакомились", "разлаком"},
		{"разлегся", "разлег"},
		{"разлейся", "разл"},
		{"разлетясь", "разлет"},
		{"разлетятся", "разлет"},
		{"разлив", "разл"},
		{"разливалась", "разлива"},
		{"разливались", "разлива"},
		{"разливался", "разлива"},
		{"разливах", "разлив"},
		{"разливающейся", "разлива"},
		{"разливе", "разлив"},
		{"разливных", "разливн"},
		{"разлилась", "разл"},
		{"разлились", "разл"},
		{"разлилось", "разл"},
		{"разлился", "разл"},
		{"разлито", "разлит"},
		{"разлитого", "разлит"},
		{"разлитому", "разлит"},
		{"разлитый", "разлит"},
		{"различал", "различа"},
		{"различала", "различа"},
		{"различать", "различа"},
		{"различаю", "различа"},
		{"различен", "различ"},
		{"различил", "различ"},
		{"различить", "различ"},
		{"различия", "различ"},
		{"различно", "различн"},
		{"различные", "различн"},
		{"различным", "различн"},
		{"различными", "различн"},
		{"различных", "различн"},
		{"разложения", "разложен"},
		{"разложенными", "разложен"},
		{"разложено", "разлож"},
		{"разложенье", "разложен"},
		{"разломал", "разлома"},
		{"разлука", "разлук"},
		{"разлуке", "разлук"},
		{"разлуки", "разлук"},
		{"разлучаться", "разлуча"},
		{"разлучен", "разлуч"},
		{"разлучимся", "разлуч"},
		{"разлучиться", "разлуч"},
		{"разлюбил", "разлюб"},
		{"разлюбите", "разлюб"},
		{"разлюбить", "разлюб"},
		{"разлюбишь", "разлюб"},
		{"разлюблю", "разлюбл"},
		{"разлюбя", "разлюб"},
		{"размазывать", "размазыва"},
		{"размарать", "размара"},
		{"размах", "разм"},
		{"размаха", "размах"},
		{"размахи", "размах"},
		{"размахивается", "размахива"},
		{"размахивай", "размахива"},
		{"размахивать", "размахива"},
		{"размахиваются", "размахива"},
		{"размахивая", "размахив"},
		{"размахнувшись", "размахнувш"},
		{"размахнулся", "размахнул"},
		{"размаху", "размах"},
		{"размашисты", "размашист"},
		{"размен", "разм"},
		{"разменивать", "разменива"},
		{"разменявший", "разменя"},
		{"разменявшись", "разменя"},
		{"разменял", "разменя"},
		{"разменялся", "разменя"},
		{"разменянного", "разменя"},
		{"разменять", "разменя"},
		{"размер", "размер"},
		{"размера", "размер"},
		{"размерам", "размер"},
		{"размере", "размер"},
		{"размеренно", "размерен"},
		{"размеров", "размер"},
		{"размеры", "размер"},
		{"разметает", "размета"},
		{"размечтался", "размечта"},
		{"разминается", "размина"},
		{"разминать", "размина"},
		{"размножится", "размнож"},
		{"размозжу", "размозж"},
		{"размолвки", "размолвк"},
		{"размучен", "размуч"},
		{"размышление", "размышлен"},
		{"размышлении", "размышлен"},
		{"размышления", "размышлен"},
		{"размышляет", "размышля"},
		{"размышлял", "размышля"},
		{"размышлять", "размышля"},
		{"размышляя", "размышл"},
		{"размягчило", "размягч"},
		{"размягчилось", "размягч"},
		{"разная", "разн"},
		{"разнежась", "разнеж"},
		{"разнежил", "разнеж"},
		{"разнесет", "разнесет"},
		{"разнесется", "разнесет"},
		{"разнесла", "разнесл"},
		{"разнесшейся", "разнесш"},
		{"разнился", "разн"},
		{"разнимались", "разнима"},
		{"разница", "разниц"},
		{"разницей", "разниц"},
		{"разницу", "разниц"},
		{"разновековой", "разновеков"},
		{"разного", "разн"},
		{"разное", "разн"},
		{"разнокалиберное", "разнокалиберн"},
		{"разноличного", "разноличн"},
		{"разнообразие", "разнообраз"},
		{"разнообразило", "разнообраз"},
		{"разнообразия", "разнообраз"},
		{"разнообразна", "разнообразн"},
		{"разнообразная", "разнообразн"},
		{"разнообразно", "разнообразн"},
		{"разнообразное", "разнообразн"},
		{"разнообразные", "разнообразн"},
		{"разнообразных", "разнообразн"},
		{"разнообразят", "разнообраз"},
		{"разнородной", "разнородн"},
		{"разнорядицу", "разнорядиц"},
		{"разносились", "разнос"},
		{"разносилось", "разнос"},
		{"разносится", "разнос"},
		{"разносить", "разнос"},
		{"разностями", "разност"},
		{"разносчик", "разносчик"},
		{"разносчики", "разносчик"},
		{"разнофасонное", "разнофасон"},
		{"разнохарактерный", "разнохарактерн"},
		{"разноцветные", "разноцветн"},
		{"разноцветным", "разноцветн"},
		{"разную", "разн"},
		{"разные", "разн"},
		{"разный", "разн"},
		{"разным", "разн"},
		{"разными", "разн"},
		{"разных", "разн"},
		{"разо", "раз"},
		{"разобидел", "разобидел"},
		{"разобижен", "разобиж"},
		{"разобрал", "разобра"},
		{"разобрала", "разобра"},
		{"разобрало", "разобра"},
		{"разобрано", "разобра"},
		{"разобраны", "разобра"},
		{"разобрать", "разобра"},
		{"разобраться", "разобра"},
		{"разобьют", "разобьют"},
		{"разовьют", "разовьют"},
		{"разогнавший", "разогна"},
		{"разогнать", "разогна"},
		{"разодета", "разодет"},
		{"разодетая", "разодет"},
		{"разодетой", "разодет"},
		{"разодетых", "разодет"},
		{"разодранной", "разодра"},
		{"разодраны", "разодра"},
		{"разожженным", "разожжен"},
		{"разозлившись", "разозл"},
		{"разозлил", "разозл"},
		{"разозлился", "разозл"},
		{"разойдемся", "разойд"},
		{"разойтись", "разойт"},
		{"разольет", "разольет"},
		{"разольется", "разольет"},
		{"разом", "раз"},
		{"разомкнуть", "разомкнут"},
		{"разорвал", "разорва"},
		{"разорвала", "разорва"},
		{"разорвали", "разорва"},
		{"разорвалось", "разорва"},
		{"разорванного", "разорва"},
		{"разорванное", "разорва"},
		{"разорванной", "разорва"},
		{"разорванные", "разорва"},
		{"разорванным", "разорва"},
		{"разорвано", "разорва"},
		{"разорвать", "разорва"},
		{"разорвусь", "разорв"},
		{"разорение", "разорен"},
		{"разорены", "разор"},
		{"разорила", "разор"},
		{"разорили", "разор"},
		{"разорились", "разор"},
		{"разорился", "разор"},
		{"разоряет", "разоря"},
		{"разорять", "разоря"},
		{"разостлали", "разостла"},
		{"разостлались", "разостла"},
		{"разочаровался", "разочарова"},
		{"разочарование", "разочарован"},
		{"разочарования", "разочарован"},
		{"разочарованный", "разочарова"},
		{"разочаровать", "разочарова"},
		{"разочароваться", "разочарова"},
		{"разочаровываться", "разочаровыва"},
		{"разочек", "разочек"},
		{"разошлась", "разошл"},
		{"разошлись", "разошл"},
		{"разработана", "разработа"},
		{"разработкою", "разработк"},
		{"разработывания", "разработыван"},
		{"разразилась", "разраз"},
		{"разразились", "разраз"},
		{"разразился", "разраз"},
		{"разразит", "разраз"},
		{"разразится", "разраз"},
		{"разразиться", "разраз"},
		{"разрежены", "разреж"},
		{"разрезал", "разреза"},
		{"разрешается", "разреша"},
		{"разрешаешь", "разреша"},
		{"разрешалась", "разреша"},
		{"разрешать", "разреша"},
		{"разрешаться", "разреша"},
		{"разрешаю", "разреша"},
		{"разрешение", "разрешен"},
		{"разрешения", "разрешен"},
		{"разрешено", "разреш"},
		{"разрешенье", "разрешен"},
		{"разрешил", "разреш"},
		{"разрешила", "разреш"},
		{"разрешили", "разреш"},
		{"разрешился", "разреш"},
		{"разрешите", "разреш"},
		{"разрешится", "разреш"},
		{"разрешить", "разреш"},
		{"разрешиться", "разреш"},
		{"разрешишь", "разреш"},
		{"разрозненных", "разрознен"},
		{"разронял", "разроня"},
		{"разрослась", "разросл"},
		{"разрушает", "разруша"},
		{"разрушаете", "разруша"},
		{"разрушаешь", "разруша"},
		{"разрушал", "разруша"},
		{"разрушать", "разруша"},
		{"разрушают", "разруша"},
		{"разрушающий", "разруша"},
		{"разрушающими", "разруша"},
		{"разрушая", "разруш"},
		{"разрушение", "разрушен"},
		{"разрушений", "разрушен"},
		{"разрушения", "разрушен"},
		{"разрушенное", "разрушен"},
		{"разрушено", "разруш"},
		{"разрушила", "разруш"},
		{"разрушители", "разрушител"},
		{"разрушительного", "разрушительн"},
		{"разрушительными", "разрушительн"},
		{"разрушителями", "разрушител"},
		{"разрушиться", "разруш"},
		{"разрыв", "разр"},
		{"разрыва", "разрыв"},
		{"разрывала", "разрыва"},
		{"разрывались", "разрыва"},
		{"разрывать", "разрыва"},
		{"разрывающим", "разрыва"},
		{"разрывая", "разрыв"},
		{"разрыве", "разрыв"},
		{"разрыву", "разрыв"},
		{"разряд", "разряд"},
		{"разряда", "разряд"},
		{"разрядится", "разряд"},
		{"разрядов", "разряд"},
		{"разряду", "разряд"},
		{"разряды", "разряд"},
		{"разряженные", "разряжен"},
		{"разряженных", "разряжен"},
		{"разу", "раз"},
		{"разубедить", "разубед"},
		{"разубеждали", "разубежда"},
		{"разубран", "разубра"},
		{"разуверило", "разувер"},
		{"разуверит", "разувер"},
		{"разуверить", "разувер"},
		{"разуверять", "разуверя"},
		{"разудалого", "разудал"},
		{"разузнавать", "разузнава"},
		{"разузнал", "разузна"},
		{"разузнали", "разузна"},
		{"разузнамши", "разузнамш"},
		{"разузнать", "разузна"},
		{"разузнаю", "разузна"},
		{"разукрашенную", "разукрашен"},
		{"разукрашенные", "разукрашен"},
		{"разукрашенный", "разукрашен"},
		{"разум", "разум"},
		{"разума", "разум"},
		{"разумеет", "разумеет"},
		{"разумеется", "разумеет"},
		{"разумеешь", "разумееш"},
		{"разумел", "разумел"},
		{"разумела", "разумел"},
		{"разумели", "разумел"},
		{"разумения", "разумен"},
		{"разумихин", "разумихин"},
		{"разумихина", "разумихин"},
		{"разумихине", "разумихин"},
		{"разумихину", "разумихин"},
		{"разумихиным", "разумихин"},
		{"разумнее", "разумн"},
		{"разумно", "разумн"},
		{"разумного", "разумн"},
		{"разумное", "разумн"},
		{"разумною", "разумн"},
		{"разумные", "разумн"},
		{"разумом", "разум"},
		{"разучился", "разуч"},
		{"разъединения", "разъединен"},
		{"разъезде", "разъезд"},
		{"разъездов", "разъезд"},
		{"разъезды", "разъезд"},
		{"разъезжать", "разъезжа"},
		{"разъезжая", "разъезж"},
		{"разъехавшись", "разъеха"},
		{"разъехались", "разъеха"},
		{"разъешься", "разъеш"},
		{"разъяснением", "разъяснен"},
		{"разъяснений", "разъяснен"},
		{"разъяснения", "разъяснен"},
		{"разъяснено", "разъясн"},
		{"разъяснил", "разъясн"},
		{"разъяснилось", "разъясн"},
		{"разъяснится", "разъясн"},
		{"разъяснить", "разъясн"},
		{"разъяснял", "разъясня"},
		{"разыграется", "разыгра"},
		{"разыгралась", "разыгра"},
		{"разыгрались", "разыгра"},
		{"разыгрался", "разыгра"},
		{"разыграть", "разыгра"},
		{"разыгрывавшиеся", "разыгрыва"},
		{"разыгрывается", "разыгрыва"},
		{"разыгрываешь", "разыгрыва"},
		{"разыгрывайте", "разыгрыва"},
		{"разыгрывалась", "разыгрыва"},
		{"разыгрывались", "разыгрыва"},
		{"разыгрывался", "разыгрыва"},
		{"разыскав", "разыска"},
		{"разыскал", "разыска"},
		{"разыскали", "разыска"},
		{"разыскать", "разыска"},
		{"разыскивал", "разыскива"},
		{"разыскиваний", "разыскиван"},
		{"разыскивать", "разыскива"},
		{"разыскиваю", "разыскива"},
		{"разыщу", "разыщ"},
		{"разящий", "разя"},
		{"рай", "ра"},
		{"райские", "райск"},
		{"райского", "райск"},
		{"райское", "райск"},
		{"рак", "рак"},
		{"раковина", "раковин"},
		{"раковинами", "раковин"},
		{"рама", "рам"},
		{"раме", "рам"},
		{"рамках", "рамк"},
		{"рамке", "рамк"},
		{"рамкой", "рамк"},
		{"рамой", "рам"},
		{"рамочку", "рамочк"},
		{"рампе", "рамп"},
		{"рамы", "рам"},
		{"ран", "ран"},
		{"ранах", "ран"},
		{"ране", "ран"},
		{"раневская", "раневск"},
		{"раневской", "раневск"},
		{"ранее", "ран"},
		{"ранен", "ран"},
		{"ранена", "ран"},
		{"раненого", "ранен"},
		{"ранешенько", "ранешеньк"},
		{"ранил", "ран"},
		{"ранили", "ран"},
		{"ранит", "ран"},
		{"ранить", "ран"},
		{"раннего", "ран"},
		{"ранней", "ран"},
		{"раннею", "ран"},
		{"ранние", "ран"},
		{"ранний", "ран"},
		{"ранним", "ран"},
		{"ранняя", "ран"},
		{"рано", "ран"},
		{"рану", "ран"},
		{"раны", "ран"},
		{"раным", "ран"},
		{"раньше", "раньш"},
		{"ранят", "ран"},
		{"рапортом", "рапорт"},
		{"раскаетесь", "раска"},
		{"раскаешься", "раска"},
		{"раскаивался", "раскаива"},
		{"раскаиваться", "раскаива"},
		{"раскаиваюсь", "раскаива"},
		{"раскаиваясь", "раскаив"},
		{"раскаленные", "раскален"},
		{"раскаленный", "раскален"},
		{"раскаленными", "раскален"},
		{"раскат", "раскат"},
		{"раскаты", "раскат"},
		{"раскачнувшись", "раскачнувш"},
		{"раскаялся", "раская"},
		{"раскаяние", "раскаян"},
		{"раскаянием", "раскаян"},
		{"раскаянии", "раскаян"},
		{"раскаяния", "раскаян"},
		{"раскаяться", "раская"},
		{"раскидал", "раскида"},
		{"раскинется", "раскинет"},
		{"раскинув", "раскинув"},
		{"раскинул", "раскинул"},
		{"раскинулась", "раскинул"},
		{"раскинулся", "раскинул"},
		{"раскинутыми", "раскинут"},
		{"раскладывает", "раскладыва"},
		{"раскладывать", "раскладыва"},
		{"раскладывают", "раскладыва"},
		{"раскланивается", "раскланива"},
		{"раскланялся", "раскланя"},
		{"расклю", "раскл"},
		{"расколдуешь", "расколдуеш"},
		{"расколот", "расколот"},
		{"расколотом", "расколот"},
		{"раскольников", "раскольник"},
		{"раскольникова", "раскольников"},
		{"раскольникове", "раскольников"},
		{"раскольниковой", "раскольников"},
		{"раскольникову", "раскольников"},
		{"раскольниковым", "раскольников"},
		{"раскольничьи", "раскольнич"},
		{"раскрасавицу", "раскрасавиц"},
		{"раскрасневшаяся", "раскрасневш"},
		{"раскрасневшимися", "раскрасневш"},
		{"раскраснелся", "раскраснел"},
		{"раскричалась", "раскрича"},
		{"раскричится", "раскрич"},
		{"раскрывал", "раскрыва"},
		{"раскрывались", "раскрыва"},
		{"раскрываться", "раскрыва"},
		{"раскрывают", "раскрыва"},
		{"раскрывая", "раскрыв"},
		{"раскрыл", "раскр"},
		{"раскрыла", "раскр"},
		{"раскрыли", "раскр"},
		{"раскрылись", "раскр"},
		{"раскрылся", "раскр"},
		{"раскрытая", "раскрыт"},
		{"раскрытым", "раскрыт"},
		{"раскрытыми", "раскрыт"},
		{"раскрыть", "раскр"},
		{"раскуси", "раскус"},
		{"раскусивший", "раскус"},
		{"раскусить", "раскус"},
		{"распался", "распа"},
		{"распарит", "распар"},
		{"распахивает", "распахива"},
		{"распахнувшийся", "распахнувш"},
		{"распахнул", "распахнул"},
		{"распахнулась", "распахнул"},
		{"распахнулись", "распахнул"},
		{"распахнут", "распахнут"},
		{"распашку", "распашк"},
		{"распечатал", "распечата"},
		{"распечатали", "распечата"},
		{"распечатывай", "распечатыва"},
		{"распечатывать", "распечатыва"},
		{"распечатывая", "распечатыв"},
		{"распивочная", "распивочн"},
		{"распивочной", "распивочн"},
		{"распивочные", "распивочн"},
		{"распивочными", "распивочн"},
		{"распивочных", "распивочн"},
		{"распивошной", "распивошн"},
		{"распирать", "распира"},
		{"расписалась", "расписа"},
		{"расписался", "расписа"},
		{"расписание", "расписан"},
		{"расписки", "расписк"},
		{"расписок", "расписок"},
		{"расписочки", "расписочк"},
		{"расписывались", "расписыва"},
		{"расписываться", "расписыва"},
		{"расплакалась", "расплака"},
		{"расплата", "расплат"},
		{"расплатившись", "расплат"},
		{"расплатился", "расплат"},
		{"расплатиться", "расплат"},
		{"расплатишься", "расплат"},
		{"расплачутся", "расплачут"},
		{"расплевывался", "расплевыва"},
		{"расплескав", "расплеска"},
		{"расплести", "расплест"},
		{"расплодилось", "расплод"},
		{"расплывается", "расплыва"},
		{"расплываться", "расплыва"},
		{"расплываясь", "расплыв"},
		{"расплылось", "распл"},
		{"расплюеве", "расплюев"},
		{"распни", "распн"},
		{"распознавать", "распознава"},
		{"распознать", "распозна"},
		{"располагается", "располага"},
		{"располагаешь", "располага"},
		{"располагал", "располага"},
		{"располагать", "располага"},
		{"располагаются", "располага"},
		{"располагающий", "располага"},
		{"расползается", "располза"},
		{"расползлась", "расползл"},
		{"располнел", "располнел"},
		{"расположен", "располож"},
		{"расположение", "расположен"},
		{"расположении", "расположен"},
		{"расположения", "расположен"},
		{"расположены", "располож"},
		{"расположил", "располож"},
		{"распомаженный", "распомажен"},
		{"распорядилась", "распоряд"},
		{"распорядился", "распоряд"},
		{"распорядись", "распоряд"},
		{"распорядительнице", "распорядительниц"},
		{"распорядительности", "распорядительн"},
		{"распорядительность", "распорядительн"},
		{"распорядится", "распоряд"},
		{"распорядиться", "распоряд"},
		{"распорядок", "распорядок"},
		{"распорядятся", "распоряд"},
		{"распоряжаетесь", "распоряжа"},
		{"распоряжается", "распоряжа"},
		{"распоряжалась", "распоряжа"},
		{"распоряжался", "распоряжа"},
		{"распоряжение", "распоряжен"},
		{"распоряжений", "распоряжен"},
		{"распоряжению", "распоряжен"},
		{"распоряжения", "распоряжен"},
		{"распоряжениям", "распоряжен"},
		{"распоряжусь", "распоряж"},
		{"расправил", "расправ"},
		{"расправила", "расправ"},
		{"расправиться", "расправ"},
		{"расправляя", "расправл"},
		{"расправу", "расправ"},
		{"распределить", "распредел"},
		{"распри", "распр"},
		{"распродал", "распрода"},
		{"распростерла", "распростерл"},
		{"распростерлось", "распростерл"},
		{"распростертый", "распростерт"},
		{"распростершийся", "распростерш"},
		{"распространение", "распространен"},
		{"распространенное", "распространен"},
		{"распространены", "распростран"},
		{"распространилась", "распростран"},
		{"распространилось", "распростран"},
		{"распространился", "распростран"},
		{"распространить", "распростран"},
		{"распространяется", "распространя"},
		{"распространять", "распространя"},
		{"распрях", "распр"},
		{"распугала", "распуга"},
		{"распускается", "распуска"},
		{"распускал", "распуска"},
		{"распускала", "распуска"},
		{"распускались", "распуска"},
		{"распускать", "распуска"},
		{"распустив", "распуст"},
		{"распустившийся", "распуст"},
		{"распустил", "распуст"},
		{"распустила", "распуст"},
		{"распустили", "распуст"},
		{"распустился", "распуст"},
		{"распустить", "распуст"},
		{"распутает", "распута"},
		{"распутица", "распутиц"},
		{"распухшее", "распухш"},
		{"распущенность", "распущен"},
		{"распущенные", "распущен"},
		{"распущены", "распущ"},
		{"распылавшийся", "распыла"},
		{"распяв", "распя"},
		{"распятие", "распят"},
		{"распять", "распя"},
		{"рассадить", "рассад"},
		{"рассажены", "рассаж"},
		{"рассвет", "рассвет"},
		{"рассвета", "рассвет"},
		{"рассвете", "рассвет"},
		{"рассветет", "рассветет"},
		{"рассветом", "рассвет"},
		{"рассвирепев", "рассвиреп"},
		{"рассвирепел", "рассвирепел"},
		{"рассеет", "рассеет"},
		{"расселись", "рассел"},
		{"рассердившийся", "рассерд"},
		{"рассердившись", "рассерд"},
		{"рассердилась", "рассерд"},
		{"рассердили", "рассерд"},
		{"рассердились", "рассерд"},
		{"рассердился", "рассерд"},
		{"рассердитесь", "рассерд"},
		{"рассердится", "рассерд"},
		{"рассердиться", "рассерд"},
		{"рассердятся", "рассерд"},
		{"рассержен", "рассерж"},
		{"рассерженный", "рассержен"},
		{"рассержены", "рассерж"},
		{"рассечь", "рассеч"},
		{"рассеялось", "рассея"},
		{"рассеян", "рассея"},
		{"рассеянна", "рассеян"},
		{"рассеянно", "рассея"},
		{"рассеянности", "рассеян"},
		{"рассеянность", "рассеян"},
		{"рассеянностью", "рассеян"},
		{"рассеянны", "рассеян"},
		{"рассеянный", "рассея"},
		{"рассеянным", "рассея"},
		{"рассеяно", "рассея"},
		{"рассеять", "рассея"},
		{"рассеяться", "рассея"},
		{"расскажет", "расскажет"},
		{"расскажете", "расскажет"},
		{"расскажешь", "расскажеш"},
		{"расскажи", "расскаж"},
		{"расскажит", "расскаж"},
		{"расскажите", "расскаж"},
		{"расскажу", "расскаж"},
		{"расскажут", "расскажут"},
		{"рассказ", "рассказ"},
		{"рассказа", "рассказ"},
		{"рассказав", "рассказа"},
		{"рассказал", "рассказа"},
		{"рассказала", "рассказа"},
		{"рассказали", "рассказа"},
		{"рассказам", "рассказ"},
		{"рассказами", "рассказ"},
		{"рассказана", "рассказа"},
		{"рассказанный", "рассказа"},
		{"рассказать", "рассказа"},
		{"рассказах", "рассказ"},
		{"рассказе", "рассказ"},
		{"рассказов", "рассказ"},
		{"рассказом", "рассказ"},
		{"рассказу", "рассказ"},
		{"рассказчика", "рассказчик"},
		{"рассказчиц", "рассказчиц"},
		{"рассказы", "рассказ"},
		{"рассказывает", "рассказыва"},
		{"рассказываешь", "рассказыва"},
		{"рассказывай", "рассказыва"},
		{"рассказывайте", "рассказыва"},
		{"рассказывал", "рассказыва"},
		{"рассказывала", "рассказыва"},
		{"рассказывали", "рассказыва"},
		{"рассказывать", "рассказыва"},
		{"рассказываю", "рассказыва"},
		{"рассказывают", "рассказыва"},
		{"расслабленного", "расслаблен"},
		{"расслабленный", "расслаблен"},
		{"расслабленным", "расслаблен"},
		{"расследован", "расследова"},
		{"расслушала", "расслуша"},
		{"расслушать", "расслуша"},
		{"расслышав", "расслыша"},
		{"расслышал", "расслыша"},
		{"расслышала", "расслыша"},
		{"расслышать", "расслыша"},
		{"рассматривавшего", "рассматрива"},
		{"рассматривает", "рассматрива"},
		{"рассматривается", "рассматрива"},
		{"рассматривал", "рассматрива"},
		{"рассматривала", "рассматрива"},
		{"рассматривали", "рассматрива"},
		{"рассматривания", "рассматриван"},
		{"рассматривать", "рассматрива"},
		{"рассматривают", "рассматрива"},
		{"рассматривая", "рассматрив"},
		{"рассмеетесь", "рассмеет"},
		{"рассмешат", "рассмешат"},
		{"рассмешил", "рассмеш"},
		{"рассмешила", "рассмеш"},
		{"рассмешили", "рассмеш"},
		{"рассмешит", "рассмеш"},
		{"рассмеявшись", "рассмея"},
		{"рассмеялся", "рассмея"},
		{"рассмеяться", "рассмея"},
		{"рассмотрев", "рассмотр"},
		{"рассмотрение", "рассмотрен"},
		{"рассмотреть", "рассмотрет"},
		{"рассориться", "рассор"},
		{"расспрашивайте", "расспрашива"},
		{"расспрашивал", "расспрашива"},
		{"расспрашивала", "расспрашива"},
		{"расспрашивали", "расспрашива"},
		{"расспрашивать", "расспрашива"},
		{"расспрашивая", "расспрашив"},
		{"расспросами", "расспрос"},
		{"расспросила", "расспрос"},
		{"расспросит", "расспрос"},
		{"расспросить", "расспрос"},
		{"расспросов", "расспрос"},
		{"расспросы", "расспрос"},
		{"расставаться", "расстава"},
		{"расставив", "расстав"},
		{"расставил", "расстав"},
		{"расставила", "расстав"},
		{"расставили", "расстав"},
		{"расставлено", "расставл"},
		{"расставляет", "расставля"},
		{"расстаемся", "расста"},
		{"расстается", "расста"},
		{"рассталась", "расста"},
		{"расстались", "расста"},
		{"расстался", "расста"},
		{"расстанемся", "расстан"},
		{"расстановисто", "расстановист"},
		{"расстановками", "расстановк"},
		{"расстановкой", "расстановк"},
		{"расстаться", "расста"},
		{"расстаются", "расста"},
		{"расстегнул", "расстегнул"},
		{"расстегнуть", "расстегнут"},
		{"расстелет", "расстелет"},
		{"расстилалася", "расстила"},
		{"расстилались", "расстила"},
		{"расстилаться", "расстила"},
		{"расстояние", "расстоян"},
		{"расстоянии", "расстоян"},
		{"расстояния", "расстоян"},
		{"расстраивались", "расстраива"},
		{"расстроен", "расстро"},
		{"расстроена", "расстро"},
		{"расстроенного", "расстроен"},
		{"расстроенной", "расстроен"},
		{"расстроенную", "расстроен"},
		{"расстроенные", "расстроен"},
		{"расстроенный", "расстроен"},
		{"расстроено", "расстро"},
		{"расстроены", "расстро"},
		{"расстроивает", "расстроива"},
		{"расстроил", "расстро"},
		{"расстроили", "расстро"},
		{"расстроить", "расстро"},
		{"расстроиться", "расстро"},
		{"расстроишь", "расстро"},
		{"расстроишься", "расстро"},
		{"расстройства", "расстройств"},
		{"расстройстве", "расстройств"},
		{"расстройство", "расстройств"},
		{"расступись", "расступ"},
		{"рассуди", "рассуд"},
		{"рассудив", "рассуд"},
		{"рассудил", "рассуд"},
		{"рассудила", "рассуд"},
		{"рассудилось", "рассуд"},
		{"рассудит", "рассуд"},
		{"рассудите", "рассуд"},
		{"рассудительна", "рассудительн"},
		{"рассудительная", "рассудительн"},
		{"рассудительнее", "рассудительн"},
		{"рассудительный", "рассудительн"},
		{"рассудить", "рассуд"},
		{"рассудка", "рассудк"},
		{"рассудке", "рассудк"},
		{"рассудкина", "рассудкин"},
		{"рассудком", "рассудк"},
		{"рассудок", "рассудок"},
		{"рассуждает", "рассужда"},
		{"рассуждаете", "рассужда"},
		{"рассуждаешь", "рассужда"},
		{"рассуждал", "рассужда"},
		{"рассуждать", "рассужда"},
		{"рассуждаю", "рассужда"},
		{"рассуждают", "рассужда"},
		{"рассуждая", "рассужд"},
		{"рассуждение", "рассужден"},
		{"рассуждений", "рассужден"},
		{"рассуждению", "рассужден"},
		{"рассуждения", "рассужден"},
		{"рассуждениями", "рассужден"},
		{"рассчитав", "рассчита"},
		{"рассчитайте", "рассчита"},
		{"рассчитал", "рассчита"},
		{"рассчитали", "рассчита"},
		{"рассчитаны", "рассчита"},
		{"рассчитать", "рассчита"},
		{"рассчитываем", "рассчитыва"},
		{"рассчитывает", "рассчитыва"},
		{"рассчитываете", "рассчитыва"},
		{"рассчитывай", "рассчитыва"},
		{"рассчитывал", "рассчитыва"},
		{"рассчитывали", "рассчитыва"},
		{"рассчитывать", "рассчитыва"},
		{"рассчитываю", "рассчитыва"},
		{"рассчитывая", "рассчитыв"},
		{"рассыпается", "рассыпа"},
		{"рассыпала", "рассыпа"},
		{"рассыпалась", "рассыпа"},
		{"рассыпались", "рассыпа"},
		{"рассыпался", "рассыпа"},
		{"рассыпчатые", "рассыпчат"},
		{"расталкивает", "расталкива"},
		{"расталкивать", "расталкива"},
		{"растаял", "растая"},
		{"растворена", "раствор"},
		{"растворенных", "растворен"},
		{"растворил", "раствор"},
		{"растворилась", "раствор"},
		{"растворились", "раствор"},
		{"растворить", "раствор"},
		{"растворяя", "раствор"},
		{"растение", "растен"},
		{"растениями", "растен"},
		{"растерзанном", "растерза"},
		{"растерявшаяся", "растеря"},
		{"растерявшимся", "растеря"},
		{"растерявшись", "растеря"},
		{"растеряется", "растеря"},
		{"растеряла", "растеря"},
		{"растеряли", "растеря"},
		{"растерялся", "растеря"},
		{"растерян", "растеря"},
		{"растерянно", "растеря"},
		{"растерянные", "растеря"},
		{"растеряют", "растеря"},
		{"растет", "растет"},
		{"расти", "раст"},
		{"растлелось", "растлел"},
		{"растолкала", "растолка"},
		{"растолковал", "растолкова"},
		{"растолковать", "растолкова"},
		{"растолковывать", "растолковыва"},
		{"растолкуйте", "растолк"},
		{"растолкую", "растолк"},
		{"растопалась", "растопа"},
		{"растопчет", "растопчет"},
		{"растопыренных", "растопырен"},
		{"расторопного", "расторопн"},
		{"расторопный", "расторопн"},
		{"расточаемые", "расточа"},
		{"расточает", "расточа"},
		{"расточаю", "расточа"},
		{"расточилась", "расточ"},
		{"расточители", "расточител"},
		{"расточительности", "расточительн"},
		{"растравлять", "растравля"},
		{"растревожили", "растревож"},
		{"растреклятого", "растреклят"},
		{"растреклятым", "растреклят"},
		{"растрепанная", "растрепа"},
		{"растрепанного", "растрепа"},
		{"растрепанную", "растрепа"},
		{"растроган", "растрога"},
		{"растроганная", "растрога"},
		{"растроганной", "растрога"},
		{"растроганный", "растрога"},
		{"растут", "растут"},
		{"растущего", "растущ"},
		{"растущей", "растущ"},
		{"растягивает", "растягива"},
		{"растянет", "растянет"},
		{"растянулись", "растянул"},
		{"растянулся", "растянул"},
		{"расфранченную", "расфранчен"},
		{"расфуфыренных", "расфуфырен"},
		{"расхаживаешь", "расхажива"},
		{"расхаживают", "расхажива"},
		{"расхищения", "расхищен"},
		{"расхлебывают", "расхлебыва"},
		{"расхлябанные", "расхляба"},
		{"расход", "расход"},
		{"расхода", "расход"},
		{"расходились", "расход"},
		{"расходился", "расход"},
		{"расходится", "расход"},
		{"расходными", "расходн"},
		{"расходы", "расход"},
		{"расходясь", "расход"},
		{"расходятся", "расход"},
		{"расхохоталась", "расхохота"},
		{"расхохотался", "расхохота"},
		{"расцвела", "расцвел"},
		{"расцвело", "расцвел"},
		{"расцвет", "расцвет"},
		{"расцветала", "расцвета"},
		{"расцветало", "расцвета"},
		{"расцветание", "расцветан"},
		{"расцветания", "расцветан"},
		{"расцветая", "расцвет"},
		{"расцветшей", "расцветш"},
		{"расчесанные", "расчеса"},
		{"расчесанный", "расчеса"},
		{"расчесывает", "расчесыва"},
		{"расчесывая", "расчесыв"},
		{"расчет", "расчет"},
		{"расчета", "расчет"},
		{"расчетам", "расчет"},
		{"расчетах", "расчет"},
		{"расчетливый", "расчетлив"},
		{"расчетов", "расчет"},
		{"расчетом", "расчет"},
		{"расчету", "расчет"},
		{"расчеты", "расчет"},
		{"расчищаем", "расчища"},
		{"расчувствовался", "расчувствова"},
		{"расшалясь", "расшал"},
		{"расшаталось", "расшата"},
		{"расшевелили", "расшевел"},
		{"расшевелить", "расшевел"},
		{"расшиб", "расшиб"},
		{"расшибанья", "расшибан"},
		{"расшибет", "расшибет"},
		{"расшибется", "расшибет"},
		{"расширением", "расширен"},
		{"расширены", "расшир"},
		{"расширялась", "расширя"},
		{"расширялся", "расширя"},
		{"расширяя", "расшир"},
		{"расщедрилась", "расщедр"},
		{"расщедрился", "расщедр"},
		{"ратую", "рат"},
		{"рать", "рат"},
		{"раут", "раут"},
		{"рафаэлевой", "рафаэлев"},
		{"рафаэлевских", "рафаэлевск"},
		{"рафаэля", "рафаэл"},
		{"рацею", "рац"},
		{"рационально", "рациональн"},
		{"рационального", "рациональн"},
		{"рациональный", "рациональн"},
		{"рачителен", "рачител"},
		{"рачительно", "рачительн"},
		{"раю", "ра"},
		{"рая", "ра"},
		{"рвал", "рвал"},
		{"рвало", "рвал"},
		{"рвался", "рвал"},
		{"рвание", "рван"},
		{"рваного", "рван"},
		{"рваной", "рван"},
		{"рванула", "рванул"},
		{"рванулась", "рванул"},
		{"рванулся", "рванул"},
		{"рвануться", "рванут"},
		{"рваные", "рван"},
		{"рвать", "рват"},
		{"рваться", "рват"},
		{"рвение", "рвен"},
		{"рвет", "рвет"},
		{"рвется", "рвет"},
		{"рвешь", "рвеш"},
		{"рвешься", "рвеш"},
		{"рву", "рву"},
		{"рдеет", "рдеет"},
		{"рдели", "рдел"},
		{"рдеют", "рдеют"},
		{"рдея", "рде"},
		{"реализме", "реализм"},
		{"реализмом", "реализм"},
		{"реальное", "реальн"},
		{"ребенка", "ребенк"},
		{"ребенке", "ребенк"},
		{"ребенком", "ребенк"},
		{"ребенку", "ребенк"},
		{"ребенок", "ребенок"},
		{"ребеночек", "ребеночек"},
		{"ребер", "ребер"},
		{"ребра", "ребр"},
		{"ребре", "ребр"},
		{"ребят", "реб"},
		{"ребята", "ребят"},
		{"ребятишек", "ребятишек"},
		{"ребятишкам", "ребятишк"},
		{"ребятишками", "ребятишк"},
		{"ребятишки", "ребятишк"},
		{"ребяческая", "ребяческ"},
		{"ребяческий", "ребяческ"},
		{"ребячества", "ребячеств"},
		{"ребячий", "ребяч"},
		{"ребячьих", "ребяч"},
		{"рев", "рев"},
		{"ревель", "ревел"},
		{"реверанс", "реверанс"},
		{"реви", "рев"},
		{"ревизор", "ревизор"},
		{"ревматизм", "ревматизм"},
		{"ревматизмом", "ревматизм"},
		{"ревнивая", "ревнив"},
		{"ревнивее", "ревнив"},
		{"ревниво", "ревнив"},
		{"ревнивой", "ревнив"},
		{"ревнивый", "ревнив"},
		{"ревновал", "ревнова"},
		{"ревновала", "ревнова"},
		{"ревновать", "ревнова"},
		{"ревности", "ревност"},
		{"ревность", "ревност"},
		{"ревностью", "ревност"},
		{"ревнует", "ревн"},
		{"ревнуя", "ревну"},
		{"револьвер", "револьвер"},
		{"револьвером", "револьвер"},
		{"революции", "революц"},
		{"революционных", "революцион"},
		{"революция", "революц"},
		{"ревом", "рев"},
		{"ревущих", "ревущ"},
		{"регистраторша", "регистраторш"},
		{"регистраторши", "регистраторш"},
		{"регре", "регр"},
		{"регулировать", "регулирова"},
		{"регулярно", "регулярн"},
		{"редактора", "редактор"},
		{"редакции", "редакц"},
		{"редакций", "редакц"},
		{"редакция", "редакц"},
		{"редеет", "редеет"},
		{"редеть", "редет"},
		{"редкая", "редк"},
		{"редки", "редк"},
		{"редкие", "редк"},
		{"редкий", "редк"},
		{"редким", "редк"},
		{"редкими", "редк"},
		{"редких", "редк"},
		{"редко", "редк"},
		{"редкое", "редк"},
		{"редкой", "редк"},
		{"редкости", "редкост"},
		{"редкость", "редкост"},
		{"редька", "редьк"},
		{"редьки", "редьк"},
		{"редьку", "редьк"},
		{"реет", "реет"},
		{"реже", "реж"},
		{"режет", "режет"},
		{"режу", "реж"},
		{"резались", "реза"},
		{"резанная", "реза"},
		{"резать", "реза"},
		{"резв", "резв"},
		{"резва", "резв"},
		{"резвая", "резв"},
		{"резвому", "резв"},
		{"резвость", "резвост"},
		{"резвы", "резв"},
		{"резвые", "резв"},
		{"резвясь", "резв"},
		{"резвятся", "резв"},
		{"резедой", "резед"},
		{"резеду", "резед"},
		{"резеды", "резед"},
		{"резиденция", "резиденц"},
		{"резиновых", "резинов"},
		{"резкая", "резк"},
		{"резкие", "резк"},
		{"резкий", "резк"},
		{"резким", "резк"},
		{"резкими", "резк"},
		{"резких", "резк"},
		{"резко", "резк"},
		{"резкой", "резк"},
		{"резком", "резк"},
		{"резкость", "резкост"},
		{"резной", "резн"},
		{"резню", "резн"},
		{"резок", "резок"},
		{"резоны", "резон"},
		{"резоня", "резон"},
		{"результат", "результат"},
		{"результатах", "результат"},
		{"результате", "результат"},
		{"результатов", "результат"},
		{"результату", "результат"},
		{"результаты", "результат"},
		{"резцом", "резц"},
		{"резче", "резч"},
		{"резюме", "резюм"},
		{"рейнгольду", "рейнгольд"},
		{"рейт", "рейт"},
		{"рек", "рек"},
		{"река", "рек"},
		{"рекам", "рек"},
		{"рекамье", "рекам"},
		{"реке", "рек"},
		{"реки", "рек"},
		{"рекой", "рек"},
		{"рекомендации", "рекомендац"},
		{"рекомендацию", "рекомендац"},
		{"рекомендация", "рекомендац"},
		{"рекомендовал", "рекомендова"},
		{"рекомендовать", "рекомендова"},
		{"рекомендует", "рекоменд"},
		{"рекомендуется", "рекоменд"},
		{"рекомендую", "рекоменд"},
		{"рекомендуя", "рекоменду"},
		{"рекорд", "рекорд"},
		{"рекою", "рек"},
		{"ректор", "ректор"},
		{"реку", "рек"},
		{"религией", "религ"},
		{"религии", "религ"},
		{"религий", "религ"},
		{"религиозно", "религиозн"},
		{"рельефностью", "рельефн"},
		{"рельс", "рельс"},
		{"рельсах", "рельс"},
		{"рельсы", "рельс"},
		{"реляции", "реляц"},
		{"реляцию", "реляц"},
		{"ремень", "ремен"},
		{"ремесла", "ремесл"},
		{"ремесленник", "ремесленник"},
		{"ремесленники", "ремесленник"},
		{"ремесленное", "ремеслен"},
		{"ремеслу", "ремесл"},
		{"ремне", "ремн"},
		{"ремни", "ремн"},
		{"ренегате", "ренегат"},
		{"реникса", "реникс"},
		{"ренную", "рен"},
		{"рент", "рент"},
		{"реомюра", "реомюр"},
		{"репетитором", "репетитор"},
		{"реплики", "реплик"},
		{"репой", "реп"},
		{"репу", "реп"},
		{"репутацией", "репутац"},
		{"репутации", "репутац"},
		{"репутаций", "репутац"},
		{"репутацию", "репутац"},
		{"репы", "реп"},
		{"ресниц", "ресниц"},
		{"ресницами", "ресниц"},
		{"ресницах", "ресниц"},
		{"ресницы", "ресниц"},
		{"ресслих", "рессл"},
		{"рессорную", "рессорн"},
		{"ресторан", "рестора"},
		{"ресторана", "рестора"},
		{"ресторане", "ресторан"},
		{"ретивое", "ретив"},
		{"ретировавшегося", "ретирова"},
		{"реторту", "реторт"},
		{"ретроград", "ретроград"},
		{"ретроградна", "ретроградн"},
		{"рефлексом", "рефлекс"},
		{"реформа", "реформ"},
		{"реформы", "реформ"},
		{"рехнулись", "рехнул"},
		{"рехнулся", "рехнул"},
		{"рецепт", "рецепт"},
		{"рецепта", "рецепт"},
		{"рецептов", "рецепт"},
		{"речами", "реч"},
		{"речах", "реч"},
		{"речей", "реч"},
		{"речи", "реч"},
		{"речисто", "речист"},
		{"речка", "речк"},
		{"речке", "речк"},
		{"речки", "речк"},
		{"речкой", "речк"},
		{"речку", "речк"},
		{"речной", "речн"},
		{"речь", "реч"},
		{"речью", "реч"},
		{"решает", "реша"},
		{"решаете", "реша"},
		{"решается", "реша"},
		{"решаешься", "реша"},
		{"решай", "реша"},
		{"решайтесь", "реша"},
		{"решал", "реша"},
		{"решала", "реша"},
		{"решалась", "реша"},
		{"решались", "реша"},
		{"решался", "реша"},
		{"решатся", "решат"},
		{"решать", "реша"},
		{"решаюсь", "реша"},
		{"решают", "реша"},
		{"решаясь", "реш"},
		{"решен", "реш"},
		{"решена", "реш"},
		{"решение", "решен"},
		{"решением", "решен"},
		{"решении", "решен"},
		{"решений", "решен"},
		{"решения", "решен"},
		{"решениям", "решен"},
		{"решенном", "решен"},
		{"решенный", "решен"},
		{"решено", "реш"},
		{"решетку", "решетк"},
		{"решетчатого", "решетчат"},
		{"решетчаты", "решетчат"},
		{"решетчатые", "решетчат"},
		{"решетчатым", "решетчат"},
		{"реши", "реш"},
		{"решив", "реш"},
		{"решившегося", "реш"},
		{"решившей", "реш"},
		{"решившийся", "реш"},
		{"решившись", "реш"},
		{"решил", "реш"},
		{"решила", "реш"},
		{"решилась", "реш"},
		{"решили", "реш"},
		{"решились", "реш"},
		{"решился", "реш"},
		{"решим", "реш"},
		{"решимости", "решим"},
		{"решимость", "решим"},
		{"решимостью", "решим"},
		{"решит", "реш"},
		{"решите", "реш"},
		{"решителем", "решител"},
		{"решительная", "решительн"},
		{"решительнее", "решительн"},
		{"решительно", "решительн"},
		{"решительного", "решительн"},
		{"решительное", "решительн"},
		{"решительном", "решительн"},
		{"решительности", "решительн"},
		{"решительную", "решительн"},
		{"решительный", "решительн"},
		{"решительным", "решительн"},
		{"решится", "реш"},
		{"решить", "реш"},
		{"решиться", "реш"},
		{"решоткой", "решотк"},
		{"решусь", "реш"},
		{"решь", "реш"},
		{"реяли", "рея"},
		{"ржавой", "ржав"},
		{"ржавые", "ржав"},
		{"ржавый", "ржав"},
		{"ржавых", "ржав"},
		{"ржаной", "ржан"},
		{"ржи", "ржи"},
		{"ризами", "риз"},
		{"ризой", "риз"},
		{"рим", "рим"},
		{"рима", "рим"},
		{"римляне", "римлян"},
		{"римские", "римск"},
		{"ринется", "ринет"},
		{"ринулся", "ринул"},
		{"ринутся", "ринут"},
		{"ринуться", "ринут"},
		{"рис", "рис"},
		{"риск", "риск"},
		{"риски", "риск"},
		{"рискнул", "рискнул"},
		{"рискнула", "рискнул"},
		{"рискнули", "рискнул"},
		{"риском", "риск"},
		{"рисовал", "рисова"},
		{"рисовала", "рисова"},
		{"рисовало", "рисова"},
		{"рисовалось", "рисова"},
		{"рисовался", "рисова"},
		{"рисования", "рисован"},
		{"рисовать", "рисова"},
		{"рисовой", "рисов"},
		{"рису", "рис"},
		{"рисует", "рис"},
		{"рисуешь", "рисуеш"},
		{"рисунка", "рисунк"},
		{"рисунков", "рисунк"},
		{"рисунок", "рисунок"},
		{"рисую", "рис"},
		{"рисуюсь", "рис"},
		{"рисуя", "рису"},
		{"ритм", "ритм"},
		{"ритма", "ритм"},
		{"ритмическое", "ритмическ"},
		{"ритмы", "ритм"},
		{"риторика", "риторик"},
		{"риторики", "риторик"},
		{"рифм", "рифм"},
		{"рифмованную", "рифмова"},
		{"ро", "ро"},
		{"робел", "робел"},
		{"робела", "робел"},
		{"робели", "робел"},
		{"робенок", "робенок"},
		{"роберте", "роберт"},
		{"робеют", "робеют"},
		{"робея", "робе"},
		{"робка", "робк"},
		{"робкая", "робк"},
		{"робкие", "робк"},
		{"робкий", "робк"},
		{"робким", "робк"},
		{"робко", "робк"},
		{"робкого", "робк"},
		{"робкой", "робк"},
		{"робком", "робк"},
		{"робок", "робок"},
		{"робостию", "робост"},
		{"робость", "робост"},
		{"робостью", "робост"},
		{"робче", "робч"},
		{"ровесник", "ровесник"},
		{"ровна", "ровн"},
		{"ровнее", "ровн"},
		{"ровнехонько", "ровнехоньк"},
		{"ровно", "ровн"},
		{"ровное", "ровн"},
		{"ровной", "ровн"},
		{"ровны", "ровн"},
		{"ровные", "ровн"},
		{"ровный", "ровн"},
		{"ровным", "ровн"},
		{"ровняет", "ровня"},
		{"рог", "рог"},
		{"рога", "рог"},
		{"рогам", "рог"},
		{"рогами", "рог"},
		{"рогатого", "рогат"},
		{"рогатые", "рогат"},
		{"рогов", "рог"},
		{"роговой", "рогов"},
		{"рогожею", "рогож"},
		{"род", "род"},
		{"рода", "род"},
		{"родах", "род"},
		{"роде", "род"},
		{"родей", "род"},
		{"роденьку", "роденьк"},
		{"роди", "род"},
		{"родила", "род"},
		{"родилась", "род"},
		{"родились", "род"},
		{"родилось", "род"},
		{"родился", "род"},
		{"родименький", "родименьк"},
		{"родимой", "родим"},
		{"родимый", "родим"},
		{"родина", "родин"},
		{"родине", "родин"},
		{"родиной", "родин"},
		{"родину", "родин"},
		{"родины", "родин"},
		{"родион", "родион"},
		{"родиона", "родион"},
		{"родионом", "родион"},
		{"родиону", "родион"},
		{"родионыч", "родионыч"},
		{"родителей", "родител"},
		{"родители", "родител"},
		{"родитель", "родител"},
		{"родительская", "родительск"},
		{"родительский", "родительск"},
		{"родительского", "родительск"},
		{"родительском", "родительск"},
		{"родителя", "родител"},
		{"родителями", "родител"},
		{"родится", "род"},
		{"родиться", "род"},
		{"родная", "родн"},
		{"родне", "родн"},
		{"роднейший", "родн"},
		{"родни", "родн"},
		{"роднила", "родн"},
		{"родного", "родн"},
		{"родное", "родн"},
		{"родной", "родн"},
		{"родном", "родн"},
		{"родную", "родн"},
		{"родные", "родн"},
		{"родным", "родн"},
		{"родными", "родн"},
		{"родных", "родн"},
		{"родню", "родн"},
		{"родня", "родн"},
		{"родов", "род"},
		{"родом", "род"},
		{"родства", "родств"},
		{"родственник", "родственник"},
		{"родственника", "родственник"},
		{"родственнике", "родственник"},
		{"родственники", "родственник"},
		{"родственников", "родственник"},
		{"родственнику", "родственник"},
		{"родственница", "родственниц"},
		{"родственницей", "родственниц"},
		{"родственницу", "родственниц"},
		{"родственницы", "родственниц"},
		{"родственного", "родствен"},
		{"родственное", "родствен"},
		{"родственной", "родствен"},
		{"родственному", "родствен"},
		{"родственные", "родствен"},
		{"родственный", "родствен"},
		{"родственных", "родствен"},
		{"родство", "родств"},
		{"роду", "род"},
		{"родька", "родьк"},
		{"родьке", "родьк"},
		{"родькин", "родькин"},
		{"родэ", "родэ"},
		{"родю", "род"},
		{"родя", "род"},
		{"родясь", "род"},
		{"родятся", "род"},
		{"роем", "ро"},
		{"роется", "роет"},
		{"роешься", "роеш"},
		{"рожа", "рож"},
		{"рожами", "рож"},
		{"рождает", "рожда"},
		{"рождается", "рожда"},
		{"рождал", "рожда"},
		{"рождали", "рожда"},
		{"рождалось", "рожда"},
		{"рождался", "рожда"},
		{"рождают", "рожда"},
		{"рождающейся", "рожда"},
		{"рождающимся", "рожда"},
		{"рождающихся", "рожда"},
		{"рождение", "рожден"},
		{"рождением", "рожден"},
		{"рождении", "рожден"},
		{"рождения", "рожден"},
		{"рожденные", "рожден"},
		{"рожденья", "рожден"},
		{"рождества", "рождеств"},
		{"рождестве", "рождеств"},
		{"рождественном", "рождествен"},
		{"рождество", "рождеств"},
		{"рождеством", "рождеств"},
		{"рождеству", "рождеств"},
		{"роже", "рож"},
		{"рожи", "рож"},
		{"рожком", "рожк"},
		{"рожок", "рожок"},
		{"рожу", "рож"},
		{"рожь", "рож"},
		{"рожью", "рож"},
		{"роз", "роз"},
		{"роза", "роз"},
		{"розан", "роза"},
		{"розах", "роз"},
		{"розни", "розн"},
		{"розно", "розн"},
		{"розные", "розн"},
		{"розовеет", "розовеет"},
		{"розовеют", "розовеют"},
		{"розовея", "розове"},
		{"розовое", "розов"},
		{"розовой", "розов"},
		{"розовом", "розов"},
		{"розовые", "розов"},
		{"розовый", "розов"},
		{"розовыми", "розов"},
		{"розовых", "розов"},
		{"розой", "роз"},
		{"розоперстая", "розоперст"},
		{"розу", "роз"},
		{"розы", "роз"},
		{"роится", "ро"},
		{"рой", "ро"},
		{"рок", "рок"},
		{"рока", "рок"},
		{"роковая", "роков"},
		{"рокового", "роков"},
		{"роковое", "роков"},
		{"роковой", "роков"},
		{"роковые", "роков"},
		{"роковым", "роков"},
		{"рокот", "рокот"},
		{"рокотание", "рокотан"},
		{"рокотов", "рокот"},
		{"рокотом", "рокот"},
		{"рокоты", "рокот"},
		{"роланда", "роланд"},
		{"ролей", "рол"},
		{"роли", "рол"},
		{"роль", "рол"},
		{"ром", "ром"},
		{"роман", "рома"},
		{"романа", "рома"},
		{"романах", "роман"},
		{"романе", "роман"},
		{"романистов", "романист"},
		{"романический", "романическ"},
		{"романическим", "романическ"},
		{"романического", "романическ"},
		{"романов", "роман"},
		{"романович", "романович"},
		{"романовича", "романович"},
		{"романовичу", "романович"},
		{"романовна", "романовн"},
		{"романовне", "романовн"},
		{"романовной", "романовн"},
		{"романовну", "романовн"},
		{"романовны", "романовн"},
		{"романс", "романс"},
		{"романсов", "романс"},
		{"романчик", "романчик"},
		{"романы", "рома"},
		{"романыч", "романыч"},
		{"романыча", "романыч"},
		{"романычем", "романыч"},
		{"романычу", "романыч"},
		{"ромео", "роме"},
		{"рому", "ром"},
		{"роняет", "роня"},
		{"ронял", "роня"},
		{"ронять", "роня"},
		{"роняя", "рон"},
		{"ропот", "ропот"},
		{"ропота", "ропот"},
		{"роптал", "ропта"},
		{"роптали", "ропта"},
		{"роптать", "ропта"},
		{"ропщу", "ропщ"},
		{"рос", "рос"},
		{"роса", "рос"},
		{"росах", "рос"},
		{"росинки", "росинк"},
		{"росистую", "росист"},
		{"роскоши", "роскош"},
		{"роскошная", "роскошн"},
		{"роскошнее", "роскошн"},
		{"роскошно", "роскошн"},
		{"роскошного", "роскошн"},
		{"роскошные", "роскошн"},
		{"роскошный", "роскошн"},
		{"роскошным", "роскошн"},
		{"роскошных", "роскошн"},
		{"роскошь", "роскош"},
		{"роскошью", "роскош"},
		{"росла", "росл"},
		{"росли", "росл"},
		{"росло", "росл"},
		{"росный", "росн"},
		{"росписные", "росписн"},
		{"россией", "росс"},
		{"россии", "росс"},
		{"российский", "российск"},
		{"россию", "росс"},
		{"россия", "росс"},
		{"россияда", "россияд"},
		{"россиянину", "россиянин"},
		{"рост", "рост"},
		{"роста", "рост"},
		{"росте", "рост"},
		{"ростовщика", "ростовщик"},
		{"ростовщица", "ростовщиц"},
		{"ростом", "рост"},
		{"росту", "рост"},
		{"росы", "рос"},
		{"рот", "рот"},
		{"рошую", "рош"},
		{"роща", "рощ"},
		{"рощах", "рощ"},
		{"роще", "рощ"},
		{"рощей", "рощ"},
		{"рощи", "рощ"},
		{"рощу", "рощ"},
		{"роют", "роют"},
		{"роются", "роют"},
		{"рояле", "роял"},
		{"роялем", "роял"},
		{"рояль", "роял"},
		{"рояля", "роял"},
		{"роясь", "ро"},
		{"роящихся", "роя"},
		{"рт", "рт"},
		{"рта", "рта"},
		{"ртами", "ртам"},
		{"ртом", "ртом"},
		{"рту", "рту"},
		{"рты", "рты"},
		{"рубах", "руб"},
		{"рубахе", "рубах"},
		{"рубаху", "рубах"},
		{"рубашек", "рубашек"},
		{"рубашечку", "рубашечк"},
		{"рубашечный", "рубашечн"},
		{"рубашка", "рубашк"},
		{"рубашках", "рубашк"},
		{"рубашке", "рубашк"},
		{"рубашки", "рубашк"},
		{"рубашкой", "рубашк"},
		{"рубашку", "рубашк"},
		{"рубашонке", "рубашонк"},
		{"рубежей", "рубеж"},
		{"рубежи", "рубеж"},
		{"рубец", "рубец"},
		{"руби", "руб"},
		{"рубивших", "руб"},
		{"рубикон", "рубикон"},
		{"рубила", "руб"},
		{"рубили", "руб"},
		{"рубин", "рубин"},
		{"рубини", "рубин"},
		{"рубинштейн", "рубинштейн"},
		{"рубит", "руб"},
		{"рубить", "руб"},
		{"рубища", "рубищ"},
		{"рубище", "рубищ"},
		{"рубиянить", "рубиян"},
		{"рублев", "рубл"},
		{"рублевым", "рублев"},
		{"рублей", "рубл"},
		{"рубликов", "рублик"},
		{"рубль", "рубл"},
		{"рублю", "рубл"},
		{"рубля", "рубл"},
		{"рублям", "рубл"},
		{"рублями", "рубл"},
		{"рублях", "рубл"},
		{"рубнуть", "рубнут"},
		{"ру