package dbus

import (
	"bytes"
	"encoding/binary"
	"reflect"
	"testing"
)

func TestEncodeByte(t *testing.T) {
	val := byte(10)
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(val)
	if err != nil {
		t.Fatal(err)
	}

	expected := []byte{0xa}
	if !bytes.Equal(buf.Bytes(), expected) {
		t.Errorf("not equal: got '%#v', want '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(val))
	if err != nil {
		t.Fatal(err)
	}
	var out byte
	if err := Store(v, &out); err != nil {
		t.Fatal(err)
	}
	if !reflect.DeepEqual(out, val) {
		t.Errorf("not equal: got '%v', want '%v'",
			out, val)
	}
}

func BenchmarkEncodeByte(b *testing.B) {
	val := byte(10)
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	enc := newEncoder(buf, binary.LittleEndian, fds)
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(val)
	}
}

func TestEncodeBool(t *testing.T) {
	val := true
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(val)
	if err != nil {
		t.Fatal(err)
	}

	expected := []byte{0x1, 0x0, 0x0, 0x0}
	if !bytes.Equal(buf.Bytes(), expected) {
		t.Errorf("not equal: got '%#v', want '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(val))
	if err != nil {
		t.Fatal(err)
	}
	var out bool
	if err := Store(v, &out); err != nil {
		t.Fatal(err)
	}
	if !reflect.DeepEqual(out, val) {
		t.Errorf("not equal: got '%v', want '%v'",
			out, val)
	}
}

func BenchmarkEncodeBool(b *testing.B) {
	val := true
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	enc := newEncoder(buf, binary.LittleEndian, fds)
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(val)
	}
}

func TestEncodeInt(t *testing.T) {
	val := 10
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(val)
	if err != nil {
		t.Fatal(err)
	}

	expected := []byte{0xa, 0x0, 0x0, 0x0}
	if !bytes.Equal(buf.Bytes(), expected) {
		t.Errorf("not equal: got '%#v', want '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(val))
	if err != nil {
		t.Fatal(err)
	}
	var out int
	if err := Store(v, &out); err != nil {
		t.Fatal(err)
	}
	if !reflect.DeepEqual(out, val) {
		t.Errorf("not equal: got '%v', want '%v'",
			out, val)
	}
}

func BenchmarkEncodeIntToInt(b *testing.B) {
	val := 10
	buf := bytes.NewBuffer(make([]byte, 0, 4))
	fds := make([]int, 0)
	enc := newEncoder(buf, binary.LittleEndian, fds)
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(val)
	}
}

func TestEncodeIntToNonCovertible(t *testing.T) {
	val := 150
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(val)
	if err != nil {
		t.Fatal(err)
	}

	expected := []byte{0x96, 0x0, 0x0, 0x0}
	if !bytes.Equal(buf.Bytes(), expected) {
		t.Errorf("not equal: got '%#v', want '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(val))
	if err != nil {
		t.Fatal(err)
	}
	var out bool
	err = Store(v, &out)
	if err == nil {
		t.Logf("%t\n", out)
		t.Fatal("Type mismatch should have occurred")
	}
}

func TestEncodeUint(t *testing.T) {
	val := uint(10)
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(val)
	if err != nil {
		t.Fatal(err)
	}

	expected := []byte{0xa, 0x0, 0x0, 0x0}
	if !bytes.Equal(buf.Bytes(), expected) {
		t.Errorf("not equal: got '%#v', want '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(val))
	if err != nil {
		t.Fatal(err)
	}
	var out uint
	if err := Store(v, &out); err != nil {
		t.Fatal(err)
	}
	if !reflect.DeepEqual(out, val) {
		t.Errorf("not equal: got '%v', want '%v'",
			out, val)
	}
}

func BenchmarkEncodeUInt(b *testing.B) {
	val := uint(10)
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	enc := newEncoder(buf, binary.LittleEndian, fds)
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(val)
	}
}

func TestEncodeUintToNonCovertible(t *testing.T) {
	val := uint(10)
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(val)
	if err != nil {
		t.Fatal(err)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(val))
	if err != nil {
		t.Fatal(err)
	}
	var out bool
	err = Store(v, &out)
	if err == nil {
		t.Fatal("Type mismatch should have occurred")
	}
}

type boolean bool

func TestEncodeOfAssignableType(t *testing.T) {
	val := boolean(true)
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(val)
	if err != nil {
		t.Fatal(err)
	}

	expected := []byte{0x1, 0x0, 0x0, 0x0}
	if !bytes.Equal(buf.Bytes(), expected) {
		t.Errorf("not equal: got '%#v', want '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(val))
	if err != nil {
		t.Fatal(err)
	}
	var out boolean
	err = Store(v, &out)
	if err != nil {
		t.Fatal(err)
	}
	if !reflect.DeepEqual(out, val) {
		t.Errorf("not equal: got '%v', want '%v'",
			out, val)
	}
}

func BenchmarkEncodeOfAssignableType(b *testing.B) {
	val := bool(true)
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	enc := newEncoder(buf, binary.LittleEndian, fds)
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(val)
	}
}

func TestEncodeArrayOfMaps(t *testing.T) {
	tests := []struct {
		name         string
		vs           []interface{}
		bigEndian    [][]byte
		littleEndian [][]byte // Three are a few ways that the underlying code may read the map
	}{
		{
			name: "aligned at 8 at start of array",
			vs: []interface{}{
				"12345",
				[]map[string]Variant{
					{
						"abcdefg": MakeVariant("foo"),
						"cdef":    MakeVariant(uint32(2)),
					},
				},
			},
			bigEndian: [][]byte{
				{0x0, 0x0, 0x0, 0x5, 0x31, 0x32, 0x33, 0x34, 0x35, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x30, 0x0, 0x0, 0x0, 0x28, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x7, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x66, 0x6f, 0x6f, 0x0, 0x0, 0x0, 0x0, 0x4, 0x63, 0x64, 0x65, 0x66, 0x0, 0x1, 0x75, 0x0, 0x0, 0x0, 0x0, 0x2},
				{0x0, 0x0, 0x0, 0x5, 0x31, 0x32, 0x33, 0x34, 0x35, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x30, 0x0, 0x0, 0x0, 0x28, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x63, 0x64, 0x65, 0x66, 0x0, 0x1, 0x75, 0x0, 0x0, 0x0, 0x0, 0x2, 0x0, 0x0, 0x0, 0x7, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x66, 0x6f, 0x6f, 0x0},
			},
			littleEndian: [][]byte{
				{0x5, 0x0, 0x0, 0x0, 0x31, 0x32, 0x33, 0x34, 0x35, 0x0, 0x0, 0x0, 0x30, 0x0, 0x0, 0x0, 0x28, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x7, 0x0, 0x0, 0x0, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0, 0x4, 0x0, 0x0, 0x0, 0x63, 0x64, 0x65, 0x66, 0x0, 0x1, 0x75, 0x0, 0x2, 0x0, 0x0, 0x0},
				{0x5, 0x0, 0x0, 0x0, 0x31, 0x32, 0x33, 0x34, 0x35, 0x0, 0x0, 0x0, 0x30, 0x0, 0x0, 0x0, 0x28, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x63, 0x64, 0x65, 0x66, 0x0, 0x1, 0x75, 0x0, 0x2, 0x0, 0x0, 0x0, 0x7, 0x0, 0x0, 0x0, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0},
			},
		},
		{
			name: "not aligned at 8 for start of array",
			vs: []interface{}{
				"1234567890",
				[]map[string]Variant{
					{
						"abcdefg": MakeVariant("foo"),
						"cdef":    MakeVariant(uint32(2)),
					},
				},
			},
			bigEndian: [][]byte{
				{0x0, 0x0, 0x0, 0xa, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x0, 0x0, 0x0, 0x0, 0x0, 0x2c, 0x0, 0x0, 0x0, 0x28, 0x0, 0x0, 0x0, 0x7, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x66, 0x6f, 0x6f, 0x0, 0x0, 0x0, 0x0, 0x4, 0x63, 0x64, 0x65, 0x66, 0x0, 0x1, 0x75, 0x0, 0x0, 0x0, 0x0, 0x2},
				{0xa, 0x0, 0x0, 0x0, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x0, 0x0, 0x2c, 0x0, 0x0, 0x0, 0x28, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x63, 0x64, 0x65, 0x66, 0x0, 0x1, 0x75, 0x0, 0x2, 0x0, 0x0, 0x0, 0x7, 0x0, 0x0, 0x0, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0},
				{0x0, 0x0, 0x0, 0xa, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x0, 0x0, 0x0, 0x0, 0x0, 0x2c, 0x0, 0x0, 0x0, 0x28, 0x0, 0x0, 0x0, 0x4, 0x63, 0x64, 0x65, 0x66, 0x0, 0x1, 0x75, 0x0, 0x0, 0x0, 0x0, 0x2, 0x0, 0x0, 0x0, 0x7, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x66, 0x6f, 0x6f, 0x0},
			},
			littleEndian: [][]byte{
				{0xa, 0x0, 0x0, 0x0, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x0, 0x0, 0x2c, 0x0, 0x0, 0x0, 0x28, 0x0, 0x0, 0x0, 0x7, 0x0, 0x0, 0x0, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0, 0x4, 0x0, 0x0, 0x0, 0x63, 0x64, 0x65, 0x66, 0x0, 0x1, 0x75, 0x0, 0x2, 0x0, 0x0, 0x0},
				{0xa, 0x0, 0x0, 0x0, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x0, 0x0, 0x2c, 0x0, 0x0, 0x0, 0x28, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x63, 0x64, 0x65, 0x66, 0x0, 0x1, 0x75, 0x0, 0x2, 0x0, 0x0, 0x0, 0x7, 0x0, 0x0, 0x0, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0},
			},
		},
	}
	for _, order := range []binary.ByteOrder{binary.LittleEndian, binary.BigEndian} {
		for _, tt := range tests {
			buf := new(bytes.Buffer)
			fds := make([]int, 0)
			enc := newEncoder(buf, order, fds)
			if err := enc.Encode(tt.vs...); err != nil {
				t.Fatal(err)
			}
			data := buf.Bytes()
			expected := tt.littleEndian
			if order == binary.BigEndian {
				expected = tt.bigEndian
			}

			found := false
			for _, e := range expected {
				if bytes.Equal(data, e) {
					found = true
					break
				}
			}

			if !found {
				t.Errorf("%q: (%v) not equal: got '%#v', want one of '%#v'", tt.name, order, data, expected)
				continue
			}

			dec := newDecoder(buf, order, enc.fds)
			v, err := dec.Decode(SignatureOf(tt.vs...))
			if err != nil {
				t.Errorf("%q: decode (%v) failed: %v", tt.name, order, err)
				continue
			}
			if !reflect.DeepEqual(v, tt.vs) {
				t.Errorf("%q: (%v) not equal: got '%v', want '%v'", tt.name, order, v, tt.vs)
				continue
			}
		}
	}
}

func BenchmarkEncodeArrayOfMaps(b *testing.B) {
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	enc := newEncoder(buf, nativeEndian, fds)
	vs := []interface{}{
		"12345",
		[]map[string]Variant{
			{
				"abcdefg": MakeVariant("foo"),
				"cdef":    MakeVariant(uint32(2)),
			},
		},
	}
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(vs...)
	}
}

func TestEncodeMapStringInterface(t *testing.T) {
	val := map[string]interface{}{"foo": "bar"}
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(val)
	if err != nil {
		t.Fatal(err)
	}
	expected := []byte{0x14, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x72, 0x0}
	if !bytes.Equal(buf.Bytes(), expected) {
		t.Errorf("not equal: got '%#v', want '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(val))
	if err != nil {
		t.Fatal(err)
	}
	out := map[string]interface{}{}
	if err := Store(v, &out); err != nil {
		t.Fatal(err)
	}
	if !reflect.DeepEqual(out, val) {
		t.Errorf("not equal: got '%v', want '%v'",
			out, val)
	}
}

func BenchmarkEncodeMapStringInterface(b *testing.B) {
	val := map[string]interface{}{"foo": "bar"}
	buf := &bytes.Buffer{}
	fds := make([]int, 0)

	b.ReportAllocs()
	b.ResetTimer()
	enc := newEncoder(buf, binary.LittleEndian, fds)
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(val)
	}
}

type empty interface{}

func TestEncodeMapStringNamedInterface(t *testing.T) {
	val := map[string]empty{"foo": "bar"}
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(val)
	if err != nil {
		t.Fatal(err)
	}
	expected := []byte{0x14, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x72, 0x0}
	if !bytes.Equal(buf.Bytes(), expected) {
		t.Errorf("not equal: got '%#v', want '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(val))
	if err != nil {
		t.Fatal(err)
	}
	out := map[string]empty{}
	if err := Store(v, &out); err != nil {
		t.Fatal(err)
	}
	if !reflect.DeepEqual(out, val) {
		t.Errorf("not equal: got '%v', want '%v'",
			out, val)
	}
}

func BenchmarkEncodeMapStringNamedInterface(b *testing.B) {
	val := map[string]empty{"foo": "bar"}
	buf := &bytes.Buffer{}
	fds := make([]int, 0)

	b.ReportAllocs()
	b.ResetTimer()
	enc := newEncoder(buf, binary.LittleEndian, fds)
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(val)
	}
}

type fooer interface {
	Foo()
}

type fooimpl string

func (fooimpl) Foo() {}

func TestEncodeMapStringNonEmptyInterface(t *testing.T) {
	val := map[string]fooer{"foo": fooimpl("bar")}
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(val)
	if err != nil {
		t.Fatal(err)
	}
	expected := []byte{0x14, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x72, 0x0}
	if !bytes.Equal(buf.Bytes(), expected) {
		t.Errorf("not equal: got '%#v', want '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(val))
	if err != nil {
		t.Fatal(err)
	}
	out := map[string]fooer{}
	err = Store(v, &out)
	if err == nil {
		t.Fatal("Shouldn't be able to convert to non empty interfaces")
	}
}

func BenchmarkEncodeMapStringNonEmptyInterface(b *testing.B) {
	val := map[string]fooer{"foo": fooimpl("bar")}
	buf := &bytes.Buffer{}
	fds := make([]int, 0)

	b.ReportAllocs()
	b.ResetTimer()
	enc := newEncoder(buf, binary.LittleEndian, fds)
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(val)
	}
}

func TestEncodeSliceInterface(t *testing.T) {
	val := []interface{}{"foo", "bar"}
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(val)
	if err != nil {
		t.Fatal(err)
	}
	expected := []byte{0x18, 0x0, 0x0, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x72, 0x0}
	if !bytes.Equal(buf.Bytes(), expected) {
		t.Errorf("not equal: got '%#v', want '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(val))
	if err != nil {
		t.Fatal(err)
	}
	out := []interface{}{}
	if err := Store(v, &out); err != nil {
		t.Fatal(err)
	}
	if !reflect.DeepEqual(out, val) {
		t.Errorf("not equal: got '%v', want '%v'",
			out, val)
	}
}

func BenchmarkEncodeSliceInterface(b *testing.B) {
	val := []interface{}{"foo", "bar"}
	buf := &bytes.Buffer{}
	fds := make([]int, 0)

	b.ReportAllocs()
	b.ResetTimer()
	enc := newEncoder(buf, binary.LittleEndian, fds)
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(val)
	}
}

func TestEncodeSliceNamedInterface(t *testing.T) {
	val := []empty{"foo", "bar"}
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(val)
	if err != nil {
		t.Fatal(err)
	}
	expected := []byte{0x18, 0x0, 0x0, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x72, 0x0}
	if !bytes.Equal(buf.Bytes(), expected) {
		t.Errorf("not equal: got '%#v', want '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(val))
	if err != nil {
		t.Fatal(err)
	}
	out := []empty{}
	if err := Store(v, &out); err != nil {
		t.Fatal(err)
	}
	if !reflect.DeepEqual(out, val) {
		t.Errorf("not equal: got '%v', want '%v'",
			out, val)
	}
}

func BenchmarkEncodeSliceNamedInterface(b *testing.B) {
	val := []empty{"foo", "bar"}
	buf := &bytes.Buffer{}
	fds := make([]int, 0)

	b.ReportAllocs()
	b.ResetTimer()
	enc := newEncoder(buf, binary.LittleEndian, fds)
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(val)
	}
}

func TestEncodeNestedInterface(t *testing.T) {
	val := map[string]interface{}{
		"foo": []interface{}{
			"1", "2", "3", "5",
			map[string]interface{}{
				"bar": "baz",
			},
		},
		"bar": map[string]interface{}{
			"baz":  "quux",
			"quux": "quuz",
		},
	}
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(val)
	if err != nil {
		t.Fatal(err)
	}

	expected := [][]byte{
		{0xad, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0, 0x2, 0x61, 0x76, 0x0, 0x54, 0x0, 0x0, 0x0, 0x1, 0x73, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x31, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x32, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x33, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x35, 0x0, 0x5, 0x61, 0x7b, 0x73, 0x76, 0x7d, 0x0, 0x0, 0x0, 0x0, 0x14, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x72, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x7a, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x72, 0x0, 0x5, 0x61, 0x7b, 0x73, 0x76, 0x7d, 0x0, 0x0, 0x2d, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x7a, 0x0, 0x1, 0x73, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x71, 0x75, 0x75, 0x78, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x71, 0x75, 0x75, 0x78, 0x0, 0x1, 0x73, 0x0, 0x4, 0x0, 0x0, 0x0, 0x71, 0x75, 0x75, 0x7a, 0x0},
		{0xad, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0, 0x2, 0x61, 0x76, 0x0, 0x54, 0x0, 0x0, 0x0, 0x1, 0x73, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x31, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x32, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x33, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x35, 0x0, 0x5, 0x61, 0x7b, 0x73, 0x76, 0x7d, 0x0, 0x0, 0x0, 0x0, 0x14, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x72, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x7a, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x72, 0x0, 0x5, 0x61, 0x7b, 0x73, 0x76, 0x7d, 0x0, 0x0, 0x2d, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x71, 0x75, 0x75, 0x78, 0x0, 0x1, 0x73, 0x0, 0x4, 0x0, 0x0, 0x0, 0x71, 0x75, 0x75, 0x7a, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x7a, 0x0, 0x1, 0x73, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x71, 0x75, 0x75, 0x78, 0x0},
		{0xac, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x72, 0x0, 0x5, 0x61, 0x7b, 0x73, 0x76, 0x7d, 0x0, 0x0, 0x2d, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x7a, 0x0, 0x1, 0x73, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x71, 0x75, 0x75, 0x78, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x71, 0x75, 0x75, 0x78, 0x0, 0x1, 0x73, 0x0, 0x4, 0x0, 0x0, 0x0, 0x71, 0x75, 0x75, 0x7a, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0, 0x2, 0x61, 0x76, 0x0, 0x54, 0x0, 0x0, 0x0, 0x1, 0x73, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x31, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x32, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x33, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x35, 0x0, 0x5, 0x61, 0x7b, 0x73, 0x76, 0x7d, 0x0, 0x0, 0x0, 0x0, 0x14, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x72, 0x0, 0x1, 0x73, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x7a, 0x0},
	}
	found := false
	for _, e := range expected {
		if bytes.Equal(buf.Bytes(), e) {
			found = true
			break
		}
	}

	if !found {
		t.Errorf("not equal: got '%#v', want one of '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(val))
	if err != nil {
		t.Fatal(err)
	}
	out := map[string]interface{}{}
	if err := Store(v, &out); err != nil {
		t.Fatal(err)
	}
	if !reflect.DeepEqual(out, val) {
		t.Errorf("not equal: got '%#v', want '%#v'",
			out, val)
	}
}

func BenchmarkEncodeNestedInterface(b *testing.B) {
	val := map[string]interface{}{
		"foo": []interface{}{
			"1", "2", "3", "5",
			map[string]interface{}{
				"bar": "baz",
			},
		},
		"bar": map[string]interface{}{
			"baz":  "quux",
			"quux": "quuz",
		},
	}
	buf := &bytes.Buffer{}
	fds := make([]int, 0)

	b.ReportAllocs()
	b.ResetTimer()
	enc := newEncoder(buf, binary.LittleEndian, fds)
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(val)
	}
}

func TestEncodeVariant(t *testing.T) {
	var res map[ObjectPath]map[string]map[string]Variant
	src := map[ObjectPath]map[string]map[string]Variant{
		ObjectPath("/foo/bar"): {
			"foo": {
				"bar": MakeVariant(10),
				"baz": MakeVariant("20"),
			},
		},
	}
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(src)
	if err != nil {
		t.Fatal(err)
	}

	expected := [][]byte{
		{0x4b, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x8, 0x0, 0x0, 0x0, 0x2f, 0x66, 0x6f, 0x6f, 0x2f, 0x62, 0x61, 0x72, 0x0, 0x0, 0x0, 0x0, 0x33, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0, 0x23, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x72, 0x0, 0x1, 0x69, 0x0, 0x0, 0xa, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x7a, 0x0, 0x1, 0x73, 0x0, 0x0, 0x2, 0x0, 0x0, 0x0, 0x32, 0x30, 0x0},
		{0x50, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x8, 0x0, 0x0, 0x0, 0x2f, 0x66, 0x6f, 0x6f, 0x2f, 0x62, 0x61, 0x72, 0x0, 0x0, 0x0, 0x0, 0x38, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0, 0x28, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x7a, 0x0, 0x1, 0x73, 0x0, 0x0, 0x2, 0x0, 0x0, 0x0, 0x32, 0x30, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x62, 0x61, 0x72, 0x0, 0x1, 0x69, 0x0, 0x0, 0xa, 0x0, 0x0, 0x0},
	}
	found := false
	for _, e := range expected {
		if bytes.Equal(buf.Bytes(), e) {
			found = true
			break
		}
	}

	if !found {
		t.Errorf("not equal: got '%#v', want one of '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(src))
	if err != nil {
		t.Fatal(err)
	}
	err = Store(v, &res)
	if err != nil {
		t.Fatal(err)
	}
	_ = res[ObjectPath("/foo/bar")]["foo"]["baz"].Value().(string)
}

func BenchmarkEncodeVariant(b *testing.B) {
	src := map[ObjectPath]map[string]map[string]Variant{
		ObjectPath("/foo/bar"): {
			"foo": {
				"bar": MakeVariant(10),
				"baz": MakeVariant("20"),
			},
		},
	}
	buf := &bytes.Buffer{}
	fds := make([]int, 0)

	b.ReportAllocs()
	b.ResetTimer()
	enc := newEncoder(buf, binary.LittleEndian, fds)
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(src)
	}
}

func TestEncodeVariantToList(t *testing.T) {
	var res map[string]Variant
	src := map[string]interface{}{
		"foo": []interface{}{"a", "b", "c"},
	}
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(src)
	if err != nil {
		t.Fatal(err)
	}
	expected := []byte{0x32, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0, 0x2, 0x61, 0x76, 0x0, 0x22, 0x0, 0x0, 0x0, 0x1, 0x73, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x61, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x62, 0x0, 0x1, 0x73, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x63, 0x0}
	if !bytes.Equal(buf.Bytes(), expected) {
		t.Errorf("not equal: got '%#v', want '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(src))
	if err != nil {
		t.Fatal(err)
	}
	err = Store(v, &res)
	if err != nil {
		t.Fatal(err)
	}
	_ = res["foo"].Value().([]Variant)
}

func BenchmarkEncodeVariantToList(b *testing.B) {
	src := map[string]interface{}{
		"foo": []interface{}{"a", "b", "c"},
	}
	buf := &bytes.Buffer{}
	fds := make([]int, 0)

	b.ReportAllocs()
	b.ResetTimer()
	enc := newEncoder(buf, binary.LittleEndian, fds)
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(src)
	}
}

func TestEncodeVariantToUint64(t *testing.T) {
	var res map[string]Variant
	src := map[string]interface{}{
		"foo": uint64(10),
	}
	buf := new(bytes.Buffer)
	fds := make([]int, 0)
	order := binary.LittleEndian
	enc := newEncoder(buf, binary.LittleEndian, fds)
	err := enc.Encode(src)
	if err != nil {
		t.Fatal(err)
	}

	expected := []byte{0x18, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x66, 0x6f, 0x6f, 0x0, 0x1, 0x74, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xa, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0}
	if !bytes.Equal(buf.Bytes(), expected) {
		t.Errorf("not equal: got '%#v', want '%#v'",
			buf.Bytes(), expected)
	}

	dec := newDecoder(buf, order, enc.fds)
	v, err := dec.Decode(SignatureOf(src))
	if err != nil {
		t.Fatal(err)
	}
	err = Store(v, &res)
	if err != nil {
		t.Fatal(err)
	}
	_ = res["foo"].Value().(uint64)
}

func BenchmarkEncodeVariantToUint64(b *testing.B) {
	src := map[string]interface{}{
		"foo": uint64(10),
	}
	buf := &bytes.Buffer{}
	fds := make([]int, 0)

	b.ReportAllocs()
	b.ResetTimer()
	enc := newEncoder(buf, binary.LittleEndian, fds)
	for n := 0; n < b.N; n++ {
		buf.Reset()
		enc.Reset(buf, nativeEndian, fds)
		enc.Encode(src)
	}
}
