// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package adsense provides access to the AdSense Management API.
//
// For product documentation, see: https://developers.google.com/adsense/management/
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/adsense/v2"
//	...
//	ctx := context.Background()
//	adsenseService, err := adsense.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate.
// To restrict scopes, use [google.golang.org/api/option.WithScopes]:
//
//	adsenseService, err := adsense.NewService(ctx, option.WithScopes(adsense.AdsenseReadonlyScope))
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	adsenseService, err := adsense.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	adsenseService, err := adsense.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package adsense // import "google.golang.org/api/adsense/v2"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "adsense:v2"
const apiName = "adsense"
const apiVersion = "v2"
const basePath = "https://adsense.googleapis.com/"
const basePathTemplate = "https://adsense.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://adsense.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// View and manage your AdSense data
	AdsenseScope = "https://www.googleapis.com/auth/adsense"

	// View your AdSense data
	AdsenseReadonlyScope = "https://www.googleapis.com/auth/adsense.readonly"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/adsense",
		"https://www.googleapis.com/auth/adsense.readonly",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Accounts = NewAccountsService(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Accounts *AccountsService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewAccountsService(s *Service) *AccountsService {
	rs := &AccountsService{s: s}
	rs.Adclients = NewAccountsAdclientsService(s)
	rs.Alerts = NewAccountsAlertsService(s)
	rs.Payments = NewAccountsPaymentsService(s)
	rs.PolicyIssues = NewAccountsPolicyIssuesService(s)
	rs.Reports = NewAccountsReportsService(s)
	rs.Sites = NewAccountsSitesService(s)
	return rs
}

type AccountsService struct {
	s *Service

	Adclients *AccountsAdclientsService

	Alerts *AccountsAlertsService

	Payments *AccountsPaymentsService

	PolicyIssues *AccountsPolicyIssuesService

	Reports *AccountsReportsService

	Sites *AccountsSitesService
}

func NewAccountsAdclientsService(s *Service) *AccountsAdclientsService {
	rs := &AccountsAdclientsService{s: s}
	rs.Adunits = NewAccountsAdclientsAdunitsService(s)
	rs.Customchannels = NewAccountsAdclientsCustomchannelsService(s)
	rs.Urlchannels = NewAccountsAdclientsUrlchannelsService(s)
	return rs
}

type AccountsAdclientsService struct {
	s *Service

	Adunits *AccountsAdclientsAdunitsService

	Customchannels *AccountsAdclientsCustomchannelsService

	Urlchannels *AccountsAdclientsUrlchannelsService
}

func NewAccountsAdclientsAdunitsService(s *Service) *AccountsAdclientsAdunitsService {
	rs := &AccountsAdclientsAdunitsService{s: s}
	return rs
}

type AccountsAdclientsAdunitsService struct {
	s *Service
}

func NewAccountsAdclientsCustomchannelsService(s *Service) *AccountsAdclientsCustomchannelsService {
	rs := &AccountsAdclientsCustomchannelsService{s: s}
	return rs
}

type AccountsAdclientsCustomchannelsService struct {
	s *Service
}

func NewAccountsAdclientsUrlchannelsService(s *Service) *AccountsAdclientsUrlchannelsService {
	rs := &AccountsAdclientsUrlchannelsService{s: s}
	return rs
}

type AccountsAdclientsUrlchannelsService struct {
	s *Service
}

func NewAccountsAlertsService(s *Service) *AccountsAlertsService {
	rs := &AccountsAlertsService{s: s}
	return rs
}

type AccountsAlertsService struct {
	s *Service
}

func NewAccountsPaymentsService(s *Service) *AccountsPaymentsService {
	rs := &AccountsPaymentsService{s: s}
	return rs
}

type AccountsPaymentsService struct {
	s *Service
}

func NewAccountsPolicyIssuesService(s *Service) *AccountsPolicyIssuesService {
	rs := &AccountsPolicyIssuesService{s: s}
	return rs
}

type AccountsPolicyIssuesService struct {
	s *Service
}

func NewAccountsReportsService(s *Service) *AccountsReportsService {
	rs := &AccountsReportsService{s: s}
	rs.Saved = NewAccountsReportsSavedService(s)
	return rs
}

type AccountsReportsService struct {
	s *Service

	Saved *AccountsReportsSavedService
}

func NewAccountsReportsSavedService(s *Service) *AccountsReportsSavedService {
	rs := &AccountsReportsSavedService{s: s}
	return rs
}

type AccountsReportsSavedService struct {
	s *Service
}

func NewAccountsSitesService(s *Service) *AccountsSitesService {
	rs := &AccountsSitesService{s: s}
	return rs
}

type AccountsSitesService struct {
	s *Service
}

// Account: Representation of an account.
type Account struct {
	// CreateTime: Output only. Creation time of the account.
	CreateTime string `json:"createTime,omitempty"`
	// DisplayName: Output only. Display name of this account.
	DisplayName string `json:"displayName,omitempty"`
	// Name: Output only. Resource name of the account. Format: accounts/pub-[0-9]+
	Name string `json:"name,omitempty"`
	// PendingTasks: Output only. Outstanding tasks that need to be completed as
	// part of the sign-up process for a new account. e.g.
	// "billing-profile-creation", "phone-pin-verification".
	PendingTasks []string `json:"pendingTasks,omitempty"`
	// Premium: Output only. Whether this account is premium.
	Premium bool `json:"premium,omitempty"`
	// State: Output only. State of the account.
	//
	// Possible values:
	//   "STATE_UNSPECIFIED" - State unspecified.
	//   "READY" - The account is open and ready to serve ads.
	//   "NEEDS_ATTENTION" - There are some issues with this account. Publishers
	// should visit AdSense in order to fix the account.
	//   "CLOSED" - The account is closed and can't serve ads.
	State string `json:"state,omitempty"`
	// TimeZone: The account time zone, as used by reporting. For more information,
	// see changing the time zone of your reports
	// (https://support.google.com/adsense/answer/9830725).
	TimeZone *TimeZone `json:"timeZone,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CreateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Account) MarshalJSON() ([]byte, error) {
	type NoMethod Account
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AdBlockingRecoveryTag: Representation of an ad blocking recovery tag. See
// https://support.google.com/adsense/answer/11575177.
type AdBlockingRecoveryTag struct {
	// ErrorProtectionCode: Error protection code that can be used in conjunction
	// with the tag. It'll display a message to users if an ad blocking extension
	// blocks their access to your site
	// (https://support.google.com/adsense/answer/11575480).
	ErrorProtectionCode string `json:"errorProtectionCode,omitempty"`
	// Tag: The ad blocking recovery tag. Note that the message generated by the
	// tag can be blocked by an ad blocking extension. If this is not your desired
	// outcome, then you'll need to use it in conjunction with the error protection
	// code.
	Tag string `json:"tag,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ErrorProtectionCode") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ErrorProtectionCode") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AdBlockingRecoveryTag) MarshalJSON() ([]byte, error) {
	type NoMethod AdBlockingRecoveryTag
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AdClient: Representation of an ad client. An ad client represents a user's
// subscription with a specific AdSense product.
type AdClient struct {
	// Name: Output only. Resource name of the ad client. Format:
	// accounts/{account}/adclients/{adclient}
	Name string `json:"name,omitempty"`
	// ProductCode: Output only. Reporting product code of the ad client. For
	// example, "AFC" for AdSense for Content. Corresponds to the `PRODUCT_CODE`
	// dimension, and present only if the ad client supports reporting.
	ProductCode string `json:"productCode,omitempty"`
	// ReportingDimensionId: Output only. Unique ID of the ad client as used in the
	// `AD_CLIENT_ID` reporting dimension. Present only if the ad client supports
	// reporting.
	ReportingDimensionId string `json:"reportingDimensionId,omitempty"`
	// State: Output only. State of the ad client.
	//
	// Possible values:
	//   "STATE_UNSPECIFIED" - State unspecified.
	//   "READY" - The ad client is ready to show ads.
	//   "GETTING_READY" - Running some checks on the ad client before it is ready
	// to serve ads.
	//   "REQUIRES_REVIEW" - The ad client hasn't been checked yet. There are tasks
	// pending before AdSense will start the review.
	State string `json:"state,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Name") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Name") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AdClient) MarshalJSON() ([]byte, error) {
	type NoMethod AdClient
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AdClientAdCode: Representation of the AdSense code for a given ad client.
// For more information, see About the AdSense code
// (https://support.google.com/adsense/answer/9274634).
type AdClientAdCode struct {
	// AdCode: Output only. The AdSense code snippet to add to the head of an HTML
	// page.
	AdCode string `json:"adCode,omitempty"`
	// AmpBody: Output only. The AdSense code snippet to add to the body of an AMP
	// page.
	AmpBody string `json:"ampBody,omitempty"`
	// AmpHead: Output only. The AdSense code snippet to add to the head of an AMP
	// page.
	AmpHead string `json:"ampHead,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AdCode") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdCode") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AdClientAdCode) MarshalJSON() ([]byte, error) {
	type NoMethod AdClientAdCode
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AdUnit: Representation of an ad unit. An ad unit represents a saved ad unit
// with a specific set of ad settings that have been customized within an
// account.
type AdUnit struct {
	// ContentAdsSettings: Required. Settings specific to content ads (AFC).
	ContentAdsSettings *ContentAdsSettings `json:"contentAdsSettings,omitempty"`
	// DisplayName: Required. Display name of the ad unit, as provided when the ad
	// unit was created.
	DisplayName string `json:"displayName,omitempty"`
	// Name: Output only. Resource name of the ad unit. Format:
	// accounts/{account}/adclients/{adclient}/adunits/{adunit}
	Name string `json:"name,omitempty"`
	// ReportingDimensionId: Output only. Unique ID of the ad unit as used in the
	// `AD_UNIT_ID` reporting dimension.
	ReportingDimensionId string `json:"reportingDimensionId,omitempty"`
	// State: Required. State of the ad unit.
	//
	// Possible values:
	//   "STATE_UNSPECIFIED" - State unspecified.
	//   "ACTIVE" - Ad unit has been activated by the user.
	//   "ARCHIVED" - Ad unit has been archived by the user. Note that archived ad
	// units are only removed from the default view in the UI. Archived ad units
	// can still serve ads.
	State string `json:"state,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ContentAdsSettings") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ContentAdsSettings") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AdUnit) MarshalJSON() ([]byte, error) {
	type NoMethod AdUnit
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AdUnitAdCode: Representation of the ad unit code for a given ad unit. For
// more information, see About the AdSense code
// (https://support.google.com/adsense/answer/9274634) and Where to place the
// ad code in your HTML (https://support.google.com/adsense/answer/9190028).
type AdUnitAdCode struct {
	// AdCode: Output only. The code snippet to add to the body of an HTML page.
	AdCode string `json:"adCode,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AdCode") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdCode") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AdUnitAdCode) MarshalJSON() ([]byte, error) {
	type NoMethod AdUnitAdCode
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Alert: Representation of an alert.
type Alert struct {
	// Message: Output only. The localized alert message. This may contain HTML
	// markup, such as phrase elements or links.
	Message string `json:"message,omitempty"`
	// Name: Output only. Resource name of the alert. Format:
	// accounts/{account}/alerts/{alert}
	Name string `json:"name,omitempty"`
	// Severity: Output only. Severity of this alert.
	//
	// Possible values:
	//   "SEVERITY_UNSPECIFIED" - Unspecified severity.
	//   "INFO" - Info.
	//   "WARNING" - Warning.
	//   "SEVERE" - Severe.
	Severity string `json:"severity,omitempty"`
	// Type: Output only. Type of alert. This identifies the broad type of this
	// alert, and provides a stable machine-readable identifier that will not be
	// translated. For example, "payment-hold".
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Message") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Message") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Alert) MarshalJSON() ([]byte, error) {
	type NoMethod Alert
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Cell: Cell representation.
type Cell struct {
	// Value: Value in the cell. The dimension cells contain strings, and the
	// metric cells contain numbers.
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Value") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Value") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Cell) MarshalJSON() ([]byte, error) {
	type NoMethod Cell
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ContentAdsSettings: Settings specific to content ads (AFC).
type ContentAdsSettings struct {
	// Size: Required. Size of the ad unit. e.g. "728x90", "1x3" (for responsive ad
	// units).
	Size string `json:"size,omitempty"`
	// Type: Required. Type of the ad unit.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Unspecified ad unit type.
	//   "DISPLAY" - Display ad unit.
	//   "FEED" - In-feed ad unit.
	//   "ARTICLE" - In-article ad unit.
	//   "MATCHED_CONTENT" - Matched content unit.
	//   "LINK" - Link ad unit. Note that link ad units have now been retired, see
	// https://support.google.com/adsense/answer/9987221.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Size") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Size") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ContentAdsSettings) MarshalJSON() ([]byte, error) {
	type NoMethod ContentAdsSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CustomChannel: Representation of a custom channel.
type CustomChannel struct {
	// Active: Whether the custom channel is active and collecting data. See
	// https://support.google.com/adsense/answer/10077192.
	Active bool `json:"active,omitempty"`
	// DisplayName: Required. Display name of the custom channel.
	DisplayName string `json:"displayName,omitempty"`
	// Name: Output only. Resource name of the custom channel. Format:
	// accounts/{account}/adclients/{adclient}/customchannels/{customchannel}
	Name string `json:"name,omitempty"`
	// ReportingDimensionId: Output only. Unique ID of the custom channel as used
	// in the `CUSTOM_CHANNEL_ID` reporting dimension.
	ReportingDimensionId string `json:"reportingDimensionId,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Active") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Active") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CustomChannel) MarshalJSON() ([]byte, error) {
	type NoMethod CustomChannel
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Date: Represents a whole or partial calendar date, such as a birthday. The
// time of day and time zone are either specified elsewhere or are
// insignificant. The date is relative to the Gregorian Calendar. This can
// represent one of the following: * A full date, with non-zero year, month,
// and day values. * A month and day, with a zero year (for example, an
// anniversary). * A year on its own, with a zero month and a zero day. * A
// year and month, with a zero day (for example, a credit card expiration
// date). Related types: * google.type.TimeOfDay * google.type.DateTime *
// google.protobuf.Timestamp
type Date struct {
	// Day: Day of a month. Must be from 1 to 31 and valid for the year and month,
	// or 0 to specify a year by itself or a year and month where the day isn't
	// significant.
	Day int64 `json:"day,omitempty"`
	// Month: Month of a year. Must be from 1 to 12, or 0 to specify a year without
	// a month and day.
	Month int64 `json:"month,omitempty"`
	// Year: Year of the date. Must be from 1 to 9999, or 0 to specify a date
	// without a year.
	Year int64 `json:"year,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Day") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Day") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Date) MarshalJSON() ([]byte, error) {
	type NoMethod Date
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// Header: The header information of the columns requested in the report.
type Header struct {
	// CurrencyCode: The ISO-4217 currency code
	// (https://en.wikipedia.org/wiki/ISO_4217) of this column. Only present if the
	// header type is METRIC_CURRENCY.
	CurrencyCode string `json:"currencyCode,omitempty"`
	// Name: Required. Name of the header.
	Name string `json:"name,omitempty"`
	// Type: Required. Type of the header.
	//
	// Possible values:
	//   "HEADER_TYPE_UNSPECIFIED" - Unspecified header.
	//   "DIMENSION" - Dimension header type.
	//   "METRIC_TALLY" - Tally header type.
	//   "METRIC_RATIO" - Ratio header type.
	//   "METRIC_CURRENCY" - Currency header type.
	//   "METRIC_MILLISECONDS" - Milliseconds header type.
	//   "METRIC_DECIMAL" - Decimal header type.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CurrencyCode") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CurrencyCode") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Header) MarshalJSON() ([]byte, error) {
	type NoMethod Header
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// HttpBody: Message that represents an arbitrary HTTP body. It should only be
// used for payload formats that can't be represented as JSON, such as raw
// binary or an HTML page. This message can be used both in streaming and
// non-streaming API methods in the request as well as the response. It can be
// used as a top-level request field, which is convenient if one wants to
// extract parameters from either the URL or HTTP template into the request
// fields and also want access to the raw HTTP body. Example: message
// GetResourceRequest { // A unique request id. string request_id = 1; // The
// raw HTTP body is bound to this field. google.api.HttpBody http_body = 2; }
// service ResourceService { rpc GetResource(GetResourceRequest) returns
// (google.api.HttpBody); rpc UpdateResource(google.api.HttpBody) returns
// (google.protobuf.Empty); } Example with streaming methods: service
// CaldavService { rpc GetCalendar(stream google.api.HttpBody) returns (stream
// google.api.HttpBody); rpc UpdateCalendar(stream google.api.HttpBody) returns
// (stream google.api.HttpBody); } Use of this type only changes how the
// request and response bodies are handled, all other features will continue to
// work unchanged.
type HttpBody struct {
	// ContentType: The HTTP Content-Type header value specifying the content type
	// of the body.
	ContentType string `json:"contentType,omitempty"`
	// Data: The HTTP request/response body as raw binary.
	Data string `json:"data,omitempty"`
	// Extensions: Application specific response metadata. Must be set in the first
	// response for streaming APIs.
	Extensions []googleapi.RawMessage `json:"extensions,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ContentType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ContentType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s HttpBody) MarshalJSON() ([]byte, error) {
	type NoMethod HttpBody
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListAccountsResponse: Response definition for the account list rpc.
type ListAccountsResponse struct {
	// Accounts: The accounts returned in this list response.
	Accounts []*Account `json:"accounts,omitempty"`
	// NextPageToken: Continuation token used to page through accounts. To retrieve
	// the next page of the results, set the next request's "page_token" value to
	// this.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Accounts") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Accounts") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListAccountsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListAccountsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListAdClientsResponse: Response definition for the ad client list rpc.
type ListAdClientsResponse struct {
	// AdClients: The ad clients returned in this list response.
	AdClients []*AdClient `json:"adClients,omitempty"`
	// NextPageToken: Continuation token used to page through ad clients. To
	// retrieve the next page of the results, set the next request's "page_token"
	// value to this.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AdClients") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdClients") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListAdClientsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListAdClientsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListAdUnitsResponse: Response definition for the adunit list rpc.
type ListAdUnitsResponse struct {
	// AdUnits: The ad units returned in the list response.
	AdUnits []*AdUnit `json:"adUnits,omitempty"`
	// NextPageToken: Continuation token used to page through ad units. To retrieve
	// the next page of the results, set the next request's "page_token" value to
	// this.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AdUnits") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdUnits") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListAdUnitsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListAdUnitsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListAlertsResponse: Response definition for the alerts list rpc.
type ListAlertsResponse struct {
	// Alerts: The alerts returned in this list response.
	Alerts []*Alert `json:"alerts,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Alerts") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Alerts") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListAlertsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListAlertsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListChildAccountsResponse: Response definition for the child account list
// rpc.
type ListChildAccountsResponse struct {
	// Accounts: The accounts returned in this list response.
	Accounts []*Account `json:"accounts,omitempty"`
	// NextPageToken: Continuation token used to page through accounts. To retrieve
	// the next page of the results, set the next request's "page_token" value to
	// this.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Accounts") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Accounts") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListChildAccountsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListChildAccountsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListCustomChannelsResponse: Response definition for the custom channel list
// rpc.
type ListCustomChannelsResponse struct {
	// CustomChannels: The custom channels returned in this list response.
	CustomChannels []*CustomChannel `json:"customChannels,omitempty"`
	// NextPageToken: Continuation token used to page through alerts. To retrieve
	// the next page of the results, set the next request's "page_token" value to
	// this.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CustomChannels") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CustomChannels") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListCustomChannelsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListCustomChannelsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListLinkedAdUnitsResponse: Response definition for the ad units linked to a
// custom channel list rpc.
type ListLinkedAdUnitsResponse struct {
	// AdUnits: The ad units returned in the list response.
	AdUnits []*AdUnit `json:"adUnits,omitempty"`
	// NextPageToken: Continuation token used to page through ad units. To retrieve
	// the next page of the results, set the next request's "page_token" value to
	// this.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AdUnits") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdUnits") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListLinkedAdUnitsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListLinkedAdUnitsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListLinkedCustomChannelsResponse: Response definition for the custom
// channels linked to an adunit list rpc.
type ListLinkedCustomChannelsResponse struct {
	// CustomChannels: The custom channels returned in this list response.
	CustomChannels []*CustomChannel `json:"customChannels,omitempty"`
	// NextPageToken: Continuation token used to page through alerts. To retrieve
	// the next page of the results, set the next request's "page_token" value to
	// this.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CustomChannels") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CustomChannels") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListLinkedCustomChannelsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListLinkedCustomChannelsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListPaymentsResponse: Response definition for the payments list rpc.
type ListPaymentsResponse struct {
	// Payments: The payments returned in this list response.
	Payments []*Payment `json:"payments,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Payments") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Payments") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListPaymentsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListPaymentsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListPolicyIssuesResponse: Response definition for the policy issues list
// rpc. Policy issues are reported only if the publisher has at least one AFC
// ad client in READY or GETTING_READY state. If the publisher has no such AFC
// ad client, the response will be an empty list.
type ListPolicyIssuesResponse struct {
	// NextPageToken: Continuation token used to page through policy issues. To
	// retrieve the next page of the results, set the next request's "page_token"
	// value to this.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// PolicyIssues: The policy issues returned in the list response.
	PolicyIssues []*PolicyIssue `json:"policyIssues,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListPolicyIssuesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListPolicyIssuesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListSavedReportsResponse: Response definition for the saved reports list
// rpc.
type ListSavedReportsResponse struct {
	// NextPageToken: Continuation token used to page through reports. To retrieve
	// the next page of the results, set the next request's "page_token" value to
	// this.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// SavedReports: The reports returned in this list response.
	SavedReports []*SavedReport `json:"savedReports,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListSavedReportsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListSavedReportsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListSitesResponse: Response definition for the sites list rpc.
type ListSitesResponse struct {
	// NextPageToken: Continuation token used to page through sites. To retrieve
	// the next page of the results, set the next request's "page_token" value to
	// this.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Sites: The sites returned in this list response.
	Sites []*Site `json:"sites,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListSitesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListSitesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListUrlChannelsResponse: Response definition for the url channels list rpc.
type ListUrlChannelsResponse struct {
	// NextPageToken: Continuation token used to page through url channels. To
	// retrieve the next page of the results, set the next request's "page_token"
	// value to this.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// UrlChannels: The url channels returned in this list response.
	UrlChannels []*UrlChannel `json:"urlChannels,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListUrlChannelsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListUrlChannelsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Payment: Representation of an unpaid or paid payment. See Payment timelines
// for AdSense (https://support.google.com/adsense/answer/7164703) for more
// information about payments and the YouTube homepage and payments account
// (https://support.google.com/adsense/answer/11622510) article for information
// about dedicated payments accounts for YouTube.
type Payment struct {
	// Amount: Output only. The amount of unpaid or paid earnings, as a formatted
	// string, including the currency. E.g. "¥1,235 JPY", "$1,234.57", "£87.65".
	Amount string `json:"amount,omitempty"`
	// Date: Output only. For paid earnings, the date that the payment was
	// credited. For unpaid earnings, this field is empty. Payment dates are always
	// returned in the billing timezone (America/Los_Angeles).
	Date *Date `json:"date,omitempty"`
	// Name: Output only. Resource name of the payment. Format: -
	// accounts/{account}/payments/unpaid for unpaid (current) AdSense earnings. -
	// accounts/{account}/payments/youtube-unpaid for unpaid (current) YouTube
	// earnings. - accounts/{account}/payments/yyyy-MM-dd for paid AdSense
	// earnings. - accounts/{account}/payments/youtube-yyyy-MM-dd for paid YouTube
	// earnings.
	Name string `json:"name,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Amount") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Amount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Payment) MarshalJSON() ([]byte, error) {
	type NoMethod Payment
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PolicyIssue: Representation of a policy issue for a single entity (site,
// site-section, or page). All issues for a single entity are represented by a
// single PolicyIssue resource, though that PolicyIssue can have multiple
// causes (or "topics") that can change over time. Policy issues are removed if
// there are no issues detected recently or if there's a recent successful
// appeal for the entity.
type PolicyIssue struct {
	// Action: Required. The most severe action taken on the entity over the past
	// seven days.
	//
	// Possible values:
	//   "ENFORCEMENT_ACTION_UNSPECIFIED" - The action is unspecified.
	//   "WARNED" - No ad serving enforcement is currently present, but enforcement
	// will start on the `warning_escalation_date` if the issue is not resolved.
	//   "AD_SERVING_RESTRICTED" - Ad serving demand has been restricted on the
	// entity.
	//   "AD_SERVING_DISABLED" - Ad serving has been disabled on the entity.
	//   "AD_SERVED_WITH_CLICK_CONFIRMATION" - Ads are being served for the entity
	// but Confirmed Click is being applied to the ads. See
	// https://support.google.com/adsense/answer/10025624.
	//   "AD_PERSONALIZATION_RESTRICTED" - Ad personalization is restricted because
	// the ad requests coming from the EEA and UK do not have a TCF string or the
	// Consent Management Platform (CMP) indicated by the TCF string is not Google
	// certified. As a result, basic/limited ads will be served. See
	// https://support.google.com/adsense/answer/13554116.
	Action string `json:"action,omitempty"`
	// AdClients: Optional. List of ad clients associated with the policy issue
	// (either as the primary ad client or an associated host/secondary ad client).
	// In the latter case, this will be an ad client that is not owned by the
	// current account.
	AdClients []string `json:"adClients,omitempty"`
	// AdRequestCount: Required. Total number of ad requests affected by the policy
	// violations over the past seven days.
	AdRequestCount int64 `json:"adRequestCount,omitempty,string"`
	// EntityType: Required. Type of the entity indicating if the entity is a site,
	// site-section, or page.
	//
	// Possible values:
	//   "ENTITY_TYPE_UNSPECIFIED" - The entity type is unspecified.
	//   "SITE" - The enforced entity is an entire website.
	//   "SITE_SECTION" - The enforced entity is a particular section of a website.
	// All the pages with this prefix are enforced.
	//   "PAGE" - The enforced entity is a single web page.
	EntityType string `json:"entityType,omitempty"`
	// FirstDetectedDate: Required. The date (in the America/Los_Angeles timezone)
	// when policy violations were first detected on the entity.
	FirstDetectedDate *Date `json:"firstDetectedDate,omitempty"`
	// LastDetectedDate: Required. The date (in the America/Los_Angeles timezone)
	// when policy violations were last detected on the entity.
	LastDetectedDate *Date `json:"lastDetectedDate,omitempty"`
	// Name: Required. Resource name of the entity with policy issues. Format:
	// accounts/{account}/policyIssues/{policy_issue}
	Name string `json:"name,omitempty"`
	// PolicyTopics: Required. Unordered list. The policy topics that this entity
	// was found to violate over the past seven days.
	PolicyTopics []*PolicyTopic `json:"policyTopics,omitempty"`
	// Site: Required. Hostname/domain of the entity (for example "foo.com" or
	// "www.foo.com"). This _should_ be a bare domain/host name without any
	// protocol. This will be present for all policy issues.
	Site string `json:"site,omitempty"`
	// SiteSection: Optional. Prefix of the site-section having policy issues (For
	// example "foo.com/bar-section"). This will be present if the `entity_type` is
	// `SITE_SECTION` and will be absent for other entity types.
	SiteSection string `json:"siteSection,omitempty"`
	// Uri: Optional. URI of the page having policy violations (for example
	// "foo.com/bar" or "www.foo.com/bar"). This will be present if the
	// `entity_type` is `PAGE` and will be absent for other entity types.
	Uri string `json:"uri,omitempty"`
	// WarningEscalationDate: Optional. The date (in the America/Los_Angeles
	// timezone) when the entity will have ad serving demand restricted or ad
	// serving disabled. This is present only for issues with a `WARNED`
	// enforcement action. See https://support.google.com/adsense/answer/11066888.
	WarningEscalationDate *Date `json:"warningEscalationDate,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Action") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Action") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PolicyIssue) MarshalJSON() ([]byte, error) {
	type NoMethod PolicyIssue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PolicyTopic: Information about a particular policy topic. A policy topic
// represents a single class of policy issue that can impact ad serving for
// your site. For example, sexual content or having ads that obscure your
// content. A single policy issue can have multiple policy topics for a single
// entity.
type PolicyTopic struct {
	// MustFix: Required. Deprecated. Always set to false.
	MustFix bool `json:"mustFix,omitempty"`
	// Topic: Required. The policy topic. For example, "sexual-content" or
	// "ads-obscuring-content"."
	Topic string `json:"topic,omitempty"`
	// Type: Optional. The type of policy topic. For example, "POLICY" represents
	// all the policy topics that are related to the Google Publisher Policy (GPP).
	// See https://support.google.com/adsense/answer/15689616.
	//
	// Possible values:
	//   "POLICY_TOPIC_TYPE_UNSPECIFIED" - The type is unspecified.
	//   "POLICY" - Topics that are primarily related to the Google Publisher
	// Policy (GPP) (https://support.google.com/publisherpolicies/answer/10502938)
	// or the Google Publisher Restrictions (GPR) policies
	// (https://support.google.com/publisherpolicies/answer/10437795).
	//   "ADVERTISER_PREFERENCE" - Topics that are related to advertiser
	// preferences. Certain advertisers may choose not to bid on content that are
	// labeled with certain policies.
	//   "REGULATORY" - Any topics that are a result of a country or regional
	// regulatory requirement body.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "MustFix") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MustFix") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PolicyTopic) MarshalJSON() ([]byte, error) {
	type NoMethod PolicyTopic
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReportResult: Result of a generated report.
type ReportResult struct {
	// Averages: The averages of the report. This is the same length as any other
	// row in the report; cells corresponding to dimension columns are empty.
	Averages *Row `json:"averages,omitempty"`
	// EndDate: Required. End date of the range (inclusive).
	EndDate *Date `json:"endDate,omitempty"`
	// Headers: The header information; one for each dimension in the request,
	// followed by one for each metric in the request.
	Headers []*Header `json:"headers,omitempty"`
	// Rows: The output rows of the report. Each row is a list of cells; one for
	// each dimension in the request, followed by one for each metric in the
	// request.
	Rows []*Row `json:"rows,omitempty"`
	// StartDate: Required. Start date of the range (inclusive).
	StartDate *Date `json:"startDate,omitempty"`
	// TotalMatchedRows: The total number of rows matched by the report request.
	TotalMatchedRows int64 `json:"totalMatchedRows,omitempty,string"`
	// Totals: The totals of the report. This is the same length as any other row
	// in the report; cells corresponding to dimension columns are empty.
	Totals *Row `json:"totals,omitempty"`
	// Warnings: Any warnings associated with generation of the report. These
	// warnings are always returned in English.
	Warnings []string `json:"warnings,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Averages") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Averages") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReportResult) MarshalJSON() ([]byte, error) {
	type NoMethod ReportResult
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Row: Row representation.
type Row struct {
	// Cells: Cells in the row.
	Cells []*Cell `json:"cells,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Cells") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Cells") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Row) MarshalJSON() ([]byte, error) {
	type NoMethod Row
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SavedReport: Representation of a saved report.
type SavedReport struct {
	// Name: Output only. Resource name of the report. Format:
	// accounts/{account}/reports/{report}
	Name string `json:"name,omitempty"`
	// Title: Report title as specified by publisher.
	Title string `json:"title,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Name") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Name") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SavedReport) MarshalJSON() ([]byte, error) {
	type NoMethod SavedReport
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Site: Representation of a Site.
type Site struct {
	// AutoAdsEnabled: Whether auto ads is turned on for the site.
	AutoAdsEnabled bool `json:"autoAdsEnabled,omitempty"`
	// Domain: Domain (or subdomain) of the site, e.g. "example.com" or
	// "www.example.com". This is used in the `OWNED_SITE_DOMAIN_NAME` reporting
	// dimension.
	Domain string `json:"domain,omitempty"`
	// Name: Output only. Resource name of a site. Format:
	// accounts/{account}/sites/{site}
	Name string `json:"name,omitempty"`
	// ReportingDimensionId: Output only. Unique ID of the site as used in the
	// `OWNED_SITE_ID` reporting dimension.
	ReportingDimensionId string `json:"reportingDimensionId,omitempty"`
	// State: Output only. State of a site.
	//
	// Possible values:
	//   "STATE_UNSPECIFIED" - State unspecified.
	//   "REQUIRES_REVIEW" - Either: * The site hasn't been checked yet. * The site
	// is inactive and needs another review before it can show ads again. Learn how
	// to [request a review for an inactive
	// site](https://support.google.com/adsense/answer/9393996).
	//   "GETTING_READY" - Google is running some checks on the site. This usually
	// takes a few days, but in some cases it can take two to four weeks.
	//   "READY" - The site is ready to show ads. Learn how to [set up ads on the
	// site](https://support.google.com/adsense/answer/7037624).
	//   "NEEDS_ATTENTION" - Publisher needs to fix some issues before the site is
	// ready to show ads. Learn what to do [if a new site isn't
	// ready](https://support.google.com/adsense/answer/9061852).
	State string `json:"state,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AutoAdsEnabled") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AutoAdsEnabled") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Site) MarshalJSON() ([]byte, error) {
	type NoMethod Site
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TimeZone: Represents a time zone from the IANA Time Zone Database
// (https://www.iana.org/time-zones).
type TimeZone struct {
	// Id: IANA Time Zone Database time zone. For example "America/New_York".
	Id string `json:"id,omitempty"`
	// Version: Optional. IANA Time Zone Database version number. For example
	// "2019a".
	Version string `json:"version,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Id") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Id") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TimeZone) MarshalJSON() ([]byte, error) {
	type NoMethod TimeZone
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UrlChannel: Representation of a URL channel. URL channels allow you to track
// the performance of particular pages in your site; see URL channels
// (https://support.google.com/adsense/answer/2923836) for more information.
type UrlChannel struct {
	// Name: Output only. Resource name of the URL channel. Format:
	// accounts/{account}/adclients/{adclient}/urlchannels/{urlchannel}
	Name string `json:"name,omitempty"`
	// ReportingDimensionId: Output only. Unique ID of the custom channel as used
	// in the `URL_CHANNEL_ID` reporting dimension.
	ReportingDimensionId string `json:"reportingDimensionId,omitempty"`
	// UriPattern: URI pattern of the channel. Does not include "http://" or
	// "https://". Example: www.example.com/home
	UriPattern string `json:"uriPattern,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Name") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Name") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UrlChannel) MarshalJSON() ([]byte, error) {
	type NoMethod UrlChannel
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AccountsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets information about the selected AdSense account.
//
// - name: Account to get information about. Format: accounts/{account}.
func (r *AccountsService) Get(name string) *AccountsGetCall {
	c := &AccountsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsGetCall) Fields(s ...googleapi.Field) *AccountsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsGetCall) IfNoneMatch(entityTag string) *AccountsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsGetCall) Context(ctx context.Context) *AccountsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Account.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsGetCall) Do(opts ...googleapi.CallOption) (*Account, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Account{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsGetAdBlockingRecoveryTagCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetAdBlockingRecoveryTag: Gets the ad blocking recovery tag of an account.
//
//   - name: The name of the account to get the tag for. Format:
//     accounts/{account}.
func (r *AccountsService) GetAdBlockingRecoveryTag(name string) *AccountsGetAdBlockingRecoveryTagCall {
	c := &AccountsGetAdBlockingRecoveryTagCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsGetAdBlockingRecoveryTagCall) Fields(s ...googleapi.Field) *AccountsGetAdBlockingRecoveryTagCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsGetAdBlockingRecoveryTagCall) IfNoneMatch(entityTag string) *AccountsGetAdBlockingRecoveryTagCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsGetAdBlockingRecoveryTagCall) Context(ctx context.Context) *AccountsGetAdBlockingRecoveryTagCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsGetAdBlockingRecoveryTagCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsGetAdBlockingRecoveryTagCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}/adBlockingRecoveryTag")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.getAdBlockingRecoveryTag", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.getAdBlockingRecoveryTag" call.
// Any non-2xx status code is an error. Response headers are in either
// *AdBlockingRecoveryTag.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsGetAdBlockingRecoveryTagCall) Do(opts ...googleapi.CallOption) (*AdBlockingRecoveryTag, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AdBlockingRecoveryTag{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.getAdBlockingRecoveryTag", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all accounts available to this user.
func (r *AccountsService) List() *AccountsListCall {
	c := &AccountsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// accounts to include in the response, used for paging. If unspecified, at
// most 10000 accounts will be returned. The maximum value is 10000; values
// above 10000 will be coerced to 10000.
func (c *AccountsListCall) PageSize(pageSize int64) *AccountsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListAccounts` call. Provide this to retrieve the subsequent
// page. When paginating, all other parameters provided to `ListAccounts` must
// match the call that provided the page token.
func (c *AccountsListCall) PageToken(pageToken string) *AccountsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsListCall) Fields(s ...googleapi.Field) *AccountsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsListCall) IfNoneMatch(entityTag string) *AccountsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsListCall) Context(ctx context.Context) *AccountsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/accounts")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListAccountsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsListCall) Do(opts ...googleapi.CallOption) (*ListAccountsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListAccountsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsListCall) Pages(ctx context.Context, f func(*ListAccountsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsListChildAccountsCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// ListChildAccounts: Lists all accounts directly managed by the given AdSense
// account.
//
//   - parent: The parent account, which owns the child accounts. Format:
//     accounts/{account}.
func (r *AccountsService) ListChildAccounts(parent string) *AccountsListChildAccountsCall {
	c := &AccountsListChildAccountsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// accounts to include in the response, used for paging. If unspecified, at
// most 10000 accounts will be returned. The maximum value is 10000; values
// above 10000 will be coerced to 10000.
func (c *AccountsListChildAccountsCall) PageSize(pageSize int64) *AccountsListChildAccountsCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListChildAccounts` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListChildAccounts` must match the call that provided the page token.
func (c *AccountsListChildAccountsCall) PageToken(pageToken string) *AccountsListChildAccountsCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsListChildAccountsCall) Fields(s ...googleapi.Field) *AccountsListChildAccountsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsListChildAccountsCall) IfNoneMatch(entityTag string) *AccountsListChildAccountsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsListChildAccountsCall) Context(ctx context.Context) *AccountsListChildAccountsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsListChildAccountsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsListChildAccountsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}:listChildAccounts")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.listChildAccounts", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.listChildAccounts" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListChildAccountsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsListChildAccountsCall) Do(opts ...googleapi.CallOption) (*ListChildAccountsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListChildAccountsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.listChildAccounts", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsListChildAccountsCall) Pages(ctx context.Context, f func(*ListChildAccountsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsAdclientsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the ad client from the given resource name.
//
//   - name: The name of the ad client to retrieve. Format:
//     accounts/{account}/adclients/{adclient}.
func (r *AccountsAdclientsService) Get(name string) *AccountsAdclientsGetCall {
	c := &AccountsAdclientsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsGetCall) Fields(s ...googleapi.Field) *AccountsAdclientsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsAdclientsGetCall) IfNoneMatch(entityTag string) *AccountsAdclientsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsGetCall) Context(ctx context.Context) *AccountsAdclientsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *AdClient.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsAdclientsGetCall) Do(opts ...googleapi.CallOption) (*AdClient, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AdClient{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsAdclientsGetAdcodeCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetAdcode: Gets the AdSense code for a given ad client. This returns what
// was previously known as the 'auto ad code'. This is only supported for ad
// clients with a product_code of AFC. For more information, see About the
// AdSense code (https://support.google.com/adsense/answer/9274634).
//
//   - name: Name of the ad client for which to get the adcode. Format:
//     accounts/{account}/adclients/{adclient}.
func (r *AccountsAdclientsService) GetAdcode(name string) *AccountsAdclientsGetAdcodeCall {
	c := &AccountsAdclientsGetAdcodeCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsGetAdcodeCall) Fields(s ...googleapi.Field) *AccountsAdclientsGetAdcodeCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsAdclientsGetAdcodeCall) IfNoneMatch(entityTag string) *AccountsAdclientsGetAdcodeCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsGetAdcodeCall) Context(ctx context.Context) *AccountsAdclientsGetAdcodeCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsGetAdcodeCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsGetAdcodeCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}/adcode")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.getAdcode", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.getAdcode" call.
// Any non-2xx status code is an error. Response headers are in either
// *AdClientAdCode.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsAdclientsGetAdcodeCall) Do(opts ...googleapi.CallOption) (*AdClientAdCode, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AdClientAdCode{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.getAdcode", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsAdclientsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all the ad clients available in an account.
//
//   - parent: The account which owns the collection of ad clients. Format:
//     accounts/{account}.
func (r *AccountsAdclientsService) List(parent string) *AccountsAdclientsListCall {
	c := &AccountsAdclientsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of ad
// clients to include in the response, used for paging. If unspecified, at most
// 10000 ad clients will be returned. The maximum value is 10000; values above
// 10000 will be coerced to 10000.
func (c *AccountsAdclientsListCall) PageSize(pageSize int64) *AccountsAdclientsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListAdClients` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListAdClients` must match the call that provided the page token.
func (c *AccountsAdclientsListCall) PageToken(pageToken string) *AccountsAdclientsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsListCall) Fields(s ...googleapi.Field) *AccountsAdclientsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsAdclientsListCall) IfNoneMatch(entityTag string) *AccountsAdclientsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsListCall) Context(ctx context.Context) *AccountsAdclientsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/adclients")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListAdClientsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsAdclientsListCall) Do(opts ...googleapi.CallOption) (*ListAdClientsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListAdClientsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsAdclientsListCall) Pages(ctx context.Context, f func(*ListAdClientsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsAdclientsAdunitsCreateCall struct {
	s          *Service
	parent     string
	adunit     *AdUnit
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates an ad unit. This method can be called only by a restricted
// set of projects, which are usually owned by AdSense for Platforms
// (https://developers.google.com/adsense/platforms/) publishers. Contact your
// account manager if you need to use this method. Note that ad units can only
// be created for ad clients with an "AFC" product code. For more info see the
// AdClient resource
// (/adsense/management/reference/rest/v2/accounts.adclients). For now, this
// method can only be used to create `DISPLAY` ad units. See:
// https://support.google.com/adsense/answer/9183566
//
//   - parent: Ad client to create an ad unit under. Format:
//     accounts/{account}/adclients/{adclient}.
func (r *AccountsAdclientsAdunitsService) Create(parent string, adunit *AdUnit) *AccountsAdclientsAdunitsCreateCall {
	c := &AccountsAdclientsAdunitsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.adunit = adunit
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsAdunitsCreateCall) Fields(s ...googleapi.Field) *AccountsAdclientsAdunitsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsAdunitsCreateCall) Context(ctx context.Context) *AccountsAdclientsAdunitsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsAdunitsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsAdunitsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.adunit)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/adunits")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.adunits.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.adunits.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *AdUnit.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsAdclientsAdunitsCreateCall) Do(opts ...googleapi.CallOption) (*AdUnit, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AdUnit{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.adunits.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsAdclientsAdunitsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets an ad unit from a specified account and ad client.
//
//   - name: AdUnit to get information about. Format:
//     accounts/{account}/adclients/{adclient}/adunits/{adunit}.
func (r *AccountsAdclientsAdunitsService) Get(name string) *AccountsAdclientsAdunitsGetCall {
	c := &AccountsAdclientsAdunitsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsAdunitsGetCall) Fields(s ...googleapi.Field) *AccountsAdclientsAdunitsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsAdclientsAdunitsGetCall) IfNoneMatch(entityTag string) *AccountsAdclientsAdunitsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsAdunitsGetCall) Context(ctx context.Context) *AccountsAdclientsAdunitsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsAdunitsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsAdunitsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.adunits.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.adunits.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *AdUnit.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsAdclientsAdunitsGetCall) Do(opts ...googleapi.CallOption) (*AdUnit, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AdUnit{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.adunits.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsAdclientsAdunitsGetAdcodeCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetAdcode: Gets the ad unit code for a given ad unit. For more information,
// see About the AdSense code
// (https://support.google.com/adsense/answer/9274634) and Where to place the
// ad code in your HTML (https://support.google.com/adsense/answer/9190028).
//
//   - name: Name of the adunit for which to get the adcode. Format:
//     accounts/{account}/adclients/{adclient}/adunits/{adunit}.
func (r *AccountsAdclientsAdunitsService) GetAdcode(name string) *AccountsAdclientsAdunitsGetAdcodeCall {
	c := &AccountsAdclientsAdunitsGetAdcodeCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsAdunitsGetAdcodeCall) Fields(s ...googleapi.Field) *AccountsAdclientsAdunitsGetAdcodeCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsAdclientsAdunitsGetAdcodeCall) IfNoneMatch(entityTag string) *AccountsAdclientsAdunitsGetAdcodeCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsAdunitsGetAdcodeCall) Context(ctx context.Context) *AccountsAdclientsAdunitsGetAdcodeCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsAdunitsGetAdcodeCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsAdunitsGetAdcodeCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}/adcode")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.adunits.getAdcode", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.adunits.getAdcode" call.
// Any non-2xx status code is an error. Response headers are in either
// *AdUnitAdCode.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsAdclientsAdunitsGetAdcodeCall) Do(opts ...googleapi.CallOption) (*AdUnitAdCode, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AdUnitAdCode{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.adunits.getAdcode", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsAdclientsAdunitsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all ad units under a specified account and ad client.
//
//   - parent: The ad client which owns the collection of ad units. Format:
//     accounts/{account}/adclients/{adclient}.
func (r *AccountsAdclientsAdunitsService) List(parent string) *AccountsAdclientsAdunitsListCall {
	c := &AccountsAdclientsAdunitsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of ad
// units to include in the response, used for paging. If unspecified, at most
// 10000 ad units will be returned. The maximum value is 10000; values above
// 10000 will be coerced to 10000.
func (c *AccountsAdclientsAdunitsListCall) PageSize(pageSize int64) *AccountsAdclientsAdunitsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListAdUnits` call. Provide this to retrieve the subsequent
// page. When paginating, all other parameters provided to `ListAdUnits` must
// match the call that provided the page token.
func (c *AccountsAdclientsAdunitsListCall) PageToken(pageToken string) *AccountsAdclientsAdunitsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsAdunitsListCall) Fields(s ...googleapi.Field) *AccountsAdclientsAdunitsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsAdclientsAdunitsListCall) IfNoneMatch(entityTag string) *AccountsAdclientsAdunitsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsAdunitsListCall) Context(ctx context.Context) *AccountsAdclientsAdunitsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsAdunitsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsAdunitsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/adunits")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.adunits.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.adunits.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListAdUnitsResponse.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsAdclientsAdunitsListCall) Do(opts ...googleapi.CallOption) (*ListAdUnitsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListAdUnitsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.adunits.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsAdclientsAdunitsListCall) Pages(ctx context.Context, f func(*ListAdUnitsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsAdclientsAdunitsListLinkedCustomChannelsCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// ListLinkedCustomChannels: Lists all the custom channels available for an ad
// unit.
//
//   - parent: The ad unit which owns the collection of custom channels. Format:
//     accounts/{account}/adclients/{adclient}/adunits/{adunit}.
func (r *AccountsAdclientsAdunitsService) ListLinkedCustomChannels(parent string) *AccountsAdclientsAdunitsListLinkedCustomChannelsCall {
	c := &AccountsAdclientsAdunitsListLinkedCustomChannelsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// custom channels to include in the response, used for paging. If unspecified,
// at most 10000 custom channels will be returned. The maximum value is 10000;
// values above 10000 will be coerced to 10000.
func (c *AccountsAdclientsAdunitsListLinkedCustomChannelsCall) PageSize(pageSize int64) *AccountsAdclientsAdunitsListLinkedCustomChannelsCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListLinkedCustomChannels` call. Provide this to retrieve
// the subsequent page. When paginating, all other parameters provided to
// `ListLinkedCustomChannels` must match the call that provided the page token.
func (c *AccountsAdclientsAdunitsListLinkedCustomChannelsCall) PageToken(pageToken string) *AccountsAdclientsAdunitsListLinkedCustomChannelsCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsAdunitsListLinkedCustomChannelsCall) Fields(s ...googleapi.Field) *AccountsAdclientsAdunitsListLinkedCustomChannelsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsAdclientsAdunitsListLinkedCustomChannelsCall) IfNoneMatch(entityTag string) *AccountsAdclientsAdunitsListLinkedCustomChannelsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsAdunitsListLinkedCustomChannelsCall) Context(ctx context.Context) *AccountsAdclientsAdunitsListLinkedCustomChannelsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsAdunitsListLinkedCustomChannelsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsAdunitsListLinkedCustomChannelsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}:listLinkedCustomChannels")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.adunits.listLinkedCustomChannels", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.adunits.listLinkedCustomChannels" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListLinkedCustomChannelsResponse.ServerResponse.Header or (if a response
// was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsAdclientsAdunitsListLinkedCustomChannelsCall) Do(opts ...googleapi.CallOption) (*ListLinkedCustomChannelsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListLinkedCustomChannelsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.adunits.listLinkedCustomChannels", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsAdclientsAdunitsListLinkedCustomChannelsCall) Pages(ctx context.Context, f func(*ListLinkedCustomChannelsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsAdclientsAdunitsPatchCall struct {
	s          *Service
	name       string
	adunit     *AdUnit
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates an ad unit. This method can be called only by a restricted
// set of projects, which are usually owned by AdSense for Platforms
// (https://developers.google.com/adsense/platforms/) publishers. Contact your
// account manager if you need to use this method. For now, this method can
// only be used to update `DISPLAY` ad units. See:
// https://support.google.com/adsense/answer/9183566
//
//   - name: Output only. Resource name of the ad unit. Format:
//     accounts/{account}/adclients/{adclient}/adunits/{adunit}.
func (r *AccountsAdclientsAdunitsService) Patch(name string, adunit *AdUnit) *AccountsAdclientsAdunitsPatchCall {
	c := &AccountsAdclientsAdunitsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.adunit = adunit
	return c
}

// UpdateMask sets the optional parameter "updateMask": The list of fields to
// update. If empty, a full update is performed.
func (c *AccountsAdclientsAdunitsPatchCall) UpdateMask(updateMask string) *AccountsAdclientsAdunitsPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsAdunitsPatchCall) Fields(s ...googleapi.Field) *AccountsAdclientsAdunitsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsAdunitsPatchCall) Context(ctx context.Context) *AccountsAdclientsAdunitsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsAdunitsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsAdunitsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.adunit)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.adunits.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.adunits.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *AdUnit.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsAdclientsAdunitsPatchCall) Do(opts ...googleapi.CallOption) (*AdUnit, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AdUnit{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.adunits.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsAdclientsCustomchannelsCreateCall struct {
	s             *Service
	parent        string
	customchannel *CustomChannel
	urlParams_    gensupport.URLParams
	ctx_          context.Context
	header_       http.Header
}

// Create: Creates a custom channel. This method can be called only by a
// restricted set of projects, which are usually owned by AdSense for Platforms
// (https://developers.google.com/adsense/platforms/) publishers. Contact your
// account manager if you need to use this method.
//
//   - parent: The ad client to create a custom channel under. Format:
//     accounts/{account}/adclients/{adclient}.
func (r *AccountsAdclientsCustomchannelsService) Create(parent string, customchannel *CustomChannel) *AccountsAdclientsCustomchannelsCreateCall {
	c := &AccountsAdclientsCustomchannelsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.customchannel = customchannel
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsCustomchannelsCreateCall) Fields(s ...googleapi.Field) *AccountsAdclientsCustomchannelsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsCustomchannelsCreateCall) Context(ctx context.Context) *AccountsAdclientsCustomchannelsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsCustomchannelsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsCustomchannelsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.customchannel)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/customchannels")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.customchannels.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.customchannels.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *CustomChannel.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsAdclientsCustomchannelsCreateCall) Do(opts ...googleapi.CallOption) (*CustomChannel, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CustomChannel{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.customchannels.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsAdclientsCustomchannelsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a custom channel. This method can be called only by a
// restricted set of projects, which are usually owned by AdSense for Platforms
// (https://developers.google.com/adsense/platforms/) publishers. Contact your
// account manager if you need to use this method.
//
//   - name: Name of the custom channel to delete. Format:
//     accounts/{account}/adclients/{adclient}/customchannels/{customchannel}.
func (r *AccountsAdclientsCustomchannelsService) Delete(name string) *AccountsAdclientsCustomchannelsDeleteCall {
	c := &AccountsAdclientsCustomchannelsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsCustomchannelsDeleteCall) Fields(s ...googleapi.Field) *AccountsAdclientsCustomchannelsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsCustomchannelsDeleteCall) Context(ctx context.Context) *AccountsAdclientsCustomchannelsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsCustomchannelsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsCustomchannelsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.customchannels.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.customchannels.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsAdclientsCustomchannelsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.customchannels.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsAdclientsCustomchannelsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets information about the selected custom channel.
//
//   - name: Name of the custom channel. Format:
//     accounts/{account}/adclients/{adclient}/customchannels/{customchannel}.
func (r *AccountsAdclientsCustomchannelsService) Get(name string) *AccountsAdclientsCustomchannelsGetCall {
	c := &AccountsAdclientsCustomchannelsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsCustomchannelsGetCall) Fields(s ...googleapi.Field) *AccountsAdclientsCustomchannelsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsAdclientsCustomchannelsGetCall) IfNoneMatch(entityTag string) *AccountsAdclientsCustomchannelsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsCustomchannelsGetCall) Context(ctx context.Context) *AccountsAdclientsCustomchannelsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsCustomchannelsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsCustomchannelsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.customchannels.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.customchannels.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *CustomChannel.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsAdclientsCustomchannelsGetCall) Do(opts ...googleapi.CallOption) (*CustomChannel, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CustomChannel{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.customchannels.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsAdclientsCustomchannelsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all the custom channels available in an ad client.
//
//   - parent: The ad client which owns the collection of custom channels.
//     Format: accounts/{account}/adclients/{adclient}.
func (r *AccountsAdclientsCustomchannelsService) List(parent string) *AccountsAdclientsCustomchannelsListCall {
	c := &AccountsAdclientsCustomchannelsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// custom channels to include in the response, used for paging. If unspecified,
// at most 10000 custom channels will be returned. The maximum value is 10000;
// values above 10000 will be coerced to 10000.
func (c *AccountsAdclientsCustomchannelsListCall) PageSize(pageSize int64) *AccountsAdclientsCustomchannelsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListCustomChannels` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListCustomChannels` must match the call that provided the page token.
func (c *AccountsAdclientsCustomchannelsListCall) PageToken(pageToken string) *AccountsAdclientsCustomchannelsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsCustomchannelsListCall) Fields(s ...googleapi.Field) *AccountsAdclientsCustomchannelsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsAdclientsCustomchannelsListCall) IfNoneMatch(entityTag string) *AccountsAdclientsCustomchannelsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsCustomchannelsListCall) Context(ctx context.Context) *AccountsAdclientsCustomchannelsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsCustomchannelsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsCustomchannelsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/customchannels")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.customchannels.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.customchannels.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListCustomChannelsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsAdclientsCustomchannelsListCall) Do(opts ...googleapi.CallOption) (*ListCustomChannelsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListCustomChannelsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.customchannels.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsAdclientsCustomchannelsListCall) Pages(ctx context.Context, f func(*ListCustomChannelsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsAdclientsCustomchannelsListLinkedAdUnitsCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// ListLinkedAdUnits: Lists all the ad units available for a custom channel.
//
//   - parent: The custom channel which owns the collection of ad units. Format:
//     accounts/{account}/adclients/{adclient}/customchannels/{customchannel}.
func (r *AccountsAdclientsCustomchannelsService) ListLinkedAdUnits(parent string) *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall {
	c := &AccountsAdclientsCustomchannelsListLinkedAdUnitsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of ad
// units to include in the response, used for paging. If unspecified, at most
// 10000 ad units will be returned. The maximum value is 10000; values above
// 10000 will be coerced to 10000.
func (c *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall) PageSize(pageSize int64) *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListLinkedAdUnits` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListLinkedAdUnits` must match the call that provided the page token.
func (c *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall) PageToken(pageToken string) *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall) Fields(s ...googleapi.Field) *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall) IfNoneMatch(entityTag string) *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall) Context(ctx context.Context) *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}:listLinkedAdUnits")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.customchannels.listLinkedAdUnits", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.customchannels.listLinkedAdUnits" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListLinkedAdUnitsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall) Do(opts ...googleapi.CallOption) (*ListLinkedAdUnitsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListLinkedAdUnitsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.customchannels.listLinkedAdUnits", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsAdclientsCustomchannelsListLinkedAdUnitsCall) Pages(ctx context.Context, f func(*ListLinkedAdUnitsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsAdclientsCustomchannelsPatchCall struct {
	s             *Service
	name          string
	customchannel *CustomChannel
	urlParams_    gensupport.URLParams
	ctx_          context.Context
	header_       http.Header
}

// Patch: Updates a custom channel. This method can be called only by a
// restricted set of projects, which are usually owned by AdSense for Platforms
// (https://developers.google.com/adsense/platforms/) publishers. Contact your
// account manager if you need to use this method.
//
//   - name: Output only. Resource name of the custom channel. Format:
//     accounts/{account}/adclients/{adclient}/customchannels/{customchannel}.
func (r *AccountsAdclientsCustomchannelsService) Patch(name string, customchannel *CustomChannel) *AccountsAdclientsCustomchannelsPatchCall {
	c := &AccountsAdclientsCustomchannelsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.customchannel = customchannel
	return c
}

// UpdateMask sets the optional parameter "updateMask": The list of fields to
// update. If empty, a full update is performed.
func (c *AccountsAdclientsCustomchannelsPatchCall) UpdateMask(updateMask string) *AccountsAdclientsCustomchannelsPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsCustomchannelsPatchCall) Fields(s ...googleapi.Field) *AccountsAdclientsCustomchannelsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsCustomchannelsPatchCall) Context(ctx context.Context) *AccountsAdclientsCustomchannelsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsCustomchannelsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsCustomchannelsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.customchannel)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.customchannels.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.customchannels.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *CustomChannel.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsAdclientsCustomchannelsPatchCall) Do(opts ...googleapi.CallOption) (*CustomChannel, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CustomChannel{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.customchannels.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsAdclientsUrlchannelsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets information about the selected url channel.
//
//   - name: The name of the url channel to retrieve. Format:
//     accounts/{account}/adclients/{adclient}/urlchannels/{urlchannel}.
func (r *AccountsAdclientsUrlchannelsService) Get(name string) *AccountsAdclientsUrlchannelsGetCall {
	c := &AccountsAdclientsUrlchannelsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsUrlchannelsGetCall) Fields(s ...googleapi.Field) *AccountsAdclientsUrlchannelsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsAdclientsUrlchannelsGetCall) IfNoneMatch(entityTag string) *AccountsAdclientsUrlchannelsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsUrlchannelsGetCall) Context(ctx context.Context) *AccountsAdclientsUrlchannelsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsUrlchannelsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsUrlchannelsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.urlchannels.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.urlchannels.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *UrlChannel.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsAdclientsUrlchannelsGetCall) Do(opts ...googleapi.CallOption) (*UrlChannel, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &UrlChannel{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.urlchannels.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsAdclientsUrlchannelsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists active url channels.
//
//   - parent: The ad client which owns the collection of url channels. Format:
//     accounts/{account}/adclients/{adclient}.
func (r *AccountsAdclientsUrlchannelsService) List(parent string) *AccountsAdclientsUrlchannelsListCall {
	c := &AccountsAdclientsUrlchannelsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of url
// channels to include in the response, used for paging. If unspecified, at
// most 10000 url channels will be returned. The maximum value is 10000; values
// above 10000 will be coerced to 10000.
func (c *AccountsAdclientsUrlchannelsListCall) PageSize(pageSize int64) *AccountsAdclientsUrlchannelsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListUrlChannels` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListUrlChannels` must match the call that provided the page token.
func (c *AccountsAdclientsUrlchannelsListCall) PageToken(pageToken string) *AccountsAdclientsUrlchannelsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAdclientsUrlchannelsListCall) Fields(s ...googleapi.Field) *AccountsAdclientsUrlchannelsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsAdclientsUrlchannelsListCall) IfNoneMatch(entityTag string) *AccountsAdclientsUrlchannelsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAdclientsUrlchannelsListCall) Context(ctx context.Context) *AccountsAdclientsUrlchannelsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAdclientsUrlchannelsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAdclientsUrlchannelsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/urlchannels")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.urlchannels.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.adclients.urlchannels.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListUrlChannelsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsAdclientsUrlchannelsListCall) Do(opts ...googleapi.CallOption) (*ListUrlChannelsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListUrlChannelsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.adclients.urlchannels.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsAdclientsUrlchannelsListCall) Pages(ctx context.Context, f func(*ListUrlChannelsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsAlertsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all the alerts available in an account.
//
//   - parent: The account which owns the collection of alerts. Format:
//     accounts/{account}.
func (r *AccountsAlertsService) List(parent string) *AccountsAlertsListCall {
	c := &AccountsAlertsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// LanguageCode sets the optional parameter "languageCode": The language to use
// for translating alert messages. If unspecified, this defaults to the user's
// display language. If the given language is not supported, alerts will be
// returned in English. The language is specified as an IETF BCP-47 language
// code (https://en.wikipedia.org/wiki/IETF_language_tag).
func (c *AccountsAlertsListCall) LanguageCode(languageCode string) *AccountsAlertsListCall {
	c.urlParams_.Set("languageCode", languageCode)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsAlertsListCall) Fields(s ...googleapi.Field) *AccountsAlertsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsAlertsListCall) IfNoneMatch(entityTag string) *AccountsAlertsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsAlertsListCall) Context(ctx context.Context) *AccountsAlertsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsAlertsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsAlertsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/alerts")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.alerts.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.alerts.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListAlertsResponse.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsAlertsListCall) Do(opts ...googleapi.CallOption) (*ListAlertsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListAlertsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.alerts.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsPaymentsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all the payments available for an account.
//
//   - parent: The account which owns the collection of payments. Format:
//     accounts/{account}.
func (r *AccountsPaymentsService) List(parent string) *AccountsPaymentsListCall {
	c := &AccountsPaymentsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsPaymentsListCall) Fields(s ...googleapi.Field) *AccountsPaymentsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsPaymentsListCall) IfNoneMatch(entityTag string) *AccountsPaymentsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsPaymentsListCall) Context(ctx context.Context) *AccountsPaymentsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsPaymentsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsPaymentsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/payments")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.payments.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.payments.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListPaymentsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsPaymentsListCall) Do(opts ...googleapi.CallOption) (*ListPaymentsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListPaymentsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.payments.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsPolicyIssuesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets information about the selected policy issue.
//
//   - name: Name of the policy issue. Format:
//     accounts/{account}/policyIssues/{policy_issue}.
func (r *AccountsPolicyIssuesService) Get(name string) *AccountsPolicyIssuesGetCall {
	c := &AccountsPolicyIssuesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsPolicyIssuesGetCall) Fields(s ...googleapi.Field) *AccountsPolicyIssuesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsPolicyIssuesGetCall) IfNoneMatch(entityTag string) *AccountsPolicyIssuesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsPolicyIssuesGetCall) Context(ctx context.Context) *AccountsPolicyIssuesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsPolicyIssuesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsPolicyIssuesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.policyIssues.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.policyIssues.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *PolicyIssue.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsPolicyIssuesGetCall) Do(opts ...googleapi.CallOption) (*PolicyIssue, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &PolicyIssue{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.policyIssues.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsPolicyIssuesListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all the policy issues where the specified account is involved,
// both directly and through any AFP child accounts.
//
//   - parent: The account for which policy issues are being retrieved. Format:
//     accounts/{account}.
func (r *AccountsPolicyIssuesService) List(parent string) *AccountsPolicyIssuesListCall {
	c := &AccountsPolicyIssuesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// policy issues to include in the response, used for paging. If unspecified,
// at most 10000 policy issues will be returned. The maximum value is 10000;
// values above 10000 will be coerced to 10000.
func (c *AccountsPolicyIssuesListCall) PageSize(pageSize int64) *AccountsPolicyIssuesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListPolicyIssues` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListPolicyIssues` must match the call that provided the page token.
func (c *AccountsPolicyIssuesListCall) PageToken(pageToken string) *AccountsPolicyIssuesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsPolicyIssuesListCall) Fields(s ...googleapi.Field) *AccountsPolicyIssuesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsPolicyIssuesListCall) IfNoneMatch(entityTag string) *AccountsPolicyIssuesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsPolicyIssuesListCall) Context(ctx context.Context) *AccountsPolicyIssuesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsPolicyIssuesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsPolicyIssuesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/policyIssues")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.policyIssues.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.policyIssues.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListPolicyIssuesResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsPolicyIssuesListCall) Do(opts ...googleapi.CallOption) (*ListPolicyIssuesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListPolicyIssuesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.policyIssues.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsPolicyIssuesListCall) Pages(ctx context.Context, f func(*ListPolicyIssuesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsReportsGenerateCall struct {
	s            *Service
	account      string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Generate: Generates an ad hoc report.
//
//   - account: The account which owns the collection of reports. Format:
//     accounts/{account}.
func (r *AccountsReportsService) Generate(account string) *AccountsReportsGenerateCall {
	c := &AccountsReportsGenerateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.account = account
	return c
}

// CurrencyCode sets the optional parameter "currencyCode": The ISO-4217
// currency code (https://en.wikipedia.org/wiki/ISO_4217) to use when reporting
// on monetary metrics. Defaults to the account's currency if not set.
func (c *AccountsReportsGenerateCall) CurrencyCode(currencyCode string) *AccountsReportsGenerateCall {
	c.urlParams_.Set("currencyCode", currencyCode)
	return c
}

// DateRange sets the optional parameter "dateRange": Date range of the report,
// if unset the range will be considered CUSTOM.
//
// Possible values:
//
//	"REPORTING_DATE_RANGE_UNSPECIFIED" - Unspecified date range.
//	"CUSTOM" - A custom date range specified using the `start_date` and
//
// `end_date` fields. This is the default if no ReportingDateRange is provided.
//
//	"TODAY" - Current day.
//	"YESTERDAY" - Yesterday.
//	"MONTH_TO_DATE" - From the start of the current month to the current day.
//
// e.g. if the current date is 2020-03-12 then the range will be [2020-03-01,
// 2020-03-12].
//
//	"YEAR_TO_DATE" - From the start of the current year to the current day.
//
// e.g. if the current date is 2020-03-12 then the range will be [2020-01-01,
// 2020-03-12].
//
//	"LAST_7_DAYS" - Last 7 days, excluding current day.
//	"LAST_30_DAYS" - Last 30 days, excluding current day.
func (c *AccountsReportsGenerateCall) DateRange(dateRange string) *AccountsReportsGenerateCall {
	c.urlParams_.Set("dateRange", dateRange)
	return c
}

// Dimensions sets the optional parameter "dimensions": Dimensions to base the
// report on.
//
// Possible values:
//
//	"DIMENSION_UNSPECIFIED" - Unspecified dimension.
//	"DATE" - Date dimension in YYYY-MM-DD format (e.g. "2010-02-10").
//	"WEEK" - Week dimension in YYYY-MM-DD format, representing the first day
//
// of each week (e.g. "2010-02-08"). The first day of the week is determined by
// the language_code specified in a report generation request (so e.g. this
// would be a Monday for "en-GB" or "es", but a Sunday for "en" or "fr-CA").
//
//	"MONTH" - Month dimension in YYYY-MM format (e.g. "2010-02").
//	"ACCOUNT_NAME" - Account name. The members of this dimension match the
//
// values from Account.display_name.
//
//	"AD_CLIENT_ID" - Unique ID of an ad client. The members of this dimension
//
// match the values from AdClient.reporting_dimension_id.
//
//	"HOSTED_AD_CLIENT_ID" - Unique ID of a sub-account's ad client. The
//
// members of this dimension match the values from
// AdClient.reporting_dimension_id (for the sub-account).
//
//	"PRODUCT_NAME" - Localized product name (e.g. "AdSense for Content",
//
// "AdSense for Search").
//
//	"PRODUCT_CODE" - Product code (e.g. "AFC", "AFS"). The members of this
//
// dimension match the values from AdClient.product_code.
//
//	"AD_UNIT_NAME" - Ad unit name (within which an ad was served). The members
//
// of this dimension match the values from AdUnit.display_name.
//
//	"AD_UNIT_ID" - Unique ID of an ad unit (within which an ad was served).
//
// The members of this dimension match the values from
// AdUnit.reporting_dimension_id.
//
//	"AD_UNIT_SIZE_NAME" - Localized size of an ad unit (e.g. "728x90",
//
// "Responsive").
//
//	"AD_UNIT_SIZE_CODE" - The size code of an ad unit (e.g. "728x90",
//
// "responsive").
//
//	"CUSTOM_CHANNEL_NAME" - Custom channel name. The members of this dimension
//
// match the values from CustomChannel.display_name.
//
//	"CUSTOM_CHANNEL_ID" - Unique ID of a custom channel. The members of this
//
// dimension match the values from CustomChannel.reporting_dimension_id.
//
//	"HOSTED_CUSTOM_CHANNEL_ID" - Not supported.
//	"OWNED_SITE_DOMAIN_NAME" - Domain name of a verified site (e.g.
//
// "example.com"). The members of this dimension match the values from
// Site.domain.
//
//	"OWNED_SITE_ID" - Unique ID of a verified site. The members of this
//
// dimension match the values from Site.reporting_dimension_id.
//
//	"PAGE_URL" - URL of the page upon which the ad was served. This is a
//
// complete URL including scheme and query parameters. Note that the URL that
// appears in this dimension may be a canonicalized version of the one that was
// used in the original request, and so may not exactly match the URL that a
// user might have seen. Note that there are also some caveats to be aware of
// when using this dimension. For more information, see [Page URL
// breakdown](https://support.google.com/adsense/answer/11988478).
//
//	"URL_CHANNEL_NAME" - Name of a URL channel. The members of this dimension
//
// match the values from UrlChannel.uri_pattern.
//
//	"URL_CHANNEL_ID" - Unique ID of a URL channel. The members of this
//
// dimension match the values from UrlChannel.reporting_dimension_id.
//
//	"BUYER_NETWORK_NAME" - Name of an ad network that returned the winning ads
//
// for an ad request (e.g. "Google AdWords"). Note that unlike other "NAME"
// dimensions, the members of this dimensions are not localized.
//
//	"BUYER_NETWORK_ID" - Unique (opaque) ID of an ad network that returned the
//
// winning ads for an ad request.
//
//	"BID_TYPE_NAME" - Localized bid type name (e.g. "CPC bids", "CPM bids")
//
// for a served ad.
//
//	"BID_TYPE_CODE" - Type of a bid (e.g. "cpc", "cpm") for a served ad.
//	"CREATIVE_SIZE_NAME" - Localized creative size name (e.g. "728x90",
//
// "Dynamic") of a served ad.
//
//	"CREATIVE_SIZE_CODE" - Creative size code (e.g. "728x90", "dynamic") of a
//
// served ad.
//
//	"DOMAIN_NAME" - Localized name of a host on which an ad was served, after
//
// IDNA decoding (e.g. "www.google.com", "Web caches and other",
// "bücher.example").
//
//	"DOMAIN_CODE" - Name of a host on which an ad was served (e.g.
//
// "www.google.com", "webcaches", "xn--bcher-kva.example").
//
//	"COUNTRY_NAME" - Localized region name of a user viewing an ad (e.g.
//
// "United States", "France").
//
//	"COUNTRY_CODE" - CLDR region code of a user viewing an ad (e.g. "US",
//
// "FR").
//
//	"PLATFORM_TYPE_NAME" - Localized platform type name (e.g. "High-end mobile
//
// devices", "Desktop").
//
//	"PLATFORM_TYPE_CODE" - Platform type code (e.g. "HighEndMobile",
//
// "Desktop").
//
//	"TARGETING_TYPE_NAME" - Localized targeting type name (e.g. "Contextual",
//
// "Personalized", "Run of Network").
//
//	"TARGETING_TYPE_CODE" - Targeting type code (e.g. "Keyword",
//
// "UserInterest", "RunOfNetwork").
//
//	"CONTENT_PLATFORM_NAME" - Localized content platform name an ad request
//
// was made from (e.g. "AMP", "Web").
//
//	"CONTENT_PLATFORM_CODE" - Content platform code an ad request was made
//
// from (e.g. "AMP", "HTML").
//
//	"AD_PLACEMENT_NAME" - Localized ad placement name (e.g. "Ad unit", "Global
//
// settings", "Manual").
//
//	"AD_PLACEMENT_CODE" - Ad placement code (e.g. "AD_UNIT",
//
// "ca-pub-123456:78910", "OTHER").
//
//	"REQUESTED_AD_TYPE_NAME" - Localized requested ad type name (e.g.
//
// "Display", "Link unit", "Other").
//
//	"REQUESTED_AD_TYPE_CODE" - Requested ad type code (e.g. "IMAGE",
//
// "RADLINK", "OTHER").
//
//	"SERVED_AD_TYPE_NAME" - Localized served ad type name (e.g. "Display",
//
// "Link unit", "Other").
//
//	"SERVED_AD_TYPE_CODE" - Served ad type code (e.g. "IMAGE", "RADLINK",
//
// "OTHER").
//
//	"AD_FORMAT_NAME" - Localized ad format name indicating the way an ad is
//
// shown to the users on your site (e.g. "In-page", "Anchor", "Vignette").
//
//	"AD_FORMAT_CODE" - Ad format code indicating the way an ad is shown to the
//
// users on your site (e.g. "ON_PAGE", "ANCHOR", "INTERSTITIAL").
//
//	"CUSTOM_SEARCH_STYLE_NAME" - Custom search style name.
//	"CUSTOM_SEARCH_STYLE_ID" - Custom search style id.
//	"DOMAIN_REGISTRANT" - Domain registrants.
//	"WEBSEARCH_QUERY_STRING" - Query strings for web searches.
func (c *AccountsReportsGenerateCall) Dimensions(dimensions ...string) *AccountsReportsGenerateCall {
	c.urlParams_.SetMulti("dimensions", append([]string{}, dimensions...))
	return c
}

// EndDateDay sets the optional parameter "endDate.day": Day of a month. Must
// be from 1 to 31 and valid for the year and month, or 0 to specify a year by
// itself or a year and month where the day isn't significant.
func (c *AccountsReportsGenerateCall) EndDateDay(endDateDay int64) *AccountsReportsGenerateCall {
	c.urlParams_.Set("endDate.day", fmt.Sprint(endDateDay))
	return c
}

// EndDateMonth sets the optional parameter "endDate.month": Month of a year.
// Must be from 1 to 12, or 0 to specify a year without a month and day.
func (c *AccountsReportsGenerateCall) EndDateMonth(endDateMonth int64) *AccountsReportsGenerateCall {
	c.urlParams_.Set("endDate.month", fmt.Sprint(endDateMonth))
	return c
}

// EndDateYear sets the optional parameter "endDate.year": Year of the date.
// Must be from 1 to 9999, or 0 to specify a date without a year.
func (c *AccountsReportsGenerateCall) EndDateYear(endDateYear int64) *AccountsReportsGenerateCall {
	c.urlParams_.Set("endDate.year", fmt.Sprint(endDateYear))
	return c
}

// Filters sets the optional parameter "filters": A list of filters
// (/adsense/management/reporting/filtering) to apply to the report. All
// provided filters must match in order for the data to be included in the
// report.
func (c *AccountsReportsGenerateCall) Filters(filters ...string) *AccountsReportsGenerateCall {
	c.urlParams_.SetMulti("filters", append([]string{}, filters...))
	return c
}

// LanguageCode sets the optional parameter "languageCode": The language to use
// for translating report output. If unspecified, this defaults to English
// ("en"). If the given language is not supported, report output will be
// returned in English. The language is specified as an IETF BCP-47 language
// code (https://en.wikipedia.org/wiki/IETF_language_tag).
func (c *AccountsReportsGenerateCall) LanguageCode(languageCode string) *AccountsReportsGenerateCall {
	c.urlParams_.Set("languageCode", languageCode)
	return c
}

// Limit sets the optional parameter "limit": The maximum number of rows of
// report data to return. Reports producing more rows than the requested limit
// will be truncated. If unset, this defaults to 100,000 rows for
// `Reports.GenerateReport` and 1,000,000 rows for `Reports.GenerateCsvReport`,
// which are also the maximum values permitted here. Report truncation can be
// identified (for `Reports.GenerateReport` only) by comparing the number of
// rows returned to the value returned in `total_matched_rows`.
func (c *AccountsReportsGenerateCall) Limit(limit int64) *AccountsReportsGenerateCall {
	c.urlParams_.Set("limit", fmt.Sprint(limit))
	return c
}

// Metrics sets the optional parameter "metrics": Required. Reporting metrics.
//
// Possible values:
//
//	"METRIC_UNSPECIFIED" - Unspecified metric.
//	"PAGE_VIEWS" - Number of page views.
//	"AD_REQUESTS" - Number of ad units that requested ads (for content ads) or
//
// search queries (for search ads). An ad request may result in zero, one, or
// multiple individual ad impressions depending on the size of the ad unit and
// whether any ads were available.
//
//	"MATCHED_AD_REQUESTS" - Requests that returned at least one ad.
//	"TOTAL_IMPRESSIONS" - Impressions. An impression is counted for each ad
//
// request where at least one ad has been downloaded to the user's device and
// has begun to load. It is the number of ad units (for content ads) or search
// queries (for search ads) that showed ads.
//
//	"IMPRESSIONS" - Impressions. An impression is counted for each ad request
//
// where at least one ad has been downloaded to the user's device and has begun
// to load. It is the number of ad units (for content ads) or search queries
// (for search ads) that showed ads.
//
//	"INDIVIDUAL_AD_IMPRESSIONS" - Ads shown. Different ad formats will display
//
// varying numbers of ads. For example, a vertical banner may consist of 2 or
// more ads. Also, the number of ads in an ad unit may vary depending on
// whether the ad unit is displaying standard text ads, expanded text ads or
// image ads.
//
//	"CLICKS" - Number of times a user clicked on a standard content ad.
//	"PAGE_VIEWS_SPAM_RATIO" - Fraction of page views considered to be spam.
//
// Only available to premium accounts.
//
//	"AD_REQUESTS_SPAM_RATIO" - Fraction of ad requests considered to be spam.
//
// Only available to premium accounts.
//
//	"MATCHED_AD_REQUESTS_SPAM_RATIO" - Fraction of ad requests that returned
//
// ads considered to be spam. Only available to premium accounts.
//
//	"IMPRESSIONS_SPAM_RATIO" - Fraction of impressions considered to be spam.
//
// Only available to premium accounts.
//
//	"INDIVIDUAL_AD_IMPRESSIONS_SPAM_RATIO" - Fraction of ad impressions
//
// considered to be spam. Only available to premium accounts.
//
//	"CLICKS_SPAM_RATIO" - Fraction of clicks considered to be spam. Only
//
// available to premium accounts.
//
//	"AD_REQUESTS_COVERAGE" - Ratio of requested ad units or queries to the
//
// number returned to the site.
//
//	"PAGE_VIEWS_CTR" - Ratio of individual page views that resulted in a
//
// click.
//
//	"AD_REQUESTS_CTR" - Ratio of ad requests that resulted in a click.
//	"MATCHED_AD_REQUESTS_CTR" - Ratio of clicks to matched requests.
//	"IMPRESSIONS_CTR" - Ratio of IMPRESSIONS that resulted in a click.
//	"INDIVIDUAL_AD_IMPRESSIONS_CTR" - Ratio of individual ad impressions that
//
// resulted in a click.
//
//	"ACTIVE_VIEW_MEASURABILITY" - Ratio of requests that were measurable for
//
// viewability.
//
//	"ACTIVE_VIEW_VIEWABILITY" - Ratio of requests that were viewable.
//	"ACTIVE_VIEW_TIME" - Mean time an ad was displayed on screen.
//	"ESTIMATED_EARNINGS" - Estimated earnings of the publisher. Note that
//
// earnings up to yesterday are accurate, more recent earnings are estimated
// due to the possibility of spam, or exchange rate fluctuations.
//
//	"PAGE_VIEWS_RPM" - Revenue per thousand page views. This is calculated by
//
// dividing the estimated revenue by the number of page views multiplied by
// 1000.
//
//	"AD_REQUESTS_RPM" - Revenue per thousand ad requests. This is calculated
//
// by dividing estimated revenue by the number of ad requests multiplied by
// 1000.
//
//	"MATCHED_AD_REQUESTS_RPM" - Revenue per thousand matched ad requests. This
//
// is calculated by dividing estimated revenue by the number of matched ad
// requests multiplied by 1000.
//
//	"IMPRESSIONS_RPM" - Revenue per thousand ad impressions. This is
//
// calculated by dividing estimated revenue by the number of ad impressions
// multiplied by 1000.
//
//	"INDIVIDUAL_AD_IMPRESSIONS_RPM" - Revenue per thousand individual ad
//
// impressions. This is calculated by dividing estimated revenue by the number
// of individual ad impressions multiplied by 1000.
//
//	"COST_PER_CLICK" - Amount the publisher earns each time a user clicks on
//
// an ad. CPC is calculated by dividing the estimated revenue by the number of
// clicks received.
//
//	"ADS_PER_IMPRESSION" - Number of ad views per impression.
//	"TOTAL_EARNINGS" - Total earnings are the gross estimated earnings from
//
// revenue shared traffic before any parent and child account revenue share is
// applied.
//
//	"WEBSEARCH_RESULT_PAGES" - Number of results pages. This metric can only
//
// be used when generating a report in the Google timezone, not the account
// timezone. Since the account timezone is the default for report generation,
// this metric can only be used by explicitly specifying
// `reportingTimeZone=GOOGLE_TIME_ZONE`.
//
//	"FUNNEL_REQUESTS" - Number of requests for non-ad units (for example a
//
// related search unit). For more information, see [Funnel
// requests](https://support.google.com/adsense/answer/11586959).
//
//	"FUNNEL_IMPRESSIONS" - Number of requests for non-ad units ads that
//
// returned content that was shown to the user. For more information, see
// [Funnel impressions](https://support.google.com/adsense/answer/11585767).
//
//	"FUNNEL_CLICKS" - Number of times a user clicked on a non-ad unit,
//
// triggering further ad requests. For more information, see [Funnel
// clicks](https://support.google.com/adsense/answer/11586382).
//
//	"FUNNEL_RPM" - Revenue per thousand funnel impressions. This is calculated
//
// by dividing estimated revenue by the number of funnel impressions multiplied
// by 1000. For more information, see [Funnel
// RPM](https://support.google.com/adsense/answer/11585979).
func (c *AccountsReportsGenerateCall) Metrics(metrics ...string) *AccountsReportsGenerateCall {
	c.urlParams_.SetMulti("metrics", append([]string{}, metrics...))
	return c
}

// OrderBy sets the optional parameter "orderBy": The name of a dimension or
// metric to sort the resulting report on, can be prefixed with "+" to sort
// ascending or "-" to sort descending. If no prefix is specified, the column
// is sorted ascending.
func (c *AccountsReportsGenerateCall) OrderBy(orderBy ...string) *AccountsReportsGenerateCall {
	c.urlParams_.SetMulti("orderBy", append([]string{}, orderBy...))
	return c
}

// ReportingTimeZone sets the optional parameter "reportingTimeZone": Timezone
// in which to generate the report. If unspecified, this defaults to the
// account timezone. For more information, see changing the time zone of your
// reports (https://support.google.com/adsense/answer/9830725).
//
// Possible values:
//
//	"REPORTING_TIME_ZONE_UNSPECIFIED" - Unspecified timezone.
//	"ACCOUNT_TIME_ZONE" - Use the account timezone in the report.
//	"GOOGLE_TIME_ZONE" - Use the Google timezone in the report
//
// (America/Los_Angeles).
func (c *AccountsReportsGenerateCall) ReportingTimeZone(reportingTimeZone string) *AccountsReportsGenerateCall {
	c.urlParams_.Set("reportingTimeZone", reportingTimeZone)
	return c
}

// StartDateDay sets the optional parameter "startDate.day": Day of a month.
// Must be from 1 to 31 and valid for the year and month, or 0 to specify a
// year by itself or a year and month where the day isn't significant.
func (c *AccountsReportsGenerateCall) StartDateDay(startDateDay int64) *AccountsReportsGenerateCall {
	c.urlParams_.Set("startDate.day", fmt.Sprint(startDateDay))
	return c
}

// StartDateMonth sets the optional parameter "startDate.month": Month of a
// year. Must be from 1 to 12, or 0 to specify a year without a month and day.
func (c *AccountsReportsGenerateCall) StartDateMonth(startDateMonth int64) *AccountsReportsGenerateCall {
	c.urlParams_.Set("startDate.month", fmt.Sprint(startDateMonth))
	return c
}

// StartDateYear sets the optional parameter "startDate.year": Year of the
// date. Must be from 1 to 9999, or 0 to specify a date without a year.
func (c *AccountsReportsGenerateCall) StartDateYear(startDateYear int64) *AccountsReportsGenerateCall {
	c.urlParams_.Set("startDate.year", fmt.Sprint(startDateYear))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsReportsGenerateCall) Fields(s ...googleapi.Field) *AccountsReportsGenerateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsReportsGenerateCall) IfNoneMatch(entityTag string) *AccountsReportsGenerateCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsReportsGenerateCall) Context(ctx context.Context) *AccountsReportsGenerateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsReportsGenerateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsReportsGenerateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+account}/reports:generate")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"account": c.account,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.reports.generate", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.reports.generate" call.
// Any non-2xx status code is an error. Response headers are in either
// *ReportResult.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsReportsGenerateCall) Do(opts ...googleapi.CallOption) (*ReportResult, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ReportResult{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.reports.generate", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsReportsGenerateCsvCall struct {
	s            *Service
	account      string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GenerateCsv: Generates a csv formatted ad hoc report.
//
//   - account: The account which owns the collection of reports. Format:
//     accounts/{account}.
func (r *AccountsReportsService) GenerateCsv(account string) *AccountsReportsGenerateCsvCall {
	c := &AccountsReportsGenerateCsvCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.account = account
	return c
}

// CurrencyCode sets the optional parameter "currencyCode": The ISO-4217
// currency code (https://en.wikipedia.org/wiki/ISO_4217) to use when reporting
// on monetary metrics. Defaults to the account's currency if not set.
func (c *AccountsReportsGenerateCsvCall) CurrencyCode(currencyCode string) *AccountsReportsGenerateCsvCall {
	c.urlParams_.Set("currencyCode", currencyCode)
	return c
}

// DateRange sets the optional parameter "dateRange": Date range of the report,
// if unset the range will be considered CUSTOM.
//
// Possible values:
//
//	"REPORTING_DATE_RANGE_UNSPECIFIED" - Unspecified date range.
//	"CUSTOM" - A custom date range specified using the `start_date` and
//
// `end_date` fields. This is the default if no ReportingDateRange is provided.
//
//	"TODAY" - Current day.
//	"YESTERDAY" - Yesterday.
//	"MONTH_TO_DATE" - From the start of the current month to the current day.
//
// e.g. if the current date is 2020-03-12 then the range will be [2020-03-01,
// 2020-03-12].
//
//	"YEAR_TO_DATE" - From the start of the current year to the current day.
//
// e.g. if the current date is 2020-03-12 then the range will be [2020-01-01,
// 2020-03-12].
//
//	"LAST_7_DAYS" - Last 7 days, excluding current day.
//	"LAST_30_DAYS" - Last 30 days, excluding current day.
func (c *AccountsReportsGenerateCsvCall) DateRange(dateRange string) *AccountsReportsGenerateCsvCall {
	c.urlParams_.Set("dateRange", dateRange)
	return c
}

// Dimensions sets the optional parameter "dimensions": Dimensions to base the
// report on.
//
// Possible values:
//
//	"DIMENSION_UNSPECIFIED" - Unspecified dimension.
//	"DATE" - Date dimension in YYYY-MM-DD format (e.g. "2010-02-10").
//	"WEEK" - Week dimension in YYYY-MM-DD format, representing the first day
//
// of each week (e.g. "2010-02-08"). The first day of the week is determined by
// the language_code specified in a report generation request (so e.g. this
// would be a Monday for "en-GB" or "es", but a Sunday for "en" or "fr-CA").
//
//	"MONTH" - Month dimension in YYYY-MM format (e.g. "2010-02").
//	"ACCOUNT_NAME" - Account name. The members of this dimension match the
//
// values from Account.display_name.
//
//	"AD_CLIENT_ID" - Unique ID of an ad client. The members of this dimension
//
// match the values from AdClient.reporting_dimension_id.
//
//	"HOSTED_AD_CLIENT_ID" - Unique ID of a sub-account's ad client. The
//
// members of this dimension match the values from
// AdClient.reporting_dimension_id (for the sub-account).
//
//	"PRODUCT_NAME" - Localized product name (e.g. "AdSense for Content",
//
// "AdSense for Search").
//
//	"PRODUCT_CODE" - Product code (e.g. "AFC", "AFS"). The members of this
//
// dimension match the values from AdClient.product_code.
//
//	"AD_UNIT_NAME" - Ad unit name (within which an ad was served). The members
//
// of this dimension match the values from AdUnit.display_name.
//
//	"AD_UNIT_ID" - Unique ID of an ad unit (within which an ad was served).
//
// The members of this dimension match the values from
// AdUnit.reporting_dimension_id.
//
//	"AD_UNIT_SIZE_NAME" - Localized size of an ad unit (e.g. "728x90",
//
// "Responsive").
//
//	"AD_UNIT_SIZE_CODE" - The size code of an ad unit (e.g. "728x90",
//
// "responsive").
//
//	"CUSTOM_CHANNEL_NAME" - Custom channel name. The members of this dimension
//
// match the values from CustomChannel.display_name.
//
//	"CUSTOM_CHANNEL_ID" - Unique ID of a custom channel. The members of this
//
// dimension match the values from CustomChannel.reporting_dimension_id.
//
//	"HOSTED_CUSTOM_CHANNEL_ID" - Not supported.
//	"OWNED_SITE_DOMAIN_NAME" - Domain name of a verified site (e.g.
//
// "example.com"). The members of this dimension match the values from
// Site.domain.
//
//	"OWNED_SITE_ID" - Unique ID of a verified site. The members of this
//
// dimension match the values from Site.reporting_dimension_id.
//
//	"PAGE_URL" - URL of the page upon which the ad was served. This is a
//
// complete URL including scheme and query parameters. Note that the URL that
// appears in this dimension may be a canonicalized version of the one that was
// used in the original request, and so may not exactly match the URL that a
// user might have seen. Note that there are also some caveats to be aware of
// when using this dimension. For more information, see [Page URL
// breakdown](https://support.google.com/adsense/answer/11988478).
//
//	"URL_CHANNEL_NAME" - Name of a URL channel. The members of this dimension
//
// match the values from UrlChannel.uri_pattern.
//
//	"URL_CHANNEL_ID" - Unique ID of a URL channel. The members of this
//
// dimension match the values from UrlChannel.reporting_dimension_id.
//
//	"BUYER_NETWORK_NAME" - Name of an ad network that returned the winning ads
//
// for an ad request (e.g. "Google AdWords"). Note that unlike other "NAME"
// dimensions, the members of this dimensions are not localized.
//
//	"BUYER_NETWORK_ID" - Unique (opaque) ID of an ad network that returned the
//
// winning ads for an ad request.
//
//	"BID_TYPE_NAME" - Localized bid type name (e.g. "CPC bids", "CPM bids")
//
// for a served ad.
//
//	"BID_TYPE_CODE" - Type of a bid (e.g. "cpc", "cpm") for a served ad.
//	"CREATIVE_SIZE_NAME" - Localized creative size name (e.g. "728x90",
//
// "Dynamic") of a served ad.
//
//	"CREATIVE_SIZE_CODE" - Creative size code (e.g. "728x90", "dynamic") of a
//
// served ad.
//
//	"DOMAIN_NAME" - Localized name of a host on which an ad was served, after
//
// IDNA decoding (e.g. "www.google.com", "Web caches and other",
// "bücher.example").
//
//	"DOMAIN_CODE" - Name of a host on which an ad was served (e.g.
//
// "www.google.com", "webcaches", "xn--bcher-kva.example").
//
//	"COUNTRY_NAME" - Localized region name of a user viewing an ad (e.g.
//
// "United States", "France").
//
//	"COUNTRY_CODE" - CLDR region code of a user viewing an ad (e.g. "US",
//
// "FR").
//
//	"PLATFORM_TYPE_NAME" - Localized platform type name (e.g. "High-end mobile
//
// devices", "Desktop").
//
//	"PLATFORM_TYPE_CODE" - Platform type code (e.g. "HighEndMobile",
//
// "Desktop").
//
//	"TARGETING_TYPE_NAME" - Localized targeting type name (e.g. "Contextual",
//
// "Personalized", "Run of Network").
//
//	"TARGETING_TYPE_CODE" - Targeting type code (e.g. "Keyword",
//
// "UserInterest", "RunOfNetwork").
//
//	"CONTENT_PLATFORM_NAME" - Localized content platform name an ad request
//
// was made from (e.g. "AMP", "Web").
//
//	"CONTENT_PLATFORM_CODE" - Content platform code an ad request was made
//
// from (e.g. "AMP", "HTML").
//
//	"AD_PLACEMENT_NAME" - Localized ad placement name (e.g. "Ad unit", "Global
//
// settings", "Manual").
//
//	"AD_PLACEMENT_CODE" - Ad placement code (e.g. "AD_UNIT",
//
// "ca-pub-123456:78910", "OTHER").
//
//	"REQUESTED_AD_TYPE_NAME" - Localized requested ad type name (e.g.
//
// "Display", "Link unit", "Other").
//
//	"REQUESTED_AD_TYPE_CODE" - Requested ad type code (e.g. "IMAGE",
//
// "RADLINK", "OTHER").
//
//	"SERVED_AD_TYPE_NAME" - Localized served ad type name (e.g. "Display",
//
// "Link unit", "Other").
//
//	"SERVED_AD_TYPE_CODE" - Served ad type code (e.g. "IMAGE", "RADLINK",
//
// "OTHER").
//
//	"AD_FORMAT_NAME" - Localized ad format name indicating the way an ad is
//
// shown to the users on your site (e.g. "In-page", "Anchor", "Vignette").
//
//	"AD_FORMAT_CODE" - Ad format code indicating the way an ad is shown to the
//
// users on your site (e.g. "ON_PAGE", "ANCHOR", "INTERSTITIAL").
//
//	"CUSTOM_SEARCH_STYLE_NAME" - Custom search style name.
//	"CUSTOM_SEARCH_STYLE_ID" - Custom search style id.
//	"DOMAIN_REGISTRANT" - Domain registrants.
//	"WEBSEARCH_QUERY_STRING" - Query strings for web searches.
func (c *AccountsReportsGenerateCsvCall) Dimensions(dimensions ...string) *AccountsReportsGenerateCsvCall {
	c.urlParams_.SetMulti("dimensions", append([]string{}, dimensions...))
	return c
}

// EndDateDay sets the optional parameter "endDate.day": Day of a month. Must
// be from 1 to 31 and valid for the year and month, or 0 to specify a year by
// itself or a year and month where the day isn't significant.
func (c *AccountsReportsGenerateCsvCall) EndDateDay(endDateDay int64) *AccountsReportsGenerateCsvCall {
	c.urlParams_.Set("endDate.day", fmt.Sprint(endDateDay))
	return c
}

// EndDateMonth sets the optional parameter "endDate.month": Month of a year.
// Must be from 1 to 12, or 0 to specify a year without a month and day.
func (c *AccountsReportsGenerateCsvCall) EndDateMonth(endDateMonth int64) *AccountsReportsGenerateCsvCall {
	c.urlParams_.Set("endDate.month", fmt.Sprint(endDateMonth))
	return c
}

// EndDateYear sets the optional parameter "endDate.year": Year of the date.
// Must be from 1 to 9999, or 0 to specify a date without a year.
func (c *AccountsReportsGenerateCsvCall) EndDateYear(endDateYear int64) *AccountsReportsGenerateCsvCall {
	c.urlParams_.Set("endDate.year", fmt.Sprint(endDateYear))
	return c
}

// Filters sets the optional parameter "filters": A list of filters
// (/adsense/management/reporting/filtering) to apply to the report. All
// provided filters must match in order for the data to be included in the
// report.
func (c *AccountsReportsGenerateCsvCall) Filters(filters ...string) *AccountsReportsGenerateCsvCall {
	c.urlParams_.SetMulti("filters", append([]string{}, filters...))
	return c
}

// LanguageCode sets the optional parameter "languageCode": The language to use
// for translating report output. If unspecified, this defaults to English
// ("en"). If the given language is not supported, report output will be
// returned in English. The language is specified as an IETF BCP-47 language
// code (https://en.wikipedia.org/wiki/IETF_language_tag).
func (c *AccountsReportsGenerateCsvCall) LanguageCode(languageCode string) *AccountsReportsGenerateCsvCall {
	c.urlParams_.Set("languageCode", languageCode)
	return c
}

// Limit sets the optional parameter "limit": The maximum number of rows of
// report data to return. Reports producing more rows than the requested limit
// will be truncated. If unset, this defaults to 100,000 rows for
// `Reports.GenerateReport` and 1,000,000 rows for `Reports.GenerateCsvReport`,
// which are also the maximum values permitted here. Report truncation can be
// identified (for `Reports.GenerateReport` only) by comparing the number of
// rows returned to the value returned in `total_matched_rows`.
func (c *AccountsReportsGenerateCsvCall) Limit(limit int64) *AccountsReportsGenerateCsvCall {
	c.urlParams_.Set("limit", fmt.Sprint(limit))
	return c
}

// Metrics sets the optional parameter "metrics": Required. Reporting metrics.
//
// Possible values:
//
//	"METRIC_UNSPECIFIED" - Unspecified metric.
//	"PAGE_VIEWS" - Number of page views.
//	"AD_REQUESTS" - Number of ad units that requested ads (for content ads) or
//
// search queries (for search ads). An ad request may result in zero, one, or
// multiple individual ad impressions depending on the size of the ad unit and
// whether any ads were available.
//
//	"MATCHED_AD_REQUESTS" - Requests that returned at least one ad.
//	"TOTAL_IMPRESSIONS" - Impressions. An impression is counted for each ad
//
// request where at least one ad has been downloaded to the user's device and
// has begun to load. It is the number of ad units (for content ads) or search
// queries (for search ads) that showed ads.
//
//	"IMPRESSIONS" - Impressions. An impression is counted for each ad request
//
// where at least one ad has been downloaded to the user's device and has begun
// to load. It is the number of ad units (for content ads) or search queries
// (for search ads) that showed ads.
//
//	"INDIVIDUAL_AD_IMPRESSIONS" - Ads shown. Different ad formats will display
//
// varying numbers of ads. For example, a vertical banner may consist of 2 or
// more ads. Also, the number of ads in an ad unit may vary depending on
// whether the ad unit is displaying standard text ads, expanded text ads or
// image ads.
//
//	"CLICKS" - Number of times a user clicked on a standard content ad.
//	"PAGE_VIEWS_SPAM_RATIO" - Fraction of page views considered to be spam.
//
// Only available to premium accounts.
//
//	"AD_REQUESTS_SPAM_RATIO" - Fraction of ad requests considered to be spam.
//
// Only available to premium accounts.
//
//	"MATCHED_AD_REQUESTS_SPAM_RATIO" - Fraction of ad requests that returned
//
// ads considered to be spam. Only available to premium accounts.
//
//	"IMPRESSIONS_SPAM_RATIO" - Fraction of impressions considered to be spam.
//
// Only available to premium accounts.
//
//	"INDIVIDUAL_AD_IMPRESSIONS_SPAM_RATIO" - Fraction of ad impressions
//
// considered to be spam. Only available to premium accounts.
//
//	"CLICKS_SPAM_RATIO" - Fraction of clicks considered to be spam. Only
//
// available to premium accounts.
//
//	"AD_REQUESTS_COVERAGE" - Ratio of requested ad units or queries to the
//
// number returned to the site.
//
//	"PAGE_VIEWS_CTR" - Ratio of individual page views that resulted in a
//
// click.
//
//	"AD_REQUESTS_CTR" - Ratio of ad requests that resulted in a click.
//	"MATCHED_AD_REQUESTS_CTR" - Ratio of clicks to matched requests.
//	"IMPRESSIONS_CTR" - Ratio of IMPRESSIONS that resulted in a click.
//	"INDIVIDUAL_AD_IMPRESSIONS_CTR" - Ratio of individual ad impressions that
//
// resulted in a click.
//
//	"ACTIVE_VIEW_MEASURABILITY" - Ratio of requests that were measurable for
//
// viewability.
//
//	"ACTIVE_VIEW_VIEWABILITY" - Ratio of requests that were viewable.
//	"ACTIVE_VIEW_TIME" - Mean time an ad was displayed on screen.
//	"ESTIMATED_EARNINGS" - Estimated earnings of the publisher. Note that
//
// earnings up to yesterday are accurate, more recent earnings are estimated
// due to the possibility of spam, or exchange rate fluctuations.
//
//	"PAGE_VIEWS_RPM" - Revenue per thousand page views. This is calculated by
//
// dividing the estimated revenue by the number of page views multiplied by
// 1000.
//
//	"AD_REQUESTS_RPM" - Revenue per thousand ad requests. This is calculated
//
// by dividing estimated revenue by the number of ad requests multiplied by
// 1000.
//
//	"MATCHED_AD_REQUESTS_RPM" - Revenue per thousand matched ad requests. This
//
// is calculated by dividing estimated revenue by the number of matched ad
// requests multiplied by 1000.
//
//	"IMPRESSIONS_RPM" - Revenue per thousand ad impressions. This is
//
// calculated by dividing estimated revenue by the number of ad impressions
// multiplied by 1000.
//
//	"INDIVIDUAL_AD_IMPRESSIONS_RPM" - Revenue per thousand individual ad
//
// impressions. This is calculated by dividing estimated revenue by the number
// of individual ad impressions multiplied by 1000.
//
//	"COST_PER_CLICK" - Amount the publisher earns each time a user clicks on
//
// an ad. CPC is calculated by dividing the estimated revenue by the number of
// clicks received.
//
//	"ADS_PER_IMPRESSION" - Number of ad views per impression.
//	"TOTAL_EARNINGS" - Total earnings are the gross estimated earnings from
//
// revenue shared traffic before any parent and child account revenue share is
// applied.
//
//	"WEBSEARCH_RESULT_PAGES" - Number of results pages. This metric can only
//
// be used when generating a report in the Google timezone, not the account
// timezone. Since the account timezone is the default for report generation,
// this metric can only be used by explicitly specifying
// `reportingTimeZone=GOOGLE_TIME_ZONE`.
//
//	"FUNNEL_REQUESTS" - Number of requests for non-ad units (for example a
//
// related search unit). For more information, see [Funnel
// requests](https://support.google.com/adsense/answer/11586959).
//
//	"FUNNEL_IMPRESSIONS" - Number of requests for non-ad units ads that
//
// returned content that was shown to the user. For more information, see
// [Funnel impressions](https://support.google.com/adsense/answer/11585767).
//
//	"FUNNEL_CLICKS" - Number of times a user clicked on a non-ad unit,
//
// triggering further ad requests. For more information, see [Funnel
// clicks](https://support.google.com/adsense/answer/11586382).
//
//	"FUNNEL_RPM" - Revenue per thousand funnel impressions. This is calculated
//
// by dividing estimated revenue by the number of funnel impressions multiplied
// by 1000. For more information, see [Funnel
// RPM](https://support.google.com/adsense/answer/11585979).
func (c *AccountsReportsGenerateCsvCall) Metrics(metrics ...string) *AccountsReportsGenerateCsvCall {
	c.urlParams_.SetMulti("metrics", append([]string{}, metrics...))
	return c
}

// OrderBy sets the optional parameter "orderBy": The name of a dimension or
// metric to sort the resulting report on, can be prefixed with "+" to sort
// ascending or "-" to sort descending. If no prefix is specified, the column
// is sorted ascending.
func (c *AccountsReportsGenerateCsvCall) OrderBy(orderBy ...string) *AccountsReportsGenerateCsvCall {
	c.urlParams_.SetMulti("orderBy", append([]string{}, orderBy...))
	return c
}

// ReportingTimeZone sets the optional parameter "reportingTimeZone": Timezone
// in which to generate the report. If unspecified, this defaults to the
// account timezone. For more information, see changing the time zone of your
// reports (https://support.google.com/adsense/answer/9830725).
//
// Possible values:
//
//	"REPORTING_TIME_ZONE_UNSPECIFIED" - Unspecified timezone.
//	"ACCOUNT_TIME_ZONE" - Use the account timezone in the report.
//	"GOOGLE_TIME_ZONE" - Use the Google timezone in the report
//
// (America/Los_Angeles).
func (c *AccountsReportsGenerateCsvCall) ReportingTimeZone(reportingTimeZone string) *AccountsReportsGenerateCsvCall {
	c.urlParams_.Set("reportingTimeZone", reportingTimeZone)
	return c
}

// StartDateDay sets the optional parameter "startDate.day": Day of a month.
// Must be from 1 to 31 and valid for the year and month, or 0 to specify a
// year by itself or a year and month where the day isn't significant.
func (c *AccountsReportsGenerateCsvCall) StartDateDay(startDateDay int64) *AccountsReportsGenerateCsvCall {
	c.urlParams_.Set("startDate.day", fmt.Sprint(startDateDay))
	return c
}

// StartDateMonth sets the optional parameter "startDate.month": Month of a
// year. Must be from 1 to 12, or 0 to specify a year without a month and day.
func (c *AccountsReportsGenerateCsvCall) StartDateMonth(startDateMonth int64) *AccountsReportsGenerateCsvCall {
	c.urlParams_.Set("startDate.month", fmt.Sprint(startDateMonth))
	return c
}

// StartDateYear sets the optional parameter "startDate.year": Year of the
// date. Must be from 1 to 9999, or 0 to specify a date without a year.
func (c *AccountsReportsGenerateCsvCall) StartDateYear(startDateYear int64) *AccountsReportsGenerateCsvCall {
	c.urlParams_.Set("startDate.year", fmt.Sprint(startDateYear))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsReportsGenerateCsvCall) Fields(s ...googleapi.Field) *AccountsReportsGenerateCsvCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsReportsGenerateCsvCall) IfNoneMatch(entityTag string) *AccountsReportsGenerateCsvCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsReportsGenerateCsvCall) Context(ctx context.Context) *AccountsReportsGenerateCsvCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsReportsGenerateCsvCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsReportsGenerateCsvCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+account}/reports:generateCsv")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"account": c.account,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.reports.generateCsv", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.reports.generateCsv" call.
// Any non-2xx status code is an error. Response headers are in either
// *HttpBody.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsReportsGenerateCsvCall) Do(opts ...googleapi.CallOption) (*HttpBody, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &HttpBody{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.reports.generateCsv", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsReportsGetSavedCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetSaved: Gets the saved report from the given resource name.
//
//   - name: The name of the saved report to retrieve. Format:
//     accounts/{account}/reports/{report}.
func (r *AccountsReportsService) GetSaved(name string) *AccountsReportsGetSavedCall {
	c := &AccountsReportsGetSavedCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsReportsGetSavedCall) Fields(s ...googleapi.Field) *AccountsReportsGetSavedCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsReportsGetSavedCall) IfNoneMatch(entityTag string) *AccountsReportsGetSavedCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsReportsGetSavedCall) Context(ctx context.Context) *AccountsReportsGetSavedCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsReportsGetSavedCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsReportsGetSavedCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}/saved")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.reports.getSaved", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.reports.getSaved" call.
// Any non-2xx status code is an error. Response headers are in either
// *SavedReport.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsReportsGetSavedCall) Do(opts ...googleapi.CallOption) (*SavedReport, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &SavedReport{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.reports.getSaved", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsReportsSavedGenerateCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Generate: Generates a saved report.
//
//   - name: Name of the saved report. Format:
//     accounts/{account}/reports/{report}.
func (r *AccountsReportsSavedService) Generate(name string) *AccountsReportsSavedGenerateCall {
	c := &AccountsReportsSavedGenerateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// CurrencyCode sets the optional parameter "currencyCode": The ISO-4217
// currency code (https://en.wikipedia.org/wiki/ISO_4217) to use when reporting
// on monetary metrics. Defaults to the account's currency if not set.
func (c *AccountsReportsSavedGenerateCall) CurrencyCode(currencyCode string) *AccountsReportsSavedGenerateCall {
	c.urlParams_.Set("currencyCode", currencyCode)
	return c
}

// DateRange sets the optional parameter "dateRange": Date range of the report,
// if unset the range will be considered CUSTOM.
//
// Possible values:
//
//	"REPORTING_DATE_RANGE_UNSPECIFIED" - Unspecified date range.
//	"CUSTOM" - A custom date range specified using the `start_date` and
//
// `end_date` fields. This is the default if no ReportingDateRange is provided.
//
//	"TODAY" - Current day.
//	"YESTERDAY" - Yesterday.
//	"MONTH_TO_DATE" - From the start of the current month to the current day.
//
// e.g. if the current date is 2020-03-12 then the range will be [2020-03-01,
// 2020-03-12].
//
//	"YEAR_TO_DATE" - From the start of the current year to the current day.
//
// e.g. if the current date is 2020-03-12 then the range will be [2020-01-01,
// 2020-03-12].
//
//	"LAST_7_DAYS" - Last 7 days, excluding current day.
//	"LAST_30_DAYS" - Last 30 days, excluding current day.
func (c *AccountsReportsSavedGenerateCall) DateRange(dateRange string) *AccountsReportsSavedGenerateCall {
	c.urlParams_.Set("dateRange", dateRange)
	return c
}

// EndDateDay sets the optional parameter "endDate.day": Day of a month. Must
// be from 1 to 31 and valid for the year and month, or 0 to specify a year by
// itself or a year and month where the day isn't significant.
func (c *AccountsReportsSavedGenerateCall) EndDateDay(endDateDay int64) *AccountsReportsSavedGenerateCall {
	c.urlParams_.Set("endDate.day", fmt.Sprint(endDateDay))
	return c
}

// EndDateMonth sets the optional parameter "endDate.month": Month of a year.
// Must be from 1 to 12, or 0 to specify a year without a month and day.
func (c *AccountsReportsSavedGenerateCall) EndDateMonth(endDateMonth int64) *AccountsReportsSavedGenerateCall {
	c.urlParams_.Set("endDate.month", fmt.Sprint(endDateMonth))
	return c
}

// EndDateYear sets the optional parameter "endDate.year": Year of the date.
// Must be from 1 to 9999, or 0 to specify a date without a year.
func (c *AccountsReportsSavedGenerateCall) EndDateYear(endDateYear int64) *AccountsReportsSavedGenerateCall {
	c.urlParams_.Set("endDate.year", fmt.Sprint(endDateYear))
	return c
}

// LanguageCode sets the optional parameter "languageCode": The language to use
// for translating report output. If unspecified, this defaults to English
// ("en"). If the given language is not supported, report output will be
// returned in English. The language is specified as an IETF BCP-47 language
// code (https://en.wikipedia.org/wiki/IETF_language_tag).
func (c *AccountsReportsSavedGenerateCall) LanguageCode(languageCode string) *AccountsReportsSavedGenerateCall {
	c.urlParams_.Set("languageCode", languageCode)
	return c
}

// ReportingTimeZone sets the optional parameter "reportingTimeZone": Timezone
// in which to generate the report. If unspecified, this defaults to the
// account timezone. For more information, see changing the time zone of your
// reports (https://support.google.com/adsense/answer/9830725).
//
// Possible values:
//
//	"REPORTING_TIME_ZONE_UNSPECIFIED" - Unspecified timezone.
//	"ACCOUNT_TIME_ZONE" - Use the account timezone in the report.
//	"GOOGLE_TIME_ZONE" - Use the Google timezone in the report
//
// (America/Los_Angeles).
func (c *AccountsReportsSavedGenerateCall) ReportingTimeZone(reportingTimeZone string) *AccountsReportsSavedGenerateCall {
	c.urlParams_.Set("reportingTimeZone", reportingTimeZone)
	return c
}

// StartDateDay sets the optional parameter "startDate.day": Day of a month.
// Must be from 1 to 31 and valid for the year and month, or 0 to specify a
// year by itself or a year and month where the day isn't significant.
func (c *AccountsReportsSavedGenerateCall) StartDateDay(startDateDay int64) *AccountsReportsSavedGenerateCall {
	c.urlParams_.Set("startDate.day", fmt.Sprint(startDateDay))
	return c
}

// StartDateMonth sets the optional parameter "startDate.month": Month of a
// year. Must be from 1 to 12, or 0 to specify a year without a month and day.
func (c *AccountsReportsSavedGenerateCall) StartDateMonth(startDateMonth int64) *AccountsReportsSavedGenerateCall {
	c.urlParams_.Set("startDate.month", fmt.Sprint(startDateMonth))
	return c
}

// StartDateYear sets the optional parameter "startDate.year": Year of the
// date. Must be from 1 to 9999, or 0 to specify a date without a year.
func (c *AccountsReportsSavedGenerateCall) StartDateYear(startDateYear int64) *AccountsReportsSavedGenerateCall {
	c.urlParams_.Set("startDate.year", fmt.Sprint(startDateYear))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsReportsSavedGenerateCall) Fields(s ...googleapi.Field) *AccountsReportsSavedGenerateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsReportsSavedGenerateCall) IfNoneMatch(entityTag string) *AccountsReportsSavedGenerateCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsReportsSavedGenerateCall) Context(ctx context.Context) *AccountsReportsSavedGenerateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsReportsSavedGenerateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsReportsSavedGenerateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}/saved:generate")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.reports.saved.generate", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.reports.saved.generate" call.
// Any non-2xx status code is an error. Response headers are in either
// *ReportResult.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsReportsSavedGenerateCall) Do(opts ...googleapi.CallOption) (*ReportResult, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ReportResult{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.reports.saved.generate", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsReportsSavedGenerateCsvCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GenerateCsv: Generates a csv formatted saved report.
//
//   - name: Name of the saved report. Format:
//     accounts/{account}/reports/{report}.
func (r *AccountsReportsSavedService) GenerateCsv(name string) *AccountsReportsSavedGenerateCsvCall {
	c := &AccountsReportsSavedGenerateCsvCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// CurrencyCode sets the optional parameter "currencyCode": The ISO-4217
// currency code (https://en.wikipedia.org/wiki/ISO_4217) to use when reporting
// on monetary metrics. Defaults to the account's currency if not set.
func (c *AccountsReportsSavedGenerateCsvCall) CurrencyCode(currencyCode string) *AccountsReportsSavedGenerateCsvCall {
	c.urlParams_.Set("currencyCode", currencyCode)
	return c
}

// DateRange sets the optional parameter "dateRange": Date range of the report,
// if unset the range will be considered CUSTOM.
//
// Possible values:
//
//	"REPORTING_DATE_RANGE_UNSPECIFIED" - Unspecified date range.
//	"CUSTOM" - A custom date range specified using the `start_date` and
//
// `end_date` fields. This is the default if no ReportingDateRange is provided.
//
//	"TODAY" - Current day.
//	"YESTERDAY" - Yesterday.
//	"MONTH_TO_DATE" - From the start of the current month to the current day.
//
// e.g. if the current date is 2020-03-12 then the range will be [2020-03-01,
// 2020-03-12].
//
//	"YEAR_TO_DATE" - From the start of the current year to the current day.
//
// e.g. if the current date is 2020-03-12 then the range will be [2020-01-01,
// 2020-03-12].
//
//	"LAST_7_DAYS" - Last 7 days, excluding current day.
//	"LAST_30_DAYS" - Last 30 days, excluding current day.
func (c *AccountsReportsSavedGenerateCsvCall) DateRange(dateRange string) *AccountsReportsSavedGenerateCsvCall {
	c.urlParams_.Set("dateRange", dateRange)
	return c
}

// EndDateDay sets the optional parameter "endDate.day": Day of a month. Must
// be from 1 to 31 and valid for the year and month, or 0 to specify a year by
// itself or a year and month where the day isn't significant.
func (c *AccountsReportsSavedGenerateCsvCall) EndDateDay(endDateDay int64) *AccountsReportsSavedGenerateCsvCall {
	c.urlParams_.Set("endDate.day", fmt.Sprint(endDateDay))
	return c
}

// EndDateMonth sets the optional parameter "endDate.month": Month of a year.
// Must be from 1 to 12, or 0 to specify a year without a month and day.
func (c *AccountsReportsSavedGenerateCsvCall) EndDateMonth(endDateMonth int64) *AccountsReportsSavedGenerateCsvCall {
	c.urlParams_.Set("endDate.month", fmt.Sprint(endDateMonth))
	return c
}

// EndDateYear sets the optional parameter "endDate.year": Year of the date.
// Must be from 1 to 9999, or 0 to specify a date without a year.
func (c *AccountsReportsSavedGenerateCsvCall) EndDateYear(endDateYear int64) *AccountsReportsSavedGenerateCsvCall {
	c.urlParams_.Set("endDate.year", fmt.Sprint(endDateYear))
	return c
}

// LanguageCode sets the optional parameter "languageCode": The language to use
// for translating report output. If unspecified, this defaults to English
// ("en"). If the given language is not supported, report output will be
// returned in English. The language is specified as an IETF BCP-47 language
// code (https://en.wikipedia.org/wiki/IETF_language_tag).
func (c *AccountsReportsSavedGenerateCsvCall) LanguageCode(languageCode string) *AccountsReportsSavedGenerateCsvCall {
	c.urlParams_.Set("languageCode", languageCode)
	return c
}

// ReportingTimeZone sets the optional parameter "reportingTimeZone": Timezone
// in which to generate the report. If unspecified, this defaults to the
// account timezone. For more information, see changing the time zone of your
// reports (https://support.google.com/adsense/answer/9830725).
//
// Possible values:
//
//	"REPORTING_TIME_ZONE_UNSPECIFIED" - Unspecified timezone.
//	"ACCOUNT_TIME_ZONE" - Use the account timezone in the report.
//	"GOOGLE_TIME_ZONE" - Use the Google timezone in the report
//
// (America/Los_Angeles).
func (c *AccountsReportsSavedGenerateCsvCall) ReportingTimeZone(reportingTimeZone string) *AccountsReportsSavedGenerateCsvCall {
	c.urlParams_.Set("reportingTimeZone", reportingTimeZone)
	return c
}

// StartDateDay sets the optional parameter "startDate.day": Day of a month.
// Must be from 1 to 31 and valid for the year and month, or 0 to specify a
// year by itself or a year and month where the day isn't significant.
func (c *AccountsReportsSavedGenerateCsvCall) StartDateDay(startDateDay int64) *AccountsReportsSavedGenerateCsvCall {
	c.urlParams_.Set("startDate.day", fmt.Sprint(startDateDay))
	return c
}

// StartDateMonth sets the optional parameter "startDate.month": Month of a
// year. Must be from 1 to 12, or 0 to specify a year without a month and day.
func (c *AccountsReportsSavedGenerateCsvCall) StartDateMonth(startDateMonth int64) *AccountsReportsSavedGenerateCsvCall {
	c.urlParams_.Set("startDate.month", fmt.Sprint(startDateMonth))
	return c
}

// StartDateYear sets the optional parameter "startDate.year": Year of the
// date. Must be from 1 to 9999, or 0 to specify a date without a year.
func (c *AccountsReportsSavedGenerateCsvCall) StartDateYear(startDateYear int64) *AccountsReportsSavedGenerateCsvCall {
	c.urlParams_.Set("startDate.year", fmt.Sprint(startDateYear))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsReportsSavedGenerateCsvCall) Fields(s ...googleapi.Field) *AccountsReportsSavedGenerateCsvCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsReportsSavedGenerateCsvCall) IfNoneMatch(entityTag string) *AccountsReportsSavedGenerateCsvCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsReportsSavedGenerateCsvCall) Context(ctx context.Context) *AccountsReportsSavedGenerateCsvCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsReportsSavedGenerateCsvCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsReportsSavedGenerateCsvCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}/saved:generateCsv")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.reports.saved.generateCsv", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.reports.saved.generateCsv" call.
// Any non-2xx status code is an error. Response headers are in either
// *HttpBody.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsReportsSavedGenerateCsvCall) Do(opts ...googleapi.CallOption) (*HttpBody, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &HttpBody{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.reports.saved.generateCsv", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsReportsSavedListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists saved reports.
//
//   - parent: The account which owns the collection of reports. Format:
//     accounts/{account}.
func (r *AccountsReportsSavedService) List(parent string) *AccountsReportsSavedListCall {
	c := &AccountsReportsSavedListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// reports to include in the response, used for paging. If unspecified, at most
// 10000 reports will be returned. The maximum value is 10000; values above
// 10000 will be coerced to 10000.
func (c *AccountsReportsSavedListCall) PageSize(pageSize int64) *AccountsReportsSavedListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListSavedReports` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListSavedReports` must match the call that provided the page token.
func (c *AccountsReportsSavedListCall) PageToken(pageToken string) *AccountsReportsSavedListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsReportsSavedListCall) Fields(s ...googleapi.Field) *AccountsReportsSavedListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsReportsSavedListCall) IfNoneMatch(entityTag string) *AccountsReportsSavedListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsReportsSavedListCall) Context(ctx context.Context) *AccountsReportsSavedListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsReportsSavedListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsReportsSavedListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/reports/saved")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.reports.saved.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.reports.saved.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListSavedReportsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsReportsSavedListCall) Do(opts ...googleapi.CallOption) (*ListSavedReportsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListSavedReportsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.reports.saved.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsReportsSavedListCall) Pages(ctx context.Context, f func(*ListSavedReportsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsSitesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets information about the selected site.
//
// - name: Name of the site. Format: accounts/{account}/sites/{site}.
func (r *AccountsSitesService) Get(name string) *AccountsSitesGetCall {
	c := &AccountsSitesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsSitesGetCall) Fields(s ...googleapi.Field) *AccountsSitesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsSitesGetCall) IfNoneMatch(entityTag string) *AccountsSitesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsSitesGetCall) Context(ctx context.Context) *AccountsSitesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsSitesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsSitesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.sites.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.sites.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Site.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsSitesGetCall) Do(opts ...googleapi.CallOption) (*Site, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Site{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.sites.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsSitesListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all the sites available in an account.
//
//   - parent: The account which owns the collection of sites. Format:
//     accounts/{account}.
func (r *AccountsSitesService) List(parent string) *AccountsSitesListCall {
	c := &AccountsSitesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of sites
// to include in the response, used for paging. If unspecified, at most 10000
// sites will be returned. The maximum value is 10000; values above 10000 will
// be coerced to 10000.
func (c *AccountsSitesListCall) PageSize(pageSize int64) *AccountsSitesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListSites` call. Provide this to retrieve the subsequent
// page. When paginating, all other parameters provided to `ListSites` must
// match the call that provided the page token.
func (c *AccountsSitesListCall) PageToken(pageToken string) *AccountsSitesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsSitesListCall) Fields(s ...googleapi.Field) *AccountsSitesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsSitesListCall) IfNoneMatch(entityTag string) *AccountsSitesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsSitesListCall) Context(ctx context.Context) *AccountsSitesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsSitesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsSitesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/sites")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "adsense.accounts.sites.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "adsense.accounts.sites.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListSitesResponse.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsSitesListCall) Do(opts ...googleapi.CallOption) (*ListSitesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListSitesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "adsense.accounts.sites.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsSitesListCall) Pages(ctx context.Context, f func(*ListSitesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}
