//---------------------------------------------------------------------------
// Copyright (C) 2000 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
// Read the configuration list from the Weather Station software and 
// create a TAG based file.
//
// Version 2.00
//

#include "msginc.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#define MAX_DATABUF            1024
#define MAX_FILEBUF            5000
#define MAX_FILEBUF            5000
#define COUNT_ROM              64
#define SWITCH_ROM             72
#define TEMP_ROM               80

extern int AddTAGGroup(ushort, TAGType *);
extern int EndTAGGroup(TAGType *);
extern int AddTAGData(ushort, void *, int, TAGType *);
extern int TAGToString(TAGType *, char *, int);
extern int ParseTAG(TAGType *,int,int);
extern int EnterHex(char *, int, ulong *);
extern int EnterNum(char *msg, int, long *, long, long);
extern int EnterString(char *, char *, int, int);
extern int ToHex(char ch);

int iButtonFound(void);
int ParseData(char *, int, uchar *, int);
int ReadRoms(FILE *, uchar *, int);

//----------------------------------------------------------------------
//  This is the Main routine for list2obm.
//
int main(short argc, char **argv)
{
   uchar databuf[MAX_DATABUF];
   int i,j,mcnt,ln;
   long tagtype=0,getstrings=0;
   FILE *infp,*outfp;
   TAGType OutTAG;
   uchar rombuf[88];
   char msg[80],buf[4096];
   ulong ul;
   time_t tlong;

   // check arguments to see if request instruction with '?' or too many
   if (argc < 2)
   {
       printf("\nusage: mktag input_filename <output_filename <tag_type>>\n"
              "  - Create an TAG Tag based on the ROM numbers\n"
              "    in the config file\n"
              "  - argument 1, input_filename: input filename\n"
              "  - optional argument 2, output_filename: output filename\n"
              "  - optional argument 3, tag_type: specifies tag type number\n"
              "  - version 2.00\n");
       exit(1);
   }

   // open the input file
   infp = fopen(argv[1],"r+");
   if(infp == NULL)
   {
      printf("ERROR, Could not open input file!\n");
      exit(1);
   }
   else
      printf("File '%s' opened to read.\n",
              argv[1]);

   // open the output file
   if (argc >= 3)
   {
      outfp = fopen(argv[2],"wb");
      if(outfp == NULL)
      {
         printf("ERROR, Could not open output file!\n");
         exit(1);
      }
      else
         printf("File '%s' opened to write ouput TAG Tag.\n",
                 argv[2]);
   }
   else
   {
      // use standard out
      printf("Output will only be sent to standard out\n");      
      outfp = NULL;
   }

   if (argc >= 4)
      tagtype = atoi(argv[3]);
   else
   {
      // prompt to erase current mission
      if (!EnterNum("\n(0) Weather Station (01,01,01,01,01,01,01,01,1D,12,10)\n"
         "(1) Single temp (10)\n"
         "(2) Dual Temp (10,10)\n\n"
         "Enter the tag type: ",1, &tagtype, 0, 2))
         exit(1);

      // prompt to see if 
      if (!EnterNum("Prompt custom settings?\n  (1) yes\n  (0) no (use default)\nAnswer:",1, &getstrings, 0, 1))
         exit(1);
   }   

   // print tag type being used
   printf("Tag Type: %d\n\n",tagtype);   

   // create a TAG using the ROM info
   switch (tagtype)
   {
      case 0:
         // Big weather Station
         // read the rom numbers from the input file
         ReadRoms(infp, rombuf, 11);
         // {G: ParseData} [
         AddTAGGroup(GD_ParseData,&OutTAG);
            // {D: Description ()}
            mcnt = sprintf(msg,"1-Wire Weather Station");
            if (getstrings)
               mcnt = EnterString("Weather Station, Description",
                      msg,0,80);
            if (mcnt)
               AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);
            // {D: Cluster ID ()}
            for (i = 6; i >= 0; i--)
               if (rombuf[SWITCH_ROM + i] != 0)
                  break;
            ln = i + 1; j = 0;
            for (; i >= 0; i--)
               databuf[j++] = rombuf[SWITCH_ROM + i];
            AddTAGData(DD_ClusterID,databuf,ln,&OutTAG);
            // {D: Cluster Revision ()}
            ul = 0x00;
            if (getstrings)
               EnterNum("Weather Station Cluster, Revision", 3, &ul, 0, 255);
            AddTAGData(DD_ClusterRev,&ul,1,&OutTAG);
            // {D: DD_Manufacturer ()}
            mcnt = sprintf(msg,"Dallas Semiconductor");
            if (getstrings)
               mcnt = EnterString("Weather Station, Manufacturer",
                      msg,0,80);
            if (mcnt)
               AddTAGData(DD_Manufacturer,&msg[0],strlen(msg),&OutTAG);
            // {D: DD_ManufacturerCode ()}
            ul = 0;
            if (getstrings)
               EnterHex("Weather Station, Manufacturer Code",8,&ul);
            for (i = 0; i < 4; i++)
            {
               databuf[3 - i] = (uchar)(ul & 0xFF);
               ul >>= 8;
            }
            AddTAGData(DD_ManufacturerCode,databuf,4,&OutTAG);

            // {D: DD_SecondsSince1970 ()}
            time(&tlong);
            AddTAGData(DD_SecondsSince1970,&tlong,4,&OutTAG);
            // {G: OWCluster} [
            AddTAGGroup(GD_OWCluster,&OutTAG);
               // {D: Description ()}
               sprintf(msg,"Isolation Branch");  
               if (getstrings)
                  mcnt = EnterString("Weather Station, Branch for Wind Direction",
                         msg,0,80);
               if (mcnt)
                  AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);
               // {D: ROM Array (8)}
               AddTAGData(DD_RomData,&rombuf[SWITCH_ROM],8,&OutTAG);
               // {D: ChannelMask (1) }
               ul = 0x02;
               AddTAGData(DD_ChannelMask,&ul,1,&OutTAG);
               // {D: ChannelState (1) }
               ul = 0x02;
               AddTAGData(DD_ChannelState,&ul,1,&OutTAG);
               // {D: AccessMethod () }
               ul = AM_SWITCH_2406;
               AddTAGData(DD_AccessMethod,&ul,1,&OutTAG);
               // {G: OWBranch} [
               AddTAGGroup(GD_OWBranch,&OutTAG);
                  // {G: OWSensor} [
                  AddTAGGroup(GD_OWSensor,&OutTAG);
                     // {D: Description ()}
                     sprintf(msg,"Wind Direction");  
                      if (getstrings)
                        mcnt = EnterString("Weather Station, Wind Direction Sensor",
                               msg,0,80);
                     if (mcnt)
                        AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);
                     // {D: AccessMethod () }
                     ul = AM_ORDER_ROM_SWITCH_LIST;
                     AddTAGData(DD_AccessMethod,&ul,1,&OutTAG);
                     // {D: Init State ()}
                     AddTAGData(DD_InitState,&rombuf[0],64,&OutTAG);
                  // ]
                  EndTAGGroup(&OutTAG);
               // ]
               EndTAGGroup(&OutTAG);

               // {G: OWSensor} [
               AddTAGGroup(GD_OWSensor,&OutTAG);
                  // {D: Description ()}
                  sprintf(msg,"Enclosure");  
                  if (getstrings)
                     mcnt = EnterString("Weather Station, temperature in enclosure ",
                            msg,0,80);
                  if (mcnt)
                     AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);
                  // {D: ROM Array (8)}
                  AddTAGData(DD_RomData,&rombuf[TEMP_ROM],8,&OutTAG);
                  // {D: AccessMethod () }
                  ul = AM_TEMPERATURE_1820;
                  AddTAGData(DD_AccessMethod,&ul,1,&OutTAG);
               // ]
               EndTAGGroup(&OutTAG);

               // {G: OWSensor} [
               AddTAGGroup(GD_OWSensor,&OutTAG);
                  // {D: Description ()}
                  sprintf(msg,"Wind Speed");  
                  if (getstrings)
                     mcnt = EnterString("Weather Station, Wind Speed sensor",
                            msg,0,80);
                  if (mcnt)
                     AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);
                  // {D: ROM Array (8)}
                  AddTAGData(DD_RomData,&rombuf[COUNT_ROM],8,&OutTAG);
                  // {D: Unit Name ()}
                  sprintf(msg,"MPH");  
                  AddTAGData(DD_UnitName,&msg[0],strlen(msg),&OutTAG);
                  // {D: Scale FactorM (1) }
                  ul = 0x2FE9;
                  AddTAGData(DD_ScaleFactorM,&ul,2,&OutTAG);
                  // {D: Scale FactorD (1) }
                  ul = 0x2710;
                  AddTAGData(DD_ScaleFactorD,&ul,2,&OutTAG);
                  // {D: Init State ()}
                  msg[0] = 15;
                  AddTAGData(DD_InitState,&msg[0],1,&OutTAG);
                  // {D: AccessMethod () }
                  ul = AM_COUNT_VELOCITY_2423;
                  AddTAGData(DD_AccessMethod,&ul,1,&OutTAG);
               // ]
               EndTAGGroup(&OutTAG);
            // ]
            EndTAGGroup(&OutTAG);
         // ]
         EndTAGGroup(&OutTAG);
         break;

      case 1:
         // Single temp
         // read the rom numbers from the input file
         ReadRoms(infp, rombuf, 1);
         // {G: ParseData} [
         AddTAGGroup(GD_ParseData,&OutTAG);

            // {D: Description ()}
            mcnt = sprintf(msg,"Single-Temperature Cluster");  
            if (getstrings)
               mcnt = EnterString("Single-Temperature Cluster, Description",
                      msg,0,80);
            if (mcnt)
               AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);

            // {D: Cluster ID ()}
            for (i = 6; i >= 0; i--)
               if (rombuf[i] != 0)
                  break;
            ln = i + 1; j = 0;
            for (; i >= 0; i--)
               databuf[j++] = rombuf[i];
            AddTAGData(DD_ClusterID,databuf,ln,&OutTAG);
            // {D: Cluster Revision ()}
            ul = 0x00;
            if (getstrings)
               EnterNum("Single-Temperature Cluster, Revision", 3, &ul, 0, 255);
            AddTAGData(DD_ClusterRev,&ul,1,&OutTAG);

            // {G: OWCluster} [
            AddTAGGroup(GD_OWCluster,&OutTAG);
               // {G: OWSensor} [
               AddTAGGroup(GD_OWSensor,&OutTAG);
                  // {D: Description ()}
                  sprintf(msg,"Freezer");  
                  if (getstrings)
                     mcnt = EnterString("Single-Temperature Cluster, temperature",
                            msg,0,80);
                  if (mcnt)
                     AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);
                  // {D: ROM Array (8)}
                  AddTAGData(DD_RomData,&rombuf[0],8,&OutTAG);
                  // {D: AccessMethod () }
                  ul = AM_TEMPERATURE_1820;
                  AddTAGData(DD_AccessMethod,&ul,1,&OutTAG);
               // ]
               EndTAGGroup(&OutTAG);
            // ]
            EndTAGGroup(&OutTAG);
         // ]
         EndTAGGroup(&OutTAG);
         break;

      case 2:
         // Dual Temp
         // read the rom numbers from the input file
         ReadRoms(infp, rombuf, 2);
         // {G: ParseData} [
         AddTAGGroup(GD_ParseData,&OutTAG);

            // {D: Description ()}
            mcnt = sprintf(msg,"Dual-Temperature Cluster");  
            if (getstrings)
               mcnt = EnterString("Dual-Temperature Cluster, Description",
                      msg,0,80);
            if (mcnt)
               AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);

            // {D: Cluster ID ()}
            for (i = 6; i >= 0; i--)
               if (rombuf[i] != 0)
                  break;
            ln = i + 1; j = 0;
            for (; i >= 0; i--)
               databuf[j++] = rombuf[i];
            AddTAGData(DD_ClusterID,databuf,ln,&OutTAG);
            // {D: Cluster Revision ()}
            ul = 0x00;
            if (getstrings)
               EnterNum("Dual-Temperature Cluster, Revision", 3, &ul, 0, 255);
            AddTAGData(DD_ClusterRev,&ul,1,&OutTAG);
            // {D: Enumeration ()}
            ul = 0x14;
            AddTAGData(DD_Enum,&ul,1,&OutTAG);
            // {G: OWCluster} [
            AddTAGGroup(GD_OWCluster,&OutTAG);
               // {G: OWSensor} [
               AddTAGGroup(GD_OWSensor,&OutTAG);
                  // {D: Description ()}
                  sprintf(msg,"Inside");  
                  if (getstrings)
                     mcnt = EnterString("Dual-Temperature Cluster, temperature #1",
                            msg,0,80);
                  if (mcnt)
                     AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);
                  // {D: ROM Array (8)}
                  AddTAGData(DD_RomData,&rombuf[0],8,&OutTAG);
                  // {D: AccessMethod () }
                  ul = AM_TEMPERATURE_1820;
                  AddTAGData(DD_AccessMethod,&ul,1,&OutTAG);
               // ]
               EndTAGGroup(&OutTAG);

               // {G: OWSensor} [
               AddTAGGroup(GD_OWSensor,&OutTAG);
                  // {D: Description ()}
                  sprintf(msg,"Outside");  
                  if (getstrings)
                     mcnt = EnterString("Dual-Temperature Cluster, temperature #1",
                            msg,0,80);
                  if (mcnt)
                     AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);
                  // {D: ROM Array (8)}
                  AddTAGData(DD_RomData,&rombuf[8],8,&OutTAG);
                  // {D: AccessMethod () }
                  ul = AM_TEMPERATURE_1820;
                  AddTAGData(DD_AccessMethod,&ul,1,&OutTAG);
               // ]
               EndTAGGroup(&OutTAG);
            // ]
            EndTAGGroup(&OutTAG);
         // ]
         EndTAGGroup(&OutTAG);
         break;
      default:
         printf("Incorrect tag type %d\n",tagtype);
         exit(1);
         break;
   }   

   // print out the TAG Tag in human readable form
   TAGToString(&OutTAG,buf,sizeof(buf));
   printf("\n%s\n",buf);

   // write the output file
   if (outfp != NULL)
   {   
      // write to the output file
      fwrite(&OutTAG.msg[0], 1, OutTAG.msglen, outfp);
      // close output file
      printf("File '%s' closed.\n",
              argv[2]);
      fclose(outfp);
   }
   
   // normal exit
   exit(0);

   return 0;
}


//----------------------------------------------------------------------
// Read the specified number of ROM numbers from the input file
//
int ReadRoms(FILE *infp, uchar *rombuf, int NUMROMS)
{
   char filebuf[MAX_FILEBUF];
   uchar databuf[MAX_DATABUF];
   int i,j,fllen,buflen;

   // read the input file 
   fllen = fread(filebuf,1,MAX_FILEBUF,infp);

   // clean up opened files
   fclose(infp);
   printf("Input File closed.\n\n");

   // attempt to convert the text into an array of uchars 
   buflen = ParseData(filebuf, fllen, databuf, MAX_DATABUF);

   // verify correct length
   if (buflen != NUMROMS * 8)
   {
      printf("ERROR, input file is the incorrect length!\n");
      exit(1);
   }

   // reverse order of ROM's for ease of use
   for (i = 0; i < NUMROMS; i++)
   {
      for (j = 0; j < 8; j++)
         rombuf[7 - j + 8 * i] = databuf[j + 8 * i];
   }

   return buflen;
}


//----------------------------------------------------------------------
// Parse the raw file data in to an array of uchar's.  The data must
// be hex characters possible seperated by white space
//     12 34 456789ABCD EF
//     FE DC BA 98 7654 32 10
// would be converted to an array of 16 uchars.
// return the array length.  If an invalid hex pair is found then the
// return will be 0.
//
int ParseData(char *inbuf, int insize, uchar *outbuf, int maxsize)
{
   int ps, outlen=0, gotmnib=0;
   uchar mnib;

   // loop until end of data
   for (ps = 0; ps < insize; ps++)
   {
      // check for white space 
      if (isspace(inbuf[ps]))
         continue;
      // not white space, make sure hex
      else if (isxdigit(inbuf[ps]))
      {
         // check if have first nibble yet
         if (gotmnib)
         {  
            // this is the second nibble
            outbuf[outlen++] = (mnib << 4) | ToHex(inbuf[ps]);
            gotmnib = 0;
         }
         else
         {
            // this is the first nibble
            mnib = ToHex(inbuf[ps]);
            gotmnib = 1;
         }
      }
      else
         return 0;

      // if getting to the max return what we have
      if ((outlen + 1) >= maxsize)
         return outlen;
   }

   return outlen;
}



